/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client RPC over TCP
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccapi.h"
#include "nqapi.h"
#include "cmcrypt.h"
#include "cmapi.h"
#include "ccdcerpc.h"
#include "cmbufman.h"
#include "cmrpcdef.h"

#include "ccfile.h"
#include "cmthread.h"
#include "ccparams.h" 
#include "ccwrite.h"
#include "ccutils.h"
#include "ccrpc.h"
#include "ccepm.h"
#include "ccnetlgn.h"
#include "amspnego.h"
#include "amntlmss.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT
#ifdef UD_CC_INCLUDERPC


#define NOPAYLOAD_BUFFERSIZE    1024        /* buffer size for messages with no call payload */
#define MAX_FRAGMENT            65448       /*4280 */      /* max fragment size for calls */

#define PDU_HEADER_OPCODE_SIZE              2
#define PDU_HEADER_SIZE_MINUS_OPNUM         22                                                               /* not including the opnum */
#define PDU_HEADER_SIZE                     (PDU_HEADER_SIZE_MINUS_OPNUM + PDU_HEADER_OPCODE_SIZE)           /* including the opnum */

#define PDU_SECTRAILER_SIZE                 8

/* -- Static data -- */
static NQ_BOOL isModuleInitialized = FALSE;
static CMList rpcServers;
static NQ_COUNT callId;
static const CMRpcDcerpcSyntaxId transferSyntax = {
    CM_RPC_TRANSFERSYNTAXSIGNATURE,
    CM_RPC_NDRVERSION
};

static NQ_BOOL dceRpcWrite(CCPipe *pPipe, NQ_BYTE *buffer, NQ_UINT count, NQ_UINT *writtenSize)
{
    NQ_INT i;                       /* retry counter */
    CCRPCServer *pRpcServer;        /* pointer to server */
    NQ_BOOL result = FALSE;         /* return result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pPipe:%p buff:%p count:%u written:%p", pPipe, buffer, count, writtenSize);

    if (NULL == pPipe)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pRpcServer = pPipe->server;
    if (NULL == pRpcServer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid server handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    for (i = CC_CONFIG_RETRYCOUNT; i > 0; i--)
    {
        if (TRUE == syIsValidSocket(pRpcServer->socket))
        {
            NQ_INT msgLen = sySendSocket(pRpcServer->socket, buffer, count);
            if (NQ_FAIL != msgLen)
            {
                result = TRUE;
                break;
            }
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

static NQ_BOOL dceRpcRead(CCPipe *pPipe, NQ_BYTE *buffer, NQ_UINT count, NQ_UINT *readSize)
{
    CCRPCServer *pRpcServer;            /* pointer to server */
    NQ_BOOL result = FALSE;             /* return value */
    NQ_INT read = 0;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pipe:%p buff:%p count:%u size:%p", pPipe, buffer, count, readSize);

    if (NULL == pPipe)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pRpcServer = pPipe->server;
    if (NULL == pRpcServer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid server handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (NULL != readSize)
    {
        *readSize = 0;
    }

    read = syRecvSocketWithTimeout(pRpcServer->socket, buffer, count, UD_NQ_TCPSOCKETRECVTIMEOUT);
    if ((NQ_FAIL == read) || (0 == read))
    {
        goto Exit;
    }

Exit:
    if (NULL != readSize)
    {
        *readSize = (NQ_UINT)read;
    }
    result = (read != 0);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s, read:%d", result ? "TRUE" : "FALSE", read);
    return result;
}


static NQ_BOOL dcerpcOverTcpBindAuth(NQ_HANDLE handle, AuthParams *pAuthBlob)
{
    CMBufferWriter writer;                 /* writer for bind PDU */
    NQ_BYTE * pBuf = NULL;                 /* read and write buffer */
    NQ_COUNT dataLen;                      /* buffer length or data length into the buffer */
    NQ_BYTE * pFragLength;                 /* pointer to the fragment length field in the header */
    NQ_BOOL result = FALSE;
    CCPipe * pPipe;
    NQ_UINT32 authPad = 0;
    NQ_BYTE *pAuthHeader = NULL;
    NQ_BYTE *pAuthData = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p  pAuthBlob:%p", handle, pAuthBlob);

    if ((NULL == handle) || (NULL == pAuthBlob) || (NULL == pAuthBlob->blobOut.data))
    {
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid handle");
        goto Exit;
    }

    pPipe = (CCPipe *)handle;
    /* allocate buffer of enough size for both request and response */
    pBuf = cmBufManTake(NOPAYLOAD_BUFFERSIZE);
    if (NULL == pBuf)
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
        goto Exit;
    }

    /* compose Auth request */
    /* header */
    dataLen = NOPAYLOAD_BUFFERSIZE;
    cmBufferWriterInit(&writer, pBuf, dataLen);
    pFragLength = ccDcerpcCreateFragmentHeader(&writer, CM_RP_PKT_AUTH3, 0, CM_RP_PFCFLAG_FIRST | CM_RP_PFCFLAG_LAST, (NQ_UINT16)pAuthBlob->blobOut.len, 0);
    cmBufferWriteZeroes(&writer, 4);                    /* reserved? */
    /* auth portion */
    pAuthHeader = cmBufferWriterGetPosition(&writer);
    cmBufferWriterSkip(&writer, 8);
    authPad = cmBufferWriterAlign(&writer, cmBufferWriterGetStart(&writer), 4);
    pAuthData = cmBufferWriterGetPosition(&writer);
    cmBufferWriterSetPosition(&writer, pAuthHeader);
    cmBufferWriteByte(&writer, pAuthBlob->type);        /* auth type */
    cmBufferWriteByte(&writer, pAuthBlob->level);       /* auth level */
    cmBufferWriteByte(&writer, (NQ_BYTE)authPad);       /* authPad */
    cmBufferWriteByte(&writer, 0);                      /* reserved */
    cmBufferWriteUint32(&writer, pPipe->authCtxId);     /* auth ctx ID */
    /* copy auth blob */
    cmBufferWriterSetPosition(&writer, pAuthData);
    cmBufferWriteBytes(&writer, pAuthBlob->blobOut.data, pAuthBlob->blobOut.len);
    /* update fragment length */
    ccDcerpcSetFragmentLength(&writer, pFragLength);
    dataLen = cmBufferWriterGetDataCount(&writer);

    /* send */
    if (FALSE == dceRpcWrite(pPipe, pBuf, dataLen, &dataLen))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending Auth request");
        goto Exit;
    }
    /* no response expected on success*/
    /* seems no response is expected at all at this step*/
    syMemcpy(pPipe->sessionKey, amSpnegoClientGetMacSessionKey(pAuthBlob->securityContext)->data, sizeof(pPipe->sessionKey));
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "sessionKey", pPipe->sessionKey, sizeof(pPipe->sessionKey));
    cmCalculateNtlmSigningKey(pPipe->sessionKey, pPipe->recvSigningKey, CM_CRYPT_NTLM_FROM_SERVER_SIGNING);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "recvSigningKey", pPipe->recvSigningKey, sizeof(pPipe->recvSigningKey));
    cmCalculateNtlmSigningKey(pPipe->sessionKey, pPipe->sendSigningKey, CM_CRYPT_NTLM_TO_SERVER_SIGNING);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "sendSigningKey", pPipe->sendSigningKey, sizeof(pPipe->sendSigningKey));
    cmCalculateNtlmSigningKey(pPipe->sessionKey, pPipe->recvSealingState, CM_CRYPT_NTLM_FROM_SERVER_SEALING);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "recvSealingState", pPipe->recvSealingState, sizeof(pPipe->recvSealingState));
    cmCalculateNtlmSigningKey(pPipe->sessionKey, pPipe->sendSealingState, CM_CRYPT_NTLM_TO_SERVER_SEALING);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "sendSealingState", pPipe->sendSealingState, sizeof(pPipe->sendSealingState));

    result = TRUE;
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}


NQ_BOOL ccDcerpcOverTcpStart(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (TRUE == isModuleInitialized)
    {
       goto Exit;
    }

    callId = 1;
    cmListStart(&rpcServers);
#ifdef UD_NQ_INCLUDETRACE
    rpcServers.name = "rpcServers";
#endif /* UD_NQ_INCLUDETRACE */
    isModuleInitialized = TRUE;

 Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return TRUE;
}

void ccDcerpcOverTcpShutdown(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (TRUE == isModuleInitialized)
    {
        cmListShutdown(&rpcServers);
        isModuleInitialized = FALSE;
        dumpRpcServers(SY_LOG_FUNCTION);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}


/*
 * Explicitly dispose and disconnect rpc server:
 *  - disconnects from the server
 *  - disposes private data
 */
static void disposeRpcServer(CCRPCServer *pRpcServer)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p", pRpcServer);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "About to dispose server %s", pRpcServer->item.name ? cmWDump(pRpcServer->item.name) : "");

    /* close the socket */
    if (TRUE == syIsValidSocket(pRpcServer->socket))
    {
        syCloseSocket(pRpcServer->socket);
        pRpcServer->socket = syInvalidSocket();
    }

    /* release memory */
    if (NULL != pRpcServer->ips)
    {
        cmMemoryFree(pRpcServer->ips);
        pRpcServer->ips = NULL;
    }

    cmListShutdown(&pRpcServer->pipes);
    cmListItemRemoveAndDispose((CMItem *)pRpcServer);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}


/*
 * Callback for server unlock and disposal:
 *  - disconnects from the server
 *  - disposes private data
 */
static NQ_BOOL unlockRpcServerCallback(CMItem * pItem)
{
    disposeRpcServer((CCRPCServer *)pItem);
    return TRUE;
}

/* build fragment auth portion in a buffer (no PDU yet) */
static void dcerpcCreateFragmentAuth(CMBufferWriter * pWriter, NQ_BYTE type, NQ_BYTE level)
{
    cmBufferWriteByte(pWriter, type);                   /* auth type */
    cmBufferWriteByte(pWriter, level);                  /* auth level */
    cmBufferWriteUint16(pWriter, 0);                    /* pad + reserved*/
    cmBufferWriteUint32(pWriter, 1);                    /* call ID 1 always in auth */
}


static CCRPCServer *createNewRpcServer(const NQ_WCHAR *host, const NQ_IPADDRESS *ips, NQ_INT numIps, NQ_PORT port, const CCDcerpcPipeDescriptor *pipeDesc)
{
    CCRPCServer *pRpcServer;
    CCRPCServer *pResult = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "host:%s ips:%p numIps:%d port:%d pipeDesc:%p", cmWDump(host), ips, numIps, port, pipeDesc);

    pRpcServer = (CCRPCServer *)cmListItemCreateAndAdd(&rpcServers, sizeof(CCRPCServer), host, unlockRpcServerCallback, CM_LISTITEM_NOLOCK, FALSE);
    if (NULL == pRpcServer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Error;
    }

    /* initialize server */
    cmListStart(&pRpcServer->pipes);
    pRpcServer->ips = ips;
    pRpcServer->numIps = (NQ_COUNT)numIps;
    pRpcServer->port = port;
    pRpcServer->pipeDesc = pipeDesc;
 #ifdef UD_NQ_INCLUDESMBCAPTURE
    syMemset(&pRpcServer->captureHdr, 0, sizeof(CMCaptureHeader));
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    pResult = pRpcServer;
    goto Exit;

Error:
    syMutexGive(&rpcServers.guard);
    sySetLastError(NQ_ERR_OUTOFMEMORY);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pResult);
    return pResult;
}

static NQ_BOOL connectRpcServer(CCRPCServer *pRpcServer)
{
    NQ_BOOL result = FALSE;          /* return value */
    const NQ_IPADDRESS *ip;
    SYSocketHandle socket;           /* socket handle */
    NQ_INT numIps;                   /* number of resolved IPs */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p", pRpcServer);

    /* IPv4 only */
    socket = syCreateSocket(TRUE, CM_IPADDR_IPV4);
    if (FALSE == syIsValidSocket(socket))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create socket");
        goto Exit;
    }

    for (numIps = (NQ_INT)pRpcServer->numIps, ip = pRpcServer->ips; numIps > 0; numIps--, ip++)
    {
        if (NQ_SUCCESS == syConnectSocket(socket, ip, syHton16(pRpcServer->port)))
        {
            pRpcServer->socket = socket;
            pRpcServer->ip = *ip;
            result = TRUE;
            goto Exit;
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to server");
        }
    }

    syCloseSocket(socket);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

static CCRPCServer *createRpcServer(const NQ_WCHAR *name, const CCDcerpcPipeDescriptor *pipeDesc, NQ_PORT port)
{
    CCRPCServer *pRpcServer = NULL;  /* server pointer */
    NQ_IPADDRESS ip;                 /* server name converted to IP */
    const NQ_IPADDRESS *ips = NULL;  /* array of all server IPs */
    NQ_INT numIps;                   /* number of resolved IPs */
    NQ_BOOL nameIsIp;                /* to distinguish between a name and an IP */
    NQ_BOOL result;                  /* connect result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%s pipeDesc:%p port:%d", cmWDump(name), pipeDesc, port);

    syMutexTake(&rpcServers.guard);

    /* resolve ip if needed */
    nameIsIp = ccUtilsNameToIp(name, &ip);
    if (!nameIsIp)
    {
        /* resolve all host IPs */
        ips = cmResolverGetHostIps(NULL, name, &numIps);
        if (NULL == ips)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Cannot resolve IPs for %s", cmWDump(name));
            sySetLastError(NQ_ERR_BADPATH);
            goto Exit;
        }
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Resolved %d IPs, %s ", numIps, numIps > 0 ? cmIPDump(&ips[0]) : "");
    }
    else
    {
        numIps = 1;
        ips = (NQ_IPADDRESS *)cmMemoryAllocate(sizeof(NQ_IPADDRESS));
        if (NULL == ips)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_NOMEM);
            goto Exit;
        }
        cmWcharToIp((NQ_WCHAR *)name, (NQ_IPADDRESS *)ips);
    }

    /* create new rpc server */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Creating new RPC server: %s", cmWDump(name));
    pRpcServer = createNewRpcServer(name, ips, numIps, port, pipeDesc);
    if (NULL == pRpcServer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to create an rpc server object");
        cmMemoryFree(ips);
        goto Exit;
    }

    /* connect */
    result = connectRpcServer(pRpcServer);
    if (FALSE == result)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to rpc server");
        cmListItemUnlock((CMItem *)pRpcServer);
        pRpcServer = NULL;
        goto Exit;
    }

Exit:
    syMutexGive(&rpcServers.guard);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pRpcServer);
    return pRpcServer;
}


static CCPipe *createPipeOnRpcServer(CCRPCServer *pServer, const NQ_WCHAR *name)
{
    CCPipe *pPipe = NULL;

    pPipe = (CCPipe *)cmListItemCreateAndAdd(&pServer->pipes, sizeof(CCPipe), name, NULL /*unlockCallback*/, CM_LISTITEM_NOLOCK, FALSE);
    if (NULL == pPipe)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    pPipe->server = pServer;
    pPipe->open = TRUE;
    pPipe->disconnected = FALSE;
    pPipe->authType = 0;
    pPipe->authLevel = 0;
    pPipe->flagsAuth = 0;
    pPipe->sendSeqNum = 0;
    pPipe->recvSeqNum = 0;
    pPipe->authCtxId = 0;
    syMemset(pPipe->sessionKey, 0, sizeof(pPipe->sessionKey));
    syMemset(pPipe->sendSigningKey, 0, sizeof(pPipe->sendSigningKey));
    syMemset(pPipe->recvSigningKey, 0, sizeof(pPipe->recvSigningKey));
    syMemset(pPipe->sendSealingState, 0, sizeof(pPipe->sendSealingState));
    syMemset(pPipe->recvSealingState, 0, sizeof(pPipe->recvSealingState));
    syMemset(&pPipe->credential, 0, sizeof(pPipe->credential));
    syMemset(&pPipe->schannel, 0, sizeof(pPipe->schannel));
    pPipe->doSignPduHeader = FALSE;

Exit:
    return pPipe;
}


NQ_HANDLE ccDcerpcOverTcpBind(NQ_IPADDRESS *ip, NQ_PORT port, const NQ_WCHAR *hostName, AuthParams *pAuthBlob, const CCDcerpcPipeDescriptor *pipeDesc)
{
    CCRPCServer *pRpcServer = NULL;         /* pointer to server */
    CCPipe *pPipe = NULL;                   /* pipe handle */
    CCPipe *pPipeResult = NULL;             /* result pipe handle */
    CMRpcDcerpcPacket rpcHeader;            /* fragment header */
    CMRpcPacketDescriptor rpcDescr;         /* incoming packet parser */
    CMBufferWriter writer;                  /* writer for bind PDU */
    NQ_BYTE *pBuf = NULL;                   /* read and write buffer */
    NQ_COUNT dataLen;                       /* buffer length or data length into the buffer */
    NQ_BYTE *pFragLength;                   /* pointer to the fragment length field in the header */
    NQ_UINT16 maxLen;                       /* max fragment length (either xmit or recv) */
    NQ_INT type;                            /* packet type in response */
    NQ_BYTE flags;                          /* rpc flags */
    NQ_BOOL hasAuthBlob = FALSE;            /* whether authentication blob is present and expected from server */
#define RPC_OPENACCESS 0x2019f              /* access mask for opening an RPC pipe as a file */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "ip:%p port:%d hostName:%s pAuthBlob:%p pipeDesc:%p", ip, port, cmWDump(hostName), pAuthBlob, pipeDesc);

    /* resolve the name and connect */
    pRpcServer = createRpcServer(hostName, pipeDesc, port);
    if (NULL == pRpcServer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to connect to %s", cmWDump(hostName));
        goto Error;
    }

    /* create pipe */
    pPipe = createPipeOnRpcServer(pRpcServer, pipeDesc->name);
    if (NULL == pPipe)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create a pipe for %s", cmWDump(pipeDesc->name));
        goto Error;
    }

    /* allocate buffer of enough size for both request and response */
    pBuf = cmBufManTake(NOPAYLOAD_BUFFERSIZE);
    if (NULL == pBuf)
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
        goto Error;
    }

    /* compose Bind request */
    hasAuthBlob = (NULL != pAuthBlob) && (NULL != pAuthBlob->blobOut.data);
    dataLen = NOPAYLOAD_BUFFERSIZE;
    flags = CM_RP_PFCFLAG_FIRST | CM_RP_PFCFLAG_LAST;

    /* negotiate header sign always, server will return it's flag */
    flags |= CM_RP_PFCFLAG_HEADER_SIGN_PENDING_CANCEL;
    cmBufferWriterInit(&writer, pBuf, dataLen);
    pFragLength = ccDcerpcCreateFragmentHeader(&writer, CM_RP_PKT_BIND, 0, flags, (NQ_UINT16)((TRUE == hasAuthBlob) ? pAuthBlob->blobOut.len : 0), 0);
    /* build Bind PDU without context */
    cmBufferWriteUint16(&writer, (NQ_UINT16)MAX_FRAGMENT);     /* max xmit fragment */
    cmBufferWriteUint16(&writer, (NQ_UINT16)MAX_FRAGMENT);     /* max recv fragment */
    cmBufferWriteUint32(&writer, 0);                           /* assocGroupId */
    cmBufferWriteByte(&writer, 1);                             /* num contexts */
    cmBufferWriterAlign(&writer, pBuf, 4);
    /* buld context */
    cmBufferWriteUint16(&writer, 0);                           /* context ID */
    cmBufferWriteByte(&writer, 1);                             /* num transfer syntaxes */
    cmBufferWriterAlign(&writer, pBuf, 4);
    cmBufferWriteUuid(&writer,&pipeDesc->uuid);                /* pipe GUID */
    cmBufferWriteUint32(&writer, pipeDesc->version);           /* major/minor version */
    /* transfer syntax signature */
    cmBufferWriteUint32(&writer, (NQ_UINT32)cmGetSUint32(transferSyntax.uuid.timeLow));
    cmBufferWriteUint16(&writer, (NQ_UINT16)cmGetSUint16(transferSyntax.uuid.timeMid));
    cmBufferWriteUint16(&writer, (NQ_UINT16)cmGetSUint16(transferSyntax.uuid.timeHiVersion));
    cmBufferWriteBytes(&writer, transferSyntax.uuid.clockSeq, sizeof(transferSyntax.uuid.clockSeq));
    cmBufferWriteBytes(&writer, transferSyntax.uuid.node, sizeof(transferSyntax.uuid.node));
    cmBufferWriteUint32(&writer, transferSyntax.interfaceVersion);  /* transfer syntax version */

    if (TRUE == hasAuthBlob)
    {
        /* auth portion */
        dcerpcCreateFragmentAuth(&writer, pAuthBlob->type, pAuthBlob->level);

        /* copy auth blob */
        cmBufferWriteBytes(&writer, pAuthBlob->blobOut.data, pAuthBlob->blobOut.len);
    }

    /* update fragment length */
    ccDcerpcSetFragmentLength(&writer, pFragLength);
    dataLen = cmBufferWriterGetDataCount(&writer);

    /* bind the pipe */
    if (FALSE == dceRpcWrite(pPipe, pBuf, dataLen, &dataLen))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending Bind request");
        goto Error;
    }

    /* analyze the response */
    if (FALSE == dceRpcRead(pPipe, pBuf, NOPAYLOAD_BUFFERSIZE, &dataLen))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving Bind response");
        goto Error;
    }

    type = ccDcerpcParseFragmentHeader(pBuf, &rpcDescr, &rpcHeader);
    if (type != CM_RP_PKT_BINDACK)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to bind to the pipe");
        sySetLastError(NQ_ERR_BADACCESS);
        goto Error;
    }

    /* server supports whole pdu signing */
    if (0 != (rpcHeader.pfcFlags & CM_RP_PFCFLAG_HEADER_SIGN_PENDING_CANCEL))
    {
        pPipe->doSignPduHeader = TRUE;
    }

    cmRpcParseUint16(&rpcDescr, &maxLen);        /* max xmit fragment length */
    pPipe->maxRpcXmit = (NQ_UINT16)(maxLen > MAX_FRAGMENT ? MAX_FRAGMENT : maxLen);
    cmRpcParseUint16(&rpcDescr, &maxLen);        /* max recv fragment length */
    pPipe->maxRpcRecv = (NQ_UINT16)(maxLen > MAX_FRAGMENT ? MAX_FRAGMENT : maxLen);

    if (TRUE == hasAuthBlob)
    {
        NQ_UINT16 scndLen;
        NQ_BYTE numRes;

        if (0 == rpcHeader.authLength)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Missing auth data");
            sySetLastError(NQ_ERR_BADACCESS);
            goto Error;
        }
        pPipe->flagsAuth = pAuthBlob->flags;

        cmRpcParseSkip(&rpcDescr, 4);           /* assoc group */
        cmRpcParseUint16(&rpcDescr, &scndLen);  /* secondary address */
        cmRpcParseSkip(&rpcDescr, scndLen);
        cmRpcAllign(&rpcDescr, 4);
        cmRpcParseByte(&rpcDescr, &numRes);
        cmRpcParseSkip(&rpcDescr, 3);           /* pad */
        for (   ; numRes > 0; numRes--)
        {
            NQ_UINT16 ackResult;

            cmRpcParseUint16(&rpcDescr, &ackResult);
            if (ackResult != CM_RP_ACCEPTANCE)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Invalid auth data, bind ackResult:0x%x", ackResult);
                sySetLastError(NQ_ERR_BADACCESS);
                goto Error;
            }
            cmRpcParseSkip(&rpcDescr, 2);
            cmRpcParseSkip(&rpcDescr, 20); /* uuid and version */
        }
        cmRpcParseByte(&rpcDescr, &pPipe->authType);
        cmRpcParseByte(&rpcDescr, &pPipe->authLevel);

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "authType %d, authLevel %d", pPipe->authType, pPipe->authLevel);
        if ((pPipe->authType != pAuthBlob->type) || (pPipe->authLevel != pAuthBlob->level))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Invalid auth data");
            sySetLastError(NQ_ERR_BADACCESS);
            goto Error;
        }
        cmRpcParseSkip(&rpcDescr, 2);
        cmRpcParseUint32(&rpcDescr, &pPipe->authCtxId);

        /* read blob of length rpcHeader.authLength */
        /* copy auth blob */
        pAuthBlob->blobIn.data = (NQ_BYTE *)cmMemoryAllocate(rpcHeader.authLength);
        if (NULL == pAuthBlob->blobIn.data)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_NOMEM);
            goto Error;
        }
        pAuthBlob->blobIn.len = rpcHeader.authLength;
        cmRpcParseBytes(&rpcDescr, pAuthBlob->blobIn.data, rpcHeader.authLength);
    }
    pPipeResult = pPipe;
    goto Exit;

Error:
    if (NULL != pRpcServer)
    {
        cmListItemUnlock((CMItem *)pRpcServer);
    }
Exit:
    cmBufManGive(pBuf);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pPipeResult);
    return (NQ_HANDLE)pPipeResult;
}


void ccDcerpcOverTcpDisconnect(NQ_HANDLE pipeHandle)
{
    CCPipe *pPipe = (CCPipe *)pipeHandle;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pipe:%p", pipeHandle);

    if (NULL != pPipe->server)
    {
        disposeRpcServer(pPipe->server);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}


#define RPC_AUTH_LEN_NTLMSSP                16

/* NL_AUTH_SIGNATURE used when AES not negotiated */
#define RPC_AUTH_LEN_NTLGN_SIGN                 24
#define RPC_AUTH_LEN_NTLGN_SIGN_AND_SEAL        32

/* NL_AUTH_SHA2_SIGNATURE used when AES negotiated */
#define RPC_AUTH_LEN_NTLGN_AES_SIGN             48
#define RPC_AUTH_LEN_NTLGN_AES_SIGN_AND_SEAL    56


#define SCHANNEL_NTLGN_MAX_SIG_SIZE 56          /* when AES */
#define SCHANNEL_NTLGN_MIN_SIG_SIZE 24          /* when strong keys (non AES) */

static NQ_UINT32 align(NQ_UINT32 what, NQ_UINT32 alignment)
{
    --alignment;

    return (what + alignment) & ~alignment;
}

NQ_BOOL ccDcerpcOverTcpCall(NQ_HANDLE pipeHandle, CCDcerpcRequestCallback request, CCDcerpcResponseCallback response, void *callParams)
{
    CCPipe * pPipe;                /* casted pipe handle */
    NQ_BOOL res = FALSE;           /* operation result */
    NQ_BOOL firstFrag = TRUE;      /* whether fragment is the first one */
    NQ_BYTE * pBuf;                /* read and write buffer */
    NQ_COUNT dataLen;              /* buffer length or data length into the buffer */
    NQ_BOOL moreData;              /* whether application has more data */
    NQ_STATUS status;              /* status of callback execution */
    NQ_BYTE * pStubData;           /* pointer to stub data */
    NQ_COUNT stubLen;              /* stub length supplied by the caller (in callback) */
    NQ_UINT fragLen = 0;           /* whole rpc packet length */
    NQ_COUNT pduLen;               /* PDU length (total length of pdu header, stub data and auth part if present) */
    NQ_UINT16 authLen = 0;         /* auth verifier length */
    NQ_BYTE authPadLen = 0;        /* number of bytes coming after stub data before auth trailer */
    NQ_BYTE *pSig = NULL;          /* pointer to signature start part in the packet (format depends on mechanism used) */
    NQ_BOOL doSeal = FALSE;        /* seal the packet or unseal (encrypted, level privacy) */
    NQ_BOOL doAes = FALSE;         /* security type */
    NQ_BYTE *pDataToSign;          /* data to sign - whole pdu or stub data only */
    NQ_UINT dataLenToSign;         /* data length to sign accordingly */
    NQ_BYTE sequence[8];           /* channel sequence */
    NQ_BOOL result = FALSE;        /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pipe:%p req:%p res:%p params:%p", pipeHandle, request, response, callParams);

    if (NULL == pipeHandle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    /* allocate buffer of enough size for the request */
    pPipe = (CCPipe *)pipeHandle;
    dataLen = pPipe->maxRpcXmit;
    pBuf = cmBufManTake(dataLen);
    if (NULL == pBuf)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    do
    {
        CMBufferWriter writer;          /* writer for bind PDU */
        NQ_BYTE fragType;               /* first and/or last flags */

        /* build header and build Request PDU stub data */
        cmBufferWriterInit(&writer, pBuf, dataLen);
        cmBufferWriterSkip(&writer, PDU_HEADER_SIZE_MINUS_OPNUM); /* skip header for now */

        /* place stub data (also opcode) */
        moreData = FALSE;
        pStubData = cmBufferWriterGetPosition(&writer);
        stubLen = (*request)(pStubData, dataLen - cmBufferWriterGetDataCount(&writer), callParams, &moreData);
        if (0 == stubLen)
        {
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            res = FALSE;
            break;
        }
        cmBufferWriterSkip(&writer, stubLen);           /* skip stub length */
        stubLen -= PDU_HEADER_OPCODE_SIZE;              /* minus opcode */
        pStubData += PDU_HEADER_OPCODE_SIZE;
        
        pduLen = cmBufferWriterGetDataCount(&writer);

        /* place security trailer and authentication token */
        if (pPipe->authLevel >= CM_RP_AUTHLEVELINTEGRITY)
        {
            /* stub data size (alloc hint) should be 16 bytes aligned, otherwise pad */
            authPadLen = (NQ_BYTE)(align(stubLen, 16) - stubLen);

            switch (pPipe->authType)
            {
            case CM_RP_AUTHTYPENTLMSSP:
                authLen = RPC_AUTH_LEN_NTLMSSP;
                break;
            case CM_RP_AUTHTYPESCHANNEL:
                doSeal = (CM_RP_AUTHLEVELPRIVACY == pPipe->authLevel);
                doAes = 0 != (NETLOGON_NEG_FLAGS_SUPPORTS_AES & pPipe->schannel.creds.negotFlags);
                authLen = (TRUE == doSeal) ? ((TRUE == doAes) ? RPC_AUTH_LEN_NTLGN_AES_SIGN_AND_SEAL : RPC_AUTH_LEN_NTLGN_SIGN_AND_SEAL) :
                                             ((TRUE == doAes) ? RPC_AUTH_LEN_NTLGN_AES_SIGN_AND_SEAL : RPC_AUTH_LEN_NTLGN_SIGN);
                break;
            default:
                authLen = 0;
                break;
            }
        }
                
        /* re-build header and set fragment length */
        fragType = 0;
        if (TRUE == firstFrag)
        {
            fragType |= CM_RP_PFCFLAG_FIRST;
            firstFrag = FALSE;
        }
        if (FALSE == moreData)
        {
            fragType |= CM_RP_PFCFLAG_LAST;
        }

        /* write header */
        cmBufferWriterSetPosition(&writer, cmBufferWriterGetStart(&writer));
        /* calculate whole rpc packet length */
        fragLen = (NQ_UINT)((pPipe->authLevel >= CM_RP_AUTHLEVELINTEGRITY) ? (pduLen + authPadLen + PDU_SECTRAILER_SIZE + authLen) : pduLen);
        ccDcerpcCreateFragmentHeader(&writer, CM_RP_PKT_REQUEST, (NQ_UINT16)fragLen, fragType, authLen, stubLen);
        cmBufferWriteUint16(&writer, 0);     /* context id - we always offer 1 context, id 0 */

        /* write auth data */
        /* integrity: sign */
        if (pPipe->authLevel >= CM_RP_AUTHLEVELINTEGRITY)
        {
            /* set writer at auth trailer start */
            cmBufferWriterSetPosition(&writer, cmBufferWriterGetStart(&writer) + pduLen + authPadLen);
            cmBufferWriteByte(&writer, pPipe->authType);
            cmBufferWriteByte(&writer, pPipe->authLevel);
            cmBufferWriteByte(&writer, authPadLen);                     /* pad len */
            cmBufferWriteByte(&writer, 0);                              /* reserved */
            cmBufferWriteUint32(&writer, pPipe->authCtxId);             /* auth ctx id */

            switch (pPipe->authType)
            {
                case CM_RP_AUTHTYPENTLMSSP:
                {
                    /* write 14 bytes of ntlmssp signature */
                    cmBufferWriteUint32(&writer, 1);    /* verifier version */
                    pSig = cmBufferWriterGetPosition(&writer);

                    if (pPipe->flagsAuth & NTLMSSP_NEGOTIATE_EXTENDED_SECURITY)
                    {
                        cmCalculateDcerpcSignature(cmBufferWriterGetStart(&writer), (NQ_UINT16)(fragLen - authLen), pPipe->sendSigningKey, pPipe->sendSealingState, pPipe->sendSeqNum, TRUE, pSig);
                        cmBufferWriterSkip(&writer, NTLMSSP_SIGNATURE_LENGTH); /* skip the signature */
                        cmBufferWriteUint32(&writer, pPipe->sendSeqNum++);  /* auth tail */
                    }

                    /* privacy: integrity (sign) and also seal (encrypt) the stub data */
                    if (pPipe->authLevel == CM_RP_AUTHLEVELPRIVACY)
                    {
                        /* seal (encrypt) stub data */
                        cmArcfourCryptWithState(pStubData, stubLen + authPadLen, pPipe->sendSealingState);
                        if (0 != (pPipe->flagsAuth & NTLMSSP_NEGOTIATE_KEY_EXCH))
                        {
                            /* finally seal (encrypt) signature */
                            cmArcfourCryptWithState(pSig, NTLMSSP_SIGNATURE_LENGTH, pPipe->sendSealingState);
                        }
                    }
                    break;
                }
                case CM_RP_AUTHTYPESCHANNEL:
                {
                    /* sign always, first check whether to sign the whole pdu (minus the auth trailer) or just the stub data  */
                    pDataToSign = pPipe->doSignPduHeader ? cmBufferWriterGetStart(&writer) : pStubData;
                    dataLenToSign = pPipe->doSignPduHeader ? (fragLen - authLen): stubLen;

                    /* pointer to auth trailer in packet */
                    pSig = cmBufferWriterGetPosition(&writer);

                    /* sign the data - place the calculated data checksum and confounder in the auth trailer */
                    ccSchannelSign(&pPipe->schannel, pDataToSign, dataLenToSign, doSeal, pSig, authLen, TRUE);

                    /* generate next schannel sequence number based on current */
                    ccSchannelGenerateSequenceNumber(sequence, pPipe->schannel.sequence, TRUE);

                    if (TRUE == doSeal)
                    {
                        /* seal the data - encrypt the stub data, place modified confounder in the auth trailer */
                        ccSchannelSeal(&pPipe->schannel, pStubData, stubLen + authPadLen, sequence, pSig, authLen, TRUE);
                    }

                    /* channel sequence number - encrypt always the sequence number with session key and calculated signature, write into packet */
                    ccSchannelSequence(&pPipe->schannel, sequence, pSig);

                    break;
                }
                default:
                {
                    break;
                }
            } /* end of switch (pPipe->authType) */
        } /* end of if (pPipe->authLevel >= CM_RP_AUTHLEVELINTEGRITY) */

        res = dceRpcWrite(pPipe, pBuf, fragLen, (NQ_UINT *)&fragLen);
    }
    while ((TRUE == res) && (TRUE == moreData));

    cmBufManGive(pBuf);

    if (FALSE == res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending request");
        goto Exit;
    }

    /* allocate buffer of enough size for the response */
    dataLen = pPipe->maxRpcRecv;
    pBuf = cmBufManTake(dataLen);
    if (NULL == pBuf)
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
        goto Exit;
    }

    /* loop on recv */
    do
    {
        CMRpcPacketDescriptor reader;      /* incoming packet parser */
        CMRpcDcerpcPacket rpcHeader;       /* fragment header */
        NQ_BYTE cancelCount;               /* cancel count from PDU */
        NQ_INT type;                       /* packet type in response */

        res = dceRpcRead(pPipe, pBuf, pPipe->maxRpcRecv, &dataLen);
        if (FALSE == res)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Error reading response");
            break;
        }
        fragLen = dataLen;

        /* parse header and withdraw packet type */
        type = ccDcerpcParseFragmentHeader(pBuf, &reader, &rpcHeader);
        if (type != CM_RP_PKT_RESPONSE)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Received packet is not Response");
            sySetLastError(NQ_ERR_BADPARAM);
            res = FALSE;
            break;
        }
        authPadLen = 0;
        moreData = (0 == (rpcHeader.pfcFlags & CM_RP_PFCFLAG_LAST));

        /* parse PDU */
        cmRpcParseUint32(&reader, &stubLen);  /* alloc hint is stub len for non fragmented */
        cmRpcParseSkip(&reader, 2);           /* skip context id */
        cmRpcParseByte(&reader, &cancelCount);
        if (0 != cancelCount)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Packet received with cancel count: %d", cancelCount);
            sySetLastError(NQ_ERR_BADPARAM);
            res = FALSE;
            break;
        }
        if ((pPipe->authLevel >= CM_RP_AUTHLEVELINTEGRITY) && (0 == rpcHeader.authLength))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Auth data missing");
            sySetLastError(NQ_ERR_BADPARAM);
            res = FALSE;
            break;
        }
        cmRpcParseSkip(&reader, 1);           /* skip reserved */
        
        pStubData = cmBufferReaderGetPosition(&reader);

        if (pPipe->authLevel >= CM_RP_AUTHLEVELINTEGRITY)
        {
            NQ_BYTE tempByte;
            NQ_UINT32 tempUint32;
            
            cmBufferReaderSetPosition(&reader, cmBufferReaderGetStart(&reader) + rpcHeader.fragLength - rpcHeader.authLength - PDU_SECTRAILER_SIZE);
            cmBufferReadByte(&reader, &tempByte);
            if (pPipe->authType != tempByte)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Wrong auth type: %d", tempByte);
                sySetLastError(NQ_ERR_BADPARAM);
                res = FALSE;
                break;
            }
            cmBufferReadByte(&reader, &tempByte);
            if (pPipe->authLevel != tempByte)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Wrong auth level: %d, expected: %d", tempByte, pPipe->authLevel);
                sySetLastError(NQ_ERR_BADPARAM);
                res = FALSE;
                break;
            }
            cmBufferReadByte(&reader, &authPadLen);               /* auth pad len */
            cmBufferReaderSkip(&reader, 1);                       /* skip reserved */
            cmBufferReadUint32(&reader, &tempUint32);             /* auth ctx id */
            if (pPipe->authCtxId != tempUint32)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Wrong auth context id: %d, expected: %d", tempUint32, pPipe->authCtxId);
                sySetLastError(NQ_ERR_BADPARAM);
                res = FALSE;
                break;
            }

            /* set pointer to auth trailer in packet */
            pSig = cmBufferWriterGetPosition(&reader);
         }

        /* fix stubLen for fragmented (it's not equal to alloc hint in that case) */
        if (TRUE == moreData)
        {
            stubLen = fragLen - PDU_HEADER_SIZE - rpcHeader.authLength - authPadLen - PDU_SECTRAILER_SIZE;
        }

        switch (pPipe->authType)
        {
            case CM_RP_AUTHTYPENTLMSSP:
            {
                if (CM_RP_AUTHLEVELPRIVACY == pPipe->authLevel)
                {
                    /*  first unseal stub data */
                    cmArcfourCryptWithState(pStubData, stubLen + authPadLen, pPipe->recvSealingState);

                    if (0 != (pPipe->flagsAuth & NTLMSSP_NEGOTIATE_KEY_EXCH))
                    {
                        /* unseal signature */
                        cmArcfourCryptWithState(pSig, NTLMSSP_SIGNATURE_LENGTH, pPipe->recvSealingState);
                    }
                }
                if (CM_RP_AUTHLEVELINTEGRITY == pPipe->authLevel)
                {
                    /* check signature */
                    if (0 != (pPipe->flagsAuth & NTLMSSP_NEGOTIATE_EXTENDED_SECURITY))
                    {
                        if (FALSE == cmCheckDcerpcSignature(cmBufferReaderGetStart(&reader), (NQ_UINT16)(rpcHeader.fragLength - rpcHeader.authLength), pPipe->recvSigningKey, pPipe->recvSealingState, pPipe->recvSeqNum++, FALSE, pSig))
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, " signature doesn't match");
                            sySetLastError(NQ_ERR_BADACCESS);
                            res = FALSE;
                            goto Exit;
                        }
                    }
                 }

                break;
            }
            case CM_RP_AUTHTYPESCHANNEL:
            {
                NQ_BYTE authSig[RPC_AUTH_LEN_NTLGN_AES_SIGN_AND_SEAL] = {0};  /* buffer to compare auth trailer using max size */

                doSeal = (CM_RP_AUTHLEVELPRIVACY == pPipe->authLevel);
                doAes = 0 != (NETLOGON_NEG_FLAGS_SUPPORTS_AES & pPipe->schannel.creds.negotFlags);  /* not supported yet */
                authLen = (TRUE == doSeal) ? ((TRUE == doAes) ? RPC_AUTH_LEN_NTLGN_AES_SIGN_AND_SEAL : RPC_AUTH_LEN_NTLGN_SIGN_AND_SEAL) :
                                             ((TRUE == doAes) ? RPC_AUTH_LEN_NTLGN_AES_SIGN_AND_SEAL : RPC_AUTH_LEN_NTLGN_SIGN);

                /* generate next schannel sequence number based on current */
                ccSchannelGenerateSequenceNumber(sequence, pPipe->schannel.sequence, FALSE);

                /* first unseal the packet */
                if (TRUE == doSeal)
                {
                    /* unseal the data - decrypt the stub data, put modified confounder in the auth trailer */
                    ccSchannelSeal(&pPipe->schannel, pStubData, stubLen + authPadLen, sequence, pSig, authLen, FALSE);
                }

                syMemcpy(authSig, pSig, authLen);

                /* always signed, calculate the checksum and compare */
                /* sign the data (places the checksum and encrypted sequence in the auth trailer) */
                pDataToSign = pPipe->doSignPduHeader ? cmBufferWriterGetStart(&reader) : pStubData;
                dataLenToSign = pPipe->doSignPduHeader ? (fragLen - authLen): stubLen;

                ccSchannelSign(&pPipe->schannel, pDataToSign, dataLenToSign, doSeal, pSig, authLen, FALSE);
                /* compare actual checksum in packet with the calculated one */
                if (0 != syMemcmp(pSig + SCHANNEL_CHECKSUM_OFFSET, authSig + SCHANNEL_CHECKSUM_OFFSET, SCHANNEL_CHECKSUM_LENGTH))
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Wrong checksum");
                    sySetLastError(NQ_ERR_ACCESS);
                    res = FALSE;
                    break;
                }

                /* compare actual sequence in packet with the calculated one */
                if (0 != syMemcmp(pSig + SCHANNEL_SEQUENCE_OFFSET, authSig + SCHANNEL_SEQUENCE_OFFSET, SCHANNEL_SEQUENCE_LENGTH))
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Wrong sequence");
                    sySetLastError(NQ_ERR_ACCESS);
                    res = FALSE;
                    break;
                }

                break;
            }
            default:
            {
                break;
            }
        }

        if (FALSE == res)
        {
            break;
        }
        /* finally withdraw stub data */
        cmRpcAllign(&reader, 2);
        status = (*response)(pStubData, stubLen, callParams, moreData);
        if (NQ_SUCCESS != status)
        {
            sySetLastError((NQ_UINT32)status);
            res = FALSE;
            break;
        }
    }
    while ((TRUE == res) && (TRUE == moreData));

    cmBufManGive(pBuf);

    if ((NQ_COUNT)NQ_FAIL == dataLen)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving response");
        goto Exit;
    }

Exit:
    result = res;
    
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

static void composeNLAuthMessage(const NQ_WCHAR *ownHostName, const NQ_WCHAR *ownDomainName, CMBlob *nlAuthMessBlob)
{
    CMBufferWriter writer;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "ownHostName:%p ownDomainName:%p", ownHostName, ownDomainName);

#define NL_AUTH_MESSAGE_HEADER_SIZE             8   /* fixed size */
#define NL_AUTH_MESSAGE_TYPE_REQUEST            0
#define NL_AUTH_MESSAGE_TYPE_RESPONSE           1
#define NL_AUTH_MESSAGE_FLAGS_NETBIOS_HOST      0x02
#define NL_AUTH_MESSAGE_FLAGS_NETBIOS_DOMAIN    0x01

    if ((NULL == ownHostName) || (NULL == ownDomainName))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid parameter");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    nlAuthMessBlob->len = NL_AUTH_MESSAGE_HEADER_SIZE + cmWStrlen(ownHostName) + cmWStrlen(ownDomainName) + 2 * CM_TRAILING_NULL;
    nlAuthMessBlob->data = cmMemoryAllocate(nlAuthMessBlob->len);
    if (NULL == nlAuthMessBlob->data)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
    cmBufferWriterInit(&writer, nlAuthMessBlob->data, nlAuthMessBlob->len);
    cmBufferWriteUint32(&writer, NL_AUTH_MESSAGE_TYPE_REQUEST);
    cmBufferWriteUint32(&writer, NL_AUTH_MESSAGE_FLAGS_NETBIOS_HOST | NL_AUTH_MESSAGE_FLAGS_NETBIOS_DOMAIN);
    /* strings are in order according to flags above */
    cmBufferWriteString(&writer, TRUE, (const NQ_BYTE *)ownDomainName, TRUE, CM_BSF_WRITENULLTERM);
    cmBufferWriteString(&writer, TRUE, (const NQ_BYTE *)ownHostName, TRUE, CM_BSF_WRITENULLTERM);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

NQ_HANDLE ccDcerpcOverTcpConnect(const NQ_WCHAR *serverName, const NQ_WCHAR *ownHostName, const AMCredentials *pCredentials, const CCDcerpcPipeDescriptor *pipeDesc, NQ_BYTE authType)
{
    NQ_BOOL res;                    /* operation result */
    NQ_HANDLE pipeHandle = NULL;    /* handle */
    AuthParams authData;            /* auth data params */
    NQ_PORT port;                   /* obtained by epmapper */
    NQ_IPADDRESS ip;                /* obtained by epmapper */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "serverName:%s ownHostName:%p pCredentials:%p pipeDesc:%p authType:%d", cmWDump(serverName), ownHostName, pCredentials, pipeDesc, authType);

    syMemset(&authData, 0, sizeof(authData));

    /* first connect to server's well known epm port to obtain the ip and port for required pipe */
    if (NQ_FAIL == ccEpmapperGetPort(serverName, pipeDesc, &ip, &port))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get data from epmap on server: %s", cmWDump(serverName));
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "For pipe:%s returned port:%d, ip:%s", cmWDump(pipeDesc->name), port, cmIPDump(&ip));

    /* connects to ip and port and perform bind */
    switch (authType)
    {
        case CM_RP_AUTHTYPENTLMSSP:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "CM_RP_AUTHTYPENTLMSSP");

            authData.securityContext = amSpnegoClientAllocateContext(pCredentials, AM_MAXSECURITYLEVEL, FALSE);
            if (NULL == authData.securityContext)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Error allocating security context");
                goto Exit;
            }

            authData.sessionKey.data = NULL;
            authData.macKey.data = NULL;
            if (FALSE == amSpnegoClientNegotiateSecurity(
                                                    authData.securityContext,
                                                    NULL,
                                                    FALSE,
                                                    serverName,
                                                    &authData.sessionKey,
                                                    &authData.macKey
                                                    )
                )
            {
                amSpnegoClientFreeContext(authData.securityContext);
                if (authData.sessionKey.data != NULL)
                {
                    amSpnegoFreeKey(&authData.sessionKey);
                }
            }

            authData.type = CM_RP_AUTHTYPENTLMSSP;
#ifdef UD_CC_NETLOGONENCRYPT
            authData.level = CM_RP_AUTHLEVELPRIVACY;
#else
            authData.level = CM_RP_AUTHLEVELINTEGRITY;
#endif /* UD_CC_NETLOGONENCRYPT */
            authData.flags = (NTLMSSP_NEGOTIATE_KEY_EXCH | NTLMSSP_NEGOTIATE_128 | NTLMSSP_NEGOTIATE_EXTENDED_SECURITY
                            | NTLMSSP_NEGOTIATE_NTLM | NTLMSSP_NEGOTIATE_ALWAYS_SIGN | NTLMSSP_NEGOTIATE_SIGN
                            | NTLMSSP_NEGOTIATE_SEAL | NTLMSSP_NEGOTIATE_UNICODE | NTLMSSP_NEGOTIATE_REQUEST_TARGET);

            authData.blobOut = amSpnegoClientGenerateFirstBlob(authData.securityContext);
            if (NULL == authData.blobOut.data)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Error generating auth blob");
                goto Exit;
            }

            /* bind with auth */
            pipeHandle = ccDcerpcOverTcpBind(&ip, port, serverName, &authData, pipeDesc);
            cmMemoryFreeBlob(&authData.blobOut);
            if (NULL == pipeHandle)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Error connecting pipe");
                goto Exit;
            }

            authData.blobOut = amSpnegoClientAcceptNextBlob(authData.securityContext, &authData.blobIn);
            cmMemoryFreeBlob(&authData.blobIn);

            res = dcerpcOverTcpBindAuth(pipeHandle, &authData);
            amSpnegoClientFreeContext(authData.securityContext);
            if (FALSE == res)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "AUTH failed");
                goto Error;
            }
            else
            {
                LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "AUTH done");
                goto Exit;
            }

            break;
        }
        case CM_RP_AUTHTYPESCHANNEL:
        {
            NQ_BYTE secret[16];                                         /* hostname secret */
            const NQ_WCHAR *ownDomainName = cmNetBiosGetDomainAuthW();  /* NetBIOS form */
            NQ_UINT32 status;                                           /* operation status */
            CCPipe *pPipe;                                              /* casted pipe handle */
            CCNetlogonCredential credential;                            /* netlogon credentials used in step 1 */

            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "CM_RP_AUTHTYPESCHANNEL");

            /* host must have secret */
            syMemcpy(secret, (NQ_BYTE *)pCredentials, sizeof(secret));

            cmWStrupr((NQ_WCHAR *)serverName);

            /* bind with no auth (unprotected RPC channel) */
            pipeHandle = ccDcerpcOverTcpBind(&ip, port, serverName, NULL, pipeDesc);
            if (NULL == pipeHandle)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Cannot bind");
                goto Exit;
            }
            pPipe = (CCPipe *)pipeHandle;

            dumpRpcServers(SY_LOG_FUNCTION);

            /* Step 1: Session-key negotiation over an unprotected RPC channel. */

            /* generate client random challenge */
            ccNetlogonCredentialsRandom((NQ_BYTE *)&pPipe->credential.client, sizeof(pPipe->credential.client));

            /* send client challenge and get server random challenge */
            status = ccNetrServerReqChallengeEx(pipeHandle, serverName, ownHostName, &pPipe->credential);
            if (NQ_SUCCESS != status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "NetrServerReqChallenge failed");
                goto Error;
            }

            /* try NetrServerAuthenticate3 with AES flags requested */
            ccNetlogonCredentialsInit(&pPipe->credential, secret, NETLOGON_NEG_FLAGS_AUTH3);
            status = ccNetrServerAuthenticate3Ex(pipeHandle, serverName, ownHostName, &pPipe->credential, &pPipe->credential.negotFlags);
            if (NQ_SUCCESS != status)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "NetrServerAuthenticate3 failed");

                /* downgrade to strong key - not supported currently */
#ifndef UD_CC_NETLOGONSTRONGKEYSUPPORTED
                goto Error;
#else
                /* try NetrServerAuthenticate2 (restart the credentials) */
                ccNetlogonCredentialsRandom((NQ_BYTE *)&pPipe->credential.client, sizeof(pPipe->credential.client));

                /* send client challenge and get server random challenge */
                status = ccNetrServerReqChallenge(pipeHandle, serverName, ownHostName, &pPipe->credential);
                if (NQ_SUCCESS != status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "NetrServerReqChallenge failed");
                    goto Error;
                }

                ccNetlogonCredentialsInit(&pPipe->credential, secret, NETLOGON_NEG_FLAGS_AUTH2);
                status = ccNetrServerAuthenticate2(pipeHandle, serverName, ownHostName, &pPipe->credential, &pPipe->credential.negotFlags);
                if (NQ_SUCCESS != status)
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "NetrServerAuthenticate2 failed with error:0x%x, server's flags:0x%x", status, pPipe->credential.flags);
                    goto Error;
                }
#endif /* UD_CC_NETLOGONSTRONGKEYSUPPORTED */
            }

            /* verify server supports AES */
            if (0 == (pPipe->credential.negotFlags & NETLOGON_NEG_FLAGS_SUPPORTS_AES))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Server doesn't support AES, returned flags: 0x%x", pPipe->credential.negotFlags);
#ifndef UD_CC_NETLOGONSTRONGKEYSUPPORTED
                goto Error;
#else
                /* check here strong key is actually supported */
#endif /* UD_CC_NETLOGONSTRONGKEYSUPPORTED */
            }

            /* Step 1 done (session key calculated) */
            dumpNetlogonCredentials("After auth3", &pPipe->credential);
            credential = pPipe->credential;
            cmListItemUnlock((CMItem *)pPipe->server);

            /* Step 2: bind again over protected RPC using schannel */

            /* again connect to server's well known epm port to obtain the ip and port for required pipe */
            if (NQ_FAIL == ccEpmapperGetPort(serverName, pipeDesc, &ip, &port))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get data from epmap on server: %s", cmWDump(serverName));
                goto Exit;
            }
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "For pipe:%s returned port:%d, ip:%s", cmWDump(pipeDesc->name), port, cmIPDump(&ip));

            /* bind with auth */
            authData.type = CM_RP_AUTHTYPESCHANNEL;
#ifdef UD_CC_NETLOGONENCRYPT
            authData.level = CM_RP_AUTHLEVELPRIVACY;
#else
            authData.level = CM_RP_AUTHLEVELINTEGRITY;
#endif /* UD_CC_NETLOGONENCRYPT */
            composeNLAuthMessage(ownHostName, ownDomainName, &authData.blobOut);
            if (NULL == authData.blobOut.data)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Error generating auth blob");
                goto Exit;
            }

            pipeHandle = ccDcerpcOverTcpBind(&ip, port, serverName, &authData, pipeDesc);
            if (NULL == pipeHandle)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Cannot bind secured");
                goto Exit;
            }
            pPipe = (CCPipe *)pipeHandle;

            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Netlogon secure channel with %s established", cmWDump(serverName));

            /* init schannel using credential from step 1 */
            pPipe->authType = authData.type;
            pPipe->authLevel = authData.level;
            pPipe->authCtxId = 1;
            pPipe->schannel.creds = credential;
            cmU64Zero(&pPipe->schannel.sequence);

            dumpRpcServers(SY_LOG_FUNCTION);

            goto Exit;
        }
        default:
        {
            break;
        }
    } /* endof switch (authType) */

Error:
    if (NULL != pipeHandle)
    {
        ccDcerpcOverTcpDisconnect(pipeHandle);
    }

Exit:
    cmMemoryFreeBlob(&authData.blobOut);
    cmMemoryFreeBlob(&authData.blobIn);

    dumpRpcServers(SY_LOG_FUNCTION);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "pipe handle:%p", pipeHandle);
    return pipeHandle;
}

void dumpRpcServers(const NQ_CHAR *text)
{
#ifdef UD_NQ_INCLUDETRACE
    CMIterator iterator;
    CMList *pList = &rpcServers;
    NQ_INT i = 0;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "%s", text);

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "List [%s] %p (first: %p, last: %p):", pList->name, pList, pList->first, pList->last);
    cmListIteratorStart(pList, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        CMItem *pItem = cmListIteratorNext(&iterator);
        if (pItem->master != pList)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "  !! item %p corrupted (list: %p item: %p item->master : %p)",pItem, pList, pItem, pItem->master);
        }
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "  [%04d] item %p next: %p, prev: %p, locks: %d, name: %s", i++, pItem, pItem->next, pItem->prev, pItem->locks, pItem->name != NULL? cmWDump(pItem->name) : "<none>");
        {
            NQ_UINT j;
            NQ_IPADDRESS *pIp;
            CCRPCServer *pRpcServer = (CCRPCServer *)pItem;

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "pipe:%s, port:%d, numIps:%d, socket:%d, locks:%d", cmWDump(pRpcServer->pipeDesc->name), pRpcServer->port, pRpcServer->numIps, pRpcServer->socket, pItem->locks);
            for (j = 0, pIp = (NQ_IPADDRESS *)pRpcServer->ips; j < pRpcServer->numIps; j++, pIp++)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "ip:%s", cmIPDump(pIp));
            }
        }
        if (NULL != pItem->dump)
        {
            pItem->dump(pItem);
        }
        if (pItem->references.first != NULL)
        {
            CMIterator refIterator;
            LOGMSG( CM_TRC_LEVEL_MESS_NORMAL, "  references:");
            cmListIteratorStart(&pItem->references, &refIterator);
            while (cmListIteratorHasNext(&refIterator))
            {
                CMReference * ref = (CMReference *)cmListIteratorNext(&refIterator);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "    item %p master: %p, locks: %d", ref->ref, ref->ref->master, ref->ref->locks);
            }
            cmListIteratorTerminate(&refIterator);
        }
    }
    cmListIteratorTerminate(&iterator);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
#endif /* UD_NQ_INCLUDETRACE */
}

#endif /* UD_CC_INCLUDERPC */
#endif /* UD_NQ_INCLUDECIFSCLIENT */
