/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : NETLOGON RPC client
 *--------------------------------------------------------------------
 * MODULE        : rpc - rpccore
 * DEPENDENCIES  : None
 ********************************************************************/

#ifndef _CCNETLGN_H_
#define _CCNETLGN_H_

#include "cmapi.h"
#include "cmsdescr.h"
#include "ccdcerpc.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

/*Netlogon negotiate flags: */
#define NETLOGON_NEG_FLAGS_ACCOUNT_LOCKOUT              0x00000001
#define NETLOGON_NEG_FLAGS_PERSISTENT_SAMREPL           0x00000002
#define NETLOGON_NEG_FLAGS_ARCFOUR                      0x00000004
#define NETLOGON_NEG_FLAGS_PROMOTION_COUNT              0x00000008
#define NETLOGON_NEG_FLAGS_CHANGELOG_BDC                0x00000010
#define NETLOGON_NEG_FLAGS_FULL_SYNC_REPL               0x00000020
#define NETLOGON_NEG_FLAGS_MULTIPLE_SIDS                0x00000040
#define NETLOGON_NEG_FLAGS_REDO                         0x00000080
#define NETLOGON_NEG_FLAGS_PASSWORD_CHANGE_REFUSAL      0x00000100
#define NETLOGON_NEG_FLAGS_SEND_PASSWORD_INFO_PDC       0x00000200
#define NETLOGON_NEG_FLAGS_GENERIC_PASSTHROUGH          0x00000400
#define NETLOGON_NEG_FLAGS_CONCURRENT_RPC               0x00000800
#define NETLOGON_NEG_FLAGS_AVOID_ACCOUNT_DB_REPL        0x00001000
#define NETLOGON_NEG_FLAGS_AVOID_SECURITYAUTH_DB_REPL   0x00002000
#define NETLOGON_NEG_FLAGS_STRONG_KEYS                  0x00004000  /* aka 128 bit keys */
#define NETLOGON_NEG_FLAGS_TRANSITIVE_TRUSTS            0x00008000
#define NETLOGON_NEG_FLAGS_DNS_DOMAIN_TRUSTS            0x00010000
#define NETLOGON_NEG_FLAGS_PASSWORD_SET2                0x00020000
#define NETLOGON_NEG_FLAGS_GETDOMAININFO                0x00040000
#define NETLOGON_NEG_FLAGS_CROSS_FOREST_TRUSTS          0x00080000
#define NETLOGON_NEG_FLAGS_NEUTRALIZE_NT4_EMULATION     0x00100000
#define NETLOGON_NEG_FLAGS_RODC_PASSTHROUGH             0x00200000
#define NETLOGON_NEG_FLAGS_SUPPORTS_AES_SHA2            0x00400000
#define NETLOGON_NEG_FLAGS_SUPPORTS_AES                 0x01000000
#define NETLOGON_NEG_FLAGS_AUTHENTICATED_RPC_LSASS      0x20000000
#define NETLOGON_NEG_FLAGS_AUTHENTICATED_RPC            0x40000000  /* aka schannel */


/* 0x600fffff */
#define NETLOGON_NEG_FLAGS_BASE        (NETLOGON_NEG_FLAGS_ACCOUNT_LOCKOUT | \
                                        NETLOGON_NEG_FLAGS_PERSISTENT_SAMREPL | \
                                        NETLOGON_NEG_FLAGS_PROMOTION_COUNT | \
                                        NETLOGON_NEG_FLAGS_CHANGELOG_BDC | \
                                        NETLOGON_NEG_FLAGS_FULL_SYNC_REPL | \
                                        NETLOGON_NEG_FLAGS_MULTIPLE_SIDS | \
                                        NETLOGON_NEG_FLAGS_REDO | \
                                        NETLOGON_NEG_FLAGS_PASSWORD_CHANGE_REFUSAL | \
                                        NETLOGON_NEG_FLAGS_SEND_PASSWORD_INFO_PDC | \
                                        NETLOGON_NEG_FLAGS_GENERIC_PASSTHROUGH | \
                                        NETLOGON_NEG_FLAGS_CONCURRENT_RPC | \
                                        NETLOGON_NEG_FLAGS_AVOID_ACCOUNT_DB_REPL | \
                                        NETLOGON_NEG_FLAGS_AVOID_SECURITYAUTH_DB_REPL | \
                                        NETLOGON_NEG_FLAGS_TRANSITIVE_TRUSTS | \
                                        NETLOGON_NEG_FLAGS_DNS_DOMAIN_TRUSTS | \
                                        NETLOGON_NEG_FLAGS_PASSWORD_SET2 | \
                                        NETLOGON_NEG_FLAGS_GETDOMAININFO | \
                                        NETLOGON_NEG_FLAGS_CROSS_FOREST_TRUSTS | \
                                        NETLOGON_NEG_FLAGS_AUTHENTICATED_RPC_LSASS | \
                                        NETLOGON_NEG_FLAGS_ARCFOUR | \
                                        NETLOGON_NEG_FLAGS_STRONG_KEYS | \
                                        NETLOGON_NEG_FLAGS_AUTHENTICATED_RPC)


/* 0x602fffff */
#define NETLOGON_NEG_FLAGS_AUTH2        (NETLOGON_NEG_FLAGS_BASE | NETLOGON_NEG_FLAGS_RODC_PASSTHROUGH)

/* 0x610fffff */
#define NETLOGON_NEG_FLAGS_AUTH3        (NETLOGON_NEG_FLAGS_BASE | NETLOGON_NEG_FLAGS_SUPPORTS_AES)

#define SCHANNEL_CHECKSUM_LENGTH     8
#define SCHANNEL_SEQUENCE_LENGTH     8
#define SCHANNEL_CONFOUNDER_LENGTH   8
#define SCHANNEL_SEQUENCE_OFFSET     8
#define SCHANNEL_CHECKSUM_OFFSET     16
#define SCHANNEL_CONFOUNDER_OFFSET   24

#define RPC_SIGN_ALGO_HMAC_MDA5     0x0077
#define RPC_SIGN_ALGO_HMAC_SHA256   0x0013

#define RPC_SEAL_ALGO_RC4           0x007A
#define RPC_SEAL_ALGO_AES_128       0x001A
#define RPC_SEAL_ALGO_NONE          0xFFFF

/**
 * This struct describes Netlogon Credentials
 */
typedef struct
{
    NQ_BYTE client[8];      /**< Client challenge */
    NQ_BYTE server[8];      /**< Server challenge */
    NQ_BYTE seed[8];        /**< Seed */
    NQ_BYTE sessionKey[16]; /**< Session key calculated as a result of exchange */
    NQ_BYTE secret[16];     /**< Secret buffer */
    NQ_UINT32 negotFlags;   /**< Negotiate flags suggested */
    NQ_UINT32 sequence;     /**< Current sequence */
    NQ_UINT32 time;         /**< Time stamp */
    NQ_UINT32 flags;        /**< Flags eventually used */
}
CCNetlogonCredential;

/**
 * This struct describes Netlogon Secure Channel
 */
typedef struct
{
    NQ_UINT64 sequence;             /**< Schannel current sequence */
    CCNetlogonCredential creds;     /**< Schannel Netlogon credentials structure */
}
CCSchannel;

const CCDcerpcPipeDescriptor *ccNetlogonGetPipe(void);

NQ_UINT32 ccNetrServerReqChallenge(
    NQ_HANDLE netlogon,
    const NQ_WCHAR *server,
    const NQ_WCHAR *computer,
    CCNetlogonCredential *credential
    );

NQ_UINT32 ccNetrServerReqChallengeEx(
    NQ_HANDLE netlogon,
    const NQ_WCHAR *server,
    const NQ_WCHAR *computer,
    CCNetlogonCredential *credential
    );


NQ_UINT32 ccNetrServerAuthenticate2(
    NQ_HANDLE netlogon,
    const NQ_WCHAR *server,
    const NQ_WCHAR *computer,
    CCNetlogonCredential *credential,
    NQ_UINT32 *flags
    );

NQ_UINT32 ccNetrServerAuthenticate2Ex(
    NQ_HANDLE netlogon,
    const NQ_WCHAR *server,
    const NQ_WCHAR *computer,
    CCNetlogonCredential *credential,
    NQ_UINT32 *flags
    );

NQ_UINT32 ccNetrServerAuthenticate3Ex(
    NQ_HANDLE netlogon,
    const NQ_WCHAR *server,
    const NQ_WCHAR *computer,
    CCNetlogonCredential *credential,
    NQ_UINT32 *flags
    );

/* callback function for storing different names during enumeration */
typedef void (*CCNetrEnumerateNamesCallback)(
    const NQ_WCHAR *name,      /* next name (null terminated) */
    void *data,                /* data structure */
    void *list                 /* list to add name to */
    );

NQ_UINT32 ccDsrEnumerateDomainTrusts(
    NQ_HANDLE netlogon,
    const NQ_WCHAR *server, 
    CCNetrEnumerateNamesCallback callback,
    CMList *list
    );

/**
 * Create random buffer for Netlogon credentials (first 5 bytes are unique).
 *
 * @param buffer        Pointer to buffer.
 * @param size          Size of buffer.
 */
void ccNetlogonCredentialsRandom(NQ_BYTE *buffer, NQ_UINT32 size);

/**
 * Init Netlogon credentials structure with secret.
 *
 * @param creds         Pointer to Netlogon credentials struct.
 * @param secret        Secret to use.
 * @param flags         Flags to use.
 */
void ccNetlogonCredentialsInit(CCNetlogonCredential *creds, const NQ_BYTE secret[16], NQ_UINT32 flags);

/**
 * Generate next step in chain of Netlogon credentials exchange.
 *
 * @param creds         Pointer to Netlogon credentials struct.
 */
void ccNetlogonCredentialsNext(CCNetlogonCredential *creds);

/**
 * Generate Netlogon credentials using secret.
 *
 * @param credential    Pointer to Netlogon credentials struct.
 * @param secret        Secret to use.
 */
void ccNetlogonCredentialsGenerate(CCNetlogonCredential *credential, const NQ_BYTE secret[16]);

/**
 * Dump Netlogon credentials state
 *
 * @param text          Text.
 * @param cred          Pointer to Netlogon credentials struct.
 */
void dumpNetlogonCredentials(const NQ_CHAR *text, CCNetlogonCredential *cred);

/* generate next schannel sequence number based on current */
void ccSchannelGenerateSequenceNumber(NQ_BYTE result[8], NQ_UINT64 currSeqNum, NQ_BOOL isOutgoing);

/**
 * Encrypt the sequence number and write into packet.
 *
 * @param pSchannel     Pointer to secure channel.
 * @param sequence      Sequence of schannel.
 * @param pSig          Pointer to start of Netlogon signature in RPC packet.
 */
void ccSchannelSequence(CCSchannel *pSchannel, const NQ_BYTE sequence[8], NQ_BYTE *pSig);

/**
 * Sign the data and fill the signature trailer fields.
 *
 * @param pSchannel     Pointer to secure channel.
 * @param pData         Pointer to data.
 * @param dataLen       Data length.
 * @param doSeal        Whether to seal(encrypt).
 * @param pSig          Pointer to start of Netlogon signature in RPC packet.
 * @param sigLen        Signature length in Rpc packet.
 * @param isOutgoing    Packet out or in.
 */
void ccSchannelSign(CCSchannel *pSchannel, const NQ_BYTE *pData, NQ_UINT dataLen, NQ_BOOL doSeal, NQ_BYTE *pSig, NQ_UINT16 sigLen, NQ_BOOL isOutgoing);

/**
 * Seal (encrypt) the data, write the modified confounder into packet.
 *
 * @param pSchannel     Pointer to secure channel.
 * @param pData         Pointer to data.
 * @param dataLen       Data length.
 * @param sequence      Sequence of schannel.
 * @param pSig          Pointer to start of Netlogon signature in RPC packet.
 * @param sigLen        Signature length in Rpc packet.
 * @param isOutgoing    Packet out or in.
 */
void ccSchannelSeal(CCSchannel *pSchannel, NQ_BYTE *pData, NQ_UINT dataLen, const NQ_BYTE sequence[8], NQ_BYTE *pSig, NQ_UINT16 sigLen, NQ_BOOL isOutgoing);


#endif /* UD_NQ_INCLUDECIFSCLIENT */

#endif /* _CCNETLGN_H_ */

