/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : RAP Client Implementation
 *--------------------------------------------------------------------
 * MODULE        : rpc - browsing
 * DEPENDENCIES  : None
 ********************************************************************/

#include "ccrap.h"
#include "ccshare.h"
#include "cmapi.h"
#include "nqapi.h"
#include "ccsrvsvc.h"
#include "cmsmb1.h"

#ifdef UD_CC_INCLUDEBROWSING

/* Beginning of packed structures definition */

#include "sypackon.h"

#define NETSERVERENUM2  0x0068
#define NETSERVERENUM3  0x00D7

typedef SY_PACK_PREFIX struct {
    NQ_SUINT16  function;
    NQ_SCHAR    pReq[6];
    NQ_SCHAR    pRsp[7];
    NQ_SUINT16  level;
    NQ_SUINT16  bufLen;
}
SY_PACK_ATTR NetShareEnum1Rsp;

typedef SY_PACK_PREFIX struct {
    NQ_SUINT16  status;
    NQ_SUINT16  converter;
    NQ_SUINT16  entries;
    NQ_SUINT16  total;
}
SY_PACK_ATTR NetShareEnumRsp;

typedef SY_PACK_PREFIX struct {
    NQ_SUINT16  function;
    NQ_SCHAR    pReq[6];
    NQ_SCHAR    pRsp[7];
}
SY_PACK_ATTR NetShareInfoReq;

typedef SY_PACK_PREFIX struct {
    NQ_SUINT16  status;
    NQ_SUINT16  converter;
    NQ_SUINT16  size;
    NQ_SUINT16  reserve;
}
SY_PACK_ATTR NetShareInfoRsp;

typedef SY_PACK_PREFIX struct {
    NQ_SCHAR    netName[13];
    NQ_SBYTE    pad;
    NQ_SUINT16  type;
    NQ_SUINT16  remark;
    NQ_SUINT16  pad2;
}
SY_PACK_ATTR RapShareInfo1;

typedef SY_PACK_PREFIX struct {
    NQ_SCHAR   srvName[16];
}
SY_PACK_ATTR RapServerInfo0;

typedef SY_PACK_PREFIX struct {
    NQ_SCHAR    srvName[16];
    NQ_SBYTE    versionMajor;
    NQ_SBYTE    versionMinor;
    NQ_SUINT32  type;
    NQ_SUINT16  commentLow;
    NQ_SUINT16  commentHigh;
}
SY_PACK_ATTR RapServerInfo1;

typedef SY_PACK_PREFIX struct {
    NQ_SUINT16  function;
    NQ_SCHAR    pReq[8];
    NQ_SCHAR    pRsp[8];
    NQ_SUINT16  level;
    NQ_SUINT16  bufLen;
    NQ_SUINT32  type;
}
SY_PACK_ATTR NetServerEnum2Req;

typedef SY_PACK_PREFIX struct {
    NQ_SUINT16  function;
    NQ_SCHAR    pReq[9];
    NQ_SCHAR    pRsp[8];
    NQ_SUINT16  level;
    NQ_SUINT16  bufLen;
    NQ_SUINT32  type;
}
SY_PACK_ATTR NetServerEnum3Req;

typedef SY_PACK_PREFIX struct {
    NQ_SUINT16  status;
    NQ_SUINT16  converter;
    NQ_SUINT16  entries;
    NQ_SUINT16  total;
}
SY_PACK_ATTR NetServerEnumRsp;

#include "sypackof.h"

/* End of packed structures definition */

/* Static data */

typedef struct {
    NQ_STATUS rapError;
    NQ_STATUS nqError;
}RapToNqError;

static RapToNqError rapToNqErros[] =
{
        {SMB_RAPSTATUS_ACCESS_DENIED,            NQ_ERR_RAPACCESSDENIED},
        {SMB_RAPSTATUS_MORE_DATA,                NQ_ERR_RAPMOREDATA},
        {SMB_RAPSTATUS_ServerNotStarted,         NQ_ERR_RAPSERVERNOTSTARTED},
        {SMB_RAPSTATUS_BadTransactConfig,        NQ_ERR_RAPBADTRANSACTCONFIG},
        {SMB_RAPSTATUS_ServiceNotInstalled,      NQ_ERR_RAPSERVICENOTINSTALLED},
        {SMB_RAPSTATUS_DevNotRedirected,         NQ_ERR_RAPDEVICENOTCONNECTED},
        {SMB_RAPSTATUS_NotLocalDomain,           NQ_ERR_RAPNOTACTIVEINDOMAIN},
        {SMB_RAPSTATUS_NO_BROWSER_SERVERS_FOUND, NQ_ERR_RAPNOBROWSERSERVERSFOUND},
        {SMB_RAPSTATUS_NERR_Success,             NQ_ERR_OK}
};

static NQ_STATUS convertRAPErrorToNQError(NQ_STATUS rapError)
{
    NQ_STATUS nqErr = NQ_ERR_RAPGENERALERROR;
    NQ_INT i;

    for (i = 0; i < sizeof(rapToNqErros) / sizeof(rapToNqErros[0]); i++)
    {
        if (rapToNqErros[i].rapError == rapError)
        {
            nqErr = rapToNqErros[i].nqError;
            goto Exit;
        }
    }
Exit:
    return nqErr;
}

/* static functions */

static NQ_STATUS netShareEnumLevel1(NetShareEnumRsp * response, NQ_CHAR *data, CCRapEnumerateNamesCallback callback, void* params);
static NQ_STATUS netServerEnumLevel1(NetServerEnumRsp * response, NQ_CHAR *data, CCRapEnumerateNamesCallback callback, void* params, NQ_CHAR *lastEntry, NQ_UINT16 opCode, NQ_UINT16 *totalEntries, NQ_UINT16 *receivedEntries);

/*
 *====================================================================
 * PURPOSE: enumerate shares on remote server
 *--------------------------------------------------------------------
 * PARAMS:  IN  server name
 *          IN  protocol level
 *          IN  pointer to data buffer to store share entries in
 *          IN  data buffer length
 *          OUT number of entries returned in the data buffer
 *          OUT total number of entries available
 *
 * RETURNS: NQ_ERR_OK if succeeded, appropriate error code otherwise
 *
 * NOTES:   only level 1 is available
 *====================================================================
 */

NQ_STATUS
ccRapNetShareEnum(
    const NQ_WCHAR *server,
    CCRapEnumerateNamesCallback callback,
    void* params
   )
{
    CCShare * pShare;                       /* pointer to IPC descriptor */
    CMBlob inData;                          /* request parameters */
    CMBlob outParams = {NULL, 0};           /* response parameters + data parameters */
    CMBlob outData = {NULL, 0};             /* response data parameters */
    NetShareEnum1Rsp request1;              /* request parameters structure */
    NQ_STATUS status = NQ_ERR_GENERAL;      /* operation status */
    NQ_COUNT i, j;                          /* just a counter */
    CCServer *pServer;                      /* pointer to server object */
    NQ_BOOL security[] = {TRUE, FALSE};     /* whether to use extended security */
    NQ_BOOL anon[] = {TRUE , FALSE};        /* whether to use Anonymous or not */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "server:%s callback:%p params:%p", cmWDump(server), callback, params);

    if (!ccCifsGetSMB1Support())
    {
        status = NQ_ERR_NOSUPPORT;
        goto Exit;
    }

    cmPutSUint16(request1.function, 0);
    syStrcpy(request1.pReq, "WrLeh");
    syStrcpy(request1.pRsp, "B13BWz");
    cmPutSUint16(request1.level, cmHtol16(1));
    cmPutSUint16(request1.bufLen, cmHtol16(CIFS_MAX_DATA_SIZE16));
    inData.data = (NQ_BYTE *)&request1;
    inData.len = sizeof(request1);

    for (i = 0; i < sizeof(security)/sizeof(security[0]); i++)
    {
        pServer = ccServerFindOrCreate(server, security[i], ccCifsGetDefaultSmb());
        if (pServer != NULL)
        {
            for (j = 0 ; j < sizeof(anon)/sizeof(anon[0]); j++)
            {
                const AMCredentials * pCredentials = anon[j] ? ccUserGetAnonymousCredentials() : NULL;       /* credentials */
                pShare = ccShareConnectIpc(pServer , &pCredentials);
                if (NULL == pShare)
                {
                    cmListItemUnlock((CMItem *)pServer);
                    LOGERR(CM_TRC_LEVEL_ERROR, "Can't establish connection to %s", cmWDump(server));
                    status = (NQ_STATUS)syGetLastError();
                    goto Exit;
                }

                status = pShare->user->server->smb->doRapTransaction(pShare, &inData, &outParams, &outData);
                cmListItemUnlock((CMItem *)pShare);       /* force disconnect when not used */

                if (status == NQ_SUCCESS || status == NQ_ERR_NOSUPPORT)
                {
                    break;
                }
            }
            cmListItemUnlock((CMItem *)pServer);
            if (status == NQ_SUCCESS || status == NQ_ERR_NOSUPPORT)
            {
                break;
            }
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Can't establish connection to %s", cmWDump(server));
            status = (NQ_STATUS)syGetLastError();
            goto Exit;
        }
    }/* end of connecting "for" */

    if (NQ_SUCCESS != status)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "No valid NetShareEnum response");
        goto Exit;
    }

    status = netShareEnumLevel1((NetShareEnumRsp *)outParams.data, (NQ_CHAR *)outData.data, callback, params);

Exit:
    cmMemoryFreeBlob(&outParams);
    cmMemoryFreeBlob(&outData);
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d", status);
    return status;
}

/*
 *====================================================================
 * PURPOSE: get share information
 *--------------------------------------------------------------------
 * PARAMS:  IN  server name
 *          IN  share name
 *          OUT share type (SMB_SHARETYPE_XXXXX)
 *          OUT share remark in the supplied buffer
 *          IN  remark buffer size
 *
 * RETURNS: NQ_ERR_OK if succeeded, appropriate error code otherwise
 *
 * NOTES:   only level 1 is available
 *====================================================================
 */

NQ_STATUS
ccRapNetShareInfo(
    const NQ_WCHAR * server,
    const NQ_WCHAR * share,
    NQ_UINT16 * type,
    NQ_WCHAR * remark,
    NQ_INT maxRemarkSize,
    NQ_BOOL unicodeResult
   )
{
    CCShare * pShare;                       /* pointer to IPC descriptor */
    CMBlob inData = {NULL, 0};              /* request parameters */
    CMBlob outParams = {NULL, 0};           /* response parameters + data parameters */
    CMBlob outData = {NULL, 0};             /* response data parameters */
    NetShareInfoReq * request;              /* pointer to request parameters */
    NetShareInfoRsp * response;             /* pointer to response parameters */
    RapShareInfo1 * info;                   /* pointer to share information data */
    NQ_BYTE * p;                            /* generic pointer */
    NQ_STATUS status = NQ_ERR_GENERAL;      /* operation status */
    NQ_COUNT i, j;                          /* just a counter */
    CCServer * pServer;                     /* pointer to server object */
    NQ_BOOL security[] = {TRUE, FALSE};     /* whether to use extended security */
    NQ_BOOL anon[] = {TRUE , FALSE};        /*whether to use Anonymous or not */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "server:%s share:%s type:%p remark:%p size:%d unicode:%s", cmWDump(server), cmWDump(share), type, remark, maxRemarkSize, unicodeResult ? "TRUE" : "FALSE");

    if (!ccCifsGetSMB1Support())
    {
        status = NQ_ERR_NOSUPPORT;
        goto Exit;
    }

    if ((NULL == server) || (NULL == share)|| (NULL == type) || (NULL == remark))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    inData.len = (NQ_COUNT)(sizeof(NetShareInfoReq) + cmWStrlen(share) + 1 + (sizeof(NQ_UINT16) * 2));  /* more memory for multibyte characters */
    inData.data = (NQ_BYTE *)cmMemoryAllocate(inData.len);
    if (NULL == inData.data)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Memory overflow");
        status = NQ_ERR_NOMEM;
        goto Exit;
    }

    request = (NetShareInfoReq *)inData.data;
    cmPutSUint16(request->function, 1);
    syStrcpy(request->pReq, "zWrLh");
    syStrcpy(request->pRsp, "B13BWz");
    p = (NQ_BYTE *)(request + 1);
    cmUnicodeToAnsi((NQ_CHAR *)p, share);
    p += syStrlen((NQ_CHAR *)p) + 1;
    cmPutUint16(p, cmHtol16(1));
    p += 2;
    cmPutUint16(p, (NQ_UINT16)cmHtol16(maxRemarkSize));

    for (i = 0; i < sizeof(security)/sizeof(security[0]); i++)
    {
        pServer = ccServerFindOrCreate(server, security[i], ccCifsGetDefaultSmb());
        if (pServer != NULL)
        {
            for (j = 0 ; j < sizeof(anon)/sizeof(anon[0]); j++)
            {
                const AMCredentials * pCredentials = anon[j] ? ccUserGetAnonymousCredentials() : NULL;       /* credentials */
                pShare = ccShareConnectIpc(pServer , &pCredentials);
                if (NULL == pShare)
                {
                    cmListItemUnlock((CMItem *)pServer);
                    LOGERR(CM_TRC_LEVEL_ERROR, "Can't establish connection to %s", cmWDump(server));
                    status = (NQ_STATUS)syGetLastError();
                    goto Exit;
                }

                status = pShare->user->server->smb->doRapTransaction(pShare, &inData, &outParams, &outData);
                cmListItemUnlock((CMItem *)pShare);       /* force disconnect when not used */

                if (status == NQ_SUCCESS || status == NQ_ERR_NOSUPPORT)
                {
                    break;
                }
            }
            cmListItemUnlock((CMItem *)pServer);
            if (status == NQ_SUCCESS || status == NQ_ERR_NOSUPPORT)
            {
                break;
            }
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Can't establish connection to %s", cmWDump(server));
            status = (NQ_STATUS)syGetLastError();
            goto Exit;
        }
    }/* end of connecting "for" */

    if (NQ_SUCCESS != status)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "No valid NetShareInfo response");
        goto Exit;
    }

    response = (NetShareInfoRsp *)outParams.data;
    info = (RapShareInfo1 *)outData.data;
    if ((status = cmLtoh16(cmGetSUint16(response->status))) == NQ_ERR_OK)
    {
        *type = (NQ_UINT16)cmLtoh16(cmGetSUint16(info->type));
        if (unicodeResult)
        {
            cmAnsiToUnicode((NQ_WCHAR *)remark, (NQ_CHAR *)info + (info->remark - response->converter));
        }
        else
        {
            syStrcpy((NQ_CHAR*)remark, (NQ_CHAR *)info + (info->remark - response->converter));
    }
    }

Exit:
    cmMemoryFreeBlob(&inData);
    cmMemoryFreeBlob(&outParams);
    cmMemoryFreeBlob(&outData);
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d", status);
    return status;
}

static NQ_STATUS packRequest(NQ_UINT16 opCode, CMBlob *inData, NQ_UINT32 serverType, const NQ_CHAR * domainA, NQ_COUNT len, NQ_CHAR * lastEntry, NQ_UINT16 maxBuffSize)
{
    NQ_STATUS status = NQ_ERR_GENERAL;          /* operation status */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "opCode:%d inData:%p serverType:%d domainA:%s len:%d lastEntry:%p", opCode, inData, serverType, domainA, len, lastEntry);

    switch (opCode)
    {
        case NETSERVERENUM2:
        {
            NetServerEnum2Req * request = (NetServerEnum2Req *)inData->data;                /* request parameters structure */

            inData->len = sizeof(NetServerEnum2Req);
            inData->len +=  len;
            inData->data = (NQ_BYTE *)cmMemoryAllocate(inData->len);
            if (NULL == inData->data)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Memory overflow");
                status = NQ_ERR_NOMEM;
                goto Exit;
            }

            request = (NetServerEnum2Req *)inData->data;
            cmPutSUint16(request->function, cmHtol16(NETSERVERENUM2));
            syStrcpy(request->pReq, (NULL == domainA) ? "WrLehDO" : "WrLehDz");
            syStrcpy(request->pRsp, "B16BBDz");
            cmPutSUint16(request->level, cmHtol16(1));
            cmPutSUint16(request->bufLen, cmHtol16(maxBuffSize));
            cmPutSUint32(request->type, cmHtol32(serverType));

            if (NULL != domainA)
            {
                syStrcpy((NQ_CHAR *)(request + 1), domainA);
            }
            status = NQ_SUCCESS;
            break;
        }
        case NETSERVERENUM3:
        {
            NetServerEnum3Req * request;               /* request parameters structure */
            NQ_BYTE *p;

            inData->len = sizeof(NetServerEnum3Req) + 16; /* 16 is the max length of server name NULL terminated */
            inData->len +=  len;
            inData->data = (NQ_BYTE *)cmMemoryAllocate(inData->len);
            if (NULL == inData->data)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Memory overflow");
                status = NQ_ERR_NOMEM;
                goto Exit;
            }
            request = (NetServerEnum3Req *)inData->data;
            cmPutSUint16(request->function, cmHtol16(NETSERVERENUM3));
            syStrcpy(request->pReq, /*(NULL == domainA) ? "WrLehDO" : */"WrLehDzz");
            syStrcpy(request->pRsp, "B16BBDz");
            cmPutSUint16(request->level, cmHtol16(1));
            cmPutSUint16(request->bufLen, cmHtol16(maxBuffSize));
            cmPutSUint32(request->type, cmHtol32(serverType));

            if (NULL != domainA)
            {
                syStrcpy((NQ_CHAR *)(request + 1), domainA);
            }

            p = (NQ_BYTE *)(request + 1);
            p += 1 + ((NULL != domainA) ? syStrlen(domainA) : 0);
            syMemset(p, 0, 16);
            if (0 != lastEntry[0])
            {
                syStrcpy((NQ_CHAR *)p, lastEntry);
            }

            status = NQ_SUCCESS;
            break;
        }
    }
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "status:0x%08x", status);
    return status;
}

static NQ_STATUS processResponse(NQ_UINT16 opCode, CMBlob *outParams, NQ_CHAR *data, CCRapEnumerateNamesCallback callback, void* params, NQ_CHAR *lastEntry, NQ_UINT16 *totalEntries, NQ_UINT16 *receivedEntries)
{
    NQ_STATUS status;
    NetServerEnumRsp *response = (NetServerEnumRsp *)outParams->data;
    NQ_CHAR name[16] = {0};

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "opCode:%d outParams:%p data:%p callback:%p params:%p lastEntry:%p totalEntries:%p receivedEntries:%p", opCode, outParams, data, callback, params, lastEntry, totalEntries, receivedEntries);

    syStrcpy(name, lastEntry);
    status = netServerEnumLevel1(response, data, callback, params, lastEntry, opCode, totalEntries, receivedEntries);

    /* a case when returned entry is the same as requested, but buffer size doesn't fit more */
    if ((syStrcmp(name, lastEntry) == 0) && (*receivedEntries == 1))
    {
        status = NQ_ERR_GENERAL;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "status:0x%08x", status);
    return status;
}
/*
 *====================================================================
 * PURPOSE: enumerate hosts in domain
 *--------------------------------------------------------------------
 * PARAMS:  IN  server name to send enumeration request to
 *          IN  callback for storing names
 *          IN/OUT parameters for the callback
 *          IN  server types include in enumeration
 *          IN  enumerated domain name
 *
 * RETURNS: NQ_ERR_OK if succeeded, appropriate error code otherwise
 *
 * NOTES:   levels 0 and 1 are supported
 *====================================================================
 */

NQ_STATUS
ccRapNetServerEnum(
    const NQ_WCHAR * server,
    CCRapEnumerateNamesCallback callback,
    void * params,
    NQ_UINT32 serverType,
    const NQ_WCHAR * domain
   )
{
    CCShare * pShare;                           /* pointer to IPC descriptor */
    CMBlob inData = {NULL, 0};                  /* request parameters */
    CMBlob outParams = {NULL, 0};               /* response parameters + data parameters */
    CMBlob outData = {NULL, 0};                 /* response data parameters */
    NQ_STATUS status = NQ_ERR_GENERAL;          /* operation status */
    NQ_COUNT i, j, k;                           /* just a counter */
    CCServer * pServer;                         /* pointer to server object */
    NQ_UINT16 opCode[] = {NETSERVERENUM2, NETSERVERENUM3};/* opCode to use */
    NQ_BOOL security[] = {TRUE, FALSE};         /* whether to use extended security */
    NQ_BOOL anon[] = {TRUE , FALSE};            /* whether to use Anonymous or not */
    NQ_COUNT len;                               /* string length */
    NQ_CHAR *domainA = NULL;                    /* domain in ASCII */
    NQ_CHAR lastEntry[16] = {0};                /* last entry in ASCII, always 16 including NULL */
    NQ_BOOL switchOpCode = FALSE;               /* flag to switch opCode */
    NQ_UINT16 totalEntries = 0, total = 0;      /* number of total entries in response */
    NQ_UINT16 receivedEntries = 0, received = 0;/* number of received so far entries in response */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "server:%s callback:%p params:%p serverType:%u domain:%s", cmWDump(server), callback, params, serverType, cmWDump(domain));

    if (FALSE == ccCifsGetSMB1Support())
    {
        status = NQ_ERR_NOSUPPORT;
        goto Exit;
    }

    /* check required space for domain */
    if (NULL == domain)
    {
        len = 0;
    }
    else
    {
        NQ_CHAR *dot = NULL;

        len = (NQ_COUNT)((cmWStrlen(domain) + 1) * sizeof(NQ_CHAR) * 2); /* add more space for multibyte character sets */
        domainA = (NQ_CHAR *)cmMemoryAllocate(len);
        if (NULL == domainA)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Memory overflow");
            status = NQ_ERR_NOMEM;
            goto Exit;
        }

        cmUnicodeToAnsi(domainA, domain);
        if (NULL != (dot = syStrchr(domainA, '.')))
        {
            *dot = '\0';
        }

        len = (NQ_COUNT)(syStrlen(domainA) + 1); /* check actual string length */
    }

    for (i = 0 ; i < sizeof(security)/sizeof(security[0]) ; i++)
    {
        pServer = ccServerFindOrCreate(server, security[i], ccCifsGetDefaultSmb());
        if (NULL == pServer)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Can't establish connection to %s", cmWDump(server));
            status = (NQ_STATUS)syGetLastError();
            goto Exit;
        }

        for (j = 0 ; j < sizeof(anon)/sizeof(anon[0]) ; j++)
        {
            const AMCredentials * pCredentials = anon[j] ? ccUserGetAnonymousCredentials() : NULL;       /* credentials */

            pShare = ccShareConnectIpc(pServer, &pCredentials);
            if (NULL == pShare)
            {
                cmListItemUnlock((CMItem *)pServer);
                LOGERR(CM_TRC_LEVEL_ERROR, "Can't establish connection to %s", cmWDump(server));
                status = (NQ_STATUS)syGetLastError();
                goto Exit;
            }

            for (k = 0 ; k < sizeof(opCode)/sizeof(opCode[0]) ; k++)
            {
                NQ_BOOL doContinue = TRUE;

                do
                {
                    cmMemoryFreeBlob(&inData);
                    cmMemoryFreeBlob(&outParams);
                    cmMemoryFreeBlob(&outData);

                    if (NQ_SUCCESS != packRequest(opCode[k], &inData, serverType, domainA, len, lastEntry, (NQ_UINT16)(pServer->maxTrans - sizeof(SMB_HEADERSIZE) - sizeof(CMCifsTransactionResponse) - sizeof(NetServerEnumRsp))))
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Can't pack request");
                        status = (NQ_STATUS)syGetLastError();
                        goto ExitUnlock;
                    }

                    status = pShare->user->server->smb->doRapTransaction(pShare, &inData, &outParams, &outData);
                    if (NQ_SUCCESS != status)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Response error:0x%x", status);
                        goto ExitUnlock;
                    }

                    status = processResponse(opCode[k], &outParams, (NQ_CHAR *)outData.data, callback, params, lastEntry, &totalEntries, &receivedEntries);
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "opCode:%d totalEntries:%d receivedEntries:%d status:0x%x", opCode[k], totalEntries, receivedEntries, status);

                    if (NQ_ERR_GENERAL == status)
                    {
                        received = receivedEntries;
                        goto ExitUnlock;
                    }

                    switch (opCode[k])
                    {
                        case NETSERVERENUM2:
                        {
                            /* Means that the buffer is not enough for 1 entry */
                            if ((0 != totalEntries) && (0 == receivedEntries))
                            {
                                status = NQ_ERR_MOREDATA;
                                goto ExitUnlock;
                            }

                            if ((totalEntries == receivedEntries) && (NQ_SUCCESS == status))
                            {
                                goto ExitUnlock;
                            }

                            doContinue = FALSE;
                            switchOpCode = TRUE;
                            lastEntry[0] = 0;
                            received = total = 0;
                            break;
                        }
                        case NETSERVERENUM3:
                        {
                            /* Means that the buffer is not enough for 1 entry */
                            if ((0 != totalEntries) && (0 == receivedEntries))
                            {
                                status = NQ_ERR_MOREDATA;
                                goto ExitUnlock;
                            }

                            if ((NQ_ERR_RAPMOREDATA != status) && (NQ_SUCCESS != status))
                            {
                                LOGERR(CM_TRC_LEVEL_ERROR, "Can't receive response");
                                goto ExitUnlock;
                            }

                            if (TRUE == switchOpCode)
                            {
                                /* one time init after switching opCode*/
                                total = totalEntries;
                                received = receivedEntries;
                                switchOpCode = FALSE;
                            }
                            else
                            {
                                received = (NQ_UINT16)(received + receivedEntries - 1);
                            }

                            if ((totalEntries != receivedEntries) || (NQ_ERR_RAPMOREDATA == status))
                            {
                                doContinue = TRUE;
                                break;
                            }

                            if (received == total)
                            {
                                goto ExitUnlock;
                            }
                        }
                    } /* end of switch */
                }
                while (doContinue);
            } /* end of opCodes "for" */

            cmListItemUnlock((CMItem *)pShare);
        } /* end of anon "for" */

        cmListItemUnlock((CMItem *)pServer);
    }/* end of security "for" */

    goto Exit;

ExitUnlock:
    cmListItemUnlock((CMItem *)pShare);
    cmListItemUnlock((CMItem *)pServer);
Exit:
    cmMemoryFreeBlob(&inData);
    cmMemoryFreeBlob(&outParams);
    cmMemoryFreeBlob(&outData);
    cmMemoryFree(domainA);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:0x%x", status);
    return status;
}

/*
 *====================================================================
 * PURPOSE: parse enumerate shares level 1 response
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to net share enumeration response
 *          IN  pointer to received shares data to be parsed
 *          IN  pointer to callback function
 *          IN/OUT pointer to callback parameters
 *
 * RETURNS: NQ_ERR_OK if succeeded, NQ_ERR_MOREDATA if the output buffer
 *          is too small
 *
 * NOTES:
 *====================================================================
 */

static
NQ_STATUS
netShareEnumLevel1(
    NetShareEnumRsp *response,
    NQ_CHAR *data,
    CCRapEnumerateNamesCallback callback,
    void* params
   )
{
    ShareInfo1 *info;
    ShareCallbackItem          pItem;
    NQ_UINT16 count = (NQ_UINT16)cmLtoh16(cmGetSUint16(response->entries));
    NQ_STATUS status = convertRAPErrorToNQError(cmLtoh16(cmGetSUint16(response->status)));
    NQ_UINT16 i = 0;

    if (NULL == data)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "No rap data is present");
        goto Exit;
    }

    info = (ShareInfo1 *)data;
    pItem.params = params;

    for (; i < count; i++, info++)
    {
        pItem.type = info->type;
        pItem.comment = NULL;
        (*callback)(info->netName, NULL, &pItem);
    }

Exit:
    return status;
}

/*
 *====================================================================
 * PURPOSE: parse enumerate servers level 1 response
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to net server enumeration response
 *          IN  pointer to data buffer to store server entries in
 *          IN  callback for storing parameters
 *          IN/OUT  parameters for callback
 *
 * RETURNS: NQ_ERR_OK if succeeded, NQ_ERR_MOREDATA if the output buffer
 *          is too small
 *
 * NOTES:
 *====================================================================
 */

static
NQ_STATUS
netServerEnumLevel1(
    NetServerEnumRsp *response,
    NQ_CHAR *data,
    CCRapEnumerateNamesCallback callback,
    void* params,
    NQ_CHAR *lastEntry,
    NQ_UINT16 opCode,
    NQ_UINT16 *totalEntries,
    NQ_UINT16 *receivedEntries
    )
{
    NQ_CHAR * pData;
    RapServerInfo1 * pInfo;
    NQ_UINT16 count = (NQ_UINT16)cmLtoh16(cmGetSUint16(response->entries));
    NQ_UINT16 total = (NQ_UINT16)cmLtoh16(cmGetSUint16(response->total));
    NQ_STATUS status = convertRAPErrorToNQError(cmLtoh16(cmGetSUint16(response->status)));
    NQ_UINT16 offset = (NQ_UINT16)cmLtoh16(cmGetSUint16(response->converter));
    NQ_UINT16 i;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "response:%p data:%p callback:%p params:%p lastEntry:%p opCode:%d totalEntries:%p receivedEntries:%p", response, data, callback, params, lastEntry, opCode, totalEntries, receivedEntries);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "opCode:%d count:%d total:%d status:0x%x offset:%d", opCode, count, total, status, offset);

    *totalEntries = total;
    *receivedEntries = count;

    if (NULL == data)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "No rap data is present");
        goto Exit;
    }

    pData = (NQ_CHAR*)cmAllignTwo(data);
    pInfo = (RapServerInfo1 *)pData;

    for (i = 0; i < count; i++, pInfo++)
    {
        CCRapServerInfo1 serverInfo;

        serverInfo.srvName = cmMemoryCloneAString(pInfo->srvName);
        serverInfo.versionMajor = pInfo->versionMajor;
        serverInfo.versionMinor = pInfo->versionMinor;
        serverInfo.type = cmLtoh32(cmGetSUint32(pInfo->type));
        serverInfo.comment = cmMemoryCloneAString((NQ_CHAR *)((NQ_BYTE *)data + cmLtoh16(cmGetSUint16(pInfo->commentLow)) - offset));

        (*callback)(pInfo->srvName, &serverInfo, params);

        if (lastEntry && (i == (count - 1)))
        {
            syStrcpy(lastEntry, pInfo->srvName);
        }
    }
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", status);
    return status;
}

#endif /* UD_CC_INCLUDEBROWSING */
