/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client smb30 operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccapi.h"
#include "ccsmb20.h"
#include "cctransport.h"
#include "ccsmb30.h"
#include "ccfile.h"
#include "ccutils.h"
#include "ccerrors.h"
#include "ccparams.h"
#include "ccsmb2common.h"
#include "ccsearch.h"
#include "ccinfo.h"
#include "cmthread.h"
#include "cmfsutil.h"
#include "cmsmb2.h"
#include "cmbufman.h"
#include "cmcrypt.h"
#include "cmsdescr.h"
#include "cmuuid.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT
#ifdef UD_NQ_INCLUDESMB3

/**< When defined SMB2 Validate Negotiate request is issued for Servers SMB3 and SMB3.02 */
#define CC_REQUIRESECURENEGOTIATE
#undef CC_REQUIRESECURENEGOTIATE

/* special callbacks */
static void writeCallback(CCServer * pServer, Match * pContext);
static void readCallback(CCServer * pServer, Match * pContext);
/* notification handles */
static void handleBreakNotification(CCServer * pServer, Response * pResponse, CCFile *pFile);
static void handleWaitingNotifyResponse(void *pServer, void *pFile);

static const Command commandDescriptors[] = /* SMB2 descriptor */
{
#ifdef UD_NQ_INCLUDESMB311
    { 190,  36,     65,     NULL,               NULL},                      /* SMB2 NEGOTIATE 0x0000 with extra contexts */
#else
    { 128,  36,     65,     NULL,               NULL},                      /* SMB2 NEGOTIATE 0x0000 */
#endif
    { 26,   25,     9,      NULL,               NULL},                      /* SMB2 SESSION_SETUP 0x0001 */
    { 6,    4,      4,      NULL,               NULL},                      /* SMB2 LOGOFF 0x0002 */
    { 10,   9,      16,     NULL,               NULL},                      /* SMB2 TREE_CONNECT 0x0003 */
    { 6,    4,      4,      NULL,               NULL},                      /* SMB2 TREE_DISCONNECT 0x0004 */
    { 4096, 57,     89,     NULL,               NULL},                      /* SMB2 CREATE 0x0005 */
    { 26,   24,     60,     NULL,               NULL},                      /* SMB2 CLOSE 0x0006 */
    { 26,   24,     4,      NULL,               NULL},                      /* SMB2 FLUSH 0x0007 */
    { 64,   49,     17,     readCallback,       NULL},                      /* SMB2 READ 0x0008 */
    { 64,   49,     17,     writeCallback,      NULL},                      /* SMB2 WRITE 0x0009 */
    { 0,    0,      0,      NULL,               NULL},                      /* SMB2 LOCK 0x000A */
    { 100,  57,     49,     NULL,               NULL},                      /* SMB2 IOCTL 0x000B */
    { 0,    0,      0,      NULL,               NULL},                      /* SMB2 CANCEL 0x000C */
    { 4,    4,      4,      NULL,               NULL},                      /* SMB2 ECHO 0x000D */
    { 40,   33,     9,      NULL,               NULL},                      /* SMB2 QUERY_DIRECTORY 0x000E */
    { 0,    0,      0,      NULL,               NULL},                      /* SMB2 CHANGE_NOTIFY 0x000F */
    { 44,   41,     9,      NULL,               NULL},                      /* SMB2 QUERY_INFO 0x0010 */
    { 80,   33,     2,      NULL,               NULL},                      /* SMB2 SET_INFO 0x0011 */
    { 100,  24,     0,      NULL,               handleBreakNotification },  /* SMB2 OPLOCK_BREAK 0x0012 */
};

#define COMMAND_MAX     ((sizeof(commandDescriptors) / sizeof(commandDescriptors[0])) - 1)

/* CCCifsSmb methods */
static NQ_STATUS sendRequest(CCServer * pServer, CCUser * pUser, Request * pRequest, Match * pMatch, NQ_BOOL (*callback)(CMItem * pItem));
static NQ_STATUS sendReceive(CCServer * pServer, CCUser * pUser, Request * pRequest, Response * pResponse);
static void anyResponseCallback(void * transport);

static NQ_STATUS keyDerivation(void * user);
static NQ_BOOL validateNegotiate(void *pServ, void *_pUser, void* _pShareIPC);

static  CCCifsSmb dialect300;
static  CCCifsSmb dialect302;

/* -- API Functions */

NQ_BOOL ccSmb30Start()
{
    CCCifsSmb tempDialect;

    syMemcpy(&tempDialect ,ccSmb20GetCifs202(), sizeof(CCCifsSmb));

    tempDialect.name = SMB3_DIALECTSTRING;
    tempDialect.revision = SMB3_DIALECTREVISION;
    tempDialect.anyResponseCallback = anyResponseCallback;
    tempDialect.sendRequest = (NQ_STATUS (*)(void * pServer, void * pUser, void * pRequest, void * pMatch, NQ_BOOL (*callback)(CMItem * pItem)))sendRequest;
    tempDialect.sendReceive = (NQ_STATUS (*)(void * pServer, void * pUser, void * pRequest, void * pResponse))sendReceive;
    tempDialect.handleWaitingNotifyResponses = handleWaitingNotifyResponse;
    tempDialect.keyDerivation = keyDerivation;
    tempDialect.validateNegotiate = validateNegotiate;

    syMemcpy(&dialect300, &tempDialect, sizeof(CCCifsSmb));

    /* initialize pointer of dialect 302 */
    dialect302 = dialect300;
    dialect302.name = SMB3_0_2_DIALECTSTRING;
    dialect302.revision = SMB3_0_2_DIALECTREVISION;

    return TRUE;
}

NQ_BOOL ccSmb30Shutdown()
{
    return TRUE;
}

const CCCifsSmb * ccSmb30GetCifs300(void)
{
    return &dialect300;
}

const CCCifsSmb * ccSmb30GetCifs302(void)
{
    return &dialect302;
}

NQ_BOOL ccSmb30CheckMessageSignatureSMB3(CCUser *pUser, NQ_BYTE *pHeaderIn, NQ_COUNT headerDataLength, NQ_IOBufPos buffer, NQ_COUNT bufLength)
{
    NQ_BYTE sigReceived[SMB2_SECURITY_SIGNATURE_SIZE];
    NQ_BYTE *sig = pHeaderIn + SMB2_SECURITY_SIGNATURE_OFFSET;
    NQ_BOOL result = TRUE;
    NQ_IOBufPos bufHeaderPos;
    IOBUF_POSCONSTRUCTORINIT(bufHeaderPos)

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pUser:%p pHeaderIn:%p headerDataLength:%d buffer:%p bufLength:%d", pUser, pHeaderIn, headerDataLength, buffer, bufLength);

    syMemcpy(sigReceived, sig, sizeof(sigReceived));
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "Received Signature" , sigReceived , SMB2_SECURITY_SIGNATURE_SIZE);
    syMemset(sig, 0, SMB2_SECURITY_SIGNATURE_SIZE);

    IOBUF_POSCONSTRUCTOR(bufHeaderPos, pHeaderIn, headerDataLength)
    cmSmb3CalculateMessageSignature(pUser->macSessionKey.data, pUser->macSessionKey.len, bufHeaderPos, headerDataLength, buffer, bufLength, sig);

    result = syMemcmp(sigReceived, sig, SMB2_SECURITY_SIGNATURE_SIZE) == 0;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "signatures %smatch", result ? "" : "don't ");
    return result;
}

void ccComposeEncryptionNonce(NQ_BYTE * buf ,NQ_UINT32 midLow)
{
    CMBufferWriter  writer;
    CMTime  time;
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    if (buf == NULL)
    {
        return;
    }

    /* although nonce size different for AES GCM and AES CCM,
     * this nonce algorithm qualifies for both. */

    IOBUF_POSCONSTRUCTOR(bufPos, buf, SMB2_AES128_CCM_NONCE_SIZE)
    cmBufferWriterInit(&writer, bufPos, SMB2_AES128_CCM_NONCE_SIZE);
    cmGetCurrentTime(&time);
    cmBufferWriteUint32(&writer,time.low);
    cmBufferWriterSkip(&writer,3);
    cmBufferWriteUint32(&writer,midLow);

    return;
}

/* -- Static functions -- */

static NQ_STATUS keyDerivation(void * user)
{
    CCUser * pUser = (CCUser *)user;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", pUser);

    if (NULL != pUser->macSessionKey.data)
    {
        if (pUser->macSessionKey.len > pUser->server->smb->maxSigningKeyLen)
        {
            pUser->macSessionKey.len = pUser->server->smb->maxSigningKeyLen;  /* restrict bigger keys */
        }
        cmMemoryFreeBlob(&pUser->encryptionKey);
        cmMemoryFreeBlob(&pUser->decryptionKey);
        cmMemoryFreeBlob(&pUser->applicationKey);

        pUser->encryptionKey.data = (NQ_BYTE *)cmMemoryAllocate(sizeof(NQ_BYTE) * SMB2_CRYPTO_KEY_SIZE);
        pUser->encryptionKey.len = SMB2_CRYPTO_KEY_SIZE;
        pUser->decryptionKey.data = (NQ_BYTE *)cmMemoryAllocate(sizeof(NQ_BYTE) * SMB2_CRYPTO_KEY_SIZE);
        pUser->decryptionKey.len = SMB2_CRYPTO_KEY_SIZE;
        pUser->applicationKey.data = (NQ_BYTE *)cmMemoryAllocate(sizeof(NQ_BYTE) * SMB2_CRYPTO_KEY_SIZE);
        pUser->applicationKey.len = SMB2_CRYPTO_KEY_SIZE;
        cmKeyDerivation( pUser->macSessionKey.data, pUser->encryptionKey.len  , (NQ_BYTE*)"SMB2AESCCM\0" , 11 , (NQ_BYTE*)"ServerOut\0" , 10, (NQ_BYTE *)pUser->encryptionKey.data );
        cmKeyDerivation( pUser->macSessionKey.data, pUser->decryptionKey.len  , (NQ_BYTE*)"SMB2AESCCM\0" , 11 , (NQ_BYTE*)"ServerIn \0" , 10, (NQ_BYTE *)pUser->decryptionKey.data );
        cmKeyDerivation( pUser->macSessionKey.data, pUser->applicationKey.len , (NQ_BYTE*)"SMB2APP\0"    , 8  , (NQ_BYTE*)"SmbRpc\0"    , 7 , (NQ_BYTE *)pUser->applicationKey.data );
        cmKeyDerivation( pUser->macSessionKey.data, pUser->macSessionKey.len  , (NQ_BYTE*)"SMB2AESCMAC\0", 12 , (NQ_BYTE*)"SmbSign\0"   , 8 , (NQ_BYTE *)pUser->macSessionKey.data );
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return NQ_SUCCESS;
}

static NQ_STATUS sendRequest(CCServer * pServer, CCUser * pUser, Request * pRequest, Match * pMatch, NQ_BOOL (*callback)(CMItem * pItem))
{
    NQ_UINT32 packetLen;        /* packet length of both in and out packets */
    CMBufferWriter writer;      /* to write down MID */
    Context * pContext;         /* server context */
    NQ_STATUS result = NQ_SUCCESS; /* return value */
    NQ_BYTE * encryptedBuf = NULL; /* encrypted buffer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p user:%p request:%p match:%p", pServer, pUser, pRequest, pMatch);

    if (pServer->smbContext == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, " smbContext in CCServer object is missing");
        result = NQ_ERR_NOTCONNECTED;
        goto Error;
    }

    if (!ccServerWaitForCredits(pServer, (NQ_COUNT)((pServer->capabilities & CC_CAP_LARGEMTU) ? pRequest->header.creditCharge : 1)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "No credits");
        result = NQ_ERR_TIMEOUT;
        goto Error;
    }    

    ccTransportLock(&pServer->transport);
    cmListItemTake(&pServer->item);

    if (!pServer->transport.connected || !pUser->logged)
    {
        if (!pServer->transport.connected)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, " transport isn't connected");
            result = NQ_ERR_NOTCONNECTED;
            goto Exit;
        }

        if (!pUser->logged && pRequest->header.command != SMB2_CMD_SESSIONSETUP)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "User: %s isn't logged, probably reconnect failed.", pUser->credentials && pUser->credentials->user ? cmWDump(pUser->credentials->user) : "");
            result = NQ_ERR_SESSIONREAUTHREQUIRED;
            goto Exit;
        }
    }
    /* set response as not received */
    pMatch->response->wasReceived = FALSE;

    /* write down MID */
    pContext = (Context *)pServer->smbContext;
    syMutexTake(&pContext->midGuard);
    pRequest->header.mid = pMatch->mid = pContext->mid;

    /* prepare MID for next request */
    cmU64AddU32(&pContext->mid, (NQ_UINT32)(pRequest->header.creditCharge > 0 ? pRequest->header.creditCharge : 1));
    syMutexGive(&pContext->midGuard);

    pMatch->cmd = pRequest->header.command;
    packetLen = cmBufferWriterGetDataCount(&pRequest->writer) - 4;  /* NBT header */
    cmBufferWriterInit(&writer, IOBUF_SKIPBYTE(pRequest->buffer, SEQNUMBEROFFSET), (NQ_COUNT)packetLen);
    IOBUF_SKIPBACKBYTE(pRequest->buffer, SEQNUMBEROFFSET);
    cmBufferWriteUint64(&writer, &pRequest->header.mid);

    /* add match to list only after mid was set */
    cmListItemAdd(&pServer->expectedResponses, (CMItem *)pMatch, callback);



    /* compose signature */
    if (
            !pRequest->encrypt && (
#ifdef UD_NQ_INCLUDESMB311
            ( (pServer->smb->revision == SMB3_1_1_DIALECTREVISION) && ccUserUseSignatures(pUser) &&
                    ((pRequest->command == SMB2_CMD_TREECONNECT) || (pRequest->command == SMB2_CMD_IOCTL && pRequest->subCommand == SMB_IOCTL_VALIDATE_NEGOTIATE)))
            ||
#endif /* UD_NQ_INCLUDESMB311 */
            (ccServerUseSignatures(pServer) && ccUserUseSignatures(pUser) && (pRequest->header.command != SMB2_CMD_SESSIONSETUP))
            )
        )
    {
        NQ_IOBufPos bufTmpPos;

        bufTmpPos = pRequest->buffer;
        IOBUF_MOVEBYTES(bufTmpPos, 4);
        IOBUF_BUFFER_CANUSEFLAT_ASSERT(pRequest->header._start, (SMB2_SECURITY_SIGNATURE_OFFSET + 16));
        cmSmb3CalculateMessageSignature(
            pUser->macSessionKey.data,
            pUser->macSessionKey.len,
            bufTmpPos,
            (NQ_UINT)packetLen,
            pRequest->tail.data,
            pRequest->tail.len,
            IOBUF_GETBYTEPTR(pRequest->header._start) + SMB2_SECURITY_SIGNATURE_OFFSET
            );
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    {
        NQ_IOBufPos tempBuf;

        tempBuf = pRequest->buffer;
        IOBUF_MOVEBYTES(tempBuf , 4);
        IOBUF_IOVECSET_START(tempBuf , NULL);

        pServer->captureHdr.receiving = FALSE;
        cmCapturePacketWriteStart(&pServer->captureHdr , (NQ_UINT)(packetLen + pRequest->tail.len));
        cmCapturePacketWritePacket( tempBuf, (NQ_UINT)packetLen);
        if (pRequest->tail.len > 0)
        {
            cmCapturePacketWritePacket(pRequest->tail.data, pRequest->tail.len);
        }
        cmCapturePacketWriteEnd();
        IOBUF_IOVEC_RESTORE(tempBuf , NULL);
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Request: command=%u, credit charge=%d, credits req=%d, mid=%u/%u, sid.low=0x%x, signed:%d, async:%d, pid(async.high)=0x%x, tid(async.low)=0x%x",
        pRequest->header.command, pRequest->header.creditCharge, pRequest->header.credits, pRequest->header.mid.high, pRequest->header.mid.low, pRequest->header.sid.low, (pRequest->header.flags & SMB2_FLAG_SIGNED) > 0,
        (pRequest->header.flags & SMB2_FLAG_ASYNC_COMMAND) > 0,
        pRequest->header.flags & SMB2_FLAG_ASYNC_COMMAND ? pRequest->header.aid.high : pRequest->header.pid,
        pRequest->header.flags & SMB2_FLAG_ASYNC_COMMAND ? pRequest->header.aid.low : pRequest->header.tid);

    if (pRequest->encrypt && ccUserUseSignatures(pUser) && pRequest->header.command != SMB2_CMD_SESSIONSETUP)
    {
        NQ_UINT32 msgLen = packetLen + pRequest->tail.len;

        encryptedBuf = (NQ_BYTE *)cmMemoryAllocate((NQ_UINT)(msgLen + SMB2_TRANSFORMHEADER_SIZE + 4));
        if (encryptedBuf != NULL)
        {
            CMSmb2TransformHeader   transformHeader;
            CMBufferWriter  writer;
            NQ_IOBufPos msgPoint;
            NQ_IOBufPos addPoint;
            NQ_IOBufPos addPoint1;
            NQ_IOBufPos bufEncPos;
            IOBUF_POSCONSTRUCTORINIT(bufEncPos)

            IOBUF_POSCONSTRUCTOR(bufEncPos, encryptedBuf, msgLen + SMB2_TRANSFORMHEADER_SIZE + 4)
            syMemset(&transformHeader, 0, sizeof(transformHeader));
            transformHeader.encryptionArgorithm = SMB2_ENCRYPTION_AES128_CCM;
            transformHeader.originalMsgSize = msgLen;
            transformHeader.sid = pUser->uid;
            ccComposeEncryptionNonce(transformHeader.nonce , pMatch->mid.low);
            cmBufferWriterInit(&writer, IOBUF_SKIPBYTE(bufEncPos, 4), (NQ_COUNT)(msgLen + SMB2_TRANSFORMHEADER_SIZE));
            IOBUF_SKIPBACKBYTE(bufEncPos, 4);
            addPoint1 = addPoint = cmBufferWriterGetPosition(&writer);
            IOBUF_MOVEBYTES(addPoint, 20);
            IOBUF_MOVEBYTES(addPoint1, 4);
            cmSmb2TransformHeaderWrite(&transformHeader, &writer);
            msgPoint = cmBufferWriterGetPosition(&writer);
            IOBUF_BUFFER_CANUSEFLAT_ASSERT(pRequest->buffer , packetLen + 4);
            cmBufferWriteBytes(&writer, IOBUF_GETBYTEPTR(pRequest->buffer) + 4  , (NQ_COUNT)packetLen);
            cmBufferWriteIOBytes(&writer, pRequest->tail.data, pRequest->tail.len);
            cmSmb3EncryptMessage(pUser->decryptionKey.data, transformHeader.nonce, msgPoint, (NQ_UINT)msgLen, addPoint,
                SMB2_TRANSFORMHEADER_SIZE - 20, IOBUF_GETBYTEPTR(addPoint1), pUser->server->isAesGcm);

            if (!ccTransportSend(
                            &pServer->transport,
                            bufEncPos,
                            (NQ_COUNT)(msgLen + SMB2_TRANSFORMHEADER_SIZE),
                            (NQ_COUNT)(msgLen + SMB2_TRANSFORMHEADER_SIZE)
                            )
                        )
            {
                result = (NQ_STATUS)syGetLastError();
                goto Exit;
            }
        }
        else
        {
            sySetLastError(NQ_ERR_NOMEM);
            LOGERR(CM_TRC_LEVEL_ERROR, "Allocating memory for encrypted buffer failed.");
            result = NQ_ERR_NOMEM;
            goto Exit;
        }
    }
    else
    {
#ifdef UD_NQ_INCLUDESMB311
        if (pUser->isPreauthIntegOn && (pRequest->header.command == SMB2_CMD_SESSIONSETUP) && (pServer->smb->revision == SMB3_1_1_DIALECTREVISION))
        {
            /* calculate message hash - all messages till session setup success */
            NQ_BYTE *packetBuf;

            if (pRequest->tail.len > 0)
            {
                NQ_IOBufPos ioBufPosTmp;
                IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

                packetBuf = cmBufManTake((packetLen + pRequest->tail.len));
                if (packetBuf == NULL)
                {
                    sySetLastError(NQ_ERR_NOMEM);
                    LOGERR(CM_TRC_LEVEL_ERROR, "Allocating memory for hashing message failed.");
                    result = NQ_ERR_NOMEM;
                    goto Exit;
                }
                IOBUF_MEMCPY_V2F(packetBuf, IOBUF_SKIPBYTE(pRequest->buffer, 4), packetLen);
                IOBUF_SKIPBACKBYTE(pRequest->buffer, 4);
                IOBUF_MEMCPY_V2F(packetBuf + packetLen, pRequest->tail.data, pRequest->tail.len);
                IOBUF_POSCONSTRUCTOR(ioBufPosTmp, packetBuf, packetLen + pRequest->tail.len)
                cmSmb311CalcMessagesHash(ioBufPosTmp, (packetLen + pRequest->tail.len), pUser->preauthIntegHashVal, NULL);
                cmBufManGive(packetBuf);
            }
            else
            {
                NQ_IOBufPos ioBufPosTmp = pRequest->buffer;

                IOBUF_MOVEBYTES(ioBufPosTmp, 4);
                cmSmb311CalcMessagesHash(ioBufPosTmp, packetLen, pUser->preauthIntegHashVal, NULL);
            }
        }
#endif /* UD_NQ_INCLUDESMB311 */

        if (!ccTransportSend(
                &pServer->transport,
                pRequest->buffer,
                (NQ_COUNT)(packetLen + pRequest->tail.len),
                (NQ_COUNT)packetLen
                )
            )
        {
            result = (NQ_STATUS)syGetLastError();
            goto Exit;
        }

        if (0 != pRequest->tail.len &&
            !ccTransportSendTail(&pServer->transport, pRequest->tail.data, pRequest->tail.len)
            )
        {
            result = (NQ_STATUS)syGetLastError();
        }
    }

Exit:
    cmListItemGive(&pServer->item);
    ccTransportUnlock(&pServer->transport);
    cmMemoryFree(encryptedBuf);

Error:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", result);
    return result;
}

static NQ_STATUS sendReceive(CCServer * pServer, CCUser * pUser, Request * pRequest, Response * pResponse)
{
    NQ_STATUS res;                  /* send result */
    CMThread * pThread;             /* current thread */
    Match * pMatch;                 /* match structure pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p user:%p request:%p response:%p", pServer, pUser, pRequest, pResponse);

    IOBUF_POSINIT(pResponse->buffer);
    pThread = cmThreadGetCurrent();
    if (NULL == pThread)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, ">>>No thread object.");
        res = NQ_ERR_GETDATA;
        goto Exit;
    }

    pMatch = (Match *)cmThreadGetContextAsStatItem(pThread, sizeof(Match));
    if (NULL == pMatch)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_OUTOFMEMORY;
        goto Exit;
    }

    pMatch->thread = pThread;
    pMatch->response = pResponse;
    pMatch->cond = &pThread->syncCond;
    pMatch->server = pServer;
    pMatch->userId = pUser->uid;
    pMatch->isResponseAllocated = FALSE;
    pMatch->item.locks = 0;
    pMatch->matchExtraInfo = MATCHINFO_NONE;

    cmThreadCondClear(pMatch->cond); /* Cleaning up the condition socket before sending*/

    res = pServer->smb->sendRequest(pServer, pUser, pRequest, pMatch, NULL);
    if (NQ_SUCCESS != res)
    {
        if (NQ_ERR_SESSIONREAUTHREQUIRED == res)
        {
            if (FALSE == ccUserRelogon(pUser))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "user relogon failed");
                res = NQ_ERR_USERRELOGONFAILED;
            }
            else
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "user relogon succeeded, trying again");
                res = NQ_ERR_TRYAGAIN;
            }
        }

        if (NULL != pMatch->thread->element.item.guard)
        {
            syMutexDelete(pMatch->thread->element.item.guard);
            cmMemoryFree(pMatch->thread->element.item.guard);
            pMatch->thread->element.item.guard = NULL;
        }

        cmListItemRemove((CMItem *)pMatch);
        goto Exit;
    }

    if (FALSE == cmThreadCondWait(pMatch->cond, ccConfigGetTimeout()))
    {
        NQ_BOOL wasReceived;    /* temp variable for pMatch->response->wasReceived */

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "No response for %d sec.", ccConfigGetTimeout());

        cmListItemRemove((CMItem *)pMatch);
        /* Double check if a response was arrived */
        cmListItemTake((CMItem *)pMatch);
        wasReceived = pMatch->response->wasReceived;
        cmListItemGive((CMItem *)pMatch);

        if ((FALSE == wasReceived))
        {
            res = NQ_ERR_TIMEOUT;
            if (((FALSE == pServer->transport.connected) || (IOBUF_ISNULL(pResponse->buffer)))
                && (pRequest->command != SMB2_CMD_NEGOTIATE) && (pRequest->command != SMB2_CMD_SESSIONSETUP)
                && !((pRequest->command == SMB2_CMD_IOCTL) && (pRequest->subCommand == SMB_IOCTL_VALIDATE_NEGOTIATE)))
            {
                res = NQ_ERR_TRYAGAIN;

                if (FALSE == ccServerReconnect(pServer))
                {
                    res = NQ_ERR_NOTCONNECTED;
                }
                else
                {
                    pServer->smb->signalAllMatch(&pServer->transport);
                }
            }
            else
            {
                pServer->smb->signalAllMatch(&pServer->transport);
            }

            if (NULL != pMatch->thread->element.item.guard)
            {
                syMutexDelete(pMatch->thread->element.item.guard);
                cmMemoryFree(pMatch->thread->element.item.guard);
                pMatch->thread->element.item.guard = NULL;
            }

            goto Exit;
        }

    }

    if (pServer->connectionBroke)
    {
        if (!ccServerReconnect(pServer))
        {
            res = NQ_ERR_NOTCONNECTED;
            goto Exit;
        }

        cmListItemTake((CMItem *) pServer);
        pServer->connectionBroke = FALSE;
        cmListItemGive((CMItem *)pServer);
        res = NQ_ERR_TRYAGAIN;
        goto Exit;
    }

    /* check connection */
    if (!pServer->transport.connected)
    {
        pServer->smb->signalAllMatch(&pServer->transport);
        if (pRequest->command != SMB2_CMD_NEGOTIATE && pRequest->command != SMB2_CMD_SESSIONSETUP)
        {
            if (ccServerReconnect(pServer))
            {
                /* simulate timeout - causing retry */
                res = NQ_ERR_TRYAGAIN;
                goto Exit;
            }
        }
        res = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

    if (FALSE == pMatch->response->wasReceived)
    {
        if (NULL != pMatch->thread->element.item.guard)
        {
            syMutexDelete(pMatch->thread->element.item.guard);
            cmMemoryFree(pMatch->thread->element.item.guard);
            pMatch->thread->element.item.guard = NULL;
        }

        if (pMatch->matchExtraInfo & MATCHINFO_WASSIGNALED)
        {
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Match was signaled, return error code to try again the command.");
            res = NQ_ERR_TRYAGAIN;
        }
        else
        {
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Invalid response, do no try again the command.");
            res = NQ_ERR_GETDATA;
        }

        cmListItemRemove((CMItem *)pMatch);
        goto Exit;
    }

    if(((SMB_STATUS_USER_SESSION_DELETED == pResponse->header.status) || (SMB_STATUS_NETWORK_SESSION_EXPIRED == pResponse->header.status)) && (SMB2_CMD_SESSIONSETUP != pResponse->header.command))
    {
        if (FALSE == ccUserRelogon(pUser))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "user relogon failed");
            res = NQ_ERR_USERRELOGONFAILED;
            goto Exit;
        }

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "user relogon succeeded, trying again");
        res = NQ_ERR_TRYAGAIN;
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Response: command=%u, credit charge=%d, credits granted=%d, mid=%u/%u, sid.low=0x%x, status=0x%x, signed:%d, async:%d, pid(async.high)=0x%x, tid(async.low)=0x%x",
        pResponse->header.command, pResponse->header.creditCharge, pResponse->header.credits, pResponse->header.mid.high, pResponse->header.mid.low, pResponse->header.sid.low, pResponse->header.status, (pResponse->header.flags & SMB2_FLAG_SIGNED) > 0,
        (pResponse->header.flags & SMB2_FLAG_ASYNC_COMMAND) > 0,
        pResponse->header.flags & SMB2_FLAG_ASYNC_COMMAND ? pResponse->header.aid.high : pResponse->header.pid,
        pResponse->header.flags & SMB2_FLAG_ASYNC_COMMAND ? pResponse->header.aid.low : pResponse->header.tid);

#ifdef UD_NQ_INCLUDESMB311
    if (pUser->isPreauthIntegOn && (pResponse->header.command == SMB2_CMD_SESSIONSETUP) && (pServer->smb->revision == SMB3_1_1_DIALECTREVISION))
    {
        if ((IOBUF_ISNULL(pResponse->buffer)) && (pResponse->tailLen > 0))
        {
            /* dialect 3.1.1 if signature fails in this stage we should fail the session setup */
            LOGERR(CM_TRC_LEVEL_ERROR, "Received empty data in response.");
            res = NQ_ERR_GETDATA;
            goto Exit;
        }

        if (pResponse->header.status != SMB_STATUS_SUCCESS &&
                pResponse->header.status != SMB_STATUS_MORE_PROCESSING_REQUIRED)
        {
            /* restart the preauthIntegHashVal to the start of session setup phase */
            syMemcpy(pUser->preauthIntegHashVal , pServer->preauthIntegHashVal , SMB3_PREAUTH_INTEG_HASH_LENGTH);
            LOGERR(CM_TRC_LEVEL_ERROR, "Session setup error: 0x%08x.", pResponse->header.status);
            /* otherwise real error code returned by server is lost */
            res = (NQ_STATUS)ccErrorsStatusToNq(pResponse->header.status, TRUE);
            sySetLastError((NQ_UINT32)res);
            goto Exit;
        }
        else if (pResponse->header.status != SMB_STATUS_SUCCESS)
        {
            NQ_IOBufPos ioBufPosTmp;
            IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)
            NQ_BYTE *buf;

            buf = cmBufManTake((NQ_COUNT)SMB2_HEADERANDSTRUCTSIZE + pResponse->tailLen);
            if (NULL == buf)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
                sySetLastError(NQ_ERR_OUTOFMEMORY);
                goto Exit;
            }

            syMemcpy(buf, &pMatch->hdrBuf, SMB2_HEADERANDSTRUCTSIZE);

            if (pResponse->tailLen > 0)
            {
                IOBUF_MEMCPY_V2F(buf + SMB2_HEADERANDSTRUCTSIZE, pResponse->buffer, pResponse->tailLen);
            }
            IOBUF_POSCONSTRUCTOR(ioBufPosTmp, buf, (NQ_COUNT)SMB2_HEADERANDSTRUCTSIZE + pResponse->tailLen)
            cmSmb311CalcMessagesHash(ioBufPosTmp, (NQ_COUNT)SMB2_HEADERANDSTRUCTSIZE + pResponse->tailLen, pUser->preauthIntegHashVal, NULL);
            cmBufManGive(buf);
        }
        else /* session setup success */
        {
            NQ_UINT16 sessionFlags;

            cmBufferReadUint16(&pResponse->reader, &sessionFlags);
            cmBufferReaderSetPosition(&pResponse->reader, cmBufferReaderGetStart(&pResponse->reader));

            /* on dialect 3.1.1 and above, the last session setup (when success) should be signed and validated */
            if (!(sessionFlags & SMB2SESSIONFLAG_IS_GUEST) && !pUser->isAnonymous)
            {
                pUser->isPreauthIntegOn = FALSE; /* hashing process done. turn off hash flag */

                /* we save the response to be signature validated  because for some authentication mechanisms
                   the blob in response should be processed first in order to generate signing key */
                pUser->savedResponse.len = (NQ_COUNT)(SMB2_HEADERANDSTRUCTSIZE + pResponse->tailLen);
                pUser->savedResponse.data = cmMemoryAllocate(pUser->savedResponse.len);
                if (NULL == pUser->savedResponse.data)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    res = NQ_ERR_OUTOFMEMORY;
                    goto Exit;
                }

                syMemcpy(pUser->savedResponse.data, pMatch->hdrBuf, SMB2_HEADERANDSTRUCTSIZE);
                syMemcpy(pUser->savedResponse.data + SMB2_HEADERANDSTRUCTSIZE, pResponse->buffer, pResponse->tailLen);
            }
        }
    }
#endif /* UD_NQ_INCLUDESMB311 */

    /* check signatures */
    if (!pRequest->encrypt && ccServerUseSignatures(pServer) && ccUserUseSignatures(pUser) 
        && (pResponse->header.flags & SMB2_FLAG_SIGNED) && (pResponse->header.command != SMB2_CMD_SESSIONSETUP))
    {
        /* on reconnect all encryption data is erased. we have to take server and avoid calling during reconnect */
        cmListItemTake(&pServer->item);
        if (FALSE == ccSmb30CheckMessageSignatureSMB3(pUser, pMatch->hdrBuf, (NQ_UINT)SMB2_HEADERANDSTRUCTSIZE, pResponse->buffer, pResponse->tailLen))
        {
            cmListItemGive(&pServer->item);
            LOGERR(CM_TRC_LEVEL_ERROR, "Signature mismatch in incoming packet");
            res = NQ_ERR_SIGNATUREFAIL;
            goto Exit;
        }
        cmListItemGive(&pServer->item);
    }

    res = (NQ_STATUS)ccErrorsStatusToNq(pResponse->header.status, TRUE);
    sySetLastError((NQ_UINT32)res);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", res);
    return res;
}

/*
 * Callback for waiting break Item dispose:
 *  - free response memory
 */
static NQ_BOOL waitingResponseItemDispose(CMItem * pItem)
{
    waitingResponse *pRespItem = (waitingResponse *)pItem;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pItem:%p ", pItem);

    cmIOBufManGive(pRespItem->notifyResponse->buffer);
    cmBufManGive((NQ_BYTE *)pRespItem->notifyResponse);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "TRUE");
    return TRUE;
}

/*
     * Server sends notify responses == response without request. Ex: break notification
     * If file ID for sent response isn't found. we save the response and try again on newly created files.
     * To avoid missing a break notification that is handled while file creation on our side still in process.
     */
static void handleWaitingNotifyResponse(void *pserver, void *pfile)
{
    CMIterator responseIterator;
    CCFile *pFile = (CCFile *)pfile;
    CCServer *pServer = (CCServer *)pserver;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pServer:%p pFile:%p", pServer, pFile);

    cmListIteratorStart(&pServer->waitingNotifyResponses, &responseIterator);

    while (cmListIteratorHasNext(&responseIterator))
    {
        waitingResponse *pResponsItem;
        pResponsItem = (waitingResponse *) cmListIteratorNext(&responseIterator);
        if (0 == syMemcmp(pResponsItem->fid, pFile->fid, sizeof(pFile->fid)))
        {
            /* parse notification */
            if (NULL == commandDescriptors[pResponsItem->notifyResponse->header.command].notificationHandle)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Wrong response saved in list, %d", pResponsItem->notifyResponse->header.command);
                /* no need to keep it */
                cmListItemRemoveAndDispose((CMItem *)pResponsItem);
                goto Exit;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Handling waiting response.");
            commandDescriptors[pResponsItem->notifyResponse->header.command].notificationHandle(pServer, pResponsItem->notifyResponse, pFile);
            if (!IOBUF_ISNULL(pResponsItem->notifyResponse->buffer))
            {
                cmIOBufManGive(pResponsItem->notifyResponse->buffer);
            }

            cmBufManGive((NQ_BYTE *)pResponsItem->notifyResponse);
            cmListItemRemoveAndDispose((CMItem *)pResponsItem);
            pServer->numOfNotifyResponses--;
        }
    }

Exit:
    cmListIteratorTerminate(&responseIterator);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void handleNotification(CCServer *pServer, CMSmb2Header *header, CMBlob *decryptPacket)
{
    Response* pResponse;
    NQ_BYTE * pFid;    /* pointer to file ID in the notification */
    CCFile *pFile;
    NQ_IOBufPos bufPos;
#ifdef UD_NQ_INCLUDESMBCAPTURE
    NQ_BOOL closePacketCapture = TRUE;
#endif

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pServer:%p header:%p decryptPacket:%p", pServer, header, decryptPacket);

    /* this is a notification == response without request */
    pResponse = (Response *)cmBufManTake(sizeof(Response));
    if (NULL == pResponse)
    {
        sySetLastError(NQ_ERR_NOMEM);
        LOGERR(CM_TRC_LEVEL_ERROR, "Allocating memory for waiting response failed.");
        goto Exit;
    }

    IOBUF_POSINIT(pResponse->buffer);

    if (decryptPacket->data != NULL)
    {
        /* handle notification should release transport when done - for decrypted packet case. reading from transport is done already */
        ccTransportReceiveEnd(&pServer->transport);

        pResponse->tailLen = (NQ_COUNT)(decryptPacket->len - SMB2_HEADERANDSTRUCTSIZE);
        pResponse->buffer = cmIOBufManTake(pResponse->tailLen);
        if (IOBUF_ISNULL(pResponse->buffer))
        {
            sySetLastError(NQ_ERR_NOMEM);
            LOGERR(CM_TRC_LEVEL_ERROR, "Allocating memory for waiting response failed.");
            cmMemoryFreeBlob(decryptPacket);
            goto Exit1;
        }

        IOBUF_MEMCPY_F2V(pResponse->buffer, decryptPacket->data + SMB2_HEADERANDSTRUCTSIZE, pResponse->tailLen);

        cmMemoryFreeBlob(decryptPacket);
    }
    else if (pServer->transport.recv.remaining > 0)
    {
        if (pServer->smbContext == NULL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, " smbContext in CCServer object is missing");
            goto Error;
        }

        pResponse->tailLen = pServer->transport.recv.remaining;
        pResponse->buffer = cmIOBufManTake(pResponse->tailLen);
        if (IOBUF_ISNULL(pResponse->buffer))
        {
            sySetLastError(NQ_ERR_NOMEM);
            LOGERR(CM_TRC_LEVEL_ERROR, "Allocating memory for waiting response failed.");
            goto Error;
        }

        if (pResponse->tailLen != ccTransportReceiveBytes(&pServer->transport, IOBUF_GETBUFPTR(pResponse->buffer), pResponse->tailLen))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Transport receive error.");
            goto Error;
        }

        ccTransportReceiveEnd(&pServer->transport);
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    cmCapturePacketWritePacket(pResponse->buffer, pResponse->tailLen);
    cmCapturePacketWriteEnd();
    closePacketCapture = FALSE;
#endif /* UD_NQ_INCLUDESMBCAPTURE */

    if (header->status != SMB_STATUS_SUCCESS)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Notification with error status: 0x%08x", header->status);
        goto Exit1;
    }

    cmBufferReaderInit(&pResponse->reader, pResponse->buffer, pResponse->tailLen);
    pResponse->header = *header;

    /* parse notification and find fid */
    cmBufferReaderSkip(&pResponse->reader, 6);  /* oplock + reserved + reserved 2 */
    bufPos = cmBufferReaderGetPosition(&pResponse->reader);
    IOBUF_BUFFER_CANUSEFLAT_ASSERT(bufPos, 16);
    pFid = IOBUF_GETBYTEPTR(bufPos);

    /* init reader again to take it back to start of packet */
    cmBufferReaderInit(&pResponse->reader, pResponse->buffer, pResponse->tailLen);

    /* handle or save notification response */
    pFile = ccFileFindById(pServer, pFid);

    if (NULL == pFile)
    {
        /* save this response for later handling. relevant file create might be happening concurrently */
        CMIterator responseIterator;
        waitingResponse *pWaitingResponse;

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "File ID not found, saving response.");
        cmListIteratorStart(&pServer->waitingNotifyResponses, &responseIterator);

        /* if maximum responses has reached replace the oldest item with the new one */
        if (MAX_WAITING_NOTIFY_RESPONSES <= pServer->numOfNotifyResponses)
        {
            waitingResponse *oldestResponsItem = NULL;
            NQ_TIME minArrivalTime;

            /* initialize minimum time */
            minArrivalTime.high = 0xFFFFFFFF;
            minArrivalTime.low = 0xFFFFFFFF;

            while(cmListIteratorHasNext(&responseIterator))
            {
                waitingResponse *currentResponsItem;

                currentResponsItem = (waitingResponse *) cmListIteratorNext(&responseIterator);
                if (1 != cmU64Cmp(&currentResponsItem->arrivalTime, &minArrivalTime))
                {
                    oldestResponsItem = currentResponsItem;
                    cmU64AssignU64(&minArrivalTime, &currentResponsItem->arrivalTime);
                }
            }

            if (oldestResponsItem != NULL)
            {
                if (!IOBUF_ISNULL(oldestResponsItem->notifyResponse->buffer))
                {
                    cmIOBufManGive(oldestResponsItem->notifyResponse->buffer);
                }

                cmBufManGive((NQ_BYTE *)oldestResponsItem->notifyResponse);
                oldestResponsItem->notifyResponse = pResponse;
                syMemcpy(oldestResponsItem->fid, pFid, sizeof(oldestResponsItem->fid));
                oldestResponsItem->arrivalTime = syGetTimeInMsec();
                cmListIteratorTerminate(&responseIterator);
                goto Exit;
            }
            else
            {
                cmListIteratorTerminate(&responseIterator);
                goto Exit1;
            }
        }

        pWaitingResponse = (waitingResponse *) cmListItemCreateAndAdd(&pServer->waitingNotifyResponses, sizeof(waitingResponse),
                    NULL, waitingResponseItemDispose, CM_LISTITEM_NOLOCK, FALSE);

        if (NULL == pWaitingResponse)
        {
            cmListIteratorTerminate(&responseIterator);
            sySetLastError(NQ_ERR_NOMEM);
            LOGERR(CM_TRC_LEVEL_ERROR, "Allocating memory for waiting response failed.");
            goto Exit1;
        }

        pWaitingResponse->notifyResponse = pResponse;
        syMemcpy(pWaitingResponse->fid, pFid, sizeof(pWaitingResponse->fid));
        pWaitingResponse->arrivalTime = syGetTimeInMsec();
        pServer->numOfNotifyResponses++;
        cmListIteratorTerminate(&responseIterator);

        goto Exit;
    }

    /* file found - handle notification message */
    commandDescriptors[header->command].notificationHandle(pServer, pResponse, pFile);

    goto Exit1;

Error:
    ccTransportReceiveEnd(&pServer->transport);

Exit1:
    if (!IOBUF_ISNULL(pResponse->buffer))
    {
        cmIOBufManGive(pResponse->buffer);
    }
    cmBufManGive((NQ_BYTE *)pResponse);

Exit:
#ifdef UD_NQ_INCLUDESMBCAPTURE
    if (closePacketCapture)
    {
        cmCapturePacketWriteEnd();
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void anyResponseCallback(void * transport)
{
    CCTransport *   pTransport = (CCTransport *)transport;  /* casted to transport entry */
    CCServer *      pServer;                        /* casted pointer */
    CMIterator      iterator;                       /* iterates through expected responses */
    CMSmb2Header    header;                         /* response header */
    CMBufferReader  reader;                         /* to parse header */
    NQ_COUNT        res;                            /* bytes read */
    NQ_BYTE         buffer[SMB2_HEADERANDSTRUCTSIZE];/* header + structure size */
    CMBlob          decryptPacket;
    NQ_BYTE*        tHdr = NULL;
    NQ_IOBufPos     bufPos;
    NQ_IOBufPos     bufCryptPos;
    NQ_IOBufPos     bufHdrPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)
    IOBUF_POSCONSTRUCTORINIT(bufCryptPos)
    IOBUF_POSCONSTRUCTORINIT(bufHdrPos)
    IOBUF_POSINIT(bufCryptPos);

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transport:%p", transport);

    decryptPacket.data = NULL;
    decryptPacket.len = 0;
    pServer = (CCServer *)pTransport->context;
    
    if (!pTransport->connected) /* proceed disconnect */
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Connection broken with %s", cmWDump(pServer->item.name));

        /* match with request */
        cmListItemTake((CMItem *)pServer);
        cmListIteratorStart(&pServer->expectedResponses, &iterator);
        while (cmListIteratorHasNext(&iterator))
        {
            Match * pMatch;

            pMatch = (Match *)cmListIteratorNext(&iterator);
            cmListItemTake((CMItem *)pMatch);
            if (pMatch->cond != NULL)
            {
                cmThreadCondSignal(pMatch->cond);
            }

            pMatch->matchExtraInfo |= MATCHINFO_WASSIGNALED;
            if (pMatch->isResponseAllocated)
            {
                cmMemoryFree(pMatch->response);
                pMatch->response = NULL;
                pMatch->isResponseAllocated = FALSE;
            }

            cmListItemGive((CMItem *)pMatch);
        }

        cmListIteratorTerminate(&iterator);
        if (NULL != pTransport->cleanupCallback)
        {
            (*pTransport->cleanupCallback)(pTransport->cleanupContext);
        }

        cmListItemGive((CMItem *)pServer);

        LOGERR(CM_TRC_LEVEL_ERROR, "Transport not connected: %s", cmWDump(pServer->item.name));
        goto Error;
    }

    /* read & parse SMB signature */
    IOBUF_POSCONSTRUCTOR(bufPos, buffer, sizeof(buffer))
    res = ccTransportReceiveBytes(pTransport, IOBUF_GETBUFPTR(bufPos), 4);
    if ((NQ_COUNT)NQ_FAIL == res)
    {
        goto Error;
    }
    if (0 == IOBUF_MEMCMP_V2F(bufPos, cmSmb2TrnsfrmHdrProtocolId, sizeof(cmSmb2TrnsfrmHdrProtocolId)))
    {   
        CMSmb2TransformHeader   transHeader;
        CMIterator              userItr;
        CCUser          *       pUser = NULL;
        CMList                  fakeUserList;
       
        tHdr = (NQ_BYTE *)cmMemoryAllocate(SMB2_TRANSFORMHEADER_SIZE);
        if (NULL == tHdr)
        {
            goto Error;
        }

        syMemcpy(tHdr ,cmSmb2TrnsfrmHdrProtocolId , sizeof(cmSmb2TrnsfrmHdrProtocolId));
        IOBUF_POSCONSTRUCTOR(bufHdrPos, tHdr, SMB2_TRANSFORMHEADER_SIZE)
        IOBUF_MOVEBYTES(bufHdrPos, sizeof(cmSmb2TrnsfrmHdrProtocolId));
        IOBUF_IOVECSET_START(bufHdrPos, NULL);
        res = ccTransportReceiveBytes(pTransport, IOBUF_GETBUFPTR(bufHdrPos), SMB2_TRANSFORMHEADER_SIZE - 4);
        if ((NQ_COUNT)NQ_FAIL == res)
        {
            goto Error;
        }
        IOBUF_IOVEC_RESTORE(bufHdrPos, NULL);
        IOBUF_MOVEBYTES(bufHdrPos, -(NQ_INT)(sizeof(cmSmb2TrnsfrmHdrProtocolId)));

        cmBufferReaderInit(&reader, bufHdrPos, SMB2_TRANSFORMHEADER_SIZE);
        cmSmb2TransformHeaderRead(&transHeader, &reader);
        /* Create Fake List with the same values as pServer->users to avoid deadlock (mutex is different)*/
        fakeUserList.first = pServer->users.first;
        fakeUserList.last = pServer->users.last;
        fakeUserList.isUsed = TRUE;
        syMutexCreate(&fakeUserList.guard);
        cmListIteratorStart(&fakeUserList,&userItr);
        while (cmListIteratorHasNext(&userItr) && pUser == NULL)
        {
            CCUser * pTemp = (CCUser *)cmListIteratorNext(&userItr);

            if (cmU64Cmp(&transHeader.sid,&pTemp->uid) == 0)
            {
                pUser = pTemp;
                break;
            }
        }
        cmListIteratorTerminate(&userItr);
        syMutexDelete(&fakeUserList.guard);
        if (pUser == NULL)
        {
            goto Error;
        }
        /* decrypt packet here*/ 
        decryptPacket.data = (NQ_BYTE *)cmMemoryAllocate((NQ_UINT)transHeader.originalMsgSize);
        decryptPacket.len = (NQ_COUNT)transHeader.originalMsgSize;
        IOBUF_POSCONSTRUCTOR(bufCryptPos, decryptPacket.data, decryptPacket.len)
        res = ccTransportReceiveBytes(pTransport, IOBUF_GETBUFPTR(bufCryptPos) , (NQ_COUNT)transHeader.originalMsgSize);
        if ((NQ_COUNT)NQ_FAIL == res)
        {
            goto Error;
        }
        
        if (FALSE == cmSmb3DecryptMessage(pUser->encryptionKey.data,
                                    transHeader.nonce,
                                    bufCryptPos,
                                    decryptPacket.len,
                                    IOBUF_SKIPBYTE(bufHdrPos, 20),
                                    (NQ_COUNT)(SMB2_TRANSFORMHEADER_SIZE - 20),
                                    transHeader.signature,
                                    pUser->server->isAesGcm
                                    ))
        {
            goto Error;
        }
        cmMemoryFree(tHdr);
        tHdr = NULL;
        IOBUF_MEMCPY_V2V(bufPos , bufCryptPos , sizeof(buffer));
    }
    else
    {
        /* receive header */
        IOBUF_MOVEBYTES(bufPos, sizeof(cmSmb2ProtocolId))
        IOBUF_IOVECSET_START(bufPos, NULL);
        res = ccTransportReceiveBytes(pTransport, IOBUF_GETBUFPTR(bufPos), sizeof(buffer) - sizeof(cmSmb2ProtocolId));
        if ((NQ_COUNT)NQ_FAIL == res)
        {
            goto Error;
        }
        IOBUF_IOVEC_RESTORE(bufPos, NULL);
        IOBUF_MOVEBYTES(bufPos, -(NQ_INT)(sizeof(cmSmb2ProtocolId)));
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    pServer->captureHdr.receiving = TRUE;
    cmCapturePacketWriteStart(&pServer->captureHdr , (NQ_UINT)(decryptPacket.data != NULL ? decryptPacket.len : SMB2_HEADERANDSTRUCTSIZE + pServer->transport.recv.remaining));
    cmCapturePacketWritePacket( bufPos, SMB2_HEADERANDSTRUCTSIZE);
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    cmBufferReaderInit(&reader, bufPos, res); /* starting from SMB header */
    syMemset(&header, 0, sizeof(CMSmb2Header));
    cmSmb2HeaderRead(&header, &reader);

    /* match with request */
    cmListIteratorStart(&pServer->expectedResponses, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        Match * pMatch;
        
        pMatch = (Match *)cmListIteratorNext(&iterator);
        cmListItemTake((CMItem*)pMatch);
        if (pMatch->server == pServer && 0 == cmU64Cmp(&pMatch->mid, &header.mid))
        {
            /* the received response message ID matches this waiting match structure */
            NQ_UINT16 structSize;           /* structure size */

            if (NULL == pMatch->response)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Invalid response");
                cmListItemGive((CMItem*)pMatch);
                ccTransportReceiveEnd(&pServer->transport);
                cmListIteratorTerminate(&iterator);
                goto Exit;
            }

            pMatch->response->header = header; /* header start address will not match the command start address  */
            cmBufferReadUint16(&reader, &structSize); /* structure size */

            if ((header.command > COMMAND_MAX) || (header.command != pMatch->cmd))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Invalid command code: %u, expected: %u", header.command, pMatch->cmd);
                pMatch->response->wasReceived = FALSE;
                ccTransportReceiveEnd(&pServer->transport);
                if (NULL != pMatch->cond)
                {
                    cmThreadCondSignal(pMatch->cond);
                }

                cmListItemGive((CMItem*)pMatch);
                cmListIteratorTerminate(&iterator);
                goto Exit;
            }

            if (((SMB_STATUS_SUCCESS == header.status) && (structSize != commandDescriptors[header.command].responseStructSize))
                ||
                    ((SMB_STATUS_SUCCESS != header.status)
                    &&
                    (structSize != SMB2ERRORRESPONSE_STRUCTSIZE)
                    &&
                    ((SMB2_CMD_IOCTL != header.command) || (structSize != commandDescriptors[header.command].responseStructSize))
                    )
                )
            {
                LOGERR( CM_TRC_LEVEL_ERROR, "Unexpected structure length in response: %d, expected %d, command %d, status 0x%08x",
                        structSize, commandDescriptors[header.command].responseStructSize, header.command, header.status);
                pMatch->response->wasReceived = FALSE;
                ccTransportReceiveEnd(&pServer->transport);
                if (NULL != pMatch->cond)
                {
                    cmThreadCondSignal(pMatch->cond);
                }

                cmListItemGive((CMItem *)pMatch);
                cmListIteratorTerminate(&iterator);
                goto Exit;
            }

            /* check for interim response */
            if ((header.flags & SMB2_FLAG_ASYNC_COMMAND) && (header.status == SMB_STATUS_PENDING))
            {
#ifdef UD_NQ_INCLUDESMBCAPTURE
                NQ_BYTE * tempBuf;
                NQ_COUNT len = 0;
                NQ_IOBufPos tmpBuf;
                IOBUF_POSCONSTRUCTORINIT(tmpBuf);

                IOBUF_POSINIT(tmpBuf);
                if (decryptPacket.data != NULL)
                {
                    len = (NQ_COUNT)(decryptPacket.len - SMB2_HEADERANDSTRUCTSIZE);
                    tempBuf = (NQ_BYTE *)cmMemoryAllocate(len);
                    if (NULL == tempBuf)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                        cmListItemGive((CMItem*)pMatch);
                        goto Error;
                    }
                    IOBUF_POSCONSTRUCTOR(tmpBuf, tempBuf, len)
                    syMemcpy(tempBuf , decryptPacket.data + SMB2_HEADERANDSTRUCTSIZE , len);
                }
                else
                {
                    len = pServer->transport.recv.remaining;
                    tempBuf = (NQ_BYTE *)cmMemoryAllocate(len);
                    if (NULL == tempBuf)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                        cmListItemGive((CMItem*)pMatch);
                        goto Error;
                    }
                    IOBUF_POSCONSTRUCTOR(tmpBuf, tempBuf, len)
                    ccTransportReceiveBytes(&pServer->transport,IOBUF_GETBUFPTR(tmpBuf), len);
                }

                cmCapturePacketWritePacket(tmpBuf , len);
                cmCapturePacketWriteEnd();
                cmMemoryFree(tempBuf);
#endif /* UD_NQ_INCLUDESMBCAPTURE */
                ccTransportReceiveEnd(pTransport);
                if (header.command == SMB2_CMD_WRITE || header.command == SMB2_CMD_READ)
                {
                    WriteMatch  *   wMatch = NULL;
                    void        **  tmp = NULL;
                    NQ_BOOL     *   isPending = NULL;

                    wMatch = (WriteMatch *)pMatch;

                    tmp = (void **)((CMItem *)wMatch->context + 1);
                    isPending = (NQ_BOOL *)*tmp;

                    *isPending = TRUE;

                    /* when pending response received the timeout is extended. */
                    wMatch->setTimeout = wMatch->setTimeout + (wMatch->setTimeout * PENDING_TIMEOUT_EXTENTION);
                }

                cmListItemGive((CMItem*)pMatch);
                ccTransportDiscardReceive(pTransport);
            } /* if ((header.flags & SMB2_FLAG_ASYNC_COMMAND) && (header.status == SMB_STATUS_PENDING)) */
            else
            {
                NQ_IOBufPos tmpBuf = bufCryptPos;

                cmListItemRemove((CMItem *)pMatch);
                if (NULL != pMatch->thread->element.item.guard)
                {
                    syMutexDelete(pMatch->thread->element.item.guard);
                    cmMemoryFree(pMatch->thread->element.item.guard);
                    pMatch->thread->element.item.guard = NULL;
                }

                if (pServer->useSigning)
                {
                    syMemcpy(pMatch->hdrBuf, buffer, SMB2_HEADERANDSTRUCTSIZE);
                }

                if ((SMB_STATUS_SUCCESS != header.status) && (SMB_STATUS_MORE_PROCESSING_REQUIRED != header.status))
                {
                    pMatch->thread->status = header.status;
                }

#ifdef UD_NQ_USEIOVECS
                /* for iovec set the byteOffset of header from command start. */
                pMatch->response->header._start.byteOffset = 0;
#endif

                if (NULL != commandDescriptors[header.command].callback)
                {
                    Response * pResponse = pMatch->response;  /* associated response */

                    if (!IOBUF_ISNULL(bufCryptPos))
                    {
                        pResponse->tailLen = (NQ_COUNT)(decryptPacket.len - SMB2_HEADERANDSTRUCTSIZE);

                        pResponse->buffer = cmIOBufManTake(pResponse->tailLen);

                        if (IOBUF_ISNULL(pResponse->buffer))
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                            cmListItemGive((CMItem*)pMatch);
                            cmListIteratorTerminate(&iterator);
                            goto ErrorAndCredits;
                        }
                        IOBUF_MOVEBYTES(tmpBuf, SMB2_HEADERANDSTRUCTSIZE);
                        IOBUF_MEMCPY_V2V(pResponse->buffer, tmpBuf, pResponse->tailLen);
                    }
                    else
                    {
                        pResponse->tailLen = pServer->transport.recv.remaining;
                        IOBUF_POSINIT(pResponse->buffer);
                    }

                    pMatch->response->wasReceived = TRUE;
                    commandDescriptors[header.command].callback(pServer, pMatch);
                    /* ccTransportReceiveEnd(&pServer->transport) called inside callback,
                     * notice that Match mutex is taken at the beginning of the loop and should be given
                     * inside the callback before disposing the Match */
                } /* if (NULL != commandDescriptors[header.command].callback) */
                else
                {   
                    /* this packet was encrypted */
                    if (!IOBUF_ISNULL(bufCryptPos))
                    {
                        Response * pResponse = pMatch->response;  /* associated response */

                        pResponse->tailLen = (NQ_COUNT)(decryptPacket.len - SMB2_HEADERANDSTRUCTSIZE);
                        pResponse->buffer = cmIOBufManTake(pResponse->tailLen);
                        if (!IOBUF_ISNULL(pResponse->buffer))
                        {
                            IOBUF_MOVEBYTES(tmpBuf, SMB2_HEADERANDSTRUCTSIZE);
                            IOBUF_MEMCPY_V2V(pResponse->buffer, tmpBuf, pResponse->tailLen);
#ifdef UD_NQ_INCLUDESMBCAPTURE
                            cmCapturePacketWritePacket( pResponse->buffer, pResponse->tailLen);
#endif /* UD_NQ_INCLUDESMBCAPTURE */
                            cmBufferReaderInit(&pResponse->reader, pResponse->buffer, pResponse->tailLen);
#ifndef UD_NQ_USEIOVECS
                            /* set virtual header start - shift back header size. will be used to calculate offsets from header start,
                             * should not be used to read header */
                            pResponse->header._start =  pResponse->buffer - SMB2_HEADERANDSTRUCTSIZE;
#else
                            /* this value will be used to calculate offset inside command */
                            pResponse->header._start.byteOffset = SMB2_HEADERANDSTRUCTSIZE;
#endif
                        }
                    } /* if (!IOBUF_ISNULL(bufCryptPos)) */
                    else if (pServer->transport.recv.remaining > 0)
                    {
                        Response * pResponse = pMatch->response;  /* associated response */
                        NQ_COUNT receivedBytes;

                        pResponse->tailLen = pServer->transport.recv.remaining;
                        pResponse->buffer = cmIOBufManTake(pResponse->tailLen);
                        if (!IOBUF_ISNULL(pResponse->buffer))
                        {
                            if (pResponse->tailLen == (receivedBytes = ccTransportReceiveBytes(&pServer->transport, IOBUF_GETBUFPTR(pResponse->buffer), pResponse->tailLen)))
                            {
#ifdef UD_NQ_INCLUDESMBCAPTURE
                                cmCapturePacketWritePacket( pResponse->buffer, pResponse->tailLen);
#endif /* UD_NQ_INCLUDESMBCAPTURE */
                                cmBufferReaderInit(&pResponse->reader, pResponse->buffer, pResponse->tailLen);
#ifdef UD_NQ_USEIOVECS
                               /* for iovec set the byteOffset of header from command start. */
                               pResponse->header._start.byteOffset = SMB2_HEADERANDSTRUCTSIZE;
#else
                               /* set virtual header start - shift back header size. will be used to calculate offsets from header start,
                               * should not be used to read header */
                               pResponse->header._start =   pResponse->buffer - SMB2_HEADERANDSTRUCTSIZE;
#endif /* UD_NQ_USEIOVECS */
#ifdef UD_NQ_INCLUDESMB311
                                /* Message header is required for hash calculation - all messages till session setup success */
                                if (pServer->smb->revision == SMB3_1_1_DIALECTREVISION && header.command == SMB2_CMD_SESSIONSETUP)
                                {
                                    IOBUF_MEMCPY_V2F(pMatch->hdrBuf, bufPos, SMB2_HEADERANDSTRUCTSIZE);
                                }
#endif /* UD_NQ_INCLUDESMB311 */
                            }
                            else
                            {
                                LOGERR(CM_TRC_LEVEL_ERROR, ">>>Number of network received bytes: %d not as expected: %d.", receivedBytes, pResponse->tailLen);
                                cmListItemGive((CMItem*)pMatch);
                                cmListIteratorTerminate(&iterator);
                                goto ErrorAndCredits;
                            }
                        }
                        else
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, ">>>Out of memory");
                            cmListItemGive((CMItem*)pMatch);
                            cmListIteratorTerminate(&iterator);
                            goto ErrorAndCredits;
                        } /* else */
                    } /* else if (pServer->transport.recv.remaining > 0) */
                    else
                    {
                        pMatch->response->tailLen = 0;
                    }
#ifdef UD_NQ_INCLUDESMBCAPTURE
                    cmCapturePacketWriteEnd();
#endif /* UD_NQ_INCLUDESMBCAPTURE */
                    ccTransportReceiveEnd(&pServer->transport);
                    pMatch->response->wasReceived = TRUE;
                    cmThreadCondSignal(pMatch->cond);
                    cmListItemGive((CMItem*)pMatch);
                } /* else */
            } /* else */

            cmListIteratorTerminate(&iterator);
            if (header.credits > 0)
            {
                ccServerPostCredits(pServer, header.credits);
            }

            goto Exit;
        } /*if (pMatch->server == pServer && 0 == cmU64Cmp(&pMatch->mid, &header.mid))*/

        cmListItemGive((CMItem*)pMatch);
    } /* while (cmListIteratorHasNext(&iterator)) */

    cmListIteratorTerminate(&iterator);

    /* No match request matched this response, check for notification message */
    if (header.command > COMMAND_MAX)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid command code");
        goto Error;
    }

    if (NULL != commandDescriptors[header.command].notificationHandle)
    {
        handleNotification(pServer, &header, &decryptPacket);
        /* ccTransportReceiveEnd(&pServer->transport) called inside callback */
        goto Exit;
    }
    else
    {
#ifdef UD_NQ_INCLUDESMBCAPTURE
        cmCapturePacketWriteEnd();
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        LOGERR(CM_TRC_LEVEL_ERROR, "Response not matched. Mid: %d:%d server: %s", header.mid.high, header.mid.low, cmWDump(pServer->item.name));
    }
ErrorAndCredits:
    /* for some reason the match wasn't found. or some other error occurred. still update credits. */
    if (header.credits > 0)
    {
        ccServerPostCredits(pServer, header.credits);
    }
Error:
    ccTransportReceiveEnd(&pServer->transport);
#ifdef UD_NQ_INCLUDESMBCAPTURE
    cmCapturePacketWriteEnd();
#endif /* UD_NQ_INCLUDESMBCAPTURE */

Exit:
    cmMemoryFree(tHdr);
    cmMemoryFreeBlob(&decryptPacket);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void writeCallback(CCServer * pServer, Match * pContext)
{
    WriteMatch * pMatch = (WriteMatch *)pContext;   /* casted pointer */
    NQ_BYTE buffer[20];                             /* buffer for structure */
    NQ_UINT tailLen = pServer->transport.recv.remaining;    /* bytes remaining */
    Response * pResponse = pContext->response;              /* response structure pointer */
    NQ_UINT32 count = 0;                                    /* bytes written */
    NQ_UINT32     currentTime;                    /* Current Time for checking timed-out responses*/
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p context:%p", pServer, pContext);

    IOBUF_POSCONSTRUCTOR(bufPos, buffer, sizeof(buffer))
    /* receive the rest of command */
    if (!IOBUF_ISNULL(pResponse->buffer))
    {
        IOBUF_MEMCPY_V2V(bufPos , pResponse->buffer , pResponse->tailLen > 20 ? 20 : pResponse->tailLen );
        cmIOBufManGive(pResponse->buffer);
#ifdef UD_NQ_INCLUDESMBCAPTURE
        count = 1;
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    }
    else if ( tailLen != ccTransportReceiveBytes(&pServer->transport, IOBUF_GETBUFPTR(bufPos), tailLen))
    {
        ccTransportReceiveEnd(&pServer->transport);
        goto Exit;
    }
    ccTransportReceiveEnd(&pServer->transport);
#ifdef UD_NQ_INCLUDESMBCAPTURE
    cmCapturePacketWritePacket( bufPos, (NQ_UINT)(count == 1 ? pResponse->tailLen : tailLen));
    cmCapturePacketWriteEnd();
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    cmBufferReaderInit(&pResponse->reader, bufPos, tailLen);

    /* parse the response */
    if (SMB_STATUS_SUCCESS == pResponse->header.status)
    {
        cmBufferReaderSkip(&pResponse->reader, sizeof(NQ_UINT16));  /* reserved */
        cmBufferReadUint32(&pResponse->reader, &count); /* count */
    }
    currentTime = (NQ_UINT32)syGetTimeInSec();

    /* check if session was deleted, if yes mark the user as logged off */
    if ((SMB_STATUS_USER_SESSION_DELETED == pResponse->header.status) || (SMB_STATUS_NETWORK_SESSION_EXPIRED == pResponse->header.status))
    {
        CCUser * pUser;

        pUser = ccUserFindById(pServer, pResponse->header.sid);
        if (NULL != pUser)
        {
            cmListItemTake((CMItem*)pUser);
            pUser->logged = FALSE;
            cmListItemGive((CMItem*)pUser);
        }
    }

    /* call up */
    if ((pMatch->timeCreated + pMatch->setTimeout) > currentTime)
    {
        pMatch->callback(pResponse->header.status == 0 ? 0 : (NQ_STATUS)ccErrorsStatusToNq(pResponse->header.status, TRUE), (NQ_UINT)count, pMatch->context);
    }
    else
    {
        /* response timed out */
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Write response arrived after timeout:%lu. Mid:%hu Expected:%lu arrived:%lu. Will not be handled.",
                (NQ_ULONG)pMatch->setTimeout, pMatch->match.mid, (NQ_ULONG)(pMatch->timeCreated + pMatch->setTimeout), (NQ_ULONG)currentTime);
    }

    /* release context */
    if (NULL != pMatch->match.thread->element.item.guard)
    {
        syMutexDelete(pMatch->match.thread->element.item.guard);
        cmMemoryFree(pMatch->match.thread->element.item.guard);
        pMatch->match.thread->element.item.guard = NULL;
    }

    cmMemoryFree(pMatch->match.response);
    /* the mutex is taken at the beginning of the while that iterates over Matches list */
    cmListItemGive((CMItem *)pMatch);
    cmListItemDispose((CMItem *)pMatch);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void readCallback(CCServer * pServer, Match * pContext)
{
#define READSTRUCT_SIZE 14
    ReadMatch * pMatch = (ReadMatch *)pContext;     /* casted pointer */
    NQ_BYTE buffer[64];                             /* buffer for structure and padding */
    Response * pResponse = pContext->response;      /* response structure pointer */
    NQ_UINT32 count = 0;                            /* bytes read */
    NQ_BYTE offset;                                 /* data offset */
    NQ_UINT32 currentTime;                      /* Current Time for checking timed-out responses*/
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p context:%p", pServer, pContext);

    IOBUF_POSCONSTRUCTOR(bufPos, buffer, sizeof(buffer))
    /* receive the structure but not the buffer (payload) */
    if (!IOBUF_ISNULL(pResponse->buffer))
    {
        if (pResponse->tailLen < READSTRUCT_SIZE )
        {
#ifdef UD_NQ_INCLUDESMBCAPTURE
            cmCapturePacketWritePacket( bufPos, pResponse->tailLen);
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            count = 0;
        }
        else
        {
            NQ_IOBufPos bufTmpPos;

            IOBUF_MEMCPY_V2V(bufPos , pResponse->buffer ,READSTRUCT_SIZE );
#ifdef UD_NQ_INCLUDESMBCAPTURE
            cmCapturePacketWritePacket( bufPos,READSTRUCT_SIZE );
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            cmBufferReaderInit(&pResponse->reader, bufPos, sizeof(buffer));
            cmBufferReadByte(&pResponse->reader, &offset);              /* data offset */
            cmBufferReaderSkip(&pResponse->reader, sizeof(NQ_BYTE));    /* reserved */
            cmBufferReadUint32(&pResponse->reader, &count); /* data length */
            bufTmpPos = pResponse->buffer;
            IOBUF_MOVEBYTES(bufTmpPos, READSTRUCT_SIZE + (offset - (SMB2_HEADERSIZE + 16)));
            IOBUF_MEMCPY_V2V(pMatch->buffer , bufTmpPos, count );
#ifdef UD_NQ_INCLUDESMBCAPTURE
            cmCapturePacketWritePacket( pMatch->buffer, (NQ_UINT)count);
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        }
        cmIOBufManGive(pResponse->buffer);
    }
    else if (pResponse->tailLen < READSTRUCT_SIZE)
    {
#ifdef UD_NQ_INCLUDESMBCAPTURE
        NQ_COUNT res =
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        ccTransportReceiveBytes(&pServer->transport, IOBUF_GETBUFPTR(bufPos), pResponse->tailLen );
#ifdef UD_NQ_INCLUDESMBCAPTURE
        if (res > 0)
        {
            cmCapturePacketWritePacket(bufPos,res );
        }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        count = 0;
    }
    else if (READSTRUCT_SIZE == ccTransportReceiveBytes(&pServer->transport, IOBUF_GETBUFPTR(bufPos), READSTRUCT_SIZE))
    {
#ifdef UD_NQ_INCLUDESMBCAPTURE
        cmCapturePacketWritePacket( bufPos,READSTRUCT_SIZE );
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        /* parse the response */
        cmBufferReaderInit(&pResponse->reader, bufPos, sizeof(buffer));
        if (SMB_STATUS_SUCCESS == pResponse->header.status)
        {
            cmBufferReadByte(&pResponse->reader, &offset);              /* data offset */
            cmBufferReaderSkip(&pResponse->reader, sizeof(NQ_BYTE));    /* reserved */
            cmBufferReadUint32(&pResponse->reader, &count); /* data length */
            offset = (NQ_BYTE)(offset - (SMB2_HEADERSIZE + 16));    /* bytes to skip */
            if (offset > 0 )
            {
                ccTransportReceiveBytes(&pServer->transport, IOBUF_GETBUFPTR(bufPos), (NQ_COUNT)offset);    /* read padding */
#ifdef UD_NQ_INCLUDESMBCAPTURE
                cmCapturePacketWritePacket(bufPos,offset );
#endif /* UD_NQ_INCLUDESMBCAPTURE */
            }
            ccTransportReceiveBytes(&pServer->transport, IOBUF_GETBUFPTR(pMatch->buffer), (NQ_COUNT)count); /* read into application buffer */
#ifdef UD_NQ_INCLUDESMBCAPTURE
            cmCapturePacketWritePacket(pMatch->buffer, (NQ_UINT)count);
#endif /* UD_NQ_INCLUDESMBCAPTURE */
        }
    }
    else
    {
        count = 0;
    }
    ccTransportReceiveEnd(&pServer->transport);
#ifdef UD_NQ_INCLUDESMBCAPTURE
    cmCapturePacketWriteEnd();
#endif /* UD_NQ_INCLUDESMBCAPTURE */

    currentTime = (NQ_UINT32)syGetTimeInSec();

    /* check if session was deleted, if yes mark the user as logged off */
    if ((SMB_STATUS_USER_SESSION_DELETED == pResponse->header.status) || (SMB_STATUS_NETWORK_SESSION_EXPIRED == pResponse->header.status))
    {
        CCUser * pUser;

        pUser = ccUserFindById(pServer, pResponse->header.sid);
        if (NULL != pUser)
        {
            cmListItemTake((CMItem*)pUser);
            pUser->logged = FALSE;
            cmListItemGive((CMItem*)pUser);
        }
    }

    /* call up */
    if ((pMatch->timeCreated + pMatch->setTimeout) > currentTime)
    {
        pMatch->callback(pResponse->header.status == SMB_STATUS_SUCCESS ? 0 : (NQ_STATUS)ccErrorsStatusToNq(pResponse->header.status, TRUE), (NQ_UINT)count, pMatch->context, count < pMatch->count);
    }
    else
    {
        /* response timed out */
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Read response arrived after timeout:%lu. Mid:%hu Expected:%lu arrived:%lu. Will not be handled.",
                (NQ_ULONG)pMatch->setTimeout, pMatch->match.mid, (NQ_ULONG)(pMatch->timeCreated + pMatch->setTimeout), (NQ_ULONG)currentTime);
    }

    /* release */
    if (NULL != pMatch->match.thread->element.item.guard)
    {
        syMutexDelete(pMatch->match.thread->element.item.guard);
        cmMemoryFree(pMatch->match.thread->element.item.guard);
        pMatch->match.thread->element.item.guard = NULL;
    }

    cmMemoryFree(pMatch->match.response);
    /* the mutex is taken at the beginning of the while that iterates over Matches list */
    cmListItemGive((CMItem *)pMatch);
    cmListItemDispose((CMItem *)pMatch);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void handleBreakNotification(CCServer * pServer, Response * notifyResponse, CCFile *pFile)
{
    Request request;            /* request descriptor */
    NQ_COUNT packetLen;         /* packet length of both in and out packets */
    NQ_BYTE oplockLevel;        /* new oplock level */
    CCUser * pUser;             /* user pointer */
    CCShare * pShare;           /* share pointer */
    Context * pContext;         /* SMB context */
    NQ_UINT64 negMid = {0xffffffff , 0xffffffff}; /* -1 mid */
    NQ_BYTE * encryptedBuf = NULL;
    NQ_IOBufPos bufTmpPos;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pServer:%p notifyResponse:%p pFile:%p", pServer, notifyResponse, pFile);

    if (SMB2_OPLOCK_LEVEL_NONE == pFile->grantedOplock)
    {
        /* no action is required */
        goto Exit;
    }

    /* parse notification */
    cmBufferReadByte(&notifyResponse->reader, &oplockLevel);    /* oplock */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "oplockLevel: %d", oplockLevel);

    if ((SMB2_OPLOCK_LEVEL_NONE == oplockLevel && pFile->grantedOplock == SMB2_OPLOCK_LEVEL_II) || cmU64Cmp(&notifyResponse->header.mid, &negMid) != 0)
    {
        /* don't have to reply - only update oplock */
        pFile->grantedOplock = oplockLevel;
        goto Exit;
    }

    pFile->grantedOplock = oplockLevel;
    if (pFile->durableState == DURABLE_GRANTED)
    {
        pFile->durableState = DURABLE_CANCELED;
        pFile->durableFlags = 0;
        cmGenerateUuid(&pFile->durableHandle); /* New Handle*/
    }

    pShare = pFile->share;
    pUser = pShare->user;

    /* compose ack (allocates request.buffer) */
    if (!ccSmb20PrepareSingleRequestByShare(&request, pShare, SMB2_CMD_OPLOCKBREAK, 0, 0))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "ccSmb20PrepareSingleRequestByShare() failed");
        goto Exit;
    }

    cmSmb2HeaderWrite(&request.header, &request.writer);
    cmBufferWriteUint16(&request.writer, commandDescriptors[request.command].requestStructSize);
    cmBufferWriteByte(&request.writer, oplockLevel);        /* oplock */
    cmBufferWriteZeroes(&request.writer, 5);                /* reserved + reserved 2 */
    cmBufferWriteBytes(&request.writer, pFile->fid, sizeof(pFile->fid));/* file id */
    packetLen = cmBufferWriterGetDataCount(&request.writer) - 4;    /* NBT header */

    /* write down MID */
    pContext = (Context *)pServer->smbContext;
    request.header.mid = pContext->mid;
    bufTmpPos = request.buffer;
    IOBUF_MOVEBYTES(bufTmpPos, SEQNUMBEROFFSET);
    cmBufferWriterSetPosition(&request.writer, bufTmpPos);
    cmBufferWriteUint64(&request.writer, &pContext->mid);

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Request: command=%u, credit charge=%d, credits req=%d, mid=%u/%u, sid.low=0x%x, signed:%d, async:%d, pid(async.high)=0x%x, tid(async.low)=0x%x",
        request.header.command, request.header.creditCharge, request.header.credits, request.header.mid.high, request.header.mid.low, request.header.sid.low, (request.header.flags & SMB2_FLAG_SIGNED) > 0,
        (request.header.flags & SMB2_FLAG_ASYNC_COMMAND) > 0,
        request.header.flags & SMB2_FLAG_ASYNC_COMMAND ? request.header.aid.high : request.header.pid,
        request.header.flags & SMB2_FLAG_ASYNC_COMMAND ? request.header.aid.low : request.header.tid);

    /* prepare MID for next request */
    cmU64Inc(&pContext->mid);

    /* compose signature */
    if (!request.encrypt && ccServerUseSignatures(pServer) && ccUserUseSignatures(pUser))
    {
        NQ_IOBufPos bufZeroPos;

        IOBUF_POSINIT(bufZeroPos)
        bufTmpPos = request.buffer;
        IOBUF_MOVEBYTES(bufTmpPos, 4);
        IOBUF_BUFFER_CANUSEFLAT_ASSERT(request.header._start, (SMB2_SECURITY_SIGNATURE_OFFSET + 16));
        cmSmb3CalculateMessageSignature(
            pUser->macSessionKey.data,
            pUser->macSessionKey.len,
            bufTmpPos,
            packetLen,
            bufZeroPos,
            0,
            IOBUF_GETBYTEPTR(request.header._start) + SMB2_SECURITY_SIGNATURE_OFFSET
            );
    }

#ifdef UD_NQ_INCLUDESMBCAPTURE
    {
        NQ_IOBufPos tempBuf;

        tempBuf = request.buffer;
        IOBUF_MOVEBYTES(tempBuf , 4);
        IOBUF_IOVECSET_START(tempBuf , NULL);

        pServer->captureHdr.receiving = FALSE;
        cmCapturePacketWriteStart(&pServer->captureHdr ,packetLen);
        cmCapturePacketWritePacket( tempBuf, packetLen);
        cmCapturePacketWriteEnd();
        IOBUF_IOVEC_RESTORE(tempBuf , NULL);
    }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    ccTransportLock(&pServer->transport);
    if (request.encrypt)
    {
        encryptedBuf = (NQ_BYTE *)cmMemoryAllocate(packetLen + SMB2_TRANSFORMHEADER_SIZE + 4);
        if (encryptedBuf != NULL)
        {
            CMSmb2TransformHeader   transformHeader;
            CMBufferWriter  writer;
            NQ_IOBufPos msgPoint;
            NQ_IOBufPos addPoint;
            NQ_IOBufPos addPoint1;
            NQ_IOBufPos bufEncPos;
            IOBUF_POSCONSTRUCTORINIT(bufEncPos)

            IOBUF_POSCONSTRUCTOR(bufEncPos, encryptedBuf, packetLen + SMB2_TRANSFORMHEADER_SIZE + 4)
            syMemset(&transformHeader , 0 , sizeof(transformHeader));
            transformHeader.encryptionArgorithm = SMB2_ENCRYPTION_AES128_CCM;
            transformHeader.originalMsgSize = packetLen;
            transformHeader.sid = pUser->uid;
            ccComposeEncryptionNonce(transformHeader.nonce , request.header.mid.low);
            cmBufferWriterInit(&writer , IOBUF_SKIPBYTE(bufEncPos, 4) , packetLen + SMB2_TRANSFORMHEADER_SIZE);
            IOBUF_SKIPBACKBYTE(bufEncPos, 4);
            addPoint1 = addPoint = cmBufferWriterGetPosition(&writer);
            IOBUF_MOVEBYTES(addPoint, 20);
            IOBUF_MOVEBYTES(addPoint1, 4);
            cmSmb2TransformHeaderWrite(&transformHeader , &writer);
            msgPoint = cmBufferWriterGetPosition(&writer);
            IOBUF_BUFFER_CANUSEFLAT_ASSERT(request.buffer, packetLen + 4);
            IOBUF_MOVEBYTES(request.buffer,4)
            cmBufferWriteBytes(&writer, IOBUF_GETBYTEPTR(request.buffer), packetLen);
            cmSmb3EncryptMessage(pUser->decryptionKey.data , transformHeader.nonce , msgPoint , packetLen, addPoint ,
                SMB2_TRANSFORMHEADER_SIZE - 20 , IOBUF_GETBYTEPTR(addPoint1), pUser->server->isAesGcm);
            IOBUF_MOVEBYTES(request.buffer,(NQ_INT)(-4))
            /* send - since we running inside the receiving thread - this is done inline */
            if (!ccTransportSendSync(
                            &pServer->transport,
                            bufEncPos,
                            packetLen + SMB2_TRANSFORMHEADER_SIZE,
                            packetLen + SMB2_TRANSFORMHEADER_SIZE
                            )
                        )
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "ccTransportSendSync() failed");
            }
        }
    }
    else
    {
        /* send - since we running inside the receiving thread - this is done inline */
        if (!ccTransportSendSync(
                &pServer->transport,
                request.buffer,
                packetLen,
                packetLen
                )
            )
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "ccTransportSendSync() failed");
        }
    }

    ccTransportUnlock(&pServer->transport);
    cmIOBufManGive(request.buffer);
    cmMemoryFree(encryptedBuf);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}

static NQ_BOOL validateNegotiate(void *pServ, void *_pUser, void* _pShareIPC)
{
    /* according to MS-SMB2 Validate Negotiate is not a MUST
     * will be re added in future releases only as optional functionality, disabled by default */
#ifdef CC_REQUIRESECURENEGOTIATE
    Request request;                /* request descriptor */
    Response response;              /* response descriptor */
    NQ_STATUS res = NQ_FAIL;        /* exchange result */
    NQ_IOBufPos pInputOffset;       /* pointer to the input offset field in the request */
    NQ_UINT32 offset, temp32Uint;   /* offset relative to the header */
    NQ_IOBufPos pTemp;              /* pointer in the buffer */
    NQ_UINT16 temp16Uint;
    CCShare   *pShare = (CCShare *)_pShareIPC;
    CCServer *pServer = (CCServer *)pServ;
    CCUser *pUser = (CCUser *)_pUser;
    const AMCredentials * pCredentials = pUser->credentials;
    NQ_UINT16 actualDialects = 0;
    NQ_COUNT i;
    NQ_BOOL result = FALSE;
    NQ_GUID serverGUID;
    NQ_UINT32 capabilities = 0;
    
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pServer: %p pUser: %p _pShareIPC:%p", pServer, pUser, _pShareIPC);

    if (!ccUserUseSignatures(pUser))
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "No validation when signing off.");
        result = TRUE;
        goto Exit;
    }

    IOBUF_POSINIT(request.buffer)
    IOBUF_POSINIT(response.buffer)

    if (NULL == _pShareIPC)
    {
        pShare = ccShareConnectIpc(pServer, &pCredentials);
        cmListItemUnlock((CMItem *)pUser); /* unlock the lock of ccShareConnectIpc */
        if (NULL == pShare)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to server's IPC");
            goto Exit;
        }
    }

    /* ccShareConnectIpc should have performed the negotiate validate, so recheck it here. */
    if (TRUE == pServer->isNegotiationValidated)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Already validated");
        result = TRUE;
        goto Exit;
    }

    if (!ccSmb20PrepareSingleRequestByShare(&request, pShare, SMB2_CMD_IOCTL, SMB_IOCTL_VALIDATE_NEGOTIATE, 0))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_OUTOFMEMORY;
        goto Error;
    }

    /* signature a MUST */
    request.header.flags |= SMB2_FLAG_SIGNED;

    /* compose request - start with header*/
    cmSmb2HeaderWrite(&request.header, &request.writer);
    cmBufferWriteUint16(&request.writer, 0x39);                         /* ioctl structure size*/

    /* ioctl header */
    cmBufferWriteUint16(&request.writer, 0x0);                          /* reserved */
    cmBufferWriteUint32(&request.writer, request.subCommand);           /* CtlCode: FSCTL_VALIDATE_NEGOTIATE_INFO */
    cmBufferWriteUint32(&request.writer, 0xFFFFFFFF);                   /* set special file ID */
    cmBufferWriteUint32(&request.writer, 0xFFFFFFFF);                   /* file ID */
    cmBufferWriteUint32(&request.writer, 0xFFFFFFFF);                   /* file ID */
    cmBufferWriteUint32(&request.writer, 0xFFFFFFFF);                   /* file ID */
    pInputOffset = cmBufferWriterGetPosition(&request.writer);
    cmBufferWriterSkip(&request.writer, sizeof(NQ_UINT32) * 3);         /* InputOffset/Count + MaxInputResponse */
    cmBufferWriteUint32(&request.writer, 0);                            /* Output offset */
    cmBufferWriteUint32(&request.writer, 0);                            /* Output count */
    cmBufferWriteUint32(&request.writer, 24);                           /* Max Output Response */
    cmBufferWriteUint32(&request.writer, SMB2_0_IOCTL_IS_FSCTL);        /* flags: FSCTL */
    cmBufferWriteUint32(&request.writer, 0);                            /* reserved */

    /* end of IOCTL header - start of IOCTL payload */
    offset = cmSmb2HeaderGetWriterOffset(&request.header, &request.writer);
#ifdef UD_NQ_INCLUDESMB3
    capabilities |= (SMB2_CAPABILITY_ENCRYPTION | SMB2_CAPABILITY_LARGE_MTU);
#endif /* UD_NQ_INCLUDESMB3 */
#ifdef UD_CC_INCLUDEDFS
    capabilities |= SMB2_CAPABILITY_DFS;
#endif /* UD_CC_INCLUDEDFS */
    cmBufferWriteUint32(&request.writer, capabilities);                 /* client capabilities */
    cmBufferWriteUint32(&request.writer, pServer->clientGuidPartial);   /* client GUID */
    cmBufferWriteUint32(&request.writer, 0);                            /* client GUID */
    cmBufferWriteUint32(&request.writer, 0);                            /* client GUID */
    cmBufferWriteUint32(&request.writer, 0);                            /* client GUID */
    cmBufferWriteUint16(&request.writer, ccGetSigningPolicy()? 1: 0);  /* security mode */
    for (i = 0; i < sizeof(pServer->clientDialectRevision) / sizeof(pServer->clientDialectRevision[0]); i++)
    {
        if (pServer->clientDialectRevision[i] != CCCIFS_ILLEGALSMBREVISION)
        {
            actualDialects++;
        }
    }
    cmBufferWriteUint16(&request.writer, actualDialects);               /* number of dialects */

    for (i = 0; i < sizeof(pServer->clientDialectRevision) / sizeof(pServer->clientDialectRevision[0]); i++)
    {
        if (pServer->clientDialectRevision[i] != CCCIFS_ILLEGALSMBREVISION)
        {
            cmBufferWriteUint16(&request.writer, pServer->clientDialectRevision[i]);    /* write actual dialects */
        }
    }

    /* set input fields in ioctl header  */
    pTemp = cmBufferWriterGetPosition(&request.writer);
    cmBufferWriterSetPosition(&request.writer, pInputOffset);
    cmBufferWriteUint32(&request.writer, offset);               /* Input Offset in bytes*/
    /* input count = 4 (capabilities) + 16 (client GUID) + 2 (security mode) + 2 (num dialects) + 2 * numDialects (dialects themselves) */
    cmBufferWriteUint32(&request.writer, (4 + 16 + 2 + 2 + (NQ_UINT32)(2 * actualDialects)));   /* InputCount */
    cmBufferWriteUint32(&request.writer, 0);                    /* MaxInputResponse */
    cmBufferWriterSetPosition(&request.writer, pTemp);

    /* make sure this packet will be signed */
    {
        NQ_BOOL useSigning;
        NQ_UINT32 capab;

        cmListItemTake((CMItem *)pServer);

        useSigning = pServer->useSigning;
        capab = pServer->capabilities;

        pServer->useSigning = TRUE;
        pServer->capabilities |= CC_CAP_MESSAGESIGNING;

        res = pServer->smb->sendReceive(pServer, pUser, &request, &response);

        pServer->useSigning = useSigning;
        pServer->capabilities = capab;

        cmListItemGive((CMItem *)pServer);
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR," sendReceive() failed");

        if ((SMB_STATUS_INVALID_PARAMETER == response.header.status) ||
            (SMB_STATUS_FILE_CLOSED == response.header.status) ||
            (SMB_STATUS_INVALID_DEVICE_REQUEST == response.header.status) ||
            (SMB_STATUS_NOT_SUPPORTED == response.header.status))
        {
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL," validation not supported on server - skipping");
            result = TRUE;
        }
        goto Error;
    }

    if (!request.encrypt && 0 == (response.header.flags & SMB2_FLAG_SIGNED))
    {
        /* validate negotiate packet must be signed */
        LOGERR(CM_TRC_LEVEL_ERROR," Validate negotiate response isn't signed. Validation failure.");
        goto Error;
    }

    /* parse response */
    cmBufferReaderSkip(&response.reader, sizeof(NQ_UINT16));    /* reserved */
    cmBufferReadUint32(&response.reader, &temp32Uint);          /* CtlCode */
    if (temp32Uint != SMB_IOCTL_VALIDATE_NEGOTIATE)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Bad CtlCode: %x in negotiate validation.", temp32Uint);
        goto Error;
    }

    for (i = 4; i > 0; --i)
    {
        cmBufferReadUint32(&response.reader, &temp32Uint);              /* file ID */
        if (temp32Uint != 0xFFFFFFFF)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Bad file ID value: %x in negotiate validation.", temp32Uint);
            goto Error;
        }
    }

    cmBufferReaderSkip(&response.reader, sizeof(NQ_UINT32));    /* InputOffset */
    cmBufferReadUint32(&response.reader, &temp32Uint);          /* InputCount */
    if (temp32Uint != 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Bad input count: %u in negotiate validation.", temp32Uint);
        goto Error;
    }
    cmBufferReadUint32(&response.reader, &offset);              /* OutputOffset */
    cmBufferReaderSkip(&response.reader, sizeof(NQ_UINT32));    /* OutoutCount */
    cmBufferReaderSkip(&response.reader, sizeof(NQ_UINT32));    /* flags - currently no relevant response flags */
    cmBufferReaderSkip(&response.reader, sizeof(NQ_UINT32));    /* reserved */

    /* end of IOCTL header and start of IOCTL payload */
    cmSmb2HeaderSetReaderOffset(&response.header, &response.reader, (NQ_UINT16)offset);
    cmBufferReadUint32(&response.reader, &temp32Uint);              /* server capabilities*/
    if (temp32Uint != pServer->serverCapabilites)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Capabilities mismatch, in negotiate validation.");
        goto Error;
    }
    cmBufferReadUuid(&response.reader, &serverGUID);
    if (!cmUuidIsEqual(&serverGUID, &pServer->serverGUID))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Received server GUID doesn't match, in negotiate validation.");
        goto Error;
    }
    cmBufferReadUint16(&response.reader, &temp16Uint);                      /* security mode */
    if (temp16Uint != pServer->serverSecurityMode)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Security mode mismatch, in negotiate validation.");
        goto Error;
    }
    cmBufferReadUint16(&response.reader, &temp16Uint);                      /* server selected Dialect Revision*/
    if (temp16Uint != pServer->serverDialectRevision)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Dialect revision mismatch, in negotiate validation.");
        goto Error;
    }
    result = TRUE;

Error:
    cmIOBufManGive(request.buffer);
    cmIOBufManGive(response.buffer);
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result: %d", result);
    return (result);
#else /* not defined CC_REQUIRESECURENEGOTIATE */

    return TRUE;

#endif /* CC_REQUIRESECURENEGOTIATE */
}

#endif /* UD_NQ_INCLUDESMB3 */
#endif /* UD_NQ_INCLUDECIFSCLIENT */
