/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client start and shut down operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccapi.h"
#include "cccifs.h"
#include "ccsmb311.h"
#include "ccsmb30.h"
#include "ccsmb20.h"
#include "ccsmb10.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

/* -- Static data -- */
static NQ_BOOL isModuleInitialized = FALSE;

typedef struct{
    const CCCifsSmb *dialect;
    NQ_BOOL isActive;
}CCCifsSmbDialect;

typedef struct{
    CCCifsSmbDialect dialects[CCCIFS_SMB_NUM_DIALECTS];
    NQ_INT           numberOfActiveDialects;
    SYMutex          dialectsGuard;
}CCCifsDialects;

static CCCifsDialects cifsDialects;

static void setDefaultCifsDialects(CCCifsDialects *cifsDialects)
{
    NQ_INT dialectCount = 0;

    cifsDialects->numberOfActiveDialects = 0;
#ifdef UD_NQ_INCLUDESMB1
    /* add NT LM 0.12 (smb1) */
    cifsDialects->dialects[dialectCount].dialect = ccSmb10GetCifs();
    cifsDialects->dialects[dialectCount].isActive = FALSE;
    if (udDefGetClientSMB1Support())
    {
        cifsDialects->dialects[dialectCount++].isActive = TRUE;
        cifsDialects->numberOfActiveDialects++;
    }
#endif /* UD_NQ_INCLUDESMB1 */
#ifdef UD_NQ_INCLUDESMB2
    /* add dialect 202 */
    cifsDialects->dialects[dialectCount].dialect = ccSmb20GetCifs202();
    cifsDialects->dialects[dialectCount++].isActive = TRUE;
    cifsDialects->numberOfActiveDialects++;
    /* add dialect 210 */
    cifsDialects->dialects[dialectCount].dialect = ccSmb20GetCifs210();
    cifsDialects->dialects[dialectCount++].isActive = TRUE;
    cifsDialects->numberOfActiveDialects++;

#ifdef UD_NQ_INCLUDESMB3
    /* add dialect 300 */
    cifsDialects->dialects[dialectCount].dialect = ccSmb30GetCifs300();
    cifsDialects->dialects[dialectCount++].isActive = TRUE;
    cifsDialects->numberOfActiveDialects++;
    /* add dialect 302 */
    cifsDialects->dialects[dialectCount].dialect = ccSmb30GetCifs302();
    cifsDialects->dialects[dialectCount++].isActive = TRUE;
    cifsDialects->numberOfActiveDialects++;

#ifdef UD_NQ_INCLUDESMB311
    /* add dialect 311 */
    cifsDialects->dialects[dialectCount].dialect = ccSmb311GetCifs();
    cifsDialects->dialects[dialectCount++].isActive = TRUE;
    cifsDialects->numberOfActiveDialects++;
#endif /* UD_NQ_INCLUDESMB311 */
#endif /* UD_NQ_INCLUDESMB3 */
#endif /* UD_NQ_INCLUDESMB2 */
#if defined(UD_NQ_INCLUDESMB3) || defined(CCCIFS_SMB_ANY_SMB2)
    {
        static CCCifsSmb anySmb2Dialect;
        static const NQ_CHAR * anySmb2String = "SMB 2.???";

        anySmb2Dialect = *ccSmb20GetCifs202(); /* copy SMB2.0 dialect */
        anySmb2Dialect.name = anySmb2String;
        anySmb2Dialect.revision = CCCIFS_ILLEGALSMBREVISION;
        cifsDialects->dialects[dialectCount].dialect = &anySmb2Dialect;
        cifsDialects->dialects[dialectCount++].isActive = TRUE;
    }
#endif /* defined(UD_NQ_INCLUDESMB3) || defined(CCCIFS_SMB_ANY_SMB2) */
}


/* -- API Functions */

NQ_BOOL ccCifsStart(void)
{
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        goto Exit;
    }

    setDefaultCifsDialects(&cifsDialects);
    if (0 == cifsDialects.numberOfActiveDialects)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "All dialects are disabled");
        result = FALSE;
        goto Exit;
    }

    syMutexCreate(&cifsDialects.dialectsGuard);

    isModuleInitialized = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

void ccCifsShutdown(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        syMutexDelete(&cifsDialects.dialectsGuard);
        isModuleInitialized = FALSE;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

const CCCifsSmb * ccCifsGetDefaultSmb(void)
{
    NQ_INT i;
    const CCCifsSmb * dialect;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    syMutexTake(&cifsDialects.dialectsGuard);
    for (i = 0; i < CCCIFS_SMB_NUM_DIALECTS; i++)
    {
        if (TRUE == cifsDialects.dialects[i].isActive)
        {
            dialect = cifsDialects.dialects[i].dialect;
            goto Exit;
        }
    }

    LOGERR(CM_TRC_LEVEL_ERROR, "No dialect was found");
    dialect = NULL;

Exit:
    syMutexGive(&cifsDialects.dialectsGuard);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "dialect:%p", dialect);
    return dialect;
}

NQ_INT ccCifsGetDialects(const CCCifsSmb *** dialects)
{
    NQ_BOOL isSMB1Only = TRUE;
    NQ_INT dialectsNum = 0;
    NQ_COUNT i;
    const CCCifsSmb **dialectsActive = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "dialects:%p", dialects);

    dialectsActive = (const CCCifsSmb **)cmMemoryAllocate(CCCIFS_SMB_NUM_DIALECTS * sizeof(cifsDialects.dialects[0].dialect));
    if (NULL == dialectsActive)
    {
        *dialects = NULL;
        goto Exit;
    }

    syMutexTake(&cifsDialects.dialectsGuard);
    for (i = 0; i < CCCIFS_SMB_NUM_DIALECTS; i++)
    {
        if ((NULL != cifsDialects.dialects[i].dialect) && (TRUE == cifsDialects.dialects[i].isActive))
        {
            dialectsActive[dialectsNum++] = cifsDialects.dialects[i].dialect;
            if ((cifsDialects.dialects[i].dialect->revision >= SMB2_DIALECTREVISION) && (cifsDialects.dialects[i].dialect->revision != CCCIFS_ILLEGALSMBREVISION))
            {
                isSMB1Only = FALSE;
            }
        }
    }
    dialectsNum = isSMB1Only ? 1 : dialectsNum;
    *dialects = dialectsActive;
    syMutexGive(&cifsDialects.dialectsGuard);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "dialectsNum:%d", dialectsNum);
    return dialectsNum;
}

NQ_BOOL ccSetSmbDialect(CCSmbDialect dialect, NQ_BOOL setActive)
{
    NQ_BOOL result = FALSE;
    NQ_INT i;
    NQ_UINT16 revision;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "dialect:%d setActive:%d", dialect, setActive);

    switch (dialect)
    {
    case CC_SMB100:
        revision = CCCIFS_ILLEGALSMBREVISION;
        break;
    case CC_SMB202:
        revision = SMB2_DIALECTREVISION;
        break;
    case CC_SMB210:
        revision = SMB2_1_DIALECTREVISION;
        break;
    case CC_SMB300:
        revision = SMB3_DIALECTREVISION;
        break;
    case CC_SMB302:
        revision = SMB3_0_2_DIALECTREVISION;
        break;
    case CC_SMB311:
        revision = SMB3_1_1_DIALECTREVISION;
        break;
    default:
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid dialect:0x%x", dialect);
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "dialect:0x%x", revision);

    syMutexTake(&cifsDialects.dialectsGuard);

    for (i = 0; i < CCCIFS_SMB_NUM_DIALECTS; i++)
    {
        if ((NULL != cifsDialects.dialects[i].dialect) && (cifsDialects.dialects[i].dialect->revision == revision) && (0 != syStrncmp(cifsDialects.dialects[i].dialect->name, "SMB 2.???", syStrlen("SMB 2.???"))))
        {
            if ((FALSE == setActive) && (TRUE == cifsDialects.dialects[i].isActive))
            {
                if (1 == cifsDialects.numberOfActiveDialects)
                {
                    result = FALSE;
                    LOGERR(CM_TRC_LEVEL_ERROR, "Can not disable the last dialect");
                    break;
                }
                cifsDialects.numberOfActiveDialects--;
            }

            if ((TRUE == setActive) && (FALSE == cifsDialects.dialects[i].isActive))
            {
                cifsDialects.numberOfActiveDialects++;
            }

            cifsDialects.dialects[i].isActive = setActive;
            result = TRUE;
            break;
        }
    }

    syMutexGive(&cifsDialects.dialectsGuard);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}


/*
 *====================================================================
 * PURPOSE: whether SMB1 dialect is supported
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: TRUE
 *          FALSE
 *
 * NOTES:   Assuming SMB1 is first in the dialects array
 *====================================================================
 */
NQ_BOOL
ccCifsGetSMB1Support(void)
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

#ifdef UD_NQ_INCLUDESMB1
    syMutexTake(&cifsDialects.dialectsGuard);

    if ((NULL != cifsDialects.dialects[0].dialect) && (0 == syStrncmp(cifsDialects.dialects[0].dialect->name, "NT LM 0.12", syStrlen("NT LM 0.12"))))
    {
        result = cifsDialects.dialects[0].isActive;
    }

    syMutexGive(&cifsDialects.dialectsGuard);
#endif /* UD_NQ_INCLUDESMB1 */

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

#ifdef UD_NQ_INCLUDESMB1
/*
 *====================================================================
 * PURPOSE: Activates or deactivates of SMB1 dialect.
 *--------------------------------------------------------------------
 * PARAMS:  isSupport - TRUE to enable SMB1 dialect
 *                    FLASE to disable SMB1 dialect
 *
 * RETURNS:
 *          None
 *
 * NOTES:   Assuming SMB1 is first in the dialects array
 *====================================================================
 */
void
ccCifsSetSMB1Support(NQ_BOOL isSupport)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "isSupport:%d", isSupport);

    syMutexTake(&cifsDialects.dialectsGuard);

    if ((NULL != cifsDialects.dialects[0].dialect) && (0 == syStrncmp(cifsDialects.dialects[0].dialect->name, "NT LM 0.12", syStrlen("NT LM 0.12"))))
    {
        if ((FALSE == isSupport) && (TRUE == cifsDialects.dialects[0].isActive))
        {
            if (1 == cifsDialects.numberOfActiveDialects)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Can not disable the last dialect");
                goto Exit;
            }
            cifsDialects.numberOfActiveDialects--;
        }

        if ((TRUE == isSupport) && (FALSE == cifsDialects.dialects[0].isActive))
        {
            cifsDialects.numberOfActiveDialects++;
        }

        cifsDialects.dialects[0].isActive = isSupport;
    }

Exit:
    syMutexGive(&cifsDialects.dialectsGuard);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}
#endif /* UD_NQ_INCLUDESMB1 */


#endif /* UD_NQ_INCLUDECIFSCLIENT */

