/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Configuration management
 *--------------------------------------------------------------------
 * MODULE        : Linux - UD
 * DEPENDENCIES  :
 ********************************************************************/

#include "syinclud.h"
#include "udparams.h"
#include "sycompil.h"
#include "udapi.h"
#include "udconfig.h"
#include "udparser.h"
#include "cmapi.h"
#include "cmcrypt.h"
#include "aminaesgcm.h"

/*
  This file contains a sample implementation of UD configuration functions.

  Configuration parameters reside in several files. Each file has a format described in
  "udparser.c"

  Functions in this file are not reentrant. This is appropriate since for each of them
  there is exactly one caller.

  The following configuration files are used:
    1) COMMON configuration file defines network parameters common for both server
       and client.

    2) CIFS Server configuration file defines shared directories on target (shares)

    3) CIFS Client configuration file defines mounted remote file systems that CIFS
       client provides access to

    4) User (password) list for CIFS Client

 */

 /*
    NQ Parameters
    ----------------
  */

/* default values for NQ parameters */

#define NQ_TASKPRIORITY         1
#define NQ_DOMAIN               "WORKGROUP"
#define NQ_ISWORKGROUP          TRUE
#define NQ_SERVERCOMMENT        "YNQ SMB Server"   /* server description in host announcement */
#define NQ_CCDRVNAME            "/net"
#define NQ_CCUSERNAME           "user"
#define NQ_CCPASSWORD           "password"

#define NQ_NOT_INITIALIZED         (-1)           /* indicate that a parameter is not initialized yet */
#define NQ_NOT_PRESENT             (-2)           /* indicate that a parameter is not present in the configuration file */
#define NQ_CONFIG_FILE_NAME_LEN    100
#define NQ_CONFIG_FILE_VALUE_LEN   350

#ifndef NQ_DNSDOMAIN
  #define NQ_DNSDOMAIN NQ_DOMAIN
#endif
#ifndef NQ_DNSADDRESS         /* max number of DNS servers should not exceed UD_NQ_MAXDNSSERVERS */
/* Notice: when defining more than one address, format is: "x.x.x.x;y.y.y.y", avoid spaces */
  #define NQ_DNSADDRESS       "192.168.19.2"

#endif

#define SECRET_LENGTH   16   /* domain computer account secret length */

#define NQ_HASHED_PASSWORD_LENGTH   16 /* length of hashed passwords both for MD4 and DES */

/*#define UDCONFIG_DEBUG*/

#define NQ_ENCRYPTED_POSTFIX_LEN        2
#define NQ_PWD_SALT_LEN                 5
#define NQ_NUM_OF_SALTS                 3
#define NQ_MAX_ENCRYPTED_PASS_SIZE      (NQ_PWD_SALT_LEN * NQ_NUM_OF_SALTS + CM_USERNAMELENGTH + UD_NQ_MAXPWDLEN)

/* size of ace-gcm key is 16 bytes */
static unsigned char key[16] =   {0x12, 0xab, 0x8b, 0xc1, 0x9d, 0x85, 0x2f, 0xf1, 0xaf, 0x5b, 0xcb, 0xc5, 0xd2, 0xae, 0xb7, 0xec};
/* size of ace-gcm nonce is 12 bytes */
static unsigned char nonce[12] = {0x35, 0xea, 0x68, 0x51, 0x7b, 0xbe, 0xfe, 0x92, 0xaa, 0x8b, 0xcb, 0xc4};


typedef struct
{
/* buffers in the user space. The default implementation uses static buffers. */
    unsigned char staticBuffers[5*UD_NS_BUFFERSIZE];
/* data for access to the configuration files. Some of them are read only once. */
    int fileNamesReady;      /* will be set to 1 when full passes will set for configuration files (only once) */
    int commonConfigFlag;    /* will be set to 1 after the NB configuration file will be read (only once) */
    int readingShares;       /* will be set to 1 while reading shares */
    int defaultShareReported;/* if the default share was already reported */
    int hiddenShareReported; /* if the hidden share was already reported */
    int readingMounts;       /* will be set to 1 while reading mounts */
    const char* scopeIdPtr;  /* will be set after the scope id will be read from the NB configuration (only once) */
    int isSecretFileLoaded;  /* will be set to 1 after loading secret from file */
    int isSecretAvailable;   /* will be set to 1 if secret is available */
    char secret[16];         /* buffer for domain computer account secret */
    char scopeId[100];       /* buffer for scope id */
    ParseContext mountParser;/* parser for reading the mount table */
    ParseContext shareParser;/* parser for reading the share table */
/* storage for configuration parameters */
    char cmConfigFile[250];  /* full path to common configuration file name */
    char csConfigFile[250];  /* full path to server configuration file name */
    char ccConfigFile[250];  /* full path to client configuration file name */
    char passwordFile[250];  /* full path to password file name */
    char shareSdFile[250];   /* full path to file for share SD */
    char tempFileName[250];  /* full path to a temporary file */
    char secretFilePath[250];/* full path to a secret file */
    char secretFile[250];    /* full path to a secret file */
    char secretFileName[250];/* secret file name (not path), according to domain, format: domainDNS.secret.domainNB */
#ifdef UD_NQ_INCLUDEEVENTLOG
    char eventLogFile[250];       /* full path to file for event log */
    int eventLogBaseFolderTooLong; /* indicate that the path of event log base folder is too long */
#endif /* UD_NQ_INCLUDEEVENTLOG */
    /* char userName[CM_USERNAMELENGTH]; */
    /* char password[65]; */
    char domainNameOfClient[CM_NQ_HOSTNAMESIZE];
    int  isWorkgroupName;                           /* TRUE if the domain name is a workgroup */
    char winsAddresses[UD_NQ_MAXWINSSERVERS * UD_NS_MAXADAPTERS * CM_IP4ADDR_MAXLEN]; /* WINS addresses string (; separated) */
    FILE* logHandle;                                /* event log file */
    int logHandleFlag;                              /* open flag for event log file */
    char userName[CM_USERNAMELENGTH];               /* buffer for user's name - set in udSetCredentials*/
    char password[UD_NQ_MAXPWDLEN];                 /* buffer for user's password - set in udSetCredentials*/
    char domainNameOfServer[CM_NQ_HOSTNAMESIZE];    /* buffer for domain name with which the client connects to the server */
    int isGlobalEncryptionServer;                         /* global encryption values: -1 not initialized, 0 - off, 1 - global (effective for SMB3) */
    int signingPolicyServer;                        /* signing policy values: -1 not initialized, 0 - disabled, 1 - enabled, 2 - required */
    char dnsAddresses[UD_NQ_MAXDNSSERVERS * UD_NS_MAXADAPTERS * CM_IPADDR_MAXLEN];  /* DNS addresses string (; separated) */
#ifdef UD_NQ_INCLUDESMBCAPTURE
    int useInternalCapture;                         /* whether to use internal capture: -1 not initialized, 0 - do not use, 1 - use */
    char captureFileBaseFolder[UD_FS_FILENAMELEN];  /* internal capture file base folder path */
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    int useInternalTrace;                           /* whether to use internal trace: -1 not initialized, 0 - do not use, 1 - use */
    char hostname[CM_NB_NAMELEN];                   /* NetBIOS hostname - 15 symbols + NULL termination */
    char logFileBaseFolder[UD_FS_FILENAMELEN];      /* log file base folder path */
#ifdef UD_NQ_INCLUDECODEPAGE
    NQ_INT currentCodePage;                         /* the current code page in use */
#endif /* UD_NQ_INCLUDECODEPAGE */
#ifdef UD_NQ_INCLUDESMB1
    int supportSMB1ForServer;                       /* whether to support SMB1 dialect for server: -1 not initialized, 0 - do not support, 1 - support */
#endif /* UD_NQ_INCLUDESMB1 */
    int isAnonymousAccessAllowedServer;             /* anonymous access allowed: -1 not initialized, 0 - no, 1 - yes */
    int maxDiskSizeServer;                          /* max disk size for shares of YNQ Server */
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/* Format of Share SD file:
 *      fix size records of
 *          - 256 bytes - share name padded by zeroes
 *          - 2 bytes ";;"
 *          - 600 bytes of SD padded by zeroes (2 bytes of ascii per 1byte of binary)
 *          - '\n\r'
 *      total: 860 bytes per record
 */

#define SD_SHARENAMELEN 256
#define SD_SDLEN 300
#define SD_RECLEN (SD_SHARENAMELEN + 2 + SD_SDLEN * 2 + 2)

/* parsing configurations and storing parameters */

static NQ_BOOL
parseCommonConfig(
    void
    );

static void
setFileNames(
    void
    );

static NQ_BOOL
convertHex2Ascii(
    char* text,
    NQ_INT outputStrLen
    );

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
static void
convertPasswordToText(
    const NQ_BYTE* password,
    NQ_CHAR* buffer
    );
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

#ifdef UD_NQ_INCLUDEEVENTLOG
static void
saveEvent(
    char* format,
    ...
    );
#endif /* UD_NQ_INCLUDEEVENTLOG */



/*====================================================================
 * PURPOSE: set new client credentials
 *--------------------------------------------------------------------
 * PARAMS:  IN user name for client login
 *          IN password for client login
 *          IN domain name
 *
 * RETURNS: NONE
 *
 * NOTES:
 * Regarding domainName and domainNameOfClient:
 * There are 3 possible cases:
 * 1. domainName was never changed -> the domainNameOfClient is NQ_DOMAIN
 * 2. domainName was changed, and domain was not received from the
 *    command line -> domainNameOfClient and domainNameOfServer are the same.
 * 3. domain was received from command line -> domainNameOfClient and
 *    domainNameOfServer could be different.
 *========================================================================
 */
void
udSetCredentials(
    const char* user,
    const char* pwd,
    const char* domain
    )
{
    char *userNameDelimiter = NULL;

    if (user == NULL && pwd == NULL && domain == NULL)
    {
        syPrintf ("/=========================================================================\n");
        syPrintf ("In order to use this function properly, the format should be as follows:\n\n");
        syPrintf ("udSetCredentials \"username\",\"password\", \"domain\"\n\n");
        syPrintf ("It is possible to omit the fields, then, the default hardcoded fields\nwill be used\n");
        syPrintf ("=========================================================================/\n\n/");
        goto Exit;
    }

    /* First one should check if the entered pointers are not NULL */
    if (user == NULL)
    {
        syPrintf ("You have entered wrong user name. Please try again...\n");
        goto Exit;
    }
    if (pwd == NULL)
    {
        syPrintf ("You have entered wrong password. Please try again...\n");
        goto Exit;
    }

    userNameDelimiter = syStrchr(user, '\\');

    if (domain != NULL)
    {
        if (syStrlen(domain) > 0 && userNameDelimiter != NULL)
        {
            syPrintf ("Ambiguity in user name and domain name. Please try again...\n");
            goto Exit;
        }
        /* If we here the entered string for 'domain' is not NULL */
        syStrncpy(staticData->domainNameOfClient, domain, sizeof(staticData->domainNameOfClient));
        staticData->domainNameOfClient[sizeof(staticData->domainNameOfClient) - 1] = '\0';
        /* capitalize domain name */
        {
            char *s = staticData->domainNameOfClient;

            while ((*s = syToupper(*s)) != 0)
            {
                s++;
            }
        }
    }
    else
    {
        if (userNameDelimiter != NULL)
        {
            syStrncpy(staticData->domainNameOfClient, user, (size_t)(userNameDelimiter - user));
            staticData->domainNameOfClient[userNameDelimiter - user] = 0;
        }
        else
        {
            syPrintf ("Using predefined value (the domain of the client).\n");
            /* The default value of domainNameOfServer is the value in domainName */
            syStrncpy(staticData->domainNameOfClient, staticData->domainNameOfServer, sizeof(staticData->domainNameOfClient));
            staticData->domainNameOfClient[sizeof(staticData->domainNameOfClient) - 1] = '\0';
        }
    }

    /* If we here the entered strings for 'use' and 'pwd' are not NULL */
    if (userNameDelimiter != NULL)
    {
        syStrncpy(staticData->userName, userNameDelimiter + 1, syStrlen(userNameDelimiter + 1));
        staticData->userName[syStrlen(userNameDelimiter + 1)] = 0;
    }
    else
    {
        syStrncpy(staticData->userName, user, sizeof(staticData->userName));
        staticData->userName[sizeof(staticData->userName) - 1] = '\0';
    }
    syStrncpy(staticData->password, pwd, sizeof(staticData->password));
    staticData->password[sizeof(staticData->password) - 1] = '\0';

#ifdef NQDEBUG
    syPrintf ("udSetCredentials: the parameters that were set are:\n");
    syPrintf ("                  User name is: %s\n",staticData->userName);
/*  syPrintf ("                  Password is: %s\n",staticData->password );*/
    syPrintf ("                  Domain name of the client: %s\n",staticData->domainNameOfClient);
    syPrintf ("                  Domain name of the server is: %s\n",staticData->domainNameOfServer);
#endif /* NQDEBUG */
Exit:
    return;
}

/*====================================================================
 * PURPOSE: set client's credentials (UNICODE version)
 *--------------------------------------------------------------------
 * PARAMS:  user - user name
 *          pwd  - password
 *          domain - domain name
 *
 * RETURNS: None
 *
 *====================================================================
 */
void udSetCredentialsW (
        const NQ_WCHAR* user,
        const NQ_WCHAR* pwd,
        const NQ_WCHAR* domain
        )
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user name:%s", cmWDump(user));
    LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "domain name:%s", cmWDump(domain));

    if ( user == NULL && pwd == NULL && domain == NULL )
    {
        /* Printing brief help about the function */
        syPrintf ("/=========================================================================\n");
        syPrintf ("In order to use this function properly, the format should be as follows:\n\n");
        syPrintf ("udSetCredentials \"username\",\"password\", \"domain\"\n\n");
        syPrintf ("It is possible to omit the fields, then, the default hardcoded fields\nwill be used\n");
        syPrintf ("=========================================================================/\n\n/");
        goto Exit;
    }

    /* First one should check if the enterd pointers are not NULL */
    if ( user == NULL )
    {
        syPrintf ("You have entered wrong user name. Please try again...\n");
        goto Exit;
    }

    if ( pwd == NULL )
    {
        syPrintf ("You have entered wrong password. Please try again...\n");
        goto Exit;
    }

    if ( domain == NULL )
    {
        syPrintf ("Using predefined value (the domain of the client).\n");
        /* The default value of domainNameOfServer is the value in domainName */
        syStrncpy(staticData->domainNameOfClient,staticData->domainNameOfServer, sizeof(staticData->domainNameOfClient));
        staticData->domainNameOfClient[sizeof(staticData->domainNameOfClient) - 1] = '\0';
    }
    else
    {
        /* If we here the entered string for 'domain' is not NULL */
        cmUnicodeToAnsiN(staticData->domainNameOfClient, sizeof(staticData->domainNameOfClient), domain,  CM_IGNORE_LENGTH);
        /* capitalize domain name */
        {
            char *s = staticData->domainNameOfClient;

            while ((*s = syToupper(*s)) != 0)
            {
                s++;
            }
        }
    }

    /* If we here the entered strings for 'use' and 'pwd' are not NULL */
    cmUnicodeToAnsiN(staticData->userName, sizeof(staticData->userName), user, CM_IGNORE_LENGTH);
    cmUnicodeToAnsiN(staticData->password, sizeof(staticData->password), pwd, CM_IGNORE_LENGTH);

#ifdef NQDEBUG
    syPrintf ("udSetCredentials: the parameters that were set are:\n");
    syPrintf ("                  User name is: %s\n",staticData->userName );
/*  syPrintf ("                  Password is: %s\n",staticData->password );*/
    syPrintf ("                  Domain name of the client: %s\n",staticData->domainNameOfClient );
    syPrintf ("                  Domain name of the server is: %s\n",staticData->domainNameOfServer  );
#endif /* NQDEBUG */

    Exit:
        LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
        return;
}

/*====================================================================
 * PURPOSE: initialize the configuration
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NQ_SUCCESS on success
 *          NQ_FAIL on error
 *
 * NOTES:   Inits this module
 *====================================================================
 */

NQ_STATUS
udDefInit(
    void
    )
{
    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)syCalloc(1, sizeof(StaticData));
    if (NULL == staticData)
    {
        TRCERR("Unable to allocate UD config data");
        TRCE();
        return NQ_FAIL;
    }
#endif /* SY_FORCEALLOCATION */

    staticData->fileNamesReady = 0;
    staticData->readingShares = 0;
    staticData->readingMounts = 0;
    staticData->scopeIdPtr = NULL;
    strcpy(staticData->userName, NQ_CCUSERNAME);
    strcpy(staticData->password, NQ_CCPASSWORD);
    strcpy(staticData->domainNameOfClient, NQ_DOMAIN);
    strcpy(staticData->domainNameOfServer, NQ_DOMAIN);
    staticData->isWorkgroupName = NQ_ISWORKGROUP;
    staticData->winsAddresses[0] = '\0';
    staticData->defaultShareReported = 0;
    staticData->commonConfigFlag = 0;
    staticData->logHandleFlag = 0;
    staticData->isSecretFileLoaded = 0;
    staticData->isSecretAvailable = 0;
    staticData->isGlobalEncryptionServer = NQ_NOT_INITIALIZED;
    staticData->isAnonymousAccessAllowedServer = NQ_NOT_INITIALIZED;
    staticData->signingPolicyServer = NQ_NOT_INITIALIZED;
    staticData->dnsAddresses[0] = '\0';
    staticData->useInternalTrace = NQ_NOT_INITIALIZED;
    staticData->hostname[0] = 0;
    syGetHostName(staticData->hostname, sizeof(staticData->hostname) - 1);
    if (staticData->hostname[0] != 0)
    {
        staticData->hostname[sizeof(staticData->hostname) - 1] = 0;
    }

#ifdef UD_CM_LOG_BASEFOLDER
    syStrncpy(staticData->logFileBaseFolder, UD_CM_LOG_BASEFOLDER, sizeof(staticData->logFileBaseFolder) - CM_TRAILING_NULL);
    staticData->logFileBaseFolder[sizeof(staticData->logFileBaseFolder) - 1] = '\0';
#else /* UD_CM_LOG_BASEFOLDER */
    staticData->logFileBaseFolder[0] = '\0';
#endif /* UD_CM_LOG_BASEFOLDER */

#ifdef UD_NQ_INCLUDESMBCAPTURE
    staticData->useInternalCapture = NQ_NOT_INITIALIZED;
#ifdef UD_CM_CAPTURE_BASEFOLDER
    syStrncpy(staticData->captureFileBaseFolder, UD_CM_CAPTURE_BASEFOLDER, sizeof(staticData->captureFileBaseFolder) - CM_TRAILING_NULL);
    staticData->captureFileBaseFolder[sizeof(staticData->captureFileBaseFolder) - 1] = '\0';
#else /* UD_CM_CAPTURE_BASEFOLDER */
    staticData->captureFileBaseFolder[0] = '\0';
#endif /* UD_CM_CAPTURE_BASEFOLDER */
#endif /* UD_NQ_INCLUDESMBCAPTURE */

#if defined(UD_NQ_INCLUDECODEPAGE) && defined(UD_NQ_CODEPAGE437)
    staticData->currentCodePage = UD_NQ_CODEPAGE437;
#endif
#ifdef UD_NQ_INCLUDESMB1
    staticData->supportSMB1ForServer = NQ_NOT_INITIALIZED;
#endif /* UD_NQ_INCLUDESMB1 */
    staticData->maxDiskSizeServer = NQ_NOT_INITIALIZED;

#ifdef UD_PRINT_STARTUP_INFO
    {
        int fd;

        setFileNames();
        if (ERROR == (fd = open(staticData->passwordFile, O_RDONLY, 0777)))
        {
            syPrintf("No USER database\n");
        }
        else
        {
            close(fd);
            syPrintf("With USER database\n");
        }
    }
#endif /* UD_PRINT_STARTUP_INFO */

    return NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: stop the CIFS configuration
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udDefStop(
    void
    )

{
    /* release memory */
#ifdef SY_FORCEALLOCATION
    if (NULL != staticData)
    {
        syFree(staticData);
    }

    staticData = NULL;
#endif /* SY_FORCEALLOCATION */
}

/*
 *====================================================================
 * PURPOSE: reset server parameters
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: None
 *
 * NOTES: This function should be called during server restart
 *====================================================================
 */
void
udDefResetServerParams(
    void
    )
{
    staticData->signingPolicyServer = NQ_NOT_INITIALIZED;
    staticData->isGlobalEncryptionServer = NQ_NOT_INITIALIZED;
#ifdef UD_NQ_INCLUDESMB1
    staticData->supportSMB1ForServer = NQ_NOT_INITIALIZED;
#endif /* UD_NQ_INCLUDESMB1 */
    staticData->isAnonymousAccessAllowedServer = NQ_NOT_INITIALIZED;
    staticData->maxDiskSizeServer = NQ_NOT_INITIALIZED;
}

/*
 *====================================================================
 * PURPOSE: get scope id
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udDefGetScopeID(
    NQ_WCHAR* buffer
    )
{
    if (!staticData->commonConfigFlag)
        parseCommonConfig();
    if (staticData->scopeIdPtr != 0)
    {
        syAnsiToUnicode(buffer, staticData->scopeIdPtr);
    }
    else
    {
        syAnsiToUnicode(buffer, "");
    }
}

/*
 *====================================================================
 * PURPOSE: get WINS addresses
 *--------------------------------------------------------------------
 * PARAMS:  OUT A semicolon\-delimited list of WINS servers
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void              /* wins addresses in NBO or 0 */
udDefGetWins(
    NQ_WCHAR *pServers        /* The WINS servers IP addresses */
    )
{

    if (!staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }

    syAnsiToUnicode(pServers, (0 != *staticData->winsAddresses) ? staticData->winsAddresses : "\0");
}

/*
 *====================================================================
 * PURPOSE: get domain name
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udDefGetDomain(
    NQ_WCHAR *buffer,
    NQ_BOOL  *isWorkgroup
    )
{
    if (!staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }

    syAnsiToUnicode(buffer, staticData->domainNameOfServer);
    *isWorkgroup = staticData->isWorkgroupName;
}

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
/*
 *====================================================================
 * PURPOSE: get DNS initialization parameters
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the default domain target belongs to
 *          OUT The DNS server address: IPv4 or IPv6
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udDefGetDnsParams(
    NQ_WCHAR *domain,
    NQ_WCHAR *server
    )
{
    /*syAnsiToUnicode(domain, NQ_DNSDOMAIN);*/
    if (!staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }
    syAnsiToUnicode(domain, staticData->domainNameOfServer);

    syAnsiToUnicode(server, (*staticData->dnsAddresses != 0) ? staticData->dnsAddresses : NQ_DNSADDRESS);
    /*printf("%s dns:%s \n", __FUNCTION__, cmWDump(server));*/
}

#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */

/*
 *====================================================================
 * PURPOSE: get authentication parameters
 *--------------------------------------------------------------------
 * PARAMS:  IN: URI about to connect to
 *          OUT buffer for user name
 *          OUT buffer for password
 *          OUT buffer for domain name
 *
 * RETURNS: TRUE - success
 *          FALSE - fail
 *
 * NOTES:   Not implemented yet
 *====================================================================
 */

NQ_BOOL
udDefGetCredentials(
    const void* resource,
    NQ_WCHAR* userName,
    NQ_WCHAR* password,
    NQ_WCHAR* domain
    )
{
    if (userName)
        syAnsiToUnicode(userName, staticData->userName);
    if (password)
        syAnsiToUnicode(password, staticData->password);
    if (domain)
        syAnsiToUnicode(domain, staticData->domainNameOfClient);

    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get next share in the list of shares for CIFS Server
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for share name
 *          OUT buffer for the map path
 *          OUT pointer to variable getting 0 for file system share and 1 for a print queue
 *          OUT buffer for the share description
 *          OUT pointer to variable getting 0 for regular share and 1 for encrypted share (SMB3)
 *
 * RETURNS: TRUE - next share read
 *          FALSE - no more shares
 *
 * NOTES:   Consequently parses the CIFS configuration file for shares
 *          if there is no configuration file, this function returns on its first call:
 *          "Root", NQ_CONFIGPATH, "Default root (no configuration file provided)"
 *====================================================================
 */

NQ_BOOL
udDefGetNextShareEx(
    NQ_WCHAR* name,
    NQ_WCHAR* map,
    NQ_BOOL* isPrinter,
    NQ_WCHAR* description,
    NQ_BOOL* isEncrypted
    )
{
    NQ_STATIC char nameA[UD_FS_MAXSHARELEN];
    NQ_STATIC char mapA[UD_FS_MAXPATHLEN];
    NQ_STATIC char descriptionA[UD_FS_MAXDESCRIPTIONLEN];
    NQ_STATIC char bufferA[UD_FS_MAXPATHLEN];

    if (!staticData->readingShares)
    {
        staticData->readingShares = 1;      /* start reading shares */
        setFileNames();
        if (!parseInit(&staticData->shareParser, staticData->csConfigFile)) /* start parsing */
        {
            parseStop(&staticData->shareParser);
            staticData->readingShares = 0;

            if (staticData->defaultShareReported)
            {
                return FALSE;      /* no share read */
            }
            else
            {
                staticData->defaultShareReported = 1;
                syAnsiToUnicode(name, "Root");
                syAnsiToUnicode(map, NQ_CONFIGPATH);
                syAnsiToUnicode(description, "Default root (no configuration file provided, edit cs_cfg.txt)");
                *isPrinter = FALSE;
                *isEncrypted = FALSE;
                return TRUE;
            }
        }
    }

    /* cycle by lines of the parameter file */

    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&staticData->shareParser))    /* EOF */
        {
            if (!staticData->hiddenShareReported)
            {
                /*  return default hidden C$ share */
                syAnsiToUnicode(name, "C$");
                syAnsiToUnicode(map, NQ_CONFIGPATH);
                syAnsiToUnicode(description, "Default share");
                *isPrinter = FALSE;
                *isEncrypted = FALSE;
                staticData->hiddenShareReported = 1;
                return TRUE;
            }
            staticData->readingShares = 0;
            return FALSE;                   /* no share read */
        }

        /* <share name>;<path>;<share comment>;<type> type - printer or encrypted */
        *isPrinter = FALSE;
        *isEncrypted = FALSE;
        parseSkipSpaces(&staticData->shareParser);
        if (!parseAtLineEnd(&staticData->shareParser) && (ch = parseGet(&staticData->shareParser)) != '#')    /* comment  line? */
        {
            parseUnget(&staticData->shareParser, ch);
            parseSkipSpaces(&staticData->shareParser);
            parseValue(&staticData->shareParser, nameA, sizeof(nameA), ';');      /* share name */
            if (parseDelimiter(&staticData->shareParser, ';'))
            {
                parseValue(&staticData->shareParser, mapA, sizeof(mapA), ';');  /* share map */
                if (parseDelimiter(&staticData->shareParser, ';'))
                {
                    parseValue(&staticData->shareParser, descriptionA, sizeof(descriptionA), ';'/*(char)0*/);  /* share description */
                    if (parseDelimiter(&staticData->shareParser, ';'))
                    {
                        parseValue(&staticData->shareParser, bufferA, sizeof(bufferA), ';'/*(char)0*/);  /* printer or encrypted */
                        if (syStrncmp(bufferA, "printer", syStrlen("printer")) == 0)
                        {
                            *isPrinter = TRUE;
                        }
                        else if (syStrncmp(bufferA, "encrypted", syStrlen("encrypted")) == 0)
                        {
                            *isEncrypted = TRUE;
                        }
                        if (parseDelimiter(&staticData->shareParser, ';'))
                        {
                            parseValue(&staticData->shareParser, bufferA, sizeof(bufferA), ';'/*(char)0*/);  /* printer or encrypted */
                            if (syStrncmp(bufferA, "printer", syStrlen("printer")) == 0)
                            {
                                *isPrinter = TRUE;
                            }
                            else if (syStrncmp(bufferA, "encrypted", syStrlen("encrypted")) == 0)
                            {
                                *isEncrypted = TRUE;
                            }
                        }
                    }
                    syAnsiToUnicode(name, nameA);
                    syAnsiToUnicode(map, mapA);
                    syAnsiToUnicode(description, descriptionA);
                    return TRUE;   /* a share read */
                }
            }
        }
        parseSkipLine(&staticData->shareParser);
    }
}

/*
 *====================================================================
 * PURPOSE: get next mount in the list of mounted file systems for CIFS
 *          Client
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for volume name
 *          OUT buffer for the map path
 *
 * RETURNS: TRUE - a mount read FALSE - no more mounts
 *
 * NOTES:   Consequently parses the mount file
 *====================================================================
 */

NQ_BOOL
udDefGetNextMount(
    NQ_WCHAR* name,
    NQ_WCHAR* map
    )
{
    NQ_STATIC char nameA[UD_FS_MAXSHARELEN];
    NQ_STATIC char mapA[UD_FS_MAX_SHARE_NET_PATH_LEN];

    if (!staticData->readingMounts)
    {
        staticData->readingMounts = 1;      /* start reading shares */
        setFileNames();
        if (!parseInit(&staticData->mountParser, staticData->ccConfigFile)) /* start parsing */
        {
            staticData->readingMounts = 0;
            return FALSE;      /* no mount read */
        }
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&staticData->mountParser))    /* EOF */
        {
            staticData->readingMounts = 0;
            return FALSE;                   /* no mount read */
        }

        parseSkipSpaces(&staticData->mountParser);
        if ((ch = parseGet(&staticData->mountParser)) != '#')    /* comment  line? */
        {
            parseUnget(&staticData->mountParser, ch);
            parseSkipSpaces(&staticData->mountParser);
            parseValue(&staticData->mountParser, nameA, sizeof(nameA), ';');      /* mount name */
            if (parseDelimiter(&staticData->mountParser, ';'))
            {
                parseValue(&staticData->mountParser, mapA, sizeof(mapA), ';');   /* share path */
                parseSkipLine(&staticData->mountParser);
                syAnsiToUnicode(name, nameA);
                syAnsiToUnicode(map, mapA);
                return TRUE;
            }
        }
        parseSkipLine(&staticData->mountParser);
    }
}


/*
 *====================================================================
 * PURPOSE: get server message signing policy
 *--------------------------------------------------------------------
 * PARAMS:  OUT pointer to parameter getting 0 - disabled, 1 - enabled, 2 - required
 *
 * RETURNS: TRUE - implemented
 *          FALSE - not implemented (use NQ default)
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
udDefGetMessageSigningPolicy(
        NQ_INT* policy
        )
{
    ParseContext parser;
    NQ_BOOL result = FALSE;

    if (NULL == policy)
    {
        goto Exit;
    }

    if (staticData->signingPolicyServer != NQ_NOT_INITIALIZED)
    {
        *policy = staticData->signingPolicyServer;
        result = TRUE;
        goto Exit;
    }

    setFileNames();
    if (!parseInit(&parser, staticData->csConfigFile)) /* start parsing */
    {
        parseStop(&staticData->shareParser);
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&parser))      /* EOF */
        {
            goto Exit;
        }

        parseSkipSpaces(&parser);

        if ((ch = parseGet(&parser)) != '#')       /* comment  line? */
        {
            char name[NQ_CONFIG_FILE_NAME_LEN];    /* buffer for name */
            char value[NQ_CONFIG_FILE_VALUE_LEN];   /* buffer for value */

            parseUnget(&parser, ch);
            parseSkipSpaces(&parser);
            parseName(&parser, name, sizeof(name));                  /* parameter name */
            parseSkipSpaces(&parser);

            if (parseDelimiter(&parser, '='))
            {
                parseSkipSpaces(&parser);
                parseValue(&parser, value, sizeof(value), (char)0);  /* parameter value - no delimiter */

                *policy = NQ_NOT_INITIALIZED;
                if (syStrncmp(name, "MESSAGESIGNING", strlen("MESSAGESIGNING")) == 0)
                {
                    if (syStrncmp(value, "DISABLED", strlen("DISABLED")) == 0)
                    {
                        *policy = 0;
                    }
                    else if (syStrncmp(value, "ENABLED", strlen("ENABLED")) == 0)
                    {
                        *policy = 1;
                    }
                    else if (syStrncmp(value, "REQUIRED", strlen("REQUIRED")) == 0)
                    {
                        *policy = 2;
                    }
                    result = (*policy != NQ_NOT_INITIALIZED);
                    if (result)
                    {
                        staticData->signingPolicyServer = *policy;
                    }
                    parseStop(&parser);
                    goto Exit;
                }
            }
        }
        parseSkipLine(&parser);
    }

Exit:
    /*printf("%s result:%s policy:%d\n", __FUNCTION__, result ? "TRUE" : "FALSE", *policy);*/
    return result;
}

/*
 *====================================================================
 * PURPOSE: get server global encryption
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE - global encryption
 *          FALSE - no global encryption
 *
 * NOTES: This setting is controlled by UD only,
 * so when parameter not found in config it means FALSE.
 *====================================================================
 */
NQ_BOOL
udDefGetGlobalEncryption(void)
{
    ParseContext parser;
    NQ_BOOL result = FALSE;

    if (staticData->isGlobalEncryptionServer != NQ_NOT_INITIALIZED)
    {
        result = staticData->isGlobalEncryptionServer;
        goto Exit;
    }

    setFileNames();
    if (!parseInit(&parser, staticData->csConfigFile)) /* start parsing */
    {
        parseStop(&parser);
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&parser))      /* EOF */
        {
            goto Exit;
        }

        parseSkipSpaces(&parser);

        if ((ch = parseGet(&parser)) != '#')    /* comment  line? */
        {
            char name[NQ_CONFIG_FILE_NAME_LEN];    /* buffer for name */
            char value[NQ_CONFIG_FILE_VALUE_LEN];   /* buffer for value */

            parseUnget(&parser, ch);
            parseSkipSpaces(&parser);
            parseName(&parser, name, sizeof(name));                  /* parameter name */
            parseSkipSpaces(&parser);

            if (parseDelimiter(&parser, '='))
            {
                int encryption = NQ_NOT_INITIALIZED;

                parseSkipSpaces(&parser);
                parseValue(&parser, value, sizeof(value), (char)0);  /* parameter value - no delimiter */

                if (syStrncmp(name, "GLOBALENCRYPTION", strlen("GLOBALENCRYPTION")) == 0)
                {
                    if (syStrncmp(value, "TRUE", strlen("TRUE")) == 0)
                    {
                        encryption = 1;
                    }
                    else if (syStrncmp(value, "FALSE", strlen("FALSE")) == 0)
                    {
                        encryption = 0;
                    }
                    staticData->isGlobalEncryptionServer = (encryption == 1) ? 1 : 0;
                    result = staticData->isGlobalEncryptionServer;
                    parseStop(&parser);
                    goto Exit;
                }
            }
        }
        parseSkipLine(&parser);
    }

Exit:
    /*printf("%s result:%s policy:%d\n", __FUNCTION__, result ? "TRUE" : "FALSE", staticData->isGlobalEncryptionServer);*/
    return result;
}

/*
 *====================================================================
 * PURPOSE: get server anonymous access setting
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT access - 1 allowed, 0 - not allowed
 *
 * RETURNS: TRUE - when anonymous access parameter was found in config file
 *          FALSE - when anonymous access parameter was not found in config file
 *
 * NOTES: Return value of access parameter has no meaning when method returns FALSE
 *====================================================================
 */
NQ_BOOL
udDefGetServerAnonymousAllowed(NQ_BOOL *access)
{
    ParseContext parser;
    NQ_BOOL result = FALSE;

    if (NULL == access)
    {
        goto Exit;
    }

    if (staticData->isAnonymousAccessAllowedServer != NQ_NOT_INITIALIZED)
    {
        *access = staticData->isAnonymousAccessAllowedServer;
        result = TRUE;
        goto Exit;
    }

    setFileNames();
    if (!parseInit(&parser, staticData->csConfigFile)) /* start parsing */
    {
        syPrintf("Failed to open cs_cfg.txt\n");
        parseStop(&parser);
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&parser))      /* EOF */
        {
            goto Exit;
        }

        parseSkipSpaces(&parser);

        if ((ch = parseGet(&parser)) != '#')    /* comment  line? */
        {
            char name[NQ_CONFIG_FILE_NAME_LEN];    /* buffer for name */
            char value[NQ_CONFIG_FILE_VALUE_LEN];   /* buffer for value */

            parseUnget(&parser, ch);
            parseSkipSpaces(&parser);
            parseName(&parser, name, sizeof(name));                  /* parameter name */
            parseSkipSpaces(&parser);

            if (parseDelimiter(&parser, '='))
            {
                parseSkipSpaces(&parser);
                parseValue(&parser, value, sizeof(value), (char)0);  /* parameter value - no delimiter */

                *access = NQ_NOT_INITIALIZED;
                if (syStrncmp(name, "ANONYMOUSALLOWED", strlen("ANONYMOUSALLOWED")) == 0)
                {
                    if (syStrncmp(value, "TRUE", strlen("TRUE")) == 0)
                    {
                        *access = 1;
                    }
                    else if (syStrncmp(value, "FALSE", strlen("FALSE")) == 0)
                    {
                        *access = 0;
                    }
                    result = (*access != NQ_NOT_INITIALIZED);
                    if (result)
                    {
                        staticData->isAnonymousAccessAllowedServer = *access;
                    }
                    parseStop(&parser);
                    goto Exit;
                }
            }
        }
        parseSkipLine(&parser);
    }

Exit:
    return result;
}


#ifdef UD_NQ_INCLUDESMBCAPTURE
/*
 *====================================================================
 * PURPOSE: whether to use internal capture
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE
 *          FALSE
 *
 * NOTES: Returns FALSE only when parameter was present and FALSE in config file,
 *        parameter not present in config file is considered as TRUE, meaning default
 *        NQ core setting is not affected
 *====================================================================
 */
NQ_BOOL
udDefGetInternalCapture(void)
{
    if (NQ_NOT_INITIALIZED == staticData->useInternalCapture && !staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }

    return (staticData->useInternalCapture == 0) ? FALSE : TRUE;
}


/*
*====================================================================
* PURPOSE: get base folder path of internal capture file
*--------------------------------------------------------------------
* PARAMS:  buffer - buffer for the path
*          length - buffer size
*
* NOTES:
*====================================================================
*/
void
udDefGetCaptureFileBaseFolder(
    NQ_CHAR* buffer,
    NQ_UINT  size
    )
{
    if (!staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }

    strncpy(buffer, staticData->captureFileBaseFolder, size > sizeof(staticData->captureFileBaseFolder) ? sizeof(staticData->captureFileBaseFolder) : size);
}
#endif /* UD_NQ_INCLUDESMBCAPTURE */


/*
 *====================================================================
 * PURPOSE: whether to use internal trace
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE
 *          FALSE
 *
  * NOTES: Returns FALSE only when parameter was present and FALSE in config file,
 *        parameter not present in config file is considered as TRUE, meaning default
 *        NQ core setting is not affected
 *====================================================================
 */
NQ_BOOL
udDefGetInternalTrace(void)
{
    if (NQ_NOT_INITIALIZED == staticData->useInternalTrace && !staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }

    return (staticData->useInternalTrace == 0) ? FALSE : TRUE;
}

/*
 *====================================================================
 * PURPOSE: get task priorities
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: task priority
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
udDefGetTaskPriorities(
    void
    )
{
    return NQ_TASKPRIORITY;
}


/*
 *====================================================================
 * PURPOSE: get server comment string
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result (fits 256 NQ_WCHARs)
 *
 * RETURNS: None
 *
 * NOTES:
 *
 *====================================================================
 */

void
udDefGetServerComment(
    NQ_WCHAR* buffer
    )
{
    syAnsiToUnicode(buffer, NQ_SERVERCOMMENT);
}

/*
 *====================================================================
 * PURPOSE: get next user name and password from the list of passwords
 *--------------------------------------------------------------------
 * PARAMS:  IN login name
 *          OUT buffer for password
 *          OUT TRUE if the password is hashed, FALSE - otherwise
 *          OUT user number while administrative users have numbers < 0
 *
 * RETURNS: NQ_CS_PWDFOUND - user found equivalent to 3 (deprecated)
 *          NQ_CS_PWDNOAUTH - authentication is not required
 *          NQ_CS_PWDNOUSER - no such user
 *          NQ_CS_PWDLMHASH - user found and password is LM hash (*pwdIsHashed value has to
 *              be TRUE in this case)
 *          NQ_CS_PWDANY - user found and password is either LM and NTLM hash or plain
 *              text depending on the *pwdIsHashed value
 *
 * NOTES:   Opens the file, parses it and stores parameter values if
 *          those parameters were found. User number is returned as ID from
 *             the pwd file
 *====================================================================
 */

NQ_INT
udDefGetPassword(
    const NQ_WCHAR* userName,
    NQ_CHAR* password,
    NQ_BOOL* pwdIsHashed,
    NQ_UINT32* userNumber
    )
{
    ParseContext userParser;    /* parser for reading the password list */
    char name[CM_USERNAMELENGTH + CM_TRAILING_NULL];              /* next user name */
    char userNameA[CM_USERNAMELENGTH + CM_TRAILING_NULL];        /* user name in ASCII */
    NQ_UINT i;
    char userNumText[12];

    /* start parsing passwords */

    setFileNames();
    if (!parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        return NQ_CS_PWDNOAUTH;           /* proceed without authentication */
    }

    syUnicodeToAnsi(userNameA, userName);

    for (i = 0; i < strlen(userNameA); i++)
    {
        userNameA[i] = (char)tolower(((int)userNameA[i]));
    }

    if (0 == syStrcmp(userNameA, "anonymous logon"))
    {
        *userNumber = 666;
#ifdef UD_CS_AUTHENTICATEANONYMOUS
        /* up to UD to decide whether to allow login for anonymous:
           NQ_CS_PWDNOUSER - will fail the anonymous login
           any other - will allow the anonymous login */

        return NQ_CS_PWDNOUSER;           /* no such user, result login failure */
#else
        return NQ_CS_PWDNOAUTH;           /* proceed without authentication */
#endif /* UD_CS_AUTHENTICATEANONYMOUS */
    }

    /* cycle by lines of the parameter file */

    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&userParser))    /* EOF */
        {
            parseStop(&userParser);
            return NQ_CS_PWDNOUSER;                   /* user not found */
        }

        parseSkipSpaces(&userParser);
        if (parseAtLineEnd(&userParser))            /* empty line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        parseUnget(&userParser, ch);
        parseSkipSpaces(&userParser);
        parseName(&userParser, name, sizeof(name));      /* user name */
        for (i = 0; i < strlen(name); i++)
        {
            name[i] = (char)tolower(((int)name[i]));
        }
        if (parseDelimiter(&userParser, ':'))
        {
            /* the hashed passwords are stored in hex format in pwd.txt thats why the input string has double size */
            parseValue(&userParser, password, UD_NQ_MAXPWDLEN, ':');  /* password */
        }
        if (parseDelimiter(&userParser, ':'))
        {
            parseValue(&userParser, userNumText, sizeof(userNumText) - 1, ':');  /* ID */
        }
        parseSkipLine(&userParser);

        if (strcmp(userNameA, name) == 0)
        {
            int pwdlen = (int)strlen(password);

            *userNumber = (NQ_UINT32)atoi(userNumText);
            *pwdIsHashed = (pwdlen == (NQ_HASHED_PASSWORD_LENGTH * 2) || pwdlen == (NQ_HASHED_PASSWORD_LENGTH * 4));

            parseStop(&userParser);

            /* if the password is hashed convert it to a binary form */
            if (*pwdIsHashed)
            {
                if (!convertHex2Ascii(password, NQ_HASHED_PASSWORD_LENGTH * 2))
                {
                    return NQ_CS_PWDNOUSER;   /* report user not found  */
                }

                /* password is LM and NTLM hash if its length equals to 64, otherwise consider it LM hash */
                return (pwdlen == (NQ_HASHED_PASSWORD_LENGTH * 4))? NQ_CS_PWDANY : NQ_CS_PWDLMHASH;
            }

            return NQ_CS_PWDANY; /* the password is plain text */
        }
    }
}

#ifdef UD_NQ_INCLUDECODEPAGE

/*
 *====================================================================
 * PURPOSE: get the current code page
 *--------------------------------------------------------------------
 * PARAMS:
 *
 * RETURNS: code page number
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
udDefGetCodePage(
    void
    )
{
    return staticData->currentCodePage;
}


/*
 *====================================================================
 * PURPOSE: set the current code page
 *--------------------------------------------------------------------
 * PARAMS: code page number.
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udDefSetCodePage(
    NQ_INT codePage
    )
{
    staticData->currentCodePage = codePage;
}

#endif /* UD_NQ_INCLUDECODEPAGE */

/*
 *====================================================================
 * PURPOSE: allocate buffer in the user space
 *--------------------------------------------------------------------
 * PARAMS:  IN buffer index zero based
 *          IN total number of buffers
 *          IN buffer size in bytes
 *
 * RETURNS: pointer to the buffer
 *
 * NOTES:   Include any project-level processing here
 *====================================================================
 */

NQ_BYTE*
udDefAllocateBuffer(
    NQ_INT idx,
    NQ_COUNT numBufs,
    NQ_UINT bufferSize
    )
{
    return staticData->staticBuffers + (NQ_UINT)idx*(bufferSize);
}

/*
 *====================================================================
 * PURPOSE: Return CIFS driver name
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the result
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udDefGetDriverName(
    NQ_CHAR* buffer
    )
{
    strcpy(buffer, NQ_CCDRVNAME);
}

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)

/*
 *====================================================================
 * PURPOSE: get unique ID for the current machine
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer of 12 bytes length
 *
 * RETURNS: None
 *
 * NOTES:   The returned 12-byte value should be:
 *              - "statistically unique" for the given machine
 *              - persistently the same for each call
 *             Recommended methods are:
 *              - MAC address of the default adapter
 *              - product serial number when available
 *             This reference implementation returns the same number for
 *             each computer
 *====================================================================
 */

void
udDefGetComputerId(
    NQ_BYTE* buf
    )
{
    NQ_INT i;

    for (i = 0; i < 12 ; i++)
    {
        *buf++ = 0xA5;
    }
}

/*
 *====================================================================
 * PURPOSE: Get persistent security descriptor for share
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *          OUT buffer for SD
 *          IN buffer length
 *
 * RETURNS: SD length or zero on error
 *
 * NOTES:
 *====================================================================
 */

NQ_COUNT
udDefLoadShareSecurityDescriptor(
    const NQ_WCHAR* shareName,
    NQ_BYTE* buffer,
    NQ_COUNT bufferLen
    )
{
#define isHex(_a)   isxdigit((int)_a)
#define ascToHex(_a) ((_a) > '9'? tolower((int)_a) - 'a' + 10 : (int)(_a) - '0')
    NQ_STATIC char nameA[256];
    int file;
    NQ_COUNT res = 0;

    syUnicodeToAnsi(nameA, shareName);

    file = open(staticData->shareSdFile, O_RDONLY, 0777); /* start parsing */
    if (ERROR == file)
        return 0;      /* no share read */

    /* cycle by lines of the SD file */

    while (1)
    {
        static char in[SD_RECLEN];                        /* next record */

        if (read(file, in, SD_RECLEN) != SD_RECLEN)
        {
            res = 0;
            break;
        }
        if (0 == strcmp(nameA, in))     /* share name match */
        {
            char* pc = in + SD_SHARENAMELEN + 2;

            while (0 != *pc)
            {
                int val;

                if (bufferLen <=0)
                {
                    syPrintf("UDCONFIG - Buffer overflow while loading share SD\n");
                    res = 0;
                    break;
                }
                if (!isHex(*pc))
                {
                    syPrintf("UDCONFIG - Non-hexadecimal character found in Share SD file: %s\n", pc);
                    res = 0;
                    break;
                }
                val = ascToHex(*pc);
                pc++;
                if (!isHex(*pc))
                {
                    syPrintf("UDCONFIG - Non-hexadecimal character found in Share SD file: %s\n", pc);
                    res = 0;
                    break;
                }
                val = 16 * val + ascToHex(*pc);
                pc++;
                *buffer++ = (NQ_BYTE)val;
                res++;
                bufferLen--;
            }
            break;
        }
    }
    close(file);

    return res;
}

/*
 *====================================================================
 * PURPOSE: Save persistent security descriptor for share
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *          IN pointer to SD
 *          IN SD length
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
udDefSaveShareSecurityDescriptor(
    const NQ_WCHAR* shareName,
    const NQ_BYTE* sd,
    NQ_COUNT sdLen
    )
{
    NQ_STATIC char nameA[UD_FS_MAXSHARELEN];
    int file;
    int recNum;

    syUnicodeToAnsi(nameA, shareName);

    if (sdLen > SD_SDLEN)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "security descriptor length too long");
        goto Exit;
    }

    file = open(staticData->shareSdFile, O_CREAT , 0777); /* start parsing */
    if (ERROR == file)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to create file:%s", staticData->shareSdFile);
        goto Exit;
    }

    close(file);
    file = open(staticData->shareSdFile, O_RDWR , 0777); /* start parsing */
    if (ERROR == file)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to open file:%s", staticData->shareSdFile);
        goto Exit;
    }

    /* cycle by lines of the SD file */

    for (recNum = 0; ; recNum++)
    {
        NQ_STATIC char in[SD_RECLEN];                        /* next record */

        if (read(file, in, SD_RECLEN) != SD_RECLEN)
        {
            break;
        }
        if (0 == strcmp(nameA, in))     /* share name match */
        {
            int i;

            close (file);
            file = open(staticData->shareSdFile, O_RDWR , 0777); /* start parsing again */
            if (ERROR == file)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "failed to open file:%s", staticData->shareSdFile);
                goto Exit;
            }

            for (i = recNum; i > 0; i--)
            {
                if (read(file, in, SD_RECLEN) != SD_RECLEN)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "read failed");
                    goto Exit1;
                }
            }

            break;
        }
    }

    /* update record (either old or new) */
    {
        NQ_STATIC char temp[SD_SDLEN * 2];
        NQ_STATIC char name[SD_SHARENAMELEN];
        char* pc = temp;

        memset(name, 0, sizeof(name));
        strncpy(name, nameA, sizeof(name));
        if (write(file, name, sizeof(name)) != sizeof(name))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "share name write failed");
            goto Exit1;
        }

        if (write(file, ";;", 2) != 2)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "write \";;\" failed");
            goto Exit1;
        }

        memset(temp, 0, sizeof(temp));
        while (sdLen > 0)
        {
            sprintf(pc, "%02x", *sd++);
            sdLen--;
            pc += 2;
        }

        if (write(file, temp, sizeof(temp)) != sizeof(temp))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "security descriptor write failed");
            goto Exit1;
    }

        if (write(file, "\n\r", 2) != 2)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "write \"\n\r\" failed");
            goto Exit1;
        }
    }

Exit1:
    close(file);
Exit:
    return;
}

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

/*
 *====================================================================
 * PURPOSE: get number of local users
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: Number of local users
 *
 * NOTES:
 *====================================================================
 */

NQ_COUNT
udDefGetUserCount(
    void
    )
{
    ParseContext userParser;    /* parser for reading the password list */
    NQ_COUNT count = 0;

    setFileNames();
    if (!parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        return 0;    /* no users */
    }

    /* cycle by lines of the parameter file */

    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&userParser))    /* EOF */
        {
            parseStop(&userParser);
            return count;
        }

        parseSkipSpaces(&userParser);
        if (parseAtLineEnd(&userParser))            /* empty line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        parseUnget(&userParser, ch);
        parseSkipLine(&userParser);
        count++;
    }
}

/*
 *====================================================================
 * PURPOSE: get user ID by name
 *--------------------------------------------------------------------
 * PARAMS:  IN user name
 *          OUT buffer for user ID
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDefGetUserRidByName(
    const NQ_WCHAR* name,
    NQ_UINT32* rid
    )
{
    ParseContext userParser;    /* parser for reading the password list */
    NQ_STATIC char nextName[CM_USERNAMELENGTH];     /* next user name */
    NQ_STATIC char userNameA[CM_USERNAMELENGTH];    /* user name in ASCII */
    NQ_STATIC char password[UD_NQ_MAXPWDLEN];       /* password in ASCII */
    unsigned int i;
    char userNumText[12];

    /* start parsing passwords */

    syUnicodeToAnsi(userNameA, name);

    for (i = 0; i < strlen(userNameA); i++)
    {
        userNameA[i] = (char)tolower(((int)userNameA[i]));
    }

    setFileNames();
    if (!parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        return FALSE;           /* proceed without authentication */
    }

    /* cycle by lines of the parameter file */

    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&userParser))    /* EOF */
        {
            parseStop(&userParser);
            return FALSE;                   /* user not found */
        }

        parseSkipSpaces(&userParser);
        if (parseAtLineEnd(&userParser))            /* empty line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        parseUnget(&userParser, ch);
        parseSkipSpaces(&userParser);
        parseName(&userParser, nextName, sizeof(nextName) - CM_TRAILING_NULL);      /* user name */
        for (i = 0; i < strlen(nextName); i++)
        {
            nextName[i] = (char)tolower(((int)nextName[i]));
        }
        if (parseDelimiter(&userParser, ':'))
        {
            parseValue(&userParser, password, sizeof(password) - CM_TRAILING_NULL, ':');  /* password */
        }
        if (parseDelimiter(&userParser, ':'))
        {
            parseValue(&userParser, userNumText, sizeof(userNumText) - CM_TRAILING_NULL, ':');  /* ID */
        }
        parseSkipLine(&userParser);

        if (strcmp(userNameA, nextName) == 0)
        {
            *rid = (NQ_UINT32)atoi(userNumText);
            parseStop(&userParser);
            return TRUE; /* user found */
        }
    }
}

/*
 *====================================================================
 * PURPOSE: get user name by ID
 *--------------------------------------------------------------------
 * PARAMS:  IN user ID
 *          OUT buffer for user name
 *          OUT buffer for user name
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDefGetUserNameByRid(
    NQ_UINT32 rid,
    NQ_WCHAR* nameBuffer,
    NQ_WCHAR* fullNameBuffer
    )
{
    ParseContext userParser;    /* parser for reading the password list */
    char name[CM_USERNAMELENGTH];              /* next user name */
    char userNumText[12];
    NQ_STATIC char password[UD_NQ_MAXPWDLEN];     /* password in ASCII */

    setFileNames();
    if (!parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        return FALSE;           /* proceed without authentication */
    }

    /* cycle by lines of the parameter file */

    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&userParser))    /* EOF */
        {
            parseStop(&userParser);
            return FALSE;                   /* user not found */
        }

        parseSkipSpaces(&userParser);
        if (parseAtLineEnd(&userParser))            /* empty line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        parseUnget(&userParser, ch);
        parseSkipSpaces(&userParser);
        parseName(&userParser, name, sizeof(name) - CM_TRAILING_NULL);      /* user name */
        if (parseDelimiter(&userParser, ':'))
        {
            parseValue(&userParser, password, sizeof(password) - CM_TRAILING_NULL, ':');  /* password */
        }
        if (parseDelimiter(&userParser, ':'))
        {
            parseValue(&userParser, userNumText, sizeof(userNumText) - 1, ':');  /* ID */
        }
        parseSkipLine(&userParser);

        if ((NQ_INT)rid == atoi(userNumText))
        {
            syAnsiToUnicode(nameBuffer, name);
            syAnsiToUnicode(fullNameBuffer, name);
            parseStop(&userParser);
            return TRUE; /* user found */
        }
    }
}

/*
 *====================================================================
 * PURPOSE: enumerate users
 *--------------------------------------------------------------------
 * PARAMS:  IN user index (zero based)
 *          OUT buffer for user id
 *          OUT buffer for user name
 *          OUT buffer for user's full name
 *          OUT buffer for user description
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDefGetUserInfo(
    NQ_UINT index,
    NQ_UINT32* rid,
    NQ_WCHAR* shortName,
    NQ_WCHAR* fullName,
    NQ_WCHAR* description
    )
{
    ParseContext userParser;    /* parser for reading the password list */
    char name[CM_USERNAMELENGTH];              /* next user name */
    char userNumText[12];
    NQ_STATIC char password[UD_NQ_MAXPWDLEN];     /* password in ASCII */

    setFileNames();
    if (!parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        return FALSE;           /* proceed without authentication */
    }

    /* cycle by lines of the parameter file */

    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&userParser))    /* EOF */
        {
            parseStop(&userParser);
            return FALSE;                   /* user not found */
        }

        parseSkipSpaces(&userParser);
        if (parseAtLineEnd(&userParser))            /* empty line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
        {
            parseSkipLine(&userParser);
            continue;
        }
        parseUnget(&userParser, ch);
        parseSkipSpaces(&userParser);
        parseName(&userParser, name, sizeof(name) - CM_TRAILING_NULL);      /* user name */
        if (parseDelimiter(&userParser, ':'))
        {
            parseValue(&userParser, password, sizeof(password) - CM_TRAILING_NULL, ':');  /* password */
        }
        if (parseDelimiter(&userParser, ':'))
        {
            parseValue(&userParser, userNumText, sizeof(userNumText) - 1, ':');  /* ID */
        }
        parseSkipLine(&userParser);

        if (index-- <= 0)
        {
            *rid = (NQ_UINT32)atoi(userNumText);
            syAnsiToUnicode(shortName, name);
            syAnsiToUnicode(fullName, name);
            syAnsiToUnicode(description, ((NQ_INT)*rid) < 0? "Administrator":"Ordinary user");
            parseStop(&userParser);
            return TRUE; /* user found */
        }
    }
}

/*
 *====================================================================
 * PURPOSE: modify user
 *--------------------------------------------------------------------
 * PARAMS:  IN user RID
 *          IN user name
 *          IN full user name
 *          IN user description
 *          IN Unicode text password or NULL
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDefSetUserInfo
(
    NQ_UINT32 rid,
    const NQ_WCHAR* name,
    const NQ_WCHAR* fullName,
    const NQ_WCHAR* description,
    const NQ_WCHAR* password
    )
{
    ParseContext userParser;            /* parser for reading the password list */
    NQ_STATIC char nextName[CM_USERNAMELENGTH];     /* next user name */
    NQ_STATIC char userNameA[CM_USERNAMELENGTH];    /* user name in ASCII */
    char userNumText[12];               /* next RID */
    char matchUserNumText[12];          /* required user RID */
    NQ_STATIC char oldPassword[UD_NQ_MAXPWDLEN];    /* password in ASCII hex pairs */
    NQ_STATIC char matchPassword[UD_NQ_MAXPWDLEN];  /* required user password in ASCII */
    FILE* tempFile;                     /* temporary file fd */
    int oldUser = 0;                    /* user exists */
    NQ_INT nextRid;

    syUnicodeToAnsi(userNameA, name);

    /* create temporary file */
    tempFile = fopen((const char*)staticData->tempFileName, "a+");
    if (NULL == tempFile)
        return FALSE;

    /* parse existing file */
    setFileNames();
    if (parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        /* cycle by lines of the parameter file */
        while (1)
        {
            char ch;                        /* next character */

            if (parseAtFileEnd(&userParser))    /* EOF */
            {
                parseStop(&userParser);
                break;
            }

            parseSkipSpaces(&userParser);
            if (parseAtLineEnd(&userParser))            /* empty line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            parseUnget(&userParser, ch);
            parseSkipSpaces(&userParser);
            parseName(&userParser, nextName, sizeof(nextName) - CM_TRAILING_NULL);      /* user name */
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, oldPassword, sizeof(oldPassword) - CM_TRAILING_NULL, ':');  /* password */
            }
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, userNumText, sizeof(userNumText) - CM_TRAILING_NULL, ':');  /* ID */
            }
            parseSkipLine(&userParser);

            nextRid = atoi(userNumText);

            if ((NQ_INT)rid == nextRid)
            {
                oldUser = 1;
                syStrcpy(matchPassword, oldPassword);
                syStrcpy(matchUserNumText, userNumText);
            }
            else
            {
                syFprintf(tempFile, "%s:%s:%s\n", nextName, oldPassword, userNumText);
            }
        }
    }
    if (NULL != password)
    {
        NQ_STATIC NQ_BYTE encryptedPassword[32];
        NQ_STATIC NQ_BYTE asciiPassword[256];

        cmUnicodeToAnsi((NQ_CHAR *)asciiPassword, password);
        cmHashPassword(asciiPassword, encryptedPassword);    /* LM */
        cmMD4(
            encryptedPassword + 16,
            (NQ_BYTE*)password,
            (NQ_UINT)(cmWStrlen(password) * sizeof(NQ_WCHAR))
            );                                                 /* NTLM */
        convertPasswordToText(encryptedPassword, matchPassword);
    }
    matchPassword[64] = '\0';
    if (oldUser)
    {
        syFprintf(tempFile, "%s:%s:%s\n", userNameA, matchPassword, matchUserNumText);
        fclose(tempFile);
        unlink(staticData->passwordFile);
        rename((const char*)staticData->tempFileName, staticData->passwordFile);
        return TRUE;
    }
    else
    {
        fclose(tempFile);
        unlink(staticData->tempFileName);
        return FALSE;
    }
}

/*
 *====================================================================
 * PURPOSE: add user
 *--------------------------------------------------------------------
 * PARAMS:  IN user name
 *          IN full user name
 *          IN user description
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:   This function either creates a new user or modifies an existing one.
 *====================================================================
 */

NQ_BOOL
udDefCreateUser(
    const NQ_WCHAR* name,
    const NQ_WCHAR* fullName,
    const NQ_WCHAR* description
    )
{
    ParseContext userParser;            /* parser for reading the password list */
    NQ_STATIC char nextName[CM_USERNAMELENGTH];     /* next user name */
    NQ_STATIC char userNameA[CM_USERNAMELENGTH];    /* user name in ASCII */
    NQ_STATIC char userNumText[12];     /* next RID */
    NQ_STATIC char password[UD_NQ_MAXPWDLEN];       /* password in ASCII */
    FILE* tempFile;                     /* temporary file fd */
    NQ_INT maxRid = 0;
    NQ_INT nextRid;

    syUnicodeToAnsi(userNameA, name);

    /* create temporary file */
    tempFile = fopen((const char*)staticData->tempFileName, "a+");
    if (NULL == tempFile)
        return FALSE;

    /* parse existing file */
    setFileNames();
    if (parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        /* cycle by lines of the parameter file */
        while (1)
        {
            char ch;                        /* next character */

            if (parseAtFileEnd(&userParser))    /* EOF */
            {
                parseStop(&userParser);
                break;
            }

            parseSkipSpaces(&userParser);
            if (parseAtLineEnd(&userParser))            /* empty line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            parseUnget(&userParser, ch);
            parseSkipSpaces(&userParser);
            parseName(&userParser, nextName, sizeof(nextName) - CM_TRAILING_NULL);      /* user name */
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, password, sizeof(password) - CM_TRAILING_NULL, ':');  /* password */
            }
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, userNumText, sizeof(userNumText) - CM_TRAILING_NULL, ':');  /* ID */
            }
            parseSkipLine(&userParser);

            nextRid = atoi(userNumText);
            if ((nextRid > 0 && nextRid > maxRid) || (nextRid < 0 && -nextRid > maxRid))
            {
                maxRid = nextRid > 0? nextRid : -nextRid;
            }

            if (0 == syStrcmp(userNameA, nextName))
            {
                 fclose(tempFile);
                unlink(staticData->tempFileName);
                return FALSE;
            }
            else
            {
                syFprintf(tempFile, "%s:%s:%s\n", nextName, password, userNumText);
            }
        }
    }

    syFprintf(tempFile, "%s:%s:%d\n", userNameA, "", maxRid + 1);
    fclose(tempFile);
    unlink(staticData->passwordFile);
    rename((const char*)staticData->tempFileName, staticData->passwordFile);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: set user administrative rights
 *--------------------------------------------------------------------
 * PARAMS:  IN user RID
 *          IN TRUE to make user an administrator
 *
 * RETURNS: TRUE when user was found
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDefSetUserAsAdministrator(
    NQ_UINT32 rid,
    NQ_BOOL    isAdmin
    )
{
    ParseContext userParser;            /* parser for reading the password list */
    NQ_STATIC char nextName[CM_USERNAMELENGTH];     /* next user name */
    NQ_STATIC char matchName[CM_USERNAMELENGTH];    /* required user name */
    char userNumText[12];               /* next RID */
    char matchUserNumText[12];          /* required user RID */
    NQ_STATIC char password[UD_NQ_MAXPWDLEN];       /* password in ASCII */
    NQ_STATIC char matchPassword[UD_NQ_MAXPWDLEN];  /* required user password in ASCII */
    FILE* tempFile;                     /* temporary file fd */
    int oldUser = 0;                    /* user exists */
    NQ_INT nextRid;

    if (((int)rid < 0) && isAdmin) return TRUE;   /* nothing to do */
    if (((int)rid > 0) && !isAdmin) return TRUE;   /* nothing to do */

    /* create temporary file */
    tempFile = fopen((const char*)staticData->tempFileName, "a+");
    if (NULL == tempFile)
        return FALSE;

    /* parse existing file */
    setFileNames();
    if (parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        /* cycle by lines of the parameter file */
        while (1)
        {
            char ch;                        /* next character */

            if (parseAtFileEnd(&userParser))    /* EOF */
            {
                parseStop(&userParser);
                break;
            }

            parseSkipSpaces(&userParser);
            if (parseAtLineEnd(&userParser))            /* empty line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            parseUnget(&userParser, ch);
            parseSkipSpaces(&userParser);
            parseName(&userParser, nextName, sizeof(nextName) - CM_TRAILING_NULL);      /* user name */
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, password, sizeof(password) - CM_TRAILING_NULL, ':');  /* password */
            }
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, userNumText, sizeof(userNumText) - CM_TRAILING_NULL, ':');  /* ID */
            }
            parseSkipLine(&userParser);

            nextRid = atoi(userNumText);

            if ((NQ_INT)rid == nextRid)
            {
                oldUser = 1;
                syStrcpy(matchName, nextName);
                syStrcpy(matchPassword, password);
                syStrcpy(matchUserNumText, userNumText);
            }
            else
            {
                syFprintf(tempFile, "%s:%s:%s\n", nextName, password, userNumText);
            }
        }
    }
    if (!oldUser)
    {
        fclose(tempFile);
        unlink(staticData->tempFileName);
        return FALSE;
    }
    else
    {
        rid  = (NQ_UINT32)(-(NQ_INT)rid);
        syFprintf(tempFile, "%s:%s:%d\n", matchName, matchPassword, (NQ_INT)rid);
    }

    fclose(tempFile);
    unlink(staticData->passwordFile);
    rename((const char*)staticData->tempFileName, staticData->passwordFile);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: remove user
 *--------------------------------------------------------------------
 * PARAMS:  IN user RID
 *
 * RETURNS: TRUE when user was deleted
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDefDeleteUserByRid(
    NQ_UINT32 rid
    )
{
    ParseContext userParser;            /* parser for reading the password list */
    NQ_STATIC char nextName[CM_USERNAMELENGTH];     /* next user name */
    char userNumText[12];               /* next RID */
    NQ_STATIC char password[UD_NQ_MAXPWDLEN];       /* password in ASCII */
    FILE* tempFile;                     /* temporary file fd */
    int userDeleted = 0;                /* user exists */
    NQ_INT nextRid;

    /* create temporary file */
    tempFile = fopen((const char*)staticData->tempFileName, "a+");
    if (NULL == tempFile)
        return FALSE;

    /* parse existing file */
    setFileNames();
    if (parseInit(&userParser, staticData->passwordFile)) /* start parsing */
    {
        /* cycle by lines of the parameter file */
        while (1)
        {
            char ch;                        /* next character */

            if (parseAtFileEnd(&userParser))    /* EOF */
            {
                parseStop(&userParser);
                break;
            }

            parseSkipSpaces(&userParser);
            if (parseAtLineEnd(&userParser))            /* empty line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            if ((ch = parseGet(&userParser)) == '#')    /* comment  line? */
            {
                parseSkipLine(&userParser);
                continue;
            }
            parseUnget(&userParser, ch);
            parseSkipSpaces(&userParser);
            parseName(&userParser, nextName, sizeof(nextName) - CM_TRAILING_NULL);      /* user name */
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, password, sizeof(password) - CM_TRAILING_NULL, ':');  /* password */
            }
            if (parseDelimiter(&userParser, ':'))
            {
                parseValue(&userParser, userNumText, sizeof(userNumText) - CM_TRAILING_NULL, ':');  /* ID */
            }
            parseSkipLine(&userParser);

            nextRid = atoi(userNumText);

            if ((NQ_INT)rid == nextRid)
            {
                userDeleted = 1;
            }
            else
            {
                syFprintf(tempFile, "%s:%s:%s\n", nextName, password, userNumText);
            }
        }
    }

    fclose(tempFile);
    unlink(staticData->passwordFile);
    rename((const char*)staticData->tempFileName, staticData->passwordFile);
    return userDeleted? TRUE : FALSE;
}

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)*/

#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION

/*
 *====================================================================
 * PURPOSE: modify/create share information in a persistent store
 *--------------------------------------------------------------------
 * PARAMS:  IN share name to change or NULL for a new share
 *          IN share name
 *          IN share path
 *          IN share descriptor
 *
 * RETURNS: TRUE on success, FALSE on failure
 *
 * NOTES:   user-level should return TRUE in the following cases:
 *          1) new share was perisistently stored
 *          2) existing share was peristently modified
 *          3) new share was not persistently stored but it should
 *             be exposed until server shutdown
 *          4) share was not persistently modified but this modification
 *             should be exposed until server shutdown
 *          user-level should return FALSE when a new share should not \
 *          be created or an existing share should not be modified
 *
 *====================================================================
 */

NQ_BOOL
udDefSaveShareInformation(
    const NQ_WCHAR* name,
    const NQ_WCHAR* newName,
    const NQ_WCHAR* newMap,
    const NQ_WCHAR* newDescription
    )
{
    NQ_STATIC char nameA[UD_FS_MAXSHARELEN];
    NQ_STATIC char newNameA[UD_FS_MAXSHARELEN];
    NQ_STATIC char mapA[UD_FS_MAXPATHLEN];
    NQ_STATIC char descriptionA[UD_FS_MAXDESCRIPTIONLEN];
    FILE* defFile;
    FILE* tempFile;
    NQ_STATIC char buffer[256];
    NQ_BOOL result = FALSE;

    setFileNames();
    if (1 == staticData->readingShares)
    {
        goto Exit;
    }

    unlink((const char*)staticData->tempFileName);
    if (NULL != name)
    {
        syUnicodeToAnsi(nameA, name);

        defFile = fopen((const char*)staticData->csConfigFile, "r");
        if (NULL != defFile)
        {
            tempFile = fopen((const char*)staticData->tempFileName, "w+");
            if (NULL == tempFile)
            {
                fclose(defFile);
                goto Exit;
            }

            /* cycle by lines of the parameter file */
            while (NULL != fgets(buffer, sizeof(buffer), defFile))
            {
                if (0 != strncmp(nameA, buffer, strlen(nameA)))
                {
                    /* If this line is not the desired line, copy it to the temp file */
                    fputs(buffer, tempFile);
                }
            }

            fclose(defFile);
            fclose(tempFile);
        }
    }
    else
    {
        if (OK != rename((const char*)staticData->csConfigFile, (const char*)staticData->tempFileName))
        {
            goto Exit;
        }
    }

    tempFile = fopen((const char*)staticData->tempFileName, "a+");
    if (NULL == tempFile)
    {
       goto Exit;
    }

    syUnicodeToAnsi(newNameA, newName);
    syUnicodeToAnsi(mapA, newMap);
    syUnicodeToAnsi(descriptionA, newDescription);
    syFprintf(tempFile, "%s;%s;%s\n", newNameA, mapA, descriptionA);
    fclose(tempFile);

    unlink((const char*)staticData->csConfigFile);
    if (OK != rename((const char*)staticData->tempFileName, (const char*)staticData->csConfigFile))
    {
        goto Exit;
    }

    result = TRUE;

Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: remove share from the persistent store
 *--------------------------------------------------------------------
 * PARAMS:  IN share name
 *
 * RETURNS: TRUE on success, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
udDefRemoveShare(
    const NQ_WCHAR* name
    )
{
    NQ_STATIC char nameA[UD_FS_MAXSHARELEN];
    FILE* defFile;
    FILE* tempFile;
    NQ_STATIC char buffer[256];
    NQ_BOOL result = FALSE;

    setFileNames();
    if (staticData->readingShares)
    {
       goto Exit;
    }

    unlink((const char*)staticData->tempFileName);
    syUnicodeToAnsi(nameA, name);

    defFile = fopen((const char*)staticData->csConfigFile, "r");
    if (NULL == defFile)
    {
        goto Exit;
    }

    tempFile = fopen((const char*)staticData->tempFileName, "w");
    if (NULL == tempFile)
    {
        fclose(defFile);
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (NULL != fgets(buffer, sizeof(buffer), defFile))
    {
        if (0 != strncmp(nameA, buffer, strlen(nameA)))
        {
            fputs(buffer, tempFile);
        }
    }

    fclose(defFile);
    fclose(tempFile);
    unlink((const char*)staticData->csConfigFile);
    if (OK != rename((const char*)staticData->tempFileName, (const char*)staticData->csConfigFile))
    {
        goto Exit;
    }

    result = TRUE;

Exit:
    return result;
}

#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */

#ifdef UD_NQ_INCLUDEEVENTLOG

/*
 *====================================================================
 * PURPOSE: event log function
 *--------------------------------------------------------------------
 * PARAMS:  IN code of NQ module that originated this event
 *          IN event class code
 *          IN event type
 *          IN pointer to the user name string
 *          IN IP address on the second side of the connection
 *          IN zero if the operation has succeeded or error code on failure
 *             for server event this code is the same that will be transmitted
 *             to the client
 *             for an NQ CIFS client event this value is the same that will be
 *             installed as system error
 *          IN pointer to a structure that is filled with event data
 *             actual structure depends on event type
 *
 * RETURNS: None
 *
 * NOTES:   Sample implementation
 *====================================================================
 */

void
udDefEventLog (
    NQ_UINT module,
    NQ_UINT eventClass,
    NQ_UINT type,
    const NQ_WCHAR* userName,
    const NQ_IPADDRESS* pIp,
    NQ_UINT32 status,
    const NQ_BYTE* parameters
    )
{

#define UNIQUECLASS(_mod, _class)  (_mod * 100 + _class)
#define UNIQUEEVENT(_mod, _class, _type)  (_mod * 10000 + _class * 100 + _type)
    const char* modName;
    const char* className;
    const char* typeName;
    NQ_CHAR    ip[CM_IPADDR_MAXLEN];
    NQ_BOOL   isUnicode = TRUE;
    NQ_CHAR * tempName = NULL;

    if (userName != NULL)
    {
        tempName = cmMemoryCloneWStringAsAscii(userName);
    }

    if (pIp != NULL)
    {
        cmIpToAscii(ip, pIp);
    }
    else
    {
        syStrcpy(ip, "<NULL>");
    }

    switch (module)
    {
    case UD_LOG_MODULE_CS:
        modName = "SERVER";
        break;
    case UD_LOG_MODULE_CC:
        modName = "CLIENT";
        break;
    default:
        modName = "UNKNOWN";
    }

    switch (UNIQUECLASS(module, eventClass))
    {
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_GEN):
        className = "GENERIC   ";
        break;
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE):
        className = "FILE      ";
        break;
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_SHARE):
        className = "SHARE     ";
        break;
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_USER):
        className = "USER      ";
        break;
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_CONNECTION):
        className = "CONNECTION";
        break;
    default:
        className = "UNKNOWN   ";
    }

    switch (UNIQUEEVENT(module, eventClass, type))
    {
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_GEN, UD_LOG_GEN_START):
        typeName = "START     ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_GEN, UD_LOG_GEN_STOP):
        typeName = "STOP      ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_GEN, UD_LOG_GEN_NAMECONFLICT):
        typeName = "NAME CONF ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_CREATE):
        typeName = "CREATE    ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_OPEN):
        typeName = "OPEN      ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_CLOSE):
        typeName = "CLOSE     ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_DELETE):
        typeName = "DELETE    ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_RENAME):
        typeName = "RENAME    ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_ATTRIBSET):
        typeName = "ATTRIB SET";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_ATTRIBGET):
        typeName = "ATTRIB GET";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_SIZESET):
        typeName = "SIZE SET  ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_VOLUMEINFO):
        typeName = "VOLUMEINFO";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_QUERYDIRECTORY):
        typeName = "QUERYDIR  ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_SEEK):
        typeName = "SEEK      ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_LOCK):
        typeName = "LOCK      ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE, UD_LOG_FILE_UNLOCK):
        typeName = "UNLOCK    ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_SHARE, UD_LOG_SHARE_CONNECT):
        typeName = "CONNECT   ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_SHARE, UD_LOG_SHARE_DISCONNECT):
        typeName = "DISCONNECT";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_USER, UD_LOG_USER_LOGON):
        typeName = "LOGON     ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_USER, UD_LOG_USER_LOGOFF):
        typeName = "LOGOFF    ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_CONNECTION, UD_LOG_CONNECTION_CONNECT):
        typeName = "CONNECT   ";
        break;
    case UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_CONNECTION, UD_LOG_CONNECTION_DISCONNECT):
        typeName = "DISCONNECT";
        break;
    default:
        typeName = "UNKNOWN   ";
    }

    /*saveEvent(
        "#NQ event in %s class %s type %s user %s IP 0x%08lx error %ld",
        modName,
        className,
        typeName,
        NULL == userName? "<NONE>" : cmWDump(userName),
        NULL == pIp? 0 : CM_IPADDR_GET4(*pIp),
        status
        );*/
    saveEvent(
        "#NQ Event in: %s Class: %s Type: %s User: %s IP: %s",
        modName,
        className,
        typeName,
        NULL == userName? "<NONE>" : isUnicode ? tempName :(NQ_CHAR *)userName,
        ip
        );
    if (status != NQ_SUCCESS)
        saveEvent(" ERROR: %x" , status);
        
    if (userName != NULL)
        cmMemoryFree(tempName);
    switch(status)
    {
    case (SMB_STATUS_USER_SESSION_DELETED):
            saveEvent(" UNEXPECTED DISCONNECT");
            break;
    case ((NQ_UINT32)NQ_ERR_NORESOURCE):
            saveEvent(" OVERFLOW");
            break;
    default:
        if (status != NQ_SUCCESS)
                saveEvent(" ERROR: %x" , status);
        break;
    }

    switch (UNIQUECLASS(module, eventClass))
    {
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_FILE):
        {
            UDFileAccessEvent* event = (UDFileAccessEvent*)parameters;
            NQ_CHAR * tempFileName = NULL;

            if (event == NULL)
            {
                break;
            }


            if (event->fileName != NULL)
            {
                tempFileName = cmMemoryCloneWStringAsAscii(event->fileName);
            }

            saveEvent(" TID: %d" , event->tid == (NQ_UINT32)(-1) ? (NQ_UINT32)(-1) : event->tid);
            saveEvent("  File: %s", NULL == event->fileName? "<NONE>" : (isUnicode && tempFileName)? tempFileName : (NQ_CHAR *)event->fileName);

            if (event->fileName != NULL)
            {
                cmMemoryFree(tempFileName);
                tempFileName = NULL;
            }

            event->before ? saveEvent(" BEFORE") : saveEvent(" AFTER");
            switch(type)
            {
            case UD_LOG_FILE_CREATE:
                saveEvent(" CREATE");
                saveEvent(" RID: '%d' ", event->rid);
                break;
            case UD_LOG_FILE_DELETE:
                saveEvent(" DELETE");
                saveEvent(" RID: '%d' ", event->rid);
                break;
            case UD_LOG_FILE_OPEN:
            {
                const char* access;

                switch (event->access & 0xF)
                {
                case 0:
                    access = "READ";
                    break;
                case 1:
                    access = "WRITE";
                    break;
                case 2:
                    access = "READ/WRITE";
                    break;
                case 3:
                    access = "EXECUTE";
                    break;
                case 0xF:
                    access = "READ";
                    break;
                default:
                    access = "UNKNOWN";
                }
                saveEvent(" Access: %s", access);
                saveEvent(" RID: '%d' ", event->rid);
            }
            break;
            case UD_LOG_FILE_CLOSE:
            {
                const char* access;

                switch (event->access & 0xF)
                {
                case 0:
                    access = "READ";
                    break;
                case 1:
                    access = "WRITE";
                    break;
                case 2:
                    access = "READ/WRITE";
                    break;
                case 3:
                    access = "EXECUTE";
                    break;
                case 0xF:
                    access = "READ";
                    break;
                default:
                    access = "UNKNOWN";
                }
                saveEvent(" Access: %s", access);
                saveEvent(" RID: '%d' ", event->rid);
            }
            break;
           case UD_LOG_FILE_RENAME:
               if (event->newName != NULL)
               {
                    tempFileName = cmMemoryCloneWStringAsAscii(event->newName);
               }

               saveEvent(" New: %s", (isUnicode && tempFileName) ? tempFileName : (NQ_CHAR *)event->newName);
               saveEvent(" RID: '%d' ", event->rid);

                if (event->newName != NULL)
                {
                    cmMemoryFree(tempFileName);
                    tempFileName = NULL;
                }

               break;
           case UD_LOG_FILE_ATTRIBSET:
               {
                   saveEvent(" Mode: %08lx", event->access);
                   saveEvent(" RID: '%d' ", event->rid);
               }
               break;
           default:
                saveEvent(" RID: '%d' ", event->rid);
                break;
            }
        }
        break;
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_SHARE):
        {
            UDShareAccessEvent* event = (UDShareAccessEvent*)parameters;
            NQ_CHAR * tempShareName = NULL;

            if (event == NULL)
            {
                break;
            }

            tempShareName = cmMemoryCloneWStringAsAscii(event->shareName);

            saveEvent(" Share: '%s'", isUnicode ? tempShareName : (NQ_CHAR *)event->shareName);
            saveEvent(" Tid: %d" , event->tid );

            cmMemoryFree(tempShareName);

            if (event->ipc)
            {
                saveEvent(" IPC");
            }
            if (event->printQueue)
            {
                saveEvent(" PRINT QUEUE");
            }
            saveEvent(" RID: '%d' ", event->rid);
        }
        break;
    case UNIQUECLASS(UD_LOG_MODULE_CS, UD_LOG_CLASS_USER):
    {
        UDUserAccessEvent * event = (UDUserAccessEvent *)parameters;

        if (event == NULL)
        {
            break;
        }

        saveEvent(" RID: '%d' ", event->rid);
    }
    default:
        className = "UNKNOWN";
    }
    saveEvent("\n");
    if (UNIQUEEVENT(UD_LOG_MODULE_CS, UD_LOG_CLASS_GEN, UD_LOG_GEN_STOP) ==
        UNIQUEEVENT(module, eventClass, type)
       )
       saveEvent(NULL);
}

#endif /* UD_NQ_INCLUDEEVENTLOG */

/*
 *====================================================================
 * PURPOSE: read and parse common configuration file
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE on success,
 *          FALSE on error
 *
 * NOTES:   Opens the file, parses it and stores parameter values if
 *          those parameters were found
 *====================================================================
 */

static NQ_BOOL
parseCommonConfig(
    void
    )
{
    ParseContext    parser;            /* parser context */
    NQ_BOOL         result = TRUE;     /* result - TRUE on success, FALSE otherwise */

    setFileNames();
    if (!parseInit(&parser, staticData->cmConfigFile)) /* start parsing */
    {
        result = FALSE;
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&parser))      /* EOF */
        {
            staticData->commonConfigFlag = 1;
            goto Exit;
        }

        parseSkipSpaces(&parser);

        if ((ch = parseGet(&parser)) != '#')    /* comment  line? */
        {
            char name[NQ_CONFIG_FILE_NAME_LEN];    /* buffer for name */
            char value[NQ_MAX_ENCRYPTED_PASS_SIZE * 2 + NQ_ENCRYPTED_POSTFIX_LEN];  /* multiples by 2 because every byte is represented by 2 hexadecimal digits */

            parseUnget(&parser, ch);
            parseSkipSpaces(&parser);
            parseName(&parser, name, sizeof(name));              /* parameter name */

            parseSkipSpaces(&parser);

            if (parseDelimiter(&parser, '='))
            {
                parseSkipSpaces(&parser);
                if (!strncmp(name, "DNS", strlen("DNS")) || !strncmp(name, "LOGBASEFOLDER", strlen("LOGBASEFOLDER")) || !strncmp(name, "INTERNALCAPTUREBASEFOLDER", strlen("INTERNALCAPTUREBASEFOLDER")) || !strncmp(name, "EVENTLOGBASEFOLDER", strlen("EVENTLOGBASEFOLDER")) || !strncmp(name, "PASSWORD", strlen("PASSWORD")))
                {
                    parseValue(&parser, value, sizeof(value), ' '); /* IP version 6 DNS value contain ':', hence different delimiter was picked */
                }
                else
                {
                    parseValue(&parser, value, sizeof(value), ':');  /* parameter value - no delimiter */
                }

                if (!strncmp(name, "WINS", strlen("WINS")))
                {
                    strcpy(staticData->winsAddresses, value);
                }
                else if (!strncmp(name, "SCOPE_ID", strlen("SCOPE_ID")))
                {
                    strcpy(staticData->scopeId, value);
                    staticData->scopeIdPtr = staticData->scopeId;
                }

                if (!strncmp(name, "DOMAIN", strlen("DOMAIN")))
                {
                    /* value of type DOMAIN_NAME[:D] */
                    strcpy(staticData->domainNameOfServer, value);
                    /* same for client */
                    strcpy(staticData->domainNameOfClient, value);

                    parseSkipSpaces(&parser);
                    staticData->isWorkgroupName = TRUE;

                    if (parseDelimiter(&parser, ':'))
                    {
                        parseSkipSpaces(&parser);
                        parseValue(&parser, value, 2, (char)0);
                        staticData->isWorkgroupName = (value[0] != 'D');
                    }
                }

                if (!strncmp(name, "DNS", strlen("DNS")))
                {
                    strcpy(staticData->dnsAddresses, value);
                }

                if (!strncmp(name, "USERNAME", strlen("USERNAME")))
                {
                    strcpy(staticData->userName, value);
                }

                if (!strncmp(name, "PASSWORD", strlen("PASSWORD")))
                {
                    char fullPassString[NQ_MAX_ENCRYPTED_PASS_SIZE];
                    NQ_UINT32 receivedStrLen = (NQ_UINT32)strlen(value);

                    /* check that the last characters are ':' and 'E' - for encrypted */
                    if ((receivedStrLen > 2) && (':' == value[receivedStrLen - NQ_ENCRYPTED_POSTFIX_LEN]) && ('E' == value[receivedStrLen - 1]))
                    {
                        NQ_BYTE passwordSize;
                        char checksum = 0;
                        NQ_INT  i;
                        NQ_UINT32 encryptedStringLen = (receivedStrLen - NQ_ENCRYPTED_POSTFIX_LEN) / 2;

                        /* check that the length of encrypted string longer than the salts */
                        if ((encryptedStringLen <=  NQ_PWD_SALT_LEN * NQ_NUM_OF_SALTS) || (0 != (receivedStrLen % 2)))
                        {
                            syPrintf("Error during password decryption, no password has been set\n");
                            staticData->password[0] = '\0';
                        }
                        else
                        {
                            /* convert hexadecimal representation of a encrypted password string into ASCI */
                            convertHex2Ascii(value, (NQ_INT)(encryptedStringLen));

                            /* decrypt password */
                            aesGcmEnc(key, nonce, (NQ_BYTE *)value, encryptedStringLen, (NQ_BYTE *)fullPassString);

                            /* get password length from last entry of the first salt */
                            passwordSize = (NQ_BYTE)fullPassString[NQ_PWD_SALT_LEN - 1];

                            /* calculate checksum */
                            for (i = 0; i < (encryptedStringLen - 1); i++)
                            {
                                checksum = (char)(checksum ^ fullPassString[i]);
                            }

                            /* compare computed checksum to expected checksum and check that the length of dencrypted string longer than the salts plus password length */
                            if ((checksum != fullPassString[i]) || ((NQ_PWD_SALT_LEN * NQ_NUM_OF_SALTS + passwordSize) >= encryptedStringLen))
                            {
                                syPrintf("Error during password decryption, no password has been set\n");
                                staticData->password[0] = '\0';
                            }
                            else
                            {
#ifdef UDCONFIG_DEBUG
                                if ((encryptedStringLen != (passwordSize + strlen(staticData->userName) + NQ_PWD_SALT_LEN * NQ_NUM_OF_SALTS)) || (0 != strncmp(staticData->userName, &fullPassString[NQ_PWD_SALT_LEN * 2 + passwordSize], strlen(staticData->userName))))
                                {
                                    syFprintf(stderr, "[%s:%d][%s()] user name mismatch, you possibly supplied another user's password (user name from encrypted string is not match to user name from common configuration file) %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
                                }
#endif /* UDCONFIG_DEBUG */

                                /* copy decrypted password string to static variable */
                                syStrncpy(staticData->password, &fullPassString[NQ_PWD_SALT_LEN], passwordSize);
                                staticData->password[passwordSize] = '\0';
                            }
                        }
                    }
                    else
                    {
                        syStrcpy(staticData->password, value);
                    }
                }

#ifdef UD_NQ_INCLUDESMBCAPTURE
                if (!strncmp(name, "INTERNALCAPTURE", strlen("INTERNALCAPTURE")))
                {
                    if (!strncmp(value, "TRUE", strlen("TRUE")))
                    {
                        staticData->useInternalCapture = 1;
                    }
                    else if (!strncmp(value, "FALSE", strlen("FALSE")))
                    {
                        staticData->useInternalCapture = 0;
                    }
                }

                if (!strncmp(name, "INTERNALCAPTUREBASEFOLDER", strlen("INTERNALCAPTUREBASEFOLDER")))
                {
                    if (strlen(value) > 0)
                    {
                        strncpy(staticData->captureFileBaseFolder, value, sizeof(staticData->captureFileBaseFolder) - CM_TRAILING_NULL);
                        staticData->captureFileBaseFolder[sizeof(staticData->captureFileBaseFolder) - 1] = 0;
                    }
                }
#endif /* UD_NQ_INCLUDESMBCAPTURE */

                if (!strncmp(name, "INTERNALTRACE", strlen("INTERNALTRACE")))
                {
                    if (!strncmp(value, "TRUE", strlen("TRUE")))
                    {
                        staticData->useInternalTrace = 1;
                    }
                    else if (!strncmp(value, "FALSE", strlen("FALSE")))
                    {
                        staticData->useInternalTrace = 0;
                    }
                }

                if (!strncmp(name, "HOSTNAME", strlen("HOSTNAME")))
                {
                    if (strlen(value) > 0)
                    {
                        strncpy(staticData->hostname, value, sizeof(staticData->hostname) - 1);
                        staticData->hostname[sizeof(staticData->hostname) - 1] = 0;
                    }
                }

                if (!strncmp(name, "LOGBASEFOLDER", strlen("LOGBASEFOLDER")))
                {
                    if (strlen(value) > 0)
                    {
                        strncpy(staticData->logFileBaseFolder, value, sizeof(staticData->logFileBaseFolder) - CM_TRAILING_NULL);
                        staticData->logFileBaseFolder[sizeof(staticData->logFileBaseFolder) - 1] = 0;
                    }
                }

#ifdef UD_NQ_INCLUDEEVENTLOG
                if (!strncmp(name, "EVENTLOGBASEFOLDER", strlen("EVENTLOGBASEFOLDER")))
                {
                    NQ_UINT len = (NQ_UINT)syStrlen(value);

                    staticData->eventLogBaseFolderTooLong = 0;
                    if (len > 0)
                    {
                        if (len + syStrlen(UD_CM_EVENTLOG_FILENAME) + sizeof(SY_PATHSEPARATOR) < sizeof(staticData->eventLogFile))
                        {
                            syStrcpy(staticData->eventLogFile, value);
                            if (staticData->eventLogFile[len - 1] != SY_PATHSEPARATOR)
                            {
                                staticData->eventLogFile[len] = SY_PATHSEPARATOR;
                                staticData->eventLogFile[len + 1] = '\0';
                                len++;
                            }

                            syStrncat(staticData->eventLogFile, UD_CM_EVENTLOG_FILENAME, sizeof(staticData->eventLogFile) - len);
                            len += (NQ_UINT)sizeof(UD_CM_EVENTLOG_FILENAME);
                            staticData->eventLogFile[len] = '\0';
                        }
                        else
                        {
                            staticData->eventLogFile[0] = '\0';
                            staticData->eventLogBaseFolderTooLong = 1;
                        }
                    }
                    else if(syStrlen(UD_CM_EVENTLOG_FILENAME) < sizeof(staticData->eventLogFile))
                    {
                        syStrncpy(staticData->eventLogFile, UD_CM_EVENTLOG_FILENAME, sizeof(staticData->eventLogFile));
                        staticData->eventLogFile[syStrlen(UD_CM_EVENTLOG_FILENAME)] = '\0';
                    }
                    else
                    {
                        staticData->eventLogFile[0] = '\0';
                    }
                }
#endif /* UD_NQ_INCLUDEEVENTLOG */
            }
        }

        parseSkipLine(&parser);
    }

Exit:
    return result;
}


/*
 *====================================================================
 * PURPOSE: Prepare file names for future use
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:   This function does its work only once by using a singleton
 *====================================================================
 */

static void
setFileNames(
    void
    )
{
    if (!staticData->fileNamesReady)
    {
        strcpy(staticData->cmConfigFile, NQ_CONFIGPATH);
        if (*(staticData->cmConfigFile + strlen(staticData->cmConfigFile) - 1) != '/')
        {
            strcat(staticData->cmConfigFile, "/");
        }
        
        strcat(staticData->cmConfigFile, "cm_cfg.txt");
        strcpy(staticData->csConfigFile, NQ_CONFIGPATH);
        if (*(staticData->csConfigFile + strlen(staticData->csConfigFile) - 1) != '/')
        {
            strcat(staticData->csConfigFile, "/");
        }
        
        strcat(staticData->csConfigFile, "cs_cfg.txt");
        strcpy(staticData->ccConfigFile, NQ_CONFIGPATH);
        if (*(staticData->ccConfigFile + strlen(staticData->ccConfigFile) - 1) != '/')
        {
            strcat(staticData->ccConfigFile, "/");
        }
        
        strcat(staticData->ccConfigFile, "cc_cfg.txt");
        strcpy(staticData->passwordFile, NQ_CONFIGPATH);
        if (*(staticData->passwordFile + strlen(staticData->passwordFile) - 1) != '/')
        {
            strcat(staticData->passwordFile, "/");
        }
        
        strcat(staticData->passwordFile, "pwd_list.txt");
        strcpy(staticData->shareSdFile, NQ_CONFIGPATH);
        if (*(staticData->shareSdFile + strlen(staticData->shareSdFile) - 1) != '/')
        {
            strcat(staticData->shareSdFile, "/");
        }

        strcat(staticData->shareSdFile, "share_sd.txt");
        strcpy(staticData->tempFileName, NQ_CONFIGPATH);
        if (*(staticData->tempFileName + strlen(staticData->tempFileName) - 1) != '/')
        {
            strcat(staticData->tempFileName, "/");
        }

        strcat(staticData->tempFileName, "__temp__");
#ifdef UD_NQ_INCLUDEEVENTLOG
#ifdef UD_CM_EVENTLOG_BASEFOLDER
        {
            NQ_UINT len = (NQ_UINT)syStrlen(UD_CM_EVENTLOG_BASEFOLDER);

            staticData->eventLogBaseFolderTooLong = 0;
            if (len > 0)
            {
                if (len + syStrlen(UD_CM_EVENTLOG_FILENAME) + sizeof(SY_PATHSEPARATOR) < sizeof(staticData->eventLogFile))
                {
                    syStrcpy(staticData->eventLogFile, UD_CM_EVENTLOG_BASEFOLDER);
                    if (staticData->eventLogFile[len - 1] != SY_PATHSEPARATOR)
                    {
                        staticData->eventLogFile[len] = SY_PATHSEPARATOR;
                        staticData->eventLogFile[len + 1] = '\0';
                        len++;
                    }

                    syStrncat(staticData->eventLogFile, UD_CM_EVENTLOG_FILENAME, sizeof(staticData->eventLogFile) - len);
                    len += (NQ_UINT)sizeof(UD_CM_EVENTLOG_FILENAME);
                    staticData->eventLogFile[len] = '\0';
                }
                else
                {
                    staticData->eventLogBaseFolderTooLong = 1;
                    staticData->eventLogFile[0] = '\0';
                }
            }
            else if(syStrlen(UD_CM_EVENTLOG_FILENAME) < sizeof(staticData->eventLogFile))
            {
                syStrncpy(staticData->eventLogFile, UD_CM_EVENTLOG_FILENAME, sizeof(staticData->eventLogFile));
                staticData->eventLogFile[syStrlen(UD_CM_EVENTLOG_FILENAME)] = '\0';
            }
            else
            {
                staticData->eventLogFile[0] = '\0';
            }
        }
#else /* UD_CM_EVENTLOG_BASEFOLDER */
        staticData->eventLogBaseFolderTooLong = 0;
        if(syStrlen(UD_CM_EVENTLOG_FILENAME) < sizeof(staticData->eventLogFile))
        {
            syStrncpy(staticData->eventLogFile, UD_CM_EVENTLOG_FILENAME, sizeof(staticData->eventLogFile));
            staticData->eventLogFile[syStrlen(UD_CM_EVENTLOG_FILENAME)] = '\0';
        }
        else
        {
            staticData->eventLogFile[0] = '\0';
        }
#endif /* UD_CM_EVENTLOG_BASEFOLDER */
#endif /* UD_NQ_INCLUDEEVENTLOG */

        staticData->fileNamesReady = 1;
    }
}

/*
 *====================================================================
 * PURPOSE: Save/print logged event
 *--------------------------------------------------------------------
 * PARAMS:  IN print format
 *          VARARG parameters
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

#ifdef UD_NQ_INCLUDEEVENTLOG
static void
saveEvent(
    char* format,
    ...
    )
{
    va_list va;                         /* parameter list */

    if (!staticData->logHandleFlag)
    {
        if (!staticData->commonConfigFlag)
        {
            parseCommonConfig();
        }

        if (1 == staticData->eventLogBaseFolderTooLong)
        {
            syPrintf("Could not create event log file. The path of event log file base folder is too long.\n");
            staticData->logHandleFlag = 1;
            staticData->logHandle = NULL;
            goto Exit;
        }

        staticData->logHandle = fopen(staticData->eventLogFile, "w");
        if (NULL == staticData->logHandle)
        {
            syPrintf("Could not create event log file:%s, error: %d\n", staticData->eventLogFile, syGetLastError());
        }

        staticData->logHandleFlag = 1;
    }

    if (NULL == staticData->logHandle)
    {
        goto Exit;
    }

    if (NULL == format)
    {
        staticData->logHandleFlag = 0;
        fclose(staticData->logHandle);
        goto Exit;
    }

    va_start(va, format);
    vfprintf(staticData->logHandle, format, va);
    fflush(staticData->logHandle);

Exit:
    return;
}
#endif /* UD_NQ_INCLUDEEVENTLOG */

/*
 *====================================================================
 * PURPOSE: Convert HEX representation of a password string to text
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT HEX string. This buffer is used for conversion
 *          IN output string length
 *
 * RETURNS: TRUE for success FALSE for error
 *
 * NOTES:   HEX representation is an ASCII representation where each ACSII character
 *          is represented as its HEX equivalent
 *====================================================================
 */

static NQ_BOOL
convertHex2Ascii(
    char* text,
    NQ_INT outputStrLen
    )
{
    int  src;           /* index in the source (hex) string */
    int  dst;           /* index in the target (ascii) string*/
    unsigned char tmp;           /* temporary holds the half-character being converted */


    /* we use the same buffer for the destination string
       the size password in HEX should be of exact length */

    for ( src = 0, dst = 0; text[src] > 0 && dst < outputStrLen; dst++ )
    {
        /* check if next character is a hex numbers */

        tmp = (unsigned char)toupper((int)text[src]);
        src++;

        if ( !(   ((tmp >= '0') && (tmp <= '9') )
               || ((tmp >= 'A') && (tmp <= 'F') )
              )
           )
        {
            return FALSE;
        }

        /* get the real number of the high hex character */

        tmp = (unsigned char)(tmp - (unsigned char)((tmp < 'A')? 0x30: 0x37));
        text[dst] = (char)(tmp << 4);   /* high half-octet */

        /* check if the second character is a hex numbers */

        tmp = (unsigned char)toupper((int)text[src]);
        src++;

        if ( !(   ((tmp >= '0') && (tmp <= '9') )
               || ((tmp >= 'A') && (tmp <= 'F') )
              )
           )
        {
            return FALSE;
        }

        /* get the real number of the high hex character */

        tmp = (unsigned char)(tmp - (unsigned char)((tmp < 'A')? 0x30: 0x37));
        text[dst] = (char)(text[dst] + tmp);       /* low half-octet */
    }

    text[dst] = '\0';

    return TRUE;
}

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

/*
 *====================================================================
 * PURPOSE: Convert binary representation of a password to text
 *--------------------------------------------------------------------
 * PARAMS:  IN 32-byte binary password
 *             OUT 64-byte buffer
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

static void
convertPasswordToText(
    const NQ_BYTE* password,
    NQ_CHAR* buffer
    )
{
    int i;

    for (i = 0; i < 32; i++)
    {
        *buffer = (NQ_CHAR)((((*password)/16) < 10)? (char)((*password)/16 + '0') : (char)((*password)/16 - 10 + 'a'));
        buffer++;
        *buffer = (NQ_CHAR)((((*password)%16) < 10)? (char)((*password)%16 + '0') : (char)((*password)%16 - 10 + 'a'));
        buffer++;
        password++;
    }
}

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

#ifdef UD_CC_INCLUDEDOMAINMEMBERSHIP

/* check whether secret is for domain other than default, already loaded */
static NQ_BOOL isNewSecretFileName()
{
    NQ_CHAR secretFilePath[UD_FS_MAXPATHLEN];   /* secret file path */
    NQ_CHAR *secretFileName;                    /* secret file name */
    DIR* dir;
    struct dirent* de;
    NQ_BOOL isNewSecret = TRUE;

    /* format: domainDNS.secret.domainNB */
    syStrcpy(secretFilePath, NQ_CONFIGPATH);
    if (*(secretFilePath + syStrlen(secretFilePath) - 1) != '/')
    {
        syStrcat(secretFilePath, "/");
    }
    secretFileName = &secretFilePath[syStrlen(secretFilePath)];

    dir = opendir(secretFilePath);
    if (NULL != dir)
    {
        syStrcpy(secretFileName, cmGetFullDomainName() ? cmGetFullDomainName() : cmNetBiosGetDomain()->name);
        cmAStrupr(secretFileName);
        syStrcat(secretFileName, ".secret.");

        /* find domainDNS.secret.* file */
        while (NULL != (de = readdir(dir)))
        {
            if (syStrncmp(secretFileName, de->d_name, syStrlen(secretFileName)) == 0)
            {
                syStrcpy(staticData->secretFileName, de->d_name);
                syStrcpy(secretFileName, de->d_name);
                syStrcpy(staticData->secretFilePath, secretFilePath);
                isNewSecret = FALSE;
                break;
            }
        }
        closedir(dir);
    }

    if (isNewSecret)
    {
        NQ_UINT isSlashAppended = 0;

        /* format: domainDNS.secret.domainNB */
        syStrcpy(secretFileName, cmGetFullDomainName() ? cmGetFullDomainName() : cmNetBiosGetDomain()->name);
        cmAStrupr(secretFileName);
        syStrcat(secretFileName, ".secret.");
        syStrcat(secretFileName, cmNetBiosGetDomainAuth()->name);
        syStrcpy(staticData->secretFileName, secretFileName);

        syStrcpy(staticData->secretFilePath, NQ_CONFIGPATH);
        if (*(staticData->secretFilePath + strlen(staticData->secretFilePath) - 1) != '/')
        {
            syStrcat(staticData->secretFilePath, "/");
            isSlashAppended = 1;
        }

        /* subtract (sizeof(NQ_CONFIGPATH) + isSlashAppended) because staticData->secretFilePath already contain string of length (sizeof(NQ_CONFIGPATH) + isSlashAppended) */
        syStrncat(staticData->secretFilePath, staticData->secretFileName, sizeof(staticData->secretFilePath) - sizeof(NQ_CONFIGPATH) - isSlashAppended - CM_TRAILING_NULL);
        staticData->secretFilePath[sizeof(staticData->secretFilePath) - 1] = '\0';
    }

    return isNewSecret;
}

NQ_BOOL
udDefGetComputerSecretByDomain(
    NQ_BYTE *secret,
    const NQ_WCHAR *domainDNS,
    NQ_WCHAR *domainNB
    )
{
    NQ_CHAR secretFilePath[UD_FS_MAXPATHLEN];
    NQ_CHAR *secretFileName, *p;
    int file = NQ_FAIL;
    DIR* dir;
    struct dirent* de;
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "secret:%p domainDNS:%s domainNB:%p", secret, domainDNS ? cmWDump(domainDNS) : "", domainNB);

    if (NULL == secret || NULL == domainDNS)
    {
        goto Exit;
    }

    /* format: domainDNS.secret.domainNB */
    syStrcpy(secretFilePath, NQ_CONFIGPATH);
    dir = opendir(secretFilePath);
    if (NULL == dir)
    {
        goto Exit;
    }

    if (*(secretFilePath + syStrlen(secretFilePath) - 1) != '/')
    {
        syStrcat(secretFilePath, "/");
    }
    secretFileName = &secretFilePath[syStrlen(secretFilePath)];
    p = secretFilePath + syStrlen(secretFilePath);
    cmUnicodeToAnsi(p, domainDNS);
    cmAStrupr(p);
    syStrcat(secretFilePath, ".secret.");

    /* find domainDNS.secret.* file */
    while (NULL != (de = readdir(dir)))
    {
        if (syStrncmp(secretFileName, de->d_name, syStrlen(secretFileName)) == 0)
        {
            result = TRUE;
            syStrcpy(secretFileName, de->d_name);
            break;
        }
    }
    closedir(dir);

    if (!result)
    {
        goto Exit;
    }
    if (domainNB)
    {
        p = syStrrchr(secretFileName, '.');
        cmAStrupr(++p);
        cmAnsiToUnicode(domainNB, p);
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "domainNB: %s", cmWDump(domainNB));
    }

    /* open existing */
    file = open(secretFilePath, O_RDWR, 0777);
    if (NQ_FAIL == file)
    {
        result = FALSE;
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to open secret file:%s", secretFilePath);
#ifdef UDCONFIG_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] Failed to open secret file:%s %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, secretFilePath, errno, strerror(errno));
#endif /* UDCONFIG_DEBUG */
        goto Exit;
    }

    if (read(file, (void *)secret, SECRET_LENGTH) != SECRET_LENGTH)
    {
        result = FALSE;
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to read secret file:%s", secretFilePath);
#ifdef UDCONFIG_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] Failed to read secret file:%s %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, secretFilePath, errno, strerror(errno));
#endif /* UDCONFIG_DEBUG */
        goto Exit;
    }
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "secret", secret, SECRET_LENGTH);
    result = TRUE;

Exit:
    if (file != NQ_FAIL)
        close(file);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL
udDefGetComputerSecret(
    NQ_BYTE **secret
    )
{
    int file = NQ_FAIL;
    NQ_BOOL result = FALSE;
    NQ_BOOL doLoad = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "secret:%p", secret);

    /* take care of case when default domain in newly joined, but may be not loaded yet */
    doLoad = isNewSecretFileName() ? TRUE : !staticData->isSecretFileLoaded;
    if (doLoad)
    {
        setFileNames();

        file = open(staticData->secretFilePath, O_RDWR, 0777);
        if (file == NQ_FAIL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to open secret file: %s", staticData->secretFilePath);
            goto Exit;
        }
        if (read(file, staticData->secret, sizeof(staticData->secret)) != sizeof(staticData->secret))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to read secret file: %s", staticData->secretFilePath);
            goto Exit;
        }
        /* load secret file once per domain */
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Secret loaded from file: %s", staticData->secretFilePath);
        staticData->isSecretFileLoaded = TRUE;
        LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "secret", staticData->secret, SECRET_LENGTH);
    }

    if (staticData->isSecretFileLoaded && secret)
    {
        *secret = (NQ_BYTE *)&staticData->secret;
        LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "secret", *secret, SECRET_LENGTH);
    }

    result = staticData->isSecretFileLoaded;

Exit:
    if (file != NQ_FAIL)
        close(file);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}


void
udDefSetComputerSecretByDomain(
    const NQ_BYTE *secret,
    const NQ_WCHAR *domainDNS,
    const NQ_WCHAR *domainNB
    )
{
    int file;
    NQ_CHAR secretFileName[UD_FS_MAXPATHLEN];
    NQ_CHAR *p = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "secret:%p domainDNS:%s", secret, cmWDump(domainDNS));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "domainNB:%s", cmWDump(domainNB));

    /* format: domainDNS.secret.domainNB */
    syStrcpy(secretFileName, NQ_CONFIGPATH);
    if (*(secretFileName + syStrlen(secretFileName) - 1) != '/')
    {
        syStrcat(secretFileName, "/");
    }
    p = secretFileName + syStrlen(secretFileName);
    cmUnicodeToAnsi(p, domainDNS);
    cmAStrupr(p);
    syStrcat(secretFileName, ".secret.");
    if (domainNB)
    {
        p = secretFileName + syStrlen(secretFileName);
        cmUnicodeToAnsi(p, domainNB);
        cmAStrupr(p);
    }

    /* overwrite existing */
    file = open(secretFileName, O_RDWR | O_CREAT, 0777);
    if (NQ_FAIL == file)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to create secret file:%s", secretFileName);
#ifdef UDCONFIG_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] Failed to create secret file:%s %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, secretFileName, errno, strerror(errno));
#endif /* UDCONFIG_DEBUG */
        goto Exit;
    }

    if (write(file, (const char *)secret, SECRET_LENGTH) != SECRET_LENGTH)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to write computer secret into:%s", secretFileName);
        close(file);
        goto Exit;
    }

    close(file);
#ifdef UDCONFIG_DEBUG
    syFprintf(stderr, "[%s:%d][%s()] Created secret file:%s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, secretFileName);
#endif /* UDCONFIG_DEBUG */
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void
udDefSetComputerSecret(
    NQ_BYTE *secret
    )
{
    int file;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "secret:%p", secret);

    if (!secret)
        goto Exit;

    syMemcpy(staticData->secret, secret, sizeof(staticData->secret));
    staticData->isSecretFileLoaded = TRUE;

    file = open(staticData->secretFilePath, O_RDWR | O_CREAT, 0777);
    if (NQ_FAIL == file)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to create secret file:%s", staticData->secretFilePath);
#ifdef UDCONFIG_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] Failed to create secret file:%s %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, staticData->secretFilePath, errno, strerror(errno));
#endif /* UDCONFIG_DEBUG */
        goto Exit;
    }

    if (write(file, staticData->secret, sizeof(staticData->secret)) != sizeof(staticData->secret))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to write computer secret into:%s", staticData->secret);
        close(file);
        goto Exit;
    }

    close(file);

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Created secret file:%s", staticData->secretFilePath);
#ifdef UDCONFIG_DEBUG
    syFprintf(stderr, "[%s:%d][%s()] Created secret file:%s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, staticData->secretFilePath);
#endif /* UDCONFIG_DEBUG */
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

#endif /* UD_CC_INCLUDEDOMAINMEMBERSHIP */

void
udDefGetHostName(
    NQ_CHAR* buffer,
    NQ_UINT length
    )
{
    if (!staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }
    strncpy(buffer, staticData->hostname, length > sizeof(staticData->hostname) ? sizeof(staticData->hostname) : length);
}


#ifdef UD_NQ_INCLUDESMB1
/*
 *====================================================================
 * PURPOSE: whether SMB1 dialect is supported for client
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE
 *          FALSE
 *
 * NOTES:  Returns FALSE only when the parameter was present and equal to
 *         FALSE in client config file. If the parameter does not present in
 *         client config file it is considered as TRUE.
 *====================================================================
 */
NQ_BOOL
udDefGetClientSMB1Support(void)
{
    ParseContext parser;
    NQ_BOOL support = TRUE;

    setFileNames();
    if (!parseInit(&parser, staticData->ccConfigFile)) /* start parsing */
    {
        parseStop(&staticData->mountParser);
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&parser))      /* EOF */
        {
            goto Exit;
        }

        parseSkipSpaces(&parser);

        if ((ch = parseGet(&parser)) != '#')       /* comment  line? */
        {
            char name[NQ_CONFIG_FILE_NAME_LEN];    /* buffer for name */
            char value[NQ_CONFIG_FILE_VALUE_LEN];   /* buffer for value */

            parseUnget(&parser, ch);
            parseSkipSpaces(&parser);
            parseName(&parser, name, sizeof(name));                  /* parameter name */
            parseSkipSpaces(&parser);

            if (parseDelimiter(&parser, '='))
            {
                parseSkipSpaces(&parser);
                parseValue(&parser, value, sizeof(value), (char)0);  /* parameter value - no delimiter */

                if (syStrncmp(name, "SUPPORTSMB1", strlen("SUPPORTSMB1")) == 0)
                {
                    if (!strncmp(value, "TRUE", strlen("TRUE")))
                    {
                        support = TRUE;
                        goto Exit;
                    }
                    else if (!strncmp(value, "FALSE", strlen("FALSE")))
                    {
                        support = FALSE;
                        goto Exit;
                    }

                    parseStop(&parser);
                    goto Exit;
                }
            }
        }
        parseSkipLine(&parser);
    }

Exit:
    return support;
}


/*
 *====================================================================
 * PURPOSE: Activates or deactivates of SMB1 dialect for server.
 *--------------------------------------------------------------------
 * PARAMS:  support - TRUE to enable SMB1 dialect for server
 *                    FLASE to disable SMB1 dialect for server
 *
 * RETURNS:
 *          None
 *
 * NOTES:
 *====================================================================
 */
NQ_STATUS
udDefSetServerSMB1Support(
        NQ_BOOL isSupport
    )
{
    NQ_STATUS result = NQ_FAIL;
    /* if SMB2 isn't enabled SMB1 can't be disabled */
#ifdef UD_NQ_INCLUDESMB2
    result = NQ_SUCCESS;
    staticData->supportSMB1ForServer = isSupport;
#endif /* UD_NQ_INCLUDESMB2 */
    return result;
}
#endif /* UD_NQ_INCLUDESMB1 */


/*
 *====================================================================
 * PURPOSE: whether SMB1 dialect is supported for server
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE
 *          FALSE
 *
 * NOTES:  Returns FALSE only when the parameter was present and equal to
 *         FALSE in server config file. If the parameter does not present in
 *         server config file it is considered as TRUE.
 *====================================================================
 */
NQ_BOOL
udDefGetServerSMB1Support(
        void
        )
{
    NQ_BOOL result = TRUE;

#ifndef UD_NQ_INCLUDESMB1
    result = FALSE;
    goto Exit;
#else /* UD_NQ_INCLUDESMB1 */
    ParseContext parser;

    if (staticData->supportSMB1ForServer != NQ_NOT_INITIALIZED)
    {
        result = staticData->supportSMB1ForServer;
        goto Exit;
    }

    setFileNames();
    if (!parseInit(&parser, staticData->csConfigFile)) /* start parsing */
    {
        parseStop(&staticData->shareParser);
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&parser))      /* EOF */
        {
            staticData->supportSMB1ForServer = 1;
            result = TRUE;
            goto Exit;
        }

        parseSkipSpaces(&parser);

        if ((ch = parseGet(&parser)) != '#')       /* comment  line? */
        {
            char name[NQ_CONFIG_FILE_NAME_LEN];    /* buffer for name */
            char value[NQ_CONFIG_FILE_VALUE_LEN];   /* buffer for value */

            parseUnget(&parser, ch);
            parseSkipSpaces(&parser);
            parseName(&parser, name, sizeof(name));                  /* parameter name */
            parseSkipSpaces(&parser);

            if (parseDelimiter(&parser, '='))
            {
                parseSkipSpaces(&parser);
                parseValue(&parser, value, sizeof(value), (char)0);  /* parameter value - no delimiter */

                if (syStrncmp(name, "SUPPORTSMB1", strlen("SUPPORTSMB1")) == 0)
                {
                    if (syStrncmp(value, "FALSE", strlen("FALSE")) == 0)
                    {
                        staticData->supportSMB1ForServer = 0;
                        result = FALSE;
                    }
                    else if (syStrncmp(value, "TRUE", strlen("TRUE")) == 0)
                    {
                        staticData->supportSMB1ForServer = 1;
                        result = TRUE;
                    }

                    parseStop(&parser);
                    goto Exit;
                }
            }
        }
        parseSkipLine(&parser);
    }
#endif /* UD_NQ_INCLUDESMB1 */

Exit:
    return result;
}

/*
*====================================================================
* PURPOSE: get base folder path of log file
*--------------------------------------------------------------------
* PARAMS:  buffer - buffer for the path
*          size   - buffer size
*
* NOTES:
*====================================================================
*/
void
udDefGetLogFileBaseFolder(
    NQ_CHAR* buffer,
    NQ_UINT  size
    )
{
    if (!staticData->commonConfigFlag)
    {
        parseCommonConfig();
    }

    strncpy(buffer, staticData->logFileBaseFolder, size > sizeof(staticData->logFileBaseFolder) ? sizeof(staticData->logFileBaseFolder) : size);
}

/*
 *====================================================================
 * PURPOSE: get disk size limit
 *--------------------------------------------------------------------
 * PARAMS:  size - pointer to size limit
 *
 * RETURNS:
 *         None
 *
 *====================================================================
 */
void
udDefGetMaxDiskSize(
        NQ_UINT32 *size
        )
{
    ParseContext parser;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "size :%p", size);

    setFileNames();
    if (staticData->maxDiskSizeServer != NQ_NOT_INITIALIZED)
    {
        if (staticData->maxDiskSizeServer != NQ_NOT_PRESENT)
        {
            *size = (NQ_UINT32)staticData->maxDiskSizeServer;
        }

        goto Exit;
    }

    if (!parseInit(&parser, staticData->csConfigFile)) /* start parsing */
    {
        parseStop(&staticData->shareParser);
        goto Exit;
    }

    /* cycle by lines of the parameter file */
    while (1)
    {
        char ch;                        /* next character */

        if (parseAtFileEnd(&parser))      /* EOF */
        {
            staticData->maxDiskSizeServer = NQ_NOT_PRESENT;
            goto Exit;
        }

        parseSkipSpaces(&parser);

        if ((ch = parseGet(&parser)) != '#')       /* comment  line? */
        {
            char name[NQ_CONFIG_FILE_NAME_LEN];    /* buffer for name */
            char value[NQ_CONFIG_FILE_VALUE_LEN];   /* buffer for value */

            parseUnget(&parser, ch);
            parseSkipSpaces(&parser);
            parseName(&parser, name, sizeof(name));                  /* parameter name */
            parseSkipSpaces(&parser);

            if (parseDelimiter(&parser, '='))
            {
                parseSkipSpaces(&parser);
                parseValue(&parser, value, sizeof(value), (char)0);  /* parameter value - no delimiter */

                if (syStrncmp(name, "MAXDISKSIZE", strlen("MAXDISKSIZE")) == 0)
                {
                    if (strlen(value) > 0)
                    {
                        NQ_INT i;

                        for (i = 0; '\0' != value[i]; i++)
                        {
                            if ((value[i] < '0') || (value[i] > '9'))
                            {
                                LOGERR(CM_TRC_LEVEL_ERROR, "MAXDISKSIZE contains an illegal character");
                                staticData->maxDiskSizeServer = NQ_NOT_PRESENT;
                                parseStop(&parser);
                                goto Exit;
                            }
                        }

                        staticData->maxDiskSizeServer = atoi(value);
                        *size = (NQ_UINT32)staticData->maxDiskSizeServer;
                    }

                    parseStop(&parser);
                    goto Exit;
                }
            }
        }
        parseSkipLine(&parser);
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}
