/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : OS-dependent definitions
 *                 This file is expected to be modified during porting
 *--------------------------------------------------------------------
 * MODULE        : Linux - SY
 * DEPENDENCIES  :
 ********************************************************************/

#ifndef _SYOPSYST_H_
#define _SYOPSYST_H_

#include "sycommon.h"
#ifdef UD_NQ_CODEPAGEUTF8
#include <iconv.h>
#endif /* UD_NQ_CODEPAGEUTF8 */ 

/** @defgroup syopsyst System Dependent */

#define MUTEX_DEBUG     /**< When defined each mutex create, take, destroy will be logged */
#undef MUTEX_DEBUG

/* OS Name */

#define SY_OSNAME       "RedHat Linux"                   /**< Operating system name */

#define syAssert(_stat_)        assert(_stat_)          /**< Assert macro */

/** @ingroup syopsyst */
/** @defgroup time Time
 *
    Time functions
    -------------------------

    @see syGetTimeAccuracy() in the platform-dependent sypltfrm.h file

 */
/** @{ */

#define syGetTimeInSec()     time(0)         /**< System (Posix) time in seconds from 1-Jan-1970 */

/**
 * Get system time in POSIX format in milliseconds
 * @return The number of milliseconds elapsed since Jan 1, 1970 in milliseconds
 */
NQ_TIME syGetTimeInMsec(void);

/**
 *Convert the <b>Timespec</b> to time in milliseconds
 * @param val  <b>Timespec</b> value format
 * @return The converted time from <b>Timespec</b> to a time in milliseconds
 */
NQ_TIME syConvertTimeSpecToTimeInMsec(void * val);

/**
 * Convert time in milliseconds to seconds
 * @param timeMsec Time in milliseconds.
 * @return Time in seconds
 */
NQ_UINT32
syConvertTimeInMsecToSec(
    NQ_TIME * timeMsec
    );

/**
 * Get time zone difference in minutes
 * @return The number of minutes to be added to the local time to
 *          get GMT. This number is negative for GMT+ and positive for
 *          GMT-
 */
NQ_INT
syGetTimeZone(
    void
    );

/**
 * Decompose system time into fragments
 * @param time System time
 * @param decomposed Structure of file fragments
 */
void
syDecomposeTime(
    NQ_UINT32 time,
    SYTimeFragments* decomposed
    );

/**
 * Compose system time from fragments
 * @param decomposed Structure of file fragments
 * @return Composed system time
 */
NQ_UINT32
syComposeTime(
    const SYTimeFragments* decomposed
    );


#define sySleep(_secs_) sleep(_secs_) /**< Wait a number of seconds. */
#define syUSleep(_msecs_) usleep(_msecs_) /**< Wait a number of milliseconds. */

#if defined(CM_NQ_STORAGE) || defined(DOXYGEN)
/** This function converts system time into GMT time and prints
   it according to the format (<i>fmt</i>).

   <b><i><c> GMT-%Y.%m.%d-%H.%M.%S</c></i></b>

   @param strTime   The buffer to return the string.
   @param size      The buffer size.
   @param t         Time in seconds from Jan 1, 1970 (Unix time).
   @param fmt       The string format.

   @returns
   <i>TRUE</i> on success, <i>FALSE</i> on error.
   @note
   The format string in the example is currently the only one
   that need be supported.                                     */
NQ_BOOL syGmtToString(NQ_BYTE * strTime, NQ_COUNT size, NQ_UINT32 t, const NQ_CHAR * fmt);
#endif

/** @} end of group time Time */

/** @ingroup syopsyst */
/**@defgroup threads Threads
 *
    Threads
    -------

  Thread management calls.
 */
/** @{ */

#define SYThread                    pthread_t               /**< TID - thread handle */
#define syIsValidThread(_taskId_)   TRUE

#define syThreadGetCurrent          pthread_self

/**Create an internal thread
 *
 * @param isRT Whether to set priority level according to priorityLevel
 * @param priorityLevel The priority level of the thread
 * @param taskIdPtr Pointer to a thread handle to create
 * @param startpoint Pointer to the function to be used as the thread startpoint
 * @param background TRUE if the thread should have low priority. FALSE if the thread should have normal priority.
 */
void syThreadStart(NQ_BOOL isRT, NQ_INT priorityLevel, SYThread *taskIdPtr, void (*startpoint)(void), NQ_BOOL background);

#define syThreadDestroy(_taskId_)   pthread_cancel(_taskId_);

/** @} end of group threads Threads */

/** @ingroup syopsyst */
/** @defgroup mutex Mutexes */
/** @{ */

/* mutex */

#ifndef MUTEX_DEBUG
#define SYMutex                       pthread_mutex_t
#else
typedef struct SYMutex
{
    pthread_mutex_t mutex;
    int lockCounter;
    char who[3][120];
    NQ_UINT where[3];
}_SYMutex;
#endif

/** Create a mutual exception semaphore
 *
 * @param _m Pointer to a semaphore variable to create
 */
void syMutexCreate(SYMutex* _m);

#if defined(MUTEX_DEBUG) || defined(DOXYGEN)

/* Mutex functions - add prints per mutex create, take, give, destroy */
/** Dispose a mutual exception semaphore
 * @param _m Pointer to a semaphore variable to dispose
 */
void syMutexDelete(SYMutex* _m);

/** Lock a resource protected by a mutex
 * @param _m Pointer to a mutex semaphore
 * @param text File name for debug
 * @param line Line number for drbug
 */
void syMutexTakeDebug(SYMutex *_m, const NQ_CHAR *text, const NQ_UINT line);
#define syMutexTake(_m) syMutexTakeDebug(_m, SY_LOG_FILE, SY_LOG_LINE)

/** Unlock a resource protected by a mutex
 * @param _m Pointer to a mutex semaphore
 */
void syMutexGive(SYMutex* _m);

#else
/* regular mutex operations */
#define syMutexDelete(_m)             pthread_mutex_destroy(_m)
#define syMutexTake(_m)               pthread_mutex_lock(_m)
#define syMutexGive(_m)               pthread_mutex_unlock(_m)
#endif

/** @} end of group mutex Mutexes */

/** @ingroup syopsyst */
/** @defgroup semaphores Semaphores
 *
    Semaphores
    ----------

 1) We use mutex semaphores or simulate their behavior if the OS does not
 support pure mutex semaphores.
 2) We use binary semaphores */
/** @{ */

#define SY_SEMAPHORE_AVAILABLE /**< Enable this define when semaphores are supported on system */

#if defined(SY_SEMAPHORE_AVAILABLE) || defined(DOXYGEN)

#define SYSemaphore                   sem_t /**< Counting semaphore */

/**
 * Create semaphore
 * @param semId Pointer to the semaphore id
 * @param count Number of resources
 * @return NQ_SUCCESS when semaphore was created or NQ_FAIL on error
 */
NQ_STATUS
sySemaphoreCreate(
    SYSemaphore* semId,
    NQ_UINT count
    );

/**
 * Reset semaphore counter
 * @param pSemID Pointer to the semaphore id
 */
void
sySemaphoreResetCounter(
    SYSemaphore* pSemID
    );

#define sySemaphoreDelete(_s)         sem_destroy(&_s)
#define sySemaphoreTake(_s)           sem_wait(&_s)
#define sySemaphoreGive(_s)           sem_post(&_s)

/**
 * Lock a resource protected by a binary semaphore
 * @param sem Binary semaphore
 * @param timeout Timeout in seconds to wait for lock
 * @return NQ_SUCCESS when the resource was locked or NQ_FAIL on error
 */
NQ_INT  sySemaphoreTimedTake( SYSemaphore *sem , NQ_INT timeout);

#endif /* SY_SEMAPHORE_AVAILABLE */

/** @} end of group semaphores Semaphores */

/** @ingroup syopsyst */
/**@defgroup network Network */
/** @{ */

#if defined(UD_NQ_USETRANSPORTIPV6) || defined(DOXYGEN)

#ifdef SY_LITTLEENDIANHOST
#define SY_LOCALHOSTIP4 /*{0, 0} */  {0x007f, 0x0100}
#define SY_LOCALHOSTIP6 {0, 0, 0, 0, 0, 0, 0, 0x0100} /* {0, 0, 0, 0, 0, 0, 0, 0} */
#define SY_LINKLOCALIP  0x80fe
#else /* SY_LITTLEENDIANHOST */
#define SY_LOCALHOSTIP4 /*{0, 0} */ {0x7f00 , 0x0001}
#define SY_LOCALHOSTIP6 {0, 0, 0, 0, 0, 0, 0, 1} /* {0, 0, 0, 0, 0, 0, 0, 0} */
#define SY_LINKLOCALIP  0xfe80
#endif /* SY_LITTLEENDIANHOST */
#define SY_ANYIP4       {0, 0}
#define SY_ANYIP6       {0, 0, 0, 0, 0, 0, 0, 0}
#define SY_ZEROIP       {0, 0, 0, 0, 0, 0, 0, 0}

#else /* UD_NQ_USETRANSPORTIPV6 */

#ifdef SY_LITTLEENDIANHOST
#define SY_LOCALHOSTIP     0x0100007f
#else /* SY_LITTLEENDIANHOST */
#define SY_LOCALHOSTIP     0x7f000001
#endif /* SY_LITTLEENDIANHOST */
#define SY_ANYIP        0L
#define SY_ZEROIP       0L

#endif /* UD_NQ_USETRANSPORTIPV6 */

#define SY_ZEROIP4      0L

#if defined(UD_NQ_USETRANSPORTIPV6) || defined(DOXYGEN)
/** Get IPv6 scope ID
 * @param ip The IPv6 address
 * @return 0..n - The scope id to be used
 * @note 0 - Unknown network interface
 */
NQ_UINT32
syGetIPv6ScopeId(
    const NQ_IPADDRESS6 ip
    );
#endif /* UD_NQ_USETRANSPORTIPV6 */

/** @} end of group network Network */

/** @ingroup syopsyst */
/**@defgroup sockets Sockets
 *
    Sockets
    -------

 Most socket operations are BSD 4.x standard calls. However, a few very specific
 operations are OS-dependent. For a BSD-compliant system use definitions below

   Definition of "loopback address". This may be different for different OS. The
   standard value is 127.0.0.1 for IPv4 and ::1 for IPv6 (in NBO), however, some
   OS require a value of O. */
/** @{ */


#define SY_INTERNALSOCKETPOOL       /**< Define this parameter to use internal socket pool, comment for per-task pool */
#undef SY_INTERNALSOCKETPOOL

#define SYSocketHandle              NQ_INT
#define SYSocketSet                 fd_set
#define syIsValidSocket(_sock)      (_sock != ERROR)

/** Detecting whether a socket is still alive
 * @param sock Socket id
 * @return This method is said to work on any BSD socket system: issue select()
 *          with a zero timeout. on dead socket this should return error instead of zero
 */
NQ_BOOL
syIsSocketAlive(
    SYSocketHandle sock
    );

#define syInvalidSocket()                   (ERROR)
#define syAddSocketToSet(_sock, _set)       FD_SET((_sock), (_set))
#define syIsSocketSet(_sock, _set)          FD_ISSET((_sock), (_set))
#define syClearSocketSet(_set)              FD_ZERO((_set))
/**
   Remove a socket from a socket set.

   @param _sock   The socket to remove.
   @param _set :   Socket set.
                              */
#define syClearSocketFromSet(_sock, _set)   FD_CLR((_sock), (_set))

/**
 * Stop socket operations and disconnect the socket if it was connected
 * @param sock Socket id
 * @return NQ_SUCCESS or NQ_FAIL
 * @note This method is said to work on any BSD socket system
 */
NQ_STATUS
syShutdownSocket(
    SYSocketHandle sock
    );

/** Close socket
 *
 * @param sock Socket id
 * @return NQ_SUCCESS or NQ_FAIL
 */NQ_STATUS
syCloseSocket(
    SYSocketHandle sock
    );

/** Listen on server socket
 *
 * @param sock Socket id
 * @param backlog Max number of requests in queue
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syListenSocket(
    SYSocketHandle sock,
    NQ_INT backlog
    );

/**Create new socket
 *
 * @param stream TRUE for TCP socket, FALSE for UDP socket
 * @param family  CM_IPADDR_IPV4 for IPv4, CM_IPADDR_IPV6 for IPv6
 * @return New socket or invalid socket handle
 */
SYSocketHandle
syCreateSocket(
    NQ_BOOL stream,
    NQ_UINT family
    );

/** Bind socket to IP and port
 *
 * @param sock Socket handle
 * @param ip IP to bind to in NBO
 * @param port Port to bind to in NBO
 * @param reuseAddress Whether to reuse address
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syBindSocket(
    SYSocketHandle sock,
    const NQ_IPADDRESS *ip,
    NQ_PORT port,
    NQ_BOOL reuseAddress
    );

/** Allow broadcasts on an UDP socket
 *
 * @param sock Socket handle
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syAllowBroadcastsSocket(
    SYSocketHandle sock
    );

/** Tune new client socket
 *
 * @param sock Socket handle
 */
void
sySetClientSocketOptions(
    SYSocketHandle sock
    );

/** Get IP and port the socket is bound to
 *
 * @param sock Socket handle
 * @param ip Buffer for IP address in NBO
 * @param port Buffer for port number in NBO
 */
void
syGetSocketPortAndIP(
    SYSocketHandle sock,
    NQ_IPADDRESS *ip,
    NQ_PORT *port
    );

/** Send a UDP message to a specific addressee
 *
 * @param sock Socket handle
 * @param buf Data to send
 * @param len Number of bytes to send
 * @param ip Number of bytes to send
 * @param port Port number to send to in NBO
 * @return Number of bytes sent or NQ_FAIL
 */
NQ_INT
sySendToSocket(
    SYSocketHandle sock,
    const NQ_BYTE* buf,
    NQ_COUNT len,
    const NQ_IPADDRESS *ip,
    NQ_PORT port
    );

/**
 * Connect to a remote server port
 * @param sock Socket handle
 * @param ip IP address of the server in NBO
 * @param port Port number of the server in NBO
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syConnectSocket(
    SYSocketHandle sock,
    const NQ_IPADDRESS *ip,
    NQ_PORT port
    );

/**
 * Send bytes over a connected socket
 * @param sock Socket handle
 * @param buf Data to send
 * @param len Number of bytes to send
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_INT
sySendSocket(
    SYSocketHandle sock,
    const NQ_BYTE* buf,
    NQ_COUNT len
    );

/**
  * Send bytes asynchronously over a connected socket
  * @param sock Socket handle
  * @param buf Data to send
  * @param len Number of bytes to send
  * @param releaseFunc Callback function for releasing the buffer
  * @return NQ_SUCCESS or NQ_FAIL
  */
NQ_STATUS
sySendSocketAsync(
    SYSocketHandle sock,
    const NQ_BYTE* buf,
    NQ_COUNT len,
    void (*releaseFunc)(const NQ_BYTE*)
    );

/**
 * Select on sockets
 * @param pset Pointer to the file set
 * @param timeout Timeout in seconds
 * @return Number of sockets with data pending, zero on timeout or NQ_FAIL on error
 */
NQ_INT
sySelectSocket(
    SYSocketSet* pset,
    NQ_UINT32 timeout
    );

/** Receive a UDP message
 *
 * @param sock Socket handle
 * @param buf Receive buffer
 * @param len Buffer length
 * @param ip Buffer for sender IP address in NBO
 * @param port Buffer for sender port in NBO
 * @return Number of bytes received or NQ_FAIL
 */
NQ_INT
syRecvFromSocket(
    SYSocketHandle sock,
    NQ_BYTE* buf,
    NQ_COUNT len,
    NQ_IPADDRESS* ip,
    NQ_PORT* port
    );

/** Receive a UDP message from any sender
 *
 * @param sock Socket handle
 * @param buf Receive buffer
 * @param len Buffer length
 * @return Number of bytes received or NQ_FAIL
 */
NQ_INT
syRecvSocket(
    SYSocketHandle sock,
    NQ_BYTE* buf,
    NQ_COUNT len
    );

/**
 Receive from a datagram or a TCP stream or time out if no data on sockets.

 @param sock  Socket handle.
 @param buf  Receive buffer.
 @param len  Buffer size.
 @param secs  Number of seconds to wait for data on sockets.
 @return
 Number of bytes received or <i>NQ_FAIL</i> on error. */
NQ_INT syRecvSocketWithTimeout(SYSocketHandle sock, NQ_BYTE * buf, unsigned int len,  unsigned int secs);

/** Accept client socket
 *
 * @param sock Server socket handle
 * @param ip Buffer for client IP address in NBO
 * @param port Buffer for client port in NBO
 * @return New socket ID or invalid handle
 */
SYSocketHandle
syAcceptSocket(
    SYSocketHandle sock,
    NQ_IPADDRESS* ip,
    NQ_PORT* port
    );

#define sySetDatagramSocketOptions(_sock)
#define sySetStreamSocketOptions(_sock)

/** Send multicast datagram
 *
 * @param socket Socket handle
 * @param buffer Data to send
 * @param length Number of bytes to send
 * @param ip Destination IP
 * @param port Destination port
 * @return NQ_FAIL or number of bytes sent
 */
NQ_STATUS sySendMulticast(
    SYSocketHandle socket,
    const NQ_BYTE * buffer,
    NQ_COUNT length,
    const NQ_IPADDRESS *ip,
    NQ_PORT port);

/**Subscribe IP address to remote socket as a multicast
 *
 * @param socket Remote socket handle
 * @param ip Pointer to the IP address to subscribe. This value should
   be a multicast address in Network Byte Order (NBO).
 * @note This function should analyze the IP address type (either IPV4 or IPv6)
 *  and builds IP address in the system format (e.g., -sockaddr_in/sockaddr_in6)
 *  accordingly before subscribe it.
 */
void sySubscribeToMulticast(SYSocketHandle socket,
        const NQ_IPADDRESS *ip
        );

/** @} end of group sockets Sockets */

/** @ingroup syopsyst */
/** @defgroup tasks Tasks
 *
    Tasks
    -----

  Task management calls. We assume that the target system answers the following
  generic model:
  - a task has a unique id (PID) that may be mapped onto a unique 32 bit number
 */
/** @{ */

#define syGetPid()  getpid()

/** @} end of group tasks Tasks */

/** @ingroup syopsyst */
/**@defgroup directories Directories */
/** @{ */

 /*
    Directories
    -----------

 */

#define SYDirectory                     DIR*
#define syInvalidateDirectory(_pd)      *(_pd) = NULL
#define syIsValidDirectory(_d)          (_d != NULL)

/** Create directory
 *
 * @param name Full directory path
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syCreateDirectory(
    const NQ_WCHAR* name
    );

/** Delete directory
 *
 * @param name Full directory path
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syDeleteDirectory(
    const NQ_WCHAR* name
    );

/** Open directory by name
 *
 * @param name Full directory path
 * @return Directory handle or invalid handle
 */
SYDirectory
syOpenDirectory(
    const NQ_WCHAR* name
    );

/** Open directory and read the first entry
 *
 * @param name Full directory path
 * @param pDir Buffer for directory handle
 * @param fileName Buffer for a pointer to the file name
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syFirstDirectoryFile(
    const NQ_WCHAR* name,
    SYDirectory* pDir,
    const NQ_WCHAR** fileName
    );

/** Read next directory entry
 *
 * @param dir Directory handle
 * @param fileName Buffer for a pointer to the file name
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syNextDirectoryFile(
    SYDirectory dir,
    const NQ_WCHAR** fileName
    );

/** Close directory handle
 *
 * @param dir Directory handle
 * @return NQ_SUCCESS if the operation succeeded, NQ_FAIL if an error occurred.
 */
NQ_STATUS
syCloseDirectory(
    SYDirectory dir
    );

/** @} end of group directories Directories */

/** @ingroup syopsyst */
/**@defgroup files Files */
/** @{ */

/*
    Files
    -----

 */

#define SY_EXTENDFILENOTSUPPORTED /**<On some platforms ftruncate() might not support extending file. In this case open this define to simulate extending files */
#undef SY_EXTENDFILENOTSUPPORTED

#define SY_PATHSEPARATOR    '/'         /**< Path separator character */

#define SYFile                          int             /**< File handle */
#define syInvalidateFile(_f)            *_f = ERROR     /**< Set invalid file handle */
#define syIsValidFile(_file)            (_file!=ERROR)  /**< Check file handle */
#define syInvalidFile()                 (ERROR)


#define SY_CP_FIRSTILLEGALCHAR          {0xe5} /**< Characters which are not acceptable by the file system as a file name */
#define SY_CP_ANYILLEGALCHAR            {0x7c, 0x5c} /**< Characters which are not acceptable by the file system as a file name */


/** Create and open new file
 *
 * @param name File name
 * @param denyread True - to deny sharing for read
 * @param denyexecute True - to deny sharing for execute
 * @param denywrite True - to deny sharing for write
 * @return File handle or invalid handle
 */
SYFile
syCreateFile(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    );

/** Delete file
 *
 * @param name File name
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syDeleteFile(
    const NQ_WCHAR* name
    );

/** Rename file
 *
 * @param oldName File name
 * @param newName New file name
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syRenameFile(
    const NQ_WCHAR* oldName,
    const NQ_WCHAR* newName
    );

/** Open file for reading
 *
 * @param name File name
 * @param denyread True - to deny sharing for read
 * @param denyexecute True - to deny sharing for execute
 * @param denywrite True - to deny sharing for write
 * @return File handle or invalid handle
 */
SYFile
syOpenFileForRead(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    );

/** Open file for writing
 *
 * @param name File name
 * @param denyread True - to deny sharing for read
 * @param denyexecute True - to deny sharing for execute
 * @param denywrite True - to deny sharing for write
 * @return File handle or invalid handle
 */
SYFile
syOpenFileForWrite(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    );

/** Open file for reading and writing
 *
 * @param name File name
 * @param denyread True - to deny sharing for read
 * @param denyexecute True - to deny sharing for execute
 * @param denywrite True - to deny sharing for write
 * @return File handle or invalid handle
 */
SYFile
syOpenFileForReadWrite(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    );

/** Truncate file
 *
 * @param file File handle
 * @param offLow Offset low
 * @param offHigh Offset high
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syTruncateFile(
    SYFile file,
    NQ_UINT32 offLow,
    NQ_UINT32 offHigh
    );

#define syFlushFile(_file)                         ((fsync(_file)==OK)? NQ_SUCCESS:NQ_FAIL)

/** Read bytes from file
 *
 * @param file File handle
 * @param buf Buffer for data
 * @param len Number of bytes to read
 * @return Number of bytes read, zero on end of file, or NQ_FAIL
 */
NQ_INT
syReadFile(
    SYFile file,
    NQ_BYTE* buf,
    NQ_COUNT len
    );

/** Write bytes into file
 *
 * @param file File handle
 * @param buf Bytes to write
 * @param len Number of bytes to write
 * @return  Number of bytes written or NQ_FAIL
 */
NQ_INT
syWriteFile(
    SYFile file,
    const NQ_BYTE* buf,
    NQ_COUNT len
    );

/** Close opened file
 *
 * @param fd Handle of the file
 * @return NQ_SUCCESS if the file was close or NQ_FAIL on error
 */
NQ_STATUS
syCloseFile(
    SYFile fd
    );

/** Position file relatively from the current position
 *
 * @param file File handle
 * @param off Low 32 bits of the offset
 * @param offHigh High 32 bits of the offset
 * @return New file position or NQ_FAIL
 */
NQ_UINT32
sySeekFileCurrent(
    SYFile file,
    NQ_INT32 off,
    NQ_INT32 offHigh
    );

/** Position file from the beginning
 *
 * @param file File handle
 * @param off Low 32 bits of the offset
 * @param offHigh High 32 bits of the offset
 * @return New file position or NQ_FAIL
 */
NQ_UINT32
sySeekFileStart(
    SYFile file,
    NQ_UINT32 off,
    NQ_UINT32 offHigh
    );

/** Position file from the end
 *
 * @param file File handle
 * @param off Low 32 bits of the offset
 * @param offHigh High 32 bits of the offset
 * @return New file position or NQ_FAIL
 */
NQ_UINT32
sySeekFileEnd(
    SYFile file,
    NQ_INT32 off,
    NQ_INT32 offHigh
    );

/** Read file information structure by file handle
 *
 * @param file File id
 * @param fileName File name
 * @param fileInfo File information structure
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syGetFileInformation(
    SYFile file,
    const NQ_WCHAR* fileName,
    SYFileInformation* fileInfo
    );

/** Read file information structure by file name
 *
 * @param fileName File name
 * @param fileInfo File information structure
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syGetFileInformationByName(
    const NQ_WCHAR* fileName,
    SYFileInformation* fileInfo
    );

/** Read file size by file handle
 *
 * @param file File id
 * @param size File size
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syGetFileSize(
        SYFile file,
        NQ_UINT64 *size
        );

/** Update file information by either file name or file handle
 *
 * @param fileName File name
 * @param handle File handle
 * @param fileInfo File information structure
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
sySetFileInformation(
    const NQ_WCHAR* fileName,
    SYFile handle,
    const SYFileInformation* fileInfo
    );

/** Query volume information
 *
 * @param name Volume name
 * @param info Buffer for information
 * @return NQ_SUCCESS or NQ_FAIL
 */
NQ_STATUS
syGetVolumeInformation(
    const NQ_WCHAR* name,
    SYVolumeInformation *info
    );

#define syGetSecurityDescriptor(_f, _i, _b)     udGetSecurityDescriptor(_f, _i, _b) /**< This function is actually user-defined. These macros are used for properly redefining file descriptors */
#define sySetSecurityDescriptor(_f, _i, _b, _l) udSetSecurityDescriptor(_f, _i, _b, _l) /**< This function is actually user-defined. These macros are used for properly redefining file descriptors */

/** @} end of group files Files */

/** @ingroup syopsyst */
/** @defgroup io Input Output
 * @{
 */
#define syPrintf        printf
#define syFprintf       fprintf
#define sySprintf       sprintf
#define sySnprintf      snprintf
#define syVsnprintf     vsnprintf
#define sySscanf        sscanf
#define syScanf         scanf
#define syGetchar       getchar
#define syFflush        fflush

/** @} end of group io Input Output */

/** @ingroup syopsyst */
/** @defgroup file_locking File Locking
 *
 *
    File locking
    ------------

  The default implementation is mere a placeholder
 */
/** @{ */
                                                            
#define syUnlockFile(       \
            _file,          \
            _offsetHigh,    \
            _offsetLow,     \
            _lengthHigh,    \
            _lengthLow,     \
            _timeout        \
            )           ((_file==ERROR)? NQ_FAIL:NQ_SUCCESS)
#define syLockFile(         \
            _file,          \
            _offsetHigh,    \
            _offsetLow,     \
            _lengthHigh,    \
            _lengthLow,     \
            _lockType,      \
            _oplockLevel    \
            )           ((_file==ERROR)? NQ_FAIL:NQ_SUCCESS)

/** @} end of group file_locking File Locking */

/** @addtogroup network
 * @{ */

/*              
    Networking
    ----------

 */

#define   syGetHostName(_name, _nameLen)       gethostname((_name), (_nameLen))

/** Find host IP by its name
 *
 * @param name Host name
 * @return Host IP
 */
NQ_IPADDRESS4
syGetHostByName(
    const char* name
    );

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) || defined(DOXYGEN)
/** Returns the DNS initializations parameters
 * @param domain The default domain target belongs to
 * @param server The DNS server IP address
 * */
void
syGetDnsParams(
    NQ_CHAR *domain,
    NQ_IPADDRESS *server
    );
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */

/** Get MAC address by IP4
 *
 * @param ip Next IP address
 * @param macBuffer  Buffer for mac address
 */
void
syGetMacAddress(
    NQ_IPADDRESS4 ip,
    NQ_BYTE* macBuffer
    );

/** Get adapter information
 *
 * @param adapterIdx Adapter number (zero based)
 * @param osIndex Buffer for adapter index as defined by the OS
 * @param pIp Buffer for adapter IP in NBO
 * @param ip6 Buffer for adapter IPv6 in NBO
 * @param pSubnet Buffer for subnet address in NBO
 * @param pBcast Buffer for bcast address in NBO
 * @param pWins Buffer for semicolon delimited list of WINS servers
 * @param pDns Buffer for semicolon delimited list of DNS servers
 * @return NQ_FAIL when there is no adapter with the given index,
                               NQ_SUCCESS when adapter information available
 */
NQ_STATUS
syGetAdapter(
    NQ_INDEX adapterIdx,
    NQ_INDEX * osIndex,
    NQ_IPADDRESS4* pIp,
    NQ_IPADDRESS6 *ip6,
    NQ_IPADDRESS4* pSubnet,
    NQ_IPADDRESS4* pBcast,
    NQ_WCHAR* pWins,
    NQ_WCHAR* pDns
    );

/** @} end of group network Network */

#if defined(UD_CS_INCLUDEDIRECTTRANSFER) || defined(DOXYGEN)

/** @ingroup network */
/**@defgroup direct Direct Transfer */
/** @{ */

/*
    Socket-to-file-to-socket data transfer
    --------------------------------------

 */


/** Start fragmented packet
 * @param sock Socket handle
 * @return NQ_FAIL when this operation is not avaiiable
           NQ_SUCCESS when operation succeeded
 * */
NQ_STATUS
syDtStartPacket(
        SYSocketHandle sock
        );

/** End fragmented packet
 * @param sock Socket handle  */
void
syDtEndPacket(
        SYSocketHandle sock /* socket handle */
        );

/** Transfer bytes from socket to file
 * @param sock Socket handle
 * @param file File handle
 * @param len Number of bytes to transfer, OUT bytes transferred
 * @return NQ_FAIL on error or NQ_SUCCESS when operation succeeded
 * */
NQ_STATUS
syDtFromSocket(
        SYSocketHandle sock,
        SYFile file,
        NQ_COUNT * len
        );

/** Transfer bytes from file to socket
 * @param sock Socket handle
 * @param file File handle
 * @param len Number of bytes to transfer, OUT bytes transferred
 * @return NQ_FAIL on error or NQ_SUCCESS when operation succeeded
 * */
NQ_STATUS
syDtToSocket(
        SYSocketHandle sock,
        SYFile file,
        NQ_COUNT * len
        );

/** @} end of group direct Direct Transfer */
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

/** @ingroup syopsyst */
/**@defgroup encoding Encoding */
/** @{ */

#define SY_UNICODEFILESYSTEM        /**< Whether the filesystem supports Unicode. Otherwise all filenames will be converted to ANSI even if CIFS is supporting UNICODE */
#undef  SY_UNICODEFILESYSTEM

#if defined(UD_CC_INCLUDELDAP) || defined(UD_NQ_CODEPAGEUTF8) || defined(DOXYGEN)

/** Convert Unicode string to UTF8
 * @param outStr Pointer to the result string in UTF8
 * @param outLength Length of the result buffer
 * @param inWStr Pointer to string in UNICODE to be converted
 * @param inLength Length of string to be converted
 *  */
void
syUnicodeToUTF8N(
    NQ_CHAR *outStr,
    NQ_UINT outLength,
    const NQ_WCHAR *inWStr,
    NQ_UINT inLength
    );


/** Convert UTF8 string to Unicode
 * @param outWStr Pointer to the result string in UNICODE
 * @param outLength Length of the result buffer
 * @param inStr Pointer to input string in UTF8
 * @param inLength Length of string to be converted*/
void
syUTF8ToUnicodeN(
    NQ_WCHAR *outWStr,
    NQ_UINT outLength,
    const NQ_CHAR *inStr,
    NQ_UINT inLength
    );

#endif /* defined(UD_CC_INCLUDELDAP) || defined(UD_NQ_CODEPAGEUTF8) */

/*
    Using OS library for UTF 8
    --------------------------
*/
#if defined(UD_NQ_CODEPAGEUTF8) || defined(DOXYGEN)

/**Check whether conversion is available on the platform
 * @return <i>TRUE</i> on success, <i>FALSE</i> on error.
 */
NQ_BOOL initCodePageUTF8(void);
                
#endif /* UD_NQ_CODEPAGEUTF8 */

/** @} end of group encoding Encoding */

/** @ingroup syopsyst */
/**@defgroup utils Utils
 * @{ */
#define sySetRand()     srand((unsigned int)time(0))  /**< Take seed from the system time */
#define syRand()        rand()          /**< Next (pseudo)random value */

/** Convert UNIX file permissions to DOS file attributes
 * @param mode UNIX file permissions
 * @return Matching DOS file attributes
 * @note
            \* DOS read only is represented in UNIX by removing everyone's write bit.
            \* DOS archive is represented in UNIX by the user's execute bit.
            \* DOS system is represented in UNIX by the group's execute bit.
            \* DOS hidden is represented in UNIX by the other's execute bit.
            \* DOS directory is represented in UNIX by UNIX 's directory bit.*/
int
syUnixMode2DosAttr(
    int mode
    );

/** Convert last system error to SMB error
 *
 * @return SMB error
 */
NQ_UINT32
syGetLastSmbError(
    void
    );

/** Convert NQ error into system error
 * @param nqErr NQ error
 */
void
sySetLastNqError(
    NQ_STATUS nqErr
    );

/** @} end of group utils Utils */

#endif  /* _SYOPSYST_H_ */
