/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : WS-Discovery (WSD) Client light
 *--------------------------------------------------------------------
 * MODULE        : Service
 * DEPENDENCIES  :
 *************************************************************************/
#include "cmapi.h"
#include "wsdclient.h"

#if defined(UD_CM_INCLUDEWSDCLIENT)

#include "cmresolver.h"
#include "cmuuid.h"

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>


#define WSDCLIENT_IPV4_MULTICAST_ADDRESS "239.255.255.250"
#define WSDCLIENT_IPV6_MULTICAST_ADDRESS "FF02::C"
#define WSD_PORT 3702


#define PROBE_MESSAGE_START "<soap:Envelope xmlns:soap=\"http://www.w3.org/2003/05/soap-envelope\" xmlns:pub=\"http://schemas.microsoft.com/windows/pub/2005/07\" xmlns:wsa=\"http://schemas.xmlsoap.org/ws/2004/08/addressing\" xmlns:wsd=\"http://schemas.xmlsoap.org/ws/2005/04/discovery\" xmlns:wsdp=\"http://schemas.xmlsoap.org/ws/2006/02/devprof\"><soap:Header><wsa:Action>http://schemas.xmlsoap.org/ws/2005/04/discovery/Probe</wsa:Action><wsa:MessageID>urn:uuid:"
#define PROBE_MESSAGE_BODY  "</wsa:MessageID><wsa:To>urn:schemas-xmlsoap-org:ws:2005:04:discovery</wsa:To></soap:Header><soap:Body><wsd:Probe><wsd:Types>"
#define PROBE_MESSAGE_END   "</wsd:Types></wsd:Probe></soap:Body></soap:Envelope>"

#define WSD_XMLNAMESPACES   "soap=http://www.w3.org/2003/05/soap-envelope \
                            wsa=http://schemas.xmlsoap.org/ws/2004/08/addressing \
                            wsd=http://schemas.xmlsoap.org/ws/2005/04/discovery \
                            wsdp=http://schemas.xmlsoap.org/ws/2006/02/devprof \
                            wprt=http://schemas.microsoft.com/windows/2006/08/wdp/print \
                            wscn=http://schemas.microsoft.com/windows/2006/08/wdp/scan \
                            pwg=http://www.hp.com/schemas/imaging/con/pwg/sm/1.0 \
                            dd=http://www.hp.com/schemas/imaging/con/dictionaries/1.0 \
                            hpd=http://www.hp.com/schemas/imaging/con/discovery/2006/09/19 \
                            ledm=http://www.hp.com/schemas/imaging/con/ledm/discoverytree/2007/07/01"

#define WSD_XPATH_MESSAGE_ID            "/soap:Envelope/soap:Header/wsa:RelatesTo"
#define WSD_XPATH_PROBE_MATCHES_TYPES   "/soap:Envelope/soap:Body/wsd:ProbeMatches/wsd:ProbeMatch/wsd:Types"

#define WSD_TYPE_PRINTER    "PrintDeviceType"
#define WSD_TYPE_COMPUTER   "Computer"

#define WSD_XADDR_IP        0x1
#define WSD_XADDR_NAME      0x2

#define WSD_MAX_RESPONSE_ITEMS  10

typedef struct
{
    NQ_UINT32 timeout;
    NQ_IPADDRESS wsdIp4;
    NQ_IPADDRESS wsdIp6;
    SYMutex guard;
}
StaticData;

static NQ_BOOL isModuleInitialized = FALSE;
static NQ_COUNT moduleInitCount = 0;
#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

static NQ_BOOL registerXmlNamespaces(xmlXPathContextPtr xpathCtx, const xmlChar* nsList)
{
    xmlChar* nsListDup = NULL;
    xmlChar* prefix = NULL;
    xmlChar* href = NULL;
    xmlChar* next = NULL;
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "xpathCtx:%p nsList:%p", xpathCtx, nsList);

    nsListDup = xmlStrdup(nsList);
    if (NULL == nsListDup)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "xmlStrdup failed");
        goto Exit;
    }

    next = nsListDup;
    while (NULL != next)
    {
        /* skip spaces */
        while ((*next) == ' ')
        {
            next++;
        }
        if ((*next) == '\0')
        {
            break;
        }

        /* find prefix */
        prefix = next;
        next = (xmlChar*)xmlStrchr(next, '=');
        if (NULL == next)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Invalid namespaces list format");
            goto Exit;
        }
        *(next++) = '\0';

        /* find href */
        href = next;
        next = (xmlChar*)xmlStrchr(next, ' ');
        if (NULL != next)
        {
            *(next++) = '\0';
        }

        /* register namespace */
        if (0 != xmlXPathRegisterNs(xpathCtx, prefix, href))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to register namespace");
            goto Exit;
        }
        result = TRUE;
    }
Exit:
    if (NULL != nsListDup)
    {
        xmlFree(nsListDup);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

static const NQ_CHAR *getDeviceType(NQ_UINT deviceType)
{
    switch (deviceType)
    {
    case DEVICE_TYPE_PRINTER:
        return "wsdp:Device";
    case DEVICE_TYPE_COMPUTER:
        return "wsdp:Device"; /* currently the same, need more research */
    default:
        return NULL;
    }
}

static NQ_COUNT createProbeMessage(NQ_BYTE *buffer, NQ_COUNT bufferSize, const NQ_CHAR *deviceType, CMUuid *uuid)
{
    CMBufferWriter writer;
    NQ_COUNT written = 0;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buffer:%p bufferSize:%d deviceType:%s uuid:%p", buffer, bufferSize, deviceType, uuid);

    cmBufferWriterInit(&writer, buffer, bufferSize);
    cmBufferWriterSetByteOrder(&writer, FALSE);
    cmBufferWriteString(&writer, TRUE, (NQ_BYTE *)PROBE_MESSAGE_START, FALSE, CM_BSF_NOFLAGS);
    cmBufferWriteUuidAsString(&writer, uuid);
    cmBufferWriteString(&writer, TRUE, (NQ_BYTE *)PROBE_MESSAGE_BODY, FALSE, CM_BSF_NOFLAGS);
    cmBufferWriteString(&writer, TRUE, (NQ_BYTE *)deviceType, FALSE, CM_BSF_NOFLAGS);
    cmBufferWriteString(&writer, TRUE, (NQ_BYTE *)PROBE_MESSAGE_END, FALSE, CM_BSF_NOFLAGS);
    written = cmBufferWriterGetDataCount(&writer);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "written:%d", written);
    return written;
}

static xmlChar *xpathGetData(xmlChar *xpathExpr, xmlXPathContextPtr xpathCtx)
{
    xmlChar *data = NULL;
    xmlXPathObjectPtr xpathObj = NULL;
    xmlChar *content = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "xpathExpr:%s xpathCtx:%p", xpathExpr, xpathCtx);

    /* Evaluate xpath expression */
    xpathObj = xmlXPathEvalExpression(xpathExpr, xpathCtx);
    if (NULL == xpathObj)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "xmlXPathEvalExpression failed");
        goto Exit;
    }

    if ((NULL != xpathObj->nodesetval) && (0 != xpathObj->nodesetval->nodeNr))
    {
        xmlNodePtr cur = NULL;

        cur = xpathObj->nodesetval->nodeTab[0];
        content = xmlNodeGetContent(cur);
        if (NULL != content)
        {
            data = (xmlChar *)cmMemoryCloneAStringAsAscii((const NQ_CHAR *)content);
            if (NULL == data)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            }
        }
    }

Exit:
    if (NULL != content)
    {
        xmlFree(content);
    }
    if (NULL != xpathObj)
    {
        xmlXPathFreeObject(xpathObj);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "data:%p", data);
    return data;
}

static NQ_BOOL validateMessageID(xmlXPathContextPtr xpathCtx, NQ_Uuid *uuid)
{
    xmlChar *messageID = NULL;
    NQ_CHAR *uuidToString = cmUuidToString(uuid);
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "xpathCtx:%p uuid:%p", xpathCtx, uuid);

    messageID = xpathGetData((xmlChar *)WSD_XPATH_MESSAGE_ID, xpathCtx);
    if (NULL == messageID)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Can't extract message ID");
        goto Exit;
    }

    if (NULL != uuidToString)
    {
        NQ_CHAR *p;

        p = syStrrchr((NQ_CHAR *)messageID, ':');
        if (NULL == p)
        {
            goto Exit;
        }
        if (0 != syStricmp(uuidToString, ++p))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unexpected messId: %s, expected: %s", p, uuidToString);
            goto Exit;
        }
        result = TRUE;
    }
Exit:
    cmMemoryFree(messageID);
    cmMemoryFree(uuidToString);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/* find one string of requested type inside XAddrs, add it to list (advance the start pointer in list according to value of num parameter) */
static NQ_BOOL parseXAddrsByType(xmlNodePtr current, const NQ_CHAR **what, NQ_COUNT whatSize, xmlChar ending, NQ_INT type, void *list, NQ_INT *num)
{
    xmlChar *start = NULL, *end = NULL, *content = NULL;
    NQ_COUNT i;
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "current:%p what:%p whatSize:%u ending:%c type:%d list:%p num:%d", current, what, whatSize, ending, type, list, *num);

    if (WSD_MAX_RESPONSE_ITEMS == *num)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Reached max items:%d", *num);
        goto Exit;
    }

    content = xmlNodeGetContent(current);
    if (NULL == content)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "xmlNodeGetContent failed");
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "XAddrs: '%s'", content);

    for (i = 0; i < whatSize; i++)
    {
        start = (xmlChar *)xmlStrstr(content, (xmlChar *)what[i]);
        if (NULL == start)
        {
            continue;
        }

        start += syStrlen((const NQ_CHAR *)what[i]);
        end = (xmlChar *)xmlStrchr((xmlChar *)start, ending);
        if (NULL != end)
        {
            *end = '\0';
        }

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "type:%s, '%s'", WSD_XADDR_IP == type ? "IP" : ((WSD_XADDR_NAME == type) ? "name" : "error"), start);

        switch (type)
        {
            case WSD_XADDR_IP:
            {
                NQ_IPADDRESS *listIps = (NQ_IPADDRESS *)list;

                /* advance the start pointer in list */
                listIps += *num;

                if (NQ_SUCCESS == cmAsciiToIp((NQ_CHAR *)start, listIps))
                {
                   ++(*num);
                   result = TRUE;
                }

                break;
            }
            case WSD_XADDR_NAME:
            {
                NQ_WCHAR *listNames = (NQ_WCHAR *)list;
                NQ_INT n = *num;  /* current number of names in list */
                NQ_WCHAR *p;
                NQ_IPADDRESS dummyIp;

                /* advance the start pointer in list */
                while (n > 0)
                {
                   p = cmWStrchr(listNames, cmWChar(0));
                   if (NULL == p)
                   {
                       break;
                   }

                   listNames = ++p;
                   --n;
                }

                /* not interested in IPs */
                if (NQ_FAIL == cmAsciiToIp((NQ_CHAR *)start, &dummyIp))
                {
                    cmAnsiToUnicode(listNames, (NQ_CHAR *)start);
                    ++(*num);
                }
                else
                {
                    result = TRUE;
                }

                break;
            }
            default:
            {
                break;
            }
        }
    }

Exit:
    if (NULL != content)
    {
        xmlFree(content);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/* find all strings of requested type inside all XML nodes, add them to list (advance the start pointer in list according to value of num parameter) */
static NQ_BOOL getEntriesbyDeviceType(xmlXPathContextPtr xpathCtx, NQ_UINT deviceType, NQ_IPADDRESS *listIps, NQ_INT *pNumIps, NQ_WCHAR *listNames, NQ_INT *pNumNames)
{
    NQ_BOOL result = FALSE;
    NQ_INT size, i;
    xmlNodeSetPtr nodes = NULL;
    xmlXPathObjectPtr xpathObj = NULL;
    xmlChar *contentType = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "xpathCtx:%p deviceType:0x%x listIps:%p pNumIps:%p listNames:%p pNumNames:%p", xpathCtx, deviceType, listIps, pNumIps, listNames, pNumNames);

    if (NULL != pNumIps)
    {
        *pNumIps = 0;
    }
    if (NULL != pNumNames)
    {
        *pNumNames = 0;
    }

    /* Evaluate xpath expression */
    xpathObj = xmlXPathEvalExpression((xmlChar *)WSD_XPATH_PROBE_MATCHES_TYPES, xpathCtx);
    if (NULL == xpathObj)
    {
        goto Exit;
    }
    nodes = xpathObj->nodesetval;
    size = (NULL == nodes) ? 0 : nodes->nodeNr;

    for (i = 0; i < size; ++i)
    {
        if (NULL == nodes->nodeTab[i])
        {
            goto Exit;
        }

        if (nodes->nodeTab[i]->type == XML_ELEMENT_NODE)
        {
            const xmlChar *strType = NULL;

            contentType = xmlNodeGetContent(nodes->nodeTab[i]);
            if (NULL == contentType)
            {
                goto Exit;
            }

            switch (deviceType)
            {
                case DEVICE_TYPE_PRINTER:
                {
                    strType = (xmlChar *) WSD_TYPE_PRINTER;
                    break;
                }
                case DEVICE_TYPE_COMPUTER:
                {
                    strType = (xmlChar *) WSD_TYPE_COMPUTER;
                    break;
                }
                default:
                {
                    break;
                }
            }

            if (NULL != xmlStrcasestr(contentType, strType))
            {
                xmlNodePtr cur = (xmlNodePtr)nodes->nodeTab[i];

                /* look for XAddrs */
                while (NULL != cur)
                {
                    if ((cur->type == XML_ELEMENT_NODE) && (NULL != cur->name) && (0 == xmlStrcmp(cur->name, (xmlChar *)"XAddrs")))
                    {
                        static const NQ_CHAR *prefixIPv4[] = {"http://", "https://"};
                        NQ_BOOL ret = FALSE;

                        /* Check if the IP is in the form of IPv4 */
                        /* find in content IP and add to the list */
                        ret = parseXAddrsByType(cur, prefixIPv4, sizeof(prefixIPv4) / sizeof(prefixIPv4[0]), ':', WSD_XADDR_IP, (void *)listIps, pNumIps);
                        if (FALSE == ret)
#ifdef UD_NQ_USETRANSPORTIPV6
                        {
                            static const NQ_CHAR *prefixIPv6[] = {"http://[", "https://["};

                            /* If not Check if the IP is in the form of IPv6 */
                            ret = parseXAddrsByType(cur, prefixIPv6, sizeof(prefixIPv6) / sizeof(prefixIPv6[0]), ']', WSD_XADDR_IP, (void *)listIps, pNumIps);

                        }
#endif /* UD_NQ_USETRANSPORTIPV6 */

                        if (FALSE == ret)
                        {
                            /* If not then the it must be a name, name is in the form of IPv4 only, IPv4:PORT, find in content name and add to the list */
                            parseXAddrsByType(cur, prefixIPv4, sizeof(prefixIPv4) / sizeof(prefixIPv4[0]), ':', WSD_XADDR_NAME, (void *)listNames, pNumNames);
                        }
                    } /* end of if */
                    cur = cur->next;
                } /* end of while */
            }
        } /* end of if */
    } /* end of for */

    result = TRUE;
Exit:
    if (NULL != contentType)
    {
        xmlFree(contentType);
    }
    if (NULL != xpathObj)
    {
        xmlXPathFreeObject(xpathObj);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}

static NQ_STATUS parseProbeMatchesMessage(NQ_BYTE *buffer, NQ_INT bufferSize, WsdClientContext *context, NQ_IPADDRESS *listIPs, NQ_INT *pNumIps, NQ_WCHAR *listNames, NQ_INT *pNumNames)
{
    NQ_STATUS res = NQ_FAIL;
    xmlDocPtr doc = NULL;
    xmlXPathContextPtr xpathCtx = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buffer:%p bufferSize:%d context:%p listIPs:%p pNumIps:%p", buffer, bufferSize, context, listIPs, pNumIps);

    /* Init libxml */
    xmlInitParser();

    /* parse an XML in-memory block and build a tree */
    doc = xmlParseMemory((const char *)buffer, (int)bufferSize);
    if (NULL == doc)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "xmlParseMemory failed");
        goto Exit;
    }

    /* prints the xml with indentation */

    /*
    {
        xmlChar *mem;
        int size;

        xmlKeepBlanksDefault(0);
        xmlDocDumpFormatMemory(doc, &mem, &size, 1);
        syPrintf("DUMP: \n%s\n", mem);
        xmlFree(mem);
    }
    */

    /* Create xpath evaluation context */
    xpathCtx = xmlXPathNewContext(doc);
    if (NULL == xpathCtx)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "xmlXPathNewContext failed");
        goto Exit;
    }

    /* Register namespaces */
    if (!registerXmlNamespaces(xpathCtx, (xmlChar *) WSD_XMLNAMESPACES))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "registerXmlNamespaces failed");
        goto Exit;
    }

    /* validate MessageID */
    if (!validateMessageID(xpathCtx, &context->uuid))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Wrong message ID");
        goto Exit;
    }

    /* get devices IPs and names by device type */
    if (!getEntriesbyDeviceType(xpathCtx, context->deviceType, listIPs, pNumIps, listNames, pNumNames))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "No devices by requested type");
        goto Exit;
    }
    res = NQ_SUCCESS;

Exit:
    if (NULL != xpathCtx)
    {
        xmlXPathFreeContext(xpathCtx);
    }
    if (NULL != doc)
    {
        xmlFreeDoc(doc);
    }
    /* Shutdown libxml */
    xmlCleanupParser();

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "res:0x%x", res);
    return res;
}


static NQ_STATUS wsdRequestByIp(SYSocketHandle socket, const NQ_IPADDRESS *ip, void *ctxt, const NQ_IPADDRESS *serverIp, NQ_COUNT * numOfSentRequests)
{
    NQ_BYTE buffer[CM_NB_DATAGRAMBUFFERSIZE];       /* output buffer */
    NQ_UINT length;                                 /* outgoing packet length */
    NQ_STATUS result = NQ_SUCCESS;                  /* operation result */
    WsdClientContext *context = (WsdClientContext *)ctxt; /* casted context */
    const NQ_CHAR *deviceType;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d ip:%p ctxt:%p serverIp:%p", socket, ip, context, serverIp);

    *numOfSentRequests = 0;
    deviceType = getDeviceType(context->deviceType);
    if (NULL == deviceType)
    {
        result = NQ_ERR_BADPARAM;
        goto Exit;
    }

    /* save sent uuid to match with responses */
    cmGenerateUuid(&context->uuid);

    length = createProbeMessage(buffer, sizeof(buffer), deviceType, &context->uuid);

#ifdef UD_NQ_USETRANSPORTIPV4
    if (CM_IPADDR_EQUAL(staticData->wsdIp4, *serverIp))
    {
        result = sySendMulticast(socket, buffer, length, serverIp, syHton16(WSD_PORT));
    }
#ifdef UD_NQ_USETRANSPORTIPV6
    else
#endif /* UD_NQ_USETRANSPORTIPV6 */
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    if (CM_IPADDR_EQUAL(staticData->wsdIp6, *serverIp))
    {
        result = sySendToSocket(socket, buffer, length, serverIp, syHton16(WSD_PORT));
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */

    if (result > 0)
    {
        (*numOfSentRequests)++;
        result = NQ_SUCCESS;
    }
    else
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Send failed");
        result = NQ_ERR_SOCKETSEND;
    }
Exit:
    sySetLastError(result);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", result);
    return result;
}

static NQ_STATUS wsdResponseByName(SYSocketHandle socket, NQ_IPADDRESS **pAddressArray, NQ_INT *pNumIps, void **pCtxt)
{
    NQ_IPADDRESS IPs[WSD_MAX_RESPONSE_ITEMS];                       /* resulted IPs */
    NQ_WCHAR names[(CM_NQ_HOSTNAMESIZE + CM_TRAILING_NULL) * WSD_MAX_RESPONSE_ITEMS];    /* resulted names */
    NQ_BYTE buffer[4096];                                           /* input buffer */
    NQ_INT count;                                                   /* number of bytes in the incoming datagram */
    NQ_IPADDRESS srcIp;                                             /* source IP */
    NQ_PORT srcPort;                                                /* source port */
    WsdClientContext *context = (WsdClientContext *)*pCtxt;         /* casted context (contains names list pointer) */
    NQ_INT *pNumNames = &context->numNames;                         /* names number pointer */
    NQ_STATUS res = NQ_FAIL;                                        /* operation result */
    NQ_STATUS lastError = NQ_ERR_OK;                                /* last operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%d pAddressArray:%p numIps:%p pCtxt:%p", socket, pAddressArray, pNumIps, pCtxt);

    *pNumIps = 0;
    *pNumNames = 0;

    count = syRecvFromSocket(socket, buffer, sizeof(buffer), &srcIp, &srcPort);
    if (count <= 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving WSD response");
        lastError = (NQ_STATUS)syGetLastError();
        goto Exit;
    }

    /* get IPs and names */
    lastError = parseProbeMatchesMessage(buffer, count, context, IPs, pNumIps, names, pNumNames);
    if (NQ_SUCCESS != lastError)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error parsing WSD response");
        lastError = NQ_FAIL;
        goto Exit;
    }
    if (*pNumIps > 0)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Returned %d ips, 1st address: %s", *pNumIps, cmIPDump(&IPs[0]));

        *pAddressArray = (NQ_IPADDRESS *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_IPADDRESS) * (NQ_UINT)(*pNumIps)));
        if (NULL == *pAddressArray)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            lastError = NQ_ERR_OUTOFMEMORY;
            *pNumIps = 0;
        }
        else
        {
            syMemcpy(*pAddressArray, IPs, sizeof(NQ_IPADDRESS) * (NQ_UINT)(*pNumIps));
#ifdef UD_NQ_INCLUDETRACE
            cmDumpIPsList(*pAddressArray, *pNumIps);
#endif /* UD_NQ_INCLUDETRACE */
        }
    }

    if (*pNumNames > 0)
    {
        NQ_UINT     size, i;
        NQ_WCHAR    *p;

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Resolved %d names, 1st name: %s", *pNumNames, cmWDump(names));

        /* calculate size for delimited list of names */
        for (size = 0, i = 0, p = names; i < *pNumNames; i++, p += cmWStrlen(names) + CM_TRAILING_NULL)
        {
            size += (NQ_UINT)(sizeof(NQ_WCHAR) * (cmWStrlen(p) + CM_TRAILING_NULL));
        }

        context->names = (NQ_WCHAR *)cmMemoryAllocate(size);
        if (NULL == context->names)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            lastError = NQ_ERR_OUTOFMEMORY;
            *pNumNames = 0;
        }
        else
        {
            NQ_WCHAR delimiter[2] = {cmWChar(CM_NQ_DELIMITER), cmWChar('\0')};

            context->names[0] = cmWChar('\0');
            for (i = 0, p = names; i < *pNumNames; i++, p += cmWStrlen(names) + CM_TRAILING_NULL)
            {
                cmWStrcat(context->names, p);
                if ((i + 1) < *pNumNames)
                {
                    cmWStrcat(context->names, delimiter);
                }
            }
        }
#ifdef UD_NQ_INCLUDETRACE
        cmDumpNamesList(context->names);
#endif /* UD_NQ_INCLUDETRACE */
    }

Exit:
    res = ((0 != *pNumIps) || (0 != *pNumNames)) ? NQ_SUCCESS : lastError;
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%x", res);
    return res;
}

void static registerResolverMethods()
{
    CMResolverMethodDescriptor method;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    method.type = NQ_RESOLVER_WSD;
    method.isMulticast = TRUE;
    method.activationPriority = 1;
    method.timeout.low = staticData->timeout;   /* milliseconds */
    method.timeout.high = 0;                    /* milliseconds */
    method.waitAnyway = TRUE;
    method.requestByName = NULL;
    method.responseByName = wsdResponseByName;
    method.requestByIp = wsdRequestByIp;
    method.responseByIp = NULL;
#ifdef UD_NQ_USETRANSPORTIPV4
    cmAsciiToIp(WSDCLIENT_IPV4_MULTICAST_ADDRESS, &staticData->wsdIp4);
    cmResolverRemoveMethod(&method, &staticData->wsdIp4);
    if (!cmResolverRegisterMethod(&method, &staticData->wsdIp4))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to register method wsd ipv4");
    }
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    cmAsciiToIp(WSDCLIENT_IPV6_MULTICAST_ADDRESS, &staticData->wsdIp6);
    cmResolverRemoveMethod(&method, &staticData->wsdIp6);
    if (!cmResolverRegisterMethod(&method, &staticData->wsdIp6))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to register method wsd ipv6");
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

NQ_BOOL wsdClientStart(void)
{
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        ++moduleInitCount;
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate WSD data");
        result = FALSE;
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    syMutexCreate(&staticData->guard);

    staticData->timeout = UD_CM_WSDCLIENTTIMEOUT;

    registerResolverMethods();

    isModuleInitialized = TRUE;
    ++moduleInitCount;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

void wsdClientShutdown(void)
{
    if (isModuleInitialized && (--moduleInitCount == 0))
    {
        syMutexDelete(&staticData->guard);

        /* release memory */
#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
        {
            cmMemoryFreeShutdown(staticData);
            staticData = NULL;
        }
#endif /* SY_FORCEALLOCATION */
        isModuleInitialized = FALSE;
    }
}

NQ_BOOL wsdClientSetTimeout(NQ_UINT32 milliseconds)
{
    NQ_BOOL result = TRUE;

    if (isModuleInitialized)
    {
        syMutexTake(&staticData->guard);
        staticData->timeout = milliseconds;

        /* register methods with new timeout */
        registerResolverMethods();

        syMutexGive(&staticData->guard);
        goto Exit;
    }
    result = FALSE;
Exit:
    return result;
}

#endif /* defined(UD_CM_INCLUDEWSDCLIENT) */
