/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : NT Security Descriptors
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#ifndef _CMSDESCR_H_
#define _CMSDESCR_H_

#include "cmapi.h"

/* security descriptor */

#define SD_BUFFERS_LARGE_SIZE_BYTE 0x8000 /* 32 KB  see https://blogs.msdn.microsoft.com/arvind7in/2010/05/26/what-is-64k-acl-limit/ */

typedef struct
{
    NQ_BYTE     data[UD_CM_SECURITYDESCRIPTORLENGTH];
    NQ_UINT32   length;
}
CMSdSecurityDescriptor;

typedef struct
{
    NQ_BYTE     data[SD_BUFFERS_LARGE_SIZE_BYTE];
    NQ_UINT32   length;
}
CMShareSecurityDescriptor;


/* relative ID */
typedef NQ_UINT32 CMSdRid;

/* admin SID */
#define cmSdIsAdmin(rid)            ((NQ_INT)rid < 0)

#include "sypackon.h"

/*  Domain SID (security ID). Sub-authorities follow this structure. Each sub-authority is
    a 32-bit value. Number of values is defined by numAuth field. */

typedef SY_PACK_PREFIX struct
{
    NQ_BYTE revision;               /* SID revision number */
    NQ_BYTE numAuths;               /* number of sub-authorities */
    NQ_BYTE idAuth[6];              /* identifier authority */
    NQ_UINT32 subs[6];              /* sub authorities */
}
SY_PACK_ATTR CMSdDomainSid;

typedef SY_PACK_PREFIX struct
{
    NQ_BYTE type;       /* ace type - e.g allowed / denied etc */
    NQ_BYTE flags;      /* see below */
    NQ_UINT16 size;
    NQ_UINT32 accessMask;
    CMSdDomainSid trustee;
}
SY_PACK_ATTR CMSdAce;

/* type values */
#define CM_SD_ALLOW 0
#define CM_SD_DENY  1

/* Security Information Flags */
#define CM_SD_OWNER     0x00000001
#define CM_SD_GROUP     0x00000002
#define CM_SD_DACL      0x00000004
#define CM_SD_SACL      0x00000008
#define CM_SD_LABEL     0x00000010

/* RID/name types */
#define CM_SD_RIDTYPE_USENONE    0  /* NOTUSED */
#define CM_SD_RIDTYPE_USER       1  /* user */
#define CM_SD_RIDTYPE_DOMGRP     2  /* domain group */
#define CM_SD_RIDTYPE_DOMAIN     3  /* domain name */
#define CM_SD_RIDTYPE_ALIAS      4  /* local group */
#define CM_SD_RIDTYPE_WKNGRP     5  /* well-known group */
#define CM_SD_RIDTYPE_DELETED    6  /* deleted account: needed for c2 rating */
#define CM_SD_RIDTYPE_INVALID    7  /* invalid account */
#define CM_SD_RIDTYPE_UNKNOWN    8  /* */

/*  An array of CMCifsSecurityAce structures follows this structure. The number of
    following CMCifsSecurityAce values is defined by the numAces field. */
typedef SY_PACK_PREFIX struct
{
    NQ_UINT16 revision;
    NQ_UINT16 size;
    NQ_UINT32 numAces;
}
SY_PACK_ATTR CMSdAcl;

/* Finally - security descriptor */
typedef SY_PACK_PREFIX struct
{
    NQ_UINT16 revision;             /* format version */
    NQ_UINT16 type;                 /* see flags in cmsdesc.c -"type bits for security descriptor"  */
    NQ_UINT32 ownerSid;             /* offset to owner SID */
    NQ_UINT32 groupSid;             /* offset to main group SID */
    NQ_UINT32 sacl;                 /* offset to system ACL */
    NQ_UINT32 dacl;                 /* offset to DACL */
}
SY_PACK_ATTR CMSdSecurityDescriptorHeader;

/* well known groups in SID */
#define CM_SD_BUILTIN_GROUP     32
#define CM_SD_NONUNIQUE_GROUP   21

/* Well-known users */

#define SD_UID_DOMAINUSER_ADMIN          (500L)
#define SD_UID_DOMAINUSER_GUEST          (501L)
#define SD_UID_DOMAINUSER_KRBTGT         (502L)

/* Well-known groups */

#define SD_UID_DOMAINGROUP_ADMINS        (512L)
#define SD_UID_DOMAINGROUP_USERS         (513L)
#define SD_UID_DOMAINGROUP_GUESTS        (514L)
#define SD_UID_DOMAINGROUP_COMPUTERS     (515L)
#define SD_UID_DOMAINGROUP_CONTROLLERS   (516L)
#define SD_UID_DOMAINGROUP_CERTADMINS    (517L)
#define SD_UID_DOMAINGROUP_SCHEMAADMINS  (518L)
#define SD_UID_DOMAINGROUP_ENTERPADMINS  (519L)

/* Well-known aliases */

#define SD_UID_DOMAINALIAS_EVERYONE      (0L)
#define SD_UID_DOMAINALIAS_ADMINS        (544L)
#define SD_UID_DOMAINALIAS_USERS         (545L)
#define SD_UID_DOMAINALIAS_GUESTS        (546L)
#define SD_UID_DOMAINALIAS_POWERUSERS    (547L)
#define SD_UID_DOMAINALIAS_ACCOUNTOPS    (548L)
#define SD_UID_DOMAINALIAS_SYSTEMOPS     (549L)
#define SD_UID_DOMAINALIAS_PRINTOPS      (550L)
#define SD_UID_DOMAINALIAS_BACKUPOPS     (551L)
#define SD_UID_DOMAINALIAS_REPLICATOR    (552L)


/* "Well known RIDs for groups and aliases */

#define CM_SD_RIDANONYMOUS              7       /* user anonymous */
#define CM_SD_RIDLOCALSYSTEM            18      /* local system */
#define CM_SD_RIDADMINISTRATOR          500     /* local administrator */
#define CM_SD_RIDGUEST                  501     /* guest user */
#define CM_SD_RIDGROUPADMINS            512     /* administrators group */
#define CM_SD_RIDGROUPUSERS             513     /* users group */
#define CM_SD_RIDGROUPGUESTS            514     /* guests group */
#define CM_SD_RIDGROUPDOMAINCOMPUTERS   515     /* domain computers */
#define CM_SD_RIDGROUPPOLICYCREATORS    520     /* group policy creator owners */
#define CM_SD_RIDALIASADMIN             544     /* any administrators */
#define CM_SD_RIDALIASUSER              545     /* any users */
#define CM_SD_RIDALIASGUEST             546     /* any guests */
#define CM_SD_RIDALIASACCOUNTOP         548     /* account operators */

#include "sypackof.h"

/* desired access */

typedef  NQ_UINT32 CMSdAccessFlags;

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)

/* initialize this module */
NQ_STATUS        /* NQ_SUCCESS or NQ_FAIL */
cmSdInit(
    void
    );

/* release this module */
void
cmSdExit(
    void
    );

/* get large security descriptor buffer */
NQ_BYTE* cmSdGetLargeSDBuf(NQ_COUNT *bufSize);

/* release large buffer */
NQ_BOOL cmSdReleaseLargeSDBuf(NQ_BYTE* buffer);

/* check security descriptor */
NQ_BOOL                                 /* TRUE when valid */
cmSdIsValid(
    const CMBlob *pSd   /* descriptor to check */
    );

/* user identification structure (access token) contains all information for
 * querying particular access of user to an object protected by an ACL */

typedef struct
{
    NQ_BOOL isAdmin;                        /* is user admin */
    NQ_BOOL isAnon;                         /* is user anonymous */
    CMSdDomainSid domain;                   /* domain SID */
    NQ_UINT32 numRids;                      /* actual number of RIDs */
    CMSdRid *rids;                          /* array of user's RIDs (user's rid first, group rid second, then all groups memberships) */
}
CMSdAccessToken;

/* determine user's access by ACL */
NQ_BOOL                     /* TRUE when allowed */
cmSdHasAccess(
    const void * token,         /* user access token (may be NULL) */
    const NQ_BYTE * sdData,         /* security descriptor data */
    NQ_UINT32 requestedAccess       /* desired access flags */
    );

NQ_BOOL                     /* TRUE when allowed  */
cmSdHasPermissions(
    NQ_UINT32 requestedAttributeFlags,
    NQ_UINT32 fileAccessGranted
    );

/* check whether user token contains sd's owner */
NQ_BOOL
cmSdIsOwnerSidInToken(
    const CMSdAccessToken* token,
    const NQ_BYTE* sd
    );

/* determine user access by ACL */
NQ_UINT32 cmSdGetAccess(const CMBlob * sd, void * token         /* user access token (may be NULL) */
    );

/* create new security descriptor from parent security descriptor */
NQ_UINT32 cmSdInherit(const CMBlob * parentSd, const CMBlob * receivedSd, const void * userToken, NQ_BOOL isContainer, CMBlob * resultSd);

NQ_UINT32 cmSdCreateModifySecurityDescriptor(const CMBlob * oldSd, const CMBlob * receivedSd, const void * userToken, NQ_BOOL isContainer, NQ_BOOL isOnlyModifySD, CMBlob * resultSd);
NQ_UINT32 cmSdModifySecurityDescriptor(const CMBlob * oldSd, const CMBlob * newSd, const void * userToken, NQ_BOOL isContainer, CMBlob * resultSd);

/* check whether SD has OWNER or GROUP or DACL to set  */
NQ_BOOL cmSdHasDataToSet(const CMBlob * sd);

/* Map generic access rights to object specific access rights */
void cmSdMapAccessRights(NQ_BYTE* sdData);

/* determine if this user has administrative access */
NQ_BOOL                                 /* TRUE when allowed */
cmSdIsAdministrator(
    const void * userToken              /* user access token */
    );

/* get default SD with no owner */
NQ_BOOL                                 /* TRUE on success */
cmSdGetDefaultSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get default security descriptor according to user type.
 * user type is added by flags */
NQ_BOOL
cmSdGetDefaultSecurityDescriptorByFlags(
        const NQ_UINT32 flags,
        CMBlob *pSd
);

/* get SD with owner (and DACL) */
NQ_BOOL                                 /* TRUE on success */
cmSdGetOwnerAndDaclSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get SD with group (and DACL) */
NQ_BOOL                                 /* TRUE on success */
cmSdGetGroupAndDaclSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get default SD for a share */
NQ_BOOL                                 /* TRUE on success */
cmSdGetShareSecurityDescriptor(
    CMBlob *pSd                             /* buffer for descriptor */
    );

NQ_BOOL
cmSdGetShareSecurityDescriptorByToken(
    CMBlob* pSd,
    const void * userToken
    );

/* get empty SD for a share */
NQ_BOOL                                 /* TRUE on success */
cmSdGetEmptyShareSecurityDescriptor(
    CMBlob *psd                         /* buffer for descriptor */
    );
    
/* get "non-supported" SD */
NQ_BOOL                                 /* TRUE on success */
cmSdGetNoneSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get "administrative access only" SD */
NQ_BOOL                                 /* TRUE on success */
cmSdGetOwnerSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get "group access only" SD */
NQ_BOOL                                 /* TRUE on success */
cmSdGetGroupSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get "administrative and group access only" SD */
NQ_BOOL                                 /* TRUE on success */
cmSdGetOwnerAndGroupSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get "administrative, group and DACL access SD */
NQ_BOOL                                 /* TRUE on success */
cmSdGetOwnerAndGroupAndDaclSecurityDescriptor(
    CMBlob *pSd                             /* buffer for descriptor */
    );

/* get read only SD for a share */
NQ_BOOL                                 /* TRUE on success */
cmSdGetReadonlyShareSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );

/* get admin only SD for a share */
NQ_BOOL                                 /* TRUE on success */
cmSdGetAdminonlyShareSecurityDescriptor(
    CMBlob *pSd                         /* buffer for descriptor */
    );
    
/* get default ACE for group Everyone */
const NQ_BYTE*
cmSdGetEveryoneGroupACE(
    void
    );

/* get default ACE for group Administrators */
const NQ_BYTE*
cmSdGetAdministratorsGroupACE(
    void
    );

/* get default ACE for group Users */
const NQ_BYTE*
cmSdGetUsersGroupACE(
    void
    );

/* get default SD by token */
NQ_BOOL                                 /* TRUE on success */
cmSdGetDefaultSecurityDescriptorByToken(
    const void * userToken,       /* pointer to owner's handle */
    CMBlob *pSd                         /* buffer for descriptor */
    );

NQ_BOOL
cmSdGetDefaultSecurityDescriptorByTokenWinStyle(
    const void * userToken,
    CMBlob *pSd
    );

/* get default SD for user */
NQ_BOOL                                 /* TRUE on success */
cmSdGetLocalSecurityDescriptorForUser(
    CMSdRid rid,                        /* owner's RID */
    CMSdSecurityDescriptor* pSd         /* buffer for descriptor */
    );

/* set host domain's SID */
void
cmSdSetDomainSid(
    const CMSdDomainSid* sid        /* SID to set */
    );

/* Get host domain's SID */
const CMSdDomainSid*                /* domain SID */
cmSdGetDomainSid(
    void
    );

/* Get local domain SID alias */
const CMSdDomainSid*                /* domain SID */
cmSdGetLocalDomainAlias(
    void
    );

/* Get computer SID */
const CMSdDomainSid*                /* computer SID */
cmSdGetComputerSid(
    void
    );

/* checks if domain SID was set */
NQ_BOOL                             /* TRUE if already set */
cmSdIsDomainSidSet(
    void
    );

/* parse domain SID */
void
cmSdParseSid(
    CMRpcPacketDescriptor* in,      /* incoming packet descriptor */
    CMSdDomainSid* sid              /* buffer for SID */
    );

/* pack domain SID */
void
cmSdPackSid(
    CMRpcPacketDescriptor* out,     /* outgoing packet descriptor */
    const CMSdDomainSid* sid        /* SID to pack */
    );

/* pack full SID including domain SID and user RID */
void
cmSdPackSidRid(
    CMRpcPacketDescriptor* out,
    const CMSdDomainSid* sid,
    NQ_UINT32 rid
    );

/* parse Security Descriptor */
void
cmSdParseSecurityDescriptor(
    CMRpcPacketDescriptor* in,      /* incoming packet descriptor */
    CMBlob *pSd                         /* buffer for SD */
    );

/* pack Security Descriptor */
void
cmSdPackSecurityDescriptor(
    CMRpcPacketDescriptor* out,         /* outgoing packet descriptor */
    const CMBlob* pSd,  /* SD to pack */
    NQ_UINT32 flags
    );

/* parse Access Control List */
void
cmSdParseAcl(
    CMRpcPacketDescriptor* in,      /* incoming packet descriptor */
    CMSdAcl* pAcl,                  /* buffer for ACL */
    const NQ_BYTE* limit            /* the highest address to use */
    );

/* pack Access Control List */
void
cmSdPackAcl(
    CMRpcPacketDescriptor* out,         /* outgoing packet descriptor */
    const CMSdAcl* pAcl                 /* ACL to pack */
    );

/* parse Access Control Entry */
void
cmSdParseAce(
    CMRpcPacketDescriptor* in,      /* incoming packet descriptor */
    CMSdAce* pAce                   /* buffer for ACE */
    );

/* pack Access Control Entry */
void
cmSdPackAce(
    CMRpcPacketDescriptor* out,         /* outgoing packet descriptor */
    const CMSdAce* pAce                 /* ACE to pack */
    );

/* checks if the required SID is "Any" domain SID */

NQ_BOOL                         /* TRUE on match */
cmSdIsAnySid(
    const CMSdDomainSid* sid    /* SID to match */
    );

/* checks if the required SID is computer SID */

NQ_BOOL                         /* TRUE on match */
cmSdIsComputerSid(
    const CMSdDomainSid* sid    /* SID to match */
    );

/* check an alias */

NQ_BOOL                         /* TRUE when we support this alias */
cmSdCheckAlias(
    const CMSdDomainSid* sid,   /* SID + alias RID */
    CMSdRid* alias              /* buffer for alias RID */
    );

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

/* find local name by RID */

NQ_BOOL                         /* TRUE when found */
cmSdLookupRid(
    CMSdRid rid,                /* rid */
    NQ_WCHAR* nameBuffer,       /* name buffer */
    NQ_WCHAR* fullNameBuffer    /* name buffer */
    );

/* find RID by local name */

NQ_BOOL                         /* TRUE when found */
cmSdLookupName(
    const NQ_WCHAR* name,       /* name */
    CMSdRid* rid                /* rid buffer */
    );

/* get RID type */

NQ_UINT32                       /* RID type */
cmSdGetRidType(
    CMSdRid rid                 /* rid */
    );

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

/* create SD with exclusive rights for a given user */

NQ_BOOL                         /* TRUE on success, FALSE on failure */
cmSdCreateExclusiveSecurityDescriptor(
    const CMSdAccessToken* token,    /* user token */
    CMSdSecurityDescriptor* sd       /* buffer for the result */
    );

/* check if the given SD has exclusive rights for a given user */

NQ_BOOL                         /* TRUE when exclusive */
cmSdIsExclusiveSecurityDescriptor(
    const CMSdAccessToken* token,       /* user token */
    const CMSdSecurityDescriptor* sd    /* descriptor to check */
    );

NQ_BOOL
cmCheckAnonymousAccess(
    CMBlob * CurrSd,
    CMBlob * sd
);

NQ_BOOL                         /* TRUE on success, FALSE on failure */
cmSdCompareToken(
    const CMSdAccessToken* token1,      /* user token1 */
    const CMSdAccessToken* token2       /* user token2 */
    );
    
#ifdef UD_NQ_INCLUDETRACE
void cmSdDumpDomainSid(const NQ_CHAR *title, const CMSdDomainSid *sid);
void cmSdDumpSecurityDescriptor(const NQ_WCHAR *shareName, const CMSdSecurityDescriptor *sd, NQ_COUNT sdLen);
void cmSdDumpAccessToken(const CMSdAccessToken *token);
#endif /* UD_NQ_INCLUDETRACE */

#else /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP)|| defined(UD_CS_INCLUDEPASSTHROUGH)*/

/* a placeholder for user identification structure (access token) */

typedef struct
{
    NQ_UINT32 dummy;
    NQ_BOOL isAnon; /* is user anonymous*/
}
CMSdAccessToken;

/* determine user access by ACL */
NQ_BOOL                     /* TRUE when allowed */
cmSdHasAccess(
    const void * token,         /* user access token (may be NULL) */
    const NQ_BYTE * sdData,         /* security descriptor data */
    NQ_UINT32 requestedAccess       /* desired access flags */
    );

#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)*/

#endif  /* _CMSDESCR_H_ */
