/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : RPC packet parsing and creation
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmapi.h"
#include "cmrpcdef.h"

/*====================================================================
 * PURPOSE: attach data to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to data
 *          IN  byte order indicator (TRUE if network)
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcSetDescriptor(
    CMRpcPacketDescriptor *pDesc,
    NQ_IOBufPos pData,
    NQ_BOOL nbo
    )
{
    pDesc->origin = pData;
    pDesc->nbo = nbo;
    pDesc->token = NULL;
    cmRpcResetDescriptor(pDesc);
    pDesc->length = 0;
}

/*====================================================================
 * PURPOSE: attach data and user token to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to data
 *          IN  byte order indicator (TRUE if network)
 *          IN  pointer to user token
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcSetTokenDescriptor(
    CMRpcPacketDescriptor *pDesc,
    NQ_IOBufPos pData,
    NQ_BOOL nbo,
    NQ_BYTE *token
    )
{
    pDesc->origin = pData;
    pDesc->nbo = nbo;
    pDesc->token = token;
    cmRpcResetDescriptor(pDesc);
}

/*====================================================================
 * PURPOSE: copy descriptor fields to another descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the source descriptor
 *          IN  pointer to the destination descriptor
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcCloneDescriptor(
    CMRpcPacketDescriptor *pSrc,
    CMRpcPacketDescriptor *pDst
    )
{
    pDst->origin = pSrc->origin;
    pDst->current = pSrc->current;
    pDst->length = pSrc->length;
    pDst->token = pSrc->token;
    pDst->nbo = pSrc->nbo;
}

/*====================================================================
 * PURPOSE: reset descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcResetDescriptor(
    CMRpcPacketDescriptor *pDesc
    )
{
    pDesc->current = pDesc->origin;
}

/*====================================================================
 * PURPOSE: set descriptor byte order
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  byte order indicator (TRUE if network)
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcDescriptorSetByteOrder(
    CMRpcPacketDescriptor *pDesc,
    NQ_BOOL nbo
    )
{
    pDesc->nbo = nbo;
}


/*====================================================================
 * PURPOSE: read one byte from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place a byte in
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseByte(
    CMRpcPacketDescriptor *pDesc,
    NQ_BYTE *pRes
    )
{
    *pRes = *(IOBUF_GETBYTEPTR(pDesc->current));
    IOBUF_MOVEBYTES(pDesc->current, sizeof(NQ_BYTE));
}

/*====================================================================
 * PURPOSE: read UINT16 from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place UINT16 value in
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseUint16(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT16 *pRes
    )
{
    NQ_UINT16 temp;
    cmIOGetUint16(temp, pDesc->current);
    *pRes = (NQ_UINT16)((pDesc->nbo)? syNtoh16(temp) : cmLtoh16(temp));

    IOBUF_MOVEBYTES(pDesc->current, 2);
}

/*====================================================================
 * PURPOSE: read UINT16 from a descriptor by Byte Order
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place UINT16 value in
 *          IN  Network byte order or host
 *
 * RETURNS: NONE
 *
 * NOTES: Parse the value ignoring Byte Order set in descriptor
 *====================================================================
 */
void
cmRpcParseUint16ByByteOrder(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT16 *pRes,
    NQ_BOOL nbo
    )
{
    NQ_UINT16 temp;
    cmIOGetUint16(temp, pDesc->current);
    *pRes = (NQ_UINT16)(nbo ? syNtoh16(temp) : cmLtoh16(temp));

    IOBUF_MOVEBYTES(pDesc->current, 2);
}

/*====================================================================
 * PURPOSE: read UINT32 from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place UINT32 value in
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseUint32(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT32 *pRes
    )
{
    NQ_UINT32 temp;
    cmIOGetUint32(temp, pDesc->current);
    *pRes = (NQ_UINT32)((pDesc->nbo)? syNtoh32((NQ_UINT)temp) : cmLtoh32(temp));

    IOBUF_MOVEBYTES(pDesc->current, 4);
}

/*====================================================================
 * PURPOSE: read UINT32 from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place UINT32 value in
 *          IN  Network byte order or host
 *
 * RETURNS: NONE
 *
 * NOTES: Parse the value ignoring Byte Order set in descriptor
 *====================================================================
 */
void
cmRpcParseUint32ByByteOrder(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT32 *pRes,
    NQ_BOOL nbo
    )
{
    NQ_UINT32 temp;
    cmIOGetUint32(temp, pDesc->current);
    *pRes = (NQ_UINT32)(nbo ? syNtoh32((NQ_UINT)temp) : cmLtoh32(temp));

    IOBUF_MOVEBYTES(pDesc->current, 4);
}

/*====================================================================
 * PURPOSE: read UINT64 from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place UINT64 value in
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseUint64(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT64 *pRes
    )
{
    if (pDesc->nbo)
    {
        cmRpcParseUint32(pDesc, &pRes->high);
        cmRpcParseUint32(pDesc, &pRes->low);
    }
    else
    {
        cmRpcParseUint32(pDesc, &pRes->low);
        cmRpcParseUint32(pDesc, &pRes->high);
    }
}

/*void
cmRpcCopyParseUnicode(
    CMRpcPacketDescriptor *pDesc,
    CMRpcUnicodeString *pRes,
    NQ_WCHAR *destName,
    NQ_UINT16 flags
    )
{
    NQ_BOOL calculateSize = FALSE;

    if (flags & CM_RP_SIZE32)
    {
        cmRpcParseUint32(pDesc, &pRes->size);
    }
    else
    {
        calculateSize = TRUE;
    }

    if (flags & CM_RP_FRAGMENT32)
    {
        cmRpcParseUint32(pDesc, &pRes->offset);
        cmRpcParseUint32(pDesc, &pRes->length);
    }
    else
    {
        pRes->offset = 0;
        pRes->length = pRes->size;
    }

    IOBUF_WSTRCPY_V2F(pDesc->current, destName);

    if (calculateSize)
    {
        pRes->size = cmWStrlen(destName) + 1;
    }
    pRes->length = pRes->size;

    pRes->text = (NQ_WCHAR*)pDesc->current;
    IOBUF_MOVEBYTES(pDesc->current, (pRes->length * sizeof(NQ_WCHAR)));

    if ((pRes->length % 2) == 1)
        IOBUF_MOVEBYTES(pDesc->current, sizeof(NQ_WCHAR));   undocumented
}*/

/*====================================================================
 * PURPOSE: read unicode string from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place the unicode string in
 *          IN  flags
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseUnicode(
    CMRpcPacketDescriptor *pDesc,
    CMRpcUnicodeString *pRes,
    NQ_UINT16 flags
    )
{
    NQ_BOOL calculateSize = FALSE;

    if (flags & CM_RP_SIZE32)
    {
        cmRpcParseUint32(pDesc, &pRes->size);   /* max count */
    }
    else
    {
        calculateSize = TRUE;
    }

    if (flags & CM_RP_FRAGMENT32)
    {
        cmRpcParseUint32(pDesc, &pRes->offset); /* offset */
        cmRpcParseUint32(pDesc, &pRes->length); /* actual count */
    }
    else
    {
        pRes->offset = 0;
        pRes->length = pRes->size;
    }

    if (calculateSize)
    {
        pRes->size = IOBUF_WSTRLEN(pDesc->current) + 1;
    }

    IOBUF_BUFFER_CANUSEFLAT_ASSERT(pDesc->current, pRes->length);

    pRes->text = (NQ_WCHAR*) IOBUF_GETBYTEPTR(pDesc->current);
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)(pRes->length * sizeof(NQ_WCHAR)));

    if ((pRes->length % 2) == 1)
        IOBUF_MOVEBYTES(pDesc->current, sizeof(NQ_WCHAR));  /* undocumented */
}

/*====================================================================
 * PURPOSE: read ascii string from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place the ascii string in
 *          IN  flags
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseAscii(
    CMRpcPacketDescriptor *pDesc,
    CMRpcAsciiString *pRes,
    NQ_UINT16 flags
    )
{
    NQ_BOOL calculateSize = FALSE;

    if (flags & CM_RP_SIZE32)
    {
        cmRpcParseUint32(pDesc, &pRes->size);
    }
    else
    {
        if (flags & CM_RP_SIZE16)
        {
            NQ_UINT16 temp;

            cmRpcParseUint16(pDesc, &temp);
            pRes->size = (NQ_UINT32)temp;
        }
        else
        {
            calculateSize = TRUE;
        }
    }

    if (flags & CM_RP_FRAGMENT32)
    {
        cmRpcParseUint32(pDesc, &pRes->offset);
        cmRpcParseUint32(pDesc, &pRes->length);
    }
    else
    {
        pRes->offset = 0;
        pRes->length = pRes->size;
    }

    if (calculateSize)
    {
        pRes->size = (NQ_UINT32)IOBUF_STRLEN(pDesc->current);
    }
    pRes->length = pRes->size;

    IOBUF_BUFFER_CANUSEFLAT_ASSERT(pDesc->current, pRes->size);

    pRes->text = (NQ_CHAR *)IOBUF_GETBYTEPTR(pDesc->current);
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)pRes->size);
}

/*====================================================================
 * PURPOSE: read bytes from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  address to place the bytes in
 *          IN  number of bytes to read
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseBytes(
    CMRpcPacketDescriptor *pDesc,
    NQ_BYTE *pRes,
    NQ_UINT32 num
    )
{
    IOBUF_MEMCPY_V2F(pRes, pDesc->current, num);
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)num);
}

/*====================================================================
 * PURPOSE: skip bytes in a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  number of bytes to skip
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseSkip(
    CMRpcPacketDescriptor *pDesc,
    NQ_INT32 num
    )
{
    IOBUF_MOVEBYTES(pDesc->current, num);
}

/*====================================================================
 * PURPOSE: make alignment in a descriptor's buffer
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  number of bytes to align to
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
NQ_COUNT
cmRpcAllign(
    CMRpcPacketDescriptor *pDesc,
    NQ_INT align
    )
{
    NQ_INT offset = (NQ_INT)IOBUF_GETPOINTERSDIF(pDesc->current, pDesc->origin);
    NQ_INT mod = (NQ_INT)((offset + align) % align);
    NQ_INT add = (NQ_INT)((align - mod) % align);

    IOBUF_MOVEBYTES(pDesc->current, add);
    return (NQ_COUNT)add;
}

/*====================================================================
 * PURPOSE: read UUID from a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to UUID structure
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcParseUuid(
    CMRpcPacketDescriptor *pDesc,
    CMRpcUuid *pUuid
    )
{
    NQ_UINT32 temp32;
    NQ_UINT16 temp16;

    cmRpcParseUint32(pDesc, &temp32);
    cmPutSUint32(pUuid->timeLow, temp32);
    cmRpcParseUint16(pDesc, &temp16);
    cmPutSUint16(pUuid->timeMid, temp16);
    cmRpcParseUint16(pDesc, &temp16);
    cmPutSUint16(pUuid->timeHiVersion, temp16);
    cmRpcParseBytes(pDesc, pUuid->clockSeq, sizeof(pUuid->clockSeq));
    cmRpcParseBytes(pDesc, pUuid->node, sizeof(pUuid->node));
}

/*====================================================================
 * PURPOSE: write a byte to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  value to write
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcPackByte(
    CMRpcPacketDescriptor *pDesc,
    NQ_BYTE src
    )
{
    *IOBUF_GETBYTEPTR(pDesc->current) = src;
    IOBUF_MOVEBYTES(pDesc->current, sizeof(NQ_BYTE));
}

/*====================================================================
 * PURPOSE: write a UINT16 value to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  value to write
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcPackUint16(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT16 src
    )
{
    NQ_UINT16 temp = (NQ_UINT16)((pDesc->nbo)? syHton16(src) : cmHtol16(src));

    cmIOPutUint16(pDesc->current, temp);
    IOBUF_MOVEBYTES(pDesc->current, sizeof(NQ_UINT16));
}

/*====================================================================
 * PURPOSE: write a UINT16 value by Byte Order
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  value to write
 *          IN  Network byte order or host
 *
 * RETURNS: NONE
 *
 * NOTES: Pack the value ignoring Byte Order set in descriptor
 *====================================================================
 */
void
cmRpcPackUint16ByByteOrder(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT16 src,
    NQ_BOOL nbo
    )
{
    NQ_UINT16 temp = (NQ_UINT16)(nbo ? syHton16(src) : cmHtol16(src));

    cmIOPutUint16(pDesc->current, temp);
    IOBUF_MOVEBYTES(pDesc->current, sizeof(NQ_UINT16));
}

/*====================================================================
 * PURPOSE: write a UINT32 value to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  value to write
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcPackUint32(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT32 src
    )
{
    NQ_COUNT temp = (NQ_UINT32)((pDesc->nbo)? syHton32((NQ_UINT)src) : cmHtol32(src));

    cmIOPutUint32(pDesc->current, temp);
    IOBUF_MOVEBYTES(pDesc->current, sizeof(src));
}

/*====================================================================
 * PURPOSE: write a UINT32 value by Byte Order
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  value to write
 *          IN  Network byte order or host
 *
 * RETURNS: NONE
 *
 * NOTES: Pack the value ignoring Byte Order set in descriptor
 *====================================================================
 */
void
cmRpcPackUint32ByByteOrder(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT32 src,
    NQ_BOOL nbo
    )
{
    NQ_COUNT temp = (NQ_UINT32)(nbo ? syHton32((NQ_UINT)src) : cmHtol32(src));

    cmIOPutUint32(pDesc->current, temp);
    IOBUF_MOVEBYTES(pDesc->current, sizeof(src));
}

void
cmRpcPackInt32(
    CMRpcPacketDescriptor *pDesc,
    NQ_INT32 src
    )
{
    NQ_COUNT temp = (NQ_UINT32)((pDesc->nbo)? syHton32((NQ_UINT)src) : cmHtol32((NQ_UINT)src));

    cmIOPutUint32(pDesc->current, temp);
    IOBUF_MOVEBYTES(pDesc->current, sizeof(src));
}


/*====================================================================
 * PURPOSE: write a 64-bit value to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  low 32 bits of the value to write
 *          IN  high 32 bits of the value to write
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

void
cmRpcPackUint64(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT32 low,
    NQ_UINT32 high
    )
{
    if (pDesc->nbo)
    {
        cmRpcPackUint32(pDesc, high);
        cmRpcPackUint32(pDesc, low);
    }
    else
    {
        cmRpcPackUint32(pDesc, low);
        cmRpcPackUint32(pDesc, high);
    }
}

/*====================================================================
 * PURPOSE: write a 32-bit time value as 64-bit UTC time
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  time to write
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

void cmRpcPackTimeAsUTC(
    CMRpcPacketDescriptor *pDesc,
    NQ_TIME time
    )
{
    NQ_UINT32 timeLow;     /* low part of UTC time */
    NQ_UINT32 timeHigh;    /* high part of UTC time */

    cmCifsTimeToUTC(time, &timeLow, &timeHigh);
    cmRpcPackUint64(pDesc, timeLow, timeHigh);
}

/*====================================================================
 * PURPOSE: write a unicode string to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to the unicode string to write
 *          IN  flags
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
NQ_UINT32 cmRpcPackUnicode(CMRpcPacketDescriptor *pDesc, const NQ_WCHAR *str, NQ_UINT16 flags)
{
    NQ_UINT32 size = cmWStrlen(str),
              fullSize = (flags & CM_RP_NULLTERM)? size + 1 : size;
    
    if (flags & CM_RP_SIZE32)
    {
        NQ_UINT32 m = (flags & CM_RP_INCMAXCOUNT)? fullSize + 1 : fullSize;
        cmRpcPackUint32(pDesc, m);                                                      /* Max count */
    }

    if (flags & CM_RP_FRAGMENT32)
    {
        cmRpcPackUint32(pDesc, 0);                                                      /* Offset */
        cmRpcPackUint32(pDesc,  (flags & CM_RP_DECACTCOUNT)? fullSize - 1 : fullSize);  /* Actual count */
    }

    if (pDesc->nbo) 
    {
        NQ_COUNT i;
        for (i = 0; i < size; i++)
        {
#ifdef SY_LITTLEENDIANHOST  
            cmRpcPackUint16(pDesc, str[i]);
#else
            NQ_UINT16 temp = (NQ_UINT16)(cmLtoh16(str[i]));
            cmPutUint16(pDesc->current, temp);
            pDesc->current += 2;
#endif
        }
    }
    else
    {
        IOBUF_MEMCPY_F2V(pDesc->current, (const NQ_BYTE*)str, (NQ_COUNT)(size * sizeof(NQ_WCHAR)));
        IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)(size * sizeof(NQ_WCHAR)));
    }
    
    if (flags & CM_RP_NULLTERM)
    {
#ifndef UD_NQ_USEIOVECS
        *(pDesc->current)++ = 0;
        *(pDesc->current)++ = 0;
#else /* UD_NQ_USEIOVECS */
        *IOBUF_GETBYTEPTR(pDesc->current) = 0;
        IOBUF_MOVEBYTES(pDesc->current, 1);
        *IOBUF_GETBYTEPTR(pDesc->current) = 0;
        IOBUF_MOVEBYTES(pDesc->current, 1);
#endif /* UD_NQ_USEIOVECS */
    }

    if (flags & CM_RP_SIZE32)
    {
        cmRpcAllignZero(pDesc, 4);
    }
    return 0;
}

/*====================================================================
 * PURPOSE: write a unicode string as ascii to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to the unicode string to write
 *          IN  flags
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
NQ_UINT32 cmRpcPackUnicodeAsAscii(CMRpcPacketDescriptor *pDesc, const NQ_WCHAR *str, NQ_UINT16 flags)
{
    NQ_CHAR * temp = cmMemoryCloneWStringAsAscii(str);
    NQ_UINT32 result = (NQ_UINT32)NQ_FAIL;

    if (NULL == temp)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    cmRpcPackAscii(pDesc, temp, flags);
    cmMemoryFree(temp);
    result = NQ_SUCCESS;

Exit:
    return result;
}

/*====================================================================
 * PURPOSE: write an ascii string to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to the ascii string to write
 *          IN  flags
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcPackAscii(
    CMRpcPacketDescriptor *pDesc,
    const NQ_CHAR *str,
    NQ_UINT16 flags
    )
{
    NQ_UINT32 size = (NQ_UINT32)syStrlen(str),
              fullSize = (flags & CM_RP_NULLTERM)? size + 1 : size;

    if (flags & CM_RP_SIZE32)
    {
        cmRpcPackUint32(pDesc, fullSize);
    }
    else
    {
        if (flags & CM_RP_SIZE16)
        {
            cmRpcPackUint16(pDesc, (NQ_UINT16)fullSize);
        }
    }

    if (flags & CM_RP_FRAGMENT32)
    {
        cmRpcPackUint32(pDesc, 0);
        cmRpcPackUint32(pDesc, fullSize);
    }

    IOBUF_MEMCPY_F2V(pDesc->current, (const NQ_BYTE*)str, size);
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)size);

    if (flags & CM_RP_NULLTERM)
    {
        cmRpcPackByte(pDesc, 0);
    }
}

/*====================================================================
 * PURPOSE: write bytes to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to the bytes to write
 *          IN  number of bytes to write
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcPackBytes(
    CMRpcPacketDescriptor *pDesc,
    const NQ_BYTE *pRes,
    NQ_UINT32 num
    )
{
    IOBUF_MEMCPY_F2V(pDesc->current, pRes, num);
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)num);
}

void
cmRpcPackIOBytes(
    CMRpcPacketDescriptor *pDesc,
    NQ_IOBufPos pRes,
    NQ_UINT32 num
    )
{
    IOBUF_MEMCPY_V2V(pDesc->current, pRes, num);
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)num);
}


/*====================================================================
 * PURPOSE: skip bytes in a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  number of bytes to skip
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcPackSkip(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT32 num
    )
{
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)num);
}


/*====================================================================
* PURPOSE: pack zeroes in a descriptor
*--------------------------------------------------------------------
* PARAMS:  IN  pointer to the descriptor
*          IN  number of zeroes to pack
*
* RETURNS: NONE
*
* NOTES:
*====================================================================
*/
void
cmRpcPackZeroes(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT32 num
    )
{
    for (; num > 0; num--)
    {
#ifndef UD_NQ_USEIOVECS
        *(pDesc->current)++ = 0;
#else
        *IOBUF_GETBYTEPTR(pDesc->current) = 0;
        IOBUF_MOVEBYTES(pDesc->current, sizeof(NQ_BYTE));
#endif
    }
}

/*====================================================================
 * PURPOSE: align and write zeros
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  number of bytes to align to
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcAllignZero(
    CMRpcPacketDescriptor *pDesc,
    NQ_UINT16 align
    )
{
    NQ_INT offset = (NQ_INT)cmRpcGetDataCount(pDesc);
    NQ_INT mod = (offset + align) % align;
    NQ_INT add = (align - mod) % align;

    IOBUF_MEMSET(pDesc->current, 0, (NQ_COUNT)add);
    IOBUF_MOVEBYTES(pDesc->current, (NQ_INT)add);
}

/*====================================================================
 * PURPOSE: write UUID to a descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN  pointer to the descriptor
 *          IN  pointer to the UUID to write
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */
void
cmRpcPackUuid(
    CMRpcPacketDescriptor *pDesc,
    const CMRpcUuid *pUuid
    )
{
    cmRpcPackUint32(pDesc, cmGetSUint32(pUuid->timeLow));
    cmRpcPackUint16(pDesc, cmGetSUint16(pUuid->timeMid));
    cmRpcPackUint16(pDesc, cmGetSUint16(pUuid->timeHiVersion));
    cmRpcPackBytes(pDesc, pUuid->clockSeq, sizeof(pUuid->clockSeq));
    cmRpcPackBytes(pDesc, pUuid->node, sizeof(pUuid->node));
}

void
cmRpcPackNQUuid(
    CMRpcPacketDescriptor *pDesc,
    const NQ_Uuid *pUuid
    )
{
    cmRpcPackUint32(pDesc, cmGetSUint32(pUuid->timeLow));
    cmRpcPackUint16(pDesc, cmGetSUint16(pUuid->timeMid));
    cmRpcPackUint16(pDesc, cmGetSUint16(pUuid->timeHiVersion));
    cmRpcPackBytes(pDesc, pUuid->clockSeq, sizeof(pUuid->clockSeq));
    cmRpcPackBytes(pDesc, pUuid->node, sizeof(pUuid->node));
}

NQ_UINT32 cmRpcSpace(CMRpcPacketDescriptor *pDesc)
{
    return (NQ_UINT32)(pDesc->length - cmRpcGetDataCount(pDesc));
}

/*====================================================================
 * PURPOSE: pack ascii string as Unicode string
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT packet descriptor to use
 *          IN string to pack
 *          IN pack flags ad required by cmRpcPackUnicode()
 *
 * RETURNS: 0 or error code on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32 cmRpcPackAsciiAsUnicode(
    CMRpcPacketDescriptor * desc,
    const NQ_CHAR * source,
    NQ_INT flags
    )
{
    NQ_WCHAR * temp = cmMemoryCloneAString(source);
    NQ_UINT32 result = (NQ_UINT32)NQ_FAIL;

    if (NULL == temp)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    cmRpcPackUnicode(desc, temp, (NQ_UINT16)flags);
    cmMemoryFree(temp);
    result = NQ_SUCCESS;

Exit:
    return result;
}

/*====================================================================
 * PURPOSE: pack WCHAR string as Unicode string
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT packet descriptor to use
 *          IN string to pack
 *          IN pack flags ad required by cmRpcPackUnicode()
 *
 * RETURNS: 0 or error code on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32 cmRpcPackWcharAsUnicode(
    CMRpcPacketDescriptor * desc,
    const NQ_WCHAR * source,
    NQ_INT flags
    )
{
    const NQ_WCHAR * temp = source;

    if (source != NULL)
    {
        cmRpcPackUnicode(desc, temp, (NQ_UINT16)flags);
    }
    else
    {
        static const NQ_WCHAR nullStr[] = {0};
        temp = nullStr;
        cmRpcPackUnicode(desc, temp, (NQ_UINT16)flags);
    }
    return NQ_SUCCESS;
}

NQ_COUNT cmRpcGetDataCount(
    const CMRpcPacketDescriptor * desc
    )
{
    return (NQ_COUNT)IOBUF_GETPOINTERSDIF(desc->current, desc->origin);
}

/*====================================================================
 * PURPOSE: Get remaining data count
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT packet descriptor to use
 *
 * RETURNS: Number of bytes available to read from buffer or to write
 *          into the buffer
 *
 *====================================================================
 */
NQ_COUNT cmRpcGetRemaining(
    const CMRpcPacketDescriptor * desc
    )
{
    return (desc->length - cmRpcGetDataCount(desc));
}

/*  Description: get current position plus offset.
 *  When offset is 0 returns current position.
 *  Params:
 *      dsec - in - packet descriptor
 *      offsetFromCurrent - in - offset from currenct position
 *      position - out - the returned position.
 *
 *  Returne:
 *      returned position. same as position parameter.
 */
NQ_IOBufPos cmRpcGetPosition(
    const CMRpcPacketDescriptor * desc,
    NQ_COUNT offsetFromCurrent,
    NQ_IOBufPos *position)
{
    NQ_IOBufPos tmpPos = desc->current;
    IOBUF_MOVEBYTES(tmpPos, (NQ_INT)offsetFromCurrent);
    if (position != NULL)
        *position = tmpPos;

    return tmpPos;
}

/* for functinos that will use this pointer */
NQ_BYTE * cmRpcGetPositionBytePtr(
    const CMRpcPacketDescriptor * desc,
    NQ_COUNT offsetFromCurrent,
    NQ_COUNT requestSizeVerify)
{
    NQ_IOBufPos tmpIOBufPos = desc->current;
    if (desc->length > 0 && (desc->length - cmRpcGetDataCount(desc) < requestSizeVerify))
    {
        return NULL;
    }

    IOBUF_MOVEBYTES(tmpIOBufPos, (NQ_INT)offsetFromCurrent);
#ifdef UD_NQ_USEIOVECS
        /* verify size doesnt overflow iovec fragment */
    if (tmpIOBufPos.byteOffset + requestSizeVerify > tmpIOBufPos.ioBuf->iovec[tmpIOBufPos.segment].iov_len)
    {
        /*can't use this position as flat buffer */
        syAssert(FALSE);
        return NULL;
    }
#endif
    return (IOBUF_GETBYTEPTR((tmpIOBufPos)));
}


NQ_IOBufPos cmRpcGetStartPosition(
    const CMRpcPacketDescriptor * desc,
    NQ_COUNT offsetFromStart,
    NQ_IOBufPos *position)
{
    NQ_IOBufPos tmpPos  = desc->origin;

    IOBUF_MOVEBYTES(tmpPos, (NQ_INT)offsetFromStart);

    if (position != NULL)
        *position = tmpPos;

    return tmpPos ;
}

NQ_BOOL cmRpcSetPosition(
    CMRpcPacketDescriptor * desc,
    NQ_IOBufPos position)
{
    desc->current = position;
    return TRUE;
}

/* when I know this byte pointer is on the same segment as current */
NQ_BOOL cmRpcSetPositionBytePtr(
    CMRpcPacketDescriptor * desc,
    NQ_BYTE *position)
{
#ifndef UD_NQ_USEIOVECS
    desc->current = position;
    return TRUE;
#else
    syAssert(IOBUF_BUFFER_ISINSEGMENT(desc->current, position));
    desc->current.byteOffset = (NQ_COUNT)(position - desc->current.ioBuf->iovec[desc->current.segment].iov_base);
    return TRUE;
#endif /* UD_NQ_USEIOVECS */
}

NQ_BOOL cmRpcSetPositionWithOffset(
    CMRpcPacketDescriptor * desc,
    NQ_IOBufPos position,
    NQ_COUNT offsetFromPosition)
{
    desc->current = position;
    IOBUF_MOVEBYTES(desc->current, (NQ_INT)offsetFromPosition);
    return TRUE;
}


NQ_BOOL
cmRpcCopyBytes(
    NQ_IOBufPos pSrc,
    NQ_IOBufPos pDst,
    NQ_UINT32 numBytes
    )
{
#ifdef UD_NQ_USEIOVECS
    IOBUF_MEMCPYBACK_V2V(pDst, pSrc, numBytes);
#else

    NQ_UINT i;
    for (i = numBytes; i>0; i--)
    {
        *--pDst = *--pSrc;
    }
#endif
    return TRUE;
}

NQ_INT cmRpcMemCmpCurrent(
    const CMRpcPacketDescriptor * desc,
    NQ_BYTE* data,
    NQ_COUNT compareSize
    )
{
    return IOBUF_MEMCMP_V2F(desc->current, data, compareSize);
}

NQ_COUNT cmRpcGetCountFromOffset(
    const CMRpcPacketDescriptor * desc,
    NQ_IOBufPos offset
    )
{
    return (NQ_COUNT)IOBUF_GETPOINTERSDIF(desc->current, offset);
}

NQ_BOOL
cmRpcIsCurrentPositionSmaller(
    const CMRpcPacketDescriptor * desc,
    NQ_IOBufPos position
    )
{
    return (IOBUF_ISPTRSMALLER(desc->current, position));
}

NQ_BOOL
cmRpcIsCurrentPositionBigger(
    const CMRpcPacketDescriptor * desc,
    NQ_IOBufPos position
    )
{
    return (IOBUF_ISPTRBIGGER(desc->current, position));
}
