/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common resolver operations
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _CMRESOLVER_H_
#define _CMRESOLVER_H_

#include "udparams.h"
#include "udapi.h"
#include "syapi.h"
#include "cmcommon.h"

/* -- API structures -- */

/* Description
   A descriptor of one resolver method. */
typedef struct 
{
        /* one of NQ_RESOLVER_NETBIOS or NQ_RESOLVER_DNS or NQ_RESOLVER_EXTERNAL_METHOD */
    NQ_INT type;
        /* This value should be <i>TRUE</i> for a multicast method. NQ uses unicast methods first.*/
    NQ_BOOL isMulticast;
        /* Methods will be activated according to priority, if first priority fails to resolve next priority is activated etc */
        /* for all unicast we set priority 2. for all multicast priority 4.
         * this way external method that is registered by user can be priority 1 or 3 or 5. */
    NQ_UINT activationPriority;
        /* timeout in milliseconds to use with this method */
    NQ_TIME timeout;
        /* wait for this method result even if another one has already succeeded */
    NQ_BOOL waitAnyway;
    /* Routine for composing and sending a name resolution request
       Parameters:
        socket :    Socket handle to use for sending
        name :      Name to resolve
        context :   Pointer to a method-specific context. This value may be NULL on the first call.
        serverIp :  Pointer to  the IP of the server to query or NULL for multicast
       Return:
        NQ_SUCCESS           request sent
        1-n                  a positive number means that more then one request was sent
        NQ_ERR_<*>           error
     */
    NQ_STATUS (* requestByName)(SYSocketHandle socket, const NQ_WCHAR * name, void * context, const NQ_IPADDRESS * serverIp, NQ_COUNT * numOfSentRequests);
    /* Routine for receiving and parsing a name resolution response
       Parameters
        socket:             Socket handle to use for sending
        pAddressArray:      Address of the pointer which this call sets to an array of
                            resolved IP addresses. It is caller's responsibility to release this array.
                            On error, this pointer remains untouched.
        numIps:             Pointer to the number of resolved IPs.
        pContext:           Double pointer to a method-specific context. Method may dispose
                            context and create a new one.
        numOfSentRequests:  Return the number of requests that were sent.

       Return:
        NQ_SUCCESS           name successfully resolved
        NQ_ERR_MOREDATA      more exchange expected
        NQ_ERR_NOACCESS      more comprehensive method with the same code should be used
        NQ_ERR_<*>           error
     */
    NQ_STATUS (* responseByName)(SYSocketHandle socket, NQ_IPADDRESS ** pAddressArray, NQ_INT * numIps, void ** pContext);
    /* Routine for composing and sending an IP resolution request
       Parameters:
        socket :    Socket handle to use for sending
        ip :        Pointer to the IP address to resolve.
        context :   Pointer to a method-specific context. This value may be NULL on the first call.
        serverIp :  Pointer to  the IP of the server to query or NULL for multicast
       Return:
       NQ_SUCCESS           request sent
       NQ_ERR_<*>           error
     */
    NQ_STATUS (* requestByIp)(SYSocketHandle socket, const NQ_IPADDRESS * ip, void * context, const NQ_IPADDRESS * serverIp, NQ_COUNT * numOfSentRequests);
    /* Routine for receiving and parsing a name resolution response

       Parameters
        socket:             Socket handle to use for sending
        pName:              Double pointer to the resolved name. On success, this variable will
                            point to a newly allocated name. Its is caller's responsibility to release it later.
        pContext:           Double pointer to a method-specific context. Method may dispose
                            context and create a new one.
        numOfSentRequests:  Return the number of requests that were sent.

       Return:
       NQ_SUCCESS           name successfully resolved
       NQ_ERR_MOREDATA      more exchange expected
       NQ_ERR_NOACCESS      more comprehensive method with the same code should be used
       NQ_ERR_<*>           error
     */
    NQ_STATUS (* responseByIp)(SYSocketHandle socket, const NQ_WCHAR ** pName, void ** pContext);
} 
CMResolverMethodDescriptor;

#define RESOLVER_MAX_ACTIVATION_PRIORITY 5

/* -- API Functions */

/* Description
   Initialize this module.
   Returns 
   None
 */
NQ_BOOL cmResolverStart(void);

/* Description
   Release resources used by this module.
   Returns 
   None
 */
void cmResolverShutdown(void);

/* Description
   This function registers a resolution method.

   Resolution method may be NetBIOS (Name Service), DNS or LLMNR. 
   Parameters
   descriptor : Pointer to a method descriptor. 
   serverIp :  Pointer to  the IP of the server to query or NULL for multicast
   Returns
   TRUE on success or FALSE on failure. */

NQ_BOOL cmResolverRegisterMethod(const CMResolverMethodDescriptor * descriptor, const NQ_IPADDRESS * serverIp);

/* Description
   This function removes a previously registered method.

   This function removes the first method that matches the following parameters:
   * type;
   * multicast flag on/off;
   * server IP. 

   if there are more methods matching the same parameters - they will remain registered.
   Parameters
   descriptor : Pointer to a method descriptor. 
   serverIp :  Pointer to  the IP of the server to query or NULL for multicast
   Returns
   None. */
void cmResolverRemoveMethod(const CMResolverMethodDescriptor * descriptor, const NQ_IPADDRESS * serverIp);

/*  Description
    This function sets resolver cache state.

    Resolver cache is enabled by default.

    Parameters
    timeout : timeout/ttl value for entry cache (0 to disable cache)
    Returns
    None. */
void cmResolverCacheSet(NQ_UINT32 timeout);

/*  Description
    This function remove cache entry.

    Parameters
    name: name of cache entry to remove.

    Returns
    None. */
void cmResolverRemoveFromCache(const NQ_WCHAR * name);

#ifdef UD_CM_INCLUDEWSDCLIENT

/* Description
   This function resolves devices (IPs and names, whatever the responses contain) using WS-Discovery method.

   The array of IP addresses is created in allocated memory so
   that it is caller's responsibility to free this memory.

   List of null terminated names is created in allocated memory so
   that it is caller's responsibility to free this memory.

   NQ uses multicast methods over IPv4 and IPv6 for this resolution.
   If those methods did not succeed, NQ calls external method if
   this method has been installed in <link cmResolverSetExternal>()
   call.
   Parameters
   deviceType :    Type of device (currently supported computers and printers).
   listIPs : Pointer to array of IP addresses.
   numIps :  Pointer to variable that on exit gets the number of
             received IP addresses.
   listNames : Pointer to list of null terminated names.
   numNames :  Pointer to variable that on exit gets the number of
             received names.
   Returns
   TRUE or FALSE.
   Note
   It is caller's responsibility to release both lists with <i>cmMemoryFree()</i> function.
   Please note that IPs and names returned can correspond to the same device, it's beyond this method's responsibility
   to resolve IPs into names or names into IPs. This method returns the WSD data only in the form it is received from the wire. */
NQ_BOOL cmResolverGetDevices(const NQ_UINT deviceType, const NQ_IPADDRESS **listIPs, NQ_INT *numIps, const NQ_WCHAR **listNames, NQ_INT *numNames);

/* Description
   This function sets timeout used while resolving devices IPs using WS-Discovery method.
   See <link cmResolverGetDevicesIps>() call.

   Parameters
   milliseconds :    Timeout in milliseconds to wait for responses.

   Returns
   TRUE or FALSE.
   Note
   Default timeout value is set by UD_CM_WSDCLIENTTIMEOUT.
   */
NQ_BOOL cmResolverSetTimeoutForDevicesIps(NQ_UINT32 milliseconds);

#endif /* UD_CM_INCLUDEWSDCLIENT */

#ifdef UD_NQ_INCLUDETRACE

void cmDumpIPsList(const NQ_IPADDRESS *pList, NQ_INT numOfIPs);

void cmDumpNamesList(NQ_WCHAR *list);

#endif /* UD_NQ_INCLUDETRACE */

#endif /* _CMRESOLVER_H_ */
