/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Code page module, CP API functions
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmapi.h"
#include "cmcp.h"

#ifdef UD_NQ_INCLUDECODEPAGE

#ifdef UD_NQ_CODEPAGE437
const CMCodepage*
cmCpInit437(
    void
    );
#endif /* UD_NQ_CODEPAGE437 */

#ifdef UD_NQ_CODEPAGE850
CMCodepage*
cmCpInit850(
    void
    );
#endif /* UD_NQ_CODEPAGE850 */

#ifdef UD_NQ_CODEPAGE852
CMCodepage*
cmCpInit852(
    void
    );
#endif /* UD_NQ_CODEPAGE852 */

#ifdef UD_NQ_CODEPAGE858
CMCodepage*
cmCpInit858(
    void
    );
#endif /* UD_NQ_CODEPAGE858 */

#ifdef UD_NQ_CODEPAGE862
CMCodepage*
cmCpInit862(
    void
    );
#endif /* UD_NQ_CODEPAGE862 */

#ifdef UD_NQ_CODEPAGE932
const CMCodepage*
cmCpInit932(
    void
    );
#endif /* UD_NQ_CODEPAGE932 */

#ifdef UD_NQ_CODEPAGE936
CMCodepage*
cmCpInit936(
    void
    );
#endif /* UD_NQ_CODEPAGE936 */

#ifdef UD_NQ_CODEPAGE949
CMCodepage*
cmCpInit949(
    void
    );
#endif /* UD_NQ_CODEPAGE949 */

#ifdef UD_NQ_CODEPAGE950
CMCodepage*
cmCpInit950(
    void
    );
#endif /* UD_NQ_CODEPAGE950 */

#ifdef UD_NQ_CODEPAGEUTF8
CMCodepage*
cmCpInitUtf8(
    void
    );
#endif /* UD_NQ_CODEPAGEUTF8 */

const CMCodepage* encodings[30];

#if (defined(SY_CP_FIRSTILLEGALCHAR) && defined(SY_CP_ANYILLEGALCHAR))

static const NQ_BYTE firstIllegalChar[] = SY_CP_FIRSTILLEGALCHAR;
static const NQ_BYTE anyIllegalChar[] = SY_CP_ANYILLEGALCHAR;

#define FIRST_ILLEGAL_CHAR_NUM    (sizeof(firstIllegalChar) / sizeof(NQ_BYTE))
#define ANY_ILLEGAL_CHAR_NUM      (sizeof(anyIllegalChar) / sizeof(NQ_BYTE))

#endif /* (defined(SY_CP_FIRSTILLEGALCHAR) && defined(SY_CP_ANYILLEGALCHAR)) */

static
NQ_INT
cmCpGetCodePage(
    void
    );

static
void
cmCpInitEncodings(
    void
    );

NQ_BOOL
cmCodepageAdd(
    const CMCodepage * codePage
    )
{
    NQ_INT index;
    NQ_BOOL result = TRUE;

    if (NULL == codePage)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        result = FALSE;
        goto Exit;
    }

    cmCpInitEncodings();

    for (index = 0; index < sizeof(encodings)/sizeof(CMCodepage * ); index++)
    {
        if (encodings[index] != NULL)
        {
            if (encodings[index]->id == codePage->id)
            {
                encodings[index] = codePage;
                goto Exit;
            }
        }
    }
    for (index = 0; index < sizeof(encodings)/sizeof(CMCodepage *); index++)
    {
        if (encodings[index] == NULL)
        {
            encodings[index] = codePage;
            goto Exit;
        }
    }
    result = FALSE;

Exit:
    return result;
}

/*
Uninstalls code page.
codePage - descriptor should have the same id value as was specified in the cmCodepageAdd call. Other fields are ignored.
This call can also remove a pre-defined page.

*/
NQ_BOOL
cmCodepageRemove(
    const CMCodepage * codePage
    )
{
    NQ_INT index;
    NQ_BOOL result = TRUE;

    if (NULL == codePage)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        result = FALSE;
        goto Exit;
    }

    for (index = 0; index < sizeof(encodings)/sizeof(CMCodepage *); index++)
    {

        if ( encodings[index] != NULL)
        {
            if (encodings[index]->id == codePage->id)
            {
                encodings[index] = NULL;
                goto Exit;
            }
        }
    }
    sySetLastError(NQ_ERR_NOTFOUND);
    result = FALSE;

Exit:
    return result;
}


/*
 *====================================================================
 * PURPOSE: Convert Ascii string to Unicode according to codepage
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT result Unicode string
 *          IN Ascii string to be converted
 *          IN length of result Unicode string (in bytes)
 *          IN length of Ascii string (in bytes)
 *
 * RETURNS: length of result string, 0 if failed
 *
 * NOTES:

 *====================================================================
 */

NQ_INT
cmCpAnsiToUnicode(
    NQ_WCHAR* outWStr,
    NQ_UINT outLength,
    const NQ_CHAR* inStr,
    NQ_UINT inLength
    )
{
    NQ_INT codePageIdx = cmCpGetCodePage();
    NQ_INT result;

    if (encodings[codePageIdx] != NULL && encodings[codePageIdx]->a2uTab)
    {
        result = cmCpSingleByteAnsiToUnicode(outWStr, outLength, inStr, inLength, encodings[codePageIdx]->a2uTab);
        goto Exit;
    }

    if (encodings[codePageIdx] != NULL)
        result = encodings[codePageIdx]->toUnicode(outWStr, outLength, inStr, inLength);
    else
        result = 0;

Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: Convert Unicode string to Ascii according to codepage
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT Ascii result string
 *          IN Unicode string to be converted
 *          IN length of result Ascii string (in bytes)
 *          IN length of Unicode string (in bytes)
 *
 * RETURNS: length of result string, 0 if failed
 *
 * NOTES:

 *====================================================================
 */

NQ_INT
cmCpUnicodeToAnsi(
    NQ_CHAR* outStr,
    NQ_UINT outLength,
    const NQ_WCHAR* inWStr,
    NQ_UINT inLength
    )
{
    NQ_INT codePageIdx = cmCpGetCodePage();
    NQ_INT result;

    if (encodings[codePageIdx] != NULL)
        result = encodings[codePageIdx]->toAnsi(outStr, outLength, inWStr,inLength );
    else
        result = 0;

    return result;
}

/*
 *====================================================================
 * PURPOSE: Capitalize a single Ascii symbol
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT pointer to result
 *          IN pointer to character to be converted
 *
 * RETURNS: length of result string, 0 if failed
 *
 * NOTES:
 *====================================================================
 */

NQ_INT
cmCpAToUpper(
    NQ_CHAR* dest,
    const NQ_CHAR* src
    )
{
    NQ_INT codePageIdx = cmCpGetCodePage();
    NQ_INT result;

    if (encodings[codePageIdx] != NULL)
        result = encodings[codePageIdx]->toUpper(dest, src);
    else
        result = 0;

    return result;
}

/*
 *====================================================================
 * PURPOSE: convert string from ANSI with possible two-byte encoding
 *          to a string acceptable by the file system as a file name
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT Ascii string/converted string
 *          IN its lenth before and after conversion
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
cmCpAnsiToFs(
    NQ_CHAR *str,
    NQ_INT size
    )
{
#if (defined(SY_CP_FIRSTILLEGALCHAR) && defined(SY_CP_ANYILLEGALCHAR))
    NQ_INT codePageIdx = cmCpGetCodePage();
    if (encodings[codePageIdx] != NULL && encodings[codePageIdx]->ansiToFs)
    {
        encodings[codePageIdx]->ansiToFs(str, size, firstIllegalChar,
                                         anyIllegalChar, FIRST_ILLEGAL_CHAR_NUM, ANY_ILLEGAL_CHAR_NUM );
    }
#endif
}

/*
 *====================================================================
 * PURPOSE: convert file name (encoded by the previous call)
 *          into a valid ANSI string
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT Ascii string/converted string
 *          IN its lenth before and after conversion
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
cmCpFsToAnsi(
    NQ_CHAR *str,
    NQ_INT size
    )
{
#if (defined(SY_CP_FIRSTILLEGALCHAR) && defined(SY_CP_ANYILLEGALCHAR))
    NQ_INT codePageIdx = cmCpGetCodePage();
    if (encodings[codePageIdx] != NULL && encodings[codePageIdx]->fsToAnsi)
    {
        encodings[codePageIdx]->fsToAnsi(str, size, firstIllegalChar,
                                         anyIllegalChar, FIRST_ILLEGAL_CHAR_NUM, ANY_ILLEGAL_CHAR_NUM );
    }
#endif
}


NQ_INT
cmCpSingleByteAnsiToUnicode(
    NQ_WCHAR* outWStr,
    NQ_UINT outLength,
    const NQ_CHAR* inStr,
    NQ_UINT inLength,
    const NQ_WCHAR* a2uTable
    )
{
    NQ_WCHAR* pW = outWStr;
    const NQ_CHAR* pA = inStr;
    NQ_INT length = 0;
    NQ_UINT outNumOfCharacters;
    NQ_BOOL ignoreInLength = (inLength == CM_IGNORE_LENGTH);
    NQ_BOOL ignoreOutLength = (outLength == CM_IGNORE_LENGTH);
    NQ_INT result = 0;

    outNumOfCharacters = outLength / sizeof(NQ_WCHAR);  /* calculate number of characters for destination buffer */
    outNumOfCharacters--;                               /* reserve place for trailing zero */
    if (outWStr && inStr)
    {
        for ( ; *pA && (ignoreInLength || (inLength > 0)) && (ignoreOutLength || (length < (NQ_INT)outNumOfCharacters)); pA++, pW++, inLength--, length++)
        {
            NQ_BYTE a = (NQ_BYTE)*pA;

            if ( a <= 0x7F)
            {
                *pW = cmHtol16((NQ_WCHAR)a);
            }
            else
            {
                *pW = cmHtol16(a2uTable[a-0x80]);
            }
        }
    }
    else
    {
        if (!outWStr)
            goto Exit;
    }

    *pW = 0;                /* Place trailing zero */
    result = length * (NQ_INT)sizeof(NQ_WCHAR);
Exit:
    return result;
}

static
NQ_INT
cmCpGetCodePage(
    void
    )
{
    NQ_INT i = 0;
    NQ_INT codePage = udGetCodePage();
    NQ_INT result;

    cmCpInitEncodings();

    for (i = 0; i < sizeof(encodings)/sizeof(CMCodepage *); i++)
    {
        if (encodings[i] != NULL)
        {
            if (encodings[i]->id == codePage)
            {
                result = i;
                goto Exit;
            }
        }
    }
    for (i = 0; i < sizeof(encodings)/sizeof(CMCodepage *); i++)
    {
        if (encodings[i] != NULL)
        {
            result = i;  /* return first existing codepage*/
            goto Exit;
        }
    }
    result = 0;

Exit:
    return result;
}

static
void
cmCpInitEncodings(
    void
    )
{
    static NQ_BOOL isInit = FALSE;
    NQ_INT index;

    if(!isInit)
    {
        isInit = TRUE;
        for (index = 0; index < sizeof(encodings)/sizeof(CMCodepage *); index++)
        {
            encodings[index] = NULL;
        }

#ifdef UD_NQ_CODEPAGE437
        cmCodepageAdd(cmCpInit437());
#endif /* UD_NQ_CODEPAGE437 */

#ifdef UD_NQ_CODEPAGE850
        cmCodepageAdd(cmCpInit850());
#endif /* UD_NQ_CODEPAGE850 */

#ifdef UD_NQ_CODEPAGE852
        cmCodepageAdd(cmCpInit852());
#endif /* UD_NQ_CODEPAGE852 */

#ifdef UD_NQ_CODEPAGE858
        cmCodepageAdd(cmCpInit858());
#endif /* UD_NQ_CODEPAGE858 */

#ifdef UD_NQ_CODEPAGE862
        cmCodepageAdd(cmCpInit862());
#endif /* UD_NQ_CODEPAGE862 */

#ifdef UD_NQ_CODEPAGE932
        cmCodepageAdd(cmCpInit932());
#endif /* UD_NQ_CODEPAGE932 */

#ifdef UD_NQ_CODEPAGE936
        cmCodepageAdd(cmCpInit936());
#endif /* UD_NQ_CODEPAGE936 */

#ifdef UD_NQ_CODEPAGE949
        cmCodepageAdd(cmCpInit949());
#endif /* UD_NQ_CODEPAGE949 */

#ifdef UD_NQ_CODEPAGE950
        cmCodepageAdd(cmCpInit950());
#endif /* UD_NQ_CODEPAGE950 */

#ifdef UD_NQ_CODEPAGEUTF8
            cmCodepageAdd(cmCpInitUtf8());
#endif /* UD_NQ_CODEPAGEUTF8 */
    }

    return;
}


void cmSplitUint16(NQ_UINT16 src, NQ_BYTE *msb, NQ_BYTE *lsb)
{
    *lsb = (NQ_BYTE)(src);
    *msb = (NQ_BYTE)((src) >> 8);
}

void cmSplitUnicode(NQ_WCHAR src, NQ_BYTE *msb, NQ_BYTE *lsb)
{
    NQ_WCHAR _u = cmLtoh16(src);
    cmSplitUint16(_u, msb, lsb);
}

#endif /* UD_NQ_INCLUDECODEPAGE*/

