/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : CIFS pass-through and local authentication
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  : None
 ********************************************************************/

#include "cmapi.h"
#include "cmcrypt.h"
#include "cmlist.h"
#include "nsapi.h"
#include "nqapi.h"
#include "cmfinddc.h"
#include "csauth.h"
#include <amspnego.h>
#include <amntlmss.h>
#include "cmlist.h"
#ifdef UD_CS_INCLUDEPASSTHROUGH
#include "ccapi.h"
#include "ccserver.h"
#include "ccuser.h"
#include "ccdomain.h"
#include "ccdcerpc.h"
#include "cclsarpc.h"
#include "ccsamrpc.h"
#endif /*UD_CS_INCLUDEPASSTHROUGH*/

#ifdef UD_NQ_INCLUDECIFSSERVER

/*
    PDC data structure
*/

typedef struct {
    NQ_WCHAR name[CM_BUFFERLENGTH(NQ_WCHAR, CM_NQ_HOSTNAMESIZE)];   /* PDC name for the server domain */
#ifdef UD_CS_INCLUDEPASSTHROUGH
    CCServer * server;                  /* used for communicating CIFS with PDC */
    CCUser user;                        /* dummy user - used only as an UID store */
#endif /* UD_CS_INCLUDEPASSTHROUGH */
}
Pdc;

/*
    static data declarations
*/

typedef struct
{
    NQ_CHAR buffer[CM_BUFFERLENGTH(NQ_CHAR, CM_NQ_HOSTNAMESIZE)];   /* temp buffer */
    Pdc pdc;
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

static NQ_UINT encryptLevel =   CS_AUTH_ENCRYPTION_ALL;

typedef struct 
{
    NQ_BOOL hashed;               /* 1 - password is hashed, 0 - plain text */
    NQ_WCHAR unicode[64+1];       /* plain text UNICODE password */
    NQ_BYTE lm[16];               /* hashed LM password */
    NQ_BYTE ntlm[16];             /* hashed NTLM password */
} LocalPassword;

/* key type definition is applied to session key */
#define KEY_LM 1
#define KEY_NTLM 2
#define KEY_NTLMV2 3

/*
    Static functions
    ----------------
 */
#if defined (UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY)
/* Generate session key */
static void
generateSessionKey(
    const LocalPassword* in,
    NQ_BYTE* out,
    NQ_INT keyType
    );
#endif

#ifdef UD_CS_MESSAGESIGNINGPOLICY
static void                         /* create context for signing calculation */
createSigningContextSmb1(
    CSSession *pSession,            /* session pointer */
    CSUser *pUser,                  /* user pointer */
    AMNtlmDescriptor *pDescr        /* NTLM blob descriptor */
    );
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

static NQ_BOOL                      /* check NTLM password */
encryptNTLMv2(
    const NQ_BYTE* key,             /* session key */
    const NQ_BYTE* v2hash,          /* v2 hash */
    const NQ_BYTE* ntlm,            /* NTLM password data in request */
    NQ_INT ntlmlen,                 /* data size */
    CSUser* pUser                   /* user structure */
    );

static NQ_BOOL                      /* check NTLM password */
encryptLMv2(
    const NQ_BYTE* key,             /* session key */
    const NQ_BYTE* v2hash,          /* v2 hash */
    const NQ_BYTE* lm,              /* LM password data in request */
    NQ_INT lmlen,                   /* data size */
    CSUser* pUser                   /* user structure */
    );

static NQ_UINT32                                    /* local status */
authenticateNtlm(
    const CMCifsSessionSetupAndXRequest* pRequest,  /* request pointer */
    NQ_BOOL unicodeRequired,                        /* TRUE when client sends UNICODE strings */
    AMNtlmDescriptor* descr,                        /* NTLM blob descriptor */
    NQ_WCHAR* userName,                             /* buffer for user name */
    NQ_WCHAR* domainName,                           /* buffer for pointer to the domain name */
    const NQ_BYTE** osName                          /* buffer for pointer to the 1st byte of non-parsed data */
    );


#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
static NQ_UINT32                                    /* local status */
authenticateSpnego(
    NQ_IOBufPos pRequest,                           /* request pointer */
    CSSession * pSession,                           /* session structure */
    AMNtlmDescriptor * descr,                       /* NTLM blob descriptor */
    NQ_WCHAR * userName,                            /* buffer for user name */
    NQ_WCHAR * pDomain,                             /* buffer for pointer to the domain name */
    NQ_WCHAR * pClientHostName,                     /* buffer for client host name */
    const NQ_BYTE ** pSessionKey,                   /* buffer for session key pointer or NULL if none */
    NQ_IOBufPos resBlob,                            /* buffer for response blob */
    NQ_COUNT* resBlobLen,                           /* buffer for blob length */
    const NQ_BYTE ** osName,                        /* buffer for pointer to the 1st byte of non-parsed data */
    NQ_IOBufPos *inBlob,                            /* buffer for pointer to incoming blob */
    NQ_UINT16* inBlobLength                         /* pointer to incoming blob length */
);
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */

static NQ_BOOL                  /* returns TRUE is there is a local user */
isLocalUserOk(
    CSUser* pUser,              /* user structure pointer */
    const NQ_WCHAR* domain,     /* domain name */
    const NQ_WCHAR* user,       /* user name */
    const NQ_BYTE* key,         /* session key */
    AMNtlmDescriptor * pBlob    /* incoming blob descriptor */
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    , NQ_BOOL  isExtendedSecurity
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
    , CMSdRid* userRid           /* buffer for user RID */
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    );


/*
    actual pass-through authentication implementation
*/

NQ_UINT getCurrentSessionKey(NQ_BYTE ** key, NQ_BYTE **nonce)
{
    CSSession* session = csGetSessionBySocket();     /* session structure */
    if (NULL == session)                             /* malformed command or there was no Negotiate yet */
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown session by socket");
        LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
        return AM_STATUS_GENERIC;
    }

    *key = session->encryptionKey;
    *nonce = session->sessionNonce;
    return sizeof(session->encryptionKey);
}

/*
 *====================================================================
 * PURPOSE: initialize the pass-through authentication module
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: TRUE if succeeded, FALSE otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
csAuthInit(
    void
    )
{
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocate(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    staticData->pdc.name[0] = '\0';

#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    amSpnegoServerSetSessionKeyCallback(getCurrentSessionKey);
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Own domain name: %s:%s", cmNetBiosGetDomainAuth()->name, (cmNetBiosGetDomainAuth()->isGroup ? "W" : "D"));

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE: shut down the pass-through authentication module
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

void
csAuthShutdown(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    /* release memory */
#ifdef SY_FORCEALLOCATION
    if (NULL != staticData)
        cmMemoryFree(staticData);
    staticData = NULL;
#endif /* SY_FORCEALLOCATION */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 *====================================================================
 * PURPOSE: get PDC name
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: pointer to PDC name or NULL
 *
 * NOTES:
 *====================================================================
 */

const NQ_WCHAR*
csAuthGetPDCName(
    void
    )
{
    const NQ_WCHAR* pResult = NULL;

    if (staticData->pdc.name[0] != '\0' ||
        cmGetDCName(staticData->buffer, NULL) == NQ_SUCCESS)
    {
        cmAnsiToUnicode(staticData->pdc.name, staticData->buffer);
        pResult = staticData->pdc.name;
    }

    return pResult;
}

/*
*====================================================================
* PURPOSE: Perform user authentication
*--------------------------------------------------------------------
* PARAMS:  IN pointer to the command in the message
*          IN pointer to the session descriptor
*          OUT pointer to the blob buffer
*          OUT pointer to the blob length
*          IN TRUE when client sends UNICODE strings
*          OUT place for pointer to user descriptor
*          OUT place for the pointer to OS name
* RETURNS: error code or NQ_SUCCESS
*
* NOTES:   none
*====================================================================
*/
NQ_UINT32
csAuthenticateUser(
    const NQ_IOBufPos pReq,
    CSSession *pSession,
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    NQ_IOBufPos pOutBlob,               /* place to generate response blob */
    NQ_COUNT* blobLength,               /* pointer to blob length */
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
    NQ_BOOL unicodeRequired,
    CSUser** pUser,
    const NQ_BYTE** pOsName
    )
{
    NQ_STATIC NQ_WCHAR domainName[CM_BUFFERLENGTH(NQ_WCHAR, CM_NQ_HOSTNAMESIZE)];       /* domain name */
    NQ_STATIC NQ_WCHAR userName[CM_BUFFERLENGTH(NQ_WCHAR, CM_USERNAMELENGTH)];          /* user name */
    NQ_STATIC NQ_WCHAR clientHostName[CM_BUFFERLENGTH(NQ_WCHAR, CM_NQ_HOSTNAMESIZE)];   /* client host name */
    NQ_STATIC NQ_BYTE passwords[CM_CRYPT_MAX_NTLMV2NTLMSSPRESPONSESIZE];                /* saved passwords */
    AMNtlmDescriptor descr;                                                             /* descriptor of LM/NTLM blobs */
    NQ_INT credentialsLen;                                                              /* lengths of the credentials to save */
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
#ifdef UD_CS_INCLUDEPASSTHROUGH                                    
    const NQ_WCHAR * pOwnHostName = NULL;                                               /* own host name */
    NQ_STATIC NQ_CHAR domainA[CM_BUFFERLENGTH(NQ_CHAR, CM_NQ_HOSTNAMESIZE)];            /* buffer for domain name in CHAR */
    NQ_BOOL doPassThrough = FALSE;
    NQ_BYTE secret[16];
    const NQ_WCHAR *domainDNS = NULL;
    NQ_WCHAR * pDomainInRequest = domainName;                                           /* pointer to domain name in request */
#endif /* UD_CS_INCLUDEPASSTHROUGH */
    const NQ_BYTE* pSessionKey = NULL;                                                  /* pointer to session key in session setup auth message */
    NQ_BYTE* inBlob = NULL;                                                             /* pointer to incoming blob */
    NQ_UINT16 inBlobLength = 0;                                                         /* its length */
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
    NQ_UINT32 res = AM_STATUS_GENERIC;                                                  /* local response */
    const CMCifsSessionSetupAndXRequest* pRequest = (const CMCifsSessionSetupAndXRequest*)pReq;
    NQ_UINT32 result = NQ_SUCCESS;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "request:%p connection:%p unicode:%d user:%p name:%p", IOBUF_GETBYTEPTR(pReq), pSession, unicodeRequired ? 1 : 0, pUser, pOsName);

    syMemset(&descr, 0, sizeof(descr));
 
    switch (pSession->dialect)
    {
    case CS_DIALECT_SMB1:
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
        if (pRequest->wordCount == SMB_SESSIONSETUPANDX_REQUEST_WORDCOUNT)  /* NTLM logon */
        {
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
            res = authenticateNtlm(
                    pRequest,
                    unicodeRequired,
                    &descr, 
                    userName,
                    domainName,
                    pOsName
                    ); 
            *clientHostName = cmWChar(0);
            break;            
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
        }
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
        /* fall through  - if word count is different, SMB1 will use SPNEGO */
    case CS_DIALECT_SMB2:
    case CS_DIALECT_SMB210:
    case CS_DIALECT_SMB30:
    case CS_DIALECT_SMB311:
    default:
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    {
        NQ_BYTE *request = (NQ_BYTE *)pReq;

        res = authenticateSpnego(
                (NQ_IOBufPos)request,
                pSession,
                &descr, 
                userName,
                domainName,
                clientHostName,
                &pSessionKey,
                pOutBlob,
                blobLength, 
                pOsName,
                &inBlob,
                &inBlobLength
                ); 
    }
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
        break;
    }

    /* allocate user if not yet */
    if (*pUser == NULL)
    {
        /* find an empty record in the session table */
        *pUser = csGetNewUser(pSession);
        if (*pUser == NULL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "User table overflow");
            result = csErrorReturn(SMB_STATUS_NO_MEMORY, DOS_ERRnomem);
            goto Exit;
        }
    }

    if (CM_CRYPT_MAX_NTLMV2NTLMSSPRESPONSESIZE < (descr.lmLen + descr.ntlmLen))
    {
        /* we later copy these parameters to passwords[CM_CRYPT_MAX_NTLMV2NTLMSSPRESPONSESIZE]. size has to fit */
        TRCERR("password sizes too long for passwords array. NTLM length: %d, lm length: %d", descr.ntlmLen, descr.lmLen);
        res = AM_STATUS_BAD_FORMAT;
    }

    switch (res)
    {
        case AM_STATUS_AUTHENTICATED:
        {
            goto Exit;
        }
        case AM_STATUS_NOT_AUTHENTICATED:         /* NTLM login or ntlmssp auth message (final) */
        {
            res = NQ_SUCCESS;
            break;
        }
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
        case AM_STATUS_MORE_PROCESSING_REQUIRED:  /* ntlmssp negot */
        {
            result = csErrorReturn(SMB_STATUS_MORE_PROCESSING_REQUIRED, (NQ_UINT32)NQ_ERR_MOREDATA);
            goto Exit;
        }
#endif  /* UD_CS_INCLUDEEXTENDEDSECURITY */
        case AM_STATUS_BAD_FORMAT:
        {
            if (*pUser != NULL)
            {
                csReleaseUser((*pUser)->uid, FALSE);
            }

            result = csErrorReturn(SMB_STATUS_INVALID_PARAMETER, DOS_ERRbadformat);
            goto Exit;
        }
        case AM_STATUS_INSUFFICIENT_RESOURCES:
        {
            if (*pUser != NULL)
            {
                csReleaseUser((*pUser)->uid, FALSE);
            }

            result = csErrorReturn(SMB_STATUS_INSUFFICIENT_RESOURCES, DOS_ERRnomem);
            goto Exit;
        }
        case AM_STATUS_GENERIC:
        default:
        {
            if (*pUser != NULL)
            {
                csReleaseUser((*pUser)->uid, FALSE);
            }

            result = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
            goto Exit;
        }
    }

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Domain in request: %s", cmWDump(domainName));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "User in request: %s", cmWDump(userName));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Client's host in request: %s", cmWDump(clientHostName));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Own domain: %s", (NQ_CHAR *)cmNetBiosGetDomainAuth()->name);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Own host: %s", (NQ_CHAR *)cmNetBiosGetHostNameZeroed());

    /* anonymous connection */
    if ((descr.lmLen == 0 || descr.lmLen == 1) && (descr.ntlmLen == 0 || descr.ntlmLen == 1) && 
       ((pSession->dialect > CS_DIALECT_SMB1) ? TRUE : (pRequest->wordCount == SMB_SESSIONSETUPANDXSSP_REQUEST_WORDCOUNT ? TRUE : *(NQ_BYTE*)(pRequest + 1) == 0)))
    {
        NQ_BOOL isAnonymousAllowed = csGetAnonymousAllowed();

        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "this is anonymous connection - %s allowed", isAnonymousAllowed ? "" : "not");

        if (FALSE == isAnonymousAllowed)
        {
            csReleaseUser((*pUser)->uid, TRUE);
            LOGERR(CM_TRC_LEVEL_ERROR, "anonymous connection isn't allowed");
            result = csErrorReturn(SMB_STATUS_LOGON_TYPE_NOT_GRANTED, DOS_ERRnoaccess);
            goto Exit;
        }
        else
        {
            /* deprecated - UD decides whether to allow anonymous */
#ifdef UD_CS_AUTHENTICATEANONYMOUS
            NQ_INT res;
            NQ_UINT32 fakeRid;
            NQ_BOOL fakePassHashed;
            NQ_CHAR fakePass;
#endif  /* UD_CS_AUTHENTICATEANONYMOUS */
            CSUser *pAnononymous;

            /* set user */
            syAnsiToUnicode(userName, "anonymous logon");

            /* deprecated  - UD can restrict the access */
#ifdef UD_CS_AUTHENTICATEANONYMOUS
            res = udGetPassword(userName, &fakePass, &fakePassHashed, &fakeRid);
            if (res == NQ_CS_PWDNOUSER)
            {
                csReleaseUser((*pUser)->uid, TRUE);
                LOGERR(CM_TRC_LEVEL_ERROR, "anonymous connection isn't allowed");
                result = csErrorReturn(SMB_STATUS_LOGON_TYPE_NOT_GRANTED, DOS_ERRnoaccess);
                goto Exit;
            }
#endif  /* UD_CS_AUTHENTICATEANONYMOUS */
            pAnononymous = csGetUserByNameAndCredentials(userName, NULL, 0);
            if (NULL != pAnononymous && pAnononymous->authenticated)  /* user already authenticated in previous steps */
            {
                csReleaseUser((*pUser)->uid, TRUE);
                *pUser = pAnononymous;
                goto Exit;
            }

            syAnsiToUnicode((*pUser)->name, "anonymous");
            (*pUser)->isAnonymous = TRUE;
            (*pUser)->token.isAnon = TRUE;
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
            (*pUser)->token.numRids = 1;
            (*pUser)->token.rids = (CMSdRid *)cmMemoryAllocate((*pUser)->token.numRids * (NQ_UINT)sizeof(CMSdRid));
            if (NULL == (*pUser)->token.rids)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "token allocation failed");
                csReleaseUser((*pUser)->uid, FALSE);
                result = csErrorReturn(SMB_STATUS_LOGON_FAILURE, DOS_ERRnoaccess);
                goto Exit;
            }
            (*pUser)->token.rids[0] = CM_SD_RIDALIASGUEST; /* CM_SD_RIDGUEST */
            syMemcpy(&(*pUser)->token.domain, cmSdGetComputerSid(), sizeof((*pUser)->token.domain));
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
#ifdef UD_CS_AUTHENTICATEANONYMOUS
            (*pUser)->rid = fakeRid;
#endif  /* UD_CS_AUTHENTICATEANONYMOUS */

#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
            /* for anonymous session key is 0 */
            syMemset((*pUser)->sessionKey, 0, sizeof((*pUser)->sessionKey));
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Performing local authentication for anonymous");
            if (pSessionKey)  /* client supplied session key */
            {
                if (descr.flags & NTLMSSP_NEGOTIATE_KEY_EXCH) /* session key should be decrypted*/
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Client supplied encrypted session key");
                    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "encrypted session key (sess setup auth mess)", pSessionKey, 16);
                    cmArcfourCrypt((NQ_BYTE*)pSessionKey, 16, (*pUser)->sessionKey, sizeof((*pUser)->sessionKey));
                }
                syMemcpy((*pUser)->sessionKey, pSessionKey, sizeof((*pUser)->sessionKey));
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Created session key from local key 0x00 and client session key");
                LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "Final session key for anonymous", (*pUser)->sessionKey, 16);
            }
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
#ifdef UD_CS_MESSAGESIGNINGPOLICY
            if ((TRUE == pSession->signingOn) && (CS_DIALECT_SMB1 == pSession->dialect))
            {
                createSigningContextSmb1(pSession, *pUser, &descr);
            }
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

            goto Exit;
        } /* end of if (FALSE == isAnonymousAllowed) */
    } /* end of anonymous */

    /* save user name, credentials, session key */
    syWStrcpy((*pUser)->name, userName);
    syMemcpy(passwords, descr.pLm, descr.lmLen);
    syMemcpy(passwords + descr.lmLen, descr.pNtlm, descr.ntlmLen);
    credentialsLen = descr.lmLen + descr.ntlmLen;
    if (credentialsLen > (NQ_INT)sizeof((*pUser)->credentials))
    {
        credentialsLen = sizeof((*pUser)->credentials);
    }
    syMemcpy((*pUser)->credentials, passwords, credentialsLen);
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    if (NULL != pSessionKey)
    {
        syMemcpy((*pUser)->sessionKey, pSessionKey, sizeof((*pUser)->sessionKey));
    }
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */

    /* find an existing user (SMB1 only) */
    if (pSession->dialect == CS_DIALECT_SMB1)
    {
        CSUser *pPreviousUser = csGetUserByNameAndCredentials(userName, passwords, credentialsLen);
        if (NULL != pPreviousUser && pPreviousUser->authenticated)  /* user already authenticated by previous setups */
        {
            csReleaseUser((*pUser)->uid, TRUE);
            *pUser = pPreviousUser;
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "user already authenticated by previous setups");
            goto Exit;
        }
    }

    /* try pass through authentication */
#if defined(UD_CS_INCLUDEEXTENDEDSECURITY) && defined(UD_CS_INCLUDEPASSTHROUGH)
    cmUnicodeToAnsiN(domainA, sizeof(domainA), domainName, sizeof(domainName));

    /* perform pass through if domain in client's request is same as own FQDN or NetBIOS domain name */
    if (!cmNetBiosGetDomainAuth()->isGroup && pSession->usePassthrough)
    {
        NQ_WCHAR *_pDomainInRequest = pDomainInRequest; /* never NULL */

        /* empty domain name when UPN format used */
        if (*pDomainInRequest == cmWChar(0))
        {
            if (NULL != (_pDomainInRequest = cmWStrchr(userName, cmWChar('@'))))
            {
                ++_pDomainInRequest;
            }
        }

        if ((NULL != _pDomainInRequest) && ((cmWStricmp(cmNetBiosGetDomainAuthW(), _pDomainInRequest) == 0) ||
                (cmGetFullDomainNameW() && cmWStricmp(cmGetFullDomainNameW(), _pDomainInRequest) == 0)))
        {
            doPassThrough = TRUE;
            domainDNS = cmGetFullDomainNameW() ? cmGetFullDomainNameW() : pDomainInRequest;
        }
    }

    if (TRUE == doPassThrough)
    {
        NQ_BOOL isExtendedSecurity = ((CS_DIALECT_SMB1 == pSession->dialect) && (SMB_SESSIONSETUPANDX_REQUEST_WORDCOUNT == pRequest->wordCount)) ? FALSE : TRUE;

        (*pUser)->isDomainUser = TRUE;

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL,"Passthrough authentication is required");

        pOwnHostName = cmNetBiosGetHostNameZeroedW();
        if (FALSE == udGetComputerSecretByDomain(secret, domainDNS, NULL))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Secret not available (passthrough NetLogon)");
        }
        else
        {
            /* secure channel over tcp */
            if (FALSE == ccNetLogonEx(domainName,
                            userName,
                            clientHostName,
                            pOwnHostName,
                            pSession->encryptionKey, 
                            descr.pLm, 
                            descr.lmLen, 
                            descr.pNtlm, 
                            descr.ntlmLen, 
                            NULL,
                            secret,
                            isExtendedSecurity,
                            NULL,               /* DNS ips list (maybe NULL) */
                            domainDNS,          /* domain DNS name, required for DC resolution */
                            (*pUser)->sessionKey,
                            &(*pUser)->token))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Passthrough (NetLogon secure channel) authentication failed");

                /* fallback to rpc over smb */
                if (FALSE == ccNetLogon(domainName,
                                userName,
                                clientHostName,
                                pOwnHostName,
                                pSession->encryptionKey,
                                descr.pLm,
                                descr.lmLen,
                                descr.pNtlm,
                                descr.ntlmLen,
                                NULL,
                                secret,
                                isExtendedSecurity,
                                NULL,               /* DNS ips list (maybe NULL) */
                                domainDNS,          /* domain DNS name, required for DC resolution */
                                (*pUser)->sessionKey,
                                &(*pUser)->token))
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Passthrough (NetLogon over smb) authentication failed");
                    /* try local authentication */
                    goto LocalAuthentication;
                }
                else
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS,"Passthrough (NetLogon over smb) authentication succeeded");
                }
            }
            else
            {
                LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS,"Passthrough (NetLogon secure channel) authentication succeeded");
            }

            /*TRCDUMP("Session key:", (*pUser)->sessionKey, 16);*/

            if (0 != (descr.flags & NTLMSSP_NEGOTIATE_EXTENDED_SECURITY) && (NTLMSSP_LM_NTLM_DATA_LENGTH == descr.lmLen) && (NTLMSSP_LM_NTLM_DATA_LENGTH == descr.ntlmLen))
            {
                LOGMSG(CM_TRC_LEVEL_MESS_SOME,"Generating extended security (ntlmv2) session key");
                cmGenerateExtSecuritySessionKey((*pUser)->sessionKey, pSession->sessionNonce, (*pUser)->sessionKey);
            }

            if (NULL != pSessionKey)  /* client supplied session key */
            {
                if (0 != (descr.flags & NTLMSSP_NEGOTIATE_KEY_EXCH)) /* session key should be decrypted*/
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_SOME, "Client supplied encrypted session key");
                    /*LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "encrypted session key (sess setup auth mess)", pSessionKey, 16);*/
                    cmArcfourCrypt((NQ_BYTE*)pSessionKey, 16, (*pUser)->sessionKey, sizeof((*pUser)->sessionKey));
                }
                syMemcpy((*pUser)->sessionKey, pSessionKey, sizeof((*pUser)->sessionKey));
            }

            /*LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "Session key (final)", (*pUser)->sessionKey, 16);*/
            (*pUser)->isExtendSecAuth = isExtendedSecurity;
            (*pUser)->authBySamlogon = TRUE;
            descr.isNtlmAuthenticated = TRUE;
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
            if (0 == (*pUser)->token.domain.revision)
            {
                syMemcpy(&(*pUser)->token.domain, cmSdGetComputerSid(), sizeof((*pUser)->token.domain));
            }
#ifdef UD_NQ_INCLUDETRACE
            cmSdDumpAccessToken(&(*pUser)->token);
#endif /* UD_NQ_INCLUDETRACE */
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
#ifdef UD_CS_MESSAGESIGNINGPOLICY
            if ((TRUE == pSession->signingOn) && (CS_DIALECT_SMB1 == pSession->dialect))
            {
                createSigningContextSmb1(pSession, *pUser, &descr);
            }
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
#ifdef UD_NQ_INCLUDESMB3
#ifdef UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE
            (*pUser)->isEncrypted = (csIsServerEncrypted() && (0 != (pSession->capabilities & SMB2_CAPABILITY_ENCRYPTION)));
#else /* UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE */
            (*pUser)->isEncrypted = csIsServerEncrypted();
#endif /* UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE */
#endif /* UD_NQ_INCLUDESMB3 */
            goto Exit;
        }
    } /* end of TRUE == doPassThrough */

LocalAuthentication:
#endif /* defined(UD_CS_INCLUDEEXTENDEDSECURITY) && defined(UD_CS_INCLUDEPASSTHROUGH) */

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Performing local authentication");

    #ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
    (*pUser)->token.numRids = 4;
    (*pUser)->token.rids = (CMSdRid *)cmMemoryAllocate((*pUser)->token.numRids * (NQ_UINT)sizeof(CMSdRid));
    if (NULL == (*pUser)->token.rids)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "token allocation failed");
        csReleaseUser((*pUser)->uid, FALSE);
        result = csErrorReturn(SMB_STATUS_LOGON_FAILURE, DOS_ERRnoaccess);
        goto Exit;
    }
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    (*pUser)->isDomainUser = FALSE;

    if (isLocalUserOk(
            *pUser,
            domainName,
            userName,
            pSession->encryptionKey,
            &descr
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
            ,
            pRequest->wordCount == SMB_SESSIONSETUPANDXSSP_REQUEST_WORDCOUNT
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */           
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
            ,
            &(*pUser)->token.rids[0]
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
            )
        )
    {
        if ((*pUser)->isGuest)
        {
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
            (*pUser)->token.numRids = 1;
            (*pUser)->token.rids = (CMSdRid *)cmMemoryAllocate((*pUser)->token.numRids * (NQ_UINT)sizeof(CMSdRid));
            if (NULL == (*pUser)->token.rids)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "token allocation failed");
                csReleaseUser((*pUser)->uid, FALSE);
                result = csErrorReturn(SMB_STATUS_LOGON_FAILURE, DOS_ERRnoaccess);
                goto Exit;
            }

            (*pUser)->token.rids[0] = CM_SD_RIDALIASGUEST; /* CM_SD_RIDGUEST */
            syMemcpy(&(*pUser)->token.domain, cmSdGetComputerSid(), sizeof((*pUser)->token.domain));
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Guest login");
            goto Exit;
        }

#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
        if (NULL != pSessionKey)
        {
            if (descr.flags & NTLMSSP_NEGOTIATE_KEY_EXCH) /* client supplied encrypted session key */
            {
                cmArcfourCrypt((NQ_BYTE*)pSessionKey, 16, (*pUser)->sessionKey, sizeof((*pUser)->sessionKey));
            }
            syMemcpy((*pUser)->sessionKey, pSessionKey, sizeof((*pUser)->sessionKey));
        }     
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */
#ifdef UD_CS_MESSAGESIGNINGPOLICY
        if ((TRUE == pSession->signingOn) && (CS_DIALECT_SMB1 == pSession->dialect))
        {
            createSigningContextSmb1(pSession, *pUser, &descr);
        }
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
    }
    else
    {
#ifdef UD_NQ_INCLUDEEVENTLOG
        UDUserAccessEvent   eventInfo;
#endif /* UD_NQ_INCLUDEEVENTLOG */
        TRCERR("Local authentication failed");
#ifdef UD_NQ_INCLUDEEVENTLOG
        eventInfo.rid = csGetUserRid((*pUser));
        udEventLog(UD_LOG_MODULE_CS,
                   UD_LOG_CLASS_USER,
                   UD_LOG_USER_LOGON,
                   (*pUser)->name,
                   (*pUser)->ip,
                   csErrorReturn(SMB_STATUS_LOGON_FAILURE, DOS_ERRnoaccess),
                   (const NQ_BYTE *)&eventInfo);
#endif /* UD_NQ_INCLUDEEVENTLOG */
        csReleaseUser((*pUser)->uid, TRUE);
        result = csErrorReturn(SMB_STATUS_LOGON_FAILURE, DOS_ERRnoaccess);
        goto Exit;
    }

#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
    (*pUser)->token.rids[1] = CM_SD_RIDALIASUSER;

    if (cmSdIsAdmin((*pUser)->token.rids[0]))         /* administrator */
    {
        (*pUser)->token.rids[2] = CM_SD_RIDGROUPADMINS;
        (*pUser)->token.rids[3] = CM_SD_RIDALIASADMIN;
    }
    else
    {
        (*pUser)->token.numRids = 2;
    }

    syMemcpy(&(*pUser)->token.domain, cmSdGetComputerSid(), sizeof((*pUser)->token.domain));
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */

#ifdef UD_NQ_INCLUDESMB3
#ifdef UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE
    (*pUser)->isEncrypted = (csIsServerEncrypted() && (0 != (pSession->capabilities & SMB2_CAPABILITY_ENCRYPTION)));
#else /* UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE */
    (*pUser)->isEncrypted = csIsServerEncrypted();
#endif /* UD_CS_ALLOW_NONENCRYPTED_ACCESS_TO_ENCRYPTED_SHARE */
#endif /* UD_NQ_INCLUDESMB3 */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%x", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Perform local user authentication
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the domain name
 *          IN pointer to the user name
 *          IN pointer to the encryption key
 *          IN pointer to the LM password
 *          IN pointer to the incoming blob
 *          IN TRUE for extended security 
 *          OUT buffer for user RID
 *
 * RETURNS: TRUE if authenticated, FALSE otherwise
 *
 * NOTES:   If there is no user list or it has no records - we treat
 *          this as automatic authentication and allow any access by
 *          any user
 *====================================================================
 */

static
NQ_BOOL
isLocalUserOk(
    CSUser* pUser,
    const NQ_WCHAR* domain,
    const NQ_WCHAR* user,
    const NQ_BYTE* key,
    AMNtlmDescriptor * pBlob
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    ,
    NQ_BOOL  isExtendedSecurity
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */    
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
    ,
    CMSdRid* userRid
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    )
{
    NQ_STATIC LocalPassword password;
    NQ_STATIC NQ_BYTE encrypted[24];
    NQ_STATIC NQ_BYTE v2hash[16];
    NQ_STATIC NQ_CHAR buffer[UD_NQ_MAXPWDLEN];
#ifndef UD_CS_INCLUDESECURITYDESCRIPTORS
    NQ_UINT32 userRid[1];                   /* dummy for user RID */
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    const NQ_WCHAR zeroWStr[] = {0,0};
    NQ_UINT16 enclen = 0;
#if (defined(UD_CS_INCLUDELOCALUSERMANAGEMENT) && defined(UD_CS_INCLUDEEXTENDEDSECURITY)) || (defined(UD_CS_MESSAGESIGNINGPOLICY)) && defined(UD_CS_INCLUDEEXTENDEDSECURITY)
    CSSession* session;                     /* session structure */      
#endif

    TRCB();

#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    pUser->isExtendSecAuth = isExtendedSecurity;
#endif
        
    switch (udGetPassword(user, buffer, &password.hashed, userRid))
    {
        case NQ_CS_PWDNOAUTH:
            pUser->isGuest = TRUE;
            pUser->rid = *userRid;
            TRC("No authentication required (no user database)");
            TRCE();
            return TRUE;        /* automatic authentication when there is no user list */

        case NQ_CS_PWDNOUSER:
            TRCERR("Unknown user");
            pUser->rid = CS_ILLEGALID;
            TRC1P("  user name: %s", cmWDump(user));
            break;

        case NQ_CS_PWDANY:
        {
            /* both LM and NTML passwords present */
            pUser->rid = *userRid;
            if (password.hashed)
            {
                syMemcpy(password.lm, buffer, 16);
                syMemcpy(password.ntlm, buffer + 16, 16);
            }
            else
            {
                cmHashPassword((NQ_BYTE *)buffer, password.lm);
                cmAnsiToUnicode(password.unicode, buffer);
                cmMD4(password.ntlm, (NQ_BYTE*)password.unicode, (NQ_UINT)(cmWStrlen(password.unicode) * sizeof(NQ_WCHAR)));
            }

            if (pBlob->ntlmLen > 0)
            {
                if (encryptLevel & CS_AUTH_ENCRYPTION_NTLMV2 || encryptLevel & CS_AUTH_ENCRYPTION_NTLM )
                {
                    TRC("trying NTLM");

                    /* check NTLM password */
                    cmEncryptNTLMPassword(key, password.ntlm, encrypted, &enclen);
                }
                if (encryptLevel & CS_AUTH_ENCRYPTION_NTLM)
                {
                    if (pBlob->ntlmLen == enclen && syMemcmp(pBlob->pNtlm, encrypted, enclen) == 0)
                    {
#if defined (UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY)                
                        generateSessionKey(&password, pUser->sessionKey, KEY_NTLM);
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
                        if ((session = csGetSessionBySocket()) == NULL)
                        {
                            TRCERR("Unknown session by socket");
                            TRCE();
                            return FALSE;
                        }
                        if ((isExtendedSecurity && (pBlob->flags & NTLMSSP_NEGOTIATE_EXTENDED_SECURITY)) || session->dialect != CS_DIALECT_SMB1)
                        {
                            cmGenerateExtSecuritySessionKey(pUser->sessionKey, session->sessionNonce, pUser->sessionKey);
                        }
#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */    
#endif /* defined (UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY) */
                        pBlob->isNtlmAuthenticated = TRUE;
                        TRC("NTLM passwords match");
                        TRCE();
                        return TRUE;
                    }
                }

                if (encryptLevel & CS_AUTH_ENCRYPTION_NTLMV2)
                {
                    TRC("trying NTLMv2 (csd)");

                    /* check NTLMv2 passwords */
                    cmCreateV2Hash(domain, TRUE, user, password.ntlm, sizeof(password.ntlm), v2hash);
                    if (encryptNTLMv2(key, v2hash, pBlob->pNtlm, pBlob->ntlmLen, pUser))
                    {
                        pBlob->isNtlmAuthenticated = TRUE;
                        TRC(" NTLMv2 (csd) passwords match");
                        TRCE();
                        return TRUE;
                    }

                    TRC("trying NTLMv2");

                    /* check NTLMv2 passwords */
                    cmCreateV2Hash(domain, FALSE, user, password.ntlm, sizeof(password.ntlm), v2hash);
                    if (encryptNTLMv2(key, v2hash, pBlob->pNtlm, pBlob->ntlmLen, pUser))
                    {
                        pBlob->isNtlmAuthenticated = TRUE;
                        TRC(" NTLMv2 passwords match");
                        TRCE();
                        return TRUE;
                    }

                    TRC("trying NTLMv2 - null domain");

                    /* check NTLMv2 passwords */
                    cmCreateV2Hash(zeroWStr, FALSE, user, password.ntlm, sizeof(password.ntlm), v2hash);
                    if (encryptNTLMv2(key, v2hash, pBlob->pNtlm, pBlob->ntlmLen, pUser))
                    {
                        pBlob->isNtlmAuthenticated = TRUE;
                        TRC(" NTLMv2 null domain passwords match");
                        TRCE();
                        return TRUE;
                    }
                }
            }
            if (pBlob->lmLen > 0)
            {
                if (encryptLevel & CS_AUTH_ENCRYPTION_LMV2)
                {
                    TRC("trying LMv2 (csd)");

                    /* check LMv2 passwords with case sensitive domain */
                    cmCreateV2Hash(domain, TRUE, user, password.ntlm, sizeof(password.ntlm), v2hash);
                    if (encryptLMv2(key, v2hash, pBlob->pLm, pBlob->lmLen, pUser))
                    {
                        pBlob->isLmAuthenticated = TRUE;
                        TRC(" LMv2 (csd) passwords match");
                        TRCE();
                        return TRUE;
                    }

                    TRC("trying LMv2");

                    /* check LMv2 passwords */
                    cmCreateV2Hash(domain, FALSE, user, password.ntlm, sizeof(password.ntlm), v2hash);
                    if (encryptLMv2(key, v2hash, pBlob->pLm, pBlob->lmLen, pUser))
                    {
                        pBlob->isLmAuthenticated = TRUE;
                        TRC(" LMv2 passwords match");
                        TRCE();
                        return TRUE;
                    }

                    TRC("trying LMv2 - null domain");

                    /* check LMv2 passwords */
                    cmCreateV2Hash(zeroWStr, FALSE, user, password.ntlm, sizeof(password.ntlm), v2hash);
                    if (encryptLMv2(key, v2hash, pBlob->pLm, pBlob->lmLen, pUser))
                    {
                        pBlob->isLmAuthenticated = TRUE;
                        TRC(" LMv2 null domain passwords match");
                        TRCE();
                        return TRUE;
                    }
                }
            }
        }
        /* continue to the next case */
        case NQ_CS_PWDLMHASH: /* only LM password presents */
        default:
            pUser->rid = *userRid;
            if (pBlob->lmLen > 0)
            {
                if (password.hashed)
                    syMemcpy(password.lm, buffer, 16);
                else
                    cmHashPassword((NQ_BYTE *)buffer, password.lm);

                TRC("trying LM");
                if (encryptLevel & CS_AUTH_ENCRYPTION_LM)
                {
                    cmEncryptLMPassword(key, password.lm, encrypted, &enclen);

                    if(pBlob->lmLen == enclen && syMemcmp(pBlob->pLm, encrypted, enclen) == 0)
                    {
#if defined (UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY)               
                        generateSessionKey(&password, pUser->sessionKey, KEY_LM);
#endif /* defined (UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY) */                  
                        pBlob->isLmAuthenticated = TRUE;
                        TRC("LM passwords match");
                        TRCE();
                        return TRUE;
                    }
                }
            }
    }

    TRCE();
    return FALSE;
}

#if defined (UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY)

/*
 *====================================================================
 * PURPOSE: Generate session key
 *--------------------------------------------------------------------
 * PARAMS:  IN 16 byte hash
 *          OUT buffer for session key
 *          IN key type as LM or NTLM
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

static void
generateSessionKey(
    const LocalPassword* in,
    NQ_BYTE* out,
    NQ_INT keyType
    )
{
    TRCB();
    switch (keyType)
    {
        case KEY_LM:
            TRC("Generated LM session key");
            syMemset(out, 0, SMB_SESSIONKEY_LENGTH);
            syMemcpy(out, in->lm, 8);
            break;
        case KEY_NTLM:
        default:
            TRC("Generated NTLM session key");
            cmMD4(out, (NQ_BYTE*)in->ntlm, SMB_SESSIONKEY_LENGTH);
            break;
    }
    TRCE();
}

#endif

/*
 *====================================================================
 * PURPOSE: check NTLM password
 *--------------------------------------------------------------------
 * PARAMS:  IN session key
 *          IN generated v2 hash
 *          IN NTLM password data in the request
 *          IN data size
 *          IN pointer to user structure
 *
 * RETURNS: TRUE if match, FALSE if not
 *
 * NOTES:
 *====================================================================
 */

static NQ_BOOL
encryptNTLMv2(
    const NQ_BYTE* key,
    const NQ_BYTE* v2hash,
    const NQ_BYTE* ntlm,
    NQ_INT ntlmlen,
    CSUser* pUser
    )
{
    NQ_STATIC NQ_BYTE encrypted[24];
    NQ_UINT16 enclen = 0;

    cmEncryptNTLMv2Password(key, v2hash, ntlm + CM_CRYPT_ENCLMv2HMACSIZE, (NQ_UINT16)(ntlmlen - CM_CRYPT_ENCLMv2HMACSIZE), encrypted, &enclen);
    if (syMemcmp(ntlm, encrypted, enclen) == 0)
    {
#if defined(UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY)
        cmGenerateExtSecuritySessionKey(v2hash, encrypted, pUser->sessionKey);
#endif /* defined(UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY) */
        return TRUE;
    }
    return FALSE;
}

/*
 *====================================================================
 * PURPOSE: check NTLM password
 *--------------------------------------------------------------------
 * PARAMS:  IN session key
 *          IN generated v2 hash
 *          IN LM password data in the request
 *          IN data size
 *          IN pointer to user structure
 *
 * RETURNS: TRUE if match, FALSE if not
 *
 * NOTES:
 *====================================================================
 */

static NQ_BOOL
encryptLMv2(
    const NQ_BYTE* key,
    const NQ_BYTE* v2hash,
    const NQ_BYTE* lm,
    NQ_INT lmlen,
    CSUser* pUser
    )
{
    NQ_BOOL result = FALSE;
    NQ_BYTE encrypted[24];
    NQ_UINT16 enclen = 0;

    if (lmlen < 24)
        goto Exit;

    cmEncryptNTLMv2Password(key, v2hash, lm + CM_CRYPT_ENCLMv2HMACSIZE, 8, encrypted, &enclen);
    if (syMemcmp(lm, encrypted, enclen) == 0)
    {
#if defined(UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY)
        cmGenerateExtSecuritySessionKey(v2hash, encrypted, pUser->sessionKey);
#endif /* defined(UD_CS_INCLUDELOCALUSERMANAGEMENT) || defined(UD_CS_MESSAGESIGNINGPOLICY) */
        result = TRUE;
    }
Exit:
    return result;
}


/*
 *====================================================================
 * PURPOSE: Parse SPNEGO Session Setup request
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the request
 *          IN TRUE when Unicode desired
 *          OUT populate this structure with blob pointers and sizes  
 *          OUT buffer for user name
 *          OUT buffer for domain name pointer
 *          OUT buffer for pointer to the 1st byte after parsed data
 *
 * RETURNS: status, including:
 *      AM_STATUS_NOT_AUTHENTICATED         - was parsed but not authenticated yet
 *      <any other>                         - parse error
 *
 * NOTES:   
 *====================================================================
 */
static NQ_UINT32 authenticateNtlm(
    const CMCifsSessionSetupAndXRequest * pRequest,  
    NQ_BOOL unicodeRequired,                
    AMNtlmDescriptor * descr,
    NQ_WCHAR * userName,
    NQ_WCHAR * domainName,
    const NQ_BYTE ** pOsName                              
    )
{
    NQ_BYTE * pBuf;
    NQ_UINT size;

    TRCB();

    descr->lmLen = (NQ_UINT16)cmLtoh16(cmGetSUint16(pRequest->caseInsensitivePasswordLength));
    descr->ntlmLen = (NQ_UINT16)cmLtoh16(cmGetSUint16(pRequest->caseSensitivePasswordLength));
    descr->pLm = (NQ_BYTE*)pRequest + sizeof(*pRequest);
    descr->pNtlm = descr->pLm + descr->lmLen;

    pBuf = (NQ_BYTE *)(descr->pNtlm + descr->ntlmLen);   /* pointer to the account name in the message */
    if (unicodeRequired)
    {
        /* UNICODE string padding (skip 1 byte) */
        pBuf = (NQ_BYTE *)cmAllignTwo(pBuf);
        cmWStrncpy(userName, (NQ_WCHAR *)pBuf, CM_USERNAMELENGTH - 1);
        userName[CM_USERNAMELENGTH - 1] = cmWChar('\0');
        size = 1 + cmWStrlen(userName);
        pBuf += size * sizeof(NQ_WCHAR);
        cmWStrncpy(domainName, (NQ_WCHAR *)pBuf, CM_NQ_HOSTNAMESIZE - 1);
        domainName[CM_NQ_HOSTNAMESIZE - 1] = cmWChar('\0');
        size = 1 + cmWStrlen(domainName);
        pBuf += size * sizeof(NQ_WCHAR);
        *pOsName = pBuf;
    }
    else
    {
        cmAnsiToUnicodeN(userName, CM_USERNAMELENGTH * sizeof(NQ_WCHAR), (const NQ_CHAR *)pBuf, CM_USERNAMELENGTH);
        /* the next works only if the user name is always in UNICODE */
        size = 1 + cmWStrlen(userName);
        pBuf += size;
        cmAnsiToUnicodeN(domainName, CM_NQ_HOSTNAMESIZE * sizeof(NQ_WCHAR), (const NQ_CHAR *)pBuf, CM_NQ_HOSTNAMESIZE);
        size = 1 + cmWStrlen(domainName);
        pBuf += size;
        *pOsName = pBuf;
    }

    TRCE();
    return AM_STATUS_NOT_AUTHENTICATED;
}

#ifdef UD_CS_INCLUDEEXTENDEDSECURITY

/*
 *====================================================================
 * PURPOSE: Parse SPNEGO Session Setup request
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the request
 *          IN/OUT session structure
 *          OUT populate this structure with blob pointers and sizes
 *          OUT buffer for user name
 *          OUT buffer for domain name pointer
 *          OUT buffer for pointer to session key or NULL if none
 *          OUT buffer for response blob
 *          OUT buffer for response blob length
 *          OUT buffer for pointer to the 1st byte after parsed data
 *
 * RETURNS: status, including:
 *      AM_STATUS_AUTHENTICATED            - was authenticated
 *      AM_STATUS_NOT_AUTHENTICATED        - was parsed but not authenticated yet
 *      AM_STATUS_MORE_PROCESSING_REQUIRED - was recognized but requires more exchange
 *      <any other>                        - parse error
 *
 * NOTES:   
 *====================================================================
 */

static NQ_UINT32                                   
authenticateSpnego(
    NQ_IOBufPos pRequest, /*const CMCifsSessionSetupAndXRequest * pRequest,*/
    CSSession * pSession,
    AMNtlmDescriptor * descr,
    NQ_WCHAR * userName,
    NQ_WCHAR * domain,
    NQ_WCHAR * clientHostName,
    const NQ_BYTE ** pSessionKey,
    NQ_BYTE * resBlob,                              
    NQ_COUNT * resBlobLen,                            
    const NQ_BYTE ** pOsName,
    NQ_BYTE ** inBlob,
    NQ_UINT16 * inBlobLength
    )
{
    CMIOBlob spnegoIn;
    CMIOBlob spnegoOut;
    const NQ_WCHAR *pOwnDomain;
    const NQ_WCHAR *pOwnHostname;
    const CMCifsSessionSetupAndXSSPRequest* pSpnego;
    NQ_UINT32 result;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "req:%p conn:%p des:%p user:%s domain:%p key:%p res:%p resLen:%p osName:%p in:%p inLen:%p", IOBUF_GETBYTEPTR(pRequest), pSession, descr, cmWDump(userName), domain, pSessionKey, resBlob, resBlobLen, pOsName, inBlob, inBlobLength);

    if (pSession->dialect != CS_DIALECT_SMB1)
    {
        CMBufferReader r;
        NQ_IOBufPos inBlobTmp;
        NQ_BYTE* bufPtr;
        *inBlob = pRequest;
        inBlobTmp = *inBlob;
        IOBUF_MOVEBYTES(inBlobTmp, (NQ_INT)(-10));
        bufPtr = IOBUF_GETBYTEPTR(inBlobTmp);

        cmBufferReaderInit(&r, bufPtr, sizeof(*inBlobLength));
        cmBufferReadUint16(&r, inBlobLength);
    }
    else
    {
        IOBUF_BUFFER_CANUSEFLAT_ASSERT(pRequest, (sizeof(CMCifsSessionSetupAndXSSPRequest) + 2));
        pSpnego = (const CMCifsSessionSetupAndXSSPRequest*)IOBUF_GETBYTEPTR(pRequest);
        *pOsName = (NQ_BYTE*)(pSpnego + 1) + cmLtoh16(cmGetSUint16(pSpnego->blobLength));
        *pOsName = cmAllignTwo(*pOsName);
        IOBUF_MOVEBYTES(pRequest, sizeof(CMCifsSessionSetupAndXSSPRequest));
        *inBlob = pRequest;
        *inBlobLength = (NQ_UINT16)cmLtoh16(cmGetSUint16(pSpnego->blobLength));
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "blob length = %d", *inBlobLength);
    spnegoIn.data = *inBlob;
    spnegoIn.len = *inBlobLength;
    spnegoOut.data = resBlob;
    pOwnDomain = cmNetBiosGetDomainAuthW();
    pOwnHostname = cmNetBiosGetHostNameZeroedW();

    result = amSpnegoServerAcceptBlob(
                                        &pSession->securityMech,
                                        pOwnDomain,
                                        pOwnHostname,
                                        &spnegoIn,
                                        &spnegoOut,
                                        userName,
                                        clientHostName,
                                        domain,
                                        pSessionKey,
                                        descr
                                        );
    *resBlobLen = spnegoOut.len;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%u", result);
    return result;
}

#endif /* UD_CS_INCLUDEEXTENDEDSECURITY */

#ifdef UD_CS_MESSAGESIGNINGPOLICY
/*
 *====================================================================
 * PURPOSE: Create context for message signing SMB1
 *--------------------------------------------------------------------
 * PARAMS:  IN     pointer session structure
 *          IN/OUT pointer user structure
 *          IN     pointer NTLM blob descriptor
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */

static 
void
createSigningContextSmb1(
    CSSession *pSession,
    CSUser *pUser,    
    AMNtlmDescriptor *pDescr
    )
{    
    NQ_BYTE *password;
    NQ_COUNT passwordLen;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pSession:%p pUser:%p pDescr:%p", pSession, pUser, pDescr);
    
#ifdef UD_CS_INCLUDEEXTENDEDSECURITY
    if (pUser->isExtendSecAuth)
    {
        password = NULL;
        passwordLen = 0;
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "extended security authentication");
    }
    else
#endif  /* UD_CS_INCLUDEEXTENDEDSECURITY */            
    {
        password = (NQ_BYTE *)(pDescr->isLmAuthenticated ? pDescr->pLm : (pDescr->isNtlmAuthenticated ? pDescr->pNtlm : NULL));
        passwordLen = (NQ_COUNT)(pDescr->isLmAuthenticated ? pDescr->lmLen : (pDescr->isNtlmAuthenticated ? pDescr->ntlmLen : 0));
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "authenticated by %s", pDescr->isLmAuthenticated ? "LM" : "NTLM");
    }

    /* on first "real" logged in user save session key for signing throughout the whole session */
    if (pSession->isBsrspyl && !pUser->isAnonymous && !pUser->isGuest)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "no more BSRSPYL session, saving session key");
        pSession->isBsrspyl = FALSE;
        syMemcpy(pSession->sessionKey, pUser->sessionKey, sizeof(pUser->sessionKey));
        TRCDUMP("session key", pSession->sessionKey, sizeof(pUser->sessionKey));
    }

    if (password != NULL)
    {
        pUser->password.data = (NQ_BYTE *)cmMemoryAllocate(passwordLen);
        pUser->password.len = passwordLen;
        syMemcpy(pUser->password.data , password , passwordLen);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

NQ_BOOL
csChangeAuthenticationEncryptionLevel(
        NQ_UINT mask
        )
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "mask: 0x%x", mask);

    if (CS_AUTH_ENCRYPTION_ALL < mask || 0 == mask)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal mask");
        goto Exit;
    }

    encryptLevel = mask;
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}

#endif /* UD_NQ_INCLUDECIFSSERVER */

