/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client API
 *--------------------------------------------------------------------
 * MODULE        : NQ
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _CCAPI_H_
#define _CCAPI_H_

#include "syapi.h"
#include "cmapi.h"
#include <amcredentials.h>
#include <ccparams.h>

/**@defgroup client Client */

/* ********************************************************************
 * Constants
 ********************************************************************/

/** @ingroup user_cred */
/** @defgroup credential Credential fields length
 * @{
 */
/* *** Credential fields length ****/
#define CM_USERNAMELENGTH   256                     /**< maximum user name length */
#define PASSWORD_LENGTH     UD_NQ_MAXPWDLEN         /**< maximum password length */
#define DOMAIN_LENGTH       (CM_NQ_HOSTNAMESIZE)    /**< maximum domain name length */
/** @} end of group Credential fields length */

/** @ingroup error */
/** @defgroup notification Driver Notification Events
 * @{
 */
/* *** Notification events ****/
#define NOTIFY_SRCH_HANDLE_ERROR    1   /**< An error occurred in Find File handle. Application should clean the related resources. */
#define NOTIFY_FILE_HANDLE_ERROR    2   /**< An error occurred in File handle. Application should clean the related resources. */
/** @} end of group notification Driver Notification Events */

/** @ingroup file */
/** @defgroup attributes File attributes
 * @{
 */
/* *** File attributes ****/
#define CIFS_ATTR_READONLY      0x01 /**< file is read-only */
#define CIFS_ATTR_HIDDEN        0x02 /**< file is hidden */
#define CIFS_ATTR_SYSTEM        0x04 /**< file is a system file */
#define CIFS_ATTR_VOLUME        0x08 /**< this is a volume, not a file */
#define CIFS_ATTR_DIR           0x10 /**< the file is directory */
#define CIFS_ATTR_ARCHIVE       0x20 /**< file is scheduled for archiving */
#define CIFS_ATTR_ALL           (CIFS_ATTR_HIDDEN | CIFS_ATTR_SYSTEM | CIFS_ATTR_DIR)
/** @} end of group attributes File attributes */

/** @ingroup file */
/** @defgroup access File access mode
 * @{
 */
/* *** File access mode ****/
#define FILE_AM_READ            0   /**< read-only */
#define FILE_AM_WRITE           1   /**< write-only */
#define FILE_AM_READ_WRITE      2   /**< read and write */
#define FILE_AM_SPECIAL_MASK    0x8000000
/**< When this bit is set in the <i>access</i> value of <i>ccCreateFile()</i>
   it has special meaning. After stripping off this bit, NQ uses it
   as <i>DesiredAccess</i> in the <i>NtCreateAndX</i>(SMB) command
   or in the <i>Create</i>(SMB2) command.                        */

/** @} end of group access File access mode */

/** @ingroup file */
/** @defgroup share File share mode
 * @{
 */
/* *** File share mode ****/
#define FILE_SM_COMPAT          0   /**< compatibility mode */
#define FILE_SM_EXCLUSIVE       1   /**< deny read/write/execute (exclusive) */
#define FILE_SM_DENY_WRITE      2   /**< deny write */
#define FILE_SM_DENY_READ       3   /**< deny read/execute */
#define FILE_SM_DENY_NONE       4   /**< deny none */
/** @} end of group share File share mode*/

/** @ingroup file */
/** @defgroup locality File locality
 * @{
 */
/* *** File locality ****/
#define FILE_LCL_UNKNOWN        0   /**< locality of reference is unknown */
#define FILE_LCL_SEQUENTIAL     1   /**< mainly sequential access */
#define FILE_LCL_RANDOM         2   /**< mainly random access */
#define FILE_LCL_MIXED          3   /**< random access with some locality */
/** @} end of group locality File locality */

/** @ingroup file */
/** @defgroup create File create action
 * @{
 */
/* *** File create action ****/
#define FILE_CA_FAIL            0   /**< if file does not exist - fail */
#define FILE_CA_CREATE          1   /**< if file does not exist - create it */
/* @note: create action can be one of those values not both */
/** @} end of group create File create action */

/** @ingroup file */
/** @defgroup open File open action
 * @{
 */
/* *** File open action ****/
#define FILE_OA_FAIL            0   /**< if file exists - fail */
#define FILE_OA_OPEN            1   /**< if file exists - open it */
#define FILE_OA_TRUNCATE        2   /**< if file exists - open it and truncate */
/* @note: open action can be one of those values but not a combination */
/** @} end of group open File open action */

/** @ingroup file */
/** @defgroup seek File seek constants
 * @{
 */
/* File seek method */
#define SEEK_FILE_BEGIN         0   /**< seek from start of file */
#define SEEK_FILE_CURRENT       1   /**< seek from the current position in file */
#define SEEK_FILE_END           2   /**< seek from the file end */
/** @} end of group seek File seek constants */


/** @ingroup special */
/** @defgroup authentication Authentication Level
 * @{
 */
/* *** LAN Manager Authentication Level ****/
#define NQ_CC_AUTH_LM_AND_NTLM      1   /**< forces CIFS Client to send both LM and NTLM encrypted password */
#define NQ_CC_AUTH_NTLM             2   /**< forces CIFS Client to send NTLM encrypted password only */
#define NQ_CC_AUTH_NTLM_V2          3   /**< forces CIFS Client to use NTLMv2 authentication */
#if defined(UD_CC_INCLUDEEXTENDEDSECURITY_KERBEROS) || defined(DOXYGEN)
#define NQ_CC_AUTH_SPNEGO_KERBEROS  4   /**< forces CIFS Client to use SPNEGO authentication with an extended security mechanism */
#endif /* UD_CC_INCLUDEEXTENDEDSECURITY_KERBEROS */
/** @} end of group authentication Authentication Level */

/** @ingroup mount_info */
/** @defgroup user_logon_type User logon type
 * @{
 */
/* User logon type */
#define CC_LOGON_TYPE_REGULAR         0   /**< regular user */
#define CC_LOGON_TYPE_ANONYMOUS       1   /**< anonymous user */
#define CC_LOGON_TYPE_GUEST           2   /**< guest user */
/** @} end of group logon User logon type */

/** @ingroup mount_info */
/** @defgroup server_capabilities Server capabilities
 * @{
 */
/* server capabilities */
#define CC_CAPABILITY_DFS                   0x00000001            /**< DFS capability */
#define CC_CAPABILITY_LEASING               0x00000002            /**< leasing capability */
#define CC_CAPABILITY_LARGE_MTU             0x00000004            /**< multi credit operations */
#define CC_CAPABILITY_MULTI_CHANNEL         0x00000008            /**< multi channel capability */
#define CC_CAPABILITY_PERSISTENT_HANDLES    0x00000010            /**< persistent handle capability */
#define CC_CAPABILITY_DIRECTORY_LEASING     0x00000020            /**< directory leasing capability */
#define CC_CAPABILITY_ENCRYPTION            0x00000040            /**< encryption capability */
/** @} end of group server_capabilities Server capabilities */

/** @ingroup mount_info */
/** @defgroup signing_policy signing policy
 * @{
 */
/* signing policy */
#define CC_SIGNING_ENABLED       0x0001       /**< signing enabled */
#define CC_SIGNING_REQUIRED      0x0002       /**< signing required */
/** @} end of group signing_policy signing policy */

/** @ingroup dialects */
/**
    Available SMB dialects.

    See  ccSetSmbDialect() */
typedef enum
{
    CC_SMB100, /**< SMB 1 */
    CC_SMB202, /**< SMB 2.0.2 */
    CC_SMB210, /**< SMB 2.1.0 */
    CC_SMB300, /**< SMB 3.0.0 */
    CC_SMB302, /**< SMB 3.0.2 */
    CC_SMB311  /**< SMB 3.1.1 */
}
CCSmbDialect;

/** @ingroup file */
/** Prototype for a callback function for server side copy chunk response.
   This function is called after each response from the server arrives.
   @param status                The response status
   @param totalBytesWritten     Total bytes that was written in the file
   @param totalFileSize         Total file size
   @param context               A context pointer supplied by the
                                application. By using context, an application
                                can distinguish between different copy chunk operations.
               */
typedef void (* CCServerSideCopyCallback)(NQ_STATUS status, NQ_UINT32 totalBytesWritten, NQ_UINT64 totalFileSize, void* context);

/**@ingroup info
 * This structure is a 64-bit value representing the number of
   100-nanosecond intervals since January 1, 1601 (UTC). The
   application can use the NQ system abstraction layer calls for
   converting time from OS native format to CCFileTime.          */
typedef struct {
    NQ_UINT32 timeLow;      /**< high 32 bits of file time */
    NQ_UINT32 timeHigh;     /**< low 32 bits of file time */
} CCFileTime;

/** @ingroup search
 * This structure is used when calling ccFindFirstFile()
   and ccFindNextFile() calls.                            */
typedef struct CCFindFileDataA{
    NQ_UINT32 fileAttributes;       /**< file attributes as defined in @ref attributes module */
    NQ_UINT32 creationTimeLow;      /**< low 32 bits of file creation time */
    NQ_UINT32 creationTimeHigh;     /**< high 32 bits of file creation time */
    NQ_UINT32 lastAccessTimeLow;    /**< low 32 bits of file last access time */
    NQ_UINT32 lastAccessTimeHigh;   /**< high 32 bits of file last access time */
    NQ_UINT32 lastWriteTimeLow;     /**< low 32 bits of file last write time */
    NQ_UINT32 lastWriteTimeHigh;    /**< high 32 bits of file last write time */
    NQ_UINT32 changeTimeLow;        /**< low 32 bits of file change time */
    NQ_UINT32 changeTimeHigh;       /**< high 32 bits of file change time */
    NQ_UINT32 fileSizeLow;          /**< low 32 bits of file data size */
    NQ_UINT32 fileSizeHigh;         /**< high 32 bits of file data size */
    NQ_UINT32 allocationSizeLow;    /**< low 32 bits of file allocation size */
    NQ_UINT32 allocationSizeHigh;   /**< high 32 bits of file allocation size */
    NQ_UINT32 fileNameLength;       /**< file name length in characters */
    NQ_CHAR fileName[CM_BUFFERLENGTH(NQ_CHAR, UD_FS_FILENAMECOMPONENTLEN)];    /**< file name buffer */
} FindFileDataA_t;

/** @ingroup search */
typedef struct {
    NQ_UINT32 fileAttributes;       /**< file attributes as defined in @ref attributes module*/
    NQ_UINT32 creationTimeLow;      /**< low 32 bits of file creation time */
    NQ_UINT32 creationTimeHigh;     /**< high 32 bits of file creation time */
    NQ_UINT32 lastAccessTimeLow;    /**< low 32 bits of file last access time */
    NQ_UINT32 lastAccessTimeHigh;   /**< high 32 bits of file last access time */
    NQ_UINT32 lastWriteTimeLow;     /**< low 32 bits of file last write time */
    NQ_UINT32 lastWriteTimeHigh;    /**< high 32 bits of file last write time */
    NQ_UINT32 changeTimeLow;        /**< low 32 bits of file change time */
    NQ_UINT32 changeTimeHigh;       /**< high 32 bits of file change time */
    NQ_UINT32 fileSizeLow;          /**< low 32 bits of file data size */
    NQ_UINT32 fileSizeHigh;         /**< high 32 bits of file data size */
    NQ_UINT32 allocationSizeLow;    /**< low 32 bits of file allocation size */
    NQ_UINT32 allocationSizeHigh;   /**< high 32 bits of file allocation size */
    NQ_UINT32 fileNameLength;       /**< file name length in characters */
    NQ_WCHAR fileName[CM_BUFFERLENGTH(NQ_WCHAR, UD_FS_FILENAMECOMPONENTLEN)];    /**< file name buffer */
} CCFindFileData;

/** @ingroup info
 * This structure is used when calling File Information related CIFS Client API calls,
 *  it is used by ccGetFileInformationByNameA() for backward compatibility. */
typedef struct CCFileInfo_t{
    NQ_UINT32 creationTimeLow;      /**< low 32 bits of file creation time */
    NQ_UINT32 creationTimeHigh;     /**< high 32 bits of file creation time */
    NQ_UINT32 lastAccessTimeLow;    /**< low 32 bits of file last access time */
    NQ_UINT32 lastAccessTimeHigh;   /**< high 32 bits of file last access time */
    NQ_UINT32 lastWriteTimeLow;     /**< low 32 bits of file last write time */
    NQ_UINT32 lastWriteTimeHigh;    /**< high 32 bits of file last write time */
    NQ_UINT32 attributes;           /**< file attributes as defined in @ref attributes module*/
    NQ_UINT32 volumeSerialNumber;   /**< volume serial number */
    NQ_UINT32 allocationSizeLow;    /**< low 32 bits of file allocation size */
    NQ_UINT32 allocationSizeHigh;   /**< high 32 bits of file allocation size */
    NQ_UINT32 fileSizeLow;          /**< low 32 bits of file data size */
    NQ_UINT32 fileSizeHigh;         /**< high 32 bits of file data size */
    NQ_UINT32 numberOfLinks;        /**< number of hard links to this file */
    NQ_UINT32 fileIndexLow;         /**< low 32 bits of file ID (always zero) */
    NQ_UINT32 fileIndexHigh;        /**< high 32 bits of file ID (always zero) */
} FileInfo_t;

/**@ingroup info
   This structure describes file information.
   It combines all possible file information fields that may be
   used in information calls. The file information class
   equivalent is <i>FileAllInformation</i>.                     */
typedef struct
{
    NQ_UINT64 creationTime;         /**< File creation time. */
    NQ_UINT64 lastAccessTime;       /**< File last access time. */
    NQ_UINT64 lastWriteTime;        /**< File last write time. */
    NQ_UINT64 changeTime;           /**< File change time. */
    NQ_UINT64 endOfFile;            /**< File size. */
    NQ_UINT64 allocationSize;       /**< File allocation size. */
    NQ_UINT32 attributes;           /**< File attributes. */
    NQ_UINT32 numberOfLinks;        /**< Number of hard links to this file. */
    NQ_UINT64 fileIndex;            /**< File index. */
    NQ_UINT64 volumeId;             /**< Volume Id within which the file is opened.*/
} CCFileInfo; /* File information structure. */

/**@ingroup info
   This structure describes volume information.
   */
typedef struct
{
    NQ_UINT sectorsPerCluster;  /* Number of sectors per cluster. */
    NQ_UINT bytesPerSector;     /* Number of bytes per sector. */
    NQ_UINT64 freeClusters;     /* Number of free clusters. */
    NQ_UINT64 totalClusters;    /* Number of clusters on disk. */
    NQ_UINT fsType;             /* File system type as disk or DC. */
    NQ_UINT serialNumber;       /* Disk serial number. */
} CCVolumeInfo; /* Volume information structure. */

/**@ingroup mount_info
   This structure contain statistics on data transferring
   */
typedef struct
{
    NQ_FLOAT readAmount;    /**< amount of data which was read from share's files in GB */
    NQ_FLOAT writeAmount;   /**< amount of data which was written to share's files in GB */
}CCSStatistics; /* data transferring statistics */

/**@ingroup mount_info
   This structure describes server information.
   */
typedef struct
{
    CCSmbDialect  dialect;                    /**< SMB dialect, see: #CCSmbDialect */
    NQ_UINT32     serverCapabilities;         /**< server capabilities, combination of the following bits: #CC_CAPABILITY_DFS ,#CC_CAPABILITY_LEASING, #CC_CAPABILITY_LARGE_MTU, #CC_CAPABILITY_MULTI_CHANNEL, #CC_CAPABILITY_PERSISTENT_HANDLES, #CC_CAPABILITY_DIRECTORY_LEASING, #CC_CAPABILITY_ENCRYPTION */
    NQ_IPADDRESS  *ips;                       /**< pointer to server IP addresses. */
    NQ_COUNT      numIps;                     /**< number of IP addresses in the array above.*/
    NQ_INT        serverSigningPolicy;        /**< server signing policy, combination of the following bits: #CC_SIGNING_ENABLED ,#CC_SIGNING_REQUIRED */
} CCServerInfo; /* server information structure. */

/**@ingroup mount_info
   This structure describes mount information. This is ASCII version of CCMountInfo.
   */
typedef struct
{
    NQ_TIME       creationTime;               /**< mount point creation time. */
    NQ_INT        userLogonType;              /**< user logon type, see: #CC_LOGON_TYPE_REGULAR, #CC_LOGON_TYPE_ANONYMOUS, #CC_LOGON_TYPE_GUEST */
    NQ_CHAR       *serverName;                /**< server name */
    CCServerInfo  serverInfo;                 /**< server information */
    CCSStatistics statistics;                 /**< statistics on data transferring from and to the share */
} CCMountInfoA; /* ASCII version of mount information structure. */

/**@ingroup mount_info
   This structure describes mount information.
   */
typedef struct
{
    NQ_TIME       creationTime;               /**< mount point creation time. */
    NQ_INT        userLogonType;              /**< user logon type, see: #CC_LOGON_TYPE_REGULAR, #CC_LOGON_TYPE_ANONYMOUS, #CC_LOGON_TYPE_GUEST */
    NQ_WCHAR      *serverName;                /**< server name */
    CCServerInfo  serverInfo;                 /**< server information */
    CCSStatistics statistics;                 /**< statistics on data transferring from and to the share */
} CCMountInfo; /* Mount information structure. */

/**@ingroup cc_network
 * This structure is used when calling ccNetworkGetNextShareItem()
 *  CCNetShareItem - ASCII version */
typedef struct{
    NQ_CHAR *   name;               /**< share name */
    NQ_UINT32   type;               /**< share type */
    NQ_CHAR *   comment;            /**< share comment */
}CCNetShareItemA;

/**@ingroup cc_network */
/* CCNetShareItem */
typedef struct{
    NQ_WCHAR *  name;               /**< share name */
    NQ_UINT32   type;               /**< share type */
    NQ_WCHAR *  comment;            /**< share comment */
}CCNetShareItem;

/**@ingroup cc_network */
/* CCNetServerItem */
typedef struct{
    NQ_WCHAR *  name;               /**< server name */
    NQ_UINT32   type;               /**< type */
    NQ_WCHAR * comment;             /**< server comment */
    NQ_BYTE majorVersion;           /**< major version */
    NQ_BYTE minorVersion;           /**< minor version */
}CCNetServerItem;

/**@ingroup cc_network */
/* CCNetDomainItem */
typedef struct{
    NQ_WCHAR *  name;               /**< domain name */
    NQ_UINT32   type;               /**< type */
    NQ_WCHAR * comment;             /**< domain comment */
    NQ_BYTE majorVersion;           /**< major version */
    NQ_BYTE minorVersion;           /**< minor version */
}CCNetDomainItem;

/*********************************************************************
 * API functions
 ********************************************************************/

/** @ingroup client */
/** @defgroup init_shut Initialization and Shutdown
 * @{
 */

/**
   Initialization functions. Application should call this
   function at its startup to initialize NQ Client.

   The <i>fsNotify</i> parameter is used for callback
   notification. Application can provide this function to become
   informed of important events inside NQ Notification. It is
   useful when, for instance, NQ Client is being wrapped as a local
   file system.

   @param fsdNotify  Pointer to the notification function. This value can
               be NULL.
   @returns
   None  Application can examine the error code for the failure reason. */
NQ_BOOL ccInit(void (*fsdNotify)(NQ_INT eventId, NQ_ULONG param));

/**
   This function stops NQ Client and releases all its resources.
                                                          */
void ccShutdown(void);

/**
   This function checks whether NQ Client has started.
   @returns
   TRUE when NQ Client was initialized and FALSE otherwise.          */
NQ_BOOL ccIsInitialized(void);

/** @} end of group init_shut Initialization and Shutdown */

/** @ingroup client */
/** @defgroup mount Mount and Server
 *  @{
 */
/**
   This function mounts the remote share as a local sub
   directory under the local virtual network file system.
   @param mountPoint  Local mount point for the share. The
                      leading backslash is required. Example: <i>\\remoteShare1.</i>
   @param remotePath  Path to the remote share, starting from
                      host name. Path syntax is <i>\\\\\<host\>\\\<share\>.</i>
   @param connect     A value of TRUE means that NQ will
                      immediately connect to the remote share.
                      Use FALSE to postpone connection to the
                      first operation over this mount point. 
   @return This function returns 0 if the mount point was
   connected successfully or -1 otherwise. Application can
   examine the error code for the failure reason.

   @see
   ccRemoveMount()

   @note
   This is an ASCII version of ccAddMount() */
NQ_INT ccAddMountA(const NQ_CHAR *mountPoint, const NQ_CHAR *remotePath, NQ_BOOL connect);

/**
   This function mounts the remote share as a local sub
   directory under the local virtual network file system.
   @param mountPoint  Local mount point for the share. The
                      leading backslash is required. Example: <i>\\remoteShare1.</i>
   @param remotePath  Path to the remote share, starting from
                      host name. Path syntax is <i>\\\\\<host\>\\\<share\>.</i>
   @param connect     A value of TRUE means that NQ will
                      immediately connect to the remote share.
                      Use FALSE to postpone connection to the
                      first operation over this mount point. 
   @return This function returns 0 if the mount point was
   connected successfully or -1 otherwise. Application can
   examine the error code for the failure reason.

   @see ccRemoveMount()                              */
NQ_INT ccAddMount(const NQ_WCHAR *mountPoint, const NQ_WCHAR *remotePath, NQ_BOOL connect);

/**
   This function mounts the remote share as a local sub directory
   under the local virtual network file system.
  @param mountPoint Local mount point for the share. The leading
                     backslash is required. Example: <i>\\remoteShare1.</i>
   @param remotePath Path to the remote share, starting from host
                     name. Path syntax is <i>\\\\\<host\>\\\<share\>.</i>
   @param connect    A value of TRUE means that NQ will immediately
                     connect to the remote share. Use FALSE to
                     postpone connection to the first operation over
                     this mount point.
   @param userCredentials  Pointer to a structure, which stores the
                     user credentials. See AMCredentialsA structure in amapi.h file.
                     If NULL is supplied anonymous credentials will be used.
                     To initialize that structure use amCredentialsInitA() function from amcredentials file.
                     In order to use anonymous credentials pass empty strings to amCredentialsInitA():
                     amCredentialsInitA(credentials, "", "", "", 0).
                     In order to use anonymous credentials with password pass empty strings and password to amCredentialsInitA():
                     amCredentialsInitA(credentials, "", "", passwordString, 0).
                     It is the caller responsibility to release that structure.

   @return   This function returns 0 if the mount point was connected
   successfully or -1 otherwise. Application can examine the
   error code for the failure reason.
   @see
   ccRemoveMount()

   @note
   This is an ASCII version of ccAddMountWithCreds() */
NQ_INT ccAddMountWithCredsA(const NQ_CHAR * mountPoint, const NQ_CHAR * remotePath, NQ_BOOL connect, const AMCredentialsA * userCredentials);

/**
   This function mounts the remote share as a local sub directory
   under the local virtual network file system.
   @param mountPoint Local mount point for the share. The leading
                     backslash is required. Example: <i>\\remoteShare1.</i>
   @param remotePath Path to the remote share, starting from host
                     name. Path syntax is <i>\\\\\<host\>\\\<share\>.</i>
   @param connect    A value of TRUE means that NQ will immediately
                     connect to the remote share. Use FALSE to
                     postpone connection to the first operation over
                     this mount point.
   @param userCredentials  Pointer to a structure, which stores the
                     user credentials. See AMCredentials structure in amapi.h file.
                     If NULL is supplied anonymous credentials will be used.
                     To initialize that structure use amCredentialsInitW() from amcredentials file.
                     In order to use anonymous credentials pass empty strings to amCredentialsInitW():
                     amCredentialsInitW(credentials, emptyWideCharString, emptyWideCharString, emptyWideCharString, 0).
                     In order to use anonymous credentials with password pass empty strings and password to amCredentialsInitW():
                     amCredentialsInitW(credentials, emptyWideCharString, emptyWideCharString, passwordString, 0).
                     It is the caller responsibility to release that structure.
   @return   This function returns 0 if the mount point was connected
   successfully or -1 otherwise. Application can examine the
   error code for the failure reason.
   @see
   ccRemoveMount()                                             */
NQ_INT ccAddMountWithCreds(const NQ_WCHAR * mountPoint, const NQ_WCHAR * remotePath, NQ_BOOL connect, const AMCredentials * userCredentials);

/**
   This function makes mount point unavailable. It does not
   close the files opened over this mount point.
   @param mountPoint  The name of the mount point to be removed. 
   @return This function returns 0 if the mount point has been
   successfully removed or -1 otherwise. Application can examine
   the error code for the failure reason.

   @see
   ccAddMount()

   @note
   This is an ASCII version of ccRemoveMount()*/
NQ_INT ccRemoveMountA(const NQ_CHAR *mountPoint);

/**
   This function makes mount point unavailable. It does not
   close the files opened over this mount point.
   @param mountPoint  The name of the mount point to be removed. 
   @return This function returns 0 if the mount point has been
   successfully removed or -1 otherwise. Application can examine
   the error code for the failure reason.

   @see
   ccAddMount() */
NQ_INT ccRemoveMount(const NQ_WCHAR *mountPoint);

/**
   This function returns globally unique server identifier
   (GUID) associated with the given mount point.
   @param mountpoint  The name of the mount point.
   @return Globally unique server identifier 16 bytes long if
   available, zero identifier otherwise.
   @note
     \* Server GUID must not be used by a client as a secure
       method of identifying a server. Windows-based servers
       generate a new GUID each time they are started.
     \* This is an ASCII version of ccGetServerId()*/
NQ_GUID ccGetServerIdA(const NQ_CHAR * mountpoint);

/**
   This function returns globally unique server identifier (GUID) associated with the
   given mount point.

   @param mountpoint   The name of the mount point.
   @returns
   Globally unique server identifier 16 bytes long if available, zero identifier otherwise.
   @note
   Server GUID must not be used by a client as a secure method of identifying a server.
   Windows-based servers generate a new GUID each time they are started.      */
NQ_GUID ccGetServerId(const NQ_WCHAR * mountpoint);

/** @} end of group mount Mount and Server */

/** @ingroup client */
/** @defgroup mount_info Mount information
 * @{
 */

/**
   This function is called by application to get information about specific mount point.

   @param mountPoint       The name of the mount point.
   @param mountPointInfo   pointer to CCMountInfoA structure to fill.
   @return
   TRUE if the information was written successfully and FALSE otherwise.
   @see
   CCMountInfoA structure.
   @note
   This is an ASCII version of ccGetMountInformation().
   Memory for ips (field of CCServerInfo) and serverName (field of CCMountInfoA) is
   allocated by this function and it is the caller responsibility to release it    */
NQ_BOOL ccGetMountInformationA(const NQ_CHAR * mountPoint, CCMountInfoA * mountPointInfo);

/**
   This function is called by application to get information about specific mount point.

   @param mountPoint       The name of the mount point.
   @param mountPointInfo   pointer to CCMountInfo structure to fill.
   @returns
   None.
   @see
   CCMountInfo structure.
   @return
   TRUE if the information was written successfully and FALSE otherwise.
   @note
   Memory for ips (field of CCServerInfo) and serverName (field of CCMountInfoA) is
   allocated by this function and it is the caller responsibility to release it    */
NQ_BOOL ccGetMountInformation(const NQ_WCHAR * mountPoint, CCMountInfo * mountPointInfo);
/** @} end of group mount_info Mount information */

/** @ingroup client */
/** @defgroup search Search Operations
 * @{
 */

/**
   This function is called by application to find a first file
   matching the specified search criteria. On success, this
   function returns a search handle which can be used by
   application for getting more search results (ccFindNextFile()).
   At the end of the search the application should close this
   handle by calling ccFindClose().
   
   The <i>srchPath</i> argument (see below) designates either a
   local or a remote path. A remote path starts with a mount
   point name. Any other path, which does not start from a mount
   point is considered local. A local path, regardless of its
   form, designates a list of mount points.
   @param srchPath      Path and the filename to search on the
                        remote share. Wildcard characters are
                        applicable for the filename. 
   @param findFileData  Pointer to a structure, where NQ places
                        the search result data in CCFindFileData structure.
   @param extractFirst  This flag specifies whether NQ should
                        place the search results data for this
                        call (if TRUE), or it should only open
                        the search handle for the specified path
                        for future search data retrieval (if
                        FALSE) by calling ccFindNextFile().
   @return This function returns NULL if it cannot create a
   handle or a valid handle otherwise. The application can
   inspect the error code for the failure reason. Error code
   NQ_ERR_OK means that there are no files on a remote share
   matching the search criteria CCFindFileData structure
   @see
   ccFindNextFile()
   ccFindClose()
   @note
     \* When application calls this function with a local path,
       NQ Client will lock the list of mount points until the ccFindClose()
       will be called. During that period any other application
       thread trying to call a function from NQ Client API may
       remain locked.
     \* This is an ASCII version of ccFindFirstFile() */
NQ_HANDLE ccFindFirstFileA(const NQ_CHAR *srchPath, FindFileDataA_t *findFileData, NQ_BOOL extractFirst);

/**
   This function is called by application to find a first file
   matching the specified search criteria. On success, this
   function returns a search handle which can be used by
   application for getting more search results (ccFindNextFile()).
   At the end of the search the application should close this
   handle by calling ccFindClose().
   
   The <i>srchPath</i> argument (see below) designates either a
   local or a remote path. A remote path starts with a mount
   point name. Any other path, which does not start from a mount
   point is considered local. Specifying a mount point name for example '\\mnt' or '\\*' or '\\*.*' will return a list of mount points currently mounted.
   @param srchPath      Path and the filename to search on the
                        remote share. Wildcard characters are
                        applicable for the filename. 
   @param findFileData  Pointer to a structure, where NQ places
                        the search result data in CCFindFileData structure.
   @param extractFirst  This flag specifies whether NQ should
                        place the search results data for this
                        call (if TRUE), or it should only open
                        the search handle for the specified path
                        for future search data retrieval (if
                        FALSE) by calling ccFindNextFile().
   @return This function returns NULL if it cannot create a
   handle or a valid handle otherwise. The application can
   inspect the error code for the failure reason. Error code
   NQ_ERR_OK means that there are no files on a remote share
   matching the search criteria.

   @see

   CCFindFileData structure
   ccFindNextFile()
   ccFindClose()

   @note
   When application calls this function with a local path, NQ
   Client will lock the list of mount points until the ccFindClose()
   will be called. During that period any other application
   thread trying to call a function from NQ Client API may
   remain locked.                                                                                                    */
NQ_HANDLE ccFindFirstFile(const NQ_WCHAR *srchPath, CCFindFileData *findFileData, NQ_BOOL extractFirst);

/**
   This function is called by application to find a next file
   matching the specified wildcard.
   @param handle        Handle value returned by calling ccFindFirstFile().
   @param findFileData  Pointer to a structure, where NQ places
                        the search result data in CCFindFileData structure.
   @return This function returns TRUE if the next file was found
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason. Error code NQ_ERR_OK
   means that there are no more files on a remote share matching
   the search criteria.

   @see

   CCFindFileData structure
   ccFindFirstFile()
   ccFindClose()
   @note
   This is an ASCII version of ccFindNextFile()                                            */
NQ_BOOL ccFindNextFileA(NQ_HANDLE handle, FindFileDataA_t *findFileData);

/**
   This function is called by application to find a next file
   matching the specified wildcard.
   @param handle        Handle value returned by calling ccFindFirstFile().
   @param findFileData  Pointer to a structure, where NQ places
                        the search result data in CCFindFileData structure.
   @return This function returns TRUE if the next file was found
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason. Error code NQ_ERR_OK
   means that there are no more files on a remote share matching
   the search criteria.

   @see
   CCFindFileData structure
   ccFindFirstFile()
   ccFindClose()                                                                                               */
NQ_BOOL ccFindNextFile(NQ_HANDLE handle, CCFindFileData *findFileData);

/**
   This function is called by application to close the search
   handle
   @param handle  Handle value returned by calling the ccFindFirstFile()
   @return This function returns TRUE if the search handle is
   closed successfully or FALSE otherwise. The application can
   inspect the error code for the failure reason.                                                                                          */
NQ_BOOL ccFindClose(NQ_HANDLE handle);

/** @} end of group search Search Operations */

/** @ingroup client */
/** @defgroup directory Directory Operations
 * @{
 */

/**
   This function is called by application to create a directory.
   @param pathName  Path of the directory to be created
   @return This function returns TRUE if directory is created
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason.
   @note
   This is an ASCII version of ccCreateDirectory()*/
NQ_BOOL ccCreateDirectoryA(const NQ_CHAR *pathName);

/**
   This function is called by application to create a directory.

   @param pathName  Path of the directory to be created
   @returns
   This function returns TRUE if directory is created
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason.                        */
NQ_BOOL ccCreateDirectory(const NQ_WCHAR *pathName);

/**
   This function is called by application to remove a directory.
   @param pathName  Path of the directory to be removed.
   @return This function returns TRUE if the specified directory
   is removed successfully or FALSE otherwise. The application
   can inspect the error code for the failure reason. The
   directory should be empty, otherwise error condition will
   occur.
   @note
   This is an ASCII version of ccRemoveDirectory()*/
NQ_BOOL ccRemoveDirectoryA(const NQ_CHAR *pathName);

/**
   This function is called by application to remove a directory.

   @param pathName  Path of the directory to be removed.
   @returns
   This function returns TRUE if the specified directory is
   removed successfully or FALSE otherwise. The application can
   inspect the error code for the failure reason. The directory
   should be empty, otherwise error condition will occur.        */
NQ_BOOL ccRemoveDirectory(const NQ_WCHAR *pathName);

/** @} end of group directory Directory Operations */

/** @ingroup client */
/** @defgroup file File Operations
 * @{
 */

/**
   This function is called by application to create or open a file.
   @param fileName      Path of the file to be created/opened. 
   @param access        Desired file access mode (see @ref access module).
   @param shareMode     Desired file share mode (see @ref share module).
   @param locality      Desired file locality (see @ref locality module).
   @param writeThrough  If TRUE then no read ahead or write
                        behind allowed on this file or device.
                        When the response is returned, data is
                        expected to be on the disk or device. 
   @param attributes    File attributes for a newly created file
                        (see @ref attributes module)
   @param createAction  Desired file create action (see @ref create module).
   @param openAction    Desired file open action (see @ref open module).
   @return This function returns NULL if it cannot create/open a
   file or a valid handle otherwise. The application can
   inspect the error code for the failure reason.
   @note
   This is an ASCII version of ccCreateFile()*/
NQ_HANDLE ccCreateFileA(const NQ_CHAR *fileName, NQ_INT access, NQ_INT shareMode,
                        NQ_INT locality, NQ_BOOL writeThrough, NQ_UINT16 attributes,
                        NQ_INT createAction, NQ_INT openAction);

/**
   This function is called by application to create or open a
   file.
   @param fileName      Path of the file to be created/opened. 
   @param access        Desired file access mode (see @ref access
                        module: #FILE_AM_READ, #FILE_AM_SPECIAL_MASK,
                        #FILE_AM_READ_WRITE, #FILE_AM_WRITE).
   @param shareMode     Desired file share mode (see @ref share
                        module: #FILE_SM_COMPAT, #FILE_SM_DENY_NONE,
                        #FILE_SM_DENY_READ, #FILE_SM_DENY_WRITE,
                        #FILE_SM_EXCLUSIVE).
   @param locality      Desired file locality (see
                        @ref locality module: #FILE_LCL_MIXED, #FILE_LCL_RANDOM,
                        #FILE_LCL_SEQUENTIAL, #FILE_LCL_UNKNOWN).
   @param writeThrough  If TRUE then no read ahead or write
                        behind allowed on this file or device.
                        When the response is returned, data is
                        expected to be on the disk or device. 
   @param attributes    File attributes for a newly created file
                        (see @ref attributes module: #CIFS_ATTR_ALL,
                        #CIFS_ATTR_ARCHIVE, #CIFS_ATTR_DIR,
                        #CIFS_ATTR_HIDDEN, #CIFS_ATTR_READONLY,
                        #CIFS_ATTR_SYSTEM, #CIFS_ATTR_VOLUME).
   @param createAction  Desired file create action, can be one of the following values:
                        #FILE_CA_CREATE, #FILE_CA_FAIL.
   @param openAction    Desired file open action, can be one of the following values:
                        #FILE_OA_FAIL, #FILE_OA_OPEN,
                        #FILE_OA_TRUNCATE.
   @return This function returns NULL if it cannot create/open a
   file or a valid handle otherwise. The application can
   inspect the error code for the failure reason.                                     */
NQ_HANDLE ccCreateFile(const NQ_WCHAR *fileName, NQ_INT access, NQ_INT shareMode,
                        NQ_INT locality, NQ_BOOL writeThrough, NQ_UINT16 attributes,
                        NQ_INT createAction, NQ_INT openAction);

/**
   This function is called by application to delete a file.
   @param fileName  Path of the directory to be created
   @return This function returns TRUE if the specified file is
   deleted successfully or FALSE otherwise. The application can
   inspect the error code for the failure reason.
   @note
   This is an ASCII version of ccDeleteFile()*/
NQ_BOOL ccDeleteFileA(const NQ_CHAR *fileName);

/**
   This function is called by application to delete a file.

   @param fileName  Path of the directory to be created
   @returns
   This function returns TRUE if the specified file is deleted
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason.                        */
NQ_BOOL ccDeleteFile(const NQ_WCHAR *fileName);

/**
   This function is called by application to move (rename) a
   file/directory to a different location. note, that a file
   cannot be moved between different remote shares.
   @param oldFilename  Path of the file/directory to be moved. 
   @param newFilename  Path of the file/directory to be moved to.
   @return This function returns TRUE if the specified file is
   moved successfully or FALSE otherwise. The application can
   inspect the error code for the failure reason.
   @note
   This is an ASCII version of ccMoveFile()*/
NQ_BOOL ccMoveFileA(const NQ_CHAR *oldFilename, const NQ_CHAR *newFilename);

/**
   This function is called by application to move (rename) a
   file/directory to a different location. note, that a file
   cannot be moved between different remote shares.

   @param oldFilename Path of the file/directory to be moved.
   @param newFilename  Path of the file/directory to be moved to.
   @returns
   This function returns TRUE if the specified file is moved
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason.                       */
NQ_BOOL ccMoveFile(const NQ_WCHAR *oldFilename, const NQ_WCHAR *newFilename);

/**
   This function is called by application to read data from the
   opened file.
   
   NQ attempts to use the biggest available payload. However, if
   the <i>count</i> parameter is big enough, calling this
   function results in transmitting multiple SMB requests. NQ
   Client attempts to send these requests concurrently. Since
   this call is synchronous, NQ will wait for all responses to
   come back from the server.
   
   NQ Client reads file bytes from its current position. For
   random access, use function ccSetFilePointer().
   @param hndl      Handle value returned by calling ccCreateFile()
   @param buffer    Pointer to a buffer to read the file data to 
   @param count     Number of bytes to read from the file 
   @param readSize  The pointer to a variable which on exit
                    receives the number of bytes actually read.
                    This value can be NULL. 
   @return This function returns TRUE if the data is read
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason.                                                                                                                */
NQ_BOOL ccReadFile(NQ_HANDLE hndl, NQ_IOBufPos buffer, NQ_UINT count, NQ_UINT *readSize);

/**
   This function is called by application to read data from the
   opened file.
   
   NQ attempts to use the biggest available payload. However, if
   the <i>count</i> parameter is big enough, calling this
   function results in transmitting multiple SMB requests. NQ
   Client attempts to send these requests concurrently. Since
   this call is asynchronous, NQ will not wait for all responses
   to come back from the server but will rather call the <i>callback</i>
   function after receiving the last response.
   
   NQ places the actual number of data read into the first
   argument of the call to the <i>callback </i>function.
   
   NQ Client reads bytes starting from the current position in
   the file. For random read use function ccSetFilePointer().
   
   This function is not thread-protected over the same file. It
   assumes that there is just one thread reading file through
   the given handle. Otherwise, the result is undefined. It is
   possible, however, to read the same file over two different
   handles simultaneously.
   @param hndl      Handle value returned by calling ccCreateFile().
   @param buffer    Pointer to a buffer to use for reading. 
   @param count     Number of bytes to write to the file. 
   @param context   A context pointer supplied by the
                    application. By using context, an application
                    can distinguish between different write
                    operations. For casting purposes first structure field must be of
                     type NQ_BOOL representing whether response is postponed by the server, used internally by NQ Client.
   @param callback  Pointer to a callback function supplied by
                    application. This function accepts operation
                    status, the actual number of bytes read and
                    an abstract context pointer supplied by the
                    application. 
   @return This function returns TRUE if the read operations
   where successfully queued or FALSE otherwise.
   
   The application can inspect the error code for the failure
   reason.
   
   Because of the asynchronous character of this operation the
   value does not reflect read results.
   
   Application should use <i>callback</i> to analyze read
   results.                                                                                                                                               */
NQ_BOOL ccReadFileAsync(NQ_HANDLE hndl, NQ_IOBufPos buffer, NQ_UINT count, void * context, void (* callback)(NQ_STATUS, NQ_UINT, void *));

/**
   This function is called by application to write data to an
   open file.
   
   Upon successful completion NQ places the actual number of
   data written into the memory location pointed by parameter
   writtenSize. On an error, this value is left unmodified.
   
   NQ Client writes bytes starting from the current position in
   the file. For random write use function ccSetFilePointer().
   
   The ccWriteFile() function can be also called for file
   truncation. Use ccSetFilePointer()
   function first to set file pointer into the desired
   end-of-file position. Then call the current function with
   zero data size. This will truncate the file.
   @param hndl         Handle value returned by calling ccCreateFile().
   @param buffer       Pointer to a buffer with bytes to be
                       written.
   @param count        Number of bytes to write to the file.
   @param writtenSize  Pointer to a variable which will receive
                       the number of bytes actually written. This
                       value can be NULL.
   @return This function returns TRUE if the data is written
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason.                                                                                                                    */
NQ_BOOL ccWriteFile(NQ_HANDLE hndl, NQ_IOBufPos buffer, NQ_UINT count, NQ_UINT *writtenSize);

/**
   This function is called by application to write data to an
   open file using asynchronous operations.
   
   Upon successful completion NQ schedules necessary write
   operations. When all writes will be complete, NQ will call
   the <i>callback</i> function.
   
   When calling the <i>callback</i> function, NQ places the
   actual number of data written into the first argument of this
   call.
   
   NQ Client writes bytes starting from the current position in
   the file. For random write use function ccSetFilePointer().
   
   The ccWriteFile() function can be also called for file
   truncation. Use ccSetFilePointer()
   function first to set file pointer into the desired
   end-of-file position. Then call the current function with
   zero data size. This will truncate the file.
   
   This function is not thread-protected over the same file. It
   assumes that there is just one thread writing to the file
   through the given handle. Otherwise, the result is undefined.
   It is possible, however, to write the same file over two
   different handles simultaneously. The permission to open the
   same file twice for read depends on the server's policies.

   @param hndl      Handle value returned by calling ccCreateFile().
   @param buffer    Pointer to a buffer with bytes to be written.
   @param count     Number of bytes to write to the file. 
   @param context   A context pointer supplied by the
                    application. By using context, an application
                    can distinguish between different write
                    operations. 
   @param callback  Pointer to a callback function supplied by
                    application. This function accepts the actual
                    number of bytes written and an abstract
                    context pointer supplied by the application. For casting purposes first structure field must be of
                     type NQ_BOOL representing whether response is postponed by the server, used internally by NQ Client. 
   @return This function returns TRUE if the write operations
   where successfully queued or FALSE otherwise. The application
   can inspect the error code for the failure reason.
   
   Because of the asynchronous character of this operation the
   value does not reflect write results. Application should use <i>callback</i>
   to analyze write results.                                                                                                                              */
NQ_BOOL ccWriteFileAsync(NQ_HANDLE hndl, NQ_IOBufPos buffer, NQ_UINT count, void * context, void (* callback)(NQ_STATUS , NQ_UINT, void *));

/**
   This function is called by application to force server to
   synchronize its local buffers with the file contents.
   @param handle  Handle value returned by calling ccCreateFile()
   @return This function returns TRUE if the next file was found
   successfully or FALSE otherwise. The application can inspect
   the error code for the failure reason.                                                                                                                    */
NQ_BOOL ccFlushFile(NQ_HANDLE handle);

/**
   This function is called by application to close the file
   handle
   @return This function returns TRUE if the file handle is
   closed successfully or FALSE otherwise. The application can
   inspect the error code for the failure reason.
   @param handle  Handle value returned by calling ccCreateFile() */
NQ_BOOL ccCloseHandle(NQ_HANDLE handle);

/** @} end of group file File Operations */

/** @ingroup client */
/** @defgroup info File Info Operations
 * @{
 */

/**
   This function is called by application to withdraw the
   current position in the file. The result is returned as a
   64-bit number.
   @param handle  Handle value returned by calling ccCreateFile().
   @param offset  Pointer to 64 bit value of the file offset.
   @return This function returns TRUE if the specified file
   information was retrieved successfully or FALSE otherwise.
   The application can inspect the error code for the failure
   reason.                                                                                                                                                    */
NQ_BOOL ccGetFilePointer(NQ_HANDLE handle, NQ_UINT64 * offset);

/**
   This function is called by application to set a file pointer
   of the file read/write location in the opened file.
   @param handle  Handle value returned by calling ccCreateFile().
   @param offset  Pointer to 64 bit value of the file offset.
                  Upon successful return this variable contains
                  new offset.
   @param method  File seek method (see #SEEK_FILE_BEGIN, #SEEK_FILE_CURRENT
                  and #SEEK_FILE_END Macros in @ref seek module)
   @return This function returns TRUE if the specified file
   information was set successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason. Upon successful return offset variable contains new
   offset. Use cmS64Assign() for initializing offset parameter.                                                                                         */
NQ_BOOL ccSetFilePointer(NQ_HANDLE handle, NQ_INT64 * offset, NQ_INT method);

/**
   This function is called by application to perform server side
   copy to a new location. The source and the destination files must be on the same server.
   Some servers do not support server-side data copy operation such as
   SMB1 servers, old versions of SAMBA and NetApp.
   In this case the error code that will be set is NQ_ERR_NOTSUPPORTED.
   Some servers do not support server-side data copy across different shares such as
   old versions of SAMBA and lower or equal to SMB 3.0.0 versions of MAC.
   In this case the error code that will be set is NQ_ERR_NOTSUPPORTED.
   @param srcPath       Path to the source file.
   @param dstPath       Path to the destination file.
   @param context       A pointer to a context for the callBack function, if needed.
   @param callBack      User defined method that will be called at a completion of an asynchronous
                        server-side data copy operation, can be null. Currently not in use.
   @return This function returns TRUE if file was copy successfully or FALSE otherwise.
   The application can inspect the error code for the failure reason.
   @note
   This is an ASCII version of ccServerSideDataCopy()   */
NQ_BOOL ccServerSideDataCopyA(NQ_CHAR* srcPath, NQ_CHAR* dstPath, void* context, CCServerSideCopyCallback* callBack);


/**
   This function is called by application to perform server side
   copy to a new location. The source and the destination files must be on the same server.
   Some servers do not support server-side data copy operation such as
   SMB1 servers, old versions of SAMBA and NetApp.
   In this case the error code that will be set is NQ_ERR_NOTSUPPORTED.
   Some servers do not support server-side data copy across different shares such as
   old versions of SAMBA and lower or equal to SMB 3.0.0 versions of MAC.
   In this case the error code that will be set is NQ_ERR_NOTSUPPORTED.
   @param srcPath       Path to the source file.
   @param dstPath       Path to the destination file.
   @param context       A pointer to a context for the callBack function, if needed.
   @param callBack      User defined method that will be called at a completion of an asynchronous
                        server-side data copy operation, can be null. Currently not in use.
   @return This function returns TRUE if file was copy successfully or FALSE otherwise.
   The application can inspect the error code for the failure reason.   */
NQ_BOOL ccServerSideDataCopy(NQ_WCHAR* srcPath, NQ_WCHAR* dstPath, void* context, CCServerSideCopyCallback* callBack);

/**
   This function is called by application to get the free disk
   space information for the specified remote share.
   @param pathName           Path of any object on the remote
                             share the free disk space
                             information is requested for. 
   @param sectorsPerCluster  Pointer to a variable which will
                             receive the number of sectors per
                             cluster on the remote share. 
   @param bytesPerSector     Pointer to a variable which will
                             receive the number of bytes per
                             sector on the remote share. 
   @param freeClusters       Pointer to a variable which will
                             receive the number of free clusters
                             on the remote share. 
   @param totalClusters      Pointer to a variable which will
                             receive the number of total clusters
                             on the remote share. 
   @param fsType             Pointer to a variable which will
                             receive the type of the file system.
   @param serialNumber       Pointer to a variable which will
                             receive the serial number of the
                             file system
   @return This function returns TRUE if the remote disk
   information is received successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.
   @note
   This is an ASCII version of ccGetDiskFreeSpace() */
NQ_BOOL ccGetDiskFreeSpaceA(const NQ_CHAR *pathName, NQ_UINT *sectorsPerCluster,
                            NQ_UINT *bytesPerSector, NQ_UINT *freeClusters, NQ_UINT *totalClusters,
                            NQ_UINT *fsType, NQ_UINT *serialNumber);

/**
   This function is called by application to get the free disk
   space information for the specified remote share.

   @param pathName            Path of any object on the remote share
                              the free disk space information is
                              requested for.
   @param sectorsPerCluster   Pointer to a variable which will receive
                              the number of sectors per cluster on the
                              remote share.
   @param bytesPerSector      Pointer to a variable which will receive
                              the number of bytes per sector on the
                              remote share.
   @param freeClusters        Pointer to a variable which will receive
                              the number of free clusters on the remote
                              share.
   @param totalClusters       Pointer to a variable which will receive
                              the number of total clusters on the
                              remote share.
   @param fsType              Pointer to a variable which will receive
                              the type of the file system.
   @param serialNumber        Pointer to a variable which will receive
                              the serial number of the file system
   @returns
   This function returns TRUE if the remote disk information is
   received successfully or FALSE otherwise. The application can
   inspect the error code for the failure reason.                 */
NQ_BOOL ccGetDiskFreeSpace(const NQ_WCHAR *pathName, NQ_UINT *sectorsPerCluster,
                            NQ_UINT *bytesPerSector, NQ_UINT *freeClusters, NQ_UINT *totalClusters,
                            NQ_UINT *fsType, NQ_UINT *serialNumber);

/**
   Extended version of ccGetDiskFreeSpace().
   When disk is larger than 4 giga clusters - only the extended
   version gives the correct result.
   @param pathName           Path of any object on the remote
                             share the free disk space
                             information is requested for. 
   @param sectorsPerCluster  Pointer to a variable which will
                             receive the number of sectors per
                             cluster on the remote share. 
   @param bytesPerSector     Pointer to a variable which will
                             receive the number of bytes per
                             sector on the remote share. 
   @param freeClusters       Pointer to a variable which will
                             receive the number of free clusters
                             on the remote share. 
   @param totalClusters      Pointer to a variable which will
                             receive the number of total clusters
                             on the remote share. 
   @param fsType             Pointer to a variable which will
                             receive the type of the file system.
   @param serialNumber       Pointer to a variable which will
                             receive the serial number of the
                             file system
   @return This function returns TRUE if the remote disk
   information is received successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.                                                                                                                                           */
NQ_BOOL ccGetDiskFreeSpaceExA(const NQ_CHAR *pathName, NQ_UINT *sectorsPerCluster,
                            NQ_UINT *bytesPerSector, NQ_UINT64 *freeClusters, NQ_UINT64 *totalClusters,
                            NQ_UINT *fsType, NQ_UINT *serialNumber);

/**
   Extended version of ccGetDiskFreeSpace().
   When disk is larger than 4 giga clusters - only the extended
   version gives the correct result.
   @param pathName    Path of any object on the remote share the
                      free disk space information is requested
                      for. 
   @param volumeInfo  Pointer to a variable which will receive
                      the volume information, see CCVolumeInfo.
   @return This function returns TRUE if the remote disk
   information is received successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.                                                                                                                                           */
NQ_BOOL ccGetDiskFreeSpaceEx(const NQ_WCHAR *pathName, CCVolumeInfo *volumeInfo);

/**
   This function is called by application to get attributes of
   the specified file.

   @param fileName   Path to the file.
   @returns
   This function returns file attributes if successful (see @ref attributes module) or NQ_ERR_ATTRERROR in case of failure.
   The application can inspect the error code for the failure reason.                 */
NQ_UINT32 ccGetFileAttributesA(const NQ_CHAR *fileName);

/**
   This function is called by application to get attributes of
   the specified file.

   @param fileName   Path to the file.
   @param attributes  Pointer to attributes value.
   @returns
   This function returns TRUE if successful and the file attributes in attributes parameter (see @ref attributes module)
   otherwise FALSE. The application can inspect the error code for the failure reason.                 */
NQ_BOOL ccGetFileAttributes(const NQ_WCHAR *fileName, NQ_UINT32 *attributes);

/**
   This function is called by application to set attributes of
   the specified file.
   @param fileName    Path to the file. 
   @param attributes  Attributes to set. See @ref attributes module
                      for values. 
   @return This function returns TRUE if the specified
   attributes are set successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.
   @note
   This is an ASCII version of ccSetFileAttributes()*/
NQ_BOOL ccSetFileAttributesA(const NQ_CHAR *fileName, NQ_UINT32 attributes);

/**
   This function is called by application to set attributes of
   the specified file.

   @param fileName     Path to the file.
   @param attributes   Attributes to set. See @ref attributes module
                 for values.
   @returns
   This function returns TRUE if the specified attributes are
   set successfully or FALSE otherwise. The application can
   inspect the error code for the failure reason.
   @see
   @ref attributes module                                             */
NQ_BOOL ccSetFileAttributes(const NQ_WCHAR *fileName, NQ_UINT32 attributes);

/**
   This function is called by application to get the information
   of the specified file.
   @param fileName  Path to the file. 
   @param fileInfo  Pointer to a structure which will receive the
                    file information (see CCFileInfo Structure).
   @return This function returns TRUE if the specified file
   information is obtained successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.
   @note
   This is an ASCII version of ccGetFileInformationByName()*/
NQ_BOOL ccGetFileInformationByNameA(const NQ_CHAR *fileName, FileInfo_t *fileInfo);

/**
   This function is called by application to get the information
   of the specified file.

   @param fileName   Path to the file.
   @param fileInfo   Pointer to a structure which will receive the file
               information (see CCFileInfo Structure).
   @returns
   This function returns TRUE if the specified file information
   is obtained successfully or FALSE otherwise. The application
   can inspect the error code for the failure reason.                       */
NQ_BOOL ccGetFileInformationByName(const NQ_WCHAR *fileName, CCFileInfo *fileInfo);

/**
   This function is called by application to set file size when
   file is specified by its name.

   @param fileName   Path to file.
   @param sizeLow    Low 32 bit desired file size.
   @param sizeHigh   High 32 bit desired file size.
   @returns
   This function returns TRUE if the file size was modified
   successfully or FALSE otherwise.

   When the specified file size is less than the current file
   size, this operation truncates the file to the specified
   size.                                                        */
NQ_BOOL ccSetFileSizeByNameA(const NQ_CHAR *fileName, NQ_UINT32 sizeLow, NQ_UINT32 sizeHigh);

/**
   This function is called by application to set file size when
   file is specified by its name.

   @param fileName   Path to file.
   @param size    64 bit desired file size.
   @returns
   This function returns TRUE if the file size was modified
   successfully or FALSE otherwise.

   When the specified file size is less than the current file
   size, this operation truncates the file to the specified
   size.                                                        */
NQ_BOOL ccSetFileSizeByName(const NQ_WCHAR *fileName, NQ_UINT64 size);

/**
   This function is called by application to apply size to an
   open file. When the specified file size is less than the
   current file size, this operation truncates the file to the
   specified size.
   @param handle  Handle value returned by calling ccCreateFile().
   @param size    64 bit desired file size.
   @return This function returns TRUE if the file size was
   modified successfully or FALSE otherwise.                                                                                                            */
NQ_BOOL ccSetFileSizeByHandle(NQ_HANDLE handle, NQ_UINT64 size);

/**
   This function is called by application to get the time
   information of the specified file.
   @param handle       Handle value returned by calling ccCreateFile()
   @param pCreation    Pointer to a buffer for file creation
                       time. This buffer should hold the time
                       structure (see CCFileTime Structure)
   @param pLastAccess  Pointer to a buffer for file last access
                       time. This buffer should hold the time
                       structure (see CCFileTime Structure)
   @param pLastWrite   Pointer to a buffer for file last write
                       time. This buffer should hold the time
                       structure (see CCFileTime Structure)
   @return This function returns TRUE if the specified file time
   information is obtained successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.                                                                                                                                                  */
NQ_BOOL ccGetFileTime(NQ_HANDLE handle, CCFileTime *pCreation, CCFileTime *pLastAccess, CCFileTime *pLastWrite);

/**
   This function is called by application to set the time
   information of the specified file.
   
   NQ Client sends time modification request using the provided
   time values but the server interpretation of this request may
   apply additional calculations, so that the resulted file
   times may differ. Null can be passed when no change is
   required in one of the times.
   @param handle       Handle value returned by calling ccCreateFile().
   @param pCreation    Pointer to a structure holding the file
                       creation time (see CCFileTime Structure)
   @param pLastAccess  Pointer to a structure holding the file
                       last access time (see CCFileTime Structure)
   @param pLastWrite   Pointer to a structure holding the file
                       last write time (see CCFileTime Structure)
   @return This function returns TRUE if the specified file time
   information is set successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.                                                                                                                                                   */
NQ_BOOL ccSetFileTime(NQ_HANDLE handle, CCFileTime *pCreation, CCFileTime *pLastAccess, CCFileTime *pLastWrite);

/**
   This function is called by application to get the size of the
   specified file.
   @param handle    Handle value returned by calling ccCreateFile()
   @param fileSize  Pointer to a buffer for the NQ_UINT64 value
   @return This function returns TRUE if the specified file
   information is obtained successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.                                                                                                                                               */
NQ_BOOL ccGetFileSize(NQ_HANDLE handle, NQ_UINT64 *fileSize);

/**
   This function is called by application to get the information
   of an open file
   @param handle    Handle value returned by calling ccCreateFile()
   @param fileInfo  Pointer to a structure holding the file
                    information (see CCFileInfo Structure)
   @return This function returns TRUE if the specified file
   information is obtained successfully or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.                                                                                                                                                     */
NQ_BOOL ccGetFileInformationByHandle(NQ_HANDLE handle, CCFileInfo *fileInfo);

/** @} end of group info File Info Operations */

/** @ingroup client */
/** @defgroup timeouts Timeout Operations
 * @{
 */

/**
   Set SMB timeout for waiting a response.

   @param secs   Number of seconds to wait for an SMB response. The
           default value is 15 seconds.
                                                      */
void ccConfigSetTimeout(NQ_UINT32 secs);

/**
   Get the current value of SMB timeout.
   @returns
   Number of seconds to wait for an SMB response. The default value is 15 seconds.
 */
NQ_UINT32 ccConfigGetTimeout(void);

#if defined(UD_NQ_USETRANSPORTNETBIOS) || defined(DOXYGEN)
/**
   This function allows to modify the default timeout
   value for the GetBackupList operation.

   This operation is sent as a broadcast. NQ issues this
   operation when starting network browsing.

   The default value is defined in the
   UD_CC_CLIENTRESPONSETIMEOUT parameter.

   @param seconds   Timeout in seconds.

   @see ccNetworkEnumerateDomains().
   @returns
   This function returns TRUE if seconds is valid parameter or FALSE otherwise.    */
NQ_BOOL ccSetGetBackupListTimeout(NQ_UINT32 seconds);

#endif /* UD_NQ_USETRANSPORTNETBIOS */

/** @} end of group timeouts Timeout Operations */

#if defined(UD_CC_INCLUDEBROWSING) || defined(DOXYGEN)
/** @ingroup client */
/** @defgroup cc_network Network Browsing
 * @{
 */

#if defined(UD_NQ_USETRANSPORTNETBIOS) || defined(DOXYGEN)
/**
   This function defines the name of domain/workgroup that will
   be used as the default in subsequent calls to ccNetwork
   methods.
   @param domain  The name of the default domain/workgroup to be
                  used in subsequent calls. 
   @return This function returns TRUE if domain is valid
   parameter or FALSE otherwise.
   @note
     \* This function is only available when NQ supports the
       NetBIOS transport.
     \* This is an ASCII version of ccNetworkSetDefaultDomain()*/
NQ_BOOL ccNetworkSetDefaultDomainA(const NQ_CHAR *domain);

/**
   This function defines the name of domain/workgroup that will
   be used as the default in subsequent calls to ccNetwork
   methods.
   @param domain  The name of the default domain/workgroup to be
                  used in subsequent calls.
   @return This function returns TRUE if domain is valid
   parameter or FALSE otherwise.
   @note
      This function is only available when NQ supports the
       NetBIOS transport.                                */
NQ_BOOL ccNetworkSetDefaultDomain(const NQ_WCHAR *domain);

/**
    This function sets network browsing cache state.

    Network browsing cache is enabled by default.

    @param timeout Timeout/ttl value for entry cache (0 to disable cache)
    @note
    When cache is enabled domains, servers and shares per server are cached.
    This function is part of the new browser API.     */
void ccNetworkCacheSet(NQ_UINT32 timeout);

/**
   This function is called by application to get the name of the
   workgroup/domain that CIFS Client will use by default. This
   default will be used in those CIFS Client functions where
   workgroup/domain is not explicitly specified.
   @return Buffer pointer to the name of the default
   domain/workgroup.
   @note
     \* This function is relevant when NQ supports the NetBIOS
       transport.
     \* This function is part of the new Browser API. The
       functions of the new Browser API are designated with a <i>ccNetwork...
       </i>prefix.
     \* This an ASCII version of ccNetworkGetDefaultDomain()           */
const NQ_CHAR* ccNetworkGetDefaultDomainA();

/**
   This function is called by application to get the name of the
   workgroup/domain that CIFS Client will use by default. This
   default will be used in those CIFS Client functions where
   workgroup/domain is not explicitly specified.
   @return Buffer pointer to the name of the default
   domain/workgroup.
   @note
     \* This function is relevant when NQ supports the NetBIOS
       transport.
     \* This function is part of the new Browser API. The
       functions of the new Browser API are designated with a <i>ccNetwork...
       </i>prefix.                                                            */
const NQ_WCHAR* ccNetworkGetDefaultDomain();

/**
   This function is called by application to start enumerating
   domains/workgroups. It executes necessary transactions to
   withdraw domains/workgroups announced on the network. After
   successful return from this call, NQ Client creates a list of
   domains/workgroups designated by an abstract handle.
   
   Application may withdraw domain/workgroup names one by one by
   calling ccNetworkGetNextDomainItem()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @return This function returns an enumeration handle.
   Application should use this abstract handle in subsequent
   calls to ccNetworkGetNextDomainItem().
   On error this function returns a NULL handle. Application can
   examine the error code for the failure reason.
   @note
     \* The resulted handle is not thread safe.
     \* This function is only available when NQ supports the
       NetBIOS transport.
                                               */
NQ_HANDLE ccNetworkEnumerateDomains(void);

/**
   Application calls this function after successfully starting a
   domain enumeration (ccNetworkEnumerateDomains()).

   Application may withdraw CCNetDomainItem Items one by one by calling ccNetworkGetNextDomainItem()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.

   @param handle   Enumeration handle as obtained in one of Browser
             enumeration calls.
   @returns
   This function returns a pointer to the next CCNetDomainItem structure in the
   enumeration. This function returns a NULL
   pointer when NQ Client reaches the end of enumeration.  */
CCNetDomainItem * ccNetworkGetNextDomainItem(NQ_HANDLE handle);
#endif /* UD_NQ_USETRANSPORTNETBIOS */

/**
   This function is called by application to start enumerating
   servers of a particular domain/workgroups. It executes
   necessary transactions to withdraw the server list. After
   successful return from this call, NQ Client creates a list of
   servers designated by an abstract handle.
   
   Application may withdraw server names one by one by calling ccNetworkGetNextItemName()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param domain  Domain/workgroup name. This call will return a
                  list of servers for this domain/workgroup. When
                  domain name not supplied the default one will
                  be used see ccNetworkSetDefaultDomain(),
                  in addition all hosts discoverable over
                  WS-Discovery will be returned.
   @return This function returns an enumeration handle.
   Application should use this abstract handle in subsequent
   calls to ccNetworkGetNextItemName().
   On error this function returns a NULL handle. Application can
   examine the error code for the failure reason.
   @note
     \* The resulted handle is not thread-safe.
     \* This is an ASCII version of ccNetworkEnumerateServers()                    */
NQ_HANDLE ccNetworkEnumerateServersA(const NQ_CHAR * domain);

/**
   This function is called by application to start enumerating
   servers of a particular domain/workgroups. It executes
   necessary transactions to withdraw the server list. After
   successful return from this call, NQ Client creates a list of
   servers designated by an abstract handle.
   
   Application may withdraw server names one by one by calling ccNetworkGetNextItemName()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param domain  Domain/workgroup name. This call will return a
                  list of servers for this domain/workgroup. When
                  domain name not supplied the default one will
                  be used see ccNetworkSetDefaultDomain(),
                  in addition all hosts discoverable over
                  WS-Discovery will be returned.
   @return This function returns an enumeration handle.
   Application should use this abstract handle in subsequent
   calls to ccNetworkGetNextItemName().
   On error this function returns a NULL handle. Application can
   examine the error code for the failure reason.
   @note
   The resulted handle is not thread-safe.                                                                                           */
NQ_HANDLE ccNetworkEnumerateServers(const NQ_WCHAR * domain);

/**
   Application calls this function after successfully starting a
   server enumeration (ccNetworkEnumerateServers()).
   
   Application may withdraw CCNetServerItem Items one by one by
   calling ccNetworkGetNextServerItem()
   with the handle obtained in the current call. When finished
   with server enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param handle  Enumeration handle as obtained in one of
                  Browser enumeration calls.
   @return This function returns a pointer to the next
   CCNetServerItem structure in the enumeration. This function
   returns a NULL pointer when NQ Client reaches the end of
   enumeration.                                                                                                 */

CCNetServerItem * ccNetworkGetNextServerItem(NQ_HANDLE handle);

/**
   This function is called by application to start enumerating
   shares on a particular server. It executes necessary
   transactions to withdraw the list of shares. After successful
   return from this call, NQ Client creates a list of shares
   designated by an abstract handle.
   
   Application may withdraw share names one by one by calling ccNetworkGetNextItemName()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param server  Server name. This call will return a list of
                  shares for this server. 
   @return This function returns an enumeration handle.
   Application should use this abstract handle in subsequent
   calls to ccNetworkGetNextItemName().
   On error this function returns a NULL handle. Application can
   examine the error code for the failure reason.
   @note
     \* The resulted handle is not thread-safe.
     \* This is an ASCII version of ccNetworkEnumerateShares()               */
NQ_HANDLE ccNetworkEnumerateSharesA(const NQ_CHAR * server);

/**
   This function is called by application to start enumerating
   shares on a particular server. It executes necessary
   transactions to withdraw the list of shares. After successful
   return from this call, NQ Client creates a list of shares
   designated by an abstract handle.
   
   Application may withdraw share names one by one by calling ccNetworkGetNextItemName()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param server  Server name. This call will return a list of
                  shares for this server.
   @return This function returns an enumeration handle.
   Application should use this abstract handle in subsequent
   calls to ccNetworkGetNextItemName().
   On error this function returns a NULL handle. Application can
   examine the error code for the failure reason.
   @note
   The resulted handle is not thread-safe.                                                                                          */
NQ_HANDLE ccNetworkEnumerateShares(const NQ_WCHAR * server);

/**
   Application calls this function after successfully starting a
   browser enumeration ( ccNetworkEnumerateDomains(),ccNetworkEnumerateServers()
   or ccNetworkEnumerateShares()).
   Application may withdraw share names one by one by calling ccNetworkGetNextItemName()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param handle  Enumeration handle as obtained in one of
                  Browser enumeration calls. 
   @return This function returns a pointer to the next item name
   in the enumeration. Item type depends on the handle source -
   one of the browser enumeration calls. This function returns a
   NULL pointer when NQ Client reaches the end of enumeration.
   @note
   This is an ASCII version of ccNetworkGetNextItemName()                               */
const NQ_CHAR * ccNetworkGetNextItemNameA(NQ_HANDLE handle);

/**
   Application calls this function after successfully starting a
   browser enumeration (ccNetworkEnumerateDomains(), ccNetworkEnumerateServers()
   or  ccNetworkEnumerateShares()).
   Application may withdraw share names one by one by calling ccNetworkGetNextItemName>()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param handle  Enumeration handle as obtained in one of
                  Browser enumeration calls.
   @return This function returns a pointer to the next item name
   in the enumeration. Item type depends on the handle source -
   one of the browser enumeration calls. This function returns a
   NULL pointer when NQ Client reaches the end of enumeration.                                                                      */
const NQ_WCHAR * ccNetworkGetNextItemName(NQ_HANDLE handle);

/**
   Application calls this function after successfully starting a
   share enumeration (ccNetworkEnumerateShares()).
   
   Application may withdraw CCNetShareItem Items one by one by
   calling ccNetworkGetNextShareItem()
   with the handle obtained in the current call. When finished
   with domain enumeration, application must call ccNetworkCloseHandle()
   to release resources associated with this handle.
   @param handle  Enumeration handle as obtained in one of
                  Browser enumeration calls. 
   @return This function returns a pointer to the next item name
   in the enumeration. Item type depends on the handle source -
   one of the browser enumeration calls. This function returns a
   NULL pointer when NQ Client reaches the end of enumeration.                                                  */
CCNetShareItem * ccNetworkGetNextShareItem(NQ_HANDLE handle);

/**
   This function is called by application to get information
   about a particular share on the specified host.
   @param server        Server name. 
   @param share         Share name. 
   @param type          Share types as:
                        - directory tree
                        - print queue
                        - serial device
                        - IPC 0x80000000
                        - hidden share
   @param remarkBuffer  Buffer to place the share remark into. 
   @param bufferSize    Size of the remark buffer. The
                        recommended value is 256 characters. 
   @return This function returns TRUE if the share information
   is obtained successfully or FALSE otherwise. Application can
   examine the error code for the failure reason.
   @note
   This is an ASCII version of ccNetworkGetShareInfo()*/
NQ_BOOL ccNetworkGetShareInfoA(const NQ_CHAR * server, const NQ_CHAR * share, NQ_UINT16 * type, NQ_CHAR * remarkBuffer, NQ_INT bufferSize);

/**
   This function is called by application to get information
   about a particular share on the specified host.
   Parameters
   @param server         Server name.
   @param share          Share name.
   @param type           Share types as:
                             - directory tree
                             - print queue
                             - serial device
                             - IPC 0x80000000
                             - hidden share
   @param remarkBuffer   Buffer to place the share remark into.
   @param bufferSize     Size of the remark buffer. The recommended
                   value is 256 characters.
   @returns
   This function returns TRUE if the share information is
   obtained successfully or FALSE otherwise. Application can
   examine the error code for the failure reason.                 */
NQ_BOOL ccNetworkGetShareInfo(const NQ_WCHAR * server, const NQ_WCHAR * share, NQ_UINT16 * type, NQ_WCHAR * remarkBuffer, NQ_INT bufferSize);

/**
   This function is called by application to finish a browser
   enumeration and release all resources (e.g., - item names)
   associated with that enumeration.
   @param handle  An abstract handle as obtained from one of the
                  browser enumeration calls.
   @return TRUE on success or FALSE on error.
   @note
   Currently, this function always return TRUE.

   @see
   ccNetworkEnumerateDomains()
   ccNetworkEnumerateServers()
   ccNetworkEnumerateShares()  */
NQ_BOOL ccNetworkCloseHandle(NQ_HANDLE handle);

/**
   This function is called by application to restart a browser
   enumeration designated by an abstract handle. If this handle
   was already used in one or more of the ccNetworkGetNextItemName()
   calls, NQ Client will revert it to the state it has right
   after the respective Browser enumeration call (see below).
   @param handle  An abstract handle as obtained from one of the
                  browser enumeration calls. 
   @return TRUE on success or FALSE on error.
   @note
   Currently, this function always return TRUE.

   @see
   ccNetworkEnumerateDomains()
   ccNetworkEnumerateServers()
   ccNetworkEnumerateShares()                                 */
NQ_BOOL ccNetworkResetHandle(NQ_HANDLE handle);

/** @} end of group cc_network Network Browsing */

#endif /* UD_CC_INCLUDEBROWSING */

/** @ingroup client */
/** @defgroup domain Domain Operations
 * @{
 */

#if defined(UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(DOXYGEN)

/**
   This function joins target computer to the domain.
   
   After successfully joining domain, this function returns a
   computer password obtained from domain controller to be used
   for further logging into domain. This operation requires
   domain administrative credentials.
   @param domain    Name of the domain to join. This may be a
                    fully qualified name. 
   @param computer  Name of the computer (host) that joins. 
   @param admin     Administrative credentials for its domain. 
   @param secret    Buffer for computer password. This buffer
                    should accommodate at least 16 bytes . 
   @return This function returns TRUE if computer succeeded to
   join the domain or FALSE otherwise. The application can
   inspect the error code for the failure reason.
   
   @see
   ccDomainLogon()
   ccDomainLeave()
   AMCredentials structure in amapi.h file.
   @note
   This is an ASCII version of ccDomainJoin()*/
NQ_BOOL ccDomainJoinA(const NQ_CHAR *domain, const NQ_CHAR *computer, const AMCredentialsA *admin, NQ_BYTE secret[16]);

/**
   This function joins target computer to the domain.
   
   After successfully joining domain, this function returns a
   computer password obtained from domain controller to be used
   for further logging into domain. This operation requires
   domain administrative credentials.
   @param domain    Name of the domain to join. This may be a
                    fully qualified name. 
   @param computer  Name of the computer (host) that joins. 
   @param admin     Administrative credentials for its domain. 
   @param secret    Buffer for computer password. This buffer
                    should accommodate at least 16 bytes . 
   @return This function returns TRUE if computer succeeded to
   join the domain or FALSE otherwise. The application can
   inspect the error code for the failure reason.

   @see
   ccDomainLogon()
   ccDomainLeave()
   AMCredentials structure in amapi.h file.                                                                                      */
NQ_BOOL ccDomainJoin(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin, NQ_BYTE secret[16]);

/**
   This function joins target computer to the domain.
   
   After successfully joining domain, this function returns a
   computer password obtained from domain controller to be used
   for further logging into domain. This operation requires
   domain administrative credentials.
   @param domain    Name of the domain to join. This may be a
                    fully qualified name.
   @param computer  Name of the computer (host) that joins.
   @param admin     Administrative credentials for its domain.
   @param dns       A semicolon delimited list of DNS servers,
                    can be NULL.
   @return This function returns TRUE if computer succeeded to
   join the domain or FALSE otherwise. The application can
   inspect the error code for the failure reason.
   
   @see
   ccDomainLogon()
   ccDomainLeave()
   AMCredentials structure in amapi.h file.                                                                                            */
NQ_BOOL ccDomainJoinPersistent(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin, const NQ_WCHAR * dns);

/**
   This function joins target computer to the domain or updates the password (secret)
   for previously joined computer.

   On success this function returns a computer password obtained from domain controller to be used
   for further logging into domain. This operation requires domain administrative credentials.

   @param domain    Name of the domain to join. This may be a fully qualified name.
   @param computer  Name of the computer (host) that joins.
   @param admin Administrative credentials for its domain.
   @param secret    Buffer for computer password. This buffer should accommodate at least 16 bytes.
   @return
   This function returns TRUE if computer succeeded to join the
   domain or when computer account password was updated (in case when account already existed on domain), FALSE otherwise.
   The application can inspect the error code for the failure reason.

   @see
   ccDomainJoin()
   ccDomainLogon()
   ccDomainLeave()
   AMCredentials structure in amapi.h file.
   @note
   This is an ASCII version of ccDomainUpdateSecret()*/
NQ_BOOL ccDomainUpdateSecretA(const NQ_CHAR *domain, const NQ_CHAR *computer, const AMCredentialsA *admin, NQ_BYTE secret[16]);

/**
   This function joins target computer to the domain or updates the password (secret)
   for previously joined computer.

   On success this function returns a computer password obtained from domain controller to be used
   for further logging into domain. This operation requires domain administrative credentials.

   @param domain    Name of the domain to join. This may be a fully qualified name.
   @param computer  Name of the computer (host) that joins.
   @param admin Administrative credentials for its domain.
   @param secret    Buffer for computer password. This buffer should accommodate at least 16 bytes.
   @return
   This function returns TRUE if computer succeeded to join the
   domain or when computer account password was updated (in case when account already existed on domain), FALSE otherwise.
   The application can inspect the error code for the failure reason.

   @see
   ccDomainJoin()
   ccDomainLogon()
   ccDomainLeave()
   AMCredentials structure in amapi.h file.                                 */
NQ_BOOL ccDomainUpdateSecret(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin, NQ_BYTE secret[16]);

/**
   This function cancels domain membership of the target
   computer.
   @param domain    Name of the domain to leave. 
   @param computer  Name of the computer (host) that leaves. 
   @param admin     Administrative credentials for its domain. 
   @return This function returns TRUE if computer successfully
   leaves the domain or FALSE otherwise. The application can
   inspect the error code for the failure reason.

   @see
   ccDomainJoin()
   AMCredentials structure in amapi.h file.

   @note
   This is an ASCII version of ccDomainLeave()*/
NQ_BOOL ccDomainLeaveA(const NQ_CHAR *domain, const NQ_CHAR *computer, const AMCredentialsA *admin);

/**
   This function cancels domain membership of the target
   computer.
   @param domain    Name of the domain to leave. 
   @param computer  Name of the computer (host) that leaves. 
   @param admin     Administrative credentials for its domain. 
   @return This function returns TRUE if computer successfully
   leaves the domain or FALSE otherwise. The application can
   inspect the error code for the failure reason.

   @see
   ccDomainJoin()
   AMCredentials structure in amapi.h file.                                                                                     */
NQ_BOOL ccDomainLeave(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin);

/**
   This function performs domain logon operation supplying a
   password obtained from domain controller by calling ccDomainJoin().
   @param domain    Name of the domain to log on. 
   @param computer  Computer (host) name. 
   @param admin      User credentials.
   @param secret    Password as obtained when joining domain.
   @return This function returns TRUE if the computer has
   successfully logged with the domain controller or FALSE
   otherwise. The application can inspect the error code for the
   failure reason.
   @see AMCredentials structure in amapi.h file.
   @note
   This is an ASCII version of ccDomainLogon()                      */
NQ_BOOL ccDomainLogonA(const NQ_CHAR * domain, const NQ_CHAR * computer, const AMCredentialsA * admin, NQ_BYTE secret[16]);

/**
   This function performs domain logon operation supplying a
   password obtained from domain controller by calling ccDomainJoin().
   @param domain    Name of the domain to log on. 
   @param computer  Computer (host) name. 
   @param admin      User credentials.
   @param secret    Password as obtained when joining domain.
   @return This function returns TRUE if the computer has
   successfully logged with the domain controller or FALSE
   otherwise. The application can inspect the error code for the
   failure reason.
   @see AMCredentials structure in amapi.h file.                                                                                                                                          */
NQ_BOOL ccDomainLogon(const NQ_WCHAR *domain, const NQ_WCHAR * computer, const AMCredentials * admin, NQ_BYTE secret[16]);

#endif /* UD_CC_INCLUDEDOMAINMEMBERSHIP */

/** @} end of group domain Domain Operations */

/** @ingroup client */
/** @defgroup special Special Operations
* @{
*/

/**
  This function is called by application to get the current
  Message Signing mode.

  @returns
  TRUE if Message Signing is enabled and FALSE otherwise                            */
NQ_BOOL ccGetSigningPolicy(void);

/**
  This function is called by application to set the current
  Message Signing mode.

  @param enable  If set to TRUE NQ Client will sign messages.
            if set to FALSE NQ Client will not sign  messages.
  @note
  Default is TRUE.
  It's not recommended to set to FALSE unless the server shows unusual behavior.    */
void ccSetSigningPolicy(NQ_BOOL enable);

/**
   This function is called by application to set the NQ CIFS Client security minimal level. Before this function is called
   NQ CIFS Client uses the default security configuration (see AM_MAXSECURITYLEVEL in NQ Library Reference - Authentication)

   @param level   LAN Manager authentication level. The
                          value of this parameter corresponds to table see amClientDefineLevel in NQ Library Reference - Authentication
   @returns
   NQ_SUCCESS or NQ_FAIL when level requested is invalid.
*/
NQ_STATUS ccSetMinSecurityLevel(NQ_INT level);

/**
   This function is called by application to get the NQ CIFS Client security minimal level.

   @returns
   Current NQ CIFS Client security minimal level.
   Value returned corresponds to table see amClientDefineLevel in NQ Library Reference - Authentication
*/
NQ_INT ccGetMinSecurityLevel(void);

/**
   This function is called by application to set the NQ CIFS Client security maximal level. Before this function is called
   NQ CIFS Client uses the default security configuration (see AM_MAXSECURITYLEVEL in NQ Library Reference - Authentication)

   @param level   LAN Manager authentication level. The
                          value of this parameter corresponds to table see amClientDefineLevel in NQ Library Reference - Authentication
   @returns
   NQ_SUCCESS or NQ_FAIL when level requested is invalid.
*/
NQ_STATUS ccSetMaxSecurityLevel(NQ_INT level);

/**
   This function is called by application to get the NQ CIFS Client security maximal level.

   @returns
   Current NQ CIFS Client security maximal level.
   Value returned corresponds to table see amClientDefineLevel in NQ Library Reference - Authentication
*/
NQ_INT ccGetMaxSecurityLevel(void);

#if defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined(DOXYGEN)

/**
   This function sets or removes exclusive access rights or
   restores default access rights.
   
   This function applies access rights to a file. When
   makeExclusive parameter is TRUE the following access rights
   will be applied:
   
     \* Owner - full access
     \* Administrators (group) - full access
     \* All others - no access
     \* When applied to a folder, access rights will be
       propagated down to subfolders/files, unless they have
       non-inheriting security descriptors
     \* The ownership is changed to the user authenticated.
     \* Access rights of the parent directory will be no more inherited.
   When the second parameter is FALSE file descriptor is
   modified so that all access rights are inherited from the
   parent directory.
   
   When this function returns TRUE it means that the remote
   server has accepted new access rights but it does not
   guarantee that those rights were effectively modified. To
   check the latter use ccIsExclusiveAccessToFile().
   @param fileName       Path to the file. 
   @param makeExclusive  TRUE to set exclusive access, FALSE to
                         revert to default access rights. 
   @return TRUE when operation succeeded, FALSE on error. The
   application can inspect the error code for the failure
   reason.
   @note
   This is an ASCII version of ccSetExclusiveAccessToFile()*/
NQ_BOOL ccSetExclusiveAccessToFileA(NQ_CHAR *fileName, NQ_BOOL makeExclusive);

/**
   This function sets or removes exclusive access rights or
   restores default access rights.
   
   This function applies access rights to a file. When
   makeExclusive parameter is TRUE the following access rights
   will be applied:
   
     \* Owner - full access
     \* Administrators (group) - full access
     \* All others - no access
     \* When applied to a folder, access rights will be
       propagated down to subfolders/files, unless they have
       non-inheriting security descriptors
     \* The ownership is changed to the user authenticated.
     \* Access rights of the parent directory will be no more inherited.
   When the second parameter is FALSE file descriptor is
   modified so that all access rights are inherited from the
   parent directory.
   
   When this function returns TRUE it means that the remote
   server has accepted new access rights but it does not
   guarantee that those rights were effectively modified. To
   check the latter use ccIsExclusiveAccessToFile().
   @param fileName       Path to the file. 
   @param makeExclusive  TRUE to set exclusive access, FALSE to
                         revert to default access rights. 
   @return TRUE when operation succeeded, FALSE on error. The
   application can inspect the error code for the failure
   reason.                                                                                        */
NQ_BOOL ccSetExclusiveAccessToFile(NQ_WCHAR *fileName, NQ_BOOL makeExclusive);

/**
   This function is called by application to determine whether
   the file/directory specified by the parameter has exclusive
   access rights.
   @param fileName  Path to the file.
   @return TRUE if the file specified by the parameter has
   exclusive access rights or FALSE otherwise.

   @see

   ccSetExclusiveAccessToFile()

   @note
   This is an ASCII version of ccIsExclusiveAccessToFile()*/
NQ_BOOL ccIsExclusiveAccessToFileA(NQ_CHAR *fileName);

/**
   This function is called by application to determine whether
   the file/directory specified by the parameter has exclusive
   access rights.
   @param fileName  Path to the file.
   @return TRUE if the file specified by the parameter has
   exclusive access rights or FALSE otherwise.

   @see
   ccSetExclusiveAccessToFile() */
NQ_BOOL ccIsExclusiveAccessToFile(NQ_WCHAR *fileName);

#endif /* UD_CC_INCLUDESECURITYDESCRIPTORS */

/** @} end of group special Special Operations */


/** @ingroup client */
/** @defgroup dialects Dialects Operations
 * @{
 */

/**
   Activates or deactivates selected SMB dialect.
   
   This function affects new connections and it does not affect
   the existing connections.
   @param dialect    SMB dialect value, see #CCSmbDialect
   @param setActive  <i>TRUE to enable this dialect or FALSE</i>
                     to disable it. 
   @return <i>TRUE or FALSE</i> .                                */
NQ_BOOL ccSetSmbDialect(CCSmbDialect dialect, NQ_BOOL setActive);

/**
    Determine whether SMB1 dialect is supported.

    @returns
     <i>TRUE</i> or  <i>FALSE</i>
 */
NQ_BOOL ccCifsGetSMB1Support(void);

#if defined(UD_NQ_INCLUDESMB1) || defined(DOXYGEN)
/**
    Activates or deactivates of SMB1 dialect.

    @param   isSupport  <i>TRUE</i> to enable SMB1 dialect, <i>FALSE</i> to disable SMB1 dialect.
 */
void
ccCifsSetSMB1Support(
        NQ_BOOL isSupport
        );
#endif /* UD_NQ_INCLUDESMB1 */

/** @} end of group dialects Dialects Operations */

/** @ingroup client */
/** @defgroup user_cred User Credentials
 * @{
 */
/**
   This function clears user credentials associated with the
   given mount point.

   After this call, NQ will query application in any of the
   following cases:
     \* Reconnecting one of the shares, associated with this
       mount point as a result of a temporary server disconnect.
     \* Establishing a connection with additional servers as a
       result of DFS redirection.

   This call does not affect those shares that were connected
   before this call.
   @param mountpoint  The name of the mount point to clear
                      credentials.
   @return TRUE on success, FALSE on failure.
   @note
   This is an ASCII version of ccResetCredentails() */
NQ_BOOL ccResetCredentailsA(const NQ_CHAR * mountpoint);

/**
   This function clears user credentials associated with the
   given mount point.

   After this call, NQ will query application in any of the
   following cases:
     \* Reconnecting one of the shares, associated with this
       mount point as a result of a temporary server disconnect.
     \* Establishing a connection with additional servers as a
       result of DFS redirection.
   This call does not affect those shares that were connected
   before this call.

   @param mountpoint  The name of the mount point to clear credentials.
   @returns
   TRUE on success, FALSE on failure.                                                               */
NQ_BOOL ccResetCredentails(const NQ_WCHAR * mountpoint);

/**
   This function is called by application to verify that the
   user credentials supplied by udGetCredentials() are
   sufficient on the specified server. Those credentials can
   be set by udSetCredentials() before calling this function.
   @param server  Server name the credentials should be verified
                  against. If this parameter is NULL then NQ
                  Client will try to automatically discover the
                  Domain Controller and verify the credentials
                  against it.
   @return
   This function returns TRUE if the user is
   successfully authenticated or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.
   @note
   This is an ASCII version of ccValidateUser() */

NQ_BOOL ccValidateUserA(NQ_CHAR *server);

/**
   This function is called by application to verify that the
   user credentials supplied by udGetCredentials() are
   sufficient on the specified server. Those credentials can
   be set by udSetCredentials() before calling this function.
   @param server  Server name the credentials should be verified
                  against. If this parameter is NULL then NQ Client
                  will try to automatically discover the Domain
                  Controller and verify the credentials against it.
   @return
   This function returns TRUE if the user is successfully
   authenticated or FALSE otherwise. The application can inspect
   the error code for the failure reason */

NQ_BOOL ccValidateUser(NQ_WCHAR *server);

/**
   This function is called by application to verify that the supplied
   user credentials sufficient on the specified server.
   @param server  Server name the credentials should be verified
                  against. If this parameter is NULL then NQ
                  Client will try to automatically discover the
                  Domain Controller and verify the credentials
                  against it.
   @param userCredentials  Pointer to a structure, which stores the
                  user credentials. See AMCredentialsA structure.
                  If NULL is supplied the credentials will be queried by udGetCredentials().
                  To initialize that structure use amCredentialsInitA().
                  It is the caller responsibility to release that structure.
   @return
   This function returns TRUE if the user is
   successfully authenticated or FALSE otherwise. The
   application can inspect the error code for the failure
   reason.
   @note
   This is an ASCII version of ccValidateUserByCredentials() */

NQ_BOOL ccValidateUserByCredentialsA(NQ_CHAR * server, const AMCredentialsA * userCredentials);

/**
   This function is called by application to verify that the supplied
   user credentials sufficient on the specified server.
   @param server  Server name the credentials should be verified
                  against. If this parameter is NULL then NQ Client
                  will try to automatically discover the Domain
                  Controller and verify the credentials against it.
   @param userCredentials  Pointer to a structure, which stores the
                  user credentials. See AMCredentials structure.
                  If NULL is supplied the credentials will be queried by udGetCredentials().
                  To initialize that structure use amCredentialsInitW().
                  It is the caller responsibility to release that structure.
   @return
   This function returns TRUE if the user is successfully
   authenticated or FALSE otherwise. The application can inspect
   the error code for the failure reason */

NQ_BOOL ccValidateUserByCredentials(NQ_WCHAR * server, const AMCredentials * userCredentials);

/** @} end of group user_cred User Credentials */

/** @ingroup client */
/** @defgroup error Errors
 * @{
 */

/**
   Returns last SMB NT error status for application thread.

   Status returned is for last erroneous SMB command response from server.
   @note That last erroneous NT status code may not reflect the real problem.
   Use syGetLastError() for proper NQ error code.
   @returns
   Error code.                                      */
NQ_UINT32 ccGetNTStatus(void);

/**
   This function is called by the application when some
    returned NQ error value should be translated to a SMB
    error.

    @param nqError The error code returned from some nq function.

    @returns
    SMB error code.                                    */
NQ_UINT32 ccErrorToSmbStatus(NQ_STATUS nqError);

/** @} end of group error Errors */

/** @ingroup client */
/** @defgroup aux Auxiliary methods
 * @{
 */

/**
   This function must be called only by the driver.
   When the FUSE driver runs in regular mode (not debug) it creates a
   hard fork and closes parent process. This causes the receiving thread
   to be killed.
   This function is used to restart the receiving thread.

   @returns
   TRUE on success or FALSE on error.

   @note
   Currently, this function always return TRUE. */
NQ_BOOL ccTransportRestartRecieveThread();

/**
  This function is called by application to print all the CIFS
  Client tables. It is used for debugging purposes only.
                                                       */
void ccDump(void);

/**
  This function is called by application to close all open
  connections and to release resources associated with them. In
  this function call NQ will do the following:
    \* Close all files;
    \* Release outstanding searches;
    \* Disconnect from shares (trees);
    \* Logoff users;
    \* Disconnect from servers.
                                                       */
void ccCloseAllConnections(void);

/**
  This function is called by application to those connections
  that were used implicitly and are not associated with any
  mount. For instance, upon this call NQ will close those
  connections that were open solely for network browsing. NQ
  will release all resources associated with the connections
  being closed, namely:
    \* Close open files;
    \* Release outstanding searches;
    \* Disconnect from shares (trees);
    \* Logoff users;
    \* Disconnect from servers.
                                                    */
void ccCloseHiddenConnections(void);

/**
   Register the current system thread.

   This function creates an NQ object describing the current system (application) thread.
   NQ expects the application to call ccThreadUnsubscribe() to release NQ resources associated with
   this thread. If this thread was already registered, nothing will happen.
                                    */
void ccThreadSubscribe(void);

/**
   Release the resources associated with the current system thread.

   NQ assumes that the current system (application) thread was registered with a
   ccThreadSubscribe() call. If this is not true, nothing will happen.
                                    */
void ccThreadUnsubscribe(void);

/** @} end of group aux Auxiliary methods */

#endif /* _CCAPI_H_ */
