/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Authentication API
 *--------------------------------------------------------------------
 * MODULE        : NQ
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _AMAPI_H_ 
#define _AMAPI_H_

#include "cmapi.h"

/* -- Constants -- */

/** @defgroup amapi Authentication
 * @{ */

/* security levels: */
#define AM_MAXSECURITYLEVEL 4   /**< Maximum security level. Security levels are counted from zero to this value. Used by amClientDefineLevel() */

/** @} end of group amapi Authentication */

/** @ingroup amapi */
/**@defgroup enc_alg Encryption Algorithms
 * Available encryption algorithms
 * @{
 */

/* available encryption algorithms: */

#define AM_CRYPTER_NONE     0   /**< Encryption is skipped. */

#define AM_CRYPTER_LM       1   /**< Fixed-length 128-bit LM response. */

#define AM_CRYPTER_NTLM     2   /**< Fixed-length 192-bit NTLM response. */

#define AM_CRYPTER_LM2      3   /**< Fixed-length 128-bit LMv2 response. */

#define AM_CRYPTER_NTLM2    4   /**< Variable-length NTLMv2 response. */

/** @} end of group enc_alg Encryption Algorithms */

/** @ingroup amapi */
/**@defgroup sec_mech Security Mechanisms
 * @{
 */

/* security mechanisms: */
#define AM_MECH_NTLMSSP     1   /**< NTLMSSP security mechanism. This value is used in a security mechanism mask (see amClientDefineLevel()). */
#define AM_MECH_KERBEROS    2   /**< Kerberos security mechanism. This value is used in a security mechanism mask (see amClientDefineLevel()). */

/** @} end of group sec_mech Security Mechanisms */

/* -- Structures --*/

/** @addtogroup amapi
 * @{ */

/** This structure is used for domain description. Two fields of this
  structure compose together a fully qualified domain name. */
typedef struct {
    NQ_CHAR name[CM_BUFFERLENGTH(NQ_CHAR, 256)];     /**< Domain name */
    NQ_CHAR realm[CM_BUFFERLENGTH(NQ_CHAR, 256)];    /**< Domain realm name */
}
AMDomainA;

typedef struct {
    NQ_WCHAR name[CM_BUFFERLENGTH(NQ_WCHAR, 256)];   /**< Domain name */
    NQ_WCHAR realm[CM_BUFFERLENGTH(NQ_WCHAR, 256)];  /**< Domain realm name */
}
AMDomain;

/** This structure carries user credentials as used for
   authentication against server in the following cases:
     \* SMB/SMB2 authentication
     \* Domain join
     \* Domain logon                                     */
/* Account credentials */
typedef struct {
    AMDomainA domain;                                   /**< Domain name */
    NQ_CHAR user[CM_BUFFERLENGTH(NQ_CHAR, 257)];        /**< User name */
    NQ_CHAR password[CM_BUFFERLENGTH(NQ_CHAR, 257)];    /**< Password buffer */
    NQ_UINT type;                                       /**< Password type, see @ref pass_type*/
}
AMCredentialsA;

/** Account credentials */
typedef struct {
    AMDomain domain;                                    /**< Domain name */
    NQ_WCHAR user[CM_BUFFERLENGTH(NQ_WCHAR, 257)];      /**< User name */
    NQ_WCHAR password[CM_BUFFERLENGTH(NQ_WCHAR, 257)];  /**< Password buffer */
    NQ_UINT type;                                       /**< Password type, see @ref pass_type  */
}
AMCredentials;

/** @} end of group amapi Authentication */

/** @ingroup amapi */
/**@defgroup pass_type Security Password Type
 * @{
 */

#define AM_CREDENTIALS_PASSWORD_PLAINTEXT   0
#define AM_CREDENTIALS_PASSWORD_MD4         1

/** @} end of group pass_type Security Password Type */

/* -- Functions -- */

/** @addtogroup amapi
 * @{ */

/**This function initializes this module.
   @returns
   TRUE on success and FALSE on failure.  */
NQ_BOOL amStart(void);

/** This function disposes resources used by this module.
                                               */
void amShutdown(void);

#if defined(UD_NQ_INCLUDECIFSCLIENT) || defined(DOXYGEN)

/** NQ Authentication module conveys authentication according to
   the required level of security.
   
   This function assigns parameters for one authentication
   level:
     \* NQ combines two encryption algorithms in one
       authentication blob. User can choose between LM, NTLM and
       NTLMv2 encryptions.
     \* User can choose security mechanisms for extended security
       negotiations. Currently, the available mechanisms are:
       NTLMSSP and Kerberos.

   @param level Authentication level to define. This value should
                be greater or equal to zero and it should not
                exceed the maximum security level as defined in #AM_MAXSECURITYLEVEL.
                An illegal value will have no effect.
   @param crypter1    The first encryption algorithm. Available values
                are:
                \* #AM_CRYPTER_LM
                \* #AM_CRYPTER_NTLM
                \* #AM_CRYPTER_LM2
                \* #AM_CRYPTER_NTLM2
                \* #AM_CRYPTER_NONE
   @param crypter2 The second encryption algorithm. Available values
                are:
                \* #AM_CRYPTER_LM
                \* #AM_CRYPTER_NTLM
                \* #AM_CRYPTER_LM2
                \* #AM_CRYPTER_NTLM2
                \* #AM_CRYPTER_NONE
   @param mehanisms  Available security mechanisms. This value is a
                bit mask of the following:
                \* #AM_MECH_KERBEROS
                \* #AM_MECH_NTLMSSP
                User can specify one of those masks, both or
                none. When two masks are specified, Kerberos is
                considered first.
   @note
     \* Level parameters are assigned globally so that two
       concurrent applications using the same level are sharing the
       same parameters. If one of them assigns new parameters this
       also affects the second one.
     \* This function is not thread-safe.
     \* The default per-level parameters are:

   Level    |   Crypter 1|  Crypter 2 | NTLMSSP  |   KERBEROS
   -------  |----------  |------------|----------|-----------
   0        |LM          |none        |-         |-
   1        |LM          |NTLM        |-         |-
   2        |LM          |NTLM        |Yes       |Yes
   3        |LMv2        |NTLMv2      |Yes       |-
   4        |LMv2        |NTLMv2      |Yes       |Yes
                                                                                 */
void amClientDefineLevel(NQ_UINT level, NQ_UINT crypter1, NQ_UINT crypter2, NQ_UINT32 mehanisms);

/** Some old authentication methods - used in SMB 1 only - are
   considered today non secure. By default those non secure
   methods will be disabled. Use this function to enable /
   disable non secure methods.

   @param enableNonSecureAuthentication  TRUE - non secure
                                    authentication will be used.
                                    FALSE - non secure authentication
                                    will not be used.
   @note
     \* This function uses amClientDefineLevel()
       to set authentication algorithms. The amClientDefineLevel()
       call can be used to further modify the levels set by this
       function.
     \* See also notes in amClientDefineLevel().
                                                                                                                               */
void amSetNonSecureAuthentication(NQ_BOOL enableNonSecureAuthentication);

#endif /* UD_NQ_INCLUDECIFSCLIENT */

/**@} end of group amapi Authentication */

#endif /* _AMAPI_H_ */
