/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client transport operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "cctransport.h"
#include "ccapi.h"
#include "cmbufman.h"
#include "cmfinddc.h"
#include "cmcommon.h"
#include "cmlist.h"
#include "ccserver.h"
#include "nssocket.h"
#ifdef SY_EPOLL_AVAILABLE 
#include "syepoll.h"
#endif /* SY_EPOLL_AVAILABLE */

/* -- Constants -- */
#define TRANSPORT_IDLETIMEOUT             (15*60) /* 15 min Timeout for transport , this mimics windows disconnection after 15 min*/
#define DODISCONNECT_TIMEOUT              30
#define RECEIVETHREAD_SHUTDOWN_TIMEOUT    2 /* 2 second timeout to allow receive thread run to completion*/


/*#define SIMULATE_DISCONNECT*/ /* simulate transport disconnect - debug purposes only */
#ifdef SIMULATE_DISCONNECT
#define SIMULATE_DISCONNECT_AFTER 30
#endif /* SIMULATE_DISCONNECT */

/* -- Static data -- */

static NQ_BOOL isModuleInitialized = FALSE;     /* was init done  */
static NQ_BOOL doReceive;               /* when TRUE - receive responses */ 
static CMList connections;              /* list of sockets to listen */
static SYThread receiveThread;          /* receiving thread */
static SYSocketHandle notifyingSocket = syInvalidSocket();  /* this socket is used for sending messages to receive thread */
static SYSocketHandle connectionsListHasChangedSocket = syInvalidSocket();   /* dummy message is sent to this socket to indicate
                                            that the connections list has changed */
static SYSocketHandle terminationSocket = syInvalidSocket();  /* dummy message is sent to this socket to cause
                                            the receive thread to terminate */
static NQ_PORT terminationPort;           /* port to use for termination */
static NQ_PORT connectionsListHasChangedPort; /* port is used to indicate that the connections list has changed */
static NQ_PORT notifyingPort;           /* notifying socket is bound to this port */
#ifdef SY_EPOLL_AVAILABLE
static SYEpollHandle epoll;             /* for epoll */
#endif /* SY_EPOLL_AVAILABLE */

static const NQ_UINT defaultTransports[] = {
#ifdef UD_NQ_USETRANSPORTIPV4
      NS_TRANSPORT_IPV4,
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTNETBIOS
      NS_TRANSPORT_NETBIOS,
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_NQ_USETRANSPORTIPV6
      NS_TRANSPORT_IPV6,
#endif /* UD_NQ_USETRANSPORTIPV6 */
      0 } ;

static  CMThreadCond    receiveThreadShutDownCond;

/* -- Static functions -- */

/* signal that the list of connections has changed */
static void notifyListChange()
{
    static NQ_BYTE dummyMsg[] = {0};   /* a voluntary message to send over the notify socket */

    if (sySendToSocket(notifyingSocket, dummyMsg, sizeof(dummyMsg), cmSelfipGetLocalHostIp(), syHton16(connectionsListHasChangedPort)) < 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to return error response");
    }
}

/*
 * Receive thread body
 */
static void receiveThreadBody(void)
{
#ifdef SIMULATE_DISCONNECT
    static NQ_INT cmdCount = 0; /* count commands and disconnect after SIMULATE_DISCONNECT_AFTER of them */
#endif /* SIMULATE_DISCONNECT */

    while (doReceive)
    {
        CMIterator iterator;                /* to go through the list */
        NQ_STATUS res = 0;                  /* operation status */
#ifdef SY_EPOLL_AVAILABLE
        void * ctxTransport = NULL;             /* to pass to thread */
#else
        NSSocketSet readList;               /* socket set for select */
#endif /* SY_EPOLL_AVAILABLE */
        
        /* Prepare socket descriptor */
#ifdef SY_EPOLL_AVAILABLE
#else
        nsClearSocketSet(&readList);
#endif /* SY_EPOLL_AVAILABLE */
        cmListIteratorStart(&connections, &iterator);
        while (cmListIteratorHasNext(&iterator))
        {
            CCTransport * pTransport;   /* casted pointer */

            pTransport = (CCTransport *)cmListIteratorNext(&iterator);
            if ((!nsIsSocketAlive(pTransport->socket) || pTransport->doDisconnect) && !pTransport->isSettingUp)
            {
                pTransport->connected = FALSE;
                cmListItemRemove((CMItem *)pTransport);
#ifdef SY_EPOLL_AVAILABLE
                syEpollRemoveSocket(epoll , SYSOCK(pTransport->socket));
#endif /* SY_EPOLL_AVAILABLE */
                syMutexTake(&pTransport->guard);
                syMutexGive(&pTransport->guard);
                syMutexDelete(&pTransport->guard);
                syMutexDelete(pTransport->item.guard);
                cmMemoryFree(pTransport->item.guard);
                pTransport->item.guard = NULL;
                if (NULL != pTransport->cleanupCallback)
                {
                    (*pTransport->cleanupCallback)(pTransport->cleanupContext);
                }
                nsClose(pTransport->socket);
                if (TRUE == pTransport->isWaitingDisconectCond)
                {
                    cmThreadCondSignal(&pTransport->disconnectCond);
                }
                continue;
            }
            syMutexTake(&pTransport->guard);
            if (!pTransport->isReceiving && !pTransport->isSettingUp)
            {
                res++;
#ifndef SY_EPOLL_AVAILABLE
                nsAddSocketToSet(&readList, pTransport->socket);
#endif /* SY_EPOLL_AVAILABLE */
            }
            syMutexGive(&pTransport->guard);
        }
        cmListIteratorTerminate(&iterator);

        res = 0;
#ifdef SY_EPOLL_AVAILABLE
        res = syEpollWait(epoll, &ctxTransport, 1);
#else
        syAddSocketToSet(connectionsListHasChangedSocket, &readList);
        syAddSocketToSet(terminationSocket, &readList);
        res = nsSelect(&readList, 1);   /* each second */
#endif /* SY_EPOLL_AVAILABLE */
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Select returned %d", res);

        if (res == 0)
        {
            LOGERR(CM_TRC_LEVEL_LOW_ERROR, "Select() timeout");
            continue;
        }
        if (res == NQ_FAIL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Select() error");
            continue;
        }


#ifdef SIMULATE_DISCONNECT
        cmdCount++;
#endif /* SIMULATE_DISCONNECT */

#ifdef SY_EPOLL_AVAILABLE
        if (ctxTransport == &terminationSocket)
#else
        if (syIsSocketSet(terminationSocket, &readList))
#endif /* SY_EPOLL_AVAILABLE */
        {
            NQ_BYTE buf[2];         /* buffer for dummy */
            NQ_IPADDRESS ip;        /* dummy ip */
            NQ_PORT port;           /* dummy port */

            res = syRecvFromSocket(terminationSocket, buf, sizeof(buf), &ip, &port);
            if ((0 == res) || (NQ_FAIL == res))
            {
               LOGERR(CM_TRC_LEVEL_ERROR, "Receive error");
            }

#ifdef SY_EPOLL_AVAILABLE
            syEpollRearmSocket(epoll, terminationSocket, &terminationSocket);
#endif /* SY_EPOLL_AVAILABLE */
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "receiveThread received termination signal");
            break;
        }

        /* check notify */
#ifdef SY_EPOLL_AVAILABLE
        if (ctxTransport == &connectionsListHasChangedSocket)
#else
        if (syIsSocketSet(connectionsListHasChangedSocket, &readList))
#endif /* SY_EPOLL_AVAILABLE */
        {
            NQ_BYTE buf[2];         /* buffer for dummy */
            NQ_IPADDRESS ip;        /* dummy ip */
            NQ_PORT port;           /* dummy port */

            res = syRecvFromSocket(connectionsListHasChangedSocket, buf, sizeof(buf), &ip, &port);
            if ((0 == res) || (NQ_FAIL == res))
            {
               LOGERR(CM_TRC_LEVEL_ERROR, "Receive error");
            }
#ifdef SY_EPOLL_AVAILABLE
            syEpollRearmSocket(epoll, connectionsListHasChangedSocket, &connectionsListHasChangedSocket);
#endif /* SY_EPOLL_AVAILABLE */
            continue;
        }

#ifndef SY_EPOLL_AVAILABLE
        cmListIteratorStart(&connections, &iterator);
        while (cmListIteratorHasNext(&iterator))
#endif /* SY_EPOLL_AVAILABLE */
        {
            CCTransport * pTransport = NULL;    /* casted pointer */

#ifdef SY_EPOLL_AVAILABLE
            pTransport =  (CCTransport *)ctxTransport;
#else
            pTransport = (CCTransport *)cmListIteratorNext(&iterator);
#endif /* SY_EPOLL_AVAILABLE */
#ifdef SIMULATE_DISCONNECT
            if (cmdCount >= SIMULATE_DISCONNECT_AFTER)
            {
                nsClose(pTransport->socket);
            }
            else
#endif /* SIMULATE_DISCONNECT */
            {
                if (NULL != pTransport->socket && nsIsSocketAlive(pTransport->socket)
#ifndef SY_EPOLL_AVAILABLE
                        && nsSocketInSet(&readList, pTransport->socket)
#endif /* SY_EPOLL_AVAILABLE */
                )
                {
                    NQ_INT packetLength;         /* number of bytes or error */

                    ccTransportLock(pTransport);

                    packetLength = ccTransportReceivePacketLength(pTransport);
                    if (packetLength > 0)
                    {
                        pTransport->callback(pTransport);
#ifdef SY_EPOLL_AVAILABLE
                        syEpollRearmSocket(epoll, SYSOCK(pTransport->socket), pTransport);
#endif /* SY_EPOLL_AVAILABLE */
                        ccTransportUnlock(pTransport);
                    }
                    else
                    {
                        CCServer *pServer = (CCServer *)pTransport->server;

                        ccTransportReceiveEnd(pTransport);

                        if (packetLength < 0)
                        {
                            if (pServer->isReconnecting || pServer->item.beingDisposed || !pServer->item.findable)
                            {
                                pTransport->doDisconnect = TRUE;
                                ccTransportUnlock(pTransport);
                                continue;
                            }
                            else
                            {
                                pTransport->doDisconnect = TRUE;
                                pTransport->connected = FALSE;
                                ccTransportUnlock(pTransport);

                                cmListItemTake((CMItem *)pServer);
                                pServer->smb->signalAllMatch(pTransport);
                                pServer->connectionBroke = TRUE;
                                cmListItemGive((CMItem *)pServer);
                            }
                        }
                        else /* packetLength = 0 */
                        {
                            ccTransportUnlock(pTransport);
                        }
                    }
                }
            }
        } /* while (cmListIteratorHasNext(&iterator)) */
#ifndef SY_EPOLL_AVAILABLE
        cmListIteratorTerminate(&iterator);
#endif /* SY_EPOLL_AVAILABLE */
#ifdef SIMULATE_DISCONNECT
        if (cmdCount >= SIMULATE_DISCONNECT_AFTER)
        {
            cmdCount = 0;
        }
#endif /* SIMULATE_DISCONNECT */
    } /* while (doReceive) */

    cmThreadCondSignal(&receiveThreadShutDownCond);
}

/*
 * An attempt to connect server 
 */
static NSSocketHandle connectOneTransportByOneIp(NQ_INT transportType, const NQ_IPADDRESS * ip, CMNetBiosNameInfo * nbInfo)
{
    NQ_STATUS res;              /* operation result */
    NSSocketHandle socket;      /* socket handle */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transportType:%d ip:%p nbInfo:%p", transportType, ip, nbInfo);
    /*LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Transport: %d", transportType);*/

    /* Create TCP socket */
    socket = nsSocket(NS_SOCKET_STREAM, (NQ_UINT)transportType);
    if (socket == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "nsSocket() failed");
        goto Exit;
    }

    /* Connect to remote Server */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "before nsConnect: %s", nbInfo->name);
    res = nsConnect(socket, (NQ_IPADDRESS *)ip, nbInfo);
    if (res == NQ_FAIL)
    {
        nsClose(socket);
        socket = NULL;
        LOGERR(CM_TRC_LEVEL_ERROR, "nsConnect() failed");
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Connected to %s, %s", cmIPDump(ip), nbInfo->name);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", socket);
    return socket;
}


/* -- API Functions */

NQ_UINT ccGetNumOfAvailableTransports(void)
{
    return (NQ_UINT)(sizeof(defaultTransports) / sizeof(NQ_UINT) - 1);
}

/*
 *====================================================================
 * PURPOSE: returns the list of transports by their priorities
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: the pointer to array of transports following by zero
 *
 * NOTES:
 *====================================================================
 */

void ccGetTransportPriorities(NQ_UINT * pBuf)
{
    const NQ_UINT *t;
    NQ_INT i, p;

    if (pBuf != NULL)
    {
        for (p = 3, i = 0; p > 0; p--)
        {
            /* lets try all transports */
            for (t = defaultTransports; *t; t++)
            {
                if (udGetTransportPriority(*t) == p)
                    pBuf[i++] = *t;
            }
        }
        pBuf[i] = 0;
    }
}


NQ_BOOL ccTransportStart(void)
{
    NQ_BOOL result = FALSE;
    NQ_INT error = NQ_ERR_OK;

   LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

   if (isModuleInitialized)
   {
      result = TRUE;
      goto Exit;
   }

    result = cmThreadCondSet(&receiveThreadShutDownCond);
    if (FALSE == result)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Did not set thread cond, possibly that 'SY_SEMAPHORE_AVAILABLE' is off ");
    }

    doReceive = TRUE;
    cmListStart(&connections);

    connectionsListHasChangedSocket = syCreateSocket(FALSE, CM_IPADDR_IPV4);
    notifyingSocket = syCreateSocket(FALSE, CM_IPADDR_IPV4);
    terminationSocket = syCreateSocket(FALSE, CM_IPADDR_IPV4);

    if (!syIsValidSocket(connectionsListHasChangedSocket) || !syIsValidSocket(notifyingSocket) || !syIsValidSocket(terminationSocket))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "syCreateSocket() failed");
        error = NQ_ERR_SOCKETCREATE;
        goto Error;
    }


   notifyingPort = cmDynamicPortBindSocket(notifyingSocket, FALSE);
   connectionsListHasChangedPort = cmDynamicPortBindSocket(connectionsListHasChangedSocket, TRUE);
   terminationPort = cmDynamicPortBindSocket(terminationSocket, FALSE);

   if ((0 == connectionsListHasChangedPort) || (0 == notifyingPort) || (0 == terminationPort))
   {
        LOGERR(CM_TRC_LEVEL_ERROR, "cmDynamicPortBindSocket() failed");
        error = NQ_ERR_SOCKETBIND;
        goto Error;
   }

#ifdef SY_EPOLL_AVAILABLE
    epoll = syEpollCreate(23);
    syEpollAddSocket(epoll, connectionsListHasChangedSocket , &connectionsListHasChangedSocket);
    syEpollAddSocket(epoll, terminationSocket , &terminationSocket);
#endif /* SY_EPOLL_AVAILABLE */

    syThreadStart(FALSE, NQ_THREAD_PRIORITY_NONE, &receiveThread, receiveThreadBody, TRUE);
    result = TRUE;
    goto Exit;

Error:
    if (syIsValidSocket(connectionsListHasChangedSocket))
    {
        syCloseSocket(connectionsListHasChangedSocket);
    }

    if (syIsValidSocket(notifyingSocket))
    {
        syCloseSocket(notifyingSocket);
    }

    if (syIsValidSocket(terminationSocket))
    {
        syCloseSocket(terminationSocket);
    }

    sySetLastError(error);

Exit:
    if (result)
    {
        isModuleInitialized = TRUE;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL ccTransportRestartRecieveThread(void)
{
    syThreadStart(FALSE, NQ_THREAD_PRIORITY_NONE, &receiveThread, receiveThreadBody, TRUE);

    return TRUE;
}
void ccTransportShutdown(void)
{
    CMIterator  itr;
    NQ_BYTE     dummyMsg[] = {0};   /* a voluntary message to send over the terminate socket */
    NQ_INT      res = NQ_SUCCESS;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (FALSE == isModuleInitialized)
    {
        goto Exit;
    }

    doReceive = FALSE;
    res = sySendToSocket(notifyingSocket, dummyMsg, sizeof(dummyMsg), cmSelfipGetLocalHostIp(), syHton16(terminationPort));
    if (NQ_FAIL == res)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Failed to send socket: %d", notifyingSocket);
    }

    /* make sure receiveThread is finished before killing it. */
    if (!cmThreadCondWait(&receiveThreadShutDownCond, RECEIVETHREAD_SHUTDOWN_TIMEOUT))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "time out occur while waiting for receive thread to finish");
    }

    /* connections list shut down - can't use regular shutdown because this item isn't allocated in cmmemory. it is part of a server item. */
    cmListIteratorStart(&connections, &itr);
    while (cmListIteratorHasNext(&itr))
    {
        CCTransport *pTransport;
        pTransport = (CCTransport *)cmListIteratorNext(&itr);
        cmListItemRemove((CMItem *)pTransport);

        LOGERR(CM_TRC_LEVEL_ERROR, "Bad shutdown. ccServer item: %p wan't released after usage.", pTransport->server);

        /* transport is part of the server item. if any transport in the list we should remove its corresponding server item */
        cmListItemRemoveAndDispose((CMItem *) pTransport->server);
    }
    cmListIteratorTerminate(&itr);

    cmListShutdown(&connections);

    cmThreadCondRelease(&receiveThreadShutDownCond);
    syThreadDestroy(receiveThread);
#ifdef SY_EPOLL_AVAILABLE
    syEpollDelete(epoll);
#endif /* SY_EPOLL_AVAILABLE */
    if (syIsValidSocket(connectionsListHasChangedSocket))
    {
        syCloseSocket(connectionsListHasChangedSocket);
    }

    if (syIsValidSocket(notifyingSocket))
    {
        syCloseSocket(notifyingSocket);
    }

    if (syIsValidSocket(terminationSocket))
    {
        syCloseSocket(terminationSocket);
    }

    cmDynamicPortRelease(terminationPort);
    cmDynamicPortRelease(connectionsListHasChangedPort);
    cmDynamicPortRelease(notifyingPort);

Exit:
    isModuleInitialized = FALSE;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void ccTransportInit(CCTransport * transport)
{
    transport->connected = FALSE;
    transport->callback  = NULL;
    cmListItemInit(&transport->item);
}

NQ_BOOL ccTransportConnect(
    CCTransport * pTransport, 
    const NQ_IPADDRESS * ips, 
    NQ_INT numIps, 
    const NQ_WCHAR * host,
    CCTransportCleanupCallback cleanupCallback,
    void * cleanupContext
#ifdef UD_NQ_USETRANSPORTNETBIOS
    ,NQ_BOOL forceNBSocket
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_NQ_INCLUDESMBCAPTURE
    ,CMCaptureHeader    *   captureHdr
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    )
{
    NSSocketHandle socket;      /* resulting handle */
    CMNetBiosNameInfo nbInfo;   /* NetBIOS name information */
    NQ_CHAR * aHost = NULL;     /* host name in ASCII */
    NQ_UINT * transportTypes = NULL;   /* transport types ordered by priorities */
    NQ_UINT * transportList = NULL;   /* transport types ordered by priorities */
    NQ_BOOL result = FALSE;
#ifdef UD_NQ_USETRANSPORTNETBIOS
    NQ_BOOL hasNBTransport = FALSE;
#endif /* UD_NQ_USETRANSPORTNETBIOS */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transport:%p ips:%p numIps:%d host:%s callback:%p context:%p", pTransport, ips, numIps, cmWDump(host), cleanupCallback, cleanupContext);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "host: %s", cmWDump((const NQ_WCHAR *) host));

    pTransport->item.guard = (SYMutex *)cmMemoryAllocate(sizeof(*pTransport->item.guard));
    if (NULL == pTransport->item.guard)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    syMutexCreate(pTransport->item.guard);
    pTransport->connected = FALSE;
    pTransport->doDisconnect = FALSE;
    pTransport->isWaitingDisconectCond = FALSE;
    pTransport->cleanupCallback = cleanupCallback;
    pTransport->cleanupContext = cleanupContext;
    /* compose NetBIOS name */
    aHost = cmMemoryCloneWStringAsAscii(host);
    if (NULL == aHost)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    cmNetBiosNameCreate(nbInfo.name, aHost, CM_NB_POSTFIX_SERVER);
    nbInfo.isGroup = FALSE;

    transportList = (NQ_UINT *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_UINT) * (ccGetNumOfAvailableTransports() + 1)));
    if (NULL == transportList)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    transportTypes = transportList;

#ifdef UD_NQ_USETRANSPORTNETBIOS
    if (forceNBSocket)
    {
        NQ_UINT i = 0;

        ccGetTransportPriorities(transportList);
        for (i = 0; i < ccGetNumOfAvailableTransports(); i++)
        {

            if (*transportList == NS_TRANSPORT_NETBIOS)
            {
                hasNBTransport = TRUE;
                break;
            }
            transportList++;
        }
        transportList = transportTypes;
    }
#endif /* UD_NQ_USETRANSPORTNETBIOS */

    for (ccGetTransportPriorities(transportList); *transportTypes != 0; transportTypes++)
    {
        NQ_INT i;   /* just a counter */
        for (i = 0; i < numIps; i++)
        {   
#ifdef UD_NQ_USETRANSPORTIPV6                    
            NQ_UINT addrType;        /* address type */

            addrType = CM_IPADDR_VERSION(ips[i]);

            if (addrType == CM_IPADDR_IPV4 && *transportTypes == NS_TRANSPORT_IPV6)
                    continue;
            if (addrType == CM_IPADDR_IPV6 && *transportTypes != NS_TRANSPORT_IPV6)
                continue;
#endif /* UD_NQ_USETRANSPORTIPV6 */   

#ifdef UD_NQ_USETRANSPORTNETBIOS
            if (hasNBTransport && forceNBSocket && *transportTypes != NS_TRANSPORT_NETBIOS)
            {
                continue;
            }
#endif /* UD_NQ_USETRANSPORTNETBIOS */

            if (NULL != (socket = connectOneTransportByOneIp((NQ_INT)*transportTypes, &ips[i], &nbInfo)))
            {
#ifdef UD_NQ_INCLUDESMBCAPTURE
                {
                    SocketSlot *    serverSock = (SocketSlot *)socket;
                    NQ_IPADDRESS    serverIp;
                    NQ_PORT         serverPort;

                    syGetSocketPortAndIP(serverSock->socket , &serverIp , &serverPort);

                    captureHdr->dstIP = ips[i];
                    captureHdr->dstPort = *transportTypes == NS_TRANSPORT_NETBIOS ? CM_NB_SESSIONSERVICEPORT : CM_NB_SESSIONSERVICEPORTIP ;
                    captureHdr->srcIP = serverIp;
                    captureHdr->srcPort = 0;
                    captureHdr->receiving = FALSE;
                }
#endif /* UD_NQ_INCLUDESMBCAPTURE */
                pTransport->socket = socket;
                pTransport->connected = TRUE;
                pTransport->isReceiving = TRUE;
                pTransport->isSettingUp = TRUE;
                syMutexCreate(&pTransport->guard);
                cmListItemAdd(&connections, (CMItem *)pTransport, NULL);
                notifyListChange();
                result = TRUE;
                goto Exit;
            }
        }
    }

Exit:
    cmMemoryFree(aHost);
    cmMemoryFree(transportList);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL ccTransportIsTimeoutExpired(CCTransport * transport)
{
    NQ_TIME t, to = {TRANSPORT_IDLETIMEOUT*1000, 0 };
    NQ_TIME curr = syGetTimeInMsec();

    cmU64SubU64U64(&t, &curr, &transport->lastTime);

    return cmU64Cmp(&t,&to) > 0;
}

NQ_BOOL ccTransportIsConnected(CCTransport * pTransport)
{
    return nsIsSocketAlive(pTransport->socket) && pTransport->connected;
}

NQ_BOOL ccTransportDisconnect(CCTransport * pTransport)
{   
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transport:%p", pTransport);

    if (pTransport->connected && pTransport->item.guard != NULL)
    {
        NQ_BOOL res = FALSE;

        pTransport->isSettingUp = FALSE;
        pTransport->doDisconnect = TRUE;
        pTransport->isWaitingDisconectCond = TRUE;
        if (!cmThreadCondSet(&pTransport->disconnectCond))
            goto Exit;
        notifyListChange();

        res = cmThreadCondWait(&pTransport->disconnectCond, DODISCONNECT_TIMEOUT);
        cmThreadCondRelease(&pTransport->disconnectCond);
        pTransport->isWaitingDisconectCond = FALSE;
        if (res)
        {
            result = TRUE;
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR , "Waiting for disconnect timed out");
        }
    }
    else
    {   
        CCServer    *   pServer = NULL;

        pServer = (CCServer *)pTransport->server;
        pServer->smb->signalAllMatch(pTransport);
        result = FALSE;
    }
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

void ccTransportLock(CCTransport * transport)
{
    if (transport->item.guard != NULL)   /* just to check that ccTransportDisconnect wasn't called previously */
        syMutexTake(&transport->guard);
}

void ccTransportUnlock(CCTransport * transport)
{
    if (transport->item.guard != NULL)   /* just to check that ccTransportDisconnect wasn't called previously */
        syMutexGive(&transport->guard);
}

NQ_BOOL ccTransportSend(CCTransport * pTransport, NQ_IOBufPos buffer, NQ_COUNT packetLen, NQ_COUNT dataLen)
{
    NQ_BOOL result = FALSE;
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transport:%p buff:%p packetLen:%d dataLen:%d", pTransport, buffer, packetLen, dataLen);

    if (!nsIsSocketAlive(pTransport->socket))
    {
        ccTransportDisconnect(pTransport);
        sySetLastError(NQ_ERR_RECONNECTREQUIRED);
        LOGERR(CM_TRC_LEVEL_ERROR, "Socket is not alive");
        goto Exit;
    }

    /* Send the packet through NetBIOS */
    dataLen = nsPrepareNBBuffer(buffer, packetLen, dataLen);
    if(0 == dataLen)
    {
        pTransport->connected = FALSE;
        sySetLastError(NQ_ERR_RECONNECTREQUIRED);
        LOGERR(CM_TRC_LEVEL_ERROR, "prepare buffer failed");
        goto Exit;
    }

    if (NQ_FAIL == nsSendFromBuffer(pTransport->socket, buffer, packetLen, dataLen, NULL))
    {
        pTransport->connected = FALSE;
        sySetLastError(NQ_ERR_RECONNECTREQUIRED);
        LOGERR(CM_TRC_LEVEL_ERROR, "Sending failed");
        goto Exit;
    }
    pTransport->lastTime = syGetTimeInMsec();
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL ccTransportSendSync(CCTransport * pTransport, NQ_IOBufPos buffer, NQ_COUNT packetLen, NQ_COUNT dataLen)
{
    NQ_BOOL result = FALSE;
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "transport:%p buff:%p packetLen:%d dataLen:%d", pTransport, buffer, packetLen, dataLen);

    if (!nsIsSocketAlive(pTransport->socket))
    {
        ccTransportDisconnect(pTransport);
        sySetLastError(NQ_ERR_RECONNECTREQUIRED);
        LOGERR(CM_TRC_LEVEL_ERROR, "Socket is not alive");
        goto Exit;
    }
    /* Send the packet through NetBIOS */
    dataLen = nsPrepareNBBuffer(buffer, packetLen, dataLen);
    if(0 == dataLen)
    {
        pTransport->connected = FALSE;
        sySetLastError(NQ_ERR_RECONNECTREQUIRED);
        LOGERR(CM_TRC_LEVEL_ERROR, "prepare buffer failed");
        goto Exit;
    }

    if (NQ_FAIL == nsSendFromBuffer(pTransport->socket, buffer, packetLen, dataLen, NULL))
    {
        pTransport->connected = FALSE;
        sySetLastError(NQ_ERR_RECONNECTREQUIRED);
        LOGERR(CM_TRC_LEVEL_ERROR, "Sending failed");
        goto Exit;
    }
    pTransport->lastTime = syGetTimeInMsec();
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL ccTransportSendTail(CCTransport * pTransport, NQ_IOBufPos data, NQ_COUNT dataLen)
{
    NQ_INT result;
#ifdef UD_NQ_USEIOVECS
    NQ_IOBufState bufState;
    NQ_IOBufOrigin bufOrig;
#endif
    NQ_IOBuf sendBuf;
    sendBuf = IOBUF_GETBUFPTR(data);
    IOBUF_PREPARELENGTH(sendBuf, bufOrig, bufState, dataLen)

    result = ((NQ_INT) dataLen == sySendSocket(nsGetSySocket(pTransport->socket), sendBuf, dataLen));

    IOBUF_IOVECBUF_RESTORE(sendBuf, bufState, bufOrig)

    return result;
}

void ccTransportSetResponseCallback(CCTransport * pTransport, CCTransportResponseCallback callback, void * context)
{
    pTransport->callback = callback;
    pTransport->context = context;
}

NQ_IOBufPos ccTransportReceiveAll(CCTransport * pTransport, NQ_COUNT * dataLen)
{
    NQ_INT res;                 /* Various: number of bytes expected, call result */
    NQ_IOBufPos pRecvBuffer;    /* Receive buffer pointer */
    NQ_IOBufPos pResult;        /* Receive buffer pointer */

    IOBUF_POSINIT(pResult);
    res = ccTransportReceivePacketLength(pTransport);
    if (res == 0 || res == NQ_FAIL)
    {
        goto Exit;
    }
    pRecvBuffer = cmIOBufManTake((NQ_COUNT)res);
    if (IOBUF_ISNULL(pRecvBuffer))
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to take buffer");
        goto Exit;
    }
    if ((NQ_COUNT) NQ_FAIL == ccTransportReceiveBytes(pTransport, IOBUF_GETBUFPTR(pRecvBuffer), (NQ_COUNT)res))
    {
        cmIOBufManGive(pRecvBuffer);
        goto Exit;
    }
    *dataLen = (NQ_COUNT)res;
    pResult = pRecvBuffer;

Exit:
    ccTransportReceiveEnd(pTransport);
    return pResult;
}

/* the following 3 methods are a triplet (ccTransportReceiveEnd() must be called always) */
NQ_INT ccTransportReceivePacketLength(CCTransport * pTransport)
{
    NQ_INT res; /* bytes read */

    pTransport->isReceiving = TRUE;
    res = nsStartRecvIntoBuffer(pTransport->socket, &pTransport->recv);
    if (res == 0 || res == NQ_FAIL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to read NBT header or a control message received");
        pTransport->isReceiving = FALSE;
    }
    return res;
}

NQ_COUNT ccTransportReceiveBytes(CCTransport * pTransport, NQ_IOBuf buffer, NQ_COUNT dataLen)
{
    NQ_INT res;             /* Various: number of bytes expected, call result */ 

    res = nsRecvIntoBuffer(&pTransport->recv, buffer, dataLen);
    if (NQ_FAIL == res)
    {
        pTransport->isReceiving = FALSE;
        LOGERR(CM_TRC_LEVEL_ERROR, "Recv() failed");
    }
    return (NQ_COUNT)res;
}

NQ_COUNT ccTransportReceiveEnd(CCTransport * pTransport)
{
    NQ_COUNT res;       /* operation result */

    pTransport->isReceiving = FALSE;
    res = (NQ_COUNT)nsEndRecvIntoBuffer(&pTransport->recv);

    return res;
}

void ccTransportDiscardReceive(CCTransport * pTransport)
{
    ccTransportLock(pTransport);
    pTransport->isReceiving = FALSE;
    ccTransportUnlock(pTransport);
}

void ccTransportDiscardSettingUp(CCTransport * pTransport)
{
    ccTransportLock(pTransport);
    pTransport->isSettingUp = FALSE;
#ifdef SY_EPOLL_AVAILABLE
    syEpollAddSocket(epoll, SYSOCK(pTransport->socket) , pTransport);
#endif /* SY_EPOLL_AVAILABLE */
    notifyListChange();
    ccTransportUnlock(pTransport);
}
