/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client info operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccinfo.h"
#include "ccmount.h"
#include "ccfile.h"
#include "ccdfs.h"
#include "ccutils.h"
#include "ccparams.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

static NQ_WCHAR emptyFileName[] = { 0 };            /* empty file name */

#define isEmptyName(fileName) ((*fileName == cmWChar('\\')) ? syMemcmp(fileName + 1, emptyFileName, sizeof(NQ_WCHAR)) == 0 : syMemcmp(fileName, emptyFileName, sizeof(NQ_WCHAR)) == 0)

/* -- Static functions -- */

static NQ_STATUS getFileInformationByName(const NQ_WCHAR * fileName, CCFileInfo * pInfo, CCShare ** resolvedShare, NQ_WCHAR ** resolvedPath)
{
    CCMount * pMount;                           /* mount point descriptor */
    NQ_WCHAR * filePath = NULL;                 /* path component local to remote share or DFS full path */
    NQ_WCHAR * filePathFromLocalPath = NULL;    /* file path component */
    NQ_INT counter;                             /* operation attempts counter */
    CCShare * pShare;                           /* pointer to the hosting share */
#ifdef UD_CC_INCLUDEDFS
    NQ_BOOL dfsResolved = FALSE;
    CCDfsContext dfsContext = {CC_DFS_NUMOFRETRIES, 0, NULL, NULL}; /* DFS operations context */
    CCDfsResult dfsResult = {NULL, NULL, NULL};               /* result of DFS resolution */
    NQ_BOOL isDfs = FALSE;                      /* whether share is dfs (then full path should be build) */
#endif /* UD_CC_INCLUDEDFS */
    CCMountIdentifier mountPointID;             /* mount point identifier */
    NQ_STATUS result = NQ_FAIL;                 /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s pInfo:%p resolvedShare:%p resolvedPath:%p", cmWDump(fileName), pInfo, resolvedShare, resolvedPath);

    pMount = ccMountFind(fileName);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot find mount point");
        result = (NQ_STATUS)NQ_ERR_BADPATH;
        goto Exit;
    }
    pShare = pMount->share;
    if (pShare->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot get info from a print share");
        result = (NQ_STATUS)NQ_ERR_BADPARAM;
        goto Exit;
    }

#ifdef UD_CC_INCLUDEDFS
    if (pShare->flags & CC_SHARE_IN_DFS)
    {
        isDfs = TRUE;
    }
#endif /* UD_CC_INCLUDEDFS */

    filePathFromLocalPath = ccUtilsFilePathFromLocalPath(fileName,
#ifdef UD_CC_INCLUDEDFS
                                                        isDfs ? NULL :
#endif /* UD_CC_INCLUDEDFS */
                                                        pMount->pathPrefix,
                                                        pShare->user->server->smb->revision == CCCIFS_ILLEGALSMBREVISION,
                                                        TRUE);
    if (NULL == filePathFromLocalPath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        result = (NQ_STATUS)NQ_ERR_OUTOFMEMORY;
        goto Exit;
    }
    if (!ccTransportIsConnected(&pShare->user->server->transport) && !ccServerReconnect(pShare->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        result = (NQ_STATUS)NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

#ifdef UD_CC_INCLUDEDFS
    if (isDfs && NULL != fileName)
    {
        filePath = ccUtilsComposeRemotePathToFileByMountPath(pMount->path, filePathFromLocalPath, FALSE);
        if (NULL == filePath)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            result = (NQ_STATUS)NQ_ERR_OUTOFMEMORY;
            goto Exit;
        }

        cmMemoryFree(filePathFromLocalPath);
        filePathFromLocalPath = ccUtilsFilePathFromRemotePath(filePath, TRUE);
        if (NULL == filePathFromLocalPath)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            result = (NQ_STATUS)NQ_ERR_OUTOFMEMORY;
            goto Exit;
        }
    }
    else
#endif /* UD_CC_INCLUDEDFS */
    {
        filePath = cmMemoryCloneWString(filePathFromLocalPath);
        cmMemoryFree(filePathFromLocalPath);
        filePathFromLocalPath = NULL;
    }
    if (NULL == filePath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        result = (NQ_STATUS)NQ_ERR_OUTOFMEMORY;
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "filePath: %s", cmWDump(filePath));

    ccInitializeMountPointIdentifier(pMount, &mountPointID);
    for (counter = CC_CONFIG_RETRYCOUNT; counter > 0; counter--)
    {
        NQ_STATUS res = pShare->user->server->smb->doQueryFileInfoByName(
                    pShare,
                    filePath,
                    pInfo
                    );
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "doQueryFileInfoByName result: 0x%x", res);
#ifdef UD_CC_INCLUDEDFS
        if (NQ_SUCCESS != res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }
        }

        if (ccDfsIsError(dfsContext.lastError = res))
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "DFS related error %s", res == NQ_ERR_PATHNOTCOVERED ? ": NQ_ERR_PATHNOTCOVERED" : "");

            if (--dfsContext.counter < 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "DFS failed to resolve path: too many attempts");
                result = (NQ_STATUS)res;
                goto Exit;
            }

            dfsResult = ccDfsResolvePath(pMount, pShare, filePathFromLocalPath, &dfsContext);
            if (dfsResult.path)
            {
                cmMemoryFree(filePath);
                cmMemoryFree(filePathFromLocalPath);
                filePath = NULL;
                filePathFromLocalPath = NULL;
                pShare = dfsResult.share;
                filePath = dfsResult.share->flags & CC_SHARE_IN_DFS ? cmMemoryCloneWString(dfsResult.path) : ccUtilsFilePathFromRemotePath(dfsResult.path, FALSE);
                if (NULL == filePath)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    result = (NQ_STATUS)NQ_ERR_OUTOFMEMORY;
                    goto Exit;
                }
                dfsResolved = TRUE;
                LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "resolved filePath: %s", cmWDump(filePath));
                filePathFromLocalPath = ccUtilsFilePathFromRemotePath(dfsResult.path, TRUE);
                if (NULL == filePathFromLocalPath)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    result = (NQ_STATUS)NQ_ERR_OUTOFMEMORY;
                    goto Exit;
                }

                ccDfsResolveDispose(&dfsResult);
                dfsResult.path = NULL;
                counter++;
                continue;
            }
            else
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                    break;
                }
            }
       }
#endif /* UD_CC_INCLUDEDFS */
        cmMemoryFree(filePathFromLocalPath);
        filePathFromLocalPath = NULL;

        result = res;
        if (result == NQ_SUCCESS)
        {
#ifdef UD_CC_INCLUDEDFS
            if (dfsResolved && resolvedShare != NULL)
            {
                *resolvedShare = pShare;
            }
#endif /* UD_CC_INCLUDEDFS */
            break;
        }
        if (result == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pShare->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pShare->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

    if (resolvedPath && filePath)
    {
        *resolvedPath = cmMemoryCloneWString(filePath);
    }

Exit:
    if (NQ_SUCCESS != result)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to withdraw file information: 0x%x", result);
        sySetLastError((NQ_UINT32)result);        
    }
    cmMemoryFree(filePathFromLocalPath);
    cmMemoryFree(filePath);
#ifdef UD_CC_INCLUDEDFS
    if (NULL != dfsResult.path)
    {
        ccDfsResolveDispose(&dfsResult);
    }
    if (NULL != dfsContext.dfsPath)
    {
        cmMemoryFree(dfsContext.dfsPath);
    }
    if (NULL != dfsContext.referral)
    {
        cmMemoryFree(dfsContext.referral);
    }
#endif /* UD_CC_INCLUDEDFS */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}


/* --- API functions --- */

NQ_BOOL ccGetDiskFreeSpaceA(
    const NQ_CHAR * pathName,
    NQ_UINT * sectorsPerCluster,
    NQ_UINT * bytesPerSector,
    NQ_UINT * freeClusters,
    NQ_UINT * totalClusters,
    NQ_UINT * fsType,
    NQ_UINT * serialNumber
   )
{
    NQ_BOOL result = FALSE;         /* Unicode result */
    NQ_WCHAR * pathNameW = NULL;    /* the same in Unicode */
    CCVolumeInfo infoVolume;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pathName:%s sectorsPerCluster:%p bytesPerSector:%p freeClusters:%p totalClusters:%p fsType:%p serialNumber:%p",
                                    pathName, sectorsPerCluster, bytesPerSector, freeClusters, totalClusters, fsType, serialNumber);

    if ((NULL == pathName) || (NULL == sectorsPerCluster && NULL == bytesPerSector && NULL == freeClusters && NULL == totalClusters && NULL == fsType && NULL == serialNumber))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pathNameW = (NQ_WCHAR *)cmMemoryCloneAString(pathName);
    if (NULL == pathNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    result = ccGetDiskFreeSpaceEx(pathNameW, &infoVolume);
    if (result)
    {
        if (NULL != sectorsPerCluster)
        {
            *sectorsPerCluster = infoVolume.sectorsPerCluster;
        }
        if (NULL != bytesPerSector)
        {
            *bytesPerSector = infoVolume.bytesPerSector;
        }
        if (NULL != freeClusters)
        {
            *freeClusters = cmNQ_UINT64toU32(infoVolume.freeClusters);
        }
        if (NULL != totalClusters)
        {
            *totalClusters = cmNQ_UINT64toU32(infoVolume.totalClusters);
        }
        if (NULL != fsType)
        {
            *fsType = infoVolume.fsType;
        }
        if (NULL != serialNumber)
        {
            *serialNumber = infoVolume.serialNumber;
        }
    }
Exit:
    if (NULL != pathNameW)
    {
        cmMemoryFree(pathNameW);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetDiskFreeSpace(
    const NQ_WCHAR * pathName,
    NQ_UINT * sectorsPerCluster,
    NQ_UINT * bytesPerSector,
    NQ_UINT * freeClusters,
    NQ_UINT * totalClusters,
    NQ_UINT * fsType,
    NQ_UINT * serialNumber
   )
{
    NQ_BOOL result = FALSE;
    CCVolumeInfo infoVolume;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pathName:%s sectorsPerCluster:%p bytesPerSector:%p freeClusters:%p totalClusters:%p fsType:%p serialNumber:%p",
                                    cmWDump(pathName), sectorsPerCluster, bytesPerSector, freeClusters, totalClusters, fsType, serialNumber);

    if ((NULL == pathName) || (NULL == sectorsPerCluster && NULL == bytesPerSector && NULL == freeClusters && NULL == totalClusters && NULL == fsType && NULL == serialNumber))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    result = ccGetDiskFreeSpaceEx(pathName, &infoVolume);
    if (result)
    {
        if (NULL != sectorsPerCluster)
        {
            *sectorsPerCluster = infoVolume.sectorsPerCluster;
        }
        if (NULL != bytesPerSector)
        {
            *bytesPerSector = infoVolume.bytesPerSector;
        }
        if (NULL != freeClusters)
        {
            *freeClusters = cmNQ_UINT64toU32(infoVolume.freeClusters);
        }
        if (NULL != totalClusters)
        {
            *totalClusters = cmNQ_UINT64toU32(infoVolume.totalClusters);
        }
        if (NULL != fsType)
        {
            *fsType = infoVolume.fsType;
        }
        if (NULL != serialNumber)
        {
            *serialNumber = infoVolume.serialNumber;
        }
    }
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetDiskFreeSpaceExA(
    const NQ_CHAR * pathName,
    NQ_UINT * sectorsPerCluster,
    NQ_UINT * bytesPerSector,
    NQ_UINT64 * freeClusters,
    NQ_UINT64 * totalClusters,
    NQ_UINT * fsType,
    NQ_UINT * serialNumber
   )
{
    NQ_BOOL result = FALSE;           /* Unicode result */
    NQ_WCHAR * pathNameW = NULL;      /* the same in Unicode */
    CCVolumeInfo infoVolume;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pathName:%s sectorsPerCluster:%p bytesPerSector:%p freeClusters:%p totalClusters:%p fsType:%p serialNumber:%p",
                                    pathName, sectorsPerCluster, bytesPerSector, freeClusters, totalClusters, fsType, serialNumber);

    if ((NULL == pathName) || (NULL == sectorsPerCluster && NULL == bytesPerSector && NULL == freeClusters && NULL == totalClusters && NULL == fsType && NULL == serialNumber))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pathNameW = (NQ_WCHAR *)cmMemoryCloneAString(pathName);
    if (NULL == pathNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    result = ccGetDiskFreeSpaceEx(pathNameW, &infoVolume);
    if (result)
    {
        if (NULL != sectorsPerCluster)
        {
            *sectorsPerCluster = infoVolume.sectorsPerCluster;
        }
        if (NULL != bytesPerSector)
        {
            *bytesPerSector = infoVolume.bytesPerSector;
        }
        if (NULL != freeClusters)
        {
            freeClusters->high = infoVolume.freeClusters.high;
            freeClusters->low = infoVolume.freeClusters.low;
        }
        if (NULL != totalClusters)
        {
            totalClusters->high = infoVolume.totalClusters.high;
            totalClusters->low = infoVolume.totalClusters.low;
        }
        if (NULL != fsType)
        {
            *fsType = infoVolume.fsType;
        }
        if (NULL != serialNumber)
        {
            *serialNumber = infoVolume.serialNumber;
        }
    }
Exit:
    if (NULL != pathNameW)
    {
        cmMemoryFree(pathNameW);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetDiskFreeSpaceEx(
    const NQ_WCHAR * pathName,
    CCVolumeInfo *infoVolume
   )
{
    CCMount *       pMount;         /* mount point descriptor */
    NQ_STATUS       res;            /* operation result */
    CCShare *       pShare;         /* resolved share (DFS) */
    CCFileInfo      infoFile;       /* file info */
    NQ_COUNT        counter;        /* simple counter */
    NQ_WCHAR *      resolvedPath = NULL;
    NQ_WCHAR *      folderPath = NULL;
    CCMountIdentifier mountPointID; /* mount point identifier */
    NQ_BOOL         result = FALSE; /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "path:%s infoVolume:%p", cmWDump(pathName), infoVolume);

    if (NULL == pathName || NULL == infoVolume)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (ccUtilsPathIsLocal(pathName))
    {
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "local path");
        infoVolume->sectorsPerCluster = 1;
        infoVolume->bytesPerSector = 512;
        cmU64Zero(&infoVolume->freeClusters);
        cmU64Zero(&infoVolume->totalClusters);
        infoVolume->fsType = 0;
        infoVolume->serialNumber = 0;

        result = TRUE;
        goto Exit;
    }

    pMount = ccMountFind(pathName);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot find mount point");
        sySetLastError(NQ_ERR_BADPATH);
        goto Exit;
    }
    pShare = pMount->share;
    res = getFileInformationByName(pathName, &infoFile, &pShare, &resolvedPath);
    if (NQ_SUCCESS != res || NULL == pShare)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to withdraw file information:%d", res);
        sySetLastError(res);
        goto Exit;
    }
    if (!(infoFile.attributes & CIFS_ATTR_DIR))
    {
        /* get parent folder */
        folderPath = ccUtilsDirectoryFromPath(resolvedPath);
        cmMemoryFree(resolvedPath);
    }
    else
    {
        folderPath = resolvedPath;
    }

    ccInitializeMountPointIdentifier(pMount, &mountPointID);
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pShare->user->server->smb->doQueryFsInfo(pShare, folderPath, infoVolume);
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pShare->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pShare->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if (res != NQ_SUCCESS && !isEmptyName(folderPath))
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            --counter;
            *folderPath = cmWChar(0); /* try empty path */
        }
        else
        {
            break;
        }
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to withdraw volume information:0x%x", res);
        sySetLastError(res);
        goto Exit;
    }

    result = TRUE;

Exit:
    if (NULL != folderPath)
    {
        cmMemoryFree(folderPath);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_UINT32 ccGetFileAttributesA(const NQ_CHAR *fileName)
{
    NQ_STATUS result = NQ_SUCCESS;      /* operation result */
    NQ_WCHAR * fileNameW = NULL;        /* file Name in Unicode */
    NQ_UINT32 attributes = (NQ_UINT32)NQ_ERR_ATTRERROR;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s", fileName);

    if (NULL == fileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input parameters");
        result = NQ_ERR_BADPARAM;
        goto Exit;
    }
    fileNameW = cmMemoryCloneAString(fileName);
    if (NULL == fileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to allocate memory");
        result = NQ_ERR_NOMEM;
        goto Exit;
    }

    if (!ccGetFileAttributes(fileNameW, &attributes))
    {
         LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get attributes");
         attributes = (NQ_UINT32)NQ_ERR_ATTRERROR;
         result = syGetLastError();
         goto Exit;
    }

Exit:
    if (NULL != fileNameW)
    {
        cmMemoryFree(fileNameW);
    }
    if (NQ_SUCCESS != result)
    {
        sySetLastError(result);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%x error:0x%x", attributes, syGetLastError());
    return attributes;
}

NQ_BOOL ccGetFileAttributes(const NQ_WCHAR *fileName, NQ_UINT32 *attributes)
{
    CCFileInfo info;                     /* file info */
    NQ_STATUS res = NQ_FAIL;             /* operation status */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s attributes:%p", cmWDump(fileName), attributes);

    if (fileName == NULL || attributes == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input parameters");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (ccUtilsPathIsLocal(fileName))
    {
        info.attributes = CIFS_ATTR_DIR | CIFS_ATTR_READONLY;
        res = NQ_SUCCESS;
    }
    else
    {
        res = getFileInformationByName(fileName, &info, NULL, NULL);
    }
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to withdraw file information:0x%x", res);
        sySetLastError(res);
        goto Exit;
    }

    *attributes = info.attributes;
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "attributes:0x%x", *attributes);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", (res == NQ_SUCCESS) ? "TRUE" : "FALSE", syGetLastError());
    return (res == NQ_SUCCESS);
}

NQ_BOOL ccSetFileAttributesA(const NQ_CHAR * fileName, NQ_UINT32 attributes)
{
    NQ_BOOL result = FALSE;           /* Unicode result */
    NQ_WCHAR * fileNameW = NULL;      /* the same in Unicode */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s attributes:0x%x", fileName, attributes);

    if (fileName == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input parameters");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    fileNameW = cmMemoryCloneAString(fileName);
    if (NULL == fileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    result = ccSetFileAttributes(fileNameW, attributes);

Exit:
    if (NULL != fileNameW)
    {
       cmMemoryFree(fileNameW);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccSetFileAttributes(const NQ_WCHAR * fileName, NQ_UINT32 attributes)
{
    CCFile *    pFile = NULL;           /* open file handle */
    NQ_STATUS   res;                    /* operation result */
    NQ_INT      counter;                /* simple counter*/
    CCMountIdentifier mountPointID;     /* mount point identifier */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s attributes:0x%x", cmWDump(fileName), attributes);

    if (fileName == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input parameters");
        res = NQ_ERR_BADPARAM;
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        res = NQ_ERR_NOTREADY;
        goto Exit;
    }

    if (0 == attributes)
    {
        attributes |= SMB_ATTR_NORMAL;
    }
    pFile = (CCFile *)ccCreateFile(
            fileName,
            FILE_AM_SPECIAL_MASK | SMB_DESIREDACCESS_WRITEATTRIBUTES,
            FILE_SM_DENY_NONE,
            FILE_LCL_UNKNOWN,
            FALSE,
            0,
            FILE_CA_FAIL,
            FILE_OA_OPEN
            );
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot open file");
        res = NQ_ERR_BADPARAM;
        goto Exit;
    }
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot set attributes for a print file");
        res = NQ_ERR_BADPARAM;
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doSetFileAttributes(
                pFile,
                attributes
                );
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to write file information:0x%x", res);
    }

Exit:
    if (NULL != pFile)
    {
        ccCloseHandle(pFile);
    }
    if (NQ_SUCCESS != res)
    {
        sySetLastError((NQ_UINT32)res);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", (res == NQ_SUCCESS) ? "TRUE" : "FALSE", syGetLastError());
    return (res == NQ_SUCCESS);
}

NQ_BOOL ccGetFileInformationByNameA(const NQ_CHAR * fileName, FileInfo_t * fileInfoT)
{
    NQ_BOOL result = FALSE;           /* Unicode result */
    NQ_WCHAR * fileNameW = NULL;      /* the same in Unicode */
    CCFileInfo fileInfo;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s fileInfoT:%p", fileName, fileInfoT);

    if (NULL == fileName || NULL == fileInfoT)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    fileNameW = cmMemoryCloneAString(fileName);
    if (NULL == fileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    result = ccGetFileInformationByName(fileNameW, &fileInfo);
    if (result)
    {
        fileInfoT->allocationSizeHigh = fileInfo.allocationSize.high;
        fileInfoT->allocationSizeLow = fileInfo.allocationSize.low;
        fileInfoT->attributes = fileInfo.attributes;
        fileInfoT->creationTimeHigh = fileInfo.creationTime.high;
        fileInfoT->creationTimeLow = fileInfo.creationTime.low;
        fileInfoT->fileIndexHigh = fileInfo.fileIndex.high;
        fileInfoT->fileIndexLow = fileInfo.fileIndex.low;
        fileInfoT->fileSizeHigh = fileInfo.endOfFile.high;
        fileInfoT->fileSizeLow = fileInfo.endOfFile.low;
        fileInfoT->lastAccessTimeHigh = fileInfo.lastAccessTime.high;
        fileInfoT->lastAccessTimeLow = fileInfo.lastAccessTime.low;
        fileInfoT->lastWriteTimeHigh = fileInfo.lastWriteTime.high;
        fileInfoT->lastWriteTimeLow = fileInfo.lastWriteTime.low;
        fileInfoT->numberOfLinks = fileInfo.numberOfLinks;
        fileInfoT->volumeSerialNumber = fileInfo.volumeId.low;
    }

Exit:
    if (NULL != fileNameW)
    {
        cmMemoryFree(fileNameW);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetFileInformationByName(const NQ_WCHAR * fileName, CCFileInfo * fileInfo)
{
    NQ_STATUS res;            /* operation result */
    NQ_BOOL result = FALSE;   /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s fileInfo:%p", cmWDump(fileName), fileInfo);

    if (NULL == fileName || NULL == fileInfo)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (ccUtilsPathIsLocal(fileName))
    {
        syMemset(fileInfo, 0, sizeof(CCFileInfo));
        fileInfo->attributes = CIFS_ATTR_DIR | CIFS_ATTR_READONLY;
        fileInfo->endOfFile.low = 512;
        fileInfo->endOfFile.high = 0;
        result = TRUE;
        goto Exit;
    }
    res = getFileInformationByName(fileName, fileInfo, NULL, NULL);
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot get file information:%d", res);
        sySetLastError(res);
        result = FALSE;
        goto Exit;
    }
    result = TRUE;
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetFileTime(NQ_HANDLE handle, CCFileTime * creationTime, CCFileTime * lastAccessTime, CCFileTime * lastWriteTime)
{
    CCFile *        pFile = NULL;    /* open file handle */
    NQ_STATUS       res;             /* operation result */
    CCFileInfo      info;            /* file info */
    NQ_INT          counter;         /* simple counter*/
    CCMountIdentifier mountPointID;  /* mount point identifier */
    NQ_BOOL         result = FALSE;  /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p creationTime:%p lastAccessTime:%p lastWriteTime:%p", handle, creationTime, lastAccessTime, lastWriteTime);

    if ((NULL == handle) || (NULL == creationTime && NULL == lastAccessTime && NULL == lastWriteTime))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (!ccValidateFileHandle(handle))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)handle;
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot get file time for a print file");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (!pFile->open)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "File not opened");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (!ccTransportIsConnected(&pFile->share->user->server->transport) && !ccServerReconnect(pFile->share->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doQueryFileInfoByHandle(
                    pFile,
                    &info
                    );
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }

    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to query file:0x%x", res);
        goto Exit;
    }

    if (NULL != creationTime)
    {
        creationTime->timeHigh = info.creationTime.high;
        creationTime->timeLow = info.creationTime.low;
    }

    if (NULL != lastAccessTime)
    {
        lastAccessTime->timeHigh = info.lastAccessTime.high;
        lastAccessTime->timeLow = info.lastAccessTime.low;
    }

    if (NULL != lastWriteTime)
    {
        lastWriteTime->timeHigh = info.lastWriteTime.high;
        lastWriteTime->timeLow = info.lastWriteTime.low;

    }

    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccSetFileTime(NQ_HANDLE handle, CCFileTime * creationTime, CCFileTime * lastAccessTime, CCFileTime * lastWriteTime)
{
    CCFile *    pFile;                  /* open file handle */
    NQ_STATUS   res;                    /* operation result */
    NQ_UINT64   creationT;              /* time copy */
    NQ_UINT64   lastAccessT;            /* time copy */
    NQ_UINT64   lastWriteT;             /* time copy */
    NQ_INT      counter;                /* simple counter*/
    CCMountIdentifier mountPointID;     /* mount point identifier */
    NQ_BOOL     result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p creationTime:%p lastAccessTime:%p lastWriteTime:%p", handle, creationTime, lastAccessTime, lastWriteTime);

    if ((NULL == handle) || (NULL == creationTime && NULL == lastAccessTime && NULL == lastWriteTime))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "NULL Handle");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (!ccValidateFileHandle(handle))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)handle;
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot set file time for a print file");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    if (!pFile->open)
    {
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }
    if (NULL != creationTime)
    {
        creationT.low = creationTime->timeLow;
        creationT.high = creationTime->timeHigh;
    }
    else
    {
        creationT.low = 0;
        creationT.high = 0;
    }
    if (NULL != lastAccessTime)
    {
        lastAccessT.low = lastAccessTime->timeLow;
        lastAccessT.high = lastAccessTime->timeHigh;
    }
    else
    {
        lastAccessT.low = 0;
        lastAccessT.high = 0;
    }
    if (NULL != lastWriteTime)
    {
        lastWriteT.low = lastWriteTime->timeLow;
        lastWriteT.high = lastWriteTime->timeHigh;
    }
    else
    {
        lastWriteT.low = 0;
        lastWriteT.high = 0;
    }

    if (!ccTransportIsConnected(&pFile->share->user->server->transport) && !ccServerReconnect(pFile->share->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doSetFileTime(
                    pFile,
                    creationT,
                    lastAccessT,
                    lastWriteT
                    );
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set file time:0x%x", res);
        goto Exit;
    }

    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetFileSize(NQ_HANDLE hndl, NQ_UINT64 *fileSize)
{
    CCFile *        pFile;          /* open file handle */
    NQ_STATUS       res;            /* operation result */
    CCFileInfo      info;           /* file info */
    NQ_INT          counter;        /* simple counter */
    CCMountIdentifier mountPointID; /* mount point identifier */
    NQ_BOOL         result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "hndl:%p fileSize:%p", hndl, fileSize);

    if (NULL == hndl || NULL == fileSize)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (!ccValidateFileHandle(hndl))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)hndl;
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot get file size for a print file");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    if (!pFile->open)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "File not opened");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (!ccTransportIsConnected(&pFile->share->user->server->transport) && !ccServerReconnect(pFile->share->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doQueryFileInfoByHandle(
                    pFile,
                    &info
                    );
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to query file by handle:0x%x", res);
        goto Exit;
    }

    cmU64AssignU64(fileSize, &info.endOfFile);
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetFileInformationByHandle(NQ_HANDLE hndl, CCFileInfo *fileInfo)
{
    CCFile *    pFile;                  /* open file handle */
    NQ_STATUS   res;                    /* operation result */
    NQ_INT      counter;                /* simple counter*/
    CCMountIdentifier mountPointID;     /* mount point identifier */
    NQ_BOOL     result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "hndl:%p info:%p", hndl, fileInfo);

    if (NULL == hndl || NULL == fileInfo)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (!ccValidateFileHandle(hndl))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)hndl;
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot get file info for a print file");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (!pFile->open)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "File not opened");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (!ccTransportIsConnected(&pFile->share->user->server->transport) && !ccServerReconnect(pFile->share->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doQueryFileInfoByHandle(
                    pFile,
                    fileInfo
                    );
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to query file by handle:0x%x", res);
        goto Exit;
    }

    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccSetFileSizeByNameA(const NQ_CHAR *fileName, NQ_UINT32 sizeLow, NQ_UINT32 sizeHigh)
{
    NQ_BOOL result = FALSE;    /* Unicode result */
    NQ_WCHAR * fileNameW;      /* the same in Unicode */
    NQ_UINT64 size;            /* required size */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s sizeLow:%u sizeHigh:%u", fileName, sizeLow, sizeHigh);

    if (NULL == fileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    fileNameW = cmMemoryCloneAString(fileName);
    if (NULL == fileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    size.low = sizeLow;
    size.high = sizeHigh;
    result = ccSetFileSizeByName(fileNameW, size);
    cmMemoryFree(fileNameW);
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccSetFileSizeByName(const NQ_WCHAR * fileName, NQ_UINT64 size)
{
    CCFile *    pFile = NULL;           /* open file handle */
    NQ_STATUS   res = NQ_ERR_BADPARAM;  /* operation result */
    NQ_INT  counter;                    /* simple counter */
    CCMountIdentifier mountPointID;     /* mount point identifier */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%s size.low:%u size.high:%u", cmWDump(fileName), size.low, size.high);

    if (NULL == fileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        res = NQ_ERR_NOTREADY;
        goto Exit;
    }

    pFile = (CCFile *)ccCreateFileNoBatch(
            fileName,
            FILE_AM_SPECIAL_MASK | SMB_DESIREDACCESS_WRITEDATA,
            FILE_SM_DENY_NONE,
            FILE_LCL_UNKNOWN,
            FALSE,
            0,
            FILE_CA_FAIL,
            FILE_OA_OPEN
            );
    if (NULL == pFile)
    {
        res = syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot open file");
        goto Exit;
    }

    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot set file size for a print file");
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doSetFileSize(
                    pFile,
                    size
                    );
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set file size:0x%x", res);
    }

Exit:
    if (NULL != pFile)
    {
        ccCloseHandle(pFile);
    }
    sySetLastError((NQ_UINT32)res);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", (NQ_SUCCESS == res) ? "TRUE" : "FALSE", syGetLastError());
    return (NQ_SUCCESS == res);
}

NQ_BOOL ccSetFileSizeByHandle(NQ_HANDLE handle, NQ_UINT64 size)
{
    CCFile *        pFile;            /* open file handle */
    NQ_STATUS       res;              /* operation result */
    NQ_INT          counter;          /* simple counter */
    CCMountIdentifier mountPointID;   /* mount point identifier */
    NQ_BOOL         result = FALSE;   /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p size.low:%u size.high:%u", handle, size.low, size.high);

    if (NULL == handle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (!ccValidateFileHandle(handle))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)handle;
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot set file size for a print file");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    if (!pFile->open)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "File not opened");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }
    if (!ccTransportIsConnected(&pFile->share->user->server->transport) && !ccServerReconnect(pFile->share->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doSetFileSize(
                    pFile,
                    size
                    );
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set file size:0x%x", res);
        goto Exit;
    }
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccCheckPath(const NQ_WCHAR * path, NQ_BOOL stripLast, NQ_WCHAR **resolvedPath)
{
    NQ_STATUS status;
    CCFileInfo fileInfo;
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "path:%s stripLast:%s resolvedPath:%p", cmWDump(path), stripLast ? "TRUE" : "FALSE", resolvedPath);

    *resolvedPath = NULL;
    if (stripLast)
    {
        NQ_WCHAR * pathStripped;

        pathStripped = ccUtilsFilePathStripLastComponent(path);
        if (NULL == pathStripped)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }
        status = getFileInformationByName(pathStripped, &fileInfo, NULL, resolvedPath);
        cmMemoryFree(pathStripped);
    }
    else
    {
        status = getFileInformationByName(path, &fileInfo, NULL, resolvedPath);
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "resolvedPath: %s", (resolvedPath && *resolvedPath) ? cmWDump(*resolvedPath) : "");
    result = ( NQ_SUCCESS == status );

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

#endif /* UD_NQ_INCLUDECIFSCLIENT */
