/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Trace definitions
 *--------------------------------------------------------------------
 * MODULE        : Linux - SY
 * DEPENDENCIES  :
 ********************************************************************/

#ifndef _SYTRACE_H_
#define _SYTRACE_H_

#include <assert.h>

/*@@UD_NQ_EXTERNALTRACE
   Description
   By default NQ writes debug message through its internal
   mechanism called Trace Log. NQ can write traces on screen,
   into a file or just send them over the network as UDP
   datagrams.
   
   It is possible to delegate traces to an external mechanism.
   When this parameter is omitted, NQ uses its internal tracing,
   while when it is define, NQ delegates traces to an external
   logger.
   
   When external tracer is used, the following macros must be
   redefined:
     * <link syTraceInit, syTraceInit Macro>
     * <link syTraceShutdown, syTraceShutdown Macro>
     * <link syTraceMessage, syTraceMessage Macro>
     * <link syTraceDump, syTraceDump Macro>
     */
/*#define UD_NQ_EXTERNALTRACE*/

/*@@UD_CM_LOG_THRESHOLD
   Description
   This is the maximum level of the log message that NQ will
   \output. Messages with greater levels will be skipped.    */
#ifndef UD_CM_LOG_THRESHOLD
#define UD_CM_LOG_THRESHOLD 1000
#endif /* UD_CM_LOG_THRESHOLD */

/*@@UD_CM_LOG_TOCONSOLE
   Description
   Define preferred log output. When this parameter is defined
   NQ will delegate trace messages to the default console.
                                                               */
/*#define UD_CM_LOG_TOCONSOLE*/

/*@@UD_CM_LOG_TOFILE
   Description
   Define preferred log output. When this parameter is defined,
   NQ will save Trace Log messages into the file defined with
   the <link UD_CM_LOG_FILENAME, UD_CM_LOG_FILENAME Macro>.     */
#define UD_CM_LOG_TOFILE

/*@@UD_CM_LOG_TOREMOTE
   Description
   Define preferred log output. When this parameter is defined,
   NQ will transmit Trace Log messages as UDP datagrams.
   
   The server machine as designated with the <link UD_CM_LOG_REMOTESRVIP, UD_CM_LOG_REMOTESRVIP Macro>
   must run an application that will be listening to those
   datagrams.
   See Also
   <link UD_CM_LOG_REMOTEBROADCAST, UD_CM_LOG_REMOTEBROADCAST Macro>
   
   <link UD_CM_LOG_REMOTESRVIP, UD_CM_LOG_REMOTESRVIP Macro>
   
   <link UD_CM_LOG_REMOTESRVPORT, UD_CM_LOG_REMOTESRVPORT Macro>                                       */
/*#define UD_CM_LOG_TOREMOTE*/

/*@@UD_CM_LOG_REMOTEBROADCAST
   Description
   When this parameter is defined, NQ transmits a remote log
   message as a broadcast datagram. This parameter is only used
   when the <link UD_CM_LOG_TOREMOTE, UD_CM_LOG_TOREMOTE Macro>
   is defined.                                                  */
/*#define UD_CM_LOG_REMOTEBROADCAST*/

#ifdef UD_CM_LOG_TOFILE
/*@@UD_CM_LOG_FILENAME
   Description
   The name of the file for Trace Log output. This parameter is
   only used when the <link UD_CM_LOG_TOFILE, UD_CM_LOG_TOFILE Macro>
   is defined.                                                        */
#define UD_CM_LOG_FILENAME  "nqlog.log"
/* old log files are deleted when SY_LOG_NUMBEROFFILES is reached */
#define UD_CM_LOG_NUMBEROFFILES 5
#define UD_CM_LOG_FILESIZE_BYTES 52428800 /* 50MB */
#endif /* UD_CM_LOG_TOFILE */

/*@@UD_CM_LOG_REMOTESRVPORT
   Description
   NQ uses this parameter when the <link UD_CM_LOG_TOREMOTE, UD_CM_LOG_TOREMOTE Macro>
   is defined. NQ uses the value of this parameter as the port
   number on the remote machine which is listening for Trace Log
   datagrams.                                                                          */
#define UD_CM_LOG_REMOTESRVPORT 12121

#ifdef UD_CM_LOG_TOREMOTE
#ifdef UD_CM_LOG_REMOTEBROADCAST
#define UD_CM_LOG_REMOTESRVIP "255.255.255.255"
#else /* UD_CM_LOG_TOREMOTE */
/*@@UD_CM_LOG_REMOTESRVIP
   Description
   NQ uses this parameter when the <link UD_CM_LOG_TOREMOTE, UD_CM_LOG_TOREMOTE Macro>
   is defined and the <link UD_CM_LOG_REMOTEBROADCAST, UD_CM_LOG_REMOTEBROADCAST Macro>
   is not defined. NQ uses the value of this parameter as the IP
   address of the remote machine which is listening for Trace
   Log datagrams.                                                                       */
#define UD_CM_LOG_REMOTESRVIP "192.168.19.151"
#endif /* UD_CM_LOG_REMOTEBROADCAST */
#endif /* UD_CM_LOG_TOREMOTE */

#ifdef UD_CM_LOG_TOFILE
#define UD_CM_LOG_BASEFOLDER "./"
#endif /* UD_CM_LOG_TOFILE */

/* File and folder names for capture output. Those parameters are only used when
   the UD_NQ_INCLUDESMBCAPTURE Macro is defined. */
#define UD_CM_CAPTURE_FILENAME "nq.pcap"
#define UD_CM_CAPTURE_BASEFOLDER "./"

/* File and folder names for event log output. Those parameters are only used when
   the UD_NQ_INCLUDEEVENTLOG Macro is defined. */
#define UD_CM_EVENTLOG_FILENAME "event_log.txt"
#define UD_CM_EVENTLOG_BASEFOLDER "./"

/*@@SY_LOG_FUNCTION
   Description
   This is a way how the compiler specifies the enclosing
   function name. Example:
   
   __FUNCTION__
   
   If such macro is not available redefine this parameter as an
   empty string.                                                */
#define SY_LOG_FUNCTION     __FUNCTION__

/*@@SY_LOG_LINE
   Description
   This is a way how the compiler specifies line number in the
   source file. Example:
   
   __LINE__
   
   If such macro is not available redefine this parameter as an
   empty string.                                                */
#define SY_LOG_LINE         __LINE__

/*@@SY_LOG_FILE
   Description
   This is a way how the compiler specifies source file name.
   \Example:
   
   <c>__FILE__</c>
   
   If such macro is not available redefine this parameter as an
   empty string.                                                */
#define SY_LOG_FILE         __FILE__

typedef enum {SY_MSG_TYPE_ENTER, SY_MSG_TYPE_LEAVE, SY_MSG_TYPE_ERR, SY_MSG_TYPE_MSG, SY_MSG_TYPE_START, SY_MSG_TYPE_STOP} SYMsgType;

/*@@
 Description
 External tracer initialization. NQ issues this call once on its startup.
 Returns
 None
 */
#define syTraceInit()

/*@@
   Description
   External tracer shutdown. NQ issues this call once on its
   shutdown.
 Returns
 None
 */
#define syTraceShutdown()

/*@@syTraceMessage
   Description
   Create a trace message. This call may contain a variable
   argument list in accordance with the <i>_format</i>
   parameter.
   Parameters
   _file :      Name of the source file.
   _function :  Name of the enclosing function.
   _line :      Line number in the source file.
   _level :     Trace priority.
   _type :      Record type as\:
                * 0 &#45; function call entrance.
                * 1 &#45; function return.
                * 2 &#45; error message.
                * 3 &#45; generic text message.
                * 4 &#45; task/thread start.
                * 5 &#45; task/thread shutdown.
   _format :    The format string in the same form as used in a <i>printf</i>
                call.
   Returns
   None   */
void syTraceMessage(const NQ_CHAR *file, const NQ_CHAR *function, NQ_UINT line, NQ_UINT level, NQ_UINT type, const NQ_CHAR *format, ...);

/*@@
   Description
   Create and open new trace file
   Parameters
   name :      Trace file name
   Returns
   file handle or invalid handle   */
SYFile syTraceCreateFile(const NQ_WCHAR* name);

/*@@
   Description
   Delete trace file
   Parameters
   name :      Trace file name
   Returns
   NQ_SUCCESS on success otherwise NQ_FAIL   */
NQ_STATUS syTraceDeleteFile(const NQ_WCHAR* name);

/*@@syTraceDump
   Description
   Dump an array of bytes into Trace Log.
   Parameters
   _file :      Name of the source file.
   _function :  Name of the enclosing function.
   _line :      Line number in the source file.
   _level :     Trace priority.
   _str :       Explanatory text.
   _addr :      Pointer to the first byte to dump.
   _nBytes :    Number of bytes to dump.
   Returns
   None                                            */
#define syTraceDump(_file, _function, _line, _level, _str, _addr, _nBytes)
#define syTraceIODump(_file, _function, _line, _level, _str, _addr, _nBytes)

#endif  /* _SYTRACE_H_ */


