/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : OS-dependent functions
 *--------------------------------------------------------------------
 * MODULE        : Linux - SY
 * DEPENDENCIES  :
 ********************************************************************/

#include "udparams.h"
#include <syapi.h>
#include <udapi.h>
#include <cmapi.h>
#include <errno.h>

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
#include <sys/sendfile.h>
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

#include <sys/statvfs.h>

#ifdef UD_NQ_USETRANSPORTIPV6
/* IPv6 related includes here */
#endif /* UD_NQ_USETRANSPORTIPV6 */
#if defined(SY_UNICODEFILESYSTEM) || defined(UD_CC_INCLUDELDAP) || defined(UD_NQ_CODEPAGEUTF8)
#include <iconv.h>
#endif /* defined(SY_UNICODEFILESYSTEM) || defined(UD_CC_INCLUDELDAP) || defined(UD_NQ_CODEPAGEUTF8) */

/*#define SYOPSYST_DEBUG*/

#if defined(SY_UNICODEFILESYSTEM)
#define UNICODEFILENAMES
#define UTF8_BYTES_REPRESENTATION 4                                         /* While representing UTF8 string using ASCII string using 1 to 4 bytes for each symbol */
#define UTF8_BUFF_SIZE (UTF8_BYTES_REPRESENTATION * UD_FS_FILENAMELEN + 1)  /* Use this size while representing UTF8 string using ASCII strings.
                                                                               Each 4 cells will represent 1 UTF8 character with max size of UD_FS_FILENAMELEN, +1 for trailing NULL */
#endif /* defined(SY_UNICODEFILESYSTEM) */

/* not following symbolic links */
#define DONTFOLLOW_SYMLINKS

#ifdef DONTFOLLOW_SYMLINKS
#define OPEN_FLAGS_LINKS O_NOFOLLOW
#else
#define OPEN_FLAGS_LINKS 0
#endif /* DONTFOLLOW_SYMLINKS */

/* 64 bit offsets support */
#define LONG_FILES_SUPPORT

#if defined(LONG_FILES_SUPPORT) && !(defined(_FILE_OFFSET_BITS) && _FILE_OFFSET_BITS == 64)
#define stat        stat64
#define lstat       lstat64
#define fstat       fstat64
#define lseek       lseek64
#define ftruncate   ftruncate64
#define off_t       off64_t
#define fsblkcnt_t  fsblkcnt64_t
#define OPEN_FLAGS_LONGFILES O_LARGEFILE
#else
#define OPEN_FLAGS_LONGFILES 0
#endif /* defined(LONG_FILES_SUPPORT) && !(defined(_FILE_OFFSET_BITS) && _FILE_OFFSET_BITS == 64) */

#define OPEN_RDONLY          (O_RDONLY           | OPEN_FLAGS_LONGFILES | OPEN_FLAGS_LINKS)
#define OPEN_WRONLY          (O_WRONLY           | OPEN_FLAGS_LONGFILES | OPEN_FLAGS_LINKS)
#define OPEN_RDWR            (O_RDWR             | OPEN_FLAGS_LONGFILES | OPEN_FLAGS_LINKS)
#define OPEN_RDWR_CREAT      (O_RDWR   | O_CREAT | OPEN_FLAGS_LONGFILES | OPEN_FLAGS_LINKS)

#define S_IWUGO (S_IWUSR | S_IWGRP | S_IWOTH)
#define S_IRUGO (S_IRUSR | S_IRGRP | S_IROTH)

#ifdef SYOPSYST_DEBUG
#define PRINT_SOCKET_OPTIONS
#endif /* SYOPSYST_DEBUG */
/*
    Static functions & data
    -----------------------
 */
#if (defined(SYOPSYST_DEBUG) && defined(PRINT_SOCKET_OPTIONS))
typedef struct          /* descriptor for a control command */
{
    int level;
    int opt;
    NQ_CHAR * str;
    int param;
}
SocketOpt;

static const SocketOpt sockOpt[] =
{
    { SOL_SOCKET    , SO_KEEPALIVE      , "SO_KEEPALIVE" , 1},      /* 0 */
    { SOL_SOCKET    , SO_REUSEADDR      , "SO_REUSEADDR" , 1},      /* 1 */
    { SOL_SOCKET    , SO_BROADCAST      , "SO_BROADCAST" , 0},      /* 2 */
    { IPPROTO_TCP   , TCP_NODELAY       , "TCP_NODELAY" , 1},       /* 3 */
    { SOL_TCP       , TCP_KEEPCNT       , "TCP_KEEPCNT" , 9},           /* 4 */
    { SOL_TCP       , TCP_KEEPIDLE      , "TCP_KEEPIDLE" , 7200},       /* 5 */
    { SOL_TCP       , TCP_KEEPINTVL     , "TCP_KEEPINTVL" , 75},        /* 6 */
    { IPPROTO_IP    , IP_TOS            , "IPTOS_LOWDELAY and IPTOS_THROUGHPUT" , 1},       /* 7 */
    { SOL_SOCKET    , SO_SNDBUF         , "SO_SNDBUF" , 43690},         /* 9 */
    { SOL_SOCKET    , SO_RCVBUF         , "SO_RCVBUF" , 184640},            /* 10 */
    { SOL_SOCKET    , SO_SNDLOWAT       , "SO_SNDLOWAT" , 1},           /* 11 */
    { SOL_SOCKET    , SO_RCVLOWAT       , "SO_RCVLOWAT" , 1},           /* 12 */
    { SOL_SOCKET    , SO_SNDTIMEO       , "SO_SNDTIMEO" , 0},       /* 13 */
    { SOL_SOCKET    , SO_RCVTIMEO       , "SO_RCVTIMEO" , 0},       /* 14 */
    { IPPROTO_TCP   , TCP_QUICKACK      , "TCP_QUICKACK" , 1},      /* 15 */
    { SOL_TCP       , TCP_DEFER_ACCEPT  , "TCP_DEFER_ACCEPT" , 0},  /* 16 */
    { IPPROTO_TCP   , TCP_MAXSEG        , "TCP_MAXSEG" , 536},      /* 17 */
    { IPPROTO_TCP   , TCP_CORK          , "TCP_CORK" , 0},              /* 18 */
};
#endif /* defined(SYOPSYST_DEBUG) && defined(PRINT_SOCKET_OPTIONS) */


/*
    Static functions & data
    -----------------------
 */

typedef struct
{
  /* buffer for converting strings */
#ifdef UNICODEFILENAMES
    char utf8Name[UTF8_BUFF_SIZE];
    char newName[UTF8_BUFF_SIZE];
#ifdef UD_CM_LOG_TOFILE
    char traceUtf8Name[UTF8_BUFF_SIZE];
#endif /* UD_CM_LOG_TOFILE */
#else
    char asciiName[CM_BUFFERLENGTH(char, UD_FS_FILENAMELEN)];
    char newName[CM_BUFFERLENGTH(char, UD_FS_FILENAMELEN)];
#ifdef UD_CM_LOG_TOFILE
    char traceAsciiName[CM_BUFFERLENGTH(char, UD_FS_FILENAMELEN)];
#endif /* UD_CM_LOG_TOFILE */
#endif /* UNICODEFILENAMES */
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
    int pipe[2];
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/* check that file name is valid */
static int checkFileName(const NQ_WCHAR * name)
{
        return NULL == syWStrrchr(name, cmWChar(':'));
}

/*
 *====================================================================
 * PURPOSE: Convert file name from UTF-16 LE to UTF-8
 *          Buffer should be twice as big as for UTF-16, since UTF-8 may
 *          (theoretically) have up to 4 bytes for a symbol
 *--------------------------------------------------------------------
 * PARAMS:  IN inName UTF-16 LE name buffer
 *          OUT utf8Name UTF-8 name buffer
 *          IN utf8NameSize out buffer size
 *
 * RETURNS: none
 *
 * NOTES:
 *
 *====================================================================
 */
#ifdef UNICODEFILENAMES
static void filenameToUtf8(const NQ_UINT16* inName, NQ_CHAR *utf8Name, size_t utf8NameSize)
{
    iconv_t convertor;
    size_t inbytesleft = (syWStrlen(inName) + 1) * sizeof(NQ_UINT16);
    size_t outbytesleft = utf8NameSize;
    NQ_CHAR *in = (NQ_CHAR *)inName;
    NQ_CHAR *out = utf8Name;

    if ((convertor = iconv_open("UTF-8", "UTF-16LE")) == (iconv_t)-1)
    {
        syPrintf("!! Unable to convert UTF-16LE to UTF-8 (iconv_open failed with errno %d)\n", errno);
        return;
    }

    if (iconv(convertor, &in, &inbytesleft, &out, &outbytesleft) == (size_t)-1)
    {
        syPrintf("!! Unable to convert UTF-16LE to UTF-8 (iconv failed with errno %d)\n", errno);
        iconv_close(convertor);
        return;
    }

    iconv_close(convertor);
}

/*
 *====================================================================
 * PURPOSE: Convert file name from UTF-8 to UTF-16 LE
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer UTF-16 LE name buffer
 *          IN size out buffer size
 *
 * RETURNS: none
 *
 * NOTES:   The input name will be taken from 'staticData->utf8Name'
 *
 *====================================================================
 */
static void filenameFromUtf8(NQ_UINT16* buffer, NQ_INT size)
{
    iconv_t convertor;
    size_t inbytesleft = strlen(staticData->utf8Name) + 1;
    size_t outbytesleft = (size_t)size;
    NQ_CHAR *in = staticData->utf8Name;
    NQ_CHAR *out = (NQ_CHAR*)buffer;

    if ((convertor = iconv_open("UTF-16LE", "UTF-8")) == (iconv_t)-1)
    {
        syPrintf("!! Unable to convert UTF-8 to UTF-16LE (iconv_open failed with errno %d)\n", errno);
        return;
    }

    if (iconv(convertor, &in, &inbytesleft, &out, &outbytesleft) == (size_t)-1)
    {
        syPrintf("!! Unable to convert UTF-8 to UTF-16LE (iconv failed with errno %d)\n", errno);
        iconv_close(convertor);
        return;
    }

    iconv_close(convertor);
}
#endif /* UNICODEFILENAMES */


#ifdef UD_NS_ASYNCSEND
static void (*bufferReleaseCallback)(const unsigned char *buf);      /* callback function for releasing a z-buffer */

/* release Z-buffer */

static
void
freeSocketBuffer(
    caddr_t buf,        /* buffer to release */
    int freeArg         /* argument */
    );
#endif /* UD_NS_ASYNCSEND */

static
NQ_BOOL
buildSockaddr(
    struct sockaddr* saddr,
    int *size,
    const NQ_IPADDRESS *ip,
    NQ_PORT port
        );

static
NQ_BOOL
parseSockaddr(
    struct sockaddr* saddr,
    NQ_IPADDRESS *ip,
    NQ_PORT *port
        );

/* convert file information from system to system-independent format */

static void
statToFileInformation(
    const struct stat* from,
    const struct statvfs* fromvfs,
    SYFileInformation* to
    );

#ifdef SY_EXTENDFILENOTSUPPORTED
/* for Linux versions where ftruncate() doesn't support extending file */

/* extend file with a specified amount of bytes by writing zeros and positioning
 the file at the end */

static int              /* NQ_SUCCESS or NQ_FAIL */
extendFile(
    int file,           /* fd */
    int off,            /* number of bytes to add */
    int fSize           /* file size before */
);

static const unsigned char zeroArray[] =        /* array of zero bytes for extendFile() use */
    {
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
};
#endif /* SY_EXTENDFILENOTSUPPORTED */


static fsblkcnt_t convertBlocks(unsigned long to, unsigned long from, fsblkcnt_t blocks)
{
    fsblkcnt_t res = 0;

    if (from == to)
    {
        res = blocks;
    }
    else if (from > to)
    {
        res = blocks * (from / to);
    }
    else
    {
        res = blocks / (to / from);
    }
    return res;
}

/*====================================================================
 * PURPOSE: initialize resources
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
syInit(
    void
    )
{
    NQ_BOOL res = FALSE;

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData*) syCalloc(1, sizeof(*staticData));
    if (NULL == staticData)
    {
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
    staticData->pipe[0] = -1;
    staticData->pipe[1] = -1;
    if (pipe(staticData->pipe) < 0)
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
#ifdef SY_FORCEALLOCATION
        syFree(staticData);
        staticData = NULL;
#endif /* SY_FORCEALLOCATION */
        goto Exit;
    }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

    res = TRUE;

Exit:
    return res;
}

/*====================================================================
 * PURPOSE: release resources
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
syStop(
    void
    )
{
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
    close(staticData->pipe[0]);
    close(staticData->pipe[1]);
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
    /* release memory */
#ifdef SY_FORCEALLOCATION
    if (NULL != staticData)
    {
        syFree(staticData);
    }
    staticData = NULL;
#endif /* SY_FORCEALLOCATION */
}

/*
 *====================================================================
 * Convert UNIX file permissions to DOS file attributes
 *--------------------------------------------------------------------
 * NOTES: dos readonly is represented in unix by removing everyone's write bit
 *        dos archive is represented in unix by the user's execute bit
 *        dos system is represented in unix by the group's execute bit
 *        dos hidden is represented in unix by the other's execute bit
 *        dos directory is represented in unix by unix's dir bit
 *====================================================================
 */


int
syUnixMode2DosAttr(
    int mode
    )
{
    int attributes = 0;

    if ((mode & S_IFDIR) != 0)
        attributes = SY_ATTR_DIRECTORY;

    if ((mode & S_IWUGO) == 0)
        attributes |= SY_ATTR_READONLY;
/*  Note: the next attributes are incorrect
    if ((attributes & SY_ATTR_HIDDEN) != 0)
        mode |= S_IXOTH;
    if ((attributes & SY_ATTR_SYSTEM) != 0)
        mode |= S_IXGRP;*/
    if ((mode & S_IXUSR) != 0 && (mode & S_IFDIR) == 0)
        attributes |= SY_ATTR_ARCHIVE;

    return (attributes == 0 ? SY_ATTR_NORMAL : attributes);
}

/*
 *====================================================================
 * Convert DOS file attributes to UNIX file permissions
 *--------------------------------------------------------------------
 * NOTES: dos readonly is represented in unix by removing everyone's write bit
 *        dos archive is represented in unix by the user's execute bit
 *        dos system is represented in unix by the group's execute bit
 *        dos hidden is represented in unix by the other's execute bit
 *        dos directory is represented in unix by unix's dir bit
 *====================================================================
 */

static
int
dos2Unix(
    int attributes
    )
{
    int mode = S_IRUGO;

    if ((attributes & SY_ATTR_DIRECTORY) != 0)
        mode |= S_IFDIR;

    if ((attributes & SY_ATTR_READONLY) == 0)
        mode |= S_IWUGO;
/*  Note: the next attributes are incorrect
    if ((attributes & SY_ATTR_HIDDEN) != 0)
        mode |= S_IXOTH;
    if ((attributes & SY_ATTR_SYSTEM) != 0)
        mode |= S_IXGRP;*/
    if ((attributes & SY_ATTR_ARCHIVE) != 0)
        mode |= S_IXUSR;

    return mode;
}

/*
 *====================================================================
 * PURPOSE: Convert last system error into SMB error
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: SMB Error converted from system error
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_UINT32
syGetLastSmbError(
    void
    )
{
    if (errno == 0)
        return 0;

    return udGetSmbError((NQ_UINT32)errno);
}

/*
 *====================================================================
 * PURPOSE: Set NQ error into system error
 *--------------------------------------------------------------------
 * PARAMS:  IN SMB error in NT format
 *
 * RETURNS: NONE
 *
 * NOTES:
 *
 *====================================================================
 */

void
sySetLastNqError(
    NQ_STATUS nqErr
    )
{
    errno = (int)nqErr;
}

/*
 *====================================================================
 * PURPOSE: fill a static file information structure
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          OUT file information structure
 *
 * RETURNS: 0 on success, -1 on error
 *
 * NOTES:
 *
 *====================================================================
 */

int
syGetFileInformationByName(
    const NQ_WCHAR* fileName,
    SYFileInformation* fileInfo
    )
{
    struct stat tmp;
    struct statvfs vfs;

#ifdef UNICODEFILENAMES
    filenameToUtf8(fileName, staticData->utf8Name, sizeof(staticData->utf8Name));
    if (-1 == stat(staticData->utf8Name, &tmp))
    {
        return NQ_FAIL;
    }
    if (-1 == statvfs(staticData->utf8Name, &vfs))
    {
        return NQ_FAIL;
    }
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), fileName, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    if (-1 == stat(staticData->asciiName, &tmp))
    {
        return NQ_FAIL;
    }
    if (-1 == statvfs(staticData->asciiName, &vfs))
    {
        return NQ_FAIL;
    }
#endif /* UNICODEFILENAMES */

    statToFileInformation(&tmp, &vfs, fileInfo);

    return NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: fill a static file information structure
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          IN file name (not used in VxWorks)
 *          OUT file information structure
 *
 * RETURNS: NONE
 *
 * NOTES:
 *
 *====================================================================
 */

int
syGetFileInformation(
    SYFile file,
    const NQ_WCHAR* fileName,
    SYFileInformation* fileInfo
    )
{
    struct stat tmp;
    struct statvfs vfs;

    if (-1 == fstat(file, &tmp))
    {
        return NQ_FAIL;
    }

    if (-1 == fstatvfs(file, &vfs))
    {
        return NQ_FAIL;
    }

    statToFileInformation(&tmp, &vfs, fileInfo);

    return NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: Get file size by file handle
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          IN/OUT size
 *
 * RETURNS: NQ_FAIL or NQ_SUCCESS
 *
 * NOTES:
 *
 *====================================================================
 */
int
syGetFileSize(
        SYFile file,
        NQ_UINT64 *size
        )
{
    struct stat tmp;

    if (-1 == fstat(file, &tmp))
    {
        return NQ_FAIL;
    }
#ifdef LONG_FILES_SUPPORT
    size->high = (NQ_UINT32)(tmp.st_size >> 32);
    size->low = (NQ_UINT32)(tmp.st_size & 0xFFFFFFFF);
#else
    size->high = 0;
    size->low = tmp.st_size;
#endif /* LONG_FILES_SUPPORT */

    return NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: fill a static file information structure
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN file handle (use handle if it is valid)
 *          OUT file information structure
 *
 * RETURNS: 0 - success, -1 - error
 *
 * NOTES:
 *
 *====================================================================
 */

int
sySetFileInformation(
    const NQ_WCHAR* fileName,
    SYFile file,
    const SYFileInformation* fileInfo
    )
{
    struct utimbuf timeBuf;     /* time buffer */
    struct stat statBuf;        /* file status info */
    const char *name;
    NQ_BOOL doClose = FALSE;    /* whether to close the file or not */
    int result = NQ_FAIL;

#ifdef UNICODEFILENAMES
    filenameToUtf8(fileName, staticData->utf8Name, sizeof(staticData->utf8Name));
    name = staticData->utf8Name;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), fileName, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    name = staticData->asciiName;
#endif /* UNICODEFILENAMES */

    if (!(syIsValidFile(file)))
    {
        if ((file = open(name, OPEN_RDONLY, 0777)) == -1)
        {
            goto Exit;
        }
        doClose = TRUE;
    }

#ifdef DONTFOLLOW_SYMLINKS
    if (lstat(name, &statBuf) == -1)
    {
        goto Exit;
    }
    if (S_ISLNK(statBuf.st_mode))
    {
        errno = EACCES;
        goto Exit;
    }
#endif /* DONTFOLLOW_SYMLINKS */

    /* set attributes */
    if (fchmod(file, (mode_t)dos2Unix((int)fileInfo->attributes)) == -1)
    {
        goto Exit;
    }

    /* read file statistics */
    if (fstat(file, &statBuf) == -1)
    {
        goto Exit;
    }

    if (   (cmTimeConvertMSecToSec((NQ_TIME *)&fileInfo->lastAccessTime) != 0 && cmTimeConvertMSecToSec((NQ_TIME *)&fileInfo->lastAccessTime) != (NQ_UINT32)statBuf.st_atime)
        || (cmTimeConvertMSecToSec((NQ_TIME *)&fileInfo->lastWriteTime)  != 0 && cmTimeConvertMSecToSec((NQ_TIME *)&fileInfo->lastWriteTime)  != (NQ_UINT32)statBuf.st_mtime)
       )
    {
        NQ_TIME zero = {0, 0};

        if (0 == cmU64Cmp((NQ_TIME *)&fileInfo->lastAccessTime, &zero))
        {
            timeBuf.actime = statBuf.st_atime;
        }
        else
        {
            timeBuf.actime = (time_t)cmTimeConvertMSecToSec((NQ_TIME *)&fileInfo->lastAccessTime);
        }
        if (0 == cmU64Cmp((NQ_TIME *)&fileInfo->lastWriteTime, &zero))
        {
            timeBuf.modtime = statBuf.st_mtime;
        }
        else
        {
            timeBuf.modtime = (time_t)cmTimeConvertMSecToSec((NQ_TIME *)&fileInfo->lastWriteTime);
        }

        if (utime(name, &timeBuf) == -1)
        {
            goto Exit;
        }
    }

    result = NQ_SUCCESS;

Exit:
    if (doClose)
    {
        close(file);
    }

    return result;
}

/*
 *====================================================================
 * PURPOSE: query volume information
 *--------------------------------------------------------------------
 * PARAMS:  IN volume name
 *          OUT buffer for information
 *
 * RETURNS: 0 - success, -1 - error
 *
 * NOTES:
 *
 *====================================================================
 */

int
syGetVolumeInformation(
    const NQ_WCHAR* name,
    SYVolumeInformation *info
    )
{
    const char *converted_name;
    struct statfs fs_stat;
    struct stat file_stat;
    dev_t root_dev;
    DIR *root_dir;
    struct dirent *entry;
    fsblkcnt_t total_sum = 0;
    fsblkcnt_t free_sum = 0;
    int ret = NQ_FAIL;

#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    converted_name = staticData->utf8Name;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    converted_name = staticData->asciiName;
#endif /* UNICODEFILENAMES */

    syMemset(info, 0, sizeof(*info));

    /* inspect the share root */
    if (NULL == (root_dir = opendir(converted_name)))
    {
        goto Exit;
    }

    if (-1 == fstatfs(dirfd(root_dir), &fs_stat))
    {
        goto ExitClose;
    }

    if (-1 == fstat(dirfd(root_dir), &file_stat))
    {
        goto ExitClose;
    }

    root_dev = file_stat.st_dev;
    total_sum += fs_stat.f_blocks;
    free_sum += fs_stat.f_bfree;
    info->fileSystemIdLow = UD_FS_FILESYSTEMID;
    info->creationTimeLow = 0L; /* simulate 1-1-1970 */
    info->serialNumberLow = 0L; /* we do not report serial number */
    info->blockSizeLow = (NQ_UINT32)fs_stat.f_bsize;
    info->blocksPerUnitLow = 1;

    /* accumulate the free space of file systems mounted within  */
     while ((entry = readdir(root_dir)))
     {
        int current_fd = -1;

        if (entry->d_type != DT_DIR)
        {
            continue;
        }

        if (0 == strncmp(entry->d_name, "..", sizeof(entry->d_name)))
        {
            continue;
        }

        current_fd = openat(dirfd(root_dir), entry->d_name, O_RDONLY | O_DIRECTORY);
        if (current_fd < 0)
        {
            continue;
        }
        if (-1 == fstat(current_fd, &file_stat))
        {
            goto Next;
        }
        if (file_stat.st_dev == root_dev)
        {
            goto Next;
        }
        if (-1 == fstatfs(current_fd, &fs_stat))
        {
            goto Next;
        }

        total_sum += convertBlocks(info->blockSizeLow,
                                    (unsigned long)fs_stat.f_bsize,
                                    fs_stat.f_blocks);
        free_sum += convertBlocks(info->blockSizeLow,
                                    (unsigned long)fs_stat.f_bsize,
                                    fs_stat.f_bfree);
Next:
        close(current_fd);
    } /* end of while ((entry = readdir(root_dir))) */

    info->totalUnitsLow = (NQ_UINT32)total_sum;
    info->freeUnitsLow = (NQ_UINT32)free_sum;
#ifdef LONG_FILES_SUPPORT
    info->totalUnitsHigh = (NQ_UINT32)(total_sum >> 32);
    info->freeUnitsHigh = (NQ_UINT32)(free_sum >> 32);
#endif /* LONG_FILES_SUPPORT */
    ret = NQ_SUCCESS;

ExitClose:
    closedir(root_dir);
Exit:
    return ret;
}

/*
 *====================================================================
 * PURPOSE: get MAC address by IP4
 *--------------------------------------------------------------------
 * PARAMS:  IN ip address
 *          OUT buffer for MAC address
 *
 * RETURNS: NONE
 *
 * NOTES:   if MAC address is not available - fill buffer by zeroes
 *
 *====================================================================
 */

void
syGetMacAddress(
    NQ_IPADDRESS4 ip4,
    NQ_BYTE* macBuffer
    )
{
    static const NQ_BYTE mac[6] = {0,0,0,0,0,0};
    syMemcpy(macBuffer, mac, sizeof(mac));
    return;
}

/*
 *====================================================================
 * PURPOSE: get next adapter information
 *--------------------------------------------------------------------
 * PARAMS:  IN adapter number (zero based)
 *          OUT buffer for adapter IP in NBO
 *          OUT buffer for subnet mask in NBO
 *          OUT buffer for wins IP in NBO (may be zero for a B-node)
 *
 * RETURNS: 0 when there is no adapter with the given index,
 *          1 when adapter information awailable
 *
 * NOTES:
 *
 *====================================================================
 */

/*static struct ifreq *next_ifr (struct ifreq *ifr)
{
    char *ptr = (char *)ifr;
    ptr += (sizeof (*ifr)  - sizeof (struct sockaddr) + ifr->ifr_ifru.ifru_addr.sa_len );
    return (struct ifreq *)ptr;
}*/

#define MAX_EXT_ADDRESSES (UD_NS_MAXADAPTERS + 5)

NQ_STATUS
syGetAdapter(
    NQ_INDEX adapterIdx,    /* adapter number (zero based) */
    NQ_INDEX * osIndex,     /* buffer for adapter index as defined by the OS */
    NQ_IPADDRESS4* pIp,     /* buffer for adapter IP in NBO */
    NQ_IPADDRESS6 *ip6,     /* buffer for adapter IPv6 in NBO */
    NQ_IPADDRESS4* pSubnet, /* buffer for subnet address in NBO */
    NQ_IPADDRESS4* pBcast,  /* buffer for bcast address in NBO */
    NQ_WCHAR* pWins,        /* buffer for semicolon delimited list of WINS servers */
    NQ_WCHAR* pDns          /* buffer for semicolon delimited list of DNS servers */
    )
{
    unsigned int idx = 0;                    /* as counted by NQ */
    unsigned int osIdx = 0;                  /* as counted by the OS */
    int status;
    unsigned long nelem;
    int ntSockFd;
    int result = NQ_FAIL;
    struct ifaddrs* pIfa;
    struct ifaddrs* saved = NULL;
#ifdef UD_NQ_USETRANSPORTIPV6
    struct in6_addr any6 = IN6ADDR_ANY_INIT;
#endif /* UD_NQ_USETRANSPORTIPV6 */

    /*
     * nelem is set to the maximum interfaces
     * on one machine here
     */
    nelem = 4*MAX_EXT_ADDRESSES;

    /* try to get the IP/Bcast from the system */

    ntSockFd = socket(AF_INET, SOCK_DGRAM, 0);

    /* check socket error */
    if (-1 == ntSockFd)
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
       goto Exit;
    }

    status = getifaddrs(&pIfa);
    if (status < 0)
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif
        goto Exit;
    }

    if (0 == status)
    {
        saved = pIfa;
        for (idx = 0; adapterIdx > 0 && idx <= *osIndex; pIfa = pIfa->ifa_next, ++idx )
        {
            /* do nothing */
        }
        *osIndex = idx;

        for (idx = 0 , osIdx = *osIndex; pIfa != NULL && osIdx < nelem ; pIfa = pIfa->ifa_next, ++osIdx)
        {
#ifdef UD_NQ_USETRANSPORTIPV6
            memset(ip6, 0, 16);
#endif /* UD_NQ_USETRANSPORTIPV6 */
            *pIp = CM_IPADDR_ZERO4;
            *pSubnet = CM_IPADDR_ZERO4;
            if (pIfa->ifa_addr && (pIfa->ifa_addr->sa_family == AF_INET || pIfa->ifa_addr->sa_family == AF_INET6))
            {
#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
                unsigned short domain[CM_NQ_HOSTNAMESIZE];       /* buffer for domain name */
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */
                /*
                 * Don't bother with interfaces that have been disabled
                 */
                if (!(pIfa->ifa_flags & IFF_UP))
                {
                    continue;
                }

                /*
                 * Don't use the loop back interface
                 */
                if (pIfa->ifa_flags & IFF_LOOPBACK)
                {
                    continue;
                }

                /*
                 * If its not an internet inteface then dont use it.
                 */

                if(0 && pIfa->ifa_addr->sa_family != AF_INET)
                {
                    continue;
                }

                /*
                 * If this is an interface that supports
                 * broadcast fetch the broadcast address.
                 */
                if (!(pIfa->ifa_flags & IFF_BROADCAST))
                {
                    continue;
                }

                /* Get WINS servers as supplied by User Defined level (for example cm_cfg.txt) */
                udGetWins(pWins);
#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
                /* Get DNS servers as supplied by User Defined level (for example cm_cfg.txt) */
                udGetDnsParams(domain, pDns);
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */
                *osIndex = osIdx;
                if (pIfa->ifa_addr != NULL)
                {
                    if (pIfa->ifa_addr->sa_family == AF_INET)
                    {
                        /* Get interface IP Address */
                        *pIp = ((struct sockaddr_in*)pIfa->ifa_addr)->sin_addr.s_addr;
                        /* Get interface mask Address */
                        *pSubnet = ((struct sockaddr_in*)pIfa->ifa_netmask)->sin_addr.s_addr;
                        /* Get interface broadcast Address */
                        *pBcast = ((struct sockaddr_in*)pIfa->ifa_ifu.ifu_broadaddr)->sin_addr.s_addr;
                    }
#ifdef UD_NQ_USETRANSPORTIPV6
                    else if (pIfa->ifa_addr->sa_family == AF_INET6)
                    {
                        struct sockaddr_in6* p6 = (struct sockaddr_in6*)pIfa->ifa_addr;
                        if (memcmp(p6->sin6_addr.s6_addr16, &any6, sizeof(any6)) == 0)
                            continue;
                        memcpy(ip6, p6->sin6_addr.s6_addr16, 16);
                    }
#endif /* UD_NQ_USETRANSPORTIPV6 */
                    result = NQ_SUCCESS;
                    goto Exit;
                }
            }
        }
    }

Exit:
    if (NULL != saved)
    {
        freeifaddrs(saved);
    }

    if (-1 != ntSockFd)
    {
        close(ntSockFd);
    }

    return result;
}


#ifdef UD_NQ_USETRANSPORTIPV6
/*
 *====================================================================
 * PURPOSE: get IPv6 scope ID
 *--------------------------------------------------------------------
 * PARAMS:  ip - the IPv6 address
 *
 * RETURNS: 0..n - the scope id to be used
 *
 * NOTES:  0 - unknown network interface
 *====================================================================
 */

NQ_UINT32
syGetIPv6ScopeId(
    const NQ_IPADDRESS6 ip
    )
{
    struct ifaddrs *ifaddr, *ifa;
    unsigned int scopeId = 0;

    /* only link-local ip addresses need scopeId */
    if ((ip[0] & SY_LINKLOCALIP) != SY_LINKLOCALIP)
        return 0;

    if (getifaddrs(&ifaddr) == -1)
        return 0;

    for (ifa = ifaddr; ifa != NULL; ifa = ifa->ifa_next)
    {
       if (ifa->ifa_addr == NULL)
          continue;

       if (!(ifa->ifa_flags & IFF_UP) || (ifa->ifa_flags & IFF_LOOPBACK))
          continue;

       if (ifa->ifa_addr->sa_family == AF_INET6)
       {
           scopeId = if_nametoindex(ifa->ifa_name);
           if (scopeId != 0)
              break;
       }
    }

    freeifaddrs(ifaddr);
    return scopeId;
}

#endif /* UD_NQ_USETRANSPORTIPV6 */

/*
 *====================================================================
 * PURPOSE: Get system time in POSIX format in milliseconds
 *--------------------------------------------------------------------
 * PARAMS:
 *
 * RETURNS: The number of milliseconds elapsed since Jan 1, 1970 in milliseconds
 *
 * NOTES:
 *
 *====================================================================
 */
NQ_TIME syGetTimeInMsec(void)
{
    NQ_TIME curTime;
    struct timeval tv;

    gettimeofday(&tv, NULL);
    cmU64MultU32U32(&curTime, (NQ_UINT32)tv.tv_sec, 1000);
    curTime.low += (NQ_UINT32)(tv.tv_usec/1000);

    return curTime;
}

/*
 *====================================================================
 * PURPOSE: Get time offset
 *--------------------------------------------------------------------
 * PARAMS:
 *
 * RETURNS: The number of minutes to be added to the local time to
 *          get GMT. This number is negative for GMT+ and positive for
 *          GMT-
 *
 * NOTES:
 *
 *====================================================================
 */

int
syGetTimeZone(
    void
    )
{
  struct tm utc, local;
  time_t current;

  time ( &current );

  gmtime_r ( &current, &utc );
  localtime_r ( &current, &local );

  return (utc.tm_hour - local.tm_hour) * 60 + utc.tm_min - local.tm_min;
}

/*
 *====================================================================
 * PURPOSE: Covert the <b>timespec</b> to time in milliseconds
 *--------------------------------------------------------------------
 * PARAMS:  IN <b>timespec</b> value format
 *
 * RETURNS: The converted time from <b>timespec</b> to a time in milliseconds
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_TIME syConvertTimeSpecToTimeInMsec(void * val)
{
    NQ_TIME curTime;
    struct timespec * tv = (struct timespec *)val;

    cmU64MultU32U32(&curTime, (NQ_UINT32)tv->tv_sec, 1000);
    curTime.low += (NQ_UINT32)(tv->tv_nsec/1000000);

    return curTime;
}

/*
 *====================================================================
 * PURPOSE: Convert time in milliseconds to seconds
 *--------------------------------------------------------------------
 * PARAMS:  timeMsec - time in milliseconds.
 *
 * RETURNS:  time in seconds
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_UINT32 syConvertTimeInMsecToSec(NQ_TIME * timeMsec)
{
    NQ_UINT32 a0;        /* 16 bit, low    bits */
    NQ_UINT32 a1;        /* 16 bit, medium bits */
    NQ_UINT32 a2;        /* 32 bit, high   bits */

    /* Copy the time values to a2/a1/a0 */

    a2 =  timeMsec->high;
    a1 = timeMsec->low >> 16;
    a0 = timeMsec->low & 0xffff;

    /* divide a by 1000 (a = a2/a1/a0), put the rest into r. */

    a1 += (a2 % 1000) << 16;
    a2 /=       1000;
    a0 += (a1 % 1000) << 16;
    a1 /=       1000;
    a0 /=       1000;

    return (a1 << 16) + a0;
}

/*
 *====================================================================
 * PURPOSE: Decompose system time into fragments
 *--------------------------------------------------------------------
 * PARAMS:  IN system time
 *          OUT structure of file fragments
 *
 * RETURNS: NONE
 *
 * NOTES:
 *
 *====================================================================
 */

void
syDecomposeTime(
    NQ_UINT32 time,
    SYTimeFragments* decomposed
    )
{
    struct tm sysTime;
    time_t t = (time_t)time;

    localtime_r(&t, &sysTime);

    decomposed->year   = (NQ_UINT16)sysTime.tm_year;
    decomposed->day    = (NQ_UINT16)sysTime.tm_mday;
    decomposed->month  = (NQ_UINT16)sysTime.tm_mon;
    decomposed->hour   = (NQ_UINT16)sysTime.tm_hour;
    decomposed->min    = (NQ_UINT16)sysTime.tm_min;
    decomposed->sec    = (NQ_UINT16)sysTime.tm_sec;
}

/*
 *====================================================================
 * PURPOSE: compose system time from fragments
 *--------------------------------------------------------------------
 * PARAMS:  OUT structure of file fragments
 *
 * RETURNS: composed system time
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_UINT32
syComposeTime(
    const SYTimeFragments* decomposed
    )
{
    struct tm* timer;
    time_t localTime;

    localTime = time(NULL);

    timer = localtime(&localTime);

    timer->tm_year   = decomposed->year;
    timer->tm_mday   = decomposed->day;
    timer->tm_mon    = decomposed->month;
    timer->tm_hour   = decomposed->hour;
    timer->tm_min    = decomposed->min;
    timer->tm_sec    = decomposed->sec;

    return (NQ_UINT32)mktime(timer);
}

#ifdef UD_NQ_USETRANSPORTIPV6
#define MAX_SOCKADDR_SIZE sizeof(struct sockaddr_in6)
#else /* UD_NQ_USETRANSPORTIPV6 */
#define MAX_SOCKADDR_SIZE sizeof(struct sockaddr_in)
#endif /* UD_NQ_USETRANSPORTIPV6 */

static
NQ_BOOL
buildSockaddr(
    struct sockaddr* saddr,
    int *size,
    const NQ_IPADDRESS *ip,
    NQ_PORT port
        )
{
#ifdef UD_NQ_INCLUDETRACE
    if (NULL == ip)
    {
        TRCERR("Invalid ip pointer: NULL");
        return FALSE;
    }
#endif /* UD_NQ_INCLUDETRACE */

#ifdef UD_NQ_USETRANSPORTIPV6
    switch (ip->version)
    {
        case CM_IPADDR_IPV4:
#endif /* UD_NQ_USETRANSPORTIPV6 */

        {
            struct sockaddr_in *sin = (struct sockaddr_in*)saddr;

            memset(sin, 0, sizeof(struct sockaddr_in));
            sin->sin_family = AF_INET;
            sin->sin_port = port;
            sin->sin_addr.s_addr = (in_addr_t)CM_IPADDR_GET4(*ip);
            *size = sizeof(struct sockaddr_in);
            return TRUE;
        }

#ifdef UD_NQ_USETRANSPORTIPV6
        case CM_IPADDR_IPV6:
        {
            struct sockaddr_in6 *sin6 = (struct sockaddr_in6*)saddr;

            memset(sin6, 0, sizeof(struct sockaddr_in6));
            sin6->sin6_family = AF_INET6;
            sin6->sin6_port = port;
            syMemcpy(sin6->sin6_addr.s6_addr, CM_IPADDR_GET6(*ip), sizeof(NQ_IPADDRESS6));
            sin6->sin6_scope_id = (uint32_t)syGetIPv6ScopeId(ip->addr.v6);
            *size = sizeof(struct sockaddr_in6);
            return TRUE;
        }

        default:
            TRC1P("Invalid ip version: %d", ip->version);
            return FALSE;
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */
}

static
NQ_BOOL
parseSockaddr(
    struct sockaddr* saddr,
    NQ_IPADDRESS *ip,
    NQ_PORT *port
        )
{
#ifdef UD_NQ_USETRANSPORTIPV6
    switch (saddr->sa_family)
    {
    case AF_INET:
#endif /* UD_NQ_USETRANSPORTIPV6 */

    {
        struct sockaddr_in *sin = (struct sockaddr_in*)saddr;
        *port = sin->sin_port;
        CM_IPADDR_ASSIGN4(*ip, sin->sin_addr.s_addr);
        return TRUE;
    }

#ifdef UD_NQ_USETRANSPORTIPV6
    case AF_INET6:
    {
        struct sockaddr_in6 *sin6 = (struct sockaddr_in6*)saddr;
        *port = sin6->sin6_port;
        CM_IPADDR_ASSIGN6(*ip, sin6->sin6_addr.s6_addr);
        return TRUE;
    }

    default:
    {
        NQ_IPADDRESS zero = CM_IPADDR_ZERO;
        TRC1P("Unknown address family: %d", saddr->sa_family);
        *port = 0;
        *ip = zero;
        return FALSE;
    }
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */
}

/*
 *====================================================================
 * PURPOSE: Detecting whether a socket is still alive
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *
 * RETURNS: TRUE or FALSE
 *
 * NOTES:   this method is said to work on any BSD socket system: issue select()
 *          with a zero timeout. on dead socket this should return error instead of zero
 *====================================================================
 */

NQ_BOOL
syIsSocketAlive(
    SYSocketHandle sock
    )
{
    fd_set set;             /* temporary set for checking the socket */
    struct timeval tv;      /* timeout value */
    int ret;                /* the select result */

    tv.tv_sec = 0;
    tv.tv_usec = 0;

    FD_ZERO(&set);
    FD_SET(sock, &set);

    ret = select(FD_SETSIZE, &set, NULL, NULL, &tv);

    return (ret >= 0)? TRUE : FALSE;
}

/*
 *====================================================================
 * PURPOSE: Stop socket operations and disconnect the socket if it was connected
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   this method is said to work on any BSD socket system
 *
 *====================================================================
 */

NQ_STATUS
syShutdownSocket(
    SYSocketHandle sock
    )
{
    return (shutdown(sock, 2)==ERROR)? NQ_FAIL : NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: Close socket
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */
NQ_STATUS
syCloseSocket(
    SYSocketHandle sock
    )
{
    return (close(sock) == ERROR) ? NQ_FAIL : NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: Listen on server socket
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          IN max number of requests in queue
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */
NQ_STATUS
syListenSocket(
    SYSocketHandle sock,
    NQ_INT backlog
    )
{
    return (listen(sock, backlog) == OK) ? NQ_SUCCESS : NQ_FAIL;
}

/*
 *====================================================================
 * PURPOSE: Create new socket
 *--------------------------------------------------------------------
 * PARAMS:  OUT pointer to socket id
 *
 * RETURNS: None
 *
 * NOTES: For TCP sockets send timeout is set
 *
 *====================================================================
 */
SYSocketHandle
syCreateSocket(
    NQ_INT stream,
    NQ_UINT family
    )
{
    int sock = ERROR;

#ifdef UD_NQ_USETRANSPORTIPV6
    switch (family)
    {
        case CM_IPADDR_IPV4:
#endif /* UD_NQ_USETRANSPORTIPV6 */
            sock = socket(AF_INET, ((stream)? SOCK_STREAM : SOCK_DGRAM), 0);
#ifdef UD_NQ_USETRANSPORTIPV6
            break;
        case CM_IPADDR_IPV6:
            sock = socket(AF_INET6, ((stream)? SOCK_STREAM : SOCK_DGRAM), 0);
            break;
        default:
            TRC1P("Invalid socket family: %d", family);
            goto Exit;
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */

#ifdef UD_NQ_TCPSOCKETSENDTIMEOUT
    if ((ERROR != sock) && stream)
    {
        struct timeval timeout;
        timeout.tv_sec = UD_NQ_TCPSOCKETSENDTIMEOUT;
        timeout.tv_usec = 0;

        if (ERROR == setsockopt(sock, SOL_SOCKET, SO_SNDTIMEO, (char *)&timeout, sizeof(timeout)))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = SOL_SOCKET, __optname = SO_SNDTIMEO");
#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        }
    }
#endif /* UD_NQ_TCPSOCKETSENDTIMEOUT */
#ifdef UD_NQ_USETRANSPORTIPV6
Exit:
#endif /* UD_NQ_USETRANSPORTIPV6 */
    return sock;
}

/*
 *====================================================================
 * PURPOSE: Bind socket to IP and port
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          IN IP address in NBO
 *          IN port number in NBO
 *          IN whether to reuse address
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syBindSocket(
    SYSocketHandle sock,
    const NQ_IPADDRESS *ip,
    NQ_PORT port,
    NQ_BOOL reuseAddress
    )
{
    char buffer[MAX_SOCKADDR_SIZE];
    struct sockaddr *saddr = (struct sockaddr*)buffer;
    int size;
    int yes = 1;               /* setting socket options */

#ifdef UD_NQ_USETRANSPORTIPV6
    if (ip->version == CM_IPADDR_IPV6)
    {
        setsockopt(sock, IPPROTO_IPV6, IPV6_V6ONLY, (char*)&yes, sizeof(yes));
    }
#endif

    if (reuseAddress)
    {
        if (ERROR == setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, (char*)&yes, sizeof(yes)))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = SOL_SOCKET, __optname = SO_REUSEADDR");
#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
            return NQ_FAIL;
        }
    }

    if (buildSockaddr(saddr, &size, ip, port) &&
        bind(sock, (struct sockaddr*)saddr, (socklen_t)size) == OK)
    {
        if (ERROR == setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (char*)&yes, sizeof(yes)))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = SOL_SOCKET, __optname = SO_BROADCAST");
#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        }
        if (ERROR == setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char*)&yes, sizeof(yes)))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = IPPROTO_TCP, __optname = TCP_NODELAY");
#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        }
        if (ERROR == setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char*)&yes, sizeof(yes)))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = SOL_SOCKET, __optname = SO_KEEPALIVE");
#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        }

        return NQ_SUCCESS;
    }

    return NQ_FAIL;
}

/*
 *====================================================================
 * PURPOSE: Allow broadcasts on socket
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syAllowBroadcastsSocket(
    SYSocketHandle sock
    )
{
    int on = 1;

    return setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (char*)&on, sizeof(on));
}


/*
 *====================================================================
 * PURPOSE: Tune a new client socket
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *
 * RETURNS: None
 *
 * NOTES:
 *
 *====================================================================
 */

void
sySetClientSocketOptions(
    SYSocketHandle sock
    )
{
    struct linger l;    /* for setting linger options */
#if (UD_NS_BUFFERSIZE > 8192)
    int valBuf;         /* for setting buffer lengths */
#endif

    l.l_onoff = 0;
    l.l_linger = 1;
    if (ERROR == setsockopt(sock, SOL_SOCKET, SO_LINGER, (char*)&l, sizeof(l)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = SOL_SOCKET, __optname = SO_LINGER");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
    }

#if (UD_NS_BUFFERSIZE > 8192)
    valBuf = UD_NS_BUFFERSIZE;
    if (ERROR == setsockopt(sock, SOL_SOCKET, SO_SNDBUF, (char*)&valBuf, sizeof(valBuf)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = SOL_SOCKET, __optname = SO_SNDBUF");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
    }

    if (ERROR == setsockopt(sock, SOL_SOCKET, SO_RCVBUF, (char*)&valBuf, sizeof(valBuf)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = SOL_SOCKET, __optname = SO_RCVBUF");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
    }
#endif /* (UD_NS_BUFFERSIZE > 8192) */

#if (defined(SYOPSYST_DEBUG) && defined(PRINT_SOCKET_OPTIONS))
    {
        int i;
        int sendbuff;
        socklen_t optlen;

        optlen = sizeof(sendbuff);
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "sySetClientSocketOptions: sock = %d\n",sock);
#endif /* SYOPSYST_DEBUG */
        for (i = 0; i < sizeof(sockOpt)/sizeof(sockOpt[0]); i++)
        {
            getsockopt(sock, sockOpt[i].level, sockOpt[i].opt, &sendbuff, &optlen);
#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "\t\t\t%s = %d\n", sockOpt[i].str, sendbuff);
#endif /* SYOPSYST_DEBUG */
        }
    }
#endif /* defined(SYOPSYST_DEBUG) && defined(PRINT_SOCKET_OPTIONS) */
}

/*
 *====================================================================
 * PURPOSE: Get IP and port the socket is bound on
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          OUT buffer for IP address in NBO
 *          OUT buffer for port number in NBO
 *
 * RETURNS: None
 *
 * NOTES:
 *
 *====================================================================
 */

void
syGetSocketPortAndIP(
    SYSocketHandle sock,
    NQ_IPADDRESS *ip,
    NQ_PORT *port
    )
{
    char buffer[MAX_SOCKADDR_SIZE];
    struct sockaddr *saddr = (struct sockaddr*)buffer;
    socklen_t size = sizeof(buffer);

    if (getsockname(sock, (struct sockaddr*)saddr, &size) == ERROR)
    {
        NQ_IPADDRESS zero = CM_IPADDR_ZERO;

        *port = 0;
        *ip = zero;
    }
    else
        parseSockaddr(saddr, ip, port);
}

/*
 *====================================================================
 * PURPOSE: Send a UDP message to a specific addressee
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          IN buffer to send
 *          IN number of bytes to send
 *          IN IP address of the addressee in NBO
 *          IN port number of the addressee in NBO
 *
 * RETURNS: NQ_FAIL or number of bytes sent
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
sySendToSocket(
    SYSocketHandle sock,
    const NQ_BYTE *buf,
    NQ_COUNT len,
    const NQ_IPADDRESS *ip,
    NQ_PORT port
    )
{
    char buffer[MAX_SOCKADDR_SIZE];
    struct sockaddr *saddr = (struct sockaddr*)buffer;
    int size, res;

    if (!buildSockaddr(saddr, &size, ip, port))
        return NQ_FAIL;

#ifdef MSG_NOSIGNAL
    res = (int)sendto(sock, (char*)buf, (size_t)len, MSG_NOSIGNAL, saddr, (socklen_t)size);
#else
    res = (int)sendto(sock, (char*)buf, (size_t)len, 0, saddr, (socklen_t)size);
#endif

    return (res == ERROR) ? NQ_FAIL : res;
}

/*
 *====================================================================
 * PURPOSE: Connect to a remote server port
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          IN IP address of the server in NBO
 *          IN port number of the server in NBO
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES: This call temporary uses non blocking socket in order to have configurable timeout
 *        otherwise connect() for blocking sockets would block for a long time.
 *
 *====================================================================
 */
NQ_STATUS
syConnectSocket(
    SYSocketHandle sock,
    const NQ_IPADDRESS *ip,
    NQ_PORT port
    )
{
    char buffer[MAX_SOCKADDR_SIZE];
    struct sockaddr *saddr = (struct sockaddr*)buffer;
    int size, val0 = 0, val1 = 1;
#ifdef UD_NQ_SOCKETCONNECTTIMEOUT
    int flags;
#endif /* UD_NQ_SOCKETCONNECTTIMEOUT */
    NQ_STATUS result = NQ_FAIL;

    if (!buildSockaddr(saddr, &size, ip, port))
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }

#ifdef UD_NQ_SOCKETCONNECTTIMEOUT
    /* get current socket flags */
    if ((flags = fcntl(sock, F_GETFL, 0)) < 0)
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }

    /* make it non blocking */
    if (fcntl(sock, F_SETFL, flags | O_NONBLOCK) < 0)
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }

    if (connect(sock, saddr, (socklen_t)size) < 0)
    {
        if (errno == EINPROGRESS)
        {
            SYSocketSet writeSet;
            struct timeval tv;
            int retVal;
            int error;

            FD_ZERO(&writeSet);
            FD_SET(sock, &writeSet);
            tv.tv_sec = (time_t)UD_NQ_SOCKETCONNECTTIMEOUT;
            tv.tv_usec = 0;

            retVal = select(sock + 1, NULL, &writeSet, NULL, &tv);
            if (retVal == 0 || retVal < 0)
            {
                /* timeout or error */
                goto Restore;
            }

            if (FD_ISSET(sock, &writeSet))
            {
                socklen_t len = sizeof(error);
                error = 0;
                if (getsockopt(sock, SOL_SOCKET, SO_ERROR, &error, &len) < 0)
                {
#ifdef SYOPSYST_DEBUG
                    syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
                    goto Restore;
                }

                if (error > 0)
                {
                    goto Restore;
                }
            }
            else
            {
                goto Restore;
            }

            result = NQ_SUCCESS;
        }
Restore:
        /* restore socket flags */
        if ((flags = fcntl(sock, F_SETFL, flags)) < 0)
        {
#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
            result = NQ_FAIL;
            goto Exit;
        }
    }
#else
    if (connect(sock, saddr, (socklen_t)size) < 0)
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }
    result = NQ_SUCCESS;
#endif /* UD_NQ_SOCKETCONNECTTIMEOUT */

    if ( -1 == setsockopt(sock, SOL_SOCKET, SO_KEEPALIVE, (char*)&val0, sizeof(val0)) )
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }

    if ( -1 == setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char*)&val1, sizeof(val1)) )
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }

    if ( -1 == setsockopt(sock, IPPROTO_TCP, TCP_QUICKACK, &val1, sizeof(val1)))
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }
Exit:
    return result;
}


/*
 *====================================================================
 * PURPOSE: Send bytes over a connected socket
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          IN buffer to send
 *          IN number of bytes to send
 *
 * RETURNS: NQ_FAIL or number of bytes sent
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
sySendSocket(
    SYSocketHandle sock,
    const unsigned char* buf,
    unsigned int len
    )
{
    int res;                /* operation result */

#ifdef MSG_NOSIGNAL
    res = (int)send(sock, (const char*)buf, len, MSG_NOSIGNAL);
#else
    res = (int)send(sock, (const char*)buf, len, 0);
#endif

    return (res == ERROR)? NQ_FAIL : res;
}

#ifdef UD_NS_ASYNCSEND
#error Zero buffers are not supported (UD_NS_ASYNCSEND)
#endif /* UD_NS_ASYNCSEND */

/*
 *====================================================================
 * PURPOSE: Select on sockets
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to file set
 *          IN select timeout in seconds
 *
 * RETURNS: number of sockets with data pending, zero on timeout or
 *          NQ_FAIL on error
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
sySelectSocket(
    SYSocketSet* pSet,
    NQ_UINT32 timeout
    )
{
    struct timeval tv;      /* timeout */
    int num;                /* the result of select() */

    tv.tv_sec = (time_t)timeout;
    tv.tv_usec = 0;
    num = select (FD_SETSIZE, pSet, NULL, NULL, &tv);
    return (num == ERROR)? NQ_FAIL : num;
}

/*
 *====================================================================
 * PURPOSE: Receive a UDP message
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          IN receive buffer
 *          IN buffer length
 *          OUT buffer for sender IP address in NBO
 *          OUT buffer for sender port number in NBO
 *
 * RETURNS: NQ_FAIL or number of bytes received
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
syRecvFromSocket(
    SYSocketHandle sock,
    NQ_BYTE* buf,
    NQ_COUNT len,
    NQ_IPADDRESS* ip,
    NQ_PORT* port
    )
{
    char buffer[MAX_SOCKADDR_SIZE];
    struct sockaddr *saddr = (struct sockaddr*)buffer;
    socklen_t size = sizeof(buffer);
    int res = (int)recvfrom(sock, (char*)buf, len, 0, saddr, &size);

    if (ERROR == res)
    {
        /* errno was set by Linux in this case */
        LOGERR(CM_TRC_LEVEL_ERROR, "recvfrom() was failed, error: 0x%x", syGetLastError());
        res = NQ_FAIL;
        goto Exit;
    }

    if (0 == res)
    {
        sySetLastError(NQ_ERR_BADCONNECTION);
        LOGERR(CM_TRC_LEVEL_ERROR, "Connection has been gracefully closed");
        goto Exit;
    }

    if (FALSE == parseSockaddr(saddr, ip, port))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Could not parse socket address");
        sySetLastError(NQ_ERR_SOCKETADDRESS);
        res = NQ_FAIL;
        goto Exit;
    }

Exit:
    return res;
}

/*
 *====================================================================
 * PURPOSE: Receive a UDP message from any sender
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          IN receive buffer
 *          IN buffer length
 *
 * RETURNS: NQ_FAIL or number of bytes received
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
syRecvSocket(
    SYSocketHandle sock,
    unsigned char* buf,
    unsigned int len
    )
{
    int res;                        /* operation result */

    res = (int)recv(sock, (char*)buf, len, 0);
    return (res==ERROR)? NQ_FAIL : res;
}

/*
 *====================================================================
 * PURPOSE: Receive from a datagram or a TCP stream or time out
 *--------------------------------------------------------------------
 * PARAMS:  IN socket id
 *          OUT receive buffer
 *          IN buffer length
 *          IN timeout
 *
 * RETURNS: NQ_FAIL or number of bytes received
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
syRecvSocketWithTimeout(
    SYSocketHandle sock,
    unsigned char* buf,
    unsigned int len,
    unsigned int secs
    )
{
    int res;                         /* operation result */
    fd_set socketSet;                /* for select */
    struct timeval tv;               /* timeout */

    tv.tv_sec = (time_t)secs;
    tv.tv_usec = 0;

    FD_ZERO(&socketSet);
    FD_SET(sock, &socketSet);
    res = select (FD_SETSIZE, &socketSet, NULL, NULL, &tv);
#ifdef SYOPSYST_DEBUG
    if (NQ_FAIL == res)
    {
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
    }
#endif /* SYOPSYST_DEBUG */

    if (res > 0)
    {
        res = (int)recv(sock, (char*)buf, len, 0);
#ifdef SYOPSYST_DEBUG
        if (NQ_FAIL == res)
        {
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
        }
#endif /* SYOPSYST_DEBUG */
    }

    return res;
}


/*
 *====================================================================
 * PURPOSE: Accept client socket
 *--------------------------------------------------------------------
 * PARAMS:  IN server socket id
 *          OUT buffer for sender IP address in NBO
 *          OUT buffer for sender port number in NBO
 *
 * RETURNS: new socket ID or invalid handle
 *
 * NOTES:
 *
 *====================================================================
 */

SYSocketHandle
syAcceptSocket(
    SYSocketHandle sock,
    NQ_IPADDRESS* ip,
    NQ_PORT* port
    )
{
    char buffer[MAX_SOCKADDR_SIZE];
    struct sockaddr *saddr = (struct sockaddr*)buffer;
    socklen_t size = sizeof(buffer);
    SYSocketHandle newSock = accept(sock, saddr, &size);

    parseSockaddr(saddr, ip, port);

    return newSock;
}
#ifdef CM_NQ_STORAGE
/*
 *====================================================================
 * PURPOSE: Insert GMT time to strTime as string in fmt format.
 *--------------------------------------------------------------------
 * PARAMS:  OUT time string buffer
 *             IN  buffer size
 *             IN  time
 *             IN  string format
 *
 * RETURNS: TRUE on success ,FALSE on error
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
syGmtToString(NQ_BYTE * strTime, NQ_COUNT size, NQ_UINT32 t, const NQ_CHAR * fmt)
{
    struct tm *tmp;
    char outstr[size];
    NQ_BOOL result = FALSE;
    time_t tt = (time_t)t;

    tmp = gmtime(&tt);
    if (NULL == tmp)
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }

    if(0 == strftime(outstr, sizeof(outstr), fmt, tmp))
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
        goto Exit;
    }
    syMemcpy(strTime, outstr, size);

    result = TRUE;

Exit:
    return result;
}
#endif
/*
 *====================================================================
 * PURPOSE: Open directory by name
 *--------------------------------------------------------------------
 * PARAMS:  IN directory name
 *
 * RETURNS: Directory handle or invalid handle
 *
 * NOTES:
 *
 *====================================================================
 */

SYDirectory
syOpenDirectory(
    const NQ_WCHAR* dirName
    )
{
    const char *name;
    const NQ_WCHAR root[] = {cmWChar('/'), cmWChar('\0')};
    SYDirectory dir = NULL;

    if (*dirName == cmWChar(0))
    {
        dirName = root;
    }

#ifdef UNICODEFILENAMES
    filenameToUtf8(dirName, staticData->utf8Name, sizeof(staticData->utf8Name));
    name = staticData->utf8Name;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), dirName, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    name = staticData->asciiName;
#endif /* UNICODEFILENAMES */

#ifdef DONTFOLLOW_SYMLINKS
    {
        struct stat statInfo;

        if (lstat(name, &statInfo) == -1)
        {
            goto Exit;
        }
        if (S_ISLNK(statInfo.st_mode))
        {
            errno = EACCES;
            goto Exit;
        }
    }
#endif /* DONTFOLLOW_SYMLINKS */

    dir = opendir(name);

Exit:
    return dir;
}

/*
 *====================================================================
 * PURPOSE: Open directory and read the first entry
 *--------------------------------------------------------------------
 * PARAMS:  IN directory name
 *          OUT buffer for directory handle
 *          OUT buffer for a pointer to the first file name
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syFirstDirectoryFile(
    const NQ_WCHAR* dirName,
    SYDirectory* pDir,
    const NQ_WCHAR** fileName
    )
{
    const char *name;
    const NQ_WCHAR root[] = {cmWChar('/'), cmWChar('\0')};

    *pDir = NULL;

    if (*dirName == cmWChar(0))
    {
        dirName = root;
    }

 #ifdef UNICODEFILENAMES
    filenameToUtf8(dirName, staticData->utf8Name, sizeof(staticData->utf8Name));
    name = staticData->utf8Name;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), dirName, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    name = staticData->asciiName;
#endif /* UNICODEFILENAMES */

#ifdef DONTFOLLOW_SYMLINKS
    {
        struct stat statInfo;

        if (lstat(name, &statInfo) == -1)
        {
            goto Exit;
        }
        if (S_ISLNK(statInfo.st_mode))
        {
            errno = EACCES;
            goto Exit;
        }
    }
#endif /* DONTFOLLOW_SYMLINKS */

    *pDir = opendir(name);

Exit:
    return (*pDir == NULL) ? NQ_FAIL : syNextDirectoryFile(*pDir, fileName);
}

/*
 *====================================================================
 * PURPOSE: Read next directory entry
 *--------------------------------------------------------------------
 * PARAMS:  IN directory handle
 *          OUT buffer for a pointer to the next file name
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syNextDirectoryFile(
    SYDirectory dir,
    const NQ_WCHAR** fileName
    )
{
    struct dirent* de;

    errno = OK;
    de = readdir(dir);
    if (de == NULL)
    {
        *fileName = NULL;
        return (errno == OK)? NQ_SUCCESS: NQ_FAIL;
    }
    else
    {
        static NQ_WCHAR tcharName[CM_BUFFERLENGTH(NQ_WCHAR, UD_FS_FILENAMELEN)];

#ifdef UNICODEFILENAMES
        strcpy(staticData->utf8Name, de->d_name);
        filenameFromUtf8(tcharName, sizeof(tcharName));
#else
        strcpy(staticData->asciiName, de->d_name);
        cmFsToAnsi(staticData->asciiName, sizeof(staticData->asciiName));
        syAnsiToUnicode(tcharName, staticData->asciiName);
#endif /* UNICODEFILENAMES */
        *fileName = tcharName;
        return NQ_SUCCESS;
    }
}

/*
 *====================================================================
 * PURPOSE: Open file for read
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN TRUE to deny further openings for read
 *          IN TRUE to deny further openings for execute
 *          IN TRUE to deny further openings for write
 *
 * RETURNS: file handle or invalid handle
 *
 * NOTES:
 *
 *====================================================================
 */

SYFile
syOpenFileForRead(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    return open(staticData->utf8Name, OPEN_RDONLY);
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    return open(staticData->asciiName, OPEN_RDONLY);
#endif /* UNICODEFILENAMES */
}

/*
 *====================================================================
 * PURPOSE: Open file for write
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN TRUE to deny further openings for read
 *          IN TRUE to deny further openings for execute
 *          IN TRUE to deny further openings for write
 *
 * RETURNS: file handle or invalid handle
 *
 * NOTES:
 *
 *====================================================================
 */

SYFile
syOpenFileForWrite(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    return open(staticData->utf8Name, OPEN_WRONLY);
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    return open(staticData->asciiName, OPEN_WRONLY);
#endif /* UNICODEFILENAMES */
}

/*
 *====================================================================
 * PURPOSE: Open file for read and write
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN TRUE to deny further openings for read
 *          IN TRUE to deny further openings for execute
 *          IN TRUE to deny further openings for write
 *
 * RETURNS: file handle or invalid handle
 *
 * NOTES:
 *
 *====================================================================
 */

SYFile
syOpenFileForReadWrite(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    return open(staticData->utf8Name, OPEN_RDWR);
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    return open(staticData->asciiName, OPEN_RDWR);
#endif /* UNICODEFILENAMES */
}

/*
 *====================================================================
 * PURPOSE: Delete directory
 *--------------------------------------------------------------------
 * PARAMS:  IN directory name
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syDeleteDirectory(
    const NQ_WCHAR* name
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    return rmdir(staticData->utf8Name) == OK? NQ_SUCCESS : NQ_FAIL;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    return rmdir(staticData->asciiName) == OK ? NQ_SUCCESS : NQ_FAIL;
#endif /* UNICODEFILENAMES */
}

/*
 *====================================================================
 * PURPOSE: Create new directory
 *--------------------------------------------------------------------
 * PARAMS:  IN directory name
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syCreateDirectory(
    const NQ_WCHAR* name
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    return mkdir(staticData->utf8Name, 0766) == OK ? NQ_SUCCESS : NQ_FAIL;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    return mkdir(staticData->asciiName, 0766) == OK ? NQ_SUCCESS : NQ_FAIL;
#endif /* UNICODEFILENAMES */
}

/*
 *====================================================================
 * PURPOSE: Rename a file
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN new file name
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syRenameFile(
    const NQ_WCHAR* oldName,
    const NQ_WCHAR* newName
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(newName, staticData->utf8Name, sizeof(staticData->utf8Name));
    strcpy(staticData->newName, staticData->utf8Name);
    filenameToUtf8(oldName, staticData->utf8Name, sizeof(staticData->utf8Name));
    return rename(staticData->utf8Name, staticData->newName) == OK ? NQ_SUCCESS : NQ_FAIL;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), oldName, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    syUnicodeToAnsiN(staticData->newName, sizeof(staticData->newName), newName, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->newName, sizeof(staticData->newName));
    return rename(staticData->asciiName, staticData->newName) == OK ? NQ_SUCCESS : NQ_FAIL;
#endif /* UNICODEFILENAMES */
}

/*
 *====================================================================
 * PURPOSE: Create and open new file
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *          IN TRUE to deny further openings for read
 *          IN TRUE to deny further openings for execute
 *          IN TRUE to deny further openings for write
 *
 * RETURNS: file handle or invalid handle
 *
 * NOTES:
 *
 *====================================================================
 */

SYFile
syCreateFile(
    const NQ_WCHAR* name,
    NQ_BOOL denyread,
    NQ_BOOL denyexecute,
    NQ_BOOL denywrite
    )
{
    if (!checkFileName(name))
    {
        errno = ENAMETOOLONG;
        return ERROR;
    }
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    return open(staticData->utf8Name, OPEN_RDWR_CREAT, 0700);
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    return open(staticData->asciiName, OPEN_RDWR_CREAT, 0700);
#endif /* UNICODEFILENAMES */
}

#ifdef UD_CM_LOG_TOFILE
/*
 *====================================================================
 * PURPOSE: Create and open new trace file
 *--------------------------------------------------------------------
 * PARAMS:  IN trace file name
 *
 * RETURNS: file handle or invalid handle
 *
 * NOTES:   Use a different function for traces since YNQ internal trace mechanism run in a different thread.
 *          This will lead to file corruption when using the same staticData->* variables with a different threads.
 *
 *====================================================================
 */

SYFile
syTraceCreateFile(
    const NQ_WCHAR* name
    )
{
    if (!checkFileName(name))
    {
        errno = ENAMETOOLONG;
        return ERROR;
    }
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->traceUtf8Name, sizeof(staticData->traceUtf8Name));
    return open(staticData->traceUtf8Name, OPEN_RDWR_CREAT, 0700);
#else
    syUnicodeToAnsiN(staticData->traceAsciiName, sizeof(staticData->traceAsciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->traceAsciiName, sizeof(staticData->traceAsciiName));
    return open(staticData->traceAsciiName, OPEN_RDWR_CREAT, 0700);
#endif /* UNICODEFILENAMES */
}
#endif /* UD_CM_LOG_TOFILE */

/*
 *====================================================================
 * PURPOSE: Read bytes from file
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          OUT buffer for data
 *          IN number of bytes to read
 *
 * RETURNS: number of bytes read, zero on end of file, NQ_FAIL on error
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
syReadFile(
    SYFile file,
    NQ_BYTE* buf,
    NQ_COUNT len
    )
{
    int res = (int)read(file, (char*)buf, len);
    return (res == ERROR)? NQ_FAIL : res;
}

/*
 *====================================================================
 * PURPOSE: Write bytes into file
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          IN data to write
 *          IN number of bytes to write
 *
 * RETURNS: number of bytes written, NQ_FAIL on error
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_INT
syWriteFile(
    SYFile file,
    const NQ_BYTE* buf,
    NQ_COUNT len
    )
{
    int res = (int)write(file, (char*)buf, len);
    return (res == ERROR)? NQ_FAIL : res;
}

#ifdef SY_EXTENDFILENOTSUPPORTED
/* for Linux versions where ftruncate() doesn't support extending file */

/*
 *====================================================================
 * PURPOSE: Extend file
 *--------------------------------------------------------------------
 * PARAMS:  IN file id
 *          IN number of bytes to add
 *          IN file length before
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   write zeros and position the file at the end
 *
 *====================================================================
 */

static int
extendFile(
    int file,
    int off,
    int fSize
)
{
    off_t status;          /* operation result */
    long int fillAmount;    /* number of bytes to add at once */

    /* seek to the end of the file first */

    errno = 0;
    status = lseek(file, fSize, SEEK_SET);
    if (status != fSize)
    {
        return NQ_FAIL;
    }

    /* fill file with zeroes to bring length up to 'offset' */

    while (off > 0)
    {
        if (off > (int)sizeof(zeroArray))
        {
            fillAmount = sizeof(zeroArray);
        }
        else
        {
            fillAmount = off;
        }

        errno = 0;
        status = write(file, (char*)zeroArray, (size_t)fillAmount);
        if (status != fillAmount)
        {
            return NQ_FAIL;
        }

        off = off - (int)fillAmount;
    }

    return NQ_SUCCESS;
}
#endif /* SY_EXTENDFILENOTSUPPORTED */

/*
 *====================================================================
 * PURPOSE: Position file relatively from the current position
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          IN low 32 bits of the offset
 *          IN high 32 bits of the offset
 *
 * RETURNS: new file position or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32
sySeekFileCurrent(
    SYFile file,
    NQ_INT32 off,
    NQ_INT32 offHigh
    )
{
#ifdef LONG_FILES_SUPPORT
    off_t pos = (off_t)off + ((off_t)offHigh * ((off_t)1 << 32));
    pos = lseek(file, (off_t)pos, SEEK_CUR);
#else
    off_t pos;
    pos = lseek(file, (off_t)off, SEEK_CUR);
#endif /* LONG_FILES_SUPPORT */
    return (pos == ERROR) ? (NQ_UINT32)NQ_FAIL : (NQ_UINT32)pos;
}

/*
 *====================================================================
 * PURPOSE: Position file from the beginning
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          IN low 32 bits of the offset
 *          IN high 32 bits of the offset
 *
 * RETURNS: new file position or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32
sySeekFileStart(
    SYFile file,
    NQ_UINT32 off,
    NQ_UINT32 offHigh
    )
{
#ifdef LONG_FILES_SUPPORT
    off_t pos = (off_t)off + ((off_t)offHigh * ((off_t)1 << 32));
    pos = lseek(file, (off_t)pos, SEEK_SET);
#else
    off_t pos;
    pos = lseek(file, (off_t)off, SEEK_SET);
#endif /* LONG_FILES_SUPPORT */
    return (pos == ERROR) ? (NQ_UINT32)NQ_FAIL : (NQ_UINT32)pos;
}

/*
 *====================================================================
 * PURPOSE: Position file from the end
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          IN low 32 bits of the offset
 *          IN high 32 bits of the offset
 *
 * RETURNS: new file position or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32
sySeekFileEnd(
    SYFile file,
    NQ_INT32 off,
    NQ_INT32 offHigh
    )
{
    struct stat s;
    off_t pos = ERROR;

    if (0 != fstat(file, &s))
    {
       goto Exit;
    }

#ifdef LONG_FILES_SUPPORT
    /* avoid negative resulting file offset */
    pos = s.st_size + (off_t)off + ((off_t)offHigh * ((off_t)1 << 32));
    if (pos < 0)
    {
        pos = 0;
    }
    pos = lseek(file, (off_t)pos, SEEK_SET);
#else
    /* avoid negative resulting file offset */
    off += (NQ_INT32)s.st_size;
    if (off < 0)
    {
        off = 0;
    }
    pos = lseek(file, (off_t)off, SEEK_SET);
#endif /* LONG_FILES_SUPPORT */
Exit:
    return (pos == ERROR) ? (NQ_UINT32)NQ_FAIL : (NQ_UINT32)pos;
}

/*====================================================================
 * PURPOSE: Truncate file
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *          IN low 32 bits of the offset
 *          IN high 32 bits of the offset
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_STATUS
syTruncateFile(
    SYFile file,
    NQ_UINT32 offLow,
    NQ_UINT32 offHigh
    )
{
    struct stat fileStat;
    NQ_STATUS   status;
#ifdef LONG_FILES_SUPPORT
    off64_t len = (loff_t)offLow + ((loff_t)offHigh * ((loff_t)1 << 32));
#endif

    if (ERROR == fstat(file, &fileStat))
    {
        status = NQ_FAIL;
        goto Exit;
    }

#ifdef LONG_FILES_SUPPORT
    if (ftruncate(file, (loff_t)len) == ERROR)
#else
    if (ftruncate(file, (off_t)offLow) == ERROR)
#endif
    {
        if (EIO == errno)
        {
            status = NQ_FAIL;
            goto Exit;
        }

#ifdef SY_EXTENDFILENOTSUPPORTED
        /* if ftruncate() doesn't support extending file, file can be extended by writing zeros */
        extendFile(file, (NQ_INT)offLow - (NQ_INT)fileStat.st_size, (NQ_INT)fileStat.st_size);
#endif /* SY_EXTENDFILENOTSUPPORTED */
    }

    status = NQ_SUCCESS;

Exit:
    return status;
}


/*
 *====================================================================
 * PURPOSE: find host IP by its name
 *--------------------------------------------------------------------
 * PARAMS:  IN host name
 *
 * RETURNS: host IP
 *
 * NOTES:   this function is used by client when UD_NB_INCLUDENAMESERVICE
 *          is not defined. This implementation uses DNS which is a temporary solution
 *          for vxWorks
 *====================================================================
 */

NQ_IPADDRESS4
syGetHostByName(
    const char* name
    )
{
    struct hostent* h = gethostbyname(name);
    return h != NULL ? ((struct sockaddr_in*)h->h_addr)->sin_addr.s_addr : 0;
}

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
/*
 *====================================================================
 * PURPOSE: get DNS initialization parameters
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for the default domain target belongs to
 *          OUT The DNS server address: IPv4 or IPv6
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
syGetDnsParams(
    NQ_CHAR *domain,           /* The default domain target belongs to */
    NQ_IPADDRESS *server       /* The DNS server IP address */
    )
{
    NQ_WCHAR DomainW[CM_NQ_HOSTNAMESIZE];
    NQ_WCHAR ServerW[CM_IPADDR_MAXLEN];
    NQ_CHAR  aServer[CM_IPADDR_MAXLEN];

    udGetDnsParams(DomainW, ServerW);
    syUnicodeToAnsi(domain, DomainW);
    syUnicodeToAnsi(aServer, ServerW);
    cmAsciiToIp(aServer, server);
}
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */

/*
 *====================================================================
 * PURPOSE: copy system file information into system-independent structure
 *--------------------------------------------------------------------
 * PARAMS:  IN system structure
 *          OUT independent structure
 *
 * RETURNS: NONE
 *
 * NOTES:
 *
 *====================================================================
 */

static void
statToFileInformation(
    const struct stat* from,
    const struct statvfs* fromvfs,
    SYFileInformation* to
    )
{
    NQ_TIME mTime, aTime, cTime;
    void *timeValue;

    timeValue = (void *)&from->st_mtime;
    mTime = syConvertTimeSpecToTimeInMsec(timeValue);
    timeValue = (void *)&from->st_atime;
    aTime = syConvertTimeSpecToTimeInMsec(timeValue);
    timeValue = (void *)&from->st_ctime;
    cTime = syConvertTimeSpecToTimeInMsec(timeValue);

    to->lastAccessTime = aTime;
    to->lastChangeTime = to->lastWriteTime = mTime;

    /* since POSIX does not support creation time, we set it to the least of the three file
       times */
    to->creationTime   = cTime;
    if (cmU64Cmp(&mTime, &to->creationTime) < 0)
    {
        to->creationTime = mTime;
    }
    if (cmU64Cmp(&aTime, &to->creationTime) < 0)
    {
        to->creationTime = aTime;
    }

    /* avoid zero timestamps */
    if (0 == to->creationTime.low && 0 == to->creationTime.high)
    {
        NQ_TIME now = syGetTimeInMsec();

        to->creationTime = now;
        to->lastAccessTime = now;
        to->lastChangeTime = now;
        to->lastWriteTime = now;
    }

    to->attributes = (NQ_UINT32)syUnixMode2DosAttr((int)from->st_mode);
    to->isDeleted = 0;

#ifdef LONG_FILES_SUPPORT
    to->fileIdHigh     = (NQ_UINT32)(from->st_ino >> 32);
    to->fileIdLow      = (NQ_UINT32)(from->st_ino & 0xFFFFFFFF);
#else
    to->fileIdHigh     = 0;
    to->fileIdLow      = (NQ_UINT32)from->st_ino;
#endif /* LONG_FILES_SUPPORT */

    if ((to->attributes & SY_ATTR_DIRECTORY) != 0)
    {
        to->sizeHigh       = 0;
        to->sizeLow        = 0;
        to->allocSizeHigh  = 0;
        to->allocSizeLow   = 0;
        to->numLinks       = 1;
    }
    else
    {
        to->numLinks       = 0;
#ifdef LONG_FILES_SUPPORT
        to->sizeHigh       = (NQ_UINT32)(from->st_size >> 32);
        to->sizeLow        = (NQ_UINT32)(from->st_size & 0xFFFFFFFF);
        to->allocSizeHigh  = (NQ_UINT32)((from->st_blocks * 512) >> 32);        /* standard block size for UNIX file system */
        to->allocSizeLow   = (NQ_UINT32)((from->st_blocks * 512) & 0xFFFFFFFF); /* standard block size for UNIX file system */
#else
        to->sizeHigh       = 0;
        to->sizeLow        = (NQ_UINT32)from->st_size;
        to->allocSizeHigh  = 0;
        to->allocSizeLow   = (NQ_UINT32)(from->st_blocks * 512); /* standard block size for UNIX file system */
#endif /* LONG_FILES_SUPPORT */
    }
}

/*
 *====================================================================
 * PURPOSE: Close file
 *--------------------------------------------------------------------
 * PARAMS:  IN file handle
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syCloseFile(
    SYFile fd
    )
{
    return close(fd) != ERROR ? NQ_SUCCESS : NQ_FAIL;
}

/*
 *====================================================================
 * PURPOSE: Close directory
 *--------------------------------------------------------------------
 * PARAMS:  IN directory handle
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syCloseDirectory(
    SYDirectory dir
    )
{
    return closedir(dir) != ERROR ? NQ_SUCCESS : NQ_FAIL;
}

/*
 *====================================================================
 * PURPOSE: Delete file
 *--------------------------------------------------------------------
 * PARAMS:  IN file name
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *
 *====================================================================
 */

NQ_STATUS
syDeleteFile(
    const NQ_WCHAR* name
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->utf8Name, sizeof(staticData->utf8Name));
    return unlink(staticData->utf8Name) == OK ? NQ_SUCCESS : NQ_FAIL;
#else
    syUnicodeToAnsiN(staticData->asciiName, sizeof(staticData->asciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->asciiName, sizeof(staticData->asciiName));
    return unlink(staticData->asciiName) == OK ? NQ_SUCCESS : NQ_FAIL;
#endif /* UNICODEFILENAMES */
}

#ifdef UD_CM_LOG_TOFILE
/*
 *====================================================================
 * PURPOSE: Delete trace file
 *--------------------------------------------------------------------
 * PARAMS:  IN trace file name
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   Use a different function for traces since YNQ internal trace mechanism run in a different thread.
 *          This will lead to file corruption when using the same staticData->* variables with a different threads.
 *
 *====================================================================
 */

NQ_STATUS
syTraceDeleteFile(
    const NQ_WCHAR* name
    )
{
#ifdef UNICODEFILENAMES
    filenameToUtf8(name, staticData->traceUtf8Name, sizeof(staticData->traceUtf8Name));
    return unlink(staticData->traceUtf8Name) == OK ? NQ_SUCCESS : NQ_FAIL;
#else
    syUnicodeToAnsiN(staticData->traceAsciiName, sizeof(staticData->traceAsciiName), name, CM_IGNORE_LENGTH);
    cmAnsiToFs(staticData->traceAsciiName, sizeof(staticData->traceAsciiName));
    return unlink(staticData->traceAsciiName) == OK ? NQ_SUCCESS : NQ_FAIL;
#endif /* UNICODEFILENAMES */
}
#endif /* UD_CM_LOG_TOFILE */

#ifdef UD_CS_INCLUDEDIRECTTRANSFER

/*
 *====================================================================
 * PURPOSE: Start fragmented packet
 *--------------------------------------------------------------------
 * PARAMS:  IN socket handle
 *
 * RETURNS: NQ_FAIL when this operation is not available
 *          NQ_SUCCESS when operation succeeded
 *
 * NOTES:   This function removes TCP_NODELAY and sets TCP_CORK on socket
 *
 *====================================================================
 */

NQ_STATUS
syDtStartPacket(
    SYSocketHandle sock
    )
{
    int opt;
    NQ_STATUS result = NQ_SUCCESS;

    /* unset TCP_NODELAY, so that data is buffered until there is a sufficient amount to send out */
    opt = 0;
    if ( -1 == setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char*)&opt, sizeof(opt)) )
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock);
#endif /* SYOPSYST_DEBUG */
        result = NQ_FAIL;
    }

    /* set TCP_CORK, useful for prepending headers before calling sendfile() */
    opt = 1;
    if (-1 == setsockopt(sock, IPPROTO_TCP, TCP_CORK, (char*)&opt, sizeof(opt)))
    {
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock);
#endif /* SYOPSYST_DEBUG */
        result = NQ_FAIL;
    }
    return result;
}

/*
 *====================================================================
 * PURPOSE: End fragmented packet
 *--------------------------------------------------------------------
 * PARAMS:  IN socket handle
 *
 * RETURNS: NONE
 *
 * NOTES:   This function sets TCP_NODELAY and remove TCP_CORK on socket
 *
 *====================================================================
 */

void
syDtEndPacket(
    SYSocketHandle sock
    )
{
    int opt;
    int res;

    opt = 1;
    res = setsockopt(sock, IPPROTO_TCP, TCP_NODELAY, (char*)&opt, sizeof(opt));
    if (ERROR == res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = IPPROTO_TCP, __optname = TCP_NODELAY");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock);
#endif /* SYOPSYST_DEBUG */
    }

    opt = 0;
    res = setsockopt(sock, IPPROTO_TCP, TCP_CORK, (char*)&opt, sizeof(opt));
    if (ERROR == res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = IPPROTO_TCP, __optname = TCP_CORK");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock);
#endif /* SYOPSYST_DEBUG */
    }
}

/*
    DT setup:
    The following code is a sample implementation. Real implementation may vary from platform to
    platform.
    For Z-copy we can use 1) sendfile() 2) revcfile() 3) splice() and 4) something proprietary
    Since recvfile() is not available in most linux platforms, this example uses splice() and sendfile().
    Notice that some 2.6.x kernels (2.6.31 in particular) have a buggy splice() that hangs up on heavy load.
*/

#define USE_DT_READ               /* use DT for reads: sendfile() if available or simulate */
#define USE_DT_WRITE              /* use DT for writes: splice() if available or simulate */
#define SPLICE_WRITE_AVAILABLE    /* splice() function is available on the target platform (for write ) */
#define SENDFILE_READ_AVAILABLE   /* sendfile() from file to socket is available on the target platform (for read) */


#if defined(SPLICE_WRITE_AVAILABLE)
/*
Fedora does not properly export "splice", remove this on a clean platform
*/
#define SPLICE_F_MOVE        1       /* Move pages instead of copying.  */
#define SPLICE_F_NONBLOCK    2       /* Don't block on the pipe splicing */
#define SPLICE_F_MORE        4       /* Expect more data.  */
#define SPLICE_F_GIFT        8       /* Pages passed in are a gift.  */
/*long splice(int fd_in, off_t *off_in, int fd_out, off_t *off_out, size_t len, unsigned int flags);*/

static size_t nqSplice(int from, int to, size_t len)
{
    ssize_t bytes, sent, in_pipe;
    size_t total_sent = 0;
    size_t result = (size_t)NQ_FAIL;

    while (total_sent < len)
    {
        /* workaround for kernel 2.6.31 hanging on first splice() call, but seems to happen on later kernels as well */
        if ((sent = splice(from, NULL, staticData->pipe[1], NULL, (len - total_sent) < 16384 ? len - total_sent : 16384, SPLICE_F_MORE | SPLICE_F_MOVE)) <= 0)
        /* if ((sent = splice(from, NULL, staticData->pipe[1], NULL, len - total_sent, SPLICE_F_MORE | SPLICE_F_MOVE)) <= 0) */
        {
            if (errno == EINTR || errno == EAGAIN)
            {
                continue;
            }

#ifdef SYOPSYST_DEBUG
            syFprintf(stderr, "[%s:%d][%s()] %d %s, from:%d, to:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), from, to);
#endif /* SYOPSYST_DEBUG */
            goto Exit;
        }
        in_pipe = sent;
        while (in_pipe > 0)
        {
            if ((bytes = splice(staticData->pipe[0], NULL, to, NULL, (size_t)in_pipe, SPLICE_F_MORE | SPLICE_F_MOVE)) <= 0)
            {
                if (errno == EINTR || errno == EAGAIN)
                {
                    continue;
                }

#ifdef SYOPSYST_DEBUG
                syFprintf(stderr, "[%s:%d][%s()] %d %s, from:%d, to:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), from, to);
#endif /* SYOPSYST_DEBUG */
                goto Exit;
            }
            in_pipe -= bytes;
        }
        total_sent += (size_t)sent;
    }
    result = total_sent;

Exit:
#ifdef SYOPSYST_DEBUG
    if (total_sent != len)
    {
        syFprintf(stderr, "[%s:%d][%s()] data remains, from:%d, to:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, from, to);
        /* means that data remains unread, socket needs to be drained */
    }
    else
    {
        syFprintf(stderr, "[%s:%d][%s()] from:%d, to:%d, send:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, from, to, len);
    }
#endif /* SYOPSYST_DEBUG */
    return result;
}
#endif /* defined(USE_DT_WRITE) && defined(SPLICE_WRITE_AVAILABLE) */

#if defined(SENDFILE_READ_AVAILABLE)
static size_t nqSendFile(int from, int to, size_t len)
{
    size_t res = (size_t)sendfile(to, from, NULL, len);
#ifdef SYOPSYST_DEBUG
    if (NQ_FAIL == res)
    {
        syFprintf(stderr, "[%s:%d][%s()] %d %s from:%d to:%d len:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), from, to, len);
    }
    else
    {
        syFprintf(stderr, "[%s:%d][%s()] from:%d, to:%d, requested to send:%d sent:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, from, to, len, res);
    }
#endif /* SYOPSYST_DEBUG */
    return res;
}
#endif /* defined(SENDFILE_READ_AVAILABLE) */

/*
 *====================================================================
 * PURPOSE: Transfer bytes from socket to file
 *--------------------------------------------------------------------
 * PARAMS:  IN socket handle
 *          IN file handle
 *          IN/OUT number of bytes to transfer/number of bytes transferred
 *
 * RETURNS: NQ_FAIL on error or NQ_SUCCESS when operation succeeded
 *
 * NOTES:   This function uses splice() or simulation code
 *
 *====================================================================
 */

NQ_STATUS
syDtFromSocket(
    SYSocketHandle sock,
    SYFile file,
    NQ_COUNT * len
    )
{
    NQ_STATUS result = NQ_FAIL;

#if defined(USE_DT_WRITE)
#if defined(SPLICE_WRITE_AVAILABLE)
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Using DT for writing, fileID: %d", file);
    *len = (NQ_COUNT)nqSplice(sock, file, (size_t)*len);
    result = (ERROR == *len) ? NQ_FAIL : NQ_SUCCESS;
#else  /* simulation */
    {
        static NQ_BYTE buf[UD_NS_BUFFERSIZE];
        NQ_COUNT cnt1, cnt2, total = 0;

        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Using DT simulation for writing, fileID: %d", file);
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] USING DT SIMULATION \n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock, file);
#endif /* SYOPSYST_DEBUG */
        while (*len > 0)
        {
            cnt1 = (NQ_COUNT)recv(sock, (char*)buf, *len, 0);
            if (cnt1 == NQ_FAIL)
            {
#ifdef SYOPSYST_DEBUG
                syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d, file:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock, file);
#endif /* SYOPSYST_DEBUG */
                goto Exit;
            }
#ifdef UD_FS_USEFILESTREAMS
            cnt2 = (NQ_COUNT)fwrite((char*)buf, 1, cnt1, file);
#else
            cnt2 = (NQ_COUNT)write(file, (char*)buf, cnt1);
#endif /* UD_FS_USEFILESTREAMS */
            if (cnt2 == NQ_FAIL || cnt2 != cnt1)
            {
#ifdef SYOPSYST_DEBUG
                syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d, file:%d, cnt1:%u, cnt2:%u\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock, file, cnt1, cnt2);
#endif /* SYOPSYST_DEBUG */
                goto Exit;
            }
            *len -= cnt2;
            total += cnt2;
        }
        *len = total;
        result = NQ_SUCCESS;
    }
Exit:
#endif /* simulation */
#endif /* defined(USE_DT_WRITE) */

    return result;
}


/*
 *====================================================================
 * PURPOSE: Transfer bytes from file to socket
 *--------------------------------------------------------------------
 * PARAMS:  IN socket handle
 *          IN file handle
 *          IN/OUT number of bytes to transfer/number of bytes transferred
 *
 * RETURNS: NQ_FAIL on error or NQ_SUCCESS when operation succeeded
 *
 * NOTES:   This function uses sendfile() or simulation code
 *
 *====================================================================
 */

NQ_STATUS                   /* NQ_FAIL on error or NQ_SUCCESS when operation succeeded */
syDtToSocket(
    SYSocketHandle sock,    /* socket handle */
    SYFile file,            /* file handle */
    NQ_COUNT * len          /* IN number of bytes to transfer, OUT bytes transferred */
    )
{
    NQ_STATUS result = NQ_FAIL;

#if defined(USE_DT_READ)
#if defined(SENDFILE_READ_AVAILABLE)
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Using DT for reading, fileID: %d", file);
    *len = (NQ_COUNT)nqSendFile(file, sock, *len);
    result = (ERROR == *len) ? NQ_FAIL : NQ_SUCCESS;
#else /* simulation */
    {
        static NQ_BYTE buf[UD_NS_BUFFERSIZE];

        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Using DT simulation for reading, fileID: %d", file);
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] USING DT SIMULATION \n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock, file);
#endif /* SYOPSYST_DEBUG */
        *len = read(file, (char*)buf, *len);
        if (*len==NQ_FAIL)
        {
            syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d, file:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock, file);
            goto Exit;
        }

        *len = send(sock, (char*)buf, *len, 0);
#ifdef SYOPSYST_DEBUG
        if (*len == NQ_FAIL)
        {
            syFprintf(stderr, "[%s:%d][%s()] %d %s, socket:%d, file:%d\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno), sock, file);
        }
#endif /* SYOPSYST_DEBUG */

        result = (NQ_FAIL == *len) ? NQ_FAIL : NQ_SUCCESS;
    }

Exit:
#endif /* simulation */
#endif /* defined(USE_DT_READ) */
    return result;
}

#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

#if defined(UD_CC_INCLUDELDAP) || defined(UD_NQ_CODEPAGEUTF8)

/* Convert Unicode UTF-16LE string to UTF8 */
void
syUnicodeToUTF8N(
    NQ_CHAR *outStr,
    NQ_UINT outLength,
    const NQ_WCHAR *inWStr,
    NQ_UINT inLength
    )
{
    iconv_t convertor = (iconv_t)-1;
    size_t inBytesLeft;
    size_t outBytesLeft;
    char *in = (char *)inWStr;
    char *out = outStr;
    NQ_UINT i, beforeConversionOutBytes;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL,"outStr:%p outLength:%d inWStr:%p inLength:%d",outStr, outLength, inWStr, inLength);

    if (inLength != CM_IGNORE_LENGTH)
    {
        inBytesLeft = inLength;
        for (i = 0; i < (inLength / sizeof(NQ_WCHAR)); i++)
        {
            if (0 == inWStr[i])
            {
                inBytesLeft = (i + CM_TRAILING_NULL) * sizeof(NQ_WCHAR);
                break;
            }
        }
    }
    else
    {
        inBytesLeft = (syWStrlen(inWStr) + CM_TRAILING_NULL) * sizeof(NQ_WCHAR);
    }

    if (outLength != CM_IGNORE_LENGTH)
    {
        outBytesLeft = outLength;
    }
    else
    {
        outBytesLeft = inBytesLeft / sizeof(NQ_WCHAR);
    }

    beforeConversionOutBytes = (NQ_UINT)outBytesLeft;
    *outStr = 0;
    if ((convertor = iconv_open("UTF-8", "UTF-16LE")) == (iconv_t)-1)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to open converter (UTF-16LE to UTF-8)");
        sySetLastError(NQ_ERR_ICONVOPEN);
        goto Exit;
    }

    if (iconv(convertor, &in, &inBytesLeft, &out, &outBytesLeft) == (size_t)-1)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to convert UTF-16LE to UTF-8");
        sySetLastError(NQ_ERR_ICONVERROR);
    }

    /* if there is no place left for trailing zero overwrite the last character */
    if (outBytesLeft < sizeof(NQ_CHAR))
    {
        outStr[beforeConversionOutBytes - 1] = 0;
    }
    else
    {
        outStr[beforeConversionOutBytes - outBytesLeft] = 0;
    }

Exit:
    iconv_close(convertor);
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
}

void
syUTF8ToUnicodeN(
    NQ_WCHAR *outWStr,
    NQ_UINT outLength,
    const NQ_CHAR *inStr,
    NQ_UINT inLength
    )
{
    iconv_t convertor = (iconv_t)-1;
    size_t inBytesLeft;
    size_t outBytesLeft;
    char *in = (char *)inStr;
    char *out = (char *)outWStr;
    NQ_UINT i, beforeConversionOutBytes;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL,"outWStr:%p outLength:%d inStr:%p inLength:%d",outWStr, outLength, inStr, inLength);

    if (inLength != CM_IGNORE_LENGTH)
    {
        inBytesLeft = inLength;
        for (i = 0; i < inLength; i++)
        {
            if (0 == inStr[i])
            {
                inBytesLeft = i + CM_TRAILING_NULL;
                break;
            }
        }
    }
    else
    {
        inBytesLeft = strlen(inStr) + CM_TRAILING_NULL;
    }

    if (outLength != CM_IGNORE_LENGTH)
    {
        outBytesLeft = outLength;
    }
    else
    {
        outBytesLeft = inBytesLeft * sizeof(NQ_WCHAR);
    }

    beforeConversionOutBytes = (NQ_UINT)outBytesLeft;
    *outWStr = cmWChar(0);
    if ((convertor = iconv_open("UTF-16LE", "UTF-8")) == (iconv_t)-1)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to open converter (UTF-8 to UTF-16LE)");
        sySetLastError(NQ_ERR_ICONVOPEN);
        goto Exit;
    }

    if (iconv(convertor, &in, &inBytesLeft, &out, &outBytesLeft) == (size_t)-1)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to convert UTF-8 to UTF-16LE");
        sySetLastError(NQ_ERR_ICONVERROR);
    }

    /* if there is no place left for trailing zero overwrite the last character */
    if (outBytesLeft < sizeof(NQ_WCHAR))
    {
        outWStr[(beforeConversionOutBytes / sizeof(NQ_WCHAR)) - 1] = 0;
    }
    else
    {
        outWStr[(beforeConversionOutBytes - outBytesLeft) / sizeof(NQ_WCHAR)] = 0;
    }

Exit:
    iconv_close(convertor);
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
}
#endif /* defined(UD_CC_INCLUDELDAP) || defined(UD_NQ_CODEPAGEUTF8) */

#ifdef MUTEX_DEBUG /* debug mutex issues. */
void syMutexDelete(SYMutex* _m)
{
    /* will assert on many client operations, as we destroy item while its being locked */
    /* if (_m->lockCounter > 0)
    {
        syAssert(FALSE);
    }
    */
    pthread_mutex_destroy(&_m->mutex);
}

void syMutexTakeDebug(SYMutex *_m, const NQ_CHAR *text, const NQ_UINT line)
{
    if (_m)
    {
        pthread_mutex_lock(&_m->mutex);
        ++_m->lockCounter;
        if (_m->lockCounter == 4)
        {
            syAssert(FALSE);
        }
        syStrncpy(_m->who[_m->lockCounter - 1], text, 119);
        _m->where[_m->lockCounter - 1] = line;
    }
}


void syMutexGive(SYMutex* _m)
{
    --_m->lockCounter;

    pthread_mutex_unlock(&_m->mutex);

    if (_m->lockCounter < 0)
    {
        syAssert(FALSE);
    }
    syMemset(&_m->who[_m->lockCounter], 0, 120);
    _m->where[_m->lockCounter] = 0;
}

#endif /* MUTEX_DEBUG */

void
syMutexCreate(SYMutex* _m)
{
    pthread_mutexattr_t attr;

    pthread_mutexattr_init(&attr);
    pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE);

#ifdef MUTEX_DEBUG
    LOGERR(CM_TRC_LEVEL_MESS_NORMAL,"Create mutex: %p", _m);
    _m->lockCounter = 0;
    syMemset(&_m->where,  0, sizeof(_m->where));
    syMemset(&_m->who,  0, sizeof(_m->who));
    pthread_mutex_init(&_m->mutex, &attr);
#else
    pthread_mutex_init(_m, &attr);
#endif /* MUTEX_DEBUG */
    pthread_mutexattr_destroy(&attr);
}

NQ_STATUS sySendMulticast(
    SYSocketHandle socket,
    const NQ_BYTE * buffer,
    NQ_COUNT length,
    const NQ_IPADDRESS *ip,
    NQ_PORT port)
{
    NQ_STATUS res;          /* operation result */
    struct ip_mreq mreg;

    mreg.imr_multiaddr.s_addr = (in_addr_t)CM_IPADDR_GET4(*ip);
    mreg.imr_interface.s_addr = INADDR_ANY;

    if (ERROR == setsockopt (socket, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreg, sizeof(mreg)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = IPPROTO_IP, __optname = IP_ADD_MEMBERSHIP");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
    }

    res = sySendToSocket(socket, buffer, length, ip, port);
    if (ERROR == setsockopt (socket, IPPROTO_IP, IP_DROP_MEMBERSHIP, &mreg, sizeof(mreg)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = IPPROTO_IP, __optname = IP_DROP_MEMBERSHIP");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
    }

    return res;
}

void sySubscribeToMulticast(SYSocketHandle socket,
        const NQ_IPADDRESS *ip
        )
{
    struct ip_mreq mreg;

    mreg.imr_multiaddr.s_addr = (in_addr_t)CM_IPADDR_GET4(*ip);
    mreg.imr_interface.s_addr = INADDR_ANY;
    if (ERROR == setsockopt (socket, IPPROTO_IP, IP_ADD_MEMBERSHIP, &mreg, sizeof(mreg)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to set socket options __level = IPPROTO_IP, __optname = IP_ADD_MEMBERSHIP");
#ifdef SYOPSYST_DEBUG
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
#endif /* SYOPSYST_DEBUG */
    }
}

#ifdef SY_SEMAPHORE_AVAILABLE
NQ_STATUS sySemaphoreCreate(SYSemaphore* semId, NQ_UINT count)
{
    NQ_INT res = NQ_SUCCESS;

    if (NQ_SUCCESS != sem_init(semId, 0, count))
    {
        res = NQ_FAIL;
    }

    return res;
}

void sySemaphoreResetCounter(SYSemaphore *pSemID)
{
    NQ_INT lockCount = 0 , i = 0;

    sem_getvalue(pSemID, &lockCount);
    for (i = 0; i < lockCount ; i++)
    {
        sem_wait(pSemID);
    }
}

NQ_INT sySemaphoreTimedTake(SYSemaphore *sem , NQ_INT timeout)
{
    NQ_INT res = NQ_SUCCESS;
    struct timespec semTimeout;

    semTimeout.tv_sec = (time_t)syGetTimeInSec();
    semTimeout.tv_sec += timeout;
    semTimeout.tv_nsec = 0;

    while (sem_timedwait(sem, &semTimeout))
    {
        NQ_INT semErr = errno;

        if (ETIMEDOUT != semErr)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "semaphore error: %d. %s", semErr, strerror(semErr));
            if (EINTR == semErr)
            {
                /* interrupted by a signal, retry the wait */
                continue;
            }
        }
        res = NQ_FAIL;
        break;
    }

    return res;
}
#endif /* SY_SEMAPHORE_AVAILABLE */

void syThreadStart(NQ_BOOL isRT, NQ_INT priorityLevel, SYThread *taskIdPtr, void (*startpoint)(void), NQ_BOOL background)
{
    pthread_attr_t attr;
    struct sched_param param;
#ifdef SYOPSYST_DEBUG
    NQ_INT res;
#endif /* SYOPSYST_DEBUG */

    pthread_attr_init(&attr);

    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
#ifdef SYOPSYST_DEBUG
    res =
#endif /* SYOPSYST_DEBUG */
    pthread_create(taskIdPtr, &attr, (void * (*)(void *))startpoint, NULL);
#ifdef SYOPSYST_DEBUG
    if (0 != res)
    {
        syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
    }
#endif /* SYOPSYST_DEBUG */

    if (isRT)
    {
        switch (priorityLevel)
        {
            case NQ_THREAD_PRIORITY_TRANSPORT_MAIN:
                param.sched_priority = 10;
                break;
            case NQ_THREAD_PRIORITY_TRANSPORT_WORK:
                param.sched_priority = 30;
                break;
            default:
                param.sched_priority = 55;
        }

#ifdef SYOPSYST_DEBUG
        res =
#endif /* SYOPSYST_DEBUG */
        pthread_setschedparam(*taskIdPtr, SCHED_FIFO, &param);
#ifdef SYOPSYST_DEBUG
        if (0 != res)
        {
            syFprintf(stderr, "[%s:%d][%s()] %d %s\n", SY_LOG_FILE, SY_LOG_LINE, SY_LOG_FUNCTION, errno, strerror(errno));
        }
#endif /* SYOPSYST_DEBUG */
    }

    pthread_attr_destroy(&attr);
}

#ifdef UD_NQ_CODEPAGEUTF8

/* check whether conversion is available on the platform */
NQ_BOOL initCodePageUTF8()
{
    iconv_t convertor;
    NQ_BOOL result = FALSE;

    convertor = iconv_open("UTF-8", "UTF-16LE");
    if (convertor == (iconv_t)-1)
    {
        if (errno == EINVAL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "conversion from UTF-16LE to UTF-8 is not available");
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "failed to open conversion from UTF-16LE to UTF-8");
        }
        goto Exit;
    }
    iconv_close(convertor);

    convertor = iconv_open("UTF-16LE", "UTF-8");
    if (convertor == (iconv_t)-1)
    {
        if (errno == EINVAL)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "conversion from UTF-8 to UTF-16LE is not available");
        }
        else
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "failed to open conversion from UTF-8 to UTF-16LE");
        }
        goto Exit;
    }
    iconv_close(convertor);

    result = TRUE;
Exit:
    return result;
}

#endif /* UD_NQ_CODEPAGEUTF8 */



