/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Pool of sockets for internal communications with
 *                 Name Daemon and Datagram Daemon
 *--------------------------------------------------------------------
 * MODULE        : Network
 * DEPENDENCIES  :
 ********************************************************************/

#include "nsinsock.h"

/*
 NS module communicates with localhost Name and Datagram Daemons over a socket.
 Since NS is reenterant, using the same socket may significantly decrease performance.
 To avoid this problem we use a pool of pre-connected sockets.

 The pool uses preallocated socket slots. It is controlled by an array of "free socket"
 pointers. This array is a cyclical array of pointers to free socket slots. Two indexes
 (of the 1-st and of the last cell) "roll around" in this array.

 Access to the buffer pool is protected by a mutex. The overflow condition is controlled
 by a binary semaphore.
 */

/*
    Static data & functions
    -----------------------
 */

/* Initialize an internal socket */

static NQ_STATUS                    /* NQ_SUCCESS or NQ_FAIL */
initInternalSocketHandle(
    SYSocketHandle *socket          /* pointer to socket handle */
    );

#ifdef SY_INTERNALSOCKETPOOL
static void
releaseDaemonInternalSockets(
    InternalSocket *socketSlots,
    NQ_UINT numOfSlots
    );
#endif /* SY_INTERNALSOCKETPOOL */

typedef struct
{
    InternalSocket   slotsND[UD_NS_NUMNDCHANNELS];   /* NS socket pool */
    InternalSocket*  freeND[UD_NS_NUMNDCHANNELS];    /* free slot pointers */
    NQ_INT firstFreeND;             /* Index of the pointer to the 1st free ND socket */
    NQ_INT lastFreeND;              /* Index of the pointer to the last free ND socket */
    SYMutex      poolGuardND;       /* Mutex for exclusive access to data */
#ifdef SY_SEMAPHORE_AVAILABLE
    SYSemaphore  overflowGuardND;   /* Binary semaphore for resolving pool
                                       overflow. If pool is empty, a task waits
                                       for this semaphore until another task
                                       releases a buffer. */
#endif /* SY_SEMAPHORE_AVAILABLE */
    InternalSocket   slotsDD[UD_NS_NUMDDCHANNELS];   /* DD socket pool */
    InternalSocket*  freeDD[UD_NS_NUMDDCHANNELS];    /* Array of pointers to free DD sockets */
    NQ_INT firstFreeDD;             /* Index of the pointer to the 1st free DD socket */
    NQ_INT lastFreeDD;              /* Index of the pointer to the last free DD socket */
    SYMutex      poolGuardDD;       /* Mutex for exclusive access to data */
#ifdef SY_SEMAPHORE_AVAILABLE
    SYSemaphore  overflowGuardDD;   /* Binary semaphore for resolving pool
                                       overflow. If pool is empty, a task waits
                                       for this semaphore until another task
                                       releases a buffer. */
#endif /* SY_SEMAPHORE_AVAILABLE */
}
StaticData;

static NQ_BOOL isModuleInitialized = FALSE;
static NQ_COUNT moduleInitCount = 0;
#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/*
 *====================================================================
 * PURPOSE: Initialize the pool
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   1) create sockets
 *          2) bind each of them to a dynamic port and any available IP
 *====================================================================
 */

NQ_STATUS
nsInitInternalSockets(
    void
    )
{
    NQ_INDEX i;       /* just a counter */
    NQ_STATUS result = NQ_SUCCESS;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        ++moduleInitCount;
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        result = NQ_FAIL;
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate internal sockets");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    syMutexCreate(&staticData->poolGuardND);
    syMutexCreate(&staticData->poolGuardDD);

#ifdef SY_SEMAPHORE_AVAILABLE
    if (sySemaphoreCreate(&staticData->overflowGuardND, UD_NS_NUMNDCHANNELS) != NQ_SUCCESS)
    {
        result = NQ_FAIL;
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create overflowGuardND Semaphore");
        sySetLastError(NQ_ERR_NOMEM);
        goto Error1;
    }

    if (sySemaphoreCreate(&staticData->overflowGuardDD, UD_NS_NUMDDCHANNELS) != NQ_SUCCESS)
    {
        result = NQ_FAIL;
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create overflowGuardDD Semaphore");
        sySetLastError(NQ_ERR_NOMEM);
        goto Error2;
    }
#endif /* SY_SEMAPHORE_AVAILABLE */

    syMutexTake(&staticData->poolGuardND);
    syMutexTake(&staticData->poolGuardDD);

    staticData->firstFreeND = 0;
    staticData->lastFreeND  = -1;
    staticData->firstFreeDD = 0;
    staticData->lastFreeDD  = -1;

    /* Name Daemon sockets */
    for (i = 0; i < UD_NS_NUMNDCHANNELS; i++)
    {
#ifdef SY_INTERNALSOCKETPOOL
        staticData->slotsND[i].socket = syInvalidSocket();
        if (initInternalSocketHandle(&staticData->slotsND[i].socket) == NQ_FAIL)
        {
            releaseDaemonInternalSockets(staticData->slotsND, UD_NS_NUMNDCHANNELS);
            result = NQ_FAIL;
            goto Error3;
        }
#endif /* SY_INTERNALSOCKETPOOL */
        staticData->freeND[i] = &staticData->slotsND[i];
        staticData->slotsND[i].idx = i;
    }

    /* Datagram Daemon sockets */
    for (i = 0; i < UD_NS_NUMDDCHANNELS; i++)
    {
#ifdef SY_INTERNALSOCKETPOOL
        staticData->slotsDD[i].socket = syInvalidSocket();
        if (initInternalSocketHandle(&staticData->slotsDD[i].socket) == NQ_FAIL)
        {
            releaseDaemonInternalSockets(staticData->slotsDD, UD_NS_NUMDDCHANNELS);
            result = NQ_FAIL;
            goto Error3;
        }
#endif /* SY_INTERNALSOCKETPOOL */
        staticData->freeDD[i] = &staticData->slotsDD[i];
        staticData->slotsDD[i].idx = i;
    }

    isModuleInitialized = TRUE;
    syMutexGive(&staticData->poolGuardDD);
    syMutexGive(&staticData->poolGuardND);
    ++moduleInitCount;
    goto Exit;

#ifdef SY_INTERNALSOCKETPOOL
Error3:
    syMutexGive(&staticData->poolGuardDD);
    syMutexGive(&staticData->poolGuardND);
#ifdef SY_SEMAPHORE_AVAILABLE
    sySemaphoreDelete(staticData->overflowGuardDD);
#endif /* SY_SEMAPHORE_AVAILABLE */
#endif /* SY_INTERNALSOCKETPOOL */
#ifdef SY_SEMAPHORE_AVAILABLE
Error2:
    sySemaphoreDelete(staticData->overflowGuardND);
Error1:
    syMutexDelete(&staticData->poolGuardND);
    syMutexDelete(&staticData->poolGuardDD);
#endif /* SY_SEMAPHORE_AVAILABLE */
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Clean up the pool
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:   1) close sockets
 *          2) delete semaphores
 *====================================================================
 */

NQ_STATUS
nsExitInternalSockets(
    void
    )
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized && (--moduleInitCount == 0))
    {
        syMutexTake(&staticData->poolGuardND);
        syMutexTake(&staticData->poolGuardDD);

        /* Name Daemon sockets */
#ifdef SY_INTERNALSOCKETPOOL
        {
            NQ_INT i;       /* just a counter */

            for (i=0; i<UD_NS_NUMNDCHANNELS; i++)
            {
                if (syIsValidSocket(staticData->slotsND[i].socket))
                {
                    syCloseSocket(staticData->slotsND[i].socket);
                }
            }
        }
#endif
        /* Datagram Daemon sockets */
#ifdef SY_INTERNALSOCKETPOOL
        {
            NQ_INT i;       /* just a counter */

            for (i=0; i<UD_NS_NUMDDCHANNELS; i++)
            {
                if (syIsValidSocket(staticData->slotsDD[i].socket))
                {
                    syCloseSocket(staticData->slotsDD[i].socket);
                }
            }
        }
#endif

        syMutexGive(&staticData->poolGuardND);
        syMutexGive(&staticData->poolGuardDD);
        syMutexDelete(&staticData->poolGuardND);
        syMutexDelete(&staticData->poolGuardDD);
#ifdef SY_SEMAPHORE_AVAILABLE
        sySemaphoreDelete(staticData->overflowGuardND);
        sySemaphoreDelete(staticData->overflowGuardDD);
#endif /* SY_SEMAPHORE_AVAILABLE */

        /* release memory */
#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
        {
            cmMemoryFreeShutdown(staticData);
        }

        staticData = NULL;
#endif /* SY_FORCEALLOCATION */
        isModuleInitialized = FALSE;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", NQ_SUCCESS);
    return NQ_SUCCESS;
}

/*
 *====================================================================
 * PURPOSE: Get an ND socket from the pool
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: socket pointer or NULL on failure
 *====================================================================
 */

InternalSocket*
getInternalSocketND(
    void
    )
{
    InternalSocket* socket = NULL;  /* pointer to return */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

#ifdef SY_SEMAPHORE_AVAILABLE
    sySemaphoreTake(staticData->overflowGuardND);
#endif /* SY_SEMAPHORE_AVAILABLE */
    syMutexTake(&staticData->poolGuardND);

    socket = staticData->freeND[staticData->firstFreeND];       /* get the 1st */
    staticData->firstFreeND++;                      /* shift to next */
    staticData->firstFreeND %= UD_NS_NUMNDCHANNELS; /* wrap around */

    syMutexGive(&staticData->poolGuardND);

#ifndef SY_INTERNALSOCKETPOOL
    if (initInternalSocketHandle(&(socket->socket))==NQ_FAIL)
    {
        socket = NULL;
    }
#endif
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", socket);
    return socket;
}

/*
 *====================================================================
 * PURPOSE: Return an internal ND socket
 *--------------------------------------------------------------------
 * PARAMS:  Pointer to the socket to return
 *
 * RETURNS: none
 *====================================================================
 */

void
putInternalSocketND(
    InternalSocket* socket
    )
{
#ifndef SY_INTERNALSOCKETPOOL
    syCloseSocket(socket->socket);
#endif

    syMutexTake(&staticData->poolGuardND);

    staticData->lastFreeND++;                       /* if was -1 will start from index 0 */
    staticData->lastFreeND %= UD_NS_NUMNDCHANNELS;  /* wrap around */
    staticData->freeND[staticData->lastFreeND] = socket;        /* write a free slot pointer */

    syMutexGive(&staticData->poolGuardND);

#ifdef SY_SEMAPHORE_AVAILABLE
    sySemaphoreGive(staticData->overflowGuardND);
#endif /* SY_SEMAPHORE_AVAILABLE */
}


/*
 *====================================================================
 * PURPOSE: Get an DD socket from the pool
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: socket pointer or NULL on failure
 *====================================================================
 */

InternalSocket*
getInternalSocketDD(
    void
    )
{
    InternalSocket* socket;  /* pointer to return */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

#ifdef SY_SEMAPHORE_AVAILABLE
    sySemaphoreTake(staticData->overflowGuardDD);
#endif /* SY_SEMAPHORE_AVAILABLE */
    syMutexTake(&staticData->poolGuardDD);

    socket = staticData->freeDD[staticData->firstFreeDD];               /* get 1st free */
    staticData->firstFreeDD++;                              /* shift to the next */
    staticData->firstFreeDD %= UD_NS_NUMDDCHANNELS;         /* wrap around */

    syMutexGive(&staticData->poolGuardDD);

#ifndef SY_INTERNALSOCKETPOOL
    if (initInternalSocketHandle(&(socket->socket))==NQ_FAIL)
    {
        socket = NULL;
    }
#endif

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", socket);
    return socket;
}

/*
 *====================================================================
 * PURPOSE: Return an internal DD socket
 *--------------------------------------------------------------------
 * PARAMS:  Pointer to the socket to return
 *
 * RETURNS: none
 *====================================================================
 */

void
putInternalSocketDD(
    InternalSocket* socket
    )
{
#ifndef SY_INTERNALSOCKETPOOL
    syCloseSocket(socket->socket);
#endif

    syMutexTake(&staticData->poolGuardDD);

    staticData->lastFreeDD++;                       /* if was -1 will start from index 0 */
    staticData->lastFreeDD %= UD_NS_NUMDDCHANNELS;  /* wrap around */
    staticData->freeDD[staticData->lastFreeDD] = socket;        /* write a free slot pointer */

    syMutexGive(&staticData->poolGuardDD);

#ifdef SY_SEMAPHORE_AVAILABLE
    sySemaphoreGive(staticData->overflowGuardDD);
#endif /* SY_SEMAPHORE_AVAILABLE */
}

/*
 *====================================================================
 * PURPOSE: Init an internal socket
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT Pointer to the socket handle
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *====================================================================
 */

static NQ_STATUS
initInternalSocketHandle(
    SYSocketHandle *socket
    )
{
    NQ_PORT port;      /* dynamic port number */
    NQ_IPADDRESS ip;   /* assigned IP */
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket:%p", socket);

    *socket = syCreateSocket(FALSE, CM_IPADDR_IPV4);    /* datagram socket */
    if(!syIsValidSocket(*socket))       /* error */
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create internal communication socket");
        sySetLastError(NQ_ERR_SOCKETCREATE);
        goto Exit;
    }

    if (syBindSocket(*socket, cmSelfipGetLocalHostIp(), 0, TRUE) == NQ_FAIL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to bind internal communication socket");
        sySetLastError(NQ_ERR_SOCKETBIND);
        goto Error;
    }

    syGetSocketPortAndIP(*socket, &ip, &port);
    if (port == 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to get internal communication socket's port");
        sySetLastError(NQ_ERR_SOCKETNAME);
        goto Error;
    }

    result = NQ_SUCCESS;
    goto Exit;

Error:
    syCloseSocket(*socket);
    *socket = syInvalidSocket();

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

#ifdef SY_INTERNALSOCKETPOOL
/*
 *====================================================================
 * PURPOSE: release the internal daemon sockets slots
 *--------------------------------------------------------------------
 * PARAMS:  IN Pointer to the sockets slot
 *          IN Number of sockets in the slot
 *
 * RETURNS:
 *====================================================================
 */

static void
releaseDaemonInternalSockets(
    InternalSocket *socketSlots,
    NQ_UINT numOfSlots
    )
{
    NQ_INT i;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "sockets slots:%p , number of slots:%d", socketSlots, numOfSlots);

    for (i = 0 ; i < (NQ_INT)numOfSlots ; i++)
    {
        if (syInvalidSocket() != socketSlots[i].socket)
        {
            syCloseSocket(socketSlots[i].socket);
            socketSlots[i].socket = syInvalidSocket();
        }
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}
#endif /* SY_INTERNALSOCKETPOOL */


