/*************************************************************************
* Copyright (c) 2021 by Visuality Systems, Ltd.
*
*                     All Rights Reserved
*
* This item is the property of Visuality Systems, Ltd., and contains
* confidential, proprietary, and trade-secret information. It may not
* be transferred from the custody or control of Visuality Systems, Ltd.,
* except as expressly authorized in writing by an officer of Visuality
* Systems, Ltd. Neither this item nor the information it contains may
* be used, transferred, reproduced, published, or disclosed, in whole
* or in part, and directly or indirectly, except as expressly authorized
* by an officer of Visuality Systems, Ltd., pursuant to written agreement.
*
*************************************************************************
* FILE NAME     : $Workfile:$
* ID            : $Header:$
* REVISION      : $Revision:$
*--------------------------------------------------------------------
* DESCRIPTION   : Common storage operations
*--------------------------------------------------------------------
* MODULE        : Common
* DEPENDENCIES  :
*************************************************************************/

#ifndef _CMSTORAGE_H_
#define _CMSTORAGE_H_

#include "cmapi.h"
#include "cmrepository.h"

#define STORAGE_IS_TREE     0x0010
#define STORAGE_IS_LIMITED  0x0100 /* used for SMB1 only */

typedef struct _cmstorageitem{
    NQ_UINT64               itemIndex;
    NQ_BOOL                 isUsed;
    NQ_BOOL                 isFindable;
    struct _cmstorageitem * next;
}CMStorageItem;

typedef struct{
    NQ_UINT32       index;          /* list index */
    void *          items;          /* item holder - doesn't change */
    void *          freeItems;      /* list of currently free items */
    void *          lastFreeItem;   /* pointer to the last free item in the 'freeItems' list */
    SYMutex         guard;          /* list guard */
    NQ_UINT16       usedItems;      /* number of used items in this list */
}CMStorageList;

typedef NQ_BOOL (* cmStorageItemInitCallback)(void* item);
typedef NQ_BOOL (* cmStorageItemDisposeCallback)(void* item);

typedef struct{
    NQ_UINT         itemSize;
    CMStorageList * list;
    NQ_UINT         numOfLists;
    NQ_UINT         numOfListItems;
    NQ_UINT32       numOfUsedItems;
    NQ_BOOL         reserveFirstList;
    NQ_BOOL         extend;
    NQ_BOOL         waitingOnCond;
    CMThreadCond    cond;
    SYMutex         guard;
    cmStorageItemInitCallback itemInitCallBack;
    cmStorageItemDisposeCallback itemDisposeCallBack;
#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(NQDEBUG) && !defined(UD_NQ_DYNAMICALLOCATION))
    NQ_COUNT        numFailedItems;
    NQ_CHAR         *listName;
#endif /* #if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(NQDEBUG) && !defined(UD_NQ_DYNAMICALLOCATION)) */
#ifdef REPOSITORY_SIZE_TUNING
    NQ_COUNT        initialNumLists;
    NQ_COUNT        finalNumLists;
#endif /* REPOSITORY_SIZE_TUNING */
}CMStorageData;

typedef struct{
    CMStorageData * pData;
    NQ_UINT         listNumber;
    NQ_UINT         itemNumber;
    SYMutex         itrGuard;
}CMStorageIterator;

NQ_BOOL cmStorageListInit(CMStorageData * pData, NQ_UINT itemSize, NQ_UINT numOfLists, NQ_UINT numOfListItems,
                          NQ_BOOL reserveFirstList, NQ_CHAR* listName, cmStorageItemInitCallback itemInitCallback, cmStorageItemDisposeCallback itemDisposeCallBack);
void cmStorageListShutdown(CMStorageData * pData);
CMStorageItem * cmStorageItemGetNew(CMStorageData * pData, NQ_INT condition);
CMStorageItem * cmStorageItemFind(CMStorageData * pData, NQ_UINT row, NQ_UINT cell, NQ_BOOL isFindAll);
void cmStorageItemRemove(CMStorageData * pData, NQ_UINT row, NQ_UINT cell);

void cmStorageIteratorStart(CMStorageData * pData, CMStorageIterator * Itr);
NQ_BOOL cmStorageIteratorHasNext(CMStorageIterator * Itr);
CMStorageItem * cmStorageIteratorNext(CMStorageIterator * Itr);
void cmStorageIteratorTerminate(CMStorageIterator * Itr);

typedef struct{
    CMItem  item;
    void *  storeItem;
}CMHashItem;

#define HASH_N  17

typedef NQ_UINT32(*CMStorageHashFunction)(const void * value);

typedef struct{
    CMList                  list[HASH_N];
    CMStorageHashFunction   doHash;
    NQ_BOOL                 isUsed;
}CMStorageHashList;

NQ_BOOL cmStorageHashListStart(CMStorageHashList * pHList, CMStorageHashFunction doHash);
void cmStorageHashListShutdown(CMStorageHashList * pHList);
void cmStorageHashListDispose(CMStorageHashList * pHList);
CMList * cmStorageHashFindList(CMStorageHashList * pHList, const void * value);

NQ_UINT32 cmStorageNameHash(const void * name);

#endif /* _CMSTORAGE_H_ */
