/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common part of common library
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmapi.h"
#include "nsapi.h"

static
NQ_INT
parseDigit(
    NQ_CHAR ch
    );

static
NQ_STATUS
cmAsciiToIp4(
    NQ_CHAR *str,
    NQ_IPADDRESS *ip
   );

static
NQ_STATUS
cmIpToAscii4(
    NQ_CHAR *ascii,
    NQ_IPADDRESS4 ip
   );

#ifdef UD_NQ_USETRANSPORTIPV6

static
NQ_STATUS
cmAsciiToIp6(
    NQ_CHAR *str,
    NQ_IPADDRESS *ip
   );

static
NQ_STATUS
cmIpToAscii6(
    NQ_CHAR *ascii,
    const NQ_IPADDRESS *ip
   );

#endif /* UD_NQ_USETRANSPORTIPV6 */

NQ_STATUS
cmAsciiToIp(
    NQ_CHAR *ascii,
    NQ_IPADDRESS *ip
   )
{
    if (cmAsciiToIp4(ascii, ip) == NQ_SUCCESS)
    {
       return NQ_SUCCESS;
    }

#ifdef UD_NQ_USETRANSPORTIPV6
    if (cmAsciiToIp6(ascii, ip) == NQ_SUCCESS)
    {
        return NQ_SUCCESS;
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */

    return NQ_FAIL;
}

NQ_STATUS
cmWcharToIp(
    NQ_WCHAR *wchar,
    NQ_IPADDRESS *ip
)
{
    static NQ_CHAR ascii[256];
    cmUnicodeToAnsi(ascii, wchar);

    return cmAsciiToIp(ascii, ip);
}

NQ_STATUS
cmIpToAscii(
    NQ_CHAR *ascii,
    const NQ_IPADDRESS *ip
   )
{
    *ascii = '\0';
#ifdef UD_NQ_USETRANSPORTIPV6
    switch (CM_IPADDR_VERSION(*ip))
    {
        case CM_IPADDR_IPV4:
#endif /* UD_NQ_USETRANSPORTIPV6 */
            return cmIpToAscii4(ascii, CM_IPADDR_GET4(*ip));

#ifdef UD_NQ_USETRANSPORTIPV6
        case CM_IPADDR_IPV6:
            return cmIpToAscii6(ascii, ip);
#ifdef UD_NQ_USETRANSPORTIPVR
        case CM_IPADDR_IPVR:
        {
/*          NQ_IPADDRESS ipvr;

            if (CM_IPADDR_IPV6 == ip->addr.vr.version)
                CM_IPADDR_ASSIGN6(ipvr, ip->addr.vr.addr.v6)
            else
                CM_IPADDR_ASSIGN4(ipvr, ip->addr.vr.addr.v4)
            return cmIpToAscii(ascii, (const NQ_IPADDRESS *)&ipvr);*/
            return NQ_SUCCESS;
        }
#endif /* UD_NQ_USETRANSPORTIPVR */
        default:
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Invalid ip version: %d", ip->version);
            *ascii = '\0';  /* cleanup */
            return NQ_FAIL;
    }
#endif /* UD_NQ_USETRANSPORTIPV6 */
}

NQ_CHAR * cmIPDump(const NQ_IPADDRESS *ip)
{
    static NQ_CHAR temp[CM_IPADDR_MAXLEN];

    if (NULL != ip)
    {
        cmIpToAscii(temp, ip);
    }
    else
    {
        temp[0] = '\0';
    }

    return temp;
}

NQ_CHAR * cmIPDumpV4(NQ_IPADDRESS4 ip)
{
    static NQ_CHAR temp[CM_IPADDR_MAXLEN];

    cmIpToAscii4(temp, ip);
    return temp;
}

static
NQ_INT
parseDigit(
    NQ_CHAR ch
    )
{
    switch (ch)
    {
    case '0':
    case '1':
    case '2':
    case '3':
    case '4':
    case '5':
    case '6':
    case '7':
    case '8':
    case '9':
        return ch - '0';

#ifdef UD_NQ_USETRANSPORTIPV6
    case 'a':
    case 'b':
    case 'c':
    case 'd':
    case 'e':
    case 'f':
        return ch - 'a' + 10;

    case 'A':
    case 'B':
    case 'C':
    case 'D':
    case 'E':
    case 'F':
        return ch - 'A' + 10;
#endif /* UD_NQ_USETRANSPORTIPV6 */

    default:
        return -1;
    }
}

static
NQ_STATUS
cmAsciiToIp4(
    NQ_CHAR *str,
    NQ_IPADDRESS *ip
   )
{
    NQ_BYTE result[4];
    NQ_BYTE *tmp = result;
    NQ_INT counter, digit, num;
    NQ_IPADDRESS4 *resultIp;
    NQ_STATUS res = NQ_FAIL;

    if (!str || *str == '.')
        goto Exit;

    counter = num = 0;

    while (*str)
    {
        if (*str == '.')
        {
            *tmp++ = (NQ_BYTE)num;
            num = 0;
            counter++;
            str++;
            if (counter > 3 || *str == 0 || *str == '.')
                goto Exit;

            continue;
        }

        digit = parseDigit(*str++);

        if (digit < 0 || digit > 9)
            goto Exit;

        num = num * 10 + digit;

        if (num > 255)
            goto Exit;
    }

    *tmp = (NQ_BYTE)num;
    counter++;
    if (counter < 4)
        goto Exit;

    /* result is already in BIG ENDIAN, no need to convert to the network order */
    resultIp = (NQ_IPADDRESS4*)result;
    CM_IPADDR_ASSIGN4(*ip, *resultIp);
    res = NQ_SUCCESS;

Exit:
    return res;
}

static
NQ_STATUS
cmIpToAscii4(
    NQ_CHAR *ascii,
    NQ_IPADDRESS4 ip
   )
{
    NQ_UINT32 i = syNtoh32((NQ_UINT)ip);

    sySprintf(
        ascii,
        "%u.%u.%u.%u",
        (NQ_INT)((i >> 24) & 0xff),
        (NQ_INT)((i >> 16) & 0xff),
        (NQ_INT)((i >> 8) & 0xff),
        (NQ_INT)(i & 0xff));

    return NQ_SUCCESS;
}

NQ_BOOL cmIsIPv6Literal(NQ_WCHAR *name)
{
    NQ_WCHAR ipv6LiteralName[] = {cmWChar('.'), cmWChar('i'), cmWChar('p'), cmWChar('v'), cmWChar('6'), cmWChar('-'), cmWChar('l'),
            cmWChar('i'), cmWChar('t'), cmWChar('e'), cmWChar('r'), cmWChar('a'), cmWChar('l'), cmWChar('.'), cmWChar('n'),
            cmWChar('e'), cmWChar('t'), cmWChar('\0')};
    NQ_BOOL res = FALSE;

    /* check whether address is ipv6-literal.net */
    if (cmWStrincmp(name + syWStrlen(name) - syWStrlen(ipv6LiteralName), ipv6LiteralName, (NQ_COUNT)(syWStrlen(ipv6LiteralName))) == 0)
        res = TRUE;

    return res;
}

const NQ_WCHAR * cmIPCShareName(void)
{
    static const NQ_WCHAR ipcName[] = {cmWChar('I'), cmWChar('P'), cmWChar('C'), cmWChar('$'), cmWChar(0)};
    return ipcName;
}

#ifdef UD_NQ_USETRANSPORTIPV6

#define NQ_IPADDRESS6_WORDS sizeof(NQ_IPADDRESS6)/2

static
NQ_STATUS
cmAsciiToIp6(
    NQ_CHAR *str,
    NQ_IPADDRESS *ip
   )
{
    NQ_IPADDRESS6 result = { 0, 0, 0, 0, 0, 0, 0, 0 };
    NQ_IPADDRESS ip4;
    NQ_UINT16 *tmp;
    NQ_CHAR *last, *input;
    NQ_BOOL mixed;
    NQ_INT zeros, zeroslen, counter, num, limit, digit, i;
    NQ_CHAR ipv6LiteralName[] = ".IPV6-LITERAL.NET";
    NQ_STATUS res = NQ_FAIL;
    NQ_CHAR temp[40];

    /* check whether address is ipv6-literal.net */
    input = str;
    if (syStrlen(str) > syStrlen(ipv6LiteralName) && 
        cmAStrincmp(str + syStrlen(str) - syStrlen(ipv6LiteralName), ipv6LiteralName, (NQ_COUNT)(syStrlen(ipv6LiteralName))) == 0)
    {
        NQ_CHAR *p;
        NQ_COUNT i;

        for (i = 0, p = str; *p != '.'; p++, i++)
        {
            switch (*p)
            {
                case '-':
                    temp[i] = ':';
                    break;
                case 'S':
                case 's':
                    temp[i] = '%';
                    break;
                default:
                    temp[i] = *p;
                    break;                   
            }
        }
        temp[i] = '\0';
        input = temp;
    }

    tmp = result;
    last = NULL;
    zeros = -1;
    counter = num = 0;
    mixed = syStrchr(input, '.') != NULL;
    limit = mixed ? 7 : 8 ;

    while (*input)
    {
        if (*input == ':' && input[1] == ':')
        {
            last = ++input;

            if (zeros >= 0)
            {
                goto Exit;
            }

            zeros = counter + 1;
        }

        if (*input == ':')
        {
            last = input;
            *tmp++ = syHton16((NQ_UINT16)num);
            num = 0;
            counter++;
            input++;
            if (counter > (limit - 1) || *input == 0)
            {
                goto Exit;
            }

            continue;
        }

        if (*input == '.' || *input == '%')
        {
            break;
        }

        digit = parseDigit(*input++);
        if (digit < 0)
        {
            goto Exit;
        }

        num = (num << 4) + digit;
        if (num > 0xffff)
        {
            goto Exit;
        }
    }

    *tmp++ = syHton16((NQ_UINT16)num);
    counter++;

    if (counter < limit && zeros < 0)
    {
        goto Exit;
    }

    if (counter == limit && zeros >= 0)
    {
        goto Exit;
    }

    if (zeros >= 0)
    {
        zeroslen = limit - counter;

        for (i = limit - 1; i >= (zeros + zeroslen); i--)
        {
            result[i] = result[i - zeroslen];
            result[i - zeroslen] = 0;
        }
    }

    if (mixed)
    {
        if (cmAsciiToIp4(last + 1, &ip4) == NQ_FAIL)
        {
            goto Exit;
        }

        syMemcpy(&result[6], &ip4.addr.v4, sizeof(NQ_IPADDRESS4));
    }

    CM_IPADDR_ASSIGN6(*ip, *(NQ_IPADDRESS6*)result);
    res = NQ_SUCCESS;

Exit:
    return res;
}

static
NQ_STATUS
cmIpToAscii6(
    NQ_CHAR *ascii,
    const NQ_IPADDRESS *ip
   )
{
    NQ_IPADDRESS6 ip6;
    NQ_INT zeros, zeroslen, i, j;
    NQ_CHAR str[8];

    syMemcpy(ip6, ip->addr.v6, sizeof(NQ_IPADDRESS6));
    *ascii = 0;
    zeros = -1;
    zeroslen = 1;
    for (i = 0; i < (NQ_INT)NQ_IPADDRESS6_WORDS; i = j + 1)
    {
        for (j = i; j < (NQ_INT)NQ_IPADDRESS6_WORDS && !ip6[j]; j++);

        if ((j - i) > zeroslen)
        {
            zeros = i;
            zeroslen = j - i;
        }
    }

    for (i = 0; i < (NQ_INT)NQ_IPADDRESS6_WORDS;)
    {
        if (i == zeros)
        {
            syStrcat(ascii, i ? ":" : "::");
            i += zeroslen;
            continue;
        }

        sySprintf(str, "%x", syNtoh16(ip6[i]));
        if (++i < (NQ_INT)NQ_IPADDRESS6_WORDS)
        {
            syStrcat(str, ":");
        }

        syStrcat(ascii, str);
    }

    return NQ_SUCCESS;
}

#endif /* UD_NQ_USETRANSPORTIPV6 */

#ifdef UD_NQ_USEIOVECS /*UD_NQ_USEIOVECS*/

NQ_IOVecPosition cmIovecSkipBytes(NQ_IOVecPosition *buffer, NQ_INT numBytes)
{
    NQ_INT offset = (NQ_INT)buffer->byteOffset;
    NQ_INT segment = (NQ_INT)buffer->segment;

    /* validation */
    ASSERT_IOBUFPOSPTR(buffer);

    /* move forward*/
    if (numBytes >= 0)
    {
        while (offset + numBytes > buffer->ioBuf->iovec[segment].iov_len)
        {
            numBytes -= ((NQ_INT)buffer->ioBuf->iovec[segment].iov_len - offset);
            if (++segment >= buffer->ioBuf->iovSegmentCnt)
            {
                if (numBytes > 0)
                    syAssert(FALSE);
                /* legal state. finished move and at the end of iovec buffer. */
            }
            offset = 0;
        }
        offset += numBytes;
        if (offset == buffer->ioBuf->iovec[segment].iov_len && segment + 1 < buffer->ioBuf->iovSegmentCnt)
        {
            ++segment;
            offset = 0;
        }
    }
    else
    {
        /* skip value is negative - move backwards */
        numBytes *= -1;

        while (offset - numBytes < 0)
        {
            if (--segment < 0)
            {
                goto Error;
            }
            numBytes -= offset;
            offset = (NQ_INT)buffer->ioBuf->iovec[segment].iov_len;
        }
        offset -= numBytes;
    }
    buffer->segment = (NQ_COUNT)segment;
    buffer->byteOffset = (NQ_COUNT)offset;

    goto Exit;

Error:
    LOGERR(CM_TRC_LEVEL_ERROR, "buffer over flow. bad skip value: %d", numBytes);
    syAssert(FALSE);
Exit:
    return *buffer;
}

NQ_COUNT cmIovecNumByteDiff(const NQ_IOVecPosition *base, const NQ_IOVecPosition *curr)
{
    NQ_COUNT result = (NQ_COUNT)NQ_FAIL;

    syAssert(base->ioBuf == curr->ioBuf);

    if (base->ioBuf != curr->ioBuf ||
        base->segment > curr->segment ||
        (base->segment == curr->segment && base->byteOffset > curr->byteOffset))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Can't calculate diff. base buf: %p, curr buf: %p, base seg: %d, curr seg: %d, base offset: %d. curr offset: %d",
                base->ioBuf, curr->ioBuf, base->segment,curr->segment, base->byteOffset, curr->byteOffset);
        goto Exit;
    }

    if (base->segment == curr->segment)
    {
        result = curr->byteOffset - base->byteOffset;
    }
    else
    {
        NQ_COUNT segment = base->segment + 1;
        /* first add 2 "tails" = non full segments */
        result = curr->byteOffset + ((base->ioBuf->iovec[base->segment].iov_len - base->byteOffset));
        /* add all segment lengths */
        for (;segment < curr->segment; ++segment)
        {
            result += base->ioBuf->iovec[segment].iov_len;
        }
    }

Exit:
    return result;

}

void* cmIovecMemcpyVec2Flat(NQ_BYTE* to, NQ_IOVecPosition fromIOBuf, NQ_COUNT numBytes)
{
    void* result = to;
    NQ_COUNT segment = fromIOBuf.segment;
    NQ_COUNT byteOffset = fromIOBuf.byteOffset;

    while (numBytes > 0)
    {
        if (numBytes >= (NQ_COUNT)sizeof(NQ_UINT32) && byteOffset + (NQ_COUNT)sizeof(NQ_UINT32) < fromIOBuf.ioBuf->iovec[segment].iov_len)
        {
            *((NQ_UINT32 *)to) = *((NQ_UINT32 *)(fromIOBuf.ioBuf->iovec[segment].iov_base + byteOffset));
            /* copy 4 bytes per iteration */
            to += (NQ_COUNT)sizeof(NQ_UINT32);
            byteOffset += (NQ_COUNT)sizeof(NQ_UINT32);
            numBytes -= (NQ_COUNT)sizeof(NQ_UINT32);
        }
        else
        {
            /* copy byte per iteration */
            *to++ = *(fromIOBuf.ioBuf->iovec[segment].iov_base + byteOffset);

            if (--numBytes == 0)
                break;

            /* now move in iovec */
            if (++byteOffset >= fromIOBuf.ioBuf->iovec[segment].iov_len)
            {
                if (segment + 1 >= fromIOBuf.ioBuf->iovSegmentCnt)
                {
                    result = NULL;
                    goto Exit;
                }
                else
                {
                    ++segment;
                    byteOffset = 0;
                }
            }
        }
    }

Exit:
    return result;
}

NQ_IOVecPosition cmIovecMemcpyFlat2Vec(NQ_IOVecPosition *toIOBuf, const NQ_BYTE* from, NQ_COUNT numBytes)
{
    NQ_IOVecPosition result;
    NQ_COUNT segment, byteOffset;
    segment = toIOBuf->segment;
    byteOffset = toIOBuf->byteOffset;

    IOBUF_POSINIT(result);

    while(numBytes > 0)
    {
        if (numBytes >= (NQ_COUNT)sizeof(NQ_UINT32) && byteOffset + (NQ_COUNT)sizeof(NQ_UINT32) < toIOBuf->ioBuf->iovec[segment].iov_len)
        {
            *((NQ_UINT32 *)(toIOBuf->ioBuf->iovec[segment].iov_base + byteOffset)) = *((NQ_UINT32 *)from);
            from += sizeof(NQ_UINT32);
            byteOffset += (NQ_COUNT)sizeof(NQ_UINT32);
            numBytes -= (NQ_COUNT)sizeof(NQ_UINT32);
        }
        else
        {
            *(toIOBuf->ioBuf->iovec[segment].iov_base + byteOffset) = *from++;

            if (--numBytes == 0)
                break;

            /* now move in iovec */
            if (++byteOffset >= toIOBuf->ioBuf->iovec[segment].iov_len)
            {
                if (segment + 1 >= toIOBuf->ioBuf->iovSegmentCnt)
                {
                    goto Exit;
                }
                else
                {
                    ++segment;
                    byteOffset = 0;
                }
            }
        }
    }
    result = *toIOBuf;

Exit:
    return result;
}

NQ_IOVecPosition cmIovecMemcpyVec2Vec(NQ_IOVecPosition *toIOBuf, const NQ_IOVecPosition* fromIOBuf, NQ_COUNT numBytes)
{
    NQ_IOVecPosition result;
    NQ_COUNT segmentTo, offsetTo, segmentFrom, offsetFrom;
    segmentTo = toIOBuf->segment;
    offsetTo = toIOBuf->byteOffset;
    segmentFrom = fromIOBuf->segment;
    offsetFrom = fromIOBuf->byteOffset;

    IOBUF_POSINIT(result)

    while (numBytes > 0)
    {
        if (numBytes > (NQ_COUNT)sizeof(NQ_UINT32) &&
            offsetTo + (NQ_COUNT)sizeof(NQ_UINT32) < toIOBuf->ioBuf->iovec[segmentTo].iov_len &&
            offsetFrom + (NQ_COUNT)sizeof(NQ_UINT32) < fromIOBuf->ioBuf->iovec[segmentFrom].iov_len)
        {
            /* copy 4 bytes .*/
            *((NQ_UINT32 *)(toIOBuf->ioBuf->iovec[segmentTo].iov_base + offsetTo)) = *((NQ_UINT32 *)(fromIOBuf->ioBuf->iovec[segmentFrom].iov_base + offsetFrom));
            offsetFrom += (NQ_COUNT)sizeof(NQ_UINT32);
            offsetTo += (NQ_COUNT)sizeof(NQ_UINT32);
            numBytes -= (NQ_COUNT)sizeof(NQ_UINT32);
        }
        else
        {
            /* copy byte */
            *((toIOBuf->ioBuf->iovec[segmentTo].iov_base + offsetTo)) = *((fromIOBuf->ioBuf->iovec[segmentFrom].iov_base + offsetFrom));

            if (--numBytes == 0)
                break;

            /* now move in iovec */
            if (++offsetTo >= toIOBuf->ioBuf->iovec[segmentTo].iov_len)
            {
                if (segmentTo + 1 >= toIOBuf->ioBuf->iovSegmentCnt)
                {
                    goto Exit;
                }
                else
                {
                    ++segmentTo ;
                    offsetTo = 0;
                }
            }
            if (++offsetFrom >= fromIOBuf->ioBuf->iovec[segmentFrom].iov_len)
            {
                if (segmentFrom + 1 >= fromIOBuf->ioBuf->iovSegmentCnt)
                {
                    goto Exit;
                }
                else
                {
                    ++segmentFrom;
                    offsetFrom = 0;
                }
            }
        }
    }

    result = *toIOBuf;

Exit:
    return result;
}

NQ_IOVecPosition cmIovecMemcpyVec2VecBackwards(NQ_IOVecPosition *toIOBuf, const NQ_IOVecPosition* fromIOBuf, NQ_COUNT numBytes)
{
    NQ_IOVecPosition result;
    NQ_INT segmentTo, offsetTo, segmentFrom, offsetFrom;
    segmentTo = (NQ_INT)toIOBuf->segment;
    offsetTo = (NQ_INT)toIOBuf->byteOffset;
    segmentFrom = (NQ_INT)fromIOBuf->segment;
    offsetFrom = (NQ_INT)fromIOBuf->byteOffset;

    IOBUF_POSINIT(result)

    while (numBytes > 0)
    {
        if (numBytes > (NQ_COUNT)sizeof(NQ_UINT32) &&
            offsetTo > (NQ_COUNT)sizeof(NQ_UINT32) &&
            offsetFrom > (NQ_COUNT)sizeof(NQ_UINT32))
        {
            /* move backwards */
            offsetFrom -= (NQ_INT)sizeof(NQ_UINT32);
            offsetTo -= (NQ_INT)sizeof(NQ_UINT32);
            /* copy 4 bytes .*/
            *((NQ_UINT32 *)(toIOBuf->ioBuf->iovec[segmentTo].iov_base + offsetTo)) = *((NQ_UINT32 *)(fromIOBuf->ioBuf->iovec[segmentFrom].iov_base + offsetFrom));
            numBytes -= (NQ_COUNT)sizeof(NQ_UINT32);
        }
        else
        {
            /* move backwards */
            if (--offsetTo < 0)
            {
                if (--segmentTo < 0)
                {
                    goto Exit;
                }
                offsetTo = (NQ_INT)toIOBuf->ioBuf->iovec[segmentTo].iov_len;
            }

            if (--offsetFrom < 0)
            {
                if (--segmentFrom < 0)
                {
                    goto Exit;
                }
                offsetFrom = (NQ_INT)fromIOBuf->ioBuf->iovec[segmentFrom].iov_len;
            }

            /* copy byte */
            *((toIOBuf->ioBuf->iovec[segmentTo].iov_base + offsetTo)) = *((fromIOBuf->ioBuf->iovec[segmentFrom].iov_base + offsetFrom));
            --numBytes;
        }
    }

    result = *toIOBuf;

Exit:
    return result;
}

NQ_INT cmIovecMemcmpVec2Flat(const NQ_IOVecPosition *IOBuf, const NQ_BYTE* buff, NQ_COUNT numBytes)
{
    NQ_COUNT segment, offset, numCmpBytes;
    NQ_INT result = 0;
    segment = IOBuf->segment;
    offset = IOBuf->byteOffset;

    while(numBytes > 0)
    {
        /* choose the smallest fragment to compare. */
        numCmpBytes = numBytes <= (IOBuf->ioBuf->iovec[segment].iov_len - offset)? numBytes :
                     (IOBuf->ioBuf->iovec[segment].iov_len - offset);

        result = syMemcmp((NQ_BYTE *)(IOBuf->ioBuf->iovec[segment].iov_base + offset),
                          buff,
                          numCmpBytes);

        if (result)
        {
            /* not equal */
            goto Exit;
        }

        offset += numCmpBytes;
        numBytes -= numCmpBytes;

        if (offset == IOBuf->ioBuf->iovec[segment].iov_len)
        {
            if (0 == numBytes)
                goto Exit;

            if (++segment >= IOBuf->ioBuf->iovSegmentCnt)
            {
                /* non zero failure code*/
                result = -1;
                goto Exit;
            }
            offset = 0;
        }
    }

Exit:
    return result;
}

NQ_INT cmIovecMemcmpVec2Vec(const NQ_IOVecPosition *toIOBuf, const NQ_IOVecPosition* fromIOBuf, NQ_COUNT numBytes)
{
    NQ_COUNT segmentTo, offsetTo, segmentFrom, offsetFrom, numCmpBytes;
    NQ_INT result = 0;
    segmentTo = toIOBuf->segment;
    offsetTo = toIOBuf->byteOffset;
    segmentFrom = toIOBuf->segment;
    offsetFrom = toIOBuf->byteOffset;

    for (;numBytes > 0;)
    {
        /* choose the smallest fragment to compater. */
        numCmpBytes = numBytes <= (toIOBuf->ioBuf->iovec[segmentTo].iov_len - offsetTo)? numBytes :
                     (toIOBuf->ioBuf->iovec[segmentTo].iov_len - offsetTo);
        numCmpBytes = numCmpBytes <= (fromIOBuf->ioBuf->iovec[segmentFrom].iov_len - offsetFrom)? numCmpBytes :
                      (fromIOBuf->ioBuf->iovec[segmentFrom].iov_len - offsetFrom);
        result = syMemcmp((toIOBuf->ioBuf->iovec[segmentTo].iov_base + offsetTo),
                     (fromIOBuf->ioBuf->iovec[segmentFrom].iov_base + offsetFrom),
                     numCmpBytes);

        if (result)
        {
            /* not equal */
            goto Exit;
        }

        offsetTo += numCmpBytes;
        if (offsetTo == toIOBuf->ioBuf->iovec[segmentTo].iov_len)
        {
            if (++segmentTo >= toIOBuf->ioBuf->iovSegmentCnt)
            {
                /* non zero failure code*/
                return -1;
            }
            offsetTo = 0;
        }
        offsetFrom += numCmpBytes;
        if (offsetFrom == toIOBuf->ioBuf->iovec[segmentFrom].iov_len)
        {
            if (++segmentFrom >= fromIOBuf->ioBuf->iovSegmentCnt)
            {
                /* non zero failure code*/
                return -1;
            }
            offsetFrom = 0;
        }
        numBytes -= numCmpBytes;
    }

Exit:
    return result;
}

NQ_BOOL cmIovecMemset(NQ_IOVecPosition *IOBuf, NQ_INT val, NQ_COUNT numBytes)
{
    NQ_COUNT segment, offset, numSetBytes;
    NQ_BOOL result = TRUE;
    segment = IOBuf->segment;
    offset = IOBuf->byteOffset;

    while(numBytes > 0)
    {
        /* choose the smallest fragment to compater. */
        numSetBytes = numBytes <= (IOBuf->ioBuf->iovec[segment].iov_len - offset)? numBytes :
                     (IOBuf->ioBuf->iovec[segment].iov_len - offset);

        syMemset((IOBuf->ioBuf->iovec[segment].iov_base + offset),
                          val,
                          numSetBytes);

        offset += numSetBytes;
        if (offset == IOBuf->ioBuf->iovec[segment].iov_len)
        {
            if (++segment >= IOBuf->ioBuf->iovSegmentCnt)
            {

                result = FALSE;
                goto Exit;
            }
            offset = 0;
        }
        numBytes -= numSetBytes;
    }

Exit:
    return result;
}

NQ_COUNT cmIOStrlen(NQ_IOVecPosition iobuf)
{
    const NQ_CHAR* p;
    NQ_COUNT segment = iobuf.segment;
    NQ_COUNT byteOffset = iobuf.byteOffset;
    NQ_COUNT length = 0;

    p = (NQ_CHAR*)IOBUF_GETBYTEPTR(iobuf);

    for(;*p;)
    {
        if (byteOffset + sizeof(NQ_CHAR) > iobuf.ioBuf->iovec[segment].iov_len)
        {
            /* end of segment */
            if (segment >= iobuf.ioBuf->iovSegmentCnt)
            {
                /* end of buffer. bad call. */
                goto Exit;
            }
            else
            {
                p = (NQ_CHAR *)iobuf.ioBuf->iovec[++segment].iov_base;
                byteOffset = 0;
            }
        }
        else
        {
            ++p;
            byteOffset += (NQ_COUNT)sizeof(NQ_CHAR);
        }
        ++length;
    }

Exit:
    return length;
}
#endif /* UD_NQ_USEIOVECS */

NQ_CHAR * cmFileIdDump(const NQ_BYTE id[16])
{
    static NQ_CHAR str[18];
    NQ_CHAR *p = str;
    NQ_COUNT i;

    for (i = 0; i < 16; i++)
    {
        sySprintf(p, "%x", id[i]);
        p++;
        if (i == 7)
        {
            sySprintf(p, " ");
            p++;
        }
    }
    str[17] = 0;
    return str;
}

/*
 *====================================================================
 * PURPOSE: Generate random byte sequence
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT buffer
 *          IN  buffer size
 *
 * RETURNS: none
 *
 * NOTES:   none
 *====================================================================
 */

void cmCreateRandomByteSequence(NQ_BYTE *buffer, NQ_UINT32 size)
{
    for (   ; size > 0; --size, ++buffer)
    {
        *buffer = (NQ_BYTE)syRand();
    }
}

NQ_PORT cmDynamicPortBindSocket(SYSocketHandle sock, NQ_BOOL doReuseAddress)
{
    NQ_PORT res;            /* the result */
    NQ_PORT currPort = 0, loopPort;

    res = loopPort = cmPortManage();

    while (currPort != loopPort && NQ_SUCCESS != syBindSocket(sock, cmSelfipGetLocalHostIp(), syHton16(res), doReuseAddress))
    {
        currPort = res = cmPortManage();
    }
    if (currPort == loopPort)
    {
        goto Error;
    }
    goto Exit;

Error:
    res = 0;

Exit:
    return res;
}

void cmDynamicPortRelease(NQ_PORT dynPort)
{
    cmManageFreePort(dynPort);
}

CMProductInfo * cmGetCurrentProductInformation(void)
{
    static CMProductInfo curProduct;
    NQ_BOOL isRecognized = FALSE;

    /* Set product name */
    syStrcpy(curProduct.productName, CM_NQ_PRODUCT_NAME);

    /* Set product type */
    syStrcpy(curProduct.productType, "Server Corporate");
    isRecognized = TRUE;

    if (FALSE == isRecognized)
    {
        syStrcpy(curProduct.productType, "Not Recognized");
    }

    /* Set product version */
    sySprintf(curProduct.version, "%d.%d.%d", CM_SOFTWAREVERSIONMAJOR, CM_SOFTWAREVERSIONMINOR, CM_SOFTWAREVERSIONSUPPORT);

    curProduct.checksum = 0;

    return &curProduct;
}
