/*********************************************************************
*
*           Copyright (c) 2021 by Visuality Systems, Ltd.
*
*********************************************************************
* FILE NAME     : $Workfile:$
* ID            : $Header:$
* REVISION      : $Revision:$
*--------------------------------------------------------------------
* DESCRIPTION   : Buffer manipulation
*--------------------------------------------------------------------
* MODULE        : Common
* DEPENDENCIES  :
********************************************************************/

#ifndef _CMBUF_H
#define _CMBUF_H

/*
todo: replace CMRpcPacketDescriptor with the following structure -

typedef struct {
NQ_BYTE *start;
NQ_BYTE *current;
NQ_UINT size;
NQ_BOOL le;
}
CMBufferReader,
CMBufferWriter;

todo: remove cmRpcParse... and cmRpcPack... functions
todo: make CMRpcPacketDescriptor to contain buffer reader and/or writer
*/

#include "cmapi.h"

/* String flags */
typedef enum
{
    CM_BSF_NOFLAGS = 0,
    CM_BSF_WRITENULLTERM = 1
}
CMBufferStringFlags;

/* Buffer reader */
typedef CMRpcPacketDescriptor CMBufferReader;

/* Initialize buffer reader (byte order set to LE by default) */
void cmBufferReaderInit(CMBufferReader *reader, NQ_IOBufPos buffer, NQ_COUNT size);
/* Set reader start position to its current buffer position and recalculates remaining length */
void cmBufferReaderStart(CMBufferReader *reader);
/* initialize new reader from another reader */
void cmBufferReaderClone(const CMBufferReader *from, CMBufferReader *to);
/* Reset reader current position to its start */
void cmBufferReaderReset(CMBufferReader *reader);
/* Set reader byte order */
void cmBufferReaderSetByteOrder(CMBufferReader *reader, NQ_BOOL le);

/* Get reader start position */
NQ_IOBufPos cmBufferReaderGetStart(const CMBufferReader *reader);
/* Get reader current position */
NQ_IOBufPos cmBufferReaderGetPosition(const CMBufferReader *reader);
/* Get reader current position - byte ptr. required length is the size we need a flat buffer  */
NQ_BYTE* cmBufferReaderGetPositionBytePtr(const CMBufferReader *reader, NQ_COUNT requiredLength);
/* Set reader current position */
void cmBufferReaderSetPosition(CMBufferReader *reader, NQ_IOBufPos position);
/* Set current position of reader if the supplied position is valid (i.e it is inside the buffer) */
NQ_BOOL cmBufferReaderSetPositionIfValid(CMBufferReader *reader, NQ_IOBufPos position);
void cmBufferReaderSetPositionNULL(CMBufferReader *reader);
/* Set reader current offset relative to buffer start */
void cmBufferReaderSetOffset(CMBufferReader *reader, NQ_UINT32 offset);
/* Get read data count in bytes */
NQ_COUNT cmBufferReaderGetDataCount(const CMBufferReader *reader);
/* Get remaining data count */
NQ_INT cmBufferReaderGetRemaining(const CMBufferReader *reader);

/* get total size value */
NQ_COUNT cmBufferReaderGetTotalSize(const CMBufferReader *reader);

/* Align current position (anchor must be inside the buffer), possible alignments: 2, 4, 8, 16, ... */
void cmBufferReaderAlign(CMBufferReader *reader, NQ_IOBufPos anchor, NQ_UINT alignment);

void cmBufferReaderAlignToOrigin(CMBufferReader *reader, NQ_UINT alignment);
/* Move data pointer forward */
void cmBufferReaderSkip(CMBufferReader *reader, NQ_UINT bytes);
/* Move data pointer backward */
void cmBufferReaderMoveBackward(CMBufferReader *reader, NQ_UINT bytes);

/* Read 1 byte */
void cmBufferReadByte(CMBufferReader *reader, NQ_BYTE *to);
/* Read bytes */
void cmBufferReadBytes(CMBufferReader *reader, NQ_BYTE *to, NQ_COUNT size);
/* Read unsigned 16 bit value */
void cmBufferReadUint16(CMBufferReader *reader, NQ_UINT16 *to);
/* Read unsigned 32 bit value */
void cmBufferReadUint32(CMBufferReader *reader, NQ_UINT32 *to);
/* Read unsigned 64 bit value */
void cmBufferReadUint64(CMBufferReader *reader, NQ_UINT64 *to);
/* Read GUID structure */
void cmBufferReadUuid(CMBufferReader *reader, NQ_Uuid * pUuid);

NQ_BOOL cmBufferReaderIsCurrentPositionBigger(const CMBufferReader *reader, NQ_IOBufPos position);

NQ_BOOL cmBufferReaderIsCurrentPositionSmaller(const CMBufferReader *reader, NQ_IOBufPos position);

NQ_INT cmBufferReaderMemCmp(const CMBufferReader *reader, NQ_BYTE *with, NQ_COUNT size);

/* Buffer writer */
typedef CMRpcPacketDescriptor CMBufferWriter;

/* Initialize buffer writer (byte order set to LE by default) */
void cmBufferWriterInit(CMBufferWriter *writer, NQ_IOBufPos buffer, NQ_COUNT capacity);
/* Set writer start position to its current buffer position and recalculates remaining length */
void cmBufferWriterStart(CMBufferWriter *writer);
/* Reset writer current position to its start */
void cmBufferWriterReset(CMBufferWriter *writer);
/* Set writer byte order */
void cmBufferWriterSetByteOrder(CMBufferWriter *writer, NQ_BOOL le);

/* Clone writer - initialize new writer (to) with current position of the old one + offset */
void cmBufferWriterClone(const CMBufferWriter *what, CMBufferWriter *to, NQ_UINT offset);
/* Clone writer - initialize new writer (to) with current position of the old one and then skip by offset */
void cmBufferWriterCloneAndSkip(CMBufferWriter *what, CMBufferWriter *to, NQ_UINT offset);
/* Branch writer - initialize new writer (to) starting from the old writer current position + offset */
void cmBufferWriterBranch(const CMBufferWriter *what, CMBufferWriter *to, NQ_UINT offset);
/* Synchronize writer current positions */
void cmBufferWriterSync(CMBufferWriter *what, const CMBufferWriter *with);

/* Get writer start position */
NQ_IOBufPos cmBufferWriterGetStart(const CMBufferWriter *writer);
/* Get writer current position */
NQ_IOBufPos cmBufferWriterGetPosition(const CMBufferWriter *writer);
/* Get writer current position - get byte ptr. if iovec - assert to make sure can use this length as flat buffer */
NQ_BYTE* cmBufferWriterGetPositionBytePtr(const CMBufferWriter *writer, NQ_COUNT requiredLength);
/* get writer current position plus offset number */
NQ_IOBufPos cmBufferWriterGetPositionWithOff(const CMBufferWriter *writer, NQ_COUNT offset);
/* Set writer current position (must be inside the buffer) */
void cmBufferWriterSetPosition(CMBufferWriter *writer, NQ_IOBufPos position);
/* Set writer current offset relative to buffer start */
/*void cmBufferWriterSetOffset(CMBufferWriter *writer, NQ_UINT offset);*/
/* Get written data count in bytes */
NQ_COUNT cmBufferWriterGetDataCount(const CMBufferWriter *writer);
/* Get written data count in bytes from offset to current */
NQ_COUNT cmBufferWriterGetDataCountFromOffset(const CMBufferWriter *writer, NQ_IOBufPos offset);
/* Get remaining data count */
NQ_COUNT cmBufferWriterGetRemaining(const CMBufferWriter *writer);

/* Align current position and zero the gap (anchor must be inside the buffer), possible alignments: 2, 4, 8, 16, ... , return gap size */
NQ_UINT32 cmBufferWriterAlign(CMBufferWriter *writer, NQ_IOBufPos anchor, NQ_UINT alignment);

NQ_UINT32 cmBufferWriterAlignToOrigin(CMBufferWriter *writer, NQ_UINT alignment);
/* Move data pointer forward */
void cmBufferWriterSkip(CMBufferWriter *writer, NQ_UINT bytes);

/* Write 1 byte */
void cmBufferWriteByte(CMBufferWriter *writer, NQ_BYTE value);
/* Write bytes to flat buffer*/
void cmBufferWriteBytes(CMBufferWriter *writer, const NQ_BYTE *bytes, NQ_COUNT size);
/* Write bytes to iovec buffer*/
void cmBufferWriteIOBytes(CMBufferWriter *writer, NQ_IOBufPos bytes, NQ_COUNT size);
/* Write zero bytes */
void cmBufferWriteZeroes(CMBufferWriter *writer, NQ_COUNT size);
/* Write unsigned 16 bit value */
void cmBufferWriteUint16(CMBufferWriter *writer, NQ_UINT16 value);
/* Write unsigned 32 bit value */
void cmBufferWriteUint32(CMBufferWriter *writer, NQ_UINT32 value);
/* Write 32 bit value */
void cmBufferWriteInt32(CMBufferWriter *writer, NQ_INT32 value);
/* Write unsigned 64 bit value */
void cmBufferWriteUint64(CMBufferWriter *writer, const NQ_UINT64 *value);
/* Write GUID structure */
void cmBufferWriteUuid(CMBufferWriter * writer, const NQ_Uuid * pUuid);
/* Write GUID structure in textual representation */
void cmBufferWriteUuidAsString(CMBufferWriter * writer, const NQ_Uuid * pUuid);
/* Write a null terminated sequence of NQ_WCHARs as LE unicode string */
void cmBufferWriteAsciiAsUnicodeN(CMBufferWriter *writer, const NQ_CHAR *string, NQ_UINT length, CMBufferStringFlags flags);
/* Write Unicode string (null-terminated) */
void cmBufferWriteUnicode(CMBufferWriter *writer, const NQ_WCHAR *string);
/* Write Unicode string without null terminator */
void cmBufferWriteUnicodeNoNull(CMBufferWriter *writer, const NQ_WCHAR *string);
/* Write random bytes */
void cmBufferWriteRandomBytes(CMBufferWriter *writer, NQ_COUNT size);
/* Write string as ASCII or Unicode taking parameters of input string */
void cmBufferWriteString(CMBufferWriter *writer, NQ_BOOL outAscii, const NQ_BYTE *string, NQ_BOOL inUnicode, CMBufferStringFlags flags);
#endif

