/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : AES-GCM(128bit) implementation.
 *--------------------------------------------------------------------
 * MODULE        : Auth - AM
 * DEPENDENCIES  : None
 ********************************************************************/
#include "amaesgcm.h"


/*====================================================================
 * PURPOSE: S-box substitution: Word
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN/OUT    w    : pointer to data block(state); 16 bytes.
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void SubWord(NQ_UINT32* w)
{
    *w = (NQ_UINT32)(AES_SBOX[*w & 0xff]
        | AES_SBOX[*w >> 8 & 0xff] << 8
        | AES_SBOX[*w >> 16 & 0xff] << 16
        | AES_SBOX[*w >> 24 & 0xff] << 24);
}

/*====================================================================
 * PURPOSE: S-box substitution: Word ( Inverse function )
 *--------------------------------------------------------------------
 * NOTES:
 *    Not used for now, so disabled in code.
 *====================================================================
 */
/*
static void InvSubWord(NQ_UINT32* w) {
    *w = AES_INV_SBOX[*w & 0xff]
        | AES_INV_SBOX[*w >> 8 & 0xff] << 8
        | AES_INV_SBOX[*w >> 16 & 0xff] << 16
        | AES_INV_SBOX[*w >> 24 & 0xff] << 24;
}
*/

/*====================================================================
 * PURPOSE: S-box substitution: state
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN/OUT    state    : pointer to data block(state); 16 bytes.
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void SubBytes(NQ_BYTE* state)
{
    NQ_UINT32 i;

    for (i = 0; i < AES_BLOCK_SIZE; i++)
    {
        *state = AES_SBOX[*state];
        state++;
    }
}

/*====================================================================
 * PURPOSE: S-box substitution : ( Inverse function )
 * NOTES:
 *    Not used for now, so disabled in code.
 *====================================================================
 */
/*
static void InvSubBytes(NQ_BYTE* state) {
    NQ_UINT32 i;
    for (i = 0; i<AES_BLOCK_SIZE; ++i) {
        *state = AES_INV_SBOX[*state];
        state++;
    }
}
*/

/*====================================================================
 * PURPOSE: Shift one word
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN/OUT    w    : pointer to data block(state); 1 word (4 bytes).
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void ShiftWord(NQ_UINT32* w)
{
    *w = *w << 24 | *w >> 8;
}


/*====================================================================
 * PURPOSE: Expand key to round key.
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    key    : pointer to key; 16 bytes.
 *    OUT    rk    : pointer to expanded key(round key); 44 words
 *
 * RETURNS:
 *    NONE
 *
 * NOTES:
 *    Only for 128 bit key for now.
 *    https://en.wikipedia.org/wiki/File:AES-Key_Schedule_128-bit_key.svg
 *====================================================================
 */
static void ExpandsKey(const NQ_BYTE* key, NQ_UINT32* rk)
{
    NQ_UINT32 i, r;
    NQ_UINT32 Nk = 4;        /* The number of words in the Key ( 4 words == 128bits ), 6 for 192, 8 for 256 */
    NQ_UINT32 Nr = Nk + 6;    /* The number of Round. 10 times for 128 bit key. */
    NQ_UINT32* cur = rk;
    NQ_UINT32* next;

    /* prepare rk[0]~rk[3] */
    for (i = 0; i < 4; i++)
    {
        *cur++ = (NQ_UINT32)(*key) ^ (NQ_UINT32)((*(key + 1)) << 8) ^ (NQ_UINT32)((*(key + 2)) << 16) ^ (NQ_UINT32)(((*(key + 3)) << 24));
        key += 4;
    }

    cur = rk;
    next = rk + 4;

    /* generate rk[4]~rk[43] */
    for (r = 0; r < Nr; r++)
    {
        /* first Row */
        *next = *(cur + 3);
        ShiftWord(next);
        SubWord(next);
        *next ^= AES_RCON[r] ^ *cur;

        *(next + 1) = (*++cur) ^ *next;    next++;
        *(next + 1) = (*++cur) ^ *next;    next++;
        *(next + 1) = (*++cur) ^ *next;    next++;
        if (r <= Nr - 1)
        {
            next++; cur++;
        }
    }
}

/*====================================================================
 * PURPOSE: Shift Rows of Block
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN/OUT    state    : pointer to data block(state); 16 bytes.
 *
 * RETURNS:
 *    NONE
 *
 * NOTES:
 *    First Line: do nothing.
 *    Second Line: left shift 1 byte.
 *    Third Line: left shift 2 bytes.
 *    Forth Line: left shift 3 bytes.
 *
 *====================================================================
 */
static void ShiftRows(NQ_BYTE* state)
{
    NQ_UINT32* w = (NQ_UINT32*)(state + 4);
    NQ_UINT32 uint32Temp = cmLtoh32(*w);

    /* Second Line */
    uint32Temp = uint32Temp >> 8  | (uint32Temp << 24);
    *w = cmHtol32(uint32Temp);
    w++;

    /* Third Line */
    uint32Temp = cmLtoh32(*w);
    uint32Temp = uint32Temp << 16 | ((uint32Temp >> 16) & 0xFFFF);
    *w = cmHtol32(uint32Temp);
    w++;

    /* Forth Line */
    uint32Temp = cmLtoh32(*w);
    uint32Temp = uint32Temp << 8  | (uint32Temp >> 24);
    *w = cmHtol32(uint32Temp);
}

/*====================================================================
 * PURPOSE: Shift Rows ( Inverse function )
 * NOTES:
 *    Not used for now, so disabled in code.
 *====================================================================
 */
/*
static void InvShiftRows(NQ_BYTE* state) {

    NQ_UINT32* w = (NQ_UINT32*)(state + 4);

    *w = *w << 8  | (*w >> 24); w++;        // Second Line
    *w = *w << 16 | ((*w >> 16) & 0xFFFF); w++;    // Third Line
    *w = *w >> 8  | (*w << 24);            // Forth Line
}
*/

/*====================================================================
 * PURPOSE: Add round key.
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN/OUT    state    : pointer to data block(state); 4 words.
 *    IN    rk    : pointer to round key; 4 words
 *
 * RETURNS:
 *   NONE
 *
 * NOTES:
 *
 *====================================================================
 */
static void AddRoundKey(NQ_BYTE* state, const NQ_BYTE* rk)
{
    NQ_UINT32 i;

    for (i = 0; i < 4; ++i)
    {
        NQ_UINT32 uint32RK = cmHtol32((NQ_UINT32)(*rk) ^
                                      (NQ_UINT32)((*(rk + 1)) << 8) ^
                                      (NQ_UINT32)((*(rk + 2)) << 16) ^
                                      (NQ_UINT32)(((*(rk + 3)) << 24)));

        *(state) = (NQ_BYTE)((*state) ^ (uint32RK & 0x000000FF));
        *(state + 4) = (NQ_BYTE)((*(state + 4)) ^ ((uint32RK & 0x0000FF00) >> 8));
        *(state + 8) = (NQ_BYTE)((*(state + 8)) ^ ((uint32RK & 0x00FF0000) >> 16));
        *(state + 12) = (NQ_BYTE)((*(state + 12)) ^ ((uint32RK & 0xFF000000) >> 24));

        state++;
        rk += 4;
    }
}

/*====================================================================
 * PURPOSE: Multiply by 2
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    val    : pointer to input data
 *    OUT        : result
 *
 * RETURNS: 
 *   NONE
 *
 * NOTES:
 *    data << 1; XOR 0x1b if MSB is 1.
 *
 *====================================================================
 */
static NQ_BYTE MulBy2(NQ_BYTE* val)
{
    NQ_BYTE tmp = *val;
    NQ_UINT32 msb = tmp & 0x80;

    tmp = (NQ_BYTE)(tmp << 0x01);
    if (msb)
    {
        tmp ^= 0x1b;
    }
    return tmp;
}

/*====================================================================
 * PURPOSE: Mix Columns with circular MDS matrix
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN/OUT    state    : pointer to data block(state); 4 words.
 *    IN    rk    : pointer to round key; 4 words
 *
 * RETURNS:
 *    NONE
 *
 * Matrix:
 *
 *   | 02 03 01 01 |
 *   | 01 02 03 01 |
 *   | 01 01 02 03 |
 *   | 03 01 01 02 |
 *
 *====================================================================
 */
static void MixCol(NQ_BYTE* state)
{
    NQ_UINT32 i;
    NQ_BYTE org1, org2, org3, org4;
    NQ_BYTE mul2_1, mul2_2, mul2_3, mul2_4;

    for (i = 0; i < 4; ++i)
    {
        org1 = *state;
        org2 = *(state + 4);
        org3 = *(state + 8);
        org4 = *(state + 12);

        mul2_1 = MulBy2(&org1);
        mul2_2 = MulBy2(&org2);
        mul2_3 = MulBy2(&org3);
        mul2_4 = MulBy2(&org4);

        *state = mul2_1 ^ mul2_2^ org2 ^ org3 ^ org4;
        *(state + 4) = org1 ^ mul2_2 ^ mul2_3^ org3 ^ org4;
        *(state + 8) = org1 ^ org2 ^ mul2_3 ^ mul2_4 ^ org4;
        *(state + 12) = mul2_1 ^ org1 ^ org2 ^ org3 ^ mul2_4;

        state++;
    }
}

/*====================================================================
 * Multiply by n
 *====================================================================
 */
/*
static NQ_BYTE MulByN(NQ_BYTE* val, NQ_BYTE n) {
    NQ_UINT32 i;
    NQ_BYTE x = 0;
    NQ_BYTE tmp = *val;
    for (i = 0; i<8; ++i) {
        if ((n & 0x01) != 0) {
            x ^= tmp;
        }
        tmp = MulBy2(&tmp);
        n >>= 1;
    }
    return x;
}
*/

/*====================================================================
 * PURPOSE: Inverse Mix Columns ( Inverse function )
 *--------------------------------------------------------------------
 * Matrix:
 *
 *   | 0e 0b 0d 09 |
 *   | 09 0e 0b 0d |
 *   | 0d 09 0e 0b |
 *   | 0b 0d 09 0e |
 *
 * NOTES:
 *    Not used for now, so disabled in code.
 *====================================================================
 */
/*
static void InvMixCol(NQ_BYTE* state) {
    NQ_UINT32 i;
    NQ_BYTE org1, org2, org3, org4;

    for (i = 0; i < 4; ++i) {

        org1 = *(state);
        org2 = *(state + 4);
        org3 = *(state + 8);
        org4 = *(state + 12);

        *state = MulByN(&org1, 0x0e) ^ MulByN(&org2, 0x0b) ^ MulByN(&org3, 0x0d) ^ MulByN(&org4, 0x09);
        *(state + 4) = MulByN(&org1, 0x09) ^ MulByN(&org2, 0x0e) ^ MulByN(&org3, 0x0b) ^ MulByN(&org4, 0x0d);
        *(state + 8) = MulByN(&org1, 0x0d) ^ MulByN(&org2, 0x09) ^ MulByN(&org3, 0x0e) ^ MulByN(&org4, 0x0b);
        *(state + 12) = MulByN(&org1, 0x0b) ^ MulByN(&org2, 0x0d) ^ MulByN(&org3, 0x09) ^ MulByN(&org4, 0x0e);

        state++;
    }
}
*/

/*====================================================================
 * PURPOSE: Plain text block to state
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    plainText: pointer to data block; 4 words.
 *    OUT    state    : pointer to round key; 4 words
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void PlainExState(const NQ_BYTE* plainText, NQ_BYTE* state)
{
    NQ_UINT32 i;

    for (i = 0; i < 4; ++i)
    {

        *state = *(plainText);
        *(state + 4) = *(plainText + 1);
        *(state + 8) = *(plainText + 2);
        *(state + 12) = *(plainText + 3);

        state++;
        plainText += 4;
    }
}

/*====================================================================
 * PURPOSE: Encrypt one state
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    rk        : pointer to expended round key; (44 words)
 *    IN    state    : pointer to data block(state); 4 words.
 *    OUT    out    : pointer to encrypted data block; 4 words.
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void AesEncBlock(const NQ_UINT32* rk, const NQ_BYTE* block, NQ_BYTE* out)
{
    NQ_UINT32 i;
    NQ_BYTE buf[AES_BLOCK_SIZE];

    PlainExState(block, buf);

    /* Initial round */
    AddRoundKey(buf, (NQ_BYTE*)rk);        /* AddRoundKey */

    /* 9 rounds */
    for (i = 1; i < 10; ++i)
    {
        SubBytes(buf);                /* SubBytes */
        ShiftRows(buf);                /* ShifRows */
        MixCol(buf);                /* MixColumns */
        AddRoundKey(buf, (NQ_BYTE*)(rk + i * 4));    /* AddRoundKey */
    }

    /* Final round */
    SubBytes(buf);                /* SubBytes */
    ShiftRows(buf);                /* ShifRows */
    AddRoundKey(buf, (NQ_BYTE*)(rk + 40));    /* AddRoundKey */

    PlainExState(buf, out);
}


/*====================================================================
 * PURPOSE: Increase the count in the iv
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    iv    : pointer to initial vector; (16 bytes, last word contains count number)
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void IncIV(NQ_BYTE* const iv)
{
    NQ_UINT32 c = ((NQ_UINT32)(*(iv + 12)) << 24)
        | ((NQ_UINT32)(*(iv + 13)) << 16)
        | ((NQ_UINT32)(*(iv + 14)) << 8)
        | ((NQ_UINT32)(*(iv + 15)));

    c++;
    *(iv + 12) = (NQ_BYTE)(c >> 24);
    *(iv + 13) = (NQ_BYTE)(c >> 16);
    *(iv + 14) = (NQ_BYTE)(c >> 8);
    *(iv + 15) = (NQ_BYTE)(c);
}

/*====================================================================
 * PURPOSE: Initialize initial vector
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    nonce    : pointer to nonce; (12 bytes, 96 bits)
 *    OUT    iv    : pointer to vector;
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void InitIV(const NQ_BYTE* nonce, NQ_BYTE* iv)
{
    syMemset(iv, 0, AES_BLOCK_SIZE);
    syMemcpy(iv, nonce, NONCE_LEN);    /*nonce is 12bytes. */
    IncIV(iv);                /*set 1 as initial value. */
}

/*====================================================================
 * PURPOSE: XOR two blocks
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    in    : pointer to in data 1; (16 bytes)
 *    IN    out    : pointer to in data 2; (16 bytes)
 *    OUT    out    : pointer to result data; (16 bytes)
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void XorBlocks(const NQ_BYTE* in, NQ_BYTE* out)
{
    NQ_UINT32 *i = (NQ_UINT32 *)in;
    NQ_UINT32 *o = (NQ_UINT32 *)out;

    *o++ ^= *i++;
    *o++ ^= *i++;
    *o++ ^= *i++;
    *o++ ^= *i++;
}

/*====================================================================
 * PURPOSE: Right shift whole block 1 bit.
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    block    : pointer to in data; (16 bytes)
 *    OUT    block    : pointer to out data; (16 bytes)
 *
 * RETURNS:
 *    NONE
 *
 * NOTES:
 *    W = rightshift(V), then Wi = Vi-1 for 1 = i = 127 and W0 = 0.
 *    The leftmost bit is V0, and the rightmost bit is V127
 *====================================================================
 */
static void RightShiftBlock(NQ_BYTE* block)
{
    NQ_INT i;
    NQ_BYTE* tmp = (NQ_BYTE*)block;

    for (i = AES_BLOCK_SIZE - 1; i >= 0; --i)
    {
        *(tmp+i) = *(tmp+i) >> 1;
        if (i != 0 && (*(tmp+i-1) & (NQ_BYTE)0x1))
        {
            *(tmp+i) = *(tmp+i) | (NQ_BYTE)0x80;
        }
    }
}

/*====================================================================
 * PURPOSE: multiplication in GF(2^128) 
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    MultD    : pointer to multiplicand data; (16 bytes)
 *    IN    multR    : pointer to multiplier data; (16 bytes)
 *    OUT    product    : pointer to result data; (16 bytes)
 *
 * RETURNS:
 *    NONE
 * NOTES:
 *    1, If AESGCM_USE_MTABLE is defined, only used to calculate MTable when .
 *    2, If AESGCM_USE_MTABLE is NOT defined, used to multiplication (Cost times)
 *====================================================================
 */
static void MultGF128(const NQ_BYTE* MultD, const NQ_BYTE* multR, NQ_BYTE* product)
{
    /*R = 11100001 || 0^120 */
    const NQ_BYTE R[AES_BLOCK_SIZE] = { 0xE1, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0 };
    NQ_BYTE v[AES_BLOCK_SIZE];
    NQ_BYTE tmp[AES_BLOCK_SIZE];
    NQ_INT i, j;
    NQ_BYTE msk;

    syMemset(tmp, 0, AES_BLOCK_SIZE);
    syMemcpy(v, MultD, AES_BLOCK_SIZE);

    for (i = 0; i < AES_BLOCK_SIZE; ++i)
    {
        msk = 0x80;
        for (j = 0; j < 8; ++j)
        {
            if (*multR & msk)
            {
                XorBlocks(v, tmp);
            }
            msk = msk >> 1;

            if (v[15] & (NQ_BYTE)0x01)
            {
                RightShiftBlock(v);
                XorBlocks(R, v);
            }
            else
            {
                RightShiftBlock(v);
            }
        }
        multR += 1;
    }
    syMemcpy(product, tmp, AES_BLOCK_SIZE);
}

/*====================================================================
 * PURPOSE: Calculate 4K MTable
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    hashKey    : pointer to MTable data; (4K bytes)
 *    OUT    pMTable    : pointer to MTable buffer; (4K bytes)
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
#ifdef AESGCM_USE_MTABLE
static void calcMTable(const NQ_BYTE* hashKey, NQ_BYTE* pMTable)
{
    NQ_BYTE i;
    NQ_BYTE MulD[AES_BLOCK_SIZE];

    syMemset(MulD, 0, AES_BLOCK_SIZE);

    i = 0;
    while(1)
    {
        MulD[0] = i;
        MultGF128(MulD, hashKey, pMTable);
        pMTable += AES_BLOCK_SIZE;
        if (i == 255)
        {
            break;
        }
        i++;
    }
}
#endif

#ifdef AESGCM_USE_MTABLE
/*====================================================================
 * PURPOSE: multiplication in GF(2^128) with 4K MTable
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    MultD    : pointer to multiplicand data; (16 bytes)
 *    IN    pMTable    : pointer to MTable data; (4K bytes)
 *    OUT    product    : pointer to result data; (16 bytes)
 *
 * RETURNS:
 *    NONE
 * NOTE:
 *    Use external 4KB(MTable) + 512B(RTable) memory for performance.
 *====================================================================
 */
#if 1
static void MultGF128_MTable(const NQ_BYTE* MultD, const NQ_BYTE* pMTable, NQ_BYTE* product)
{
    NQ_BYTE buf[32];
    NQ_BYTE* pBuf;
    NQ_BYTE val;
    NQ_INT i;

    syMemset(buf, 0, 32);
    pBuf = buf + 15;

    /* byte(X,15) -> byte(X,1) */
    for (i = 15; i > 0; --i)
    {
        XorBlocks((pMTable + ( (*(MultD+i))*AES_BLOCK_SIZE )), pBuf);
        val = *(pBuf + 15);

        pBuf--;
        *pBuf ^= *(RTable + val * 2);
        *(pBuf+1) ^= *(RTable + val * 2 + 1);

    }
    /* byte(X, 0) */
    XorBlocks(pMTable + (*MultD) * AES_BLOCK_SIZE, pBuf);

    syMemcpy(product, buf, AES_BLOCK_SIZE);
}
#else
static void MultGF128_MTable(const NQ_BYTE* MultD, const NQ_BYTE* pMTable, NQ_BYTE* product)
{
    int i, j;
    NQ_BYTE val;
    NQ_BYTE buf[16];

    syMemset(buf, 0, 16);

    //byte(X,15) -> byte(X,1)
    for(i=15;i>0;--i)
    {
        XorBlocks((pMTable + ( (*(MultD+i))*16 )), buf);
        val = *(buf + 15);

        for(j = 15; j>0; --j)
        {
            *(buf+j) = *(buf+j -1) ;
        }
        *buf = 0;
        *buf ^= *(RTable + val * 2);
        *(buf+1) ^= *(RTable + val * 2 + 1);
    }
    //byte(X, 0)
    XorBlocks(pMTable + (*MultD) * 16, buf);

    syMemcpy(product, buf,16);
}
#endif
#endif /* #ifdef AESGCM_USE_MTABLE */

/*====================================================================
 * PURPOSE: GHASH one block
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    auth    : Previous authentication data; (16 bytes)
 *    IN    ac    : Additional authentication Data or Ciper text
 *    IN    acSize    : ac length in byte; (1-16 bytes)
 *    IN    hashKey    : Hash key; (16 bytes)
 *    OUT    auth    : Hashed authentication data; (16 bytes)
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
static void GHashBlcok( NQ_BYTE* auth, const NQ_BYTE* ac, const NQ_UINT32 acSize, const NQ_BYTE* hashKey)
{
    NQ_BYTE tmp[AES_BLOCK_SIZE];

    if (acSize < AES_BLOCK_SIZE)
    {
        syMemcpy(tmp, ac, acSize);
        syMemset((void*)(tmp + acSize), 0, (AES_BLOCK_SIZE - acSize));
        XorBlocks( tmp, auth);
    }
    else
    {
        XorBlocks( ac, auth);
    }

#ifdef AESGCM_USE_MTABLE
    MultGF128_MTable( auth, hashKey, auth);
#else
    MultGF128( auth, hashKey, auth);
#endif
}

/*====================================================================
 * PURPOSE: Calculate GMAC
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    key    : pointer to encryption key; (16 bytes).
 *    IN    nonce    : pointer to nonce; 98 bites.
 *    IN    cData    : pointer to input ciper text.
 *    IN    cDataLen: ciper text length.
 *    IN    aadBuf    : pointer to additional authentication data.
 *    IN    aadLen    : aad text length.
 *    OUT    gmac    : pointer to Message Authentication Code.
 *
 * RETURNS:
 *    NONE
 * NOTE:
 *    If AESGCM_USE_MTABLE is defined, then use Table.(much faster)
 *====================================================================
 */
static void CalcAesGmac(const NQ_BYTE* key, const NQ_BYTE* nonce, const NQ_BYTE* cData, const NQ_UINT32 cDataLen, 
        const NQ_BYTE* aadBuf, const NQ_UINT32 aadLen, NQ_BYTE* gmac)
{
    NQ_UINT32 i, round, left;
    NQ_UINT32 roundKey[EXTKEY_LEN];
    NQ_BYTE hashKey[AES_BLOCK_SIZE];
    NQ_BYTE lenBuf[AES_BLOCK_SIZE];
    NQ_BYTE iv[AES_BLOCK_SIZE];
    NQ_BYTE* pKey;
#ifdef AESGCM_USE_MTABLE
    NQ_BYTE pMTable[AES_BLOCK_SIZE * 256];
#endif
    const NQ_BYTE* p;

    /*Initialize hashKey */
    ExpandsKey(key, roundKey);
    syMemset(hashKey, 0, AES_BLOCK_SIZE);
    AesEncBlock(roundKey, hashKey, hashKey);

#ifdef AESGCM_USE_MTABLE
    calcMTable(hashKey, pMTable);
    pKey = pMTable;
#else
    pKey = hashKey;
#endif
    syMemset(gmac, 0, AES_BLOCK_SIZE);

    /*hash aad */
    round = aadLen / AES_BLOCK_SIZE;
    left = aadLen % AES_BLOCK_SIZE;
    p = aadBuf;
    for (i = 0; i < round; ++i)
    {
        GHashBlcok(gmac, p, AES_BLOCK_SIZE, pKey);
        p += AES_BLOCK_SIZE;
    }

    if (0 != left)
    {
        GHashBlcok(gmac, p, left, pKey);
    }

    /*hash cipter text */
    round = cDataLen / AES_BLOCK_SIZE;
    left = cDataLen % AES_BLOCK_SIZE;
    p = cData;
    for (i = 0; i < round; ++i)
    {
        GHashBlcok(gmac, p, AES_BLOCK_SIZE, pKey);
        p += AES_BLOCK_SIZE;
    }

    if (0 != left)
    {
        GHashBlcok(gmac, p, left, pKey);
    }

    /*hash with Len(AD)||Len(PT) */
    PUT_BE_BYTES_MUL8_IN64(lenBuf, aadLen);
    PUT_BE_BYTES_MUL8_IN64(lenBuf + 8, cDataLen);
    GHashBlcok(gmac, lenBuf, AES_BLOCK_SIZE, pKey);

    /* T = MSB_t((GHASH(H,A,C)^(E(K,Y0)) */
    InitIV(nonce, iv);
    AesEncBlock(roundKey, iv, iv);
    XorBlocks(iv, gmac);
}

/*====================================================================
 * PURPOSE: Encrypt plain text.(AES Block Encryption)
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    key         : pointer to encryption key; (128 bits, 16 bytes)
 *    IN    nonce       : pointer to nonce; (96 bits, 12 bytes)
 *    IN    inText      : pointer to input text.
 *    IN    textLen     : input text length.
 *    OUT   outText     : pointer to output text(encrypted or decrypted)
 *
 * RETURNS:
 *    NONE
 *====================================================================
 */
void aesGcmEnc(const NQ_BYTE* key, const NQ_BYTE* nonce, const NQ_BYTE* inText, const NQ_UINT32 textLen, NQ_BYTE* outText)
{
    NQ_UINT32 i;
    NQ_UINT32 roundKey[EXTKEY_LEN]; /* Expanded round key buffer */
    NQ_UINT32 blocks = textLen / AES_BLOCK_SIZE;    /* number of blocks. */
    NQ_UINT32 left = textLen % AES_BLOCK_SIZE;    /* bytes of last block. */
    NQ_BYTE buffer[AES_BLOCK_SIZE] = { 0 };
    NQ_BYTE iv[AES_BLOCK_SIZE] = { 0 };
    NQ_BYTE* out;
    NQ_BOOL flag = FALSE;

    /* Encrypt in the input memory space. */
    if (inText == outText)
    {
        flag = TRUE;
        out = buffer;
    }
    else
    { /* Encrypt in output memory space */
        out = outText;
    }

    /* Prepare round key */
    ExpandsKey(key, roundKey);

    /* Encryption */
    InitIV(nonce, iv);
    for (i = 0; i < blocks; ++i)
    {
        IncIV(iv);
        AesEncBlock(roundKey, iv, out);
        if(flag)
        {
            XorBlocks(out, outText);
            outText += AES_BLOCK_SIZE;
        }
        else
        {
            XorBlocks(inText, out);
            out += AES_BLOCK_SIZE;
        }
        inText += AES_BLOCK_SIZE;
    }
    if (left != 0)
    {
        IncIV(iv);
        AesEncBlock(roundKey, iv, buffer);
        for (i = 0; i < left; ++i)
        {
            if (flag)
            {
                *outText++ ^= *(buffer + i);
            }
            else
            {
                *out++ = *inText++ ^ *(buffer + i);
            }
        }
    }
}

/*====================================================================
 * PURPOSE: Encrypt input text and calculate GMAC
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    key    : pointer to encryption key; (16 bytes).
 *    IN    nonce    : pointer to nonce; 98 bites.
 *    IN    aad    : pointer to additional authentication data.
 *    IN/OUT    plainText    : pointer to input plain text.
 *    OUT    gmac    : pointer to Message Authentication Code.
 *    OUT    msgBuf    : pointer to text(encrypted),
 *
 * RETURNS:
 *    NONE
 *
 * NOTE:
 *  If msgBuf is a NULL value, or msgBuf == plainText->data, 
 *  encrypt data is saved in plainText->data memory space.
 *
 *====================================================================
 */
void aes128GcmEncryptInternal(const CMBlob* key, const CMBlob* nonce, const CMIOBlob* aad, CMIOBlob* plainText, NQ_BYTE* gmac, NQ_BYTE *keyBuffer, NQ_BYTE *msgBuf ) {

    NQ_BYTE* data;

    if (NULL == msgBuf || plainText->data == msgBuf)
    {
        data = plainText->data;
    }
    else
    {
        data = msgBuf;
    }

    /* Encrypt plain text */
    aesGcmEnc(key->data, nonce->data, plainText->data, plainText->len, data);

    /* Calculate GMAC */
    CalcAesGmac(key->data, nonce->data, data, plainText->len, aad->data, aad->len, gmac);
}

/*====================================================================
 * PURPOSE: Check GMAC first and Decrypt.
 *--------------------------------------------------------------------
 * PARAMS:
 *    IN    key    : pointer to encryption key; (16 bytes).
 *    IN    key1   : pointer to nonce; 98 bites.
 *    IN    prefix : pointer to additional authentication data.
 *    IN/OUT  message : pointer to ciper text.
 *    IN   auth : pointer to Message Authentication Code.
 *    OUT  keyBuffer : NOT USED
 *    OUT  encMsgBuffer : pointer to plain text(Decrypted)
 *
 * RETURNS:
 *     TRUE in GMAC matches, FALSE in mismatches.
 *
 * NOTE:
 *  If encMsgBuffer is a NULL value, or encMsgBuffer == message->data,
 *  Decrypt data is saved in message->data memory space.
 *
 *====================================================================
 */
NQ_BOOL aes128GcmDecryptInternal(const CMBlob *key, const CMBlob *key1, const CMIOBlob *prefix, CMIOBlob *message, const NQ_BYTE *auth, NQ_BYTE *keyBuffer, NQ_BYTE *encMsgBuffer)
{
    NQ_BYTE calcGmac[AES_BLOCK_SIZE];
    NQ_BYTE* data;
    NQ_BOOL res = FALSE;

    if (NULL == encMsgBuffer || message->data == encMsgBuffer)
    {
        data = message->data;
    }
    else
    {
        data = encMsgBuffer;
    }

    /* Calculate GMAC */
    CalcAesGmac(key->data, key1->data, message->data, message->len, prefix->data, prefix->len, calcGmac);
    if (0 != syMemcmp(auth, calcGmac, AES_BLOCK_SIZE))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "GMAC mismatch");
        goto Exit;
    }

    /* Decrypt plain text */
    aesGcmEnc(key->data, key1->data, message->data, message->len, data);
    res = TRUE;

Exit:
    return res;
}

