/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : AES, CMAC implementation
 *--------------------------------------------------------------------
 * MODULE        : Auth - AM
 * DEPENDENCIES  : None
 ********************************************************************/

#include "cmapi.h"

#include "amaes.h"
#include "amaesccm.h"

static void AES_XOR_128(const NQ_BYTE *a, NQ_IOBufPos b, NQ_BYTE *out)
{
    NQ_INT i;

    for (i=0;i<16; i++)
    {
        out[i] = a[i] ^ *(IOBUF_GETBYTEPTR(b));
        IOBUF_MOVEBYTES(b, (NQ_INT)(1))
    }
}

static void AES_128_ExpandKey(NQ_UINT32 expandedKey[44], const NQ_BYTE key[16])
{
    NQ_INT      i = 0; /* 10 rounds */
    NQ_UINT32     temp;

    expandedKey[0] = AES_Get32(key);
    expandedKey[1] = AES_Get32(key + 4);
    expandedKey[2] = AES_Get32(key + 8);
    expandedKey[3] = AES_Get32(key + 12);

    for (i = 0; i < 10; i++)
    {
        temp  = expandedKey[(i*4)+3];
        expandedKey[(i*4) + 4] = expandedKey[(i*4)] ^
            (AES_SBox[(temp >> 16) & 0xff] & 0xff000000) ^
            (AES_SBox[(temp >>  8) & 0xff] & 0x00ff0000) ^
            (AES_SBox[(temp      ) & 0xff] & 0x0000ff00) ^
            (AES_SBox[(temp >> 24)       ] & 0x000000ff) ^
            AES_128_Rcon[i];
        expandedKey[(i*4) + 5] = expandedKey[(i*4) + 1] ^ expandedKey[(i*4) + 4];
        expandedKey[(i*4) + 6] = expandedKey[(i*4) + 2] ^ expandedKey[(i*4) + 5];
        expandedKey[(i*4) + 7] = expandedKey[(i*4) + 3] ^ expandedKey[(i*4) + 6];
    }
}

static void AES_Encryption( NQ_BYTE state[16], NQ_UINT32 key[44], NQ_BYTE out[16])
{
    NQ_UINT32 st0, st1, st2, st3, tmp0, tmp1, tmp2, tmp3;

    /*    get state into 4 UINT32 */
    st0 = AES_Get32(state)      ^ key[0];
    st1 = AES_Get32(state +  4) ^ key[1];
    st2 = AES_Get32(state +  8) ^ key[2];
    st3 = AES_Get32(state + 12) ^ key[3];

    /* first round */
    tmp0 = AES_Table_1[st0 >> 24]  ^ AES_Table_2[(st1 >> 16) & 0xff]  ^ AES_Table_3[(st2 >> 8) & 0xff]  ^ AES_Table_4[st3 & 0xff]  ^ key[ 4];
    tmp1 = AES_Table_1[st1 >> 24]  ^ AES_Table_2[(st2 >> 16) & 0xff]  ^ AES_Table_3[(st3 >> 8) & 0xff]  ^ AES_Table_4[st0 & 0xff]  ^ key[ 5];
    tmp2 = AES_Table_1[st2 >> 24]  ^ AES_Table_2[(st3 >> 16) & 0xff]  ^ AES_Table_3[(st0 >> 8) & 0xff]  ^ AES_Table_4[st1 & 0xff]  ^ key[ 6];
    tmp3 = AES_Table_1[st3 >> 24]  ^ AES_Table_2[(st0 >> 16) & 0xff]  ^ AES_Table_3[(st1 >> 8) & 0xff]  ^ AES_Table_4[st2 & 0xff]  ^ key[ 7];
    /* second round */
    st0  = AES_Table_1[tmp0 >> 24] ^ AES_Table_2[(tmp1 >> 16) & 0xff] ^ AES_Table_3[(tmp2 >> 8) & 0xff] ^ AES_Table_4[tmp3 & 0xff] ^ key[ 8];
    st1  = AES_Table_1[tmp1 >> 24] ^ AES_Table_2[(tmp2 >> 16) & 0xff] ^ AES_Table_3[(tmp3 >> 8) & 0xff] ^ AES_Table_4[tmp0 & 0xff] ^ key[ 9];
    st2  = AES_Table_1[tmp2 >> 24] ^ AES_Table_2[(tmp3 >> 16) & 0xff] ^ AES_Table_3[(tmp0 >> 8) & 0xff] ^ AES_Table_4[tmp1 & 0xff] ^ key[10];
    st3  = AES_Table_1[tmp3 >> 24] ^ AES_Table_2[(tmp0 >> 16) & 0xff] ^ AES_Table_3[(tmp1 >> 8) & 0xff] ^ AES_Table_4[tmp2 & 0xff] ^ key[11];
    /* third round */
    tmp0 = AES_Table_1[st0 >> 24]  ^ AES_Table_2[(st1 >> 16) & 0xff]  ^ AES_Table_3[(st2 >> 8) & 0xff]  ^ AES_Table_4[st3 & 0xff]  ^ key[12];
    tmp1 = AES_Table_1[st1 >> 24]  ^ AES_Table_2[(st2 >> 16) & 0xff]  ^ AES_Table_3[(st3 >> 8) & 0xff]  ^ AES_Table_4[st0 & 0xff]  ^ key[13];
    tmp2 = AES_Table_1[st2 >> 24]  ^ AES_Table_2[(st3 >> 16) & 0xff]  ^ AES_Table_3[(st0 >> 8) & 0xff]  ^ AES_Table_4[st1 & 0xff]  ^ key[14];
    tmp3 = AES_Table_1[st3 >> 24]  ^ AES_Table_2[(st0 >> 16) & 0xff]  ^ AES_Table_3[(st1 >> 8) & 0xff]  ^ AES_Table_4[st2 & 0xff]  ^ key[15];
    /* 4th round */
    st0  = AES_Table_1[tmp0 >> 24] ^ AES_Table_2[(tmp1 >> 16) & 0xff] ^ AES_Table_3[(tmp2 >> 8) & 0xff] ^ AES_Table_4[tmp3 & 0xff] ^ key[16];
    st1  = AES_Table_1[tmp1 >> 24] ^ AES_Table_2[(tmp2 >> 16) & 0xff] ^ AES_Table_3[(tmp3 >> 8) & 0xff] ^ AES_Table_4[tmp0 & 0xff] ^ key[17];
    st2  = AES_Table_1[tmp2 >> 24] ^ AES_Table_2[(tmp3 >> 16) & 0xff] ^ AES_Table_3[(tmp0 >> 8) & 0xff] ^ AES_Table_4[tmp1 & 0xff] ^ key[18];
    st3  = AES_Table_1[tmp3 >> 24] ^ AES_Table_2[(tmp0 >> 16) & 0xff] ^ AES_Table_3[(tmp1 >> 8) & 0xff] ^ AES_Table_4[tmp2 & 0xff] ^ key[19];
    /* 5th round */
    tmp0 = AES_Table_1[st0 >> 24]  ^ AES_Table_2[(st1 >> 16) & 0xff]  ^ AES_Table_3[(st2 >> 8) & 0xff]  ^ AES_Table_4[st3 & 0xff]  ^ key[20];
    tmp1 = AES_Table_1[st1 >> 24]  ^ AES_Table_2[(st2 >> 16) & 0xff]  ^ AES_Table_3[(st3 >> 8) & 0xff]  ^ AES_Table_4[st0 & 0xff]  ^ key[21];
    tmp2 = AES_Table_1[st2 >> 24]  ^ AES_Table_2[(st3 >> 16) & 0xff]  ^ AES_Table_3[(st0 >> 8) & 0xff]  ^ AES_Table_4[st1 & 0xff]  ^ key[22];
    tmp3 = AES_Table_1[st3 >> 24]  ^ AES_Table_2[(st0 >> 16) & 0xff]  ^ AES_Table_3[(st1 >> 8) & 0xff]  ^ AES_Table_4[st2 & 0xff]  ^ key[23];
    /* 6th round */
    st0  = AES_Table_1[tmp0 >> 24] ^ AES_Table_2[(tmp1 >> 16) & 0xff] ^ AES_Table_3[(tmp2 >> 8) & 0xff] ^ AES_Table_4[tmp3 & 0xff] ^ key[24];
    st1  = AES_Table_1[tmp1 >> 24] ^ AES_Table_2[(tmp2 >> 16) & 0xff] ^ AES_Table_3[(tmp3 >> 8) & 0xff] ^ AES_Table_4[tmp0 & 0xff] ^ key[25];
    st2  = AES_Table_1[tmp2 >> 24] ^ AES_Table_2[(tmp3 >> 16) & 0xff] ^ AES_Table_3[(tmp0 >> 8) & 0xff] ^ AES_Table_4[tmp1 & 0xff] ^ key[26];
    st3  = AES_Table_1[tmp3 >> 24] ^ AES_Table_2[(tmp0 >> 16) & 0xff] ^ AES_Table_3[(tmp1 >> 8) & 0xff] ^ AES_Table_4[tmp2 & 0xff] ^ key[27];
    /* 7th round */
    tmp0 = AES_Table_1[st0 >> 24]  ^ AES_Table_2[(st1 >> 16) & 0xff]  ^ AES_Table_3[(st2 >> 8) & 0xff]  ^ AES_Table_4[st3 & 0xff]  ^ key[28];
    tmp1 = AES_Table_1[st1 >> 24]  ^ AES_Table_2[(st2 >> 16) & 0xff]  ^ AES_Table_3[(st3 >> 8) & 0xff]  ^ AES_Table_4[st0 & 0xff]  ^ key[29];
    tmp2 = AES_Table_1[st2 >> 24]  ^ AES_Table_2[(st3 >> 16) & 0xff]  ^ AES_Table_3[(st0 >> 8) & 0xff]  ^ AES_Table_4[st1 & 0xff]  ^ key[30];
    tmp3 = AES_Table_1[st3 >> 24]  ^ AES_Table_2[(st0 >> 16) & 0xff]  ^ AES_Table_3[(st1 >> 8) & 0xff]  ^ AES_Table_4[st2 & 0xff]  ^ key[31];
    /* 8th round */
    st0  = AES_Table_1[tmp0 >> 24] ^ AES_Table_2[(tmp1 >> 16) & 0xff] ^ AES_Table_3[(tmp2 >> 8) & 0xff] ^ AES_Table_4[tmp3 & 0xff] ^ key[32];
    st1  = AES_Table_1[tmp1 >> 24] ^ AES_Table_2[(tmp2 >> 16) & 0xff] ^ AES_Table_3[(tmp3 >> 8) & 0xff] ^ AES_Table_4[tmp0 & 0xff] ^ key[33];
    st2  = AES_Table_1[tmp2 >> 24] ^ AES_Table_2[(tmp3 >> 16) & 0xff] ^ AES_Table_3[(tmp0 >> 8) & 0xff] ^ AES_Table_4[tmp1 & 0xff] ^ key[34];
    st3  = AES_Table_1[tmp3 >> 24] ^ AES_Table_2[(tmp0 >> 16) & 0xff] ^ AES_Table_3[(tmp1 >> 8) & 0xff] ^ AES_Table_4[tmp2 & 0xff] ^ key[35];
    /* 9th round */
    tmp0 = AES_Table_1[st0 >> 24]  ^ AES_Table_2[(st1 >> 16) & 0xff]  ^ AES_Table_3[(st2 >> 8) & 0xff]  ^ AES_Table_4[st3 & 0xff]  ^ key[36];
    tmp1 = AES_Table_1[st1 >> 24]  ^ AES_Table_2[(st2 >> 16) & 0xff]  ^ AES_Table_3[(st3 >> 8) & 0xff]  ^ AES_Table_4[st0 & 0xff]  ^ key[37];
    tmp2 = AES_Table_1[st2 >> 24]  ^ AES_Table_2[(st3 >> 16) & 0xff]  ^ AES_Table_3[(st0 >> 8) & 0xff]  ^ AES_Table_4[st1 & 0xff]  ^ key[38];
    tmp3 = AES_Table_1[st3 >> 24]  ^ AES_Table_2[(st0 >> 16) & 0xff]  ^ AES_Table_3[(st1 >> 8) & 0xff]  ^ AES_Table_4[st2 & 0xff]  ^ key[39];

    st0  =    (AES_SBox[(tmp0 >> 24)       ] & 0xff000000) ^ (AES_SBox[(tmp1 >> 16) & 0xff] & 0x00ff0000) ^
            (AES_SBox[(tmp2 >>  8) & 0xff] & 0x0000ff00) ^ (AES_SBox[(tmp3      ) & 0xff] & 0x000000ff) ^ key[40];
    st1  =    (AES_SBox[(tmp1 >> 24)       ] & 0xff000000) ^ (AES_SBox[(tmp2 >> 16) & 0xff] & 0x00ff0000) ^
            (AES_SBox[(tmp3 >>  8) & 0xff] & 0x0000ff00) ^ (AES_SBox[(tmp0      ) & 0xff] & 0x000000ff) ^ key[41];
    st2  =    (AES_SBox[(tmp2 >> 24)       ] & 0xff000000) ^ (AES_SBox[(tmp3 >> 16) & 0xff] & 0x00ff0000) ^
            (AES_SBox[(tmp0 >>  8) & 0xff] & 0x0000ff00) ^ (AES_SBox[(tmp1      ) & 0xff] & 0x000000ff) ^ key[42];
    st3  =     (AES_SBox[(tmp3 >> 24)       ] & 0xff000000) ^ (AES_SBox[(tmp0 >> 16) & 0xff] & 0x00ff0000) ^
            (AES_SBox[(tmp1 >>  8) & 0xff] & 0x0000ff00) ^ (AES_SBox[(tmp2      ) & 0xff] & 0x000000ff) ^ key[43];
    AES_Put32(out     , st0);
    AES_Put32(out +  4, st1);
    AES_Put32(out +  8, st2);
    AES_Put32(out + 12, st3);
}

void AES_128_Encrypt(NQ_BYTE state[16], NQ_BYTE key[16], NQ_BYTE encrypted[16])
{
    NQ_UINT32 expandedKey[44];

    AES_128_ExpandKey(expandedKey, key);
    AES_Encryption(state, expandedKey, encrypted);
}

/* 8-bit CFB mode */
void AES_128_CFB8_Encrypt(const NQ_BYTE key[16], NQ_BYTE initVector[16], const NQ_BYTE *in, NQ_UINT size, NQ_BYTE *out)
{
    NQ_COUNT i;
    NQ_UINT32 expandedKey[44];
    NQ_BYTE tmp[AES_BLOCK_SIZE + 1];

    AES_128_ExpandKey(expandedKey, key);

    for (i = 0; i < size; i++)
    {
        syMemcpy(tmp, initVector, AES_BLOCK_SIZE);

        AES_Encryption(initVector, expandedKey, initVector);

        out[i] = in[i] ^ initVector[0];
        tmp[AES_BLOCK_SIZE] = out[i];

        syMemcpy(initVector, &tmp[1], AES_BLOCK_SIZE);
    }
}

/* 8-bit CFB mode */
void AES_128_CFB8_Decrypt(const NQ_BYTE key[16], NQ_BYTE initVector[16], const NQ_BYTE *in, NQ_UINT size, NQ_BYTE *out)
{
    NQ_COUNT i;
    NQ_UINT32 expandedKey[44];
    NQ_BYTE tmp[AES_BLOCK_SIZE + 1];

    AES_128_ExpandKey(expandedKey, key);

    for (i = 0; i < size; i++)
    {
        syMemcpy(tmp, initVector, AES_BLOCK_SIZE);

        AES_Encryption(initVector, expandedKey, initVector);

        tmp[AES_BLOCK_SIZE] = in[i];
        out[i] = in[i] ^ initVector[0];

        syMemcpy(initVector, &tmp[1], AES_BLOCK_SIZE);
    }
}


/*
 *====================================================================
 * PURPOSE: Encrypt SMB3 Message with AES_CCM
 *--------------------------------------------------------------------
 * PARAMS:  IN     Encrypting Key
 *          IN     Nonce
 *          IN     SMB3 Message
 *          IN     SMB3 Message length
 *          IN     Additional Message (SMB2 TRANSFORM HEADER excluding protocolID, Signature, Nonce)
 *          IN     Additional Message length
 *          OUT       Encrypted authentication value (signature for TF-Header)
 *
 * RETURNS: none
 *====================================================================
 */

void aes128ccmEncryptionInternal(const CMBlob * key, const CMBlob * key1, const CMIOBlob * prefix, CMIOBlob * message, NQ_BYTE * auth)
{
    NQ_UINT lm = (message->len % AES_BLOCK_SIZE) == 0 ? message->len / AES_BLOCK_SIZE : message->len / AES_BLOCK_SIZE + 1;
    NQ_UINT la = (prefix->len % AES_BLOCK_SIZE) == 0 ? prefix->len / AES_BLOCK_SIZE : prefix->len / AES_BLOCK_SIZE + 1;
    NQ_INT     written = 0;
    NQ_UINT remaining  = 0,  i = 0, B_offset = 0;
    NQ_BYTE    * B = NULL, * X = NULL;
    NQ_BYTE S0[16];
    NQ_BYTE * writer = NULL;
    NQ_IOBufPos ioBufPosTmp;
    IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

    B = (NQ_BYTE *)cmMemoryAllocate((lm + la + 2 +1) * AES_BLOCK_SIZE);
    X = (NQ_BYTE *)cmMemoryAllocate((lm + la + 2) * AES_BLOCK_SIZE);

    if (B == NULL || X == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "  AES_CCM_Encrypt: Couldn't allocate memory");
        goto Exit;
    }
    syMemset(B, 0, (lm + la + 2 + 1) * AES_BLOCK_SIZE);
    syMemset(X, 0, (lm + la + 2) * AES_BLOCK_SIZE);
    syMemset(auth, 0, 16);

    /* Setting Block 0 */
    B[0] = AES_128_CCM_L_tag + 8 * AES_128_CCM_M_tag;
    B[0] = (NQ_BYTE)(prefix->len > 0 ? B[0] + 64 : B[0]);
    syMemcpy(&B[1], key1->data, key1->len);
    cmPutUint32((NQ_BYTE *)&B[AES_128_CCM_M - AES_128_CCM_L], cmHtob32((NQ_UINT32)message->len));

    /* X_1 */
    AES_128_Encrypt(&B[0], key->data, &X[0]);
    /* Setting Block 1 with sizes  */
    if (prefix->len >= 0xFF00)
    {
        B[16] = 0xFF;
        B[17] =    0xFE;
        cmPutUint32((NQ_BYTE *)&B[18], cmHtob32((NQ_UINT32)prefix->len));
        B_offset = 6;
        writer = &B[22];
    }
    else if (prefix->len > 0)
    {
        cmPutUint16((NQ_BYTE *)&B[16], (NQ_UINT16)cmHtob16(prefix->len));
        B_offset = 2;
        writer = &B[18];
    }
    /* Filling B */
    IOBUF_MEMCPY_V2F(writer, prefix->data, prefix->len);
    writer += prefix->len;
    remaining = (prefix->len + B_offset) % AES_BLOCK_SIZE;
    if (remaining > 0)
    {
        syMemset(writer, 0, 16 - remaining);
        writer += 16 - remaining;
    }
    IOBUF_MEMCPY_V2F(writer, message->data, message->len);
    writer += message->len;

    written = (NQ_INT) (writer - &B[0]);
    written = (written % 16 == 0) ? written / AES_BLOCK_SIZE : written /AES_BLOCK_SIZE + 1;
    /* Filling X */
    for (i = 1; i < (NQ_UINT)written; i++)
    {
        IOBUF_POSCONSTRUCTOR(ioBufPosTmp, &X[(i-1)*16], 16)
        AES_XOR_128(&B[i*16], ioBufPosTmp, &B[i*16]);
        AES_128_Encrypt(&B[i*16], key->data, &X[i*16]);
    }
    for (i =0; i < lm + 1; i++)
    {
        NQ_BYTE A[16], S[16];
        NQ_BYTE *p;

        if (i == 0)
        {
            A[0] = AES_128_CCM_L_tag;
            p = (NQ_BYTE *)&A[1];
            syMemcpy(p, key1->data, 11);
        }
            p = (NQ_BYTE *)&A[(AES_128_CCM_M - AES_128_CCM_L)];
            cmPutUint32(p, cmHtob32((NQ_UINT32)i));
            AES_128_Encrypt((NQ_BYTE *)A, key->data, (NQ_BYTE *)S);
            if (i == 0)
                syMemcpy(S0, S, 16);

        if (i > 0 )
        {
            NQ_COUNT j;
            NQ_BYTE * tmp;

            if (i == lm && message->len % AES_BLOCK_SIZE != 0)
            {
                /* last round */
                IOBUF_MOVEBYTES(message->data, (NQ_INT)((lm - 1) * AES_BLOCK_SIZE))
                for (j = 0; j < message->len % AES_BLOCK_SIZE; j++)
                {
                    IOBUF_MOVEBYTES(message->data, (NQ_INT)(0 == j ? 0 : 1))
                    tmp = IOBUF_GETBYTEPTR(message->data);
                    *tmp = *tmp ^ S[j];
                }
                IOBUF_MOVEBYTES(message->data, -(NQ_INT)(((i - 1) * (NQ_INT)AES_BLOCK_SIZE) + j - 1));
            }
            else
            {
                IOBUF_MOVEBYTES(message->data, (NQ_INT)((i - 1) * AES_BLOCK_SIZE))
                for (j = 0; j < AES_BLOCK_SIZE; j++)
                {
                    IOBUF_MOVEBYTES(message->data, (NQ_INT)(0 == j ? 0 : 1))
                    tmp = IOBUF_GETBYTEPTR(message->data);
                    *tmp = *tmp ^ S[j];
                }
                IOBUF_MOVEBYTES(message->data, (-(((NQ_INT)i * (NQ_INT)AES_BLOCK_SIZE) - 1)));
            }
        }
    }

    IOBUF_POSCONSTRUCTOR(ioBufPosTmp, S0, sizeof(S0))
    AES_XOR_128((NQ_BYTE *)&X[(written - 1)*AES_BLOCK_SIZE],ioBufPosTmp, auth);

Exit:
    cmMemoryFree(X);
    cmMemoryFree(B);
}

/*
 *====================================================================
 * PURPOSE: Decryption SMB3 Message with AES_CCM
 *--------------------------------------------------------------------
 * PARAMS:  IN     Encrypting Key
 *          IN     Nonce
 *          IN     SMB3 Message
 *          IN     SMB3 Message length
 *          IN     Additional Message (SMB2 TRANSFORM HEADER excluding protocolID, Signature, Nonce)
 *          IN     Additional Message length
 *          OUT       Encrypted authentication value (signature for TF-Header)
 *
 * RETURNS: TRUE  -> if calculated authentication value equals to received value.
 *             FALSE -> if calculated value differs from received. The Decrypted messages should be IGNORED in this case
 *====================================================================
 */
NQ_BOOL aes128ccmDecryptionInternal(const CMBlob * key, const CMBlob * key1, const CMIOBlob * prefix, CMIOBlob * message, const NQ_BYTE * auth)
{
    NQ_UINT    lm = (message->len % 16) == 0 ? message->len / 16 : message->len / 16 + 1;
    NQ_UINT la = (prefix->len % 16) == 0 ? prefix->len / 16 : prefix->len / 16 + 1;
    NQ_UINT i = 0, remaining  = 0, B_offset = 0;
    NQ_INT    written = 0;
    NQ_BYTE    T[16], S0[16];
    NQ_BYTE    *B = NULL, *X = NULL;
    NQ_BYTE *writer = NULL;
    NQ_BOOL result = FALSE;
    NQ_IOBufPos ioBufPosTmp;
    IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

    B = (NQ_BYTE *)cmMemoryAllocate((lm + la + 2 +1) * 16);
    X = (NQ_BYTE *)cmMemoryAllocate((lm + la + 2) * 16);
    if (B == NULL || X == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "  AES_CCM_Encrypt: Couldn't allocate memory");
        goto Exit;
    }
    syMemset(B, 0, (lm + la + 2 + 1) * 16);
    syMemset(X, 0, (lm + la + 2) * 16);

    for (i =0; i < lm + 1; i++)
    {
        NQ_BYTE A[16], S[16];
            NQ_BYTE *p;

        A[0] = AES_128_CCM_L_tag;
            p = (NQ_BYTE *)&A[1];
        syMemcpy(p, key1->data, 11);
            p = (NQ_BYTE *)&A[AES_128_CCM_M - AES_128_CCM_L];
        cmPutUint32(p, cmHtob32((NQ_UINT32)i));
        AES_128_Encrypt((NQ_BYTE *)A, key->data, (NQ_BYTE *)S);
        if (i == 0)
            syMemcpy(S0, S, 16);

        if (i > 0 )
        {
            NQ_COUNT j;
            NQ_BYTE *tmp;

            if (i == lm && message->len % AES_BLOCK_SIZE != 0)
            {
                /* last round. i == lm*/
                IOBUF_MOVEBYTES(message->data, (NQ_INT)((i - 1) * AES_BLOCK_SIZE))
                for (j = 0; j < message->len % AES_BLOCK_SIZE; j++)
                {
                    IOBUF_MOVEBYTES(message->data, (NQ_INT)(0 == j ? 0 : 1))
                    tmp = IOBUF_GETBYTEPTR(message->data);
                    *tmp = *tmp ^ S[j];
                }
                IOBUF_MOVEBYTES(message->data, -(NQ_INT)(((i - 1) * (NQ_INT)AES_BLOCK_SIZE) + j - 1));
            }
            else
            {
                IOBUF_MOVEBYTES(message->data, (NQ_INT)(i - 1) * AES_BLOCK_SIZE);
                for (j = 0; j < AES_BLOCK_SIZE; j++)
                {
                    IOBUF_MOVEBYTES(message->data, (NQ_INT)(0 == j ? 0 : 1))
                    tmp = IOBUF_GETBYTEPTR(message->data);
                    *tmp = *tmp ^ S[j];
                }
                IOBUF_MOVEBYTES(message->data, (-(((NQ_INT)i * (NQ_INT)AES_BLOCK_SIZE) - 1)));
            }
        }
    }

    /* Recovering T */
    IOBUF_POSCONSTRUCTOR(ioBufPosTmp, S0, sizeof(S0))
    AES_XOR_128((NQ_BYTE *)auth, ioBufPosTmp, (NQ_BYTE *)&T );

    /* Setting Block 0 */
    B[0] = AES_128_CCM_L_tag + 8 * AES_128_CCM_M_tag;
    B[0] = (NQ_BYTE)(prefix->len > 0 ? B[0] + 64 : B[0]);
    syMemcpy(&B[1], key1->data, key1->len);
    cmPutUint32((NQ_BYTE *)&B[AES_128_CCM_M - AES_128_CCM_L], cmHtob32((NQ_UINT32)message->len));

    /* X_1 */
    AES_128_Encrypt(&B[0], key->data, &X[0]);
    /* Setting Block 1 with sizes  */
    if (prefix->len >= 0xFF00)
    {
        B[16] = 0xFF;
        B[17] =    0xFE;
        cmPutUint32((NQ_BYTE *)&B[18], cmHtob32((NQ_UINT32)prefix->len));
        B_offset = 6;
        writer = &B[22];
    }
    else if (prefix->len > 0)
    {
        cmPutUint16((NQ_BYTE *)&B[16],(NQ_UINT16)cmHtob16(prefix->len));
        B_offset = 2;
        writer = &B[18];
    }

    /* Filling B */
    IOBUF_MEMCPY_V2F(writer, prefix->data, prefix->len);
    writer += prefix->len;
    remaining = (prefix->len + B_offset) % 16;
    if (remaining > 0)
    {
        syMemset(writer, 0, 16 - remaining);
        writer += 16 - remaining;
    }
    IOBUF_MEMCPY_V2F(writer, message->data, message->len);
    writer += message->len;

    written = (NQ_INT) (writer - &B[0]);
    written = (written % 16 == 0) ? written / 16 : written /16 + 1;

    /* Filling X */
    for (i = 1; i < (NQ_UINT)written; i++)
    {
        IOBUF_POSCONSTRUCTOR(ioBufPosTmp, &X[(i-1)*16], 16)
        AES_XOR_128(&B[i*16], ioBufPosTmp, &B[i*16]);
        AES_128_Encrypt(&B[i*16], key->data, &X[i*16]);
    }

    result = syMemcmp(&T, &X[(written -1) * 16], 16) == 0;

Exit:
    cmMemoryFree(B);
    cmMemoryFree(X);
    return result;
}


 /****************************************************************/
 /* AES-CMAC with AES-128 bit                                    */
 /* CMAC     Algorithm described in SP800-38B                    */
 /****************************************************************/

/* For CMAC Calculation */
NQ_BYTE const_Rb[16] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x87
};

typedef struct{
    NQ_UINT    numOfRounds;
    NQ_BYTE mainKey[16];
    NQ_BYTE    X[16];
    NQ_BYTE M_Last[16];
    NQ_BYTE extra[16];
    NQ_BYTE key1[16];
    NQ_BYTE key2[16];
    NQ_UINT leftover;
    NQ_INT     flag;
}cmac_context;

static void AES_CMAC_ShiftBitLeft(NQ_BYTE *input, NQ_BYTE *output)
{
    NQ_INT    i;
    NQ_BYTE    overflow = 0;

    for ( i = 15; i >= 0; i--)
    {
        output[i] = (NQ_BYTE)(input[i] << 1);
        output[i] |= overflow;
        overflow = (input[i] & 0x80)?1:0;
    }
    return;
}

static void AES_CMAC_GenSubKey(NQ_BYTE *key, NQ_BYTE *K1, NQ_BYTE *K2)
{
    NQ_BYTE L[16];
    NQ_BYTE Z[16];
    NQ_BYTE temp[16];
    NQ_IOBufPos ioBufPosTmp;
    IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

    syMemset(&Z, 0, 16);

    AES_128_Encrypt(Z, key, L);

    if ( (L[0] & 0x80) == 0 )
    {
        AES_CMAC_ShiftBitLeft(L, K1);
    }
    else
    {
        IOBUF_POSCONSTRUCTOR(ioBufPosTmp, const_Rb, sizeof(const_Rb))
        AES_CMAC_ShiftBitLeft(L, temp);
        AES_XOR_128(temp, ioBufPosTmp, K1);
    }

    if ( (K1[0] & 0x80) == 0 )
    {
        AES_CMAC_ShiftBitLeft(K1, K2);
    }
    else
    {
        IOBUF_POSCONSTRUCTOR(ioBufPosTmp, const_Rb, sizeof(const_Rb))
        AES_CMAC_ShiftBitLeft(K1, temp);
        AES_XOR_128(temp, ioBufPosTmp, K2);
    }
    return;
}

static void AES_CMAC_Padding ( const NQ_BYTE * lastByte, NQ_BYTE * pad, NQ_UINT length )
{
    NQ_UINT         i;

    for (i = 0; i < 16; i++ )
    {
      if ( i < length )
      {
          pad[i] = lastByte[i];
      }
      else if ( i == length )
      {
          pad[i] = 0x80;
      }
      else
      {
          pad[i] = 0x00;
      }
    }
}

static void aes_cmac_init( const NQ_BYTE * key, NQ_UINT length, cmac_context * context)
{
    syMemset(context->mainKey, 0, 16);
    syMemset(context->key1, 0, 16);
    syMemset(context->key2, 0, 16);
    syMemset(context->X, 0, 16);
    syMemset(context->M_Last, 0, 16);
    syMemset(context->extra, 0, 16);
    context->leftover = FALSE;

    syMemcpy( context->mainKey, key, 16);

    AES_CMAC_GenSubKey(context->mainKey, context->key1, context->key2);

    context->numOfRounds = (length + 15) / 16;
    if ( context->numOfRounds == 0 )
    {
      context->numOfRounds = 1;
      context->flag = 0;
    }
    else
    {
        context->flag = (length % 16) == 0 ? 1 : 0;
    }
}

static void aes_cmac_update(cmac_context * ctx, NQ_IOBufPos buffer,  NQ_UINT length)
{
    NQ_UINT currentRounds = 0;
    NQ_COUNT i = 0;
    NQ_BYTE Y[16];
    NQ_UINT newLength = length;

    syMemset(&Y, 0, 16);

    if (ctx->leftover)
    {
        NQ_COUNT copyLen, tmpLen;
        NQ_BYTE temp[16];
        NQ_IOBufPos ioBufPosTmp;
        IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

        syMemset(&temp, 0, 16);
        syMemcpy(&temp, ctx->extra, ctx->leftover);
        tmpLen = 16 - ctx->leftover;
        copyLen = length >= tmpLen ? tmpLen : length;
        IOBUF_MEMCPY_V2F(&temp[ctx->leftover], buffer, copyLen);
        newLength = (length > tmpLen) ? length - tmpLen : 0;

        if (newLength == 0)
        {
            ctx->leftover += length;
            syMemcpy(&ctx->extra, &temp, ctx->leftover );
            return;
        }

        /* work on left over buffer + added bytes */
        IOBUF_POSCONSTRUCTOR(ioBufPosTmp, temp, sizeof(temp))
        AES_XOR_128(ctx->X, ioBufPosTmp, Y);
        AES_128_Encrypt(Y, ctx->mainKey,ctx->X);
        ctx->numOfRounds--;

        IOBUF_MOVEBYTES(buffer, (NQ_INT)tmpLen);
    }

    currentRounds = (newLength / 16);

    for (i = 0; i < currentRounds; i++)
    {
        if (ctx->numOfRounds - 1 == 0 && ctx->flag)
        {
            IOBUF_MEMCPY_V2F((NQ_BYTE *)ctx->extra, IOBUF_SKIPBYTESEP(buffer, (16 * i), (NQ_INT)(0 == i ? 0 : 16)), 16);
            ctx->leftover = 16;
            return;
        }
        AES_XOR_128(ctx->X, IOBUF_SKIPBYTESEP(buffer, (16 * i), (NQ_INT)(0 == i ? 0 : 16)), Y);
        AES_128_Encrypt(Y, ctx->mainKey, ctx->X);
        ctx->numOfRounds--;
    }

    ctx->leftover = newLength % 16;
    if (ctx->leftover != 0)
    {
        syMemset(ctx->extra, 0, 16);
        IOBUF_MEMCPY_V2F((NQ_BYTE *)ctx->extra, IOBUF_SKIPBYTESEP(buffer, (16 * i), (NQ_INT)(0 == i ? 0 : 16)), ctx->leftover);
    }
}

static void aes_cmac_final(cmac_context * ctx, NQ_BYTE * mac)
{
    NQ_BYTE Y[16];
    NQ_IOBufPos ioBufPosTmp;
    IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

    syMemset( &Y, 0, 16);

    if (ctx->leftover > 0)
    {
        if (ctx->leftover < 16)
        {
            NQ_BYTE padded[16];

            IOBUF_POSCONSTRUCTOR(ioBufPosTmp, ctx->key2, sizeof(ctx->key2))
            AES_CMAC_Padding((NQ_BYTE *)&ctx->extra, padded, ctx->leftover);
            AES_XOR_128(padded, ioBufPosTmp, ctx->M_Last);
        }
        else
        {
            IOBUF_POSCONSTRUCTOR(ioBufPosTmp, ctx->key1, sizeof(ctx->key1))
            AES_XOR_128((NQ_BYTE *)&ctx->extra, ioBufPosTmp, ctx->M_Last);
        }
    }
    IOBUF_POSCONSTRUCTOR(ioBufPosTmp, ctx->M_Last, sizeof(ctx->M_Last))
    AES_XOR_128(ctx->X, ioBufPosTmp, Y);
    AES_128_Encrypt(Y, ctx->mainKey, ctx->X);
    syMemcpy( mac, ctx->X, 16);
}

void aes128cmacInternal(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize)
{
    NQ_BYTE sig[16];
    cmac_context context;
    NQ_UINT size = 0, i;

    LOGDUMP(CM_TRC_LEVEL_CRYPT, "key", key->data, key->len);
    syMemset(buffer, 0, bufferSize); /* zero out the previous signature*/
    for (i = 0; i < numFragments; i++)
    {
        if (!IOBUF_ISNULL(dataFragments[i].data))
        {
            size += dataFragments[i].len;
        }
    }
    aes_cmac_init(key->data, size, &context);
    for (i = 0; i < numFragments; i++)
    {
        if (!IOBUF_ISNULL(dataFragments[i].data) && dataFragments[i].len > 0)
        {
            aes_cmac_update(&context, (NQ_IOBufPos)dataFragments[i].data, dataFragments[i].len);
        }
    }
    aes_cmac_final(&context, (NQ_BYTE *)&sig);

    LOGDUMP(CM_TRC_LEVEL_CRYPT, "signature", sig, bufferSize);
    syMemcpy(buffer, sig, bufferSize);
}




