/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : SMB2 Close command handler
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "csutils.h"
#include "csnotify.h"
#include "cs2disp.h"
#include "csbreak.h"
#ifdef UD_CS_INCLUDERPC_SPOOLSS
#include "csspools.h"
#endif /* UD_CS_INCLUDERPC_SPOOLSS */
#include "csdcerpc.h"

#if defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2)

#define CLOSE_FLAG_POSTQUERY_ATTRIB 1   /* use the attribute fields in the response */

#define RESPONSE_DATASIZE 60            /* length of the close response not including data */

static NQ_UINT16 flags;                 /* close flags */
static SYFileInformation fileInfo;      /* buffer for file information */

#ifdef UD_CS_INCLUDERPC_SPOOLSS

/*====================================================================
 * PURPOSE: save parameters in late response context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NONE
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static void
closeSmb2LateResponseSave(
    CSLateResponseContext* context
    )
{
    CMSmb2Header * pHeader;

    pHeader = cs2DispatchGetCurrentHeader();
    pHeader->aid.low = csSmb2SendInterimResponse(pHeader);
    pHeader->aid.high = 0;
    context->prot.smb2.commandData.close.flags = flags;
    syMemcpy(&context->prot.smb2.commandData.close.fileInfo, &fileInfo, sizeof(fileInfo));

    /* write request information into the file descriptor */
    csDispatchSaveResponseContext(context);

    return;
}

/*====================================================================
 * PURPOSE: calculate command data pointer and size
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NQ_SUCCESS or error code
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static NQ_STATUS
closeSmb2LateResponsePrepare(
    CSLateResponseContext* context
    )
{
    csDispatchPrepareLateResponse(context);
    context->commandData += RESPONSE_DATASIZE;
    context->commandDataSize -= RESPONSE_DATASIZE;

    return NQ_SUCCESS;
}

/*====================================================================
 * PURPOSE: send a response using saved context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *          IN status to return
 *          IN number of bytes to return in the data section
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   data is ignored
 *====================================================================
 */

static NQ_BOOL
closeSmb2LateResponseSend(
    CSLateResponseContext* context,
    NQ_UINT32 status,
    NQ_COUNT dataLength
    )
{
    CMBufferWriter writer;

    /* save response for subsequent Close */
    csDcerpcSaveCompleteResponse((CSFile*)context->file, context->commandData, dataLength);

    /* compose Close response */
    context->commandData -= RESPONSE_DATASIZE;
    cmBufferWriterInit(&writer, context->commandData, RESPONSE_DATASIZE);
    cmBufferWriteUint16(&writer, RESPONSE_DATASIZE);            /* structure length */
    cmBufferWriteUint16(&writer, context->prot.smb2.commandData.close.flags);
    cmBufferWriteUint32(&writer, 0);                            /* reserved */

    if (CLOSE_FLAG_POSTQUERY_ATTRIB & context->prot.smb2.commandData.close.flags)
    {
        NQ_UINT64 time;         /* utc time */

        cmCifsTimeToUTC(context->prot.smb2.commandData.close.fileInfo.creationTime, &time.low, &time.high);
        cmBufferWriteUint64(&writer, &time);
        cmCifsTimeToUTC(context->prot.smb2.commandData.close.fileInfo.lastAccessTime, &time.low, &time.high);
        cmBufferWriteUint64(&writer, &time);
        cmCifsTimeToUTC(context->prot.smb2.commandData.close.fileInfo.lastWriteTime, &time.low, &time.high);
        cmBufferWriteUint64(&writer, &time);
        cmCifsTimeToUTC(context->prot.smb2.commandData.close.fileInfo.lastChangeTime, &time.low, &time.high);
        cmBufferWriteUint64(&writer, &time);
        cmBufferWriteUint32(&writer, context->prot.smb2.commandData.close.fileInfo.allocSizeLow);   /* allocation size */
        cmBufferWriteUint32(&writer, context->prot.smb2.commandData.close.fileInfo.allocSizeHigh);
        cmBufferWriteUint32(&writer, context->prot.smb2.commandData.close.fileInfo.sizeLow);        /* EOF */
        cmBufferWriteUint32(&writer, context->prot.smb2.commandData.close.fileInfo.sizeHigh);
        cmBufferWriteUint32(&writer, context->prot.smb2.commandData.close.fileInfo.attributes);
    }
    else
    {
        cmBufferWriteZeroes(&writer, cmBufferWriterGetRemaining(&writer));
    }

    return csDispatchSendLateResponse(context, status, RESPONSE_DATASIZE);
}

#endif /* UD_CS_INCLUDERPC_SPOOLSS */

/*====================================================================
 * PURPOSE: Perform Close processing
 *--------------------------------------------------------------------
 * PARAMS:  IN in - pointer to the parsed SMB2 header descriptor
 *          OUT out - pointer to the response header structure
 *          IN reader - request reader pointing to the second command field
 *          IN connection - pointer to the session structure
 *          IN user - pointer to the user structure
 *          IN tree - pointer to the tree structure
 *          OUT writer - pointer to the response writer
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   This function is called on SMB2 Create command.
 *====================================================================
 */

NQ_UINT32 csSmb2OnClose(CMSmb2Header *in, CMSmb2Header *out, CMBufferReader *reader, CSSession *connection, CSUser *user, CSTree *tree, CMBufferWriter *writer)
{
    CSFile* pFile;                          /* pointer to file descriptor */
    CSFid fid;                              /* fid of the file to close */
    const NQ_WCHAR *pFileName;              /* file name pointer */
    const CSShare* pShare;                  /* pointer to share descriptor */
    NQ_UINT32 returnValue;                  /* return code */
#ifdef UD_NQ_INCLUDEEVENTLOG
    UDFileAccessEvent eventInfo;            /* share event information */
#endif /* UD_NQ_INCLUDEEVENTLOG */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "in:%p out:%p reader:%p connection:%p user:%p tree:%p writer:%p", in, out, reader, connection, user, tree, writer);

    /* parse request */
    cmBufferReadUint16(reader, &flags);
    cmBufferReaderSkip(reader, 4);
    cmBufferReadUint16(reader, &fid);
    cs2ParseFid(&fid);

    if ((pShare = csGetShareByUidTid(user->uid, tree->tid)) == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal TID");
        returnValue = SMB_STATUS_INVALID_PARAMETER;
        goto Exit;
    }
    
    /* find file descriptor */
    pFile = csGetFileByFid(fid, tree->tid, user->uid);
    if (pFile == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = SMB_STATUS_INVALID_HANDLE;
        goto Exit;
    }

    pFileName = csGetFileName(pFile->fid);
    if (pFileName == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "File name corrupted");
        returnValue = SMB_STATUS_UNSUCCESSFUL;
        goto Exit;
    }
#ifdef UD_NQ_INCLUDEEVENTLOG
    eventInfo.tid = tree->tid;
    eventInfo.rid = csGetUserRid(user);
    eventInfo.fileName = pFileName;
    eventInfo.access = 0;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "File name: %s", cmWDump(pFileName));   

#ifdef UD_CS_INCLUDERPC_SPOOLSS
    if (pFile->isPrint)
    {
        NQ_UINT64 zeroSize;

        syMemset(&fileInfo, 0, sizeof(fileInfo));
        cmU64Zero(&zeroSize);

        /* send spooler data content to printer */
        if ((TRUE == syIsValidFile(pFile->fileSpooler)) && (-1 == cmU64Cmp(&zeroSize, &pFile->fileSpoolerDataCount)))
        {
            syGetFileInformation(pFile->fileSpooler, NULL, &fileInfo);
            returnValue = csSpoolssSendToPrinter(pFile, closeSmb2LateResponseSave, closeSmb2LateResponsePrepare, closeSmb2LateResponseSend);
            if (NQ_SUCCESS != returnValue)
            {
                csReleaseFile(pFile->fid);      /* also closes the file */
                LOGERR(CM_TRC_LEVEL_ERROR, "WRITE to printer failed");
                goto Exit;
            }
        }
    }
    else
#endif /* UD_CS_INCLUDERPC_SPOOLSS */
    {
        /* if delete on close was requested - mark this file for deletion */
        if (pFile->options & SMB_NTCREATEANDX_DELETEONCLOSE)
        {
            CSName* pName;          /* pointer to the file name descriptor */

            pName = csGetNameByNid(pFile->nid);
            if (pName == NULL)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Internal error: file name descriptor not found");
                csReleaseFile(pFile->fid);      /* also closes the file */
                returnValue = SMB_STATUS_UNSUCCESSFUL;
                goto Exit;
            }
            pName->markedForDeletion = TRUE;
#ifdef UD_NQ_INCLUDEEVENTLOG
            if (pName->markedForDeletion && pName->deletingUserRid == CS_ILLEGALID)
            {               
                pName->deletingUserRid = csGetUserRid(user);
                pName->deletingTid = pFile->tid;
                cmIpToAscii(pName->deletingIP, user->ip);
            }
#endif /* UD_NQ_INCLUDEEVENTLOG */      
        }

        /* read file information */
#ifdef UD_CS_INCLUDERPC 
        if (pFile->isPipe)
        {
            syMemset(&fileInfo, 0, sizeof(fileInfo));
        }
        else
        {
#endif /* UD_CS_INCLUDERPC */   
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_ATTRIBGET,
                user->name,
                user->ip,
                0,
                (const NQ_BYTE*)&eventInfo
                );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
            if (syGetFileInformationByName(pFileName, &fileInfo) != NQ_SUCCESS)
            {
                returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_ATTRIBGET,
                    user->name,
                    user->ip,
                    returnValue,
                    (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_NQ_INCLUDEEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_CLOSE,
                    user->name,
                    user->ip,
                    returnValue,
                    (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                csReleaseFile(pFile->fid);      /* also closes the file */
                LOGERR(CM_TRC_LEVEL_ERROR, "Unable to read file information: %d", returnValue);
                goto Exit;
            }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_ATTRIBGET,
                user->name,
                user->ip,
                0,
                (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
#ifdef UD_CS_INCLUDERPC 
        }
#endif /* UD_CS_INCLUDERPC */           
    }

    /* compose the response */
    cmBufferWriteUint16(writer, RESPONSE_DATASIZE);     /* structure size */
    cmBufferWriteUint16(writer, flags);                 /* set flags */
    cmBufferWriteUint32(writer, 0);                     /* reserved */
    if (CLOSE_FLAG_POSTQUERY_ATTRIB & flags)
    {
        NQ_UINT64 time;         /* utc time */
        
        cmCifsTimeToUTC(fileInfo.creationTime, &time.low, &time.high);
        cmBufferWriteUint64(writer, &time);
        cmCifsTimeToUTC(fileInfo.lastAccessTime, &time.low, &time.high);
        cmBufferWriteUint64(writer, &time);
        cmCifsTimeToUTC(fileInfo.lastWriteTime, &time.low, &time.high);
        cmBufferWriteUint64(writer, &time);
        cmCifsTimeToUTC(fileInfo.lastChangeTime, &time.low, &time.high);
        cmBufferWriteUint64(writer, &time);
        cmBufferWriteUint32(writer, fileInfo.allocSizeLow);   /* allocation size */
        cmBufferWriteUint32(writer, fileInfo.allocSizeHigh);
        cmBufferWriteUint32(writer, fileInfo.sizeLow);        /* EOF */
        cmBufferWriteUint32(writer, fileInfo.sizeHigh);
        cmBufferWriteUint32(writer, fileInfo.attributes);
    }
    else
    {
        cmBufferWriteZeroes(writer, RESPONSE_DATASIZE - cmBufferWriterGetDataCount(writer));
    }
    
    /* complete oplock break operation (send late response) if required */
    if (pFile->oplockGranted && pFile->isBreakingOpLock)
    {
        CMBufferWriter packet;

        out->status = 0;
        cmBufferWriterInit(&packet, cmBufferWriterGetStart(writer) - SMB2_HEADERSIZE , 124);
        cmSmb2HeaderWrite(out, &packet);
        csBreakComplete(pFile, cmBufferWriterGetStart(&packet), in->flags);
        csReleaseFile(pFile->fid);
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Oplock break completed. fid: %d", pFile->fid);
        returnValue = SMB_STATUS_NORESPONSE;
        goto Exit;
    }

    /* release the descriptor and close the file */
    csReleaseFile(pFile->fid);          /* also closes the file */

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

#endif /* defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_INCLUDESMB2) */

