/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client RPC over TCP
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#ifndef _CCRPC_H_
#define _CCRPC_H_

#include "ccdcerpc.h"
#include "ccnetlgn.h"
#include "amapi.h"

/**
 *  Struct describing remote RPC server.
 */
typedef struct _ccrpcserver
{
    CMItem item;                                    /**< List item. */
    const NQ_IPADDRESS *ips;                        /**< Pointer to server IP addresses. */
    NQ_COUNT numIps;                                /**< Number of IP addresses in the array above.*/
    NQ_IPADDRESS ip;                                /**< Used IP of server.*/
    NQ_PORT port;                                   /**< Port used. */
    NQ_UINT32 maxRead;                              /**< Maximum read size that server accepts. */
    NQ_UINT32 maxWrite;                             /**< Max write size that server accepts. */
    CMList pipes;                                   /**< Pipes list. */
    const CCDcerpcPipeDescriptor *pipeDesc;         /**< Pointer to pipe descriptor. */
    SYSocketHandle socket;                          /**< Socket used. */
#ifdef UD_NQ_INCLUDESMBCAPTURE
    CMCaptureHeader captureHdr;                     /**< Structure for internal capture. */
#endif /* UD_NQ_INCLUDESMBCAPTURE */
}
CCRPCServer;


/**
 *  Open pipe descriptor.
 */
typedef struct _ccpipe
{
    CMItem item;                                    /**< List item. */
    CCRPCServer *server;                            /**< Pointer to the remote share descriptor. */
    NQ_BOOL open;                                   /**< TRUE when file has been open. */
    NQ_UINT64 offset;                               /**< Current offset in the file */
    NQ_UINT16 maxRpcXmit;                           /**< Maximum length of RPC transmit fragment */
    NQ_UINT16 maxRpcRecv;                           /**< Maximum length of RPC receive fragment */
    NQ_BOOL disconnected;                           /**< TRUE when connection was disconnected */
    NQ_WCHAR pipeName[CM_BUFFERLENGTH(NQ_WCHAR, CM_USERNAMELENGTH)];    /**< Pipe name */
    NQ_WCHAR userName[CM_BUFFERLENGTH(NQ_WCHAR, CM_USERNAMELENGTH)];    /**< User name */
    NQ_UINT32 flagsAuth;                            /**< Authentication flags */
    NQ_BYTE authType;                               /**< Authentication type */
    NQ_BYTE authLevel;                              /**< Authentication level */
    NQ_UINT32 authCtxId;                            /**< Authentication context ID */
    NQ_BYTE sessionKey[SMB_SESSIONKEY_LENGTH];      /**< Spnego: a key for password encryption */
    NQ_BYTE recvSigningKey[SMB_SESSIONKEY_LENGTH];  /**< Spnego: a key for password encryption */
    NQ_BYTE sendSigningKey[SMB_SESSIONKEY_LENGTH];  /**< Spnego: a key for password encryption */
    NQ_BYTE recvSealingState[258];                  /**< Spnego: a key for password encryption */
    NQ_BYTE sendSealingState[258];                  /**< Spnego: a key for password encryption */
    NQ_UINT32 sendSeqNum;                           /**< Spnego: sequence number for send */
    NQ_UINT32 recvSeqNum;                           /**< Spnego: sequence number for receive */
    CCNetlogonCredential credential;                /**< Secure channel: Netlogon credential  */
    CCSchannel schannel;                            /**< Secure channel: state */
    NQ_BOOL doSignPduHeader;                        /**< Whether server supports PDU header signing or just stub data */
    NQ_BOOL isUsed;                                 /**< Whether in use */
}
CCPipe;

/**
 * Module shutdown.
 * @return TRUE when succeeded, FALSE otherwise.
 */
NQ_BOOL ccDcerpcOverTcpStart(void);

/**
 * Module start.
 */
void ccDcerpcOverTcpShutdown(void);

/**
 * Authentication parameters struct
 */
typedef struct
{
    CMBlob blobIn;          /**< Blob incoming */
    CMBlob blobOut;         /**< Blob outgoing */
    CMBlob sessionKey;      /**< Session key */
    CMBlob macKey;          /**< Signing key */
    NQ_BYTE type;           /**< Authentication type */
    NQ_BYTE level;          /**< Authentication level */
    NQ_UINT32 flags;        /**< Flags */
    void *securityContext;  /**< Security context used by authentication mechanism */
    NQ_UINT32 status;       /**< Status */
}
AuthParams;

/**
 * Connect to server using RPC over TCP as transport.
 *
 * @param server        Server host name.
 * @param own           Own host name.
 * @param pCredentials  Pointer to credentials to use (here used secret).
 * @param pipeDesc      Pointer to RPC pipe descriptor.
 * @param authType      Authentication type (currently schannel).
 * @return handle to RPC pipe.
 */
NQ_HANDLE ccDcerpcOverTcpConnect(const NQ_WCHAR *server, const NQ_WCHAR *own, const AMCredentials *pCredentials, const CCDcerpcPipeDescriptor *pipeDesc, NQ_BYTE authType);

/**
 * Perform BIND using RPC over TCP as transport.
 *
 * @param ip            Pointer to host IP.
 * @param port          Port to use.
 * @param hostName      Host name.
 * @param pAuthBlob     Pointer to authentication blob when used.
 * @param pipeDesc      Pointer to RPC pipe descriptor.
 * @return handle to RPC pipe.
 */
NQ_HANDLE ccDcerpcOverTcpBind(NQ_IPADDRESS *ip, NQ_PORT port, const NQ_WCHAR *hostName, AuthParams *pAuthBlob, const CCDcerpcPipeDescriptor *pipeDesc);

/**
 * Perform an RPC call using RPC over TCP as transport.
 *
 * @param pipeHandle    The handle to RPC pipe.
 * @param request       Request callback function.
 * @param response      Response callback function.
 * @param callParams    Pointer to parameters used in callback functions.
 * @return TRUE when succeeded, FALSE otherwise.
 */
NQ_BOOL ccDcerpcOverTcpCall(NQ_HANDLE pipeHandle, CCDcerpcRequestCallback request, CCDcerpcResponseCallback response, void *callParams);

/**
 * Disconnect the RPC handle.
 *
 * @param pipeHandle    The handle to RPC pipe.
 */
void ccDcerpcOverTcpDisconnect(NQ_HANDLE pipeHandle);

/**
 * Dump Rpc servers list.
 *
 * @param text  Text to print.
 */
void dumpRpcServers(const NQ_CHAR *text);


#endif /* _CCRPC_H_ */
