
/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client user operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccuser.h"
#include "ccshare.h"
#include "amspnego.h"
#include "cmsmb2.h"
#include "ccfile.h"
#include "ccsearch.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

/* -- Static data -- */
static NQ_BOOL isModuleInitialized = FALSE;

typedef struct
{
    SYMutex guard;                  /* critical section guard */
    AMCredentials admin;            /* pointer to administrative credentials */
    NQ_BOOL useAdmin;               /* whether to use admin credentials */
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

static const AMCredentials anonymous = {{{0}, {0}}, {0}, {0}, 0};

/* -- Static functions --- */

#ifdef UD_NQ_INCLUDETRACE
/*
 * Print user-specific information 
 */
static void dumpOne(CMItem * pItem)
{
    CCUser * pUser = (CCUser *)pItem;
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "  User:: UID: %d:%d%s", pUser->uid.low, pUser->uid.high, pUser->isAnonymous ? " anonymous" : "");
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "  Shares: ");
    cmListDump(&pUser->shares);
}
#endif /* UD_NQ_INCLUDETRACE */

#ifdef UD_CC_INCLUDEEXTENDEDSECURITY

/* wrapper for SMB's doSessionSetup  - used as AM callback. it converts NQ_ERR_MOREDATA into NQ_SUCCESS  */
static NQ_STATUS doSessionSetup(void * pUser, const CMBlob * outBlob, CMBlob * inBlob)
{
    NQ_STATUS status; /* operation status */
    CMBlob outBlobFragment;     /* current fragment of the outgoing blob */
    NQ_COUNT remainingLen;      /* remaining data length in the outgoing blob */
    NQ_COUNT maxFragmentLen;    /* available length of an outgoing blob fragment */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pUser:%p outBlob:%p inBlob:%p", pUser, outBlob, inBlob);

    outBlobFragment = *outBlob;
    remainingLen = outBlob->len;
    maxFragmentLen = (NQ_COUNT)(((CCUser *)pUser)->server->maxTrans - 120); /* leave enough room for headers */
    
    for (;;)
    {
        inBlob->data = NULL;    /* to check that server responds with an empty blob */
        outBlobFragment.len = remainingLen > maxFragmentLen? maxFragmentLen : remainingLen; 
        status = ((CCUser *)pUser)->server->smb->doSessionSetupExtended(pUser, &outBlobFragment, inBlob);
        if ((NQ_SUCCESS != status) && (NQ_ERR_MOREDATA != status))
        {
            goto Exit;
        }
        if (remainingLen == outBlobFragment.len)
        {
            break;
        }
        remainingLen -= maxFragmentLen;
        outBlobFragment.data += maxFragmentLen;
    }
    status = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "status:0x%08x", status);
    return status;
}
#endif /* UD_CC_INCLUDEEXTENDEDSECURITY */

/*
 * Explicitly dispose and disconnect server:
 *  - disconnects from the share
 *  - disposes private data  
 */
static void disposeUser(CCUser * pUser)
{
    CCServer * pServer = pUser->server;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", pUser);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "About to dispose user %s", cmWDump(pUser->item.name));

#ifdef UD_NQ_INCLUDESMB311
    if (NULL != pUser->savedResponse.data)
    {
        cmMemoryFreeBlob(&pUser->savedResponse);
    }
#endif /* UD_NQ_INCLUDESMB311 */

    if (pServer->masterUser != (CMItem *)pUser)
    {
        ccUserLogoff(pUser);

        /* IPC share used for validate negotiate needs to be unlocked explicitly */
        if (pServer->isNegotiationValidated)
        {
            CMIterator sharesItr;

            cmListIteratorStart(&pUser->shares, &sharesItr);
            while (cmListIteratorHasNext(&sharesItr))
            {
                CCShare *pShare = (CCShare *)cmListIteratorNext(&sharesItr);
                if (cmWStrcmp(((CMItem *)pShare)->name, cmIPCShareName()) == 0)
                {
                    cmListItemUnlock((CMItem *)pShare);
                    break;
                }
            }
            cmListIteratorTerminate(&sharesItr);
        }
        cmListShutdown(&pUser->shares);
        if (NULL != pUser->credentials)
        {
            cmMemoryFree(pUser->credentials);
            pUser->credentials = NULL;
        }
        cmListItemRemoveAndDispose((CMItem *)pUser);
    }
    else
    {
        cmListItemRemoveReference((CMItem *)pUser, (CMItem *)pServer);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 * Callback for share unlock and disposal:
 *  - disconnects from the share
 *  - disposes private data  
 */
static NQ_BOOL unlockCallback(CMItem * pItem)
{
    disposeUser((CCUser *)pItem);
    return TRUE;
}

/* query user credentials from application */
static AMCredentials * queryCredentials(const NQ_WCHAR * path)
{
    AMCredentials * pCredentials = NULL;  /* credentials allocated */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "path:%s", cmWDump(path));

    pCredentials = (AMCredentials *)cmMemoryAllocate(sizeof(AMCredentials));
    if (NULL == pCredentials)
    {
        sySetLastNqError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    /* query credentials */
    if (staticData->useAdmin)
    {
         syMemcpy(pCredentials, &staticData->admin, sizeof(AMCredentials));
    }
    else
    {
        pCredentials->domain.realm[0] = 0;
        pCredentials->type = AM_CREDENTIALS_PASSWORD_PLAINTEXT;

        syMutexTake(&staticData->guard);
        if (!udGetCredentials(path, pCredentials->user, pCredentials->password, pCredentials->domain.name))
        {
            syMutexGive(&staticData->guard);
            LOGERR(CM_TRC_LEVEL_ERROR, "udGetCredentials canceled");
            sySetLastNqError(NQ_ERR_BADPARAM);
            goto Error;
        }
        syMutexGive(&staticData->guard);
    }

    cmWStrupr(pCredentials->domain.name);

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Domain: %s", cmWDump(pCredentials->domain.name));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "User: %s", cmWDump(pCredentials->user));

    goto Exit;

Error:
    cmMemoryFree(pCredentials);
    pCredentials = NULL;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pCredentials);
    return pCredentials;
}

static CCUser * createUser(CCServer * pServer, const AMCredentials * pCredentials)
{
    CCUser * pUser;                 /* user pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p credentials:%p", pServer, pCredentials);

    /* create a user and logon to it */
    pUser = (CCUser *)cmListItemCreateAndAdd(&pServer->users, sizeof(CCUser), pCredentials->user, unlockCallback , CM_LISTITEM_LOCK , FALSE);
    if (NULL == pUser)
    {
        sySetLastNqError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    pUser->server = pServer;
    pUser->logged = FALSE;
    cmListStart(&pUser->shares);
    pUser->credentials = (AMCredentials *)cmMemoryAllocate(sizeof(AMCredentials));
    if (NULL == pUser->credentials)
    {
        cmListItemCheck((CMItem *)pUser);
        sySetLastNqError(NQ_ERR_OUTOFMEMORY);
        pUser = NULL;
        goto Exit;
    }
    syMemcpy(pUser->credentials, pCredentials, sizeof(AMCredentials));
    cmListItemAddReference((CMItem *)pUser, (CMItem *)pServer);
    pUser->isAnonymous = cmWStrlen(pUser->credentials->user) == 0;
    cmU64Zero(&pUser->uid);
    pUser->macSessionKey.data = NULL;
    pUser->sessionKey.data = NULL;
    pUser->isEncrypted = FALSE;
    pUser->isLogginOff = FALSE;
#ifdef UD_NQ_INCLUDESMB3
    pUser->encryptionKey.data = NULL;
    pUser->decryptionKey.data = NULL;
    pUser->applicationKey.data = NULL;
#ifdef UD_NQ_INCLUDESMB311
    pUser->isPreauthIntegOn = FALSE;
    pUser->savedResponse.data = NULL;
#endif /* UD_NQ_INCLUDESMB311 */
#endif /* UD_NQ_INCLUDESMB3 */
    pUser->isGuest = FALSE;
#ifdef UD_NQ_INCLUDETRACE
    pUser->item.dump = dumpOne; 
#endif /* UD_NQ_INCLUDETRACE */

    if (!ccUserLogon(pUser))
    {
        cmListItemUnlock((CMItem *)pUser);
        pUser = NULL;
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pUser);
    return pUser;
}

static void cleanCallback(void * context)
{
    CCUser * pUser = (CCUser *)context;

    cmU64Zero(&pUser->uid);
}

CCUser * findUser(CCServer * pServer, const AMCredentials * pCredentials)
{
    CMIterator iterator;    /* user iterator */
    CCUser * pUser;

    cmListIteratorStart(&pServer->users, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        pUser = (CCUser *)cmListIteratorNext(&iterator);
        if (pUser->item.findable &&
            NULL != pUser->credentials &&
            0 == cmWStrcmp(pCredentials->user, pUser->credentials->user) && 
            0 == cmWStrcmp(pCredentials->domain.name, pUser->credentials->domain.name) && 
            0 == cmWStrcmp(pCredentials->password, pUser->credentials->password) &&
            pCredentials->type == pUser->credentials->type
            )
        {
            cmListItemLock((CMItem *)pUser);
            goto Exit;
        }
    }
    pUser = NULL;

Exit:
    cmListIteratorTerminate(&iterator);
    return pUser;
}

/* -- API Functions */

NQ_BOOL ccUserStart(void)
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
       goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate user data");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    syMutexCreate(&staticData->guard);
    staticData->useAdmin = FALSE;
    isModuleInitialized = TRUE;
    result = TRUE;

Exit:

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return result;
}

void ccUserShutdown(void)
{
   LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

   if (isModuleInitialized)
   {
      syMutexDelete(&staticData->guard);

#ifdef SY_FORCEALLOCATION
      if ( NULL != staticData )
      {
         cmMemoryFreeShutdown(staticData);
      }
      staticData = NULL;
#endif /* SY_FORCEALLOCATION */

      isModuleInitialized = FALSE;
   }

   LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

const AMCredentials * ccUserGetAnonymousCredentials(void)
{
    return &anonymous;
}

NQ_BOOL ccUserIsAnonymousCredentials(const AMCredentials *pCredentials)
{
    return (0 == cmWStrcmp(pCredentials->user, anonymous.user) &&
            0 == cmWStrcmp(pCredentials->domain.name, anonymous.domain.name) &&
            0 == cmWStrcmp(pCredentials->password, anonymous.password));
}

CCUser * ccUserFindById(CCServer * pServer, NQ_UINT64 uid)
{
    CMIterator iterator;    /* user iterator */
    CCUser * pNextUser;     /* next user pointer */

    ccServerIterateUsers(pServer, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        pNextUser = (CCUser *)cmListIteratorNext(&iterator);
        cmListItemTake((CMItem *)pNextUser);
        if (0 == cmU64Cmp(&uid, &pNextUser->uid))
        {
            if (FALSE == pNextUser->item.findable)
            {
                cmListItemGive((CMItem *)pNextUser);
                pNextUser = NULL;
            }
            else
            {
                cmListItemGive((CMItem *)pNextUser);
            }

            goto Exit;
        }

        cmListItemGive((CMItem *)pNextUser);
    }

    pNextUser = NULL;

Exit:
    cmListIteratorTerminate(&iterator);
    return pNextUser;
}

void ccUserIterateShares(CCUser * pUser, CMIterator * iterator)
{
    cmListIteratorStart(&pUser->shares, iterator);
}

NQ_BOOL ccUserUseSignatures(CCUser * pUser)
{
    return !pUser->isAnonymous && !pUser->isGuest;
}

void ccUserSetAdministratorCredentials(const AMCredentials * credentials)
{
    if (NULL != credentials)
    {
        staticData->admin = *credentials;
    }
    staticData->useAdmin = NULL != credentials;
}

const AMCredentials * ccUserGetAdministratorCredentials(void)
{
    return staticData->useAdmin ? &staticData->admin : NULL;
}

CCUser * ccUserGet(CCServer * pServer, const NQ_WCHAR * path, const AMCredentials ** pCredentials)
{
    CCUser * pUser = NULL;                                  /* user pointer */
    const AMCredentials * credentials = * pCredentials;    /* credentials to use */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p path:%s credentials:%p", pServer, cmWDump(path), pCredentials);

    cmListItemTake((CMItem *)pServer);

    /* find existing: this is only possible when credentials are supplied */
    if (NULL != credentials)
    {
        LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "credentials were supplied, credentials: %p", credentials);
        pUser = findUser(pServer, credentials);
        if (NULL != pUser)
        {
            if (!pUser->logged && !ccUserLogon(pUser))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "User log on failed");
                cmListItemUnlock((CMItem *)pUser);
                pUser = NULL;
                goto Exit;
            }

            goto Exit;
        }

        pUser = createUser(pServer, credentials);
        if (NULL == pUser)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Create user failed");
            goto Exit;
        }

        goto Exit;
    }
    else
    {
        credentials = queryCredentials(path);
        if (NULL == credentials)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastNqError(NQ_ERR_OUTOFMEMORY);
            pUser = NULL;
            goto Exit;
        }

        pUser = findUser(pServer, credentials);
        if (NULL == pUser || (NULL != pUser && !pUser->logged))
        {
            if (pUser != NULL)
            {
                cmListItemUnlock((CMItem *)pUser); /* remove the lock from findUser */
            }

            pUser = createUser(pServer, credentials);
            if (NULL == pUser)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Create user failed");
            }
        }

        cmMemoryFree(credentials);
    }

Exit:
    cmListItemGive((CMItem *)pServer);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pUser);
    return pUser;
}

void ccUserLogoff(CCUser * pUser)
{
    CMIterator iterator;        /* to enumerate users */
    CCServer * pServer;         /* server pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", pUser);

    pServer = pUser->server;

    cmListItemTake(&pServer->item);
    cmListItemTake(&pUser->item);

    if (!pUser->logged || pUser->isLogginOff)
    {
        cmListItemGive(&pUser->item);
        cmListItemGive(&pServer->item);
        goto Exit;
    }
    pUser->isLogginOff = TRUE;

    cmListItemGive(&pUser->item);
    cmListItemGive(&pServer->item);

    cmListIteratorStart(&pUser->shares, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        CCShare * pShare;           /* next share pointer */
        
        pShare = (CCShare *)cmListIteratorNext(&iterator);
        ccShareDisconnect(pShare);
    }
    cmListIteratorTerminate(&iterator);

    pServer = pUser->server;
    if (NULL!= pServer->smb && pUser->logged)
    {
        pUser->isLogginOff = TRUE;
        pServer->smb->doLogOff(pUser);
        pUser->isLogginOff = FALSE;
        pUser->logged = FALSE;
    }
    amSpnegoFreeKey(&pUser->sessionKey);
    amSpnegoFreeKey(&pUser->macSessionKey);
#ifdef UD_NQ_INCLUDESMB3
    cmMemoryFreeBlob(&pUser->encryptionKey);
    cmMemoryFreeBlob(&pUser->decryptionKey);
    cmMemoryFreeBlob(&pUser->applicationKey);
#endif /* UD_NQ_INCLUDESMB3 */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

NQ_BOOL ccUserLogon(CCUser * pUser)
{
    NQ_STATUS status;           /* SPNEGO status */
    CCServer * pServer;         /* server pointer */
    NQ_BOOL result = FALSE;     /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", pUser);

    cmListItemTake((CMItem *)pUser);
    if (pUser->logged)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Logon: user already logged in");
        result = TRUE;
        goto Exit;
    }

    pServer = pUser->server;
#ifdef UD_NQ_INCLUDESMB311
    syMemcpy(pUser->preauthIntegHashVal , pServer->preauthIntegHashVal , SMB3_PREAUTH_INTEG_HASH_LENGTH);
    pUser->isPreauthIntegOn = TRUE; /* hashing process starting. turn on hash flag */
#endif /* UD_NQ_INCLUDESMB311 */
    
    amSpnegoFreeKey(&pUser->sessionKey);       /* in case of reconnect, otherwise - will be NULL */
    amSpnegoFreeKey(&pUser->macSessionKey);
#ifdef UD_NQ_INCLUDESMB3
    amSpnegoFreeKey(&pUser->encryptionKey);
    amSpnegoFreeKey(&pUser->decryptionKey);
    amSpnegoFreeKey(&pUser->applicationKey);
#endif /* UD_NQ_INCLUDESMB3 */

#ifdef UD_CC_INCLUDEEXTENDEDSECURITY
    if (pServer->useExtendedSecurity)
    {
        NQ_STATUS res;              /* result of keys derivation */
        NQ_BOOL restrictCrypters;   /* whether to allow all crypters or not */
    
        restrictCrypters = (pServer->capabilities & CC_CAP_MESSAGESIGNING) && pServer->smb->restrictCrypters;
        status = amSpnegoClientLogon(
            pUser, 
            pServer->item.name, 
            pUser->credentials, 
            restrictCrypters, 
            &pServer->firstSecurityBlob, 
            &pUser->sessionKey,
            &pUser->macSessionKey,
            doSessionSetup,
            cleanCallback
            );
        if (AM_SPNEGO_SUCCESS == status)
        {
            pUser->logged = TRUE;
        }

        if ((AM_SPNEGO_SUCCESS == status) && (NULL != pUser->macSessionKey.data) && (pUser->macSessionKey.len > pServer->smb->maxSigningKeyLen))
        {
            pUser->macSessionKey.len = pServer->smb->maxSigningKeyLen;  /* restrict bigger keys */
        }

        /* fail the login if key derivation fails */
        res = pUser->server->smb->keyDerivation(pUser);
        if (NQ_SUCCESS != res)
        {
            amSpnegoFreeKey(&pUser->macSessionKey);
            status = AM_SPNEGO_DENIED;
            pUser->logged = FALSE;
            sySetLastError(res);
        }

#ifdef UD_NQ_INCLUDESMB3
        if (AM_SPNEGO_SUCCESS != status)
        {
            amSpnegoFreeKey(&pUser->encryptionKey);
            amSpnegoFreeKey(&pUser->decryptionKey);
            amSpnegoFreeKey(&pUser->applicationKey);
        }
#endif /* UD_NQ_INCLUDESMB3 */

        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Logon: %s", AM_SPNEGO_SUCCESS == status ? "ok" : "failed");
        result = (AM_SPNEGO_SUCCESS == status);
        goto ExitSigning;
    }
    else
#endif /* UD_CC_INCLUDEEXTENDEDSECURITY */
    {
        NQ_INT level;           /* encryption level */

        for (level = AM_MAXSECURITYLEVEL; level >= 0; level--)
        {
            CMBlob pass1;   /* first password blob */
            CMBlob pass2;   /* second password blob */

            if (NULL == pServer->firstSecurityBlob.data)
            {
                sySetLastError(NQ_ERR_BADPARAM);
                goto Exit;
            }
            pUser->sessionKey = cmMemoryCloneBlob(&pServer->firstSecurityBlob);
            
            if (pUser->isAnonymous || !pServer->userSecurity)
            {
                pass1.data = NULL;
                pass1.len = 0;
                pass2.data = NULL;
                pass2.len = 0;
                
                status = pServer->smb->doSessionSetup(pUser, &pass1, &pass2);
                if (NQ_SUCCESS == status)
                {
                    pUser->logged = TRUE;                   
                    result = TRUE;
                    goto ExitSigning;
                }
                amSpnegoFreeKey(&pUser->sessionKey);
                sySetLastNqError(status);
                continue;
            }
            if (AM_SPNEGO_SUCCESS == amSpnegoGeneratePasswordBlobs(pUser->credentials, level, &pass1, &pass2, &pUser->sessionKey, &pUser->macSessionKey))
            {
                status = pServer->smb->doSessionSetup(pUser, &pass1, &pass2);
                amSpnegoFreeKey(&pass1);
                amSpnegoFreeKey(&pass2);
                
                if (NQ_SUCCESS == status)
                {
                    pUser->logged = TRUE;
                    if (NULL != pUser->macSessionKey.data && pUser->macSessionKey.len > SMB2_CRYPTO_KEY_SIZE)
                    {
                        pUser->macSessionKey.len = SMB2_CRYPTO_KEY_SIZE;  /* restrict bigger keys */
                    }
                    result = TRUE;
                    goto ExitSigning;
                }
                amSpnegoFreeKey(&pUser->sessionKey);
                amSpnegoFreeKey(&pUser->macSessionKey);
#ifdef UD_NQ_INCLUDESMB3
                amSpnegoFreeKey(&pUser->encryptionKey);
                amSpnegoFreeKey(&pUser->decryptionKey);
                amSpnegoFreeKey(&pUser->applicationKey);
#endif /* UD_NQ_INCLUDESMB3 */
                sySetLastNqError(status);
            }
            else
            {
                amSpnegoFreeKey(&pUser->sessionKey);
                sySetLastNqError(NQ_ERR_NOSUPPORT);
            }
        }
    }

ExitSigning:
    if (TRUE == pUser->logged)
    {
        if ((SMB2_NEGOTIATE_SIGNINGREQUIRED & pServer->serverSecurityMode) && ccGetSigningPolicy())
        {
            pServer->useSigning = TRUE;
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Signing ON");
        }
    }
Exit:
    cmListItemGive((CMItem *)pUser);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL ccUserReconnectShares(CCUser * pUser)
{
    CMIterator  iterator;       /* to enumerate users */
    NQ_BOOL     res = FALSE;
    
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", pUser);
    
    cmListIteratorStart(&pUser->shares, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        CCShare * pShare;           /* next share pointer */
        
        pShare = (CCShare *)cmListIteratorNext(&iterator);

        /* IPC share which used for negotiate validation might already be connected */
        if (TRUE == pShare->connected)
        {
            continue;
        }

        res = ccShareConnectExisting(pShare, NULL, TRUE);
        if (res)
        {
            /* if a single file failed to restore we still return true so other files will continue previous activity */
            ccShareReopenFiles(pShare);
        }
    }
    cmListIteratorTerminate(&iterator);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", res ? "TRUE" : "FALSE");
    return res;
}

NQ_BOOL ccUserRelogon(CCUser * pUser)
{
    CMIterator  shareIterator;       /* to enumerate shares */
    NQ_BOOL     result = FALSE;      /* result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "user:%p", pUser);
    cmListItemTake((CMItem *)pUser);
    pUser->logged = FALSE;
    cmU64Zero(&pUser->uid);
    amSpnegoFreeKey(&pUser->sessionKey);
    amSpnegoFreeKey(&pUser->macSessionKey);
#ifdef UD_NQ_INCLUDESMB3
    cmMemoryFreeBlob(&pUser->encryptionKey);
    cmMemoryFreeBlob(&pUser->decryptionKey);
    cmMemoryFreeBlob(&pUser->applicationKey);
#endif /* UD_NQ_INCLUDESMB3 */
    cmListIteratorStart(&pUser->shares, &shareIterator);
    while (cmListIteratorHasNext(&shareIterator))
    {
        CCShare * pShare;               /* next share pointer */

        pShare = (CCShare *)cmListIteratorNext(&shareIterator);
        pShare->connected = FALSE;
    }

    cmListIteratorTerminate(&shareIterator);
    result = ccUserLogon(pUser);

    if (TRUE == result)
    {
        result = ccUserReconnectShares(pUser);
    }

    if (FALSE == result)
    {
        cmListItemLock((CMItem *)pUser);
        cmListIteratorStart(&pUser->shares,&shareIterator);
        while (cmListIteratorHasNext(&shareIterator))
        {
            CMIterator  mntItr , iterator;
            CCShare *   pShare = (CCShare *)cmListIteratorNext(&shareIterator);

            /* removing files and searches before removing the mount to avoid SegFault*/
            cmListIteratorStart(&pShare->files, &iterator);
            while (cmListIteratorHasNext(&iterator))
            {
                CCFile * pFile = NULL;

                pFile = (CCFile *)cmListIteratorNext(&iterator);
                cmListItemUnlock((CMItem *)pFile);
            }

            cmListIteratorTerminate(&iterator);
            cmListIteratorStart(&pShare->searches, &iterator);
            while (cmListIteratorHasNext(&iterator))
            {
                CCSearch * pSearch;

                pSearch = (CCSearch *)cmListIteratorNext(&iterator);
                cmListItemUnlock((CMItem *)pSearch);
            }

            cmListIteratorTerminate(&iterator);
            ccMountIterateMounts(&mntItr);
            while (cmListIteratorHasNext(&mntItr))
            {
                CCMount * pMount = (CCMount *)cmListIteratorNext(&mntItr);

                if (pMount->share == pShare)
                {
                    cmListItemUnlock((CMItem *)pMount);
                }
            }

            cmListIteratorTerminate(&mntItr);
        }
        cmListIteratorTerminate(&shareIterator);
        cmListItemGive((CMItem *)pUser);
        cmListItemUnlock((CMItem *)pUser);
        goto Exit;
    }

    cmListItemGive((CMItem *)pUser);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

#endif /* UD_NQ_INCLUDECIFSCLIENT */
