/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client file operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccfile.h"
#include "ccserver.h"
#include "ccutils.h"
#include "ccparams.h"
#include "cmfscifs.h"
#include "ccdfs.h"
#include "cmlist.h"
#include "ccsmb10.h"
#include "ccinfo.h"
#include "cmsmb2.h"
#include "cmuuid.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

/* -- Constants -- */
#define MAX_CHUNKS_NUM 16               /* The max number of chunks in one server-side copy operation */
#define MAX_CHUNK_COPY_SIZE 1048576     /* The max number of bytes to copy in a single chunk of one server-side copy operation - 1MB */

#define ACCESSMASKDEFAULT (       \
    SMB_DESIREDACCESS_SYNCHRONISE |   \
    SMB_DESIREDACCESS_READCONTROL |   \
    SMB_DESIREDACCESS_READATTRIBUTES)

/* -- Static functions --- */

/*
 * Explicitly close and dispose file:
 *  - disconnects from the share
 *  - disposes private data
 */
static void disposeFile(CCFile * pFile)
{
    CCServer * pServer;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "About to dispose file %s", cmWDump(pFile->item.name));

    pServer = pFile->share->user->server;
    if (NULL!= pServer->smb && pFile->open)
    {
        pServer->smb->doClose(pFile);
    }
    cmListItemRemoveAndDispose((CMItem *)pFile);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 * Callback for file unlock and disposal:
 *  - disconnects from the share
 *  - disposes private data
 */
static NQ_BOOL unlockCallback(CMItem * pItem)
{
    CCFile  *   pFile = (CCFile *)pItem;

    disposeFile(pFile);
    return TRUE;
}

#ifdef UD_CC_INCLUDEDFS
static void cloneFileData(CCFile *from, CCFile *to)
{
    if (from && to)
    {
        to->accessMask = from->accessMask;
        to->attributes = from->attributes;
        to->disposition = from->disposition;
        to->open = from->open;
        to->options = from->options;
        to->sharedAccess = from->sharedAccess;
    }
}
#endif /* UD_CC_INCLUDEDFS */

/*
 * Print share-specific information
 */
#ifdef UD_NQ_INCLUDETRACE
static void dumpOne(CMItem * pItem)
{
    CCFile * pFile = (CCFile *)pItem;
    NQ_BYTE * fid = pFile->fid;
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "  File:: FID: %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x access: %08x", \
          fid[0], fid[1], fid[2], fid[3],   \
          fid[4], fid[5], fid[6], fid[7],   \
          fid[8], fid[9], fid[10], fid[11], \
          fid[12], fid[13], fid[14], fid[15], \
          pFile->accessMask             \
           );
}
#endif /* UD_NQ_INCLUDETRACE */

/*
 * Server-side data copy operation sends to the server an array of Chunk with ranges to copy.
 * Each item in the array contains the source offset, the destination offset, and the number of bytes to copy.
 * Its under the user response to free the chunks pointer.
 */
static NQ_BOOL createChunkArray(NQ_UINT64 startOffset, NQ_UINT64 eof, NQ_UINT32 maxChunksNum, NQ_UINT32 maxChunkCopySize, CCChunks* pChunks)
{
    NQ_UINT64   bytesLeftToCopy;
    NQ_UINT32   chunksNum = 0;
    NQ_BOOL     res = FALSE;
    NQ_UINT64   curOffset;
    NQ_UINT64   tempOffset;
    NQ_UINT32   curChunkCopySize = 0;
    NQ_UINT64   minChunkCopySize;
    NQ_UINT     i = 0;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "startOffset.high:%u startOffset.low:%u eof.high:%u eof.low:%u maxChunksNum:%u maxChunkCopySize:%u pChunks:%p", startOffset.high, startOffset.low, eof.high, eof.low, maxChunksNum, maxChunkCopySize, pChunks);

    cmU64AssignU64(&curOffset, &startOffset);
    cmU64SubU64U64(&bytesLeftToCopy, &eof, &startOffset);

    /* Calculates the number of the chunks */
    if (0 != bytesLeftToCopy.high)
    {
        chunksNum = maxChunksNum;
    }
    else
    {
        if (0 == maxChunkCopySize)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "maxChunkCopySize = 0, can not divide bytesLeftToCopy.low by zero.");
            sySetLastError(NQ_ERR_BADPARAM);
            goto Exit;
        }

        chunksNum = bytesLeftToCopy.low / maxChunkCopySize;
        /* In case that chunksNum calculation will be zero but there is still data to be written */
        if ((0 == chunksNum) && (0 < bytesLeftToCopy.low))
        {
            chunksNum = 1;
        }

        chunksNum = chunksNum > maxChunksNum ? maxChunksNum : chunksNum;
    }

    pChunks->chunks = (CCChunk*)cmMemoryAllocate(chunksNum * (NQ_UINT)sizeof(CCChunk));
    if (NULL == pChunks->chunks)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    pChunks->numberOfChunks = chunksNum;

    for (i = 0 ; i < chunksNum ; i++)
    {
        tempOffset.high = curOffset.high;
        tempOffset.low = curOffset.low;
        cmU64AddU32(&tempOffset, maxChunkCopySize);

        /* Check if curOffset + maxChunkCopySize exceed the eof */
        minChunkCopySize = cmU64Min(&tempOffset, &eof);

        /* If curOffset + maxChunkCopySize exceed the eof size, add just the relative size */
        if (0 == cmU64Cmp(&minChunkCopySize, &eof))
        {
            curChunkCopySize = eof.low - curOffset.low;
        }
        else
        {
            curChunkCopySize = maxChunkCopySize;
        }

        /* Assign the right data for each chunk */
        pChunks->chunks[i].sourceOffset = curOffset;
        pChunks->chunks[i].targetOffset = curOffset;
        pChunks->chunks[i].length = curChunkCopySize;

        cmU64AddU32(&curOffset, curChunkCopySize);
    }

    res = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", res ? "TRUE" : "FALSE", syGetLastError());
    return res;
}

/*
 * Server-side data copy query resume key from the server.
 */
static NQ_BOOL queryResumeFileKey(CCFile* pSrcFile, CCMountIdentifier* mountPointID, CCResumeKey* pKey)
{
    NQ_COUNT    counter;            /* Retry loop counter */
    NQ_STATUS   res = NQ_SUCCESS;   /* Operation result */
    NQ_BOOL     result = FALSE;     /* Return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pSrcFile:%p mountPointID:%p pKey:%p", pSrcFile, mountPointID, pKey);

    if ((NULL == pSrcFile) || (NULL == mountPointID) || (NULL == pKey))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid parameter");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    syMemcpy(mountPointID, &pSrcFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0 ; counter < CC_CONFIG_RETRYCOUNT ; counter++)
    {
        res = pSrcFile->share->user->server->smb->doQueryResumeFileKey(pSrcFile, pKey);
        switch (res)
        {
            case (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED:
                pSrcFile->share->user->server->transport.connected = FALSE;
                if (!ccServerReconnect(pSrcFile->share->user->server))
                {
                    ccUpdateMountPointValidity(mountPointID);
                    if (MOUNTPOINT_INVALID == mountPointID->status)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                    }

                    goto Exit;
                }

                break;
            case (NQ_STATUS)NQ_ERR_NOTCONNECTED:
                ccUpdateMountPointValidity(mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID->status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                goto Exit;
            case (NQ_STATUS)NQ_ERR_TRYAGAIN:
                ccUpdateMountPointValidity(mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID->status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                    goto Exit;
                }

                /* possible reconnect caused a wrong user session or wrong FID */
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Operation failed. Try again");
                continue;
            default:
                goto Exit;
        }
    }

Exit:
    if (NQ_SUCCESS == res)
    {
        result = TRUE;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}


/* -- API Functions */

NQ_BOOL ccFileStart(void)
{
    return TRUE;
}

void ccFileShutdown(void)
{

}

CCFile * ccFileFind(CCShare * pShare, const NQ_WCHAR * path)
{
    return NULL;  /* this call is not expected */
}

CCFile * ccFileFindById(CCServer * pServer, const NQ_BYTE * id)
{
    CMIterator userIterator;        /* user iterator */
    CCFile * pFile;                 /* next file pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p id:%p", pServer, id);

    cmListItemTake((CMItem *)pServer);
    ccServerIterateUsers(pServer, &userIterator);
    while (cmListIteratorHasNext(&userIterator))
    {
        CCUser * pUser;                 /* next user pointer */
        CMIterator shareIterator;       /* share iterator */

        pUser = (CCUser *)cmListIteratorNext(&userIterator);
        ccUserIterateShares(pUser, &shareIterator);
        while (cmListIteratorHasNext(&shareIterator))
        {
            CCShare * pShare;               /* next share pointer */
            CMIterator  fileIterator;       /* file Iterator*/

            pShare = (CCShare *)cmListIteratorNext(&shareIterator);
            cmListIteratorStart(&pShare->files, &fileIterator);
            while (cmListIteratorHasNext(&fileIterator))
            {
                pFile = (CCFile *)cmListIteratorNext(&fileIterator);
                if (0 == syMemcmp(id, pFile->fid, sizeof(pFile->fid)))
                {
                    cmListIteratorTerminate(&shareIterator);
                    cmListIteratorTerminate(&fileIterator);
                    goto Exit;
                }
            }
            cmListIteratorTerminate(&fileIterator);
        }
        cmListIteratorTerminate(&shareIterator);
    }
    pFile = NULL;

Exit:
    cmListIteratorTerminate(&userIterator);
    cmListItemGive((CMItem *)pServer);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pFile);
    return pFile;
}

NQ_HANDLE ccGetFileHandleByName(const NQ_CHAR *path, NQ_INT desiredAccess, NQ_UINT32 desiredSharedAccess, NQ_BYTE desiredOpLock, NQ_UINT16 attributes)
{
    CCFile  *pFile = NULL;                 /* next file pointer */
    CCMount *pMount;
    CMIterator iterator;
    NQ_UINT32 desiredAccessMask;
    NQ_UINT32 requiredShareAccess;
    NQ_WCHAR *pathW = NULL;
    NQ_WCHAR *filePath = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "path:%s", path);

    if (NULL == path)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid path");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit1;
    }

    pathW = cmMemoryCloneAString(path);
    if (NULL == pathW)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Out of memory.");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit1;
    }

    if (ccUtilsPathIsLocal(pathW))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "The supplied path is local");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit1;
    }

    pMount = ccMountFind(pathW);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot find mount point");
        sySetLastError(NQ_ERR_BADPATH);
        goto Exit1;
    }

    filePath = ccUtilsFilePathFromLocalPath(pathW, pMount->pathPrefix, pMount->share->user->server->smb->revision == CCCIFS_ILLEGALSMBREVISION, TRUE);
    if (NULL == filePath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit1;
    }

    cmListIteratorStart(&(pMount->share->files), &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        pFile = (CCFile *)cmListIteratorNext(&iterator);
        if (NULL != pFile->item.name && 0 == cmWStricmp(filePath, pFile->item.name))
        {
            if (!pFile->item.findable)
            {
                continue;
            }

            /* check this item had desired properties */
            if (desiredOpLock > pFile->grantedOplock)
            {
                continue;
            }

            switch (desiredAccess)
            {
                case FILE_AM_READ:
                    desiredAccessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_READDATA;
                    break;
                case FILE_AM_WRITE:
                    if (attributes & SMB_ATTR_DIRECTORY)
                    {
                        desiredAccessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES;
                    }
                    else
                    {
                        desiredAccessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES + SMB_DESIREDACCESS_WRITEDATA;
                    }
                    break;
                case FILE_AM_READ_WRITE:
                    if (attributes & SMB_ATTR_DIRECTORY)
                    {
                        desiredAccessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES;
                    }
                    else
                    {
                        desiredAccessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES + SMB_DESIREDACCESS_READDATA + SMB_DESIREDACCESS_WRITEDATA;
                    }
                    break;
                default:
                        desiredAccessMask = ACCESSMASKDEFAULT;
                    break;
            }

            if ((pFile->accessMask & desiredAccessMask) != desiredAccessMask)
            {
                continue;
            }

            switch (desiredSharedAccess)
            {
            case FILE_SM_COMPAT:
                {
                    requiredShareAccess = SMB_SHAREACCESS_READ;
                    if (!(desiredAccessMask & SMB_DESIREDACCESS_WRITEDATA))
                    {
                        requiredShareAccess |= SMB_SHAREACCESS_WRITE;
                    }
                    if (!(desiredAccessMask & SMB_DESIREDACCESS_DELETE))
                    {
                        requiredShareAccess |= SMB_SHAREACCESS_DELETE;
                    }
                }
                break;
            case FILE_SM_DENY_NONE:
                requiredShareAccess = SMB_SHAREACCESS_WRITE | SMB_SHAREACCESS_READ | SMB_SHAREACCESS_DELETE;
                break;
            case FILE_SM_DENY_READ:
                requiredShareAccess = SMB_SHAREACCESS_WRITE | SMB_SHAREACCESS_DELETE;
                break;
            case FILE_SM_DENY_WRITE:
                requiredShareAccess = SMB_SHAREACCESS_READ | SMB_SHAREACCESS_DELETE;
                break;
            case FILE_SM_EXCLUSIVE:
                /* no break */
            default:
                requiredShareAccess = SMB_SHAREACCESS_NONE;
                break;
            }
            if ((pFile->sharedAccess & requiredShareAccess) != requiredShareAccess)
            {
                continue;
            }

            goto Exit;
        }
    }
    pFile = NULL;


Exit:
    cmListIteratorTerminate(&iterator);
Exit1:
    cmMemoryFree(pathW);
    cmMemoryFree(filePath);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pFile);
    return (NQ_HANDLE)pFile;
}


CCFile * ccFileCreate(CCShare * pShare, CCMount * pMount, const NQ_WCHAR * path)
{
    CCFile * pFile;       /* File pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p mount:%p path:%s", pShare, pMount, cmWDump(path));

    /* we do not care of another file with the same name */
    pFile = (CCFile *)cmListItemCreateAndAdd(&pShare->files, sizeof(CCFile), path, unlockCallback, CM_LISTITEM_LOCK , FALSE);
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        cmListItemCheck((CMItem *)pShare);    /* try disposal */
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "pFile: %s", cmWDump(pFile->item.name));
    pFile->grantedOplock = SMB2_OPLOCK_LEVEL_NONE;
    pFile->open = FALSE;
    pFile->share = pShare;
    ccInitializeMountPointIdentifier(pMount, &pFile->mountPointID);
    pFile->disconnected = FALSE;
#ifdef UD_NQ_INCLUDESMB2
    pFile->durableState = DURABLE_REQUIRED;
    pFile->durableFlags = 0;
    pFile->durableTimeout = 0;
    cmGenerateUuid(&pFile->durableHandle);
#endif /* UD_NQ_INCLUDESMB2 */
    cmU64Zero(&pFile->volumeId);
    cmListItemAddReference((CMItem *)pFile, (CMItem *)pShare);
#ifdef UD_NQ_INCLUDETRACE
    pFile->item.dump = dumpOne;
#endif /* UD_NQ_INCLUDETRACE */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pFile);
    return pFile;
}

NQ_HANDLE ccCreateFileA(
    const NQ_CHAR *fileName,
    NQ_INT access,
    NQ_INT shareMode,
    NQ_INT locality,
    NQ_BOOL writeThrough,
    NQ_UINT16 attributes,
    NQ_INT createAction,
    NQ_INT openAction
    )
{
    NQ_WCHAR * fileNameW = NULL; /* name in Unicode */
    NQ_HANDLE res = NULL;        /* delegated call result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (NULL == fileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid FileName");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    fileNameW = cmMemoryCloneAString(fileName);
    if (NULL == fileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    res = ccCreateFile(fileNameW, access, shareMode, locality, writeThrough, attributes, createAction, openAction);

Exit:
    cmMemoryFree(fileNameW);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p error:0x%x", res, syGetLastError());
    return res;
}

NQ_HANDLE ccCreateFile(
    const NQ_WCHAR * fileName,
    NQ_INT access,
    NQ_INT shareMode,
    NQ_INT locality,
    NQ_BOOL writeThrough,
    NQ_UINT16 attributes,
    NQ_INT createAction,
    NQ_INT openAction
    )
{
    CCMount * pMount;      /* mount point descriptor */
    CCFile * pFile = NULL; /* file handle */
    CCShare * pShare;      /* pointer to the hosting share */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%s access:%d mode:%d local:%d write:%s attr:0x%x create:%d open:%d", cmWDump(fileName), access, shareMode, locality, writeThrough ? "TRUE" : "FALSE", attributes, createAction, openAction);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "fileName: %s", cmWDump(fileName));

    if (NULL == fileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid FileName");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (ccUtilsPathIsLocal(fileName))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "The supplied path is local");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pMount = ccMountFind(fileName);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot find mount point");
        sySetLastError(NQ_ERR_BADPATH);
        goto Exit;
    }
    pShare = pMount->share;
    pFile = ccFileCreateOnServer(
            pShare,
            fileName,
            TRUE,
            access,
            shareMode,
            locality,
            writeThrough,
            attributes,
            createAction,
            openAction,
            FALSE,
            FILE_OPLOCK_LEVEL_BATCH
            );

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p error:0x%x", pFile, syGetLastError());
    return (NQ_HANDLE)pFile;
}

NQ_HANDLE ccCreateFileNoBatch(
    const NQ_WCHAR * fileName,
    NQ_INT access,
    NQ_INT shareMode,
    NQ_INT locality,
    NQ_BOOL writeThrough,
    NQ_UINT16 attributes,
    NQ_INT createAction,
    NQ_INT openAction
    )
{
    CCMount * pMount;      /* mount point descriptor */
    CCFile * pFile = NULL; /* file handle */
    CCShare * pShare;      /* pointer to the hosting share */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%s access:%d mode:%d local:%d write:%s attr:0x%x create:%d open:%d", cmWDump(fileName), access, shareMode, locality, writeThrough ? "TRUE" : "FALSE", attributes, createAction, openAction);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "fileName: %s", cmWDump(fileName));

    if (NULL == fileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid FileName");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (ccUtilsPathIsLocal(fileName))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "The supplied path is local");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pMount = ccMountFind(fileName);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot find mount point");
        sySetLastError(NQ_ERR_BADPATH);
        goto Exit;
    }
    pShare = pMount->share;
    pFile = ccFileCreateOnServer(
            pShare,
            fileName,
            TRUE,
            access,
            shareMode,
            locality,
            writeThrough,
            attributes,
            createAction,
            openAction,
            FALSE,
            FILE_OPLOCK_LEVEL_NONE
            );

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pFile);
    return (NQ_HANDLE)pFile;
}

NQ_BOOL ccCloseHandle(NQ_HANDLE handle)
{
    CCFile *    pFile = (CCFile *)handle;   /* casted pointer to file */
    NQ_STATUS   res;                        /* exchange status */
    NQ_INT      counter;                    /* simple counter*/
    CCMountIdentifier mountPointID;         /* mount point identifier */
    NQ_BOOL     result = FALSE;             /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p", handle);

    if (NULL == handle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "NULL Handle");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }
    if (!ccValidateFileHandle(handle))
    {
        cmListItemUnlock((CMItem *)pFile); /* remove one lock for one use*/
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (!ccTransportIsConnected(&pFile->share->user->server->transport) && !ccServerReconnect(pFile->share->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT ; counter++)
    {
        res = pFile->share->user->server->smb->doClose(pFile);
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            /* possible reconnect caused a wrong user session or wrong FID */
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

    pFile->open = FALSE;
    cmListItemUnlock((CMItem *)pFile);
    sySetLastError((NQ_UINT32)res);
    result = (res == NQ_SUCCESS);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

CCFile * ccFileCreateOnServer(
    CCShare * pShare,
    const NQ_WCHAR * path,
    NQ_BOOL pathIsLocal,
    NQ_INT access,
    NQ_INT shareMode,
    NQ_INT locality,
    NQ_BOOL writeThrough,
    NQ_UINT16 attributes,
    NQ_INT createAction,
    NQ_INT openAction,
    NQ_BOOL isPipe,
    NQ_INT oplockLevel
    )
{
    CCFile * pFile = NULL;                  /* file handle */
    NQ_STATUS res = NQ_SUCCESS;             /* exchange status */
    NQ_INT counter;                         /* operation attempts counter */
    const NQ_WCHAR * filePath = NULL;
#ifdef UD_CC_INCLUDEDFS
    CCDfsContext dfsContext = {CC_DFS_NUMOFRETRIES, 0, NULL, NULL};   /* DFS operations context */
    CCDfsResult dfsResult = {NULL, NULL, NULL};                 /* result of DFS resolution */
#endif /* UD_CC_INCLUDEDFS */
    NQ_BOOL isDfs = FALSE;                  /* whether share is dfs (then full path should be build) */
    CCMount *pMount = NULL;                 /* pointer to mount point */
    CCMountIdentifier mountPointID;         /* mount point identifier */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pShare:%s path:%p", cmWDump(pShare->item.name), path);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "path: %s", path ? cmWDump(path) : "");

    if (!ccTransportIsConnected(&pShare->user->server->transport) && !ccServerReconnect(pShare->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        res = NQ_ERR_NOTCONNECTED;
        goto Exit;
    }

#ifdef UD_CC_INCLUDEDFS
    if (!isPipe && !pShare->isPrinter && (pShare->flags & CC_SHARE_IN_DFS))
    {
        isDfs = TRUE;
    }
#endif /* UD_CC_INCLUDEDFS */

    if (path != NULL)
    {
        pMount = ccMountFind(path);
        filePath = ccUtilsFilePathFromLocalPath(path, (pMount && !isDfs) ? pMount->pathPrefix : NULL, pShare->user->server->smb->revision == CCCIFS_ILLEGALSMBREVISION, pathIsLocal);
        if (NULL == filePath)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            res = NQ_ERR_OUTOFMEMORY;
            goto Exit;
        }
    }
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " filePath: %s", cmWDump(filePath));
    
#ifdef UD_CC_INCLUDEDFS
    if (isDfs && NULL != pMount && NULL != filePath)
    {
        NQ_WCHAR *dfsFullPath;

        dfsFullPath = ccUtilsComposeRemotePathToFileByMountPath(pMount->path, filePath, FALSE);
        if (NULL == dfsFullPath)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }
        pFile = ccFileCreate(pShare, pMount, dfsFullPath);

        cmMemoryFree(filePath);
        filePath = ccUtilsFilePathFromRemotePath(dfsFullPath, TRUE);
        cmMemoryFree(dfsFullPath);
        if (NULL == filePath)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            res = NQ_ERR_OUTOFMEMORY;
            goto Exit;
        }
    }
    else
#endif  /* UD_CC_INCLUDEDFS */
    {
        pFile = ccFileCreate(pShare, pMount, filePath);
    }
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_OUTOFMEMORY;
        goto Exit;
    }

    pFile->isPipe = isPipe;

    /* convert parameters */
    switch (access)
    {
        case FILE_AM_READ:
            pFile->accessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_READDATA;
            break;
        case FILE_AM_WRITE:
            if (attributes & SMB_ATTR_DIRECTORY)
            {
                pFile->accessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES;
            }
            else
            {
                pFile->accessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES + SMB_DESIREDACCESS_WRITEDATA;
            }
            break;
        case FILE_AM_READ_WRITE:
            if (attributes & SMB_ATTR_DIRECTORY)
            {
                pFile->accessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES;
            }
            else
            {
                pFile->accessMask = ACCESSMASKDEFAULT + SMB_DESIREDACCESS_WRITEATTRIBUTES + SMB_DESIREDACCESS_READDATA + SMB_DESIREDACCESS_WRITEDATA;
            }
            break;
        default:
            pFile->accessMask = (NQ_UINT32)(access & ~FILE_AM_SPECIAL_MASK);
            break;
    }
    switch (shareMode)
    {
        case FILE_SM_COMPAT:
            pFile->sharedAccess = SMB_SHAREACCESS_READ;
            if (!(pFile->accessMask & SMB_DESIREDACCESS_WRITEDATA))
            {
                pFile->sharedAccess |= SMB_SHAREACCESS_WRITE;
            }
            if (!(pFile->accessMask & SMB_DESIREDACCESS_DELETE))
            {
                pFile->sharedAccess |= SMB_SHAREACCESS_DELETE;
            }
            break;
        case FILE_SM_DENY_NONE:
            pFile->sharedAccess = SMB_SHAREACCESS_WRITE | SMB_SHAREACCESS_READ | SMB_SHAREACCESS_DELETE;
            break;
        case FILE_SM_DENY_READ:
            pFile->sharedAccess = SMB_SHAREACCESS_WRITE | SMB_SHAREACCESS_DELETE;
            break;
        case FILE_SM_DENY_WRITE:
            pFile->sharedAccess = SMB_SHAREACCESS_READ | SMB_SHAREACCESS_DELETE;
            break;
        case FILE_SM_EXCLUSIVE:
            pFile->sharedAccess = SMB_SHAREACCESS_NONE;
            break;
        default:
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal share mode value %d", shareMode);
            res = NQ_ERR_BADPARAM;
            goto Error;
    }
    pFile->attributes = attributes;
    pFile->disposition = SMB2_CREATEDISPOSITION_SUPERSEDE;

    if ((openAction < FILE_OA_FAIL) || (openAction > FILE_OA_TRUNCATE))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal open action value %d", openAction);
        res = NQ_ERR_BADPARAM;
        goto Error;
    }

    if (openAction == FILE_OA_FAIL && createAction == FILE_CA_FAIL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal combination of action values %d %d", createAction, openAction);
        res = NQ_ERR_BADPARAM;
        goto Error;
    }

    switch (oplockLevel)
    {
        case FILE_OPLOCK_LEVEL_NONE:
            pFile->grantedOplock = SMB2_OPLOCK_LEVEL_NONE;
            break;
        case FILE_OPLOCK_LEVEL_II:
            pFile->grantedOplock = SMB2_OPLOCK_LEVEL_II;
            break;
        case FILE_OPLOCK_LEVEL_BATCH:
        default:
            pFile->grantedOplock = SMB2_OPLOCK_LEVEL_BATCH;
    }

    switch (createAction)
    {
        case FILE_CA_CREATE:
            pFile->disposition = openAction == FILE_OA_FAIL?
        SMB_NTCREATEANDX_FILECREATE : openAction == FILE_OA_OPEN?
        SMB_NTCREATEANDX_FILEOPENIF : SMB_NTCREATEANDX_FILEOVERWRITEIF;
            break;
        case FILE_CA_FAIL:
            pFile->disposition = openAction == FILE_OA_FAIL?
            SMB_NTCREATEANDX_FILESUPERSEDE : openAction == FILE_OA_OPEN ?
        SMB_NTCREATEANDX_FILEOPEN : SMB_NTCREATEANDX_FILEOVERWRITE;
            break;
        default:
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal combination of action values %d %d", createAction, openAction);
            res = NQ_ERR_BADPARAM;
            goto Error;
    }

    pFile->options = SMB2_CREATEOPTIONS_NONE;
    if (attributes & SMB_ATTR_DIRECTORY)
    {
        pFile->options |= SMB_NTCREATEANDX_DIRECTORY;
        pFile->options &= ~(NQ_UINT32)SMB_NTCREATEANDX_NONDIRECTORY;
    }
    if (writeThrough)
    {
        pFile->options |= SMB_NTCREATEANDX_WRITETHROUGH;
    }
    if (locality == FILE_LCL_SEQUENTIAL)
    {
        pFile->options |= SMB_NTCREATEANDX_SEQUENTIAL;
    }
    if (locality == FILE_LCL_RANDOM)
    {
        pFile->options |= SMB_NTCREATEANDX_RANDOMACCESS;
    }

    /* delegate to the protocol */
    pFile->open = FALSE;

    ccInitializeMountPointIdentifier(pMount, &mountPointID);
    for (counter = CC_CONFIG_RETRYCOUNT; counter > 0; counter--)
    {
        res = pShare->user->server->smb->doCreate(pFile);
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "doCreate result: 0x%x", res);
#ifdef UD_CC_INCLUDEDFS
        if (NQ_SUCCESS != res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }
        }

        if (ccDfsIsError(dfsContext.lastError = res) && !isPipe)
        {
            dfsResult.path = NULL;

            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "DFS related error %s", res == NQ_ERR_PATHNOTCOVERED ? ": NQ_ERR_PATHNOTCOVERED" : "");

            if (--dfsContext.counter < 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "DFS failed to resolve path: too many attempts");
                break;
            }
            dfsResult = ccDfsResolvePath(pMount, pShare, filePath, &dfsContext);
            if (dfsResult.path)
            {
                NQ_WCHAR *pTempPath;
                CCFile *pTempFile;

                cmMemoryFree(filePath);
                filePath = NULL;

                pShare = dfsResult.share;

                pTempPath = dfsResult.share->flags & CC_SHARE_IN_DFS ? cmMemoryCloneWString(dfsResult.path) : ccUtilsFilePathFromRemotePath(dfsResult.path, FALSE);
                if (NULL == pTempPath)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    sySetLastError(NQ_ERR_OUTOFMEMORY);
                    goto Error;
                }

                pTempFile = ccFileCreate(dfsResult.share, pMount, pTempPath);
                cmMemoryFree(pTempPath);
                if (NULL == pTempFile)
                {
                    sySetLastError(NQ_ERR_OUTOFMEMORY);
                    goto Error;
                }
                cloneFileData(pFile, pTempFile);
                cmListItemUnlock((CMItem *)pFile);
                pFile = pTempFile;

                filePath = ccUtilsFilePathFromRemotePath(dfsResult.path, TRUE);
                ccDfsResolveDispose(&dfsResult);
                dfsResult.path = NULL;
                if (NULL == filePath)
                {
                    sySetLastError(NQ_ERR_OUTOFMEMORY);
                    goto Error;
                }
                LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "pFile: %s", cmWDump(pFile->item.name));

                counter++;
                continue;
            }
            else
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                    break;
                }
            }
        }
#endif /* UD_CC_INCLUDEDFS */
        cmMemoryFree(filePath);
        filePath = NULL;
        if (res == NQ_SUCCESS)
        {
            pFile->open = TRUE;
            cmU64Zero(&pFile->offset);
            pShare->user->server->smb->handleWaitingNotifyResponses(pShare->user->server, pFile);
            goto Exit;
        }
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            /* possible reconnect caused a wrong user session */
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

Error:
    if (NULL != pFile)
    {
        cmListItemUnlock((CMItem *)pFile);
        pFile = NULL;
    }
#ifdef UD_CC_INCLUDEDFS
    if (NULL != dfsResult.path)
    {
        ccDfsResolveDispose(&dfsResult);
    }
#endif /* UD_CC_INCLUDEDFS */
Exit:
#ifdef UD_CC_INCLUDEDFS
    if (NULL != dfsContext.dfsPath)
    {
        cmMemoryFree(dfsContext.dfsPath);
    }
    if (NULL != dfsContext.referral)
    {
        cmMemoryFree(dfsContext.referral);
    }
#endif /* UD_CC_INCLUDEDFS */
    cmMemoryFree(filePath);
    sySetLastError(res);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p error:0x%x", pFile, syGetLastError());
    return pFile;
}

NQ_BOOL ccCreateDirectoryA(const NQ_CHAR *pathName)
{
    NQ_WCHAR * pathNameW; /* name in Unicode */
    NQ_BOOL res = FALSE;  /* delegated call result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (NULL == pathName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Path");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    pathNameW = cmMemoryCloneAString(pathName);
    if (NULL == pathNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    res = ccCreateDirectory(pathNameW);
    cmMemoryFree(pathNameW);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", res ? "TRUE" : "FALSE", syGetLastError());
    return res;
}

NQ_BOOL ccCreateDirectory(const NQ_WCHAR * remotePath)
{
    CCMount *   pMount;                 /* mount point descriptor */
    CCFile      file;                   /* file structure */
    NQ_WCHAR *  localPath = NULL;       /* path component local to remote share */
    CCShare *   pShare;                 /* pointer to the hosting share */
    NQ_BOOL     res;                    /* operation result */
    NQ_INT      counter;                /* simple counter */
#ifdef UD_CC_INCLUDEDFS
    CCDfsContext dfsContext = {CC_DFS_NUMOFRETRIES, 0, NULL, NULL}; /* DFS operations context */
    CCDfsResult dfsResult = {NULL, NULL, NULL};               /* result of DFS resolution */
    NQ_BOOL     isDfs = FALSE;     /* whether share is dfs (then full path should be build) */
#endif /* UD_CC_INCLUDEDFS */
    CCMountIdentifier mountPointID;     /* mount point identifier */
    NQ_BOOL     result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "path:%s", cmWDump(remotePath));

    /* for error handling */
    file.item.name = NULL;

    if (NULL == remotePath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Path");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    pMount = ccMountFind(remotePath);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot find mount point");
        sySetLastError(NQ_ERR_BADPATH);
        goto Exit;
    }

    pShare = pMount->share;
    if (pShare->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot create directory on a printer share");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (!ccTransportIsConnected(&pShare->user->server->transport) && !ccServerReconnect(pShare->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    file.grantedOplock = SMB2_OPLOCK_LEVEL_BATCH;
    file.accessMask = SMB_DESIREDACCESS_SYNCHRONISE | SMB_DESIREDACCESS_READCONTROL| SMB_DESIREDACCESS_DELETE |
                      SMB_DESIREDACCESS_WRITEATTRIBUTES| SMB_DESIREDACCESS_READATTRIBUTES | SMB_DESIREDACCESS_WRITEEA |
                      SMB_DESIREDACCESS_READDATA | SMB_DESIREDACCESS_WRITEDATA | SMB_DESIREDACCESS_READEA; /*0x13019b*/
    file.attributes = CIFS_ATTR_DIR;
    file.disposition = SMB2_CREATEDISPOSITION_CREATE;
    file.options = SMB2_CREATEOPTIONS_DIRECTORY_FILE;
    file.share = pShare;
    file.sharedAccess = SMB_SHAREACCESS_NONE;
#ifdef UD_NQ_INCLUDESMB2
    file.durableState = DURABLE_REQUIRED;
    file.durableFlags = 0;
    file.durableTimeout = 0;
#endif /* UD_NQ_INCLUDESMB2 */
#ifdef UD_CC_INCLUDEDFS
    if (pShare->flags & CC_SHARE_IN_DFS)
    {
        isDfs = TRUE;
        file.item.name = ccUtilsComposeRemotePathToFileByMountPath(pMount->path, remotePath, TRUE);
    }
    else
#endif  /* UD_CC_INCLUDEDFS */
    {
        file.item.name = ccUtilsFilePathFromLocalPath(remotePath, pMount->pathPrefix, pShare->user->server->smb->revision == CCCIFS_ILLEGALSMBREVISION, TRUE);
    }
    if (NULL == file.item.name)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "file path: %s", cmWDump(file.item.name));

#ifdef UD_CC_INCLUDEDFS
    if (isDfs)
    {
        localPath = ccUtilsFilePathFromRemotePath(file.item.name, TRUE);
    }
    else
#endif /* UD_CC_INCLUDEDFS */
    {
        localPath = ccUtilsFilePathFromLocalPath(remotePath, pMount->pathPrefix, pShare->user->server->smb->revision == CCCIFS_ILLEGALSMBREVISION, TRUE);
    }
    if (NULL == localPath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "localPath: %s", cmWDump(localPath));

    ccInitializeMountPointIdentifier(pMount, &mountPointID);
    for (counter = CC_CONFIG_RETRYCOUNT; counter > 0; counter--)
    {
        res = pShare->user->server->smb->doCreate(&file);
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "doCreate result: 0x%x", res);
#ifdef UD_CC_INCLUDEDFS
        if (NQ_SUCCESS != res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }
        }

        if (ccDfsIsError(dfsContext.lastError = res))
        {
            dfsResult.path = NULL;

            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "DFS related error %s", res == NQ_ERR_PATHNOTCOVERED ? ": NQ_ERR_PATHNOTCOVERED" : "");

            if (--dfsContext.counter < 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "DFS failed to resolve path: too many attempts");
                break;
            }
            dfsResult = ccDfsResolvePath(pMount, pShare, localPath, &dfsContext);
            if (dfsResult.path)
            {
                file.share = pShare = dfsResult.share;

                cmMemoryFree(file.item.name);
                file.item.name = dfsResult.share->flags & CC_SHARE_IN_DFS ?
                                    cmMemoryCloneWString(dfsResult.path) :
                                    ccUtilsFilePathFromRemotePath(dfsResult.path, FALSE);
                if (NULL == file.item.name)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    sySetLastError(NQ_ERR_OUTOFMEMORY);
                    goto Exit;
                }
                cmMemoryFree(localPath);
                localPath = ccUtilsFilePathFromRemotePath(dfsResult.path, TRUE);
                ccDfsResolveDispose(&dfsResult);
                dfsResult.path = NULL;
                if (NULL == localPath)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    sySetLastError(NQ_ERR_OUTOFMEMORY);
                    goto Exit;
                }

                counter++;
                continue;
            }
            else
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                    break;
                }
            }
        }
#endif /* UD_CC_INCLUDEDFS */

        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pShare->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pShare->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            /* possible reconnect caused a wrong user session */
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            continue;
        }
        else
        {
            break;
        }
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "status:%d", res);
        sySetLastError((NQ_UINT32)res);
        goto Exit;
    }

    pShare->user->server->smb->doClose(&file);
    result = TRUE;

Exit:
    cmMemoryFree(file.item.name);
    cmMemoryFree(localPath);
#ifdef UD_CC_INCLUDEDFS
    if (dfsResult.path)
    {
        ccDfsResolveDispose(&dfsResult);
    }
    if (NULL != dfsContext.dfsPath)
    {
        cmMemoryFree(dfsContext.dfsPath);
    }
    if (NULL != dfsContext.referral)
    {
        cmMemoryFree(dfsContext.referral);
    }
#endif /* UD_CC_INCLUDEDFS */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccRemoveDirectoryA(const NQ_CHAR * remotePath)
{
    return ccDeleteFileA(remotePath);
}

NQ_BOOL ccRemoveDirectory(const NQ_WCHAR * remotePath)
{
    return ccDeleteFile(remotePath);
}

NQ_BOOL ccDeleteFileA(const NQ_CHAR * fileName)
{
    NQ_WCHAR * fileNameW = NULL; /* name in Unicode */
    NQ_BOOL res = FALSE;  /* delegated call result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (NULL == fileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid File Name");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    fileNameW = cmMemoryCloneAString(fileName);
    if (NULL == fileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    res = ccDeleteFile(fileNameW);

Exit:
    cmMemoryFree(fileNameW);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", res ? "TRUE" : "FALSE", syGetLastError());
    return res;
}

NQ_BOOL ccDeleteFile(const NQ_WCHAR * fileName)
{
    CCFile *    pFile = NULL;           /* open file handle */
    NQ_STATUS   res = NQ_SUCCESS;       /* operation result */
    NQ_UINT32   attributes = 0;         /* file attributes */
    NQ_INT      counter;                /* simple counter */
    CCMountIdentifier mountPointID;     /* mount point identifier */
    NQ_BOOL     result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "fileName:%p", fileName);

    if (!ccGetFileAttributes(fileName, &attributes))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Failed to get attributes");
        goto Exit;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "fileName: %s, attributes: 0x%x", cmWDump(fileName), attributes);

    pFile = (CCFile *)ccCreateFile(
            fileName,
            FILE_AM_SPECIAL_MASK | SMB_DESIREDACCESS_DELETE | SMB_DESIREDACCESS_READATTRIBUTES,
            FILE_SM_DENY_NONE,
            FILE_LCL_UNKNOWN,
            FALSE,
            (NQ_UINT16)attributes,
            FILE_CA_FAIL,
            FILE_OA_OPEN
            );
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "File was not created");
        goto Exit;
    }
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot delete on a printer share");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doSetFileDeleteOnClose(pFile);
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;

        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            /* possible reconnect caused a wrong user session or wrong FID */
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to delete file or directory");
        goto Exit;
    }

    result = TRUE;

Exit:
    if (NULL != pFile)
    {
        ccCloseHandle(pFile);
    }
    if (NQ_SUCCESS != res)
    {
        sySetLastError((NQ_UINT32)res);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccMoveFileA(const NQ_CHAR * oldFileName, const NQ_CHAR * newFileName)
{
    NQ_WCHAR * oldFileNameW = NULL;  /* name in Unicode */
    NQ_WCHAR * newFileNameW = NULL;  /* name in Unicode */
    NQ_BOOL    res = FALSE;          /* delegated call result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (oldFileName == NULL || newFileName == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid FileName");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    oldFileNameW = cmMemoryCloneAString(oldFileName);
    if (NULL == oldFileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    newFileNameW = cmMemoryCloneAString(newFileName);
    if (NULL == newFileNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    res = ccMoveFile(oldFileNameW, newFileNameW);

Exit:
    cmMemoryFree(oldFileNameW);
    cmMemoryFree(newFileNameW);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", res ? "TRUE" : "FALSE", syGetLastError());
    return res;
}

NQ_BOOL ccMoveFile(const NQ_WCHAR * oldFileName, const NQ_WCHAR * newFileName)
{
    CCMount *   pMount;                   /* mount point descriptor */
    CCFile *    pFile = NULL;             /* open file handle */
    NQ_WCHAR *  newLocalPath = NULL;      /* path component local to remote share */
    NQ_STATUS   res = NQ_SUCCESS;         /* operation result */
    NQ_INT      counter;                  /* simple counter */
    NQ_BOOL     createBeforeMove = FALSE; /* saved dialect flag */
    CCMountIdentifier mountPointID;       /* mount point identifier */
    NQ_BOOL     result = FALSE;           /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "oldFileName: %p newFileName:%p", oldFileName, newFileName);

    LOGMSG(CM_TRC_LEVEL_MESS_SOME, "oldFileName: %s", cmWDump(oldFileName));
    LOGMSG(CM_TRC_LEVEL_MESS_SOME, "newFileName: %s", cmWDump(newFileName));

    if (oldFileName == NULL || newFileName == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid FileName");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* rename to itself should always succeed */
    if (0 == cmWStrcmp(oldFileName, newFileName))
    {
        result = TRUE;
        goto Exit;
    }
    pMount = ccMountFind(oldFileName);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot find mount point");
        sySetLastError(NQ_ERR_BADPATH);
        goto Exit;
    }
    if (pMount->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot move files on a printer share");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    if (pMount != ccMountFind(newFileName))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot move files between different trees");
        sySetLastError(NQ_ERR_DIFFDEVICE);
        goto Exit;
    }

    /*  SMB1 does not need file to be open, but doRename() requires valid CCFile, file handle must be closed before actual rename  */
    pFile = (CCFile *)ccCreateFile(
            oldFileName,
            FILE_AM_SPECIAL_MASK | SMB_DESIREDACCESS_DELETE,
            FILE_SM_DENY_NONE,
            FILE_LCL_UNKNOWN,
            FALSE,
            0,
            FILE_CA_FAIL,
            FILE_OA_OPEN
            );
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot create file");
        goto Exit;
    }

    /* save the dialect flag since it may be not available after rename */
    createBeforeMove = pFile->share->user->server->smb->createBeforeMove;

#ifdef UD_CC_INCLUDEDFS
    {
        NQ_WCHAR *resolvedPath = NULL;

        ccCheckPath(newFileName, FALSE, &resolvedPath);
        if (NULL != resolvedPath)
        {
            /* SMB1 requires full path for a new name, while SMB2 requires relative path */
            if (pFile->share->user->server->smb->useFullPath)
            {
                newLocalPath = cmMemoryCloneWString(resolvedPath);
            }
            else
            {
                if (pFile->share->flags & CC_SHARE_IN_DFS)
                {
                    newLocalPath = ccUtilsFilePathFromRemotePath(resolvedPath, TRUE);
                }
                else
                {
                    newLocalPath = (*resolvedPath == cmWChar('\\')) ? cmMemoryCloneWString(resolvedPath + 1) : cmMemoryCloneWString(resolvedPath);
                }
            }
            cmMemoryFree(resolvedPath);
            if (NULL == newLocalPath)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                sySetLastError(NQ_ERR_OUTOFMEMORY);
                goto Exit;
            }
        }
    }
#endif /* UD_CC_INCLUDEDFS */

    if (NULL == newLocalPath)
    {
        newLocalPath = ccUtilsFilePathFromLocalPath(newFileName, pMount->pathPrefix, pFile->share->user->server->smb->revision == CCCIFS_ILLEGALSMBREVISION, TRUE);
        if (NULL == newLocalPath)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }
    }

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS,"pFile: %s", cmWDump(pFile->item.name));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS,"pFile->share: %s", cmWDump(pFile->share->item.name));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS,"newLocalPath: %s", cmWDump(newLocalPath));

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doRename(pFile, newLocalPath);
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                /* reconnection cleans up everything, so doesn't need to close pFile anymore */
                createBeforeMove = FALSE;
                LOGERR(CM_TRC_LEVEL_ERROR, "Unable to rename file (Connection Lost)");
                sySetLastError((NQ_UINT32)res);
                goto Exit;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to rename file");
        sySetLastError((NQ_UINT32)res);
        goto Exit;
    }

    result = TRUE;

Exit:
    /* SMB2 (TRUE) requires to create the file before moving it but SMB1 (FALSE) doesn't. */
    if (TRUE == createBeforeMove)
    {
        ccCloseHandle(pFile);
        if (TRUE != result)
        {
            sySetLastError((NQ_UINT32)res);
        }
    }
    cmMemoryFree(newLocalPath);    
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetFilePointer(NQ_HANDLE handle, NQ_UINT64 * offset)
{
    CCFile * pFile;                      /* casted file pointer */
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p offset:%p", handle, offset);

    if (handle == NULL || offset == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (!ccValidateFileHandle(handle))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)handle;

    if (!pFile->open)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Not Opened");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    cmU64AssignU64(offset, (const NQ_UINT64 *)&pFile->offset);
    result = TRUE;

 Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccSetFilePointer(NQ_HANDLE handle, NQ_INT64 * offset, NQ_INT moveMethod)
{
    CCFile * pFile;  /* casted file pointer */
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p offset:%p move:%d", handle, offset, moveMethod);

    if (handle == NULL || offset == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "in offset sign:%d, high: %u, low: %u", offset->sign, offset->high, offset->low);

    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }

    if (!ccValidateFileHandle(handle))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }
    pFile = (CCFile *)handle;

    if (!pFile->open)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "File is not open");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    cmListItemTake((CMItem *)pFile);

    switch (moveMethod)
    {
        case SEEK_FILE_BEGIN:
            /* negative offset is not allowed for SEEK_FILE_BEGIN */
            if (cmS64IsNegative(offset))
            {
                cmListItemGive((CMItem *)pFile);
                LOGERR(CM_TRC_LEVEL_ERROR, "Invalid negative offset for SEEK_FILE_BEGIN");
                sySetLastError(NQ_ERR_BADPARAM);
                goto Exit;
            }
            cmU64AssignU64(&pFile->offset, (NQ_UINT64 *)offset);
            break;

        case SEEK_FILE_CURRENT:
            /* requested negative offset can't be bigger than current file offset */
            if (cmS64IsNegative(offset) && (cmU64Cmp((NQ_UINT64 *)offset, &pFile->offset) == 1))
            {
                cmListItemGive((CMItem *)pFile);
                LOGERR(CM_TRC_LEVEL_ERROR, "Invalid negative offset for SEEK_FILE_CURRENT");
                sySetLastError(NQ_ERR_SEEKERROR);
                goto Exit;
            }
            cmU64AddS64(&pFile->offset, (const NQ_INT64 *)offset);
            break;

        case SEEK_FILE_END:
        {
            NQ_UINT64 eofCurrent;
            NQ_STATUS lastError;

            if (ccGetFileSize(pFile, &eofCurrent))
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "current eof: high: %u, low: %u", eofCurrent.high, eofCurrent.low);
                /* requested negative offset can't be bigger than current EOF */
                if (cmS64IsNegative(offset) && (cmU64Cmp((NQ_UINT64 *)offset, &eofCurrent) == 1))
                {
                    cmListItemGive((CMItem *)pFile);
                    LOGERR(CM_TRC_LEVEL_ERROR, "Invalid negative offset for SEEK_FILE_END");
                    sySetLastError(NQ_ERR_SEEKERROR);
                    goto Exit;
                }
                cmU64AssignU64(&pFile->offset, &eofCurrent);
                cmU64AddS64(&pFile->offset, offset);
            }
            else
            {
                lastError = syGetLastError();
                cmListItemGive((CMItem *)pFile);
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get current EOF for SEEK_FILE_END");
                sySetLastError(lastError);
                goto Exit;
            }
            break;
        }

        default:
            cmListItemGive((CMItem *)pFile);
            LOGERR(CM_TRC_LEVEL_ERROR, "Invalid move method: %d", moveMethod);
            sySetLastError(NQ_ERR_SEEKERROR);
            goto Exit;

    }
    /* return new offset */
    cmS64AssignU64(offset, &pFile->offset);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "result offset: sign:%d, high: %u, low: %u", offset->sign, offset->high, offset->low);
    cmListItemGive((CMItem *)pFile);
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccFlushFile(NQ_HANDLE handle)
{
    CCFile *            pFile;                      /* casted file pointer */
    NQ_STATUS           res;                        /* operation result */
    NQ_INT              counter;                    /* simple counter*/
    CCMountIdentifier mountPointID;                 /* mount point identifier */
    NQ_BOOL             result = FALSE;             /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p", handle);

    if (NULL == handle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    /* Check CIFS Client is initialized */
    if (!ccIsInitialized())
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "CIFS Client is not ready");
        sySetLastError(NQ_ERR_NOTREADY);
        goto Exit;
    }
    if (!ccValidateFileHandle(handle))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)handle;
    if (pFile->share->isPrinter)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot flush to a print file");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    if (!pFile->open)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }
    if (!ccTransportIsConnected(&pFile->share->user->server->transport) && !ccServerReconnect(pFile->share->user->server))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Not connected");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0 ; counter < CC_CONFIG_RETRYCOUNT; counter++)
    {
        res = pFile->share->user->server->smb->doFlush(pFile);
        if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
        {
            pFile->share->user->server->transport.connected = FALSE;
            if (!ccServerReconnect(pFile->share->user->server))
            {
                ccUpdateMountPointValidity(&mountPointID);
                if (MOUNTPOINT_INVALID == mountPointID.status)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                }

                break;
            }
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            /* possible reconnect caused a wrong user session or wrong FID */
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else
        {
            break;
        }
    }

    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to flush file");
        sySetLastError((NQ_UINT32)res);
        goto Exit;
    }
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccFileReportDisconnect(CCFile * pFile)
{
    NQ_BOOL res = FALSE; /* call result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);

    if (pFile == NULL || !ccValidateFileHandle((NQ_HANDLE)pFile))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }
    /* We reconnect the entire server, including all its users, shares. files.
    * If after reconnect this file becomes connected - we succeeded.
    */

    res = ccServerReconnect(pFile->share->user->server);
    res = (res && pFile->open);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", res ? "TRUE" : "FALSE");
    return res;
}

NQ_BOOL ccFileRestore(CCFile * pFile)
{
    NQ_STATUS res = FALSE;                  /* call result */
    NQ_BOOL result = FALSE;
    NQ_COUNT counter;
    CCMountIdentifier mountPointID;         /* mount point identifier */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "file:%p", pFile);

    if (pFile == NULL || !ccValidateFileHandle((NQ_HANDLE)pFile))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    syMemcpy(&mountPointID, &pFile->mountPointID, sizeof(CCMountIdentifier));
    for (counter = 0; counter < CC_CONFIG_RETRYCOUNT ; counter++)
    {
        res = pFile->share->user->server->smb->doRestoreHandle(pFile);
        if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                break;
            }

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "operation failed. Try again");
            continue;
        }
        else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
        {
            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_INVALID == mountPointID.status)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
            }

            break;
        }
        else
        {
            break;
        }
    }

    result = (NQ_SUCCESS == res);
    pFile->open = result;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", res ? "TRUE" : "FALSE");
    return res;
}

NQ_BOOL ccValidateFileHandle(NQ_HANDLE handle)
{
    NQ_BOOL     result = FALSE;
    CCFile * pFile = (CCFile *)handle;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "handle:%p", handle);

    if (NULL == handle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Null Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (pFile->disconnected)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "disconnected Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (((CMItem *)pFile)->beingDisposed)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "being disposed Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

void ccFileTake(NQ_HANDLE handle)
{
    if (handle)
    {
        CCFile *    pFile = (CCFile *)handle;
        cmListItemTake((CMItem *)pFile);
    }
}

void ccFileGive(NQ_HANDLE handle)
{
    if (handle)
    {
        CCFile *    pFile = (CCFile *)handle;
        cmListItemGive((CMItem *)pFile);
    }
}

NQ_BOOL ccServerSideDataCopyA(NQ_CHAR* srcPath, NQ_CHAR* dstPath, void* context, CCServerSideCopyCallback* callBack)
{
    NQ_WCHAR*  dstPathW = NULL;
    NQ_WCHAR*  srcPathW = NULL;
    NQ_BOOL    res = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "srcPath:%p dstPath:%p context:%p callBack:%p", srcPath, dstPath, context, callBack);

    if ((NULL == srcPath) || (NULL == dstPath))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid parameter");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    srcPathW = cmMemoryCloneAString(srcPath);
    if (NULL == srcPathW)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Out of memory.");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    dstPathW = cmMemoryCloneAString(dstPath);
    if (NULL == dstPathW)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Out of memory.");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    res = ccServerSideDataCopy(srcPathW, dstPathW, context, callBack);

Exit:
    if (NULL != dstPathW)
    {
        cmMemoryFree(dstPathW);
    }

    if (NULL != srcPathW)
    {
        cmMemoryFree(srcPathW);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", res ? "TRUE" : "FALSE", syGetLastError());
    return res;
}

NQ_BOOL ccServerSideDataCopy(NQ_WCHAR* srcPath, NQ_WCHAR* dstPath, void* context, CCServerSideCopyCallback* callBack)
{
    NQ_BOOL             result = FALSE;                             /* Return value*/
    NQ_STATUS           res = NQ_SUCCESS;                           /* Operation result */
    CCMount*            pSrcMount = NULL;                           /* Mount point descriptor */
    CCMount*            pDstMount = NULL;                           /* Mount point descriptor */
    CCFile*             pSrcFile = NULL;                            /* Open file handle */
    CCFile*             pDstFile = NULL;                            /* Open file handle */
    CCMountIdentifier   mountPointID;                               /* Mount point identifier */
    CCResumeKey*        pKey = NULL;                                /* Resume key pointer */
    NQ_UINT64           srcFileEOF = {0, 0};                        /* Temp variable to hold the EOF offset of the source file */
    CCChunks            pChunks;                                    /* Chunks struct pointer, contains the CCChunk array pointer and the number of chunks */
    NQ_UINT32           maxChunkCopySize = MAX_CHUNK_COPY_SIZE;     /* The maximum size for each chunk */
    NQ_UINT32           maxChunksNum = MAX_CHUNKS_NUM;              /* Maximum supported chunks by the server */
    NQ_UINT64           dstFileCurOffset = {0, 0};                  /* Temp variable to hold the current offset of the destination file */
    NQ_COUNT            counter = 0;                                /* Retry loop counter */
    CCChunksStatus      chunkStatus;                                /* A struct that will hold the server response for each copy-chunk request */
    NQ_BOOL             isFirstStatusInvalidParameter = TRUE;       /* A variable that is needed only for the first NQ_ERR_BADPARAM that will be return from the server */
    NQ_BOOL             isCopyChunkError = FALSE;                   /* Indicates if there was an error while performing a copy_chunk requests */
    NQ_WCHAR*           pSrcFileName = NULL;                        /* Source full remote path pointer */
    NQ_WCHAR*           pDstFileName = NULL;                        /* Destination full remote path pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "srcPath:%p dstPath:%p context:%p callBack:%p", srcPath, dstPath, context, callBack);

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS,"srcPath: %s", cmWDump(srcPath));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS,"dstPath: %s", cmWDump(dstPath));

    if ((NULL == srcPath) || (NULL == dstPath))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid parameter");
        res = NQ_ERR_BADPARAM;
        goto Exit;
    }

    /* Open source file with read access only */
    pSrcFile = (CCFile *)ccCreateFile(
            srcPath,
            FILE_AM_READ,
            FILE_SM_DENY_NONE,
            FILE_LCL_UNKNOWN,
            FALSE,
            0,
            FILE_CA_FAIL,
            FILE_OA_OPEN
            );
    if (NULL == pSrcFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot open source file");
        res = syGetLastError();
        goto Exit;
    }

    pSrcMount = ccMountFind(srcPath);
    if (NULL == pSrcMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot find source mount point");
        res = NQ_ERR_BADPATH;
        goto Exit;
    }

    pDstMount = ccMountFind(dstPath);
    if (NULL == pDstMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Cannot find destination mount point");
        res = NQ_ERR_BADPATH;
        goto Exit;
    }

    /* Check if the files are located on the same server, if not fail */
    if (0 != cmWStrcmp(pSrcMount->server->item.name, pDstMount->server->item.name))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot copy files from a different trees");
        res = NQ_ERR_DIFFDEVICE;
        goto Exit;
    }

    /* Get remote path of source file */
    pSrcFileName = ccUtilsComposeRemotePathToFileByMountPath(pSrcMount->path, srcPath, TRUE);
    if (NULL == pSrcFileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_OUTOFMEMORY;
        goto Exit;
    }

    /* Get remote path of destination file */
    pDstFileName = ccUtilsComposeRemotePathToFileByMountPath(pDstMount->path, dstPath, TRUE);
    if (NULL == pDstFileName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_OUTOFMEMORY;
        goto Exit;
    }

    /* Copy to itself should always succeed */
    if (0 == cmWStrcmp(pSrcFileName, pDstFileName))
    {
        res = NQ_SUCCESS;
        result = TRUE;
        goto Exit;
    }

    pKey = (CCResumeKey *)cmMemoryAllocate((NQ_UINT)sizeof(CCResumeKey));
    if (NULL == pKey)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        res = NQ_ERR_OUTOFMEMORY;
        goto Exit;
    }

    /* Query the server for resume key */
    result = queryResumeFileKey(pSrcFile, &mountPointID, pKey);
    if (FALSE == result)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to query resume key, server side copy might not supported in this server");
        res = syGetLastError();
        goto Exit;
    }

    /* Get source file size */
    result = ccGetFileSize(pSrcFile, &srcFileEOF);
    if (FALSE == result)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get srcPath info");
        res = syGetLastError();
        goto Exit;
    }

    /* Open destination file with read and write access, if file exist fail, otherwise create it */
    pDstFile = (CCFile *)ccCreateFile(
            dstPath,
            FILE_AM_READ_WRITE,
            FILE_SM_DENY_NONE,
            FILE_LCL_UNKNOWN,
            FALSE,
            0,
            FILE_CA_CREATE,
            FILE_OA_FAIL
            );
    if (NULL == pDstFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot create destination file");
        result = FALSE;
        res = syGetLastError();
        goto Exit;
    }

    /* Set destination file size */
    result = ccSetFileSizeByHandle(pDstFile, srcFileEOF);
    if (FALSE == result)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to set dstPath file size");
        res = syGetLastError();
        goto Exit1;
    }

    if ((0 != srcFileEOF.high) || (0 != srcFileEOF.low))
    {

        /* While (srcFileEOF > dstFileCurOffset)*/
        while ((srcFileEOF.high >= dstFileCurOffset.high) && (srcFileEOF.low > dstFileCurOffset.low))
        {
            result = createChunkArray(dstFileCurOffset, srcFileEOF, maxChunksNum, maxChunkCopySize, &pChunks);
            if (FALSE == result)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to create chunks array");
                res = syGetLastError();
                goto Exit1;
            }

            syMemcpy(&mountPointID, &pSrcFile->mountPointID, sizeof(CCMountIdentifier));
            for (counter = 0 ; counter < CC_CONFIG_RETRYCOUNT ; counter++)
            {
                res = pSrcFile->share->user->server->smb->doServerSideDataCopy(pDstFile, TRUE, pKey, &pChunks, &chunkStatus);
                if (res == (NQ_STATUS)NQ_ERR_RECONNECTREQUIRED)
                {
                    pSrcFile->share->user->server->transport.connected = FALSE;
                    if (!ccServerReconnect(pSrcFile->share->user->server))
                    {
                        ccUpdateMountPointValidity(&mountPointID);
                        if (MOUNTPOINT_INVALID == mountPointID.status)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                        }

                        isCopyChunkError = TRUE;
                        break;
                    }
                }
                else if ((NQ_STATUS)NQ_ERR_NOTCONNECTED == res)
                {
                    ccUpdateMountPointValidity(&mountPointID);
                    if (MOUNTPOINT_INVALID == mountPointID.status)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                    }

                    isCopyChunkError = TRUE;
                    break;
                }
                else if ((NQ_STATUS)NQ_ERR_TRYAGAIN == res)
                {
                    ccUpdateMountPointValidity(&mountPointID);
                    if (MOUNTPOINT_INVALID == mountPointID.status)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point was disposed");
                        isCopyChunkError = TRUE;
                        break;
                    }

                    /* possible reconnect caused a wrong user session or wrong FID */
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Operation failed. Try again");
                    continue;
                }
                else if ((NQ_STATUS)NQ_ERR_BADPARAM == res)
                {
                    /* if the server returned IOCTL response with SMB_STATUS_INVALID_PARAMETER and structure size 9 we set 0 in chunkStatus.chunksWritten and chunkStatus.chunkBytesWritten */
                    if ((0 == chunkStatus.chunksWritten) && (0 == chunkStatus.chunkBytesWritten))
                    {
                        if (0 != cmWStrcmp(pSrcFile->share->item.name, pDstFile->share->item.name))
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "The server does not support server-side data copy with different shares");
                            res = NQ_ERR_NOSUPPORT;
                            isCopyChunkError = TRUE;
                            break;
                        }
                        else
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "Operation failed");
                            isCopyChunkError = TRUE;
                            break;
                        }
                    }
                    else
                    {
                        if (TRUE == isFirstStatusInvalidParameter)
                        {
                            /**
                             * If the status field is NQ_ERR_BADPARAM it means that the array pChunks with ranges to copy that was sent contains wrong values.
                             * chunkStatus.chunksWritten - indicates the maximum number of chunks that the server will accept in a single request.
                             * chunkStatus.chunkBytesWritten - indicates the maximum number of bytes the server will allow to be written in a single chunk.
                             * So, we are resetting the values for the next operation.
                             */
                            maxChunksNum = chunkStatus.chunksWritten;
                            maxChunkCopySize = chunkStatus.chunkBytesWritten;
                            isFirstStatusInvalidParameter = FALSE;
                        }
                        else
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "Operation failed");
                            isCopyChunkError = TRUE;
                            break;
                        }
                    }
                }
                else if ((NQ_STATUS)NQ_SUCCESS == res)
                {
                    /* offset + total bytes written */
                    cmU64AddU32(&dstFileCurOffset, chunkStatus.totalBytesWritten);
                    break;
                }
                else
                {
                    isCopyChunkError = TRUE;
                    break;
                }
            }

            /* Free the array that was allocated in createChunkArray() after each iteration */
            cmMemoryFree(pChunks.chunks);
            pChunks.numberOfChunks = 0;
            if (TRUE == isCopyChunkError)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Error while trying to copy_chunk");
                break;
            }
        }
    }

Exit1:
    if (NQ_SUCCESS != res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to copy file");
        result = FALSE;
        if (NULL != pDstFile)
        {
            ccCloseHandle(pDstFile);
            pDstFile = NULL;
            ccDeleteFile(dstPath);
        }

        goto Exit;
    }

    result = TRUE;

Exit:
    if (NULL != pKey)
    {
        cmMemoryFree(pKey);
    }

    if (NULL != pDstFile)
    {
        ccCloseHandle(pDstFile);
    }

    if (NULL != pDstFileName)
    {
        cmMemoryFree(pDstFileName);
    }

    if (NULL != pSrcFileName)
    {
        cmMemoryFree(pSrcFileName);
    }

    if (NULL != pSrcFile)
    {
        ccCloseHandle(pSrcFile);
    }

    sySetLastError((NQ_UINT32)res);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}


#endif /* UD_NQ_INCLUDECIFSCLIENT */
