/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client dfs operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccdfs.h"
#include "ccutils.h"
#include "cmresolver.h"
#include "cmfinddc.h"
#include "ccdfscache.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

/* -- Static functions -- */

#ifdef UD_CC_INCLUDEDFS

#define CCDFS_MAXREFERRALSTOCACHE   (CM_MAX_UINT16)  /* unlimited number to be cached */
#define CCDFS_CHECKREFERRALVALIDITY (FALSE)          /* do not check referral validity - meaning do not connect */

static NQ_BOOL dfsIsOn = TRUE; /* turn DFS on/off */

static void logPrintResult(CCDfsResult result)
{
    if (result.path)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Resolved: path: %s", result.path ? cmWDump(result.path) : "");
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Resolved: server: %s", result.server ? cmWDump(result.server->item.name) : "");
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Resolved: share: %s", result.share ? cmWDump(result.share->item.name) : "");
    }
    else
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to resolve path");
    }
}

#ifdef UD_NQ_INCLUDETRACE

static void printReferrals(CMList *pList)
{
    CMIterator iterator;

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Referrals:");
    cmListIteratorStart(pList, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        CMItem * item = cmListIteratorNext(&iterator);
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s", cmWDump(item->name));
    }
    cmListIteratorTerminate(&iterator);
}

static void printDfsContext(CCDfsContext *pCtxt)
{
    if (NULL != pCtxt)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "DfsContext: %p", pCtxt);
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "counter: %d, lastError: 0x%x", pCtxt->counter, pCtxt->lastError);
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "dfsPath: %s", cmWDump(pCtxt->dfsPath));
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "referral: %s", cmWDump(pCtxt->referral));
    }
}

#endif /* UD_NQ_INCLUDETRACE */

static void createResultPath(const NQ_WCHAR *path, NQ_UINT16 numPathConsumed, CMItem *item, CCDfsResult *res)
{
    const NQ_WCHAR oneSlash[] = { cmWChar('\\'), cmWChar(0) };

    if (item != NULL && res->share != NULL)
    {
        res->path = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_WCHAR) * (cmWStrlen(item->name) + ((NULL != path) ? cmWStrlen(path) : 0) + cmWStrlen(oneSlash) + CM_TRAILING_NULL)));
        if (NULL != res->path)
        {
            const NQ_WCHAR *from = ((NULL != path) && (cmWStrlen(path) > numPathConsumed)) ? path + numPathConsumed : NULL;

            cmWStrcpy(res->path, item->name);

            if (NULL != from)
            {
                if ((res->path[cmWStrlen(res->path) - 1] != cmWChar('\\')) && (from[0] != cmWChar('\\')))
                {
                    cmWStrcat(res->path, oneSlash);
                }
                cmWStrcat(res->path, from);
            }
        }
        else
        {
             LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        }
        res->server = res->share->user->server;
    }
}

/*
 * Explicitly close and dispose referral (see above):
 *  - disconnects from the share
 *  - disposes private data
 */
static NQ_BOOL disposeReferralCallback(CMItem * pItem)
{
    CCDfsReferral * pRef;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "item:%p", pItem);

    pRef = (CCDfsReferral *)pItem;
    if (NULL!= pRef->dfsPath)
    {
        cmMemoryFree(pRef->dfsPath);
    }
    if (NULL!= pRef->netPath)
    {
        cmMemoryFree(pRef->netPath);
    }
    cmListItemRemoveAndDispose(pItem);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return TRUE;
}


/* Parser referrals. This function gets a reader pointed to the beginning of the
 * DFS response structure. This structure is the same for any SMB protocol.
 */
static void parseReferralCallback(CMBufferReader * pReader, CMList * pList)
{
    NQ_UINT16 vers;             /* referral version */
    NQ_UINT16 pathConsumed;     /* path consumed in bytes including null terminating (unicode) */
    NQ_UINT16 serverType;       /* server type */
    NQ_UINT16 flags;            /* referral entry flags */
    NQ_UINT32 ttl;              /* time to live */
    NQ_IOBufPos entryStart;     /* start of the referral entry */
    NQ_UINT16 offset;           /* various offsets to strings */
    const NQ_WCHAR * dfsPath;   /* DFS path */
    const NQ_WCHAR * netPath;   /* resolved */
    CCDfsReferral * ref;        /* referral entry */
    NQ_UINT16 numRefs;          /* number of referrals */

    cmBufferReadUint16(pReader, &pathConsumed);             /* PathConsumed */
    cmBufferReadUint16(pReader, &numRefs);                  /* NumberOfReferrals */
    cmBufferReaderSkip(pReader, sizeof(NQ_UINT32));         /* flags */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "numRefs: %d", numRefs);
    for ( ; numRefs > 0; numRefs--)
    {
        entryStart = cmBufferReaderGetPosition(pReader);
        cmBufferReadUint16(pReader, &vers);                 /* version number */
        cmBufferReaderSkip(pReader, sizeof(NQ_UINT16));     /* size */
        cmBufferReadUint16(pReader, &serverType);           /* server type */
        cmBufferReadUint16(pReader, &flags);                /* referral entry flags */
        switch (vers)
        {
            case 1:
                dfsPath = NULL;
                netPath = (const NQ_WCHAR *)cmBufferReaderGetPositionBytePtr(pReader, 0);
                ttl = 0;
                break;
            case 2:
                cmBufferReaderSkip(pReader, sizeof(NQ_UINT32));      /* proximity */
                /* continue */
            case 3:
            case 4:
            {
                NQ_IOBufPos bufPos = entryStart;

                cmBufferReadUint32(pReader, &ttl);                   /* time to live */
                cmBufferReadUint16(pReader, &offset);                /* DFS path offset */

                IOBUF_MOVEBYTES(bufPos, offset);
                if (flags & 2) /* domain referral */
                {
                    NQ_UINT16 num, expOffset;

                    dfsPath = (const NQ_WCHAR *)(IOBUF_GETBYTEPTR(bufPos));
                    cmBufferReadUint16(pReader, &num);                /* number of expanded names */
                    cmBufferReadUint16(pReader, &expOffset);          /* expanded names offset */
                    if (num == 0)
                    {
                        dfsPath = NULL;
                        netPath = (const NQ_WCHAR *)(IOBUF_GETBYTEPTR(bufPos));
                    }
                    else
                    {
                        netPath = (const NQ_WCHAR *)(IOBUF_GETBYTEPTR(bufPos));
                    }
                }
                else
                {
                    dfsPath = (const NQ_WCHAR *)(IOBUF_GETBYTEPTR(bufPos));
                    cmBufferReaderSkip(pReader, sizeof(NQ_UINT16));   /* alternate path offset */
                    cmBufferReadUint16(pReader, &offset);             /* network address offset */
                    cmBufferReaderSkip(pReader, 16);                  /* GUID */
                    bufPos = IOBUF_SKIPBYTE(entryStart, offset);
                    netPath = (const NQ_WCHAR *)(IOBUF_GETBYTEPTR(bufPos));
                }
            }
                break;
            default:
                return;
        }

        ref = (CCDfsReferral *)cmListItemCreate(sizeof(CCDfsReferral), NULL, CM_LISTITEM_NOLOCK, FALSE);
        if (NULL != ref)
        {
            ref->numPathConsumed = (NQ_UINT16)(pathConsumed / 2 + 1);
            ref->serverType = (serverType == DFS_ROOT_TARGET);
            ref->flags = flags;
            ref->ttl = ttl;
            ref->dfsPath = (dfsPath == NULL) ? NULL : cmMemoryCloneWString(dfsPath);
            ref->netPath = cmMemoryCloneWString(netPath);
            ref->isIOPerformed = FALSE;
            ref->lastIOStatus = NQ_SUCCESS;
            cmListItemAdd(pList, (CMItem *)ref, disposeReferralCallback);
        }
    } /* end of for ( ; numRefs > 0; numRefs--) */
}

static NQ_WCHAR * getDomainReferral(const NQ_WCHAR *dcName, const NQ_WCHAR *path)
{
    CCShare *pShare;
    NQ_WCHAR *ipcPath;
    NQ_WCHAR *resultReferral = NULL;
    const AMCredentials *pCredentials = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "dcName:%p path:%p", cmWDump(dcName), cmWDump(path));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "dcName: %s", cmWDump((const NQ_WCHAR *)dcName));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "path: %s", cmWDump(path));

    /* connect to IPC$ share */
    ipcPath = ccUtilsComposeRemotePathToShare(dcName, cmIPCShareName());
    if (NULL == ipcPath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    pShare = ccShareConnect(ipcPath, NULL, &pCredentials, FALSE);
    cmMemoryFree(ipcPath);
    if (NULL != pShare)
    {
        CMList domainRefs;
        CMIterator domainIter;
        NQ_STATUS status;

        cmListStart(&domainRefs);
        /* send get referrals (get domain dfs name) */
        status = pShare->user->server->smb->doQueryDfsReferrals(pShare, path + 1, parseReferralCallback, &domainRefs);
        if (NQ_SUCCESS == status)
        {
            cmListIteratorStart(&domainRefs, &domainIter);
            while (cmListIteratorHasNext(&domainIter))
            {
                CCDfsReferral * pDomainRef;
                CCShare * pShareReferral;

                pDomainRef = (CCDfsReferral *)cmListIteratorNext(&domainIter);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "got domain dfs referral: %s, rootOrLink: %s", cmWDump(pDomainRef->netPath), pDomainRef->serverType == DFS_ROOT_TARGET ? "root" : "link");

                /* connect to referral */
                pCredentials = pShare->user->credentials;
                pShareReferral = ccShareConnect(pDomainRef->netPath, NULL, &pCredentials, FALSE);
                if (pCredentials !=  pShare->user->credentials)
                {
                    cmMemoryFree(pCredentials);
                }
                if (pShareReferral != NULL)
                {
                    /* if connected - add to cache */
                    ccDfsCacheAddPath(path, pDomainRef);
                    resultReferral = cmMemoryCloneWString(pDomainRef->netPath);
                    if (NULL == resultReferral)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                        sySetLastError(NQ_ERR_OUTOFMEMORY);
                    }
                    break;
                }
            }
            cmListIteratorTerminate(&domainIter);
        }

        cmListIteratorStart(&domainRefs, &domainIter);
        while (cmListIteratorHasNext(&domainIter))
        {
            CMItem * pItem;

            pItem = cmListIteratorNext(&domainIter);
            cmListItemCheck(pItem);
        }
        cmListIteratorTerminate(&domainIter);
        cmListShutdown(&domainRefs);
        cmListItemUnlock((CMItem *)pShare);
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", resultReferral);
    return resultReferral;
}

static NQ_BOOL getReferrals(CCMount *pMount, CCShare *pShare, const NQ_WCHAR *path, CCCifsParseReferral parser)
{
    CCShare * ipc;
    CMList refs;
    NQ_STATUS result = NQ_FAIL;
    NQ_BOOL isPathAdded = FALSE;
    const AMCredentials * pCredentials = pShare->user->credentials;
    NQ_STATUS lastError = NQ_ERR_OK;  /* status to handle failed connect retries*/
    CMIterator iterator;
    NQ_WCHAR * pDomain = NULL;
    NQ_WCHAR * domainReferral = NULL;
    NQ_WCHAR * netPath = NULL;
    NQ_WCHAR * serverHostComponent = NULL;
    NQ_WCHAR * pathComponent = NULL;
    CCShare *pShareReferral;        /* pointer to referral */
    NQ_COUNT totalReferrals = 0;    /* number of referrals */
    NQ_COUNT cachedReferrals = 0;   /* number of cached referrals */
    NQ_COUNT maxReferralsToSave = CCDFS_MAXREFERRALSTOCACHE;
    NQ_BOOL doCheckReferralValidity = CCDFS_CHECKREFERRALVALIDITY;
    CCMountIdentifier mountPointID; /* mount point identifier */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "share:%p path:%s parser:%p", pShare, cmWDump(path), parser);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "share name: %s", cmWDump((const NQ_WCHAR *)pShare->item.name));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "maxReferralsToSave: %u, doCheckReferralValidity: %s", maxReferralsToSave, doCheckReferralValidity ? "yes" : "no");

    ccInitializeMountPointIdentifier(pMount, &mountPointID);

    /* connect to IPC$ share */
    ipc = ccShareConnectIpc(pShare->user->server, &pCredentials);
    if (NULL == ipc)
    {
        lastError = syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to IPC$");
        goto Exit;
    }
    if (pCredentials != pShare->user->credentials)
    {
        cmMemoryFree(pCredentials);
        pCredentials = NULL;
    }

    /* send get referrals request */
    cmListStart(&refs);
    result = pShare->user->server->smb->doQueryDfsReferrals(ipc, path + 1, parseReferralCallback, &refs);
    if (NQ_SUCCESS != result)
    {
        lastError = syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get referrals");
        goto Error1;
    }

    /* cache referrals */
    cmListIteratorStart(&refs, &iterator);
    while (cmListIteratorHasNext(&iterator) && (cachedReferrals < maxReferralsToSave))
    {
        CCDfsReferral * pRef;            /* next referral */
        CCDfsCacheEntry * pEntry;        /* DFS cache entry */

        pRef = (CCDfsReferral *)cmListIteratorNext(&iterator);

        totalReferrals++;

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "referral #%d: %s, rootOrLink: %s, pathConsumed: %d", totalReferrals, cmWDump(pRef->netPath), pRef->serverType == DFS_ROOT_TARGET ? "root" : "link", pRef->numPathConsumed);

        /* look in domain cache by first path component */
        pDomain = ccUtilsHostFromRemotePath(pRef->netPath);
        if (NULL == pDomain)
        {
            lastError = NQ_ERR_OUTOFMEMORY;
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            goto Error;
        }
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "possible domain name: %s", cmWDump(pDomain));
#ifndef UD_NQ_USETRANSPORTIPV6
        if (cmIsIPv6Literal(pDomain))
        {
            /* if we received IPV6 literal and IPv6 not supported, we shouldn't handle this result
             * otherwise it will be identified as IP on server find or create */
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "ipv6-literal.net - skipping");
            cmMemoryFree(pDomain);
            pDomain = NULL;
            continue;
        }
#endif /* UD_NQ_USETRANSPORTIPV6 */
        pEntry = ccDfsCacheFindDomain(pDomain);
        cmMemoryFree(pDomain);
        pDomain = NULL;
        if (NULL != pEntry && NULL != pEntry->refList && NULL != pEntry->refList->first)
        {
            /* printReferrals(pEntry->refList); */
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Found in domain cache DC: %s", cmWDump(pEntry->refList->first->name));

            /* ask domain's dc for referral */
            serverHostComponent = ccUtilsHostShareFromRemotePath(pRef->netPath);
            if (NULL == serverHostComponent)
            {
                lastError = NQ_ERR_OUTOFMEMORY;
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                goto Error;
            }

            pathComponent = ccUtilsFilePathFromRemotePath(pRef->netPath, TRUE);
            if (NULL == pathComponent)
            {
                lastError = NQ_ERR_OUTOFMEMORY;
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                goto Error;
            }
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "serverHostComponent: %s", cmWDump(serverHostComponent));
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "pathComponent: %s", cmWDump(pathComponent));

            domainReferral = getDomainReferral(pEntry->refList->first->name, serverHostComponent);
            cmMemoryFree(serverHostComponent);
            serverHostComponent = NULL;
            if (NULL != domainReferral)
            {
                netPath = ccUtilsComposePath(domainReferral, pathComponent);
                cmMemoryFree(domainReferral);
                domainReferral = NULL;
                if (NULL == netPath)
                {
                    lastError = NQ_ERR_OUTOFMEMORY;
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    goto Error;
                }
                LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "New path: %s", cmWDump(netPath));
            }
            cmMemoryFree(pathComponent);
            pathComponent = NULL;
        }

        if (NULL == netPath)
        {
            netPath = cmMemoryCloneWString(pRef->netPath);
            if (NULL == netPath)
            {
                lastError = NQ_ERR_OUTOFMEMORY;
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                goto Error;
            }
        }
        else
        {
            cmMemoryFree(pRef->netPath);
            pRef->netPath = cmMemoryCloneWString(netPath);
            if (NULL == pRef->netPath)
            {
                lastError = NQ_ERR_OUTOFMEMORY;
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                cmMemoryFree(netPath);
                netPath = NULL;
                goto Error;
            }
        }

        if (doCheckReferralValidity)
        {
            /* cache only those successfully connected */
            pCredentials = pShare->user->credentials;
            pShareReferral = ccShareConnect(netPath, pMount, &pCredentials, FALSE);
            if (NULL == pShareReferral)
            {
                lastError = syGetLastError();
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "#%d failed to connect", totalReferrals);
                cmMemoryFree(netPath);
                netPath = NULL;
                continue;
            }

            if (pCredentials != pShare->user->credentials)
            {
                cmMemoryFree(pCredentials);
                pCredentials = NULL;
            }

            ccUpdateMountPointValidity(&mountPointID);
            if (MOUNTPOINT_VALID == mountPointID.status)
            {
                ccMountAddShareLink(pMount, pShareReferral);
            }
        }

        /* add to cache */
        if (NULL != ccDfsCacheAddPath(path, pRef))
        {
            isPathAdded = TRUE;
            cachedReferrals++;
        }

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "#%d %s #%d", totalReferrals, isPathAdded ? "added" : "not added", cachedReferrals);

        cmMemoryFree(netPath);
        netPath = NULL;
    } /* end of while (cmListIteratorHasNext(&iterator) && (cachedReferrals < maxReferralsToSave)) */
    cmListIteratorTerminate(&iterator);

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Referrals cached: %u out of total: %u", cachedReferrals, totalReferrals);

    /* perform possible cleanup of referrals */
    cmListIteratorStart(&refs, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        CMItem * pItem;

        pItem = cmListIteratorNext(&iterator);
        cmListItemCheck(pItem);
    }
    lastError = syGetLastError();
    cmListIteratorTerminate(&iterator);

    cmListShutdown(&refs);
    cmListItemUnlock((CMItem *)ipc);

    goto Exit;

Error:
    result = NQ_FAIL;
    cmListIteratorTerminate(&iterator);

    cmListIteratorStart(&refs, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        CMItem * pItem;

        pItem = cmListIteratorNext(&iterator);
        cmListItemCheck(pItem);
    }
    cmListIteratorTerminate(&iterator);

Error1:
    cmListShutdown(&refs);
    cmListItemUnlock((CMItem *)ipc);

Exit:
    cmMemoryFree(pDomain);
    cmMemoryFree(domainReferral);
    cmMemoryFree(netPath);
    cmMemoryFree(serverHostComponent);
    cmMemoryFree(pathComponent);

    sySetLastError(lastError);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", (result == NQ_SUCCESS && isPathAdded) ? "TRUE" : "FALSE");
    return (result == NQ_SUCCESS && isPathAdded);
}

static void getDomainDC(const NQ_WCHAR *dcName)
{
    CCShare *pShare;
    NQ_WCHAR *ipcPath;
    NQ_STATUS result;
    const AMCredentials *pCredentials = NULL;
    static const NQ_WCHAR emptyPath[] = {cmWChar('\0')};

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%s", cmWDump(dcName));
    /*LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "dc name: %s", cmWDump((const NQ_WCHAR *)dcName));*/

    /* connect to IPC$ share */
    ipcPath = ccUtilsComposeRemotePathToShare(dcName, cmIPCShareName());
    if (NULL == ipcPath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    pShare = ccShareConnect(ipcPath, NULL, &pCredentials, FALSE);
    cmMemoryFree(ipcPath);
    if (NULL != pShare)
    {
        CMList domainRefs;
        CMIterator domainIter;

        cmListStart(&domainRefs);
        /* send get referrals (get domain dfs name) */
        result = pShare->user->server->smb->doQueryDfsReferrals(pShare, emptyPath, parseReferralCallback, &domainRefs);
        if (NQ_SUCCESS == result)
        {
            cmListIteratorStart(&domainRefs, &domainIter);
            while (cmListIteratorHasNext(&domainIter))
            {
                CCDfsReferral * pDomainRef;
                CMList dcRefs;
                CMIterator dcIter;

                pDomainRef = (CCDfsReferral *)cmListIteratorNext(&domainIter);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "got domain referral: %s, rootOrLink: %s", cmWDump(pDomainRef->netPath), pDomainRef->serverType == DFS_ROOT_TARGET ? "root" : "link");
                cmListStart(&dcRefs);
                /* send get referrals (for each domain dfs name get it's dc) */
                result = pShare->user->server->smb->doQueryDfsReferrals(pShare, pDomainRef->netPath, parseReferralCallback, &dcRefs);
                if (NQ_SUCCESS == result)
                {
                    cmListIteratorStart(&dcRefs, &dcIter);
                    while (cmListIteratorHasNext(&dcIter))
                    {
                        CCDfsReferral * pRootRef;

                        pRootRef = (CCDfsReferral *)cmListIteratorNext(&dcIter);
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "got dc referral: %s, rootOrLink: %s", cmWDump(pRootRef->netPath), pRootRef->serverType == DFS_ROOT_TARGET ? "root" : "link");

                        /* store into referral cache pair of domain and it's dc */
                        ccDfsCacheAddDomain(pDomainRef->netPath + 1, pRootRef->netPath + 1, 0);
                        cmListItemCheck((CMItem *)pRootRef);
                    }
                    cmListIteratorTerminate(&dcIter);
                }
                cmListIteratorStart(&dcRefs, &dcIter);
                while (cmListIteratorHasNext(&dcIter))
                {
                    CMItem * pItem;

                    pItem = cmListIteratorNext(&dcIter);
                    cmListItemCheck(pItem);
                }
                cmListIteratorTerminate(&dcIter);
                cmListShutdown(&dcRefs);
                cmListItemCheck((CMItem *)pDomainRef);
            }
            cmListIteratorTerminate(&domainIter);
        }
        cmListIteratorStart(&domainRefs, &domainIter);
        while (cmListIteratorHasNext(&domainIter))
        {
            CMItem * pItem;

            pItem = cmListIteratorNext(&domainIter);
            cmListItemCheck(pItem);
        }
        cmListIteratorTerminate(&domainIter);
        cmListShutdown(&domainRefs);
        cmListItemUnlock((CMItem *)pShare);
    }
    cmMemoryFree(pCredentials);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}
#endif /* UD_CC_INCLUDEDFS */

static NQ_STATUS errorCodes[] =
{
    NQ_ERR_PATHNOTCOVERED,
    NQ_ERR_BADSHARE,
    NQ_ERR_SHARINGVIOLATION,
    NQ_ERR_BADPATH,
    NQ_ERR_NETWORKERROR,
    NQ_ERR_BADMEDIA,
    NQ_ERR_DIFFDEVICE,
    NQ_ERR_BADDFS,
    NQ_ERR_NOMEM,
    NQ_ERR_IOTIMEOUT,
    NQ_ERR_NOSUPPORT,
    NQ_ERR_NORESOURCE,
    NQ_ERR_BADCONNECTION,
    NQ_ERR_SHARINGPAUSED,
    NQ_ERR_BADACCESS,
    NQ_ERR_TRYAGAIN,
    NQ_ERR_TIMEOUT,
    NQ_ERR_NOTCONNECTED,
    NQ_ERR_RECONNECTREQUIRED
};

/* -- API Functions */

void ccDfsResolveOn(NQ_BOOL on)
{
#ifdef UD_CC_INCLUDEDFS
    dfsIsOn = on;
#endif /* UD_CC_INCLUDEDFS */
}

NQ_BOOL ccDfsIsError(NQ_STATUS errorCode)
{
    NQ_COUNT i;
    NQ_BOOL result = TRUE;

    for (i = 0; i < sizeof(errorCodes) / sizeof(errorCodes[0]); i++)
    {
        if (errorCodes[i] == errorCode)
        {
            goto Exit;
        }
    }

    result = FALSE;

Exit:
    return result;
}

NQ_BOOL ccDfsStart(void)
{
#ifdef UD_CC_INCLUDEDFS
    NQ_BOOL result = FALSE;
    NQ_CHAR *dcNameA = NULL;
    NQ_WCHAR *dcName = NULL;
    const NQ_CHAR *domainA = NULL;
    NQ_WCHAR *domain = NULL;
    NQ_STATUS status;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    ccDfsResolveOn(TRUE);

    /* get default domain dc */
    dcNameA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_CHAR) * (CM_NQ_HOSTNAMESIZE + 1));
    if (NULL == dcNameA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    domainA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_CHAR) * (CM_NQ_HOSTNAMESIZE + 1));
    if (NULL == domainA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    if ((status = cmGetDCName(dcNameA, &domainA)) != NQ_SUCCESS)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "dfs: default domain dc not resolved, status: %d", status);
        result = TRUE; /* DFS module starts even if DC not resolved */
        goto Exit;
    }

    dcName = cmMemoryCloneAString(dcNameA);
    domain = cmMemoryCloneAString(domainA);
    if (NULL == dcName || NULL == domain)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    ccDfsCacheAddDomain(domain, dcName, 0);
    getDomainDC(dcName);

    result = TRUE;

Exit:
    cmMemoryFree(dcNameA);
    cmMemoryFree(domainA);
    cmMemoryFree(dcName);
    cmMemoryFree(domain);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
#else /* UD_CC_INCLUDEDFS */
    return TRUE;
#endif /* UD_CC_INCLUDEDFS */
}

void ccDfsShutdown(void)
{

}

const NQ_WCHAR * ccDfsResolveHost(const NQ_WCHAR * host)
{
#ifdef UD_CC_INCLUDEDFS
    CCDfsCacheEntry * pEntry;          /* DFS cache entry */
    const NQ_WCHAR * pResult = NULL;   /* return value */
    const NQ_CHAR *domainA = NULL;
    const NQ_WCHAR * dcNameW = NULL;
    NQ_CHAR * dcNameA = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "host:%s", cmWDump(host));
    /*LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "host: %s", cmWDump(host));*/

    if (!dfsIsOn)
    {
        pResult = cmMemoryCloneWString(host);
        goto Exit;
    }

    /* assume required host is a domain name */
    /* look in domain cache first */
    pEntry = ccDfsCacheFindDomain(host);
    if (NULL != pEntry && NULL != pEntry->refList && NULL != pEntry->refList->first)
    {
#ifdef UD_NQ_INCLUDETRACE
        printReferrals(pEntry->refList);
#endif /* UD_NQ_INCLUDETRACE */
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Found in domain cache: %s", cmWDump(pEntry->refList->first->name));
        pResult = cmMemoryCloneWString(pEntry->refList->first->name);
        if (NULL == pResult)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
        }
        goto Exit;
    }
    else
    {
        NQ_STATUS status;

        domainA = cmMemoryCloneWStringAsAscii(host);
        dcNameA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_CHAR) * (CM_NQ_HOSTNAMESIZE + 1));
        if (NULL == dcNameA || NULL == domainA)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }

        /* try to resolve it as domain, return DC name, add to domain cache */
        status = cmGetDCNameByDomain(domainA, NULL, dcNameA);
        if (status != NQ_SUCCESS)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "status: %d", status);
            sySetLastError(syGetLastError());
            goto Exit;
        }
        dcNameW = cmMemoryCloneAString(dcNameA);
        if (NULL == dcNameW)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }
        ccDfsCacheAddDomain(host, dcNameW, 0);
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Resolved DC: %s", cmWDump(dcNameW));
        pResult = dcNameW;
    }

Exit:
    cmMemoryFree(dcNameA);
    cmMemoryFree(domainA);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", pResult ? cmWDump(pResult) : "null");
    return pResult;

#else /* UD_CC_INCLUDEDFS */

    return cmMemoryCloneWString(host);

#endif /* UD_CC_INCLUDEDFS */
}

CCDfsResult ccDfsResolvePath(CCMount *pMount, CCShare * pShare, const NQ_WCHAR * file, CCDfsContext * context)
{
#ifdef UD_CC_INCLUDEDFS
    CCDfsCacheEntry     *pCache;                   /* cache entry pointer */
    NQ_WCHAR            *path = NULL;              /* network path to file */
    const AMCredentials *pCredentials;             /* pointer to credentials */
    CCMountIdentifier   mountPointID;              /* mount point identifier */
#endif /* UD_CC_INCLUDEDFS */
    CCDfsResult         res = {NULL, NULL, NULL};  /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "mount:%p share:%p file:%p context:%p", pMount, pShare, file, context);

#ifdef UD_CC_INCLUDEDFS
    if (!dfsIsOn)
#endif /* UD_CC_INCLUDEDFS */
    {
        goto Exit1;
    }

#ifdef UD_CC_INCLUDEDFS

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "server: %s", cmWDump((const NQ_WCHAR *)pShare->user->server->item.name));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "share: %s", cmWDump((const NQ_WCHAR *)pShare->item.name));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "file: %s", file ? cmWDump((const NQ_WCHAR *)file) : "null");

#ifdef UD_NQ_INCLUDETRACE
    printDfsContext(context);
#endif /* UD_NQ_INCLUDETRACE */

    ccInitializeMountPointIdentifier(pMount, &mountPointID);

    if (context && context->referral && context->dfsPath && context->lastError != NQ_ERR_PATHNOTCOVERED)
    {
        /* here a required path was fully resolved in a previous step and IO operation was tried on it,
           but some IO error occurred, so need to try another referral (if available) */
        CCDfsReferral * referral = NULL;
        CMIterator iterator;
        NQ_BOOL allRefsHaveIOError = FALSE;

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "IO error on context %p current referral: %s, trying other referrals, retry: %u", context, cmWDump(context->referral), CC_DFS_NUMOFRETRIES - context->counter);

        /* first find in cache the referral itself, might be expired at this point */
        pCache = ccDfsCacheFindPath(context->dfsPath);
        if (NULL == pCache)
        {
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Not found in cache");
            goto Exit2;
        }

        /* found in cache, iterate the referrals */
        cmListIteratorStart(pCache->refList, &iterator);
        while (cmListIteratorHasNext(&iterator))
        {
            CMItem * item = cmListIteratorNext(&iterator);
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s", cmWDump(item->name));

            /* current referral, mark it */
            if (0 == cmWStricmp(item->name, context->referral))
            {
                allRefsHaveIOError = TRUE;
                ((CCDfsReferral *)item)->isIOPerformed = TRUE;
                ((CCDfsReferral *)item)->lastIOStatus = context->lastError;
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s, current, continue", cmWDump(item->name));
            }
            else
            {
                /* use the other not tried before */
                if (FALSE == ((CCDfsReferral *)item)->isIOPerformed)
                {
                     pCredentials = pShare->user->credentials;
                     res.share = ccShareConnect(item->name, pMount, &pCredentials, FALSE);
                     if (pCredentials != pShare->user->credentials)
                     {
                         cmMemoryFree(pCredentials);
                     }
                     if (NULL == res.share)
                     {
                         /* IO was not actually performed due to connectivity issue, we still consider it as if and mark */
                         allRefsHaveIOError = TRUE;

                         ((CCDfsReferral *)item)->isIOPerformed = TRUE;
                         ((CCDfsReferral *)item)->lastIOStatus = syGetLastError();
                         LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "%s failed to connect (lastIOStatus 0x%x)", cmWDump(item->name), ((CCDfsReferral *)item)->lastIOStatus);
                     }
                     else
                     {
                         allRefsHaveIOError = FALSE;

                         cmMemoryFree(context->referral);
                         context->referral = cmMemoryCloneWString(item->name);
                         if (NULL == context->referral)
                         {
                             LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                             cmListIteratorTerminate(&iterator);
                             goto Error;
                         }
                         referral = (CCDfsReferral *)item;
                         LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "%s connected", cmWDump(item->name));
                         break;
                     }
                }
            }
        } /* end of while (cmListIteratorHasNext(&iterator)) */
        cmListIteratorTerminate(&iterator);

        /* all referrals marked as having IO error, but probably some are back alive */
        if (TRUE == allRefsHaveIOError)
        {
            CMItem * currentRef = NULL;  /* pointer to current referral */

            /* iterate through referrals, if no one alive remove from cache */
            cmListIteratorStart(pCache->refList, &iterator);
            while (cmListIteratorHasNext(&iterator))
            {
               CMItem * item = cmListIteratorNext(&iterator);

               /* current referral, don't use it, probably still not ok */
               if (0 == cmWStricmp(item->name, context->referral))
               {
                   LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "%s continue, current", cmWDump(item->name));
                   currentRef = item;
               }
               else
               {
                   ccUpdateMountPointValidity(&mountPointID);
                   if (MOUNTPOINT_VALID == mountPointID.status)
                   {
                       pCredentials = pShare->user->credentials;
                       res.share = ccShareConnect(item->name, pMount, &pCredentials, FALSE);
                       if (NULL != res.share)
                       {
                           if (pCredentials != pShare->user->credentials)
                           {
                               cmMemoryFree(pCredentials);
                           }
                           LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "%s connected", cmWDump(item->name));
                           cmMemoryFree(context->referral);
                           context->referral = cmMemoryCloneWString(item->name);
                           if (NULL == context->referral)
                           {
                               LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                               cmListIteratorTerminate(&iterator);
                               goto Error;
                           }
                           referral = (CCDfsReferral *)item;
                           /* clear it's status */
                           ((CCDfsReferral *)item)->isIOPerformed = FALSE;
                           ((CCDfsReferral *)item)->lastIOStatus = NQ_SUCCESS;
                           cmListIteratorTerminate(&iterator);
                           createResultPath(file, 0, &referral->item, &res);
                           goto Exit2;
                       }
                   }
               }
            }

            /* do not terminate yet the iterator (list guard) */
            /* last resort - try current anyway */
            ccUpdateMountPointValidity(&mountPointID);
            if ((NULL != currentRef) && (MOUNTPOINT_VALID == mountPointID.status))
            {
                pCredentials = pShare->user->credentials;
                res.share = ccShareConnect(currentRef->name, pMount, &pCredentials, FALSE);
                if (NULL != res.share)
                {
                    if (pCredentials != pShare->user->credentials)
                    {
                        cmMemoryFree(pCredentials);
                    }
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "%s connected", cmWDump(currentRef->name));
                    referral = (CCDfsReferral *)currentRef;
                    /* clear it's status */
                    ((CCDfsReferral *)currentRef)->isIOPerformed = FALSE;
                    ((CCDfsReferral *)currentRef)->lastIOStatus = NQ_SUCCESS;
                    cmListIteratorTerminate(&iterator);
                    createResultPath(file, 0, &referral->item, &res);
                    goto Exit2;
                }
            }

            cmListIteratorTerminate(&iterator);

            /* remove from cache, fail */
            ccDfsCacheRemovePath(context->dfsPath);
        }
        else
        {
            createResultPath(file, 0, referral ? &referral->item : NULL, &res);
        }
        goto Exit2;
    }
    else if (!(pShare->user->server->capabilities & CC_CAP_DFS))
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Server: %s does not support DFS", cmWDump(pShare->user->server->item.name));
        goto Exit2;
    }

    /* construct remote path to file */
    path = ccUtilsComposeRemotePathToFile(pShare->user->server->item.name, pShare->item.name, file);
    if (NULL == path)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit2;
    }
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "remote path: %s", cmWDump((const NQ_WCHAR *)path));

    /* find in cache */
    pCache = ccDfsCacheFindPath(path);
    if (NULL != pCache)   /* found in cache */
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "found in cache %s match", pCache->isExactMatch ? "exact" : "partial");
#ifdef UD_NQ_INCLUDETRACE
        printReferrals(pCache->refList);
#endif /* UD_NQ_INCLUDETRACE */

        if (pCache->isExactMatch)
        {
            /* iterate through all referrals, return 1st successfully connected */
            CMItem * item = NULL;
            CMIterator iterator;

            cmListIteratorStart(pCache->refList, &iterator);
            while (cmListIteratorHasNext(&iterator) && res.share == NULL)
            {
                item = cmListIteratorNext(&iterator);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s", cmWDump(item->name));

                pCredentials = pShare->user->credentials;
                res.share = ccShareConnect(item->name, pMount, &pCredentials, FALSE);
                if (pCredentials !=  pShare->user->credentials)
                {
                    cmMemoryFree(pCredentials);
                }
            }
            cmListIteratorTerminate(&iterator);

            if (NULL != context)
            {
                cmMemoryFree(context->referral);
                context->referral = cmMemoryCloneWString(item->name);
                if (NULL == context->referral)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    goto Error;
                }
                cmMemoryFree(context->dfsPath);
                context->dfsPath = cmMemoryCloneWString(pCache->item.name);
                if (NULL == context->dfsPath)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    goto Error;
                }
            }
            createResultPath(path, pCache->numPathConsumed, item, &res);
            goto Exit2;
        }
        else
        {   /* not exact match */
            if (pCache->isRoot)
            {
                /* ask root for referrals */
                /* get new share and path */
                CCShare * pRootShare = NULL;

                /* connect to root */
                {
                    /* iterate through all referrals, return 1st successfully connected */
                    CMItem * item = NULL;
                    CMIterator iterator;

                    cmListIteratorStart(pCache->refList, &iterator);
                    while (cmListIteratorHasNext(&iterator) && pRootShare == NULL)
                    {
                        item = cmListIteratorNext(&iterator);
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s", cmWDump(item->name));

                        pCredentials = pShare->user->credentials;
                        pRootShare = ccShareConnect(item->name, pMount, &pCredentials, FALSE);
                        ccUpdateMountPointValidity(&mountPointID);
                        if ((NULL != pRootShare) && (MOUNTPOINT_VALID == mountPointID.status))
                        {
                            ccMountAddShareLink(pMount, pRootShare);
                        }

                        if (pCredentials != pShare->user->credentials)
                        {
                            cmMemoryFree(pCredentials);
                        }
                    }
                    cmListIteratorTerminate(&iterator);
                }

                /* ask root for referrals */
                if (pRootShare && getReferrals(pMount, pRootShare, path, parseReferralCallback))
                {
                    /* return 1st found referral */
                    pCache = ccDfsCacheFindPath(path);
                    if (NULL != pCache)   /* found in cache */
                    {
                        /* iterate through all referrals, return 1st successfully connected */
                        CMItem * item = NULL;
                        CMIterator iterator;

                        cmListIteratorStart(pCache->refList, &iterator);

                        while (cmListIteratorHasNext(&iterator) && res.share == NULL)
                        {
                            item = cmListIteratorNext(&iterator);
                            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s", cmWDump(item->name));

                            pCredentials = pShare->user->credentials;
                            res.share = ccShareConnect(item->name, pMount, &pCredentials, FALSE);
                            if (pCredentials != pShare->user->credentials)
                            {
                                cmMemoryFree(pCredentials);
                            }
                        }
                        cmListIteratorTerminate(&iterator);

                        if (NULL != context && NULL != res.share)
                        {
                            cmMemoryFree(context->referral);
                            context->referral = cmMemoryCloneWString(item->name);
                            if (NULL == context->referral)
                            {
                                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                                goto Error;
                            }
                            cmMemoryFree(context->dfsPath);
                            context->dfsPath = cmMemoryCloneWString(pCache->item.name);
                            if (NULL == context->dfsPath)
                            {
                                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                                goto Error;
                            }
                        }
                        createResultPath(path, pCache->numPathConsumed, item, &res);
                        goto Exit2;
                    }
                }
            }
            else
            {
                /* iterate through all referrals, return 1st successfully connected */
                CCDfsReferral *referral = NULL;
                CMIterator iterator;

                cmListIteratorStart(pCache->refList, &iterator);
                while (cmListIteratorHasNext(&iterator) && res.share == NULL)
                {
                    referral = (CCDfsReferral *)cmListIteratorNext(&iterator);
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s", cmWDump(referral->item.name));

                    if (referral->isIOPerformed && referral->lastIOStatus != NQ_SUCCESS)
                    {
                        continue;
                    }
                    pCredentials = pShare->user->credentials;
                    res.share = ccShareConnect(referral->item.name, pMount, &pCredentials, FALSE);
                    if (pCredentials !=  pShare->user->credentials)
                    {
                        cmMemoryFree(pCredentials);
                    }
                }
                cmListIteratorTerminate(&iterator);

                if (NULL != context && NULL != res.share)
                {
                    cmMemoryFree(context->referral);
                    context->referral = cmMemoryCloneWString(referral->item.name);
                    if (NULL == context->referral)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                        goto Error;
                    }
                    cmMemoryFree(context->dfsPath);
                    context->dfsPath = cmMemoryCloneWString(pCache->item.name);
                    if (NULL == context->dfsPath)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                        goto Error;
                    }
                }
                createResultPath(path, pCache->numPathConsumed, &referral->item, &res);
                goto Exit2;
            }
        }
    }
    else   /* not found in cache */
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "not found in cache");

        /* get referrals and add to cache */
        if (getReferrals(pMount, pShare, path, parseReferralCallback))
        {
            /* return 1st found referral */
            pCache = ccDfsCacheFindPath(path);
            if (NULL != pCache)   /* found in cache */
            {
                /* iterate through all referrals, return 1st successfully connected */
                CMItem * item = NULL;
                CMIterator iterator;

                cmListIteratorStart(pCache->refList, &iterator);
                while (cmListIteratorHasNext(&iterator) && res.share == NULL)
                {
                    item = cmListIteratorNext(&iterator);
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, " %s", cmWDump(item->name));

                    pCredentials = pShare->user->credentials;
                    res.share = ccShareConnect(item->name, pMount, &pCredentials, FALSE);
                    if (pCredentials != pShare->user->credentials)
                    {
                        cmMemoryFree(pCredentials);
                    }
                }
                cmListIteratorTerminate(&iterator);

                if (NULL != context && NULL != res.share)
                {
                    cmMemoryFree(context->referral);
                    context->referral = cmMemoryCloneWString(item->name);
                    if (NULL == context->referral)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                        goto Error;
                    }
                    cmMemoryFree(context->dfsPath);
                    context->dfsPath = cmMemoryCloneWString(pCache->item.name);
                    if (NULL == context->dfsPath)
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                        goto Error;
                    }
                }
                createResultPath(path, pCache->numPathConsumed, item, &res);
                goto Exit2;
            }
        }
        else
        {
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Failed to get referrals");
        }
    }

    goto Exit2;

Error:
    if (NULL != context->referral)
    {
        cmMemoryFree(context->referral);
        context->referral = NULL;
    }
    sySetLastError(NQ_ERR_OUTOFMEMORY);


Exit2:
    if (path != NULL)
    {
        cmMemoryFree(path);
    }

    ccUpdateMountPointValidity(&mountPointID);
    if (MOUNTPOINT_INVALID == mountPointID.status)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "mount was disposed");

        if (NULL != res.path)
        {
            cmMemoryFree(res.path);
        }
        res.path = NULL;
        res.share = NULL;
        res.server = NULL;
    }
    else if (NULL != res.share)
    {
        ccMountAddShareLink(pMount, res.share);
    }

    logPrintResult(res);
#endif /* UD_CC_INCLUDEDFS */
Exit1:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", &res);
    return res;
}

void ccDfsResolveDispose(CCDfsResult * pRes)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pRes:%p", pRes);

    if (pRes && pRes->path)
    {
        cmMemoryFree(pRes->path);
        pRes->path = NULL;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

#endif /* UD_NQ_INCLUDECIFSCLIENT */
