/*********************************************************************
 *
 *          Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Implementation of the EXTERNAL NOTIFY CHANGE mechanism
 *--------------------------------------------------------------------
 * MODULE        : Linux - SY
 * DEPENDENCIES  :
 ********************************************************************/

#include <poll.h>
#include <sys/inotify.h>
#include "syexnotify.h"

typedef struct
{
    NQ_HANDLE   context;                            /* File descriptor for external notify.*/
    CMList      watchList;                          /* List of SYWFile. */
    CMIterator  watchListIter;                      /* watchList iterator. */
    NotifyImmediatelyCallback pImmediatelyCallback; /* Pointer to csNotifyImmediatelly */
} ExStaticData;

#ifdef SY_FORCEALLOCATION
static ExStaticData *staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static ExStaticData staticDataSrc;
static ExStaticData *staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */
SYThread ExNotifyThread = 0;
NQ_BOOL doExit;  /* flag to exit from external notify monitoring thread*/

/*====================================================================
 * PURPOSE: initialize resources
 *--------------------------------------------------------------------
 * PARAMS:  IN callback - Pointer to csNotifyImmediatelly
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */
NQ_STATUS syExNotifyInit(NotifyImmediatelyCallback callback)
{
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_INFO,"callback:%p", callback);

    if (NULL == callback)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input parameters");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (ExStaticData *)cmMemoryAllocate(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    staticData->context = (NQ_INT32 *)cmMemoryAllocate(sizeof(NQ_INT32));
    if (NULL == staticData->context)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    /* Create the file descriptor for accessing the inotify API */
    *(NQ_INT32 *)(staticData->context) = inotify_init1(IN_NONBLOCK);

    /* inotify initialization fail*/
    if (EXNOTIFY_FAIL == *(NQ_INT32 *)(staticData->context))
    {
        cmMemoryFree(staticData->context);
        staticData->context = NULL;
        LOGERR(CM_TRC_LEVEL_ERROR, "ExNotify initialization failed");
        sySetLastError(NQ_ERR_ERROR);
        goto Exit;
    }

    /* Initilaize watchList */
    cmListStart(&staticData->watchList);

    staticData->pImmediatelyCallback = callback;
    doExit = FALSE;

    syThreadStart(FALSE, NQ_THREAD_PRIORITY_NONE, &ExNotifyThread, syExNotifyMonitor, FALSE);

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_INFO, "result:%d", result);
    return result;
}

/*====================================================================
 * PURPOSE: find a file in watchList providing fileCtx
 *--------------------------------------------------------------------
 * PARAMS:  IN fileCtx - file watch descriptor
 *
 * RETURNS: Pointer to a slot or NULL
 *
 * NOTES:
 *====================================================================
 */
static SYWFile *syGetSYWFileByCtx(NQ_HANDLE fileCtx)
{
    SYWFile *syf = NULL;

    LOGFB(CM_TRC_LEVEL_INFO, "fileCtx:%p", fileCtx);

    if (NULL == fileCtx)
    {
       LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input parameters");
       sySetLastError(NQ_ERR_BADPARAM);
       goto Exit;
    }

    cmListIteratorStart(&staticData->watchList, &staticData->watchListIter);

    while (cmListIteratorHasNext(&staticData->watchListIter))
    {
        syf = (SYWFile *)cmListIteratorNext(&staticData->watchListIter);
        if ((*((NQ_INT32 *)syf->fileCtx)) == (*((NQ_INT32 *)fileCtx)))
        {
            goto Exit;
        }
    }

    syf = NULL;

Exit:
    cmListIteratorTerminate(&staticData->watchListIter);
    LOGFE(CM_TRC_LEVEL_INFO, "syf:%p", syf);
    return syf;
}

/*====================================================================
 * PURPOSE: Dispose item from the inotify watch list
 *--------------------------------------------------------------------
 * PARAMS:  IN pItem - Item to dispose from external notify watch list
 *
 * RETURNS: TRUE
 *
 * NOTES
 *====================================================================
 */
static NQ_BOOL disposeItem(CMItem *pItem)
{
    SYWFile *wFile = (SYWFile *)pItem;

    LOGFB(CM_TRC_LEVEL_INFO, "pItem:%p", pItem);

    if (NULL == wFile)
    {
       LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input parameters");
       sySetLastError(NQ_ERR_BADPARAM);
       goto Exit;
    }

    if (NULL != wFile->fileCtx)
    {
        if (EXNOTIFY_FAIL == inotify_rm_watch(*(NQ_INT32 *)(staticData->context),*(NQ_INT32 *)wFile->fileCtx))
        {
           LOGERR(CM_TRC_LEVEL_ERROR, "Exnotify failed removing file from watch list: %s", cmWDump(wFile->name));
           sySetLastError(NQ_ERR_ERROR);
        }

        cmMemoryFree(wFile->fileCtx);
        wFile->fileCtx = NULL;
    }

Exit:
    LOGFE(CM_TRC_LEVEL_INFO);
    return TRUE;
}

/*====================================================================
 * PURPOSE: Adding files/ directories to external notify watch list
 *--------------------------------------------------------------------
 * PARAMS:  IN path - path of file/ directory to be monitored
 *
 * RETURNS: SYWFile - a pointer to a file object
 *
 * NOTES
 *====================================================================
 */

SYWFile *syExNotifyAddWatch(NQ_WCHAR *path)
{
    NQ_CHAR ansiName[CM_BUFFERLENGTH(NQ_WCHAR, UD_FS_FILENAMELEN)];
    NQ_INT32 fileCtx;
    SYWFile *wFile = NULL;

    LOGFB(CM_TRC_LEVEL_INFO, "path:%p", cmWDump(path));

    if (NULL == path)
    {
       LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input parameters");
       sySetLastError(NQ_ERR_BADPARAM);
       goto Exit;
    }

    cmUnicodeToAnsiN(ansiName, (NQ_UINT)sizeof(NQ_CHAR)*((NQ_UINT)syWStrlen(path)+1), path, (NQ_UINT)sizeof(NQ_WCHAR)*(NQ_UINT)syWStrlen(path));
    ansiName[syWStrlen(path)] = '\0';

    fileCtx = inotify_add_watch(*(NQ_INT32 *)(staticData->context), ansiName, IN_CREATE |IN_DELETE |IN_MODIFY |IN_MOVED_FROM |IN_MOVED_TO );

    /* Add watch fail. */
    if (EXNOTIFY_FAIL == fileCtx)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "ExNotify failed adding file to watch list: %s", ansiName);
        sySetLastError(NQ_ERR_ERROR);
        goto Exit;
    }

    /* check if file is already in the watch list */
    if (NULL != (wFile = syGetSYWFileByCtx(&fileCtx)))
    {
        wFile->regNum++;
        goto Exit;
    }

    /* Add file to watch list. */
    wFile = (SYWFile *)cmListItemCreateAndAdd(&staticData->watchList, sizeof(SYWFile), NULL, disposeItem, CM_LISTITEM_NOLOCK , FALSE);
    if (NULL == wFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    /* Set wFile params. */
    wFile->regNum = 1;
    syWStrcpy(wFile->name, path);

    /* allocate memory */
    wFile->fileCtx = (NQ_INT32 *)cmMemoryAllocate(sizeof(NQ_INT32));
    if (NULL == wFile->fileCtx)
    {
        if (EXNOTIFY_FAIL == inotify_rm_watch(*(NQ_INT32 *)(staticData->context), fileCtx))
        {
           LOGERR(CM_TRC_LEVEL_ERROR, "Exnotify failed removing file from watch list: %s", cmWDump(wFile->name));
           sySetLastError(NQ_ERR_ERROR);
        }

        syExNotifyRemoveWatch(wFile);
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    *((NQ_INT32 *)wFile->fileCtx) = fileCtx;

Exit:
    LOGFE(CM_TRC_LEVEL_INFO, "wFile:%p", wFile);
    return wFile;
}

/*====================================================================
 * PURPOSE: Remove item from the inotify watch list
 *--------------------------------------------------------------------
 * PARAMS:  IN wFile - Item to remove from external notify watch list
 *
 * RETURNS: None
 *
 * NOTES
 *====================================================================
 */
void syExNotifyRemoveWatch(SYWFile *wFile)
{
    LOGFB(CM_TRC_LEVEL_INFO, "wFile:%p", wFile);

    if (NULL == wFile)
   {
      LOGERR(CM_TRC_LEVEL_ERROR , "Invalid input parameters");
      sySetLastError(NQ_ERR_BADPARAM);
      goto Exit;
   }

    if (0 < --wFile->regNum)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Number of registration to file: %d", wFile->regNum);
        goto Exit;
    }

    disposeItem((CMItem *)wFile);
    cmListItemRemoveAndDispose((CMItem *)wFile);

Exit:
    LOGFE(CM_TRC_LEVEL_INFO);
}

void syExNotifyMonitor(void)
{
    /* Prepare for polling */
    struct pollfd fds[1];
    NQ_INT32 poll_num;
    nfds_t nfds;
    NQ_INT32 timeout = EXNOTIFY_MAX_TIMEOUT;

    LOGFB(CM_TRC_LEVEL_INFO);

    nfds = 1;   /* number of items in the fds array */

    /* Inotify input */
    fds[0].fd = *(NQ_INT32 *)(staticData->context);
    fds[0].events = POLLIN;

    /* Wait for events and/or terminal input */
    while (FALSE == doExit)
    {
       poll_num = poll(fds, nfds, timeout);
       if (EXNOTIFY_FAIL == poll_num)
       {
           if (EINTR == errno)
           {
               LOGERR(CM_TRC_LEVEL_ERROR, "Enotify faild polling");
               sySetLastError(NQ_ERR_ERROR);
               return;
           }
       }

       if (0 < poll_num)
       {
           if (TRUE == (fds[0].revents & POLLIN))
           {
               /* Inotify events are available */
               handleEvents();
           }
       }
    }

    LOGFE(CM_TRC_LEVEL_INFO);
}

void handleEvents(void)
{
   NQ_CHAR buf[4096]
       __attribute__ ((aligned(__alignof__(struct inotify_event))));
   struct inotify_event *event;
   ssize_t len;
   NQ_CHAR *ptr;
   NQ_CHAR path[CM_BUFFERLENGTH(NQ_WCHAR, UD_FS_FILENAMELEN)];
   NQ_WCHAR fullPath[CM_BUFFERLENGTH(NQ_WCHAR, UD_FS_FILENAMELEN)];
   NQ_UINT32 action = 0;
   NQ_UINT pathNameSize = 0;
   NQ_UINT fileNameSize = 0;
   SYWFile *wFile;

   LOGFB(CM_TRC_LEVEL_INFO);

   /* Loop while events can be read from inotify file descriptor. */
   for (;;)
   {
       /* Read events. */
       len = syReadFile(*(NQ_INT32 *)(staticData->context), (NQ_BYTE *)buf, sizeof( buf));
       if ((EXNOTIFY_FAIL == len) && (EAGAIN != errno))
       {
           LOGERR(CM_TRC_LEVEL_ERROR, "Exnotify failed read event");
           sySetLastError(NQ_ERR_ERROR);
           return;
       }

       /* If the nonblocking read() found no events to read, then
          it returns EXNOTIFY_FAIL with errno set to EAGAIN. In that case,
          we exit the loop. */
       if (0 >= len)
       {
           break;
       }

       /* Loop over all events in the buffer */
       for (ptr = buf; ptr < buf + len; ptr += sizeof(struct inotify_event) + event->len)
       {
           event = (struct inotify_event *) ptr;

           /* Print type of filesystem object */
           wFile = syGetSYWFileByCtx(&event->wd);
           if ((NULL != wFile) && (0 != event->len))
           {
                /* Get full path in unicode */
                pathNameSize = (NQ_UINT)syWStrlen(wFile->name);
                fileNameSize = (NQ_UINT)syStrlen(event->name);
                cmUnicodeToAnsiN(path, (NQ_UINT)sizeof(NQ_CHAR) * (pathNameSize+1), wFile->name, (NQ_UINT)sizeof(NQ_WCHAR) * pathNameSize);
                path[pathNameSize] = '/';
                path[pathNameSize + 1] = '\0';
                syStrcat(path, event->name);
                path[pathNameSize + fileNameSize + 1] = '\0';
                cmAnsiToUnicodeN(fullPath, (NQ_UINT)sizeof(NQ_WCHAR) * (pathNameSize + fileNameSize +2), path, (NQ_UINT)sizeof(NQ_CHAR) * (pathNameSize + fileNameSize +2));

                if (0 != (event->mask & IN_CREATE ))
                {
                    action = SMB_NOTIFYCHANGE_ADDED;
                }
                else if (0 != (event->mask & IN_DELETE ))
                {
                    action = SMB_NOTIFYCHANGE_REMOVED;
                }
                else if (0 != (event->mask & IN_MODIFY ))
                {
                    action = SMB_NOTIFYCHANGE_MODIFIED;
                }
                else if (0 != (event->mask & IN_MOVED_FROM ))
                {
                    action = SMB_NOTIFYCHANGE_RENAMEDOLDNAME;
                    staticData->pImmediatelyCallback(fullPath, action, COMPLETION_FILTER);
                    continue;
                }
                else if (0 != (event->mask & IN_MOVED_TO ))
                {
                    action = SMB_NOTIFYCHANGE_RENAMEDNEWNAME;
                }

                /* remove file from watch list. */
                syExNotifyRemoveWatch(wFile);
                staticData->pImmediatelyCallback(fullPath, action, COMPLETION_FILTER);
           }
       }
   }

   LOGFE(CM_TRC_LEVEL_INFO);
}

/*====================================================================
 * PURPOSE: release resources
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */
void syExNotifyExit(void)
{
    CMIterator  itr;

    LOGFB(CM_TRC_LEVEL_INFO);

    cmListIteratorStart(&staticData->watchList, &itr);
    while (cmListIteratorHasNext(&itr))
    {
        CMItem * pItem = cmListIteratorNext(&itr);

        cmListItemCheck(pItem);
        cmListItemRemoveAndDispose(pItem);
    }
    cmListIteratorTerminate(&itr);

    doExit = TRUE;
    syCloseFile(*(NQ_INT32 *)(staticData->context));
    syThreadDestroy(ExNotifyThread);

#ifdef SY_FORCEALLOCATION
    /* release memory */
    if (NULL != staticData)
    {
        cmMemoryFree(staticData->context);
        cmMemoryFree(staticData);
    }

    staticData = NULL;
#endif /* SY_FORCEALLOCATION */

    LOGFE(CM_TRC_LEVEL_INFO);
}
