/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Implementation of session data transfer functions
 *--------------------------------------------------------------------
 * MODULE        : Network
 * DEPENDENCIES  :
 ********************************************************************/

#include "nsapi.h"
#include "cmapi.h"
#include "nssocket.h"
#include "nsbuffer.h"
#include "nscommon.h"
#include "nsframes.h"

/*
 This file implements r/w functions for a not connected NetBIOS socket.

 Data is packed into a Session Message as in RFC-1002 with the following restrictions:
    nsSendFromBuffer - data should reside in one message. A message that does not fit in a buffer
              is truncated
    nsRecvIntoBuffer -  only the 1st fragment is accepted. All subsequent fragments of a multi-fragment
              message are discarded

 These calls work for not NetBIOS (pure Internet) sockets too. In this case a call is
 delegated to the underlying socket
*/

/*
 *====================================================================
 * PURPOSE: Skip space for a session header
 *--------------------------------------------------------------------
 * PARAMS:  IN socket descriptor
 *          IN buffer to use
 *
 * RETURNS: pointer to the user data in the buffer
 *====================================================================
 */


NQ_BYTE*
nsSkipHeader(
    NSSocketHandle socket,
    NQ_BYTE *buf
    )
{
    SocketSlot* pSock;                      /* actual pointer to a socket slot */
    NQ_BYTE* pResult = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "socket:%p buf:%p", socket, buf);

    pSock = (SocketSlot*)socket;

    if (pSock != NULL && !pSock->isNetBios) /* socket is not NetBIOS */
    {
        LOGMSG(CM_TRC_LEVEL_MESS_SOME, "Not a NetBIOS socket");
        pResult = buf;
        goto Exit;
    }
    pResult = buf + sizeof(CMNetBiosSessionMessage);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:%p", pResult);
    return pResult;
}

/*
 *====================================================================
 * PURPOSE: Write to a stream
 *--------------------------------------------------------------------
 * PARAMS:  IN buffer to send (including space for header)
 *          IN packetlen length of the NBT packet
 *          IN dataCount data length - may include the entire packet data or
 *                       just headers without payload
 *
 *
 * RETURNS: TRUE or FALSE
 * NOTES:
 *====================================================================
 */

NQ_COUNT
nsPrepareNBBuffer(
    NQ_IOBufPos buf,            /* buffer to use */
    NQ_UINT packetLen,          /* packet length */
    NQ_UINT dataCount           /* data length (may the entire packet data or just headers with no payload) */
    )
{
    NQ_COUNT res = 0;
    CMNetBiosSessionMessage* msgHdr;    /* casted pointer to the outgoing message */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "buf:%p packetLen:%u dataCount:%u" , buf, packetLen, dataCount);

    if (packetLen <= 0)
    {
        sySetLastError(CM_NBERR_INVALIDPARAMETER);
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid data length");
        goto Exit;
    }

    msgHdr = (CMNetBiosSessionMessage*) IOBUF_GETSTARTPTR(buf);    /* cast the pointer */

    /* create a session message */
    msgHdr->type = CM_NB_SESSIONMESSAGE;
    msgHdr->flags = (NQ_BYTE)(packetLen / 0x10000);
    packetLen = packetLen % 0x10000;
    /* possible transaction of the 17th bit - extension */
    cmPutSUint16(msgHdr->length, (NQ_UINT16)syHton16((NQ_UINT16)packetLen));
    res = (NQ_COUNT)(dataCount + sizeof(CMNetBiosSessionMessage));

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:%d", res);
    return res;
}

/*
 *====================================================================
 * PURPOSE: Write to a stream
 *--------------------------------------------------------------------
 * PARAMS:  IN socket descriptor
 *          IN buffer to send (including space for header)
 *          IN packetlen length of the packet
 *          IN dataCount data length - may include the entire packet data or
 *                       just headers without payload
 *          IN callback function releasing the buffer
 *
 * RETURNS: Number of bytes written into the stream
 *
 * NOTES:   deallocates the buffer
 *====================================================================
 */

NQ_INT
nsSendFromBuffer(
    NSSocketHandle socketHandle,
    NQ_IOBufPos buf,
    NQ_UINT packetLen,          
    NQ_UINT dataCount,          
    NSReleaseCallback release
    )
{
    SocketSlot* pSock;                  /* actual pointer to a socket slot */
    NQ_INT msgLen;                      /* this message length */
#ifndef UD_NS_ASYNCSEND
    NQ_INT dataSent;
    NQ_INT dataToSend;
#endif
    NQ_INT result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "socketHandle:%p buf:%p packetLen:%u dataCount:%u release:%p", socketHandle, buf, packetLen, dataCount, release);

    if (packetLen <= 0)
    {
        sySetLastError(CM_NBERR_INVALIDPARAMETER);
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid data length");
        goto Error;
    }

    pSock = (SocketSlot*)socketHandle;
    if (pSock == NULL || !syIsValidSocket(pSock->socket))
    {
        sySetLastError(CM_NBERR_INVALIDPARAMETER);
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal socket descriptor");
        goto Error;
    }

    /* send the message */

#ifdef UD_NS_ASYNCSEND
    if (release == NULL)
    {
        msgLen = sySendSocket(
            pSock->socket,
            IOBUF_GETBUFPTR(buf),
           dataCount
        );
    }
    else
    {
        msgLen = sySendSocketAsync(
            pSock->socket,
            IOBUF_GETBUFPTR(buf),
            dataCount,
            release
        );
    }
#else
    if (pSock->isAccepted)
    {
        syMutexTake(&pSock->guard);
    }

    {
#ifdef UD_NQ_USEIOVECS
        NQ_IOBufState bufState;
        NQ_IOBufOrigin bufOrig;
#endif /* UD_NQ_USEIOVECS */
        NQ_IOBuf sendBuf;
        sendBuf = IOBUF_GETBUFPTR(buf);
        IOBUF_PREPARELENGTH(sendBuf, bufOrig, bufState, dataCount)

        for (dataSent = 0, dataToSend = (NQ_INT)dataCount; dataToSend > 0; dataToSend -= dataSent)
        {
            NQ_COUNT offSet = (NQ_COUNT)dataSent;
            IOBUF_IOVECBUF_ADVANCE_SAVE(sendBuf, offSet, bufState)
            if (NQ_FAIL == (dataSent = sySendSocket(pSock->socket, sendBuf, (NQ_COUNT)dataToSend)))
            {
                break;
            }
        }
        msgLen = (NQ_INT)(dataCount - (dataToSend > 0? (NQ_UINT)dataToSend : 0));
        IOBUF_IOVECBUF_RESTORE(sendBuf, bufState, bufOrig)
    }

    if (pSock->isAccepted)
    {
        syMutexGive(&pSock->guard);
    }

    if (NULL != release)
    {
        (*release)(IOBUF_GETBUFPTR(buf));        /* immediately release the buffer */
    }
#endif /* UD_NS_ASYNCSEND */

    if (msgLen == NQ_FAIL) /* error */
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error while sending message");
        goto Exit;
    }

    result = msgLen;
    goto Exit;

Error:
    if (NULL != release)
    {
        (*release)(IOBUF_GETBUFPTR(buf));
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Prepare reading from an NBT stream
 *--------------------------------------------------------------------
 * PARAMS:  IN socket descriptor
 *          OUT pointer to the receive descriptor
 *
 * RETURNS: Number of bytes available for receive in the NBT packet
 *
 * NOTES:   This function should be called prior to nsRecvIntoBuffer
 *====================================================================
 */

NQ_INT
nsStartRecvIntoBuffer(
    NSSocketHandle socket,  
    NSRecvDescr * descr      
    )
{
    SocketSlot* pSock;                      /* actual pointer to a socket slot */
    NQ_INT bytesToRead;                     /* number of bytes to receive */
    NQ_UINT32 packetLen;                    /* packet length, including the extension (E) bit */
    CMNetBiosSessionMessage buffer;         /* to read NBT header */
    NQ_IOBuf pBuf;                          /* pointer into buffer */
    NQ_INT res = 0;
    NQ_INT result = NQ_FAIL;
#ifdef UD_NQ_USEIOVECS
    NQ_IOBufState bufState;
    NQ_IOBufOrigin bufOrig;
#endif /* UD_NQ_USEIOVECS */
    IOBUF_CONSTRUCTORINIT(pBuf);

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "socket: %p, descr: %p", socket, descr);

    pSock = (SocketSlot*)socket;

    if (pSock==NULL || !syIsValidSocket(pSock->socket))
    {
        sySetLastError(CM_NBERR_INVALIDPARAMETER);
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal socket descriptor");
        goto Exit;
    }

    descr->socket = pSock;
    bytesToRead = sizeof(CMNetBiosSessionMessage);
    IOBUF_CONSTRUCTOR(pBuf, (NQ_BYTE*)&buffer, sizeof(buffer));
    IOBUF_SETORIGIN(pBuf, bufOrig, bufState)
    IOBUF_PREPARELENGTH(pBuf, bufOrig, bufState, sizeof(buffer))
    while (bytesToRead > 0)
    {
        NQ_COUNT numBytes = (NQ_COUNT)res;

        IOBUF_IOVECBUF_ADVANCE_SAVE(pBuf, numBytes, bufState)
        res = syRecvSocketWithTimeout(pSock->socket, pBuf, (NQ_UINT)bytesToRead, UD_NQ_TCPSOCKETRECVTIMEOUT);

        /* if no bytes received this means that the remote end died */
        if (res == 0 || res == NQ_FAIL)
        {
            if (res == 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "0 bytes read (header)");
                goto Exit;
            }

            LOGERR(CM_TRC_LEVEL_ERROR, "Error during reading header");
            goto Exit;
        }

        if (buffer.type == CM_NB_SESSIONKEEPALIVE)
        {
            break;
        }
        
        bytesToRead -= res;
    }
    /* check for a control message */
    if (buffer.type != CM_NB_SESSIONMESSAGE)
    {
        switch (buffer.type)
        {
        case CM_NB_SESSIONKEEPALIVE:
            /* discard */
            result = 0;
            goto Exit;
        default:
            LOGERR(CM_TRC_LEVEL_ERROR, "Unexpected packet type 0x%x", buffer.type);
            goto Exit;
        }
    }

    packetLen = (NQ_UINT32)(syHton16(cmGetSUint16((NQ_UINT16)buffer.length)) & 0xFFFF);
    packetLen += ((NQ_UINT32) buffer.flags) * 0x10000;
    if (packetLen == 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unexpected zero length");
        goto Exit;
    }
/*    packetLen |= (((NQ_UINT32) buffer.flags) & CM_NB_SESSIONLENGTHEXTENSION) << 16; */
    descr->remaining = (NQ_COUNT)packetLen;
    result = (NQ_INT)packetLen;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result: %d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Prepare reading from an Rpc over Tcp stream
 *--------------------------------------------------------------------
 * PARAMS:  IN socket descriptor
 *          OUT pointer to the receive descriptor
 *
 * RETURNS: Number of bytes available for receive in the Rpc packet
 *
 * NOTES:   This function should be called prior to nsRecvFRomBuffer
 *====================================================================
 */

NQ_INT
nsStartRecvIntoRpcBuffer(
    NSSocketHandle socket,
    NSRecvDescr * descr,
    NQ_BYTE* bufPtr,
    NQ_COUNT bytesToRead
    )
{
    SocketSlot* pSock;                      /* actual pointer to a socket slot */
    NQ_COUNT totalBytesRead = 0;
    NQ_INT byteCount = (NQ_INT) bytesToRead;
    NQ_UINT16 packetLen;                    /* packet length */
    NQ_INT result = NQ_FAIL;
    NQ_INT res = 0;
    NQ_IOBuf buf;
#ifdef UD_NQ_USEIOVECS
    NQ_IOBufState bufState;
    NQ_IOBufOrigin bufOrig;
#endif /* UD_NQ_USEIOVECS */
    IOBUF_CONSTRUCTORINIT(buf)

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "socket: %p, descr: %p", socket, descr);

    IOBUF_CONSTRUCTOR(buf, bufPtr, bytesToRead);
    IOBUF_SETORIGIN(buf, bufOrig, bufState)
    IOBUF_PREPARELENGTH(buf, bufOrig, bufState, bytesToRead)

    pSock = (SocketSlot*)socket;

    if (pSock==NULL || !syIsValidSocket(pSock->socket))
    {
        sySetLastError(CM_NBERR_INVALIDPARAMETER);
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal socket descriptor");
        goto Exit;
    }

    descr->socket = pSock;

    while (byteCount > 0)
    {
        NQ_COUNT numBytes = (NQ_COUNT)res;

        IOBUF_IOVECBUF_ADVANCE_SAVE(buf, numBytes, bufState);
        res = syRecvSocket(pSock->socket, buf, (NQ_COUNT)bytesToRead);

        /* if no bytes received this means that the remote end died */
        if (res == 0 || res == NQ_FAIL)
        {
            if (res == 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "0 bytes read (header)");
                goto Exit;
            }

            LOGERR(CM_TRC_LEVEL_ERROR, "Error during reading header");
            goto Exit;
        }

        totalBytesRead += (NQ_COUNT)res;
        byteCount -= res;
    }

    IOBUF_IOVECBUF_RESTORE(buf, bufState, bufOrig);

    packetLen = *(NQ_UINT16 *)(IOBUF_IOVEC_GETBYTEPTR(buf) + 8);
    if (packetLen == 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unexpected zero length");
        goto Exit;
    }
    descr->remaining = (NQ_COUNT)packetLen;
    result = (NQ_INT)packetLen;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result: %d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: End reading from an NBT stream
 *--------------------------------------------------------------------
 * PARAMS:  IN socket descriptor
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:   Usually this call does nothing. On possible buffer overflow it
 *          may discard the tail of the NBT packet 
 *====================================================================
 */

NQ_STATUS
nsEndRecvIntoBuffer(
        NSRecvDescr * descr      
    )
{
    SocketSlot* pSock;                      /* actual pointer to a socket slot */
    NQ_INT bytesRead;                       /* number of bytes received */
    NQ_UINT32 lenToRead;                    /* data length we can read, may be less on buffer limits */
    static NQ_BYTE buffer[300];             /* buffer for discarded bytes */
    NQ_IOBuf ioBuf;
    NQ_STATUS result = NQ_FAIL;
    IOBUF_CONSTRUCTORINIT(ioBuf);

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "descr: %p", descr);

    pSock = (SocketSlot*)descr->socket;

    if (pSock==NULL || !syIsValidSocket(pSock->socket))
    {
        sySetLastError(CM_NBERR_INVALIDPARAMETER);
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal socket descriptor");
        goto Exit;
    }

    if (descr->remaining != 0)
    { 
        while (descr->remaining > 0)
        {
#ifdef UD_NQ_USEIOVECS
            NQ_IOBufState bufState;
            NQ_IOBufOrigin bufOrig;
            NQ_COUNT x = 1;
#endif /* UD_NQ_USEIOVECS */
            lenToRead = (NQ_UINT32)(descr->remaining > sizeof(buffer) ? sizeof(buffer) : descr->remaining);
            IOBUF_CONSTRUCTOR(ioBuf, buffer, sizeof(buffer))

            IOBUF_SETORIGIN(ioBuf, bufOrig, bufState)
            IOBUF_PREPARELENGTH(ioBuf, bufOrig, bufState, ((NQ_COUNT)lenToRead))

            bytesRead = syRecvSocketWithTimeout(pSock->socket, ioBuf, (NQ_COUNT)lenToRead, UD_NQ_TCPSOCKETRECVTIMEOUT);
#ifdef UD_NQ_USEIOVECS
            IOBUF_IOVECBUF_ADVANCE_SAVE(ioBuf, x, bufState)
#endif /* UD_NQ_USEIOVECS */

            if (bytesRead == 0 || bytesRead == NQ_FAIL)
            {
                if (bytesRead == 0)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "0 bytes read (message body)");
                    goto Exit;
                }
                LOGERR(CM_TRC_LEVEL_ERROR, "Error reading message body, code: %d", syGetLastError());
                goto Exit;
            }
            descr->remaining -= (NQ_COUNT)bytesRead;
        }
    }
    result = NQ_SUCCESS;

Exit:
/*  descr->remaining = 0;*/ /* this packet is finished. */
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result: %d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Read from a stream into a preallocated buffer
 *--------------------------------------------------------------------
 * PARAMS:  IN receive descriptor
 *          OUT buffer for incoming user data
 *          IN maximum number of bytes to read
 *
 * RETURNS: Number of bytes read into the user buffer or NQ_FAIL
 *
 * NOTES:   
 *====================================================================
 */

NQ_INT
nsRecvIntoBuffer(
    NSRecvDescr  * descr,
    NQ_IOBuf buf,
    NQ_COUNT len
    )
{
    SocketSlot* pSock;                      /* actual pointer to a socket slot */
    NQ_INT bytesRead = 0;                   /* number of bytes received */
    NQ_INT totalBytesRead;                  /* number of bytes received */
    NQ_INT dataToRead;
    NQ_INT result = NQ_FAIL;
#ifdef UD_NQ_USEIOVECS
    NQ_IOBufState bufState;
    NQ_IOBufOrigin bufOrig;
#endif /* UD_NQ_USEIOVECS */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "descr: %p, buf: %p, len: %d", descr, buf, len);

    dataToRead = len < descr->remaining ? (NQ_INT)len : (NQ_INT)descr->remaining;
    pSock = (SocketSlot*)descr->socket;

    if (pSock == NULL || !syIsValidSocket(pSock->socket))
    {
        sySetLastError(CM_NBERR_INVALIDPARAMETER);
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal socket descriptor");
        goto Exit2;
    }

    IOBUF_SETORIGIN(buf, bufOrig, bufState)
    IOBUF_PREPARELENGTH(buf, bufOrig, bufState, ((NQ_COUNT)dataToRead))

    if (!pSock->isNetBios) /* socket is not NetBIOS */
    {
        LOGMSG(CM_TRC_LEVEL_MESS_SOME, "Not a NetBIOS socket");

        bytesRead = syRecvSocket(
            pSock->socket,
            buf,
            (NQ_UINT)dataToRead
            );

        if (bytesRead > 0)
        {
            descr->remaining -= (NQ_COUNT)bytesRead;
        }

        result = bytesRead;
        goto Exit;
    }

    if (pSock->isAccepted)
        syMutexTake(&pSock->guard);

    totalBytesRead = 0;
    while (len > 0 && descr->remaining > 0)
    {
        NQ_COUNT numBytes;
        numBytes = (NQ_COUNT)bytesRead;

        IOBUF_IOVECBUF_ADVANCE_SAVE(buf, numBytes, bufState)

        bytesRead = syRecvSocketWithTimeout(pSock->socket, buf, (NQ_UINT)dataToRead, UD_NQ_TCPSOCKETRECVTIMEOUT);
        if (bytesRead == 0 || bytesRead == NQ_FAIL)
        {
            if (pSock->isAccepted)
                syMutexGive(&pSock->guard);
            if (bytesRead == 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "0 bytes read (message body)");
                goto Exit;
            }
            LOGERR(CM_TRC_LEVEL_ERROR, "Error reading message body, code: %d", syGetLastError());
            goto Exit;
        }

        totalBytesRead += bytesRead;
        len -= (NQ_COUNT)bytesRead;
        dataToRead -= bytesRead;
        descr->remaining -= (NQ_COUNT)bytesRead;
    }

    if (pSock->isAccepted)
        syMutexGive(&pSock->guard);

    result = (NQ_INT)totalBytesRead;

Exit:
    IOBUF_IOVECBUF_RESTORE(buf, bufState, bufOrig);

Exit2:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result: %d", result);
    return result;
}

