/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Socket pool management
 *--------------------------------------------------------------------
 * MODULE        : Network
 * DEPENDENCIES  :
 ********************************************************************/

#include "nssocket.h"
#include "cmrepository.h"


/*
 NS module keeps trek of sockets created by means of nsSocket. For this
 reason we use a pool of socket descriptors.

 This pool is organized an an array of descriptors. It is controlled by another
 array of "free descriptor" pointers. This second array is a cyclical array of
 pointers to free descriptors. Two indexes (of the 1-st and of the last cell) "roll around"
 in this array.

 Access to the pool is protected by a mutex.
 Slot index is reserved for a future use. SInce it is referenced only in initSocketPool()
 this does not affect the performance.
*/

/*
    Static data
    -----------
 */


#define FIRST_PORT  6000                            /* bottom of the port pool for sockets */

typedef struct
{
    CMRepository socketSlots;
    NQ_BOOL isUp;
}
StaticData;

static NQ_BOOL isModuleInitialized = FALSE;
static NQ_COUNT moduleInitCount = 0;
#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

static void initSlot(CMItem * pItem)
{
    SocketSlot * pSlot = (SocketSlot *)pItem;

    pSlot->socket = syInvalidSocket();    /* no socket yet */
}

static  void disposeSlot(CMItem * pItem)
{
    SocketSlot * pSlot = (SocketSlot *)pItem;

    if (syIsValidSocket(pSlot->socket))
        syCloseSocket(pSlot->socket);
}

/*
 *====================================================================
 * PURPOSE: Initialize the pool
 *--------------------------------------------------------------------
 * PARAMS:  component: server, nd daemon, client
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *====================================================================
 */

NQ_STATUS
nsInitSocketPool(
    NQ_UINT32 component
    )
{
#ifdef CM_NQ_STORAGE
    NQ_COUNT basicServerPorts = 0;
    NQ_UINT32 numToAlloc = 35;
#else
    NQ_UINT32 numToAlloc = UD_NS_NUMSOCKETS;
#endif /* CM_NQ_STORAGE */
    NQ_STATUS result = NQ_SUCCESS;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "component:0x%x", component);

    if (isModuleInitialized)
    {
        ++moduleInitCount;
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate socket pool");
        result = NQ_FAIL;
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

#ifdef CM_NQ_STORAGE
    if (component != NQ_NDDAEMON)
        numToAlloc = 1 + 1 + (UD_CS_NUMBER_CONNECTIONS * 2);
    /* client, browse */
    basicServerPorts = 3 + 1 + 1;
    /* dcerpc - 3 (1037, 135, pipe)
     * port 139 - 1
     * port 445 - 1
     * */
#ifdef UD_NQ_USETRANSPORTIPV6
    basicServerPorts *= 2;
#endif /* UD_NQ_USETRANSPORTIPV6 */
    numToAlloc += basicServerPorts;
#ifdef UD_CS_USE_EXTERNAL_SMB_TRANSPORT
    numToAlloc = 10;
#endif
#endif /* CM_NQ_STORAGE */
    cmRepositoryInit(&staticData->socketSlots, 0, initSlot, disposeSlot, "socketSlots");
    cmRepositoryItemPoolAlloc(&staticData->socketSlots, numToAlloc, sizeof(SocketSlot));
    staticData->isUp = TRUE;
    isModuleInitialized = TRUE;
    ++moduleInitCount;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Release the pool
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *====================================================================
 */

void
nsExitSocketPool(
    void
    )
{
    if (isModuleInitialized && (--moduleInitCount == 0))
    {
        if (staticData->isUp)
        {
            staticData->isUp = FALSE;
            cmRepositoryShutdown(&staticData->socketSlots);
        }
#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
            cmMemoryFreeShutdown(staticData);
        staticData = NULL;
#endif /* SY_FORCEALLOCATION */
        isModuleInitialized = FALSE;
    }
}

/*
 *====================================================================
 * PURPOSE: Get underlying socket's handle
 *--------------------------------------------------------------------
 * PARAMS:  NS level socket handle
 *
 * RETURNS: SY level socket handle
 *
 * NOTES:
 *====================================================================
 */

SYSocketHandle nsGetSySocket(NSSocketHandle socket)
{
    SocketSlot* slot = (SocketSlot*)socket;

    return slot->socket;
}

/*
 *====================================================================
 * PURPOSE: Get a socket descriptor from the pool
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: buffer pointer or NULL on failure
 *
 * NOTES:   Takes a slot from the 1st free pointer.
 *          On pool overflow the operation fails
 *====================================================================
 */

SocketSlot * getSocketSlot(
    void
    )
{
    SocketSlot * res = NULL;  /* pointer to return on exit */

    res = (SocketSlot *)cmRepositoryGetNewItem(&staticData->socketSlots);
    return res;
}

/*
 *====================================================================
 * PURPOSE: Return a buffer to the pool
 *--------------------------------------------------------------------
 * PARAMS:  Pointer to the buffer to return
 *
 * RETURNS: none
 *
 * NOTES:   fills the next free slot ptr with a reference to the releases slot
 *====================================================================
 */

void putSocketSlot(SocketSlot * slot)
{
    if (syIsValidSocket(slot->socket))
    {
        if (syIsSocketAlive(slot->socket) && syShutdownSocket(slot->socket) != NQ_SUCCESS)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to shutdown a connection");
        }

        if (syCloseSocket(slot->socket) != NQ_SUCCESS)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unable to close a socket");
        }

        slot->socket = syInvalidSocket();
        cmRepositoryReleaseItem(&staticData->socketSlots, (CMItem *)slot);
    }
}
