/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Parsing delimited string
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmstrparse.h"

/* -- Static data -- */

static CMList list;
static NQ_BOOL isModuleInitialized = FALSE;

/* functions */

/*
 *====================================================================
 * PURPOSE: Starts this module
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: TRUE on success, FALSE on failure
 *
 *====================================================================
 */
NQ_BOOL cmSpStart(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
       goto Exit;
    }

    cmListStart(&list);
    isModuleInitialized = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: Shutdown this module
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 *====================================================================
 */
void cmSpShutdown(void)
{
    CMIterator  iterator;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        cmListIteratorStart(&list, &iterator);
        while (cmListIteratorHasNext(&iterator))
        {
            CMItem * pItem;

            pItem = cmListIteratorNext(&iterator);
            cmListItemRemoveAndDispose(pItem);
        }

        cmListIteratorTerminate(&iterator);
        cmListShutdown(&list);
        isModuleInitialized = FALSE;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 *====================================================================
 * PURPOSE: start parsing delimited string - allocate memory
 *--------------------------------------------------------------------
 * PARAMS:  IN source - list of strings delimited by 'delimiter'
 *          IN delimiter - the delimiter which in use
 *          IN isUnicode - whether the strings are UNICODE or ANSI
 *          OUT firstString - pointer to first string from the list
 *          IN isExtractString - whether to find the first string and
 *                               return it by 'firstString'.
 *
 * RETURNS: parse handle which can be used to extract the strings one by one
 *          from the list
 *
 * NOTES:  Multiple parsing can occur concurrently because we are
 *         using handles.
 *====================================================================
 */
NQ_HANDLE cmSpStartParsing(void * source, void * delimiter, NQ_BOOL isUnicode, void ** firstString, NQ_BOOL isExtractString)
{
    CMParsingdata *parseData = NULL;
    NQ_HANDLE result = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "source:%p, delimiter:%p, isUnicode:%s", source, delimiter, isUnicode ? "TRUE" : "FASLE");

    if ((NULL == delimiter) || (NULL == source))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (TRUE == isUnicode)
    {
        NQ_WCHAR  *sorceW;
        NQ_WCHAR  *delimiterW;

        source = (void *)cmAllignTwo(source);
        sorceW = (NQ_WCHAR *)source;
        delimiter = (void *)cmAllignTwo(delimiter);
        delimiterW = (NQ_WCHAR *)delimiter;

        if (0 == cmWStrlen(sorceW))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "empty string");
            sySetLastError(NQ_ERR_BADPARAM);
            goto Exit;
        }

        if (cmWChar('\0') == *delimiterW)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal delimeter");
            sySetLastError(NQ_ERR_BADPARAM);
            goto Exit;
        }
    }
    else
    {
        NQ_CHAR *sorceA = (NQ_CHAR *)source;
        NQ_CHAR *delimiterA = (NQ_CHAR *)delimiter;

        if (0 == syStrlen(sorceA))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "empty string");
            sySetLastError(NQ_ERR_BADPARAM);
            goto Exit;
        }

        if ('\0' == *delimiterA)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal delimeter");
            sySetLastError(NQ_ERR_BADPARAM);
            goto Exit;
        }
    }

    parseData = (CMParsingdata *)cmListItemCreateAndAdd(&list, sizeof(CMParsingdata), NULL, NULL, CM_LISTITEM_NOLOCK , FALSE);
    if (NULL == parseData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    parseData->source = source;
    parseData->nextEntry = source;
    parseData->delimiter = delimiter;
    parseData->isUnicode = isUnicode;
    parseData->isParseFinished = FALSE;
    result = (NQ_HANDLE)parseData;

    if (TRUE == isExtractString)
    {
        if (NULL == firstString)
        {
            goto Exit;
        }
        else
        {
            *firstString = cmSpGetNextString(result);
            if (NULL == *firstString)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "couldn't find first string");
                goto Exit;
            }
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "handle:%p", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: get next string form the list
 *--------------------------------------------------------------------
 * PARAMS:  IN parseHandle - parse handle
 *
 * RETURNS: the next string from the list
 *
 * NOTES:
 *====================================================================
 */
void * cmSpGetNextString(NQ_HANDLE parseHandle)
{
    CMParsingdata *parseData = (CMParsingdata *)parseHandle;
    void      *result = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "parseHandle:%p", parseHandle);

    if (NULL == parseHandle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "NULL Handle");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (TRUE == parseData->isParseFinished)
    {
        goto Exit;
    }

    if (TRUE == parseData->isUnicode)
    {
        NQ_WCHAR *sorceW = (NQ_WCHAR *)parseData->source;
        NQ_WCHAR *delimiterW = (NQ_WCHAR *)parseData->delimiter;
        NQ_WCHAR * pointerToNextDelimeter;

        /* if it is first string do not write back the delimeter */
        if (parseData->source == parseData->nextEntry)
        {
            pointerToNextDelimeter = cmWStrchr(sorceW, *delimiterW);
            if (NULL == pointerToNextDelimeter)
            {
                result = (NQ_HANDLE)sorceW;
                parseData->isParseFinished = TRUE;
            }
            else
            {
                *pointerToNextDelimeter = cmWChar('\0');
                result = (NQ_HANDLE)sorceW;
                pointerToNextDelimeter++;
                parseData->nextEntry = (void *)pointerToNextDelimeter;
            }
        }
        else /* not the first string */
        {
            NQ_WCHAR *nextEnryW = (NQ_WCHAR *)parseData->nextEntry;

            /* write back previous delimeter */
            nextEnryW--;
            *nextEnryW = *delimiterW;
            nextEnryW++;

            if (0 == cmWStrlen(nextEnryW))
            {
                parseData->isParseFinished = TRUE;
            }
            else
            {
                pointerToNextDelimeter = cmWStrchr(nextEnryW, *delimiterW);
                if (NULL == pointerToNextDelimeter)
                {
                    result = (NQ_HANDLE)nextEnryW;
                    parseData->isParseFinished = TRUE;
                }
                else
                {
                    *pointerToNextDelimeter = cmWChar('\0');
                    result = (NQ_HANDLE)nextEnryW;
                    pointerToNextDelimeter++;
                    parseData->nextEntry = (void *)pointerToNextDelimeter;
                }
            }
        }
    }
    else
    {
        NQ_CHAR *sorceA = (NQ_CHAR *)parseData->source;
        NQ_CHAR *delimiterA = (NQ_CHAR *)parseData->delimiter;
        NQ_CHAR * pointerToNextDelimeter;

        /* if it is first string do not write back the delimeter */
        if (parseData->source == parseData->nextEntry)
        {
            pointerToNextDelimeter = syStrchr(sorceA, *delimiterA);
            if (NULL == pointerToNextDelimeter)
            {
                result = (NQ_HANDLE)sorceA;
                parseData->isParseFinished = TRUE;
            }
            else
            {
                *pointerToNextDelimeter = '\0';
                result = (NQ_HANDLE)sorceA;
                pointerToNextDelimeter++;
                parseData->nextEntry = (void *)pointerToNextDelimeter;
            }
        }
        else /* not the first string */
        {
            NQ_CHAR *nextEnryA = (NQ_CHAR *)parseData->nextEntry;

            /* write back previous delimeter */
            nextEnryA--;
            *nextEnryA = *delimiterA;
            nextEnryA++;

            if (0 == syStrlen(nextEnryA))
            {
                parseData->isParseFinished = TRUE;
            }
            else
            {
                pointerToNextDelimeter = syStrchr(nextEnryA, *delimiterA);
                if (NULL == pointerToNextDelimeter)
                {
                    result = (NQ_HANDLE)nextEnryA;
                    parseData->isParseFinished = TRUE;
                }
                else
                {
                    *pointerToNextDelimeter = '\0';
                    result = (NQ_HANDLE)nextEnryA;
                    pointerToNextDelimeter++;
                    parseData->nextEntry = (void *)pointerToNextDelimeter;
                }
            }
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "result:%p", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: terminate parsing - release the memory that was allocated
 *--------------------------------------------------------------------
 * PARAMS:  IN parseHandle - parse handle
 *
 * RETURNS: None
 *
 * NOTES: This function should be called when the parsing of a delimited
 *        string was finished. After the parsing is finished the user can
 *        be sure that the source string has returned to its original form.
 *====================================================================
 */
void cmSpTerminateParsing(NQ_HANDLE parseHandle)
{
    CMParsingdata *parseData = (CMParsingdata *)parseHandle;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "parseHandle:%p", parseHandle);

    if (NULL == parseHandle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "NULL Handle");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* if the parsing stopped in the middle - write back the delimiter */
    if (FALSE == parseData->isParseFinished)
    {
        if (TRUE == parseData->isUnicode)
        {
            NQ_WCHAR *delimiterW = (NQ_WCHAR *)parseData->delimiter;
            NQ_WCHAR *nextEnryW = (NQ_WCHAR *)parseData->nextEntry;

            /* write back previous delimeter */
            nextEnryW--;
            *nextEnryW = *delimiterW;
        }
        else
        {
            NQ_CHAR *delimiterA = (NQ_CHAR *)parseData->delimiter;
            NQ_CHAR *nextEnryA = (NQ_CHAR *)parseData->nextEntry;

            /* write back previous delimeter */
            nextEnryA--;
            *nextEnryA = *delimiterA;
        }
    }

    cmListItemRemoveAndDispose((CMItem *)parseHandle);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
    return;
}

