/*************************************************************************
* Copyright (c) 2021 by Visuality Systems, Ltd.
*
*                     All Rights Reserved
*
* This item is the property of Visuality Systems, Ltd., and contains
* confidential, proprietary, and trade-secret information. It may not
* be transferred from the custody or control of Visuality Systems, Ltd.,
* except as expressly authorized in writing by an officer of Visuality
* Systems, Ltd. Neither this item nor the information it contains may
* be used, transferred, reproduced, published, or disclosed, in whole
* or in part, and directly or indirectly, except as expressly authorized
* by an officer of Visuality Systems, Ltd., pursuant to written agreement.
*
*************************************************************************
* FILE NAME     : $Workfile:$
* ID            : $Header:$
* REVISION      : $Revision:$
*--------------------------------------------------------------------
* DESCRIPTION   : Common storage operations
*--------------------------------------------------------------------
* MODULE        : Common
* DEPENDENCIES  :
*************************************************************************/

#include "cmstorage.h"

#define STORAGE_ITEM_TIMEOUT_SEC 3

static NQ_BOOL storageListEnlarge(CMStorageData * pData)
{
    CMStorageList * pList = NULL;
    NQ_BYTE * pTemp = NULL;
    NQ_UINT32 i = 0;
    NQ_UINT32 newNumOfLists = 0;
    NQ_BOOL result = FALSE;

    pTemp = (NQ_BYTE *)cmMemoryAllocate((NQ_UINT)(pData->numOfLists * sizeof(CMStorageList)));
    if (NULL == pTemp)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memoroy");
        goto Exit;
    }

    syMemcpy(pTemp, pData->list, pData->numOfLists * sizeof(CMStorageList));
    cmMemoryFree(pData->list);

    newNumOfLists = 2 * pData->numOfLists;
    pData->list = (CMStorageList *)cmMemoryAllocate((NQ_UINT)(newNumOfLists * sizeof(CMStorageList)));
    if (NULL == pData->list)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memoroy");
        goto Exit;
    }

    syMemcpy(pData->list, pTemp, pData->numOfLists * sizeof(CMStorageList));

    for (i = pData->numOfLists; i < newNumOfLists; i++)
    {
        NQ_UINT32   j = 0;
        CMStorageItem   *   pItem;

        pList = &pData->list[i];
        pList->items = cmMemoryAllocate((NQ_UINT)(pData->numOfListItems * pData->itemSize));
        if (NULL == pList->items)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            goto Exit;
        }

        pItem = (CMStorageItem *)pList->items;
        pList->freeItems = pItem;
        syMutexCreate(&pList->guard);
        for (j = 0; j < pData->numOfListItems; j++)
        {
            pItem->itemIndex.high = i;
            pItem->itemIndex.low = j + 1;
            pItem->isUsed = FALSE;
            pItem->isFindable = FALSE;
            if (j < pData->numOfListItems - 1)
                pItem->next = (CMStorageItem *)((NQ_BYTE *)pItem + pData->itemSize);
            else
            {
                pItem->next = NULL;
                pList->lastFreeItem = pItem;
            }

            if (NULL != pData->itemInitCallBack)
            {
                if (FALSE == pData->itemInitCallBack(pItem))
                {
                    result = FALSE;
                    goto Exit;
                }
            }

            pItem = (CMStorageItem *)((NQ_BYTE *)pItem + pData->itemSize);
        }
        pList->index = i;
    }
#ifdef REPOSITORY_SIZE_TUNING
    pData->finalNumLists = pData->numOfLists;
    syPrintf("Enlarge storage %s, from %d to %d lists", pData->listName, pData->numOfLists, newNumOfLists);
#endif


    pData->numOfLists = newNumOfLists;
    result = TRUE;

Exit:
    cmMemoryFree(pTemp);
    return result;
}


NQ_BOOL cmStorageListInit(CMStorageData * pData, NQ_UINT itemSize, NQ_UINT numOfLists, NQ_UINT numOfListItems,
                          NQ_BOOL reserveFirstList, NQ_CHAR* listName, cmStorageItemInitCallback itemInitCallback, cmStorageItemDisposeCallback itemDisposeCallBack)
{
    NQ_UINT32 i = 0;
    CMStorageList * pList = NULL;
    NQ_BOOL result = FALSE;

    syMutexCreate(&pData->guard);
    pData->extend = cmGetDynamicAllocation();
    pData->reserveFirstList = reserveFirstList;
    pData->itemSize = itemSize;
    pData->numOfUsedItems = 0;
    if (!cmThreadCondSet(&pData->cond))
        goto Exit;
    pData->waitingOnCond = FALSE;
    pData->list = (CMStorageList *)cmMemoryAllocateStartup((NQ_UINT)(numOfLists * sizeof(CMStorageList)));
    if (pData->list == NULL)
        goto Exit;

    pData->itemInitCallBack = itemInitCallback;
    pData->itemDisposeCallBack = itemDisposeCallBack;

#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(NQDEBUG) && !defined(UD_NQ_DYNAMICALLOCATION))
    pData->listName = (NQ_CHAR *)cmMemoryAllocateStartup((NQ_UINT)(syStrlen(listName) + 1));
    syStrcpy(pData->listName, listName);
#endif /* #if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(NQDEBUG) && !defined(UD_NQ_DYNAMICALLOCATION)) */

    pData->numOfLists = numOfLists;
#ifdef REPOSITORY_SIZE_TUNING
    pData->initialNumLists = pData->finalNumLists = numOfLists;
#endif
#if defined(REPOSITORY_SIZE_TUNING) || defined(REPOSITORY_DEBUG)
    pData->numFailedItems = 0;
#endif
    for (i = 0; i < numOfLists; i++)
    {
        CMStorageItem   *pItem;
        NQ_UINT32       j = 0;

        pList = &pData->list[i];
        pList->items = cmMemoryAllocateStartup(numOfListItems * pData->itemSize);
        if (pList->items == NULL)
            goto Exit;

        pItem = (CMStorageItem *)pList->items;
        pList->freeItems = pItem;
        pList->usedItems = 0;
        syMutexCreate(&pList->guard);
        pData->numOfListItems = numOfListItems;

        for (j = 0; j < numOfListItems; j++)
        {
            pItem->itemIndex.high = i;
            pItem->itemIndex.low = j + 1;
            pItem->isUsed = FALSE;
            pItem->isFindable = FALSE;
            if (j < numOfListItems - 1)
                pItem->next = (CMStorageItem *)((NQ_BYTE *)pItem + pData->itemSize);
            else
            {
                pItem->next = NULL;
                pList->lastFreeItem = pItem;
            }

            if (NULL != pData->itemInitCallBack)
            {
                if (FALSE == pData->itemInitCallBack(pItem))
                {
                    result = FALSE;
                    goto Exit;
                }
            }

            pItem = (CMStorageItem *)((NQ_BYTE *)pItem + pData->itemSize);
        }
        pList->index = i;
    }

    result = TRUE;

Exit:
    return result;
}

CMStorageItem * cmStorageItemGetNew(CMStorageData * pData, NQ_INT condition)
{
    CMStorageItem * pItem = NULL;
    CMStorageItem * pResult = NULL;
    CMStorageList * pList = NULL;
    NQ_BOOL         limit = condition & STORAGE_IS_TREE;
    NQ_BOOL         hasFreeItems = FALSE;
    NQ_UINT         firstList = 0;
    NQ_UINT         i;

    syMutexTake(&pData->guard);

    /* if can't extend the list, we will wait for an item to return for STORAGE_ITEM_TIMEOUT_SEC */
    if (!pData->extend)
    {
        if (pData->reserveFirstList && (limit ? condition - STORAGE_IS_TREE == STORAGE_IS_LIMITED : condition == STORAGE_IS_LIMITED))
        {
            hasFreeItems = pData->list[0].freeItems != NULL;
        }
        else
        {
            for (i = firstList; i <  pData->numOfLists; i++)
            {
                if (pData->list[i].freeItems != NULL)
                    hasFreeItems = TRUE;
            }
        }
        if (!hasFreeItems)
        {
            NQ_BOOL res;

#if defined(REPOSITORY_SIZE_TUNING) || defined(REPOSITORY_DEBUG)
            if (pData->numFailedItems % REPO_MESSAGE_THROTTLING == 1)
            {
                syPrintf("Item allocation failed %d times. list: %s\n", pData->numFailedItems, pData->listName);
            }
            ++pData->numFailedItems;
#endif /* defined(REPOSITORY_SIZE_TUNING) || defined(REPOSITORY_DEBUG) */
            pData->waitingOnCond = TRUE;
            syMutexGive(&pData->guard);

            res = cmThreadCondWait(&pData->cond , STORAGE_ITEM_TIMEOUT_SEC);
            pData->waitingOnCond = FALSE;
            syMutexTake(&pData->guard);
            if (!res)
            {
                goto Error;
            }
        }
    }

    if (pData->reserveFirstList)
    {
        if (limit ? condition - STORAGE_IS_TREE == STORAGE_IS_LIMITED : condition == STORAGE_IS_LIMITED)
        {
            pList = &pData->list[0];

            if (pList->freeItems == NULL|| pList->usedItems == 0xFFFD)
            {
                goto Error;
            }

            goto Exit;
        }
        firstList = 1;
    }

    for (i = firstList; i < pData->numOfLists; i++)
    {
        pList = &pData->list[i];
        if (pList->freeItems != NULL)
        {
            if (limit && i >= 0xFFFD)
            {
                goto Error;
            }
            goto Exit;
        }
    }

    if (limit && pData->numOfLists >= 0xFFFD)
    {
        goto Error;
    }

    i = pData->numOfLists;
    if (pData->extend && storageListEnlarge(pData))
    {
        pList = &pData->list[i];
        if (pList->freeItems != NULL)
        {
            goto Exit;
        }
    }
    goto Error;

Exit:
    pItem = (CMStorageItem *)pList->freeItems;
    syMutexTake(&pList->guard);
    if (pList->freeItems != NULL)
    {
        pList->freeItems = pItem->next;
        if (NULL == pList->freeItems)
            pList->lastFreeItem = NULL;
    }
    pList->usedItems++;
    if (NULL != pItem)
    {
        pItem->isUsed = TRUE;
        pItem->isFindable = TRUE;
    }
    syMutexGive(&pList->guard);
    pData->numOfUsedItems++;
    pResult = pItem;

Error:
    syMutexGive(&pData->guard);
    return pResult;
}

CMStorageItem * cmStorageItemFind(CMStorageData * pData, NQ_UINT row, NQ_UINT cell, NQ_BOOL isFindAll)
{
    CMStorageList * pList = NULL;
    CMStorageItem * pItem = NULL;

    cell--;
    syMutexTake(&pData->guard);

    if (row < pData->numOfLists && cell < pData->numOfListItems)
    {
        pList = &pData->list[row];
        pItem = (CMStorageItem *)((NQ_BYTE *)pList->items + (pData->itemSize * cell));
        pItem = ((pItem->isUsed && (isFindAll || pItem->isFindable)) ? pItem : NULL);
    }

    syMutexGive(&pData->guard);
    return pItem;

}

void cmStorageItemRemove(CMStorageData * pData, NQ_UINT row, NQ_UINT cell)
{
    CMStorageList * pList = NULL;
    CMStorageItem * pItem = NULL;

    cell--;
    if (row < pData->numOfLists && cell < pData->numOfListItems)
    {
        syMutexTake(&pData->guard);
        pList = &pData->list[row];
        pItem = (CMStorageItem *)((NQ_BYTE *)pList->items + (pData->itemSize * cell));
        if (pItem->isUsed)
        {
            CMStorageItem * pTmpItem;

            pItem->isUsed = FALSE;
            pItem->isFindable = FALSE;
            pTmpItem = (CMStorageItem *)pList->lastFreeItem;
            syMutexTake(&pList->guard);
            if (NULL == pTmpItem)
            {
                pList->freeItems = pList->lastFreeItem = pItem;
            }
            else
                pList->lastFreeItem = pTmpItem->next = pItem;
            pItem->next = NULL;

            if (pList->usedItems != 0)
                pList->usedItems--;

            syMutexGive(&pList->guard);
            pData->numOfUsedItems--;

            if (pData->waitingOnCond)
            {
                cmThreadCondSignal(&pData->cond);
            }
        }
        syMutexGive(&pData->guard);
    }
}

void cmStorageListShutdown(CMStorageData * pData)
{
    NQ_UINT i = 0;

    for (i = 0; i < pData->numOfLists; i++)
    {
        CMStorageItem *pItem;
        NQ_COUNT j;

        syMutexDelete(&pData->list[i].guard);

        pItem = (CMStorageItem *)pData->list[i].items;

        /* trigger item dispose callback if exists */
        for (j = 0; j < pData->numOfListItems ; j++)
        {
            if (pData->itemDisposeCallBack)
            {
                pData->itemDisposeCallBack(pItem);
            }

            pItem = (CMStorageItem *)((NQ_BYTE *)pItem + pData->itemSize);
        }

        cmMemoryFreeShutdown(pData->list[i].items);
        pData->list[i].items = NULL;
    }

    cmMemoryFreeShutdown(pData->list);
    syMutexDelete(&pData->guard);
    cmThreadCondRelease(&pData->cond);
    pData->list = NULL;
#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(NQDEBUG) && !defined(UD_NQ_DYNAMICALLOCATION))
    cmMemoryFreeShutdown(pData->listName);
#endif /* defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(NQDEBUG) && !defined(UD_NQ_DYNAMICALLOCATION)) */
#ifdef REPOSITORY_SIZE_TUNING

    syPrintf("storage list: %s", pData->listName);
    if(pData->finalNumLists > pData->initialNumLists)
    {
        syPrintf("    initial num lists: %d final num lists: %d\n", pData->initialNumLists, pData->finalNumLists);
    }
    else if(pData->numFailedItems > 0)
    {
        syPrintf("!!!!!!!!!!!!!!!  failed: %d items.    list size: %d, numLists:%d ", pData->numFailedItems, pData->numOfListItems, pData->numOfLists);
    }
    else
    {
        syPrintf("  good.\n");
    }
#endif /* REPOSITORY_SIZE_TUNING */

    return;
}

void cmStorageIteratorStart(CMStorageData * pData, CMStorageIterator * Itr)
{
    syMutexTake(&Itr->itrGuard);
    syMutexTake(&pData->guard);
    Itr->pData = pData;
    Itr->listNumber = 0;
    Itr->itemNumber = 0;
}

CMStorageItem * cmStorageIteratorNext(CMStorageIterator * Itr)
{
    CMStorageItem * pItem = NULL;

    if (Itr->pData != NULL)
    {
        if (Itr->listNumber < Itr->pData->numOfLists)
        {
            if (Itr->itemNumber < Itr->pData->numOfListItems)
            {
                pItem = (CMStorageItem *)((NQ_BYTE *)Itr->pData->list[Itr->listNumber].items + Itr->pData->itemSize * Itr->itemNumber);
                Itr->itemNumber++;
                if (Itr->itemNumber == Itr->pData->numOfListItems)
                {
                    Itr->listNumber++;
                    Itr->itemNumber = 0;
                }
                goto Exit;
            }
        }
    }

Exit:
    return pItem;
}

NQ_BOOL cmStorageIteratorHasNext(CMStorageIterator * Itr)
{
    NQ_BOOL result = FALSE;

    if (Itr->pData != NULL && Itr->pData->numOfUsedItems > 0)
    {
        if (Itr->listNumber < Itr->pData->numOfLists && Itr->itemNumber <= Itr->pData->numOfListItems)
        {
            CMStorageList   *   pList;

            pList = &Itr->pData->list[Itr->listNumber];
            if (pList->usedItems != 0)
            {
                while (Itr->itemNumber < Itr->pData->numOfListItems)
                {
                    CMStorageItem * pItem = NULL;

                    pItem = (CMStorageItem *)((NQ_BYTE *)Itr->pData->list[Itr->listNumber].items + Itr->pData->itemSize * Itr->itemNumber);
                    if (pItem->isUsed && pItem->isFindable)
                    {
                        result = TRUE;
                        goto Exit;
                    }
                    Itr->itemNumber++;
                }
                Itr->itemNumber = 0;
                Itr->listNumber++;
                pList = &Itr->pData->list[Itr->listNumber];
            }

            while (Itr->listNumber < Itr->pData->numOfLists && pList->usedItems == 0)
            {
                Itr->itemNumber = 0;
                Itr->listNumber++;
                pList = &Itr->pData->list[Itr->listNumber];
            }

            if (Itr->listNumber < Itr->pData->numOfLists && pList->usedItems != 0)
            {
                while (Itr->itemNumber < Itr->pData->numOfListItems)
                {
                    CMStorageItem * pItem = NULL;

                    pItem = (CMStorageItem *)((NQ_BYTE *)Itr->pData->list[Itr->listNumber].items + Itr->pData->itemSize * Itr->itemNumber);
                    if (pItem->isUsed && pItem->isFindable)
                    {
                        result = TRUE;
                        goto Exit;
                    }
                    Itr->itemNumber++;
                }
            }
        }
    }

Exit:
    return result;
}

void cmStorageIteratorTerminate(CMStorageIterator * Itr)
{
    if (Itr->pData != NULL)
    {
        syMutexGive(&Itr->pData->guard);
        Itr->pData = NULL;
        Itr->listNumber = 0;
        Itr->itemNumber = 0;
        syMutexGive(&Itr->itrGuard);
    }
}

NQ_BOOL cmStorageHashListStart(CMStorageHashList * pHList, CMStorageHashFunction doHash)
{
    NQ_UINT32   i;

    for (i = 0; i < HASH_N; i++)
    {
        cmListStart(&pHList->list[i]);
    }

    pHList->doHash = doHash;
    pHList->isUsed = TRUE;

    return TRUE;
}

CMList * cmStorageHashFindList(CMStorageHashList * pHList, const void * value)
{
    NQ_UINT32 hashedV = 0;
    CMList * pResult = NULL;

    if (pHList->isUsed)
    {
        hashedV = pHList->doHash(value);
        pResult = &pHList->list[hashedV];
    }

    return pResult;
}

void cmStorageHashListShutdown(CMStorageHashList * pHList)
{
    NQ_UINT32   i;
    CMIterator iterator;

    if (pHList->isUsed)
    {
        for (i = 0; i < HASH_N; i++)
        {
            cmListIteratorStart(&pHList->list[i], &iterator);
            while (cmListIteratorHasNext(&iterator))
            {
                CMItem  *   pSItem = NULL;

                pSItem = cmListIteratorNext(&iterator);
                cmMemoryFree(pSItem->name);
                pSItem->name = NULL;
                cmListItemRemove(pSItem);
                syMutexDelete(pSItem->guard);
                cmMemoryFree(pSItem->guard);

            }
            cmListIteratorTerminate(&iterator);
            syMutexDelete(&pHList->list[i].guard);
        }
        pHList->doHash = NULL;
        pHList->isUsed = FALSE;
    }
}

void cmStorageHashListDispose(CMStorageHashList * pHList)
{
    NQ_UINT32   i;
    CMIterator iterator;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pHList:%p", pHList);

    if (pHList->isUsed)
    {
        for (i = 0; i < HASH_N; i++)
        {
            cmListIteratorStart(&pHList->list[i], &iterator);
            while (cmListIteratorHasNext(&iterator))
            {
                CMItem  *pSItem = cmListIteratorNext(&iterator);

                cmListItemCheck(pSItem);
                cmListItemRemoveAndDispose(pSItem);
            }
            cmListIteratorTerminate(&iterator);
            syMutexDelete(&pHList->list[i].guard);
        }
        pHList->doHash = NULL;
        pHList->isUsed = FALSE;
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}


NQ_UINT32 cmStorageNameHash(const void * name)
{
    const NQ_WCHAR  *   pName = (const NQ_WCHAR *)name;
    const NQ_UINT32 prime = 0x01000193; /*   16777619*/
    const NQ_UINT32 seed = 0x811C9DC5; /* 2166136261*/
    NQ_UINT32 hash = seed;
    NQ_UINT32 numBytes;

#ifdef NQDEBUG
    syAssert(pName);
#endif /* NQDEBUG */
    numBytes = cmWStrlen(pName);

    while (numBytes--)
    {
        hash = pName[numBytes] ^ hash * prime;
    }

    return hash % HASH_N;
    /*return (NQ_UINT32)((pName[0] + pName[cmWStrlen(pName)] + pName[cmWStrlen(pName)/2 -1]) % HASH_N); - old hash*/
}
