/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : API for user-defined features
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmsdescr.h"
#include "cmrepository.h"

/*
 * Local definitions, functions, data
 * ----------------------------------
 */

/* Definition of an "empty" domain SID */
#define cmSdClearDomainSid(_pSid)   (_pSid)->revision = -1
#define cmSdIsEmptyDomainSid(_pSid) (_pSid)->revision == -1


#define cmSdTokenIsValid(_token)    ((_token) && (_token->rids) && (_token->numRids > 0))

/* local defines */

/* open this definition to use old version of dumpSid(), dumpDomainSid() and cmSdDumpAccessToken() */
/*#define CM_SD_USEOLDVERSIONDEBUG*/
#define NOTHING_TO_INHERIT              0xfff


/* default revision for Security Descriptors */
#define CM_SD_SD_REVISION                1

/* default revision for ACLs */
#define CM_SD_DACL_REVISION              2

/* default revision for ACEs */
#define CM_SD_SID_REVISION               1

/* type bits for security descriptor */
#define CM_SD_OWNERDEFAULTED        0x0001
#define CM_SD_GROUPDEFAULTED        0x0002
#define CM_SD_DACLPRESENT           0x0004
#define CM_SD_DACLDEFAULTED         0x0008
#define CM_SD_SACLPRESENT           0x0010
#define CM_SD_SACLDEFAULTED         0x0020
#define CM_SD_DACLTRUSTED           0x0040
#define CM_SD_SERVER_SECURITY       0x0080
#define CM_SD_DACLAUTOINHERITREQ    0x0100
#define CM_SD_SACLAUTOINHERITREQ    0x0200
#define CM_SD_DACLAUTOINHERITED     0x0400
#define CM_SD_SACLAUTOINHERITED     0x0800
#define CM_SD_DACLPROTECTED         0x1000
#define CM_SD_SACLPROTECTED         0x2000
#define CM_SD_RM_CONTROLVALID       0x4000
#define CM_SD_SELF_RELATIVE         0x8000

/* access mask bits and masks */
#define CM_SD_SPECIFICRIGHTSMASK    0x0000ffff
#define CM_SD_STANDARDRIGHTSMASK    0x001f0000
#define CM_SD_GENERCIRIGHTSMASK     0xf0000000

/* Flag bits in ACE */
#define CM_SD_ACEAUDITFAILEDACCESS      0x40
#define CM_SD_ACEAUDITSUCCESSFULLACCESS 0x20
#define CM_SD_ACEINHERITEDACE           0x10
#define CM_SD_ACEINHERITONLY            0x08
#define CM_SD_ACENONPROPAGATEINHERIT    0x04
#define CM_SD_ACECONTAINERINHERIT       0x02
#define CM_SD_ACEOBJECTINHERIT          0x01
#define CM_SD_NOFLAGS                   0x00

   /* generic rights */
#define SD_ACR_GENERIC_READ              (0x80000000L)
#define SD_ACR_GENERIC_WRITE             (0x40000000L)
#define SD_ACR_GENERIC_EXECUTE           (0x20000000L)
#define SD_ACR_GENERIC_ALL               (0x10000000L)

    /* standard rights */
#define SD_ACR_STANDARD_DELETE           (0x00010000L)
#define SD_ACR_STANDARD_READCONTROL      (0x00020000L)
#define SD_ACR_STANDARD_WRITEDAC         (0x00040000L)
#define SD_ACR_STANDARD_WRITEOWNER       (0x00080000L)
#define SD_ACR_STANDARD_SYNCHRONIZE      (0x00100000L)
#define SD_ACR_STANDARD_REQUIRED         (0x000F0000L)
#define SD_ACR_STANDARD_READ             (SD_ACR_STANDARD_READCONTROL)
#define SD_ACR_STANDARD_WRITE            (SD_ACR_STANDARD_READCONTROL)
#define SD_ACR_STANDARD_EXECUTE          (SD_ACR_STANDARD_READCONTROL)
#define SD_ACR_STANDARD_ALL              (0x001F0000L)

    /* file / folder specific rights */
#define SD_ACR_FILE_READDATA             (0x0001)    /* works on files & pipes */
#define SD_ACR_FILE_LISTDIRECTORY        (0x0001)    /* directories */
#define SD_ACR_FILE_WRITEDATA            (0x0002)    /* files & pipes */
#define SD_ACR_FILE_ADDFILE              (0x0002)    /* directories */
#define SD_ACR_FILE_APPENDDATA           (0x0004)    /* files */
#define SD_ACR_FILE_ADDSUBDIRECTORY      (0x0004)    /* directories */
#define SD_ACR_FILE_READEA               (0x0008)    /* files & directories */
#define SD_ACR_FILE_WRITEEA              (0x0010)    /* files & directories */
#define SD_ACR_FILE_EXECUTE              (0x0020)    /* files */
#define SD_ACR_FILE_TRAVERSE             (0x0020)    /* directories */
#define SD_ACR_FILE_DELETECHILD          (0x0040)    /* directories */
#define SD_ACR_FILE_READATTRIBUTES       (0x0080)    /* all */
#define SD_ACR_FILE_WRITEATTRIBUTES      (0x0100)    /* all */
#define SD_ACR_FILE_ALLACCESS            (SD_ACR_STANDARD_REQUIRED | SD_ACR_STANDARD_SYNCHRONIZE | 0x3FF)

#define SD_ACR_FILE_GENERICREAD          (SD_ACR_STANDARD_READ |\
                                          SD_ACR_FILE_READDATA |\
                                          SD_ACR_FILE_READATTRIBUTES |\
                                          SD_ACR_FILE_READEA |\
                                          SD_ACR_STANDARD_SYNCHRONIZE)

#define SD_ACR_FILE_GENERICWRITE         (SD_ACR_STANDARD_WRITE |\
                                          SD_ACR_FILE_WRITEDATA |\
                                          SD_ACR_FILE_WRITEATTRIBUTES |\
                                          SD_ACR_FILE_WRITEEA |\
                                          SD_ACR_FILE_APPENDDATA |\
                                          SD_ACR_STANDARD_SYNCHRONIZE)

#define SD_ACR_FILE_GENERICEXECUTE       (SD_ACR_STANDARD_EXECUTE |\
                                          SD_ACR_FILE_READATTRIBUTES |\
                                          SD_ACR_FILE_EXECUTE |\
                                          SD_ACR_STANDARD_SYNCHRONIZE)

#define SD_ACR_SPECIFIC_ALL              (0x0000FFFFL)

    /* masks used by Remote Share Management */

/* 0x001200A9 */
#define SD_READ_ACCESS                   (SD_ACR_FILE_READDATA |\
                                          SD_ACR_FILE_READEA |\
                                          SD_ACR_FILE_READATTRIBUTES |\
                                          SD_ACR_STANDARD_READ |\
                                          SD_ACR_STANDARD_SYNCHRONIZE)
                                          /*SD_ACR_FILE_EXECUTE*/

/* 0x001301BF */
#define SD_CHANGE_ACCESS                 (SD_ACR_STANDARD_DELETE |\
                                          SD_ACR_STANDARD_READCONTROL |\
                                          SD_ACR_STANDARD_SYNCHRONIZE |\
                                          SD_ACR_FILE_READDATA |\
                                          SD_ACR_FILE_WRITEDATA |\
                                          SD_ACR_FILE_APPENDDATA |\
                                          SD_ACR_FILE_READEA |\
                                          SD_ACR_FILE_WRITEEA |\
                                          SD_ACR_FILE_EXECUTE |\
                                          SD_ACR_FILE_READATTRIBUTES |\
                                          SD_ACR_FILE_WRITEATTRIBUTES)

/* 0x001F01FF */
#define SD_FULL_ACCESS                   (SD_ACR_STANDARD_DELETE |\
                                          SD_ACR_STANDARD_READCONTROL |\
                                          SD_ACR_STANDARD_WRITEDAC |\
                                          SD_ACR_STANDARD_WRITEOWNER |\
                                          SD_ACR_STANDARD_SYNCHRONIZE |\
                                          SD_ACR_FILE_READDATA |\
                                          SD_ACR_FILE_WRITEDATA |\
                                          SD_ACR_FILE_APPENDDATA |\
                                          SD_ACR_FILE_READEA |\
                                          SD_ACR_FILE_WRITEEA |\
                                          SD_ACR_FILE_EXECUTE |\
                                          SD_ACR_FILE_DELETECHILD |\
                                          SD_ACR_FILE_READATTRIBUTES |\
                                          SD_ACR_FILE_WRITEATTRIBUTES)


#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)
#ifdef UD_NQ_INCLUDETRACE
static void dumpSid(const NQ_CHAR *title, NQ_BYTE *buffer, NQ_INT length, NQ_INT offset);
static NQ_BYTE *dumpDomainSid(const NQ_CHAR *title, const CMSdDomainSid *sid);
static void dumpAcl(const NQ_CHAR *title, NQ_BYTE *buffer, NQ_INT length, NQ_INT offset);
static CMSdAce *dumpAce(const CMSdAce *ace);
static NQ_CHAR *ridToStr(CMSdRid rid);
#endif /* UD_NQ_INCLUDETRACE */


CMRepository sdBuffRepo;

/* static variables */

static const NQ_UINT sidFixedPartSize = (sizeof(NQ_BYTE/*sid.revision*/) + sizeof(NQ_BYTE/*sid.numAuths*/) + sizeof(NQ_BYTE) * 6 /* sid.idAuth */);
static const NQ_UINT aclFixedPartSize = (NQ_UINT)(sizeof(NQ_UINT16/*acl.revision*/) + sizeof(NQ_UINT16/*acl.size*/) + sizeof(NQ_UINT32/*acl.numAces*/));
static const NQ_UINT aceFixedPartSize = ((NQ_UINT)(sizeof(NQ_BYTE/*ace.type*/) + sizeof(NQ_BYTE/*ace.flags*/) + sizeof(NQ_UINT16/*ace.size*/)
                                         + sizeof(NQ_UINT32/*ace.accessMask*/)) +
                                         (sizeof(NQ_BYTE/*sid.revision*/) + sizeof(NQ_BYTE/*sid.numAuths*/) + sizeof(NQ_BYTE) * 6 /* sid.idAuth */));


#ifdef UD_NQ_INCLUDECIFSSERVER
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
/* ACE: S-1-5-32 + one unknown sub-authority */
static const CMSdDomainSid defDomain = {
  0x01,0x02,{0,0,0,0,0,0x05},
  {0x20,0,0,0,0,0}
};

/* ACE: S-1-5-21-<domain SID>-domain RID */
static const CMSdDomainSid domainSIDDomainRid = {
      0x01,0x05,{0,0,0,0,0,0x05},
      {0x15,0,0,0,0,0}
    };

/* ACE: S-1-1-0 */
static const CMSdDomainSid everyone = {
  0x01,0x01,{0,0,0,0,0,0x01},
  {0,0,0,0,0,0}
};

/* ACE: S-1-5-32-544 Builtin Administrators */
static const CMSdDomainSid builtinAdmins = {
  0x01,0x02,{0,0,0,0,0,0x05},
  {0x20,0x220,0,0,0,0}
};
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
#endif /* UD_NQ_INCLUDECIFSSERVER */

/* ACE: S-1-5-18 Local System */
static const CMSdDomainSid localSystem = {
  0x01,0x01,{0,0,0,0,0,0x05},
  {0x12,0,0,0,0}
};

typedef struct
{
    /* Singletons */
#ifdef UD_NQ_INCLUDECIFSSERVER
    NQ_BOOL domainAliasSidSet;
    NQ_BOOL domainSidSet;
    NQ_BOOL computerSidSet;
    CMSdDomainSid computerSid;      /* SID for the computer of the server */
    CMSdDomainSid domainAliasSid;   /* SID for the domain alias */
    CMSdDomainSid domainSid;        /* SID for the domain of the server */
#endif /* UD_NQ_INCLUDECIFSSERVER */
    NQ_CHAR tempName[256];          /* name in ASCII */
}
StaticData;

static NQ_BOOL isModuleInitialized = FALSE;
#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

#ifdef UD_NQ_INCLUDECIFSSERVER
static const CMSdDomainSid constDomainSid =
    {0x01,0x01,{0,0,0,0,0,0x05},{32,0,0,0,0,0}};
#endif /* UD_NQ_INCLUDECIFSSERVER */
static const CMSdDomainSid constCreatorOwner = /* Well-known SID: S-1-3-0  (Creator Owner)  */
    {0x01,0x01,{0,0,0,0,0,0x03},{0,0,0,0,0,0}};
static const CMSdDomainSid constCreatorGroup = /* Well-known SID: S-1-3-1  (Creator Group)  */
    {0x01,0x01,{0,0,0,0,0,0x03},{1,0,0,0,0,0}};
#ifdef UD_NQ_INCLUDECIFSSERVER
static const CMSdDomainSid constComputerSid =
    {0x01,0x04,{0,0,0,0,0,0x05},{21,0,0,0,0,0}};
#endif /* UD_NQ_INCLUDECIFSSERVER */

/* local alias/group table */

typedef struct
{
    CMSdRid rid;            /* RID */
    NQ_UINT32 type;         /* RID type */
    const NQ_CHAR* name;    /* name */
    const NQ_CHAR* fullName;/* description */
} AliasEntry;

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

static const AliasEntry localAliases[] = {
    {CM_SD_RIDALIASADMIN,         CM_SD_RIDTYPE_ALIAS,      "Administrators", "Administrators have complete and unrestricted access to the computer/domain"},
    {CM_SD_RIDALIASUSER,          CM_SD_RIDTYPE_ALIAS,      "Users", "Ordinary users"},
    {CM_SD_RIDALIASGUEST,         CM_SD_RIDTYPE_ALIAS,      "Guests", "Local or domain guests"},
    {CM_SD_RIDGROUPADMINS,        CM_SD_RIDTYPE_DOMGRP,     "Administrators", "Local or domain administrators"},
    {CM_SD_RIDGROUPUSERS,         CM_SD_RIDTYPE_DOMGRP,     "None", "Local or domain users"},
    {CM_SD_RIDGROUPGUESTS,        CM_SD_RIDTYPE_DOMGRP,     "Guests", "Local or domain guests"},
/*    {CM_SD_RIDADMINISTRATOR,    CM_SD_RIDTYPE_USER,     "Administrator", "Local Administrator"},
    {CM_SD_RIDGUEST,            CM_SD_RIDTYPE_USER,     "Guest", "Guest account"},
*/
};

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

/* INFO Method descriptor */
typedef struct
{
    NQ_UINT32 flags;                /* various flags (see below) */
    NQ_BOOL
        (*handle)(                  /* method handle or NULL if not handled */
        CMBlob*                     /* IN/OUT the context */
        );
} SDDescriptor;

SDDescriptor sdMethods[] = {
    { CM_SD_OWNER, cmSdGetOwnerSecurityDescriptor },                                            /* 0x00000001 */
    { CM_SD_GROUP, cmSdGetGroupSecurityDescriptor },                                            /* 0x00000002 */
    { CM_SD_OWNER | CM_SD_GROUP, cmSdGetOwnerAndGroupSecurityDescriptor },                      /* 0x00000003 */
    { CM_SD_DACL, cmSdGetDefaultSecurityDescriptor },                                           /* 0x00000004 */
    { CM_SD_OWNER | CM_SD_DACL, cmSdGetOwnerAndDaclSecurityDescriptor },                        /* 0x00000005 */
    { CM_SD_GROUP | CM_SD_DACL, cmSdGetGroupAndDaclSecurityDescriptor },                        /* 0x00000006 */
    { CM_SD_OWNER | CM_SD_GROUP | CM_SD_DACL, cmSdGetOwnerAndGroupAndDaclSecurityDescriptor },  /* 0x00000007 */
};


/* "Current" SID for the domain of the server */
static const CMSdDomainSid anyDomainSid = {0x01,0x01,{0,0,0,0,0,0x05},{32,0,0,0,0,0} };

/* Different default security descriptors are defined in Little Endian
 * byte order
 */

/* default SD for files, printers */
static const NQ_BYTE defaultSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x04,0x84,      /* type */ /* 0x8404, Self Relative, DACL Auto Inherited, DACL Present */
    0,0,0,0,        /* owner sid */
    0,0,0,0,        /* group sid */
    0,0,0,0,        /* sacl */
    0x14,0,0,0,     /* dacl */
        /* DACL */
    2,0,            /* revision */
    96,0,           /* size */
    4,0,0,0,        /* num ACEs */
        /* ACE: S-1-5-32-544 */
    0x00,0x13,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
        /* ACE: S-1-1-0 */
    0x00,0x13,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,
        /* ACE: S-1-5-18 */
    0x00,0x13,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x05,0x12,0,0,0,
        /* ACE: S-1-5-32-545 */
    0x00,0x13,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0
};

/* default SD for files (Windows-style owner has full control, system as well
 * owner should be added as a first ace */
static const NQ_BYTE defaultSecurityDescriptorWinStyle[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x04,0x80,      /* type */ /* 0x8004, Self Relative, DACL Present */
    0,0,0,0,       /* owner sid */
    0,0,0,0,        /* group sid */
    0,0,0,0,        /* sacl */
    0,0,0,0         /* dacl */
};

/* default SD for printers jobs */
/*static const NQ_BYTE defaultJobSecurityDescriptor[] =
{        SD
    0x01, 0,         revision
    0x04,0x84,       type
    0,0,0,0,         owner sid
    0,0,0,0,         group sid
    0,0,0,0,         sacl
    0x14,0,0,0,      dacl
         DACL
    2,0,             revision
    96,0,            size
    4,0,0,0,         num ACEs
         ACE: S-1-5-32-544
    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
         ACE: S-1-1-0
    0x00,0x0b,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,
         ACE: S-1-5-18
    0x00,0x03,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x05,0x12,0,0,0,
         ACE: S-1-5-32-545
    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0
};*/

/* default SD for shares */
static const NQ_BYTE shareSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x04,0x84,      /* type */
    0x14,0,0,0,     /* owner sid */
    0x24,0,0,0,     /* group sid */
    0,0,0,0,        /* sacl */
    0x34,0,0,0,     /* dacl */
        /* ACE: S-1-5-32-544 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
        /* ACE: S-1-5-32-545 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0,
        /* DACL */
    2,0,            /* revision */
    96,0,           /* size */
    4,0,0,0,        /* num ACEs */
        /* ACE: S-1-1-0 */
    0x00,0x13,0x14,0x00,0xbf,0x01,0x13,0x00,0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,
/*    0x00,0x00,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,*/
        /* ACE: S-1-5-32-544 */
    0x00,0x13,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
        /* ACE: S-1-5-32-545 */
    0x00,0x13,0x18,0x00,0xbf,0x01,0x13,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0,
        /* ACE: S-1-5-18 */
    0x00,0x13,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x05,0x12,0,0,0,
        /* ACE: S-1-5-32-512 */
/*    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0,0x02,0,0*/
};

/* special groups default ACEs */
/* Everyone: ACE: S-1-1-0 */
static const NQ_BYTE EveryoneSID[sizeof(CMSdDomainSid)] = {0x01,0x01,0,0,0,0,0,0x01,0,0,0,0};
/* Administrators: ACE: S-1-5-32-544 */
static const NQ_BYTE AdminsSID[sizeof(CMSdDomainSid)] = {0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0};
/* Users: ACE: S-1-5-32-545 */
static const NQ_BYTE UsersSID[sizeof(CMSdDomainSid)] = {0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0};

/* default empty SD for shares */
static const NQ_BYTE shareEmptySecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x04,0x80,      /* type */
    0,0,0,0,        /* owner sid */
    0,0,0,0,        /* group sid */
    0,0,0,0,        /* sacl */
    0x14,0,0,0,     /* dacl */
        /* DACL */
    2,0,            /* revision */
    8,0,            /* size */
    0,0,0,0,        /* num ACEs */
};

/* "non-supported" SD */
static const NQ_BYTE noneSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x04,0x80,      /* type */
    0x14,0,0,0,     /* owner sid */
    0x20,0,0,0,     /* group sid */
    0,0,0,0,        /* sacl */
    0,0,0,0,     /* dacl */
        /* owner S-1-1-0 =  everyone */
    0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,
        /* group S-1-1-0 = everyone */
    0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,
};

/* "administrative access only" SD */
static const NQ_BYTE ownerSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x00,0x80,      /* type */
    0x14,0,0,0,     /* owner sid */
    0,0,0,0,        /* group sid */
    0,0,0,0,        /* sacl */
    0,0,0,0,        /* dacl */
        /* ACE: S-1-5-32-544 = group admins */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0
};

/* "administrative access only" SD */
static const NQ_BYTE ownerAndDaclSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x00,0x80,      /* type */
    0x14,0,0,0,     /* owner sid */
    0,0,0,0,        /* group sid */
    0,0,0,0,        /* sacl */
    0x24,0,0,0,     /* dacl */
        /* ACE: S-1-5-32-544 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
        /* DACL */
    2,0,            /* revision */
    32,0,           /* size */
    1,0,0,0,        /* num ACEs */
        /* ACE: S-1-5-32-544 */
    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0
};

/* "group access only" SD */
static const NQ_BYTE groupSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x00,0x80,      /* type */
    0,0,0,0,        /* owner sid */
    0x14,0,0,0,     /* group sid */
    0,0,0,0,        /* sacl */
    0,0,0,0,        /* dacl */
        /* ACE: S-1-5-32-545 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0
};

/* "group access only" SD */
static const NQ_BYTE groupAndDaclSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x00,0x80,      /* type */
    0,0,0,0,        /* owner sid */
    0x14,0,0,0,     /* group sid */
    0,0,0,0,        /* sacl */
    0x24,0,0,0,     /* dacl */
        /* ACE: S-1-5-32-545 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0,
        /* DACL */
    2,0,            /* revision */
    32,0,           /* size */
    1,0,0,0,        /* num ACEs */
        /* ACE: S-1-5-32-544 */
    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0
};

/* "administrative and group access only" SD */
static const NQ_BYTE ownerAndGroupSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x00,0x80,      /* type */
    0x14,0,0,0,     /* owner sid */
    0x24,0,0,0,     /* group sid */
    0,0,0,0,        /* sacl */
    0,0,0,0,        /* dacl */
        /* ACE: S-1-5-32-544 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
        /* ACE: S-1-5-32-545 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0
};

/* "administrative and group access only" SD */
static const NQ_BYTE ownerAndGroupAndDaclSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x00,0x80,      /* type */
    0x14,0,0,0,     /* owner sid */
    0x24,0,0,0,     /* group sid */
    0,0,0,0,        /* sacl */
    0x34,0,0,0,     /* dacl */
        /* ACE: S-1-5-32-544 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
        /* ACE: S-1-5-32-545 */
    0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0,
        /* DACL */
    2,0,            /* revision */
    96,0,           /* size */
    4,0,0,0,        /* num ACEs */
        /* ACE: S-1-5-32-544  = Group admins*/
    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0,
        /* ACE: S-1-1-0  = everyone */
    0x00,0x0b,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,
        /* ACE: S-1-5-18 = local system */
    0x00,0x03,0x14,0x00,0xff,0x01,0x1f,0x00,0x01,0x01,0,0,0,0,0,0x05,0x12,0,0,0,
        /* ACE: S-1-5-32-545 = builtin users */
    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x21,0x02,0,0
};

/* default SD for readonly shares */
static const NQ_BYTE readonlyShareSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x04,0x80,      /* type */
    0,0,0,0,        /* owner sid */
    0,0,0,0,        /* group sid */
    0,0,0,0,        /* sacl */
    0x14,0,0,0,     /* dacl */
        /* DACL */
    2,0,            /* revision */
    28,0,           /* size */
    1,0,0,0,        /* num ACEs */
        /* ACE: S-1-1-0 = everyone*/
    0x00,0x03,0x14,0x00,0xa9,0x00,0x12,0x00,0x01,0x01,0,0,0,0,0,0x01,0,0,0,0,
        /* ACE: S-1-5-32-544 = group admins */
/*    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0*/
};

/* default SD for shares with administrative access only */
static const NQ_BYTE adminonlyShareSecurityDescriptor[] =
{       /* SD */
    0x01, 0,        /* revision*/
    0x04,0x81,      /* type */
    0,0,0,0,        /* owner sid */
    0,0,0,0,        /* group sid */
    0,0,0,0,        /* sacl */
    0x14,0,0,0,     /* dacl */
        /* DACL */
    2,0,            /* revision */
    32,0,           /* size */
    1,0,0,0,        /* num ACEs */
        /* ACE: S-1-5-32-544 = group admins */
    0x00,0x03,0x18,0x00,0xff,0x01,0x1f,0x00,0x01,0x02,0,0,0,0,0,0x05,0x20,0,0,0,0x20,0x02,0,0
};

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP)

void static sdMapAceAccessMask(CMSdAce *pAce)
{
    LOGFB(CM_TRC_LEVEL_SD, "accessMask: 0x%08x", pAce->accessMask);

    if (pAce->accessMask & SD_ACR_GENERIC_ALL)
    {
        pAce->accessMask = SD_FULL_ACCESS;
        pAce->accessMask &= (NQ_UINT32)~SD_ACR_GENERIC_ALL;
    }
    if (pAce->accessMask & SD_ACR_GENERIC_READ)
    {
        pAce->accessMask |= SD_READ_ACCESS;
        pAce->accessMask &= (NQ_UINT32)~SMB_DESIREDACCESS_GENREAD;
    }
    if (pAce->accessMask & SD_ACR_GENERIC_WRITE)
    {
        pAce->accessMask |= SD_ACR_FILE_GENERICWRITE;
        pAce->accessMask &= (NQ_UINT32)~SMB_DESIREDACCESS_GENWRITE;
    }
    if (pAce->accessMask & SD_ACR_GENERIC_EXECUTE)
    {
        pAce->accessMask |= SD_ACR_FILE_GENERICEXECUTE;
        pAce->accessMask &= (NQ_UINT32)~SMB_DESIREDACCESS_GENEXECUTE;
    }

    LOGFB(CM_TRC_LEVEL_SD, "accessMask: 0x%08x", pAce->accessMask);
    return;
}

static NQ_UINT32 mapAceAccessMask(NQ_UINT32 mask)
{
    CMSdAce ace;

    LOGFB(CM_TRC_LEVEL_SD, "mask: 0x%08x", mask);
    ace.accessMask = mask;
    sdMapAceAccessMask(&ace);
    LOGFE(CM_TRC_LEVEL_SD, "result: 0x%08x", ace.accessMask);
    return ace.accessMask;
}

#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP)*/

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) && defined(UD_NQ_INCLUDECIFSSERVER)
/* check whether given SID matches user's domain and rid pair */
static NQ_BOOL                                  /* TRUE when match */
matchSid(
    const CMSdDomainSid* domainToCheck,         /* SID in ACE */
    const CMSdDomainSid* domainOfUser,          /* user's domain SID */
    CMSdRid ridToCheck,                         /* RID in ACE */
    CMSdRid ridOfUser                           /* user's RID */
    );

#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) && defined(UD_NQ_INCLUDECIFSSERVER) */

/*
 *====================================================================
 * PURPOSE: Initialise this module
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *====================================================================
 */

NQ_STATUS
cmSdInit(
    void
    )
{
    NQ_STATUS result = NQ_SUCCESS;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
        goto Exit;

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData*)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate SD data");
        result = NQ_FAIL;
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

#ifdef UD_NQ_INCLUDECIFSSERVER
    staticData->domainAliasSidSet = FALSE;
    staticData->domainSidSet = FALSE;
    syMemcpy(
        &staticData->domainSid,
        &constDomainSid,
        sizeof(staticData->domainSid)
        );
    syMemcpy(
            &staticData->computerSid,
            &constComputerSid,
            sizeof(staticData->computerSid)
            );
    staticData->computerSidSet = FALSE;
    cmSdGetComputerSid();

    /* create repository for security descriptor buffers */
    {
        NQ_COUNT numSdBuffers = 5;
#ifdef CM_NQ_STORAGE
        numSdBuffers = (UD_CS_NUMBER_CONNECTIONS / 80) > numSdBuffers ? (UD_CS_NUMBER_CONNECTIONS / 80) : numSdBuffers;
#endif /* CM_NQ_STORAGE */
        cmRepositoryInit(&sdBuffRepo, 0, NULL, NULL, "SD large buffers");
        cmRepositoryItemPoolAlloc(&sdBuffRepo, numSdBuffers, SD_BUFFERS_LARGE_SIZE_BYTE);
    }
#endif /* UD_NQ_INCLUDECIFSSERVER */
    isModuleInitialized = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return result;
}

/*
 *====================================================================
 * PURPOSE: Release data
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *====================================================================
 */

void
cmSdExit(
    void
    )
{
    if (isModuleInitialized)
    {
        /* release memory */
#ifdef UD_NQ_INCLUDECIFSSERVER
        cmRepositoryShutdown(&sdBuffRepo);
#endif /* UD_NQ_INCLUDECIFSSERVER */
#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
            cmMemoryFreeShutdown(staticData);
        staticData = NULL;
#endif /* SY_FORCEALLOCATION */
        isModuleInitialized = FALSE;
    }
}
/*
 *====================================================================
 * PURPOSE: get large security descriptor buffer
 *--------------------------------------------------------------------
 * PARAMS:  OUT: bufSize
 *
 * RETURNS: buffer pointer or null for failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BYTE *cmSdGetLargeSDBuf(NQ_COUNT *bufSize)
{
    NQ_BYTE *buff;

    LOGFB(CM_TRC_LEVEL_SD, "*bufSize:%p", bufSize);

    *bufSize = SD_BUFFERS_LARGE_SIZE_BYTE;
    buff = (NQ_BYTE*)cmRepositoryGetNewItem(&sdBuffRepo);

    LOGFE(CM_TRC_LEVEL_SD, "buff:%p bufSize:%d", buff, *bufSize);
    return buff;
}

/*
 *====================================================================
 * PURPOSE: release large security descriptor buffer
 *--------------------------------------------------------------------
 * PARAMS:  buffer pointer
 *
 * RETURNS: none
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL cmSdReleaseLargeSDBuf(NQ_BYTE* buffer)
{
    NQ_BOOL result;

    LOGFB(CM_TRC_LEVEL_SD, "*buffer:%p", buffer);

    result = cmRepositoryReleaseItem(&sdBuffRepo, (CMItem *) buffer);

    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE: check security descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN descriptor to check
 *
 * RETURNS: TRUE when descriptor is valid
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdIsValid(
    const CMBlob* pSd
    )
{
    const CMSdSecurityDescriptorHeader *sdHdr;    /* casted pointer to SD header */
    const CMSdAcl* pAcl;                          /* ACL pointer */
    const CMSdAce* pAce;                          /* running ACE pointer */
    const CMSdDomainSid *pSid;                    /* SID pointer */
    NQ_COUNT aceIdx;                              /* ACE index in ACL */
    const NQ_BYTE* limit;                         /* the highest address ACL can contain */
    NQ_BOOL result = FALSE;                       /* return value */

    LOGFB(CM_TRC_LEVEL_SD, "pSd:%p", pSd);

    if (pSd->data == NULL)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor not initialized.");
        goto Exit;
    }

    limit = (const NQ_BYTE*)(pSd->data + pSd->len);
    sdHdr = (const CMSdSecurityDescriptorHeader *) pSd->data;

    if (sdHdr->revision != CM_SD_SD_REVISION)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor has invalid revision: %d", sdHdr->revision);
        goto Exit;
    }

    if (!(sdHdr->type & CM_SD_DACLPRESENT))
    {
        /* no DACL nothing more to check*/
        LOGMSG(CM_TRC_LEVEL_MESS_SOME, "DACL not present");
    }
    else if (sdHdr->dacl == 0)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_SOME, "null DACL present.");
    }

    if ((sdHdr->type & CM_SD_OWNER) && sdHdr->ownerSid == 0)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "OWNER missing");
        result = TRUE;
        goto Exit;
    }

    if (sdHdr->ownerSid > 0)
    {
        if (pSd->data + sdHdr->ownerSid + sidFixedPartSize > limit)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor has invalid owner SID offset: %d", sdHdr->ownerSid);
            goto Exit;
        }
        pSid = (const CMSdDomainSid*) (pSd->data + sdHdr->ownerSid);
        if (pSd->data + (sidFixedPartSize + pSid->numAuths * sizeof(NQ_UINT32)) > limit)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor has invalid owner SID offset: %d", sdHdr->ownerSid);
            goto Exit;
        }
    }

    if (sdHdr->groupSid > 0)
    {
        if (pSd->data + sdHdr->groupSid + sidFixedPartSize > limit)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor has invalid group SID offset: %d", sdHdr->groupSid);
            goto Exit;
        }
        pSid = (const CMSdDomainSid*) (pSd->data + sdHdr->groupSid);
        if (pSd->data + (sidFixedPartSize + pSid->numAuths * sizeof(NQ_UINT32)) > limit)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor has invalid group SID offset: %d", sdHdr->groupSid);
            goto Exit;
        }
    }

    if (pSd->data + sdHdr->dacl + sidFixedPartSize > limit)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor has invalid DACL offset: %u", sdHdr->dacl);
        goto Exit;
    }

    pAcl = (const CMSdAcl*)(pSd->data + sdHdr->dacl);

    for (aceIdx = 0,
           pAce = (const CMSdAce*)(pAcl + 1);
         aceIdx < pAcl->numAces;
         aceIdx++,
           pAce = (const CMSdAce*)((NQ_BYTE*)pAce + pAce->size)
         )
    {
        if ((NQ_BYTE*)pAce + aceFixedPartSize > limit)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor too long, probably corrupted");
            goto Exit;
        }
        /* use numAuths only after validating its a legal address */
        if ((NQ_BYTE*)pAce + aceFixedPartSize + (pAce->trustee.numAuths * sizeof(NQ_UINT32)) > limit)
        {
           LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor too long, probably corrupted");
           goto Exit;
        }
        if (pAce->size > sizeof(*pAce))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "ACE size in descriptor too long, probably corrupted, index: %d, size in Ace: %d", aceIdx, pAce->size);
            goto Exit;
        }
    }
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE: determine if this user has administrative access
 *--------------------------------------------------------------------
 * PARAMS:  IN user access token
 *
 * RETURNS: TRUE when access is allowed
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdIsAdministrator(
    const void * userToken
    )
{
    NQ_INT i;       /* just a counter */
    NQ_BOOL result = TRUE;
    const CMSdAccessToken* token = (const CMSdAccessToken*) userToken;

    if (!cmSdTokenIsValid(token))
    {
        result = FALSE;
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
        goto Exit;
    }

    if (cmSdIsAdmin(token->rids[0]))
        goto Exit;            /* local administrator */

    for (i = 1 ; i < (NQ_INT)(token->numRids) ; i++)
    {
        if (   token->rids[i] == CM_SD_RIDADMINISTRATOR
            || token->rids[i] == CM_SD_RIDGROUPADMINS
            || token->rids[i] == CM_SD_RIDALIASADMIN
            || token->rids[i] == SD_UID_DOMAINGROUP_ADMINS
            || token->rids[i] == SD_UID_DOMAINGROUP_ENTERPADMINS
           )
            goto Exit;        /* domain administrator or alias */
    }
    result = FALSE;

Exit:
    return result;
}


NQ_BOOL cmSdIsSystem(const CMSdAccessToken* token)
{
    NQ_INT i;       /* just a counter */

    if (!cmSdTokenIsValid(token))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
    }
    else
    {
        for (i = 0 ; i < (NQ_INT)(token->numRids) ; i++)
        {
            if (token->rids[i] == CM_SD_RIDLOCALSYSTEM)
                return TRUE;
        }
    }
    return FALSE;
}

/* map SD's generic access rights into object specific rights*/
void cmSdMapAccessRights(NQ_BYTE* sdData)
{
    const CMSdSecurityDescriptorHeader* pHdr;     /* casted pointer to SD header */
    const CMSdAcl* pAcl;                          /* ACL pointer */
    CMSdAce* pAce;                                /* running ACE pointer */
    NQ_COUNT aceIdx;                              /* ACE index in ACL */

    LOGFB(CM_TRC_LEVEL_SD, "sdData: %p", sdData);

    pHdr = (CMSdSecurityDescriptorHeader*)sdData;
    if ((pHdr->type & CM_SD_DACLPRESENT) && (pHdr->dacl!= 0))
    {
        pAcl = (const CMSdAcl*)(sdData + pHdr->dacl);
        for (aceIdx = 0,
               pAce = (CMSdAce*)(pAcl + 1);
             aceIdx < pAcl->numAces;
             aceIdx++,
               pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size)
             )
        {
            if ((0 == (pAce->accessMask & SMB_DESIREDACCESS_GENMASK))/* || (pAce->flags & CM_SD_ACEINHERITONLY)*/)
            {
                /* no generic rights, nothing to do */
                /* skip mapping for inherit only aces */
                continue;
            }
            pAce->accessMask = mapAceAccessMask(pAce->accessMask);
        }
    }
    LOGFE(CM_TRC_LEVEL_SD);
}

#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP)*/
#ifdef UD_NQ_INCLUDECIFSSERVER

CMSdAccessFlags cmSdGetAccess(const CMBlob * sd, void * userToken)
{
    CMSdAccessFlags result = 0;                   /* return value */
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
    const CMSdSecurityDescriptorHeader* pHdr;     /* casted pointer to SD header */
    const CMSdAcl* pAcl;                          /* ACL pointer */
    const CMSdAce* pAce;                          /* running ACE pointer */
    NQ_COUNT aceIdx;                              /* ACE index in ACL */
    NQ_COUNT ridIdx;                              /* RID index in token */
    const NQ_BYTE* limit;                         /* the highest address ACL can contain */
    const CMSdAccessToken * token = (const CMSdAccessToken *)userToken;
    static const CMSdDomainSid defDomain = {      /* ACE: S-1-5-32 + one unknown sub-authority */
      0x01,0x02,{0,0,0,0,0,0x05},
      {0x20,0,0,0,0,0}
    };
    static const CMSdDomainSid everyone = {       /* ACE: S-1-1-0 */
      0x01,0x01,{0,0,0,0,0,0x01},
      {0,0,0,0,0,0}
    };

    LOGFB(CM_TRC_LEVEL_SD, "sd:%p sd->len:%d userToken:%p", sd, sd->len, userToken);

    if (!cmSdTokenIsValid(token))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
        goto Exit;
    }

    /* validate SD */
    if (!cmSdIsValid(sd))
    {
        LOGMSG(CM_TRC_LEVEL_SD, "Security descriptor is Invalid");
        result = 0xFFFFFFFF;
        goto Exit;
    }

#ifdef UD_NQ_INCLUDETRACE
    cmSdDumpSecurityDescriptor(NULL, (const CMSdSecurityDescriptor *)sd->data, sd->len);
    cmSdDumpAccessToken(token);
#endif /* UD_NQ_INCLUDETRACE */

    limit = sd->data + UD_CM_SECURITYDESCRIPTORLENGTH;
    pHdr = (CMSdSecurityDescriptorHeader*)sd->data;
/*  if (0 == pHdr->dacl)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "SD with no DACL");
        result = 0xFFFFFFFF;
        goto Exit;
    }*/
    pAcl = (const CMSdAcl*)(sd->data + pHdr->dacl);

    for (aceIdx = 0,
           pAce = (const CMSdAce*)(pAcl + 1);
         aceIdx < pAcl->numAces;
         aceIdx++,
           pAce = (const CMSdAce*)((NQ_BYTE*)pAce + pAce->size)
         )
    {
        if ((NQ_BYTE*)pAce > limit)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Security descriptor too long, probably corrupted");
            goto Exit;
        }
        if (syMemcmp(
                &pAce->trustee,
                &everyone,
                (NQ_UINT)sizeof(everyone) - (NQ_UINT)(6 - everyone.numAuths) * (NQ_UINT)sizeof(CMSdRid)
            ) == 0)
        {
            goto Setaccess;
        }

        for (ridIdx = 0; ridIdx < token->numRids; ridIdx++)
        {
            if (syMemcmp(
                        &pAce->trustee,
                        &defDomain,
                        (NQ_UINT)sizeof(defDomain) - (NQ_UINT)(6 - defDomain.numAuths + 1) * (NQ_UINT)sizeof(CMSdRid)
                        ) == 0 &&
                   matchSid(
                        &staticData->computerSid,
                        &token->domain,
                        pAce->trustee.subs[pAce->trustee.numAuths - 1],
                        token->rids[ridIdx]
                        )
                )
            {
                goto Setaccess;
            }
            if (matchSid(
                    &pAce->trustee,
                    &token->domain,
                    pAce->trustee.subs[pAce->trustee.numAuths - 1],
                    token->rids[ridIdx]
                    )
               )
            {
                goto Setaccess;
            }
        }
        continue;
Setaccess:
        if (pAce->type == CM_SD_DENY)
        {
            result &= ~pAce->accessMask;
        }
        else
        {
            result |= pAce->accessMask;
        }
    }

    if (0 != pHdr->ownerSid)
    {
        CMSdDomainSid * pSid;

        pSid = (CMSdDomainSid*)((NQ_BYTE*)sd->data + pHdr->ownerSid);
        if (NULL != token && (pSid->subs[pSid->numAuths - 1] == token->rids[0]))
            result |= SMB_DESIREDACCESS_WRITEDAC | SMB_DESIREDACCESS_READCONTROL;
    }
Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:0x%x", result);
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    return result;
}


/* check whether user token contains sd's owner */
NQ_BOOL
cmSdIsOwnerSidInToken(
        const CMSdAccessToken* token,
        const NQ_BYTE* sd
        )
{
#ifndef UD_CS_INCLUDESECURITYDESCRIPTORS
    NQ_BOOL result = TRUE;                       /* return value */
#else
    NQ_BOOL result = FALSE;                       /* return value */
    const CMSdSecurityDescriptorHeader* pHdr;     /* casted pointer to SD header */

    LOGFB(CM_TRC_LEVEL_SD, "token:%p sd:%p", token, sd);

    if (!cmSdTokenIsValid(token))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
        goto Exit;
    }

    pHdr = (CMSdSecurityDescriptorHeader*)sd;
    if (pHdr->ownerSid == 0)
    {
        goto Exit;
    }
    else
    {
        NQ_COUNT ridIdx;
        CMSdDomainSid *pOwnerSid = (CMSdDomainSid *)(sd + pHdr->ownerSid);

        for (ridIdx = 0; ridIdx < token->numRids; ridIdx++)
        {
            if (matchSid(pOwnerSid,
                        &token->domain,
                        pOwnerSid->subs[pOwnerSid->numAuths - 1],
                        token->rids[ridIdx]
                        ))
            {
                result = TRUE;
                goto Exit;
            }
        }
    }
Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    return result;
}


/*
 *====================================================================
 * PURPOSE: determine user's access by ACL
 *--------------------------------------------------------------------
 * PARAMS:  IN user access token
 *          IN security descriptor data
 *          IN desired access MASK flags.
 *
 * RETURNS: TRUE when access is allowed
 *
 * NOTES:   We consider DACL only by comparing each ACE until either
 *          all are parsed or the first match both access bits and
 *          the SID
 *====================================================================
 */

NQ_BOOL
cmSdHasPermissions(
    NQ_UINT32 requestedAttributeFlags,
    NQ_UINT32 fileAccessGranted
    )
{
    NQ_BOOL returnVal = TRUE;
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS

    returnVal = FALSE;

    LOGFB(CM_TRC_LEVEL_SD, "requestedAttributeFlags:%x fileAccessGranted:%x", requestedAttributeFlags, fileAccessGranted);

    if ((requestedAttributeFlags & CM_SD_SACL) && !(fileAccessGranted & SMB_DESIREDACCESS_SYSTEMSECURITY))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "insufficient access for required operation - modify system security. CM_SD_SACL.");
        goto Exit;
    }

    if ((requestedAttributeFlags & CM_SD_DACL) && !(fileAccessGranted & SMB_DESIREDACCESS_WRITEDAC))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "insufficient access for required operation - modify DACL");
        goto Exit;
    }

    if ((requestedAttributeFlags & (CM_SD_OWNER | CM_SD_GROUP | CM_SD_LABEL)) && !(fileAccessGranted & SMB_DESIREDACCESS_WRITEOWNER))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "insufficient access for required operation. modify owner | group | label.");
        goto Exit;
    }

    returnVal = TRUE;
    Exit:
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    return returnVal;
}

/*
 *====================================================================
 * PURPOSE: determine user's access by ACL
 *--------------------------------------------------------------------
 * PARAMS:  IN user access token
 *          IN security descriptor data
 *          IN desired access MASK flags.
 *
 * RETURNS: TRUE when access is allowed
 *
 * NOTES:   We consider DACL only by comparing each ACE until either
 *          all are parsed or the first match both access bits and
 *          the SID
 *====================================================================
 */

NQ_BOOL
cmSdHasAccess(
    const void * userToken,
    const NQ_BYTE * sdData,
    NQ_UINT32 requestedAccess
    )
{
#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
    const CMSdSecurityDescriptorHeader* pHdr;     /* casted pointer to SD header */
    const CMSdAcl* pAcl;                          /* ACL pointer */
    const CMSdAce* pAce;                          /* running ACE pointer */
    NQ_COUNT aceIdx;                              /* ACE index in ACL */
    NQ_COUNT ridIdx;                              /* RID index in token */
    const CMSdAccessToken* token = (const CMSdAccessToken*)userToken;
    const NQ_BYTE* sd = sdData;
    CMSdAccessFlags access = requestedAccess;
    NQ_BOOL result = FALSE;                       /* return value */

#define ANYDACLACCESS ( SMB_DESIREDACCESS_WRITEDAC | \
                            SMB_DESIREDACCESS_WRITEOWNER | \
                            SMB_DESIREDACCESS_SYNCHRONISE | \
                            SMB_DESIREDACCESS_WRITEOWNER \
                          )

    LOGFB(CM_TRC_LEVEL_SD, "token:%p sd:%p access:0x%08x", token, sd, access);
    
    if (!cmSdTokenIsValid(token))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
        goto Exit;
    }

#ifdef UD_NQ_INCLUDETRACE
    /*cmSdDumpAccessToken(token);*/
#endif /* UD_NQ_INCLUDETRACE */

   /* allow any DACL access for local administrators */
    /*
    if ((access & ANYDACLACCESS) > 0)
    {
        for (ridIdx = 0; ridIdx < token->numRids; ridIdx++)
        {
            if (CM_SD_RIDALIASADMIN == token->rids[ridIdx])
            {
                result = TRUE;
                goto Exit;
            }
        }
    }
    */

    /* validate SD */
    pHdr = (CMSdSecurityDescriptorHeader*)sd;

    /* if DACL not present or NULL DACL (offset == 0). allow access */
    if (NULL == pHdr || 0 == pHdr->dacl || !(pHdr->type & CM_SD_DACLPRESENT))
    {
        result = TRUE;
        goto Exit;
    }

    /* check for privileges */
    if ((requestedAccess == SMB_DESIREDACCESS_SYSTEMSECURITY) && token->isAdmin)
    {
        result = TRUE;
        goto Exit;
    }

    if (cmSdIsOwnerSidInToken(token, sd))
    {
        if (access & (SMB_DESIREDACCESS_READCONTROL | SMB_DESIREDACCESS_WRITEDAC))
        {
            result = TRUE;
            goto Exit;
        }
    }

    pAcl = (const CMSdAcl*)(sd + pHdr->dacl);

    for (aceIdx = 0, pAce = (const CMSdAce*)(pAcl + 1);
         aceIdx < pAcl->numAces;
         aceIdx++, pAce = (const CMSdAce*)((NQ_BYTE*)pAce + pAce->size)
         )
    {
        if (pAce->flags & CM_SD_ACEINHERITONLY)
        {
            /* this Ace only for inheritance purpose */
            continue;
        }
#ifdef UD_CS_AUTHENTICATEANONYMOUS
        if (!token->isAnon)
#endif /* UD_CS_AUTHENTICATEANONYMOUS */
        {
        /* check ace referring to Everyone S-1-1-0 */
            if (0 == syMemcmp(
                &pAce->trustee,
                &everyone,
                (NQ_UINT)sizeof(everyone) - (NQ_UINT)(6 - everyone.numAuths) * (NQ_UINT)sizeof(CMSdRid)
                ))
        {
            if ((access & pAce->accessMask) == access)
            {
                if (pAce->type == CM_SD_ALLOW)
                {
                    result = TRUE;
                    goto Exit;
                }
                else if (pAce->type == CM_SD_DENY)
                {
                    goto Exit;
                }
            }
            continue;
        }

        /* check ace referring to S-1-5-18 Local System */
            if (0 == syMemcmp(
                  &pAce->trustee,
                  &localSystem,
                  (NQ_UINT)sizeof(localSystem) - (NQ_UINT)(6 - localSystem.numAuths) * (NQ_UINT)sizeof(CMSdRid)
                ))
          {
              if ((access & pAce->accessMask) == access)
              {
                  if (pAce->type == CM_SD_ALLOW)
                  {
                    result = TRUE;
                    goto Exit;
                  }
                  else if (pAce->type == CM_SD_DENY)
                  {
                    goto Exit;
                  }
              }
              continue;
          }
        } /* end if (!token->isAnon) */

        /* check if this ace refers to default domain template: S-1-5-32 + one unknown sub-authority
           or S-1-5-21-<domain SID>-domain RID */
        if ((syMemcmp(
                    &pAce->trustee,
                    &defDomain,
                    (NQ_UINT)sizeof(defDomain) - (NQ_UINT)(6 - defDomain.numAuths + 1) * (NQ_UINT)sizeof(CMSdRid)
                    ) == 0) ||
            (syMemcmp(
                    &pAce->trustee,
                    &domainSIDDomainRid,
                    (NQ_UINT)sizeof(defDomain) - (NQ_UINT)(5 * (NQ_UINT)sizeof(NQ_UINT32)) /* compare only first authority */
                    ) == 0))
        {
            /* iterate all RIDs (relative IDs) in user token  */
            for (ridIdx = 0; ridIdx < token->numRids; ridIdx++)
            {
                 if (
                        matchSid(
                                &staticData->computerSid,
                                &token->domain,
                                pAce->trustee.subs[pAce->trustee.numAuths - 1],
                                token->rids[ridIdx]
                                )
                        ||
                        matchSid(
                                &pAce->trustee,
                                &token->domain,
                                pAce->trustee.subs[pAce->trustee.numAuths - 1],
                                token->rids[ridIdx]
                                )
                   )
                {
                    if ((access & pAce->accessMask) == access)
                    {
                        if (pAce->type == CM_SD_ALLOW)
                        {
                            result = TRUE;
                            goto Exit;
                        }
                        else if (pAce->type == CM_SD_DENY)
                        {
                            goto Exit;
                        }
                    }
                }
            }
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
    return result;
#else  /* UD_CS_INCLUDESECURITYDESCRIPTORS */
    const CMSdAccessToken* token = (const CMSdAccessToken*)userToken;
    NQ_BOOL result = FALSE;    /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "token:%p sd:%p requestedAccess:0x%x", userToken, sdData, requestedAccess);

    if (token->isAnon)
    {
        if( (requestedAccess & (SMB_DESIREDACCESS_WRITEDATA |
                SMB_DESIREDACCESS_APPENDDATA |
                SMB_DESIREDACCESS_WRITEEA |
                SMB_DESIREDACCESS_DELETECHILD |
                SMB_DESIREDACCESS_WRITEATTRIBUTES |
                SMB_DESIREDACCESS_DELETE |
                SMB_DESIREDACCESS_GENWRITE |
                SMB_DESIREDACCESS_GENALL |
                SMB_DESIREDACCESS_WRITEOWNER |
                SMB_DESIREDACCESS_WRITEDAC
                )))
        {
            result = FALSE;
        }
        else
        {
            result = TRUE;
    }
    }
    else
    {
        result = TRUE;
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */
}

#endif /* UD_NQ_INCLUDECIFSSERVER */

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS)|| defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)

/*
 *====================================================================
 * PURPOSE: get default SD with no owner
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor size
 *
 * RETURNS: TRUE on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetDefaultSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)defaultSecurityDescriptor, sizeof(defaultSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

NQ_BOOL
cmSdGetOwnerAndDaclSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)ownerAndDaclSecurityDescriptor, sizeof(ownerAndDaclSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

NQ_BOOL
cmSdGetGroupAndDaclSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)groupAndDaclSecurityDescriptor, sizeof(groupAndDaclSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get default SD for a share
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetShareSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)shareSecurityDescriptor, sizeof(shareSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}


/*
 *====================================================================
 * PURPOSE: get default SD for a share for token
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetShareSecurityDescriptorByToken (
    CMBlob* pSd,
    const void * userToken
    )
{
    return cmSdGetDefaultSecurityDescriptorByTokenWinStyle(userToken, pSd);
}


/*
 *====================================================================
 * PURPOSE: get empty SD for a share
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetEmptyShareSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)shareEmptySecurityDescriptor, sizeof(shareEmptySecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get "non-supported" SD 
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   This descriptor has "WorldSid" for owner and NULL for others
 *====================================================================
 */

NQ_BOOL
cmSdGetNoneSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)noneSecurityDescriptor, sizeof(noneSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get "administrative access only" SD
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   This descriptor has "WorldSid" for owner
 *====================================================================
 */

NQ_BOOL
cmSdGetOwnerSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)ownerSecurityDescriptor, sizeof(ownerSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get "group access only" SD
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   This descriptor has Sid for group
 *====================================================================
 */

NQ_BOOL
cmSdGetGroupSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)groupSecurityDescriptor, sizeof(groupSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get "administrative and group access only" SD
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   This descriptor has "WorldSid" for owner and SID for group
 *====================================================================
 */

NQ_BOOL
cmSdGetOwnerAndGroupSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)ownerAndGroupSecurityDescriptor, sizeof(ownerAndGroupSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

NQ_BOOL
cmSdGetOwnerAndGroupAndDaclSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)ownerAndGroupAndDaclSecurityDescriptor, sizeof(ownerAndGroupAndDaclSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get read-only SD for a share
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TREU on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetReadonlyShareSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)readonlyShareSecurityDescriptor, sizeof(readonlyShareSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: get admin-only SD for a share
 *--------------------------------------------------------------------
 * PARAMS:  OUT buffer for resulting descriptor
 *
 * RETURNS: TREU on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetAdminonlyShareSecurityDescriptor(
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;    /* packet descriptor for parsing Little Endian SD */
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)adminonlyShareSecurityDescriptor, sizeof(adminonlyShareSecurityDescriptor));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    return TRUE;
}


const NQ_BYTE*
cmSdGetEveryoneGroupACE(
    void
    )
{
    return EveryoneSID;
}

const NQ_BYTE*
cmSdGetAdministratorsGroupACE(
    void
    )
{
    return AdminsSID;
}

const NQ_BYTE*
cmSdGetUsersGroupACE(
    void
    )
{
    return UsersSID;
}

/*
 *====================================================================
 * PURPOSE: get default SD by token
 *--------------------------------------------------------------------
 * PARAMS:  IN user access token to be the owner
 *          OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetDefaultSecurityDescriptorByToken(
    const void * userToken,
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;        /* packet descriptor for parsing Little Endian SD */
    CMSdSecurityDescriptorHeader* pHdr; /* casted pointer to the header */
    CMSdAcl* pAcl;                      /* casted pointer to DACL */
    CMSdAce* pAce;                      /* casted pointer to DACL */
    const CMSdAccessToken* token = (const CMSdAccessToken*)userToken;
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)defaultSecurityDescriptor, sizeof(defaultSecurityDescriptor));

    if (cmSdTokenIsValid(token))
    {
        cmRpcSetDescriptor(&descr, bufPos, FALSE);
        cmSdParseSecurityDescriptor(&descr, pSd);
        pHdr = (CMSdSecurityDescriptorHeader*) pSd->data;
        pAcl = (CMSdAcl*) (pSd->data + pHdr->dacl);
        pAcl->numAces++;
        pAce = (CMSdAce*)(pSd->data + pSd->len);
        pAce->type = CM_SD_ALLOW;
        pAce->flags = CM_SD_ACEINHERITEDACE | CM_SD_ACECONTAINERINHERIT | CM_SD_ACEOBJECTINHERIT;
        pAce->accessMask = 0x001f01ff;
        syMemcpy(&pAce->trustee, &token->domain, sizeof(token->domain));
        pAce->trustee.subs[pAce->trustee.numAuths++] = /*(token->isAdmin) ? (~token->rids[0] + 1) : */token->rids[0];
        pAce->size = (NQ_UINT16)(((NQ_UINT16)sizeof(*pAce) - (6 - pAce->trustee.numAuths) * (NQ_UINT16)sizeof (CMSdRid)));
        pSd->len += pAce->size;
        return TRUE;
    }
    return FALSE;
}

/* OWNER (from token) full access
 * SYSTEM full access
 */
NQ_BOOL
cmSdGetDefaultSecurityDescriptorByTokenWinStyle(
    const void * userToken,
    CMBlob *pSd
    )
{
    CMRpcPacketDescriptor descr;        /* packet descriptor for parsing Little Endian SD */
    CMSdSecurityDescriptorHeader* pHdr; /* casted pointer to the header */
    CMSdAcl* pAcl;                      /* casted pointer to DACL */
    CMSdAce* pAce;                      /* casted pointer to DACL */
    NQ_IOBufPos bufPos;
    CMSdDomainSid *pSid;
    const CMSdAccessToken *token = (CMSdAccessToken *)userToken;

    IOBUF_POSCONSTRUCTORINIT(bufPos);
    IOBUF_POSCONSTRUCTOR(bufPos, (NQ_BYTE*)defaultSecurityDescriptorWinStyle, sizeof(defaultSecurityDescriptorWinStyle));

    cmRpcSetDescriptor(&descr, bufPos, FALSE);
    cmSdParseSecurityDescriptor(&descr, pSd);
    pSd->len = sizeof(CMSdSecurityDescriptorHeader);
    pHdr = (CMSdSecurityDescriptorHeader*) pSd->data;

    /* set OWNER from token */
    if (cmSdTokenIsValid(token))
    {
        NQ_BOOL wasSet = FALSE;
        NQ_COUNT ownerCopySize;
        NQ_COUNT i;
        CMSdDomainSid *pOwnerSid;

        pHdr->ownerSid = sizeof(defaultSecurityDescriptorWinStyle);
        pOwnerSid = (CMSdDomainSid *)(pSd->data + pHdr->ownerSid);

        /* iterate rids. set owner to local administrator group if exists, otherwise copy domain SID */
        for (i = 0; i < token->numRids; ++i)
        {
            if (SD_UID_DOMAINGROUP_ADMINS == token->rids[i])
            {
                CMSdDomainSid *pSid = (CMSdDomainSid *)AdminsSID;

                ownerCopySize = sidFixedPartSize + (NQ_COUNT)(pSid->numAuths) * (NQ_COUNT)sizeof(pSid->subs[0]);
                syMemcpy(pOwnerSid, &AdminsSID, ownerCopySize);
                wasSet = TRUE;
                break;
            }
        }
        if (!wasSet)
        {
            ownerCopySize = sidFixedPartSize + (NQ_COUNT)((token->domain.numAuths) * sizeof(token->domain.subs[0]));
            syMemcpy(pOwnerSid, &token->domain, ownerCopySize);
            pOwnerSid->subs[pOwnerSid->numAuths] = token->rids[0];
            pOwnerSid->numAuths++;
            ownerCopySize += (NQ_COUNT)sizeof(CMSdRid);
        }
        pSd->len += (NQ_UINT32)(ownerCopySize);
        /* set dacl offset */
        pHdr->dacl = (NQ_UINT32)(pHdr->ownerSid + ownerCopySize);
        pAcl = (CMSdAcl*) (pSd->data + pHdr->dacl);
        pAcl->revision = CM_SD_DACL_REVISION;
        pAcl->size = (NQ_UINT16)aclFixedPartSize;
        pSd->len += (NQ_UINT32)sizeof(*pAcl);

        /* create OWNER ace. copy its SID from owner SID - it has same logic as owner SID */
        pAcl->numAces = 1;
        pAce = (CMSdAce*)((NQ_BYTE *)pAcl + sizeof(*pAcl));
        pAce->type = CM_SD_ALLOW;
        pAce->flags = CM_SD_ACECONTAINERINHERIT | CM_SD_ACEOBJECTINHERIT; /* CM_SD_NOFLAGS */
        pAce->accessMask = 0x001f01ff; /* full access */
        pSid = &pAce->trustee;
        syMemcpy(pSid, pOwnerSid, sidFixedPartSize + (sizeof(pOwnerSid->subs[0]) * pOwnerSid->numAuths));
        pAce->size = (NQ_UINT16)(((NQ_UINT16)sizeof(*pAce) - (6 - pAce->trustee.numAuths) * (NQ_UINT16)sizeof(CMSdRid)));
        pSd->len += pAce->size;
        pAcl->size = (NQ_UINT16)(pAcl->size + pAce->size);
    }
    else
    {
        pHdr->ownerSid = 0;
        pHdr->dacl = sizeof(defaultSecurityDescriptorWinStyle);
        pAcl = (CMSdAcl*) (pSd->data + pHdr->dacl);
        pAcl->revision = CM_SD_DACL_REVISION;
        pAcl->size = (NQ_UINT16)aclFixedPartSize;
        pSd->len += (NQ_UINT32)sizeof(*pAcl);
        pAcl->numAces = 0;

        pAce = (CMSdAce*)((NQ_BYTE *)pAcl + sizeof(*pAcl));
    }
    /* copy SYSTEM */
    pAcl->numAces++;
    pAce = (CMSdAce*)((NQ_BYTE *)pAce + pAce->size);
    pAce->type = CM_SD_ALLOW;
    pAce->flags = CM_SD_ACECONTAINERINHERIT | CM_SD_ACEOBJECTINHERIT; /* CM_SD_NOFLAGS */
    pAce->accessMask = 0x001f01ff; /* full access */
    syMemcpy(&pAce->trustee, &localSystem, sizeof(localSystem));
    pAce->size = (NQ_UINT16)(((NQ_UINT16)sizeof(*pAce) - (6 - pAce->trustee.numAuths) * (NQ_UINT16)sizeof(CMSdRid)));
    pAcl->size = (NQ_UINT16)(pAcl->size + pAce->size);
    pSd->len += pAce->size;

    return TRUE;
}

NQ_BOOL
cmSdGetDefaultSecurityDescriptorByFlags(
        const NQ_UINT32 flags,
        CMBlob *pSd
)
{
    NQ_COUNT i;

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "loading default Security Descriptor");

    for (i = 0; i < sizeof(sdMethods) / sizeof(sdMethods[0]); i++)
    {
        if (sdMethods[i].flags == flags && sdMethods[i].handle != NULL)
        {
            sdMethods[i].handle(pSd);
            break;
        }
    }
    if (sizeof(sdMethods) / sizeof(sdMethods[0]) == i)
        cmSdGetNoneSecurityDescriptor(pSd);

    return TRUE;
}

#ifdef UD_NQ_INCLUDECIFSSERVER

/*
 *====================================================================
 * PURPOSE: get default SD for user
 *--------------------------------------------------------------------
 * PARAMS:  IN owner's RID
 *          OUT buffer for resulting descriptor
 *
 * RETURNS: TRUE on success
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdGetLocalSecurityDescriptorForUser(
    CMSdRid rid,
    CMSdSecurityDescriptor* pSd
    )
{
    CMSdSecurityDescriptorHeader* pHdr; /* casted pointer to the header */
    CMSdAcl* pAcl;                      /* casted pointer to DACL */
    CMSdAce* pAce;                      /* casted pointer to ACE */

    pHdr = (CMSdSecurityDescriptorHeader*) pSd->data;
    pHdr->revision =  CM_SD_SD_REVISION;
    pHdr->type = CM_SD_DACLPRESENT | CM_SD_SELF_RELATIVE;
    pHdr->groupSid = 0;
    pHdr->ownerSid = 0;
    pHdr->dacl = sizeof(CMSdSecurityDescriptorHeader);
    pHdr->sacl = 0;
    pAcl = (CMSdAcl*) (pSd->data + pHdr->dacl);
    pAcl->revision = CM_SD_DACL_REVISION;
    pAcl->numAces = 4;
    pAcl->size = 112;
    pAce = (CMSdAce*)((NQ_BYTE*)pAcl + sizeof(*pAcl));
    /* Everyone */
    pAce->type = CM_SD_ALLOW;
    pAce->flags = 0;
    pAce->accessMask = 0x0002035b;    /* Windows server sends 0x2035b. However,
     * from us Win client expects 0x20075, otherwise it tries to update the
     * value */
    pAce->size = 20;
    pAce->trustee.revision = CM_SD_SID_REVISION;
    pAce->trustee.numAuths = 1;
    pAce->trustee.idAuth[0] = 0;
    pAce->trustee.idAuth[1] = 0;
    pAce->trustee.idAuth[2] = 0;
    pAce->trustee.idAuth[3] = 0;
    pAce->trustee.idAuth[4] = 0;
    pAce->trustee.idAuth[5] = 1;
    pAce->trustee.subs[0] = 0;
    pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size);
    /* Administrator */
    pAce->type = CM_SD_ALLOW;
    pAce->flags = 0;
    pAce->accessMask = 0x000f07ff;
    pAce->size = 24;
    pAce->trustee.revision = CM_SD_SID_REVISION;
    pAce->trustee.numAuths = 2;
    pAce->trustee.idAuth[0] = 0;
    pAce->trustee.idAuth[1] = 0;
    pAce->trustee.idAuth[2] = 0;
    pAce->trustee.idAuth[3] = 0;
    pAce->trustee.idAuth[4] = 0;
    pAce->trustee.idAuth[5] = 5;
    pAce->trustee.subs[0] = 32;
    pAce->trustee.subs[1] = CM_SD_RIDALIASADMIN;
    pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size);
    /* Account operator */
    pAce->type = CM_SD_ALLOW;
    pAce->flags = 0;
    pAce->accessMask = 0x000f07ff;
    pAce->size = 24;
    pAce->trustee.revision = CM_SD_SID_REVISION;
    pAce->trustee.numAuths = 2;
    pAce->trustee.idAuth[0] = 0;
    pAce->trustee.idAuth[1] = 0;
    pAce->trustee.idAuth[2] = 0;
    pAce->trustee.idAuth[3] = 0;
    pAce->trustee.idAuth[4] = 0;
    pAce->trustee.idAuth[5] = 5;
    pAce->trustee.subs[0] = 32;
    pAce->trustee.subs[1] = CM_SD_RIDALIASACCOUNTOP;
    pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size);
    /* Computer SID + user RID */
    pAce->type = CM_SD_ALLOW;
    pAce->flags = 0;
    pAce->accessMask = 0x00020044;
    pAce->size = 36;
    syMemcpy(&pAce->trustee, cmSdGetComputerSid(), sizeof(CMSdDomainSid));
    pAce->trustee.subs[4] = rid;
    pAce->trustee.numAuths++;
    pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size);
    pSd->length = (NQ_UINT32)((NQ_BYTE*)pAce - pSd->data);

    return TRUE;
}

/*
 *====================================================================
 * PURPOSE: set host domain's SID
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to SID to set
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
cmSdSetDomainSid(
    const CMSdDomainSid* sid
    )
{
    if (TRUE == staticData->domainSidSet) return;
    syMemcpy(&staticData->domainSid, sid, sizeof(staticData->domainSid));
    staticData->domainSidSet = TRUE;
}

/*
 *====================================================================
 * PURPOSE: get host domain's SID
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: pointer to domain SID
 *
 * NOTES:   if domain SID was not set yet computer SID is used instead
 *====================================================================
 */

const CMSdDomainSid*
cmSdGetDomainSid(
    void
    )
{
    const CMSdDomainSid* pResult;

    if (!staticData->domainSidSet)
    {
        pResult = cmSdGetComputerSid();
        goto Exit;
    }
    pResult = &staticData->domainSid;

Exit:
    return pResult;
}

/*
 *====================================================================
 * PURPOSE: get local domain SID alias
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: pointer to domain SID
 *
 * NOTES:   returns S-1-1-5-32
 *====================================================================
 */

const CMSdDomainSid*
cmSdGetLocalDomainAlias(
    void
    )
{
    if (!staticData->domainAliasSidSet)
    {
        staticData->domainAliasSid.revision = CM_SD_SID_REVISION;
        staticData->domainAliasSid.numAuths = 1;
        staticData->domainAliasSid.idAuth[0] = 0;
        staticData->domainAliasSid.idAuth[1] = 0;
        staticData->domainAliasSid.idAuth[2] = 0;
        staticData->domainAliasSid.idAuth[3] = 0;
        staticData->domainAliasSid.idAuth[4] = 0;
        staticData->domainAliasSid.idAuth[5] = 5;
        staticData->domainAliasSid.subs[0] =  32;
        staticData->domainAliasSidSet = TRUE;
    }
    return &staticData->domainAliasSid;
}

/*
 *====================================================================
 * PURPOSE: get host computer SID
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: pointer to computer SID
 *
 * NOTES:
 *====================================================================
 */

const CMSdDomainSid*
cmSdGetComputerSid(
    void
    )
{
    if (!staticData->computerSidSet)
    {
        udGetComputerId((NQ_BYTE*)&staticData->computerSid.subs[1]);
        staticData->computerSidSet = TRUE;
    }

    return &staticData->computerSid;
}

/*
 *====================================================================
 * PURPOSE: checks if domain SID was set
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: TRUE if already set
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdIsDomainSidSet(
    void
    )
{
    return staticData->domainSidSet;
}

#endif /* UD_NQ_INCLUDECIFSSERVER */

/*
 *====================================================================
 * PURPOSE: parse domain SID
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT incoming packet descriptor
 *          OUT buffer for SID
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdParseSid(
    CMRpcPacketDescriptor* in,
    CMSdDomainSid* sid
    )
{
    NQ_INT i;   /* just a counter */

    cmRpcParseByte(in, &sid->revision);
    cmRpcParseByte(in, &sid->numAuths);
    if (sid->numAuths > sizeof(sid->subs)/sizeof(sid->subs[0]))
    {
        sid->numAuths = sizeof(sid->subs)/sizeof(sid->subs[0]);
    }
    cmRpcParseBytes(in, sid->idAuth, 6);
    for (i = 0; i < sid->numAuths; i++)
    {
        NQ_UINT32 t;
        cmRpcParseUint32(in, &t);
        sid->subs[i] = t;
    }
}

/*
 *====================================================================
 * PURPOSE: compare domain SID with "Any" SID
 *--------------------------------------------------------------------
 * PARAMS:  IN SID to compare
 *
 * RETURNS: TRUE on match
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdIsAnySid(
    const CMSdDomainSid* sid
    )
{
    return (sid->numAuths == anyDomainSid.numAuths
        && 0 == syMemcmp(sid->subs, anyDomainSid.subs, sid->numAuths * sizeof(sid->subs[0])));
}



#ifdef UD_NQ_INCLUDECIFSSERVER

/*
 *====================================================================
 * PURPOSE: checks if our server knows an alias
 *--------------------------------------------------------------------
 * PARAMS:  IN SID with domain SID + alias RID
 *          OUT buffer for alias RID
 *
 * RETURNS: TRUE when we support this alias
 *
 * NOTES:   The domain portion of the SID should match computer SID
 *          The last authority should be RID for a well known user, group
 *          or alias.
 *====================================================================
 */

NQ_BOOL
cmSdCheckAlias(
    const CMSdDomainSid* sid,
    CMSdRid* alias
    )
{
    const CMSdDomainSid* compSid;        /* computer SID */
    NQ_BOOL result = TRUE;

    compSid = cmSdGetComputerSid();
    if (   sid->numAuths != compSid->numAuths + 1
        || 0 != syMemcmp(sid->subs, compSid->subs, compSid->numAuths * sizeof(compSid->subs[0]))
       )
    {
        result = FALSE;
        goto Exit;
    }
    switch (sid->subs[sid->numAuths - 1])
    {
    case CM_SD_RIDADMINISTRATOR:
    case CM_SD_RIDGUEST:
        *alias = CM_SD_RIDALIASGUEST;
        break;
    case CM_SD_RIDGROUPADMINS:
        *alias = CM_SD_RIDALIASADMIN;
        break;
    case CM_SD_RIDGROUPUSERS:
        *alias = CM_SD_RIDALIASUSER;
        break;
    case CM_SD_RIDGROUPGUESTS:
        *alias = CM_SD_RIDALIASGUEST;
        break;
    case CM_SD_RIDALIASADMIN:
        *alias = CM_SD_RIDALIASADMIN;
        break;
    case CM_SD_RIDALIASUSER:
        *alias = CM_SD_RIDALIASUSER;
        break;
    case CM_SD_RIDALIASGUEST:
        *alias = CM_SD_RIDALIASGUEST;
        break;
    default:
        *alias = CM_SD_RIDALIASUSER;
    }

Exit:
    return result;
}

/*
 *====================================================================
 * PURPOSE: compare domain SID with computer SID
 *--------------------------------------------------------------------
 * PARAMS:  IN SID to compare
 *
 * RETURNS: TRUE on match
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdIsComputerSid(
    const CMSdDomainSid* sid
    )
{
    const CMSdDomainSid* compSid;        /* computer SID */

    compSid = cmSdGetComputerSid();
    return sid->numAuths == compSid->numAuths
        && 0 == syMemcmp(sid->subs, compSid->subs, sid->numAuths * sizeof(sid->subs[0]));
}

#endif /* UD_NQ_INCLUDECIFSSERVER */

/*
 *====================================================================
 * PURPOSE: pack domain SID
 *--------------------------------------------------------------------
 * PARAMS:  OUT outgoing packet descriptor
 *          IN SID to pack
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdPackSid(
    CMRpcPacketDescriptor* out,
    const CMSdDomainSid* sid
    )
{
    NQ_INT i;   /* just a counter */

    cmRpcPackByte(out, sid->revision);
    cmRpcPackByte(out, sid->numAuths);
    cmRpcPackBytes(out, sid->idAuth, 6);
    for (i = 0; i < sid->numAuths; i++)
        cmRpcPackUint32(out, sid->subs[i]);
}

/*
 *====================================================================
 * PURPOSE: pack full user SID
 *--------------------------------------------------------------------
 * PARAMS:  OUT outgoing packet descriptor
 *          IN SID to pack
 *          IN RID to pack
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdPackSidRid(
    CMRpcPacketDescriptor* out,
    const CMSdDomainSid* sid,
    NQ_UINT32 rid
    )
{
    NQ_INT i;   /* just a counter */

    cmRpcPackByte(out, sid->revision);
    /* number of sub authorities should include an additional RID */
    cmRpcPackByte(out, (NQ_BYTE)(sid->numAuths + 1));
    cmRpcPackBytes(out, sid->idAuth, 6);

    for (i = 0; i < sid->numAuths; i++)
        cmRpcPackUint32(out, sid->subs[i]);

    cmRpcPackUint32(out, rid);
}

/*
 *====================================================================
 * PURPOSE: parse Security Descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT incoming packet descriptor
 *          OUT buffer for SD
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdParseSecurityDescriptor(
    CMRpcPacketDescriptor* in,
    CMBlob *pSd
    )
{
    CMSdSecurityDescriptorHeader* pHdr;     /* casted pointer to SD header */
    CMSdDomainSid* pSid;                    /* casted pointer to SIDs */
    CMSdAcl* pAcl;                          /* casted pointer to ACL */
    NQ_IOBufPos sdStart;                    /* pointer to SD start */
    NQ_UINT16 tempUint16;                   /* temporary 16 bits unsigned integer */
    NQ_UINT32 tempUint32;                   /* temporary 32 bits unsigned integer */

    cmRpcGetPosition(in, 0, &sdStart);
    pHdr = (CMSdSecurityDescriptorHeader*)pSd->data;
    cmRpcParseUint16(in, &tempUint16);
    pHdr->revision = tempUint16;
    cmRpcParseUint16(in, &tempUint16);
    pHdr->type = tempUint16;
    cmRpcParseUint32(in, &tempUint32);
    pHdr->ownerSid = tempUint32;
    cmRpcParseUint32(in, &tempUint32);
    pHdr->groupSid = tempUint32;
    cmRpcParseUint32(in, &tempUint32);
    pHdr->sacl = tempUint32;
    cmRpcParseUint32(in, &tempUint32);
    pHdr->dacl = tempUint32;

    if (0 != pHdr->ownerSid)
    {
        pSid = (CMSdDomainSid*)((NQ_BYTE*)pSd->data + pHdr->ownerSid);
        cmRpcSetPositionWithOffset(in, sdStart, pHdr->ownerSid);
        cmSdParseSid(in, pSid);
    }
    if (0 != pHdr->groupSid)
    {
        pSid = (CMSdDomainSid*)((NQ_BYTE*)pSd->data + pHdr->groupSid);
        cmRpcSetPositionWithOffset(in, sdStart, pHdr->groupSid);
        cmSdParseSid(in, pSid);
    }
    if (0 != pHdr->sacl)
    {
        pAcl = (CMSdAcl*)((NQ_BYTE*)pSd->data + pHdr->sacl);
        cmRpcSetPositionWithOffset(in, sdStart, pHdr->sacl);
        cmSdParseAcl(in, pAcl, (const NQ_BYTE*)(pSd + 1));
    }
    if (0 != pHdr->dacl)
    {
        pAcl = (CMSdAcl*)((NQ_BYTE*)pSd->data + pHdr->dacl);
        cmRpcSetPositionWithOffset(in, sdStart, pHdr->dacl);
        cmSdParseAcl(in, pAcl, (const NQ_BYTE*)(pSd + 1));
    }
    pSd->len = (NQ_UINT32)cmRpcGetCountFromOffset(in, sdStart);
}

/*
 *====================================================================
 * PURPOSE: pack Security Descriptor
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT outgoing packet descriptor
 *          IN SD to pack
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdPackSecurityDescriptor(
    CMRpcPacketDescriptor* out,
    const CMBlob* pSd,
    NQ_UINT32 flags
    )
{
#define NODATA 4

    CMSdSecurityDescriptorHeader* pHdr;     /* casted pointer to SD header */
    CMSdDomainSid* pSid;                    /* casted pointer to SIDs */
    CMSdAcl* pAcl;                          /* casted pointer to ACL */
    NQ_UINT32 * length;
    NQ_IOBufPos sdStart;
    NQ_IOBufPos max;
    NQ_UINT32 offset = 20;
    NQ_UINT32 mask = 1, i = 4;
    NQ_BYTE padding[16];

    LOGFB(CM_TRC_LEVEL_SD, "out:%p pSd:%p flags:0x%08x", out, pSd, flags);

    syMemset(padding, 0, 16);

    cmRpcGetPosition(out, 0, &sdStart);
    length = (NQ_UINT32 *)&pSd->len;
    pHdr = (CMSdSecurityDescriptorHeader*)pSd->data;
    cmRpcPackUint16(out, pHdr->revision);
    cmRpcPackUint16(out, pHdr->type);
    cmRpcGetPosition(out, 0, &max);
    cmRpcPackBytes(out, padding, 16);

    while (i--)
    {
        switch (flags & mask)
        {
        case CM_SD_OWNER:
            if (0 != pHdr->ownerSid)
            {
                cmRpcSetPositionWithOffset(out, sdStart, sizeof(NQ_UINT32));
                cmRpcPackUint32(out, offset);
                pSid = (CMSdDomainSid*)((NQ_BYTE*)pSd->data + pHdr->ownerSid);
                cmRpcSetPositionWithOffset(out, sdStart, offset);
                cmSdPackSid(out, pSid);
                offset += (NQ_UINT32)(8 + (sizeof(NQ_UINT32) * pSid->numAuths));
            }
            else if (0 != pHdr->groupSid)
            {
                cmRpcSetPositionWithOffset(out, sdStart, sizeof(NQ_UINT32));
                cmRpcPackUint32(out, offset);
                pSid = (CMSdDomainSid*)((NQ_BYTE*)pSd->data + pHdr->groupSid);
                cmRpcSetPositionWithOffset(out, sdStart, offset);
                cmSdPackSid(out, pSid);
                offset += (NQ_UINT32)(8 + (sizeof(NQ_UINT32) * pSid->numAuths));
            }
            cmRpcGetPosition(out, 0, &max);
            break;
        case CM_SD_GROUP:
            if (0 != pHdr->groupSid)
            {
                cmRpcSetPositionWithOffset(out, sdStart, 2 * sizeof(NQ_UINT32));
                cmRpcPackUint32(out, offset);
                pSid = (CMSdDomainSid*)((NQ_BYTE*)pSd->data + pHdr->groupSid);
                cmRpcSetPositionWithOffset(out, sdStart, offset);
                cmSdPackSid(out, pSid);
                offset += (NQ_UINT32)(8 + (sizeof(NQ_UINT32) * pSid->numAuths));
            }
            if (cmRpcIsCurrentPositionBigger(out, max))
            {
                cmRpcGetPosition(out, 0, &max);
            }

            break;
        case CM_SD_SACL:
            if (0 != pHdr->sacl)
            {
                NQ_IOBufPos curr;

                cmRpcSetPositionWithOffset(out, sdStart, 3 * sizeof(NQ_UINT32));
                cmRpcPackUint32(out, offset);
                cmRpcSetPositionWithOffset(out, sdStart, offset);
                pAcl = (CMSdAcl*)((NQ_BYTE*)pSd->data + pHdr->sacl);
                cmRpcGetPosition(out, 0, &curr);
                cmSdPackAcl(out, pAcl);
                offset += (NQ_UINT32)cmRpcGetCountFromOffset(out, curr);
            }
            if (cmRpcIsCurrentPositionBigger(out, max))
            {
                cmRpcGetPosition(out, 0, &max);
            }

            break;
        case CM_SD_DACL:
            if (0 != pHdr->dacl)
            {
                NQ_IOBufPos curr;

                cmRpcSetPositionWithOffset(out, sdStart, 4 * sizeof(NQ_UINT32));
                cmRpcPackUint32(out, offset);
                cmRpcSetPositionWithOffset(out, sdStart, offset);
                pAcl = (CMSdAcl*)((NQ_BYTE*)pSd->data + pHdr->dacl);
                cmRpcGetPosition(out, 0, &curr);
                cmSdPackAcl(out, pAcl);
                offset += (NQ_UINT32)cmRpcGetCountFromOffset(out, curr);
            }
            break;
        }
        mask <<= 1;
    }

    /* set current to the most distant point */
    if (cmRpcIsCurrentPositionSmaller(out, max))
        cmRpcGetPosition(out, 0, &max);

    *length = (NQ_UINT32)cmRpcGetCountFromOffset(out,sdStart);

    if (NODATA == *length) /* no data had been written */
    {
        cmRpcSetPosition(out, sdStart);
        if ((flags & 0xf) == 0)
        {
            cmRpcPackByte(out, 1);              /* revision */
            cmRpcPackBytes(out, padding, 1);    /* alignment */
            cmRpcPackUint16(out, CM_SD_SELF_RELATIVE);
            cmRpcPackBytes(out, padding, 16);
            *length = (NQ_UINT32)cmRpcGetCountFromOffset(out, sdStart);
        }
        else
        {
            cmRpcPackBytes(out, padding, 4);    /* alignment */
        }
    }
    LOGFE(CM_TRC_LEVEL_SD);
}

/*
 *====================================================================
 * PURPOSE: parse Access Control List
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT outgoing packet descriptor
 *          OUT buffer for ACL
 *          IN highest address to use
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdParseAcl(
    CMRpcPacketDescriptor* in,
    CMSdAcl* pAcl,
    const NQ_BYTE* limit
    )
{
    CMSdAce* pAce;                          /* pointer to ACE */
    NQ_COUNT i;                             /* just a counter */
    NQ_UINT16 tempUint16;                   /* temporary 16 bits unsigned integer */
    NQ_UINT32 tempUint32;                   /* temporary 32 bits unsigned integer */

    if ((NQ_BYTE*)pAcl > limit - sizeof(*pAcl))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "ACL buffer overflow");
    }
    cmRpcParseUint16(in, &tempUint16);
    pAcl->revision = tempUint16;
    cmRpcParseUint16(in, &tempUint16);
    pAcl->size = tempUint16;
    cmRpcParseUint32(in, &tempUint32);
    pAcl->numAces = tempUint32;

    pAce = (CMSdAce*)(pAcl + 1);
    for (i = 0; i < pAcl->numAces; i++)
    {
        NQ_IOBufPos temp;
        cmRpcGetPosition(in, 0, &temp);

        if ((NQ_BYTE*)pAce > limit - sizeof(*pAce))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Too many ACEs");
        }
        cmSdParseAce(in, pAce);
        pAce = (CMSdAce*)((NQ_BYTE*)pAce + cmRpcGetCountFromOffset(in, temp));
    }
}

/*
 *====================================================================
 * PURPOSE: pack Access Control List
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT outgoing packet descriptor
 *          IN ACL to pack
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdPackAcl(
    CMRpcPacketDescriptor* out,
    const CMSdAcl* pAcl
    )
{
    CMSdAce* pAce;                          /* pointer to ACE */
    NQ_COUNT i;                             /* just a counter */

    cmRpcPackUint16(out, pAcl->revision);
    cmRpcPackUint16(out, pAcl->size);
    cmRpcPackUint32(out, pAcl->numAces);

    pAce = (CMSdAce*)(pAcl + 1);
    for (i = 0; i < pAcl->numAces; i++)
    {
        NQ_IOBufPos temp;
        cmRpcGetPosition(out, 0, &temp);

        cmSdPackAce(out, pAce);
        pAce = (CMSdAce*)((NQ_BYTE*)pAce + cmRpcGetCountFromOffset(out, temp));
    }
}

/*
 *====================================================================
 * PURPOSE: parse Access Control Entry
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT outgoing packet descriptor
 *          OUT buffer for ACE
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdParseAce(
    CMRpcPacketDescriptor* in,
    CMSdAce* pAce
    )
{
    NQ_UINT16 tempUint16;                   /* temporary 16 bits unsigned integer */
    NQ_UINT32 tempUint32;                   /* temporary 32 bits unsigned integer */

    cmRpcParseByte(in, &pAce->type);
    cmRpcParseByte(in, &pAce->flags);
    cmRpcParseUint16(in, &tempUint16);
    pAce->size = tempUint16;
    cmRpcParseUint32(in, &tempUint32);
    pAce->accessMask = tempUint32;
    cmSdParseSid(in, &pAce->trustee);
}

/*
 *====================================================================
 * PURPOSE: pack Access Control Entry
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT outgoing packet descriptor
 *          IN ACE to pack
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void cmSdPackAce(
    CMRpcPacketDescriptor* out,
    const CMSdAce* pAce
    )
{
    cmRpcPackByte(out, pAce->type);
    cmRpcPackByte(out, pAce->flags);
    cmRpcPackUint16(out, pAce->size);
    cmRpcPackUint32(out, pAce->accessMask);
    cmSdPackSid(out, &pAce->trustee);
}

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

/*
 *====================================================================
 * PURPOSE: find local name by RID
 *--------------------------------------------------------------------
 * PARAMS:  IN local RID
 *          OUT buffer for name
 *          OUT buffer for full name
 *
 * RETURNS: TRUE when a user or alias for this RID was found
 *
 * NOTES:   RID may be either local user or a predefined local alias
 *====================================================================
 */

NQ_BOOL
cmSdLookupRid(
    CMSdRid rid,
    NQ_WCHAR* nameBuffer,
    NQ_WCHAR* fullNameBuffer
    )
{
    NQ_UINT i;        /* just a counter */
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_SD, "rid:%d, nameBuffer:%p, fullNameBuffer:%p", rid, nameBuffer, fullNameBuffer);

    for (i = 0; i < sizeof(localAliases)/sizeof(localAliases[0]); i++)
    {
        if (rid == localAliases[i].rid)
        {
            cmAnsiToUnicode(nameBuffer, localAliases[i].name);
            cmAnsiToUnicode(fullNameBuffer, localAliases[i].fullName);
            goto Exit;
        }
    }
    result = udGetUserNameByRid(rid, nameBuffer, fullNameBuffer);

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE: find RID by local name
 *--------------------------------------------------------------------
 * PARAMS:  IN name to look for
 *          OUT buffer for rid
 *
 * RETURNS: TRUE when a user or alias was found
 *
 * NOTES:   the name may designate either a local user or a predefined
 *          local alias
 *====================================================================
 */

NQ_BOOL
cmSdLookupName(
    const NQ_WCHAR* name,
    CMSdRid* rid
    )
{
    NQ_UINT i;                        /* just a counter */
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_SD, "name:%s, rid:%p", cmWDump(name), rid);

    cmUnicodeToAnsiN(staticData->tempName, sizeof(staticData->tempName), name, sizeof(staticData->tempName) * sizeof(NQ_WCHAR));
    for (i = 0; i < sizeof(localAliases)/sizeof(localAliases[0]); i++)
    {
        if (0 == syStrcmp(staticData->tempName, localAliases[i].name))
        {
            *rid = localAliases[i].rid;
            goto Exit;
        }
    }
    result = udGetUserRidByName(name, rid);

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE:
 *--------------------------------------------------------------------
 * PARAMS:  IN user's RID
 *
 * RETURNS: RID type
 *
 * NOTES:
 *====================================================================
 */
NQ_UINT32
cmSdGetRidType(
    CMSdRid rid
    )
{
    NQ_UINT i;         /* just a counter */
    NQ_UINT32 result;

    for (i = 0; i < sizeof(localAliases)/sizeof(localAliases[0]); i++)
    {
        if (rid == localAliases[i].rid)
        {
            result = localAliases[i].type;
            goto Exit;
        }
    }
    result = CM_SD_RIDTYPE_USER;

Exit:
    return result;
}

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

/*
 *====================================================================
 * PURPOSE: create SD with exclusive rights for a given user
 *--------------------------------------------------------------------
 * PARAMS:  IN user token
 *          OUT buffer for the result
 *
 * RETURNS: TRUE on success, FALSE on failure
 *
 * NOTES:
 *====================================================================
 */

NQ_BOOL
cmSdCreateExclusiveSecurityDescriptor(
    const CMSdAccessToken* token,
    CMSdSecurityDescriptor* pSd
    )
{
    CMSdSecurityDescriptorHeader* pHdr; /* casted pointer to the header */
    CMSdAcl* pAcl;                      /* casted pointer to DACL */
    CMSdAce* pAce;                      /* casted pointer to ACE */
    CMSdDomainSid* pSid;                /* casted pointer to SID */
    NQ_BOOL result = FALSE;

    if (!cmSdTokenIsValid(token))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
        goto Exit;
    }
    pHdr = (CMSdSecurityDescriptorHeader*) pSd->data;
    pHdr->revision = CM_SD_SD_REVISION;
    pHdr->type = CM_SD_DACLPRESENT | CM_SD_SELF_RELATIVE |
            CM_SD_SACLAUTOINHERITED | CM_SD_DACLAUTOINHERITED | CM_SD_DACLAUTOINHERITREQ;
    pHdr->groupSid = 0;
    pHdr->ownerSid = 0;
    pHdr->dacl = sizeof(CMSdSecurityDescriptorHeader);
    pHdr->sacl = 0;
    pAcl = (CMSdAcl*) (pSd->data + pHdr->dacl);
    pAcl->revision = CM_SD_DACL_REVISION;
    pAcl->numAces = 2;
    pAcl->size = 68;
    pAce = (CMSdAce*)((NQ_BYTE*)pAcl + sizeof(*pAcl));
    /* DACL: */
    /*   administrator */
    pAce->type = CM_SD_ALLOW;
    pAce->flags = 0x3;  /* container and object inherit */
    pAce->accessMask = 0x0001f01ff;
    pAce->size = 24;
    pAce->trustee.revision = CM_SD_SID_REVISION;
    pAce->trustee.numAuths = 2;
    pAce->trustee.idAuth[0] = 0;
    pAce->trustee.idAuth[1] = 0;
    pAce->trustee.idAuth[2] = 0;
    pAce->trustee.idAuth[3] = 0;
    pAce->trustee.idAuth[4] = 0;
    pAce->trustee.idAuth[5] = 5;
    pAce->trustee.subs[0] = 32;
    pAce->trustee.subs[1] = CM_SD_RIDALIASADMIN;
    pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size);
    /*    user */
    pAce->type = CM_SD_ALLOW;
    pAce->flags = 0x3; /* container and object inherit */
    pAce->accessMask = 0x0001f01ff;
    pAce->size = 36;
    syMemcpy(&pAce->trustee, &token->domain, sizeof(token->domain));
    pAce->trustee.subs[pAce->trustee.numAuths] = token->rids[0];
    pAce->trustee.numAuths++;
    pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size);
    /* OWNER */
    pSid = (CMSdDomainSid*)pAce;
    pHdr->ownerSid = (NQ_UINT32)((NQ_BYTE*)pSid - pSd->data);
    syMemcpy(pSid, &token->domain, sizeof(token->domain));
    pSid->subs[pSid->numAuths] = token->rids[0];
    pSid->numAuths++;
    /* GROUP */
    pSid = (CMSdDomainSid*)((NQ_BYTE*)pSid->subs + sizeof(pSid->subs[0]) * pSid->numAuths);
    pHdr->groupSid = (NQ_UINT32)((NQ_BYTE*)pSid - pSd->data);
    syMemcpy(pSid, &token->domain, sizeof(token->domain));
    pSid->subs[pSid->numAuths] = CM_SD_RIDGROUPUSERS;
    pSid->numAuths++;

    pSid = (CMSdDomainSid*)((NQ_BYTE*)pSid->subs + sizeof(pSid->subs[0]) * pSid->numAuths);
    pSd->length = (NQ_UINT32)((NQ_BYTE*)pSid - pSd->data);
    result = TRUE;
Exit:
    return result;
}


/* check whether ACE is inheritable */
static NQ_BOOL isAceInheritable(const CMSdAce* pAce, NQ_BOOL isContainer)
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_SD, "pAce:%p (flags: 0x%04x) isContainer:%d", pAce, pAce->flags, isContainer);

    if (!isContainer)
    {
        result = (pAce->flags & CM_SD_ACEOBJECTINHERIT) != 0;
        goto Exit;
    }
    if ((pAce->flags & CM_SD_ACEOBJECTINHERIT) && !(pAce->flags & CM_SD_ACENONPROPAGATEINHERIT))
    {
        result = TRUE;
        goto Exit;
    }
    if (pAce->flags & CM_SD_ACECONTAINERINHERIT)
    {
        result = TRUE;
        goto Exit;
    }

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}

/* check whether two SIDs are the same */
static NQ_BOOL isSidEqual(const CMSdDomainSid* pSid1, const CMSdDomainSid* pSid2)
{
    NQ_COUNT i;
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_SD, "pSid1:%p pAce2:%p", pSid1, pSid2);

    if (pSid1 == pSid2)
        goto Exit;

    if (pSid1->revision != pSid2->revision)
    {
        result = FALSE;
        goto Exit;
    }

    if (pSid1->numAuths != pSid2->numAuths)
    {
        result = FALSE;
        goto Exit;
    }

    for (i = 0; i < 6; i++)
    {
        if (pSid1->idAuth[i] != pSid2->idAuth[i])
        {
            result = FALSE;
            goto Exit;
        }
    }

    for (i = 0; i < pSid1->numAuths; i++)
    {
        if (pSid1->subs[i] != pSid2->subs[i])
        {
            result = FALSE;
            goto Exit;
        }
    }

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}

/* check whether two ACEs are the same */
static NQ_BOOL isAceEqual(const CMSdAce* pAce1, const CMSdAce* pAce2)
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_SD, "pAce1:%p pAce2:%p", pAce1, pAce2);

    result = (pAce1->type == pAce2->type) && (pAce1->accessMask == pAce2->accessMask) &&
            (pAce1->trustee.numAuths == pAce2->trustee.numAuths) && isSidEqual(&pAce1->trustee, &pAce2->trustee);

    LOGFE(CM_TRC_LEVEL_SD, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}


static void initAce(CMSdAce* pAce, NQ_BYTE type, NQ_BYTE flags, NQ_UINT32 accessMask, const CMSdDomainSid *trustee)
{
    syMemset(pAce, 0, sizeof(*pAce));
    pAce->type = type;
    pAce->flags = flags;
    pAce->accessMask = mapAceAccessMask(accessMask);
    syMemcpy(&pAce->trustee, trustee, (NQ_UINT32)(sizeof(CMSdDomainSid) - (NQ_UINT32)((6 - ((NQ_UINT32)trustee->numAuths + 1)) * sizeof(CMSdRid))));
    pAce->size = (NQ_UINT16)(aceFixedPartSize + pAce->trustee.numAuths * (NQ_COUNT)sizeof(NQ_UINT32));
    /*pAce->size = (NQ_UINT16)(((NQ_UINT16)sizeof(*pAce) - (6 - pAce->trustee.numAuths) * (NQ_UINT16)sizeof (CMSdRid)));*/
}

/* check whether ACE already exists in ACL  */
static NQ_BOOL isDuplicateAce(const CMSdAce* pAce, const CMSdAcl* pAcl)
{
    NQ_BOOL result = FALSE;
    CMSdAce* pAceTemp;
    NQ_COUNT i;

    LOGFB(CM_TRC_LEVEL_SD, "pAce:%p pAcl:%p", pAce, pAcl);

    for (i = 0, pAceTemp = (CMSdAce*)((NQ_BYTE*)pAcl + sizeof(*pAcl)); i < pAcl->numAces; i++, pAceTemp = (CMSdAce*)((NQ_BYTE*)pAceTemp + pAceTemp->size))
    {
        if (isAceEqual(pAceTemp, pAce))
        {
            result = TRUE;
            goto Exit;
        }
    }
Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/* this function modifies oldSd using newSd, result saved into resultSd, caller must free allocated buffer */
NQ_UINT32 cmSdModifySecurityDescriptor(const CMBlob * oldSd, const CMBlob * newSd, const void * userToken, NQ_BOOL isContainer, CMBlob * resultSd)
{
    CMSdSecurityDescriptorHeader* pOldHdr;              /* casted pointer to the header */
    CMSdSecurityDescriptorHeader* pNewHdr;              /* casted pointer to the header */
    CMSdSecurityDescriptorHeader* pRslHdr;              /* casted pointer to the header */
    CMSdAcl *pSrcDacl = NULL;
    NQ_BOOL isAutoInherit = FALSE;                      /* whether auto inherit requested */
    NQ_UINT32 retCode = (NQ_UINT32)NQ_FAIL;             /* this function return value */
    CMSdDomainSid *pOwnerSid = NULL, *pGroupSid = NULL;
    NQ_BYTE *pRslData = NULL;                           /* will point on next empty spot in result SD */
    NQ_BOOL setEmptyDacl = FALSE;

    LOGFB(CM_TRC_LEVEL_SD, "oldSd:%p newSd:%p userToken:%p isContainer:%d resultSd:%p", oldSd, newSd, userToken, isContainer, resultSd);

    pOldHdr = (CMSdSecurityDescriptorHeader*) oldSd->data;
    pNewHdr = (CMSdSecurityDescriptorHeader*) newSd->data;
    pRslHdr = (CMSdSecurityDescriptorHeader*) resultSd->data;
    resultSd->len = sizeof(CMSdSecurityDescriptorHeader);

    *pRslHdr = *pNewHdr;
    pRslHdr->type |= CM_SD_SELF_RELATIVE;

    pRslData = resultSd->data + sizeof(CMSdSecurityDescriptorHeader); /* set packet pointer */

    /* first DACL */
    if (pNewHdr->type & CM_SD_DACLPRESENT)
    {
        if (pNewHdr->dacl == 0)
        {
            setEmptyDacl = TRUE;
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "setting Empty Dacl (null Dacl)");
        }
        else
        {
            pSrcDacl = (CMSdAcl *)(newSd->data + pNewHdr->dacl);
        }
    }
    else if (0 == pNewHdr->ownerSid && 0 == pNewHdr->groupSid)
    {
        /* according to SMB torture this is another combination that should result in empty dacl. */
        setEmptyDacl = TRUE;
    }
    else if (pOldHdr->type & CM_SD_DACLPRESENT)
    {
        if (pOldHdr->dacl == 0)
        {
            setEmptyDacl = TRUE;
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "setting Empty Dacl (null Dacl)");
        }
        else
        {
            pSrcDacl = (CMSdAcl *)(oldSd->data + pOldHdr->dacl);
        }
    }

    if (pSrcDacl)
    {
        CMSdAcl *pResultDacl = (CMSdAcl *)(pRslData);

        pRslHdr->dacl = (NQ_UINT32)(pRslData - resultSd->data);

        syMemcpy((NQ_BYTE*)pResultDacl, (NQ_BYTE*)pSrcDacl, pSrcDacl->size);
        resultSd->len += pResultDacl->size;

        pRslData = ((NQ_BYTE*)pResultDacl) + pResultDacl->size;
        pRslHdr->type |= CM_SD_DACLPRESENT;
    }
    else
    {
        pRslHdr->dacl = 0;
        if (setEmptyDacl)
        {
            pRslHdr->type |= CM_SD_DACLPRESENT;
        }
        else
        {
            pRslHdr->type &= (NQ_UINT16)(~CM_SD_DACLPRESENT);
        }
    }

    /* handle owner */
    if (pNewHdr->ownerSid > 0)
    {
        pOwnerSid = (CMSdDomainSid *)(newSd->data + pNewHdr->ownerSid);
    }
    else if (pOldHdr->ownerSid > 0)
    {
        pOwnerSid = (CMSdDomainSid *)(oldSd->data + pOldHdr->ownerSid);
    }
    else
    {
        pOwnerSid = NULL;
    }

    if (pOwnerSid)
    {
        NQ_COUNT ownerLen = 0;

        /* offset to start of owner */
        pRslHdr->ownerSid = (NQ_UINT32)(pRslData - resultSd->data);

        ownerLen = (NQ_COUNT)((NQ_BYTE *)(&pOwnerSid->subs[0] + pOwnerSid->numAuths) - (NQ_BYTE *)pOwnerSid);
        syMemcpy(pRslData, pOwnerSid, ownerLen);
        resultSd->len += ownerLen;
        pRslData += ownerLen;
    }

    /* handle Group */
    if (pNewHdr->groupSid > 0)
    {
        pGroupSid = (CMSdDomainSid *)(newSd->data + pNewHdr->groupSid);
    }
    else if (pOldHdr->groupSid > 0)
    {
        pGroupSid = (CMSdDomainSid *)(oldSd->data + pOldHdr->groupSid);
    }
    else
    {
        pGroupSid = NULL;
    }

    if (pGroupSid)
    {
        NQ_COUNT groupLen = 0;

        /* offset to start of group */
        pRslHdr->groupSid = (NQ_UINT32)(pRslData - resultSd->data);

        groupLen = (NQ_COUNT)((NQ_BYTE *)(&pGroupSid->subs[0] + pGroupSid->numAuths) - (NQ_BYTE *)pGroupSid);
        syMemcpy(pRslData, pGroupSid, groupLen);
        resultSd->len += groupLen;
        pRslData += groupLen;
    }

    /* SACL if present, place it last */
    if ((pNewHdr->type & CM_SD_SACLPRESENT) && (pNewHdr->sacl > 0))
    {
        CMSdAcl *pSrcSacl = (CMSdAcl *)(newSd->data + pNewHdr->sacl);

        /* offset to sacl */
        pRslHdr->sacl = (NQ_UINT32)(pRslData - resultSd->data);

        syMemcpy(pRslData, pSrcSacl, pSrcSacl->size);
        resultSd->len += pSrcSacl->size;
        pRslData += pSrcSacl->size;
    }

    /* map generic access rights */
    cmSdMapAccessRights(resultSd->data);

    /* set DACL type for new SD */
    isAutoInherit = (pNewHdr->type & (CM_SD_DACLAUTOINHERITED | CM_SD_DACLAUTOINHERITREQ)) == (CM_SD_DACLAUTOINHERITED | CM_SD_DACLAUTOINHERITREQ);
    pRslHdr->type &= (NQ_UINT16)(~(CM_SD_DACLAUTOINHERITED | CM_SD_DACLAUTOINHERITREQ));
    if (isAutoInherit)
        pRslHdr->type |= CM_SD_DACLAUTOINHERITED;

    retCode = NQ_SUCCESS;

    LOGFE(CM_TRC_LEVEL_SD, "result: 0x%x", retCode);
    return retCode;
}

/* check whether SD has OWNER or GROUP or DACL to set  */
NQ_BOOL cmSdHasDataToSet(const CMBlob * sd)
{
    NQ_BOOL result;
    CMSdSecurityDescriptorHeader* pSdHdr = (CMSdSecurityDescriptorHeader*)sd->data;

    result = (pSdHdr->ownerSid == 0) ? FALSE : TRUE;
    result |= (pSdHdr->groupSid == 0) ? FALSE : TRUE;
    result |= (pSdHdr->sacl == 0) ? FALSE : TRUE;
    result |= (pSdHdr->dacl == 0) ? FALSE : TRUE;
    return result;
}

NQ_UINT32 cmSdCreateModifySecurityDescriptor(const CMBlob * parentOldSd, const CMBlob * receivedSd, const void * userToken, NQ_BOOL isContainer, NQ_BOOL isOnlyModifySD, CMBlob * resultSd)
{
    NQ_UINT32 retCode = (NQ_UINT32)NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_SD, "parentOldSd:%p newSd:%p userToken:%p isContainer:%d isOnlyModifySD:%d resultSd:%p", parentOldSd, receivedSd, userToken, isContainer, isOnlyModifySD, resultSd);

    if (NULL == receivedSd)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid security descriptor received, can't update.");
        retCode = SMB_STATUS_INVALID_PARAMETER;
        goto Exit;
    }

    if (FALSE == cmSdIsValid(receivedSd))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Bad security descriptor received, can't update.");
        retCode = SMB_STATUS_DATA_ERROR;
        goto Exit;
    }

#ifdef UD_NQ_INCLUDETRACE
    cmSdDumpSecurityDescriptor((const NQ_WCHAR *)"ppaarreennttOOllddSSdd", (const CMSdSecurityDescriptor *)parentOldSd->data, parentOldSd->len);
#endif /* UD_NQ_INCLUDETRACE */

    resultSd->data = cmSdGetLargeSDBuf(&resultSd->len);
    if (NULL == resultSd->data)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "No large buffers for security descriptor.");
        goto Exit;
    }

    if (isOnlyModifySD)
    {
        retCode = cmSdModifySecurityDescriptor(parentOldSd, receivedSd, userToken, isContainer, resultSd);
        goto Exit;
    }

    /* end of modify start of create new */
    /************************************/

    /* inherit */
    /***********/
    if (NULL != parentOldSd->data)
    {
        retCode = cmSdInherit(parentOldSd, receivedSd, userToken, isContainer, resultSd);
    }

    if (NQ_SUCCESS != retCode)
    {
        /* copy received SD or if empty, use default SD */
        if (NULL == receivedSd->data)
        {
            LOGMSG(CM_TRC_LEVEL_SD, "Failed to inherit from parent SD, using received SD.");
            cmSdGetDefaultSecurityDescriptorByTokenWinStyle(userToken, (CMBlob *)resultSd);
        }
        else
        {
            LOGMSG(CM_TRC_LEVEL_SD, "Failed to inherit from parent SD, creating default SD.");
            syMemcpy(resultSd->data, receivedSd->data, receivedSd->len);
            /* map generic access rights */
            cmSdMapAccessRights(resultSd->data);
            resultSd->len = receivedSd->len;
        }
        retCode = NQ_SUCCESS;
        goto Exit;
    }

Exit:
#ifdef UD_NQ_INCLUDETRACE
    if (retCode == NQ_SUCCESS)
        cmSdDumpSecurityDescriptor((const NQ_WCHAR *)"rreessuullttSSDD", (const CMSdSecurityDescriptor *)resultSd->data, resultSd->len);
#endif /* UD_NQ_INCLUDETRACE */

    LOGFE(CM_TRC_LEVEL_SD, "result: 0x%x", retCode);
    return retCode;
}

/* create new SD (Security descriptor) from received SD and inherit of parent SD
 * PARAMS:  IN parent SD (Security descriptor) to inherit from
*           IN received SD. the new defined SD or defaulted SD.
 *          IN User security token
 *          IN is new SD container (folder)
 *          IN / OUT resulting SD. holds the new SD data. to here we inherit parent SD inheritable parts.
 *
 * RETURNS:
 *      NOTHING_TO_INHERIT - no inherit took place. parent has nothing to inherit
 *      NQ_SUCCES - inherit successful.
 *      SMB_STATUS_BUFFER_TOO_SMALL - result SD buffer is too small to inherit all data
 *
 * NOTES:*/
NQ_UINT32 cmSdInherit(const CMBlob * parentSd, const CMBlob * receivedSd, const void * userToken, NQ_BOOL isContainer, CMBlob * resultSd)
{
    const CMSdAccessToken * token = (const CMSdAccessToken *)userToken; /* user token */
    const CMSdSecurityDescriptorHeader* pParentHdr;     /* casted pointer to the header */
    CMSdSecurityDescriptorHeader* pReceivedHdr = NULL;  /* casted pointer to the header */
    CMSdSecurityDescriptorHeader* pResultHdr;           /* casted pointer to the header */
    const CMSdAcl *pParentDacl;                         /* casted pointer to DACL */
    CMSdAcl *pResultDacl = NULL;                        /* casted pointer to DACL */
    const CMSdAce* pParentAce;                          /* casted pointer to ACE */
    CMSdAce* pNextResultAce = NULL;                     /* casted pointer to ACE */
    NQ_INT i;                                           /* just a counter */
    NQ_BOOL isAutoInherit = FALSE;                      /* whether auto inherit requested */
    NQ_BOOL isParentAclAutoInherited = FALSE;
    const CMSdDomainSid *pPlaceHolderSid, *pCreatorSid;  /* casted pointer to SID - owner and group*/
    CMSdDomainSid *pOwnerSid = NULL, *pGroupSid = NULL;
    NQ_UINT32 retCode = SMB_STATUS_BUFFER_TOO_SMALL;    /* this function return value */
    NQ_BOOL domainAdminsSidSet = FALSE;
    NQ_BYTE* nextEmpty;                                 /* next point to add security descriptor data on result buffer */

    LOGFB(CM_TRC_LEVEL_SD, "parentSd:%p userToken:%p isContainer:%d newSd:%p", parentSd, userToken, isContainer, receivedSd);

    if (!cmSdTokenIsValid(token))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
        goto Exit;
    }

    /* set pointers */
    pParentHdr = (CMSdSecurityDescriptorHeader*) parentSd->data;
    pResultHdr = (CMSdSecurityDescriptorHeader*) resultSd->data;

    /* use received SD if exists */
    if (NULL != receivedSd->data)
    {
        /* copy header from received SD */
        pReceivedHdr = (CMSdSecurityDescriptorHeader*) receivedSd->data;
        *pResultHdr = *pReceivedHdr;
    }
    else
    {
        *pResultHdr = *pParentHdr;
        pResultHdr->type = 0;
    }

    /* set DACL type for new SD */
    isParentAclAutoInherited = pParentHdr->type & CM_SD_DACLAUTOINHERITED;
    isAutoInherit = (pResultHdr->type & (CM_SD_DACLAUTOINHERITED | CM_SD_DACLAUTOINHERITREQ)) == (CM_SD_DACLAUTOINHERITED | CM_SD_DACLAUTOINHERITREQ);
    pResultHdr->type &= (NQ_UINT16)(~(CM_SD_DACLAUTOINHERITED | CM_SD_DACLAUTOINHERITREQ));
    if (isAutoInherit || isParentAclAutoInherited)
        pResultHdr->type |= CM_SD_DACLAUTOINHERITED;

    pResultHdr->type |= CM_SD_SELF_RELATIVE;

    /* OWNER and GROUP SIDs */
    /************************/
    /* start with owner */
    nextEmpty = resultSd->data + sizeof(CMSdSecurityDescriptorHeader);
    pResultHdr->ownerSid = (NQ_UINT32)(nextEmpty - resultSd->data);
    pOwnerSid = (CMSdDomainSid *)(resultSd->data + pResultHdr->ownerSid);
    if (pReceivedHdr && pReceivedHdr->ownerSid)
    {
        /* owner explicitly set. use it.*/
        CMSdDomainSid *pReceivedOwnerSid = (CMSdDomainSid *)(receivedSd->data + pReceivedHdr->ownerSid);
        syMemcpy(pOwnerSid, pReceivedOwnerSid, sidFixedPartSize + sizeof(pReceivedOwnerSid->subs[0]) * pReceivedOwnerSid->numAuths);
    }
    else
    {
        if ((pResultHdr->type & CM_SD_OWNERDEFAULTED) && pParentHdr->ownerSid)
        {
            /* if no token or if owner defaulted - we take owner from parent.  */
            const CMSdDomainSid *pParentOwnerSid = (CMSdDomainSid *)(parentSd->data + pParentHdr->ownerSid);

            syMemcpy(pOwnerSid, pParentOwnerSid, sidFixedPartSize + sizeof(pParentOwnerSid->subs[0]) * pParentOwnerSid->numAuths);
        }
        else
        {
            /* iterate rids. set owner to local administrator group if exists */
            for (i = 0 ; i < (NQ_INT)(token->numRids) ; ++i)
            {
                if (SD_UID_DOMAINGROUP_ADMINS == token->rids[i])
                {
                    CMSdDomainSid *pSid = (CMSdDomainSid *)AdminsSID;

                    domainAdminsSidSet = TRUE;
                    syMemcpy(pOwnerSid, &AdminsSID, sidFixedPartSize + (NQ_COUNT)(pSid->numAuths) * (NQ_COUNT)sizeof(pSid->subs[0]));
                    break;
                }
            }

            if (!domainAdminsSidSet)
            {
                syMemcpy(pOwnerSid, &token->domain, sidFixedPartSize + ((token->domain.numAuths) * sizeof(token->domain.subs[0])));
                pOwnerSid->subs[pOwnerSid->numAuths] = token->rids[0];
                pOwnerSid->numAuths++;
            }
        }
    }

    {
        NQ_UINT32 ownerSidSize;
        ownerSidSize = (NQ_UINT32)(sidFixedPartSize + sizeof(pOwnerSid->subs[0]) * pOwnerSid->numAuths);
        nextEmpty += ownerSidSize;
    }

    /* owner group */
    pResultHdr->groupSid = (NQ_UINT32)(nextEmpty - resultSd->data);
    pGroupSid = (CMSdDomainSid *)(resultSd->data + pResultHdr->groupSid);
    if (pReceivedHdr && pReceivedHdr->groupSid)
    {
        /* owner explicitly set. use it.*/
        CMSdDomainSid *pReceivedGroupSid = (CMSdDomainSid *)(receivedSd->data + pReceivedHdr->groupSid);
        syMemcpy(pGroupSid, pReceivedGroupSid, sidFixedPartSize + sizeof(pReceivedGroupSid->subs[0]) * pReceivedGroupSid->numAuths);
    }
    else
    {
        if ((pResultHdr->type & CM_SD_GROUPDEFAULTED) && pParentHdr->groupSid)
        {
            /* if no token or if owner defaulted - we take owner from parent.  */
            const CMSdDomainSid *pParentGroupSid = (CMSdDomainSid *)(parentSd->data + pParentHdr->groupSid);

            syMemcpy(pGroupSid, pParentGroupSid, sidFixedPartSize + sizeof(pParentGroupSid->subs[0]) * pParentGroupSid->numAuths);
        }
        else
        {
            syMemcpy(pGroupSid, &token->domain, (sidFixedPartSize + ((token->domain.numAuths) * sizeof(token->domain.subs[0]))));
            pGroupSid->subs[pGroupSid->numAuths] = CM_SD_RIDGROUPUSERS;
            pGroupSid->numAuths++;
        }
    }

    {
        NQ_COUNT groupSidSize;
        groupSidSize = (NQ_COUNT)(sidFixedPartSize + sizeof(pGroupSid->subs[0]) * pGroupSid->numAuths);
        nextEmpty += groupSidSize;
    }

    /* DACL */
    /********/

    /* if received DACL, copy to result DACL */
    if (pReceivedHdr && pReceivedHdr->dacl > 0)
    {
        const CMSdAcl *pReceivedDacl;

        pResultHdr->dacl = (NQ_UINT32)(nextEmpty - resultSd->data);
        pResultDacl = (CMSdAcl *)(resultSd->data + pResultHdr->dacl);

        /* if user sent a new SD - use it. else only inheritance will take place. */
        pReceivedDacl = (CMSdAcl*)(receivedSd->data + pReceivedHdr->dacl);

        /* map generic access rights */
        cmSdMapAccessRights(receivedSd->data);

        /* check overflow */
        if (pResultHdr->dacl + pReceivedDacl->size > resultSd->len)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "result SD buffer overflow");
            goto Exit;
        }
        syMemcpy((NQ_BYTE*)pResultDacl, (NQ_BYTE*)pReceivedDacl, pReceivedDacl->size);

        /* iterate received ACL list to find place of next ACE */
        pNextResultAce = (CMSdAce*)((NQ_BYTE*)pResultDacl + sizeof(*pResultDacl));
        for (i = 0; (NQ_UINT)i < pResultDacl->numAces; i++)
        {
            pNextResultAce = (CMSdAce *)((NQ_BYTE*)pNextResultAce + pNextResultAce->size);
        }
    }
    else
    {
        if (pReceivedHdr && (pReceivedHdr->type & CM_SD_DACLPRESENT) && 0 == pReceivedHdr->dacl)
        {
            /* Received NULL DACL */
            pResultHdr->dacl = 0;
        }
        else
        {
            pResultHdr->dacl = (NQ_UINT32)(nextEmpty - resultSd->data);
            pResultDacl = (CMSdAcl *)(resultSd->data + pResultHdr->dacl);
            /* no DACL received. place next ace pointer as first ace in ACL list */
            pResultDacl->numAces = 0;
            pResultDacl->revision = CM_SD_DACL_REVISION;
            pResultDacl->size = sizeof(*pResultDacl);
            pNextResultAce = (CMSdAce *)((NQ_BYTE*)pResultDacl + sizeof(*pResultDacl));
        }
    }

    /* inherit from  parent's DACL if exists */
    /* if no DACL we skip this part. */
    /*********************************/

    if ((!(pReceivedHdr && (pReceivedHdr->type & CM_SD_DACLPRESENT) && pReceivedHdr->dacl == 0)) &&
          (pParentHdr->type & CM_SD_DACLPRESENT) && (pParentHdr->dacl > 0))
        /* if not NULL DACL and parent DACL Exists */
    {
        if (pReceivedHdr && !(pReceivedHdr->type & CM_SD_DACLAUTOINHERITREQ))
        {
            nextEmpty = resultSd->data + pResultHdr->dacl + (pResultDacl? pResultDacl->size : 0);
        }
        else
        {
            pParentDacl = (CMSdAcl*) (parentSd->data + pParentHdr->dacl);

            /* check for buffer overflow */
            if (((NQ_BYTE*)pResultDacl - resultSd->data) + (NQ_INT)sizeof (CMSdAcl) > (NQ_INT)(resultSd->len))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "SD buffer overflow");
                goto Exit;
            }

            /* inherit from parent ACEs to result DACL aces
             *  Replace owner creator ACE.
             *  replace owner group ACE
             *****/


            /* iterate through parent ACEs */
            pParentAce = (CMSdAce*)((NQ_BYTE*)pParentDacl + sizeof(*pParentDacl));
            for (i = 0; (NQ_UINT)i < pParentDacl->numAces; i++)
            {
                CMSdAce ace;
                NQ_BYTE flags = (NQ_BYTE)(isContainer ? pParentAce->flags : 0); /* new ACE flags */

                /* first check whether parent's ACE is inheritable */
                if (!isAceInheritable(pParentAce, isContainer))
                {
                    pParentAce = (CMSdAce *)((NQ_BYTE*)pParentAce + pParentAce->size);
                    continue;
                }

                /* set proper flags for new ACE */
                if (isContainer)
                {
                    flags &= (NQ_BYTE)(~(CM_SD_ACEINHERITONLY | CM_SD_ACEINHERITEDACE));
                    if (!(flags & CM_SD_ACECONTAINERINHERIT))
                        flags |= CM_SD_ACEINHERITONLY;
                    if (flags & CM_SD_ACENONPROPAGATEINHERIT)
                        flags = 0;
                }

                /* deal with CREATOR OWNER and CREATOR GROUP */
                pCreatorSid = NULL;
                if (isSidEqual(&pParentAce->trustee, &constCreatorOwner))
                {
                    /* creator owner ace. A place holder to replace with owner ace. */
                    pPlaceHolderSid = pOwnerSid;
                    pCreatorSid = &constCreatorOwner;
                }
                else if (isSidEqual(&pParentAce->trustee, &constCreatorGroup))
                {
                    /* creator group ace. a place holder to replace with group ace. */
                    pPlaceHolderSid = pGroupSid;
                    pCreatorSid = &constCreatorGroup;
                }
                else
                {
                    pPlaceHolderSid = &pParentAce->trustee;
                }

                if (isContainer)
                {
                    if ((NULL != pCreatorSid) && (flags & CM_SD_ACECONTAINERINHERIT))
                    {
                        /* special case - 2 ACEs copied */
                        if ((NQ_UINT)((NQ_BYTE*)pNextResultAce - resultSd->data) + aceFixedPartSize +
                                (NQ_UINT)(pPlaceHolderSid->numAuths * (NQ_INT)sizeof(NQ_UINT32)) > resultSd->len)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "result SD - buffer overflow");
                            goto Exit;
                        }

                        /* copy parent's ACE */
                        initAce(&ace, pParentAce->type, isParentAclAutoInherited ? CM_SD_ACEINHERITEDACE : 0, pParentAce->accessMask, pPlaceHolderSid);
                        if (!isDuplicateAce(&ace, pResultDacl))
                        {
                            syMemcpy(pNextResultAce, &ace, ace.size);
                            pResultDacl->numAces++;

                            /* set pointer for next new ACE */
                            pNextResultAce = (CMSdAce *)((NQ_BYTE*)pNextResultAce + pNextResultAce->size);
                        }

                        /* copy CREATOR ACE */
                        initAce(&ace, pParentAce->type, flags | CM_SD_ACEINHERITONLY, pParentAce->accessMask, pCreatorSid);
                        if (!isDuplicateAce(&ace, pResultDacl))
                        {
                            syMemcpy(pNextResultAce, &ace, ace.size);
                            pResultDacl->numAces++;

                            /* set pointer for next new ACE */
                            pNextResultAce = (CMSdAce *)((NQ_BYTE*)pNextResultAce + pNextResultAce->size);
                        }

                        pParentAce = (CMSdAce *)((NQ_BYTE*)pParentAce + pParentAce->size);
                        continue;
                    }
                    else if (!(pParentAce->flags & CM_SD_ACENONPROPAGATEINHERIT))
                    {
                        pPlaceHolderSid = &pParentAce->trustee;
                    }
                }

                /* check for buffer overflow */
                if ((NQ_UINT)((NQ_BYTE*)pNextResultAce - resultSd->data) + aceFixedPartSize +
                        (NQ_UINT)(pParentAce->trustee.numAuths * (NQ_INT)sizeof(pParentAce->trustee.subs[0])) > resultSd->len)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "SD buffer overflow");
                    goto Exit;
                }

                /* copy parent's ACE */
                initAce(&ace, pParentAce->type, (NQ_BYTE)(flags | (isParentAclAutoInherited ? CM_SD_ACEINHERITEDACE : 0)), pParentAce->accessMask, pPlaceHolderSid);
                if (!isDuplicateAce(&ace, pResultDacl))
                {
                    syMemcpy(pNextResultAce, &ace, ace.size);
                    pResultDacl->numAces++;

                    /* set pointer for next new ACE */
                    pNextResultAce = (CMSdAce *)((NQ_BYTE*)pNextResultAce + pNextResultAce->size);
                }

                pParentAce = (CMSdAce *)((NQ_BYTE*)pParentAce + pParentAce->size);
            } /* end of for */

            /* calculate new ACL size */
            pResultDacl->size = (NQ_UINT16)((NQ_BYTE *)pNextResultAce - (NQ_BYTE *)pResultDacl);
            nextEmpty = (NQ_BYTE*)pNextResultAce;
        }
    } /* if (pOldHdr->type & CM_SD_DACLPRESENT) */
    else
    {
        /* no DACL */
        nextEmpty = resultSd->data + pResultHdr->dacl + (pResultDacl? pResultDacl->size : 0);
    }

    if (pResultDacl)
    {
        pResultHdr->type |=  CM_SD_DACLPRESENT;
    }

    /* if result DACL should exist but empty in this stage = no new  DACL received and nothing to inherit, set default DACL */
    if (pResultDacl && pResultDacl->numAces == 0)
    {
        cmSdGetDefaultSecurityDescriptorByTokenWinStyle(userToken, (CMBlob *)resultSd);
        /* we can use pResultHdr since header in the same place */
        pResultDacl = (CMSdAcl *) (resultSd->data + pResultHdr->dacl);
        nextEmpty = resultSd->data + pResultHdr->dacl + pResultDacl->size;
        retCode = NQ_SUCCESS;
        goto Exit;
    }

    resultSd->len = (NQ_UINT32)((NQ_BYTE*)nextEmpty - resultSd->data);

    retCode = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result: 0x%x", retCode);
    return retCode;
}

/* check that the given SD has exclusive rights for a given user */
NQ_BOOL cmSdIsExclusiveSecurityDescriptor(const CMSdAccessToken* token, const CMSdSecurityDescriptor* pSd)
{
    CMSdSecurityDescriptorHeader* pHdr; /* casted pointer to the header */
    CMSdAcl* pAcl;                      /* casted pointer to DACL */
    CMSdAce* pAce;                      /* casted pointer to ACE */
    CMSdDomainSid* pSid;                /* pointer to owner */
    NQ_COUNT i;                         /* just a counter */
    NQ_BOOL result = FALSE;             /* return value */

    LOGFB(CM_TRC_LEVEL_SD, "token:%p pSd:%p", token, pSd);

    if (!cmSdTokenIsValid(token))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token");
        goto Exit;
    }

    pHdr = (CMSdSecurityDescriptorHeader*) pSd->data;
    if (pHdr->dacl == 0)
        goto Exit;

    /* Owner */
/*
    if (pHdr->ownerSid == 0)
        goto Exit;
*/

    if (pHdr->ownerSid != 0)
    {
        pSid = (CMSdDomainSid*)(pSd->data + pHdr->ownerSid);
        if (pSid->subs[pSid->numAuths - 1] != token->rids[0])
            goto Exit;
    }

    /* DACL: */
    pAcl = (CMSdAcl*) (pSd->data + pHdr->dacl);
    if (0 == pAcl->numAces)
    {
        goto Exit;
    }

    pAce = (CMSdAce*)((NQ_BYTE*)pAcl + sizeof(*pAcl));
    for (i = 0; i < pAcl->numAces; i++)
    {
        if (    pAce->type != CM_SD_ALLOW
             || (pAce->accessMask & 0x0001d0000) != (0x0001f01ff & 0x0001d0000)
             || pAce->trustee.revision != CM_SD_SID_REVISION
           )
        {
           goto Exit;
        }

           if (    pAce->size == 24
                && pAce->trustee.numAuths == 2
                && pAce->trustee.subs[0] == CM_SD_BUILTIN_GROUP
                && pAce->trustee.subs[1] == CM_SD_RIDALIASADMIN
              )
           {
           }
        else if (    pAce->size == 36
                  && 0 == syMemcmp(pAce->trustee.subs, token->domain.subs, 4 * token->domain.numAuths)
                  && pAce->trustee.numAuths == token->domain.numAuths + 1
                  && pAce->trustee.subs[pAce->trustee.numAuths - 1] == token->rids[0]
           )
        {
        }
        else
        {
            goto Exit;
        }

        pAce = (CMSdAce*)((NQ_BYTE*)pAce + pAce->size);
    }

    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}


NQ_BOOL
cmCheckAnonymousAccess(
    CMBlob * currSd,
    CMBlob * sd
)
{
    CMSdSecurityDescriptorHeader* pHdr;     /* casted pointer to SD header */
    NQ_UINT32 size;
    NQ_UINT32 flags = 0;
    NQ_BOOL res = FALSE;
    CMRpcPacketDescriptor temp;
    NQ_BYTE *bytePtr = NULL;
    NQ_IOBufPos posBlob;

    bytePtr = cmMemoryAllocate(currSd->len + IOBUF_POSCONSTRUCT_EXTRASIZE);
    if (NULL == bytePtr)
    {
        goto Exit;
    }
    IOBUF_POSCONSTRUCTOR_INPLACE(posBlob, bytePtr, currSd->len)
    cmRpcSetDescriptor(&temp, posBlob, FALSE);

    pHdr = (CMSdSecurityDescriptorHeader*)sd->data;

    if (pHdr->ownerSid)
    {
        flags |= CM_SD_OWNER;
    }
    if (pHdr->groupSid)
    {
        flags |= CM_SD_GROUP;
    }
    if (pHdr->dacl)
    {
        flags |= CM_SD_DACL;
    }
    if (pHdr->sacl)
    {
        flags |= CM_SD_SACL;
    }

    cmSdPackSecurityDescriptor(&temp, currSd, flags);
    if (sd->len < currSd->len)
    {
        goto Exit;
    }

    size = 0 == pHdr->dacl ? sd->len : pHdr->dacl;
    cmRpcResetDescriptor(&temp);
    res = cmRpcMemCmpCurrent(&temp, sd->data, size);
    size = currSd->len - size;
    if (size == 0)
    {
        goto Exit;
    }
    size -= 8;
    cmRpcParseSkip(&temp, (NQ_INT32)(pHdr->dacl + 8));
    res += cmRpcMemCmpCurrent(&temp, (sd->data + pHdr->dacl + 8), size);
    res = res == 0 ? TRUE : FALSE;

Exit:
    cmMemoryFree(bytePtr);
    return res;
}

#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)*/

#ifdef UD_CS_INCLUDESECURITYDESCRIPTORS
#ifdef UD_NQ_INCLUDECIFSSERVER
/*
 *====================================================================
 * PURPOSE: check whether given SID matches user's domain and rid pair
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the SID to check
 *          IN pointer to SID of user's domain
 *          IN RID in the SID to check
 *          IN user's RID
 *
 * RETURNS: TRUE when matches
 *
 * NOTES:
 *====================================================================
 */

static NQ_BOOL
matchSid(
    const CMSdDomainSid* domainToCheck,
    const CMSdDomainSid* domainOfUser,
    CMSdRid ridToCheck,
    CMSdRid ridOfUser
    )
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_SD, "domainToCheck:%p domainOfUser:%p ridToCheck:%d ridOfUser:%d", domainToCheck, domainOfUser, ridToCheck, ridOfUser);

    /* Builtin Admins group 544 contains groups 512 and 519 */
    if (((ridOfUser == SD_UID_DOMAINGROUP_ADMINS) || (ridOfUser == SD_UID_DOMAINGROUP_ENTERPADMINS))
            && (syMemcmp(domainToCheck->subs, &builtinAdmins.subs, builtinAdmins.numAuths * sizeof(builtinAdmins.subs[0])) == 0))
    {
        result = TRUE;
        goto Exit;
    }

    if (syMemcmp(domainToCheck->subs, domainOfUser->subs, domainOfUser->numAuths * sizeof(domainToCheck->subs[0])) != 0)
    {
#ifdef UD_NQ_INCLUDETRACE
        dumpDomainSid("domainToCheck", domainToCheck);
        dumpDomainSid("domainOfUser", domainOfUser);
#endif /* UD_NQ_INCLUDETRACE */
        goto Exit;
    }

    result = (ridToCheck == ridOfUser);

Exit:
    LOGFE(CM_TRC_LEVEL_SD, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}


NQ_BOOL                         /* TRUE on success, FALSE on failure */
cmSdCompareToken(
    const CMSdAccessToken* token1,      /* user token1 */
    const CMSdAccessToken* token2       /* user token2 */
    )
{
    NQ_BOOL res = FALSE;

    if (!cmSdTokenIsValid(token1))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token1");
        goto Exit;
    }

    if (!cmSdTokenIsValid(token2))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "invalid user token2");
        goto Exit;
    }

    if (token1->isAnon != token2->isAnon)
        goto Exit;

    res = matchSid(&token1->domain , &token2->domain , *token1->rids , *token2->rids);

Exit:
    return res;
}

#endif /* UD_NQ_INCLUDECIFSSERVER */
#endif /* UD_CS_INCLUDESECURITYDESCRIPTORS */

#if defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)
#ifdef UD_NQ_INCLUDETRACE

static void dumpSid(const NQ_CHAR *title, NQ_BYTE *buffer, NQ_INT length, NQ_INT offset)
{
#ifdef CM_SD_USEOLDVERSIONDEBUG
    syPrintf("  %s: ", title);

    if (offset > 0)
    {
        buffer += offset;
        syPrintf("offset=%d, ", offset);
        dumpDomainSid((CMSdDomainSid *)buffer);
    }
    else
    {
        syPrintf("none");
    }

    syPrintf("\n");
#else /* CM_SD_USEOLDVERSIONDEBUG */
    if (offset > 0)
    {
        buffer += offset;
        LOGMSG(CM_TRC_LEVEL_SD, "%s: offset=%d", title, offset);
        dumpDomainSid(title, (const CMSdDomainSid *)buffer);
    }
    else
    {
        LOGMSG(CM_TRC_LEVEL_SD, "%s: none", title);
    }
#endif /* CM_SD_USEOLDVERSIONDEBUG */
}

static NQ_BYTE *dumpDomainSid(const NQ_CHAR *title, const CMSdDomainSid *sid)
{
#ifdef CM_SD_USEOLDVERSIONDEBUG
    NQ_BYTE* pResult = NULL;

    if (NULL != sid)
    {
        NQ_INT i;

        syPrintf("S-%d-", (NQ_INT)sid->revision);

        for (i = 0; i < 6; i++)
        {
            if (sid->idAuth[i] > 0)
            {
                syPrintf("%d", (NQ_INT)sid->idAuth[i]);
            }
        }

        for (i = 0; i < (NQ_INT)sid->numAuths; i++)
        {
            syPrintf("-%lu", sid->subs[i]);
        }

        pResult = (NQ_BYTE *)&sid->subs[i];
    }
    return pResult;
#else /* CM_SD_USEOLDVERSIONDEBUG */
    NQ_BYTE* pResult = NULL;

    if (sid != NULL)
    {
        NQ_CHAR buff[256];
        NQ_INT printed = 0;
        NQ_CHAR *p;
        NQ_INT i;

        printed = (NQ_INT)sySprintf(buff, "S-%d-", (NQ_INT)sid->revision);
        for (i = 0; i < 6; i++)
        {
            if (sid->idAuth[i] > 0)
                printed += sySprintf(buff + printed, "%d", (NQ_INT)sid->idAuth[i]);
        }
        printed = (NQ_INT)syStrlen(buff);
        p = buff + printed;
        for (printed = 0, i = 0; i < (NQ_INT)sid->numAuths; i++, p += printed)
        {
            NQ_CHAR *t;

            if (sid->subs[i] == 0)
                sySprintf(p, "-0");
            else
                sySprintf(p, "-%-10.0u", (NQ_UINT)sid->subs[i]);
            if (NULL != (t = syStrchr(p, ' ')))
                *t = 0;
            printed = (NQ_INT)syStrlen(p);
        }
        LOGMSG(CM_TRC_LEVEL_SD, "%s: %s", title, buff);

        pResult = (NQ_BYTE *)&sid->subs[i];
    }
    return pResult;
#endif /* CM_SD_USEOLDVERSIONDEBUG */
}

static CMSdAce *dumpAce(const CMSdAce *ace)
{
    NQ_BYTE *next;

    LOGMSG(CM_TRC_LEVEL_SD, "ACE type: %d, flags: 0x%02X, size: %d, access: 0x%08X", (NQ_INT)ace->type, (NQ_INT)ace->flags, ace->size, ace->accessMask);
    next = dumpDomainSid("trustee: ", &ace->trustee);
    /*syPrintf("\n");*/

    return (CMSdAce *)next;
}

static void dumpAcl(const NQ_CHAR *title, NQ_BYTE *buffer, NQ_INT length, NQ_INT offset)
{
    if (offset > 0)
    {
        CMSdAcl *acl = (CMSdAcl *)(buffer + offset);
        CMSdAce *ace = (CMSdAce *)(acl + 1);
        NQ_UINT32 i;

        LOGMSG(CM_TRC_LEVEL_SD, "%s: offset=%d, ACL revision: %d, size: %d, numAces: %d", title, offset, acl->revision, acl->size, acl->numAces);
        for (i = 0; i < acl->numAces; i++)
            ace = dumpAce(ace);
    }
    else
        LOGMSG(CM_TRC_LEVEL_SD, "%s: none", title);
}

NQ_CHAR *dumpSdType(NQ_UINT16 type)
{
    static NQ_CHAR buffer[16 * 60] = {0};

    buffer[0] = '\0';

    if (type & CM_SD_OWNERDEFAULTED)
        syStrcat(buffer, "Owner Defaulted, ");
    if (type & CM_SD_GROUPDEFAULTED)
        syStrcat(buffer, "Group Defaulted, ");
    if (type & CM_SD_DACLPRESENT)
        syStrcat(buffer, "DACL Present, ");
    if (type & CM_SD_DACLDEFAULTED)
        syStrcat(buffer, "DACL Defaulted, ");
    if (type & CM_SD_SACLPRESENT)
        syStrcat(buffer, "SACL Present, ");
    if (type & CM_SD_SACLDEFAULTED)
        syStrcat(buffer, "SACL Defaulted, ");
    if (type & CM_SD_DACLTRUSTED)
        syStrcat(buffer, "DACL Trusted, ");
    if (type & CM_SD_SERVER_SECURITY)
        syStrcat(buffer, "Server Security, ");
    if (type & CM_SD_DACLAUTOINHERITREQ)
        syStrcat(buffer, "DACL Auto Inherit Required, ");
    if (type & CM_SD_SACLAUTOINHERITREQ)
        syStrcat(buffer, "SACL Auto Inherit Required, ");
    if (type & CM_SD_DACLAUTOINHERITED)
        syStrcat(buffer, "DACL Auto Inherited, ");
    if (type & CM_SD_SACLAUTOINHERITED)
        syStrcat(buffer, "SACL Auto Inherited, ");
    if (type & CM_SD_DACLPROTECTED)
        syStrcat(buffer, "DACL Protected, ");
    if (type & CM_SD_SACLPROTECTED)
        syStrcat(buffer, "SACL Protected, ");
    if (type & CM_SD_RM_CONTROLVALID)
        syStrcat(buffer, "RM Control Valid, ");
    if (type & CM_SD_SELF_RELATIVE)
        syStrcat(buffer, "Self Relative");

    return buffer;
}

void cmSdDumpDomainSid(const NQ_CHAR *title, const CMSdDomainSid *sid)
{
    dumpDomainSid(title, sid);
}

void cmSdDumpAccessToken(const CMSdAccessToken *token)
{
    NQ_COUNT i;

    LOGFB(CM_TRC_LEVEL_SD);

#ifdef CM_SD_USEOLDVERSIONDEBUG
    syPrintf("\n=== Access token\n");
    dumpSid("Domain SID", (NQ_BYTE *)&token->domain, sizeof(CMSdDomainSid), 0);
    syPrintf("\n  %d rids:", token->numRids);
    for (i = 0; i < token->numRids; i++)
    {
        syPrintf("\n\t0x%x (%d)", token->rids[i], token->rids[i]);
    }

    syPrintf("\n");
#else /* CM_SD_USEOLDVERSIONDEBUG */
    if (NULL == token)
    {
        LOGMSG(CM_TRC_LEVEL_SD, "NULL token");
    }
    else
    {
        LOGMSG(CM_TRC_LEVEL_SD, "Access token %s:", token->isAnon ? "(anonymous)" : (token->isAdmin ? "(admin)" : ""));

        dumpDomainSid("Domain SID: ", &token->domain);
        LOGMSG(CM_TRC_LEVEL_SD, "%d rids:", token->numRids);
        for (i = 0; i < token->numRids; i++)
        {
            LOGMSG(CM_TRC_LEVEL_SD, "0x%x (%d %s)", token->rids[i], token->rids[i], i == 0 ? "user rid" : ridToStr(token->rids[i]));
        }
    }
#endif /* CM_SD_USEOLDVERSIONDEBUG */

    LOGFE(CM_TRC_LEVEL_SD);
}

void cmSdDumpSecurityDescriptor(const NQ_WCHAR *name, const CMSdSecurityDescriptor *sd, NQ_COUNT sdLen)
{
    CMSdSecurityDescriptorHeader *h = (CMSdSecurityDescriptorHeader *)sd->data;

    LOGFB(CM_TRC_LEVEL_SD);
    if (0 == sdLen || NULL == sd)
    {
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_SD, "Security descriptor for [%s]: length %d", cmWDump(name), sdLen);
    LOGMSG(CM_TRC_LEVEL_SD, "revision %d, type 0x%04x", h->revision, h->type);
    LOGMSG(CM_TRC_LEVEL_SD, "(%s)", dumpSdType(h->type));
    dumpSid("Owner SID", (NQ_BYTE *)h, (NQ_INT)sdLen, (NQ_INT)h->ownerSid);
    dumpSid("Group SID", (NQ_BYTE *)h, (NQ_INT)sdLen, (NQ_INT)h->groupSid);
    dumpAcl("SACL", (NQ_BYTE *)h, (NQ_INT)sdLen, (NQ_INT)h->sacl);
    dumpAcl("DACL", (NQ_BYTE *)h, (NQ_INT)sdLen, (NQ_INT)h->dacl);

Exit:
    LOGFE(CM_TRC_LEVEL_SD);
}

static NQ_CHAR *ridToStr(CMSdRid rid)
{
    switch (rid)
    {
        case CM_SD_RIDGROUPPOLICYCREATORS:
            return "CM_SD_RIDGROUPPOLICYCREATORS";
        case CM_SD_RIDADMINISTRATOR:
            return "CM_SD_RIDADMINISTRATOR";
        case CM_SD_RIDGUEST:
            return "CM_SD_RIDGUEST";
        case CM_SD_RIDGROUPADMINS:
            return "CM_SD_RIDGROUPADMINS";
        case CM_SD_RIDGROUPUSERS:
            return "CM_SD_RIDGROUPUSERS";
        case CM_SD_RIDGROUPGUESTS:
            return "CM_SD_RIDGROUPGUESTS";
        case CM_SD_RIDALIASADMIN:
            return "CM_SD_RIDALIASADMIN";
        case CM_SD_RIDALIASUSER:
            return "CM_SD_RIDALIASUSER";
        case CM_SD_RIDALIASGUEST:
            return "CM_SD_RIDALIASGUEST";
        case CM_SD_RIDALIASACCOUNTOP:
            return "CM_SD_RIDALIASACCOUNTOP";
        case CM_SD_RIDGROUPDOMAINCOMPUTERS:
            return "CM_SD_RIDGROUPDOMAINCOMPUTERS";
        default:
            return "unknown";
    }
}

#endif /* UD_NQ_INCLUDETRACE */
#endif /* defined(UD_CS_INCLUDESECURITYDESCRIPTORS) || defined(UD_CC_INCLUDESECURITYDESCRIPTORS) || defined (UD_CC_INCLUDEDOMAINMEMBERSHIP) || defined(UD_CS_INCLUDEPASSTHROUGH)*/

