/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common resolver operations
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 *************************************************************************/

#include "cmapi.h"
#include "cmresolver.h"
#include "cmlist.h"
#include "cmcommon.h"
#include "nsdns.h"
#ifdef UD_CM_INCLUDEWSDCLIENT
#include "wsdclient.h"
#endif /* UD_CM_INCLUDEWSDCLIENT */

/* -- Definitions -- */

#ifdef UD_CM_RESOLVERCACHENAMETTL
#define CACHEITEM_TIMEOUT   UD_CM_RESOLVERCACHENAMETTL    /* in seconds */
#else
#define CACHEITEM_TIMEOUT 10 /* in seconds */
#endif /* UD_CM_RESOLVERCACHENAMETTL */


typedef struct 
{
    CMItem item;                        /* inherited object */
    SYSocketHandle socket;              /* socket to use with this method */
    void * context;                     /* method-specific context */
    NQ_STATUS status;                   /* method status as:
                                            NQ_ERR_MOREDATA - use this method (method was initially scheduled or it requires more iterations)
                                            NQ_SUCCESS - do not use this method (either succeeded or requests in progress)
                                            NQ_ERR_<*> - method failed - do not use 
                                        */
    NQ_IPADDRESS serverIp;              /* server IP address */
    CMResolverMethodDescriptor method;  /* method descriptor */
    NQ_BOOL enabled;                    /* TRUE when this method is enabled */
} 
Method;                                 /* resolution method */

typedef struct
{
    CMItem item;                        /* inherited object */
    NQ_UINT32 time;                     /* time when this entry was cached */
    NQ_COUNT numIps;                    /* number of IPs */
    const NQ_IPADDRESS * ips;           /* IP addresses */
}
CacheEntry;                             /* an association between host name and IP(s) */

/* -- Static data -- */

typedef struct
{
    SYMutex guard;                  /* critical section guard */
    CMList methods;                 /* list of registered methods */
    CMList cache;                   /* cached associations */
    NQ_BOOL cacheEnabled;           /* cache state (enabled by default) */
    NQ_UINT32 cacheTimeout;         /* cache item timeout */
    CMResolverNameToIpA nameToIpA;  /* name-to-ip external method (ASCII) */
    CMResolverIpToNameA ipToNameA;  /* ip-to-name external method (ASCII) */
    CMResolverNameToIp nameToIpW;   /* name-to-ip external method (UNICODE) */
    CMResolverIpToName ipToNameW;   /* ip-to-name external method (UNICODE) */
}
StaticData;

static NQ_BOOL isModuleInitialized = FALSE;
#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/* -- Static functions -- */

static NQ_BOOL methodUnlockCallback(CMItem * pItem)
{
#ifndef UD_NQ_CLOSESOCKETS
    Method * pMethod = (Method *)pItem; /* casted pointer */

    syCloseSocket(pMethod->socket);
#endif

    cmListItemRemoveAndDispose(pItem);
    return TRUE;
}

#ifdef UD_NQ_INCLUDETRACE

static NQ_CHAR *methodType(NQ_INT type)
{
    switch (type)
    {
    case NQ_RESOLVER_DNS:
        return "NQ_RESOLVER_DNS";
    case NQ_RESOLVER_NETBIOS:
        return "NQ_RESOLVER_NETBIOS";
    case NQ_RESOLVER_EXTERNAL_METHOD:
        return "NQ_RESOLVER_EXTERNAL_METHOD";
    case NQ_RESOLVER_DNS_DC:
        return "NQ_RESOLVER_DNS_DC";
    case NQ_RESOLVER_NETBIOS_DC:
        return "NQ_RESOLVER_NETBIOS_DC";
    case NQ_RESOLVER_WSD:
        return "NQ_RESOLVER_WSD";
    default:
        return "error";
    }
}

static void dumpMethods()
{
    CMIterator iterator;        /* method iterator */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator);  )
    {
        Method * pMethod = (Method *)cmListIteratorNext(&iterator);     /* casted pointer */

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method: %s, serverIP: %s, type: %s, multicast: %s, activationPriority: %d, socket: %d, status: 0x%x %s", 
            pMethod->enabled ? "enabled" : "disabled",
            cmIPDump(&pMethod->serverIp), methodType(pMethod->method.type), 
            pMethod->method.isMulticast ? "YES" : "NO", pMethod->method.activationPriority, pMethod->socket,
            pMethod->status, pMethod->status == NQ_ERR_MOREDATA ? "NQ_ERR_MOREDATA" : ((pMethod->status == NQ_SUCCESS) ? "NQ_SUCCESS" : ""));
    }
    cmListIteratorTerminate(&iterator);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void cmDumpIPsList(const NQ_IPADDRESS *pList, NQ_INT numOfIPs)
{
    const NQ_IPADDRESS *pIP = pList;
    NQ_INT i;

    for (i = 0; i < numOfIPs; i++, pIP++)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "#%d: %s", i + 1, cmIPDump(pIP));
    }
}

void cmDumpNamesList(NQ_WCHAR *list)
{
    NQ_HANDLE parseHandle;
    NQ_WCHAR delimiter = cmWChar(CM_NQ_DELIMITER);
    NQ_WCHAR *currentString = NULL;

    parseHandle = cmSpStartParsing((void *)list, (void *)&delimiter, TRUE, (void *)&currentString, TRUE);
    if (NULL != parseHandle)
    {
        NQ_UINT i = 0;
        while (NULL != currentString)
        {
            LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "#%d: %s", i + 1, cmWDump(currentString));
            i++;
            currentString = cmSpGetNextString(parseHandle);
        }

        cmSpTerminateParsing(parseHandle);
    }
}

#endif /* UD_NQ_INCLUDETRACE */

#ifdef UD_CM_INCLUDEWSDCLIENT

static void freeContexts(NQ_INT methodType)
{
    CMIterator iterator;  /* method iterator */

    for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator);  )
    {
        Method * pMethod = (Method *)cmListIteratorNext(&iterator);

        if ((methodType == pMethod->method.type) && (NULL != pMethod->context))
        {
            cmMemoryFree(pMethod->context);
            pMethod->context = NULL;
        }
    }
    cmListIteratorTerminate(&iterator);
}

#endif /* UD_CM_INCLUDEWSDCLIENT */

static NQ_BOOL bindClientSocket(Method *pMethod)
{
    NQ_STATUS res = NQ_FAIL;                   /* operation result */
    SYSocketHandle socket = syInvalidSocket();
    NQ_UINT family = (NQ_UINT)CM_IPADDR_VERSION(pMethod->serverIp);   /* IP family */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "method:%p", pMethod);

    pMethod->socket = syInvalidSocket();
    socket = syCreateSocket(FALSE, family);
    if (FALSE == syIsValidSocket(socket))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error creating socket for method: %s", cmWDump(pMethod->item.name));
        goto Exit;
    }

#if defined(UD_NQ_USETRANSPORTIPV4) && defined(UD_NQ_USETRANSPORTIPV6)
    if (CM_IPADDR_IPV4 == family)
    {
        res = syBindSocket(socket, cmSelfipGetAnyIpVersion4(), 0, TRUE);
    }
    else
    {
        res = syBindSocket(socket, cmSelfipGetAnyIpVersion6(), 0, TRUE);
    }
#else /* defined(UD_NQ_USETRANSPORTIPV4) && defined(UD_NQ_USETRANSPORTIPV6) */
#ifdef UD_NQ_USETRANSPORTIPV4
    res = syBindSocket(socket, cmSelfipGetAnyIpVersion4(), 0, TRUE);
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
    res = syBindSocket(socket, cmSelfipGetAnyIpVersion6(), 0, TRUE);
#endif /* UD_NQ_USETRANSPORTIPV6 */
#ifdef UD_NQ_USETRANSPORTNETBIOS
    if (NQ_FAIL == res)
    {
        res = syBindSocket(socket, cmSelfipGetAnyIpVersion4(), 0, TRUE);
    }
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#endif /* defined(UD_NQ_USETRANSPORTIPV4) && defined(UD_NQ_USETRANSPORTIPV6) */
    if (NQ_FAIL == res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error binding socket");
        syCloseSocket(socket);
        goto Exit;
    }

    pMethod->socket = socket;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return (NQ_FAIL == res) ? FALSE : TRUE;
}

static void releaseMethods(const NQ_WCHAR * dnsList)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "dnsList:%p", dnsList);

#ifdef UD_CS_INCLUDEMULTITENANCY
    /* remove methods with provided DNS ips
       enable original methods */
    if (dnsList)
    {
        CMIterator iterator;        /* method iterator */

        for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator);  )
        {
            Method * pMethod = (Method *)cmListIteratorNext(&iterator);     /* casted pointer */

            if (pMethod->enabled)
            {
                cmListItemCheck((CMItem *)pMethod);
            }
            else
            {
                pMethod->enabled = TRUE;
            }
        }
        cmListIteratorTerminate(&iterator);
    }
#endif /* UD_CS_INCLUDEMULTITENANCY */

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

static void prepareMethods(const NQ_WCHAR * dnsList)
{
    CMIterator iterator;        /* method iterator */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "dnsList:%s", dnsList ? cmWDump(dnsList) : "");

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)

    /* register and prepare methods with provided DNS ips */
    if (dnsList)
    {
        /* disable all existing methods */
        for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator);  )
        {
            Method * pMethod = (Method *)cmListIteratorNext(&iterator);     /* casted pointer */
            pMethod->enabled = FALSE;
        }
        cmListIteratorTerminate(&iterator);

        /* register new methods with provided DNS ips */
        if (0 == nsDnsRegisterMethods(dnsList))
        {
            for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator);  )
            {
                Method * pMethod = (Method *)cmListIteratorNext(&iterator);     /* casted pointer */
                pMethod->enabled = TRUE;
            }
            cmListIteratorTerminate(&iterator);
            goto Exit;
        }
    }
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */

    /* prepare new methods */
    for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator);  )
    {
        Method * pMethod = (Method *)cmListIteratorNext(&iterator);     /* casted pointer */
#ifndef UD_NQ_CLOSESOCKETS
        SYSocketSet set;                                                /* socket set for select */
        NQ_BOOL exit = FALSE;                                           /* clean socket indicator */
        NQ_BYTE dummyBuf[2];                                            /* dummy buffer */
        NQ_IPADDRESS dummySrc;                                          /* dummy source address */
        NQ_PORT dummyPort;                                              /* dummy source port */
#endif /* UD_NQ_CLOSESOCKETS */

        if (pMethod->enabled)
        {
            pMethod->status = NQ_ERR_MOREDATA;      /* initial status */
            pMethod->context = NULL;

#ifdef UD_NQ_CLOSESOCKETS
            bindClientSocket(pMethod);
#else
            /* cleanup sockets since they may have pending datagrams from previous requests */
            syClearSocketSet(&set);
            syAddSocketToSet(pMethod->socket, &set);

            while (!exit)
            {
                switch (sySelectSocket(&set, 0))      /* will hit data only when it is already on the socket */
                {
                case 0:
                    exit = TRUE;        /* no more */
                    break;
                case NQ_FAIL:
                    LOGERR(CM_TRC_LEVEL_ERROR, "Select error");
                    exit = TRUE;        /* no more */
                    break;
                default:
                    if (NQ_FAIL == syRecvFromSocket(pMethod->socket, dummyBuf, sizeof(dummyBuf), &dummySrc, &dummyPort))
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "recvfrom error");
                        exit = TRUE;        /* no more */
                    }
                    break;
                }
            }
#endif /* UD_NQ_CLOSESOCKETS */
        }
    }
    cmListIteratorTerminate(&iterator);

#if defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6)
Exit:
#endif /* defined(UD_NQ_USETRANSPORTIPV4) || defined(UD_NQ_USETRANSPORTIPV6) */
#ifdef UD_NQ_INCLUDETRACE
    dumpMethods();
#endif /* UD_NQ_INCLUDETRACE */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

#ifdef UD_NQ_CLOSESOCKETS
static void closeSockets()
{
    CMIterator iterator;        /* method iterator */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    /* close method sockets */
    for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
    {
        Method * pMethod = (Method *)cmListIteratorNext(&iterator);     /* casted pointer */
        if(syIsValidSocket(pMethod->socket))
        {
            syCloseSocket(pMethod->socket);
            pMethod->socket = syInvalidSocket();
        }
    }
    cmListIteratorTerminate(&iterator);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}
#endif /* UD_NQ_CLOSESOCKETS */

static NQ_BOOL mergeIpAddresses(const NQ_IPADDRESS ** pTo, NQ_INT * pNumTo, NQ_IPADDRESS ** from, NQ_INT numFrom)
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "num IPs: %d, num new IPs: %d", *pNumTo, numFrom);

    if (NULL == *pTo)
    {
        *pNumTo = numFrom;
        *pTo = (const NQ_IPADDRESS *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_IPADDRESS) * (NQ_UINT)numFrom));
        if (NULL == *pTo)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            goto Exit;
        }
        syMemcpy(*pTo, *from, sizeof(NQ_IPADDRESS) * (NQ_UINT)numFrom);
    }
    else
    {   
        NQ_IPADDRESS * newArray;                        /* new array of IPs */
        NQ_INT iTo;                                     /* just a counter */
        NQ_INT iFrom;                                   /* just a counter */
        const NQ_IPADDRESS zeroIp = CM_IPADDR_ZERO;     /* non-existent IP */
        NQ_INT numToAdd = numFrom;                      /* num of addresses to add after resolving duplicates */    

        /* check duplicate addresses */
        for (iFrom = 0; iFrom < numFrom; iFrom++)
        {
            for (iTo = 0; iTo < *pNumTo; iTo++)
            {
                if (CM_IPADDR_EQUAL((*pTo)[iTo], (*from)[iFrom]))
                {
                    (*from)[iFrom] = zeroIp;
                    numToAdd--;
                    break;
                }
            }
        }

        if (numToAdd > 0)
        {
            newArray = (NQ_IPADDRESS *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_IPADDRESS) * (NQ_UINT)(*pNumTo + numToAdd)));
            if (NULL == newArray)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                goto Exit;
            }
            syMemcpy(newArray, *pTo, sizeof(NQ_IPADDRESS) * (NQ_UINT)(*pNumTo));
            iTo = *pNumTo;
            for (iFrom = 0; iFrom < numFrom; iFrom++)
            {
                if (!CM_IPADDR_EQUAL(zeroIp, (*from)[iFrom]))
                {
                    newArray[iTo++] = (*from)[iFrom];
                }
            }
            *pNumTo += numToAdd;
            cmMemoryFree(*pTo);
            *pTo = newArray;
        }
    }
    result = TRUE;
Exit:
    cmMemoryFree(*from);
    *from = NULL;
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "num non recurring IPs: %d", *pNumTo);
    return result;
}

#ifdef UD_CM_INCLUDEWSDCLIENT
static NQ_BOOL mergeNames(NQ_WCHAR ** to, NQ_INT * numTo, const NQ_WCHAR * from, const NQ_INT numFrom)
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "to:%p, numTo:%p, from:%p, numFrom:%d", to, numTo, from, numFrom);

    if (0 == numFrom)
    {
        goto Exit;
    }

    if (NULL == *to)
    {
        *to = cmMemoryCloneWString(from);
        if (NULL == *to)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            goto Exit;
        }

        *numTo = numFrom;
    }
    else
    {
        NQ_HANDLE fromHandle;
        NQ_WCHAR *currentFrom = NULL;
        NQ_WCHAR delimiter = cmWChar(CM_NQ_DELIMITER);

        /* for each name in 'from' (new list) check for duplicate in 'to' (old list) */
        fromHandle = cmSpStartParsing((void *)from, (void *)&delimiter, TRUE, (void *)&currentFrom, TRUE);
        if (NULL == fromHandle)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to start parsing");
            goto Exit;
        }

        while(NULL != currentFrom)
        {
            NQ_HANDLE toHandle;
            NQ_WCHAR  *currentTo = NULL;
            NQ_BOOL   isDuplicate = FALSE;

            toHandle = cmSpStartParsing((void *)*to, (void *)&delimiter, TRUE, (void *)&currentTo, TRUE);
            if (NULL == toHandle)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to start parsing");
                cmSpTerminateParsing(fromHandle);
                goto Exit;
            }

            while(NULL != currentTo)
            {
                if (0 == cmWStricmp(currentTo, currentFrom))
                {
                    isDuplicate = TRUE;
                    break;
                }

                currentTo = cmSpGetNextString(toHandle);
            }

            cmSpTerminateParsing(toHandle);
            if (FALSE == isDuplicate)
            {
                NQ_WCHAR * newList = NULL;

                /* allocate new list */
                newList = (NQ_WCHAR *)cmMemoryAllocate(((NQ_UINT)sizeof(NQ_WCHAR)) * (cmWStrlen(*to) + cmWStrlen(currentFrom) + CM_TRAILING_NULL) + (NQ_UINT)sizeof(delimiter));
                if (NULL == newList)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    cmSpTerminateParsing(fromHandle);
                    goto Exit;
                }

                /* copy original */
                cmWStrcpy(newList, *to);

                /* add delimiter */
                newList[cmWStrlen(newList)] = delimiter;

                /* add new string to list */
                cmWStrcat(newList, currentFrom);

                /* release old list and replace */
                cmMemoryFree(*to);
                *to = newList;
                (*numTo)++;
            }

            currentFrom = cmSpGetNextString(fromHandle);
        }

        cmSpTerminateParsing(fromHandle);
    }

    cmMemoryFree(from);
    from = NULL;

#ifdef UD_NQ_INCLUDETRACE
    cmDumpNamesList(*to);
#endif /* UD_NQ_INCLUDETRACE */

    result = TRUE;
Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}
#endif /* UD_CM_INCLUDEWSDCLIENT */

static const NQ_IPADDRESS * externalNameToIp(const NQ_WCHAR * name, NQ_INT * numIps)
{
    NQ_IPADDRESS nextIp = CM_IPADDR_ZERO;                /* next IP in the result */
    NQ_IPADDRESS * pNext = NULL;               /* pointer to a copy of next IP */
    NQ_COUNT index;                     /* IP index */
    const NQ_IPADDRESS * ips = NULL;    /* resulted array of IPs */
#ifdef UD_NQ_USETRANSPORTIPV4
    NQ_IPADDRESS4 *ip;                  /* pointer to IP */
#endif /* UD_NQ_USETRANSPORTIPV4 */
    NQ_INT type;                        /* next IP type */
    NQ_BYTE buffer[16];                 /* buffer for IP */

    if (NULL != staticData->nameToIpW)
    {
        for (index = 0; ; index++)
        {
            type = staticData->nameToIpW(name, buffer, index);
            switch(type)
            {
#ifdef UD_NQ_USETRANSPORTIPV4
            case NQ_RESOLVER_IPV4:
                ip = (NQ_IPADDRESS4 *)buffer;
                CM_IPADDR_ASSIGN4(nextIp, (*ip));
                break;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
            case NQ_RESOLVER_IPV6:
                CM_IPADDR_ASSIGN6(nextIp, (buffer));
                break;
#endif /* UD_NQ_USETRANSPORTIPV6 */
            default:
                type = NQ_RESOLVER_NONE;
            }
            if (NQ_RESOLVER_NONE == type)
                break;
            {
                pNext = (NQ_IPADDRESS *)cmMemoryAllocate(sizeof(NQ_IPADDRESS));
                if (NULL != pNext)
                {
                    *pNext = nextIp;
                    mergeIpAddresses(&ips, numIps, &pNext, 1);   /* will release pNext */
                }
            }
        }
    }
    else if (NULL != staticData->nameToIpA)
    {
        const NQ_CHAR * nameA;      /* ASCII name */
    
        nameA = cmMemoryCloneWStringAsAscii(name);
        if (NULL != nameA)
        {
            for (index = 0; ; index++)
            {
                type = staticData->nameToIpA(nameA, buffer, index);
                switch(type)
                {
#ifdef UD_NQ_USETRANSPORTIPV4
                case NQ_RESOLVER_IPV4:
                    ip = (NQ_IPADDRESS4 *)buffer;
                    CM_IPADDR_ASSIGN4(nextIp, (*ip));
                    break;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
                case NQ_RESOLVER_IPV6:
                    CM_IPADDR_ASSIGN6(nextIp, (buffer));
                    break;
#endif /* UD_NQ_USETRANSPORTIPV6 */
                default:
                    type = NQ_RESOLVER_NONE;
                }
                if (NQ_RESOLVER_NONE == type)
                    break;
                pNext = (NQ_IPADDRESS *)cmMemoryAllocate(sizeof(NQ_IPADDRESS));
                if (NULL != pNext)
                {
                    *pNext = nextIp;
                    mergeIpAddresses(&ips, numIps, &pNext, 1);   /* will release pNext */
                }
            }
        }
        cmMemoryFree(nameA);
    }

    return ips;
}

static const NQ_WCHAR * externalIpToName(const NQ_IPADDRESS * ip)
{
    NQ_WCHAR * name = NULL;     /* resulted name */
    NQ_BOOL result = FALSE;     /* resolution result */
    NQ_INT version;             /* IP version */

    name = (NQ_WCHAR *)cmMemoryAllocate(CM_DNS_NAMELEN);
    if (NULL == name)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    version = CM_IPADDR_VERSION(*ip);
    if (NULL != staticData->ipToNameW)
    {
        switch(version)
        {
#ifdef UD_NQ_USETRANSPORTIPV4
        case CM_IPADDR_IPV4:
            result = staticData->ipToNameW(name, &(CM_IPADDR_GET4(*ip)), NQ_RESOLVER_IPV4);
            break;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
        case CM_IPADDR_IPV6:
            result = staticData->ipToNameW(name, &CM_IPADDR_GET6(*ip), NQ_RESOLVER_IPV6);
            break;
#endif /* UD_NQ_USETRANSPORTIPV6 */
#ifdef UD_NQ_USETRANSPORTIPVR
        case CM_IPADDR_IPVR:
        {
            NQ_IPADDRESS ipvr;

            if (CM_IPADDR_IPV6 == ip->addr.vr.version)
            {
                CM_IPADDR_ASSIGN6(ipvr, ip->addr.vr.addr.v6)
            }
            else
            {
                CM_IPADDR_ASSIGN4(ipvr, ip->addr.vr.addr.v4)
            }

            cmMemoryFree(name);
            name = (NQ_WCHAR *)externalIpToName((const NQ_IPADDRESS *)&ipvr);
            goto Exit;
        }
#endif /* UD_NQ_USETRANSPORTIPVR */
        default:
            break;
        }
    }
    else if (NULL != staticData->ipToNameA)
    {
        NQ_CHAR * nameA;    /* pointer to ASCII name */

        switch(version)
        {
#ifdef UD_NQ_USETRANSPORTIPV4
        case CM_IPADDR_IPV4:
            result = staticData->ipToNameA((NQ_CHAR *)name, &(CM_IPADDR_GET4(*ip)), NQ_RESOLVER_IPV4);
            break;
#endif /* UD_NQ_USETRANSPORTIPV4 */
#ifdef UD_NQ_USETRANSPORTIPV6
        case CM_IPADDR_IPV6:
            result = staticData->ipToNameA((NQ_CHAR *)name, &(CM_IPADDR_GET6(*ip)), NQ_RESOLVER_IPV6);
            break;
#endif /* UD_NQ_USETRANSPORTIPV6 */
#ifdef UD_NQ_USETRANSPORTIPVR
        case CM_IPADDR_IPVR:
        {
            NQ_IPADDRESS ipvr;

            if (CM_IPADDR_IPV6 == ip->addr.vr.version)
            {
                CM_IPADDR_ASSIGN6(ipvr, ip->addr.vr.addr.v6)
            }
            else
            {
                CM_IPADDR_ASSIGN4(ipvr, ip->addr.vr.addr.v4)
            }

            cmMemoryFree(name);
            name = (NQ_WCHAR *)externalIpToName((const NQ_IPADDRESS *)&ipvr);
            goto Exit;
        }
#endif /* UD_NQ_USETRANSPORTIPVR */
        default:
            break;
        }
        if (result)
        {
            nameA = (NQ_CHAR *)name;
            name = cmMemoryCloneAString(nameA); /* NULL is OK */
            if (NULL == name)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                goto Exit;
            }
            cmMemoryFree(nameA);
        }
    }
    if (!result)
    {
        cmMemoryFree(name);
        name = NULL;
    }

Exit:
    return name;
}

static void validateCache(NQ_UINT32 timeout)
{
    CMIterator iterator;            /* for iterating cache items */
    NQ_UINT32 curTime;                /* current time in seconds */

    curTime  = (NQ_UINT32)syGetTimeInSec();
    cmListIteratorStart(&staticData->cache, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        const CacheEntry * pEntry;   /* next cache entry */
        
        pEntry = (const CacheEntry *)cmListIteratorNext(&iterator);
        if ((timeout == 0) || (timeout < (curTime - pEntry->time)))
        {
            cmListItemCheck((CMItem *)pEntry);
            cmListItemRemoveAndDispose((CMItem *)pEntry);
        }
    }

    cmListIteratorTerminate(&iterator);
}

static const CacheEntry * lookupNameInCache(const NQ_WCHAR * name)
{
    validateCache(staticData->cacheTimeout);
    return staticData->cacheEnabled ? (const CacheEntry *)cmListItemFind(&staticData->cache, name, TRUE, FALSE) : NULL;
}

static const CacheEntry * lookupIpInCache(const NQ_IPADDRESS * ip)
{
    CMIterator iterator;               /* for iterating cache items */
    const CacheEntry * pEntry = NULL;  /* next cache entry */

    validateCache(staticData->cacheTimeout);
    if (FALSE == staticData->cacheEnabled)
    {
        goto Exit1;
    }

    cmListIteratorStart(&staticData->cache, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        NQ_COUNT i;                 /* index in IPs */

        pEntry = (const CacheEntry *)cmListIteratorNext(&iterator);
        for (i = 0; i < pEntry->numIps; i++)
        {
            if (CM_IPADDR_EQUAL(*ip, pEntry->ips[i]))
            {
                goto Exit;
            }
        }
    }

    pEntry = NULL;

Exit:
    cmListIteratorTerminate(&iterator);
Exit1:
    return pEntry;
}

/*
 * Callback for cache entry unlock and disposal:
 *  - disconnects from the server
 *  - disposes private data  
 */
static NQ_BOOL cacheEntryUnlockCallback(CMItem * pItem)
{
    const CacheEntry * pEntry = (const CacheEntry *)pItem;  /* casted pointer */
    if (NULL != pEntry->ips)
    {
        cmMemoryFree(pEntry->ips);
    }

    return FALSE;
}


static void addToCache(const NQ_WCHAR * name, const NQ_IPADDRESS * ips, NQ_COUNT numIps)
{
    CacheEntry * pEntry;        /* new cache entry */

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "name:%s, ips:%p, numIps:%d", cmWDump(name), ips, numIps);

    if (FALSE == staticData->cacheEnabled)
    {
        goto Exit;
    }

    if ((NULL == name) || (NULL == ips))
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid parameter");
        goto Exit;
    }

    pEntry = (CacheEntry *)cmListItemCreateAndAdd(&staticData->cache, sizeof(CacheEntry), name, cacheEntryUnlockCallback, CM_LISTITEM_NOLOCK , FALSE);
    if (NULL == pEntry)
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    pEntry->ips = (const NQ_IPADDRESS *)cmMemoryAllocate((NQ_UINT)(numIps * sizeof(NQ_IPADDRESS)));
    if (NULL == pEntry->ips)
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        cmListItemRemoveAndDispose((CMItem *)pEntry);
        goto Exit;
    }

    syMemcpy(pEntry->ips, ips, numIps * sizeof(NQ_IPADDRESS));
    pEntry->numIps = numIps;
    pEntry->time = (NQ_UINT32)syGetTimeInSec();

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
    return;
}

/* -- API Functions */

NQ_BOOL cmResolverStart(void)
{
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocateStartup(sizeof(*staticData));
    if (NULL == staticData)
    {
        result = FALSE;
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate resolver data");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    staticData->ipToNameA = NULL;
    staticData->ipToNameW = NULL;
    staticData->nameToIpA = NULL;
    staticData->nameToIpW = NULL;
    syMutexCreate(&staticData->guard);
    staticData->cacheEnabled = TRUE;
    staticData->cacheTimeout = CACHEITEM_TIMEOUT;
    cmListStart(&staticData->methods);
    cmListStart(&staticData->cache);
    isModuleInitialized = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return result;
}

void cmResolverShutdown(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        CMIterator  itr;

        cmListIteratorStart(&staticData->cache, &itr);
        while (cmListIteratorHasNext(&itr))
        {
            CacheEntry * pEntry;

            pEntry = (CacheEntry *)cmListIteratorNext(&itr);
            cmListItemCheck((CMItem *)pEntry);
        }
        cmListIteratorTerminate(&itr);

        cmListIteratorStart(&staticData->methods, &itr);
        while (cmListIteratorHasNext(&itr))
        {
            CMItem * pItem;

            pItem = cmListIteratorNext(&itr);
            cmListItemCheck(pItem);
        }
        cmListIteratorTerminate(&itr);

        cmListShutdown(&staticData->methods);
        cmListShutdown(&staticData->cache);
        syMutexDelete(&staticData->guard);

#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
        {
            cmMemoryFreeShutdown(staticData);
        }
        staticData = NULL;
#endif /* SY_FORCEALLOCATION */
        isModuleInitialized = FALSE;
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void cmResolverRemoveFromCache(const NQ_WCHAR * name)
{
    CacheEntry * pEntry;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "name:%s", cmWDump(name));

    if (NULL == name)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid parameter");
        goto Exit;
    }

    pEntry = (CacheEntry *)cmListItemFind(&staticData->cache, name, TRUE, FALSE);
    if (NULL == pEntry)
    {
        /* name not found in cache, nothing to do */
        goto Exit;
    }
    else
    {
        cmListItemCheck((CMItem *)pEntry);
        cmListItemRemoveAndDispose((CMItem *)pEntry);
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL);
    return;
}

void cmResolverCacheSet(NQ_UINT32 timeout)
{
    if (NULL == staticData)
    {
        return;
    }

    syMutexTake(&staticData->guard);
    if ((timeout == 0) && staticData->cacheEnabled)
    {
        /* empty cache */
        validateCache(0);
    }
    staticData->cacheEnabled = (timeout != 0);
    staticData->cacheTimeout = timeout;
    syMutexGive(&staticData->guard);
}

NQ_BOOL cmResolverRegisterExternalMethod(const CMResolverRegisteredMethodDescription * pMethod)
{
    NQ_BOOL res = FALSE;
    CMResolverMethodDescriptor methodDescriptor;
    /* validations */
    /**************/

    if (NULL == pMethod)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (NULL == pMethod->serverIP)
    {
        LOGERR(CM_TRC_LEVEL_WARNING, "A registered method must add a server IP which the queries will be sent to.");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (NULL == pMethod->requestByName && NULL == pMethod->requestByIp)
    {
        LOGERR(CM_TRC_LEVEL_WARNING, "A registered method must support at least on query type. Query by name or query by IP.");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (NULL != pMethod->requestByIp && NULL == pMethod->responseByIp)
    {
        LOGERR(CM_TRC_LEVEL_WARNING, "Each registered query method must have a corresponding response method. Add a method that will handle your query results.");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (NULL != pMethod->requestByName && NULL == pMethod->responseByName)
    {
        LOGERR(CM_TRC_LEVEL_WARNING, "Each registered query method must have a corresponding response method. Add a method that will handle your query results.");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    /* register method */
    /*******************/
    methodDescriptor.type = NQ_RESOLVER_EXTERNAL_METHOD;
    methodDescriptor.isMulticast = FALSE;

    switch (pMethod->activationPriority)
    {
        case 1:
            methodDescriptor.activationPriority = 1;
            break;
        case 2:
            methodDescriptor.activationPriority = 3;
            break;
        default:
            methodDescriptor.activationPriority = 5;
            break;
    }

    methodDescriptor.timeout = cmTimeConvertSecToMSec(pMethod->timeout);
    methodDescriptor.waitAnyway = FALSE;

    methodDescriptor.requestByIp = pMethod->requestByIp;
    methodDescriptor.requestByName = pMethod->requestByName;
    methodDescriptor.responseByIp = pMethod->responseByIp;
    methodDescriptor.responseByName = pMethod->responseByName;

    res = TRUE;

    cmResolverRegisterMethod(&methodDescriptor, pMethod->serverIP);

Exit:
    return res;
}

NQ_BOOL cmResolverRegisterMethod(const CMResolverMethodDescriptor * descriptor, const NQ_IPADDRESS * serverIp)
{
    Method * pMethod;                                                                   /* method pointer */
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "descriptor:%p serverIP:%p %s", descriptor, serverIp, serverIp ? cmIPDump(serverIp) : "");

    pMethod = (Method *)cmListItemCreateAndAdd(&staticData->methods, sizeof(Method), NULL, methodUnlockCallback, CM_LISTITEM_NOLOCK , FALSE);
    if (NULL == pMethod)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    pMethod->method = *descriptor;
    pMethod->context = NULL;
    if (NULL != serverIp)
    {
        pMethod->serverIp = *serverIp;
    }
    else
    {
        CM_IPADDR_ASSIGN4(pMethod->serverIp, 0L);
    }

    /* In case UD_NQ_CLOSESOCKETS is defined call bindClientSocket to test the socket, then close it */
    if (bindClientSocket(pMethod) == FALSE)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error binding method socket");
        goto Error1;
    }

#ifdef UD_NQ_CLOSESOCKETS
    /* Close socket until needed, it will be opened later per resolver call */
    syCloseSocket(pMethod->socket);
    pMethod->socket = syInvalidSocket();
#endif /* UD_NQ_CLOSESOCKETS */

    pMethod->enabled = TRUE;                /* method is usable */
    pMethod->status = NQ_ERR_MOREDATA;      /* initial status */

    result = TRUE;
    goto Exit;


Error1:
    cmListItemRemoveAndDispose((CMItem *)pMethod);

Exit:
#ifdef UD_NQ_INCLUDETRACE
    dumpMethods();
#endif /* UD_NQ_INCLUDETRACE */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

void cmResolverRemoveMethod(const CMResolverMethodDescriptor * descriptor, const NQ_IPADDRESS * serverIp)
{
    CMIterator iterator;            /* method iterator */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "descriptor:%p serverIP:%p %s", descriptor, serverIp, serverIp ? cmIPDump(serverIp) : "");

    syMutexTake(&staticData->guard);

    for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
    {
        Method * pMethod = (Method *)cmListIteratorNext(&iterator);

        if (pMethod->method.type == descriptor->type && CM_IPADDR_EQUAL(pMethod->serverIp, *serverIp) && pMethod->method.isMulticast == descriptor->isMulticast)
        {
            cmListItemCheck((CMItem *)pMethod);
            break;
        }
    }
    cmListIteratorTerminate(&iterator);
#ifdef UD_NQ_INCLUDETRACE
    dumpMethods();
#endif /* UD_NQ_INCLUDETRACE */
    syMutexGive(&staticData->guard);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

const NQ_WCHAR * cmResolverGetHostName(const NQ_IPADDRESS * ip)
{
    NQ_TIME selectTimeStamp, timeDiff;  /* time when the current select started */
    NQ_TIME currentTimeout;             /* total timeout in milliseconds for current priority */
    NQ_TIME zero = {0, 0}, tmpTime;
    CMIterator iterator;                /* method iterator */
    const NQ_WCHAR * pName = NULL;      /* pointer to the resulted name */
    const CacheEntry * pEntry;          /* entry in the cache */
    NQ_COUNT numPendingMethods = 0;     /* number of requests sent */
    NQ_UINT priorityToActivate = 1;     /* each resolver method has activation priority. we start from 1 which is highest*/

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "ip:%p", ip);

    if (NULL == ip)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit1;
    }

    syMutexTake(&staticData->guard);

    /* look in cache */
    pEntry = lookupIpInCache(ip);
    if (NULL != pEntry)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Found in cache");

        pName = cmMemoryCloneWString(pEntry->item.name);
        if (NULL == pName)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        }
        goto Exit2;
    }

    prepareMethods(NULL);

    /* loop by the smallest timeout until either all timeouts expire or there is at least one answer */
    for (; ;)
    {
        SYSocketSet set;                /* socket set for select */
        SYSocketSet setCopy;            /* socket set for select (copy) */
        NQ_BOOL breakWhile = FALSE;     /* for while of select() */
        NQ_UINT32 selectTimeout;        /* timeout in seconds of the current select of current priority */

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Use priority: %d", priorityToActivate);

        syClearSocketSet(&set);
        cmU64Zero(&currentTimeout);

        for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
        {
            Method * pMethod = (Method *)cmListIteratorNext(&iterator);

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method: %s, serverIP: %s, type: %s, multicast: %s, activationPriority: %d, socket: %d, status: 0x%x %s", 
                    pMethod->enabled ? "enabled" : "disabled",
                    cmIPDump(&pMethod->serverIp), methodType(pMethod->method.type),
                    pMethod->method.isMulticast ? "YES" : "NO", pMethod->method.activationPriority, pMethod->socket,
                    pMethod->status, pMethod->status == NQ_ERR_MOREDATA ? "NQ_ERR_MOREDATA" : ((pMethod->status == NQ_SUCCESS) ? "NQ_SUCCESS" : ""));

            if (!pMethod->enabled || pMethod->method.type == NQ_RESOLVER_DNS_DC || pMethod->method.type == NQ_RESOLVER_NETBIOS_DC || pMethod->method.type == NQ_RESOLVER_WSD)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Skipping method");
                continue;
            }

            if ((NQ_ERR_MOREDATA == pMethod->status) && (priorityToActivate == pMethod->method.activationPriority))
            {
                NQ_COUNT numOfSentRequests;

                pMethod->status = (*pMethod->method.requestByIp)(pMethod->socket, ip, pMethod->context, &pMethod->serverIp, &numOfSentRequests);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "method status:0x%x, sent requests:%u", pMethod->status, numOfSentRequests);
                if (NQ_SUCCESS == pMethod->status)
                {
                    syAddSocketToSet(pMethod->socket, &set);
                    numPendingMethods += numOfSentRequests;

                    if (cmU64Cmp(&pMethod->method.timeout, &currentTimeout) > 0)
                    {
                        /* set to max timeout value per priority */
                        cmU64AssignU64(&currentTimeout, &pMethod->method.timeout);
                    }
                }
            }
        }
        cmListIteratorTerminate(&iterator);

        /* save original set */
        setCopy = set;

        /* set the select() timeout to be used for current priority */
        selectTimeout = cmTimeConvertMSecToSec(&currentTimeout);

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, pending: %d, priority timeout: %d", priorityToActivate, numPendingMethods, cmTimeConvertMSecToSec(&currentTimeout));

        selectTimeStamp = syGetTimeInMsec();
        while (numPendingMethods > 0)
        {
            NQ_INT selectResult;

            if (cmU64Cmp(&currentTimeout, &zero) <= 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Timeout for method priority: %d.", priorityToActivate);
                break;
            }

            selectResult = sySelectSocket(&set, selectTimeout);
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, select result: %d, pending: %d, select timeout (seconds): %d, priority timeout (seconds): %d",
                                              priorityToActivate, selectResult, numPendingMethods, selectTimeout, cmTimeConvertMSecToSec(&currentTimeout));
            switch (selectResult)
            {
            case 0:  /* timeout */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select timeout");
                breakWhile = TRUE;
                break;

            case NQ_FAIL: /* error the select failed  */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select failed");
                goto Error;

            default: /* datagram ready for reading */
                for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
                {
                    Method * pMethod = (Method *)cmListIteratorNext(&iterator);

                    if (!pMethod->enabled || pMethod->method.type == NQ_RESOLVER_DNS_DC || pMethod->method.type == NQ_RESOLVER_NETBIOS_DC || pMethod->method.type == NQ_RESOLVER_WSD)
                    {
                        continue;
                    }
                    if (syIsSocketSet(pMethod->socket, &set))
                    {
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method socket:%d", pMethod->socket);
                        syClearSocketFromSet(pMethod->socket, &set);
                        syAddSocketToSet(pMethod->socket, &set);        /* performed because more than 1 responses might arrive, but we treat one at a time */
                        --numPendingMethods;

                        pMethod->status = (*pMethod->method.responseByIp)(pMethod->socket, &pName, &pMethod->context);
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method status:%d", pMethod->status);
                        switch (pMethod->status)
                        {
                        case NQ_SUCCESS:
                            cmMemoryFree(pMethod->context); /* will handle NULL */
                            cmListIteratorTerminate(&iterator);
                            goto Exit3;
                        case NQ_ERR_MOREDATA:
                        case NQ_FAIL:
                            cmMemoryFree(pMethod->context); /* will handle NULL */
                            break;
                        default:    /* error */
                            break;
                        }
                    }
                }
                cmListIteratorTerminate(&iterator);
            } /* end of switch (selectResult) */

            /* prepare socket set for next sySelectSocket */
            set = setCopy;

            if (breakWhile)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Breaking while");
                break;
            }

            /* recalculate timeout */
            tmpTime = syGetTimeInMsec();
            cmU64SubU64U64(&timeDiff, &tmpTime, &selectTimeStamp);
            if (cmU64Cmp(&timeDiff, &zero) > 0)
            {
                if (cmU64Cmp(&timeDiff, &currentTimeout) > 0)
                {
                    currentTimeout = zero;
                }
                else
                {
                    NQ_TIME tmp = currentTimeout;

                    cmU64SubU64U64(&currentTimeout, &tmp, &timeDiff);
                }

                cmU64AddU64(&selectTimeStamp, &timeDiff);
            }
        } /*while (numPendingMethods > 0)*/

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "End of while (numPendingMethods > 0)");

        /* activate next priority methods */
        if (priorityToActivate < RESOLVER_MAX_ACTIVATION_PRIORITY)
        {
            ++priorityToActivate;
            continue;
        }
        LOGERR(CM_TRC_LEVEL_ERROR, "All methods failed (still no result and no more priorities to use)");
        goto Error;
    } /* end of for (; ;) */

Error:
    pName = (NULL == pName) ? externalIpToName(ip) : pName;
Exit3:
    addToCache(pName, ip, 1);
Exit2:
#ifdef UD_NQ_CLOSESOCKETS
    closeSockets();
#endif /* UD_NQ_CLOSESOCKETS */
    syMutexGive(&staticData->guard);
Exit1:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", cmWDump(pName));
    return pName;
}

static NQ_BOOL orderIPAddreses(const NQ_IPADDRESS ** resolvedIPs, NQ_COUNT numIPs)
{
   const CMSelfIp   * pAdapter;
   NQ_IPADDRESS *orderedIPs, *newArray; /* Same IP list. the IPs that are on same subnet as local addresses, will appear first */
   NQ_COUNT iOrdered = 0;   /* counter for new list of ordered IPS*/
   NQ_COUNT reverseIndx = 0; /* counter for order the sub list IPs , same network IPs first and broadcast after */
   NQ_BOOL res = FALSE;
   NQ_COUNT iResolved;  /* counter for list of resolved IPs before ordering */
   NQ_IPADDRESS tempIp;
   const NQ_IPADDRESS zeroIp = CM_IPADDR_ZERO;     /* non-existent IP */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "numIps: %d", numIPs);

    orderedIPs = (NQ_IPADDRESS *) cmMemoryAllocate((NQ_UINT)(sizeof (NQ_IPADDRESS) * numIPs));
    newArray = (NQ_IPADDRESS *) cmMemoryAllocate((NQ_UINT)(sizeof (NQ_IPADDRESS) * numIPs));

    if (NULL == orderedIPs || NULL == newArray)
    {
        if (orderedIPs)
        {
            cmMemoryFree(orderedIPs);
            orderedIPs = NULL;
        }
        goto Exit;
    }

   syMemcpy(newArray, *resolvedIPs, sizeof(NQ_IPADDRESS) * (NQ_UINT)numIPs);
   syMemset(orderedIPs, 0, sizeof(NQ_IPADDRESS) * (NQ_UINT)numIPs);

    for (cmSelfipIterate(); NULL != (pAdapter = cmSelfipNext()); )
    {
        if (CM_IPADDR_VERSION(pAdapter->ip) == CM_IPADDR_IPV4)
        {
            for (iResolved = 0; iResolved < numIPs; ++iResolved)
            {
               if (!CM_IPADDR_EQUAL( zeroIp, newArray[iResolved] ) &&
                    ( ( ( CM_IPADDR_GET4( newArray[iResolved] ) & pAdapter->bcast ) == CM_IPADDR_GET4( newArray[iResolved] ) ) || 
                      ( ( pAdapter->subnet & CM_IPADDR_GET4( pAdapter->ip ) ) == ( pAdapter->subnet & CM_IPADDR_GET4( newArray[iResolved] ) ) ) 
                    ) /*broadcast or self network*/
                  )/*if*/
              {
                   /* this IP is in same subnet as this adapter */
                   orderedIPs[iOrdered++] = newArray[iResolved];
                   newArray[iResolved] = zeroIp;
    
                   /* maybe this code is unnecessary ,  we don't find scenario that broadcast and subnet mask can indicate for different network ,
                   *  but just in case and for historical reasons we decide remain the broadcast condition in previous if condition and in this while loop
                   * if we decide to remove the broadcat condition the while loop not needed.
                   */
                   reverseIndx = iOrdered - 1 ;
                   while (    reverseIndx > 0 &&
                              (
                                  (pAdapter->subnet & CM_IPADDR_GET4(pAdapter->ip)) == (pAdapter->subnet & CM_IPADDR_GET4(orderedIPs[reverseIndx]))
                              )/*in self network*/ &&
                              (
                                  (CM_IPADDR_GET4(orderedIPs[reverseIndx-1]) & pAdapter->bcast) == CM_IPADDR_GET4(orderedIPs[reverseIndx-1]) &&
                                  !((pAdapter->subnet & CM_IPADDR_GET4(pAdapter->ip)) == (pAdapter->subnet & CM_IPADDR_GET4(orderedIPs[reverseIndx-1])))
                              )/*broadcast and not in self network */
                          )/*while*/
                   {
                      /* swap - move the IPs in local network before the broadcast IPs to the beginning of the list */
                      tempIp = orderedIPs[ reverseIndx-1 ];
                      orderedIPs[ reverseIndx-1 ] = orderedIPs[reverseIndx];
                      orderedIPs[ reverseIndx ] = tempIp;
                      reverseIndx--;
                   }
              }
            }
        }
    }
    cmSelfipTerminate();

    LOGMSG(CM_TRC_LEVEL_MESS_SOME, "Number of received IPs: %d, num in same subnets: %d", numIPs, iOrdered);

    iResolved = 0;
    while ((iOrdered < numIPs) && (iResolved < numIPs))
    {
        if (!CM_IPADDR_EQUAL(zeroIp, newArray[iResolved]))
            orderedIPs[iOrdered ++] = newArray[iResolved];

        ++iResolved;
    }

    res = TRUE;

Exit:
    cmMemoryFree(newArray);
    cmMemoryFree(*resolvedIPs);
    *resolvedIPs = orderedIPs;
    return res;
}

const NQ_IPADDRESS * cmResolverGetHostIps(const NQ_WCHAR * dnsList, const NQ_WCHAR * host, NQ_INT * numIps)
{
    NQ_TIME selectTimeStamp, timeDiff;          /* time when the current select started */
    NQ_TIME currentTimeout;                     /* total timeout in milliseconds for current priority */
    NQ_TIME zero = {0, 0}, tmpTime;
    CMIterator iterator;                        /* method iterator */
    const NQ_IPADDRESS * result = NULL;         /* resulted array of IPs */
    NQ_IPADDRESS * methodIPArray = NULL;        /* one method result */
    NQ_INT methodNumIps;                        /* number of IPs returned by one method */
    const CacheEntry * pEntry;                  /* entry in the cache */
    NQ_UINT priorityToActivate = 1;             /* each resolver method has activation priority. we start from 1 which is highest*/
    NQ_COUNT numPendingMethods = 0;             /* number of requests sent */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "dnsList:%p host:%s numIps:%p", dnsList, cmWDump(host), numIps);

    if ((NULL == host) || (NULL == numIps))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit1;
    }

    syMutexTake(&staticData->guard);

    /* look in cache */
    pEntry = lookupNameInCache(host);
    if (NULL != pEntry)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Found in cache");

        result = (const NQ_IPADDRESS *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_IPADDRESS) * pEntry->numIps));
        if (NULL == result)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        }
        else
        {
            syMemcpy(result, pEntry->ips, sizeof(NQ_IPADDRESS) * pEntry->numIps);
            *numIps = (NQ_INT)pEntry->numIps;
        }
        goto Exit2;
    }

    prepareMethods(dnsList);

    *numIps = 0;

    /* loop by the smallest timeout until either all timeouts expire */
    for (; ;)
    {
        SYSocketSet set;                            /* socket set for select */
        SYSocketSet setCopy;                        /* socket set for select (copy) */
        NQ_BOOL wasResponse = FALSE;                /* there was at least one response */
        NQ_BOOL breakWhile = FALSE;                 /* for while of select() */
        NQ_UINT32 selectTimeout;                    /* timeout in seconds of the current select of current priority */

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Use priority: %d", priorityToActivate);

        syClearSocketSet(&set);
        cmU64Zero(&currentTimeout);

        for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
        {
            Method * pMethod = (Method *)cmListIteratorNext(&iterator);

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method: %s, serverIP: %s, type: %s, multicast: %s, activationPriority: %d, socket: %d, status: 0x%x %s", 
                    pMethod->enabled ? "enabled" : "disabled",
                    cmIPDump(&pMethod->serverIp), methodType(pMethod->method.type),
                    pMethod->method.isMulticast ? "YES" : "NO", pMethod->method.activationPriority, pMethod->socket,
                    pMethod->status, pMethod->status == NQ_ERR_MOREDATA ? "NQ_ERR_MOREDATA" : ((pMethod->status == NQ_SUCCESS) ? "NQ_SUCCESS" : ""));

            if (!pMethod->enabled || pMethod->method.type == NQ_RESOLVER_DNS_DC || pMethod->method.type == NQ_RESOLVER_NETBIOS_DC || pMethod->method.type == NQ_RESOLVER_WSD)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Skipping method");
                continue;
            }

            if ((NQ_ERR_MOREDATA == pMethod->status) && (priorityToActivate == pMethod->method.activationPriority))
            {
                NQ_COUNT numOfSentRequests;

                pMethod->status = (*pMethod->method.requestByName)(pMethod->socket, host, pMethod->context, &pMethod->serverIp, &numOfSentRequests);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "method status:0x%x, sent requests:%u", pMethod->status, numOfSentRequests);
                if (pMethod->status == NQ_SUCCESS)
                {
                    syAddSocketToSet(pMethod->socket, &set);
                    numPendingMethods += numOfSentRequests;

                    if (cmU64Cmp(&pMethod->method.timeout, &currentTimeout) > 0)
                    {
                        /* set to max timeout value per priority */
                        cmU64AssignU64(&currentTimeout, &pMethod->method.timeout);
                    }
                }
            }
        }
        cmListIteratorTerminate(&iterator);

        /* save original set */
        setCopy = set;

        /* set the select() timeout to be used for current priority */
        selectTimeout = cmTimeConvertMSecToSec(&currentTimeout);

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, pending: %d, priority timeout: %d", priorityToActivate, numPendingMethods, cmTimeConvertMSecToSec(&currentTimeout));

        selectTimeStamp = syGetTimeInMsec();
        while (numPendingMethods > 0)
        {
            NQ_INT selectResult;

            if (cmU64Cmp(&currentTimeout, &zero) <= 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Timeout for method priority: %d.", priorityToActivate);
                break;
            }

            selectResult = sySelectSocket(&set, selectTimeout);
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, select result: %d, pending: %d, select timeout (seconds): %d, priority timeout (seconds): %d",
                                              priorityToActivate, selectResult, numPendingMethods, selectTimeout, cmTimeConvertMSecToSec(&currentTimeout));
            switch (selectResult)
            {
            case 0:  /* timeout */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select timeout");
                breakWhile = TRUE;
                break;

            case NQ_FAIL: /* error the select failed  */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select failed");
                goto Exit3;
                break;

            default: /* datagram ready for reading */
                for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
                {
                    Method * pMethod = (Method *)cmListIteratorNext(&iterator);

                    if (!pMethod->enabled || pMethod->method.type == NQ_RESOLVER_DNS_DC || pMethod->method.type == NQ_RESOLVER_NETBIOS_DC || pMethod->method.type == NQ_RESOLVER_WSD)
                    {
                        continue;
                    }
                    if (syIsSocketSet(pMethod->socket, &set))
                    {
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method socket:%d", pMethod->socket);

                        syClearSocketFromSet(pMethod->socket, &set);
                        syAddSocketToSet(pMethod->socket, &set);        /* performed because more than 1 responses might arrive, but we treat one at a time */
                        --numPendingMethods;

                        pMethod->status = (*pMethod->method.responseByName)(pMethod->socket, &methodIPArray, &methodNumIps, &pMethod->context);
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method status:%d", pMethod->status);
                        switch (pMethod->status)
                        {
                            case NQ_SUCCESS:
                            {
                                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "ipList:%p, numIPs:%d", methodIPArray, methodNumIps);
                                if ((NULL != methodIPArray) && (0 < methodNumIps))
                                {
                                    mergeIpAddresses(&result, numIps, &methodIPArray, methodNumIps);
                                    wasResponse = TRUE;
                                }
                            }
                            case NQ_ERR_MOREDATA:
                            case NQ_FAIL:
                            default:    /* error */
                            {
                                if (NULL != pMethod->context)
                                {
                                    cmMemoryFree(pMethod->context);
                                    pMethod->context = NULL;
                                }

                                if (NULL != methodIPArray)
                                {
                                    cmMemoryFree(methodIPArray);
                                    methodIPArray = NULL;
                                }

                                break;
                            }
                        }
                    }
                }
                cmListIteratorTerminate(&iterator);
            } /* end of switch (selectResult) */

            /* prepare socket set for next sySelectSocket */
            set = setCopy;

            if (TRUE == breakWhile)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Breaking while");
                break;
            }

            /* check pending requests */
            if (wasResponse)
            {
                for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
                {
                    Method * pMethod = (Method *)cmListIteratorNext(&iterator);

                    if (NQ_ERR_MOREDATA == pMethod->status && pMethod->method.waitAnyway)
                    {
                        wasResponse = FALSE;
                        break;
                    }
                }
                cmListIteratorTerminate(&iterator);
                if ((TRUE == wasResponse) && (0 == numPendingMethods))
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "There was a response and no more pending");
                    goto Exit3;
                }
            }

            /* recalculate timeout */
            tmpTime = syGetTimeInMsec();
            cmU64SubU64U64(&timeDiff, &tmpTime, &selectTimeStamp);
            if (cmU64Cmp(&timeDiff, &zero) > 0)
            {
                if (cmU64Cmp(&timeDiff, &currentTimeout) > 0)
                {
                    currentTimeout = zero;
                }
                else
                {
                    NQ_TIME tmp = currentTimeout;

                    cmU64SubU64U64(&currentTimeout, &tmp, &timeDiff);
                }

                cmU64AddU64(&selectTimeStamp, &timeDiff);
            }
        } /*while (numPendingMethods > 0)*/

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "End of while (numPendingMethods > 0)");

        /* advance activation priority or abort */
        if (NULL == result)
        {
            if (priorityToActivate < RESOLVER_MAX_ACTIVATION_PRIORITY)
            {
                ++priorityToActivate;
                continue;
            }
            else
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "All methods failed (still no result and no more priorities to use)");
            }
        }

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Exit (current priorityToActivate:%d)", priorityToActivate);
        goto Exit3;
    } /* end of for (; ;) */

Exit3:
    if (NULL == result)
    {
        result = externalNameToIp(host, numIps);
    }

    if ((NULL != result) && (1 < *numIps))
    {
        orderIPAddreses(&result, (NQ_COUNT)*numIps);
    }

    /* add to cache an ordered list */
    addToCache(host, result, (NQ_COUNT)*numIps);

Exit2:
#ifdef UD_NQ_CLOSESOCKETS
    closeSockets();
#endif
    releaseMethods(dnsList);
    syMutexGive(&staticData->guard);
Exit1:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", result);
    return result;
}

#ifdef UD_CM_INCLUDEWSDCLIENT

NQ_BOOL cmResolverGetDevices(const NQ_UINT deviceType, const NQ_IPADDRESS **listIPs, NQ_INT *numIps, const NQ_WCHAR **listNames, NQ_INT *numNames)
{
    NQ_TIME selectTimeStamp, timeDiff;          /* time when the current select started */
    NQ_TIME timeoutForSelectMSec;               /* timeout of the current select */
    NQ_TIME zero = {0, 0}, tmpTime;             /* for time */
    CMIterator iterator;                        /* method iterator */
    const NQ_IPADDRESS * resultIPs = NULL;      /* resulted array of IPs */
    NQ_IPADDRESS * methodIPArray = NULL;        /* one method result - IPs */
    NQ_WCHAR * resultNames = NULL;              /* resulted array of names */
    NQ_INT methodNumIps;                        /* number of IPs returned by one method */
    NQ_UINT priorityToActivate = 1;             /* each resolver method has activation priority. we start from 1 which is highest*/
    NQ_COUNT numPendingMethods = 0;             /* number of requests sent */
    NQ_BOOL result = FALSE;                     /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "deviceType:0x%x numIps:%p", deviceType, numIps);

    if ((NULL == listIPs) || (NULL == numIps) || (NULL == numNames) || (NULL == listNames))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit1;
    }

    syMutexTake(&staticData->guard);

    prepareMethods(NULL);

    *numIps = 0;
    *numNames = 0;

    /* loop by the smallest timeout until either all timeouts expire */
    for (; ;)
    {
        SYSocketSet set;                /* socket set for select */
        SYSocketSet setCopy;            /* socket set for select (copy) */
        NQ_BOOL breakWhile = FALSE;     /* for while of select() */

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Use priority: %d", priorityToActivate);

        syClearSocketSet(&set);
        cmU64Zero(&timeoutForSelectMSec);

        for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator);  )
        {
            Method * pMethod = (Method *)cmListIteratorNext(&iterator);

            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method: %s, serverIP: %s, type: %s, multicast: %s, activationPriority: %d, socket: %d, status: 0x%x %s",
                    pMethod->enabled ? "enabled" : "disabled",
                    cmIPDump(&pMethod->serverIp), methodType(pMethod->method.type),
                    pMethod->method.isMulticast ? "YES" : "NO", pMethod->method.activationPriority, pMethod->socket,
                    pMethod->status, pMethod->status == NQ_ERR_MOREDATA ? "NQ_ERR_MOREDATA" : ((pMethod->status == NQ_SUCCESS) ? "NQ_SUCCESS" : ""));

            if ((FALSE == pMethod->enabled) || (NQ_RESOLVER_WSD != pMethod->method.type))
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Skipping method");
                continue;
            }

            /* allocate and init method context */
            if (NULL != pMethod->context)
            {
                if (NULL != ((WsdClientContext *)pMethod->context)->names)
                {
                    cmMemoryFree(((WsdClientContext *)pMethod->context)->names);
                }

                cmMemoryFree(pMethod->context);
            }

            pMethod->context = (void *)cmMemoryAllocate((NQ_UINT)sizeof(WsdClientContext));
            if (NULL == pMethod->context)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                sySetLastError(NQ_ERR_OUTOFMEMORY);
                goto Exit2;
            }

            /* we pass names pointer via context */
            ((WsdClientContext *)pMethod->context)->deviceType = deviceType;
            ((WsdClientContext *)pMethod->context)->names = NULL;
            ((WsdClientContext *)pMethod->context)->numNames = 0;

            if ((NQ_ERR_MOREDATA == pMethod->status) && (priorityToActivate == pMethod->method.activationPriority))
            {
                NQ_COUNT numOfSentRequests;

                pMethod->status = (*pMethod->method.requestByIp)(pMethod->socket, NULL, pMethod->context, &pMethod->serverIp, &numOfSentRequests);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "method status:0x%x, sent requests:%u", pMethod->status, numOfSentRequests);
                if (pMethod->status == NQ_SUCCESS)
                {
                    syAddSocketToSet(pMethod->socket, &set);
                    numPendingMethods += numOfSentRequests;

                    /* if (pMethod->method.timeout > currentTimeout) */
                    if (cmU64Cmp(&pMethod->method.timeout, &timeoutForSelectMSec) > 0)
                    {
                        /* set to max timeout value per priority */
                        cmU64AssignU64(&timeoutForSelectMSec, &pMethod->method.timeout);
                    }
                }
            }
        } /* end of for */
        cmListIteratorTerminate(&iterator);

        /* save original set */
        setCopy = set;

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, pending: %d, priority timeout: %d", priorityToActivate, numPendingMethods, cmTimeConvertMSecToSec(&timeoutForSelectMSec));

        /* wait for responses until timeout expires */
        selectTimeStamp = syGetTimeInMsec();
        while (TRUE) /* exit on timeout expiration */
        {
            NQ_UINT32 timeoutForSelectSec = cmTimeConvertMSecToSec(&timeoutForSelectMSec);
            NQ_INT selectResult;

            /* if (timeoutForSelectMSec <= 0) */
            if (cmU64Cmp(&timeoutForSelectMSec, &zero) <= 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Select timeout, method priority: %d.", priorityToActivate);
                break;
            }

            selectResult = sySelectSocket(&set, timeoutForSelectSec);
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, select result: %d, pending: %d, timeout (seconds): %d",
                                                priorityToActivate, selectResult, numPendingMethods, timeoutForSelectSec);

            switch (selectResult)
            {
            case 0:  /* timeout */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select timeout, method priority: %d.", priorityToActivate);
                breakWhile = TRUE;
                break;

            case NQ_FAIL: /* error the select failed  */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select failed");
                goto Exit3;
                break;

            default: /* datagram ready for reading */
                for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
                {
                    Method * pMethod = (Method *)cmListIteratorNext(&iterator);

                    if (!pMethod->enabled || pMethod->method.type != NQ_RESOLVER_WSD)
                    {
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Skipping method");
                        continue;
                    }

                    if (syIsSocketSet(pMethod->socket, &set))
                    {
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method socket:%d", pMethod->socket);

                        syClearSocketFromSet(pMethod->socket, &set);
                        syAddSocketToSet(pMethod->socket, &set);
                        --numPendingMethods;

                        pMethod->status = (*pMethod->method.responseByName)(pMethod->socket, &methodIPArray, &methodNumIps, &pMethod->context);
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method status:%d", pMethod->status);
                        switch (pMethod->status)
                        {
                            case NQ_SUCCESS:
                            {
                                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "ipList:%p, numIPs:%d, nameList:%p, numNames:%d", methodIPArray, methodNumIps, ((WsdClientContext *)pMethod->context)->names, ((WsdClientContext *)pMethod->context)->numNames);
                                if ((NULL != methodIPArray) && (0 < methodNumIps))
                                {
                                    mergeIpAddresses(&resultIPs, numIps, &methodIPArray, methodNumIps);
                                }

                                if ((NULL != pMethod->context) && (NULL != ((WsdClientContext *)pMethod->context)->names) && (0 < (((WsdClientContext *)pMethod->context)->numNames)))
                                {
                                    mergeNames(&resultNames, numNames, (const NQ_WCHAR *)(((WsdClientContext *)pMethod->context)->names), ((WsdClientContext *)pMethod->context)->numNames);
                                }
                            }
                            case NQ_ERR_MOREDATA:
                            case NQ_FAIL:
                            default:    /* error */
                            {
                                if (NULL != ((WsdClientContext *)pMethod->context)->names)
                                {
                                    cmMemoryFree(((WsdClientContext *)pMethod->context)->names);
                                }

                                if (NULL != methodIPArray)
                                {
                                    cmMemoryFree(methodIPArray);
                                    methodIPArray = NULL;
                                }

                                break;
                            }
                        }
                    } /* end of syIsSocketSet */
                }  /* end of for */
                cmListIteratorTerminate(&iterator);
            } /* end of switch (selectResult) */

            /* prepare socket set for next sySelectSocket */
            set = setCopy;

            if (breakWhile)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Breaking while");
                break;
            }

            /* recalculate timeout */
            tmpTime = syGetTimeInMsec();
            cmU64SubU64U64(&timeDiff, &tmpTime, &selectTimeStamp);
            if (cmU64Cmp(&timeDiff, &zero) > 0)
            {
                if (cmU64Cmp(&timeDiff, &timeoutForSelectMSec) > 0)
                {
                    timeoutForSelectMSec = zero;
                }
                else
                {
                    NQ_TIME tmp = timeoutForSelectMSec;

                    cmU64SubU64U64(&timeoutForSelectMSec, &tmp, &timeDiff);
                }

                cmU64AddU64(&selectTimeStamp, &timeDiff);
            }
        } /*while (numPendingMethods > 0)*/

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "End of while (numPendingMethods > 0)");

        /* advance activation priority or abort */
        if ((NULL == resultIPs) || (NULL == resultNames))
        {
            if (priorityToActivate < RESOLVER_MAX_ACTIVATION_PRIORITY)
            {
                ++priorityToActivate;
                continue;
            }
            else
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "No more priorities to use");
            }
        }

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Exit (current priorityToActivate:%d)", priorityToActivate);
        goto Exit3;
    } /* end of for (; ;) */

Exit3:
    if ((NULL != resultIPs) && (1 < *numIps))
    {
        orderIPAddreses(&resultIPs, (NQ_COUNT)*numIps);
    }

    *listIPs = resultIPs;
    *listNames = resultNames;

Exit2:
#ifdef UD_NQ_CLOSESOCKETS
    closeSockets();
#endif /* UD_NQ_CLOSESOCKETS */
    freeContexts(NQ_RESOLVER_WSD);
    syMutexGive(&staticData->guard);
    result = (0 != *numIps) || (0 != *numNames);
Exit1:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s (numIps:%d numNames:%d) error:0x%x", result ? "TRUE" : "FALSE", numIps ? *numIps : 0, numNames ? *numNames : 0, syGetLastError());
    return result;
}

NQ_BOOL cmResolverSetTimeoutForDevicesIps(NQ_UINT32 milliseconds)
{
    return wsdClientSetTimeout(milliseconds);
}

#endif /* UD_CM_INCLUDEWSDCLIENT */

NQ_BOOL cmResolverUpdateExternalMethodsPriority(NQ_INT requiredPriority)
{
    CMIterator iterator;                        /* method iterator */

    for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
    {
        Method * pMethod = (Method *)cmListIteratorNext(&iterator);

        if (NQ_RESOLVER_EXTERNAL_METHOD == pMethod->method.type)
        {
            switch (requiredPriority)
            {
                case 1:
                    pMethod->method.activationPriority = 1;
                    break;
                case 2:
                    pMethod->method.activationPriority = 3;
                    break;
                default:
                    pMethod->method.activationPriority = 5;
                    break;
            }
        }
    }
    cmListIteratorTerminate(&iterator);

    return TRUE;
}

const NQ_WCHAR * cmResolverGetDCName(const NQ_WCHAR * domain, const NQ_WCHAR * dnsList, NQ_INT * numDCs)
{
    NQ_TIME selectTimeStamp, timeDiff;          /* time when the current select started */
    NQ_TIME currentTimeout;                     /* total timeout in milliseconds for current priority */
    NQ_TIME zero = {0, 0}, tmpTime;
    CMIterator iterator;                        /* method iterator */
    NQ_UINT priorityToActivate = 1;             /* each resolver method has activation priority. we start from 1 which is highest*/
    const NQ_WCHAR *pDCName = NULL;             /* resolved domain DC name */
    const NQ_WCHAR *pResult = NULL;             /* return value */
    NQ_COUNT numPendingMethods = 0;             /* number of requests sent */
    NQ_IPADDRESS * methodIPArray = NULL;        /* one method result */
    NQ_INT methodNumIps;                        /* number of IPs returned by one method */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "domain:%s dnsList:%p numDCs:%p", cmWDump(domain), dnsList, numDCs);

    if ((NULL == domain) || (NULL == numDCs))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - NULL pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit1;
    }

    syMutexTake(&staticData->guard);

    prepareMethods(dnsList);

    /* loop by the smallest timeout until either all timeouts expire or there is at least one answer */
    for (; ;)
    {
        SYSocketSet set;                    /* socket set for select */
        SYSocketSet setCopy;                /* socket set for select (copy) */
        NQ_BOOL breakWhile = FALSE;         /* for while of select() */
        NQ_UINT32 selectTimeout;            /* timeout in seconds of the current select of current priority */
        NQ_BOOL IPReceivedForNbns = FALSE;  /* whether resolved over NetBIOS method */

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Use priority: %d", priorityToActivate);

        syClearSocketSet(&set);
        cmU64Zero(&currentTimeout);

        for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
        {
            Method * pMethod = (Method *)cmListIteratorNext(&iterator);
         
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method: %s, serverIP: %s, type: %s, multicast: %s, activationPriority: %d, socket: %d, status: 0x%x %s", 
                    pMethod->enabled ? "enabled" : "disabled",
                    cmIPDump(&pMethod->serverIp), methodType(pMethod->method.type),
                    pMethod->method.isMulticast ? "YES" : "NO", pMethod->method.activationPriority, pMethod->socket,
                    pMethod->status, pMethod->status == NQ_ERR_MOREDATA ? "NQ_ERR_MOREDATA" : ((pMethod->status == NQ_SUCCESS) ? "NQ_SUCCESS" : ""));

            if (!pMethod->enabled || (pMethod->method.type != NQ_RESOLVER_DNS_DC && pMethod->method.type != NQ_RESOLVER_NETBIOS_DC))
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Skipping method");
                continue;
            }
            if ((NQ_ERR_MOREDATA == pMethod->status) && (priorityToActivate == pMethod->method.activationPriority))
            {
                NQ_COUNT numOfSentRequests;

                pMethod->status = (*pMethod->method.requestByName)(pMethod->socket, domain, pMethod->context, &pMethod->serverIp, &numOfSentRequests);
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "method status:0x%x, sent requests:%u", pMethod->status, numOfSentRequests);
                if (pMethod->status == NQ_SUCCESS)
                {
                    syAddSocketToSet(pMethod->socket, &set);
                    numPendingMethods += numOfSentRequests;

                    if (cmU64Cmp(&pMethod->method.timeout, &currentTimeout) > 0)
                    {
                        /* set to max timeout value per priority */
                        cmU64AssignU64(&currentTimeout, &pMethod->method.timeout);
                    }
                }
            }
        }
        cmListIteratorTerminate(&iterator);

        /* save original set */
        setCopy = set;

        /* set the select() timeout to be used for current priority */
        selectTimeout = cmTimeConvertMSecToSec(&currentTimeout);

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, pending: %d, priority timeout: %d", priorityToActivate, numPendingMethods, cmTimeConvertMSecToSec(&currentTimeout));

        selectTimeStamp = syGetTimeInMsec();
        while (numPendingMethods > 0)
        {
            NQ_INT selectResult;

            if (cmU64Cmp(&currentTimeout, &zero) <= 0)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Select timeout, method priority: %d.", priorityToActivate);
                break;
            }

            selectResult = sySelectSocket(&set, selectTimeout);
            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Priority: %d, select result: %d, pending: %d, select timeout (seconds): %d, priority timeout (seconds): %d",
                                              priorityToActivate, selectResult, numPendingMethods, selectTimeout, cmTimeConvertMSecToSec(&currentTimeout));
            switch (selectResult)
            {
            case 0:  /* timeout */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select timeout");
                breakWhile = TRUE;
                break;
            case NQ_FAIL: /* error the select failed  */
                LOGERR(CM_TRC_LEVEL_ERROR, "Select failed");
                goto Exit2;

            default: /* datagram ready for reading */
                for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
                {
                    Method * pMethod = (Method *)cmListIteratorNext(&iterator);

                    if (!pMethod->enabled || (pMethod->method.type != NQ_RESOLVER_DNS_DC && pMethod->method.type != NQ_RESOLVER_NETBIOS_DC))
                    {
                        continue;
                    }
                    if (syIsSocketSet(pMethod->socket, &set))
                    {
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method socket:%d", pMethod->socket);

                        syClearSocketFromSet(pMethod->socket, &set); /* remove this socket from set */
                        --numPendingMethods;

                        if (!IPReceivedForNbns && pMethod->method.type == NQ_RESOLVER_NETBIOS_DC)
                        {
                            /* for NETBIOS we first receive IP responses - take IPs and query names*/
                            NQ_COUNT IPNum;
                            IPReceivedForNbns = TRUE;

                            pMethod->status = (*pMethod->method.responseByName)(pMethod->socket, &methodIPArray, &methodNumIps, &pMethod->context);
                            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method status:%d", pMethod->status);
                            /* Received IP responses. iterate IPs and request host names.*/
                            if (NQ_SUCCESS == pMethod->status)
                            {
                                for (IPNum = 0; methodNumIps > 0; --methodNumIps, ++IPNum)
                                {
                                    NQ_COUNT numOfSentRequests;

                                    pMethod->status = (*pMethod->method.requestByIp)(pMethod->socket, &methodIPArray[IPNum], pMethod->context, &pMethod->serverIp, &numOfSentRequests);
                                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method status:0x%x, sent requests:%u", pMethod->status, numOfSentRequests);
                                    if (NQ_SUCCESS == pMethod->status)
                                    {
                                        numPendingMethods += numOfSentRequests;

                                        if (cmU64Cmp(&pMethod->method.timeout, &currentTimeout) > 0)
                                        {
                                            /* set to max timeout value */
                                            cmU64AssignU64(&currentTimeout, &pMethod->method.timeout);
                                            selectTimeStamp = syGetTimeInMsec();
                                        }
                                        cmU64AddU32(&currentTimeout, 1000);
                                    }
                                }
                                syAddSocketToSet(pMethod->socket, &set);
                            }
                            cmMemoryFree(methodIPArray);
                            methodIPArray = NULL;
                            continue;
                        }

                        /* add removed socket back to set */
                        syAddSocketToSet(pMethod->socket, &set);

                        /* Above we query by name. but for DNS in this type we need response by IP */
                        pMethod->status = (*pMethod->method.responseByIp)(pMethod->socket, &pDCName, &pMethod->context);
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Method status:%d", pMethod->status);
                        switch (pMethod->status)
                        {
                            case NQ_SUCCESS:
                            {
                                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "dcList:%p, numDCs:%d", pDCName, *((NQ_INT *)pMethod->context));
                                if ((NULL != pMethod->context) && (NULL != pDCName))
                                {
                                    *numDCs = *((NQ_INT *)pMethod->context);
                                    cmMemoryFree(pMethod->context);
                                    pMethod->context = NULL;
                                    cmListIteratorTerminate(&iterator);
                                    pResult = pDCName;
                                    goto Exit2;
                                }
                            }
                            case NQ_ERR_MOREDATA:
                            case NQ_FAIL:
                            default:    /* error */
                            {
                                if (NULL != pMethod->context)
                                {
                                    cmMemoryFree(pMethod->context);
                                    pMethod->context = NULL;
                                }

                                if  (NULL != pDCName)
                                {
                                    cmMemoryFree(pDCName);
                                    pDCName = NULL;
                                }

                                break;
                            }
                        }
                    }
                }
                cmListIteratorTerminate(&iterator);
            } /* end of switch (selectResult) */

            /* prepare socket set for next sySelectSocket */
            set = setCopy;

            if (breakWhile)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Breaking while");
                break;
            }

            /* recalculate timeout */
            tmpTime = syGetTimeInMsec();
            cmU64SubU64U64(&timeDiff, &tmpTime, &selectTimeStamp);
            if (cmU64Cmp(&timeDiff, &zero) > 0)
            {
                if (cmU64Cmp(&timeDiff, &currentTimeout) > 0)
                {
                    currentTimeout = zero;
                }
                else
                {
                    NQ_TIME tmp = currentTimeout;

                    cmU64SubU64U64(&currentTimeout, &tmp, &timeDiff);
                }

                cmU64AddU64(&selectTimeStamp, &timeDiff);
            }
        } /*while (numPendingMethods > 0)*/

        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "End of while (numPendingMethods > 0)");

        /* modify activation priority */
        if (priorityToActivate < RESOLVER_MAX_ACTIVATION_PRIORITY)
        {
            ++priorityToActivate;
            continue;
        }
        LOGERR(CM_TRC_LEVEL_ERROR, "All methods failed (still no result and no more priorities to use)");
        goto Exit2;
    }

Exit2:
#ifdef UD_NQ_CLOSESOCKETS
    closeSockets();
#endif /* UD_NQ_CLOSESOCKETS */
    releaseMethods(dnsList);
    syMutexGive(&staticData->guard);
Exit1:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p, numDCs:%d", pResult, numDCs ? *numDCs : 0);
    return pResult;
}

/* --- API functions --- */

void cmResolverSetExternalA(CMResolverNameToIpA nameToIp, CMResolverIpToNameA ipToName)
{
    staticData->nameToIpA = nameToIp;
    staticData->ipToNameA = ipToName;
}

void cmResolverSetExternal(CMResolverNameToIp nameToIp, CMResolverIpToName ipToName)
{
    staticData->nameToIpW = nameToIp;
    staticData->ipToNameW = ipToName;
}

void cmResolverEnableMethod(NQ_INT type, NQ_BOOL unicast, NQ_BOOL multicast)
{
    CMIterator iterator;                        /* method iterator */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "type:%d unicast:%s multicast:%s", unicast, unicast ? "TRUE" : "FALSE", multicast ? "TRUE" : "FALSE");

    syMutexTake(&staticData->guard);

    for (cmListIteratorStart(&staticData->methods, &iterator); cmListIteratorHasNext(&iterator); )
    {
        Method * pMethod = (Method *)cmListIteratorNext(&iterator);

        if (type == pMethod->method.type)
        {
            if (pMethod->method.isMulticast)
            {
                pMethod->enabled = multicast;
            }
            else
            {
                pMethod->enabled = unicast;
            }
        }
    }
    cmListIteratorTerminate(&iterator);
    syMutexGive(&staticData->guard);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

