/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Common repository operations
 *--------------------------------------------------------------------
 * MODULE        : Common
 * DEPENDENCIES  :
 *************************************************************************/

#include "cmrepository.h"
#include "cmmemory.h"

#define REPOSITORY_TIMEOUT_SEC 3 /* timeout for waiting on condition when resources are limited */

#ifdef UD_NQ_DYNAMICALLOCATION
static NQ_BOOL dynamiclyAllocate = TRUE;
#else
static NQ_BOOL dynamiclyAllocate = FALSE;
#endif /* UD_NQ_DYNAMICALLOCATION */

#if defined(REPOSITRY_ZERO_SIZE) && !defined(UD_NQ_DYNAMICALLOCATION)
#error REPOSITRY_ZERO_SIZE requires UD_NQ_DYNAMICALLOCATION
#endif

#ifdef REPOSITORY_STATISTICS
void dumpStats(CMRepository * pRepo);
#endif /* REPOSITORY_STATISTICS */

void cmSetDynamicAllocation(NQ_BOOL value)
{
    dynamiclyAllocate = value;
}

NQ_BOOL cmGetDynamicAllocation(void)
{
    return dynamiclyAllocate;
}

NQ_BOOL cmRepositoryInit(
        CMRepository * pRepo,
        NQ_UINT16 flags,
        CMRepositoryItemInit doInit,
        CMRepositoryItemDispose doDispose,
        const NQ_CHAR *repoName
        )
{
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_REPOSITORY, "pRepo:%p flags:%d doInit:%p doDispose:%p repoName:%s", pRepo, flags, doInit, doDispose, repoName);

#ifdef REPOSITRY_ZERO_SIZE
    flags |= CM_REPOSITORY_RELEASE_IMMEDIATELY;
#endif
    pRepo->isUsed = TRUE;
    pRepo->guard = NULL;
    pRepo->numItemsReadyForUse = 0;
    pRepo->numUsedItems = 0;
    pRepo->flags = flags;
    pRepo->doInit = doInit;
    pRepo->doDispose = doDispose;
    cmListStart(&pRepo->usedItemlist);
    cmListStart(&pRepo->itemPool);
    pRepo->extend = dynamiclyAllocate;
#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
    pRepo->name = (NQ_CHAR *)cmMemoryAllocateStartup((NQ_UINT)(syStrlen(repoName) + 1));
    syStrcpy(pRepo->name, repoName);
#endif /* defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */
#ifdef REPOSITORY_STATISTICS
    pRepo->totalTimeWaited = 0;
    pRepo->maxTimeWaited = 0;
#endif /* REPOSITORY_STATISTICS */
    pRepo->numThreadSWaitingForSignal = 0;
    pRepo->numThreadSWaitingForItemRelease = 0;
#ifdef REPOSITORY_DEBUG
    pRepo->numOfGetNew = 0;
    pRepo->numConsecutiveFails = 0;
#endif /* REPOSITORY_DEBUG */
#if defined (REPOSITORY_DEBUG) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
    pRepo->numOfFailed = 0;
    pRepo->numOfWaitedItems = 0;
#endif /* defined (REPOSITORY_DEBUG) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */
    if (FALSE == cmThreadCondSet(&pRepo->resoCond))
    {
        result = FALSE;
        goto Exit;
    }
    if (TRUE)  /* should be flags & some flag for mutex */
    {
        pRepo->guard = (SYMutex *)cmMemoryAllocateStartup(sizeof(SYMutex));
        syMutexCreate(pRepo->guard);
    }
Exit:
    LOGFE(CM_TRC_LEVEL_REPOSITORY, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL cmRepositoryItemPoolAlloc(CMRepository * pRepo , NQ_UINT preallocateNum , NQ_UINT size)
{
    LOGFB(CM_TRC_LEVEL_REPOSITORY, "pRepo:%p preallocateNum:%u size:%u", pRepo, preallocateNum, size);

    if (NULL != pRepo->guard)
        syMutexTake(pRepo->guard);

#ifdef REPOSITRY_ZERO_SIZE
    preallocateNum = 1;
#endif

    pRepo->itemSize = size;
    while (pRepo->numItemsReadyForUse < preallocateNum)
    {
        CMRepoItem * pItem;

        pItem = (CMRepoItem *)cmListItemCreateAndAdd(&pRepo->itemPool, size + (NQ_UINT)sizeof(CMRepoItem), NULL, NULL, CM_LISTITEM_NOLOCK , TRUE);
        if (pRepo->doInit != NULL)
        {
            pRepo->doInit((&(pItem + 1)->item));
        }
        pRepo->numItemsReadyForUse++;
    }

#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
    pRepo->initialNumItems = pRepo->finalNumItems = preallocateNum;
#endif /* defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */
    if (NULL != pRepo->guard)
        syMutexGive(pRepo->guard);

    LOGFE(CM_TRC_LEVEL_REPOSITORY, "result:%s", TRUE ? "TRUE" : "FALSE");
    return TRUE;
}

void cmRepositoryShutdown(CMRepository * pRepo)
{
    LOGFB(CM_TRC_LEVEL_REPOSITORY, "repo:%p", pRepo);

    if (TRUE == pRepo->isUsed)
    {
        CMIterator itr;

#ifdef REPOSITORY_DEBUG
        if (NULL != pRepo->usedItemlist.last)
        {
            NQ_UINT32 numOfLeaked = 0;
            CMIterator  itr;
            cmListIteratorStart(&pRepo->usedItemlist , &itr);
            while (cmListIteratorHasNext(&itr))
            {
                cmListIteratorNext(&itr);
                numOfLeaked++;
            }
            cmListIteratorTerminate(&itr);
            LOGERR(CM_TRC_LEVEL_ERROR, " %d Repository [created by %s ] items leaked" , numOfLeaked , pRepo->name);
        }
        if ((0 < pRepo->numOfFailed) || (0 < pRepo->numOfWaitedItems) || (pRepo->initialNumItems < pRepo->finalNumItems))
        {
            syPrintf("\nSummary. repo: %s, item size: %d ", pRepo->name, pRepo->itemSize);
            syPrintf("initial number of items: %d, final number: %d, get new: %d, failed: %d waited: %d still used:%d\n",
                    pRepo->initialNumItems, pRepo->finalNumItems, pRepo->numOfGetNew, pRepo->numOfFailed, pRepo->numOfWaitedItems, pRepo->numUsedItems);
        }
        else
#endif /* REPOSITORY_DEBUG */
#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS)
            if (pRepo->initialNumItems < pRepo->finalNumItems)
            {
                syPrintf("\nSummary. repo: %s. item size: %d", pRepo->name, pRepo->itemSize);
                syPrintf("initial number of items: %d, final number: %d\n",
                        pRepo->initialNumItems, pRepo->finalNumItems);
            }
        else
        {
            syPrintf("\nSummary. repo: %s. item size: %d, No failures enough items: %d ", pRepo->name, pRepo->itemSize, pRepo->initialNumItems);
        }
#endif /* defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) */
#ifdef REPOSITORY_STATISTICS
        dumpStats(pRepo);
#endif /* REPOSITORY_STATISTICS */

        /* Move all Items to itemPool so that doDispose is called */
        cmListIteratorStart(&pRepo->usedItemlist, &itr);
        while (cmListIteratorHasNext(&itr))
        {
            CMRepoItem *pItem = (CMRepoItem *)cmListIteratorNext(&itr);
            cmListItemRemove(&pItem->item);
            cmListItemAdd(&pRepo->itemPool , &pItem->item , pItem->item.callback);
        }
        cmListIteratorTerminate(&itr);

        if (NULL != pRepo->doDispose)
        {
            cmListIteratorStart(&pRepo->itemPool, &itr);
            while (cmListIteratorHasNext(&itr))
            {
                CMRepoItem *pItem = (CMRepoItem *)cmListIteratorNext(&itr);
                pRepo->doDispose(&((pItem + 1)->item));
            }           
            cmListIteratorTerminate(&itr);
        }

        cmThreadCondRelease(&pRepo->resoCond);

        cmListShutdown(&pRepo->usedItemlist);
        cmListShutdown(&pRepo->itemPool);

        if (NULL != pRepo->guard)
        {
            syMutexDelete(pRepo->guard);
            cmMemoryFreeShutdown(pRepo->guard);
            pRepo->guard = NULL;
        }

#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
        cmMemoryFreeShutdown(pRepo->name);
#endif /* defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */
    }

    LOGFE(CM_TRC_LEVEL_REPOSITORY);
}

NQ_BOOL cmRepositoryAddItem(CMRepository * pRepo , CMRepoItem * pItem , NQ_BYTE * key , NQ_BOOL (*callback)(CMItem * pItem) )
{
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_REPOSITORY, "repo:%p item:%p key:%p callback:%p",  pRepo, pItem, key, callback);
    if (!pRepo->isUsed)
    {
        goto Exit;
    }

    cmListItemAdd(&pRepo->usedItemlist , &pItem->item , callback);
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_REPOSITORY, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

CMItem * cmRepositoryGetNewItem(CMRepository * pRepo)
{
    CMRepoItem * pRepoItem = NULL;
    CMItem * pResult = NULL;

#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
    LOGFB(CM_TRC_LEVEL_REPOSITORY, "repo:%p %s", pRepo, pRepo && pRepo->name ? pRepo->name : "");
#endif /* defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */

    if (FALSE == pRepo->isUsed)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Repository not in use");
        goto Exit;
    }

    if (NULL != pRepo->guard)
    {
        syMutexTake(pRepo->guard);
    }

    if (pRepo->numItemsReadyForUse == pRepo->numThreadSWaitingForItemRelease)
    {
        if (TRUE == dynamiclyAllocate)
        {
            LOGMSG(CM_TRC_LEVEL_REPOSITORY, "Repository[%p]: adding new" , pRepo);
            /* TODO: fix this so that callback will not depend on first item in list*/
            pRepoItem = (CMRepoItem *)cmListItemCreate(pRepo->itemSize + (NQ_UINT)sizeof(CMRepoItem), NULL, CM_LISTITEM_NOLOCK, FALSE);
            if (NULL == pRepoItem)
            {
                syAssert(FALSE);
                LOGERR(CM_TRC_LEVEL_ERROR, "Failed to allocate new item.");
                goto Error;
            }

            cmListItemAdd(&pRepo->usedItemlist , &pRepoItem->item , NULL);
            ++pRepo->numUsedItems;
            if (NULL != pRepo->doInit)
            {
                pRepo->doInit(&((pRepoItem + 1)->item));
            }
#if defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS)
            ++pRepo->finalNumItems;
            if (0 == ((pRepo->finalNumItems - pRepo->initialNumItems) % 20))
            {
                syPrintf("Repo: %s, so far added %d items.", pRepo->name, (pRepo->finalNumItems - pRepo->initialNumItems));
            }
#endif /* defined (REPOSITORY_DEBUG) || defined (REPOSITORY_SIZE_TUNING) || defined (REPOSITORY_STATISTICS) */
        }
        else
        {
            NQ_BOOL res;
#ifdef REPOSITORY_STATISTICS
            NQ_TIME time1, time2;
#endif /* REPOSITORY_STATISTICS */
            ++pRepo->numThreadSWaitingForItemRelease;
            ++pRepo->numThreadSWaitingForSignal;

            if (NULL != pRepo->guard)
            {
                syMutexGive(pRepo->guard);
            }
#ifdef REPOSITORY_SIZE_TUNING
            syPrintf("Repo: %s no available items\n", pRepo->name);
#endif /* REPOSITORY_SIZE_TUNING */
#ifdef REPOSITORY_STATISTICS
            time1 = syGetTimeInMsec();
            /*dumpStats(pRepo);*/
#endif /* REPOSITORY_STATISTICS */

            if (TRUE == (res = cmThreadCondWait(&pRepo->resoCond , REPOSITORY_TIMEOUT_SEC)))
            {
                if (NULL != pRepo->guard)
                {
                    syMutexTake(pRepo->guard);
                }
#ifdef REPOSITORY_STATISTICS
                NQ_TIME     waitedTime;
                NQ_UINT32   waitedInMsec = 0;

                time2 = syGetTimeInMsec();
                cmU64SubU64U64(&waitedTime , &time1 , &time2);
                waitedInMsec = waitedTime.low;
                pRepo->maxTimeWaited = waitedInMsec > pRepo->maxTimeWaited ? waitedInMsec : pRepo->maxTimeWaited;
                pRepo->totalTimeWaited += waitedInMsec;
#endif /* REPOSITORY_STATISTICS */
#ifdef REPOSITORY_DEBUG
                if (1 == pRepo->numOfWaitedItems % REPO_MESSAGE_THROTTLING)
                {
                    syPrintf("Repo: %s items waited on condition %d times.\n", pRepo->name, pRepo->numOfWaitedItems);
                }
#endif /* REPOSITORY_DEBUG */
#if (defined (UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
                if (1 == pRepo->numOfWaitedItems % REPO_MESSAGE_THROTTLING)
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Repo: %s items waited on condition %d times.", pRepo->name, pRepo->numOfWaitedItems);
                }
#endif /* (defined (UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */

                --pRepo->numThreadSWaitingForItemRelease;
                goto GettingItem;
            }
            else
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Repository - couldn't get new item. Used: %d, num free to allocate: %d, condition wait res:%d", pRepo->isUsed, pRepo->numItemsReadyForUse, res);
                if (NULL != pRepo->guard)
                {
                    syMutexTake(pRepo->guard);
                }

                --pRepo->numThreadSWaitingForItemRelease;
                --pRepo->numThreadSWaitingForSignal;
#if defined (REPOSITORY_DEBUG) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
                pRepo->numOfFailed++;
#endif /* defined (REPOSITORY_DEBUG) || (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */
#ifdef REPOSITORY_DEBUG
                if (1 == pRepo->numOfFailed % REPO_MESSAGE_THROTTLING)
                {
                    syPrintf("Repo: %s allocation failed %d times. Total items: %d\n", pRepo->name, pRepo->numOfFailed, pRepo->initialNumItems);
                }
#endif /* REPOSITORY_DEBUG */
#if (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION))
                if (1 == pRepo->numOfFailed % REPO_MESSAGE_THROTTLING)
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Repo: %s allocation failed %d times. Total items: %d", pRepo->name, pRepo->numOfFailed, pRepo->initialNumItems);
                }
#endif /* (defined(UD_NQ_INCLUDETRACE) && !defined(UD_NQ_DYNAMICALLOCATION)) */
                goto Error;
            }
        }
    }
    else
    {
GettingItem:
#ifdef REPOSITORY_DEBUG
        pRepo->numOfGetNew++;
#endif /* REPOSITORY_DEBUG */
        pRepoItem = (CMRepoItem *)pRepo->itemPool.last;
        if (NULL == pRepoItem)
        {
            /* there is still some scenario we get here and list is empty. don't erase this*/
            goto Error;
        }
        cmListItemRemove(&pRepoItem->item);
        cmListItemAdd(&pRepo->usedItemlist , &pRepoItem->item , pRepoItem->item.callback);
        --pRepo->numItemsReadyForUse;
        ++pRepo->numUsedItems;
    }

    if (NULL != pRepoItem)
    {
#ifdef REPOSITORY_DEBUG
        pRepo->numConsecutiveFails = 0;
        pRepoItem->DebugData1 = pRepoItem->DebugData2 = 192;
#endif /* REPOSITORY_DEBUG */
        pRepoItem++;
        pResult = &(pRepoItem->item);
        if (NULL != pRepo->guard)
        {
            syMutexGive(pRepo->guard);
        }

        goto Exit;
    }

Error:
#ifdef REPOSITORY_DEBUG
    if (NULL == pResult)
    {
        ++pRepo->numConsecutiveFails;
        if (10 == (pRepo->numConsecutiveFails % 20))
        {
            /* if fail to get item 20 times. print debug data per used items. */
            syPrintf("Num consecutive fails: %d, repo %s, numberItems: %d\n", pRepo->numConsecutiveFails, pRepo->name, pRepo->initialNumItems);

            /* print used items list */
            /*
            CMIterator iter;
            cmListIteratorStart(&pRepo->usedItemlist, &iter);
            syPrintf("printing used Items list. %d items, repo: %s\n", pRepo->numUsedItems, pRepo->name);
            while(cmListIteratorHasNext(&iter))
            {
                CMRepoItem *pRepoItem;
                pRepoItem = (CMRepoItem *)cmListIteratorNext(&iter);
                syPrintf("CMD: %d, MID low: %u\n", pRepoItem->DebugData1, pRepoItem->DebugData2);
            }
            cmListIteratorTerminate(&iter);
            */
        }
    }
#endif /* REPOSITORY_DEBUG */
    if (NULL != pRepo->guard)
    {
        syMutexGive(pRepo->guard);
    }

Exit:
    LOGFE(CM_TRC_LEVEL_REPOSITORY, "result:%p", pResult);
    return pResult;
}

#ifdef REPOSITORY_DEBUG
void cmRepositoryAddDebugData(CMItem *_pItem, NQ_UINT debugData1, NQ_UINT debugData2)
{
    CMRepoItem *pItem = (CMRepoItem *)_pItem;

    --pItem;
    pItem->DebugData1 = debugData1;
    pItem->DebugData2 = debugData2;
}
#endif

NQ_BOOL cmRepositoryReleaseItem(CMRepository * pRepo, CMItem * _pItem )
{
    NQ_BOOL result = FALSE;
    CMRepoItem *pItem = (CMRepoItem *)_pItem;

    LOGFB(CM_TRC_LEVEL_REPOSITORY, "repo:%p item:%p", pRepo, pItem);

    if (!pRepo->isUsed)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Repository not in use");
        goto Exit;
    }

    if (NULL != pRepo->guard)   syMutexTake(pRepo->guard);

    if ((pItem - 1)->item.master != &pRepo->usedItemlist)
    {
        if (NULL != pRepo->guard)   syMutexGive(pRepo->guard);
        LOGERR(CM_TRC_LEVEL_ERROR, "Not repository item");
        goto Exit;
    }

    if (pRepo->extend && (pRepo->flags & CM_REPOSITORY_RELEASE_IMMEDIATELY))
    {
        cmListItemRemoveAndDispose(&((pItem - 1)->item));
    }
    else
    {
        cmListItemRemove(&((pItem - 1)->item));
        cmListItemAdd(&pRepo->itemPool, &((pItem - 1)->item), NULL);
        ++pRepo->numItemsReadyForUse;
        --pRepo->numUsedItems;

        if (!dynamiclyAllocate && pRepo->numThreadSWaitingForSignal > 0) /* condition used only when no dynamic allocate */
        {
            --pRepo->numThreadSWaitingForSignal;
            cmThreadCondSignal(&pRepo->resoCond);
        }
    }
    
    if (NULL != pRepo->guard)   syMutexGive(pRepo->guard);

    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_REPOSITORY, "result:%s:", result ? "TRUE" : "FALSE");
    return result;
}

void cmRepositoryIteratorStart(CMRepository * pRepo, CMIterator * iterator)
{
    CMList * pList = &pRepo->usedItemlist;

    if (NULL != pRepo->guard)   syMutexTake(pRepo->guard);
    cmListIteratorStart(pList, iterator);
}

void cmRepositoryIteratorTerminate(CMRepository * pRepo, CMIterator * iterator)
{
    cmListIteratorTerminate(iterator);
    if (NULL != pRepo->guard)   syMutexGive(pRepo->guard);
}

CMItem * cmRepositoryIteratorNext(CMIterator * iterator)
{
    CMRepoItem * pItem = (CMRepoItem *)cmListIteratorNext(iterator);
    return (CMItem *)(pItem + 1);
}

#ifdef REPOSITORY_STATISTICS
void dumpStats(CMRepository * pRepo)
{
    syPrintf(" ******* Dump statistics for repository %p name %s *******\n" , pRepo , pRepo->name);
    syPrintf(" = Item Size = %d | Number of Pre Allocated Items = %d \n" , pRepo->itemSize , pRepo->numItemsReadyForUse);
#ifdef REPOSITORY_DEBUG
    syPrintf(" = Item Get New = %d | Number Waited Items = %d | Failed to get = %d\n" , pRepo->numOfGetNew , pRepo->numOfWaitedItems, pRepo->numOfFailed);
    if (0 < pRepo->numOfWaitedItems)
    {
        syPrintf(" = Max Waited Time = %d | Average time waited in ms = %d\n" , pRepo->maxTimeWaited , pRepo->totalTimeWaited / pRepo->numOfWaitedItems);
    }
#endif /* REPOSITORY_DEBUG */
    syPrintf(" ******* END *******\n");
}
#endif /* REPOSITORY_STATISTICS */


