/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : SHA512 encryption implementation
 *--------------------------------------------------------------------
 * MODULE        : Auth - AM
 * DEPENDENCIES  : None
 ********************************************************************/
#include "cmapi.h"

#ifdef UD_NQ_INCLUDESMB3

/* define 64 bit type */

#ifdef SY_INT64
#define SHA_U64 SY_UINT64
#else
#ifdef SY_PRAGMAPACK_DEFINED
#pragma pack(1)
#endif
#ifdef SY_BIGENDIANHOST
typedef struct {
    NQ_UINT32 high;
    NQ_UINT32 low;
} SY_PACK_ATTR SHA_U64;
#else
typedef struct {
    NQ_UINT32 low;
    NQ_UINT32 high;
} SY_PACK_ATTR SHA_U64;
#endif
#ifdef SY_PRAGMAPACK_DEFINED
#pragma pack()
#endif
#endif

/* padding */ 
static const NQ_BYTE pad[128] =
{
    0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};

/* round constants */
#ifdef SY_INT64
static const SY_UINT64 k[80] =
{
    0x428A2F98D728AE22ULL, 0x7137449123EF65CDULL, 0xB5C0FBCFEC4D3B2FULL, 0xE9B5DBA58189DBBCULL,
    0x3956C25BF348B538ULL, 0x59F111F1B605D019ULL, 0x923F82A4AF194F9BULL, 0xAB1C5ED5DA6D8118ULL,
    0xD807AA98A3030242ULL, 0x12835B0145706FBEULL, 0x243185BE4EE4B28CULL, 0x550C7DC3D5FFB4E2ULL,
    0x72BE5D74F27B896FULL, 0x80DEB1FE3B1696B1ULL, 0x9BDC06A725C71235ULL, 0xC19BF174CF692694ULL,
    0xE49B69C19EF14AD2ULL, 0xEFBE4786384F25E3ULL, 0x0FC19DC68B8CD5B5ULL, 0x240CA1CC77AC9C65ULL,
    0x2DE92C6F592B0275ULL, 0x4A7484AA6EA6E483ULL, 0x5CB0A9DCBD41FBD4ULL, 0x76F988DA831153B5ULL,
    0x983E5152EE66DFABULL, 0xA831C66D2DB43210ULL, 0xB00327C898FB213FULL, 0xBF597FC7BEEF0EE4ULL,
    0xC6E00BF33DA88FC2ULL, 0xD5A79147930AA725ULL, 0x06CA6351E003826FULL, 0x142929670A0E6E70ULL,
    0x27B70A8546D22FFCULL, 0x2E1B21385C26C926ULL, 0x4D2C6DFC5AC42AEDULL, 0x53380D139D95B3DFULL,
    0x650A73548BAF63DEULL, 0x766A0ABB3C77B2A8ULL, 0x81C2C92E47EDAEE6ULL, 0x92722C851482353BULL,
    0xA2BFE8A14CF10364ULL, 0xA81A664BBC423001ULL, 0xC24B8B70D0F89791ULL, 0xC76C51A30654BE30ULL,
    0xD192E819D6EF5218ULL, 0xD69906245565A910ULL, 0xF40E35855771202AULL, 0x106AA07032BBD1B8ULL,
    0x19A4C116B8D2D0C8ULL, 0x1E376C085141AB53ULL, 0x2748774CDF8EEB99ULL, 0x34B0BCB5E19B48A8ULL,
    0x391C0CB3C5C95A63ULL, 0x4ED8AA4AE3418ACBULL, 0x5B9CCA4F7763E373ULL, 0x682E6FF3D6B2B8A3ULL,
    0x748F82EE5DEFB2FCULL, 0x78A5636F43172F60ULL, 0x84C87814A1F0AB72ULL, 0x8CC702081A6439ECULL,
    0x90BEFFFA23631E28ULL, 0xA4506CEBDE82BDE9ULL, 0xBEF9A3F7B2C67915ULL, 0xC67178F2E372532BULL,
    0xCA273ECEEA26619CULL, 0xD186B8C721C0C207ULL, 0xEADA7DD6CDE0EB1EULL, 0xF57D4F7FEE6ED178ULL,
    0x06F067AA72176FBAULL, 0x0A637DC5A2C898A6ULL, 0x113F9804BEF90DAEULL, 0x1B710B35131C471BULL,
    0x28DB77F523047D84ULL, 0x32CAAB7B40C72493ULL, 0x3C9EBE0A15C9BEBCULL, 0x431D67C49C100D4CULL,
    0x4CC5D4BECB3E42B6ULL, 0x597F299CFC657E2AULL, 0x5FCB6FAB3AD6FAECULL, 0x6C44198C4A475817ULL
};
#else
#ifdef SY_LITTLEENDIANHOST
static const SHA_U64 k[80] =
{
    {0xD728AE22, 0x428A2F98}, {0x23EF65CD, 0x71374491}, {0xEC4D3B2F, 0xB5C0FBCF}, {0x8189DBBC, 0xE9B5DBA5},
    {0xF348B538, 0x3956C25B}, {0xB605D019, 0x59F111F1}, {0xAF194F9B, 0x923F82A4}, {0xDA6D8118, 0xAB1C5ED5},
    {0xA3030242, 0xD807AA98}, {0x45706FBE, 0x12835B01}, {0x4EE4B28C, 0x243185BE}, {0xD5FFB4E2, 0x550C7DC3},
    {0xF27B896F, 0x72BE5D74}, {0x3B1696B1, 0x80DEB1FE}, {0x25C71235, 0x9BDC06A7}, {0xCF692694, 0xC19BF174},
    {0x9EF14AD2, 0xE49B69C1}, {0x384F25E3, 0xEFBE4786}, {0x8B8CD5B5, 0x0FC19DC6}, {0x77AC9C65, 0x240CA1CC},
    {0x592B0275, 0x2DE92C6F}, {0x6EA6E483, 0x4A7484AA}, {0xBD41FBD4, 0x5CB0A9DC}, {0x831153B5, 0x76F988DA},
    {0xEE66DFAB, 0x983E5152}, {0x2DB43210, 0xA831C66D}, {0x98FB213F, 0xB00327C8}, {0xBEEF0EE4, 0xBF597FC7},
    {0x3DA88FC2, 0xC6E00BF3}, {0x930AA725, 0xD5A79147}, {0xE003826F, 0x06CA6351}, {0x0A0E6E70, 0x14292967},
    {0x46D22FFC, 0x27B70A85}, {0x5C26C926, 0x2E1B2138}, {0x5AC42AED, 0x4D2C6DFC}, {0x9D95B3DF, 0x53380D13},
    {0x8BAF63DE, 0x650A7354}, {0x3C77B2A8, 0x766A0ABB}, {0x47EDAEE6, 0x81C2C92E}, {0x1482353B, 0x92722C85},
    {0x4CF10364, 0xA2BFE8A1}, {0xBC423001, 0xA81A664B}, {0xD0F89791, 0xC24B8B70}, {0x0654BE30, 0xC76C51A3},
    {0xD6EF5218, 0xD192E819}, {0x5565A910, 0xD6990624}, {0x5771202A, 0xF40E3585}, {0x32BBD1B8, 0x106AA070},
    {0xB8D2D0C8, 0x19A4C116}, {0x5141AB53, 0x1E376C08}, {0xDF8EEB99, 0x2748774C}, {0xE19B48A8, 0x34B0BCB5},
    {0xC5C95A63, 0x391C0CB3}, {0xE3418ACB, 0x4ED8AA4A}, {0x7763E373, 0x5B9CCA4F}, {0xD6B2B8A3, 0x682E6FF3},
    {0x5DEFB2FC, 0x748F82EE}, {0x43172F60, 0x78A5636F}, {0xA1F0AB72, 0x84C87814}, {0x1A6439EC, 0x8CC70208},
    {0x23631E28, 0x90BEFFFA}, {0xDE82BDE9, 0xA4506CEB}, {0xB2C67915, 0xBEF9A3F7}, {0xE372532B, 0xC67178F2},
    {0xEA26619C, 0xCA273ECE}, {0x21C0C207, 0xD186B8C7}, {0xCDE0EB1E, 0xEADA7DD6}, {0xEE6ED178, 0xF57D4F7F},
    {0x72176FBA, 0x06F067AA}, {0xA2C898A6, 0x0A637DC5}, {0xBEF90DAE, 0x113F9804}, {0x131C471B, 0x1B710B35},
    {0x23047D84, 0x28DB77F5}, {0x40C72493, 0x32CAAB7B}, {0x15C9BEBC, 0x3C9EBE0A}, {0x9C100D4C, 0x431D67C4},
    {0xCB3E42B6, 0x4CC5D4BE}, {0xFC657E2A, 0x597F299C}, {0x3AD6FAEC, 0x5FCB6FAB}, {0x4A475817, 0x6C44198C}
};
#else
static const SHA_U64 k[80] =
{
    {0x428A2F98, 0xD728AE22}, {0x71374491, 0x23EF65CD}, {0xB5C0FBCF, 0xEC4D3B2F}, {0xE9B5DBA5, 0x8189DBBC},
    {0x3956C25B, 0xF348B538}, {0x59F111F1, 0xB605D019}, {0x923F82A4, 0xAF194F9B}, {0xAB1C5ED5, 0xDA6D8118},
    {0xD807AA98, 0xA3030242}, {0x12835B01, 0x45706FBE}, {0x243185BE, 0x4EE4B28C}, {0x550C7DC3, 0xD5FFB4E2},
    {0x72BE5D74, 0xF27B896F}, {0x80DEB1FE, 0x3B1696B1}, {0x9BDC06A7, 0x25C71235}, {0xC19BF174, 0xCF692694},
    {0xE49B69C1, 0x9EF14AD2}, {0xEFBE4786, 0x384F25E3}, {0x0FC19DC6, 0x8B8CD5B5}, {0x240CA1CC, 0x77AC9C65},
    {0x2DE92C6F, 0x592B0275}, {0x4A7484AA, 0x6EA6E483}, {0x5CB0A9DC, 0xBD41FBD4}, {0x76F988DA, 0x831153B5},
    {0x983E5152, 0xEE66DFAB}, {0xA831C66D, 0x2DB43210}, {0xB00327C8, 0x98FB213F}, {0xBF597FC7, 0xBEEF0EE4},
    {0xC6E00BF3, 0x3DA88FC2}, {0xD5A79147, 0x930AA725}, {0x06CA6351, 0xE003826F}, {0x14292967, 0x0A0E6E70},
    {0x27B70A85, 0x46D22FFC}, {0x2E1B2138, 0x5C26C926}, {0x4D2C6DFC, 0x5AC42AED}, {0x53380D13, 0x9D95B3DF},
    {0x650A7354, 0x8BAF63DE}, {0x766A0ABB, 0x3C77B2A8}, {0x81C2C92E, 0x47EDAEE6}, {0x92722C85, 0x1482353B},
    {0xA2BFE8A1, 0x4CF10364}, {0xA81A664B, 0xBC423001}, {0xC24B8B70, 0xD0F89791}, {0xC76C51A3, 0x0654BE30},
    {0xD192E819, 0xD6EF5218}, {0xD6990624, 0x5565A910}, {0xF40E3585, 0x5771202A}, {0x106AA070, 0x32BBD1B8},
    {0x19A4C116, 0xB8D2D0C8}, {0x1E376C08, 0x5141AB53}, {0x2748774C, 0xDF8EEB99}, {0x34B0BCB5, 0xE19B48A8},
    {0x391C0CB3, 0xC5C95A63}, {0x4ED8AA4A, 0xE3418ACB}, {0x5B9CCA4F, 0x7763E373}, {0x682E6FF3, 0xD6B2B8A3},
    {0x748F82EE, 0x5DEFB2FC}, {0x78A5636F, 0x43172F60}, {0x84C87814, 0xA1F0AB72}, {0x8CC70208, 0x1A6439EC},
    {0x90BEFFFA, 0x23631E28}, {0xA4506CEB, 0xDE82BDE9}, {0xBEF9A3F7, 0xB2C67915}, {0xC67178F2, 0xE372532B},
    {0xCA273ECE, 0xEA26619C}, {0xD186B8C7, 0x21C0C207}, {0xEADA7DD6, 0xCDE0EB1E}, {0xF57D4F7F, 0xEE6ED178},
    {0x06F067AA, 0x72176FBA}, {0x0A637DC5, 0xA2C898A6}, {0x113F9804, 0xBEF90DAE}, {0x1B710B35, 0x131C471B},
    {0x28DB77F5, 0x23047D84}, {0x32CAAB7B, 0x40C72493}, {0x3C9EBE0A, 0x15C9BEBC}, {0x431D67C4, 0x9C100D4C},
    {0x4CC5D4BE, 0xCB3E42B6}, {0x597F299C, 0xFC657E2A}, {0x5FCB6FAB, 0x3AD6FAEC}, {0x6C44198C, 0x4A475817}
};
#endif /* SY_LITTLEENDIANHOST */
#endif /* SY_INT64 */

#define SHA512_DIGEST_SIZE 64
#define SHA512_BLOCK_SIZE  128

#define SHFR(x, n)    (x >> n)
#define ROTR(x, n)   ((x >> n) | (x << ((sizeof(x) << 3) - n)))
#define ROTL(x, n)   ((x << n) | (x >> ((sizeof(x) << 3) - n)))
#define CH(x, y, z)  ((x & y) ^ (~x & z))
#define MAJ(x, y, z) ((x & y) ^ (x & z) ^ (y & z))

#ifdef SY_INT64
#define ADD_U64U64(a, b) a+=b;
#define ADD_U64SIZET(a, b) a+=b;
#define MUL8_U64(result, a) result = a * 8;

#define ROR64(x,n) (SHFR(x,n) | (x << (64 - n)))
#define CH1(x,y,z) ((z) ^ ((x) & ((y) ^ (z))))
#define MAJ2(x, y, z) (((x) & (y)) | ((x) & (z)) | ((y) & (z)))
#define SHA512_F1(x) (ROR64(x, 28) ^ ROR64(x, 34) ^ ROR64(x, 39))
#define SHA512_F2(x) (ROR64(x, 14) ^ ROR64(x, 18) ^ ROR64(x, 41))
#define SHA512_F3(x) (ROR64(x, 1) ^ ROR64(x, 8) ^ SHFR(x, 7))
#define SHA512_F4(x) (ROR64(x, 19) ^ ROR64(x, 61) ^ SHFR(x, 6))
#else /* #ifdef SY_INT64 */
#define ADD_U64U64(a, b)\
{\
    NQ_UINT32 low = a.low;\
    a.low += b.low;\
    if (a.low < low || a.low < b.low)\
    a.high++;\
    a.high += b.high;\
}

#define ADD_U64SIZET(a, b)\
{\
    NQ_UINT32 old = a.low;\
    a.low += b;\
    if (a.low < old || a.low < b)\
    a.high++;\
}

#define MUL8_U64(result, a)\
    result.low = a.low << 3;\
result.high = (a.high << 3) | (a.low >> 29);

#define CH1(res, x, y, z)\
    res.low = ((z.low) ^ ((x.low) & ((y.low) ^ (z.low))));\
res.high = ((z.high) ^ ((x.high) & ((y.high) ^ (z.high))));

#define ROR64(src, dst, n)\
    (dst.low = n > 32 ? src.low << (64 - n) | src.high >> (n - 32) :\
     src.low >> n | src.high << (32 - n));\
(dst.high = n > 32 ? src.low >> (n - 32) | src.high << (64 - n):\
 src.high >> n | src.low << (32 - n));

#define MAJ2(dst, x, y, z) dst.low = (((x.low) & (y.low)) | ((x.low) & (z.low)) | ((y.low) & (z.low)));\
                     dst.high = (((x.high) & (y.high)) | ((x.high) & (z.high)) | ((y.high) & (z.high)));

#define SHA512_F1(res, x)\
{\
    SHA_U64 t1;\
    ROR64(x, res, 28)\
    ROR64(x, t1, 34)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
    ROR64(x, t1, 39)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
}
#define SHA512_F2(res, x)\
{\
    SHA_U64 t1;\
    ROR64(x, res, 14)\
    ROR64(x, t1, 18)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
    ROR64(x, t1, 41)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
}
#define SHA512_F3(res, x)\
{\
    SHA_U64 t1;\
    ROR64(x, res, 1)\
    ROR64(x, t1, 8)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
    SHIFTR64(x, t1, 7)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
}
#define SHA512_F4(res, x)\
{\
    SHA_U64 t1;\
    ROR64(x, res, 19)\
    ROR64(x, t1, 61)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
    SHIFTR64(x, t1, 6)\
    res.low ^= t1.low;\
    res.high ^= t1.high;\
}
#endif /* #ifdef SY_INT64 */

typedef struct {
    union 
    {
        SHA_U64 hash [8];   /* hash working registers */
        NQ_BYTE digest [64];    /* hash result */
    }union1;

    union 
    {
        SHA_U64  workBuf [80];  /* work buffer s*/
        NQ_BYTE  dataBuf [128]; /* buffer to insert data */
    }union2;

    NQ_COUNT len;           /* currrent data length in buffer */

    SHA_U64 totalLen;
} sha512_ctx;

static void sha512Init(sha512_ctx *context)
{
    /*  initial hash value */
#ifdef SY_INT64
    SET_CONST64(context->union1.hash[0], 0x6A09E667F3BCC908ULL)
    SET_CONST64(context->union1.hash[1], 0xBB67AE8584CAA73BULL)
    SET_CONST64(context->union1.hash[2], 0x3C6EF372FE94F82BULL)
    SET_CONST64(context->union1.hash[3], 0xA54FF53A5F1D36F1ULL)
    SET_CONST64(context->union1.hash[4], 0x510E527FADE682D1ULL)
    SET_CONST64(context->union1.hash[5], 0x9B05688C2B3E6C1FULL)
    SET_CONST64(context->union1.hash[6], 0x1F83D9ABFB41BD6BULL)
    SET_CONST64(context->union1.hash[7], 0x5BE0CD19137E2179ULL)
    SET_CONST64(context->totalLen, 0)
#else /* SY_INT64 */
    SET_CONST64(context->union1.hash[0], 0xF3BCC908, 0x6A09E667)
    SET_CONST64(context->union1.hash[1], 0x84CAA73B, 0xBB67AE85)
    SET_CONST64(context->union1.hash[2], 0xFE94F82B, 0x3C6EF372)
    SET_CONST64(context->union1.hash[3], 0x5F1D36F1, 0xA54FF53A)
    SET_CONST64(context->union1.hash[4], 0xADE682D1, 0x510E527F)
    SET_CONST64(context->union1.hash[5], 0x2B3E6C1F, 0x9B05688C)
    SET_CONST64(context->union1.hash[6], 0xFB41BD6B, 0x1F83D9AB)
    SET_CONST64(context->union1.hash[7], 0x137E2179, 0x5BE0CD19)

    /* Total length of the message */
    SET_CONST64(context->totalLen, 0, 0)
#endif /* SY_INT64 */

    /* Number of bytes in the working buffer */
    context->len = 0;
}

static void sha512ProcessBlock(sha512_ctx *context)
{
    NQ_UINT t;
    SHA_U64 temp1;
    SHA_U64 temp2;
    volatile SHA_U64 *w = context->union2.workBuf;

    /* Init working registers */
    SHA_U64 a, b, c, d, e, f, g, h;
    SET_64 (a, context->union1.hash[0])
    SET_64 (b, context->union1.hash[1])
    SET_64 (c, context->union1.hash[2])
    SET_64 (d, context->union1.hash[3])
    SET_64 (e, context->union1.hash[4])
    SET_64 (f, context->union1.hash[5])
    SET_64 (g, context->union1.hash[6])
    SET_64 (h, context->union1.hash[7])

    /* process message that was copied to the buffer. 8 * 64 bit. 32 bit machines the 64 bit divided to hi, low */
#ifdef SY_LITTLEENDIANHOST
    /* Convert from big-endian byte order to host byte order if needed */
    for(t = 0; t < 16; t++)
        SWAP64(w[t], w[t])
#endif

    /* Prepare message */
    for(t = 16; t < 80; t++)
    {
#ifdef SY_INT64
        w[t] = SHA512_F4(w[t - 2]) + w[t - 7] + SHA512_F3(w[t - 15]) + w[t - 16];
#else
        SHA_U64 _t1;
        SHA512_F4(w[t], w[t-2])
        ADD_U64U64(w[t], w[t-7])
        ADD_U64U64(w[t], w[t - 16])
        SHA512_F3(_t1, w[t - 15])
        ADD_U64U64(w[t], _t1)
#endif
    }
    /* SHA-512 hashing */
    for(t = 0; t < 80; t++)
    {
#ifdef SY_INT64
        temp1 = h + SHA512_F2(e) + CH1(e, f, g) + k[t] + w[t];
        temp2 = SHA512_F1(a) + MAJ2(a, b, c);
#else
        SHA_U64 t1;
        SHA512_F2(temp1, e)
        ADD_U64U64(temp1, h)
        CH1(t1, e, f, g)
        ADD_U64U64(temp1, t1)
        ADD_U64U64(temp1, k[t])
        ADD_U64U64(temp1, w[t])
        SHA512_F1(temp2, a)
        MAJ2(t1, a, b, c)
        ADD_U64U64(temp2, t1)
#endif
        /* Update registers */
        SET_64(h, g)
        SET_64(g, f)
        SET_64(f, e)

#ifdef SY_INT64
        e = temp1 + d;
#else
        ADD_U64U64(d, temp1)
            SET_64(e, d);
#endif
        SET_64(d, c)
        SET_64(c, b)
        SET_64(b, a)
#ifdef SY_INT64
        a = temp1 + temp2;
#else
        ADD_U64U64(temp1, temp2)
        SET_64(a, temp1)
#endif
    }

    /* Update the hash value */
    ADD_U64U64(context->union1.hash[0], a)
    ADD_U64U64(context->union1.hash[1], b)
    ADD_U64U64(context->union1.hash[2], c)
    ADD_U64U64(context->union1.hash[3], d)
    ADD_U64U64(context->union1.hash[4], e)
    ADD_U64U64(context->union1.hash[5], f)
    ADD_U64U64(context->union1.hash[6], g)
    ADD_U64U64(context->union1.hash[7], h)
}

static void sha512Update(sha512_ctx *context, NQ_IOBufPos data, NQ_COUNT length)
{
    /* Process incoming data */
    while(length > 0)
    {
        /* Buffer can hold up to 128 bytes */
        NQ_COUNT len = length < (128 - context->len)? length : (128 - context->len);

        /* Copy data to buffer */
        IOBUF_MEMCPY_V2F(context->union2.dataBuf + context->len, data, len);

        /* Update context */
        context->len += len;
        ADD_U64SIZET(context->totalLen, len)

        /* Advance data pointer */
        IOBUF_MOVEBYTES(data, (NQ_INT)len);

        /* Remaining bytes from received data */
        length -= len;

        /* Process message  */
        if(context->len == 128)
        {
            /* digest buffer */
            sha512ProcessBlock(context);

            /* buffer is empty */
            context->len = 0;
        }
    }
}

static void sha512Finalize(sha512_ctx *context, NQ_IOBufPos digest)
{
#ifdef SY_LITTLEENDIANHOST
    NQ_UINT i;
#endif
    NQ_COUNT paddingSize;
    SHA_U64 totalLen;
    NQ_IOBufPos ioBufPosTmp;
    IOBUF_POSCONSTRUCTORINIT(ioBufPosTmp)

    /* Length of the original message (before padding) */
    MUL8_U64(totalLen, context->totalLen)

    /* Pad the message so that two bytes are left for closure */
    paddingSize = (context->len < 112) ? (112 - context->len) : (128 + 112 - context->len);
    IOBUF_POSCONSTRUCTOR(ioBufPosTmp, (NQ_BYTE *)pad, paddingSize)

    /* Append padding */
    sha512Update(context, ioBufPosTmp, paddingSize);

    /* Append the length of the original message */
#ifdef SY_INT64
    SET_CONST64(context->union2.workBuf[14], 0)
#else
    SET_CONST64(context->union2.workBuf[14], 0, 0)
#endif

#ifdef SY_LITTLEENDIANHOST
    /* in little endian the assignment is swapped because in process block there is swap back of the whole buffer. */
    SWAP64(context->union2.workBuf[15], totalLen)
#else
    SET_64(context->union2.workBuf[15], totalLen)
#endif

    /* Calculate the message digest */
    sha512ProcessBlock(context);

    /* Convert from host byte order to big-endian byte order if needed */
#ifdef SY_LITTLEENDIANHOST
    for(i = 0; i < 8; i++)
        SWAP64(context->union1.hash[i],context->union1.hash[i]);
#endif
    /* Copy the resulting digest */
    if(!IOBUF_ISNULL(digest))
        IOBUF_MEMCPY_F2V(digest, context->union1.digest, SHA512_DIGEST_SIZE);
    else
        syPrintf("digest is null\n");
}

void sha512Internal(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_IOBufPos buffer, NQ_COUNT bufferSize, NQ_BYTE *sha512_CtxBuf)
{
    sha512_ctx *ctx;
    NQ_COUNT i;
    NQ_BOOL isExternalCtxBuff = FALSE;

    if (sha512_CtxBuf)
    {
        ctx = (sha512_ctx *)sha512_CtxBuf;
        isExternalCtxBuff = TRUE;
    }
    else
        ctx = (sha512_ctx *)cmBufManTake(sizeof(sha512_ctx));

    if (ctx == NULL)
        goto Exit;

    sha512Init(ctx);

    for (i = 0; i < numFragments; ++i)
    {
        if (!IOBUF_ISNULL(dataFragments[i].data) && dataFragments[i].len > 0)
        {
            sha512Update(ctx, (NQ_IOBufPos)dataFragments[i].data, dataFragments[i].len);
        }
    }

    sha512Finalize(ctx, buffer);

    if (!isExternalCtxBuff)
        cmBufManGive((NQ_BYTE *)ctx);
Exit:
    return;
}

#endif /* #ifdef UD_NQ_INCLUDESMB3 */
