/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : MD5, HMACMD5 implementation
 *--------------------------------------------------------------------
 * MODULE        : Auth - AM
 * DEPENDENCIES  : None
 ********************************************************************/

#include "cmapi.h"


/*********************************************************************
 * MD5
 ********************************************************************/
typedef struct
{
    NQ_UINT32 buf[4];
    NQ_UINT32 bits[2];
    NQ_BYTE   in[64];
} MD5Context;

#define MD5_F1(x, y, z) (z ^ (x & (y ^ z)))
#define MD5_F2(x, y, z) MD5_F1(z, x, y)
#define MD5_F3(x, y, z) (x ^ y ^ z)
#define MD5_F4(x, y, z) (y ^ (x | ~z))

#define MD5_STEP(f, w, x, y, z, data, s) \
    ( w += f(x, y, z) + data,  w = w<<s | w>>(32-s),  w += x )

static void MD5_Init(
        MD5Context *ctx
        )
{
    ctx->buf[0] = 0x67452301;
    ctx->buf[1] = 0xefcdab89;
    ctx->buf[2] = 0x98badcfe;
    ctx->buf[3] = 0x10325476;

    ctx->bits[0] = 0;
    ctx->bits[1] = 0;
}

static void MD5_Transform(
        NQ_UINT32 buf[4],
        NQ_UINT32 const in[16]
        )
{
    register NQ_UINT32 a, b, c, d;

    a = buf[0];
    b = buf[1];
    c = buf[2];
    d = buf[3];

    MD5_STEP(MD5_F1, a, b, c, d, cmHtol32(in[0]) + 0xd76aa478, 7);
    MD5_STEP(MD5_F1, d, a, b, c, cmHtol32(in[1]) + 0xe8c7b756, 12);
    MD5_STEP(MD5_F1, c, d, a, b, cmHtol32(in[2]) + 0x242070db, 17);
    MD5_STEP(MD5_F1, b, c, d, a, cmHtol32(in[3]) + 0xc1bdceee, 22);
    MD5_STEP(MD5_F1, a, b, c, d, cmHtol32(in[4]) + 0xf57c0faf, 7);
    MD5_STEP(MD5_F1, d, a, b, c, cmHtol32(in[5]) + 0x4787c62a, 12);
    MD5_STEP(MD5_F1, c, d, a, b, cmHtol32(in[6]) + 0xa8304613, 17);
    MD5_STEP(MD5_F1, b, c, d, a, cmHtol32(in[7]) + 0xfd469501, 22);
    MD5_STEP(MD5_F1, a, b, c, d, cmHtol32(in[8]) + 0x698098d8, 7);
    MD5_STEP(MD5_F1, d, a, b, c, cmHtol32(in[9]) + 0x8b44f7af, 12);
    MD5_STEP(MD5_F1, c, d, a, b, cmHtol32(in[10]) + 0xffff5bb1, 17);
    MD5_STEP(MD5_F1, b, c, d, a, cmHtol32(in[11]) + 0x895cd7be, 22);
    MD5_STEP(MD5_F1, a, b, c, d, cmHtol32(in[12]) + 0x6b901122, 7);
    MD5_STEP(MD5_F1, d, a, b, c, cmHtol32(in[13]) + 0xfd987193, 12);
    MD5_STEP(MD5_F1, c, d, a, b, cmHtol32(in[14]) + 0xa679438e, 17);
    MD5_STEP(MD5_F1, b, c, d, a, cmHtol32(in[15]) + 0x49b40821, 22);

    MD5_STEP(MD5_F2, a, b, c, d, cmHtol32(in[1]) + 0xf61e2562, 5);
    MD5_STEP(MD5_F2, d, a, b, c, cmHtol32(in[6]) + 0xc040b340, 9);
    MD5_STEP(MD5_F2, c, d, a, b, cmHtol32(in[11]) + 0x265e5a51, 14);
    MD5_STEP(MD5_F2, b, c, d, a, cmHtol32(in[0]) + 0xe9b6c7aa, 20);
    MD5_STEP(MD5_F2, a, b, c, d, cmHtol32(in[5]) + 0xd62f105d, 5);
    MD5_STEP(MD5_F2, d, a, b, c, cmHtol32(in[10]) + 0x02441453, 9);
    MD5_STEP(MD5_F2, c, d, a, b, cmHtol32(in[15]) + 0xd8a1e681, 14);
    MD5_STEP(MD5_F2, b, c, d, a, cmHtol32(in[4]) + 0xe7d3fbc8, 20);
    MD5_STEP(MD5_F2, a, b, c, d, cmHtol32(in[9]) + 0x21e1cde6, 5);
    MD5_STEP(MD5_F2, d, a, b, c, cmHtol32(in[14]) + 0xc33707d6, 9);
    MD5_STEP(MD5_F2, c, d, a, b, cmHtol32(in[3]) + 0xf4d50d87, 14);
    MD5_STEP(MD5_F2, b, c, d, a, cmHtol32(in[8]) + 0x455a14ed, 20);
    MD5_STEP(MD5_F2, a, b, c, d, cmHtol32(in[13]) + 0xa9e3e905, 5);
    MD5_STEP(MD5_F2, d, a, b, c, cmHtol32(in[2]) + 0xfcefa3f8, 9);
    MD5_STEP(MD5_F2, c, d, a, b, cmHtol32(in[7]) + 0x676f02d9, 14);
    MD5_STEP(MD5_F2, b, c, d, a, cmHtol32(in[12]) + 0x8d2a4c8a, 20);

    MD5_STEP(MD5_F3, a, b, c, d, cmHtol32(in[5]) + 0xfffa3942, 4);
    MD5_STEP(MD5_F3, d, a, b, c, cmHtol32(in[8]) + 0x8771f681, 11);
    MD5_STEP(MD5_F3, c, d, a, b, cmHtol32(in[11]) + 0x6d9d6122, 16);
    MD5_STEP(MD5_F3, b, c, d, a, cmHtol32(in[14]) + 0xfde5380c, 23);
    MD5_STEP(MD5_F3, a, b, c, d, cmHtol32(in[1]) + 0xa4beea44, 4);
    MD5_STEP(MD5_F3, d, a, b, c, cmHtol32(in[4]) + 0x4bdecfa9, 11);
    MD5_STEP(MD5_F3, c, d, a, b, cmHtol32(in[7]) + 0xf6bb4b60, 16);
    MD5_STEP(MD5_F3, b, c, d, a, cmHtol32(in[10]) + 0xbebfbc70, 23);
    MD5_STEP(MD5_F3, a, b, c, d, cmHtol32(in[13]) + 0x289b7ec6, 4);
    MD5_STEP(MD5_F3, d, a, b, c, cmHtol32(in[0]) + 0xeaa127fa, 11);
    MD5_STEP(MD5_F3, c, d, a, b, cmHtol32(in[3]) + 0xd4ef3085, 16);
    MD5_STEP(MD5_F3, b, c, d, a, cmHtol32(in[6]) + 0x04881d05, 23);
    MD5_STEP(MD5_F3, a, b, c, d, cmHtol32(in[9]) + 0xd9d4d039, 4);
    MD5_STEP(MD5_F3, d, a, b, c, cmHtol32(in[12]) + 0xe6db99e5, 11);
    MD5_STEP(MD5_F3, c, d, a, b, cmHtol32(in[15]) + 0x1fa27cf8, 16);
    MD5_STEP(MD5_F3, b, c, d, a, cmHtol32(in[2]) + 0xc4ac5665, 23);

    MD5_STEP(MD5_F4, a, b, c, d, cmHtol32(in[0]) + 0xf4292244, 6);
    MD5_STEP(MD5_F4, d, a, b, c, cmHtol32(in[7]) + 0x432aff97, 10);
    MD5_STEP(MD5_F4, c, d, a, b, cmHtol32(in[14]) + 0xab9423a7, 15);
    MD5_STEP(MD5_F4, b, c, d, a, cmHtol32(in[5]) + 0xfc93a039, 21);
    MD5_STEP(MD5_F4, a, b, c, d, cmHtol32(in[12]) + 0x655b59c3, 6);
    MD5_STEP(MD5_F4, d, a, b, c, cmHtol32(in[3]) + 0x8f0ccc92, 10);
    MD5_STEP(MD5_F4, c, d, a, b, cmHtol32(in[10]) + 0xffeff47d, 15);
    MD5_STEP(MD5_F4, b, c, d, a, cmHtol32(in[1]) + 0x85845dd1, 21);
    MD5_STEP(MD5_F4, a, b, c, d, cmHtol32(in[8]) + 0x6fa87e4f, 6);
    MD5_STEP(MD5_F4, d, a, b, c, cmHtol32(in[15]) + 0xfe2ce6e0, 10);
    MD5_STEP(MD5_F4, c, d, a, b, cmHtol32(in[6]) + 0xa3014314, 15);
    MD5_STEP(MD5_F4, b, c, d, a, cmHtol32(in[13]) + 0x4e0811a1, 21);
    MD5_STEP(MD5_F4, a, b, c, d, cmHtol32(in[4]) + 0xf7537e82, 6);
    MD5_STEP(MD5_F4, d, a, b, c, cmHtol32(in[11]) + 0xbd3af235, 10);
    MD5_STEP(MD5_F4, c, d, a, b, cmHtol32(in[2]) + 0x2ad7d2bb, 15);
    MD5_STEP(MD5_F4, b, c, d, a, cmHtol32(in[9]) + 0xeb86d391, 21);

    buf[0] += a;
    buf[1] += b;
    buf[2] += c;
    buf[3] += d;
}

static void MD5_Update(
        MD5Context *ctx,
        NQ_IOBufPos buf,
        NQ_UINT len
        )
{
    NQ_UINT32 t;

    t = ctx->bits[0];
    if ((ctx->bits[0] = t + ((NQ_UINT32)len << 3)) < t)
        ctx->bits[1]++;
    ctx->bits[1] += len >> 29;

    t = (t >> 3) & 0x3f;

    if (t) 
    {
        NQ_BYTE *p = (NQ_BYTE *) ctx->in + t;

        t = 64-t;
        if (len < t) 
        {
            IOBUF_MEMCPY_V2F(p, buf, len);
            return;
        }
        IOBUF_MEMCPY_V2F(p, buf, t);
        MD5_Transform(ctx->buf, (NQ_UINT32 *) ctx->in);
        IOBUF_MOVEBYTES(buf, (NQ_INT)t);
        len = (NQ_UINT)(len - t);
    }

    while (len >= 64) 
    {
        IOBUF_MEMCPY_V2F(ctx->in, buf, 64);
        MD5_Transform(ctx->buf, (NQ_UINT32 *) ctx->in);
        IOBUF_MOVEBYTES(buf, 64);
        len -= 64;
    }

    IOBUF_MEMCPY_V2F(ctx->in, buf, len);
}

static void MD5_Final(
        MD5Context *ctx,
        NQ_BYTE *digest
        )
{
    NQ_UINT count, i;
    NQ_BYTE *p;
    NQ_UINT32 *t;

    count = (ctx->bits[0] >> 3) & 0x3F;

    p = ctx->in + count;
    *p++ = 0x80;

    count = 64 - 1 - count;

    if (count < 8) 
    {
        syMemset(p, 0, count);
        MD5_Transform(ctx->buf, (NQ_UINT32 *) ctx->in);

        syMemset(ctx->in, 0, 56);
    } 
    else 
    {
        syMemset(p, 0, count-8);
    }

    t = (NQ_UINT32 *)ctx->in;
    t[14] = cmHtol32(ctx->bits[0]);
    t[15] = cmHtol32(ctx->bits[1]);

    MD5_Transform(ctx->buf, (NQ_UINT32 *) ctx->in);

    for(i = 0; i < 4; i++)
        ctx->buf[i] = cmHtol32(ctx->buf[i]);

    syMemmove(digest, ctx->buf, 16);
    syMemset(ctx, 0, sizeof(MD5Context));
}


void md5Internal(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize)
{
    MD5Context ctx;
    NQ_COUNT i;

    MD5_Init(&ctx);
    for (i = 0; i < numFragments; i++)
    {
        if (!IOBUF_ISNULL(dataFragments[i].data) && dataFragments[i].len > 0 )
            MD5_Update(&ctx , dataFragments[i].data , dataFragments[i].len);
    }
    MD5_Final(&ctx, buffer);
}

/*********************************************************************
 * HMAC_MD5
 ********************************************************************/

#define HMAC_MAX_MD_CBLOCK 64

typedef struct {
    MD5Context md5;
    NQ_BYTE iPad[HMAC_MAX_MD_CBLOCK + 1];
    NQ_BYTE oPad[HMAC_MAX_MD_CBLOCK + 1];
} HMAC_MD5Context;

static void HMACMD5_Init(HMAC_MD5Context *ctx, const NQ_BYTE *key,
        NQ_UINT len) {

    NQ_INT i;
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    syMemset(ctx->iPad, 0, sizeof(ctx->iPad));
    syMemset(ctx->oPad, 0, sizeof(ctx->iPad));
    syMemcpy(ctx->iPad, key, len);
    syMemcpy(ctx->oPad, key, len);

    for (i = 0; i < HMAC_MAX_MD_CBLOCK; i++) {
        ctx->iPad[i] ^= 0x36;
        ctx->oPad[i] ^= 0x5c;
    }

    MD5_Init(&ctx->md5);
    IOBUF_POSCONSTRUCTOR(bufPos, ctx->iPad, sizeof(ctx->iPad))
    MD5_Update(&ctx->md5, bufPos, 64);
}

static void HMACMD5_Update(HMAC_MD5Context *ctx, const NQ_IOBufPos data,
        NQ_UINT len) {
    MD5_Update(&(ctx->md5), (NQ_IOBufPos)data, len);
}

static void HMACMD5_Final(HMAC_MD5Context *ctx, NQ_BYTE *md) {
    MD5Context tempCtx;
    NQ_IOBufPos bufPos;
    IOBUF_POSCONSTRUCTORINIT(bufPos)

    MD5_Final(&ctx->md5, md);

    MD5_Init(&tempCtx);
    IOBUF_POSCONSTRUCTOR(bufPos, ctx->oPad, sizeof(ctx->oPad))
    MD5_Update(&tempCtx, bufPos, 64);
    IOBUF_POSCONSTRUCTOR(bufPos, md, 16)
    MD5_Update(&tempCtx, bufPos, 16);
    MD5_Final(&tempCtx, md);
}

void hmacmd5Internal(const CMBlob * key, const CMBlob * key1, const CMIOBlob dataFragments[], NQ_COUNT numFragments, NQ_BYTE * buffer, NQ_COUNT bufferSize)
{
    HMAC_MD5Context ctx;
    NQ_COUNT i;

    /* key1 and bufferSize are not used */
    HMACMD5_Init(&ctx, key->data, key->len);
    for (i = 0; i < numFragments; i++)
        if (!IOBUF_ISNULL(dataFragments[i].data) && dataFragments[i].len > 0 )
            HMACMD5_Update(&ctx, dataFragments[i].data, dataFragments[i].len);
    HMACMD5_Final(&ctx, buffer);
}

