/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Implementation of file access command
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "csdataba.h"
#include "csparams.h"
#include "csdispat.h"
#include "csutils.h"
#include "csbreak.h"
#include "csdcerpc.h"
#include "csutils.h"

#ifdef UD_NQ_INCLUDECIFSSERVER

/* 
 * Local functions and data
 * ------------------------
 */ 

/* two following variables are used between command processing and saving late response (see below) */
static NQ_UINT32 dataCount;     /* number of bytes in WRITE(AndX) */
static CSFile* pFile;           /* pointer to file descriptor */

#ifdef UD_CS_INCLUDERPC

/* saving late response for Write */
static void
writeLateResponseSave(
    CSLateResponseContext* context
    );

/* preparing late response for Write */
static NQ_BOOL
writeLateResponsePrepare(
    CSLateResponseContext* context
    );

/* sending late response for Write */
static NQ_BOOL
writeLateResponseSend(
    CSLateResponseContext* context,
    NQ_UINT32 status,
    NQ_COUNT dataLength
    );

/* saving late response for WriteAndX */
static void
writeAndXLateResponseSave(
    CSLateResponseContext* context
    );

/* preparing late response for WriteAndX */
static NQ_BOOL
writeAndXLateResponsePrepare(
    CSLateResponseContext* context
    );

/* sending late response for WriteAndX */
static NQ_BOOL
writeAndXLateResponseSend(
    CSLateResponseContext* context,
    NQ_UINT32 status,
    NQ_COUNT dataLength
    );

#endif /* UD_CS_INCLUDERPC */

/* This code implements file read/write/seek commands
*/

/*====================================================================
 * PURPOSE: Perform FLUSH command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComFlush(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsFlushFileRequest* flushRequest;   /* casted request */
    CMCifsFlushFileResponse* flushResponse; /* casted response */
    NQ_UINT32 returnValue;                  /* error code in NT format or 0 for no error */
    CSFile* pFile;                          /* pointer to file descriptor */
    CSFid fid;                              /* required FID */
    CSPid pid;                              /* required PID */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pRequest:%p pHeaderOut:%p pResponse:%p", pRequest, pHeaderOut, pResponse);

    /* check space in output buffer */
    returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*flushResponse));
    if (0 != returnValue)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Buffer overflow");
        goto Exit;
    }

    /* cast pointers */
    flushRequest = (CMCifsFlushFileRequest*) pRequest;
    flushResponse = (CMCifsFlushFileResponse*) *pResponse;

    /* check format */
    if ((SMB_FLUSHFILE_REQUEST_WORDCOUNT != flushRequest->wordCount) || (0 != cmGetSUint16(flushRequest->byteCount)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal WordCount or ByteCount");
        returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }

    /* find file descriptor(s) */
    fid  = (CSFid)cmLtoh16(cmGetSUint16(flushRequest->fid));
    if (0xFFFF == fid)                  /* flush all files from the same PID */
    {
        NQ_INT numFiles = 0;

        pid = (CSPid)csGetPidFromHeader(pHeaderOut);
        fid = CS_ILLEGALID;

        /* cycle by all files of the same PID and flush them */
        while (NULL != (pFile = csGetNextFileByPid(pid, fid)))
        {
            fid = pFile->fid;
            numFiles++;
            if (NQ_SUCCESS != syFlushFile(pFile->file))
            {
                returnValue = csErrorGetLast();
            }

            /* even on error continue flushing */
        }

        if (0 == numFiles)                          /* no files flushed */
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "No files flushed for FID 0xFFFF");
            returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
            goto Exit;
        }

        if (NQ_SUCCESS != returnValue)              /* on any error - exit */
        {
            goto Exit;
        }
    }
    else
    {
        pFile = csGetFileByFid(fid, (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid)), (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
        if (NULL == pFile)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
            returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
            goto Exit;
        }

        /* flush file */
        if (NQ_SUCCESS != syFlushFile(pFile->file))
        {
            returnValue = csErrorGetLast();
            LOGERR(CM_TRC_LEVEL_ERROR, "Flush error");
            goto Exit;
        }
    }

    /* compose the response */
    flushResponse->wordCount = 0;
    cmPutSUint16(flushResponse->byteCount, 0);

    /* advance the outgoing response pointer */
    *pResponse += sizeof(*flushResponse);

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

/*====================================================================
 * PURPOSE: Perform READ command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComRead(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsReadFileRequest* readRequest;     /* casted request */
    CMCifsReadFileResponse* readResponse;   /* casted response */
    NQ_INT32 dataCount;                     /* number of bytes to read */
    CMCifsData* pDataBlock;                 /* DATA BLOCK pointer for response */
    CSFile* pFile;                          /* pointer to the file descriptor */
    NQ_INT32 actualCount;                   /* available data count */
    NQ_UINT32 offset;                       /* low bit portion of the offset */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent   eventInfo;
    CSUser          *   pUser;
    NQ_WCHAR noName[] = CM_WCHAR_NULL_STRING;
#endif /* UD_NQ_INCLUDEEVENTLOG */
    NQ_UINT32 returnValue;                  /* return code */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pRequest:%p pHeaderOut:%p pResponse:%p", pRequest, pHeaderOut, pResponse);

    /* cast pointers */
    readRequest = (CMCifsReadFileRequest*) pRequest;
    readResponse = (CMCifsReadFileResponse*) *pResponse;

    /* check format */
    if ((SMB_READFILE_REQUEST_WORDCOUNT != readRequest->wordCount) || (0 != cmGetSUint16(readRequest->byteCount)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal WordCount or ByteCount");
        returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }

    /* check available room in the buffer */
    dataCount = cmLtoh16(cmGetSUint16(readRequest->count));
    actualCount = CS_MAXBUFFERSIZE - sizeof(*readResponse) - sizeof(*pHeaderOut) - sizeof(*pDataBlock);
    if (dataCount > actualCount)
    {
        dataCount = actualCount;
    }

    /* find file */
    pFile = csGetFileByFid((CSFid)cmLtoh16(cmGetSUint16(readRequest->fid)), (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid)), (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        goto Exit;
    }

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid = cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pUser = csGetUserByUid(cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    eventInfo.rid = csGetUserRid(pUser);
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    pDataBlock = (CMCifsData*)(readResponse + 1);

#ifdef UD_CS_INCLUDERPC
    if (pFile->isPipe)
    {
        /* read from pipe */
        dataCount = cmLtoh16(cmGetSUint16(pDataBlock->length));
        dataCount = (NQ_INT32)csDcerpcRead(pFile, (NQ_BYTE*)(pDataBlock + 1), (NQ_UINT)dataCount, NULL);
        if (0 == dataCount)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Error reading from pipe");
            returnValue = csErrorReturn(SMB_STATUS_INVALID_PIPE_STATE, DOS_ERRbadpipe);
            goto Exit;
        }
    }
    else
#endif /* UD_CS_INCLUDERPC */
    {
        /* read from file */
        offset = cmLtoh32(cmGetSUint32(readRequest->offset));

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            eventInfo.offsetLow = offset;
            eventInfo.offsetHigh = 0;
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SEEK,
                pUser? pUser->name: noName,
                pUser? pUser->ip : cmSelfipGetZeroIp(),
                0,
                (const NQ_BYTE*)&eventInfo
            );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        actualCount = (NQ_INT32)sySeekFileStart(pFile->file, offset, 0);
        if (actualCount != (NQ_INT)cmLtoh32(cmGetSUint32(readRequest->offset)))
        {
            returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SEEK,
                pUser ? pUser->name : noName,
                pUser ? pUser->ip : cmSelfipGetZeroIp(),
                returnValue,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
            LOGERR(CM_TRC_LEVEL_ERROR, "LSEEK failed, Required: %lu, returned: %lu", cmLtoh32(cmGetSUint32((NQ_ULONG)(readRequest->offset))), (NQ_ULONG)actualCount);
            goto Exit;
        }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_SEEK,
            pUser? pUser->name : noName,
            pUser? pUser->ip : cmSelfipGetZeroIp(),
            0,
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

        dataCount = syReadFile(pFile->file, (NQ_BYTE*)(pDataBlock + 1), (NQ_COUNT)dataCount);
        if (dataCount < 0)
        {
            returnValue = csErrorGetLast();
            LOGERR(CM_TRC_LEVEL_ERROR, "READ failed");
            goto Exit;
        }
        
        /* update file offsets */
        pFile->offsetLow = offset + (NQ_UINT32)dataCount;
        if (pFile->offsetLow < offset)
        {
            pFile->offsetHigh++;
        }
    }

    /* compose the response */
    readResponse->wordCount = SMB_READFILE_RESPONSE_WORDCOUNT;
    cmPutSUint16(readResponse->count, cmHtol16((NQ_UINT16)dataCount));
    {
        NQ_INT i;  /* just a counter */

        for (i = 0; i < 4; i++)
        {
            cmPutSUint16(readResponse->reserved[i], 0);
        }
    }
    cmPutSUint16(readResponse->byteCount, cmHtol16((NQ_UINT16)(dataCount +(NQ_UINT16) sizeof(*pDataBlock))));
    pDataBlock->identifier = SMB_FIELD_DATABLOCK;
    cmPutSUint16(pDataBlock->length, cmHtol16((NQ_UINT16)dataCount));

    /* advance the outgoing response pointer */
    *pResponse += (NQ_UINT)(sizeof(*readResponse) + sizeof(*pDataBlock) + (NQ_UINT)dataCount);

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

/*====================================================================
 * PURPOSE: Perform READ ANDX command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComReadAndX(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsReadAndXRequest* readRequest;     /* casted request */
    CMCifsReadAndXResponse* readResponse;   /* casted response */
    NQ_UINT32 immediateDataCount;           /* number of bytes in the packet (not including DT) */
    CSFile* pFile;                          /* pointer to the file descriptor */
    NQ_BYTE* dataPtr;                       /* pointer to the data buffer */
    NQ_UINT32 actualCount;                  /* available data count */
    NQ_UINT32 offsetLow;                    /* low bit portion of the offset */
    NQ_UINT32 offsetHigh;                   /* high bit portion of the offset */
    NQ_UINT padding;                        /* number of padded bytes */
    NQ_UINT32 dataLength = 0;               /* data length total */
    CSSession *session;                     /* session pointer */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent   eventInfo;
    CSUser          *   pUser;
    NQ_WCHAR noName[] = CM_WCHAR_NULL_STRING;
#endif /* UD_NQ_INCLUDEEVENTLOG */
    NQ_UINT32 returnValue;                  /* return code */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pRequest:%p pHeaderOut:%p pResponse:%p", pRequest, pHeaderOut, pResponse);

    /* cast pointers */
    readRequest = (CMCifsReadAndXRequest*) pRequest;
    readResponse = (CMCifsReadAndXResponse*) *pResponse;

    /* calculate alignment and fill the padding with zeroes */
    dataPtr = (NQ_BYTE*)(readResponse + 1);
    dataPtr += (UD_FS_BUFFERALIGNMENT + 1) - ((NQ_ULONG)dataPtr & UD_FS_BUFFERALIGNMENT);
    padding = (NQ_UINT)((NQ_ULONG)dataPtr - (NQ_ULONG)(readResponse + 1));

    /* get data length */
    session = csGetSessionBySocket(); 
    if (session && !(session->capabilities & SMB_CAP_LARGE_READX) && (cmLtoh16(cmGetSUint16(readRequest->maxCountHigh)) > 0))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Requested data count is 64K or more");
        returnValue = csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
        goto Exit;
    }

    /* check the next AndX command */
    if ((SMB_COM_CLOSE != readRequest->andXCommand) && (0xFF != readRequest->andXCommand))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal next AndX command");
        returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }

    /* find file */
    pFile = csGetFileByFid((CSFid)cmLtoh16(cmGetSUint16(readRequest->fid)), (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid)), (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        goto Exit;
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid = cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pUser = csGetUserByUid(cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    eventInfo.rid = csGetUserRid(pUser);
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    dataLength = (NQ_UINT32)(cmLtoh16(cmGetSUint16(readRequest->maxCount)));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "dataLengthHigh: %d, dataLengthLow: %d, total: %d", cmLtoh16(cmGetSUint16(readRequest->maxCountHigh)), cmLtoh16(cmGetSUint16(readRequest->maxCount)), dataLength);

    /* check available room in the buffer */
    actualCount = (NQ_UINT32)(CS_MAXBUFFERSIZE - sizeof(*readResponse) - sizeof (*pHeaderOut) - padding);
    if (dataLength > actualCount)
    {
        dataLength = actualCount;
    }
#ifdef UD_CS_INCLUDERPC
    if (pFile->isPipe)
    {
        /* read from pipe */
        /* do not use dataCountHigh since it is Timeout */
        dataLength = (NQ_UINT32)csDcerpcRead(pFile, dataPtr, (NQ_UINT)dataLength, NULL);
        if (0 == dataLength)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "error reading from pipe");
            returnValue = csErrorReturn(SMB_STATUS_INVALID_PIPE_STATE, DOS_ERRbadpipe);
            goto Exit;
        }
        immediateDataCount = dataLength;
    }
    else
#endif /* UD_CS_INCLUDERPC */
    {
        NQ_UINT16    maxCountHigh;

        /* use dataCountHigh */
        maxCountHigh = (NQ_UINT16)cmLtoh16(cmGetSUint16(readRequest->maxCountHigh));
        maxCountHigh = (NQ_UINT16)(maxCountHigh == 0xffff ? 0 : maxCountHigh);
        dataLength |= ((NQ_UINT32)maxCountHigh << 16);

        /* shift to the read position */
        if (readRequest->wordCount == SMB_READANDX_REQUEST_WORDCOUNT1)
        {
            offsetHigh = cmLtoh32(cmGetSUint32(((CMCifsReadAndXRequest1*)readRequest)->offsetHigh));
        }
        else
        {
            offsetHigh = 0;
        }
        offsetLow = cmLtoh32(cmGetSUint32(readRequest->offset));

        if ((pFile->offsetLow != offsetLow) || (pFile->offsetHigh != offsetHigh))
        {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            eventInfo.offsetLow = offsetLow;
            eventInfo.offsetHigh = offsetHigh;
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SEEK,
                pUser? pUser->name : noName,
                pUser? pUser->ip : cmSelfipGetZeroIp(),
                0,
                (const NQ_BYTE*)&eventInfo
            );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
            if ((NQ_UINT32)NQ_FAIL == sySeekFileStart(pFile->file, offsetLow, offsetHigh))
            {
                returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SEEK,
                    pUser ? pUser->name : noName,
                    pUser ? pUser->ip : cmSelfipGetZeroIp(),
                    returnValue,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                LOGERR(CM_TRC_LEVEL_ERROR, "LSEEK failed");
                goto Exit;
            }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SEEK,
                pUser? pUser->name : noName,
                pUser? pUser->ip : cmSelfipGetZeroIp(),
                0,
                (const NQ_BYTE*)&eventInfo
            );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
            pFile->offsetLow = offsetLow;
            pFile->offsetHigh = offsetHigh;
        }

        /* read from file */
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        /* for DT determine end of file before actual reading from file into socket,
         * and actual read count (can be less than requested due to end of file),
         * because SMB header with status and dataCount is returned first, then the payload of DT */
        if (csDispatchIsDtOut() && (dataLength > 0))
        {
            NQ_UINT64 remaining;        /* remaining file portion to read */
            NQ_UINT64 fileSize;         /* file size */
            NQ_UINT64 count;            /* dataCount in uint64 */
            NQ_UINT64 offset;           /* offset in uint64 */

            /* get file size */
            if (NQ_FAIL == syGetFileSize(pFile->file, &fileSize))
            {
                returnValue = csErrorGetLast();
                LOGERR(CM_TRC_LEVEL_ERROR,  "DT Read failed: file size");
                goto Exit;
            }

            /* offset >= fileSize */
            offset.high = offsetHigh;
            offset.low = offsetLow;
            if (0 <= cmU64Cmp(&offset, &fileSize))
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "DT Read failed: end of file");
                returnValue = csErrorReturn(SMB_STATUS_END_OF_FILE, SRV_ERRqeof);
                goto Exit;
            }

            /* remaining = fileSize - offset */
            cmU64SubU64U64(&remaining, &fileSize, &offset);

            /* remaining < count */
            count.high = 0;
            count.low = dataLength;
            if (-1 == cmU64Cmp(&remaining, &count))
            {
                dataLength = remaining.low;
            }

            csDispatchDtSet(pFile->file, (NQ_COUNT)dataLength);
            immediateDataCount = 0;
        }
        else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
        {
            NQ_INT result = syReadFile(pFile->file, dataPtr, (NQ_COUNT)dataLength);

            dataLength = (NQ_UINT32)result;
            if (result < 0)
            {
                returnValue = csErrorGetLast();
                LOGERR(CM_TRC_LEVEL_ERROR, "READ failed");
                goto Exit;
            }
            immediateDataCount = dataLength;
        }
        
        /* update file offsets */
        pFile->offsetLow += (NQ_UINT32)dataLength;
        if (pFile->offsetLow < offsetLow)
        {
            pFile->offsetHigh++;
        }
    }

    /* compose the response */
    readResponse->wordCount = SMB_READANDX_RESPONSE_WORDCOUNT;
    readResponse->andXCommand = readRequest->andXCommand;
    readResponse->andXReserved = 0;
    if (0xFF == readResponse->andXCommand)
    {
        cmPutSUint16(readResponse->andXOffset, 0);
    }
    else
    {
        cmPutSUint16(readResponse->andXOffset, cmHtol16((NQ_UINT16)(*pResponse - (NQ_BYTE*)pHeaderOut)));
    }

    cmPutSUint16(readResponse->remaining, cmHtol16(0xFFFF));     /* as required by CIFS */
    cmPutSUint16(readResponse->dataCompactionMode, 0);
    cmPutSUint16(readResponse->reserved1, 0);
    cmPutSUint16(readResponse->dataLength, cmHtol16((NQ_UINT16)(dataLength & 0xFFFF)));
    cmPutSUint16(readResponse->dataOffset, cmHtol16((NQ_UINT16)(dataPtr - (NQ_BYTE*)pHeaderOut)));

    {
        NQ_INT i = 0;

        if (dataLength >= 0x00010000)  /* more or equal to 64K */
        {
            cmPutSUint16(readResponse->reserved2[i++], cmHtol16((NQ_UINT16)((dataLength & 0xFFFF0000) >> 16))); /* dataLength high */
        }
        for (   ; i < 5; i++)
        {
            cmPutSUint16(readResponse->reserved2[i], 0);
        }
    }

    cmPutSUint16(
        readResponse->byteCount,
        cmHtol16((NQ_UINT16)(dataPtr + dataLength  - (NQ_BYTE*)(&readResponse->byteCount) - (NQ_UINT16)sizeof(readResponse->byteCount)))
        );

    /* advance the outgoing response pointer */
    *pResponse = dataPtr + immediateDataCount;

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

/*====================================================================
 * PURPOSE: Perform WRITE command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComWrite(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsWriteBytesRequest* writeRequest;  /* casted request */
    NQ_UINT32 returnValue;                  /* error code in NT format or 0 for no error */
    CMCifsData* pDataBlock;                 /* DATA BLOCK pointer for response */
    NQ_UINT32 offset;                       /* required offset */
    CMCifsWriteBytesResponse* writeResponse;/* casted response */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent   eventInfo;
    CSUser  *           pUser;
    NQ_WCHAR noName[] = CM_WCHAR_NULL_STRING;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pRequest:%p pHeaderOut:%p pResponse:%p", pRequest, pHeaderOut, pResponse);

    dataCount = 0;                   /* number of bytes to read */
        
    /* check available room in the buffer and set response flags */
    writeResponse = (CMCifsWriteBytesResponse*) *pResponse;
    returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*writeResponse));
    if (0 != returnValue)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Buffer overflow");
        goto Exit;
    }

    /* cast pointers */
    writeRequest = (CMCifsWriteBytesRequest*) pRequest;
    pDataBlock = (CMCifsData*)(writeRequest + 1);

    /* check format */
    if ((SMB_WRITEBYTES_REQUEST_WORDCOUNT != writeRequest->wordCount) || (SMB_FIELD_DATABLOCK != pDataBlock->identifier))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal WordCount or BufferFormat");
        returnValue =  csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }

    /* find file */
    pFile = csGetFileByFid((CSFid)cmLtoh16(cmGetSUint16(writeRequest->fid)), (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid)), (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        goto Exit;
    }

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid = cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pUser = csGetUserByUid(cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    eventInfo.rid = csGetUserRid(pUser);
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    offset = cmLtoh32(cmGetSUint32(writeRequest->offset));

#ifdef UD_CS_INCLUDERPC
    if (pFile->isPipe)
    {
        NQ_UINT32 returnValue;  /* RPC return code */

        dataCount = (NQ_UINT32)cmLtoh16(cmGetSUint16(pDataBlock->length));
        csDcerpcSetLateResponseCallbacks(
            writeLateResponseSave,
            writeLateResponsePrepare,
            writeLateResponseSend
        );
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        /* discard possible DirectTransfer and read the payload */
        csDispatchDtDiscard();
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
        returnValue = csDcerpcWrite(pFile, (NQ_BYTE*)(pDataBlock + 1), (NQ_UINT)cmLtoh16(cmGetSUint16(pDataBlock->length)), FALSE);
        if (0 != returnValue)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Error writing to pipe");
            goto Exit;
        }
    }
    else
#endif /* UD_CS_INCLUDERPC */
    {
#ifdef UD_CS_INCLUDERPC_SPOOLSS
        if (pFile->isPrint)
        {
            NQ_INT32 written;
            SYPrinterJob *pJob = NULL;
            void *p = NULL;
            
            dataCount = cmLtoh16(cmGetSUint16(pDataBlock->length));
            csDcerpcSetLateResponseCallbacks(
                writeLateResponseSave,
                writeLateResponsePrepare,
                writeLateResponseSend
            );
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
            /* discard possible DirectTransfer and read the payload */
            csDispatchDtDiscard();
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
            written = syWritePrintData(pFile->printerHandle, (NQ_UINT32)pFile->file, (NQ_BYTE*)(pDataBlock + 1), cmLtoh16(cmGetSUint16(pDataBlock->length)), &p);
            pJob = (SYPrinterJob *)p;

            if ((NQ_UINT32)written != dataCount)
            {
                if (0 == written)
                {
                    /* 0 bytes written or response has to be delayed */
                    if ((NULL != pJob) && (NULL != pJob->delayed.ctx))
                    {
                        /* response has to be delayed */
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Response is delayed");
                        csDcerpcSaveResponseContext(FALSE, NULL, (CSDcerpcResponseContext *)pJob->delayed.ctx);
                        returnValue = SMB_STATUS_NORESPONSE;
                        goto Exit;
                    }
                }
                LOGERR(CM_TRC_LEVEL_ERROR, "WRITE failed (syWritePrintData)");
                if (written >= 0)
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Required: %d bytes, written: %d bytes", cmLtoh16(cmGetSUint16(pDataBlock->length)), written);
                }
                returnValue = csErrorGetLast();
                goto Exit;
            }
        }
        else
#endif /* UD_CS_INCLUDERPC_SPOOLSS */       
        {
            if (0 == cmGetSUint16(writeRequest->count))
            {
                /* truncate file */
                returnValue = csTruncateFile(pFile, NULL, offset, 0);
                if (NQ_SUCCESS != returnValue)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Truncate failed");
                    goto Exit;
                }
            }
            else
            {
                /* check buffer size */
#ifndef UD_CS_INCLUDEDIRECTTRANSFER
                if (cmLtoh16(cmGetSUint16(pDataBlock->length)) > (CIFS_MAX_DATA_SIZE16 - sizeof(*writeRequest) - sizeof(*pDataBlock) - sizeof (*pHeaderOut)))
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Data overflow");
                    returnValue = csErrorReturn(SMB_STATUS_BUFFER_TOO_SMALL, DOS_ERRinsufficientbuffer);
                    goto Exit;
                }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

                /* write to file */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
            eventInfo.offsetLow = offset;
            eventInfo.offsetHigh = 0;
            eventInfo.before = TRUE;
            udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_SEEK,
                pUser? pUser->name : noName,
                pUser? pUser->ip : cmSelfipGetZeroIp(),
                0,
                (const NQ_BYTE*)&eventInfo
            );
            eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                if (offset != sySeekFileStart(pFile->file, offset, 0))
                {
                    returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_SEEK,
                        pUser ? pUser->name : noName,
                        pUser ? pUser->ip : cmSelfipGetZeroIp(),
                        returnValue,
                        (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
                    LOGERR(CM_TRC_LEVEL_ERROR, "LSEEK failed");
                    goto Exit;
                }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                udEventLog(
                    UD_LOG_MODULE_CS,
                    UD_LOG_CLASS_FILE,
                    UD_LOG_FILE_SEEK,
                    pUser ? pUser->name : noName,
                    pUser ? pUser->ip : cmSelfipGetZeroIp(),
                    0,
                    (const NQ_BYTE*)&eventInfo
                );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
                if (csDispatchIsDtIn())
                {
                    csDispatchDtSet(pFile->file, dataCount);
                }
                else
#else /* UD_CS_INCLUDEDIRECTTRANSFER */
                {
                    NQ_INT result = syWriteFile(pFile->file, (NQ_BYTE*)(pDataBlock + 1), (NQ_COUNT)cmLtoh16(cmGetSUint16(pDataBlock->length)));
                    dataCount = (NQ_UINT32)result;
                    if (dataCount != cmLtoh16(cmGetSUint16(pDataBlock->length)))
                    {
                        returnValue = csErrorGetLast();
                        LOGERR(CM_TRC_LEVEL_ERROR, "WRITE failed");
                        if (result >= 0)
                        {
                            LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Required: %d bytes, written: %d bytes", cmLtoh16(cmGetSUint16(pDataBlock->length)), dataCount);
                        }
                        goto Exit;
                    }
                }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
                csGetNameByNid(pFile->nid)->isDirty = TRUE;
#ifdef UD_FS_FLUSHIFMODIFIED
                pFile->wasModified = TRUE;
#endif /* UD_FS_FLUSHIFMODIFIED */
            }

            /* update file offsets */
            pFile->offsetLow = offset + dataCount;
            if (pFile->offsetLow < offset)
            {
                pFile->offsetHigh++;
            }
        }              
    }
   
    /* compose the response */
    writeResponse->wordCount = SMB_WRITEBYTES_RESPONSE_WORDCOUNT;
    cmPutSUint16(writeResponse->count, cmHtol16((NQ_UINT16)dataCount));
    cmPutSUint16(writeResponse->byteCount, 0);

    /* advance the outgoing response pointer */
    *pResponse += sizeof(*writeResponse);

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

/*====================================================================
 * PURPOSE: Perform WRITE command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComWriteAndX(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsWriteAndXRequest* writeRequest;   /* casted request */
    CMCifsWriteAndXResponse* writeResponse; /* casted response */
    NQ_UINT32 returnValue;                  /* error code in NT format or 0 for no error */
    NQ_UINT32 offsetLow;                    /* required offset - low bits */
    NQ_UINT32 offsetHigh;                   /* required offset - high bits */
    NQ_BYTE* pData;                         /* pointer to the data block */
    NQ_UINT32 dataLength = 0;               /* data length total */
    NQ_UINT32 dataWritten = 0;              /* data written */
    CSSession *session;                     /* session pointer */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent   eventInfo;
    CSUser          *   pUser;
    NQ_WCHAR noName[] = CM_WCHAR_NULL_STRING;
#endif /* UD_NQ_INCLUDEEVENTLOG */
    
    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pRequest:%p pHeaderOut:%p pResponse:%p", pRequest, pHeaderOut, pResponse);
        
    /* check available room in the buffer and set response flags */
    returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*writeResponse));
    if (0 != returnValue)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Buffer overflow");
        goto Exit;
    }

    /* cast pointers */
    writeRequest = (CMCifsWriteAndXRequest*) pRequest;
    writeResponse = (CMCifsWriteAndXResponse*) *pResponse;

    /* check the next AndX command */
    switch(writeRequest->andXCommand)
    {
        case SMB_COM_READ:
        case SMB_COM_READ_ANDX:
        case SMB_COM_LOCK_AND_READ:
        case SMB_COM_WRITE_ANDX:
        case SMB_COM_CLOSE:
        case 0xFF:
            break;
        default:
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal next AndX command");
            returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
            goto Exit;
    }

    /* find file */
    pFile = csGetFileByFid((CSFid)cmLtoh16(cmGetSUint16(writeRequest->fid)), (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid)), (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        goto Exit;
    }

    /* get data length */
    session = csGetSessionBySocket(); 
    if (session && !(session->capabilities & SMB_CAP_LARGE_WRITEX) && (cmLtoh16(cmGetSUint16(writeRequest->dataLengthHigh)) > 0))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Requested data count is 64K or more");
        returnValue = csErrorReturn(SMB_STATUS_ACCESS_DENIED, DOS_ERRnoaccess);
        goto Exit;
    }
    dataLength = (NQ_UINT32)(cmLtoh16(cmGetSUint16(writeRequest->dataLength))) | ((NQ_UINT32)cmLtoh16(cmGetSUint16(writeRequest->dataLengthHigh)) << 16);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "dataLengthHigh: %d, dataLengthLow: %d, total: %d", cmLtoh16(cmGetSUint16(writeRequest->dataLengthHigh)), cmLtoh16(cmGetSUint16(writeRequest->dataLength)), dataLength);

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid = cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pUser = csGetUserByUid(cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    eventInfo.rid = csGetUserRid(pUser);
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    /* check buffer size */
#ifndef UD_CS_INCLUDEDIRECTTRANSFER
    if (dataLength > (NQ_INT32)(CS_MAXBUFFERSIZE - sizeof(*writeRequest) - sizeof(*pHeaderOut)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Data overflow");
        returnValue = csErrorReturn(SMB_STATUS_BUFFER_TOO_SMALL, DOS_ERRinsufficientbuffer);
        goto Exit;
    }
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

    pData = pRequest - sizeof(CMCifsHeader) + cmLtoh16(cmGetSUint16(writeRequest->dataOffset));

#ifdef UD_CS_INCLUDERPC
    if (pFile->isPipe)
    {
        NQ_UINT32 returnValue;  /* RPC return code */

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
        /* discard possible DirectTransfer and read the payload */
        csDispatchDtDiscard();
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

        csDcerpcSetLateResponseCallbacks(
            writeAndXLateResponseSave,
            writeAndXLateResponsePrepare,
            writeAndXLateResponseSend
        );
        returnValue = csDcerpcWrite(
                pFile, 
                pData, 
                (NQ_UINT)dataLength, 
                FALSE
                );
        if (0 != returnValue)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "error writing to pipe");
            goto Exit;
        }
        dataWritten = dataLength;
    }
    else
#endif /* UD_CS_INCLUDERPC */
    {
#ifdef UD_CS_INCLUDERPC_SPOOLSS
        if (pFile->isPrint)
        {
            NQ_INT32 written;
            SYPrinterJob *pJob = NULL;
            void *p = NULL;

#ifdef UD_CS_INCLUDEDIRECTTRANSFER
            /* discard possible DirectTransfer and read the payload */
            csDispatchDtDiscard();
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */

            csDcerpcSetLateResponseCallbacks(
                    writeAndXLateResponseSave,
                    writeAndXLateResponsePrepare,
                    writeAndXLateResponseSend
            );
            written = syWritePrintData(pFile->printerHandle, (NQ_UINT32)pFile->file, pData, dataLength, &p);
            pJob = (SYPrinterJob *)p;
            
            if (written <= 0)
            {
                if (0 == written)
                {
                    /* 0 bytes written or response has to be delayed */
                    if ((NULL != pJob) && (NULL != pJob->delayed.ctx))
                    {
                        /* response has to be delayed */
                        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Response is delayed");
                        csDcerpcSaveResponseContext(FALSE, NULL, (CSDcerpcResponseContext *)pJob->delayed.ctx);
                        returnValue = SMB_STATUS_NORESPONSE;
                        goto Exit;
                    }
                }
                LOGERR(CM_TRC_LEVEL_ERROR, "WRITE failed (syWritePrintData)");
                returnValue = csErrorGetLast();
                goto Exit;
            }
            dataWritten = (NQ_UINT32)written;
        }
        else
#endif /* UD_CS_INCLUDERPC_SPOOLSS */       
        {
            offsetLow = cmLtoh32(cmGetSUint32(writeRequest->offset));
            if (writeRequest->wordCount == SMB_WRITEANDX_REQUEST_WORDCOUNT1)
            {
                offsetHigh = cmLtoh32(cmGetSUint32(((CMCifsWriteAndXRequest1*)writeRequest)->offsetHigh));
            }
            else
            {
                offsetHigh = 0;
            }

            if (0 == dataLength)
            {
                /* truncate file */
                returnValue = csTruncateFile(pFile, NULL, offsetLow, offsetHigh);
                if (NQ_SUCCESS != returnValue)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Truncate failed");
                    goto Exit;
                }
            }
            else
            {
                /* shift to the write position */
                if ((pFile->offsetLow != offsetLow) || (pFile->offsetHigh != offsetHigh))
                {
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                    eventInfo.offsetLow = offsetLow;
                    eventInfo.offsetHigh = offsetHigh;
                    eventInfo.before = TRUE;
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_SEEK,
                        pUser? pUser->name : noName,
                        pUser? pUser->ip : cmSelfipGetZeroIp(),
                        0,
                        (const NQ_BYTE*)&eventInfo
                    );
                    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    if ((NQ_UINT32)NQ_FAIL == sySeekFileStart(pFile->file, offsetLow, offsetHigh))
                    {
                        returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                        udEventLog(
                            UD_LOG_MODULE_CS,
                            UD_LOG_CLASS_FILE,
                            UD_LOG_FILE_SEEK,
                            pUser ? pUser->name : noName,
                            pUser ? pUser->ip : cmSelfipGetZeroIp(),
                            returnValue,
                            (const NQ_BYTE*)&eventInfo
                        );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                        LOGERR(CM_TRC_LEVEL_ERROR, "LSEEK failed");
                        goto Exit;
                    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
                    udEventLog(
                        UD_LOG_MODULE_CS,
                        UD_LOG_CLASS_FILE,
                        UD_LOG_FILE_SEEK,
                        pUser? pUser->name : noName,
                        pUser? pUser->ip : cmSelfipGetZeroIp(),
                        0,
                        (const NQ_BYTE*)&eventInfo
                    );
#endif /* UD_NQ_INCLUDEEVENTLOG */
                    pFile->offsetLow = offsetLow;
                    pFile->offsetHigh = offsetHigh;
                }

                /* write to file */
#ifdef UD_CS_INCLUDEDIRECTTRANSFER
                if (csDispatchIsDtIn())
                {
                    csDispatchDtSet(pFile->file, dataLength);
                    dataWritten = dataLength;
                }
                else
#endif /* UD_CS_INCLUDEDIRECTTRANSFER */
                {
                    dataWritten = (NQ_UINT32)syWriteFile(pFile->file, pData, (NQ_COUNT)dataLength);
                    if ((NQ_INT)dataWritten < 0)
                    {
                        returnValue = csErrorGetLast();
                        LOGERR(CM_TRC_LEVEL_ERROR, "WRITE_ANDX failed");
                        goto Exit;
                    }
                }
                csGetNameByNid(pFile->nid)->isDirty = TRUE;
#ifdef UD_FS_FLUSHIFMODIFIED
                pFile->wasModified = TRUE;
#endif /* UD_FS_FLUSHIFMODIFIED */
            }
            
            /* update file offsets */
            pFile->offsetLow += dataWritten;
            if (pFile->offsetLow < offsetLow)
            {
                pFile->offsetHigh++;
            }
        }
    }

    /*  reset times for file */
    /*csResetFileTimes(csGetNameByNid(pFile->nid));*/

    /* advance the outgoing response pointer */
    *pResponse += sizeof(*writeResponse);

    /* compose the response */
    writeResponse->wordCount = SMB_WRITEANDX_RESPONSE_WORDCOUNT;
    writeResponse->andXCommand = writeRequest->andXCommand;
    if (writeResponse->andXCommand == 0xFF)
    {
        cmPutSUint16(writeResponse->andXOffset, 0);
    }
    else
    {
        cmPutSUint16(writeResponse->andXOffset, cmHtol16((NQ_UINT16)(*pResponse - (NQ_BYTE*)pHeaderOut)));
    }
    writeResponse->andXReserved = 0;
    cmPutSUint16(writeResponse->count, cmHtol16(dataWritten & 0xFFFF));
    cmPutSUint16(writeResponse->remaining, cmHtol16(0xFFFF)); /* should be -1 */
    cmPutSUint16(writeResponse->countHigh, (NQ_UINT16)cmHtol16((dataWritten & 0xFFFF0000) >> 16));
    cmPutSUint16(writeResponse->reserved, 0);
    cmPutSUint16(writeResponse->byteCount, 0);

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

/*====================================================================
 * PURPOSE: Perform SEEK command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComSeek(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsSeekRequest* seekRequest;         /* casted request */
    CMCifsSeekResponse* seekResponse;       /* casted response */
    NQ_UINT32 returnValue;                  /* error code in NT format or 0 for no error */
    CSFile* pFile;                          /* pointer to the file descriptor */
    NQ_UINT32 offset;                       /* required offset */
    NQ_UINT16 mode;                         /* positioning mode */
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    UDFileAccessEvent   eventInfo;
    CSUser          *   pUser;
    NQ_WCHAR noName[] = CM_WCHAR_NULL_STRING;
#endif /* UD_NQ_INCLUDEEVENTLOG */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pRequest:%p pHeaderOut:%p pResponse:%p", pRequest, pHeaderOut, pResponse);

    /* check available room in the buffer and set response flags */
    returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*seekResponse));
    if (0 != returnValue)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Buffer overflow");
        goto Exit;
    }

    /* cast pointers */
    seekRequest = (CMCifsSeekRequest*) pRequest;
    seekResponse = (CMCifsSeekResponse*) *pResponse;

    /* check format */
    if (SMB_SEEK_REQUEST_WORDCOUNT != seekRequest->wordCount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal WordCount or BufferFormat");
        returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }

    /* find file */
    pFile = csGetFileByFid((CSFid)cmLtoh16(cmGetSUint16(seekRequest->fid)), (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid)), (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        goto Exit;
    }

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid = cmLtoh16(cmGetSUint16(pHeaderOut->tid));
    pUser = csGetUserByUid(cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    eventInfo.rid = csGetUserRid(pUser);
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    /* position the file */
    mode = (NQ_UINT16)cmLtoh16(cmGetSUint16(seekRequest->mode));
    offset = (NQ_UINT32)cmLtoh32(cmGetSUint32(seekRequest->offset));

#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    eventInfo.offsetLow = offset;
    eventInfo.offsetHigh = 0;
    eventInfo.before = TRUE;
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_SEEK,
        pUser ? pUser->name : noName,
        pUser ? pUser->ip : cmSelfipGetZeroIp(),
        0,
        (const NQ_BYTE*)&eventInfo
    );
    eventInfo.before = FALSE;
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    switch (mode)
    {
        case SMB_SEEK_START:
            offset = sySeekFileStart(pFile->file, offset, 0);
            break;
        case SMB_SEEK_CURRENT:
            offset = sySeekFileCurrent(pFile->file, (NQ_INT32)offset, 0);
            break;
        case SMB_SEEK_END:
            offset = sySeekFileEnd(pFile->file, (NQ_INT32)offset, 0);
            break;
        default:
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal mode");
            returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
            goto Exit;
    }

    if ((NQ_UINT32)NQ_FAIL == offset)
    {
        returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
        udEventLog(
            UD_LOG_MODULE_CS,
            UD_LOG_CLASS_FILE,
            UD_LOG_FILE_SEEK,
            pUser ? pUser->name : noName,
            pUser ? pUser->ip : cmSelfipGetZeroIp(),
            returnValue,
            (const NQ_BYTE*)&eventInfo
        );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */
        LOGERR(CM_TRC_LEVEL_ERROR, "LSEEK failed");
        goto Exit;
    }
#ifdef UD_NQ_INCLUDEEXTENDEDEVENTLOG
    udEventLog(
        UD_LOG_MODULE_CS,
        UD_LOG_CLASS_FILE,
        UD_LOG_FILE_SEEK,
        pUser ? pUser->name : noName,
        pUser ? pUser->ip : cmSelfipGetZeroIp(),
        0,
        (const NQ_BYTE*)&eventInfo
    );
#endif /* UD_NQ_INCLUDEEXTENDEDEVENTLOG */

    /* compose the response */
    seekResponse->wordCount = SMB_SEEK_RESPONSE_WORDCOUNT;
    cmPutSUint32(seekResponse->offset, cmHtol32(offset));
    cmPutSUint16(seekResponse->byteCount, 0);

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

/*====================================================================
 * PURPOSE: Perform LOCKING ANDX command
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the command in the message
 *          IN header of the outgoing message
 *          IN/OUT double pointer to the response
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:   Function parses the command pointed by the second parameter.
 *          It composes a response and places it from the response pointer,
 *          increasing it so that it will point after the response.
 *====================================================================
 */

NQ_UINT32
csComLockingAndX(
    NQ_BYTE* pRequest,
    CMCifsHeader* pHeaderOut,
    NQ_BYTE** pResponse
    )
{
    CMCifsLockingAndXRequest* lockingRequest;   /* casted request */
    CMCifsLockingAndXResponse* lockingResponse; /* casted response */
    NQ_UINT32 returnValue;                      /* error code in NT format or 0 for no error */
    CSFile* pFile;                              /* pointer to the file descriptor */
    NQ_BYTE lockType;                           /* lock type required */
    CMCifsLockingAndXRange* pRange;             /* pointer to a locking range */
    CMCifsLockingAndXLongRange* pLongRange;     /* pointer to a long locking range */
    NQ_UINT numLocks;                           /* number of locks */
    NQ_UINT numUnlocks;                         /* number of unlocks */
#ifdef  UD_NQ_INCLUDEEVENTLOG
    CSUser *            pUser;
    UDFileAccessEvent   eventInfo;
#endif  /*UD_NQ_INCLUDEEVENTLOG*/

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pRequest:%p pHeaderOut:%p pResponse:%p", pRequest, pHeaderOut, pResponse);

    /* cast pointers */
    lockingRequest = (CMCifsLockingAndXRequest*) pRequest;
    lockingResponse = (CMCifsLockingAndXResponse*) *pResponse;

    /* check format */
    if (SMB_LOCKINGANDX_REQUEST_WORDCOUNT != lockingRequest->wordCount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal WordCount");
        returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
        goto Exit;
    }

    /* check available room in the buffer */
    returnValue = csDispatchCheckSpace(pHeaderOut, *pResponse, sizeof(*lockingResponse));
    if (0 != returnValue)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Buffer overflow");
        goto Exit;
    }

    /* check the next AndX command */
    switch (lockingRequest->andXCommand)
    {
        case SMB_COM_READ:
        case SMB_COM_CLOSE:
        case SMB_COM_WRITE:
        case SMB_COM_FLUSH:
        case SMB_COM_READ_ANDX:
        case SMB_COM_WRITE_ANDX:
        case SMB_COM_LOCKING_ANDX:
        case 0xFF:
            break;
        default:
            LOGERR(CM_TRC_LEVEL_ERROR, "Illegal next AndX command");
            returnValue = csErrorReturn(SMB_STATUS_UNSUCCESSFUL, SRV_ERRerror);
            goto Exit;
    }

    /* withdraw parameters */
    lockType = lockingRequest->lockType;
    numUnlocks = (NQ_UINT)cmLtoh16(cmGetSUint16(lockingRequest->numOfUnlocks));
    numLocks = (NQ_UINT)cmLtoh16(cmGetSUint16(lockingRequest->numOfLocks));

    /* find file */
    pFile = csGetFileByFid((CSFid)cmLtoh16(cmGetSUint16(lockingRequest->fid)), (CSTid)cmLtoh16(cmGetSUint16(pHeaderOut->tid)), (CSUid)cmLtoh16(cmGetSUint16(pHeaderOut->uid)));
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unknown FID");
        returnValue = csErrorReturn(SMB_STATUS_INVALID_HANDLE, DOS_ERRbadfid);
        goto Exit;
    }
#ifdef  UD_NQ_INCLUDEEVENTLOG

    eventInfo.before   = TRUE;
    eventInfo.fileName = csGetFileName(pFile->fid);
    eventInfo.tid      = pFile->tid;
    pUser = pFile->user;
    if (pUser == NULL)
    {
         pUser = csGetUserByUid((CSUid)cmGetSUint16(pHeaderOut->uid));
    }
    if (pUser != NULL)
    {
        eventInfo.rid = csGetUserRid(pUser);
        udEventLog(
                UD_LOG_MODULE_CS,
                UD_LOG_CLASS_FILE,
                UD_LOG_FILE_LOCK,
                pUser->name,
                pUser->ip,
                0,
                (const NQ_BYTE*)&eventInfo
                );
    }
    eventInfo.before   = FALSE;
#endif  /*UD_NQ_INCLUDEEVENTLOG*/

    /* complete oplock break operation (send late response) if required */
    if ((0 == numLocks) && (0 == numUnlocks) && (TRUE == pFile->oplockGranted) && (TRUE == pFile->isBreakingOpLock))
    {
        csBreakComplete(pFile, NULL, 0);
        pFile->oplockGranted = FALSE;
        pFile->isBreakingOpLock = FALSE;
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Oplock break completed");
        returnValue = SMB_STATUS_NORESPONSE;
        goto Exit;
    }

    /* locking file */
    {
        NQ_UINT i;                                 /* just a counter */
        NQ_STATUS status = NQ_SUCCESS;             /* system operation status */
        
        if (lockType & SMB_LOCKINGANDX_LARGEFILES)
        {
            pLongRange = (CMCifsLockingAndXLongRange*)(lockingRequest + 1);
            for (i = numUnlocks; i > 0; i--, pLongRange++)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                eventInfo.offsetHigh = cmGetSUint32(pLongRange->highOffset);
                eventInfo.offsetLow  = cmGetSUint32(pLongRange->lowOffset);
#endif /* UD_NQ_INCLUDEEVENTLOG*/
                status = syUnlockFile(
                                        pFile->file,
                                        cmLtoh32(pLongRange->highOffset),
                                        cmLtoh32(pLongRange->lowOffset),
                                        cmLtoh32(pLongRange->highLength),
                                        cmLtoh32(pLongRange->lowLength),
                                        cmLtoh16(cmGetSUint32(lockingRequest->timeout))
                                        );
                if (NQ_FAIL == status)
                {
                    returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                    if (pUser != NULL)
                    {
                        udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_UNLOCK,
                                pUser->name,
                                pUser->ip,
                                returnValue,
                                (const NQ_BYTE*)&eventInfo
                                );
                    }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
                    LOGERR(CM_TRC_LEVEL_ERROR, "Unlock failed");
                    goto Exit;
                }
#ifdef UD_NQ_INCLUDEEVENTLOG
                if (pUser != NULL)
                {
                    udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_UNLOCK,
                                pUser->name,
                                pUser->ip,
                                0,
                                (const NQ_BYTE*)&eventInfo
                            );
                }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
            }
            for (i = numLocks; i > 0; i--, pLongRange++)
            {
#ifdef UD_NQ_INCLUDEEVENTLOG
                eventInfo.offsetHigh = cmGetSUint32(pLongRange->highOffset);
                eventInfo.offsetLow  = cmGetSUint32(pLongRange->lowOffset);
#endif /* UD_NQ_INCLUDEEVENTLOG*/
                status = syLockFile(
                                        pFile->file,
                                        cmGetSUint32(cmLtoh32(pLongRange->highOffset)),
                                        cmGetSUint32(cmLtoh32(pLongRange->lowOffset)),
                                        cmGetSUint32(cmLtoh32(pLongRange->highLength)),
                                        cmGetSUint32(cmLtoh32(pLongRange->lowLength)),
                                        lockType,
                                        lockingRequest->oplockLevel
                                        );
                if (NQ_FAIL == status)
                {
                    returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                    if (pUser != NULL)
                    {
                        udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_LOCK,
                                pUser->name,
                                pUser->ip,
                                returnValue,
                                (const NQ_BYTE*)&eventInfo
                                );
                    }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
                    LOGERR(CM_TRC_LEVEL_ERROR, "Lock failed");
#ifdef UD_NQ_INCLUDEEVENTLOG
                    if (pUser != NULL)
                    {
                        udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_LOCK,
                                pUser->name,
                                pUser->ip,
                                returnValue,
                                (const NQ_BYTE*)&eventInfo
                                );
                    }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
                    goto Exit;
                }
#ifdef UD_NQ_INCLUDEEVENTLOG
                if (pUser != NULL)
                {
                    udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_LOCK,
                                pUser->name,
                                pUser->ip,
                                0,
                                (const NQ_BYTE*)&eventInfo
                            );
                }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
            }
        }
        else
        {
            pRange = (CMCifsLockingAndXRange*)(lockingRequest + 1);
            for (i = numUnlocks; i > 0; i--, pRange++)
            {
                status = syUnlockFile(
                                        pFile->file,
                                        0L,
                                        cmLtoh32(pRange->offset),
                                        0L,
                                        cmLtoh32(pRange->length),
                                        cmLtoh16(cmGetSUint32(lockingRequest->timeout))
                                        );
                if (NQ_FAIL == status)
                {
                    returnValue = csErrorGetLast();
#ifdef UD_NQ_INCLUDEEVENTLOG
                    if (pUser != NULL)
                    {
                        udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_LOCK,
                                pUser->name,
                                pUser->ip,
                                returnValue,
                                (const NQ_BYTE*)&eventInfo
                                );
                    }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
                    LOGERR(CM_TRC_LEVEL_ERROR, "Unlock failed");
                    goto Exit;
                }
#ifdef UD_NQ_INCLUDEEVENTLOG
                if (pUser != NULL)
                {
                    udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_LOCK,
                                pUser->name,
                                pUser->ip,
                                0,
                                (const NQ_BYTE*)&eventInfo
                            );
                }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
            }
            for (i = numLocks; i > 0; i--, pRange++)
            {
                status = syLockFile(
                                        pFile->file,
                                        0L,
                                        cmLtoh32(pRange->offset),
                                        0L,
                                        cmLtoh32(pRange->length),
                                        lockType,
                                        lockingRequest->oplockLevel
                                        );
                if (NQ_FAIL == status)
                {
                    returnValue = csErrorGetLast();
                    LOGERR(CM_TRC_LEVEL_ERROR, "Lock failed");
                    goto Exit;
                }
#ifdef UD_NQ_INCLUDEEVENTLOG
                if (pUser != NULL)
                {
                    udEventLog(
                                UD_LOG_MODULE_CS,
                                UD_LOG_CLASS_FILE,
                                UD_LOG_FILE_LOCK,
                                pUser->name,
                                pUser->ip,
                                0,
                                (const NQ_BYTE*)&eventInfo
                            );
                }
#endif /* UD_NQ_INCLUDEEVENTLOG*/
            }
        }
    }

    /* advance the outgoing response pointer */
    *pResponse += sizeof(*lockingResponse);

    /* compose the response */
    lockingResponse->wordCount = SMB_LOCKINGANDX_RESPONSE_WORDCOUNT;
    lockingResponse->andXCommand = lockingRequest->andXCommand;
    lockingResponse->andXReserved = 0;
    if (lockingResponse->andXCommand == 0xFF)
    {
        cmPutSUint16(lockingResponse->andXOffset, 0);
    }
    else
    {
        cmPutSUint16(lockingResponse->andXOffset, cmHtol16((NQ_UINT16)(*pResponse - (NQ_BYTE*)pHeaderOut)));
    }
    cmPutSUint16(lockingResponse->byteCount, 0);

    returnValue = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result: 0x%x", returnValue);
    return returnValue;
}

#ifdef UD_CS_INCLUDERPC

/*====================================================================
 * PURPOSE: save IOCTL parameters in late response context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NONE
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static void
writeLateResponseSave(
    CSLateResponseContext* context
    )
{
    context->prot.smb1.commandData.write.dataCount = dataCount;
    context->file = pFile;

    /* write request information into the file descriptor */
    csDispatchSaveResponseContext(context);

    return;
}

/*====================================================================
 * PURPOSE: calculate command data pointer and size
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NQ_SUCCESS or error code
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static NQ_STATUS
writeLateResponsePrepare(
    CSLateResponseContext* context
    )
{
    csDispatchPrepareLateResponse(context);
    context->commandData += sizeof(CMCifsWriteBytesResponse);
    context->commandDataSize -= (NQ_COUNT)sizeof(CMCifsWriteBytesResponse);

    return NQ_SUCCESS;
}

/*====================================================================
 * PURPOSE: send a response using saved context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *          IN status to return
 *          IN number of bytes to return in the data section
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   composes header and delegates send. Restrictions:
 *          - data section is provided and already placed into the buffer
 *          - parameter section is empty
 *          - setup is empty
 *====================================================================
 */

static NQ_BOOL
writeLateResponseSend(
    CSLateResponseContext* context,
    NQ_UINT32 status,
    NQ_COUNT dataLength
    )
{
    CMCifsWriteBytesResponse* writeResponse;            /* casted response */

    if (context->isRpc)
    {
        /* save response for subsequent READ */
        csDcerpcSaveCompleteResponse((CSFile*)context->file, context->commandData, dataLength);
    }

    /* compose Write response */    
    context->commandData -= sizeof(CMCifsWriteBytesResponse);
    writeResponse = (CMCifsWriteBytesResponse*)context->commandData;
    writeResponse->wordCount = SMB_WRITEBYTES_RESPONSE_WORDCOUNT;
    cmPutSUint16(writeResponse->count, (NQ_UINT16)context->prot.smb1.commandData.write.dataCount);
    cmPutSUint16(writeResponse->byteCount, 0);

    return csDispatchSendLateResponse(context, status, sizeof(*writeResponse));
}

/*====================================================================
 * PURPOSE: save IOCTL parameters in late response context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NONE
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static void
writeAndXLateResponseSave(
    CSLateResponseContext* context
    )
{
    context->prot.smb1.commandData.write.dataCount = dataCount;
    context->file = pFile;

    /* write request information into the file descriptor */
    csDispatchSaveResponseContext(context);

    return;
}

/*====================================================================
 * PURPOSE: calculate command data pointer and size
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *
 * RETURNS: NQ_SUCCESS or error code
 *
 * NOTES:   skips Transact header
 *====================================================================
 */

static NQ_STATUS
writeAndXLateResponsePrepare(
    CSLateResponseContext* context
    )
{
    csDispatchPrepareLateResponse(context);
    context->commandData += sizeof(CMCifsWriteAndXResponse);
    context->commandDataSize -= (NQ_COUNT)sizeof(CMCifsWriteAndXResponse);

    return NQ_SUCCESS;
}

/*====================================================================
 * PURPOSE: send a response using saved context
 *--------------------------------------------------------------------
 * PARAMS:  IN pointer to the saved context
 *          IN status to return
 *          IN number of bytes to return in the data section
 *
 * RETURNS: TRUE on success
 *
 * NOTES:   composes header and delegates send. Restrictions:
 *          - data section is provided and already placed into the buffer
 *          - parameter section is empty
 *          - setup is empty
 *====================================================================
 */

static NQ_BOOL
writeAndXLateResponseSend(
    CSLateResponseContext* context,
    NQ_UINT32 status,
    NQ_COUNT dataLength
    )
{
    CMCifsWriteAndXResponse* writeResponse;            /* casted response */
    NQ_UINT32 count = context->prot.smb1.commandData.write.dataCount;

    if (context->isRpc)
    {
        /* save response for subsequent READ */
        csDcerpcSaveCompleteResponse((CSFile*)context->file, context->commandData, dataLength);
    }

    /* compose Write response */    
    context->commandData -= sizeof(CMCifsWriteAndXResponse);
    writeResponse = (CMCifsWriteAndXResponse*)context->commandData;
    writeResponse->wordCount = SMB_WRITEANDX_RESPONSE_WORDCOUNT;
    writeResponse->andXCommand = 0xFF;
    writeResponse->andXReserved = 0;
    cmPutSUint16(writeResponse->andXOffset, 0);
    cmPutSUint16(writeResponse->count, cmHtol16(count & 0xFFFF));
    cmPutSUint16(writeResponse->remaining, cmHtol16(0xFFFF)); /* should be -1 */
    cmPutSUint16(writeResponse->countHigh, (NQ_UINT16)cmHtol16((count & 0xFFFF0000) >> 16));
    cmPutSUint16(writeResponse->reserved, 0);
    cmPutSUint16(writeResponse->byteCount, 0);

    return csDispatchSendLateResponse(context, status, sizeof(*writeResponse));
}

#endif /* UD_CS_INCLUDERPC */

#endif /* UD_NQ_INCLUDECIFSSERVER */

