
/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Server Announcement processing
 *--------------------------------------------------------------------
 * MODULE        : Server
 * DEPENDENCIES  :
 ********************************************************************/

#include "nsapi.h"
#include "nscommon.h"

#include "csbrowse.h"
#include "cstransa.h"
#include "csutils.h"

#if defined(UD_NQ_INCLUDECIFSSERVER) && defined(UD_NQ_USETRANSPORTNETBIOS)

/* This code implements HOST ANNOUNCEMENT for two cases:
    - as a response to a client's request sent as a BROWSE MAILSLOT command over the
      TRANSACTION sub protocols
    - as a periodic initiative message over the same protocol
    Message data is reused for the both cases
*/

/*
    Static data & functions
    -----------------------
 */

typedef struct
{
    NQ_UINT32 nextAnnouncementInterval;                 /* next interval between announcements, this value
                                                           will raise up to CM_FS_MINHOSTANNOUNCEMENTINTERVAL */
    NSSocketHandle ddSocket;                            /* socket for connecting to DD */
    CMNetBiosNameInfo dcName;                           /* NetBIOS name (domain controller) to send announcement to */
    CMCifsServerAnnouncementRequest frame;              /* frame to send */
    NQ_UINT16 dataSize;                                 /* frame data portion size to send */
    NQ_BOOL frameReady;                                 /* whether the frame is already filled with data */
    NQ_WCHAR serverComment[SMB_SERVER_COMMENTLEN + 1];  /* buffer for server comment */
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

/* This frame is used for preparing HOST ANNOUNCEMENT. All but the timeout is filled only once
   When this frame is used for self-announcement - it is send as is
   When it is used as response to HOST ANNOUNCEMENT REQUEST, its data is copied into the
   response frame */

static const NQ_BYTE sCifsServerAnnouncementRequest[] =
{
    /* CIFS header */
    0xFF, 'S', 'M', 'B',
    SMB_COM_TRANSACTION,
    0x00, 0x00, 0x00, 0x00, /* Error */
    0x00,                   /* Flags */
    0x00, 0x00,             /* Flags 2 */
    0x00, 0x00,             /* Process ID high */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* Signature */
    0x00, 0x00,             /* reserved (2) */
    0x00, 0x00,             /* Tree ID */
    0x00, 0x00,             /* Process ID */
    0x00, 0x00,             /* User ID */
    0x00, 0x00,             /* Multiplex ID */

    /* TRANSACTION header */
    0x11,                   /* Word count */
    0x00, 0x00,             /* Total parameter count */
    0x00, 0x00,             /* Total data count */
    0x00, 0x00,             /* Max parameter count */
    0x00, 0x00,             /* Max data count */
    0x00,                   /* Max setup count */
    0x00,                   /* Reserved */
    0x00, 0x00,             /* Flags */
    0x00, 0x00, 0x00, 0x00, /* Timeout */
    0x00, 0x00,             /* Reserved */
    0x00, 0x00,             /* Parameter count */
    0x00, 0x00,             /* Parameter offset */
    0x00, 0x00,             /* Data count */
    0x00, 0x00,             /* Data offset */
    0x03,                   /* Setup count */
    0x00,                   /* Reserved */

    /* setups */
    0x01, 0x00,     /* Write mail slot opcode */
    0x00, 0x00,     /* Priority */
    0x02, 0x00,     /* Class */

    0x00, 0x00,     /* Byte Count */

    /* browser name */
    '\\','M','A','I','L','S','L','O','T','\\','B','R','O','W','S','E', 0x00, /* SMB_SERVERANNOUNCEMENT_BROWSER, */

    /* announcement data (including server name placeholder) */
    0x01,                       /* Host Announcement command */
    0x00,                       /* Update count */
    0x00, 0x00, 0x00, 0x00,     /* Update periodically */
    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, /* Host name */
    CM_SOFTWAREVERSIONMAJOR,    /* OS major version */
    (CM_SOFTWAREVERSIONMINOR * CM_SOFTWAREVERSIONDECSEPARATOR) + CM_SOFTWAREVERSIONSUPPORT,    /* OS minor version */
    0x00, 0x00, 0x00, 0x00,     /* Server type */
    0x00, 0x04,                 /* Browser protocol major and minor version */
    0x55, 0xAA,                 /* Server announcement signature */
    0x00
};

static void initializeFrame(void);    /* one-time initialization of the response frame */

/*
 *====================================================================
 * PURPOSE: initialize server announcement interval and the
 *          communication socket
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: NQ_SUCCESS or NQ_FAIL
 *
 * NOTES:
 *====================================================================
 */

NQ_STATUS
csInitBrowse(
    void
    )
{
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocate(sizeof(*staticData));
    if (NULL == staticData)
    {
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */

    staticData->ddSocket = NULL;
    staticData->frameReady = FALSE;
    staticData->nextAnnouncementInterval = SMB_MIN_SERVER_ANNOUNCEMENT_INTERVAL;

    /* prepare communication socket, when UD_ND_INCLUDENBDAEMON is defined, use local ip, when not use any ip */
#ifdef UD_ND_INCLUDENBDAEMON
    if (TRUE == nsIsDaemonInUse())
    {
        staticData->ddSocket = csPrepareSocketWithBindType(NS_SOCKET_DATAGRAM, NS_TRANSPORT_NETBIOS, NS_BIND_DEAMON);
    }
    else
#endif /* UD_ND_INCLUDENBDAEMON */
    {
        staticData->ddSocket = csPrepareSocketWithBindType(NS_SOCKET_DATAGRAM, NS_TRANSPORT_NETBIOS, NS_BIND_SERVER);
    }

    if (NULL == staticData->ddSocket)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to create socket for communication with DD");
        goto Exit;
    }

    syMemcpy(&staticData->dcName, cmNetBiosGetDomain(), sizeof(CMNetBiosNameInfo));
    cmNetBiosNameFormat(staticData->dcName.name, CM_NB_POSTFIX_MASTERBROWSER);
    /* this is definitely NetBios group name! */
    staticData->dcName.isGroup = TRUE;
    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

/*
 *====================================================================
 * PURPOSE: release resources
 *--------------------------------------------------------------------
 * PARAMS:  None
 *
 * RETURNS: None
 *
 * NOTES:
 *====================================================================
 */

void
csStopBrowse(
    void
    )
{
    if (staticData != NULL)
    {
        if (staticData->ddSocket != NULL)
        {
            nsClose(staticData->ddSocket);
            staticData->ddSocket = NULL;
        }

        staticData->frameReady = FALSE;
    }

    /* release memory */
#ifdef SY_FORCEALLOCATION
    if (NULL != staticData)
    {
        cmMemoryFree(staticData);
    }
    staticData = NULL;
#endif /* SY_FORCEALLOCATION */
}

/*
 *====================================================================
 * PURPOSE: perform Server Announcement for our server
 *--------------------------------------------------------------------
 * PARAMS:  Next announcement interval in seconds
 *
 * RETURNS: next announcement interval in seconds
 *
 * NOTES:   Broadcasts a Server Announcement message to the
 *          domain members. This routine uses a predefined frame filling
 *          only the next announcement interval value
 *====================================================================
 */

NQ_UINT32
csAnnounceServer(
    void
    )
{
    NQ_UINT32 announcementInterval;     /* this announcement interval */

    TRCB();

    TRC1P("next host announcement in: %lu sec", (NQ_ULONG)(staticData->nextAnnouncementInterval));

    initializeFrame();

    /* next announcement interval */

    cmPutSUint32(staticData->frame.data.periodicity, cmHtol32(staticData->nextAnnouncementInterval * 1000));  /* in millisecs */

    /* ask DD to broadcast our announcement to the domain */

#ifdef UD_CS_INCLUDEHOSTANNOUNCEMENT
    if (*cmNetBiosGetHostNameZeroed() != '\0')
    {
        NQ_UINT dataLen = (NQ_UINT)(sizeof(staticData->frame) + staticData->dataSize - sizeof(staticData->frame.data));

        if (nsSendToName(
            staticData->ddSocket,
            (const NQ_BYTE*)&staticData->frame,
            dataLen,
            &staticData->dcName
            ) != dataLen)
        {
            TRCERR("Failed to send broadcast to DD");
            TRCE();
            return (NQ_UINT32)NQ_FAIL;
        }
    }
#endif

    /* calculate announcement interval */

    announcementInterval = staticData->nextAnnouncementInterval;
    staticData->nextAnnouncementInterval *= 2;
    if (staticData->nextAnnouncementInterval > SMB_MAX_SERVER_ANNOUNCEMENT_INTERVAL)
    {
        staticData->nextAnnouncementInterval = SMB_MAX_SERVER_ANNOUNCEMENT_INTERVAL;
    }

    TRCE();
    return announcementInterval;
}

/*====================================================================
 * PURPOSE: Continue processing TRANSACTION command for a MAILSLOT request
 *--------------------------------------------------------------------
 * PARAMS:  IN/OUT pointer to the TRANSACTION descriptor
 *
 * RETURNS: 0 on success or error code in NT format
 *
 * NOTES:
 *====================================================================
 */

NQ_UINT32
csMailslotBrowse(
    CSTransactionDescriptor* descriptor
    )
{
    NQ_UINT16 opcode;                               /* browse opcode */
    NQ_UINT16* pSetup;                              /* casted pointer to setup words + byteCount */
    CMCifsServerAnnouncementData* pAnnouncement;    /* pointer to announcement data */

    TRCB();

    /* find opcode */

    opcode = *(NQ_UINT16*)descriptor->dataIn;
    switch (opcode)
    {
    case 0x02:
        cmPutSUint16(descriptor->hdrOut->flags2, (NQ_UINT16)(cmGetSUint16(descriptor->hdrOut->flags2) & ~SMB_FLAGS2_UNICODE));   /* remove the (possible) UNICODE flag */
        initializeFrame();
        descriptor->paramCount = 0;
        descriptor->dataCount = staticData->dataSize;
        descriptor->setupCount = SMB_SERVERANNOUNCEMENT_SETUPCOUNT;

        /* setups */

        pSetup = (NQ_UINT16*)descriptor->pBuf;
        *pSetup++ = cmHtol16(1);
        *pSetup++ = 0;
        *pSetup++ = 0;

        /* byte count */

        *pSetup++ = (NQ_UINT16)cmHtol16(descriptor->dataCount);

        /* fill data - copy the appropriate portion of the predefined frame */

        pAnnouncement = (CMCifsServerAnnouncementData*)pSetup;
        descriptor->dataOut = (NQ_BYTE*)pSetup;
        syMemcpy(descriptor->dataOut, &staticData->frame.data, staticData->dataSize);
        cmPutSUint32(pAnnouncement->periodicity, cmHtol32(staticData->nextAnnouncementInterval * 1000));  /* in millisecs */

        break;
    default:
        TRCE();
        return csErrorReturn(SMB_STATUS_NOT_SUPPORTED, (NQ_UINT32)SRV_ERRnosupport);
    }

    TRCE();
    return 0;
}

/*
 *====================================================================
 * PURPOSE: initialize Server Announcement frame
 *--------------------------------------------------------------------
 * PARAMS:  NONE
 *
 * RETURNS: NONE
 *
 * NOTES:
 *====================================================================
 */

static void initializeFrame()
{
    NQ_UINT16 count, serverCommentLength;

    if (staticData->frameReady)
        return;         /* already done */

    staticData->frameReady = TRUE;

    syMemcpy(&staticData->frame, sCifsServerAnnouncementRequest, sizeof(sCifsServerAnnouncementRequest));

    /* write in constant values for the first time */
    syMemcpy(staticData->frame.data.serverName, cmNetBiosGetHostNameZeroed(), SMB_SERVER_NAMELEN);
    syMemset(staticData->frame.data.comment, 0, sizeof(staticData->frame.data.comment));
    udGetServerComment(staticData->serverComment);
    cmUnicodeToAnsiN((NQ_CHAR *)staticData->frame.data.comment, sizeof(staticData->frame.data.comment), staticData->serverComment, sizeof(staticData->serverComment));
    serverCommentLength = (NQ_UINT16)(syStrlen((NQ_CHAR *)staticData->frame.data.comment) + 1);
    count = (NQ_UINT16)((NQ_UINT16)sizeof(staticData->frame.data) + serverCommentLength - (NQ_UINT16)sizeof(staticData->frame.data.comment));
    staticData->dataSize = count;
    cmPutSUint16(staticData->frame.transHeader.totalDataCount, cmHtol16(count));
    cmPutSUint16(staticData->frame.transHeader.dataCount, cmHtol16(count));
    count = (NQ_UINT16)(count + sizeof(staticData->frame.name));
    cmPutSUint16(staticData->frame.byteCount, cmHtol16(count));
    count = (NQ_BYTE*)&staticData->frame.data - (NQ_BYTE*)&staticData->frame;
    cmPutSUint16(staticData->frame.transHeader.dataOffset, cmHtol16(count));

    /* swap UINT16 and UINT32 fields to little endian */
    cmPutSUint32(staticData->frame.data.installedServices, cmHtol32(csGetHostType()));
}

#endif /* UD_NQ_INCLUDECIFSSERVER */

