/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : NQ Server Control tool
 *--------------------------------------------------------------------
 * MODULE        : CS CTRL
 * DEPENDENCIES  :
 ********************************************************************/

#include "cmapi.h"
#include "cscontrl.h"
#include "csdataba.h"

#ifdef UD_CS_MESSAGESIGNINGPOLICY
#define  MAX_LVL_MSP_ADAPTER     2
#endif /* UD_CS_MESSAGESIGNINGPOLICY */

static int doTerminate(int argc, char *argv[]);
static int doRestart(int argc, char *argv[]);
#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
static int doAddShare(int argc, char *argv[]);
static int doRemoveShare(int argc, char *argv[]);
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */
static int doEnumShares(int argc, char *argv[]);
#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
static int doAddUser(int argc, char *argv[]);
static int doRemoveUser(int argc, char *argv[]);
static int doCleanUserCons(int argc, char *argv[]);
static int doEnumUsers(int argc, char *argv[]);
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */
static int doEnumClients(int argc , char *argv[]);
static int doChangeAuthenticationEncryptLevel(int argc , char *argv[]);
#ifdef UD_CS_MESSAGESIGNINGPOLICY
static int doChangeMesgSign(int argc , char *argv[]);
#endif /*UD_CS_MESSAGESIGNINGPOLICY*/
static int doEnumFiles(int argc, char *argv[]);
#ifdef UD_NQ_INCLUDESMB1
static int doSetSMB1Support(int argc, char *argv[]);
#endif /* UD_NQ_INCLUDESMB1 */
#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
static NQ_BOOL isValidShareName(NQ_CHAR* shareName);
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */
#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
static NQ_BOOL isValidUserName(NQ_CHAR* shareName);
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

typedef struct 
{
    const char * name;
    const char * code;
    int (*processor)(int argc, char *argv[]);
    int min;
    int max;
} CommandDescriptor;
static CommandDescriptor commands[] =
{
        { "Terminating CIFS Server", "S", doTerminate, 0, 0 },
        { "Restarting CIFS Server", "R", doRestart, 0, 0 },
#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
#ifdef UD_CS_INCLUDERPC_SPOOLSS
        { "Adding share", "+S", doAddShare, 3, 4 },
#else /* UD_CS_INCLUDERPC_SPOOLSS */
        { "Adding share", "+S", doAddShare, 3, 3 },
#endif /* UD_CS_INCLUDERPC_SPOOLSS */
        { "Removing share", "-S", doRemoveShare, 1, 1 },
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */
        { "Enumerating shares", "ES", doEnumShares, 0, 0 },
#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
        { "Adding user", "+U", doAddUser, 4, 5 },
        { "Removing user", "-U", doRemoveUser, 1, 1 },
        { "Closing user connections to CIFS Server", "C", doCleanUserCons, 1, 2 },
        { "Enumerating users", "EU", doEnumUsers, 0, 0 },
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */
        { "Enumerating Clients", "EC" , doEnumClients , 0 , 0},
        { "Changing Maximum Authentication Encryption Level" , "SEL" , doChangeAuthenticationEncryptLevel , 1 , 1},
#ifdef UD_CS_MESSAGESIGNINGPOLICY
        { "Changing Message Signing" , "MSG" , doChangeMesgSign , 1, 1},
#endif /*UD_CS_MESSAGESIGNINGPOLICY*/
        { "Enumerating Files", "EF" , doEnumFiles , 0 , 0},
#ifdef UD_NQ_INCLUDESMB1
        { NULL, "SMB1" , doSetSMB1Support , 1 , 1}
#endif /* UD_NQ_INCLUDESMB1 */
};

/*
 *====================================================================
 * PURPOSE: Main function
 *--------------------------------------------------------------------
 * PARAMS:  as required by 'main'
 * 
 * RETURNS: 0 - success; -1 - fail
 *
 *====================================================================
 */

int main(int argc, char *argv[])
{
    int i;
    int res = -1;

    /* initialize SY module */
    if (FALSE == syInit())
    {
        syPrintf("initialize SY module was failed\n");
        goto Exit1;
    }

    /* initialize UD module */
    if (NQ_FAIL == udInit())
    {
        syPrintf("initialize UD module was failed\n");
        goto Exit2;
    }

    if (argc <= 1)
    {
        syPrintf("-- YNQ Server Control tool --\n");
        syPrintf("Usage: nqcsctrl <command> <parameters>\n");
        syPrintf("\tS \t Stop YNQ Server\n");
        syPrintf("\tR \t Restart YNQ Server\n");
#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
        syPrintf("\t+S \t Add new share\n");
        syPrintf("\t\t");
#ifdef UD_CS_INCLUDERPC_SPOOLSS
        syPrintf(" '+S \"<name>\" \"<local path>\" \"<description>\" [P]' ; 'P' is for printer share\n");
#else /* UD_CS_INCLUDERPC_SPOOLSS */
        syPrintf(" '+S \"<name>\" \"<local path>\" \"<description>\"'\n");
#endif /* UD_CS_INCLUDERPC_SPOOLSS */
        syPrintf("\t-S \t Remove share\n");
        syPrintf("\t\t");
        syPrintf(" '-S \"<name>\"'\n");
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */
        syPrintf("\tES \t Enumerate shares\n");
#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
        syPrintf("\t+U \t Add new user\n");
        syPrintf("\t\t");
        syPrintf(" '+U \"<user name>\" \"<full name>\" \"<description>\" \"<password>\" [A]' ; 'A' grants administrative rights\n");
        syPrintf("\t-U \t Remove user\n");
        syPrintf("\t\t");
        syPrintf(" '-U \"<user name>\"'\n");
        syPrintf("\tC \t Close user connections\n");
        syPrintf("\t\t");
        syPrintf(" 'C  \"<user name>\" [D]' ; 'D' for domain user\n");
        syPrintf("\tEU \t Enumerate users\n");
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */
        syPrintf("\tEC \t Enumerate clients\n");
        syPrintf("\tEF \t Enumerate files\n");
        syPrintf("\tSEL \t Set maximum authentication encryption level\n");
        syPrintf("\t\t");
#ifdef UD_CS_MESSAGESIGNINGPOLICY
        syPrintf(" 'SEL <max level>' ; 1 - NTLM, 2 - NTLMV2\n");
        syPrintf("\tMSG \t Set message signing policy value \n");
        syPrintf("\t\t");
        syPrintf(" 'MSG <value>' ; 1 - enabled, but not required; 2 - required \n");
#else /* UD_CS_MESSAGESIGNINGPOLICY */
        syPrintf(" 'SEL <max level>' ; 1 - LM, 2 - NTLM, 3 - LMV2, 4 - NTLMV2\n");
#endif /* UD_CS_MESSAGESIGNINGPOLICY */
#ifdef UD_NQ_INCLUDESMB1
        syPrintf("\tSMB1 \t Activate or deactivate SMB1 dialect\n");
        syPrintf("\t\t");
        syPrintf(" 'SMB1 <enable/disable>'\n");
#endif /* UD_NQ_INCLUDESMB1 */

        goto Exit3;
    }

    /* switch by command */
    for (i = 0; i < sizeof(commands)/sizeof(commands[0]); i++)
    {
        if (0 == strcmp(commands[i].code, argv[1]))
        {
            if ((argc - 2) < commands[i].min || (argc - 2) > commands[i].max)
            {
                if (commands[i].min == commands[i].max)
                {
                    syPrintf("  Unexpected number of parameters: expected - %d, found - %d\n", commands[i].min, argc -2);
                }
                else
                {
                    syPrintf("  Unexpected number of parameters: expected  - %d to %d, found - %d\n", commands[i].min, commands[i].max, argc -2);
                }

                goto Exit3;
            }
            break;
        }
    }
    
    if (i == sizeof(commands)/sizeof(commands[0]))
    {
        syPrintf("  Command '%s' not found\n", argv[1]);
        goto Exit3;
    }

    if (NULL != commands[i].name)
    {
        syPrintf(" %s ...", commands[i].name);
    }

    if ((*commands[i].processor)(argc - 2, argv + 2) != NQ_SUCCESS)
    {
        syPrintf("\n failed: 0x%X\n", errno);
        goto Exit3;
    }

    res = 0;
    syPrintf("\n succeeded\n");

Exit3:
    udStop();
Exit2:
    syStop();
Exit1:
    return res;
}

/*
 * Command processors
 * 
 */

static int doTerminate(int argc, char *argv[])
{
    return csCtrlStop();
}

static int doRestart(int argc, char *argv[])
{
    return csCtrlRestart();
}

#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION

static int doAddShare(int argc, char *argv[])
{
    NQ_STATUS status = NQ_FAIL;
    NQ_BOOL     isPrinter = FALSE;

#ifdef UD_CS_INCLUDERPC_SPOOLSS
    if (argc == 4)
    {
        if (0 != strcmp(argv[3], "P"))
    {
        syPrintf("  'P' expected, %s found\n", argv[3]);
        goto Exit;
    }
        else
        {
            isPrinter = TRUE;
        }
    }
#endif /* UD_CS_INCLUDERPC_SPOOLSS */

    if (0 == syStrlen(argv[0]))
    {
        syPrintf("\n Share name field is required");
        goto Exit;
    }

    if (UD_FS_MAXSHARELEN <= syStrlen(argv[0]))
    {
        syPrintf("\n Share name exceeded the maximum allowed characters");
        goto Exit;
    }

    if (0 == syStrlen(argv[1]))
    {
        syPrintf("\n Local path field is required");
        goto Exit;
    }

    if (UD_FS_MAXPATHLEN <= syStrlen(argv[1]))
    {
        syPrintf("\n Local path exceeded the maximum allowed characters");
        goto Exit;
    }

    if (0 == syStrlen(argv[2]))
    {
        syPrintf("\n Share description field is required");
        goto Exit;
    }

    if (UD_FS_MAXDESCRIPTIONLEN <= syStrlen(argv[2]))
    {
        syPrintf("\n Share description exceeded the maximum allowed characters");
        goto Exit;
    }

    if (!isValidShareName(argv[0]))
    {
        syPrintf("\n Share name must not contain the following characters: + [ ] \" / \\ : ; | < > , ? * =");
        goto Exit;
    }

    status = csCtrlAddShareA(argv[0], argv[1], isPrinter, argv[2]);

Exit:
    return status;
}

static int doRemoveShare(int argc, char *argv[])
{
    NQ_STATUS status = NQ_FAIL;

    if (0 == syStrlen(argv[0]))
    {
        syPrintf("\n Share name field is required");
        goto Exit;
    }

    if (UD_FS_MAXSHARELEN <= syStrlen(argv[0]))
    {
        syPrintf("\n Share name exceeded the maximum allowed characters");
        goto Exit;
    }

    if (!isValidShareName(argv[0]))
    {
        syPrintf("\n Share name must not contain the following characters: + [ ] \" / \\ : ; | < > , ? * =");
        goto Exit;
    }

    status = csCtrlRemoveShareA(argv[0]);

Exit:
    return status;
}

#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */

static int doEnumShares(int argc, char *argv[])
{
    CsCtrlShare share;
    NQ_INDEX i = 0;
    NQ_CHAR * tempName = NULL;
    NQ_CHAR * tempPath = NULL;
    NQ_CHAR * tempComment = NULL;
    NQ_INT  result = NQ_FAIL;

    while (NQ_SUCCESS == csCtrlEnumShares(&share, i++))
    {
        tempName = cmMemoryCloneWStringAsAscii(share.name);
        tempPath = cmMemoryCloneWStringAsAscii(share.path);
        tempComment = cmMemoryCloneWStringAsAscii(share.comment);

        syPrintf("\n  #%d\n", i);
        syPrintf("        Share name: %s\n", tempName);
        syPrintf("        Share path: %s\n", tempPath);
        syPrintf("        Share is printer: %s\n", share.isPrinter == 1 ? "+" : "-");
        syPrintf("        Share comment: %s", tempComment);
        cmMemoryFree(tempName);
        cmMemoryFree(tempPath);
        cmMemoryFree(tempComment);
    }

    if ((NQ_ERR_ERROR == syGetLastError()) || (NQ_SUCCESS == syGetLastError()))
    {
        result = NQ_SUCCESS;
    }

    return result;
}

#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT

static int doAddUser(int argc, char *argv[])
{
    NQ_STATUS status = NQ_FAIL;

    if (0 == syStrlen(argv[0]))
    {
        syPrintf("\n User name field is required");
        goto Exit;
    }

    if (CM_USERNAMELENGTH <= syStrlen(argv[0]))
    {
        syPrintf("\n User name exceeded the maximum allowed characters");
        goto Exit;
    }

    if (!isValidUserName(argv[0]))
    {
        syPrintf("\n User name must not contain the following characters: + [ ] \" / \\ : ; | < > , ? * = @");
        goto Exit;
    }

    if (CM_USERNAMELENGTH <= syStrlen(argv[1]))
    {
        syPrintf("\n Full name exceeded the maximum allowed characters");
        goto Exit;
    }

    if (UD_FS_MAXDESCRIPTIONLEN <= syStrlen(argv[2]))
    {
        syPrintf("\n The description exceeded the maximum allowed characters");
        goto Exit;
    }

    if (0 == syStrlen(argv[3]))
    {
        syPrintf("\n Password field is required");
        goto Exit;
    }

    if (UD_NQ_MAXPWDLEN <= syStrlen(argv[3]))
    {
        syPrintf("\n The password exceeded the maximum allowed characters");
        goto Exit;
    }

    if (argc == 5 && 0 != strcmp(argv[4], "A"))
    {
        syPrintf("  'A' expected, %s found\n", argv[4]);
        goto Exit;
    }

    status = csCtrlAddUserA(argv[0], argv[1], argv[2], argv[3], argc == 5? TRUE: FALSE);

Exit:
    return status;
}

static int doRemoveUser(int argc, char *argv[])
{
    NQ_STATUS status = NQ_FAIL;

    if (0 == syStrlen(argv[0]))
    {
        syPrintf("\n User name field is required");
        goto Exit;
    }

    if (CM_USERNAMELENGTH <= syStrlen(argv[0]))
    {
        syPrintf("\n User name exceeded the maximum allowed characters");
        goto Exit;
    }

    if (!isValidUserName(argv[0]))
    {
        syPrintf("\n User name must not contain the following characters: + [ ] \" / \\ : ; | < > , ? * = @");
        goto Exit;
    }

    status = csCtrlRemoveUserA(argv[0]);

Exit:
    return status;
}

static int doCleanUserCons(int argc, char *argv[])
{
    NQ_STATUS status = NQ_FAIL;

    if (0 == syStrlen(argv[0]))
    {
        syPrintf("\n User name field is required");
        goto Exit;
    }

    if (CM_USERNAMELENGTH <= syStrlen(argv[0]))
    {
        syPrintf("\n User name exceeded the maximum allowed characters");
        goto Exit;
    }

    if (!isValidUserName(argv[0]))
    {
        syPrintf("\n User name must not contain the following characters: + [ ] \" / \\ : ; | < > , ? * = @");
        goto Exit;
    }

    if (argc == 2 && 0 != strcmp(argv[1], "D"))
    {
        syPrintf("  'D' expected, %s found\n", argv[1]);
        return -1;
    }

    status = csCtrlCleanUserConnectionsA(argv[0], argc == 2 ? TRUE: FALSE);

Exit:
    return status;
}


static int doEnumUsers(int argc, char *argv[])
{
    CsCtrlUser user;
    NQ_INDEX i = 0;
    NQ_CHAR * tempName = NULL;
    NQ_CHAR * tempFullName = NULL;
    NQ_CHAR * tempDescription = NULL;
    NQ_INT    result = NQ_FAIL;

    while (NQ_SUCCESS == csCtrlEnumUsers(&user, i++))
    {
        tempName = cmMemoryCloneWStringAsAscii(user.name);
        tempFullName = cmMemoryCloneWStringAsAscii(user.fullName);
        tempDescription = cmMemoryCloneWStringAsAscii(user.description);

        syPrintf("\n  #%d\n", i);
        syPrintf("        User name: %s\n", tempName);
        syPrintf("        User full name: %s\n", tempFullName);
        syPrintf("        User description: %s\n", tempDescription);
        syPrintf("        User is Admin: %s", user.isAdmin == 1 ? "Yes" : "No");

        cmMemoryFree(tempName);
        cmMemoryFree(tempFullName);
        cmMemoryFree(tempDescription);
    }

    if ((NQ_ERR_ERROR == syGetLastError()) || (NQ_SUCCESS == syGetLastError()))
    {
        result = NQ_SUCCESS;
    }

    return result;
}

#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */

static int doEnumClients(int argc , char *argv[])
{
    CsCtrlClient    client;
    NQ_INDEX        i = 0;
    NQ_CHAR         *ip;
    NQ_INT          result = NQ_FAIL;

    ip = (NQ_CHAR *)cmMemoryAllocate(CM_IPADDR_MAXLEN* sizeof(NQ_CHAR));
    if (NULL == ip)
    {
        goto Exit;
    }

    while (NQ_SUCCESS == csCtrlEnumClients(&client , i++))
    {
        cmIpToAscii(ip , &client.ip);
        syPrintf("\n Entry #%d \n" , i);
        syPrintf(" ip = %s\n" , ip);
        syPrintf(" SMB Version:" );
        switch (client.dialect)
        {
            case (CS_DIALECT_SMB1):
            {
                syPrintf(" SMB 1\n");
                break;
            }
#ifdef UD_NQ_INCLUDESMB2
            case (CS_DIALECT_SMB2):
            {
                syPrintf(" SMB 2.0\n");
                break;
            }
            case (CS_DIALECT_SMB210):
            {
                syPrintf(" SMB 2.1\n");
                break;
            }
#ifdef UD_NQ_INCLUDESMB3
            case (CS_DIALECT_SMB30):
            {
                syPrintf(" SMB 3.0\n");
                break;
            }
            case (CS_DIALECT_SMB311):
            {
                syPrintf(" SMB 3.1.1\n");
                break;
            }

#endif /* UD_NQ_INCLUDESMB3 */
#endif /* UD_NQ_INCLUDESMB2 */
        }
    }

    cmMemoryFree(ip);
    if ((NQ_ERR_ERROR == syGetLastError()) || (NQ_SUCCESS == syGetLastError()))
    {
        result = NQ_SUCCESS;
    }

Exit:
    return result;
}

static int doEnumFiles(int argc , char *argv[])
{
    CsCtrlFile      file;
    NQ_INDEX        i = 0;
    NQ_CHAR         *ip;
    NQ_INT          result = NQ_FAIL;

    ip = (NQ_CHAR *)cmMemoryAllocate(CM_IPADDR_MAXLEN* sizeof(NQ_CHAR));
    if (NULL == ip)
    {
        goto Exit;
    }

    while (NQ_SUCCESS == csCtrlEnumFiles(&file , i++))
    {
        cmIpToAscii(ip , &file.ip);
        syPrintf("\n Entry #%d \n" , i);
        syPrintf(" name = %s \n" , cmWDump(file.name));
        syPrintf(" user name = %s \n" , cmWDump(file.userName));
        syPrintf(" ip = %s\n" , ip);
        syPrintf(" File is Directory: %s\n", file.isDirectory ? "TRUE" : "FALSE");
    }

    cmMemoryFree(ip);

    if ((NQ_ERR_ERROR == syGetLastError()) || (NQ_SUCCESS == syGetLastError()))
    {
        result = NQ_SUCCESS;
    }

Exit:
    return result;
}

#ifdef UD_NQ_INCLUDESMB1
static int doSetSMB1Support(int argc , char *argv[])
{
    NQ_INT isSupport;
    NQ_STATUS result;

    if (cmAStricmp ( "disable", argv[0]) == 0)
    {
        syPrintf(" Disabling SMB 1 dialect ...");
        isSupport = 0;
    }
    else if (cmAStricmp("enable", argv[0]) == 0)
    {
        syPrintf(" Enabling SMB 1 dialect ...");
        isSupport = 1;
    }
    else
    {
        syPrintf(" Invalid input, type 'enable' to enable SMB1 dialect or 'disable' to disable it");
        result = NQ_FAIL;
        goto Exit;
    }

    result = csCtrlSetSMB1support(isSupport);

Exit:
    return result;
}
#endif /* UD_NQ_INCLUDESMB1 */

static int doChangeAuthenticationEncryptLevel(int argc , char *argv[])
{
    NQ_UINT     maxLevel;
    NQ_CHAR * end;
    NQ_STATUS   result = NQ_FAIL;

    maxLevel = (NQ_UINT)syStrtol(argv[0] , &end , 10);
#ifdef UD_CS_MESSAGESIGNINGPOLICY
    maxLevel = maxLevel * MAX_LVL_MSP_ADAPTER;
#endif
    if (maxLevel < CS_CONTROL_MAX_LEVEL_AUTH_ENCRYPTION_LM || maxLevel > CS_CONTROL_MAX_LEVEL_AUTH_ENCRYPTION_NTLMV2 || *end)
    {
        syPrintf(" \nInvalid Maximum Authentication Encryption Level\n");
        goto Exit;
    }

    result = csCtrlSetEncryptionMethods(maxLevel);

Exit:
    return result;
}

#ifdef UD_CS_MESSAGESIGNINGPOLICY
static int doChangeMesgSign(int argc , char *argv[])
{
    NQ_INT newPolicy;
    NQ_CHAR * end;
    NQ_STATUS result = NQ_FAIL;

    newPolicy = (NQ_INT)syStrtol(argv[0] , &end , 10);
    if (((newPolicy < 1) || (newPolicy > 2)) || *end)
    {
        syPrintf(" \nInvalid Message Signing Policy\n");
        goto Exit;
    }
    result = csCtrlSetMessageSigningPolicy(newPolicy);

Exit:
    return result;
}
#endif /*UD_CS_MESSAGESIGNINGPOLICY*/

#ifdef UD_CS_INCLUDERPC_SRVSVC_EXTENSION
static NQ_BOOL isValidShareName(NQ_CHAR* shareName)
{
    NQ_BOOL isValidName = FALSE;

    if (shareName == NULL)
    {
        goto Exit;
    }

    isValidName =
           ((syStrchr(shareName, '+') == NULL)
        && (syStrchr(shareName, '[') == NULL)
        && (syStrchr(shareName, ']') == NULL)
        && (syStrchr(shareName, '"') == NULL)
        && (syStrchr(shareName, '/') == NULL)
        && (syStrchr(shareName, '\\') == NULL)
        && (syStrchr(shareName, ':') == NULL)
        && (syStrchr(shareName, ';') == NULL)
        && (syStrchr(shareName, '|') == NULL)
        && (syStrchr(shareName, '<') == NULL)
        && (syStrchr(shareName, '>') == NULL)
        && (syStrchr(shareName, ',') == NULL)
        && (syStrchr(shareName, '?') == NULL)
        && (syStrchr(shareName, '*') == NULL)
        && (syStrchr(shareName, '=') == NULL));

Exit:
    return isValidName;
}
#endif /* UD_CS_INCLUDERPC_SRVSVC_EXTENSION */
#ifdef UD_CS_INCLUDELOCALUSERMANAGEMENT
static NQ_BOOL isValidUserName(NQ_CHAR* userName)
{
    NQ_BOOL isValidName = FALSE;

    if (userName == NULL)
    {
        goto Exit;
    }

    isValidName =
           ((syStrchr(userName, '+') == NULL)
        && (syStrchr(userName, '[') == NULL)
        && (syStrchr(userName, '@') == NULL)
        && (syStrchr(userName, ']') == NULL)
        && (syStrchr(userName, '"') == NULL)
        && (syStrchr(userName, '/') == NULL)
        && (syStrchr(userName, '\\') == NULL)
        && (syStrchr(userName, ':') == NULL)
        && (syStrchr(userName, ';') == NULL)
        && (syStrchr(userName, '|') == NULL)
        && (syStrchr(userName, '<') == NULL)
        && (syStrchr(userName, '>') == NULL)
        && (syStrchr(userName, ',') == NULL)
        && (syStrchr(userName, '?') == NULL)
        && (syStrchr(userName, '*') == NULL)
        && (syStrchr(userName, '=') == NULL));

Exit:
    return isValidName;
}
#endif /* UD_CS_INCLUDELOCALUSERMANAGEMENT */
