/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : EPMAP
 *--------------------------------------------------------------------
 * MODULE        : Client, RPC
 * DEPENDENCIES  :
 *************************************************************************/

#include "nqapi.h"
#include "ccapi.h"
#include "cmrpcdef.h"
#include "ccdcerpc.h"
#include "ccrpc.h"
#include "ccepm.h"
#include "ccerrors.h"

#ifdef UD_CC_INCLUDERPC

/* endpoint mapper (EPMAP) */

#define MAPREQUEST_OPNUM        3

#define PROTOCOL_NAMEDPIPE      0x10
#define PROTOCOL_UUID           0x0d
#define PROTOCOL_SPX            0x0b
#define PROTOCOL_RPCCONNECTION  0x0b
#define PROTOCOL_DODIP          0x09
#define PROTOCOL_DODTCP         0x07

#define EPMAP_HANDLE_SIZE       20

/* pipe: epmapper */
static const NQ_WCHAR pipeName[] = { cmWChar('e'), cmWChar('p'), cmWChar('m'), cmWChar('a'), cmWChar('p'), cmWChar('p'), cmWChar('e'), cmWChar('r'), cmWChar('\0') };
static CCDcerpcPipeDescriptor epmapperDescriptor =
{
    pipeName,
    { cmPack32(0xe1af8308), cmPack16(0x5d1f), cmPack16(0x11c9), { 0x91, 0xa4 }, { 0x08, 0x00, 0x2b, 0x14, 0xa0, 0xfa } },
    cmRpcVersion(3, 0),
    135
};

static const CCDcerpcPipeDescriptor * ccEpmapperGetPipe(void)
{
    return &epmapperDescriptor;
}

typedef struct
{
    NQ_PORT port;
    NQ_IPADDRESS4 ip;
    const CCDcerpcPipeDescriptor *pipe;             /* Pipe descriptor */
    NQ_UINT32 status;                               /* RPC operation status */
}
CCEpmCallbackParams;

typedef struct
{
    const CMRpcUuid uuid;                           /* uuid */
    const NQ_UINT16 version;                        /* interface version as major + minor * 0x10000 */
}
TransferSyntax;

typedef struct
{
    NQ_BYTE protocol;
    const TransferSyntax *transferSyntax;
    NQ_UINT16 length;
    NQ_UINT16 lhsLength;
    NQ_UINT16 rhsLength;
}
FloorDescr;

/* UUID: 32bit NDR (8a885d04-1ceb-11c9-9fe8-08002b104860) */
static const TransferSyntax ndr = {
    CM_RPC_TRANSFERSYNTAXSIGNATURE,
    CM_RPC_NDRVERSION
};


static const FloorDescr towerDescr[] = { 
                                        { PROTOCOL_UUID,            &ndr,   25, 19, 2 },
                                        { PROTOCOL_UUID,            &ndr,   25, 19, 2 },
                                        { PROTOCOL_RPCCONNECTION,   NULL,   7,  1,  2 },
                                        { PROTOCOL_DODTCP,          NULL,   7,  1,  2 }, 
                                        { PROTOCOL_DODIP,           NULL,   9,  1,  4 } 
};

static NQ_COUNT mapRequestCallback(NQ_IOBufPos buffer, NQ_COUNT size, void *params, NQ_BOOL *moreData)
{
    CMRpcPacketDescriptor desc; 
    NQ_UINT32 refId = 0;
    NQ_UINT32 towerLength = 2, maxTowers = 1;
    NQ_UINT16 i;
    CCEpmCallbackParams *callParams;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buff:%p size:%d params:%p moreData:%p", buffer, size, params, moreData);

    callParams = (CCEpmCallbackParams *)params;
    cmRpcSetDescriptor(&desc, buffer, FALSE);
    cmRpcPackUint16(&desc, MAPREQUEST_OPNUM);
    cmRpcPackUint32(&desc, ++refId);
    cmRpcPackNQUuid(&desc, &callParams->pipe->uuid);     /* requested UUID */

    cmRpcPackUint32(&desc, ++refId);
    for (i = 0; i < sizeof(towerDescr) / sizeof(towerDescr[0]); i++)
    {
        towerLength += towerDescr[i].length;
    }
    cmRpcPackUint32(&desc, towerLength);
    cmRpcPackUint32(&desc, towerLength);
    cmRpcPackUint16(&desc, sizeof(towerDescr) / sizeof(towerDescr[0]));    /* number of floors */
    /* floors */
    for (i = 0; i < sizeof(towerDescr) / sizeof(towerDescr[0]); i++)
    {
        cmRpcPackUint16(&desc, towerDescr[i].lhsLength);
        cmRpcPackByte(&desc, towerDescr[i].protocol);

        switch (towerDescr[i].protocol)
        {
        case PROTOCOL_UUID:
            if (i == 0)
            {
                cmRpcPackNQUuid(&desc, &callParams->pipe->uuid);     /* requested UUID */
                cmRpcPackUint16(&desc, (NQ_UINT16)callParams->pipe->version);
            }
            else
            {
                cmRpcPackUuid(&desc, &towerDescr[i].transferSyntax->uuid);
                cmRpcPackUint16(&desc, towerDescr[i].transferSyntax->version);
            }
            cmRpcPackUint16(&desc, towerDescr[i].rhsLength);
            cmRpcPackUint16(&desc, 0);
            break;
        case PROTOCOL_RPCCONNECTION:
            cmRpcPackUint16(&desc, towerDescr[i].rhsLength);
            cmRpcPackSkip(&desc, 2);        /* padding */
            break;
        case PROTOCOL_DODTCP:
            cmRpcPackUint16(&desc, towerDescr[i].rhsLength);
            cmRpcPackUint16ByByteOrder(&desc, callParams->port, TRUE);  /* in NBO */
            break;
        case PROTOCOL_DODIP:
            cmRpcPackUint16(&desc, towerDescr[i].rhsLength);
            cmRpcPackUint32(&desc, callParams->ip);
            cmRpcPackSkip(&desc, 1);        /* padding */
            break;
        }
    }
    cmRpcPackZeroes(&desc, EPMAP_HANDLE_SIZE);   /* zero handle */
    cmRpcPackUint32(&desc, maxTowers);

    *moreData = FALSE;

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%u", (NQ_COUNT)(desc.current - desc.origin));
    return (NQ_COUNT)((desc.current - desc.origin));
}

static NQ_STATUS mapResponseCallback(NQ_IOBufPos data, NQ_COUNT size, void *params, NQ_BOOL moreData)
{
    CMRpcPacketDescriptor desc;
    NQ_UINT32 towerLength = 2;
    NQ_UINT32 numTowers, status;
    NQ_UINT16 i, numFloors;
    CCEpmCallbackParams *callParams;
    NQ_STATUS result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "data:%p size:%d params:%p moreData:%d", data, size, params, moreData);

    callParams = (CCEpmCallbackParams *)params;
    cmRpcSetDescriptor(&desc, (NQ_BYTE *)data, FALSE);
    cmRpcParseSkip(&desc, EPMAP_HANDLE_SIZE);      /* handle */
    cmRpcParseUint32(&desc, &numTowers);

    if (numTowers == 1)
    {
        cmRpcParseSkip(&desc, EPMAP_HANDLE_SIZE);
        cmRpcParseUint32(&desc, &towerLength);
        cmRpcParseUint16(&desc, &numFloors);

        /* floors */
        for (i = 0; i < sizeof(towerDescr) / sizeof(towerDescr[0]); i++)
        {
            switch (towerDescr[i].protocol)
            {
            case PROTOCOL_UUID:
                cmRpcParseSkip(&desc, towerDescr[i].length);
                break;
            case PROTOCOL_RPCCONNECTION:
                cmRpcParseSkip(&desc, towerDescr[i].length);
                cmRpcParseSkip(&desc, 2);       /* padding */
                break;
            case PROTOCOL_DODTCP:
                cmRpcParseSkip(&desc, 3);
                cmRpcParseUint16ByByteOrder(&desc, &callParams->port, TRUE);    /* in NBO */
                break;
            case PROTOCOL_DODIP:
                cmRpcParseSkip(&desc, 5);
                cmRpcParseUint32(&desc, &callParams->ip);
                cmRpcParseSkip(&desc, 1);       /* padding */
                break;
            }
        }
        cmRpcParseUint32(&desc, &status);
        callParams->status = status;
        result = (status == 0) ? NQ_SUCCESS : NQ_FAIL;
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

static NQ_UINT32 ccEpmMap(NQ_HANDLE handle, NQ_IPADDRESS serverIp, const CCDcerpcPipeDescriptor *pipeDesc, NQ_IPADDRESS *ip, NQ_PORT *port)
{
    CCEpmCallbackParams params;

    params.ip = CM_IPADDR_GET4(serverIp);
    params.port = ccEpmapperGetPipe()->port;   /* default port */
    params.pipe = pipeDesc;

    if (TRUE == ccDcerpcOverTcpCall(handle, mapRequestCallback, mapResponseCallback, &params))
    {
        *port = params.port;
        CM_IPADDR_ASSIGN4(*ip, params.ip);
    }
    else
    {
        params.status = (params.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(params.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "EMP::Map failed");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%u", params.status);
    return params.status;
}

NQ_STATUS                                       /* status */
ccEpmapperGetPort(
    const NQ_WCHAR *server,                     /* server name */
    const CCDcerpcPipeDescriptor *pipeDesc,     /* pipe descriptor */
    NQ_IPADDRESS *ip,                           /* pointer to result ip */
    NQ_PORT *port                               /* pointer to result port */
)
{
    NQ_HANDLE pipeHandle = NULL;
    NQ_STATUS result = NQ_FAIL;     /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%s pipeDesc:%p ip:%p port:%p ", cmWDump(server), pipeDesc, ip, port);

    if ((NULL == server) || (NULL == pipeDesc) || (NULL == ip) || (NULL == port))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input parameter");
        goto Exit;
    }

    /* RPC connect to port and bind */
    pipeHandle = ccDcerpcOverTcpBind(NULL, ccEpmapperGetPipe()->port, server, NULL, ccEpmapperGetPipe());
    if (NULL == pipeHandle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error connecting EPM port");
        goto Exit;
    }

    /* get new IP or most likely just new port */
    if (NQ_SUCCESS != ccEpmMap(pipeHandle, ((CCPipe *)pipeHandle)->server->ip, pipeDesc, ip, port))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to get new IP and port");
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Mapped ip:%s port:%d", cmIPDump(ip), *port);
    result = NQ_SUCCESS;

Exit:
    if (NULL != pipeHandle)
    {
        cmListItemUnlock((CMItem *)((CCPipe *)pipeHandle)->server);
    }

    dumpRpcServers(SY_LOG_FUNCTION);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%d", result);
    return result;
}

#endif  /* UD_CC_INCLUDERPC */
