/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : DCERPC library for CIFS Client
 *--------------------------------------------------------------------
 * MODULE        : rpc - rpccore
 * DEPENDENCIES  : None
 ********************************************************************/

#include "ccdcerpc.h"
#include "ccserver.h"
#include "ccshare.h"
#include "ccfile.h"
#include "cmbufman.h"
#include "cmrpcdef.h"
#include "ccerrors.h"

#ifdef UD_CC_INCLUDERPC

/* --- Static definitions, functions & data --- */

#define NOPAYLOAD_BUFFERSIZE 1024     /* buffer size for messages with no call payload */
#define MAX_FRAGMENT 65448            /* max fragment size for calls */

static NQ_COUNT callId;     /* running number */

static const CMRpcDcerpcSyntaxId transferSyntax = {
    CM_RPC_TRANSFERSYNTAXSIGNATURE,
    CM_RPC_NDRVERSION
};

/* build fragment header in a buffer (no PDU yet) */

NQ_IOBufPos ccDcerpcCreateFragmentHeader(CMBufferWriter * pWriter, NQ_BYTE type, NQ_UINT16 fragLength, NQ_BYTE flags, NQ_UINT16 authLength, NQ_COUNT allocHint)
{
    NQ_IOBufPos pFragLength;                            /* pointer to the fragment length field */

    cmBufferWriteByte(pWriter, CM_RP_MAJORVERSION);     /* major vers */
    cmBufferWriteByte(pWriter, CM_RP_MINORVERSION);     /* minor vers */
    cmBufferWriteByte(pWriter, type);                   /* packet type (e.g. - bind) */
    cmBufferWriteByte(pWriter, flags);                  /* flags */
    cmBufferWriteByte(pWriter, CM_RP_DREPLE);           /* data representation: LE byte order */
    cmBufferWriteByte(pWriter, 0);                      /* data representation: float point */
    cmBufferWriteUint16(pWriter, 0);                    /* data representation pad */
    pFragLength = cmBufferWriterGetPosition(pWriter);
    cmBufferWriteUint16(pWriter, fragLength);           /* fragment length */
    cmBufferWriteUint16(pWriter, authLength);           /* auth length */
    cmBufferWriteUint32(pWriter, callId++);             /* call ID */
    if (type == CM_RP_PKT_REQUEST)
    {
        cmBufferWriteUint32(pWriter, allocHint);        /* allocation hint - pdu length */
    }
    return pFragLength;
}

/* update fragment length in the header */
void ccDcerpcSetFragmentLength(CMBufferWriter * pWriter, NQ_IOBufPos pFragLength)
{
    NQ_IOBufPos pTemp;           /* temporary pointer in the writer */
    NQ_UINT16 fragLength;        /* fragment length */

    fragLength = (NQ_UINT16)cmBufferWriterGetDataCount(pWriter);
    pTemp = cmBufferWriterGetPosition(pWriter);
    cmBufferWriterSetPosition(pWriter, pFragLength);
    cmBufferWriteUint16(pWriter, fragLength);        /* fragment length */
    cmBufferWriterSetPosition(pWriter, pTemp);
}

/* parse response fragment header and fill PDU descriptor */
NQ_INT ccDcerpcParseFragmentHeader(NQ_IOBufPos data, CMRpcPacketDescriptor *pDesc, CMRpcDcerpcPacket *pPack)
{
    /* byte order doesnt matter when reading single bytes, we can set the byte order later */
    cmRpcSetDescriptor(pDesc, data, FALSE);
    cmRpcParseByte(pDesc, &pPack->rpcVers);                     /* RPC version */
    cmRpcParseByte(pDesc, &pPack->rpcVersMinor);                /* RPC minor version */
    cmRpcParseByte(pDesc, &pPack->packetType);                  /* packet type */
    cmRpcParseByte(pDesc, &pPack->pfcFlags);                    /* packet flags */
    cmRpcParseByte(pDesc, &pPack->drep.flags);                  /* data representation: flags - byte order */
    cmRpcDescriptorSetByteOrder(pDesc ,(pPack->drep.flags & CM_RP_DREPLE) == 0);                /* set byte order */
    cmRpcParseByte(pDesc, &pPack->drep.fp);                     /* data representation: float point */
    cmRpcParseUint16(pDesc, (NQ_UINT16 *)&pPack->drep.pad);     /* data representation: padding */
    cmRpcParseUint16(pDesc, (NQ_UINT16 *)&pPack->fragLength);   /* fragment length */
    cmRpcParseUint16(pDesc, (NQ_UINT16 *)&pPack->authLength);   /* auth length */
    cmRpcParseUint32(pDesc, (NQ_UINT32 *)&pPack->callId);       /* call ID */
    return pPack->packetType;
}

NQ_HANDLE connectPipe(const NQ_WCHAR * hostName, const AMCredentials * pCredentials, const CCDcerpcPipeDescriptor * pipeDesc, NQ_BOOL doDfs)
{
    CCServer * pServer = NULL;             /* pointer to server */
    CCShare * pShare = NULL;               /* pointer to IPC$ share */
    CCFile * pFile = NULL;                 /* file handle */
    CMRpcDcerpcPacket rpcHeader;           /* fragment header */
    CMRpcPacketDescriptor rpcDescr;        /* incoming packet parser */
    CMBufferWriter writer;                 /* writer for bind PDU */
    NQ_IOBufPos pBuf;                      /* read and write buffer */
    NQ_COUNT dataLen;                      /* buffer length or data length into the buffer */
    NQ_IOBufPos pFragLength;               /* pointer to the fragment length field in the header */
    NQ_UINT16 maxLen;                      /* max fragment length (either xmit or recv) */
    NQ_INT type;                           /* packet type in response */
    NQ_BOOL security[] = {TRUE, FALSE};    /* whether to use extended security */
    NQ_COUNT i;                            /* just a counter */
    NQ_WCHAR * rpcNamePrefixed;            /* RPC pipe name with prefix */
    NQ_INT     lastError = NQ_ERR_OK;
#define RPC_OPENACCESS 0x2019f             /* access mask for opening an RPC pipe as a file */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "host:%s credentials:%p pipe:%p dfs:%s", cmWDump(hostName), pCredentials, pipeDesc, doDfs ? "TRUE" : "FALSE");

    IOBUF_POSINIT(pBuf);

    /* open file */
    for (i = 0; i < sizeof(security)/sizeof(security[0]); i++)
    {
        const AMCredentials * oldCredentials = pCredentials;       /* saved credentials */
        const AMCredentials * newCredentials = oldCredentials;     /* try these credentials first */

        pServer = ccServerFindOrCreate(hostName, security[i], NULL);
        if (NULL != pServer)
        {
            pShare = ccShareConnectIpc(pServer, &newCredentials);
            if (newCredentials != oldCredentials)
            {
                /* new credentials were allocated */
                cmMemoryFree(newCredentials);
            }

            /* try to log on without extended security only if pShare is NULL and server dialect is SMB1 and extended security is not supported */
            if ((NULL != pShare) || (0 != syStrncmp(pServer->smb->name , CM_DIALECT_NT_LM_012, syStrlen(CM_DIALECT_NT_LM_012))) || (TRUE == pServer->useExtendedSecurity))
            {
                cmListItemUnlock((CMItem *)pServer);
                break;
            }

            lastError = syGetLastError();
            cmListItemUnlock((CMItem *)pServer);
        }
        else if (NQ_ERR_INVNETNAME == syGetLastError())
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Invalid host name, host: %s", cmWDump(hostName));
            break;
        }
    }

    if (NULL == pShare)
    {
        if (NQ_ERR_OK != lastError)
        {
            sySetLastError(lastError);
        }
        else if (NQ_ERR_OK == syGetLastError())
        {
            sySetLastError(NQ_ERR_LOGONFAILURE);
        }

        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to connect to server");
        goto Exit;
    }

    rpcNamePrefixed = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)((cmWStrlen (pipeDesc->name) + cmWStrlen(pServer->smb->rpcNamePrefix) + 1 ) * sizeof(NQ_WCHAR)));
    if (NULL == rpcNamePrefixed)
    {
        sySetLastError(NQ_ERR_NOMEM);
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    syWStrcpy(rpcNamePrefixed, pServer->smb->rpcNamePrefix);
    syWStrcat(rpcNamePrefixed, pipeDesc->name);
    pFile = ccFileCreateOnServer(
                pShare,
                rpcNamePrefixed,
                FALSE,
                RPC_OPENACCESS,
                FILE_SM_DENY_NONE,
                0,
                FALSE,
                0,
                FILE_CA_FAIL,
                FILE_OA_OPEN,
                TRUE,
                FILE_OPLOCK_LEVEL_BATCH
                );
    cmMemoryFree(rpcNamePrefixed);
    if (NULL == pFile)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to connect to %s", cmWDump(pipeDesc->name));
        goto Error1;
    }

    cmListItemUnlock((CMItem *)pShare);
    pShare = NULL;
    /* allocate buffer of enough size for both request and response */
    pBuf = cmIOBufManTake(NOPAYLOAD_BUFFERSIZE);
    if (IOBUF_ISNULL(pBuf))
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
        goto Error2;
    }

    /* compose Bind request */
    /* header */
    dataLen = NOPAYLOAD_BUFFERSIZE;
    cmBufferWriterInit(&writer, pBuf, dataLen);
    pFragLength = ccDcerpcCreateFragmentHeader(&writer, CM_RP_PKT_BIND, 0, CM_RP_PFCFLAG_FIRST | CM_RP_PFCFLAG_LAST, 0, 0);
    /* build Bind PDU without context */
    cmBufferWriteUint16(&writer, (NQ_UINT16)MAX_FRAGMENT);     /* max xmit fragment */
    cmBufferWriteUint16(&writer, (NQ_UINT16)MAX_FRAGMENT);     /* max recv fragment */
    cmBufferWriteUint32(&writer, 0);                           /* assocGroupId */
    cmBufferWriteByte(&writer, 1);                             /* num contexts */
    cmBufferWriterAlign(&writer, pBuf, 4);
    /* build context */
    cmBufferWriteUint16(&writer, 0);                           /* context ID */
    cmBufferWriteByte(&writer, 1);                             /* num transfer syntaxes */
    cmBufferWriterAlign(&writer, pBuf, 4);
    cmBufferWriteUuid(&writer, &pipeDesc->uuid);               /* pipe GUID */
    cmBufferWriteUint32(&writer, pipeDesc->version);           /* major/minor version */

    /* transfer syntax signature */
    cmBufferWriteUint32(&writer, (NQ_UINT32)cmGetSUint32(transferSyntax.uuid.timeLow));
    cmBufferWriteUint16(&writer, (NQ_UINT16)cmGetSUint16(transferSyntax.uuid.timeMid));
    cmBufferWriteUint16(&writer, (NQ_UINT16)cmGetSUint16(transferSyntax.uuid.timeHiVersion));
    cmBufferWriteBytes(&writer, transferSyntax.uuid.clockSeq, sizeof(transferSyntax.uuid.clockSeq));
    cmBufferWriteBytes(&writer, transferSyntax.uuid.node, sizeof(transferSyntax.uuid.node));

    cmBufferWriteUint32(&writer, transferSyntax.interfaceVersion);  /* transfer syntax version */
    /* update fragment length */
    ccDcerpcSetFragmentLength(&writer, pFragLength);
    dataLen = cmBufferWriterGetDataCount(&writer);

    /* bind the pipe */
    if (FALSE == ccWriteFile(pFile, pBuf, dataLen, &dataLen))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending Bind request");
        goto Error2;
    }

    /* Analyze the response */
    if (FALSE == ccReadFile(pFile, pBuf, NOPAYLOAD_BUFFERSIZE, &dataLen))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving Bind response");
        goto Error2;
    }

    type = ccDcerpcParseFragmentHeader(pBuf, &rpcDescr, &rpcHeader);
    if (CM_RP_PKT_BINDACK != type)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to bind to the pipe");
        sySetLastError(NQ_ERR_BADACCESS);
        goto Error2;
    }

    cmRpcParseUint16(&rpcDescr, &maxLen);        /* max xmit fragment length */
    pFile->maxRpcXmit = (NQ_UINT16)(maxLen > MAX_FRAGMENT ? MAX_FRAGMENT : maxLen);
    cmRpcParseUint16(&rpcDescr, &maxLen);        /* max recv fragment length */
    pFile->maxRpcRecv = (NQ_UINT16)(maxLen > MAX_FRAGMENT ? MAX_FRAGMENT : maxLen);

    goto Exit;

Error2:
    if(NULL != pFile)
    {
        cmListItemUnlock((CMItem *)pFile);
        pFile = NULL;
    }

Error1:
    if(NULL != pShare)
    {
        cmListItemUnlock((CMItem *)pShare);
    }

Exit:
    cmIOBufManGive(pBuf);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p error:0x%x", pFile, syGetLastError());
    return (NQ_HANDLE)pFile;
}

/* -- API functions -- */

NQ_BOOL ccDcerpcStart(void)
{
    callId = 1;
    return TRUE;
}

void ccDcerpcShutdown(void)
{

}

NQ_HANDLE ccDcerpcConnect(const NQ_WCHAR * hostName, const AMCredentials * pCredentials, const CCDcerpcPipeDescriptor * pipeDesc, NQ_BOOL doDfs)
{
    NQ_HANDLE handle = NULL;    /* resulting handle */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "host:%s credentials:%p pipe:%p dfs:%s", cmWDump(hostName), pCredentials, pipeDesc, doDfs ? "TRUE" : "FALSE");

    if (NULL != pCredentials)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "provided credentials");
        handle = connectPipe(hostName, pCredentials, pipeDesc, doDfs);
        if (NQ_ERR_INVNETNAME == syGetLastError())
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to pipe");
            goto Exit;
        }
    }
#ifndef UD_CC_RPCDISABLECONNECTIONFALLBACKS
    if (NULL == handle)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "anonymous credentials");
        handle = connectPipe(hostName, ccUserGetAnonymousCredentials(), pipeDesc, doDfs);
        if (NQ_ERR_INVNETNAME == syGetLastError())
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to pipe");
            goto Exit;
        }
    }
    if (NULL == handle)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "default credentials");
        handle = connectPipe(hostName, NULL, pipeDesc, doDfs); /* default credentials */
        if (NQ_ERR_INVNETNAME == syGetLastError())
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to pipe");
            goto Exit;
        }
    }
#endif /* UD_CC_RPCDISABLECONNECTIONFALLBACKS */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", handle);
    return handle;
}

NQ_BOOL ccDcerpcCall(NQ_HANDLE pipeHandle, CCDcerpcRequestCallback request, CCDcerpcResponseCallback response, void * callParams)
{
    CCFile * pFile;                /* casted file handle */
    NQ_BOOL res = TRUE;            /* operation result */
    NQ_BOOL firstFrag = TRUE;      /* whether fragment is the first one */
    NQ_IOBufPos pBuf;              /* read and write buffer */
    NQ_COUNT dataLen;              /* buffer length or data length into the buffer */
    NQ_BOOL moreData;              /* whether application has more data */
    NQ_STATUS status;              /* status of callback execution */
    NQ_COUNT pduLen;               /* PDU length applied by the caller (in callback) */
    NQ_BOOL result = FALSE;        /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pipe:%p req:%p res:%p params:%p", pipeHandle, request, response, callParams);

    /* allocate buffer of enough size for the request */

    if (NULL == pipeHandle)
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    if (!ccValidateFileHandle(pipeHandle))
    {
        LOGERR(CM_TRC_LEVEL_ERROR , "Invalid Handle");
        sySetLastError(NQ_ERR_INVALIDHANDLE);
        goto Exit;
    }

    pFile = (CCFile *)pipeHandle;
    dataLen = pFile->maxRpcXmit;
    pBuf = cmIOBufManTake(dataLen);
    if (IOBUF_ISNULL(pBuf))
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
        goto Exit;
    }

    do
    {
        CMBufferWriter writer;          /* writer for bind PDU */
        NQ_BYTE fragType;                /* first and/or last flags */

        /* build header (just to skip it) and build Request PDU stub data */
        cmBufferWriterInit(&writer, pBuf, dataLen);
        ccDcerpcCreateFragmentHeader(&writer, CM_RP_PKT_REQUEST, 0, 0, 0, 0);
        cmBufferWriteUint16(&writer, 0);     /* context id */

        /* place stub data */
        moreData = FALSE;
        pduLen = (*request)(
                    cmBufferWriterGetPosition(&writer),
                    dataLen - cmBufferWriterGetDataCount(&writer),
                    callParams,
                    &moreData
                    );
        if (pduLen == 0)
        {
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            res = FALSE;
            break;
        }

        /* re-build header and set fragment length */
        fragType = 0;
        if (firstFrag)
        {
            fragType |= CM_RP_PFCFLAG_FIRST;
            firstFrag = FALSE;
        }
        if (!moreData)
        {
            fragType |= CM_RP_PFCFLAG_LAST;
        }
        dataLen = cmBufferWriterGetDataCount(&writer) + pduLen;
        cmBufferWriterSetPosition(&writer, pBuf);
        ccDcerpcCreateFragmentHeader(&writer, CM_RP_PKT_REQUEST, (NQ_UINT16)dataLen, fragType, 0, pduLen);
        res = ccWriteFile(pipeHandle, pBuf, dataLen, &dataLen);
    }
    while(res && moreData);

    cmIOBufManGive(pBuf);

    if (!res)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error sending request");
        goto Exit;
    }

    /* allocate buffer of enough size for the response */
    pFile = (CCFile *)pipeHandle;
    dataLen = pFile->maxRpcRecv;
    pBuf = cmIOBufManTake(dataLen);
    if (IOBUF_ISNULL(pBuf))
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        LOGERR(CM_TRC_LEVEL_ERROR, "Unable to allocate buffer");
        goto Exit;
    }

    /* loop on sending Read(AndX) SMB */
    do
    {
        CMRpcPacketDescriptor rpcDescr;    /* incoming packet parser */
        CMRpcDcerpcPacket rpcHeader;       /* fragment header */
        NQ_BYTE cancelCount;               /* cancel count from PDU */
        NQ_INT type;                       /* packet type in response */

        res = ccReadFile(pipeHandle, pBuf, pFile->maxRpcRecv, &dataLen);
        if (!res)
        {
            break;
        }
        /* parse header and withdraw packet type */
        type = ccDcerpcParseFragmentHeader(pBuf, &rpcDescr, &rpcHeader);
        if (type != CM_RP_PKT_RESPONSE)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Received packet is not Response");
            sySetLastError(NQ_ERR_BADPARAM);
            res = FALSE;
            break;
        }

        /* parse PDU */
        cmRpcParseSkip(&rpcDescr, 4 + 2);   /* alloc hint + context id */
        cmRpcParseByte(&rpcDescr, &cancelCount);
        if (0 != cancelCount)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Packet received with cancel count: %d", cancelCount);
            sySetLastError(NQ_ERR_BADPARAM);
            res = FALSE;
            break;
        }

        /* withdraw stub data */
        moreData = (0 == (rpcHeader.pfcFlags & CM_RP_PFCFLAG_LAST));
        cmRpcAllign(&rpcDescr, 2);
        status = (*response)(rpcDescr.current, (NQ_COUNT)(rpcHeader.fragLength) - cmRpcGetDataCount(&rpcDescr), callParams, moreData);
        if (NQ_SUCCESS != status)
        {
            sySetLastError(ccErrorsStatusToNq((NQ_UINT32)status, TRUE));
            res = FALSE;
            break;
        }
    }
    while (moreData && res);

    cmIOBufManGive(pBuf);

    if ((NQ_COUNT) NQ_FAIL == dataLen)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Error receiving response");
        goto Exit;
    }

    result = res;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_STATUS ccDcerpcDisconnect(NQ_HANDLE pipeHandle)
{
    return ccCloseHandle(pipeHandle);
}

#endif /* UD_CC_INCLUDERPC */
