/*********************************************************************
 *
 *           Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *********************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : CIFS Client domain related operations
 *--------------------------------------------------------------------
 * MODULE        : rpc - domain
 * DEPENDENCIES  : None
 ********************************************************************/

#include "nqapi.h"
#include "ccdomain.h"

#include "cmsdescr.h"
#include "cmfinddc.h"
#include "ccfile.h"
#include "ccparams.h"
#include "cclsarpc.h"
#include "ccsamrpc.h"
#include "ccnetlgn.h"
#include "ccuser.h"
#include "ccerrors.h"
#include "cmcrypt.h"
#include "cmbuf.h"
#ifdef UD_CS_INCLUDEMULTITENANCY
#include "csstoragedata.h"
#endif /* UD_CS_INCLUDEMULTITENANCY */
#include "ccrpc.h"
#include "amspnego.h"
#include "amntlmss.h"
#include "ccepm.h"

#ifdef UD_CC_INCLUDEDOMAINMEMBERSHIP

#define NETRLOGONSAMLOGON_OPCODE   2
#define NETRLOGONSAMLOGONEX_OPCODE 39

#define MAX_USER_NAME_LENGTH            DOMAIN_LENGTH
#define MAX_DOMAIN_NAME_LENGTH          DOMAIN_LENGTH
#define MAX_PLAIN_PASSWORD_LENGTH       PASSWORD_LENGTH
#define LM_HASH_SIZE                    HASHED_PASSWORD_SIZE
#define NTLM_HASH_SIZE                  HASHED_PASSWORD_SIZE
#define COMP_ACCOUNT_PASSWORD_LENGTH    28

#define NETLOGON_PRMCTRL_CLEARTEXT_PASSWORD_ALLOWED         0x00000002
#define NETLOGON_PRMCTRL_UPDATE_LOGON_STATISTICS            0x00000004
#define NETLOGON_PRMCTRL_RETURN_USER_PARAMETERS             0x00000008
#define NETLOGON_PRMCTRL_DONT_TRY_GUEST_ACCOUNT             0x00000010
#define NETLOGON_PRMCTRL_ALLOW_SERVER_TRUST_ACCOUNT         0x00000020
#define NETLOGON_PRMCTRL_RETURN_PASSWORD_EXPIRY             0x00000040
#define NETLOGON_PRMCTRL_USE_CLIENT_CHALLENGE               0x00000080
#define NETLOGON_PRMCTRL_TRY_GUEST_ACCOUNT_ONLY             0x00000100
#define NETLOGON_PRMCTRL_RETURN_PROFILE_PATH                0x00000200
#define NETLOGON_PRMCTRL_TRY_SPECIFIED_DOMAIN_ONLY          0x00000400
#define NETLOGON_PRMCTRL_ALLOW_WORKSTATION_TRUST_ACCOUNT    0x00000800
#define NETLOGON_PRMCTRL_DISABLE_PERSONAL_FALLBACK          0x00001000
#define NETLOGON_PRMCTRL_ALLOW_FORCE_GUEST                  0x00002000
#define NETLOGON_PRMCTRL_CLEARTEXT_PASSWORD_SUPPLIED        0x00004000
#define NETLOGON_PRMCTRL_USE_DOMAIN_FOR_ROUTING_ONLY        0x00008000
#define NETLOGON_PRMCTRL_ALLOW_MSVCHAPV2                    0x00010000
#define NETLOGON_PRMCTRL_S4U2SELF                           0x00020000
#define NETLOGON_PRMCTRL_CHECK_LOGONHOURS_FOR_S4U           0x00040000
#define NETLOGON_PRMCTRL_SUBAUTHENTICATION_DLL_EX           0x00100000
#define NETLOGON_PRMCTRL_DEFAULT                            (NETLOGON_PRMCTRL_ALLOW_WORKSTATION_TRUST_ACCOUNT | NETLOGON_PRMCTRL_ALLOW_SERVER_TRUST_ACCOUNT)


/* 0xe00500b0 */
#define SAMR_CREATE_USER_DEFAULT_ACCESS     (SAMR_AM_GENERICREAD | SAMR_AM_GENERICWRITE | SAMR_AM_GENERICEXECUTE | \
                                            SAMR_AM_WRITEDAC | SAMR_AM_DELETE | \
                                            SAMR_AM_USERSETPASSWORD | SAMR_AM_USERGETATTRIBUTES | SAMR_AM_USERSETATTRIBUTES)

/* -- Static data -- */
static NQ_BOOL isModuleInitialized = FALSE;

/* -- Static prototypes --- */

static NQ_UINT32 createComputerAccount(const NQ_WCHAR *server, const CMSdDomainSid *domain, const NQ_WCHAR *computer, NQ_BYTE secret[16], NQ_BOOL updateSecret);
static NQ_UINT32 removeComputerAccount(const NQ_WCHAR *server, const CMSdDomainSid *domain, const NQ_WCHAR *computer);
static NQ_BOOL domainLogon(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin, NQ_BYTE secret[16]);
static NQ_BOOL domainJoin(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin, NQ_BYTE secret[16], const NQ_WCHAR * dnsList, NQ_WCHAR *domainNB, CMSdDomainSid * domainSid, NQ_BOOL updateSecret);
static NQ_BOOL domainLeave(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin);
static NQ_HANDLE getNetlogonHandle(const NQ_WCHAR *dc, const NQ_WCHAR * domainDNS , const AMCredentials * admin);
static NQ_BOOL isValidADComputerAccountName(const NQ_CHAR *name);

typedef struct
{
#ifndef UD_CS_INCLUDEMULTITENANCY
    NQ_HANDLE netlogonHandle;
#endif /* UD_CS_INCLUDEMULTITENANCY */
    /*const NQ_WCHAR netlogonDc[UD_NQ_HOSTNAMESIZE];*/
    SYMutex netlogonGuard;
}
StaticData;

#ifdef SY_FORCEALLOCATION
static StaticData* staticData = NULL;
#else  /* SY_FORCEALLOCATION */
static StaticData staticDataSrc;
static StaticData* staticData = &staticDataSrc;
#endif /* SY_FORCEALLOCATION */

NQ_BOOL ccDomainStart()
{
    NQ_BOOL result = TRUE;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (TRUE == isModuleInitialized)
    {
        result = TRUE;
        goto Exit;
    }

    /* allocate memory */
#ifdef SY_FORCEALLOCATION
    staticData = (StaticData *)cmMemoryAllocate(sizeof(*staticData));
    if (NULL == staticData)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
#endif /* SY_FORCEALLOCATION */
#ifndef UD_CS_INCLUDEMULTITENANCY
    staticData->netlogonHandle = NULL;
#endif /* UD_CS_INCLUDEMULTITENANCY */

    syMutexCreate(&staticData->netlogonGuard);

    isModuleInitialized = TRUE;

Exit:

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

void ccDomainShutdown()
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (TRUE == isModuleInitialized)
    {
#ifdef SY_FORCEALLOCATION
        if (NULL == staticData)
        {
            goto Exit;
        }
#endif /* SY_FORCEALLOCATION */
#ifndef UD_CS_INCLUDEMULTITENANCY
        if (NULL != staticData->netlogonHandle)
        {
            ccDcerpcDisconnect(staticData->netlogonHandle);
            staticData->netlogonHandle = NULL;
        }
#endif /* UD_CS_INCLUDEMULTITENANCY */

        syMutexDelete(&staticData->netlogonGuard);

        /* release memory */
#ifdef SY_FORCEALLOCATION
        if (NULL != staticData)
        {
            cmMemoryFree(staticData);
        }
        staticData = NULL;
#endif /* SY_FORCEALLOCATION */

        isModuleInitialized = FALSE;
    }

#ifdef SY_FORCEALLOCATION
Exit:
#endif /* SY_FORCEALLOCATION */
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}


/* --- Static functions --- */

static NQ_HANDLE getNetlogonHandle(const NQ_WCHAR *dc, const NQ_WCHAR * domainDNS ,const AMCredentials * admin)
{
    NQ_HANDLE netlogonHandle = NULL;
#ifdef UD_CS_INCLUDEMULTITENANCY
    CSTenant *domainTenant = csTenantFind(domainDNS);

    if (NULL != domainTenant)
    {
        netlogonHandle = domainTenant->netlogonHandle;
    }
#else
    netlogonHandle = staticData->netlogonHandle;
#endif /* UD_CS_INCLUDEMULTITENANCY */

    if (FALSE == ccValidateFileHandle(netlogonHandle))
    {
        netlogonHandle = NULL;
    }

    if (NULL == netlogonHandle)
    {
        {
            netlogonHandle = ccDcerpcConnect(dc, admin, ccNetlogonGetPipe(), FALSE);
        }
#ifdef UD_CS_INCLUDEMULTITENANCY
        if (NULL != domainTenant)
        {
            domainTenant->netlogonHandle = netlogonHandle;
        }
#else
        staticData->netlogonHandle = netlogonHandle;
#endif /* UD_CS_INCLUDEMULTITENANCY */
    }
    return netlogonHandle;
}

static NQ_UINT32 createComputerAccount(
    const NQ_WCHAR * server,
    const CMSdDomainSid * domain,
    const NQ_WCHAR * computer,
    NQ_BYTE secret[16],
    NQ_BOOL updateSecret
    )
{
    NQ_BYTE password[COMP_ACCOUNT_PASSWORD_LENGTH];     /* random computer account password */
    NQ_WCHAR * name = NULL;                             /* computer name with $ postfix */
    const NQ_WCHAR dollarSign[] = {cmWChar('$'), cmWChar('\0')};  /* computer name postfix */
    NQ_HANDLE samr;                                     /* RPC SAMR file handle */
    NQ_UINT32 type;                                     /* RPC value */
    NQ_UINT32 access;                                   /* RPC SAMR access */
    NQ_UINT32 rid;                                      /* user RID */
    CMRpcPolicyHandle c, d, u;                          /* RPC policies handles */
    NQ_UINT32 flags;                                    /* RPC SAMR flags */
    NQ_UINT32 status = (NQ_UINT32)NQ_ERR_GENERAL;       /* generic result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%s domain:%p computer:%p secret:%p updateSecret:%d", cmWDump(server), domain, computer, &secret, updateSecret);

    /* connect to SAMR */
    samr = ccDcerpcConnect(server, ccUserGetAdministratorCredentials(), ccSamGetPipe(), FALSE);
    if (NULL == samr)
    {
        status = (NQ_UINT32)syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "Connect to SAMR failed");
        goto Exit;
    }

    /* SAMR::Connect2 */
    status = ccSamrConnect5(samr, 0, &c);
    if (NQ_SUCCESS != status)
    {
        status = (NQ_UINT32)syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "SAMR::Connect2 failed");
        goto ExitClosePipe;
    }

    /* SAMR::OpenDomain */
    status = ccSamrOpenDomain(samr, &c, domain, SAMR_AM_MAXIMUMALLOWED, &d);
    if (NQ_SUCCESS != status)
    {
        status = (NQ_UINT32)syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "SAMR::OpenDomain failed");
        goto ExitCloseConnection;
    }

    /* append '$' to computer name */
    name = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_WCHAR) * (cmWStrlen(computer) + 2)));
    if (NULL == name)
    {
        status = (NQ_UINT32)NQ_ERR_NOMEM;
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto ExitCloseDomain;
    }
    cmWStrcpy(name, computer);
    cmWStrcat(name, dollarSign);

    /* SAMR::CreateUser2 */
    access = SAMR_CREATE_USER_DEFAULT_ACCESS;
    status = ccSamrCreateUser2(samr, &d, name, SAMR_ACB_WSTRUST, access, &u, &rid, &access);
    if ((NQ_SUCCESS != status) && ((FALSE == updateSecret) || ((TRUE == updateSecret) && (SMB_STATUS_USER_EXISTS != status))))
    {
        status = (NQ_UINT32)syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "SAMR::CreateUser2 failed");
        goto ExitCloseDomain;
    }

    if (SMB_STATUS_USER_EXISTS == status)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Computer account already exists, updating it's secret");
        /* SAMR::Close(user) */
        ccSamrClose(samr, &u);
    }

    /* lookup account name */
    /* SAMR::LookupNames */
    status = ccSamrLookupNames(samr, &d, name, &rid, &type);
    if (NQ_SUCCESS != status)
    {
        status = (NQ_UINT32)syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "SAMR::LookupNames failed");
        goto ExitCloseDomain;
    }

    /* set password */
    /* SAMR::OpenUser */
    status = ccSamrOpenUser(samr, &d, &rid, SAMR_AM_MAXIMUMALLOWED, &u);
    if (NQ_SUCCESS != status)
    {
        status = (NQ_UINT32)syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "SAMR::OpenUser failed");
        goto ExitCloseDomain;
    }

    /* create random computer account password */
    cmCreateRandomByteSequence(password, COMP_ACCOUNT_PASSWORD_LENGTH);

    /* create hashed computer account password  - secret used in domain logons */
    cmMD4(secret, password, COMP_ACCOUNT_PASSWORD_LENGTH);
    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "secret", secret, 16);

    /* SAMR::SetUserInfo2 */
    status = ccSamrSetUserPassword(samr, &u, password, COMP_ACCOUNT_PASSWORD_LENGTH);
    if (NQ_SUCCESS != status)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "SAMR::SetUserInfo2 failed");
        goto ExitCloseUser;
    }

    /* set flags */
    /* SAMR::SetUserInfo */
    flags = SAMR_ACB_WSTRUST;
    status = ccSamrSetUserInfo2(samr, &u, 16, (NQ_BYTE *)&flags);

ExitCloseUser:
    /* SAMR::Close(user) */
    ccSamrClose(samr, &u);
ExitCloseDomain:
    /* SAMR::Close(domain) */
    ccSamrClose(samr, &d);
ExitCloseConnection:
    /* SAMR::Close(connection) */
    ccSamrClose(samr, &c);
ExitClosePipe:
    /* close SAMR */
    ccDcerpcDisconnect(samr);
Exit:
    cmMemoryFree(name);
    if (NQ_SUCCESS != status)
    {
        sySetLastError(status);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%x error:0x%x", status, syGetLastError());
    return status;
}

static NQ_UINT32 removeComputerAccount(
    const NQ_WCHAR *server,
    const CMSdDomainSid *domain,
    const NQ_WCHAR *computer
    )
{
    NQ_WCHAR * compName = NULL;                                     /* computer name with a postfix */
    const NQ_WCHAR dollarSign[] = {cmWChar('$'), cmWChar('\0')};    /* postfix */
    NQ_HANDLE samr;                                                 /* SAMR file handle */
    NQ_UINT32 status = (NQ_UINT32)NQ_ERR_GENERAL;                   /* generic result */
    CMRpcPolicyHandle c, d, u;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%s domain:%p computer:%s", cmWDump(server), domain, cmWDump(computer));

    /* connect to SAMR */
    if ((samr = ccDcerpcConnect(server, ccUserGetAdministratorCredentials(), ccSamGetPipe(), FALSE)) != NULL)
    {
        NQ_UINT32 rid, type;

        /* SAMR::Connect2 */
        if ((status = ccSamrConnect5(samr, 0, &c)) == NQ_SUCCESS)
        {
            /* SAMR::OpenDomain */
            if ((status = ccSamrOpenDomain(samr, &c, domain, SAMR_AM_MAXIMUMALLOWED, &d)) == NQ_SUCCESS)
            {
                compName = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)(sizeof(NQ_WCHAR) * (cmWStrlen(computer) + 2)));
                if (NULL == compName)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    status = (NQ_UINT32)NQ_ERR_NOMEM;
                    sySetLastError((NQ_STATUS)status);
                    goto Error;
                }
                cmWStrcpy(compName, computer);
                cmWStrcat(compName, dollarSign);

                /* SAMR::LookupNames */
                if ((status = ccSamrLookupNames(samr, &d, compName, &rid, &type)) == NQ_SUCCESS)
                {
                    /* SAMR::OpenUser */
                    if ((status = ccSamrOpenUser(samr, &d, &rid, SAMR_AM_MAXIMUMALLOWED, &u)) == NQ_SUCCESS)
                    {
                        /* SAMR::DeleteUser */
                        status = ccSamrDeleteUser(samr, &u);
                    }
                }
                /* SAMR::Close(domain) */
                ccSamrClose(samr, &d);
            }

            /* SAMR::Close(connection) */
            ccSamrClose(samr, &c);
        }

        /* close SAMR */
        ccDcerpcDisconnect(samr);
    }
    else
        status = (NQ_UINT32)syGetLastError();

    goto Exit;

Error:
    ccSamrClose(samr, &d);
    ccSamrClose(samr, &c);
    ccDcerpcDisconnect(samr);

Exit:
    cmMemoryFree(compName);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%u", status);
    return status;

}



static NQ_BOOL domainLogon(
    const NQ_WCHAR * domain,
    const NQ_WCHAR * computer,
    const AMCredentials * admin,
    NQ_BYTE secret[16]
    )
{
    const NQ_WCHAR * dc = NULL;             /* DC name in Unicode */
    NQ_CHAR * dcA = NULL;                   /* DC name in ASCII */
    const NQ_CHAR * domainA = NULL;         /* domain name in ASCII */
    NQ_UINT32 status = NQ_SUCCESS;          /* generic result */
    NQ_BOOL result = FALSE;                 /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "domain:%s computer:%s admin:%p, secret:%p", cmWDump(domain), cmWDump(computer), admin, &secret);

    if (secret == NULL)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "null secret");
        goto Exit;
    }

    if (cmWStrlen(computer) > 15)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "computer name exceeds 15 characters");
        goto Exit;
    }

    /* find domain controller by domain name */
    domainA = cmMemoryCloneWStringAsAscii(domain);
    dcA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_BYTE) * CM_DNS_NAMELEN);
    if (NULL == domainA || NULL == dcA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
    if ((status = (NQ_UINT32)cmGetDCNameByDomain(domainA, NULL, dcA)) != NQ_SUCCESS)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to get dc for domain %s", domainA);
        sySetLastError(syGetLastError());
        goto Exit;
    }
    dc = cmMemoryCloneAString(dcA);
    if (NULL == dc)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    /* replace default credentials with domain administrator ones */
    cmWStrupr((NQ_WCHAR *)admin->domain.name); /* capitalize credentials domain name */
    ccUserSetAdministratorCredentials(admin);

    /* connect to NETLOGON */   
    {
        CCNetlogonCredential credentials;
        NQ_HANDLE netLogon;

        syMutexTake(&staticData->netlogonGuard);

        /* connect to netlogon pipe */
        netLogon = getNetlogonHandle(dc, domain, admin);
        if (NULL == netLogon)
        {
            syMutexGive(&staticData->netlogonGuard);
            goto Exit;
        }

        /* generate client random challenge */
        cmCreateRandomByteSequence(credentials.client, sizeof(credentials.client));

        /* send client challenge and get server random challenge */
        if ((status = ccNetrServerReqChallenge(netLogon, dc, computer, &credentials)) == NQ_SUCCESS)
        {
            /* calculate new client challenge and new server challenge */
            ccNetlogonCredentialsGenerate(&credentials, secret);

            /* send new client challenge */
            status = ccNetrServerAuthenticate2(netLogon, dc, computer, &credentials, NULL);
        }
        syMutexGive(&staticData->netlogonGuard);
    }
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "status = 0x%x", status);

    /* restore previous user credentials */
    ccUserSetAdministratorCredentials(NULL);

    sySetLastError((NQ_UINT32)status);
    result = (status == NQ_SUCCESS);

Exit:
    cmMemoryFree(domainA);
    cmMemoryFree(dcA);
    cmMemoryFree(dc);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE: join domain
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  computer name
 *          IN  domain administrator credentials
 *          IN/OUT domain secret
 *          IN  semicolon delimited list of DNS servers
 *          IN/OUT domain name NetBIOS form
 *          IN/OUT domain SID
 *          IN updateSecret whether to just update secret when account is found
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
static NQ_BOOL domainJoin(
    const NQ_WCHAR * domain,
    const NQ_WCHAR * computer,
    const AMCredentials * admin,
    NQ_BYTE secret[16],
    const NQ_WCHAR * dnsList,
    NQ_WCHAR * domainNetBios,
    CMSdDomainSid * domainSid,
    NQ_BOOL updateSecret
    )
{
    NQ_UINT32 status;                       /* generic result */
    const NQ_CHAR * domainA = NULL;         /* domain name in ASCII */
    NQ_CHAR * dcA = NULL;                   /* DC name in ASCII */
    const NQ_WCHAR * dc = NULL;             /* DC name in Unicode */
    CCLsaPolicyInfoDomain info;             /* domain info */
    NQ_BOOL result = FALSE;                 /* return value */
    NQ_WCHAR *computerCapitalized = NULL;   /* capitalized computer name */
    NQ_CHAR  *computerNameA = NULL;         /* computer name in ASCI */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "domain:%s computer:%p admin:%p secret:%p dnsList:%p domainNetBios:%p domainSid:%p updateSecret:%d", cmWDump(domain), computer, admin, secret, dnsList, domainNetBios, domainSid, updateSecret);

    if (NULL == secret)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "null secret");
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "computer name: %s", cmWDump(computer));

    computerNameA = cmMemoryCloneWStringAsAscii(computer);
    if (FALSE == isValidADComputerAccountName(computerNameA))
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "illegal AD computer account name: %s", computerNameA);
        goto Exit;
    }

    computerCapitalized = (NQ_WCHAR *)cmMemoryCloneWString(computer);
    if (NULL == computerCapitalized)
    {
        sySetLastError(NQ_ERR_NOMEM);
        LOGERR(CM_TRC_LEVEL_ERROR, "out of memory");
        goto Exit;
    }
    cmWStrupr(computerCapitalized); /* capitalize computer name */

    /* find domain controller by domain name */
    domainA = cmMemoryCloneWStringAsAscii(domain);
    dcA = (NQ_CHAR *)cmMemoryAllocate(CM_BUFFERLENGTH(NQ_CHAR, CM_DNS_NAMELEN) * sizeof(NQ_CHAR));
    if ((NULL == domainA) || (NULL == dcA))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
    if (NQ_SUCCESS != (status = (NQ_UINT32)cmGetDCNameByDomain(domainA, dnsList, dcA)))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to get dc for domain %s", domainA);
        sySetLastError(syGetLastError());
        goto Exit;
    }
    dc = cmMemoryCloneAString(dcA);
    if (NULL == dc)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    /* replace default credentials with domain administrator ones */
    cmWStrupr((NQ_WCHAR *)admin->domain.name); /* capitalize credentials domain name */
    ccUserSetAdministratorCredentials(admin);

    /* get domain SID */
    if (NQ_SUCCESS == (status = ccLsaPolicyQueryInfoDomain(admin, dc, &info)))
    {
        /* create computer account in this domain */
        status = createComputerAccount(dc, &info.sid, computerCapitalized, secret, updateSecret);
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "status = 0x%x", status);

        if (NULL != domainSid)
        {
            *domainSid = info.sid;
        }
    }

    if (NULL != domainNetBios)
    {
        CCLsaPolicyInfoDomain domainInfo;

        if (NQ_SUCCESS == ccLsaDsRoleGetPrimaryDomainInformation(dc, admin, &domainInfo))
        {
            cmWStrcpy(domainNetBios, domainInfo.name);
        }
        else
        {
            *domainNetBios = cmWChar(0);
        }
    }

    /* restore previous user credentials */
    ccUserSetAdministratorCredentials(NULL);

    sySetLastError((NQ_STATUS)status);
    result = (status == NQ_SUCCESS);

Exit:
    if (NULL != computerNameA)
    {
        cmMemoryFree(computerNameA);
    }

    if (NULL != computerCapitalized)
    {
        cmMemoryFree(computerCapitalized);
    }

    if (NULL != domainA)
    {
        cmMemoryFree(domainA);
    }

    if (NULL != dcA)
    {
        cmMemoryFree(dcA);
    }

    if (NULL != dc)
    {
        cmMemoryFree(dc);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

typedef struct
{
    NQ_WCHAR * domainName;              /* NetBIOS name of the domain */
    NQ_WCHAR * serverName;              /* server name (DC) */
    NQ_WCHAR * ownHostname;             /* NetBIOS own host name */
    NQ_WCHAR * userName;                /* user name as in client's authentication request */
    NQ_WCHAR * workstationName;         /* NetBIOS name of the workstation from which the user is logging on */
    const NQ_BYTE  * serverChallenge;
    const NQ_BYTE  * lmPasswd;
    NQ_UINT16 lmPasswdLen;
    const NQ_BYTE  * ntlmPasswd;
    NQ_UINT16 ntlmPasswdLen;
    const CCNetlogonCredential *credential;
    NQ_BYTE *userSessionKeyOut;
    NQ_BOOL extendedSecurity;
    CMSdAccessToken * accessToken;
    NQ_UINT32 status;
}
ParamsNetrLogonSamLogon;

static NQ_COUNT
composeNetrLogonSamLogon(
    NQ_IOBufPos buffer,
    NQ_COUNT size,
    void* params,
    NQ_BOOL* moreData
    )
{
    CMBufferWriter w;           /* for composing request */
    ParamsNetrLogonSamLogon *p = (ParamsNetrLogonSamLogon *)params;
    NQ_UINT32 ref = 0;          /* ref id */
    NQ_UINT32 sz;               /* rpc string size */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buff:%p size:%d params:%p more:%p", IOBUF_GETBUFPTR(buffer), size, params, moreData);

    cmBufferWriterInit(&w, buffer, size);
    cmBufferWriteUint16(&w, NETRLOGONSAMLOGON_OPCODE);  /* opcode */

    /* server name prefixed by double back slash */
    sz = 3 + (NQ_UINT32)cmWStrlen(p->serverName);
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteAsciiAsUnicodeN(&w, "\\\\", 2, CM_BSF_NOFLAGS);
    cmWStrupr(p->serverName); /* capitalize server name */
    cmBufferWriteUnicode(&w, p->serverName);

    /* own host name */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 2), 4);        /* 4 byte alignment */
    sz = 1 + (NQ_UINT32)cmWStrlen(p->ownHostname);
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmWStrupr(p->ownHostname);                     /* capitalize own host name */
    cmBufferWriteUnicode(&w, p->ownHostname);

    /* Authenticator */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteBytes(&w, p->credential->client, sizeof(p->credential->client));
    cmBufferWriteUint32(&w, p->credential->sequence/*syGetTimeInSec()*/);

    /* Return authenticator */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteZeroes(&w, 8);
    cmBufferWriteUint32(&w, 0/*syGetTimeInSec()*/);

    /* Logon level: NetlogonServiceInformation 2 */
    cmBufferWriteUint16(&w, 2);                    /* logon level */
    cmBufferWriteUint16(&w, 2);                    /* logon level */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* Logon information */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */

    /* NetBIOS name of the domain of the account */
    sz = 2 * (NQ_UINT32)cmWStrlen(p->domainName);
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* parameter control */
    if ((NTLMSSP_LM_NTLM_DATA_LENGTH == p->lmPasswdLen) && (NTLMSSP_LM_NTLM_DATA_LENGTH == p->ntlmPasswdLen) && (FALSE == p->extendedSecurity))
    {
        cmBufferWriteUint32(&w, NETLOGON_PRMCTRL_DEFAULT | NETLOGON_PRMCTRL_CLEARTEXT_PASSWORD_ALLOWED);
    }
    else
    {
        cmBufferWriteUint32(&w, NETLOGON_PRMCTRL_DEFAULT | NETLOGON_PRMCTRL_USE_CLIENT_CHALLENGE);
    }

    /* logon id (Reserved:  MUST be set to zero)*/
    cmBufferWriteUint32(&w, 0);                    /* logon id low */
    cmBufferWriteUint32(&w, 0);                    /* logon id high */

    /* username */
    sz = 2 * (NQ_UINT32)cmWStrlen(p->userName);
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* workstation - NetBIOS name of the workstation from which the user is logging on */
    sz = 2 * (NQ_UINT32)cmWStrlen(p->workstationName);
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* serverChallenge */
    cmBufferWriteBytes(&w, p->serverChallenge, 8);

    /* ntlmPasswd */
    sz = p->ntlmPasswdLen;
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* lmPasswd */
    sz = p->lmPasswdLen;
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* domain */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)cmWStrlen(p->domainName);
    cmBufferWriteUint32(&w, sz);                    /* max count */
    cmBufferWriteUint32(&w, 0);                     /* offset */
    cmBufferWriteUint32(&w, sz);                    /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->domainName);

    /* user name */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)cmWStrlen(p->userName);
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->userName);

    /* workstation - client's workstation name*/
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)cmWStrlen(p->workstationName);
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->workstationName);

    /* ntlmPasswd */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = p->ntlmPasswdLen;
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteBytes(&w, p->ntlmPasswd, (NQ_COUNT)sz);

    /* lmPasswd */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)p->lmPasswdLen;
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteBytes(&w, p->lmPasswd, (NQ_COUNT)sz);

    /* Validation Level: 3 */
    cmBufferWriteUint16(&w, 3);                    /* validation level */

    *moreData = FALSE;
    IOBUF_SKIPBACKBYTE(buffer, 2);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return cmBufferWriterGetDataCount(&w);
}


static NQ_COUNT
composeNetrLogonSamLogonEx(
    NQ_IOBufPos buffer,
    NQ_COUNT size,
    void *params,
    NQ_BOOL *moreData
    )
{
    CMBufferWriter w;           /* for composing request */
    ParamsNetrLogonSamLogon *p = (ParamsNetrLogonSamLogon *)params;
    NQ_UINT32 ref = 0;          /* ref id */
    NQ_UINT32 sz;               /* rpc string size */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "buff:%p size:%d params:%p more:%p", IOBUF_GETBUFPTR(buffer), size, params, moreData);

    cmBufferWriterInit(&w, buffer, size);
    cmBufferWriteUint16(&w, NETRLOGONSAMLOGONEX_OPCODE);    /* opcode */
    cmBufferWriteUint32(&w, ++ref);                         /* ref ID */

    /* logon server name prefixed by double back slash */
    sz = 3 + (NQ_UINT32)cmWStrlen(p->serverName);
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteAsciiAsUnicodeN(&w, "\\\\", 2, CM_BSF_NOFLAGS);
    cmWStrupr(p->serverName); /* capitalize server name */
    cmBufferWriteUnicode(&w, p->serverName);
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */

    /* own host name */
    sz = 1 + (NQ_UINT32)cmWStrlen(p->ownHostname);
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmWStrupr(p->ownHostname);                     /* capitalize own host name */
    cmBufferWriteUnicode(&w, p->ownHostname);

    /* Logon level: NetlogonServiceInformation 2 */
    cmBufferWriteUint16(&w, 2);                    /* logon level */
    cmBufferWriteUint16(&w, 2);                    /* logon level again */

    /* Logon information */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* NetBIOS name of the domain of the account */
    sz = 2 * (NQ_UINT32)cmWStrlen(p->domainName);
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* parameter control */
    if ((NTLMSSP_LM_NTLM_DATA_LENGTH == p->lmPasswdLen) && (NTLMSSP_LM_NTLM_DATA_LENGTH == p->ntlmPasswdLen) && (FALSE == p->extendedSecurity))
    {
        cmBufferWriteUint32(&w, NETLOGON_PRMCTRL_DEFAULT | NETLOGON_PRMCTRL_CLEARTEXT_PASSWORD_ALLOWED);
    }
    else
    {
        cmBufferWriteUint32(&w, NETLOGON_PRMCTRL_DEFAULT | NETLOGON_PRMCTRL_USE_CLIENT_CHALLENGE);
    }

    /* logon id (Reserved:  MUST be set to zero)*/
    cmBufferWriteUint32(&w, 0);                    /* logon id low */
    cmBufferWriteUint32(&w, 0);                    /* logon id high */

    /* username */
    sz = 2 * (NQ_UINT32)cmWStrlen(p->userName);
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* workstation - NetBIOS name of the workstation from which the user is logging on */
    sz = 2 * (NQ_UINT32)cmWStrlen(p->workstationName);
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* serverChallenge */
    cmBufferWriteBytes(&w, p->serverChallenge, 8);

    /* ntlmPasswd */
    sz = p->ntlmPasswdLen;
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* lmPasswd */
    sz = p->lmPasswdLen;
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* max count */
    cmBufferWriteUint16(&w, (NQ_UINT16)sz);        /* actual count */
    cmBufferWriteUint32(&w, ++ref);                /* ref ID */

    /* domain */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)cmWStrlen(p->domainName);
    cmBufferWriteUint32(&w, sz);                    /* max count */
    cmBufferWriteUint32(&w, 0);                     /* offset */
    cmBufferWriteUint32(&w, sz);                    /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->domainName);

    /* user name */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)cmWStrlen(p->userName);
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->userName);

    /* workstation - client's workstation name*/
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)cmWStrlen(p->workstationName);
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteUnicodeNoNull(&w, p->workstationName);

    /* ntlmPasswd */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = p->ntlmPasswdLen;
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteBytes(&w, p->ntlmPasswd, (NQ_COUNT)sz);

    /* lmPasswd */
    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */
    sz = (NQ_UINT32)p->lmPasswdLen;
    cmBufferWriteUint32(&w, sz);                   /* max count */
    cmBufferWriteUint32(&w, 0);                    /* offset */
    cmBufferWriteUint32(&w, sz);                   /* actual count */
    cmBufferWriteBytes(&w, p->lmPasswd, (NQ_COUNT)sz);

    /* Validation Level: 3 */
    cmBufferWriteUint16(&w, 3);                    /* validation level */

    cmBufferWriterAlign(&w, IOBUF_SKIPBYTESEP(buffer, 2, 0), 4);        /* 4 byte alignment */

    /* Extra Flags */
    cmBufferWriteZeroes(&w, 4);                    /* extra flags zero */

    *moreData = FALSE;
    IOBUF_SKIPBACKBYTE(buffer, 2);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "written:%u", cmBufferWriterGetDataCount(&w));
    return cmBufferWriterGetDataCount(&w);
}


#define NETR_SAMLOGON_EXTRA_SIDS_BIT 0x20
#define NETR_SAMLOGON_RESOURCE_GROUP_BIT 0x200

    /* all packet parts in order they appear in the packet */
#define NETR_SAMLOGON_ACCOUNT_NAME      1
#define NETR_SAMLOGON_FULL_NAME         2
#define NETR_SAMLOGON_LOGON_SCRIPT      3
#define NETR_SAMLOGON_PROFILE_PATH      4
#define NETR_SAMLOGON_HOME_DIR          5
#define NETR_SAMLOGON_DIR_DRIVE         6
#define NETR_SAMLOGON_SEPARATE_RIDS     7
#define NETR_SAMLOGON_GROUP_ARRAY       8
#define NETR_SAMLOGON_USER_FLAGS_SES    9
#define NETR_SAMLOGON_SERVER            10
#define NETR_SAMLOGON_DOMAIN            11
#define NETR_SAMLOGON_SID_POINTER       12
#define NETR_SAMLOGON_USER_ACOUNT       13
#define NETR_SAMLOGON_EXTRA_SIDS        14
#define NETR_SAMLOGON_LAST              (NETR_SAMLOGON_EXTRA_SIDS + 1)


static NQ_STATUS
processNetrLogonSamLogon(
    NQ_IOBufPos data,
    NQ_COUNT size,
    void * params,
    NQ_BOOL moreData
    )
{
    CMBufferReader r;
    ParamsNetrLogonSamLogon *p = (ParamsNetrLogonSamLogon *)params;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "data:%p size:%d parameters:%p more:%s", IOBUF_GETBUFPTR(data), size, params, moreData ? "TRUE" : "FALSE");

    cmBufferReaderInit(&r, data, size);
    /* read status at end of packet */
    cmBufferReaderSetOffset(&r, size - (NQ_COUNT)sizeof(NQ_UINT32));
    cmBufferReadUint32(&r, &p->status);

    if (p->status == 0)
    {
        NQ_UINT32 SIDPointerRefID = 0;
        NQ_UINT32 num, packetPart;
        NQ_UINT32 domainSIDOffset = 0;      /* offset to domain SID in data part of packet */
        NQ_UINT32 groupMemberOffset = 0;    /* offset to group membership in data part of packet */
        NQ_UINT16 length;                   /* to read length values */
        NQ_BOOL isExtraSIDExist = FALSE;
        CMSdDomainSid *pDomainSID = (CMSdDomainSid *)&(p->accessToken->domain);
        NQ_COUNT ridsLength = 0;
        CMSdRid userRid = 0;
        CMSdRid groupRid = 0;

        cmBufferReaderSetPosition(&r, cmBufferReaderGetStart(&r));

        /* skip to account name */
        {
            NQ_COUNT skipSize = sizeof (NQ_UINT32) + sizeof(NQ_UINT64) + sizeof(NQ_UINT32) + sizeof(NQ_UINT32) + sizeof(NQ_UINT32) + (6 * sizeof(NQ_UINT64));
            /* Referent ID (uint32) + credentials (uint64) + time stamp(uint32) + validation level (uint32) + referent ID (uint32) + time UTC * 6 */
            cmBufferReaderSkip(&r, skipSize);
        }

        for (packetPart = 0; packetPart < NETR_SAMLOGON_LAST; ++packetPart)
        {
            switch (packetPart)
            {
                case NETR_SAMLOGON_ACCOUNT_NAME:
                case NETR_SAMLOGON_FULL_NAME:
                case NETR_SAMLOGON_LOGON_SCRIPT:
                case NETR_SAMLOGON_PROFILE_PATH:
                case NETR_SAMLOGON_HOME_DIR:
                case NETR_SAMLOGON_DIR_DRIVE:
                case NETR_SAMLOGON_SERVER:
                case NETR_SAMLOGON_DOMAIN:
                    cmBufferReadUint16(&r, &length);
                    if (length > 0)
                    {
                        domainSIDOffset += (NQ_UINT32)(sizeof(NQ_UINT32) + sizeof(NQ_UINT32) + sizeof(NQ_UINT32));
                                                    /* max Count (int32) + offset (int32) + actual count (int32) */
                        length = (NQ_UINT16)((NQ_UINT)length + (length % 4));   /* strings 4 bytes aligned */
                        domainSIDOffset += length;                              /* string length */
                    }
                    cmBufferReaderSkip(&r, (sizeof(NQ_UINT16) + sizeof(NQ_UINT32)));    /* size + referent ID */
                    break;
                case NETR_SAMLOGON_SEPARATE_RIDS:
                    /* skip to user RID, user GROUP RID */
                    cmBufferReaderSkip(&r, (sizeof(NQ_UINT16) + sizeof(NQ_UINT16)));    /* LOGON count + bad PW count */
                    cmBufferReadUint32(&r, &userRid);   /* user RID     */
                    cmBufferReadUint32(&r, &groupRid);  /* group RID    */
                    break;
                case NETR_SAMLOGON_GROUP_ARRAY:
                    /* group membership array */
                    cmBufferReadUint32(&r, &p->accessToken->numRids);   /* NUM Rids */
                    cmBufferReaderSkip(&r, sizeof(NQ_UINT32));          /* referent ID */

                    if (p->accessToken->numRids > 0)
                    {
                        p->accessToken->rids = (CMSdRid *)cmMemoryAllocate((2 + p->accessToken->numRids) * (NQ_UINT)sizeof(CMSdRid)); /* 2 stands for user and group rids */
                        if (NULL == p->accessToken->rids)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "token allocation failed");
                            p->accessToken->numRids = 0;
                            p->status = (NQ_UINT32)NQ_FAIL;
                            goto Exit;
                        }
                        p->accessToken->isAdmin = FALSE;
                        p->accessToken->isAnon  = FALSE;
                        p->accessToken->rids[0] = userRid;
                        p->accessToken->rids[1] = groupRid;

                        groupMemberOffset = domainSIDOffset;
                        ridsLength = (NQ_UINT32)sizeof(NQ_UINT32);
                        ridsLength += (NQ_UINT32)(p->accessToken->numRids * (sizeof(NQ_UINT32) + sizeof(NQ_UINT32)));   /* each group in array has 4 byte for RID and 4 bytes for attributes */
                        domainSIDOffset += (NQ_UINT32)ridsLength;
                    }
                    break;
                case NETR_SAMLOGON_USER_FLAGS_SES:
                    /* read user flags */
                    cmBufferReadUint32(&r, &num);
                    if (num & NETR_SAMLOGON_EXTRA_SIDS_BIT)
                    {
                        isExtraSIDExist = TRUE;
                    }

                    /* user session key */
                    cmBufferReadBytes(&r, p->userSessionKeyOut, 16);
                    break;
                case NETR_SAMLOGON_SID_POINTER:
                    cmBufferReadUint32(&r, &SIDPointerRefID);
                    break;
                case NETR_SAMLOGON_USER_ACOUNT:
                    /* dummy longs */
                    cmBufferReaderSkip(&r, (2 * sizeof(NQ_UINT32)));        /* NQ_UINT32 (dummy) * 2 */
                    cmBufferReadUint32(&r, &num);                           /* user account control */
                    cmBufferReaderSkip(&r, (7 * sizeof(NQ_UINT32)));        /* NQ_UINT32 (dummy) * 7 */
                    break;
                case NETR_SAMLOGON_EXTRA_SIDS:
                    if (isExtraSIDExist)
                    {
                        cmBufferReadUint32(&r, &num);                       /* NUM extra SIDs */
                        cmBufferReaderSkip(&r, sizeof(NQ_UINT32));          /* referent ID */
                    }
                    break;
               default:
                    break;
            } /* switch (packetPart) */
        } /*  for(packetPart = 0; packetPart < NETR_SAMLOGON_LAST; ++packetPart) */

        /* now read extra data in non fixed part of packet */
        /* read RIDs membership */
        {
            NQ_COUNT i, j;

            cmBufferReaderSkip(&r, groupMemberOffset);
            cmBufferReadUint32(&r, (NQ_UINT32 *)&i); /* max count again */
            for (j = 1; i > 0; j++, i--)
            {
                cmBufferReadUint32(&r, &p->accessToken->rids[j]);
                cmBufferReaderSkip(&r, 4); /* attributes */
            }
            p->accessToken->numRids++;

            /* skip to SID pointer */
            cmBufferReaderSkip(&r, domainSIDOffset - groupMemberOffset - ridsLength);
            if (SIDPointerRefID > 0)
            {
                cmBufferReaderSkip(&r, 4);  /* count */

                /* read domain SID */
                cmBufferReadByte(&r, &pDomainSID->revision);
                if (pDomainSID->revision != 1)
                {
                    pDomainSID->revision = 0;
                    goto Exit;
                }

                cmBufferReadByte(&r, &pDomainSID->numAuths);
                cmBufferReadBytes(&r, (NQ_BYTE *)&pDomainSID->idAuth, sizeof(pDomainSID->idAuth));

                for (i = 0; i < pDomainSID->numAuths; ++i)
                {
                    NQ_UINT32 tempUint32;                   /* temporary 32 bits unsigned integer */

                    cmBufferReadUint32(&r, &tempUint32);
                    pDomainSID->subs[i] = tempUint32;
                }
            }
        }
    }

Exit:
    if ((p->status != 0) && (p->accessToken->rids != NULL))
    {
        p->accessToken->isAdmin = FALSE;
        p->accessToken->isAnon  = FALSE;
        cmMemoryFree(p->accessToken->rids);
        p->accessToken->rids = NULL;
        p->accessToken->numRids = 0;

    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", p->status);
    return (NQ_STATUS)p->status;
}

static NQ_STATUS
processNetrLogonSamLogonEx(
    NQ_IOBufPos data,
    NQ_COUNT size,
    void *params,
    NQ_BOOL moreData
    )
{
    CMBufferReader r;
    ParamsNetrLogonSamLogon *p = (ParamsNetrLogonSamLogon *)params;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "data:%p size:%d parameters:%p more:%s", IOBUF_GETBUFPTR(data), size, params, moreData ? "TRUE" : "FALSE");

    cmBufferReaderInit(&r, data, size);
    /* read status at end of packet */
    cmBufferReaderSetOffset(&r, size - (NQ_COUNT)sizeof(NQ_UINT32));
    cmBufferReadUint32(&r, &p->status);

    if (p->status == 0)
    {
        NQ_UINT32 SIDPointerRefID = 0;
        NQ_UINT32 num, packetPart;
        NQ_UINT32 domainSIDOffset = 0;      /* offset to domain SID in data part of packet */
        NQ_UINT32 groupMemberOffset = 0;    /* offset to group membership in data part of packet */
        NQ_UINT16 length;                   /* to read length values */
        NQ_BOOL isExtraSIDExist = FALSE;
        CMSdDomainSid *pDomainSID = (CMSdDomainSid *)&(p->accessToken->domain);
        NQ_COUNT ridsLength = 0;
        CMSdRid userRid = 0;
        CMSdRid groupRid = 0;

        cmBufferReaderSetPosition(&r, cmBufferReaderGetStart(&r));

        /* skip to account name */
        {
            NQ_COUNT skipSize = sizeof(NQ_UINT32) + sizeof(NQ_UINT32) + (6 * sizeof(NQ_UINT64));
                                /* validation level (uint32) + referent ID (uint32) + time UTC * 6 */
            cmBufferReaderSkip(&r, skipSize);
        }

        for (packetPart = 0; packetPart < NETR_SAMLOGON_LAST; ++packetPart)
        {
            switch (packetPart)
            {
                case NETR_SAMLOGON_ACCOUNT_NAME:
                case NETR_SAMLOGON_FULL_NAME:
                case NETR_SAMLOGON_LOGON_SCRIPT:
                case NETR_SAMLOGON_PROFILE_PATH:
                case NETR_SAMLOGON_HOME_DIR:
                case NETR_SAMLOGON_DIR_DRIVE:
                case NETR_SAMLOGON_SERVER:
                case NETR_SAMLOGON_DOMAIN:
                    cmBufferReadUint16(&r, &length);
                    if (length > 0)
                    {
                        domainSIDOffset += (NQ_UINT32)(sizeof(NQ_UINT32) + sizeof(NQ_UINT32) + sizeof(NQ_UINT32));
                                                    /* max Count (int32) + offset (int32) + actual count (int32) */
                        length = (NQ_UINT16)((NQ_UINT)length + (length % 4));   /* strings 4 bytes aligned */
                        domainSIDOffset += length;                              /* string length */
                    }
                    cmBufferReaderSkip(&r, (sizeof(NQ_UINT16) + sizeof(NQ_UINT32)));    /* size + referent ID */
                    break;
                case NETR_SAMLOGON_SEPARATE_RIDS:
                    /* skip to user RID, user GROUP RID */
                    cmBufferReaderSkip(&r, (sizeof(NQ_UINT16) + sizeof(NQ_UINT16)));    /* LOGON count + bad PW count */
                    cmBufferReadUint32(&r, &userRid);   /* user RID     */
                    cmBufferReadUint32(&r, &groupRid);  /* group RID    */
                    break;
                case NETR_SAMLOGON_GROUP_ARRAY:
                    /* group membership array */
                    cmBufferReadUint32(&r, &p->accessToken->numRids);   /* NUM Rids */
                    cmBufferReaderSkip(&r, sizeof(NQ_UINT32));          /* referent ID */

                    if (p->accessToken->numRids > 0)
                    {
                        p->accessToken->rids = (CMSdRid *)cmMemoryAllocate((2 + p->accessToken->numRids) * (NQ_UINT)sizeof(CMSdRid)); /* 2 stands for user and group rids */
                        if (NULL == p->accessToken->rids)
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "token allocation failed");
                            p->accessToken->numRids = 0;
                            p->status = (NQ_UINT32)NQ_FAIL;
                            goto Exit;
                        }
                        p->accessToken->isAdmin = FALSE;
                        p->accessToken->isAnon  = FALSE;
                        p->accessToken->rids[0] = userRid;
                        p->accessToken->rids[1] = groupRid;

                        groupMemberOffset = domainSIDOffset;
                        ridsLength = (NQ_UINT32)sizeof(NQ_UINT32);
                        ridsLength += (NQ_UINT32)(p->accessToken->numRids * (sizeof(NQ_UINT32) + sizeof(NQ_UINT32)));   /* each group in array has 4 byte for RID and 4 bytes for attributes */
                        domainSIDOffset += (NQ_UINT32)ridsLength;
                    }
                    break;
                case NETR_SAMLOGON_USER_FLAGS_SES:
                    /* read user flags */
                    cmBufferReadUint32(&r, &num);
                    if (num & NETR_SAMLOGON_EXTRA_SIDS_BIT)
                    {
                        isExtraSIDExist = TRUE;
                    }

                    /* user session key */
                    cmBufferReadBytes(&r, p->userSessionKeyOut, 16);
                    break;
                case NETR_SAMLOGON_SID_POINTER:
                    cmBufferReadUint32(&r, &SIDPointerRefID);
                    break;
                case NETR_SAMLOGON_USER_ACOUNT:
                    /* dummy longs */
                    cmBufferReaderSkip(&r, (2 * sizeof(NQ_UINT32)));        /* NQ_UINT32 (dummy) * 2 */
                    cmBufferReadUint32(&r, &num);                           /* user account control */
                    cmBufferReaderSkip(&r, (7 * sizeof(NQ_UINT32)));        /* NQ_UINT32 (dummy) * 7 */
                    break;
                case NETR_SAMLOGON_EXTRA_SIDS:
                    if (TRUE == isExtraSIDExist)
                    {
                        cmBufferReadUint32(&r, &num);                       /* NUM extra SIDs */
                        cmBufferReaderSkip(&r, sizeof(NQ_UINT32));          /* referent ID */
                    }
                    break;
                default:
                    break;
            } /* switch (packetPart) */
        } /*  for(packetPart = 0; packetPart < NETR_SAMLOGON_LAST; ++packetPart) */

        /* now read extra data in non fixed part of packet */
        /* read RIDs membership */
        {
            NQ_COUNT i, j;

            cmBufferReaderSkip(&r, groupMemberOffset);
            cmBufferReadUint32(&r, (NQ_UINT32 *)&i); /* max count again */
            for (j = 1; i > 0; j++, i--)
            {
                cmBufferReadUint32(&r, &p->accessToken->rids[j]);
                cmBufferReaderSkip(&r, 4); /* attributes */
            }
            p->accessToken->numRids++;

            /* skip to SID pointer */
            cmBufferReaderSkip(&r, domainSIDOffset - groupMemberOffset - ridsLength);
            if (SIDPointerRefID > 0)
            {
                cmBufferReaderSkip(&r, 4);  /* count */

                /* read domain SID */
                cmBufferReadByte(&r, &pDomainSID->revision);
                if (pDomainSID->revision != 1)
                {
                    pDomainSID->revision = 0;
                    goto Exit;
                }

                cmBufferReadByte(&r, &pDomainSID->numAuths);
                cmBufferReadBytes(&r, (NQ_BYTE *)&pDomainSID->idAuth, sizeof(pDomainSID->idAuth));

                for (i = 0; i < pDomainSID->numAuths; ++i)
                {
                    NQ_UINT32 tempUint32;                   /* temporary 32 bits unsigned integer */

                    cmBufferReadUint32(&r, &tempUint32);
                    pDomainSID->subs[i] = tempUint32;
                }
            }
        }
    }

Exit:
    if ((0 != p->status) && (NULL != p->accessToken->rids))
    {
        p->accessToken->isAdmin = FALSE;
        p->accessToken->isAnon  = FALSE;
        cmMemoryFree(p->accessToken->rids);
        p->accessToken->rids = NULL;
        p->accessToken->numRids = 0;
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%08x", p->status);
    return (NQ_STATUS)p->status;
}

static
NQ_UINT32
ccNetrLogonSamLogonEx(
    NQ_HANDLE netlogon,
    NQ_WCHAR *domain,
    NQ_WCHAR *server,
    NQ_WCHAR *username,
    NQ_WCHAR *workstation,
    NQ_WCHAR *own,
    const NQ_BYTE serverChallenge[8],
    const NQ_BYTE *lmPasswd,
    NQ_UINT16 lmPasswdLen,
    const NQ_BYTE *ntlmPasswd,
    NQ_UINT16 ntlmPasswdLen,
    NQ_BOOL isExtendedSecurity,
    NQ_BYTE userSessionKeyOut[16],
    CMSdAccessToken *accessToken
    )
{
    CCPipe *pPipe = (CCPipe *)netlogon;
    ParamsNetrLogonSamLogon p;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "netlogon:%p domain:%s server:%s user:%s workstation:%s", netlogon, cmWDump(domain), cmWDump(server), cmWDump(username), cmWDump(workstation));

    p.domainName = domain;
    p.serverName = server;
    p.userName = username;
    p.workstationName = workstation;
    p.ownHostname = own;
    p.serverChallenge = serverChallenge;
    p.lmPasswdLen = lmPasswdLen;
    p.lmPasswd = lmPasswd;
    p.ntlmPasswdLen = ntlmPasswdLen;
    p.ntlmPasswd = ntlmPasswd;
    p.userSessionKeyOut = userSessionKeyOut;
    p.extendedSecurity = isExtendedSecurity;
    p.accessToken = accessToken;

    /* call NETLOGON::NetrLogonSamLogonEx */
    if (TRUE == ccDcerpcOverTcpCall(netlogon, composeNetrLogonSamLogonEx, processNetrLogonSamLogonEx, &p))
    {
         /* session key is encrypted for validation level we use */
         if (0 != (pPipe->schannel.creds.negotFlags & NETLOGON_NEG_FLAGS_SUPPORTS_AES))
         {
             NQ_BYTE initVector[16] = {0};

             AES_128_CFB8_Decrypt(pPipe->schannel.creds.sessionKey, initVector, userSessionKeyOut, 16, userSessionKeyOut);
         }
         else
         {
             /* if (pPipe->schannel.creds.negotFlags & NETLOGON_NEG_FLAGS_STRONG_KEYS) */
             /* not supported yet, less secure anyway */
             cmArcfourCrypt(userSessionKeyOut, sizeof(*userSessionKeyOut), pPipe->schannel.creds.sessionKey, sizeof(pPipe->schannel.creds.sessionKey));
         }
    }
    else
    {
        p.status = (0 == p.status) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::NetrLogonSamLogonEx");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%x", p.status);
    return p.status;
}


static
NQ_UINT32
ccNetrLogonSamLogon(
    NQ_HANDLE netlogon,
    NQ_WCHAR * domain,
    NQ_WCHAR * server,
    NQ_WCHAR * username,
    NQ_WCHAR * workstation,
    NQ_WCHAR * own,
    const NQ_BYTE serverChallenge[8],
    const NQ_BYTE * lmPasswd,
    NQ_UINT16 lmPasswdLen,
    const NQ_BYTE * ntlmPasswd,
    NQ_UINT16 ntlmPasswdLen,
    const CCNetlogonCredential * credential,
    NQ_BOOL isExtendedSecurity,
    NQ_BYTE userSessionKey[16],
    CMSdAccessToken * accessToken
    )
{
    ParamsNetrLogonSamLogon p;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "netlogon:%p domain:%s server:%s user:%s workstation:%s", netlogon, cmWDump(domain), cmWDump(server), cmWDump(username), cmWDump(workstation));

    p.domainName = domain;
    p.serverName = server;
    p.userName = username;
    p.workstationName = workstation;
    p.ownHostname = own;
    p.serverChallenge = serverChallenge;
    p.lmPasswdLen = lmPasswdLen;
    p.lmPasswd = lmPasswd;
    p.ntlmPasswdLen = ntlmPasswdLen;
    p.ntlmPasswd = ntlmPasswd;
    p.credential = credential;
    p.userSessionKeyOut = userSessionKey;
    p.extendedSecurity = isExtendedSecurity;
    p.accessToken = accessToken;

    /* call NETLOGON::NetrLogonSamLogon */
    if (0 == ccDcerpcCall(netlogon, composeNetrLogonSamLogon, processNetrLogonSamLogon, &p))
    {
        p.status = (p.status == 0) ? (NQ_UINT32)syGetLastError() : (NQ_UINT32)ccErrorsStatusToNq(p.status, TRUE);
        LOGERR(CM_TRC_LEVEL_ERROR, "NETLOGON::NetrLogonSamLogon");
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:0x%x", p.status);
    return p.status;
}



/*
 *====================================================================
 * PURPOSE: logon to domain
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  user name
 *          IN  client's computer name
 *          IN  own computer name
 *          IN  server challenge
 *          IN  lm password
 *          IN  lm password length
 *          IN  ntlm password
 *          IN  ntlm password length
 *          IN  domain administrator credentials
 *          IN  domain secret
 *          IN  whether extended security is used
 *          OUT user session key
 *          OUT access token
 *          IN admin's credentials
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
static NQ_BOOL netLogon(
    NQ_WCHAR * domain,
    NQ_WCHAR * username,
    NQ_WCHAR * workstation,
    NQ_WCHAR * own,
    const NQ_BYTE serverChallenge[8],
    const NQ_BYTE * lmPasswd,
    NQ_UINT16 lmPasswdLen,
    const NQ_BYTE * ntlmPasswd,
    NQ_UINT16 ntlmPasswdLen,
    const AMCredentials * admin,
    const NQ_BYTE secret[16],
    NQ_BOOL isExtendedSecurity,
    NQ_BYTE userSessionKey[16],
    CMSdAccessToken * accessToken,
    const NQ_WCHAR * dnsList,
    NQ_WCHAR * domainDNS
    )
{
    NQ_WCHAR * dc = NULL;               /* DC name in Unicode */
    NQ_CHAR * dcA = NULL;               /* DC name in ASCII */
    const NQ_CHAR * domainA = NULL;     /* Domain name in ASCII */
    NQ_UINT32 status = NQ_SUCCESS;      /* generic result */
    NQ_BOOL result = FALSE;             /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "lmPasswd:%p lmPasswdLen:%d ntlmPasswd:%p ntlmPasswdLen:%d isExtendedSecurity:%d", lmPasswd, lmPasswdLen, ntlmPasswd, ntlmPasswdLen, isExtendedSecurity);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "domain: %s", cmWDump(domain));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "user: %s", cmWDump(username));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "workstation: %s", cmWDump(workstation));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "own: %s", cmWDump(own));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "domainDNS: %s", cmWDump(domainDNS));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "dnsList: %s", cmWDump(dnsList));

    if (NULL == secret)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "null secret");
        goto Exit;
    }

    /* find domain controller by domain name */
    domainA = cmMemoryCloneWStringAsAscii(domainDNS);
    dcA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_BYTE) * CM_DNS_NAMELEN);
    if ((NULL == domainA) || (NULL == dcA))
    {
        sySetLastError(NQ_ERR_NOMEM);
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    status = (NQ_UINT32)cmGetDCNameByDomain(domainA, dnsList, dcA);
    if (NQ_SUCCESS != status)
    {
        sySetLastError(syGetLastError());
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to get dc for domain %s", domainA);
        goto Exit;
    }

    if ((NULL == cmAStrchr(dcA, '.')) && (syStrlen(dcA) > 15))
    {
        dcA[15] = '\0';
    }

    dc = cmMemoryCloneAString(dcA);
    if (NULL == dc)
    {
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    /* replace default credentials with domain administrator ones */
    ccUserSetAdministratorCredentials(admin);

    /* connect to NETLOGON (as admin) */
    {
        CCNetlogonCredential credentials;
        NQ_HANDLE netLogon;
        AMCredentials compCreds;
        NQ_WCHAR dollar[] = { cmWChar('$'), cmWChar(0)};

        syMutexTake(&staticData->netlogonGuard);

        /* generate client random challenge */
        ccNetlogonCredentialsRandom(credentials.client, sizeof(credentials.client));

        /* connect to netlogon pipe using DOMAIN\COMPUTER$ credentials */
        amCredentialsInitW(&compCreds, domainDNS, own, (const NQ_WCHAR *)secret, AM_CREDENTIALS_PASSWORD_MD4);
        cmWStrcat(compCreds.user, dollar);
        cmWStrupr(compCreds.user);

        netLogon = getNetlogonHandle(dc, domainDNS, &compCreds);
        if (NULL == netLogon)
        {
            syMutexGive(&staticData->netlogonGuard);
            goto Exit;
        }

        /* send client challenge and get server random challenge */
        status = ccNetrServerReqChallenge(netLogon, dc, own, &credentials);
        if (NQ_SUCCESS == status)
        {
            NQ_UINT32 flags = NETLOGON_NEG_FLAGS_AUTH2;

            /* init and generate next netlogon credentials */
            ccNetlogonCredentialsInit(&credentials, secret, flags);

            /* send new client challenge, get server's challenge */
            status = ccNetrServerAuthenticate2(netLogon, dc, own, &credentials, &flags);
            if (NQ_SUCCESS == status)
            {
                ccNetlogonCredentialsNext(&credentials);

                /* send new client challenge */
                status = ccNetrLogonSamLogon(
                                            netLogon,
                                            domain,
                                            dc,
                                            username,
                                            workstation,
                                            own,
                                            serverChallenge,
                                            lmPasswd,
                                            lmPasswdLen,
                                            ntlmPasswd,
                                            ntlmPasswdLen,
                                            &credentials,
                                            isExtendedSecurity,
                                            userSessionKey,
                                            accessToken
                                            );
                if (NQ_SUCCESS == status)
                {
                    LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "userSessionKey", userSessionKey, 16);
                    /* decrypt user session key (RC4) */
                    if (0 != (flags & 0x00000004))
                    {
                        cmArcfourCrypt(userSessionKey, 16, credentials.sessionKey, 16);
                        LOGDUMP(CM_TRC_LEVEL_MESS_NORMAL, "userSessionKey (after rc4)", userSessionKey, 16);
                    }
                }
            }
        }
        syMutexGive(&staticData->netlogonGuard);
    }

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "status = 0x%x", status);

    /* restore previous user credentials */
    if (NULL != admin)
    {
        ccUserSetAdministratorCredentials(NULL);
    }

    sySetLastError((NQ_STATUS)status);
    result = (NQ_SUCCESS == status);

Exit:
    cmMemoryFree(domainA);
    cmMemoryFree(dcA);
    cmMemoryFree(dc);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

static NQ_BOOL netLogonEx(
    NQ_WCHAR *domain,
    NQ_WCHAR *username,
    NQ_WCHAR *workstation,
    NQ_WCHAR *own,
    const NQ_BYTE serverChallenge[8],
    const NQ_BYTE *lmPasswd,
    NQ_UINT16 lmPasswdLen,
    const NQ_BYTE *ntlmPasswd,
    NQ_UINT16 ntlmPasswdLen,
    const AMCredentials *admin,
    const NQ_BYTE secret[16],
    NQ_BOOL isExtendedSecurity,
    NQ_BYTE userSessionKey[16],
    CMSdAccessToken *accessToken,
    const NQ_WCHAR *dnsList,
    NQ_WCHAR *domainDNS
    )
{
    NQ_WCHAR *dc = NULL;                /* DC name in Unicode */
    NQ_CHAR *dcA = NULL;                /* DC name in ASCII */
    const NQ_CHAR *domainA = NULL;      /* Domain name in ASCII */
    NQ_UINT32 status = NQ_SUCCESS;      /* generic result */
    NQ_BOOL result = FALSE;             /* return value */
    NQ_HANDLE netlogonSecured = NULL;   /* secured channel handle */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "lmPasswd:%p lmPasswdLen:%d ntlmPasswd:%p ntlmPasswdLen:%d isExtendedSecurity:%d", lmPasswd, lmPasswdLen, ntlmPasswd, ntlmPasswdLen, isExtendedSecurity);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "domain: %s", cmWDump(domain));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "user: %s", cmWDump(username));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "workstation: %s", cmWDump(workstation));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "own: %s", cmWDump(own));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "domainDNS: %s", cmWDump(domainDNS));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "dnsList: %s", cmWDump(dnsList));

    if (NULL == secret)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "null secret");
        goto Exit;
    }

    /* find domain controller by domain name */
    domainA = cmMemoryCloneWStringAsAscii(domainDNS);
    dcA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_BYTE) * CM_DNS_NAMELEN);
    if ((NULL == domainA) || (NULL == dcA))
    {
        sySetLastError(NQ_ERR_NOMEM);
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }
    status = (NQ_UINT32)cmGetDCNameByDomain(domainA, dnsList, dcA);
    if (NQ_SUCCESS != status)
    {
        sySetLastError(syGetLastError());
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to get dc for domain %s", domainA);
        goto Exit;
    }

    if ((NULL == cmAStrchr(dcA, '.')) && (syStrlen(dcA) > 15))
    {
        dcA[15] = '\0';
    }

    dc = cmMemoryCloneAString(dcA);
    if (NULL == dc)
    {
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    syMutexTake(&staticData->netlogonGuard);

    netlogonSecured = ccDcerpcOverTcpConnect(dc, own, (const AMCredentials *)secret, ccNetlogonGetPipe(), CM_RP_AUTHTYPESCHANNEL);
    if (NULL == netlogonSecured)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to establish secure channel with: %s", cmWDump(dc));
        goto Exit;
    }

    status = ccNetrLogonSamLogonEx(
                                netlogonSecured,
                                domain,
                                dc,
                                username,
                                workstation,
                                own,
                                serverChallenge,
                                lmPasswd,
                                lmPasswdLen,
                                ntlmPasswd,
                                ntlmPasswdLen,
                                isExtendedSecurity,
                                userSessionKey,
                                accessToken
                                );

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "status = 0x%x", status);

    sySetLastError((NQ_STATUS)status);
    result = (status == NQ_SUCCESS);

Exit:
    if (NULL != netlogonSecured)
    {
        ccDcerpcOverTcpDisconnect(netlogonSecured);
    }
    syMutexGive(&staticData->netlogonGuard);
    cmMemoryFree(domainA);
    cmMemoryFree(dcA);
    cmMemoryFree(dc);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 *====================================================================
 * PURPOSE: leave domain
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  computer name
 *          IN  domain administrator credentials
 *
 * RETURNS: TRUE if succeed, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
static
NQ_BOOL
domainLeave(
    const NQ_WCHAR * domain,
    const NQ_WCHAR * computer,
    const AMCredentials *admin
    )
{
    NQ_UINT32 status;               /* generic result */
    NQ_WCHAR * dc = NULL;           /* DC name in Unicode */
    NQ_CHAR * dcA = NULL;           /* DC name in ASCII */
    NQ_CHAR * domainA = NULL;       /* domain name in ASCII */
    CCLsaPolicyInfoDomain info;
    NQ_BOOL result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "domain:%s computer:%s admin:%p", cmWDump(domain), cmWDump(computer), admin);

    if (cmWStrlen(computer) > 15)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        LOGERR(CM_TRC_LEVEL_ERROR, "workstation name exceeds 15 characters");
        goto Exit;
    }

    /* find domain controller by domain name */
    domainA = cmMemoryCloneWStringAsAscii(domain);
    dcA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_BYTE) * CM_DNS_NAMELEN);
    if (NULL == domainA || NULL == dcA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
    if ((status = (NQ_UINT32)cmGetDCNameByDomain(domainA, NULL, dcA)) != NQ_SUCCESS)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to get dc for domain %s", domainA);
        sySetLastError(syGetLastError());
        goto Exit;
    }
    dc = cmMemoryCloneAString(dcA);
    if (NULL == dc)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    /* replace default credentials with domain administrator ones */
    cmWStrupr((NQ_WCHAR *)admin->domain.name); /* capitalize credentials domain name */
    ccUserSetAdministratorCredentials(admin);

    /* get domain SID */
    if ((status = ccLsaPolicyQueryInfoDomain(admin, dc, &info)) == NQ_SUCCESS)
    {
        /* remove computer account in this domain */
        status = removeComputerAccount(dc, &info.sid, computer);
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "status = 0x%x", status);
    }

    /* restore previous user credentials */
    ccUserSetAdministratorCredentials(NULL);

    sySetLastError((NQ_STATUS)status);
    result = (status == NQ_SUCCESS);

Exit:
    cmMemoryFree(domainA);
    cmMemoryFree(dcA);
    cmMemoryFree(dc);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

static NQ_BOOL getDomainSID(const NQ_WCHAR *domain, const NQ_WCHAR *computer, const AMCredentials *admin, NQ_BYTE secret[16], const NQ_WCHAR * dnsList, NQ_WCHAR *domainNB, CMSdDomainSid * domainSid)
{
    NQ_UINT32 status;                       /* generic result */
    const NQ_CHAR * domainA = NULL;         /* domain name in ASCII */
    NQ_CHAR * dcA = NULL;                   /* DC name in ASCII */
    const NQ_WCHAR * dc = NULL;             /* DC name in Unicode */
    CCLsaPolicyInfoDomain info;             /* domain info */
    NQ_BOOL result = FALSE;                 /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "domain:%s computer:%s admin:%p secret:%p dnsList:%p domainNetBios:%p", cmWDump(domain), cmWDump(computer), admin, secret, dnsList, domainNB);

    if (NULL == admin)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Missing credentials");
        sySetLastError(NQ_ERR_ERROR);
        goto Exit;
    }

    /* find domain controller by domain name */
    domainA = cmMemoryCloneWStringAsAscii(domain);
    dcA = (NQ_CHAR *)cmMemoryAllocate(CM_BUFFERLENGTH(NQ_CHAR, CM_DNS_NAMELEN) * sizeof(NQ_CHAR));
    if (NULL == domainA || NULL == dcA)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }
    if ((status = (NQ_UINT32)cmGetDCNameByDomain(domainA, dnsList, dcA)) != NQ_SUCCESS)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "failed to get dc for domain %s", domainA);
        sySetLastError(syGetLastError());
        goto Exit;
    }
    dc = cmMemoryCloneAString(dcA);
    if (NULL == dc)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    /* replace default credentials with domain administrator ones */
    cmWStrupr((NQ_WCHAR *)admin->domain.name); /* capitalize credentials domain name */
    ccUserSetAdministratorCredentials(admin);

    /* get domain SID */
    if ((status = ccLsaPolicyQueryInfoDomain(admin, dc, &info)) == NQ_SUCCESS)
    {
        if (NULL != domainSid)
        {
            *domainSid = info.sid;
        }
        result = TRUE;
    }
Exit:
    cmMemoryFree(dc);
    cmMemoryFree(domainA);
    cmMemoryFree(dcA);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result: %s", result ? "TRUE" : "FALSE");
    return result;
}

/* Active Directory Computer accounts names must consist of:
 * letters (a-z, A-Z) numbers (0-9) and hyphens (-) but no spaces and periods (.), and the name may not consist entirely of digits.
 * We also require the name length of 15 characters.
 */
static NQ_BOOL isValidADComputerAccountName(const NQ_CHAR *name)
{
    NQ_BOOL isValidName = FALSE;
    NQ_COUNT i = 0, length;

    LOGFB(CM_TRC_LEVEL_FUNC_TOOL, "name:%s", name);

#define isAlphaOrHyphenChar(_c)     (((_c) >= 'A' && (_c) <= 'Z') || ((_c) >= 'a' && (_c) <= 'z') || ((_c) == '-'))
#define isNumericChar(_c)           (((_c) >= '0' && (_c) <= '9'))

    if ((NULL == name) || (syStrlen(name) > CM_NB_NAMELEN - CM_NB_POSTFIXLEN))
    {
        goto Exit;
    }

    length = (NQ_COUNT)syStrlen(name);
    for (i = 0; i < (NQ_COUNT)syStrlen(name); i++)
    {
        if (FALSE == isAlphaOrHyphenChar(name[i]))
        {
            if (FALSE == isNumericChar(name[i]))
            {
                goto Exit;
            }
            else
            {
                /* names must not contain only digits */
                --length;
            }
        }
    }

    isValidName = (0 == length) ? FALSE : TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_TOOL, "res:%s", isValidName ? "TRUE" : "FALSE");
    return isValidName;
}

/*
 *====================================================================
 * PURPOSE: join domain persistent
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  computer name
 *          IN  domain administrator credentials
 *          IN  semicolon delimited list of DNS servers, can be NULL
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
ccDomainJoinPersistent(
    const NQ_WCHAR * domainDNS,
    const NQ_WCHAR * computer,
    const AMCredentials * admin,
    const NQ_WCHAR * dns
    )
{
    NQ_WCHAR domainNB[CM_BUFFERLENGTH(NQ_WCHAR, CM_NB_NAMELEN + 1)];
    CMSdDomainSid domainSid;
    NQ_BYTE secret[16];
    NQ_BOOL result = FALSE;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domainDNS:%p computer:%p admin:%p dns:%p", domainDNS, computer, admin, dns);

    if (NULL == domainDNS || NULL == computer || NULL == admin)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (udGetComputerSecretByDomain(secret, domainDNS, domainNB))
    {
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "domainDNS: %s already joined", cmWDump(domainDNS));
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "domainNB: %s", cmWDump(domainNB));
#ifndef UD_CS_INCLUDEMULTITENANCY
        cmNetBiosSetDomainAuth(domainNB, FALSE);
        cmSetFullDomainNameW(domainDNS);
#endif /* ifndef UD_CS_INCLUDEMULTITENANCY */
        getDomainSID(domainDNS, computer, admin, secret, dns, domainNB, &domainSid);
        result = TRUE;
        goto Exit;
    }

    if (domainJoin(domainDNS, computer, admin, secret, dns, domainNB, &domainSid, FALSE))
    {
        udSetComputerSecretByDomain(secret, domainDNS, domainNB);
#ifndef UD_CS_INCLUDEMULTITENANCY
        cmNetBiosSetDomainAuth(domainNB, FALSE);
        cmSetFullDomainNameW(domainDNS);
#endif /* ifndef UD_CS_INCLUDEMULTITENANCY */
        result = TRUE;
    }

Exit:
#ifdef UD_CS_INCLUDEMULTITENANCY
    if (result == TRUE && !csTenantFind(domainDNS))
        csTenantAdd(domainDNS, domainNB, computer, dns, secret, &domainSid);
#endif /* UD_CS_INCLUDEMULTITENANCY */
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}


/*
 *====================================================================
 * PURPOSE: join domain (ASCII version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  computer name
 *          IN  domain administrator credentials
 *          IN/OUT domain secret
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
ccDomainJoinA(
    const NQ_CHAR * domain,
    const NQ_CHAR * computer,
    const AMCredentialsA * admin,
    NQ_BYTE secret[16]
    )
{
    NQ_WCHAR * domainW = NULL;      /* domain name in Unicode */
    NQ_WCHAR * computerW = NULL;    /* computer name in Unicode */
    NQ_WCHAR * domainNB = NULL;     /* domain NetBIOS name in Unicode */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%s admin:%p secret:%p", domain ? domain : "", computer ? computer : "" , admin, &secret);

    if (NULL == domain || NULL == computer || NULL == admin || NULL == secret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    domainW = cmMemoryCloneAString(domain);
    computerW = cmMemoryCloneAString(computer);
    domainNB = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)sizeof(NQ_WCHAR) * (NQ_UINT)(CM_NB_NAMELEN + 1));
    if (NULL != domainW && NULL != computerW && NULL != domainNB)
    {
        AMCredentials adminW;  /* Unicode credentials */

        amCredentialsAsciiiToW(&adminW, admin);
        cmWStrupr(computerW);           /* capitalize computer name */
        result = domainJoin(domainW, computerW, &adminW, secret, NULL, domainNB, NULL, FALSE);
        if ((result == TRUE) && (cmWStrcmp(cmNetBiosGetDomainAuthW(), domainNB) != 0))
        {
            cmNetBiosSetDomainAuth(domainNB, FALSE);
        }
    }
    else
    {
        sySetLastError(NQ_ERR_NOMEM);
    }

Exit:
    cmMemoryFree(domainW);
    cmMemoryFree(computerW);
    cmMemoryFree(domainNB);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

/*
 *====================================================================
 * PURPOSE: join domain (UNICODE version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  computer name
 *          IN  domain administrator credentials
 *          IN/OUT domain secret
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL
ccDomainJoin(
    const NQ_WCHAR * domain,
    const NQ_WCHAR * computer,
    const AMCredentials * admin,
    NQ_BYTE secret[16]
    )
{
    NQ_WCHAR * computerW = NULL;    /* computer name in Unicode */
    NQ_WCHAR * domainNB = NULL;     /* domain NetBIOS name in Unicode */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%p admin:%p secret:%p", cmWDump(domain), computer, admin, &secret);

    if (NULL == domain || NULL == computer || NULL == admin || NULL == secret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    computerW = cmMemoryCloneWString(computer);
    domainNB = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)sizeof(NQ_WCHAR) * (NQ_UINT)(CM_NB_NAMELEN + 1));
    if (NULL != computerW && NULL != domainNB)
    {
        result = domainJoin(domain, computerW, admin, secret, NULL, domainNB, NULL, FALSE);

        if ((result == TRUE) && (cmWStrcmp(cmNetBiosGetDomainAuthW(), domainNB) != 0))
        {
            cmNetBiosSetDomainAuth(domainNB, FALSE);
        }
    }
    else
    {
        sySetLastError(NQ_ERR_NOMEM);
    }

Exit:
    cmMemoryFree(computerW);
    cmMemoryFree(domainNB);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

/*
 *====================================================================
 * PURPOSE: leave domain (ASCII version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  computer name
 *          IN  domain administrator credentials
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL ccDomainLeaveA(
    const NQ_CHAR * domain,
    const NQ_CHAR * computer,
    const AMCredentialsA * admin
    )
{
    NQ_WCHAR * domainW = NULL;      /* domain name in Unicode */
    NQ_WCHAR * computerW = NULL;    /* computer name in Unicode */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%s admin:%p", domain ? domain : "", computer ? computer : "" , admin);

    if (NULL == domain || NULL == computer || NULL == admin)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    domainW = cmMemoryCloneAString(domain);
    computerW = cmMemoryCloneAString(computer);
    if (NULL != domainW && NULL != computerW)
    {
        AMCredentials adminW;  /* Unicode credentials */

        cmWStrupr(computerW); /* capitalize computer name */
        amCredentialsAsciiiToW(&adminW, admin);
        result = domainLeave(domainW, computerW, &adminW);
    }
    else
    {
        sySetLastError(NQ_ERR_NOMEM);
    }

Exit:
    cmMemoryFree(domainW);
    cmMemoryFree(computerW);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

/*
 *====================================================================
 * PURPOSE: leave domain (UNICODE version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  computer name
 *          IN  domain administrator credentials
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL ccDomainLeave(
    const NQ_WCHAR * domain,
    const NQ_WCHAR * computer,
    const AMCredentials  *admin
    )
{
    NQ_WCHAR * computerW = NULL;    /* computer name in Unicode */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%s admin:%p", cmWDump(domain), cmWDump(computer), admin);

    if (NULL == domain || NULL == computer || NULL == admin)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    computerW = cmMemoryCloneWString(computer);
    if (NULL != computerW)
    {
        cmWStrupr(computerW); /* capitalize computer name */
        result = domainLeave(domain, computerW, admin);
    }
    else
    {
        sySetLastError(NQ_ERR_NOMEM);
    }

Exit:
    cmMemoryFree(computerW);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

/*
 *====================================================================
 * PURPOSE: logon to a domain (ASCII version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  domain administrator credentials
 *          IN  computer name
 *          IN  domain secret
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL ccDomainLogonA(
    const NQ_CHAR * domain,
    const NQ_CHAR * computer,
    const AMCredentialsA * admin,
    NQ_BYTE secret[16]
    )
{
    NQ_WCHAR * domainW = NULL;      /* domain name in Unicode */
    NQ_WCHAR * computerW = NULL;    /* computer name in Unicode */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%s admin:%p secret:%p", domain ? domain : "", computer ? computer : "" , admin, &secret);

    if (NULL == domain || NULL == computer || NULL == admin || NULL == secret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    domainW = cmMemoryCloneAString(domain);
    computerW = cmMemoryCloneAString(computer);
    if (NULL != domainW && NULL != computerW)
    {
        AMCredentials adminW;  /* Unicode credentials */

        cmWStrupr(computerW); /* capitalize computer name */
        amCredentialsAsciiiToW(&adminW, admin);
        result = domainLogon(domainW, computerW, &adminW, secret);
    }
    else
    {
        sySetLastError(NQ_ERR_NOMEM);
    }

Exit:
    cmMemoryFree(domainW);
    cmMemoryFree(computerW);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}


/*
 *====================================================================
 * PURPOSE: logon to a domain (UNICODE version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  domain administrator credentials
 *          IN  computer name
 *          IN  domain secret
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL ccDomainLogon(
    const NQ_WCHAR * domain,
    const NQ_WCHAR * computer,
    const AMCredentials * admin,
    NQ_BYTE secret[16]
    )
{
    NQ_WCHAR * computerW = NULL;        /* computer name in Unicode */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%s admin:%p secret:%p", cmWDump(domain), cmWDump(computer), admin, &secret);

    if (NULL == domain || NULL == computer || NULL == admin || NULL == secret)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    computerW = cmMemoryCloneWString(computer);
    if (NULL != computerW)
    {
        cmWStrupr(computerW); /* capitalize computer name */
        result = domainLogon(domain, computerW, admin, secret);
    }
    else
    {
        sySetLastError(NQ_ERR_NOMEM);
    }

Exit:
    cmMemoryFree(computerW);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

/*
 *====================================================================
 * PURPOSE: logon to a domain (ASCII version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  user name
 *          IN  client's computer name
 *          IN  own computer name
 *          IN  server challenge
 *          IN  lm password
 *          IN  lm password length
 *          IN  ntlm password
 *          IN  ntlm password length
 *          IN  domain administrator credentials
 *          IN  domain secret
 *          IN  whether extended security is used
 *          IN  DNS ips list
 *          IN  domain DNS name
 *          OUT user session key
 *          OUT user access token
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL ccNetLogonA(
    const NQ_CHAR * domain,
    const NQ_CHAR * username,
    const NQ_CHAR * workstation,
    const NQ_CHAR * own,
    const NQ_BYTE serverChallenge[8],
    const NQ_BYTE * lmPasswd,
    NQ_UINT16 lmPasswdLen,
    const NQ_BYTE * ntlmPasswd,
    NQ_UINT16 ntlmPasswdLen,
    const AMCredentialsA * admin,
    const NQ_BYTE secret[16],
    NQ_BOOL isExtendedSecurity,
    const NQ_WCHAR *dnsList,
    const NQ_CHAR *domainDNS,
    NQ_BYTE userSessionKey[16],
    CMSdAccessToken * accessToken
    )
{
    NQ_WCHAR * domainDNSW;          /* domain name in Unicode */
    NQ_WCHAR * domainW;             /* domain name in Unicode */
    NQ_WCHAR * userW;               /* user name in Unicode */
    NQ_WCHAR * workstationW;        /* client's computer name in Unicode */
    NQ_WCHAR * ownW;                /* computer name in Unicode */
    NQ_BOOL result;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s user:%s workstation:%s", domainDNS ? domainDNS : "", username ? username : "", workstation ? workstation : "");

    domainW = cmMemoryCloneAString(domain);
    domainDNSW = cmMemoryCloneAString(domainDNS);
    userW = cmMemoryCloneAString(username);
    workstationW = cmMemoryCloneAString(workstation);
    ownW = cmMemoryCloneAString(own);
    if (domainW && userW && workstationW && ownW)
    {
        AMCredentials adminW;  /* Unicode credentials */

        if (NULL != admin)
        {
            amCredentialsAsciiiToW(&adminW, admin);
            cmWStrupr(adminW.domain.name); /* capitalize credentials domain name */
        }
        result = netLogon(
                        domainW,
                        userW,
                        workstationW,
                        ownW,
                        serverChallenge,
                        lmPasswd,
                        lmPasswdLen,
                        ntlmPasswd,
                        ntlmPasswdLen,
                        admin ? &adminW : NULL,
                        secret,
                        isExtendedSecurity,
                        userSessionKey,
                        accessToken,
                        dnsList,
                        domainDNSW);
    }
    else
    {
        sySetLastError(NQ_ERR_NOMEM);
        result = FALSE;
    }

    cmMemoryFree(domainDNSW);
    cmMemoryFree(domainW);
    cmMemoryFree(userW);
    cmMemoryFree(workstationW);
    cmMemoryFree(ownW);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}


/*
 *====================================================================
 * PURPOSE: logon to a domain (UNICODE version)
 *--------------------------------------------------------------------
 * PARAMS:  IN  target domain
 *          IN  user name
 *          IN  client's computer name
 *          IN  own computer name
 *          IN  server challenge
 *          IN  lm password
 *          IN  lm password length
 *          IN  ntlm password
 *          IN  ntlm password length
 *          IN  domain administrator credentials
 *          IN  domain secret
 *          IN  whether extended security is used
 *          IN  DNS ips list
 *          IN  domain DNS name
 *          OUT user session key
 *          OUT user access token
 *
 * RETURNS: TRUE if succeeded, FAIL otherwise
 *
 * NOTES:
 *====================================================================
 */
NQ_BOOL ccNetLogon(
    const NQ_WCHAR *domain,
    const NQ_WCHAR * username,
    const NQ_WCHAR * workstation,
    const NQ_WCHAR * own,
    const NQ_BYTE serverChallenge[8],
    const NQ_BYTE * lmPasswd,
    NQ_UINT16 lmPasswdLen,
    const NQ_BYTE * ntlmPasswd,
    NQ_UINT16 ntlmPasswdLen,
    const AMCredentials * admin,
    const NQ_BYTE secret[16],
    NQ_BOOL isExtendedSecurity,
    const NQ_WCHAR * dnsList,
    const NQ_WCHAR * domainDNS,
    NQ_BYTE userSessionKey[16],
    CMSdAccessToken * accessToken
    )
{
    NQ_BOOL result;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "lmPasswd:%p lmPasswdLen:%d ntlmPasswd:%p ntlmPasswdLen:%d isExtendedSecurity:%d", lmPasswd, lmPasswdLen, ntlmPasswd, ntlmPasswdLen, isExtendedSecurity);

    result = netLogon(
                    (NQ_WCHAR *)domain,
                    (NQ_WCHAR *)username,
                    (NQ_WCHAR *)workstation,
                    (NQ_WCHAR *)own,
                    serverChallenge,
                    lmPasswd,
                    lmPasswdLen,
                    ntlmPasswd,
                    ntlmPasswdLen,
                    admin,
                    secret,
                    isExtendedSecurity,
                    userSessionKey,
                    accessToken,
                    dnsList,
                    (NQ_WCHAR *)domainDNS);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL ccNetLogonEx(
    const NQ_WCHAR *domain,
    const NQ_WCHAR *username,
    const NQ_WCHAR *workstation,
    const NQ_WCHAR *own,
    const NQ_BYTE serverChallenge[8],
    const NQ_BYTE *lmPasswd,
    NQ_UINT16 lmPasswdLen,
    const NQ_BYTE *ntlmPasswd,
    NQ_UINT16 ntlmPasswdLen,
    const AMCredentials *admin,
    const NQ_BYTE secret[16],
    NQ_BOOL isExtendedSecurity,
    const NQ_WCHAR *dnsList,
    const NQ_WCHAR *domainDNS,
    NQ_BYTE userSessionKey[16],
    CMSdAccessToken *accessToken
    )
{
    NQ_BOOL result;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "lmPasswd:%p lmPasswdLen:%d ntlmPasswd:%p ntlmPasswdLen:%d isExtendedSecurity:%d", lmPasswd, lmPasswdLen, ntlmPasswd, ntlmPasswdLen, isExtendedSecurity);

    result = netLogonEx(
                    (NQ_WCHAR *)domain,
                    (NQ_WCHAR *)username,
                    (NQ_WCHAR *)workstation,
                    (NQ_WCHAR *)own,
                    serverChallenge,
                    lmPasswd,
                    lmPasswdLen,
                    ntlmPasswd,
                    ntlmPasswdLen,
                    admin,
                    secret,
                    isExtendedSecurity,
                    userSessionKey,
                    accessToken,
                    dnsList,
                    (NQ_WCHAR *)domainDNS);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

NQ_BOOL ccDomainUpdateSecret(
    const NQ_WCHAR * domain,
    const NQ_WCHAR * computer,
    const AMCredentials * admin,
    NQ_BYTE secret[16]
    )
{
    NQ_WCHAR * computerW = NULL;    /* computer name in Unicode */
    NQ_WCHAR * domainNB = NULL;     /* domain NetBIOS name in Unicode */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%p admin:%p secret:%p", cmWDump(domain), computer, admin, &secret);

    if ((NULL == domain) || (NULL == computer) || (NULL == admin) || (NULL == secret))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    computerW = cmMemoryCloneWString(computer);
    domainNB = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)sizeof(NQ_WCHAR) * (NQ_UINT)(CM_NB_NAMELEN + 1));
    if ((NULL == computerW) || (NULL == domainNB))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "No memory");
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    result = domainJoin(domain, computerW, admin, secret, NULL, domainNB, NULL, TRUE);
    if ((TRUE == result) && (0 != cmWStrcmp(cmNetBiosGetDomainAuthW(), domainNB)))
    {
        cmNetBiosSetDomainAuth(domainNB, FALSE);
    }

Exit:
    cmMemoryFree(computerW);
    cmMemoryFree(domainNB);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccDomainUpdateSecretA(
    const NQ_CHAR * domain,
    const NQ_CHAR * computer,
    const AMCredentialsA * admin,
    NQ_BYTE secret[16]
    )
{
    NQ_WCHAR * domainW = NULL;      /* domain name in Unicode */
    NQ_WCHAR * computerW = NULL;    /* computer name in Unicode */
    NQ_WCHAR * domainNB = NULL;     /* domain NetBIOS name in Unicode */
    AMCredentials adminW;           /* Unicode credentials */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "domain:%s computer:%s admin:%p secret:%p", domain ? domain : "", computer ? computer : "" , admin, &secret);

    if ((NULL == domain) || (NULL == computer) || (NULL == admin) || (NULL == secret))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    domainW = cmMemoryCloneAString(domain);
    computerW = cmMemoryCloneAString(computer);
    domainNB = (NQ_WCHAR *)cmMemoryAllocate((NQ_UINT)sizeof(NQ_WCHAR) * (NQ_UINT)(CM_NB_NAMELEN + 1));
    if ((NULL == domainW) || (NULL == computerW) || (NULL == domainNB))
    {
        sySetLastError(NQ_ERR_NOMEM);
        goto Exit;
    }

    amCredentialsAsciiiToW(&adminW, admin);
    cmWStrupr(computerW);           /* capitalize computer name */
    result = domainJoin(domainW, computerW, &adminW, secret, NULL, domainNB, NULL, TRUE);
    if ((TRUE == result) && (0 != cmWStrcmp(cmNetBiosGetDomainAuthW(), domainNB)))
    {
        cmNetBiosSetDomainAuth(domainNB, FALSE);
    }

Exit:
    cmMemoryFree(domainW);
    cmMemoryFree(computerW);
    cmMemoryFree(domainNB);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

#endif /* UD_CC_INCLUDEDOMAINMEMBERSHIP */
