/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client server operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccserver.h"
#include "ccutils.h"
#include "ccuser.h"
#include "ccmount.h"
#include "ccshare.h"
#include "cctransport.h"
#include "ccdfs.h"
#include "ccapi.h"
#include "amspnego.h"
#include "ccfile.h"
#include "ccsearch.h"
#include "ccdummysmb.h"
#include "ccerrors.h"
#include "cmuuid.h"
#include "ccsmb2common.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

/* -- Static data -- */
static CMList servers;
static NQ_BOOL isModuleInitialized = FALSE;

/* -- local functions -- */

/*
 * Dump object
 */
#ifdef UD_NQ_INCLUDETRACE
/*
 * Print server-specific information
 */
static void dumpOne(CMItem * pItem)
{
    CCServer * pServer = (CCServer *)pItem;
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "  Server:: IP: %s dialect: %s",
            pServer->numIps == 0? "<NONE>" : cmIPDump(&pServer->ips[0]),
            pServer->smb->name);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "  Users: ");
    cmListDump(&pServer->users);
}
#endif /* UD_NQ_INCLUDETRACE */

/*
 * Explicitly dispose and disconnect server:
 *  - disconnects from the server
 *  - disposes private data
 */
static void disposeServer(CCServer * pServer)
{
    CMItem * pMasterUser = pServer->masterUser;
    CMIterator responseIterator;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p", pServer);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "About to dispose server %s", pServer->item.name ? cmWDump(pServer->item.name) : "");

    if (pMasterUser != NULL)
    {
        ccUserLogoff((CCUser *)pMasterUser);
        pServer->masterUser = NULL;
        cmListItemCheck(pMasterUser);
    }
    cmListShutdown(&pServer->users);
    if (pServer->threads.first != NULL)
    {
        CMIterator  itr;
        LOGERR(CM_TRC_LEVEL_ERROR,  "There are items in the CCServer->threads list, Going to remove them" );

        cmListIteratorStart(&pServer->threads,&itr);
        while (cmListIteratorHasNext(&itr))
        {
            CMItem *    pItem;

            pItem = cmListIteratorNext(&itr);
            cmListItemRemove(pItem);
        }
        cmListIteratorTerminate(&itr);
    }
    cmListShutdown(&pServer->threads);
    cmListShutdown(&pServer->async);
    ccServerDisconnect(pServer);
    cmListShutdown(&pServer->expectedResponses);

    cmListIteratorStart(&pServer->waitingNotifyResponses, &responseIterator);
    while(cmListIteratorHasNext(&responseIterator))
    {
        waitingResponse *responsItem;

        responsItem = (waitingResponse *) cmListIteratorNext(&responseIterator);
        if (!IOBUF_ISNULL(responsItem->notifyResponse->buffer))
        {
            cmIOBufManGive(responsItem->notifyResponse->buffer);
        }

        cmBufManGive((NQ_BYTE *)responsItem->notifyResponse);
    }

    cmListIteratorTerminate(&responseIterator);
    cmListShutdown(&pServer->waitingNotifyResponses);
    pServer->numOfNotifyResponses = 0;
    if (NULL != pServer->calledName)
    {
        cmMemoryFree(pServer->calledName);
    }

    pServer->calledName = NULL;
    syMutexDelete(pServer->creditGuard);
    cmMemoryFree(pServer->creditGuard);
    pServer->creditGuard = NULL;
    cmListItemRemoveAndDispose((CMItem *)pServer);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}


static void removeExpectedResponses(CCServer * pServer)
{
    CMIterator      iterator;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p", pServer);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "About to remove all expected responses from server %s", pServer->item.name ? cmWDump(pServer->item.name) : "");

    cmListIteratorStart(&pServer->expectedResponses, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        CMItem  * pItem = (CMItem *)cmListIteratorNext(&iterator);

        if (pItem->locks == 0)
        {
            cmListItemRemoveAndDispose(pItem);
        }
        else
        {
            cmListItemRemove(pItem);
        }
    }
    cmListIteratorTerminate(&iterator);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}
/*
 * Callback for server cleanup on connection break
 */
static void connectionBrokenCallback(void * context)
{
    CCServer *pServer;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "context:%p", context);

    pServer = (CCServer *)context;
    removeExpectedResponses(pServer);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 * Connect server (on either connect or reconnect) and convey negotiations
 */
static NQ_BOOL connectServer(CCServer * pServer, NQ_BOOL extendedSecurity)
{
    NQ_STATUS res;           /* exchange result */
    NQ_BOOL result = FALSE;  /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p extendedSecurity:%s", pServer, extendedSecurity ? "TRUE" : "FALSE");

    pServer->useExtendedSecurity = extendedSecurity;
    if (!ccTransportConnect(&pServer->transport, pServer->ips, (NQ_INT)pServer->numIps, pServer->item.name, connectionBrokenCallback, pServer
#ifdef UD_NQ_USETRANSPORTNETBIOS
    ,pServer->isTemporary
#endif /* UD_NQ_USETRANSPORTNETBIOS */
#ifdef UD_NQ_INCLUDESMBCAPTURE
    ,&pServer->captureHdr
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    ))
    {
        syMutexDelete(pServer->transport.item.guard);
        cmMemoryFree(pServer->transport.item.guard);
        pServer->transport.item.guard = NULL;
        LOGERR(CM_TRC_LEVEL_ERROR, "Transport connect error");
        sySetLastError(NQ_ERR_NOTCONNECTED);
        goto Exit;
    }
    pServer->credits = 0;
    /*
     * During negotiate phase. SMB will be pointing dummySmb so any SMB operation will not proceed until negotiation phase is done.
     * The negoSmb will be used with default SMB. this pointer is used only on negotiate phase.
     * */
    pServer->negoSmb = ccCifsGetDefaultSmb();
    if (NULL == pServer->negoSmb)
    {
        ccTransportDisconnect(&pServer->transport);
        LOGERR(CM_TRC_LEVEL_ERROR, "All dialect are disabled");
        goto Exit;
    }

    pServer->smb = ccSmbDummyGetCifs();
    pServer->isNegotiationValidated = FALSE;
    pServer->transport.server = pServer;
    /* Repeat Negotiate until success. SMB_STATUS_PENDING means iteration - step
     * more negotiations.
     * Example:
     * - default is SMB1,
     * - 1st step - negotiate SMB1  and SMB2 generic (???)
     * - server selects SMB generic (???)
     * - install SMB2.0
     * - 2n step - negotiate SMB2.0 and SMB2.2
     * - server selects SMB2.2
     * - install SMB2.2
     */
    do
    {
        if (NULL != pServer->firstSecurityBlob.data)
        {
            cmMemoryFreeBlob(&pServer->firstSecurityBlob);
        }
        res = pServer->negoSmb->doNegotiate(pServer, &pServer->firstSecurityBlob);
        if (NQ_SUCCESS != res && SMB_STATUS_PENDING != res)
        {
            ccTransportDisconnect(&pServer->transport);
            LOGERR(CM_TRC_LEVEL_ERROR, "Negotiate failed: 0x%08x", res);
            sySetLastError((NQ_UINT32)res);
            goto Exit;
        }
    }
    while (res == SMB_STATUS_PENDING);

    ccTransportDiscardSettingUp(&pServer->transport);

#ifdef UD_NQ_INCLUDETRACE
    pServer->item.dump = dumpOne;
#endif /* UD_NQ_INCLUDETRACE */

    result = TRUE;

Exit:
    if (pServer->smb != pServer->negoSmb)
    {
        /* if negotiation failed, don't want pServer->smb to keep pointing on dummysmb. */
        pServer->smb = pServer->negoSmb;
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

/*
 * Callback for server unlock and disposal:
 *  - disconnects from the server
 *  - disposes private data
 */
static NQ_BOOL unlockCallback(CMItem * pItem)
{
    disposeServer((CCServer *)pItem);
    return TRUE;
}

/* Find existing server by one of it's IPs */
static CCServer * findServerByIp(const NQ_IPADDRESS * ips, NQ_INT numIps, const CCCifsSmb *pDialect)
{
    CMIterator iterator;         /* server iterator */
    CCServer * pServer;          /* pointer to server */
    CCServer * pResult = NULL;   /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "ips:%p numIps:%d dialect:%p", ips, numIps, pDialect);
    cmListIteratorStart(&servers, &iterator);
    while (cmListIteratorHasNext(&iterator))
    {
        NQ_UINT i;
        NQ_INT j;     /* index in IP addresses */

        pServer = (CCServer *)cmListIteratorNext(&iterator);
        for (i = 0; i < pServer->numIps; i++)
        {
            for (j = 0; j < numIps; j++)
            {
                if (CM_IPADDR_EQUAL(ips[j], pServer->ips[i]))
                {
                    cmListIteratorTerminate(&iterator);
                    if (((CMItem *)pServer)->findable)
                    {
                        if ((pDialect == NULL && pServer->isTemporary) || (pDialect != NULL && pDialect != pServer->smb))
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "Invalid pDialect");
                            goto Exit;
                        }
                        cmListItemLock((CMItem *)pServer);
                        pResult = pServer;
                        goto Exit;
                    }
                    else
                    {
                        LOGERR(CM_TRC_LEVEL_ERROR, "Not findable");
                        goto Exit;
                    }
                }
            }
        }
    }
    cmListIteratorTerminate(&iterator);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pResult);
    return pResult;
}

/* Find existing server by name or by container name */
static CCServer * findServerByName(const NQ_WCHAR * name, const CCCifsSmb *pDialect)
{
    CCServer * pServer;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%s dialect:%p", cmWDump(name), pDialect);

    pServer = (CCServer *)cmListItemFind(&servers, name, TRUE, TRUE);
    if (NULL != pServer)
    {
        LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "Server found by name: %s", cmWDump(pServer->item.name));
        if (((NULL == pDialect) && (TRUE == pServer->isTemporary)) || ((NULL != pDialect) && (pDialect != pServer->smb)))
        {
            LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "Must not use temporary server");
            cmListItemUnlock((CMItem *)pServer);
            pServer = NULL;
            goto Exit;
        }
    }
#ifdef UD_CC_SUPPORT_CONTAINER_NAME
    else /* server not found, search by container name */
    {
        CMIterator  serverItr;

        LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "can't find by name, search by container name");
        cmListIteratorStart(&servers , &serverItr);
        while (cmListIteratorHasNext(&serverItr))
        {
            CCServer * currentServer;

            currentServer = (CCServer *)cmListIteratorNext(&serverItr);
            if ((NULL != currentServer->containerName) && (0 == cmWStrcmp(currentServer->containerName, name)))
            {
                if (((NULL == pDialect) && (TRUE == currentServer->isTemporary)) || ((NULL != pDialect) && (pDialect != currentServer->smb)))
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "Must not use temporary server, name: %s", cmWDump(currentServer->item.name));
                    cmListItemUnlock((CMItem *)currentServer);
                }
                else
                {
                    LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "Server found by container name: %s", cmWDump(currentServer->containerName));
                    pServer = currentServer;
                    cmListIteratorTerminate(&serverItr);
                    goto Exit;
                }
            }
        }

        cmListIteratorTerminate(&serverItr);
        LOGMSG(CM_TRC_LEVEL_FUNC_COMMON, "Could not find server");
    }
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "pServer:%p, server name: %s", pServer, pServer ? cmWDump(pServer->item.name) : "None");
    return pServer;
}

/* Create and initialize new server */
static CCServer * createNewServer(const NQ_WCHAR * host,
#ifdef UD_CC_SUPPORT_CONTAINER_NAME
        const NQ_WCHAR * containerName,
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */
        NQ_BOOL extendedSecurity, const NQ_IPADDRESS *ips, NQ_INT numIps)
{
    CCServer * pServer;
    CCServer * pResult = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "host:%s extendedSecurity:%s ips:%p numIps:%d", cmWDump(host), extendedSecurity ? "TRUE" : "FALSE", ips, numIps);

    pServer = (CCServer *)cmListItemCreateAndAdd(&servers, sizeof(CCServer), host, unlockCallback, CM_LISTITEM_LOCK , FALSE);
    if (NULL == pServer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Error;
    }

    /* initialize server */
    cmListStart(&pServer->users);
    cmListStart(&pServer->threads);
    cmListStart(&pServer->async);
    cmListStart(&pServer->expectedResponses);
    cmListStart(&pServer->waitingNotifyResponses);
    pServer->numOfNotifyResponses = 0;
    ccTransportInit(&pServer->transport);
    pServer->ips = ips;
    pServer->numIps = (NQ_COUNT)numIps;
    pServer->smbContext = NULL;
    pServer->smb = NULL;
    pServer->negoSmb = NULL;
    pServer->firstSecurityBlob.data = NULL;
    pServer->useSigning = FALSE;
    pServer->calledName = cmMemoryCloneWString(host);
#ifdef UD_CC_SUPPORT_CONTAINER_NAME
    pServer->containerName = containerName;
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */
    pServer->masterUser = NULL;
    pServer->useName = TRUE;
    pServer->isReconnecting = FALSE;
    pServer->userSecurity = TRUE;
    pServer->connectionBroke = FALSE;
    if (NULL == pServer->calledName)
    {
        cmListItemUnlock((CMItem *)pServer);
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Error;
    }
    cmZeroUuid(&pServer->serverGUID);
#ifdef UD_NQ_INCLUDESMB2
    pServer->clientGuidPartial = (NQ_UINT32)syGetTimeInSec();
#endif
    pServer->vcNumber = 1;
    pServer->isTemporary = FALSE;
    pServer->useAscii = FALSE;
    pServer->negoAscii = FALSE;
    pServer->isAesGcm = FALSE;
    pServer->capabilities = 0;
#if defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311)
    pServer->serverDialectRevision = CCCIFS_ILLEGALSMBREVISION;
#endif /* defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311) */
#ifdef UD_NQ_INCLUDESMB311
    pServer->isPreauthIntegOn = FALSE;
#endif /* UD_NQ_INCLUDESMB311 */
#ifdef UD_NQ_INCLUDESMBCAPTURE
    syMemset(&pServer->captureHdr , 0 , sizeof(CMCaptureHeader));
#endif /* UD_NQ_INCLUDESMBCAPTURE */
    pServer->creditGuard = (SYMutex *)cmMemoryAllocate(sizeof(*pServer->creditGuard));
    syMutexCreate(pServer->creditGuard);

    pResult = pServer;
    goto Exit;

Error:
    syMutexGive(&servers.guard);
    sySetLastError(NQ_ERR_OUTOFMEMORY);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pResult);
    return pResult;
}

static NQ_BOOL areUsersLoggedOn(CCServer * pServer)
{
    CMIterator  userItr;
    NQ_BOOL     result = FALSE;

    cmListIteratorStart(&pServer->users , &userItr);
    while (cmListIteratorHasNext(&userItr))
    {
        CCUser  *   pUser = NULL;

        pUser = (CCUser *)cmListIteratorNext(&userItr);
        if (pUser->isLogginOff == FALSE)
        {
            result = TRUE;
            goto Exit;
        }
    }

Exit:
    cmListIteratorTerminate(&userItr);
    return !result;
}

#ifdef UD_CC_SUPPORT_CONTAINER_NAME
/*====================================================================
 * PURPOSE: this function check if host name is container's name, if yes
 *          it returns one of its server name and its resolved ips
 *--------------------------------------------------------------------
 * PARAMS:  IN host name, suspected as container name
 *          IN resolved ips of host
 *          IN number of resolved ip
 *          OUT name of some server that contained in the container
 *          OUT resolved ips of the above server
 *          OUT number of resolved ip
 *
 * RETURNS: None
 *
 * NOTES: None
 *====================================================================
 */
void getServerNameFromContainer(const NQ_WCHAR * host,const NQ_IPADDRESS * ips, NQ_INT numIps, NQ_WCHAR ** serverName, NQ_IPADDRESS ** serverIps, NQ_INT * numberIps)
{
    NQ_WCHAR * hostUpper;
    NQ_INT i;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "host:%s ips:%p numIps:%d, serverName:%p, serverIps:%p, numberIps:%p", cmWDump(host), ips, numIps, serverName, serverIps, numberIps);

    *serverName = NULL;
    *serverIps = NULL;
    *numberIps = 0;
    hostUpper = cmMemoryCloneWString(host);
    if (NULL == hostUpper)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    cmWStrupr(hostUpper);
    cmResolverRemoveFromCache(host);
    for (i = 0; i < numIps; i++)
    {
        *serverName = (NQ_WCHAR *)cmResolverGetHostName(&ips[i]);
        if (NULL != *serverName)
        {
            NQ_WCHAR * serverNameUpper;

            LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "Resolved name: %s", cmWDump(*serverName));
            serverNameUpper = cmMemoryCloneWString(*serverName);
            if (NULL == serverNameUpper)
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                sySetLastError(NQ_ERR_OUTOFMEMORY);
                goto Exit;
            }

            cmWStrupr(serverNameUpper);
            if (0 == cmWStrcmp(serverNameUpper, hostUpper))
            {
                /* same name, it means host is not container */
                LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "Names are the same, thus %s is not container name", cmWDump(host));
                cmMemoryFree(serverNameUpper);
                cmMemoryFree(*serverName);
                *serverName = NULL;
                goto Exit;
            }

            cmMemoryFree(serverNameUpper);
            cmResolverRemoveFromCache(*serverName);
            *serverIps = (NQ_IPADDRESS *)cmResolverGetHostIps(NULL, *serverName, numberIps);
            if (NULL == *serverIps)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "Could not resolve ip's of %s", *serverName);
                cmMemoryFree(*serverName);
                *serverName = NULL;
            }
            else
            {
                int j, k;

                if (*numberIps != numIps)
                {
                    /* number of ips not equal, it means host is container */
                    LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "Number of ips not equal, thus %s is container name", cmWDump(host));
                    goto Exit;
                }
                else
                {
                    for (j = 0; j < numIps; j++)
                    {
                        NQ_BOOL isMatchFound = FALSE;

                        for (k = 0; k < numIps; k++)
                        {
                            if (TRUE == CM_IPADDR_EQUAL(ips[j], (*serverIps)[k]))
                            {
                                isMatchFound = TRUE;
                                break;
                            }
                        }

                        if (FALSE == isMatchFound)
                        {
                            /* one of the ips not equal, it means host is container */
                            LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "%s does not exist in the resolved ips list, thus %s is container name", cmIPDump(&ips[j]), cmWDump(host));
                            goto Exit;
                        }
                        else
                        {
                            isMatchFound = FALSE;
                        }
                    }

                    LOGMSG(CM_TRC_LEVEL_MESS_COMMON, "ips list are the same, thus %s is not container name", cmWDump(host));
                    cmMemoryFree(*serverName);
                    *serverName = NULL;
                    cmMemoryFree(*serverIps);
                    *serverIps = NULL;
                    goto Exit;
                }
            }
        }
    }

Exit:
    if (NULL != hostUpper)
    {
        cmMemoryFree(hostUpper);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "serverName:%s, serverIps:%p, numberIps:%d", cmWDump(*serverName), *serverIps, *numberIps);
    return;
}
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */

/* -- API Functions */

NQ_BOOL ccServerStart(void)
{
   LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

   if (isModuleInitialized)
   {
      goto Exit;
   }

    cmListStart(&servers);
#ifdef UD_NQ_INCLUDETRACE
    servers.name = "servers";
#endif /* UD_NQ_INCLUDETRACE */

   isModuleInitialized = TRUE;

Exit:

   LOGFE(CM_TRC_LEVEL_FUNC_COMMON);

   return TRUE;
}

void ccServerShutdown(void)
{
    CMIterator  serverItr;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if( isModuleInitialized )
    {
      cmListIteratorStart(&servers, &serverItr);
      while (cmListIteratorHasNext(&serverItr))
      {
         CMIterator  userItr;
         CCServer    *   pServer;

         pServer = (CCServer *)cmListIteratorNext(&serverItr);

         cmListItemLock((CMItem *)pServer);
         cmListIteratorStart(&pServer->users, &userItr);
         while (cmListIteratorHasNext(&userItr))
         {
               CMIterator  shareItr;
               CCUser  *   pUser;

               pUser = (CCUser *)cmListIteratorNext(&userItr);

               /* remove files first */
               cmListItemLock((CMItem *)pUser);
               cmListIteratorStart(&pUser->shares, &shareItr);
               while (cmListIteratorHasNext(&shareItr))
               {
                  CCShare *   pShare;
                  CMIterator  filesItr;
                  CMIterator  searchItr;

               pShare = (CCShare *)cmListIteratorNext(&shareItr);

                  cmListIteratorStart(&pShare->files, &filesItr);
                  while (cmListIteratorHasNext(&filesItr))
                  {
                     CMItem * pItem;

                     pItem = cmListIteratorNext(&filesItr);
                     cmListItemUnlock(pItem);
                  }
                  cmListIteratorTerminate(&filesItr);

                  cmListIteratorStart(&pShare->searches, &searchItr);
               while (cmListIteratorHasNext(&searchItr))
               {
                  CMItem *  pSearch;
                  NQ_COUNT  lockCntr = 0 , numOfLocks;

                  pSearch = cmListIteratorNext(&searchItr);
                  numOfLocks = pSearch->locks;
                  for (lockCntr = 0 ; lockCntr < numOfLocks; lockCntr++)
                  {
                     cmListItemUnlock(pSearch);
                  }
               }
               cmListIteratorTerminate(&searchItr);
                  /*  share can be already released at this point */
               }
               cmListIteratorTerminate(&shareItr);

               /* unlock shares */
               cmListIteratorStart(&pUser->shares, &shareItr);
               while (cmListIteratorHasNext(&shareItr))
               {
                  CCShare *   pShare;
                  NQ_COUNT  lockCntr = 0 , numOfLocks;

               pShare = (CCShare *)cmListIteratorNext(&shareItr);
               numOfLocks = ((CMItem *)pShare)->locks;
               for (lockCntr = 0 ; lockCntr < numOfLocks; lockCntr++)
               {
                  cmListItemUnlock((CMItem *)pShare);
               }
               }
               cmListIteratorTerminate(&shareItr);
               cmListItemUnlock((CMItem *)pUser);
         }
         cmListIteratorTerminate(&userItr);
         cmListItemUnlock((CMItem *)pServer);
      }/*while (cmListIteratorHasNext(&userItr))*/
      cmListIteratorTerminate(&serverItr);
      cmListShutdown(&servers);

      isModuleInitialized = FALSE;
    }/*if( isModuleInitialized )*/

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void ccServerDisconnectAll(void)
{
    cmListRemoveAndDisposeAll(&servers);
}

NQ_BOOL ccServerUseSignatures(CCServer * pServer)
{
    return pServer->useSigning && (pServer->capabilities & CC_CAP_MESSAGESIGNING);
}

void ccServerCheckTimeouts(void)
{
    CMIterator serverIterator;      /* to enumerate servers */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    cmListIteratorStart(&servers, &serverIterator);
    while (cmListIteratorHasNext(&serverIterator))
    {
        CCServer * pServer;         /* next server pointer */

        pServer = (CCServer *)cmListIteratorNext(&serverIterator);
        if (ccTransportIsTimeoutExpired(&pServer->transport))
        {
            CMIterator userIterator;        /* to enumerate users */

            cmListIteratorStart(&pServer->users, &userIterator);
            while (cmListIteratorHasNext(&userIterator))
            {
                CCUser * pUser;             /* next user pointer */
                CMIterator shareIterator;   /* to enumerate users */

                pUser = (CCUser *)cmListIteratorNext(&userIterator);

                cmListIteratorStart(&pUser->shares, &shareIterator);
                while (cmListIteratorHasNext(&shareIterator))
                {
                    CCShare * pShare;           /* next share pointer */
                    CMIterator fileIterator;   /* to enumerate users */

                    pShare = (CCShare *)cmListIteratorNext(&shareIterator);

                    cmListIteratorStart(&pShare->files, &fileIterator);
                    if (cmListIteratorHasNext(&fileIterator))
                    {
                        CCFile  *   pFile;

                        pFile = (CCFile *)cmListIteratorNext(&fileIterator);
                        cmListItemUnlock((CMItem *)pFile);
                    }
                    cmListIteratorTerminate(&fileIterator);
                    ccMountIterateMounts(&fileIterator);
                    while (cmListIteratorHasNext(&fileIterator))
                    {
                        CCMount *   pMount;
                        pMount = (CCMount *)cmListIteratorNext(&fileIterator);

                        if (pMount->share == pShare)
                        {
                            cmListItemUnlock((CMItem *)pMount);
                        }
                    }
                    cmListIteratorTerminate(&fileIterator);
                    cmListItemCheck((CMItem *)pShare);    /* will be removed if not locked */
                }
                cmListIteratorTerminate(&shareIterator);
                cmListItemCheck((CMItem *)pUser);    /* will be removed if not locked */
            }
            cmListIteratorTerminate(&userIterator);
            cmListItemCheck((CMItem *)pServer);    /* will be removed if not locked */
        }
    }
    cmListIteratorTerminate(&serverIterator);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

/*
 * Disconnect server (on shutdown or reconnect)
 */
void ccServerDisconnect(CCServer * pServer)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p", pServer);

    syMutexTake(&servers.guard);
    pServer->transport.connected = TRUE;
    if (NULL != pServer->transport.callback)
    {
        ccTransportDisconnect(&pServer->transport);
    }
    if (NULL!= pServer->smb && NULL != pServer->smbContext)
    {
        pServer->smb->freeContext(pServer->smbContext, pServer);
    }
    cmMemoryFreeBlob(&pServer->firstSecurityBlob);
    if (NULL != pServer->ips)
    {
        cmMemoryFree(pServer->ips);
        pServer->ips = NULL;
        pServer->numIps = 0;
    }
    syMutexGive(&servers.guard);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

CCServer * ccServerFindOrCreate(const NQ_WCHAR * name, NQ_BOOL extendedSecurity, const CCCifsSmb *pDialect)
{
    CCServer * pServer = NULL;       /* server pointer */
    NQ_IPADDRESS ip;                 /* server name converted to IP */
    const NQ_IPADDRESS * ips = NULL; /* array of all server IPs */
    NQ_INT numIps;                   /* number of resolved IPs */
    const NQ_WCHAR * host = NULL;    /* host name */
    NQ_BOOL nameIsIp;                /* to distinguish between a name and an IP */
    const NQ_WCHAR * pdcName = NULL; /* resolved over DFS */
#ifdef UD_CC_SUPPORT_CONTAINER_NAME
    const NQ_WCHAR * containerName = NULL; /* container name */
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */
    NQ_BOOL result;                  /* connect result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "name:%s extendedSecurity:%s dialect:%s", cmWDump(name), extendedSecurity ? "TRUE" : "FALSE", pDialect ? pDialect->name : "null");

    if (NULL == name)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input - name");
        goto Exit;
    }

    /* check server timeouts and remove those whose timeout has expired */
    ccServerCheckTimeouts();

    syMutexTake(&servers.guard);

    /* find existing server */
    nameIsIp = ccUtilsNameToIp(name, &ip);
    if (nameIsIp)
    {
        /* find by IP */
        pServer = findServerByIp(&ip, 1, pDialect);
    }
    else
    {
        /* find by name */
        pServer = findServerByName(name, pDialect);
    }

    if (NULL != pServer)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Found existing server by %s %s", nameIsIp ?  " IP" : " name", pDialect ? "(with required dialect)" : "");
        if (!ccTransportIsConnected(&pServer->transport) && !ccServerReconnect(pServer))
        {
            cmListItemUnlock((CMItem *)pServer);
            pServer = NULL;
        }

        goto Exit;
    }

    /* try domain DFS resolution (when name is actually domain's name) */
    if ((FALSE == nameIsIp) && (NULL == pDialect))
    {
        pdcName = ccDfsResolveHost(name);
        if (NULL != pdcName)
        {
            /* find in existing */
            pServer = findServerByName(pdcName, NULL);
            if (NULL != pServer)
            {
                LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Found existing");
                goto Exit;
            }
        }
    }

    /* resolve host name if supplied name is IP */
    if (TRUE == nameIsIp)
    {
        host = cmResolverGetHostName(&ip);
    }
    else
    {
        host = pdcName ? cmMemoryCloneWString(pdcName) : cmMemoryCloneWString(name);
        if (NULL == host)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            pServer = NULL;
            goto Exit;
        }
    }

    if (NULL == host) /* name is IP, but IP was not resolved to name */
    {
        host = cmMemoryCloneWString(name);
        ips = (NQ_IPADDRESS *)cmMemoryAllocate(sizeof(NQ_IPADDRESS));
        if (NULL == ips)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            pServer = NULL;
            goto Exit;
        }

        syMemcpy(ips, &ip, sizeof(NQ_IPADDRESS));
        numIps = 1;
    }
    else
    {
        /* resolve all host IPs */
        ips = cmResolverGetHostIps(NULL, host, &numIps);
        if (NULL == ips)
        {
            if (nameIsIp)
            {
                ips = (NQ_IPADDRESS *)cmMemoryAllocate(sizeof(NQ_IPADDRESS));
                if (NULL == ips)
                {
                    LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                    sySetLastError(NQ_ERR_OUTOFMEMORY);
                    pServer = NULL;
                    goto Exit;
                }

                syMemcpy(ips, &ip, sizeof(NQ_IPADDRESS));
                numIps = 1;
            }
            else
            {
                LOGERR(CM_TRC_LEVEL_ERROR, "Cannot resolve IPs for %s", cmWDump(host));
                sySetLastError(NQ_ERR_INVNETNAME);
                goto Exit;
            }
        }
#ifdef UD_CC_SUPPORT_CONTAINER_NAME
        else
        {
            /* can not be container name in the following cases: the name was resolved from domain name to dc name, it has single IP, the supplied name was IP */
            if ((NULL == pdcName) && (1 < numIps) && (FALSE == nameIsIp) && (NULL == pDialect))
            {
                NQ_WCHAR      *tempServerName = NULL;
                NQ_IPADDRESS  *tempIps = NULL;
                NQ_INT        tempNumIps;

                getServerNameFromContainer(host, ips, numIps, &tempServerName, &tempIps, &tempNumIps);
                if (NULL != tempServerName)
                {
                    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Host name is container, using new host name: %s", cmWDump(tempServerName));
                    containerName = host;
                    host = tempServerName;
                    numIps = tempNumIps;
                    cmMemoryFree(ips);
                    ips = tempIps;
                }
            }
        }
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */
    }

    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Resolved host: %s", cmWDump(host));
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Resolved %d IPs %s for this host", numIps, numIps > 0 ? cmIPDump(&ips[0]) : "");

    /* reuse existing servers, search by IP */
    pServer = findServerByIp(ips, numIps, pDialect);
    if (NULL != pServer)
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Found existing server by IP: %s with IP (1st one): %s %s", cmWDump(pServer->item.name), cmIPDump(&pServer->ips[0]), pDialect ? "(with required dialect)" : "");
        cmMemoryFree(ips);
        if (!ccTransportIsConnected(&pServer->transport) && !ccServerReconnect(pServer))
        {
            cmListItemUnlock((CMItem *)pServer);
            pServer = NULL;
        }
        goto Exit;
    }

    /* create new server */
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "Creating new server %s", pDialect ? "(with required dialect)" : "");
    pServer = createNewServer(host,
#ifdef UD_CC_SUPPORT_CONTAINER_NAME
            containerName,
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */
            extendedSecurity, ips, numIps);

    if (NULL == pServer)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to create a server object");
        cmMemoryFree(ips);
        goto Exit;
    }
    if (nameIsIp)
    {
        pServer->useName = FALSE;
    }
    if (pDialect)
    {
        pServer->isTemporary = TRUE;
    }

    /* connect to server (forcing required dialect) */
    if (pDialect)   (pDialect->setSolo)(TRUE);
    result = connectServer(pServer, extendedSecurity);
    if (pDialect)   (pDialect->setSolo)(FALSE);
    if (result == FALSE)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to server");
        cmListItemUnlock((CMItem *)pServer);
        pServer = NULL;
        goto Exit;
    }

Exit:
    /* ips is stored into pServer->ips inside the createNewServer(), so do not free ips unless createNewServer() returns NULL */
    cmMemoryFree(pdcName);
    cmMemoryFree(host);
#ifdef UD_CC_SUPPORT_CONTAINER_NAME
   if (NULL != containerName)
   {
       cmMemoryFree(containerName);
   }
#endif /* UD_CC_SUPPORT_CONTAINER_NAME */
    syMutexGive(&servers.guard);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pServer);
    return pServer;
}

NQ_BOOL ccServerConnect(CCServer * pServer, const NQ_WCHAR * name)
{
    NQ_IPADDRESS ip;            /* server name converted to IP */
    const NQ_IPADDRESS * ips;   /* array of all server IPs */
    NQ_INT numIps;              /* number of resolved IPs */
    const NQ_WCHAR * host;      /* host name */
    NQ_BOOL result = FALSE;     /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p name:%s", pServer, cmWDump(name));
    /*LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "server: %s", cmWDump(name));*/

    cmListItemTake((CMItem *)pServer);
    if (ccUtilsNameToIp(name, &ip))
    {
        host = cmResolverGetHostName(&ip);
    }
    else
    {
        host = ccDfsResolveHost(name);  /* either PDC or  a copy of itself */
    }
    if (NULL == host)
    {
        sySetLastError(NQ_ERR_BADPATH);
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot resolve server");
        goto Exit;
    }

    /* now resolve all server IPs regardless of how it was specified: by name or by a single ip */
    ips = cmResolverGetHostIps(NULL, host, &numIps);
    if (NULL == ips)
    {
        cmMemoryFree(host);
        sySetLastError(NQ_ERR_BADPATH);
        LOGERR(CM_TRC_LEVEL_ERROR, "Cannot resolve IPs for %s", cmWDump(name));
        goto Exit;
    }

    /* initialize server */
    cmListStart(&pServer->users);
    pServer->ips = ips;
    pServer->numIps = (NQ_COUNT)numIps;
    pServer->calledName = NULL;
    pServer->item.name = (NQ_WCHAR *)host;
    /* connect to server */
    if (!connectServer(pServer, TRUE))
    {
        cmMemoryFree(host);
        cmMemoryFree(pServer->ips);
        goto Exit;
    }
    result = TRUE;

Exit:
    cmListItemGive((CMItem *)pServer);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

static void reconnnectThreadBody(void)
{
    sySleep(2);

    ccErrorsModifyMapping(FALSE);
}

NQ_BOOL ccServerReconnect(CCServer * pServer)
{
    CMIterator userIterator;        /* to enumerate users */
    NQ_BOOL result = FALSE;         /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server:%p", pServer);

    cmListItemTake((CMItem *)pServer);
    if (pServer->isReconnecting)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Server is already reconnecting");
        goto Exit;
    }

    if (ccTransportIsConnected(&pServer->transport))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "False alarm - the server is still connected");
        sySetLastError(NQ_ERR_TIMEOUT);
        result = TRUE;
        goto Exit;
    }

    if (areUsersLoggedOn(pServer))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "False alarm - All users are logged off");
        result = TRUE;
        goto Exit;
    }

    ccErrorsModifyMapping(TRUE);
    pServer->isReconnecting = TRUE;
    pServer->connectionBroke = FALSE;
/*  cmListItemGive((CMItem *)pServer); */ /* To free other threads that should exit */
/*  cmListItemTake((CMItem *)pServer); */
    pServer->smb->signalAllMatch(&pServer->transport);
    pServer->transport.connected = TRUE;
    ccTransportDisconnect(&pServer->transport);
    pServer->smb->freeContext(pServer->smbContext, pServer);
    cmMemoryFreeBlob(&pServer->firstSecurityBlob);

    cmListIteratorStart(&pServer->users, &userIterator);
    while (cmListIteratorHasNext(&userIterator))
    {
        CCUser * pUser;                 /* next user pointer */
        CMIterator shareIterator;       /* to enumerate shares */

        pUser = (CCUser *)cmListIteratorNext(&userIterator);
        pUser->logged = FALSE;
        cmU64Zero(&pUser->uid);
        amSpnegoFreeKey(&pUser->sessionKey);
        amSpnegoFreeKey(&pUser->macSessionKey);
#ifdef UD_NQ_INCLUDESMB3
        cmMemoryFreeBlob(&pUser->encryptionKey);
        cmMemoryFreeBlob(&pUser->decryptionKey);
        cmMemoryFreeBlob(&pUser->applicationKey);
#endif /* UD_NQ_INCLUDESMB3 */
        cmListIteratorStart(&pUser->shares, &shareIterator);
        while (cmListIteratorHasNext(&shareIterator))
        {
            CCShare * pShare;               /* next share pointer */

            pShare = (CCShare *)cmListIteratorNext(&shareIterator);
            pShare->connected = FALSE;
        }

        cmListIteratorTerminate(&shareIterator);
    }

    cmListIteratorTerminate(&userIterator);
    if (FALSE == connectServer(pServer, pServer->useExtendedSecurity))
    {
        NQ_IPADDRESS dummyIp;

        /* check that server name is not ip */
        if (FALSE == ccUtilsNameToIp(pServer->item.name, &dummyIp))
        {
            NQ_INT               numIps = 0;
            const NQ_IPADDRESS   *ips = NULL;

            /* remove cache entry of server name from the cache, otherwise resolving has no effect */
            cmResolverRemoveFromCache(pServer->item.name);

            /* resolve ips again since the ips of the server could have changed */
            ips = cmResolverGetHostIps(NULL, pServer->item.name, &numIps);
            if ((0 < numIps) && (NULL != ips))
            {
                if (NULL != pServer->ips)
                {
                    cmMemoryFree(pServer->ips);
                }

                pServer->ips = ips;
                pServer->numIps = (NQ_COUNT)numIps;

                /* try to connect again with the newly resolved ips */
                if (FALSE == connectServer(pServer, pServer->useExtendedSecurity))
                {
                    pServer->isReconnecting = FALSE;
                }
            }
            else
            {
                pServer->isReconnecting = FALSE;
            }
        }
        else
        {
            pServer->isReconnecting = FALSE;
        }
    }

    cmListItemLock((CMItem *)pServer);
    cmListIteratorStart(&pServer->users, &userIterator);
    while (cmListIteratorHasNext(&userIterator))
    {
        CCUser * pUser;         /* next user pointer */
        NQ_BOOL  reconRes = FALSE;
        NQ_BOOL  isValidateNegotiateFailed = FALSE;

        pUser = (CCUser *)cmListIteratorNext(&userIterator);
        if (ccUserLogon(pUser))
        {
            /* try to use existing IPC share to perform negotiate validation */
            if (!pServer->isNegotiationValidated)
            {
                CMIterator sharesItr;
                CCShare *pIPCShare = NULL;

                cmListIteratorStart(&pUser->shares, &sharesItr);
                while (cmListIteratorHasNext(&sharesItr))
                {
                    CCShare *pShare = (CCShare *)cmListIteratorNext(&sharesItr);
                    if (0 == cmWStrcmp(((CMItem *)pShare)->name, cmIPCShareName()))
                    {
                        pIPCShare = pShare;
                        break;
                    }
                }

                cmListIteratorTerminate(&sharesItr);
                if (NULL != pIPCShare)
                {
                    if (TRUE == ccShareConnectExisting(pIPCShare, NULL, FALSE))
                    {
                        if (FALSE == pServer->smb->validateNegotiate(pServer, pUser, pIPCShare))
                        {
                            LOGERR(CM_TRC_LEVEL_ERROR, "Validate negotiation failed");
                            isValidateNegotiateFailed = TRUE;
                        }
                        else
                        {
                            pServer->isNegotiationValidated = TRUE;
                        }
                    }
                }
            }

            if (!isValidateNegotiateFailed)
            {
                result = ccUserReconnectShares(pUser);
                if (result)
                {
                    reconRes = TRUE;
                }
            }
        }

        if (!reconRes || isValidateNegotiateFailed)
        {
            /* user couldn't login or validate negotiate was failed, cleaning up everything */
            CMIterator  shrItr;

            LOGERR(CM_TRC_LEVEL_ERROR , "Reconnect failed.");
            cmListItemLock((CMItem *)pUser);
            cmListIteratorStart(&pUser->shares,&shrItr);
            while (cmListIteratorHasNext(&shrItr))
            {
                CMIterator  mntItr , iterator;
                CCShare *   pShare = (CCShare *)cmListIteratorNext(&shrItr);

                /* removing files and searches before removing the mount to avoid SegFault*/
                cmListIteratorStart(&pShare->files, &iterator);
                while (cmListIteratorHasNext(&iterator))
                {
                    CCFile * pFile = NULL;

                    pFile = (CCFile *)cmListIteratorNext(&iterator);
                    cmListItemUnlock((CMItem *)pFile);
                }

                cmListIteratorTerminate(&iterator);
                cmListIteratorStart(&pShare->searches, &iterator);
                while (cmListIteratorHasNext(&iterator))
                {
                    CCSearch * pSearch;

                    pSearch = (CCSearch *)cmListIteratorNext(&iterator);
                    cmListItemUnlock((CMItem *)pSearch);
                }

                cmListIteratorTerminate(&iterator);
                ccMountIterateMounts(&mntItr);
                while (cmListIteratorHasNext(&mntItr))
                {
                    CCMount * pMount = (CCMount *)cmListIteratorNext(&mntItr);

                    if (pMount->share == pShare)
                    {
                        cmListItemUnlock((CMItem *)pMount);
                    }
                }

                cmListIteratorTerminate(&mntItr);
            }

            cmListIteratorTerminate(&shrItr);
            cmListItemUnlock((CMItem *)pUser);
        }
    }

    cmListIteratorTerminate(&userIterator);
    cmListItemUnlock((CMItem *)pServer);

    if (result)
    {
        pServer->isReconnecting = FALSE;
    }

    /* start a thread that will turn off error mapping for reconnect */
    {
        SYThread thread;
        syThreadStart(FALSE, NQ_THREAD_PRIORITY_NONE, &thread, (void (*)(void))&reconnnectThreadBody, FALSE);
    }

Exit:
    cmListItemGive((CMItem *)pServer);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s", result ? "TRUE" : "FALSE");
    return result;
}

void ccServerIterateServers(CMIterator * iterator)
{
    cmListIteratorStart(&servers, iterator);
}

void ccServerIterateUsers(CCServer * server, CMIterator * iterator)
{
    cmListIteratorStart(&server->users, iterator);
}

void ccCloseAllConnections(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);
    cmListRemoveAndDisposeAll(&servers);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void ccCloseHiddenConnections(void)
{
    CMIterator serverIterator;      /* server iterator */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    cmListIteratorStart(&servers, &serverIterator);
    while (cmListIteratorHasNext(&serverIterator))
    {
        CCServer * pServer;             /* next server pointer */
        CMIterator userIterator;        /* user iterator */

        pServer = (CCServer *)cmListIteratorNext(&serverIterator);
        cmListIteratorStart(&pServer->users, &userIterator);
        while (cmListIteratorHasNext(&userIterator))
        {
            CCUser * pUser;                 /* next user pointer */
            CMIterator shareIterator;       /* share iterator */

            pUser = (CCUser *)cmListIteratorNext(&userIterator);
            cmListIteratorStart(&pUser->shares, &shareIterator);
            while (cmListIteratorHasNext(&shareIterator))
            {
                CCShare * pShare;               /* next share pointer */
                CMIterator mountIterator;       /* mount iterator */
                CMIterator  fileItr;            /* File Iterator*/
                NQ_BOOL doDisconnect = TRUE;    /* disconnection flag */


                pShare = (CCShare *)cmListIteratorNext(&shareIterator);

                ccMountIterateMounts(&mountIterator);
                while (cmListIteratorHasNext(&mountIterator))
                {
                    CCMount * pMount;               /* next mount pointer */

                    pMount = (CCMount *)cmListIteratorNext(&mountIterator);

                    if (pShare == pMount->share)
                    {
                        cmListIteratorTerminate(&mountIterator);
                        doDisconnect = FALSE;
                        break;
                    }
                }
                cmListIteratorTerminate(&mountIterator);
                cmListIteratorStart(&pShare->files, &fileItr);
                if (cmListIteratorHasNext(&fileItr))
                {
                    doDisconnect = FALSE;
                }
                cmListIteratorTerminate(&fileItr);
                if (doDisconnect)
                {
                    cmListItemCheck((CMItem *)pShare);
                }
            }
            cmListIteratorTerminate(&shareIterator);
        }
        cmListIteratorTerminate(&userIterator);
    }
    cmListIteratorTerminate(&serverIterator);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

NQ_BOOL ccServerWaitForCredits(CCServer * pServer, NQ_COUNT credits)
{
    NQ_INT newCredits;
    CMThread * curThread;   /* current thread pointer */
    NQ_BOOL res = TRUE;     /* operation result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pServer:%p credits:%d - %d", pServer, pServer->credits, credits);

    cmListItemTake(&pServer->item);
    syMutexTake(pServer->creditGuard);
    newCredits = pServer->credits - (NQ_INT)credits;
    while (newCredits <= 0)
    {
        syMutexGive(pServer->creditGuard);
        curThread = cmThreadGetCurrent();
        cmListItemAdd(&pServer->threads, &curThread->element.item, NULL);
        cmListItemGive(&pServer->item);

        res = cmThreadCondWait(&curThread->asyncCond, ccConfigGetTimeout());
        if (!res)
        {
            goto Exit;
        }
        cmListItemTake(&pServer->item);
        syMutexTake(pServer->creditGuard);
        newCredits = pServer->credits - (NQ_INT)credits;
    }
    pServer->credits = newCredits;
    syMutexGive(pServer->creditGuard);
    cmListItemGive(&pServer->item);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "%s credits: %d", res ? "TRUE" : "FALSE", pServer->credits);
    return res;
}

void ccServerPostCredits(CCServer * pServer, NQ_COUNT credits)
{
    CMThreadElement * element;  /* waiting thread */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pServer:%p credits:%d + %d", pServer, pServer->credits, credits);

    syMutexTake(pServer->creditGuard);
    pServer->credits += (NQ_INT)credits;
    syMutexGive(pServer->creditGuard);

    element = (CMThreadElement *)pServer->threads.first;
    if (NULL != element)
    {
        cmListItemRemove(&element->item);
        cmThreadCondSignal(&((CMThread *)element->thread)->asyncCond);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

#ifdef UD_NQ_INCLUDETRACE

void ccServerDump(void)
{
    cmListDump(&servers);
}

#endif /* UD_NQ_INCLUDETRACE */

#endif /* UD_NQ_INCLUDECIFSCLIENT */
