/*************************************************************************
 * Copyright (c) 2021 by Visuality Systems, Ltd.
 *
 *                     All Rights Reserved
 *
 * This item is the property of Visuality Systems, Ltd., and contains
 * confidential, proprietary, and trade-secret information. It may not
 * be transferred from the custody or control of Visuality Systems, Ltd.,
 * except as expressly authorized in writing by an officer of Visuality
 * Systems, Ltd. Neither this item nor the information it contains may
 * be used, transferred, reproduced, published, or disclosed, in whole
 * or in part, and directly or indirectly, except as expressly authorized
 * by an officer of Visuality Systems, Ltd., pursuant to written agreement.
 *
 *************************************************************************
 * FILE NAME     : $Workfile:$
 * ID            : $Header:$
 * REVISION      : $Revision:$
 *--------------------------------------------------------------------
 * DESCRIPTION   : Client mount operations
 *--------------------------------------------------------------------
 * MODULE        : Client
 * DEPENDENCIES  :
 *************************************************************************/

#include "ccmount.h"
#include "cmapi.h"
#include "ccutils.h"
#include "cmuuid.h"
#include "cmfinddc.h"

#ifdef UD_NQ_INCLUDECIFSCLIENT

#define VALIDATE_USER_MP_PREFIX     "\\validateUserMountPoint_"
#define VALIDATE_USER_MP_RAND_SIZE  10

/* -- Static data -- */

static CMList mounts;
static NQ_BOOL isModuleInitialized = FALSE;

const static NQ_WCHAR illegaMountPointNameChars[] = {
    cmWChar('"'), cmWChar('['), cmWChar(']'), cmWChar(';'), cmWChar('='), cmWChar(','),
    cmWChar('\\'), cmWChar('/'), cmWChar(':'), cmWChar('*'), cmWChar('?'), cmWChar('<'),
    cmWChar('>'), cmWChar('|')
    };


/* -- Local functions -- */

/*
 * Print mount-specific information 
 */
#ifdef UD_NQ_INCLUDETRACE
static void dumpOne(CMItem * pItem)
{
    CCMount * pMount = (CCMount *)pItem;
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "  Mount:: path: %s server %p share %p", cmWDump(pMount->path), pMount->server, pMount->share);
}

static void mountInformationDump(CCMountInfo mountPointInfo)
{
    NQ_COUNT    i;

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Mount information:");
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Creation time: 0x%08x,0x%08x", mountPointInfo.creationTime.high, mountPointInfo.creationTime.low);
    switch (mountPointInfo.userLogonType)
    {
        case CC_LOGON_TYPE_REGULAR:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Logon type: regular");
            break;
        }

        case CC_LOGON_TYPE_ANONYMOUS:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Logon type: anonymous");
            break;
        }

        case CC_LOGON_TYPE_GUEST:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Logon type: guest");
            break;
        }
        default:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Logon type: None");
        }
    }

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Server name: %s", cmWDump(mountPointInfo.serverName));
    switch (mountPointInfo.serverInfo.dialect)
    {
        case CC_SMB100:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Dialect: SMB 1");
            break;
        }
        case CC_SMB202:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Dialect: SMB 2.02");
            break;
        }

        case CC_SMB210:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Dialect: SMB 2.10");
            break;
        }

        case CC_SMB300:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Dialect: SMB 3.00");
            break;
        }

        case CC_SMB302:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Dialect: SMB 3.02");
            break;
        }

        case CC_SMB311:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Dialect: SMB 3.11");
            break;
        }
        default:
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Dialect: None");
        }
    }

#if defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311)
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Server capabilities: 0x%x",mountPointInfo.serverInfo.serverCapabilities);
#endif /* defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311) */
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Ips number: %d", mountPointInfo.serverInfo.numIps);

    if(NULL != mountPointInfo.serverInfo.ips)
    {
        for (i = 0; i < mountPointInfo.serverInfo.numIps; i++)
        {
            LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "%d: %s", i + 1, cmIPDump(&mountPointInfo.serverInfo.ips[i]));
        }
    }

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Server signing policy: 0x%x", mountPointInfo.serverInfo.serverSigningPolicy);
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Amount of data read and written data: %fGB, %fGB", mountPointInfo.statistics.readAmount, mountPointInfo.statistics.writeAmount);
}
#endif /* UD_NQ_INCLUDETRACE */

/*
 * Explicitly dispose mount point:
 *  - disconnects from the server
 *  - disposes private data  
 */
static void disposeMount(CCMount * pMount)
{
#ifdef UD_CC_INCLUDEDFS
    CMIterator shareLinkIter;
    CCShareLink *pShareLink;
#endif /* UD_CC_INCLUDEDFS */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "mount:%p", pMount);
    LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "About to dispose mount %s", cmWDump(pMount->item.name));

    if (NULL != pMount->credentials)
    {
        cmMemoryFree(pMount->credentials);
    }
    if (NULL != pMount->path)
    {
        cmMemoryFree(pMount->path);
    }
    if (NULL != pMount->pathPrefix)
    {
        cmMemoryFree(pMount->pathPrefix);
    }

#ifdef UD_CC_INCLUDEDFS
    cmListIteratorStart(&pMount->shareLinks, &shareLinkIter);

    while(cmListIteratorHasNext(&shareLinkIter))
    {
        pShareLink = (CCShareLink *)cmListIteratorNext(&shareLinkIter);
        cmListItemUnlock((CMItem *)pShareLink->pShare);
    }
    cmListIteratorTerminate(&shareLinkIter);
    cmListShutdown(&pMount->shareLinks);
#endif /* UD_CC_INCLUDEDFS */

    cmListItemRemoveAndDispose((CMItem *)pMount);
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

const NQ_WCHAR* ccMountGetIllegalCharacters(NQ_UINT *pLen)
{
    if (pLen)
    {
        *pLen = sizeof(illegaMountPointNameChars)/sizeof(illegaMountPointNameChars[0]);
    }

    return illegaMountPointNameChars;
}

/*
 * Callback for share unlock and disposal:
 *  - disconnects from the share
 *  - disposes private data  
 */
static NQ_BOOL unlockCallback(CMItem * pItem)
{
    disposeMount((CCMount *)pItem);
    return TRUE;
}

/* init mount point */
static void initMount(CCMount * pMount)
{
    pMount->path = NULL;
    pMount->pathPrefix = NULL;
    pMount->server = NULL;
    pMount->share = NULL;
    pMount->credentials = NULL;
}

/*
 * Check mount point name 
 */
static NQ_BOOL isValidMountPointName(const NQ_WCHAR * name)
{
    NQ_INT length;
    NQ_UINT i, lenIllegalChars = 0;
    NQ_BOOL result = FALSE;
    const NQ_WCHAR* illegalChars = ccMountGetIllegalCharacters(&lenIllegalChars);

    if (NULL == name)
    {
        goto Exit;
    }

    /* check length */
    length = (NQ_INT)cmWStrlen(name);
    if ((length < 2) || (length >= CC_MOUNT_NAME_LENGTH) || (name[0] != cmWChar('\\')))
    {
        goto Exit;
    }

    /* skip the first character */
    for (i = 0; i < lenIllegalChars; i++)
    {
        if (cmWStrchr(name + 1, illegalChars[i]) != NULL)
        {
            goto Exit;
        }
    }

    result = TRUE;

Exit:
    return result;
}

/*
 * Check remote path
 */

static NQ_BOOL isValidRemotePathName(const NQ_WCHAR * name)
{
    NQ_COUNT i; 
    const NQ_WCHAR * p;   
    NQ_BOOL result = FALSE;

    if ((NULL == name) || (CC_SHARE_PATH_LENGTH <= cmWStrlen(name)))
    {
        goto Exit;
    }

    for (i = 0, p = name; *p != cmWChar('\0'); p++)
    {
        if (*p == cmWChar('\\'))
        {
            i++;
        }
    }

    result = (i >= 3);

Exit:
    return result;
}

static NQ_WCHAR *stripLastCharacters(NQ_WCHAR *path)
{
    const NQ_WCHAR charsToStrip[] = {cmWChar(' ')};
    NQ_WCHAR *resultPath = path;
    NQ_INT lenPath, lenStrip;
    NQ_BOOL wasStripped = FALSE;

    if (NULL == path)
    {
        goto Exit;
    }

    lenPath = (NQ_INT)cmWStrlen(resultPath);

    for (--lenPath; lenPath >= 0; lenPath--)
    {
        lenStrip = (NQ_INT)CM_ARRAY_SIZE(charsToStrip);

        for (--lenStrip, wasStripped = FALSE; lenStrip >= 0 && !wasStripped; lenStrip--)
        {
            if (resultPath[lenPath] == charsToStrip[lenStrip])
            {
                resultPath[lenPath] = cmWChar(0);
                wasStripped = TRUE;
                break;
            }
        }
        if (!wasStripped)
        {
            break;
        }
    }

Exit:
    return resultPath;
}

static NQ_INT addMountWithCreds(const NQ_WCHAR * mountPoint, const NQ_WCHAR * remotePath, NQ_BOOL connect, const AMCredentials * userCredentials)
{
    CCMount * pMount = NULL;       /* pointer to mount */
    CCShare * pShare = NULL;       /* pointer to share */
    NQ_WCHAR * mountFilePath = NULL;
    NQ_WCHAR * filePathFromRemotePath = NULL;
    NQ_WCHAR * remotePathCopy = NULL;
    NQ_WCHAR * hostShare = NULL;
    NQ_BOOL remotePathHasSubfolder = FALSE;
    NQ_STATUS lastError = NQ_ERR_OK;
    NQ_INT result = NQ_FAIL;       /* return value */
#ifdef UD_NQ_INCLUDETRACE
    CCMountInfo mountPointInfo;
#endif /* UD_NQ_INCLUDETRACE */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "point:%p path:%p connect:%s", mountPoint, remotePath, connect ? "TRUE" : "FALSE");

    if ((NULL == mountPoint) || (NULL == remotePath))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS , "mountPoint:%s", cmWDump(mountPoint));
    LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS , "path:%s", cmWDump(remotePath));

    if (!isValidMountPointName(mountPoint))
    {
        lastError = NQ_ERR_BADPARAM;
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal mount point name: '%s'", cmWDump(mountPoint));
        goto Exit;
    }

    remotePathCopy = cmMemoryCloneWString(remotePath);
    if (NULL == remotePathCopy)
    {
        lastError = NQ_ERR_OUTOFMEMORY;
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    if (!isValidRemotePathName(stripLastCharacters(remotePathCopy)))
    {
        lastError = NQ_ERR_BADPARAM;
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal share path:'%s'", cmWDump(remotePathCopy));
        goto Exit;
    }

    pMount = (CCMount *)cmListItemFind(&mounts, mountPoint + 1, TRUE, TRUE);
    if (NULL != pMount)
    {
        cmListItemUnlock((CMItem *)pMount);
        lastError = NQ_ERR_BADPARAM;
        LOGERR(CM_TRC_LEVEL_ERROR, "Duplicate mount point");
        goto Exit;
    }

    pMount = (CCMount *)cmListItemCreateAndAdd(&mounts, sizeof(CCMount), mountPoint + 1, unlockCallback, CM_LISTITEM_LOCK, FALSE);
    if (NULL == pMount)
    {
        lastError = NQ_ERR_OUTOFMEMORY;
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        goto Exit;
    }

    cmListItemTake((CMItem *)pMount);
    initMount(pMount);
#ifdef UD_CC_INCLUDEDFS
    cmListStart(&pMount->shareLinks);
#endif /* UD_CC_INCLUDEDFS */

    pMount->creationTime = syGetTimeInMsec();
    pMount->path = remotePathCopy;
    remotePathCopy = NULL;

    /* check sub folder path existence */
    filePathFromRemotePath = ccUtilsFilePathFromRemotePath(pMount->path, TRUE);
    if (NULL != filePathFromRemotePath && (0 != cmWStrlen(filePathFromRemotePath)))
    {
        /* do not upper case the sub folder */
        hostShare = ccUtilsHostShareFromRemotePath(pMount->path);
        if (NULL != hostShare)
        {
            cmWStrupr(hostShare);
        }
        cmWStrncpy(pMount->path, hostShare, cmWStrlen(hostShare));

        remotePathHasSubfolder = TRUE;
    }
    else
    {
        cmWStrupr(pMount->path);
    }

    cmZeroUuid(&pMount->serverGUID);
    pMount->pathPrefix = NULL;
    if (NULL == userCredentials)
    {
        pMount->credentials = NULL; /* will be set later */
    }
    else
    {
        pMount->credentials = (AMCredentials *)cmMemoryAllocate(sizeof(AMCredentials));
        if (NULL == pMount->credentials)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            lastError = NQ_ERR_OUTOFMEMORY;
            goto Error;
        }

        syMemcpy(pMount->credentials, userCredentials, sizeof(AMCredentials));
    }

    pShare = ccShareConnect(pMount->path, pMount, &pMount->credentials, TRUE);
    if (NULL == pShare)
    {
        lastError = syGetLastError();
        LOGERR(CM_TRC_LEVEL_ERROR, "Failed to connect to share");
        goto Error;
    }

    pMount->serverGUID = pShare->user->server->serverGUID;

    if (NULL != pShare->dfsReferral)
    {
        cmListItemAddReference((CMItem *)pMount, (CMItem *)pShare->dfsReferral);
        pMount->share = pShare->dfsReferral;
        pMount->server = pShare->dfsReferral->user->server;
    }
    else
    {
        cmListItemAddReference((CMItem *)pMount, (CMItem *)pShare);
        pMount->share = pShare;
        pMount->server = pShare->user->server;
    }
    cmListItemUnlock((CMItem *)pShare);

    /* check sub folder path existence */
    if (remotePathHasSubfolder)
    {
        CCFileInfo fileInfo;

#ifdef UD_CC_INCLUDEDFS
        if (pMount->share->flags & CC_SHARE_IN_DFS)
        {
            mountFilePath = ccUtilsComposeLocalPathToFileByMountPoint(pMount->item.name, NULL);
            if (NULL == mountFilePath)
            {
                lastError = NQ_ERR_OUTOFMEMORY;
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                goto Error;
            }
        }
        else
#endif /* UD_CC_INCLUDEDFS */
        {
            mountFilePath = ccUtilsComposeLocalPathToFileByMountPoint(pMount->item.name, filePathFromRemotePath);
            if (NULL == mountFilePath)
            {
                lastError = NQ_ERR_OUTOFMEMORY;
                LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
                goto Error;
            }
        }
        if (!ccGetFileInformationByName(mountFilePath, &fileInfo))
        {
            lastError = syGetLastError();
            LOGERR(CM_TRC_LEVEL_ERROR, "Failed to validate sub folder");
            goto Error;
        }
        pMount->pathPrefix = cmMemoryCloneWString(filePathFromRemotePath);
        if (NULL == pMount->pathPrefix)
        {
            lastError = NQ_ERR_OUTOFMEMORY;
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            goto Error;
        }
    }
#ifdef UD_NQ_INCLUDETRACE
    pMount->item.dump = dumpOne; 

    if (FALSE == ccGetMountInformation(mountPoint, &mountPointInfo))
    {
        LOGMSG(CM_TRC_LEVEL_MESS_ALWAYS, "Failed getting mount information.");
    }
    else
    {
        mountInformationDump(mountPointInfo);

        cmMemoryFree(mountPointInfo.serverName);
        cmMemoryFree(mountPointInfo.serverInfo.ips);
    }
#endif /* UD_NQ_INCLUDETRACE */

    cmListItemGive((CMItem *)pMount);
    result = NQ_SUCCESS;
    goto Exit;

Error:
    cmListItemGive((CMItem *)pMount);
    cmListItemUnlock((CMItem *)pMount);

Exit:
    cmMemoryFree(remotePathCopy);
    cmMemoryFree(filePathFromRemotePath);
    cmMemoryFree(mountFilePath);
    cmMemoryFree(hostShare);

    sySetLastError(lastError);

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d error:0x%x", result, syGetLastError());
    return result;
}

/* -- API functions -- */

NQ_BOOL ccMountStart(void)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
       goto Exit;
    }

    cmListStart(&mounts);
#ifdef UD_NQ_INCLUDETRACE
    mounts.name = "mounts";
#endif /* UD_NQ_INCLUDETRACE */

    isModuleInitialized = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return TRUE;
}

void ccMountShutdown(void)
{
    CMIterator  iterator;

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON);

    if (isModuleInitialized)
    {
        cmListIteratorStart(&mounts, &iterator);
        while (cmListIteratorHasNext(&iterator))
        {
            CMItem * pItem;

            pItem = cmListIteratorNext(&iterator);
            cmListItemUnlock(pItem);
        }
        cmListIteratorTerminate(&iterator);
        cmListShutdown(&mounts);

        isModuleInitialized = FALSE;
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
}

void ccMountIterateMounts(CMIterator * iterator)
{
    cmListIteratorStart(&mounts, iterator);
}

void ccInitializeMountPointIdentifier(const CCMount * pMount, CCMountIdentifier * mountPointID)
{
    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "pMount: %p, mountPointID: %p", pMount, mountPointID);

    if ((NULL == pMount) || (NULL == pMount->item.name) || (0 == cmWStrlen(pMount->item.name)))
    {
        mountPointID->status = MOUNTPOINTID_CANNOTBEINITIALIZED;
        goto Exit;
    }

    mountPointID->status = MOUNTPOINTID_INITIALIZED;
    mountPointID->creationTime = pMount->creationTime;
    cmWStrcpy(mountPointID->mpName, pMount->item.name);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON);
    return;
}

void ccUpdateMountPointValidity(CCMountIdentifier * mountPointID)
{
    CCMount * pMount = NULL;           /* mount point pointer */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "mount point identifier: %p", mountPointID);

    if (MOUNTPOINTID_CANNOTBEINITIALIZED == mountPointID->status)
    {
        goto Exit;
    }

    pMount = (CCMount *)cmListItemFind(&mounts, mountPointID->mpName, TRUE, FALSE);
    if (NULL == pMount)
    {
        mountPointID->status = MOUNTPOINT_INVALID;
        LOGERR(CM_TRC_LEVEL_ERROR, "Couldn't find mount point");
        goto Exit;
    }

    if ((pMount->creationTime.high != mountPointID->creationTime.high) || (pMount->creationTime.low != mountPointID->creationTime.low))
    {
        mountPointID->status = MOUNTPOINT_INVALID;
        LOGERR(CM_TRC_LEVEL_ERROR, "Wrong creation time, creation time: 0x%08x,0x%08x, identifier creation time: 0x%08x,0x%08x", pMount->creationTime.high, pMount->creationTime.low, mountPointID->creationTime.high, mountPointID->creationTime.low);
        goto Exit;
    }

    mountPointID->status = MOUNTPOINT_VALID;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "status:%d", mountPointID->status);
    return;
}

#ifdef UD_NQ_INCLUDETRACE

void ccMountDump(void)
{
    cmListDump(&mounts);
}

#endif /* UD_NQ_INCLUDETRACE */

/* -- NQ API functions */

NQ_INT ccAddMountA(const NQ_CHAR *mountPoint, const NQ_CHAR *remotePath, NQ_BOOL connect)
{
    NQ_WCHAR * wPoint = NULL;
    NQ_WCHAR * wPath = NULL;
    NQ_INT res = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL);

    if (NULL == mountPoint || NULL == remotePath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    wPoint = cmMemoryCloneAString(mountPoint);
    wPath = cmMemoryCloneAString(remotePath);
    if (NULL == wPoint || NULL == wPath)
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    res = ccAddMount(wPoint, wPath, connect);

Exit:
    cmMemoryFree(wPoint);
    cmMemoryFree(wPath);
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d error:0x%x", res, syGetLastError());
    return res;
}

NQ_INT ccAddMount(const NQ_WCHAR * mountPoint, const NQ_WCHAR * remotePath, NQ_BOOL connect)
{
    NQ_INT result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "point:%s path:%s connect:%s", cmWDump(mountPoint), cmWDump(remotePath), connect ? "TRUE" : "FALSE");

    if (NULL == mountPoint || NULL == remotePath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    result = addMountWithCreds(mountPoint, remotePath, connect, NULL);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d error:0x%x", result, syGetLastError());
    return result;
}

NQ_INT ccAddMountWithCredsA(const NQ_CHAR * mountPoint, const NQ_CHAR * remotePath, NQ_BOOL connect, const AMCredentialsA * userCredentials)
{
    NQ_WCHAR * wPoint = NULL;
    NQ_WCHAR * wPath = NULL;
    NQ_INT result = NQ_FAIL;
    AMCredentials credentials;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "point:%s path:%s connect:%s credentials:%p", mountPoint, remotePath, connect ? "TRUE" : "FALSE", userCredentials);

    if (NULL == mountPoint || NULL == remotePath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    wPoint = cmMemoryCloneAString(mountPoint);
    wPath = cmMemoryCloneAString(remotePath);
    if (NULL == wPoint || NULL == wPath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    if (NULL == userCredentials)
    {
        result = addMountWithCreds(wPoint, wPath, connect, ccUserGetAnonymousCredentials());
    }
    else
    {
        amCredentialsAsciiiToW(&credentials, userCredentials);
        result = addMountWithCreds(wPoint, wPath, connect, &credentials);
    }

Exit:
    cmMemoryFree(wPoint);
    cmMemoryFree(wPath);
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d error:0x%x", result, syGetLastError());
    return result;
}

NQ_INT ccAddMountWithCreds(const NQ_WCHAR * mountPoint, const NQ_WCHAR * remotePath, NQ_BOOL connect, const AMCredentials * userCredentials)
{
    NQ_INT result = NQ_FAIL;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "point:%s path:%s connect:%s credentials:%p", cmWDump(mountPoint), cmWDump(remotePath), connect ? "TRUE" : "FALSE", userCredentials);

    if (NULL == mountPoint || NULL == remotePath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid input");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (NULL == userCredentials)
    {
        result = addMountWithCreds(mountPoint, remotePath, connect, ccUserGetAnonymousCredentials());
    }
    else
    {
        result = addMountWithCreds(mountPoint, remotePath, connect, userCredentials);
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d error:0x%x", result, syGetLastError());
    return result;
}

NQ_INT ccRemoveMountA(const NQ_CHAR *mountPoint)
{
    NQ_WCHAR * pointW = NULL; /* mount point in Unicode */
    NQ_INT res = NQ_FAIL;     /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL);

    if (NULL == mountPoint)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid mount point");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pointW = cmMemoryCloneAString(mountPoint);
    if (NULL == pointW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    res = ccRemoveMount(pointW);

Exit:
    cmMemoryFree(pointW);
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d error:0x%x", res, syGetLastError());
    return res;
}

NQ_INT ccRemoveMount(const NQ_WCHAR * mountPoint)
{
    CCMount * pMount = NULL;     /* mount point pointer */
    NQ_INT result = NQ_FAIL;     /* return value */
#ifdef UD_CC_INCLUDEDFS
    CMIterator shareLinkIter;
    CCShareLink *pShareLink;
#endif /* UD_CC_INCLUDEDFS */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "point:%s", cmWDump(mountPoint));

    if (NULL == mountPoint)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid mount point");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pMount = (CCMount *)cmListItemFind(&mounts, mountPoint + 1, TRUE , FALSE); /*no need to lock*/
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point not found");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

#ifdef UD_CC_INCLUDEDFS
    cmListIteratorStart(&pMount->shareLinks, &shareLinkIter);

    while(cmListIteratorHasNext(&shareLinkIter))
    {
        pShareLink = (CCShareLink *)cmListIteratorNext(&shareLinkIter);
        cmListItemUnlock((CMItem *)pShareLink->pShare);
    }
    cmListIteratorTerminate(&shareLinkIter);
    cmListShutdown(&pMount->shareLinks);
#endif /* UD_CC_INCLUDEDFS */

    cmListItemUnlock((CMItem *)pMount);

    result = NQ_SUCCESS;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%d error:0x%x", result, syGetLastError());
    return result;
}

CCMount * ccMountFind(const NQ_WCHAR * path)
{
    CCMount * pMount = NULL;           /* mount point pointer */
    const NQ_WCHAR * mpName = NULL;    /* mount point name */
    CCMount * pResult = NULL;          /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "path:%s", path ? cmWDump(path) : "");

    if (NULL == path)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid path");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    mpName = ccUtilsMountPointFromLocalPath(path);
    if (NULL == mpName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    pMount = (CCMount *)cmListItemFind(&mounts, mpName, TRUE, TRUE);
    cmMemoryFree(mpName);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point not found");
        goto Exit;
    }
    cmListItemTake((CMItem *)pMount);
    if (NULL == pMount->server || NULL == pMount->share)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point not connected");
        goto Exit;
    }
    pResult = pMount;

Exit:
    if (NULL != pMount)
    {
        cmListItemGive((CMItem *)pMount);
        cmListItemUnlock((CMItem *)pMount);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%p", pResult);
    return pResult;
}

NQ_BOOL ccResetCredentailsA(const NQ_CHAR * mountPoint)
{
    NQ_WCHAR * pointW = NULL;    /* mount pint in Unicode */
    NQ_BOOL res = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL);

    if (NULL == mountPoint)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid mount point");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pointW = cmMemoryCloneAString(mountPoint);
    if (NULL == pointW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }
    res = ccResetCredentails(pointW);

Exit:
    cmMemoryFree(pointW);
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", res ? "TRUE" : "FALSE", syGetLastError());
    return res;
}

NQ_BOOL ccResetCredentails(const NQ_WCHAR * mountPoint)
{
    CCMount * pMount = NULL;       /* mount point pointer */
    CCUser * pUser = NULL;         /* user structure pointer */
    CMIterator iServer;            /* servers iterator */
    NQ_BOOL result = FALSE;        /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "point:%s", cmWDump(mountPoint));

    if (NULL == mountPoint)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid mount point");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pMount = (CCMount *)cmListItemFind(&mounts, mountPoint + 1, TRUE , TRUE);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point not found");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    if (NULL == pMount->share)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point not connected to share");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    pUser = pMount->share->user;
    if (NULL == pUser)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Associated share is not connected");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    pMount->credentials = NULL;
    ccServerIterateServers(&iServer);
    while (cmListIteratorHasNext(&iServer))
    {
        CMIterator iUser;     /* users on server iterator */
        
        ccServerIterateUsers((CCServer *)cmListIteratorNext(&iServer), &iUser);
        while (cmListIteratorHasNext(&iUser))
        {
            CCUser * pNextUser;     /* next user pointer */

            pNextUser = (CCUser *)cmListIteratorNext(&iUser);
            if (0 == cmU64Cmp(&pUser->uid, &pNextUser->uid))
            {
                if (NULL != pNextUser->credentials)
                {
                    cmMemoryFree(pNextUser->credentials);
                    pNextUser->credentials = NULL;
                }
            }
        }
        cmListIteratorTerminate(&iUser);
    }
    cmListIteratorTerminate(&iServer);
    result = TRUE;

Exit:
    if (NULL != pMount)
    {
        cmListItemUnlock((CMItem *)pMount);
    }
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccValidateUserA(NQ_CHAR * server)
{
    NQ_WCHAR        *wServer = NULL;        /* server name in Unicode */
    NQ_BOOL         result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server: %s", server);

    if (NULL != server)
    {
        wServer = cmMemoryCloneAString(server);
        if (NULL == wServer)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }
    }

    result = ccValidateUserByCredentials(wServer, NULL);

Exit:
    if (NULL != wServer)
    {
        cmMemoryFree(wServer);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccValidateUser(NQ_WCHAR * server)
{
    NQ_BOOL         result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server: %s", cmWDump(server));

    result = ccValidateUserByCredentials(server, NULL);

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccValidateUserByCredentialsA(NQ_CHAR * server, const AMCredentialsA * userCredentials)
{
    NQ_WCHAR        *serverW = NULL;        /* server name in Unicode */
    AMCredentials   *credentialsW = NULL;   /* user credentials in Unicode */
    NQ_BOOL         result = FALSE;         /* return value */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server: %s, credentials:%p", server, userCredentials);

    if (NULL != server)
    {
        serverW = cmMemoryCloneAString(server);
        if (NULL == serverW)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }
    }

    if (NULL != userCredentials)
    {
        credentialsW = (AMCredentials *)cmMemoryAllocate(sizeof(AMCredentials));
        amCredentialsAsciiiToW(credentialsW, userCredentials);
    }

    result = ccValidateUserByCredentials(serverW, credentialsW);

Exit:
    if (NULL != serverW)
    {
        cmMemoryFree(serverW);
    }

    if (NULL != credentialsW)
    {
        cmMemoryFree(credentialsW);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccValidateUserByCredentials(NQ_WCHAR * server, const AMCredentials * userCredentials)
{
    NQ_WCHAR    *ipcPath = NULL;                        /* full network path to ipc share */
    NQ_CHAR     *dcA = NULL;                            /* DC name in ASCII */
    NQ_WCHAR    *dcW = NULL;                            /* DC name in Unicode */
    NQ_STATUS   status;                                 /* operation status */
    NQ_CHAR     *validateUserMountPointName = NULL;     /* mount point name for validate user */
    NQ_WCHAR    *validateUserMountPointNameW = NULL;    /* mount point name in Unicode for validate user */
    NQ_INT      i;                                      /* loop iterator */
    NQ_BOOL     result = FALSE;                         /* result - TRUE for success and FALSE otherwise */

    LOGFB(CM_TRC_LEVEL_FUNC_COMMON, "server: %s, credentials:%p", cmWDump(server), userCredentials);

    if ((NULL == server) || (cmWChar('\0') == *server))
    {
        LOGMSG(CM_TRC_LEVEL_MESS_NORMAL, "No server name supplied, trying to discover default domain controller name...");

        dcA = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_CHAR) * (CM_DNS_NAMELEN + CM_TRAILING_NULL));
        if (NULL == dcA)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }

        if (NQ_SUCCESS != (status = cmGetDCName(dcA, NULL)))
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Domain controller discovery failure");
            sySetLastError((NQ_UINT32)status);
            goto Exit;
        }

        dcW = cmMemoryCloneAString(dcA);
        if (NULL == dcW)
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }

        server = dcW;
    }

    ipcPath = ccUtilsComposeRemotePathToShare(server, cmIPCShareName());
    if (NULL == ipcPath)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    validateUserMountPointName = (NQ_CHAR *)cmMemoryAllocate(sizeof(NQ_CHAR) * (sizeof(VALIDATE_USER_MP_PREFIX) + VALIDATE_USER_MP_RAND_SIZE + CM_TRAILING_NULL));
    if (NULL == validateUserMountPointName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    syStrcpy(validateUserMountPointName, VALIDATE_USER_MP_PREFIX);
    for (i = (NQ_INT)syStrlen(VALIDATE_USER_MP_PREFIX); i < (NQ_INT)syStrlen(VALIDATE_USER_MP_PREFIX) + VALIDATE_USER_MP_RAND_SIZE; i++)
    {
        validateUserMountPointName[i] = (NQ_CHAR)('0' + (syRand()%10));
    }

    validateUserMountPointName[syStrlen(VALIDATE_USER_MP_PREFIX) + VALIDATE_USER_MP_RAND_SIZE] = '\0';
    validateUserMountPointNameW = cmMemoryCloneAString(validateUserMountPointName);
    if (NULL == validateUserMountPointNameW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    if (NQ_SUCCESS != addMountWithCreds(validateUserMountPointNameW, ipcPath, TRUE, userCredentials))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Could not add dummy mount");
        goto Exit;
    }

    if (NQ_SUCCESS != ccRemoveMount(validateUserMountPointNameW))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Could not remove dummy mount");
    }

    result = TRUE;

Exit:
    if (NULL != dcA)
    {
        cmMemoryFree(dcA);
    }

    if (NULL != dcW)
    {
        cmMemoryFree(dcW);
    }

    if (NULL != ipcPath)
    {
        cmMemoryFree(ipcPath);
    }

    if (NULL != validateUserMountPointName)
    {
        cmMemoryFree(validateUserMountPointName);
    }

    if (NULL != validateUserMountPointNameW)
    {
        cmMemoryFree(validateUserMountPointNameW);
    }

    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetMountInformationA(const NQ_CHAR * mountPoint, CCMountInfoA * mountPointInfo)
{
    NQ_WCHAR *  mountPointW = NULL;         /* mount point Unicode string */
    CCMountInfo mountPointInfoW;            /* mount point information */
    NQ_BOOL     result = FALSE;             /* result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "mountPoint:%s, mountPointInfo:%p", mountPoint ? mountPoint : "", mountPointInfo);

    if (NULL == mountPointInfo)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid mountPointInfo");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if (NULL == mountPoint)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal mount point name: %s", mountPoint ? mountPoint : "");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    mountPointW = cmMemoryCloneAString(mountPoint);
    if (NULL == mountPointW)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "out of memory");
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    result = ccGetMountInformation(mountPointW, &mountPointInfoW);
    cmMemoryFree(mountPointW);
    if (TRUE == result)
    {
        mountPointInfo->creationTime = mountPointInfoW.creationTime;
        mountPointInfo->serverInfo = mountPointInfoW.serverInfo;
        mountPointInfo->userLogonType = mountPointInfoW.userLogonType;
        mountPointInfo->serverName = cmMemoryCloneWStringAsAscii(mountPointInfoW.serverName);
        cmMemoryFree(mountPointInfoW.serverName);
        mountPointInfo->statistics = mountPointInfoW.statistics;
    }

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

NQ_BOOL ccGetMountInformation(const NQ_WCHAR * mountPoint, CCMountInfo * mountPointInfo)
{
    CCMount *pMount = NULL;     /* mount pointer */
    NQ_BOOL result = FALSE;     /* result */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "mountPoint:%s, mountPointInfo:%p", cmWDump(mountPoint), mountPointInfo);

    if (NULL == mountPointInfo)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Invalid mount point information pointer");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    if ((NULL == mountPoint) || !isValidMountPointName(mountPoint))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal mount point name: %s", mountPoint ? cmWDump(mountPoint) : "");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pMount = (CCMount *)cmListItemFind(&mounts, mountPoint + 1, TRUE, FALSE);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point not found: %s", cmWDump(mountPoint));
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    cmListItemTake((CMItem *)pMount);
    mountPointInfo->creationTime = pMount->creationTime;
#if defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311)
    switch (pMount->server->serverDialectRevision)
    {
        case CCCIFS_ILLEGALSMBREVISION:
        {
            mountPointInfo->serverInfo.dialect = CC_SMB100;
            break;
        }

        case SMB2_DIALECTREVISION:
        {
            mountPointInfo->serverInfo.dialect = CC_SMB202;
            break;
        }

        case SMB2_1_DIALECTREVISION:
        {
            mountPointInfo->serverInfo.dialect = CC_SMB210;
            break;
        }

        case SMB3_DIALECTREVISION:
        {
            mountPointInfo->serverInfo.dialect = CC_SMB300;
            break;
        }

        case SMB3_0_2_DIALECTREVISION:
        {
            mountPointInfo->serverInfo.dialect = CC_SMB302;
            break;
        }

        case SMB3_1_1_DIALECTREVISION:
        {
            mountPointInfo->serverInfo.dialect = CC_SMB311;
            break;
        }

        default:
        {
            LOGERR(CM_TRC_LEVEL_ERROR, "Invalid dialect:0x%x", pMount->server->serverDialectRevision);
            cmListItemGive((CMItem *)pMount);
            sySetLastError(NQ_ERR_WRONGDIALECT);
            goto Exit;
        }
    }
#endif /* defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311) */

    if (TRUE == pMount->share->user->isAnonymous)
    {
        mountPointInfo->userLogonType = CC_LOGON_TYPE_ANONYMOUS;
    }
    else if (TRUE == pMount->share->user->isGuest)
    {
        mountPointInfo->userLogonType = CC_LOGON_TYPE_GUEST;
    }
    else
    {
        mountPointInfo->userLogonType = CC_LOGON_TYPE_REGULAR;
    }

#if defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311)
    mountPointInfo->serverInfo.serverCapabilities = pMount->server->serverCapabilites;
#endif /* defined(UD_NQ_INCLUDESMB2) || defined(UD_NQ_INCLUDESMB3) || defined(UD_NQ_INCLUDESMB311) */
    mountPointInfo->serverName = cmMemoryCloneWString(pMount->server->item.name);
    if (NULL == mountPointInfo->serverName)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "out of memory");
        cmListItemGive((CMItem *)pMount);
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    if (pMount->server->numIps > 0)
    {
        mountPointInfo->serverInfo.ips = (NQ_IPADDRESS *)cmMemoryAllocate(pMount->server->numIps * (NQ_UINT)sizeof(NQ_IPADDRESS));
        syMemcpy(mountPointInfo->serverInfo.ips, pMount->server->ips, pMount->server->numIps * sizeof(NQ_IPADDRESS));
    }
    else
    {
        mountPointInfo->serverInfo.ips = NULL;
    }

    mountPointInfo->serverInfo.numIps = pMount->server->numIps;
    mountPointInfo->serverInfo.serverSigningPolicy = pMount->server->serverSecurityMode;
    mountPointInfo->statistics = pMount->share->statistics;

    cmListItemGive((CMItem *)pMount);
    result = TRUE;

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_COMMON, "result:%s error:0x%x", result ? "TRUE" : "FALSE", syGetLastError());
    return result;
}

/* Description
 * Per each mount, shares that are mounted as part of DFS
 * are kept in a list so they can be unlocked on remove mount.
 * Each share should appear once only, so if an existing share is
 * again listed, one item unlock should be performed. *
 */
void ccMountAddShareLink(CCMount *pMount, CCShare *pShare)
{
#ifdef UD_CC_INCLUDEDFS
    CMIterator shareLinkIter;
    CCShareLink *pShareLink;
    NQ_BOOL shareExists = FALSE;
#endif /* UD_CC_INCLUDEDFS */

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "pMount: %p, pShare:%s", pMount, cmWDump((const NQ_WCHAR *)pShare->item.name));

#ifdef UD_CC_INCLUDEDFS
    if (NULL == pMount)
    {
        goto Exit;
    }

    /* first check if this share is already in this mounts list */
    cmListIteratorStart(&pMount->shareLinks, &shareLinkIter);
    while (cmListIteratorHasNext(&shareLinkIter))
    {
        pShareLink = (CCShareLink *)cmListIteratorNext(&shareLinkIter);
        if (pShare == pShareLink->pShare)
        {
            /* each connect the share is locked. if this share connect is related to this mount and lokced by this mount. one time is enough. */
            cmListItemUnlock((CMItem *)pShare);
            shareExists = TRUE;
            break;
        }
    }
    cmListIteratorTerminate(&shareLinkIter);

    if (FALSE == shareExists)
    {
        pShareLink = (CCShareLink *)cmListItemCreateAndAdd(&pMount->shareLinks, sizeof(CCShareLink), pShare->item.name, NULL,  CM_LISTITEM_NOLOCK , FALSE);
        if (NULL == pShareLink)
        {
            LOGERR(CM_TRC_LEVEL_ERROR , "Out of memory");
            sySetLastError(NQ_ERR_OUTOFMEMORY);
            goto Exit;
        }

        pShareLink->pShare = pShare;
    }

Exit:
#endif /* UD_CC_INCLUDEDFS */

    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL);
}

NQ_GUID ccGetServerIdA(const NQ_CHAR * mountPoint)
{
    NQ_GUID serverGUID;
    NQ_WCHAR * wPoint = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "mountPoint:%s", mountPoint ? mountPoint : "");

    cmZeroUuid(&serverGUID);

    if (NULL == mountPoint)
    {
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    wPoint = cmMemoryCloneAString(mountPoint);
    if (NULL == wPoint)
    {
        sySetLastError(NQ_ERR_OUTOFMEMORY);
        goto Exit;
    }

    serverGUID = ccGetServerId(wPoint);

Exit:
    cmMemoryFree(wPoint);
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "serverGUID:%s error:0x%x", cmUuidDump(&serverGUID), syGetLastError());
    return serverGUID;
}

NQ_GUID ccGetServerId(const NQ_WCHAR * mountPoint)
{
    NQ_GUID serverGUID;
    CCMount *pMount = NULL;

    LOGFB(CM_TRC_LEVEL_FUNC_PROTOCOL, "mountPoint:%s", mountPoint ? cmWDump(mountPoint) : "");

    cmZeroUuid(&serverGUID);

    if ((NULL == mountPoint) || !isValidMountPointName(mountPoint))
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Illegal mount point name: %s", mountPoint ? cmWDump(mountPoint) : "");
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }

    pMount = (CCMount *)cmListItemFind(&mounts, mountPoint + 1, TRUE, TRUE);
    if (NULL == pMount)
    {
        LOGERR(CM_TRC_LEVEL_ERROR, "Mount point not found: %s", cmWDump(mountPoint));
#ifdef UD_NQ_INCLUDETRACE
        ccMountDump();
#endif /* UD_NQ_INCLUDETRACE */
        sySetLastError(NQ_ERR_BADPARAM);
        goto Exit;
    }
    serverGUID = pMount->serverGUID;
    cmListItemUnlock((CMItem *)pMount);

Exit:
    LOGFE(CM_TRC_LEVEL_FUNC_PROTOCOL, "serverGUID:%s error:0x%x", cmUuidDump(&serverGUID), syGetLastError());
    return serverGUID;
}

#endif /* UD_NQ_INCLUDECIFSCLIENT */
