package com.cogniance.acs.server;

import com.cogniance.acs.messaging.AbstractMessage;
import com.cogniance.acs.messaging.Inform;
import com.cogniance.acs.messaging.api.Message;
import com.cogniance.acs.messaging.api.helping.ACSMessage;
import com.cogniance.acs.messaging.api.helping.CPEMessage;
import com.cogniance.acs.server.api.AwaitTimeoutException;
import com.cogniance.acs.util.exception.ACSException;
import com.cogniance.acs.util.exception.ACSRuntimeException;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.xml.soap.SOAPException;
import java.io.IOException;
import java.util.concurrent.TimeUnit;

import static javax.servlet.http.HttpServletResponse.*;

class CWMPServlet extends HttpServlet {

    private static final Logger logger = LoggerFactory.getLogger(CWMPServlet.class);

    private static final String DEVICE_ATTRIBUTE_ID = DeviceImpl.class.getName();

    private final ACSImpl acs;

    public CWMPServlet(ACSImpl acs) {
        this.acs = acs;
    }

    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        try {
            CPEMessage cpeMessage = readMessage(request);
            validateSOAPHeaders(cpeMessage, request);
            handleNewHttpSession(request, cpeMessage);
            DeviceImpl device = getDeviceFromHttpSession(request.getSession(true));
            logger.debug("Received message from {}:\n{}", device, cpeMessage);
            SessionImpl session = device.getCurrentSession();
            session.registerCPEMessage(cpeMessage);
            ACSMessage acsMessage;
            try {
                acsMessage = session.awaitACSMessage(1, TimeUnit.MINUTES);
            } catch (AwaitTimeoutException e) {
                logger.error("No answer for CPE {}.", device);
                throw e;
            }
            logger.debug("Sending message to {}:{}", device, acsMessage);
            if (acsMessage == null) {
                response.setStatus(SC_NO_CONTENT);
            } else {
                response.setStatus(SC_OK);
                writeMessage(acsMessage, response);
            }
            handleSessionFinished(device.getCurrentSession(), device);
//        } catch (RespondWithHttpFailure httpFailure) {
//            response.sendError(httpFailure.getHttpStatusCode()); TODO add ability to send HTTP errors
        } catch (ValidationException e) {
            logger.warn("Received invalid package from CPE.", e);
            response.sendError(SC_BAD_REQUEST, e.getMessage());
        } catch (InterruptedException ex) {
            logger.info("Server has been interrupted.");
        } catch (Throwable ex) {
            logger.warn("Internal server error.", ex);
            response.sendError(SC_INTERNAL_SERVER_ERROR, ex.getMessage());
        }
    }

    private CPEMessage readMessage(HttpServletRequest request) throws IOException, ValidationException {
        String messageData = IOUtils.toString(request.getInputStream());
        if (messageData.length() == 0) {
            return null;
        }
        Message message = AbstractMessage.createMessageByContent(messageData);
        if (!(message instanceof CPEMessage)) {
            throw new ValidationException("Expected CPE message but found: %s", message);
        }
        return (CPEMessage) message;
    }

    private void validateSOAPHeaders(Message message, HttpServletRequest request) throws ValidationException {
        if (message != null) {
            if (request.getContentType() == null || !request.getContentType().startsWith("text/xml")) {
                throw new ValidationException("Content type of 'text/xml' must be passed if request is non-empty.");
            }
            if (request.getHeader("SOAPAction") == null) {
                throw new ValidationException("SOAPAction header must be set if request is non-empty.");
            }
        } else {
            if (request.getContentType() != null) {
                throw new ValidationException("Content type must not be passed if request is empty.");
            }
            if (request.getHeader("SOAPAction") != null) {
                throw new ValidationException("SOAPAction header must not be passed if request is empty.");
            }
        }
    }

    private void handleNewHttpSession(HttpServletRequest request, CPEMessage message) throws ValidationException {
        boolean messageIsNotAnInform = !(message instanceof Inform);
        boolean alreadyInSession = request.getSession(false) != null;
        if (messageIsNotAnInform || alreadyInSession) {
            return;
        }
        Inform inform = (Inform) message;
        if (inform.getDeviceId() == null) {
            throw new ValidationException("DeviceId shouldn't be empty.");
        }

        String serialNumber = inform.getDeviceId().getSerialNumber();
        DeviceImpl device = acs.getDevice(serialNumber);
        if (device == null) {
            throw new ValidationException("Unknown device ID: %s", serialNumber);
        }
        if (device.isInSession()) {
            device.registerProtocolViolation("Device started a new session while previous was still in progress.");
            device.closeSession();
        }
        device.initializeSession();

        request.getSession(true).setAttribute(DEVICE_ATTRIBUTE_ID, device);
    }

    private DeviceImpl getDeviceFromHttpSession(HttpSession session) throws ValidationException {
        Object value = session.getAttribute(DEVICE_ATTRIBUTE_ID);
        if (!(value instanceof DeviceImpl)) {
            throw new ValidationException("HTTP session is not attached to any device.");
        }
        return (DeviceImpl) value;
    }

    private void writeMessage(ACSMessage responseMessage, HttpServletResponse response) throws IOException {
        try {
            responseMessage.toSOAP().writeTo(response.getOutputStream());
        } catch (SOAPException e) {
            throw new ACSRuntimeException(e);
        }
        response.getOutputStream().close();
    }

    private void handleSessionFinished(SessionImpl session, DeviceImpl device) {
        if (!session.isFinished()) {
            return;
        }
        device.closeSession();
    }

    private static class ValidationException extends ACSException {
        public ValidationException(Throwable cause) {
            super(cause);
        }

        public ValidationException(String message, Object... argsAndCause) {
            super(message, argsAndCause);
        }
    }
}
