package com.cogniance.acs.messaging;

import com.cogniance.acs.messaging.api.Message;
import com.cogniance.acs.util.exception.ACSRuntimeException;

import javax.xml.soap.*;
import javax.xml.transform.stream.StreamSource;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.StringReader;
import java.util.Date;
import java.util.Iterator;
import java.util.UUID;

import static com.cogniance.acs.messaging.util.DataTypes.formatBoolean;
import static com.cogniance.acs.messaging.util.DataTypes.parseBoolean;
import static com.cogniance.acs.util.soap.SOAPUtils.getChildElementValue;

/**
 * The message class that all messages should extend. Create a MessageFactory that creates the needed message from a
 * given SOAP?
 */

abstract public class AbstractMessage implements Message {

    protected static final String URN_CWMP = "urn:dslforum-org:cwmp-1-0";
    protected static final String CWMP = "cwmp";
    protected static final String SOAP_ENC = "soapenc";

    public static final String MESSAGE_SKELETON = "<soap:Envelope xmlns:soap=\"http://schemas.xmlsoap.org/soap/envelope/\" " +
            "xmlns:soapenc=\"http://schemas.xmlsoap.org/soap/encoding/\" " +
            "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" " +
            "xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\" " +
            "xmlns:cwmp=\"urn:dslforum-org:cwmp-1-0\" " +
            "soap:encodingStyle=\"http://schemas.xmlsoap.org/soap/encoding/\">\r\n" +
            "<soap:Header>\r\n</soap:Header>\r\n<soap:Body>\r\n</soap:Body>\r\n</soap:Envelope>";

    protected static final SOAPFactory SOAP_FACTORY;
    protected static final MessageFactory MESSAGE_FACTORY;

    static {
        try {
            SOAP_FACTORY = SOAPFactory.newInstance();
            MESSAGE_FACTORY = MessageFactory.newInstance();
        } catch (SOAPException e) {
            throw new ACSRuntimeException("Failed to initialize SOAP factory for messages.", e);
        }
    }


    private final String name; // Name of the message, ie. InformResponse

    private SOAPMessage cachedSOAP;
    // Variables for the header
    private String id;
    private String holdRequests;

    /**
     * Get the name of the message request
     *
     * @param msg
     *         The soap message
     * @return The name of the message
     */
    public static String getRequestName(SOAPMessage msg) throws SOAPException {
        if (msg.getSOAPBody().hasFault())
            return "Fault";

        SOAPBodyElement sbe = null;
        Iterator iter = msg.getSOAPBody().getChildElements();
        while (iter.hasNext()) {
            Node n = (Node) iter.next();
            if (n.getNodeType() == Node.ELEMENT_NODE) {
                sbe = (SOAPBodyElement) n;
            }
        }
        String name = sbe.getNodeName();
        if (name.startsWith(AbstractMessage.CWMP + ":"))
            name = name.substring(5);

        return name;
    }

    /**
     * Create a message. If messageSoap is null an empty message is created.
     *
     * @param messageName
     *         The name of the message
     * @param soapMessage
     *         The soap message
     * @return A message or null if it couldnt be created.
     */
    public static AbstractMessage createMessage(String messageName, SOAPMessage soapMessage) throws SOAPException {
        AbstractMessage message = createMessageByName(messageName);
        if (soapMessage != null) {
            message.fromSOAP(soapMessage);
        }
        return message;
    }

    public static AbstractMessage createMessageByContent(String messageContent) throws IOException {
        if (messageContent == null || messageContent.isEmpty()) {
            return null;
        }
        try {
            SOAPMessage soapMessage = MESSAGE_FACTORY.createMessage();

            SOAPPart soapPart = soapMessage.getSOAPPart();
            StreamSource source = new StreamSource(new StringReader(messageContent));
            soapPart.setContent(source);

            return createMessageByContent(soapMessage);
        } catch (SOAPException e) {
            throw new ACSRuntimeException("Failed to create message from: %s", messageContent, e);
        }
    }

    /**
     * Create a message, based on a SOAPMessage. Gets the name using getRequestName and then calling createMessage(String msgName, SOAPMessage msgSoap)
     * @param messageContent The soap message
     * @return The message
     */
    public static AbstractMessage createMessageByContent(SOAPMessage messageContent) throws IOException, SOAPException {
        String messageName = getRequestName(messageContent);
        return createMessage(messageName, messageContent);
    }

    public static AbstractMessage createMessageByName(String messageName) {
        String packageName = AbstractMessage.class.getPackage().getName();
        try {
            Class<?> messageClass = Class.forName(packageName + "." + messageName);
            return (AbstractMessage) messageClass.newInstance();
        } catch (Exception ex) {
            throw new ACSRuntimeException("Failed to create message %s", messageName, ex);
        }
    }

    /**
     * Constructor
     *
     * @param name
     *         Name of the message
     */
    public AbstractMessage(String name) {
        this.id = UUID.randomUUID().toString();
        this.holdRequests = "false";
        this.name = name;
    }

    /**
     * Create the corresponding response message
     *
     * @return The response message
     */
    @Override
    public AbstractMessage createResponseMessage() {
        if (this.name.endsWith("Response") || this.name.equals("Fault")) {
            return null;
        }
        AbstractMessage response = createMessageByName(this.name + "Response");
        response.setId(this.id);
        return response;
    }

    /**
     * Get the SOAP body of a message
     *
     * @return The body
     */
    private SOAPBodyElement getSOAPRequestBody(SOAPBody soapBody) throws SOAPException {
        Iterator it = soapBody.getChildElements();
        while (it.hasNext()) {
            Node node = (Node) it.next();
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                return (SOAPBodyElement) node;
            }
        }
        throw new SOAPException("No request body was found.");
    }

    /**
     * Parses set SOAP message. Also calls the parseBody method.
     */
    public void fromSOAP(SOAPMessage soapMessage) throws SOAPException {
        this.cachedSOAP = soapMessage;
        parseSOAPHeader(soapMessage.getSOAPHeader());
        parseSOAPBody(getSOAPRequestBody(soapMessage.getSOAPBody()));
    }

    protected void parseSOAPBody(SOAPBodyElement body) throws SOAPException {
        throw new UnsupportedOperationException();
    }

    private void parseSOAPHeader(SOAPHeader header) {
        if (header == null) {
            return;
        }
        try {
            this.id = null;
            this.id = getChildElementValue(header, Names.ID);
        } catch (Exception ignore) {
        }
        try {
            this.holdRequests = null;
            this.holdRequests = getChildElementValue(header, Names.HoldRequests);
        } catch (Exception ignore) {
        }
    }

    /**
     * Get the SOAP message
     */
    public SOAPMessage toSOAP() throws SOAPException {
        SOAPMessage soapMessage = createSOAPMessage();
        fillSOAPHeader(soapMessage.getSOAPHeader());
        Name messageName = SOAP_FACTORY.createName(getName(), CWMP, URN_CWMP);
        fillSOAPBody(soapMessage.getSOAPBody().addBodyElement(messageName));
        this.cachedSOAP = soapMessage;
        return soapMessage;
    }

    protected void fillSOAPBody(SOAPBodyElement body) throws SOAPException {
        throw new UnsupportedOperationException();
    }

    /**
     * Set the SOAP message to an empty message
     */
    private SOAPMessage createSOAPMessage() throws SOAPException {
        ByteArrayInputStream in = new ByteArrayInputStream(MESSAGE_SKELETON.getBytes());
        SOAPMessage soapMessage;
        try {
            soapMessage = MESSAGE_FACTORY.createMessage(null, in);
        } catch (IOException e) {
            throw new ACSRuntimeException("I/O failure while working with ByteArrayInputStream :-/", e);
        }
        return soapMessage;
    }

    private void fillSOAPHeader(SOAPHeader header) throws SOAPException {
        SOAPHeaderElement idHeader = header.addHeaderElement(Names.ID);
        idHeader.setValue(this.id);
        idHeader.setAttribute("mustUnderstand", "1");
        SOAPHeaderElement holdRequestsHeader = header.addHeaderElement(Names.HoldRequests);
        holdRequestsHeader.setValue(formatBoolean(isHoldRequests()));
        holdRequestsHeader.setAttribute("mustUnderstand", "1");
    }

    /**
     * Remove invalid XML characters from a string.
     *
     * @param s
     *         The XML string
     * @return String with only valid characters
     */
    public static String removeInvalidXMLCharacters(String s) {
        StringBuilder out = new StringBuilder();// Used to hold the output.
        int curr; // Used to reference the current character.

        int i = 0;
        while (i < s.length()) {
            curr = s.codePointAt(i);// This is the unicode code of the character.
            if ((curr == 0x9) ||// Consider testing larger ranges first to improve speed.
                    (curr == 0xA) ||
                    (curr == 0xD) ||
                    ((curr >= 0x20) && (curr <= 0xD7FF)) ||
                    ((curr >= 0xE000) && (curr <= 0xFFFD)) ||
                    ((curr >= 0x10000) && (curr <= 0x10FFFF))) {

                out.append(Character.toChars(curr));
            }
            i += Character.charCount(curr);// Increment with the number of code units(java chars) needed to represent a Unicode char.
        }

        return out.toString();
    }

    @Override
    public void setHoldRequests(String holdRequests) {
        this.holdRequests = holdRequests;
    }

    @Override
    public void setHoldRequests(boolean holdRequests) {
        this.holdRequests = formatBoolean(holdRequests);
    }

    @Override
    public String getHoldRequests() {
        return holdRequests;
    }

    @Override
    public boolean isHoldRequests() {
        return holdRequests == null || parseBoolean(holdRequests);
    }

    @Override
    public void setId(String id) {
        this.id = id;
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getName() {
        return name;
    }

    /**
     * Convert the SOAPMessage to a string
     *
     * @return String representation of the SOAPMessage, or {@link Object#toString()} if any errors.
     */
    @Override
    public String toString() {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            if (cachedSOAP == null) {
                toSOAP();
            }
            cachedSOAP.writeTo(baos);
            return new String(baos.toByteArray());
        } catch (Exception ignore) {
            return super.toString();
        }
    }

    private static class Names {

        private static final Name HoldRequests;
        private static final Name ID;

        static {
            try {
                HoldRequests = SOAP_FACTORY.createName("HoldRequests", CWMP, URN_CWMP);
                ID = SOAP_FACTORY.createName("ID", CWMP, URN_CWMP);
            } catch (SOAPException e) {
                throw new ACSRuntimeException("SOAP initialization failed.", e);
            }
        }
    }


}
